/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/levy/cdf' );
import Levy = require( './../../../../../base/dists/levy/ctor' );
import entropy = require( './../../../../../base/dists/levy/entropy' );
import logcdf = require( './../../../../../base/dists/levy/logcdf' );
import logpdf = require( './../../../../../base/dists/levy/logpdf' );
import mean = require( './../../../../../base/dists/levy/mean' );
import median = require( './../../../../../base/dists/levy/median' );
import mode = require( './../../../../../base/dists/levy/mode' );
import pdf = require( './../../../../../base/dists/levy/pdf' );
import quantile = require( './../../../../../base/dists/levy/quantile' );
import stdev = require( './../../../../../base/dists/levy/stdev' );
import variance = require( './../../../../../base/dists/levy/variance' );

/**
* Interface describing the `levy` namespace.
*/
interface Namespace {
	/**
	* Lévy distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 10.0, 0.0, 3.0 );
	* // returns ~0.584
	*
	* y = ns.cdf( 0.3, 0.0, 3.0 );
	* // returns ~0.002
	*
	* var myCDF = ns.cdf.factory( 2.0, 3.0 );
	* y = myCDF( 100.0 );
	* // returns ~0.861
	*
	* y = myCDF( 10.0 );
	* // returns ~0.54
	*
	* y = myCDF( 2.0 );
	* // returns 0.0
	*/
	cdf: typeof cdf;

	/**
	* Lévy distribution.
	*/
	Levy: typeof Levy;

	/**
	* Returns the differential entropy for a Lévy distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns entropy
	*
	* @example
	* var y = ns.entropy( 0.0, 1.0 );
	* // returns ~3.324
	*
	* @example
	* var y = ns.entropy( 5.0, 2.0 );
	* // returns ~4.018
	*
	* @example
	* var y = ns.entropy( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, 0.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Lévy distribution logarithm of cumulative distribution function (logCDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 10.0, 0.0, 3.0 );
	* // returns ~-0.538
	*
	* y = ns.logcdf( 0.3, 0.0, 3.0 );
	* // returns ~-6.215
	*
	* var mylogcdf = ns.logcdf.factory( 2.0, 3.0 );
	* var y = mylogcdf( 100.0 );
	* // returns ~-0.15
	*
	* y = mylogcdf( 10.0 );
	* // returns ~-0.616
	*
	* y = mylogcdf( 2.0 );
	* // returns -Infinity
	*/
	logcdf: typeof logcdf;

	/**
	* Lévy distribution natural logarithm of probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.209
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	* y = mylogpdf( 11.0 );
	* // returns ~-1.572
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value for a Lévy distribution with location `mu` and scale `c`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns expected value
	*
	* @example
	* var y = ns.mean( 0.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.mean( 5.0, 2.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.mean( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, 0.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a Lévy distribution with location `mu` and scale `c`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 0.0, 1.0 );
	* // returns ~2.198
	*
	* @example
	* var y = ns.median( 5.0, 2.0 );
	* // returns ~9.396
	*
	* @example
	* var y = ns.median( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, 0.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode for a Lévy distribution with location `mu` and scale `c`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 0.0, 1.0 );
	* // returns ~0.333
	*
	* @example
	* var y = ns.mode( 5.0, 2.0 );
	* // returns ~5.667
	*
	* @example
	* var y = ns.mode( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, 0.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Lévy distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 1.0 );
	* // returns ~0.11
	*
	* var mypdf = ns.pdf.factory( 10.0, 2.0 );
	* y = mypdf( 11.0 );
	* // returns ~0.208
	*/
	pdf: typeof pdf;

	/**
	* Lévy distribution quantile function.
	*
	* @param p - input value
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~1.386
	*
	* var myQuantile = ns.quantile.factory( 10.0, 2.0 );
	*
	* y = myQuantile( 0.5 );
	* // returns ~14.396
	*/
	quantile: typeof quantile;

	/**
	* Returns the standard deviation for a Lévy distribution with location `mu` and scale `c`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 0.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.stdev( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, 0.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a Lévy distribution with location `mu` and scale `c`.
	*
	* ## Notes
	*
	* -   If provided `c <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param c - scale parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 0.0, 1.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.variance( 5.0, 2.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.variance( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, 0.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Lévy distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
