/* SPDX-FileCopyrightText: 2024 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import registerCommand from 'gmp/command';
import EntitiesCommand from 'gmp/commands/entities';
import EntityCommand from 'gmp/commands/entity';
import GmpHttp from 'gmp/http/gmp';
import {XmlResponseData} from 'gmp/http/transform/fastxml';
import logger from 'gmp/log';
import {Element} from 'gmp/models/model';
import Role, {RoleElement} from 'gmp/models/role';
import {isArray} from 'gmp/utils/identity';

interface RoleCommandCreateParams {
  name: string;
  comment?: string;
  users?: string[] | string;
}

interface RoleCommandSaveParams {
  id: string;
  name: string;
  comment?: string;
  users?: string[] | string;
}

const log = logger.getLogger('gmp.commands.roles');

export class RoleCommand extends EntityCommand<Role, RoleElement> {
  constructor(http: GmpHttp) {
    super(http, 'role', Role);
  }

  async create({name, comment = '', users = []}: RoleCommandCreateParams) {
    const data = {
      cmd: 'create_role',
      name,
      comment,
      users: isArray(users) ? users.join(',') : '',
    };
    log.debug('Creating new role', data);
    return await this.entityAction(data);
  }

  async save({id, name, comment = '', users = []}: RoleCommandSaveParams) {
    const data = {
      cmd: 'save_role',
      id,
      name,
      comment,
      users: isArray(users) ? users.join(',') : '',
    };
    log.debug('Saving role', data);
    await this.httpPost(data);
  }

  getElementFromRoot(root: XmlResponseData): RoleElement {
    // @ts-expect-error
    return root.get_role.get_roles_response.role;
  }
}

export class RolesCommand extends EntitiesCommand<Role> {
  constructor(http: GmpHttp) {
    super(http, 'role', Role);
  }

  getEntitiesResponse(root: Element) {
    // @ts-expect-error
    return root.get_roles.get_roles_response;
  }
}

registerCommand('role', RoleCommand);
registerCommand('roles', RolesCommand);
