/*
 * serialize.h
 *
 * Copyright (c) 1998 Michael Kropfberger <michael.kropfberger@gmx.net>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifndef __SERIALIZE_H__
#define __SERIALIZE_H__

#include <qobject.h>
#include <qstring.h>
#include <qlist.h>

#include "vartuple.h"

/********************************************************************/
/**
  *  @short Serialize is the superclass of all serializable objects. 
  *
  *  It gives the possibility to dump/load its actual State to/from 
  *  a specially formatted string. 
  *  So all pvm-sendable objects _must_ inherit from @ref #Serialize .
  *  @author Michael Kropfberger <michael.kropfberger@gmx.net>
**/
class Serialize : public QObject
{ Q_OBJECT

 public:
  /*********************************************************************
   * allows the KPvm class to dump/restore the objectState to the
   * actual @ref Serialize object
  **/ 
  friend class KPvm;

  Serialize(QObject *parent=0, const char *name=0);

 protected:

    /*********************************************************************
      *    dumps the actual object state into a QList of VarTuples
      *
      *    It has to be implemented by all subclasses!
      *    @see #setObjState
      *    @see #setFixedOrderObjState
      *    @return the state-representing QStringList.
    **/    
    virtual QList<VarTuple> objState() const = 0;

    /*********************************************************************
      *    sets the actual object state according to the given 
      *    QList of VarTuples
      *
      *    Here it may be, that the VarTags may _not_ be valid,
      *    so we have to assume the same fixed order as we used
      *    when dumping in @ref #setObjState
      *
      *    It has to be implemented by all subclasses!
      *    @see #setObjState
      *    @see #objState
      *    @param fixedOrderedState the state-representing QList of VarTuples in a fixed order
    **/    
    virtual void setFixedOrderObjState(QList<VarTuple> fixedOrderedState) = 0 ;

    /*********************************************************************
      *    sets the actual object state according to the given 
      *    QList of VarTuples
      *
      *    All VarTags have to be valid, but the order may be different
      *    not even complete from when we've dumped in @ref #setObjState
      *
      *    So we have to implement a search for finding the correct 
      *    [VarTag:VarType:Value]-Tuple for our variables to fill with.
      *    I would suggest to use QDict for quick access.
      *
      *    It has to be implemented by all subclasses!
      *    @see #objState
      *    @see #setFixedOrderObjState
      *    @param state the state-representing QList of VarTuples in any 
      *            (incomplete) order
    **/    
    virtual void setObjState(QList<VarTuple> state) = 0;

};  // Serialize


#endif
