/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include <sal/types.h>
#include <cppunit/TestAssert.h>
#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/plugin/TestPlugIn.h>

#include <rtl/ustrbuf.hxx>

#include <com/sun/star/util/DateTime.hpp>
#include <com/sun/star/util/Date.hpp>
#include <com/sun/star/util/Duration.hpp>

#include <sfx2/Metadatable.hxx>
#include <sfx2/XmlIdRegistry.hxx>

#include <boost/scoped_ptr.hpp>


using namespace ::com::sun::star;


namespace {

class MetadatableTest
    : public ::CppUnit::TestFixture
{
public:
    virtual void setUp() SAL_OVERRIDE;
    virtual void tearDown() SAL_OVERRIDE;

    void test();

    CPPUNIT_TEST_SUITE(MetadatableTest);
    CPPUNIT_TEST(test);
    CPPUNIT_TEST_SUITE_END();

private:
};

void MetadatableTest::setUp()
{
}

void MetadatableTest::tearDown()
{
}


class MockMetadatable
    : public ::sfx2::Metadatable
{
private:
    ::sfx2::IXmlIdRegistry & m_rRegistry;

public:
    MockMetadatable(::sfx2::IXmlIdRegistry & i_rReg,
            bool const i_isInClip = false)
        : m_rRegistry(i_rReg)
        , m_bInClipboard(i_isInClip), m_bInUndo(false), m_bInContent(true) {}
    bool m_bInClipboard;
    bool m_bInUndo;
    bool m_bInContent;
    virtual bool IsInClipboard() const SAL_OVERRIDE { return m_bInClipboard; }
    virtual bool IsInUndo() const SAL_OVERRIDE { return m_bInUndo; }
    virtual bool IsInContent() const SAL_OVERRIDE { return m_bInContent; }
    virtual ::sfx2::IXmlIdRegistry& GetRegistry() SAL_OVERRIDE { return m_rRegistry; }
    virtual ::com::sun::star::uno::Reference<
        ::com::sun::star::rdf::XMetadatable > MakeUnoObject() SAL_OVERRIDE { return 0; }
};

static bool operator==(beans::StringPair p1, beans::StringPair p2)
{
    return p1.First == p2.First && p1.Second == p2.Second;
}

void MetadatableTest::test()
{
    boost::scoped_ptr< ::sfx2::IXmlIdRegistry > const pReg(
        ::sfx2::createXmlIdRegistry(false) );
    boost::scoped_ptr< ::sfx2::IXmlIdRegistry > const pRegClip(
        ::sfx2::createXmlIdRegistry(true) );

    MockMetadatable m1(*pReg);
    MockMetadatable m2(*pReg);
    MockMetadatable m3(*pReg);
    MockMetadatable m4(*pReg);
    MockMetadatable m5(*pReg);
    OUString empty;
    OUString content( "content.xml" );
    OUString styles( "styles.xml" );
    OUString sid1( "id1" );
    OUString sid2( "id2" );
    OUString sid3( "id3" );
    OUString sid4( "id4" );
    beans::StringPair id1(content, sid1);
    beans::StringPair id2(content, sid2);
    beans::StringPair id3(content, sid3);
    beans::StringPair id4(styles,  sid4);
    beans::StringPair id3e(empty,  sid3);
    beans::StringPair id4e(empty,  sid4);
    m1.SetMetadataReference(id1);
    CPPUNIT_ASSERT_MESSAGE("set failed", m1.GetMetadataReference() == id1);
    try {
        m2.SetMetadataReference(id1);
        CPPUNIT_ASSERT_MESSAGE("set duplicate succeeded", false);
    } catch (const lang::IllegalArgumentException &) { }
    m1.SetMetadataReference(id1);
    CPPUNIT_ASSERT_MESSAGE("set failed (existing)",
            m1.GetMetadataReference() == id1);
    m1.EnsureMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("ensure failed (existing)",
            m1.GetMetadataReference() == id1);

    m2.EnsureMetadataReference();
    beans::StringPair m2id(m2.GetMetadataReference());
    CPPUNIT_ASSERT_MESSAGE("ensure failed", !m2id.Second.isEmpty());
    m2.EnsureMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("ensure failed (idempotent)",
            m2.GetMetadataReference() == m2id);

    m1.m_bInUndo = true;
    CPPUNIT_ASSERT_MESSAGE("move to undo failed",
            m1.GetMetadataReference().Second.isEmpty());

    m1.m_bInUndo = false;
    CPPUNIT_ASSERT_MESSAGE("move from undo failed",
            m1.GetMetadataReference() == id1);

    m1.m_bInUndo = true;
    try {
        m2.SetMetadataReference(id1); // steal!
    } catch (lang::IllegalArgumentException &) {
        CPPUNIT_FAIL("set duplicate to undo failed");
    }
    m1.m_bInUndo = false;
    CPPUNIT_ASSERT_MESSAGE("move from undo: duplicate",
            m1.GetMetadataReference().Second.isEmpty());

    m3.RegisterAsCopyOf(m2);
    CPPUNIT_ASSERT_MESSAGE("copy: source", m2.GetMetadataReference() == id1);
    CPPUNIT_ASSERT_MESSAGE("copy: duplicate",
            m3.GetMetadataReference().Second.isEmpty());
    m4.RegisterAsCopyOf(m3);
    CPPUNIT_ASSERT_MESSAGE("copy: source", m2.GetMetadataReference() == id1);
    CPPUNIT_ASSERT_MESSAGE("copy: duplicate",
            m3.GetMetadataReference().Second.isEmpty());
    CPPUNIT_ASSERT_MESSAGE("copy: duplicate",
            m4.GetMetadataReference().Second.isEmpty());
    m2.m_bInUndo = true;
    CPPUNIT_ASSERT_MESSAGE("duplicate to undo",
            m3.GetMetadataReference() == id1);
    CPPUNIT_ASSERT_MESSAGE("duplicate to undo",
            m2.GetMetadataReference().Second.isEmpty());
    m2.m_bInUndo = false;
    CPPUNIT_ASSERT_MESSAGE("duplicate from undo",
            m2.GetMetadataReference() == id1);
    CPPUNIT_ASSERT_MESSAGE("duplicate from undo",
            m3.GetMetadataReference().Second.isEmpty());

    m4.EnsureMetadataReference(); // new!
    beans::StringPair m4id(m4.GetMetadataReference());
    CPPUNIT_ASSERT_MESSAGE("ensure on duplicate",
            !m4id.Second.isEmpty() && !(m4id == id1));

    MockMetadatable mc1(*pRegClip, true); // in clipboard
    MockMetadatable mc2(*pRegClip, true);
    MockMetadatable mc3(*pRegClip, true);
    MockMetadatable mc4(*pRegClip, true);
    MockMetadatable m2p(*pReg);
    MockMetadatable m3p(*pReg);

    mc1.SetMetadataReference(id2);
    CPPUNIT_ASSERT_MESSAGE("set failed", mc1.GetMetadataReference() == id2);
    try {
        mc2.SetMetadataReference(id2);
        CPPUNIT_FAIL("set duplicate succeeded");
    } catch (const lang::IllegalArgumentException &) { }
    mc1.SetMetadataReference(id2);
    CPPUNIT_ASSERT_MESSAGE("set failed (existing)",
            mc1.GetMetadataReference() == id2);
    mc1.EnsureMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("ensure failed (existing)",
            mc1.GetMetadataReference() == id2);
    mc2.EnsureMetadataReference();
    beans::StringPair mc2id(mc2.GetMetadataReference());
    CPPUNIT_ASSERT_MESSAGE("ensure failed", !mc2id.Second.isEmpty());
    mc2.EnsureMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("ensure failed (idempotent)",
            mc2.GetMetadataReference() == mc2id);
    mc2.RemoveMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("remove failed",
            mc2.GetMetadataReference().Second.isEmpty());

    // set up mc2 as copy of m2 and mc3 as copy of m3
    mc3.RegisterAsCopyOf(m3);
    CPPUNIT_ASSERT_MESSAGE("copy to clipboard (latent)",
            mc3.GetMetadataReference().Second.isEmpty() );
    mc2.RegisterAsCopyOf(m2);
    CPPUNIT_ASSERT_MESSAGE("copy to clipboard (non-latent)",
            mc2.GetMetadataReference() == id1);
    // paste mc2 to m2p and mc3 to m3p
    m2p.RegisterAsCopyOf(mc2);
    CPPUNIT_ASSERT_MESSAGE("paste from clipboard (non-latent)",
            m2p.GetMetadataReference().Second.isEmpty() );
    m3p.RegisterAsCopyOf(mc3);
    CPPUNIT_ASSERT_MESSAGE("paste from clipboard (latent)",
            m3p.GetMetadataReference().Second.isEmpty() );
    // delete m2, m2p, m3
    m2.RemoveMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("remove failed",
            m2.GetMetadataReference().Second.isEmpty());
    CPPUNIT_ASSERT_MESSAGE("paste-remove (non-latent)",
            m2p.GetMetadataReference() == id1);
    m2p.RemoveMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("remove failed",
            m2p.GetMetadataReference().Second.isEmpty());
    CPPUNIT_ASSERT_MESSAGE("paste-remove2 (non-latent)",
            m3.GetMetadataReference() == id1);
    m3.RemoveMetadataReference();
    CPPUNIT_ASSERT_MESSAGE("remove failed",
            m3.GetMetadataReference().Second.isEmpty());
    CPPUNIT_ASSERT_MESSAGE("paste-remove (latent)",
            m3p.GetMetadataReference() == id1);
    // delete mc2
    mc2.SetMetadataReference(beans::StringPair());
    CPPUNIT_ASSERT_MESSAGE("in clipboard becomes non-latent",
            mc3.GetMetadataReference().Second.isEmpty() );
    // paste mc2
    m2p.RegisterAsCopyOf(mc2);
    CPPUNIT_ASSERT_MESSAGE("remove-paste",
            m2p.GetMetadataReference().Second.isEmpty());
    CPPUNIT_ASSERT_MESSAGE("remove-paste (stolen)",
            m3p.GetMetadataReference() == id1);

    // auto-detect stream
    m5.SetMetadataReference(id3e);
    CPPUNIT_ASSERT_MESSAGE("auto-detect (content)",
            m5.GetMetadataReference() == id3);
    m5.m_bInContent = false;
    m5.SetMetadataReference(id4e);
    CPPUNIT_ASSERT_MESSAGE("auto-detect (styles)",
            m5.GetMetadataReference() == id4);
}


CPPUNIT_TEST_SUITE_REGISTRATION(MetadatableTest);

}

CPPUNIT_PLUGIN_IMPLEMENT();

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
