#import "AppController.h"
#import <libpq-fe.h>

// When a notice is raised,  this function gets called
void handle_notice (void *arg, const char *message)
{
    AppController *me = (AppController *)arg;
    [me appendToLog:[NSString stringWithFormat:@"%s", message]
            color:[NSColor magentaColor]];
}

@implementation AppController

- (id)init
{
    [super init];
    
    // These are used for the history
    executedStatements = [[NSMutableArray alloc] init];
    currentIndex = 0;
    
    return self;
}

// Get a list of all the types from the database
- (void)fetchTypes
{
    typeResult = PQexec(connection, 
        "select oid, typname from pg_type order by oid");
}

// Clear the previous result, hold onto the latest
- (void)setLastResult:(PGresult *)aResult
{
    // Release the last result
    if (lastResult) {
        PQclear(lastResult);
    }
    
    // Set a pointer to the new result
    lastResult = aResult;
}

// Logging methods

- (void)clearLog
{
    [outputTextView setString:@""];
    [outputTextView display];
}

- (void)appendToLog:(NSString *)s color:(NSColor *)c
{   
    NSFont *font = [outputTextView font];
    NSRange r;
    NSTextStorage *ts = [outputTextView textStorage];
    NSMutableDictionary *dict = [[NSMutableDictionary alloc] init];
    [dict setObject:c forKey:NSForegroundColorAttributeName];
    [dict setObject:font forKey:NSFontAttributeName];
    NSAttributedString *toAdd = [[NSAttributedString alloc] initWithString:s attributes:dict];
    [dict release];
    [ts appendAttributedString:toAdd];
    [toAdd release];
    r.location = [ts length];
    r.length = 0;
    [outputTextView scrollRangeToVisible:r];
    [outputTextView display];
}


- (IBAction)open:(id)sender
{
    NSOpenPanel *op = [NSOpenPanel openPanel];
    int returnCode = [op runModalForTypes:nil];
    if (returnCode != NSOKButton) {
        return;
    }
    NSString *path = [op filename];
    NSString *contents = [NSString stringWithContentsOfFile:path];
    [inputTextView setString:contents];
    [inputTextView display];
}


- (void)showTuples
{
    int fields = PQnfields(lastResult);
    int difference;
    int i;
    NSTableColumn *currentColumn;
    NSArray *oldColumns;
    
    // Get the right number of columns in the table view
    oldColumns = [tableView tableColumns];
    difference = fields - [oldColumns count];
    
    // Too many? remove some columns
    if (difference < 0) {
        for (i = [oldColumns count] - 1; i >= fields; i--) {
            [tableView removeTableColumn:[oldColumns objectAtIndex:i]];
        }
    }

    // Too few? add some columns
    if (difference > 0) {
        for (i=0; i < difference; i++) {
            currentColumn = [[NSTableColumn alloc] init];
            [currentColumn setEditable:NO];
            [currentColumn setMinWidth:70.0];
            [tableView addTableColumn:currentColumn];
            [currentColumn release];
        }
    }
    
    // Relabel the columns
    oldColumns = [tableView tableColumns];
    for (i = 0; i < [oldColumns count]; i++) {
        NSString *fieldString;
        char *nameOfField;
        nameOfField = PQfname(lastResult, i);
        fieldString = [NSString stringWithUTF8String:nameOfField];
        
        // Get the column to label
        currentColumn = [oldColumns objectAtIndex:i];
        [currentColumn setIdentifier:fieldString];
        [[currentColumn headerCell] setStringValue:fieldString];
        // We will use this tag to figure out which column
        // this represents.  (Important if the columns are
        // reordered!)
        [[currentColumn headerCell] setTag:i];

    }
}

- (void)executeSQL:(NSString *)sql
{
    PGresult *result;
    ExecStatusType resultStatus;
    NSString *logString;

    // Run the SQL
    result = PQexec(connection, [sql UTF8String]);
    
    // Check the status
    resultStatus = PQresultStatus(result);
    
    // OK, with rows
    if (resultStatus == PGRES_TUPLES_OK) {
        int ntuples = PQntuples(result);
        int fields = PQnfields(result);
        logString = [NSString stringWithFormat:@"%d records, %d fields\n", ntuples, fields];
        [self appendToLog:logString color:[NSColor blueColor]];
        [self setLastResult:result];
        return;
    }
        
    // Bad
    if (resultStatus == PGRES_NONFATAL_ERROR) {
        logString = [NSString stringWithFormat:@"Non-fatal - %s", PQresultErrorMessage(result)];
        [self appendToLog:logString color:[NSColor orangeColor]];
    }
    
    // Real bad
    if (resultStatus == PGRES_FATAL_ERROR) {
         logString = [NSString stringWithFormat:@"Fatal - %s", PQresultErrorMessage(result)];
        [self appendToLog:logString color:[NSColor redColor]];
    }
    
    PQclear(result);
}

- (IBAction)execute:(id)sender
{
    int i, count;
    [self clearLog];
    [self setLastResult:NULL];
    NSArray *listOfStatements;
    
    // Read the input and add it to the history
    NSString *sql = [[inputTextView string] copy];
    [executedStatements addObject:sql];
    [sql release];
    currentIndex = [executedStatements count] - 1;
    
    // Start the progress indicator (in case of a long operation)
    [progressIndicator startAnimation:nil];
    
    // Break up the statements
    // FIXME: this won't handle escaped ; correctly.
    listOfStatements = [sql componentsSeparatedByString:@";"];

    // Execute each of the statements
    count = [listOfStatements count];

    

    for (i=0; i< count; i++) {
        NSString *currentStatement;
        currentStatement = [listOfStatements objectAtIndex:i];
        
        // execute the line
        [self executeSQL:currentStatement];
    }
        
    if (lastResult) {
        [self showTuples];
    }
    
    [self appendToLog:@"Done.\n" color:[NSColor blackColor]];
    
    // update tableView
    [tableView reloadData];
    
    // Stop progress indicator
    [progressIndicator stopAnimation:nil];

}
- (void)disconnect
{
    // At termination,  disconnect gracefully
    NSLog(@"Disconnected");
    PQfinish(connection);
    connection = NULL;
}

- (IBAction)connect:(id)sender
{
    // Read the text fields,  skip empty ones
    NSMutableString *connString = [NSMutableString string];
    NSString *input = [userTextField stringValue];
    if ([input length]) 
        [connString appendFormat:@"user=%@ ", input];

    input = [passwordTextField stringValue];
    if ([input length]) 
        [connString appendFormat:@"password=%@ ", input];
    
    input = [dbTextField stringValue];
    if ([input length]) 
        [connString appendFormat:@"dbname=%@ ", input];

    input = [hostTextField stringValue];
    if ([input length]) 
        [connString appendFormat:@"host=%@ ", input];

    NSLog(@"connection string:\'%@\'", connString);
    
    connection = PQconnectdb([connString UTF8String]);
    
    // Bring up an alert panel to show status
    if (PQstatus(connection) == CONNECTION_OK) {
        NSRunAlertPanel(@"Good", @"Connection is up", nil, nil,nil);
        [loginSheet orderOut:nil];
        
        // Prepare for executing and logging for connection
        [self fetchTypes];
        PQsetNoticeProcessor(connection, handle_notice, self);
        
        // Hide the sheet
        [NSApp endSheet:loginSheet];
    }  else {
        NSRunAlertPanel(@"Bad", @"Connection not up = %d", nil, nil,nil, PQstatus(connection));
        
        // Give them another chance (clear connection, don't hide sheet)
        [self disconnect];
    }
}

- (void)awakeFromNib
{
    // Set attributes of the progress indicator
    [progressIndicator setStyle:NSProgressIndicatorSpinningStyle];
    [progressIndicator setDisplayedWhenStopped:NO];
    
    // Immediately show login panel
    [NSApp beginSheet:loginSheet	
        modalForWindow:mainWindow 
        modalDelegate:self 
        didEndSelector:NULL 
        contextInfo:NULL];
        
    // Handle tool tips for table view explicitly
    [self registerToolTipHandler];
}

// Table view datasource methods

- (int)numberOfRowsInTableView:(NSTableView *)tableView
{
    if (lastResult)
        return PQntuples(lastResult);
    else 
        return 0;
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    char *value;
    NSString *dataString;
    int fieldNum = [[tableColumn headerCell] tag];
    BOOL isNull = PQgetisnull(lastResult, row, fieldNum);
    if (isNull) {
        return @"<NULL>";
    }
    value = PQgetvalue(lastResult, row, fieldNum);
    dataString = [NSString stringWithUTF8String:value];
    return dataString;
}

// Diconnect gracefully when the user quits

- (void)applicationWillTerminate:(NSNotification *)notification
{
    [self disconnect];
}

// Allow the user to create another connection
- (IBAction)reconnect:(id)sender
{
    [self disconnect];
    
    // This will bring up the login sheet again
    [self awakeFromNib];
}

// History methods

- (IBAction)previous:(id)sender
{
    if (currentIndex == 0) {
        NSBeep();
        return;
    }
    currentIndex--;
    NSString *statement = [executedStatements objectAtIndex:currentIndex];
    [inputTextView setString:statement];
    [inputTextView display];
}
- (IBAction)next:(id)sender
{
    if (currentIndex >= [executedStatements count] - 1) {
        NSBeep();
        return;
    }
    currentIndex++;
    NSString *statement = [executedStatements objectAtIndex:currentIndex];
    [inputTextView setString:statement];
    [inputTextView display];
}

- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{
    // Disable next and previous menu items as 
    // necessary
    if ([menuItem action] == @selector(next:)) {
        return (currentIndex < [executedStatements count] - 1);
    }
    if ([menuItem action] == @selector(previous:)) {
        return (currentIndex > 0);
    }
    return YES;
}

// For types in tool tips,  used fetched type names + OIDs

- (NSString *)typeNameForOID:(Oid)typeOID
{
    int ntuples = PQntuples(typeResult);
    int oidForRow;
    char *oidAsString;
    int i;
    for (i=0; i < ntuples; i++) {
        oidAsString = PQgetvalue(typeResult, i, 0);
        oidForRow = atoi(oidAsString);
        if (oidForRow == typeOID) {
            return [NSString stringWithFormat:@"%s",
                            PQgetvalue(typeResult, i, 1)];
        }
    }
    return @"Not found";
}
- (IBAction)quit:(id)sender
{
    [NSApp endSheet:loginSheet];
    [NSApp terminate:nil];
}


// Methods to deal with tooltips for table view

- (NSString *)view:(NSView *)view stringForToolTip:(NSToolTipTag)tag point:(NSPoint)point userData:(void *)data
{
    NSTableColumn *tc;
    int columnIndex = [tableView columnAtPoint:point];
    if (columnIndex >= 0) {
        tc = [[tableView tableColumns] objectAtIndex:columnIndex];
        int tag = [[tc headerCell] tag];
        Oid typeOID = PQftype(lastResult,tag);
        if (typeOID == 0) {
            return nil;
        }
        return [self typeNameForOID:typeOID];
    }
    return nil;
}

- (void)registerToolTipHandler
{
    int count, i;
    
    [tableView removeAllToolTips];
    
    count = [[tableView tableColumns] count];
    for (i=0; i<count; i++) {
        [tableView addToolTipRect:[tableView rectOfColumn:i] owner:self userData:nil];
    }
}

- (void)windowDidResize:(NSNotification *)notification
{
    [self registerToolTipHandler];
}

- (void)splitViewDidResizeSubviews:(NSNotification *)notification
{
    [self registerToolTipHandler];
}

@end
