% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reconc_MixCond.R
\name{reconc_MixCond}
\alias{reconc_MixCond}
\title{Probabilistic forecast reconciliation of mixed hierarchies via conditioning}
\usage{
reconc_MixCond(
  S,
  fc_bottom,
  fc_upper,
  bottom_in_type = "pmf",
  distr = NULL,
  num_samples = 20000,
  return_type = "pmf",
  suppress_warnings = FALSE,
  seed = NULL
)
}
\arguments{
\item{S}{Summing matrix (n x n_bottom).}

\item{fc_bottom}{A list containing the bottom base forecasts, see details.}

\item{fc_upper}{A list containing the upper base forecasts, see details.}

\item{bottom_in_type}{A string with three possible values:
\itemize{
\item 'pmf' if the bottom base forecasts are in the form of pmf, see details;
\item 'samples' if the bottom base forecasts are in the form of samples;
\item 'params'  if the bottom base forecasts are in the form of estimated parameters.
}}

\item{distr}{A string describing the type of bottom base forecasts ('poisson' or 'nbinom').

This is only used if \code{bottom_in_type='params'}.}

\item{num_samples}{Number of samples drawn from the reconciled distribution.
This is ignored if \code{bottom_in_type='samples'}; in this case, the number of
reconciled samples is equal to the number of samples of the base forecasts.}

\item{return_type}{The return type of the reconciled distributions.
A string with three possible values:
\itemize{
\item 'pmf' returns a list containing the reconciled marginal pmf objects;
\item 'samples' returns a list containing the reconciled multivariate samples;
\item 'all' returns a list with both pmf objects and samples.
}}

\item{suppress_warnings}{Logical. If \code{TRUE}, no warnings about samples
are triggered. If \code{FALSE}, warnings are generated. Default is \code{FALSE}. See Details.}

\item{seed}{Seed for reproducibility.}
}
\value{
A list containing the reconciled forecasts. The list has the following named elements:
\itemize{
\item \code{bottom_reconciled}: a list containing the pmf, the samples (matrix n_bottom x \code{num_samples}) or both,
depending on the value of \code{return_type};
\item \code{upper_reconciled}: a list containing the pmf, the samples (matrix n_upper x \code{num_samples}) or both,
depending on the value of \code{return_type}.
}
}
\description{
Uses importance sampling to draw samples from the reconciled
forecast distribution, obtained via conditioning, in the case of a mixed hierarchy.
}
\details{
The base bottom forecasts \code{fc_bottom} must be a list of length n_bottom, where each element is either
\itemize{
\item a PMF object (see details below), if \code{bottom_in_type='pmf'};
\item a vector of samples, if \code{bottom_in_type='samples'};
\item a list of parameters, if \code{bottom_in_type='params'}:
\itemize{
\item lambda for the Poisson base forecast if \code{distr}='poisson', see \link[stats]{Poisson};
\item size and prob (or mu) for the negative binomial base forecast if \code{distr}='nbinom',
see \link[stats]{NegBinomial}.
}
}

The base upper forecasts \code{fc_upper} must be a list containing the parameters of
the multivariate Gaussian distribution of the upper forecasts.
The list must contain only the named elements \code{mu} (vector of length n_upper)
and \code{Sigma} (n_upper x n_upper matrix)

A PMF object is a numerical vector containing the probability mass function of a discrete distribution.
Each element corresponds to the probability of the integers from 0 to the last value of the support.
See also \link{PMF.get_mean}, \link{PMF.get_var}, \link{PMF.sample}, \link{PMF.get_quantile},
\link{PMF.summary} for functions that handle PMF objects.

Warnings are triggered from the Importance Sampling step if:
\itemize{
\item weights are all zeros, then the upper forecast is ignored during reconciliation;
\item the effective sample size is < 200;
\item the effective sample size is < 1\% of the sample size.
}

Note that warnings are an indication that the base forecasts might have issues.
Please check the base forecasts in case of warnings.
}
\examples{

library(bayesRecon)

# Consider a simple hierarchy with two bottom and one upper
A <- matrix(c(1,1),nrow=1)
S <- rbind(A,diag(nrow=2))
# The bottom forecasts are Poisson with lambda=15
lambda <- 15
n_tot <- 60
fc_bottom <- list()
fc_bottom[[1]] <- apply(matrix(seq(0,n_tot)),MARGIN=1,FUN=function(x) dpois(x,lambda=lambda))
fc_bottom[[2]] <- apply(matrix(seq(0,n_tot)),MARGIN=1,FUN=function(x) dpois(x,lambda=lambda))

# The upper forecast is a Normal with mean 40 and std 5
fc_upper<- list(mu=40, Sigma=matrix(5^2))

# We can reconcile with reconc_MixCond
res.mixCond <- reconc_MixCond(S, fc_bottom, fc_upper)

# Note that the bottom distributions are slightly shifted to the right
PMF.summary(res.mixCond$bottom_reconciled$pmf[[1]])
PMF.summary(fc_bottom[[1]])

PMF.summary(res.mixCond$bottom_reconciled$pmf[[2]])
PMF.summary(fc_bottom[[2]])

# The upper distribution is slightly shifted to the left
PMF.summary(res.mixCond$upper_reconciled$pmf[[1]])
PMF.get_var(res.mixCond$upper_reconciled$pmf[[1]])

}
\references{
Corani, G., Azzimonti, D., Augusto, J.P.S.C., Zaffalon, M. (2021).
\emph{Probabilistic Reconciliation of Hierarchical Forecast via Bayes' Rule}.
ECML PKDD 2020. Lecture Notes in Computer Science, vol 12459.
\doi{10.1007/978-3-030-67664-3_13}.

Zambon, L., Agosto, A., Giudici, P., Corani, G. (2024).
\emph{Properties of the reconciled distributions for Gaussian and count forecasts}.
International Journal of Forecasting (in press).
\doi{10.1016/j.ijforecast.2023.12.004}.

Zambon, L., Azzimonti, D., Rubattu, N., Corani, G. (2024).
\emph{Probabilistic reconciliation of mixed-type hierarchical time series}.
The 40th Conference on Uncertainty in Artificial Intelligence, accepted.
}
\seealso{
\code{\link[=reconc_TDcond]{reconc_TDcond()}}, \code{\link[=reconc_BUIS]{reconc_BUIS()}}
}
