% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/limma_tidiers.R
\name{limma_tidiers}
\alias{limma_tidiers}
\alias{tidy.MArrayLM}
\alias{augment.MArrayLM}
\alias{glance.MArrayLM}
\alias{tidy.MAList}
\alias{tidy.EList}
\title{Tidiers for the output of limma (linear models for microarray analysis)}
\usage{
\method{tidy}{MArrayLM}(x, intercept = FALSE, ...)

\method{augment}{MArrayLM}(x, data, ...)

\method{glance}{MArrayLM}(x, ...)

\method{tidy}{MAList}(x, ...)

\method{tidy}{EList}(x, addTargets = FALSE, ...)
}
\arguments{
\item{x}{\code{MArrayLM}, \code{MAList}, \code{Elist} object}

\item{intercept}{whether the \code{(Intercept)} term should be included
(default FALSE)}

\item{...}{extra arguments, not used}

\item{data}{original expression matrix; if missing, \code{augment} returns only
the computed per-gene statistics}

\item{addTargets}{Add sample level information. Default is FALSE.}
}
\value{
The output of tidying functions is always a data frame without
rownames.

\code{tidy} returns one row per gene per coefficient. It always
contains the columns
  \item{gene}{The name of the gene (extracted from the rownames of the
  input matrix)}
  \item{term}{The coefficient being estimated}
  \item{estimate}{The estimate of each per-gene coefficient}

Depending on whether the object comes from \code{eBayes}, it may also
contain
  \item{statistic}{Empirical Bayes t-statistic}
  \item{p.value}{p-value computed from t-statistic}
  \item{lod}{log-of-odds score}

\code{augment} returns one row per gene, containing the original
gene expression matrix if provided. It then adds columns containing
the per-gene statistics included in the MArrayLM object, each prepended
with a .:
  \item{.gene}{gene ID, obtained from the rownames of the input}
  \item{.sigma}{per-gene residual standard deviation}
  \item{.df.residual}{per-gene residual degrees of freedom}

The following columns may also be included, depending on which have been
added by \code{lmFit} and \code{eBayes}:
  \item{.AMean}{average intensity across probes}
  \item{.statistic}{moderated F-statistic}
  \item{.p.value}{p-value generated from moderated F-statistic}
  \item{.df.total}{total degrees of freedom per gene}
  \item{.df.residual}{residual degrees of freedom per gene}
  \item{.s2.prior}{prior estimate of residual variance}
  \item{.s2.post}{posterior estimate of residual variance}

\code{glance} returns one row, containing
  \item{rank}{rank of design matrix}
  \item{df.prior}{empirical Bayesian prior degrees of freedom}
  \item{s2.prior}{empirical Bayesian prior residual standard deviation}

\code{tidy} returns a data frame with one row per gene-sample
combination, with columns
  \item{gene}{gene name}
  \item{sample}{sample name (from column names)}
  \item{value}{expressions on log2 scale}

\code{tidy} returns a data frame with one row per gene-sample
combination, with columns
  \item{gene}{gene name}
  \item{sample}{sample name (from column names)}
  \item{value}{expressions on log2 scale}
  \item{weight}{present if \code{weights} is set}
  \item{other columns}{if present and if \code{addTargets} is set}
}
\description{
Tidy, augment, and glance methods for MArrayLM objects, which contain the
results of gene-wise linear models to microarray datasets. This class is
the output of the lmFit and eBayes functions.
}
\details{
Tidying this fit computes one row per coefficient per gene, while
augmenting returns one row per gene, with per-gene statistics included.
(This is thus a rare case where the \code{augment} output has more rows
than the \code{tidy} output. This is a side effect of the fact that the
input to limma is not tidy but rather a one-row-per-gene matrix).
}
\examples{
if (require("limma")) {
    # create random data and design
    set.seed(2014)
    dat <- matrix(rnorm(1000), ncol=4)
    dat[, 1:2] <- dat[, 1:2] + .5  # add an effect
    rownames(dat) <- paste0("g", 1:nrow(dat))
    des <- data.frame(treatment = c("a", "a", "b", "b"),
                      confounding = rnorm(4))

    lfit <- lmFit(dat, model.matrix(~ treatment + confounding, des))
    eb <- eBayes(lfit)
    head(tidy(lfit))
    head(tidy(eb))

    if (require("ggplot2")) {
        # the tidied form puts it in an ideal form for plotting
        ggplot(tidy(lfit), aes(estimate)) + geom_histogram(binwidth=1) +
            facet_wrap(~ term)
        ggplot(tidy(eb), aes(p.value)) + geom_histogram(binwidth=.2) +
            facet_wrap(~ term)
    }
}
}
