% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getBestTest.R
\name{getBestTest}
\alias{getBestTest}
\title{Get the best test in a cluster}
\usage{
getBestTest(
  ids,
  tab,
  by.pval = TRUE,
  weights = NULL,
  pval.col = NULL,
  fc.col = NULL,
  fc.threshold = 0.05,
  cpm.col = NULL
)
}
\arguments{
\item{ids}{An integer vector or factor containing the cluster ID for each test.}

\item{tab}{A data.frame of results with \code{PValue} and at least one \code{logFC} field for each test.}

\item{by.pval}{Logical scalar indicating whether the best test should be selected on the basis of the smallest p-value.
If \code{FALSE}, the best test is defined as that with the highest abundance.}

\item{weights}{A numeric vector of weights for each test. 
Defaults to 1 for all tests.}

\item{pval.col}{An integer scalar or string specifying the column of \code{tab} containing the p-values.
Defaults to \code{"PValue"}.}

\item{fc.col}{An integer or character vector specifying the columns of \code{tab} containing the log-fold changes.
Defaults to all columns in \code{tab} starting with \code{"logFC"}.}

\item{fc.threshold}{A numeric scalar specifying the FDR threshold to use \emph{within} each cluster for counting tests changing in each direction, see \code{?"\link{cluster-direction}"} for more details.}

\item{cpm.col}{An integer scalar or string specifying the column of \code{tab} containing the log-CPM values.
Defaults to \code{"logCPM"}.}
}
\value{
A \linkS4class{DataFrame} with one row per cluster and various fields:
\itemize{
\item An integer field \code{num.tests}, specifying the total number of tests in each cluster.
\item Two integer fields \code{num.up.*} and \code{num.down.*} for each log-FC column in \code{tab}, containing the number of tests with log-FCs significantly greater or less than 0, respectively.
See \code{?"\link{cluster-direction}"} for more details.
\item A numeric field containing the cluster-level p-value. 
If \code{pval.col=NULL}, this column is named \code{PValue}, otherwise its name is set to \code{colnames(tab[,pval.col])}.
\item A numeric field \code{FDR}, containing the BH-adjusted cluster-level p-value.
\item A character field \code{direction} (if \code{fc.col} is of length 1), specifying the dominant direction of change for tests in each cluster.
See \code{?"\link{cluster-direction}"} for more details.
\item One integer field \code{rep.test} containing the row index (for \code{tab}) of a representative test for each cluster.
See \code{?"\link{cluster-direction}"} for more details.
\item One numeric field \code{rep.*} for each log-FC column in \code{tab}, containing a representative log-fold change for the differential tests in the cluster.
See \code{?"\link{cluster-direction}"} for more details.
}
Each row is named according to the ID of the corresponding cluster.
}
\description{
Find the test with the greatest significance or the highest abundance in each cluster.
}
\details{
Each cluster is defined as a set of tests with the same value of \code{ids} (any \code{NA} values are ignored).
If \code{by.pval=TRUE}, this function identifies the test with the lowest p-value as that with the strongest evidence against the null in each cluster.
The p-value of the chosen test is adjusted using the (Holm-)Bonferroni correction, based on the total number of tests in the parent cluster. 
This is necessary to obtain strong control of the family-wise error rate such that the best test can be taken from each cluster for further consideration.

% i.e. The configuration, in this case, is taking the best test.

The importance of each window in each cluster can be adjusted by supplying different relative \code{weights} values. 
Each weight is interpreted as a different threshold for each test in the cluster using the weighted Holm procedure. 
Larger weights correspond to lower thresholds, i.e., less evidence is needed to reject the null for tests deemed to be more important. 
This may be useful for upweighting particular tests such as those for windows containing a motif for the TF of interest.

Note the difference between this function and \code{\link{combineTests}}. 
The latter presents evidence for any rejections within a cluster. 
This function specifies the exact location of the rejection in the cluster, which may be more useful in some cases but at the cost of conservativeness. 
In both cases, clustering procedures such as \code{\link{mergeWindows}} can be used to identify the cluster.

% The vagueness of combineTests may be good enough in most applications (i.e. wanting to get a location
% to look at the genomic context, or in instances where differential binding is obvious). If error control
% at specific locations is needed, then getBestTests is probably more appropriate..

If \code{by.pval=FALSE}, the best test is defined as that with the highest log-CPM value. 
This should be independent of the p-value so no adjustment is necessary. Weights are not applied here. 
This mode may be useful when abundance is correlated to rejection under the alternative hypothesis, e.g., picking high-abundance regions that are more likely to contain peaks.

To obtain \code{ids}, a simple clustering approach for genomic windows is implemented in \code{\link{mergeWindows}}.
However, anything can be used so long as it is independent of the p-values and does not compromise type I error control, e.g., promoters, gene bodies, independently called peaks. 
Any tests with \code{NA} values for \code{ids} will be ignored.
}
\examples{
ids <- round(runif(100, 1, 10))
tab <- data.frame(logFC=rnorm(100), logCPM=rnorm(100), PValue=rbeta(100, 1, 2))
best <- getBestTest(ids, tab)
head(best)

best <- getBestTest(ids, tab, cpm.col="logCPM", pval.col="PValue")
head(best)

# With window weighting.
w <- round(runif(100, 1, 5))
best <- getBestTest(ids, tab, weight=w)
head(best)

# By logCPM.
best <- getBestTest(ids, tab, by.pval=FALSE)
head(best)

best <- getBestTest(ids, tab, by.pval=FALSE, cpm.col=2, pval.col=3)
head(best)

}
\seealso{
\code{\link{combineTests}} and \code{\link{minimalTests}}, for other methods for obtaining cluster-level p-values.

\code{\link{mergeWindows}}, to generate \code{ids}.

\code{\link{glmQLFTest}}, for one method of generating \code{tab}.
}
\author{
Aaron Lun
}
\keyword{testing}
