% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normOffsets.R
\name{normOffsets}
\alias{normOffsets}
\title{Normalize trended biases across libraries}
\usage{
normOffsets(object, ..., assay.id = "counts", se.out = TRUE)
}
\arguments{
\item{object}{A \linkS4class{SummarizedExperiment} object containing a count matrix and library sizes in the \code{totals} field of the \code{\link{colData}}.

Alternatively, a \link{DGEList} object containing a count matrix in \code{object$counts} and library sizes in \code{object$samples$lib.size}.

Alternatively, an ordinary matrix containing counts.}

\item{...}{Other arguments to be passed to \code{\link{loessFit}}.}

\item{assay.id}{An integer scalar or string specifying the assay values to use for normalization.}

\item{se.out}{A logical scalar indicating whether or not a SummarizedExperiment object should be returned.

Alternatively, a SummarizedExperiment or DGEList object in which normalization factors are to be stored.}
}
\value{
If \code{se.out=FALSE}, a numeric matrix of dimensions equal to \code{object}, containing the offset for each observation.
These offsets have already been scaled to be comparable in magnitude to the log-library sizes.

If \code{se.out=TRUE}, the same matrix is stored in the \code{offset} assay of \code{object} (if it is a SummarizedExperiment)
or \code{object$offset} (if a DGEList) and the modified \code{object} is returned.

If \code{se.out} is a separate SummarizedExperiment or DGEList object, the offset matrix instead has dimensions equal to \code{se.out}.
This matrix is stored inside \code{se.out} and the modified object is returned.
}
\description{
Calculate normalization offsets by performing a loess fit to count data from multiple libraries.
}
\details{
This function performs non-linear normalization similar to the fast loess algorithm in \code{\link{normalizeCyclicLoess}}. 
The aims is to account for mean dependencies in the efficiency biases between libraries.
For each sample, a lowess curve is fitted to the log-counts against the log-average count. 
The fitted value for each genomic window is used as an offset in a generalized linear model for that feature and sample. 
The use of the average count provides more stability than the average log-count when low counts are present for differentially bound regions.

The trend fits are always computed from \code{object}.
However, if \code{se.out} is a (different) SummarizedExperiment or DGEList object, 
the trend fits will be used to compute offsets for each entry in \code{se.out} using spline interpolation.
This is useful when \code{se.out} contains counts for windows in an endogenous genome, but the trend fits are computed using spike-in chromatin regions.

An error is raised if the library sizes in \code{se.out$totals} are not identical to \code{object$totals}.
This is because the average abundances are only comparable when the library sizes are the same.
Consistent library sizes can be achieved by using the same \code{\link{readParam}} object in \code{\link{windowCounts}} and related functions.
}
\examples{
counts <- matrix(rnbinom(400, mu=10, size=20), ncol=4)
data <- SummarizedExperiment(list(counts=counts))
data$totals <- colSums(counts)

# TMM normalization.
normFactors(data)

# Using loess-based normalization, instead.
offsets <- normOffsets(data)
head(offsets)
offsets <- normOffsets(data, span=0.4)
offsets <- normOffsets(data, iterations=1)

}
\references{
Ballman KV, Grill DE, Oberg AL, Therneau TM (2004). 
Faster cyclic loess: normalizing RNA arrays via linear models. 
\emph{Bioinformatics} 20, 2778-86.
}
\seealso{
\code{\link{loessFit}}, for the fitting algorithm.

\code{\link{normalizeCyclicLoess}}, for the original inspiration for this method.
}
\author{
Aaron Lun
}
\keyword{normalization}
