#' Group cells according to subject IDs
#'
#' @param count A raw count matrix of the single-cell data. The rows are the genes, and the columns are the cells. The matrix can be a matrix object or a sparse dgCMatrix object.
#' @param id A vector of subject IDs. The length should be the same as the number of columns of the count matrix.
#' @param pred A design matrix of the predictors. The rows are the cells and the columns are the predictors. If not specified, an intercept column will be generated by default.
#' @param offset A vector of the scaling factor. The values must be strictly positive. If not specified, a vector of all ones will be generated by default. 
#' @return count: A reordered count matrix. If the cells are already grouped, the function returns NULL.
#' @return id: A reordered subject ID vector.
#' @return pred: A reordered design matrix of the predictors.
#' @return offset: A reordered vector of the scaling factor.
#' @export
#' @examples
#' library(nebula)
#' data(sample_data)
#' pred = model.matrix(~X1+X2+cc,data=sample_data$pred)
#' df_order = group_cell(count=sample_data$count,id=sample_data$sid,pred=pred)
#' 


group_cell = function(count, id, pred = NULL, offset = NULL)
{
    ng = nrow(count)
    nc = ncol(count)
    if(nc!=length(id))
    {stop("The length of id is not equal to the number of columns of the count matrix.")}
    
    id = as.character(id)
    levels = unique(id)
    id = as.numeric(factor(id,levels=levels))
    if(is.unsorted(id)==FALSE)
    {
      cat("The cells are already grouped.")
      return(NULL)
    }
    k = length(levels)
    o = order(id)
    count = count[,o]
    id = id[o]
    if(is.null(pred)==FALSE)
    {
      if(nc!=nrow(as.matrix(pred)))
      {stop("The number of rows of the design matrix is not equal to the number of columns of the count matrix")}
      pred = as.matrix(pred)[o,,drop = FALSE]
    }
    if(is.null(offset)==FALSE)
    {
      if(nc!=length(offset))
      {stop("The length of offset is not equal to the number of columns of the count matrix")}
      if(sum(offset<=0)>0)
      {stop("Some elements in the scaling factor are not positive.")}
      offset = offset[o]
    }
    grouped = list(count=count,id=id,pred=pred,offset=offset)
    return(grouped)
}