% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.hmm.R
\name{plot.hmm}
\alias{plot.hmm}
\title{Plot hidden Markov models}
\usage{
\method{plot}{hmm}(
  x,
  layout = "horizontal",
  pie = TRUE,
  vertex.size = 40,
  vertex.label = "initial.probs",
  vertex.label.dist = "auto",
  vertex.label.pos = "bottom",
  vertex.label.family = "sans",
  loops = FALSE,
  edge.curved = TRUE,
  edge.label = "auto",
  edge.width = "auto",
  cex.edge.width = 1,
  edge.arrow.size = 1.5,
  edge.label.family = "sans",
  label.signif = 2,
  label.scientific = FALSE,
  label.max.length = 6,
  trim = 1e-15,
  combine.slices = 0.05,
  combined.slice.color = "white",
  combined.slice.label = "others",
  with.legend = "bottom",
  ltext = NULL,
  legend.prop = 0.5,
  cex.legend = 1,
  ncol.legend = "auto",
  cpal = "auto",
  cpal.legend = "auto",
  legend.order = TRUE,
  main = NULL,
  withlegend,
  ...
)
}
\arguments{
\item{x}{A hidden Markov model object of class \code{hmm} created with
\code{\link{build_hmm}} (or \code{\link{build_mm}}). Multichannel
\code{hmm} objects are automatically transformed into single-channel objects.
See function \code{\link{mc_to_sc}} for more information on the
transformation.}

\item{layout}{specifies the layout of vertices (nodes). Accepts a
numerical matrix, a \code{\link[igraph]{layout_}} function (without quotation marks),
or either of the predefined options \code{"horizontal"} (the
default) and \code{"vertical"}. Options \code{"horizontal"} and
\code{"vertical"} position vertices at the same horizontal or vertical
line. A two-column numerical matrix can be used to give x and y coordinates of
the vertices. The \code{\link[igraph]{layout_}} functions available in the
\code{igraph} package offer other automatic layouts for graphs.}

\item{pie}{Are vertices plotted as pie charts of emission probabilities?
Defaults to TRUE.}

\item{vertex.size}{Size of vertices, given as a scalar or numerical
vector. The default value is 40.}

\item{vertex.label}{Labels for vertices. Possible options include
\code{"initial.probs"}, \code{"names"}, \code{NA}, and a character or
numerical vector. The default \code{"initial.probs"} prints the initial
probabilities of the model and \code{"names"} prints the names of the
hidden states as labels. \code{NA} prints no labels.}

\item{vertex.label.dist}{Distance of the label of the vertex from its
center. The default value \code{"auto"} places the label outside the
vertex.}

\item{vertex.label.pos}{Positions of vertex labels, relative to
the center of the vertex. A scalar or numerical vector giving
position(s) as radians or one of \code{"bottom"} (\code{pi/2} as radians),
\code{"top"} (\code{-pi/2}), \code{"left"} (\code{pi}), or
\code{"right"} (\code{0}).}

\item{vertex.label.family, edge.label.family}{Font family to be used for
vertex/edge labels. See argument \code{family} in \code{\link{par}} for
more information.}

\item{loops}{Defines whether transitions back to same states are plotted.}

\item{edge.curved}{Defines whether to plot curved edges (arcs, arrows)
between vertices. A logical or numerical vector or scalar. Numerical
values specify curvatures of edges. The default value \code{TRUE}
gives curvature of 0.5 to all edges. See \code{\link{igraph.plotting}} for
more information.}

\item{edge.label}{Labels for edges. Possible options include
\code{"auto"}, \code{NA}, and a character or numerical vector. The
default \code{"auto"} prints transition probabilities as edge labels.
\code{NA} prints no labels.}

\item{edge.width}{Width(s) for edges. The default \code{"auto"} determines
widths according to transition probabilities between hidden states.
Other possibilities are a scalar or a numerical vector of widths.}

\item{cex.edge.width}{An expansion factor for edge widths. Defaults to 1.}

\item{edge.arrow.size}{Size of the arrow in edges (constant). Defaults to 1.5.}

\item{label.signif}{Rounds labels of model parameters to specified number
of significant digits, 2 by default. Ignored for user-given labels.}

\item{label.scientific}{Defines if scientific notation should be used to
describe small numbers. Defaults to \code{FALSE}, e.g. 0.0001 instead of
1e-04. Ignored for user-given labels.}

\item{label.max.length}{Maximum number of digits in labels of model
parameters. Ignored for user-given labels.}

\item{trim}{Scalar between 0 and 1 giving the highest probability of
transitions that are plotted as edges, defaults to 1e-15.}

\item{combine.slices}{Scalar between 0 and 1 giving the highest probability
of emission probabilities that are combined into one state. The dafault
value is 0.05.}

\item{combined.slice.color}{Color of the combined slice that includes
the smallest emission probabilities (only if argument
\code{"combine.slices"} is greater than 0). The default color is white.}

\item{combined.slice.label}{The label for combined states (when argument
\code{"combine.slices"} is greater than 0) to appear in the legend.}

\item{with.legend}{Defines if and where the legend of state colors is
plotted. Possible values include \code{"bottom"} (the default),
\code{"top"}, \code{"left"}, and \code{"right"}. \code{FALSE} omits the
legend.}

\item{ltext}{Optional description of (combined) observed states to appear
in the legend. A vector of character strings.  See \code{\link{seqplot}} for
more information.}

\item{legend.prop}{Proportion used for plotting the legend. A scalar between
0 and 1, defaults to 0.5.}

\item{cex.legend}{Expansion factor for setting the size of the font for
labels in the legend. The default value is 1. Values lesser than 1 will
reduce the size of the font, values greater than 1 will increase the size.}

\item{ncol.legend}{The number of columns for the legend. The default value
\code{"auto"} sets the number of columns automatically.}

\item{cpal}{Optional color palette for (combinations of) observed states.
The default value \code{"auto"} uses automatic color palette. Otherwise a
vector of length \code{x$n_symbols} is given, i.e. the argument requires a color
specified for all (combinations of) observed states even if they are not
plotted (if the probability is less than \code{combine.slices}).}

\item{cpal.legend}{Optional color palette for the legend, only considered when
legend.order is FALSE. Should match ltext.}

\item{legend.order}{Whether to use the default order in the legend, i.e., order by appearance
(first by hidden state, then by emission probability). TRUE by default.}

\item{main}{Main title for the plot. Omitted by default.}

\item{withlegend}{Deprecated. Use \code{with.legend} instead.}

\item{...}{Other parameters passed on to \code{\link{plot.igraph}} such as
\code{vertex.color}, \code{vertex.label.cex}, or \code{edge.lty}.}
}
\description{
Function \code{plot.hmm} plots a directed graph with pie charts of
emission probabilities as vertices/nodes.
}
\examples{
# Multichannel data, left-to-right model

# Loading a HMM of the biofam data
data("hmm_biofam")

# Plotting hmm object
plot(hmm_biofam)

# Plotting HMM with
plot(hmm_biofam,
  # varying curvature of edges
  edge.curved = c(0, -0.7, 0.6, 0.7, 0, -0.7, 0),
  # legend with two columns and less space
  ncol.legend = 2, legend.prop = 0.4,
  # new label for combined slice
  combined.slice.label = "States with probability < 0.05"
)

# Plotting HMM with given coordinates
plot(hmm_biofam,
  # layout given in 2x5 matrix
  # x coordinates in the first column
  # y coordinates in the second column
  layout = matrix(c(
    1, 3, 3, 5, 3,
    0, 0, 1, 0, -1
  ), ncol = 2),
  # larger vertices
  vertex.size = 50,
  # straight edges
  edge.curved = FALSE,
  # thinner edges and arrows
  cex.edge.width = 0.5, edge.arrow.size = 1,
  # varying positions for vertex labels (initial probabilities)
  vertex.label.pos = c(pi, pi / 2, -pi / 2, 0, pi / 2),
  # different legend properties
  with.legend = "top", legend.prop = 0.3, cex.legend = 1.1,
  # Fix axes to the right scale
  xlim = c(0.5, 5.5), ylim = c(-1.5, 1.5), rescale = FALSE,
  # all states (not combining states with small probabilities)
  combine.slices = 0,
  # legend with two columns
  ncol.legend = 2
)

# Plotting HMM with own color palette
plot(hmm_biofam,
  cpal = 1:10,
  # States with emission probability less than 0.2 removed
  combine.slices = 0.2,
  # legend with two columns
  ncol.legend = 2
)

# Plotting HMM without pie graph and with a layout function
require("igraph")
# Setting the seed for a random layout
set.seed(1234)
plot(hmm_biofam,
  # Without pie graph
  pie = FALSE,
  # Using an automatic layout function from igraph
  layout = layout_nicely,
  vertex.size = 30,
  # Straight edges and probabilities of moving to the same state
  edge.curved = FALSE, loops = TRUE,
  # Labels with three significant digits
  label.signif = 3,
  # Fixed edge width
  edge.width = 1,
  # Remove edges with probability less than 0.01
  trim = 0.01,
  # Hidden state names as vertex labels
  vertex.label = "names",
  # Labels insidde vertices
  vertex.label.dist = 0,
  # Fix x-axis (more space on the right-hand side)
  xlim = c(-1, 1.3)
)


# Single-channel data, unrestricted model

# Loading a hidden Markov model of the mvad data (hmm object)
data("hmm_mvad")

# Plotting the HMM
plot(hmm_mvad)

# Checking the order of observed states (needed for the next call)
require(TraMineR)
alphabet(hmm_mvad$observations)

# Plotting the HMM with own legend (note: observation "none" nonexistent in the observations)
plot(hmm_mvad,
  # Override the default order in the legend
  legend.order = FALSE,
  # Colours in the pies (ordered by the alphabet of observations)
  cpal = c("purple", "pink", "brown", "lightblue", "orange", "green"),
  # Colours in the legend (matching to ltext)
  cpal.legend = c("orange", "pink", "brown", "green", "lightblue", "purple", "gray"),
  # Labels in the legend (matching to cpal.legend)
  ltext = c("school", "further educ", "higher educ", "training", "jobless", "employed", "none")
)

require("igraph")
plot(hmm_mvad,
  # Layout in circle (layout function from igraph)
  layout = layout_in_circle,
  # Less curved edges with smaller arrows, no labels
  edge.curved = 0.2, edge.arrow.size = 0.9, edge.label = NA,
  # Positioning vertex labels (initial probabilities)
  vertex.label.pos = c("right", "right", "left", "left", "right"),
  # Less space for the legend
  legend.prop = 0.3
)
}
\seealso{
\code{\link{build_hmm}} and \code{\link{fit_model}} for building and
  fitting Hidden Markov models, \code{\link{mc_to_sc}} for transforming
  multistate \code{hmm} objects into single-channel objects,
  \code{\link{hmm_biofam}} and \code{\link{hmm_mvad}} for information on the models
  used in the examples, and
  \code{\link{plot.igraph}} for the general plotting function of directed graphs.
}
