(defpackage :collectors-test
  (:use :cl :cl-user :collectors :lisp-unit2))
(in-package :collectors-test)

(define-test make-reducer-test (:tags '(reducer))
  (let ((r (make-reducer #'+ :initial-value 0)))
    (assert-eql 0 (funcall r 0))
    (assert-eql 3 (funcall r 1 2))
    (assert-eql 9 (funcall r 1 2 3))
    (assert-eql 9 (funcall r))))

(define-test with-reducer-test (:tags '(reducer))
  (with-reducer (r #'+ :initial-value 0)
    (assert-eql 0 (r 0))
    (assert-eql 3 (r 1 2))
    (assert-eql 9 (r 1 2 3))
    (assert-eql 9 (r))))

(define-test simple-collector-test (:tags '(simple collector))
  (let* ((c (make-simple-collector)))
    (assert-equal '(1 2 3 4 (5)) (funcall c 1 2 3 4 '(5)))
    (assert-equal '(1 2 3 4 (5)) (funcall c))
    (assert-equal '(1 2 3 4 (5) 6) (funcall c 6))))

(define-test simple-collector-to-place-test (:tags '(simple collector place))
  (let* ((p)
         (c (make-simple-collector-to-place p)))
    (assert-equal '(1 2 3 4 (5)) (funcall c 1 2 3 4 '(5)))
    (assert-equal '(1 2 3 4 (5)) (funcall c))
    (assert-equal '(1 2 3 4 (5)) p)
    (assert-equal '(1 2 3 4 (5) 6) (funcall c 6))
    (assert-equal '(1 2 3 4 (5) 6) p)))

(define-test simple-appender-test (:tags '(simple appender))
  (let* ((c (make-simple-appender)))
    (assert-equal '(1 2 (3) 4 5) (funcall c 1 '(2 (3)) 4 '(5)))
    (assert-equal '(1 2 (3) 4 5) (funcall c))
    (assert-equal '(1 2 (3) 4 5 6 7) (funcall c '(6 7)))))

(define-test simple-appender-to-place-test (:tags '(simple appender place))
  (let* ((p)
         (c (make-simple-appender-to-place p)))
    (assert-equal '(1 2 (3) 4 5) (funcall c 1 '(2 (3)) 4 '(5)))
    (assert-equal '(1 2 (3) 4 5) (funcall c))
    (assert-equal '(1 2 (3) 4 5) p)
    (assert-equal '(1 2 (3) 4 5 6 7) (funcall c '(6 7)))
    (assert-equal '(1 2 (3) 4 5 6 7) p)))

(define-test with-collector-test (:tags '(collector))
  (with-collector (test)
    (test :a nil :key)
    (test :and :a)
    (test :value :make)
    (test :a :plist)
    (assert-equal
        '(:a nil :key :and :a :value :make :a :plist)
        (test)
     ))
  (with-collector (test :collect-nil nil)
    (test :a nil :key)
    (test :and :a)
    (test :value :make)
    (test :a :plist)
    (assert-equal
        '(:a :key :and :a :value :make :a :plist)
        (test)
     )))

(define-test with-collector2 (:tags '(collector))
  (with-collector (test :from-end t)
    (test :a nil :key)
    (test :and :a)
    (test :value :make)
    (test :a :plist)
    (assert-equal
        '(:plist :a :make :value :a :and :key nil :a)
        (test)
     ))
  (with-collector (test :from-end t :collect-nil nil)
    (test :a nil :key)
    (test :and :a)
    (test :value :make)
    (test :a :plist)
    (assert-equal
        '(:plist :a :make :value :a :and :key :a)
        (test)
     )))

(define-test with-appender-test (:tags '(appender))
  (with-appender (test)
    (test :a :key)
    (test '(:and :a))
    (test '(:value) '(:make))
    (test '(:a :plist))
    (assert-equal
        '(:a :key :and :a :value :make :a :plist)
        (test)
     )))

(define-test with-string-builder-test (:tags '(strings))
  (with-string-builder (test)
    (test :a :key)
    (test :and :a)
    (test :value :make)
    (test "" nil)
    (test :a :plist)
    (assert-equal
        "AKEYANDAVALUEMAKEAPLIST"
        (test)
     ))
  (with-string-builder (test :delimiter ", ")
    (test nil)
    (test :put)
    (test nil)
    (test :some)
    (test :commas)
    (test :in)
    (assert-equal
        "PUT, SOME, COMMAS, IN"
        (test)
     )))

(define-test with-formatter-test (:tags '(strings))
  (with-formatter (test)
    (test "~D ~D ~D" 0 0 0)
    (test "~A" 1)
    (test "~A" 2)
    (test "~A" 3)
    (test "ABC")
    (assert-equal "0 0 0123ABC" (test)))
  (with-formatter (test :delimiter "-")
    (test "~D ~D ~D" 0 0 0)
    (test "~A" 1)
    (test "~A" 2)
    (test "~A" 3)
    (test "ABC")
    (assert-equal "0 0 0-1-2-3-ABC" (test))))

(define-test with-mapping-collector-test (:tags '(mapping-collector collector))
  (with-mapping-collector (test (x) (* 2 x))
    (test 1)
    (test 1 2)
    (test 1 2 3)
    (test 1 2 3 4)
    (assert-equal '(2 2 4 2 4 6 2 4 6 8) (test))))

(define-test with-mapping-appender-test (:tags '(mapping-appender appender))
  (with-mapping-appender (test (x) (* 2 x))
    (test 1)
    (assert-equal '(2) (test))
    (test 1 2)
    (assert-equal '(2 2 4) (test))
    (test 1 2 3)
    (test 1 2 3 4)
    (assert-equal '(2 2 4 2 4 6 2 4 6 8) (test))))

(define-test reducer-place (:tags '(place reducer))
  (let ((reduced 1))
    (with-reducer (r #'+ :place reduced)
      (r 0)
      (assert-equal 1 reduced)
      (assert-equal 2 (r 1))
      (assert-equal 2 reduced)
      (assert-equal 7 (r 2 3))
      (assert-equal 7 reduced))))

(define-test appender-place (:tags '(place appender))
  (let ((appended '(:A)))
    (with-appender (a :place appended)
      (a 0)
      (assert-equal '(:A 0) appended)
      (assert-equal '(:A 0 1) (a 1))
      (assert-equal '(:A 0 1) appended)
      (assert-equal '(:A 0 1 2 3 4) (a '(2 3) '(4)))
      (assert-equal '(:A 0 1 2 3 4) appended))))

(define-test collector-place (:tags '(place collector))
  (let ((collected '(:A)))
    (with-collector (c :place collected)
      (c 0)
      (assert-equal '(:a 0) collected)
      (assert-equal '(:a 0 1) (c 1))
      (assert-equal '(:a 0 1) collected)
      (assert-equal '(:a 0 1 (2 3) (4)) (c '(2 3) '(4)))
      (assert-equal '(:a 0 1 (2 3) (4)) collected))))

(define-test deoperate-lists (:tags '(collector deoperate))
  (let ((c (make-collector)))
    (assert-equal '(1 2 3 4 5) (funcall c 1 2 3 4 5))
    (assert-equal '(1 2 4 5) (deoperate c 3))
    (assert-equal '(1 5) (deoperate c '(2 4 6 8)))))

(define-test nested-collectors (:tags '(collector))
  (with-collector (x)
    (with-mapping-collector (z (it) (if it
                                        (* 2 it)
                                        (invoke-restart 'collectors-signals:skip)))
      (with-collector (y :collect-nil nil)
        (assert-equal '(1 nil 2) (x 1 nil 2))
        (assert-equal '(1 2) (y 1 nil 2))
        (assert-equal '(2 4) (z 1 nil 2))
        (assert-equal '(1 2 1 2) (y 1 nil 2))
        (assert-equal '(2 4 2 4) (z 1 nil 2))
        (assert-equal '(1 nil 2 1 nil 2) (x 1 nil 2))
        ))))

(define-test simple-collectors-init-test (:tags '(collector place initial-value))
  (let* ((as) (bs 2) (cs (list 1 2))
         (a (collectors:make-simple-collector-to-place as))
         (b (collectors:make-simple-collector-to-place bs))
         (c (collectors:make-simple-collector-to-place cs)))
    (funcall a :a :a)
    (funcall b :b :b)
    (funcall c :c :c)
    (assert-equal '(:a :a) as)
    (assert-equal '(:a :a) (funcall a))

    (assert-equal '(2 :b :b) bs)
    (assert-equal '(2 :b :b) (funcall b))

    (assert-equal '(1 2 :c :c) cs)
    (assert-equal '(1 2 :c :c) (funcall c))
    ))


(define-test collector-macros-push (:tags '(collector push macros))
  (with-collector (it)
    (push-it 1)
    (push-it 2)
    (assert-equal '(5 4 3 2 1 ) (push-it 3 4 5))
    (assert-equal (it) '(5 4 3 2 1))
    ))

(define-test collector-macros-pop (:tags '(collector pop macros))
  (with-collector (it)
    (it 1 2 3 4 5 6)
    (assert-equal 1 (pop-it))
    (assert-equal '(2 3 4) (pop-it 3))
    (assert-equal '(5 6) (it))
    ))

(define-test collector-macros-enqueu (:tags '(collector enqueue macros))
  (with-collector (it)
    (enqueue-it 1)
    (enqueue-it 2)
    (assert-equal '(1 2 3 4 5) (enqueue-it 3 4 5))
    (assert-equal (it) '(1 2 3 4 5))
    ))

(define-test collector-macros-unenqueue (:tags '(collector unenqueue macros))
  (with-collector (it)
    (it 1 2 3 4 5 6)
    (assert-equal 6 (unenqueue-it))
    (assert-equal '(4 5) (unenqueue-it 2))
    (assert-equal '(1 2 3) (it))
    ))
