(in-package #:djula)

;;; The basic plan of DJULA is kind of a half-assed lexer/parser than
;;; compiles templates to functions.  I don't have the expertise to
;;; properly fix the lexing and parsing at this time, so I'm just
;;; trying to clean up and better organize the code.  Roughly, the
;;; processing stages are:

;;; 1. lex     : string      => tokens
;;; 2. parse   : tokens      => tokens/tags
;;; 3. compile : tokens/tags => functions
;;; 4. execute : functions   => string
;;; 5. filter  : string      => string

;;; The lexical analysis phase is handled via simple string search and
;;; splitting.  The parsing phase is handled by the various
;;; "processor" functions constructed using the macros below.
;;; REST-VAR is the token stream generated by the lexer.  The parser
;;; consumes tokens and produces a mix of tokens and "tags" that span
;;; multiple input tokens.  (It does not appear that nesting of tags
;;; is handled properly...)

;;; The "compiler" functions produce functions that output to a
;;; stream, looking to the dynamic variable *TEMPLATE-ARGUMENTS* for
;;; variables in the current scope.

;;; rest-var => a poorly executed stream variable

(defmacro def-token-processor (name args rest-var &body body)
  (let* ((token-processor-package (find-package "DJULA.TOKEN-PROCESSORS"))
         (function-name (intern (symbol-name name)
                                token-processor-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (arg-list)
        `(progn
           (defun ,function-name (&rest ,arg-list)
             ,@(when docstring (list docstring))
             (destructuring-bind (,rest-var ,@args)
                 ,arg-list
               ,@declarations
               ,@body))
           (export ',function-name ,token-processor-package))))))

(defun find-token-processor (name)
  "Return the token processor by the name of NAME."
  (find-symbol (symbol-name name)
               (find-package "DJULA.TOKEN-PROCESSORS")))

(defmacro def-unparsed-tag-processor (name args rest-var &body body)
  (let* ((unparsed-tag-processor-package (find-package "DJULA.UNPARSED-TAG-PROCESSORS"))
         (function-name (intern (symbol-name name)
                                unparsed-tag-processor-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (arg-list)
        `(progn
           (defun ,function-name (&rest ,arg-list)
             ,@(when docstring (list docstring))
             ,@declarations
             (destructuring-bind (,rest-var ,@args)
                 ,arg-list
               ,@body))
           (export ',function-name ,unparsed-tag-processor-package))))))

(defun find-unparsed-tag-processor (tag-name)
  "Return the unparsed tag processor by the name of TAG-NAME."
  (find-symbol (symbol-name tag-name)
               (find-package "DJULA.UNPARSED-TAG-PROCESSORS")))

(defmacro def-tag-processor (name args rest-var &body body)
  (let* ((tag-processor-package (find-package "DJULA.TAG-PROCESSORS"))
         (function-name (intern (symbol-name name)
                                tag-processor-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (arg-list)
        `(progn
           (defun ,function-name (&rest ,arg-list)
             ,@(when docstring (list docstring))
             ,@declarations
             (destructuring-bind (,rest-var ,@args)
                 ,arg-list
               ,@body))
           (export ',function-name ,tag-processor-package))))))

(defun find-tag-processor (name)
  "Return the tag processor by the name of NAME."
  (find-symbol (symbol-name name)
               (find-package "DJULA.TAG-PROCESSORS")))

(defmacro def-delimited-tag (starttag endtag delimited-name)
  `(progn
     (def-tag-processor ,starttag (&rest args) unprocessed
       (multiple-value-bind (processed-clause processed-rest present-p)
           (find-end-tag ,endtag (process-tokens unprocessed))
         (unless present-p
           (template-error "Error parsing delimited tag {% ~A %}, cannot find closing tag {% ~A %}" ,starttag ,endtag))
         (process-tokens
          (cons (list* ,delimited-name args processed-clause)
                processed-rest))))
     (def-tag-compiler ,endtag (&optional argument)
       (template-error "unmatched ending tag {% ~A~@[~A~] %}" ,endtag argument))))

(defmacro def-token-compiler (name args &body body)
  (let* ((token-compiler-package (find-package "DJULA.TOKEN-COMPILERS"))
         (function-name (intern (symbol-name name)
                                token-compiler-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (arg-list)
        `(progn
           (defun ,function-name (&rest ,arg-list)
             ,@(when docstring (list docstring))
             ,@declarations
             (destructuring-bind (,@args)
                 ,arg-list
               ,@body))
           (export ',function-name ,token-compiler-package))))))

(defun find-token-compiler (name)
  "Return the token processor by the name of NAME."
  (or (find-symbol (symbol-name name) (find-package "DJULA.TOKEN-COMPILERS"))
      (error "Token compiler not found: ~a" name)))

(defmacro def-tag-compiler (name args &body body)
  "Define a Djula tag compiler.
NAME is the name of the tag.
ARGS is the list of arguments required.
BODY is the tag compilation implementation."
  (let* ((tag-compiler-package (find-package "DJULA.TAG-COMPILERS"))
         (function-name (intern (symbol-name name)
                                tag-compiler-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (arg-list)
        `(progn
           (defun ,function-name (&rest ,arg-list)
             ,@(when docstring (list docstring))
             ,@declarations
             (destructuring-bind (,@args)
                 ,arg-list
               ,@body))
           (export ',function-name ,tag-compiler-package))))))

(defun find-tag-compiler (name)
  "Return the tag compiler by the name of NAME."
  (or (find-symbol (symbol-name name) (find-package "DJULA.TAG-COMPILERS"))
      (error "Tag compiler not found: ~a" name)))

(defmacro def-filter (name args &body body)
  "Define a Djula filter.
NAME is the name of the filter.
ARGS is the list of arguments of the filter.
BODY is the implementation of the filter."
  (let* ((filter-package (find-package "DJULA.FILTERS"))
         (function-name (intern (symbol-name name)
                                filter-package)))
    (multiple-value-bind (body declarations docstring)
        (parse-body body :documentation t)
      (with-unique-names (e msg)
        `(progn
           (defun ,function-name (,@args)
             ,@(when docstring (list docstring))
             ,@declarations
             (handler-case
                 (locally
                     ,@body)
               (template-error (,e)
                 (if (and *catch-template-errors-p*
                          (not *fancy-error-template-p*))
                     (princ-to-string ,e)
                     (error ,e)))
               (error (,e)
                 (let ((,msg (template-error-string* ,e "There was an error running filter ~A" ,name)))
                   (if (and *catch-template-errors-p*
                            (not *fancy-error-template-p*))
                       (princ-to-string ,msg)
                       (template-error ,msg))))))
           (export ',function-name ,filter-package))))))

(defun find-filter (name)
  "Return the filter by the name of NAME."
  (find-symbol (symbol-name name) (find-package "DJULA.FILTERS")))
