/*
 *  Database class for DMAP sharing
 *
 * Copyright (c) 2008 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libdmapsharing/dmap.h>

#include "dmapd-dmap-db.h"

struct DmapdDmapDbPrivate {
	gchar *db_dir;
	DmapRecordFactory *record_factory;
	gchar **acceptable_formats;
};

enum {
	PROP_0,
	PROP_DB_DIR,
	PROP_RECORD_FACTORY,
	PROP_ACCEPTABLE_FORMATS
};

static void dmapd_dmap_db_init (DmapdDmapDb *db)
{
	db->priv = DMAPD_DMAP_DB_GET_PRIVATE (db);
}

static void
_set_property (GObject *object,
               guint prop_id,
               const GValue *value,
               GParamSpec *pspec)
{
        DmapdDmapDb *db = DMAPD_DMAP_DB (object);

	switch (prop_id) {
		case PROP_DB_DIR:
			g_free (db->priv->db_dir);
			db->priv->db_dir = g_value_dup_string (value);
			break;
		case PROP_RECORD_FACTORY:
			if (db->priv->record_factory) {
				g_object_unref (db->priv->record_factory);
			}
			db->priv->record_factory = DMAP_RECORD_FACTORY (g_value_get_pointer (value));
			break;	
		case PROP_ACCEPTABLE_FORMATS:
			db->priv->acceptable_formats = g_value_get_pointer (value);
			break;	
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void
_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
        DmapdDmapDb *db = DMAPD_DMAP_DB (object);

	switch (prop_id) {
		case PROP_DB_DIR:
			g_value_set_static_string (value, db->priv->db_dir);
			break;
		case PROP_RECORD_FACTORY:
			g_value_set_pointer (value, db->priv->record_factory);
			break;	
		case PROP_ACCEPTABLE_FORMATS:
			g_value_set_pointer (value, db->priv->acceptable_formats);
			break;	
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void dmapd_dmap_db_class_init (DmapdDmapDbClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (DmapdDmapDbPrivate));

	gobject_class->set_property = _set_property;
	gobject_class->get_property = _get_property;
	/* FIXME: implement finalize */

	g_object_class_install_property (gobject_class, PROP_RECORD_FACTORY,
					 g_param_spec_pointer ("record-factory",
							       "Record factory",
							       "Record factory",
					 		        G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	
	g_object_class_install_property (gobject_class, PROP_ACCEPTABLE_FORMATS,
					 g_param_spec_pointer ("acceptable-formats",
							       "Acceptable formats",
							       "Acceptable formats",
					 		        G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	g_object_class_install_property (gobject_class, PROP_DB_DIR,
					 g_param_spec_string ("db-dir",
							      "Directory for database cache",
							      "Directory for database cache",
				 	 		       NULL,
					 		       G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	}

static guint
_add (DmapDb *db, DmapRecord *record, GError **error)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->add (db, record, error);
}

static guint
_add_with_id (DmapDb *db, DmapRecord *record, guint id, GError **error)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->add_with_id (db, record, id, error);
}

static guint
_add_path (DmapDb *db, const gchar *path, GError **error)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->add_path (db, path, error);
}

static DmapRecord *
_lookup_by_id (const DmapDb *db, guint id)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->lookup_by_id (db, id);
}

static guint
_lookup_id_by_location (const DmapDb *db, const gchar *location)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->lookup_id_by_location (db, location);
}

static void
_foreach (const DmapDb *db, DmapIdRecordFunc func, gpointer data)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->foreach (db, func, data);
}

static gint64
_count (const DmapDb *db)
{
	return DMAPD_DMAP_DB_GET_CLASS (db)->count (db);
}

static void
_dmap_db_interface_init (gpointer iface, gpointer data)
{
	DmapDbInterface *dmap_db = iface;

	g_assert (G_TYPE_FROM_INTERFACE (dmap_db) == DMAP_TYPE_DB);

	dmap_db->add = _add;
	dmap_db->add_with_id = _add_with_id;
	dmap_db->add_path = _add_path;
	dmap_db->lookup_by_id = _lookup_by_id;
	dmap_db->lookup_id_by_location = _lookup_id_by_location;
	dmap_db->foreach = _foreach;
	dmap_db->count = _count;
}

G_DEFINE_TYPE_WITH_CODE (DmapdDmapDb, dmapd_dmap_db, G_TYPE_OBJECT, 
			 G_IMPLEMENT_INTERFACE (DMAP_TYPE_DB, _dmap_db_interface_init))
