//////////////////////////////////////////////////////////////////////////////
//    Copyright 2004-2014, SenseGraphics AB
//
//    This file is part of H3D API.
//
//    H3D API is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    H3D API is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with H3D API; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//    A commercial license is also available. Please contact us at 
//    www.sensegraphics.com for more information.
//
//
/// \file PhysX3Joints.cpp
/// \brief Source file for BulletJoint classes, which maintain a link between 
/// RigidBodyPhysics joint types and their Bullet implementations
///
//
//////////////////////////////////////////////////////////////////////////////
#include <H3D/H3DPhysics/PhysX3Joints.h>
#include <H3D/H3DPhysics/PhysX3Callbacks.h>

#ifdef HAVE_PHYSX3

using namespace H3D;
using namespace physx;
using namespace PhysicsEngineParameters;

#ifdef min
#undef min
#endif

#ifdef max
#undef max
#endif

PhysX3Joint::JointDatabase::JointMap PhysX3Joint::JointDatabase::map;

PhysX3Joint::PhysX3Joint ( ConstraintParameters& jointParameters ) :
joint ( NULL ) {
  jointParameters.setConstraintId ( (H3DConstraintId)this );
}

PhysX3Joint::~PhysX3Joint () {
  remove();
}

// Create a joint instance of the specified type name
PhysX3Joint* PhysX3Joint::createJoint ( ConstraintParameters& jointParameters ) {
  CreateFunc f= JointDatabase::map[jointParameters.getType()];

  if ( f )
    return (*f)( jointParameters );
  else
    return NULL;    
}

// Remove joint from simulation
void PhysX3Joint::remove () {
  if ( joint ) {
    joint->release();
    joint= NULL;
  }
}

void PhysX3Joint::setJointFrame ( const Vec3f& position, const Rotation& orientation ) {
  global_frame= Matrix4f ( position, orientation );

  PxTransform globalFrame= PxTransform ( toPxVec3(position), toPxQuat (orientation) );
  PxTransform localFrame1= globalFrame;
  PxTransform localFrame2= globalFrame;

  PxRigidActor* actor1, *actor2;
  joint->getActors ( actor1, actor2 );

  if ( actor1 ) {
    localFrame1= actor1->getGlobalPose().getInverse()*localFrame1;
  }
  if ( actor2 ) {
    localFrame2= actor2->getGlobalPose().getInverse()*localFrame2;
  }
    
  joint->setLocalPose ( PxJointActorIndex::eACTOR0, localFrame1 );
  joint->setLocalPose ( PxJointActorIndex::eACTOR1, localFrame2 );
}

Matrix4f PhysX3Joint::getJointFrame ( Body::e body ) {
  PxRigidActor* actor1, *actor2;
  joint->getActors ( actor1, actor2 );

  PxTransform localFrame= joint->getLocalPose ( 
    (body==Body::A) ? PxJointActorIndex::eACTOR1 : PxJointActorIndex::eACTOR0 );
  PxTransform globalFrame= localFrame;

  PxRigidActor* actor= (body==Body::A) ? actor2 : actor1;
  if ( actor ) {
    globalFrame= actor->getGlobalPose() * globalFrame;
  }

  return toMatrix4f ( globalFrame );
}

H3DFloat PhysX3Joint::getRotation ( Axis::e axis, Body::e axisRelativeTo ) {
  Vec3f localY, localX;
  switch ( axis ) {
  case Axis::X:
    localY= Vec3f(0,1,0);
    localX= Vec3f(1,0,0);
    break;
  case Axis::Y:
    localY= Vec3f(1,0,0);
    localX= Vec3f(0,1,0);
    break;
  case Axis::Z:
    localY= Vec3f(0,1,0);
    localX= Vec3f(0,0,1);
    break;
  }

  Vec3f bodyA_Y= getJointFrame(Body::B).getRotationPart()*localY;
  Vec3f bodyB_Y= getJointFrame(Body::A).getRotationPart()*localY;
  Vec3f bodyB_X= getJointFrame(axisRelativeTo).getRotationPart()*localX;

  // Project bodyA_Y on to plane with normal bodyB_X
  Vec3f projBodyA_Y= bodyA_Y - bodyA_Y.dotProduct ( bodyB_X ) * bodyB_X;
  projBodyA_Y.normalizeSafe();

  // Get the angle between the two Y axes now in the same plane
  H3DFloat dot= projBodyA_Y.dotProduct ( bodyB_Y );
  dot= std::max ( -1.0f, std::min ( 1.0f, dot ) );
  H3DFloat angle= acos ( dot );

  // Get the correct sign depending on the direction of rotation
  Vec3f crossProd= projBodyA_Y.crossProduct ( bodyB_Y );
  crossProd.normalizeSafe();
  if ( crossProd.dotProduct ( bodyB_X ) < 0 ) {
    angle= -angle;
  }

  return angle;
}


// Fixed joint
//

PhysX3FixedJoint::JointDatabase
PhysX3FixedJoint::database ( "FixedJoint", &newInstance<PhysX3FixedJoint> );

PhysX3FixedJoint::PhysX3FixedJoint ( ConstraintParameters& jointParameters ) :
PhysX3Joint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  FixedJointParameters* params= static_cast < FixedJointParameters* > ( &jointParameters );

  PhysX3RigidBody* body1= (PhysX3RigidBody*)params->getBody1();
  PhysX3RigidBody* body2= (PhysX3RigidBody*)params->getBody2();

  PxRigidActor* actor1= body1 ? body1->getActor() : NULL;
  PxRigidActor* actor2= body2 ? body2->getActor() : NULL;

  joint= PxFixedJointCreate ( 
    *physx_data->sdk, 
    actor1, PxTransform::createIdentity(), 
    actor2, PxTransform::createIdentity() );

  // Anchor point is in between body 1 and body 2
  Vec3f anchorPoint;
  if ( actor1 ) {
    anchorPoint= toVec3f(actor1->getGlobalPose().p);
  }
  if ( actor2 ) {
    anchorPoint+= toVec3f(actor2->getGlobalPose().p);
  }
  if ( actor1 && actor2 ) {
    anchorPoint/= 2;
  }

  setJointFrame ( anchorPoint );
}


// Ball joint
//

PhysX3BallJoint::JointDatabase
PhysX3BallJoint::database ( "BallJoint", &newInstance<PhysX3BallJoint> );

PhysX3BallJoint::PhysX3BallJoint ( ConstraintParameters& jointParameters ) :
PhysX3Joint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  BallJointParameters* params= static_cast < BallJointParameters* > ( &jointParameters );

  PhysX3RigidBody* body1= (PhysX3RigidBody*)params->getBody1();
  PhysX3RigidBody* body2= (PhysX3RigidBody*)params->getBody2();

  joint= PxSphericalJointCreate ( 
    *physx_data->sdk, 
    body1 ? body1->getActor() : NULL, PxTransform::createIdentity(), 
    body2 ? body2->getActor() : NULL, PxTransform::createIdentity() );

  setParameters ( jointParameters );
}

// Set the bullet joint's parameters using the parameter values specified in jointParameters
void PhysX3BallJoint::setParameters ( ConstraintParameters& jointParameters ) {
  BallJointParameters* params= static_cast < BallJointParameters* > ( &jointParameters );

  // Anchor point
  if ( params->haveAnchorPoint() ) {
    setJointFrame ( params->getAnchorPoint() );
  }
}

// Modify the jointParameters argument to reflect the bullet joint's current parameters
void PhysX3BallJoint::getParameters ( ConstraintParameters& jointParameters ) {
  BallJointParameters* params= static_cast < BallJointParameters* > ( &jointParameters );

  // Body 1 anchor point
  if ( params->haveBody1AnchorPoint() ) {
    params->setBody1AnchorPoint ( getJointFrame(Body::B).getTranslationPart() );
  }

  // Body 2 anchor point
  if ( params->haveBody2AnchorPoint() ) {
    params->setBody2AnchorPoint ( getJointFrame(Body::A).getTranslationPart() );
  }
}


// Single axis hinge joint
//

PhysX3SingleAxisHingeJoint::JointDatabase
PhysX3SingleAxisHingeJoint::database ( "SingleAxisHingeJoint", &newInstance<PhysX3SingleAxisHingeJoint> );

PhysX3SingleAxisHingeJoint::PhysX3SingleAxisHingeJoint ( ConstraintParameters& jointParameters ) :
PhysX3Joint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  SingleAxisHingeJointParameters* params= static_cast < SingleAxisHingeJointParameters* > ( &jointParameters );

  PhysX3RigidBody* body1= (PhysX3RigidBody*)params->getBody1();
  PhysX3RigidBody* body2= (PhysX3RigidBody*)params->getBody2();

  joint= PxRevoluteJointCreate ( 
    *physx_data->sdk, 
    body1 ? body1->getActor() : NULL, PxTransform::createIdentity(), 
    body2 ? body2->getActor() : NULL, PxTransform::createIdentity() );

  setParameters ( jointParameters );
}

// Set the bullet joint's parameters using the parameter values specified in jointParameters
void PhysX3SingleAxisHingeJoint::setParameters ( ConstraintParameters& jointParameters ) {
  SingleAxisHingeJointParameters* params= static_cast < SingleAxisHingeJointParameters* > ( &jointParameters );
  PxRevoluteJoint* revoluteJoint= static_cast<PxRevoluteJoint*>(joint);

  // Anchor point
  if ( params->haveAnchorPoint() ) {
    setJointFrame ( params->getAnchorPoint(), Rotation(getJointFrame().getRotationPart()) );
  }

  // Axis
  if ( params->haveAxis() ) {
    Vec3f axis= params->getAxis();
    axis.normalizeSafe();
    setJointFrame ( getJointFrame().getTranslationPart(), Rotation ( Vec3f(1,0,0), axis ) );
  }

  // Limit max angle
  if ( params->haveMaxAngle () ) {
    revoluteJoint->setLimit ( 
      PxJointLimitPair ( 
        -params->getMaxAngle(), 
        revoluteJoint->getLimit().upper, 
        revoluteJoint->getLimit().contactDistance ) );
    revoluteJoint->setRevoluteJointFlag ( 
      PxRevoluteJointFlag::eLIMIT_ENABLED, revoluteJoint->getLimit().upper > revoluteJoint->getLimit().lower );
  }

  // Limit min angle
  if ( params->haveMinAngle () ) {
    revoluteJoint->setLimit ( 
      PxJointLimitPair ( 
        revoluteJoint->getLimit().lower, 
        -params->getMinAngle(), 
        revoluteJoint->getLimit().contactDistance ) );
    revoluteJoint->setRevoluteJointFlag ( 
      PxRevoluteJointFlag::eLIMIT_ENABLED, revoluteJoint->getLimit().upper > revoluteJoint->getLimit().lower );
  }

  // Stop bounce
  if ( params->haveStopBounce() ) {
    PxJointLimitPair limit= revoluteJoint->getLimit();
    limit.restitution= params->getStopBounce();
    revoluteJoint->setLimit ( limit );
  }

  /// \todo stopErrorCorrection
}

// Modify the jointParameters argument to reflect the bullet joint's current parameters
void PhysX3SingleAxisHingeJoint::getParameters ( ConstraintParameters& jointParameters ) {
  SingleAxisHingeJointParameters* params= static_cast < SingleAxisHingeJointParameters* > ( &jointParameters );

  // Body 1 anchor point
  if ( params->haveBody1AnchorPoint() ) {
    params->setBody1AnchorPoint ( getJointFrame(Body::A).getTranslationPart() );
  }

  // Body 2 anchor point
  if ( params->haveBody2AnchorPoint() ) {
    params->setBody2AnchorPoint ( getJointFrame(Body::B).getTranslationPart() );
  }

  if ( params->haveAngle() || params->haveAngleRate() ) {
    // Note: Reversed to match Bullet and ODE convension for single axis hinge
    H3DFloat angle= -getRotation ( Axis::X );

    // Angle
    if ( params->haveAngle() ) {
      params->setAngle ( angle );
    }

    // Angle rate
    if ( params->haveAngleRate() ) {
      hingeAngleRate.setCurrentValue ( angle );
      params->setAngleRate ( hingeAngleRate );
    }
  }
}


// Abstract base for 6 DoF joints
//

PhysX36DoFJoint::PhysX36DoFJoint ( ConstraintParameters& jointParameters ) :
PhysX3Joint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  JointParameters* params= static_cast < JointParameters* > ( &jointParameters );

  PhysX3RigidBody* body1= (PhysX3RigidBody*)params->getBody1();
  PhysX3RigidBody* body2= (PhysX3RigidBody*)params->getBody2();

  joint= PxD6JointCreate ( 
    *physx_data->sdk, 
    body2 ? body2->getActor() : NULL, PxTransform::createIdentity(), 
    body1 ? body1->getActor() : NULL, PxTransform::createIdentity() );
}

void PhysX36DoFJoint::makeAxesPerpendicular () {
  axis3= axis1.crossProduct ( axis2 );
  axis3.normalizeSafe();

  Vec3f newAxis2= axis3.crossProduct ( axis1 );
  newAxis2.normalizeSafe();

  if ( axis2.dotProduct ( newAxis2 ) < 1-H3DUtil::Constants::f_epsilon ) {
    Console(4) << "Warning: PhysX3 implemetation of joints only supports perpendicular axes!" << endl;
    axis2= newAxis2;
  }
}


// Double axis hinge joint
//

PhysX3DoubleAxisHingeJoint::JointDatabase
PhysX3DoubleAxisHingeJoint::database ( "DoubleAxisHingeJoint", &newInstance<PhysX3DoubleAxisHingeJoint> );

PhysX3DoubleAxisHingeJoint::PhysX3DoubleAxisHingeJoint ( ConstraintParameters& jointParameters ) :
PhysX36DoFJoint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  DoubleAxisHingeJointParameters* params= static_cast < DoubleAxisHingeJointParameters* > ( &jointParameters );

  PxD6Joint* d6Joint= static_cast<PxD6Joint*>(joint);
  d6Joint->setMotion ( PxD6Axis::eTWIST, PxD6Motion::eFREE );
  d6Joint->setMotion ( PxD6Axis::eSWING1, PxD6Motion::eFREE );

  setParameters ( jointParameters );
}

// Set the bullet joint's parameters using the parameter values specified in jointParameters
void PhysX3DoubleAxisHingeJoint::setParameters ( ConstraintParameters& jointParameters ) {
  DoubleAxisHingeJointParameters* params= static_cast < DoubleAxisHingeJointParameters* > ( &jointParameters );
  PxD6Joint* d6Joint= static_cast<PxD6Joint*>(joint);

  // Anchor point
  if ( params->haveAnchorPoint() ) {
    setJointFrame ( params->getAnchorPoint(), Rotation(getJointFrame().getRotationPart()) );
  }

  if ( params->haveAxis1() || params->haveAxis2() ) {
    // Axis 1
    if ( params->haveAxis1() ) {
      axis1= params->getAxis1();
      axis1.normalizeSafe();
    }

    // Axis 2
    if ( params->haveAxis2() ) {
      axis2= params->getAxis2();
      axis2.normalizeSafe();
    }

    // Update axes
    makeAxesPerpendicular();
    setJointFrame ( 
      getJointFrame().getTranslationPart(), 
      Rotation( Matrix3f ( axis1.x, axis2.x, axis3.x,
                           axis1.y, axis2.y, axis3.y,
                           axis1.z, axis2.z, axis3.z ) ) );
  }

  // Limit max angle 1
  if ( params->haveMaxAngle1 () ) { 
    PxJointLimitPair limit= d6Joint->getTwistLimit();
    limit.upper= params->getMaxAngle1();
    d6Joint->setTwistLimit ( limit );
    d6Joint->setMotion ( PxD6Axis::eTWIST, (limit.upper > limit.lower) ? PxD6Motion::eLIMITED : PxD6Motion::eLOCKED );
  }
  
  // Limit min angle 1
  if ( params->haveMinAngle1 () ) { 
    PxJointLimitPair limit= d6Joint->getTwistLimit();
    limit.lower= params->getMinAngle1();
    d6Joint->setTwistLimit ( limit );
    d6Joint->setMotion ( PxD6Axis::eTWIST, (limit.upper > limit.lower) ? PxD6Motion::eLIMITED : PxD6Motion::eLOCKED );
  }

  // Stop bounce
  if ( params->haveStopBounce1() ) {
    PxJointLimitPair limit= d6Joint->getTwistLimit();
    limit.restitution= params->getStopBounce1();
    d6Joint->setTwistLimit ( limit );
  }

  /// \todo stopConstantForceMix1, stopErrorCorrection1
  /// \todo suspensionErrorCorrection, suspensionForce
  /// \todo desiredAngularVelocity1, maxTorque1
  /// \todo desiredAngularVelocity2, maxTorque2
}

// Modify the jointParameters argument to reflect the bullet joint's current parameters
void PhysX3DoubleAxisHingeJoint::getParameters ( ConstraintParameters& jointParameters ) {
  DoubleAxisHingeJointParameters* params= static_cast < DoubleAxisHingeJointParameters* > ( &jointParameters );

  // Body 1 anchor point
  if ( params->haveBody1AnchorPoint() ) {
    params->setBody1AnchorPoint ( getJointFrame(Body::B).getTranslationPart() );
  }

  // Body 2 anchor point
  if ( params->haveBody2AnchorPoint() ) {
    params->setBody2AnchorPoint ( getJointFrame(Body::A).getTranslationPart() );
  }

  // Hinge 1
  if ( params->haveHinge1Angle() || params->haveHinge1AngleRate() ) {
    H3DFloat angle= getRotation ( Axis::X ); 

    // Angle
    if ( params->haveHinge1Angle() ) {
      params->setHinge1Angle ( angle );
    }

    // Angle rate
    if ( params->haveHinge1AngleRate() ) {
      hinge1AngleRate.setCurrentValue ( angle );
      params->setHinge1AngleRate ( hinge1AngleRate );
    }
  }

  // Hinge 2
  if ( params->haveHinge2Angle() || params->haveHinge2AngleRate() ) {
    H3DFloat angle= getRotation ( Axis::Y, Body::B ); 

    // Angle
    if ( params->haveHinge2Angle() ) {
      params->setHinge2Angle ( angle );
    }

    // Angle rate
    if ( params->haveHinge2AngleRate() ) {
      hinge2AngleRate.setCurrentValue ( angle );
      params->setHinge2AngleRate ( hinge2AngleRate );
    }
  }
}


// Universal joint
//

PhysX3UniversalJoint::JointDatabase
PhysX3UniversalJoint::database ( "UniversalJoint", &newInstance<PhysX3UniversalJoint> );

PhysX3UniversalJoint::PhysX3UniversalJoint ( ConstraintParameters& jointParameters ) :
PhysX36DoFJoint ( jointParameters ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  UniversalJointParameters* params= static_cast < UniversalJointParameters* > ( &jointParameters );

  PxD6Joint* d6Joint= static_cast<PxD6Joint*>(joint);
  d6Joint->setMotion ( PxD6Axis::eTWIST, PxD6Motion::eFREE );
  d6Joint->setMotion ( PxD6Axis::eSWING1, PxD6Motion::eFREE );

  setParameters ( jointParameters );
}

// Set the bullet joint's parameters using the parameter values specified in jointParameters
void PhysX3UniversalJoint::setParameters ( ConstraintParameters& jointParameters ) {
  UniversalJointParameters* params= static_cast < UniversalJointParameters* > ( &jointParameters );
  PxD6Joint* d6Joint= static_cast<PxD6Joint*>(joint);

  // Anchor point
  if ( params->haveAnchorPoint() ) {
    setJointFrame ( params->getAnchorPoint(), Rotation(getJointFrame().getRotationPart()) );
  }

  if ( params->haveAxis1() || params->haveAxis2() ) {
    // Axis 1
    if ( params->haveAxis1() ) {
      axis1= params->getAxis1();
      axis1.normalizeSafe();
    }

    // Axis 2
    if ( params->haveAxis2() ) {
      axis2= params->getAxis2();
      axis2.normalizeSafe();
    }

    // Update axes
    makeAxesPerpendicular();
    setJointFrame ( 
      getJointFrame().getTranslationPart(), 
      Rotation( Matrix3f ( axis1.x, axis2.x, axis3.x,
                           axis1.y, axis2.y, axis3.y,
                           axis1.z, axis2.z, axis3.z ) ) );
  }

  // Stop 1 bounce
  if ( params->haveStop1Bounce() ) {
    PxJointLimitPair limit= d6Joint->getTwistLimit();
    limit.restitution= params->getStop1Bounce();
    d6Joint->setTwistLimit ( limit );
  }

  // Stop 2 bounce
  if ( params->haveStop2Bounce() ) {
    PxJointLimitPair limit= d6Joint->getTwistLimit();
    limit.restitution= params->getStop2Bounce();
    d6Joint->setTwistLimit ( limit );
  }

  /// \todo stop1ErrorCorrection, stop2ErrorCorrection
}

// Modify the jointParameters argument to reflect the bullet joint's current parameters
void PhysX3UniversalJoint::getParameters ( ConstraintParameters& jointParameters ) {
  UniversalJointParameters* params= static_cast < UniversalJointParameters* > ( &jointParameters );

  // Body 1 anchor point
  if ( params->haveBody1AnchorPoint() ) {
    params->setBody1AnchorPoint ( getJointFrame(Body::B).getTranslationPart() );
  }

  // Body 2 anchor point
  if ( params->haveBody2AnchorPoint() ) {
    params->setBody2AnchorPoint ( getJointFrame(Body::A).getTranslationPart() );
  }

  // body1Axis output
  if( params->haveBody1Axis() ) {
    params->setBody1Axis( getJointFrame(Body::A).getRotationPart()*Vec3f(1,0,0) );
  }

  // body2Axis output
  if( params->haveBody2Axis() ) {
    params->setBody2Axis( getJointFrame(Body::B).getRotationPart()*Vec3f(0,1,0) );
  }
}


// Slider joint
//

PhysX3SliderJoint::JointDatabase
PhysX3SliderJoint::database ( "SliderJoint", &newInstance<PhysX3SliderJoint> );

PhysX3SliderJoint::PhysX3SliderJoint ( ConstraintParameters& jointParameters ) :
  PhysX3Joint ( jointParameters ),
  simulationCallbackId( -1 ),
  physicsEngineThread( jointParameters.getEngine() ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(jointParameters.getEngine()->getEngineSpecificData());
  SliderJointParameters* params= static_cast < SliderJointParameters* > ( &jointParameters );

  PhysX3RigidBody* body2= (PhysX3RigidBody*)params->getBody1();
  PhysX3RigidBody* body1= (PhysX3RigidBody*)params->getBody2();

  joint= PxPrismaticJointCreate ( 
    *physx_data->sdk, 
    body1 ? body1->getActor() : NULL, PxTransform::createIdentity(), 
    body2 ? body2->getActor() : NULL, PxTransform::createIdentity() );

  setParameters ( jointParameters );
}

PhysX3SliderJoint::~PhysX3SliderJoint() {
  remove();
}

// Set the bullet joint's parameters using the parameter values specified in jointParameters
void PhysX3SliderJoint::setParameters ( ConstraintParameters& jointParameters ) {
  SliderJointParameters* params= static_cast < SliderJointParameters* > ( &jointParameters );
  PxPrismaticJoint* prismaticJoint= static_cast<PxPrismaticJoint*>(joint);

  // Axis
  if ( params->haveAxis() ) {
    Vec3f axis= params->getAxis();
    axis.normalizeSafe();

    // Anchor point is in between body 1 and body 2
    PxRigidActor* actor1, *actor2;
    joint->getActors ( actor1, actor2 );
    Vec3f anchorPoint;
    if ( actor1 ) {
      anchorPoint= toVec3f(actor1->getGlobalPose().p);
    }
    if ( actor2 ) {
      anchorPoint+= toVec3f(actor2->getGlobalPose().p);
    }
    if ( actor1 && actor2 ) {
      anchorPoint/= 2;
    }

    setJointFrame ( anchorPoint, Rotation ( Vec3f(1,0,0), axis ) );
  }

  /// \todo The slider limits only work correctly when the global axis for 
  ///       the slider is along the world x-axis.
  if ( params->haveMaxSeparation () ) {
    PxJointLimitPair limit= prismaticJoint->getLimit();
    limit.upper= params->getMaxSeparation();
    prismaticJoint->setLimit ( limit );
    prismaticJoint->setPrismaticJointFlag ( 
      PxPrismaticJointFlag::eLIMIT_ENABLED, prismaticJoint->getLimit().upper > prismaticJoint->getLimit().lower );
  }

  if ( params->haveMinSeparation () ) {
    PxJointLimitPair limit= prismaticJoint->getLimit();
    limit.lower= params->getMinSeparation();
    prismaticJoint->setLimit ( limit );
    prismaticJoint->setPrismaticJointFlag ( 
      PxPrismaticJointFlag::eLIMIT_ENABLED, prismaticJoint->getLimit().upper > prismaticJoint->getLimit().lower );
  }

  if( params->haveSliderForce() ) {
    slider_force = params->getSliderForce();
    if( simulationCallbackId < 0 ) {
      if( slider_force != 0 )
        simulationCallbackId= params->getEngine()->asynchronousCallback ( &PhysX3SliderJoint::updateSlider, this );
    } else if( slider_force == 0 ) {
      params->getEngine()->removeAsynchronousCallbackNoLock( simulationCallbackId );
      simulationCallbackId = -1;
    }
  } else if( simulationCallbackId >= 0 ) {
    params->getEngine()->removeAsynchronousCallbackNoLock( simulationCallbackId );
    simulationCallbackId = -1;
  }
}

// Modify the jointParameters argument to reflect the bullet joint's current parameters
void PhysX3SliderJoint::getParameters ( ConstraintParameters& jointParameters ) {
  SliderJointParameters* params= static_cast < SliderJointParameters* > ( &jointParameters );

  if ( params->haveSeparation() ) {

    Vec3f diff= getJointFrame(Body::B).getTranslationPart()-getJointFrame(Body::A).getTranslationPart();
    H3DFloat sep= -diff.length();

    Vec3f globalAxis= getJointFrame(Body::A).getRotationPart()*Vec3f(1,0,0);
    if ( diff.dotProduct ( globalAxis ) < 0 ) {
      sep= -sep;
    }

    params->setSeparation( sep );
  }
}

// Callback used to update slider simulation
H3DUtil::PeriodicThread::CallbackCode PhysX3SliderJoint::updateSlider ( void *data )
{
  static_cast < PhysX3SliderJoint* > ( data )->update ();
  return H3DUtil::PeriodicThread::CALLBACK_CONTINUE;
}

// Member function called by static callback, used to update motor simulation
void PhysX3SliderJoint::update ()
{
  PxVec3 the_force( slider_force, 0, 0 );
  PxRigidActor* actor1, *actor2;
  joint->getActors ( actor1, actor2 );
  if( actor1 ) {
    PxTransform localFrame = joint->getLocalPose( PxJointActorIndex::eACTOR0 );
    PxTransform globalFrame= localFrame;
    globalFrame= actor1->getGlobalPose() * globalFrame;
    static_cast< PxRigidDynamic * >(actor1)->addForce( globalFrame.q.rotate( -the_force ) );
  }
  if( actor2 ) {
    PxTransform localFrame = joint->getLocalPose( PxJointActorIndex::eACTOR1 );
    PxTransform globalFrame= localFrame;
    globalFrame= actor2->getGlobalPose() * globalFrame;
    static_cast< PxRigidDynamic * >(actor2)->addForce( globalFrame.q.rotate( the_force ) );
  }
}

// Remove joint from simulation
// Override to remove simulation callback
void PhysX3SliderJoint::remove ()
{
  if( simulationCallbackId >= 0 ) {
    physicsEngineThread->removeAsynchronousCallbackNoLock ( simulationCallbackId );
    simulationCallbackId = -1;
  }
}

#endif

