/*
*  Copyright (C) 2010 Craig Robinson, Yi Pan
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with with program; see the file COPYING. If not, write to the
*  Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
*  MA  02111-1307  USA
*/


/**
 * \author Craig Robinson, Yi Pan
 *
 * \brief More recent versions of the EOB models, such as EOBNRv2 and SEOBNRv1, utilise
 * a non-quasicircular correction (NQC) to bring the peak of the EOB frequency
 * into agreement with that of NR simulations. This file contains the functions
 * used to calculate these NQC corrections, described in DCC document T1100433.
 * The fits to NR peak amplitude, frequency, and their derivatives, are taken
 * from Pan et al. PRD 84 124052 (2011), for EOBNRv2, and
 * from Taracchini et al. PRD 86, 024011 (2012), for SEOBNRv1.
 */

#include <complex.h>
#include <math.h>

#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_linalg.h>

#include "LALSimIMREOBNRv2.h"

#define OLDNSNQC 0
#define NEWNSNQC 1

#ifndef _LALSIMIMRNQCCORRECTION_C 
#define _LALSIMIMRNQCCORRECTION_C 

#ifdef __GNUC__
#define UNUSED __attribute__ ((unused))
#else
#define UNUSED
#endif

/* ------------------------------------------------
 *          Non-spin (EOBNRv2)
 * ------------------------------------------------*/

/**
 * Compute the time offset which should be used in computing the
 * non-quasicircular correction and performing the ringdown attachment.
 * These numbers were tuned to numerical relativity simulations, and
 * are taken from Pan et al, PRD84, 124052(2011), lines 1-5 of Table II.
 */
 static REAL8 XLALSimIMREOBGetNRPeakDeltaT( 
                         INT4 l,    /**<< Mode l */ 
                         INT4 m,    /**<< Mode m */
                         REAL8 eta  /**<< Symmetric mass ratio */
                         )
{
  switch ( l )
  {
    case 2:
      if ( m == 2 )
      {
        return 0.0;
      }
      else if ( m == 1 )
      {
        return 10.67 - 2.5 + 9.0*eta - 41.41 * eta + 76.1 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 3:
      if ( m == 3 )
      {
        return 3.383 + 3.847 * eta + 8.979 * eta * eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 4:
      if ( m == 4 )
      {
        return 5.57 - 49.86 * eta + 154.3 * eta * eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 5:
      if ( m == 5 )
      {
        return 6.693 - 34.47 * eta + 102.7 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
      break;
  }
}

/**
 * Function which returns a value of the expected peak amplitude
 * taken from a fit to numerical relativity simulations. The functions
 * are taken from Pan et al, PRD84, 124052(2011), lines 1-5 of Table II.
 */
static inline
REAL8 GetNRPeakAmplitude( 
                        INT4 l,   /**<< Mode l */ 
                        INT4 m,   /**<< Mode m */
                        REAL8 eta /**<< Symmetric mass ratio */
                        )
{
  switch ( l )
  {
    case 2:
      if ( m == 2 )
      {
        return eta * ( 1.422 + 0.3013 * eta + 1.246 * eta * eta );
      }
      else if ( m == 1 )
      {
        return eta * sqrt( 1.0 - 4. * eta ) * (0.4832 - 0.01032 * eta);
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 3:
      if ( m == 3 )
      {
        return eta * sqrt(1.-4.*eta) * ( 0.5761 - 0.09638 * eta + 2.715*eta*eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 4:
      if ( m == 4 )
      {
        return eta * (0.354 - 1.779 * eta + 2.834 * eta*eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 5:
      if ( m == 5 )
      {
        return eta * sqrt(1.-4.*eta) * ( 0.1353 - 0.1485 * eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
      break;
  }
}

/**
 * Function which returns second derivative of the amplitude at the peak
 * taken from a fit to numerical relativity simulations. The functions
 * are taken from Pan et al, PRD84, 124052(2011), lines 1-5 of Table II.
 */
static inline
REAL8 GetNRPeakADDot( 
                    INT4 l,   /**<< Mode l */ 
                    INT4 m,   /**<< Mode m */
                    REAL8 eta /**<< Symmetric mass ratio */
                    )
{
  switch ( l )
  {
    case 2:
      if ( m == 2 )
      {
        return -0.01 * eta * ( 0.1679 + 1.44 * eta - 2.001 * eta * eta );
      }
      else if ( m == 1 )
      {
        return -0.01 * eta * sqrt(1.-4.*eta) * (0.1867 + 0.6094 * eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 3:
      if ( m == 3 )
      {
        return -0.01 * eta * sqrt(1.-4.*eta) * (0.2518 - 0.8145*eta + 5.731*eta*eta);
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 4:
      if ( m == 4 )
      {
        return -0.01 * eta * (0.1813 - 0.9935 * eta + 1.858 * eta * eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 5:
      if ( m == 5 )
      {
        return -0.01 * eta * sqrt(1.-4.*eta) * ( 0.09051 - 0.1604 * eta );
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
      break;
  }
}


/**
 * Function which returns a value of the expected peak frequency
 * taken from a fit to numerical relativity simulations. The functions
 * are taken from Pan et al, PRD84, 124052(2011), lines 1-5 of Table II.
 */
static inline 
REAL8 GetNRPeakOmega( 
                    INT4 l,   /**<< Mode l */
                    INT4 m,   /**<< Mode m */
                    REAL8 eta /**<< Symmetric mass ratio */
                    )
{
  switch ( l )
  {
    case 2:
      if ( m == 2 )
      {
        return 0.2733 + 0.2316 * eta + 0.4463 * eta * eta;
      }
      else if ( m == 1 )
      {
        return 0.2907 - 0.08338 * eta + 0.587 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 3:
      if ( m == 3 )
      {
        return 0.4539 + 0.5376 * eta + 1.042 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 4:
      if ( m == 4 )
      {
        return 0.6435 - 0.05103 * eta + 2.216 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 5:
      if ( m == 5 )
      {
        return 0.8217 + 0.2346 * eta + 2.599 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
      break;
  }
}

/**
 * Function which returns the derivative of the expected peak frequency
 * taken from a fit to numerical relativity simulations. The functions
 * are taken from Pan et al, PRD84, 124052(2011), lines 1-5 of Table II.
 */
static inline 
REAL8 GetNRPeakOmegaDot( 
                       INT4 l,   /**<< Mode l */ 
                       INT4 m,   /**<< Mode m */
                       REAL8 eta /**<< Symmetric mass ratio */
                       )
{
  switch ( l )
  {
    case 2:
      if ( m == 2 )
      {
        return 0.005862 + 0.01506 * eta + 0.02625 * eta * eta;
      }
      else if ( m == 1 )
      {
        return 0.00149 + 0.09197 * eta - 0.1909 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 3:
      if ( m == 3 )
      {
        return 0.01074 + 0.0293 * eta + 0.02066 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 4:
      if ( m == 4 )
      {
        return 0.01486 + 0.08529 * eta - 0.2174 * eta * eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    case 5:
      if ( m == 5 )
      {
        return 0.01775 + 0.09801 * eta - 0.1686 * eta*eta;
      }
      else
      {
        XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
      break;
  }
}


/**
 * For the 2,2 mode, there are fits available for the NQC coefficients,
 * given in Eqs.(40a)-(40c) of Pan et al, PRD84, 124052(2011).
 * This function provides the values of these coefficients, so the
 * correction can be used in the dynamics prior to finding the more
 * accurate NQC values later on.
 */
UNUSED static int XLALSimIMREOBGetCalibratedNQCCoeffs( 
                                EOBNonQCCoeffs *coeffs, /**<< OUTPUT, Structure for NQC coeffs */
                                INT4            l,      /**<< Mode l */
                                INT4            m,      /**<< Mode m */
                                REAL8           eta     /**<< Symmetric mass ratio */
                                )
{

#ifndef LAL_NDEBUG
  if ( !coeffs )
  {
    XLAL_ERROR( XLAL_EINVAL );
  }
#endif

  if ( l != 2 || m != 2 )
  {
    XLALPrintError( "Mode %d,%d is not supported by this function.\n", l, m );
    XLAL_ERROR( XLAL_EINVAL );
  }

  /* All NQC coefficients are set to zero here */
  /* including coeffs->a3S, coeffs->a4 and coeffs->a5 that are not used in EOBNRv2 */
  memset( coeffs, 0, sizeof( *coeffs ) );

  coeffs->a1 = -4.55919 + 18.761 * eta - 24.226 * eta*eta;
  coeffs->a2 = 37.683 - 201.468 * eta + 324.591 * eta*eta;
  coeffs->a3 = - 39.6024 + 228.899 * eta - 387.222 * eta * eta;

  return XLAL_SUCCESS;
}

/**
 * This function calculates the non-quasicircular correction to apply to
 * the waveform. The form of this correction can be found in Pan et al,
 * PRD84, 124052(2011), Eq.(22), and also in the DCC document T1100433. Note
 * that when calling this function, the NQC coefficients should already
 * have been pre-computed.
 */
UNUSED static int  XLALSimIMREOBNonQCCorrection(
                      COMPLEX16      * restrict nqc,    /**<< OUTPUT, The NQC correction */
                      REAL8Vector    * restrict values, /**<< Dynamics r, phi, pr, pphi */
                      const REAL8               omega,  /**<< Angular frequency */
                      EOBNonQCCoeffs * restrict coeffs  /**<< NQC coefficients */
                     )

{

  REAL8 rOmega, rOmegaSq;
  REAL8 r, p, sqrtR;

  REAL8 mag, phase;


  r = values->data[0];
  p = values->data[2];

  sqrtR = sqrt(r);

  rOmega = r * omega;
  rOmegaSq = rOmega*rOmega;
/*printf("a1 = %.16e, a2 = %.16e, a3 = %.16e, a3S = %.16e, a4 = %.16e, a5 = %.16e\n",coeffs->a1,coeffs->a2,coeffs->a3,coeffs->a3S, coeffs->a4,coeffs->a5);
printf("b1 = %.16e, b2 = %.16e, b3 = %.16e, b4 = %.16e\n",coeffs->b1,coeffs->b2,coeffs->b3,coeffs->b4);*/
  /* In EOBNRv2, coeffs->a3S, coeffs->a4 and coeffs->a5 are set to zero */
  /* through XLALSimIMREOBGetCalibratedNQCCoeffs() */
  /* and XLALSimIMREOBCalculateNQCCoefficients() */
  mag = 1. + (p*p / rOmegaSq) * ( coeffs->a1
     + coeffs->a2 / r + ( coeffs->a3 + coeffs->a3S) / (r*sqrtR)
     + coeffs->a4 / (r*r) + coeffs->a5 / (r*r*sqrtR));
//printf("NQC INFO mag = %.16e, r = %.16e, p = %.16e\n",mag,r,p);
  phase = coeffs->b1 * p / rOmega + p*p*p/rOmega * ( coeffs->b2
     + coeffs->b3 / sqrtR + coeffs->b4 / r );

  *nqc = mag * cos(phase);
  *nqc += I * mag * sin(phase);
/*printf("r = %.16e, pr = %.16e, omega = %.16e\n",r,p,omega);
printf("NQC mag = %.16e, arg = %.16e\n",mag,phase);*/
  return XLAL_SUCCESS;

}



/**
 * This function calculates the non-quasicircular correction to apply to
 * the waveform. The form of this correction can be found in Pan et al,
 * PRD84, 124052(2011), Eq.(22), and also in the DCC document T1100433. Note
 * that when calling this function, the NQC coefficients should already
 * have been pre-computed.
 * This version is for generic precesing case where the dynamics variable
 * values are given in Catesean coordinates.
 */
UNUSED static int  XLALSimIMRSpinEOBNonQCCorrection(
                      COMPLEX16      * restrict nqc,    /**<< OUTPUT, The NQC correction */
                      REAL8Vector    * restrict values, /**<< Dynamics r, phi, pr, pphi */
                      const REAL8               omega,  /**<< Angular frequency */
                      EOBNonQCCoeffs * restrict coeffs  /**<< NQC coefficients */
                     )

{

  REAL8 rOmega, rOmegaSq;
  REAL8 r, p, sqrtR;

  REAL8 mag, phase;


  r = sqrt(values->data[0]*values->data[0]+values->data[1]*values->data[1]+values->data[2]*values->data[2]);
  p = (values->data[0]*values->data[3]+values->data[1]*values->data[4]+values->data[2]*values->data[5]) / r;

  sqrtR = sqrt(r);

  rOmega = r * omega;
  rOmegaSq = rOmega*rOmega;

  /* In EOBNRv2, coeffs->a3S, coeffs->a4 and coeffs->a5 are set to zero */
  /* through XLALSimIMREOBGetCalibratedNQCCoeffs() */
  /* and XLALSimIMREOBCalculateNQCCoefficients() */
  mag = 1. + (p*p / rOmegaSq) * ( coeffs->a1
     + coeffs->a2 / r + ( coeffs->a3 + coeffs->a3S) / (r*sqrtR)
     + coeffs->a4 / (r*r) + coeffs->a5 / (r*r*sqrtR));

  phase = coeffs->b1 * p / rOmega + p*p*p/rOmega * ( coeffs->b2
     + coeffs->b3 / sqrtR + coeffs->b4 / r );

  *nqc = mag * cos(phase);
  *nqc += I * mag * sin(phase);

  return XLAL_SUCCESS;

}



/**
 * This function computes the coefficients a1, a2, etc. used in the
 * non-quasicircular correction. The details of the calculation of these
 * coefficients are found in the DCC document T1100433.
 */
UNUSED static int XLALSimIMREOBCalculateNQCCoefficients(
                 EOBNonQCCoeffs * restrict coeffs,    /**<< OUTPUT, NQC coefficients */
                 REAL8Vector    * restrict amplitude, /**<< Waveform amplitude, func of time */
                 REAL8Vector    * restrict phase,     /**<< Waveform phase(rad), func of time */
                 REAL8Vector    * restrict q1,        /**<< Function of dynamics (see DCC doc) */
                 REAL8Vector    * restrict q2,        /**<< Function of dynamics (see DCC doc) */
                 REAL8Vector    * restrict q3,        /**<< Function of dynamics (see DCC doc) */
                 REAL8Vector    * restrict p1,        /**<< Function of dynamics (see DCC doc) */
                 REAL8Vector    * restrict p2,        /**<< Function of dynamics (see DCC doc) */
                 INT4                      l,         /**<< Mode l */
                 INT4                      m,         /**<< Mode m */
                 REAL8                     timePeak,  /**<< Time of peak orbital frequency */
                 REAL8                     deltaT,    /**<< Sampling interval */
                 REAL8                     eta        /**<< Symmetric mass ratio */
                 )
{

  UINT4 i;

  int signum;

  REAL8Vector * restrict timeVec = NULL;

  /* Since the vectors we actually want are q etc * A, we will have to generate them here */
  REAL8Vector *q1LM = NULL;
  REAL8Vector *q2LM = NULL;
  REAL8Vector *q3LM = NULL; 

  REAL8 a, aDot, aDDot;
  REAL8 omega, omegaDot;

  REAL8 nra, nraDDot;
  REAL8 nromega, nromegaDot;

  REAL8 nrDeltaT, nrTimePeak;

  /* Stuff for finding numerical derivatives */
  gsl_spline    *spline = NULL;
  gsl_interp_accel *acc = NULL;

  /* Matrix stuff for calculating coefficients */
  gsl_matrix *qMatrix = NULL, *pMatrix = NULL;
  gsl_vector *aCoeff  = NULL, *bCoeff  = NULL;

  gsl_vector *amps = NULL, *omegaVec = NULL;

  gsl_permutation *perm1 = NULL, *perm2 = NULL;

  /* All NQC coefficients are set to zero here */ 
  /* including coeffs->a4 that is not used in EOBNRv2 */
  memset( coeffs, 0, sizeof( EOBNonQCCoeffs ) );

  /* Populate the time vector */
  /* It is okay to assume initial t = 0 */
  timeVec = XLALCreateREAL8Vector( q1->length );
  q1LM    = XLALCreateREAL8Vector( q1->length );
  q2LM    = XLALCreateREAL8Vector( q2->length );
  q3LM    = XLALCreateREAL8Vector( q3->length );

  /* Populate vectors as necessary */
  for ( i = 0; i < timeVec->length; i++ )
  {
    timeVec->data[i] = i * deltaT;
    q1LM->data[i]    = q1->data[i] * amplitude->data[i];
    q2LM->data[i]    = q2->data[i] * amplitude->data[i];
    q3LM->data[i]    = q3->data[i] * amplitude->data[i];
  }

  /* Allocate all the memory we need */
  XLAL_CALLGSL(
    /* a stuff */
    qMatrix = gsl_matrix_alloc( 3, 3 );
    aCoeff  = gsl_vector_alloc( 3 );
    amps    = gsl_vector_alloc( 3 );
    perm1   = gsl_permutation_alloc( 3 );

    /* b stuff */
    pMatrix  = gsl_matrix_alloc( 2, 2 );
    bCoeff   = gsl_vector_alloc( 2 );
    omegaVec = gsl_vector_alloc( 2 );
    perm2    = gsl_permutation_alloc( 2 );
  );

  if ( !qMatrix || !aCoeff || !amps || !pMatrix || !bCoeff || !omegaVec )
  {
    gsl_matrix_free( qMatrix );
    gsl_vector_free( amps );
    gsl_vector_free( aCoeff );
    gsl_permutation_free( perm1 );
    gsl_matrix_free( pMatrix );
    gsl_vector_free( omegaVec );
    gsl_vector_free( bCoeff );
    gsl_permutation_free( perm2 );
    XLALDestroyREAL8Vector( q1LM );
    XLALDestroyREAL8Vector( q2LM );
    XLALDestroyREAL8Vector( q3LM );
    XLALDestroyREAL8Vector( timeVec );
    XLAL_ERROR( XLAL_ENOMEM );
  }

  /* The time we want to take as the peak time depends on l and m */
  /* Calculate the adjustment we need to make here */
  nrDeltaT = XLALSimIMREOBGetNRPeakDeltaT( l, m, eta );
  if ( XLAL_IS_REAL8_FAIL_NAN( nrDeltaT ) )
  {
    XLALDestroyREAL8Vector( q1LM );
    XLALDestroyREAL8Vector( q2LM );
    XLALDestroyREAL8Vector( q3LM );
    XLALDestroyREAL8Vector( timeVec );
    XLAL_ERROR( XLAL_EFUNC );
  }

  nrTimePeak = timePeak + nrDeltaT;

  /* We are now in a position to use the interp stuff to calculate the derivatives we need */
  /* We will start with the quantities used in the calculation of the a coefficients */
  spline = gsl_spline_alloc( gsl_interp_cspline, amplitude->length );
  acc    = gsl_interp_accel_alloc();

  /* Q1 */
  gsl_spline_init( spline, timeVec->data, q1LM->data, q1LM->length );
  gsl_matrix_set( qMatrix, 0, 0, gsl_spline_eval( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 1, 0, gsl_spline_eval_deriv( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 2, 0, gsl_spline_eval_deriv2( spline, nrTimePeak, acc ) );

  /* Q2 */
  gsl_spline_init( spline, timeVec->data, q2LM->data, q2LM->length );
  gsl_interp_accel_reset( acc );
  gsl_matrix_set( qMatrix, 0, 1, gsl_spline_eval( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 1, 1, gsl_spline_eval_deriv( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 2, 1, gsl_spline_eval_deriv2( spline, nrTimePeak, acc ) );

  /* Q3 */
  gsl_spline_init( spline, timeVec->data, q3LM->data, q3LM->length );
  gsl_interp_accel_reset( acc );
  gsl_matrix_set( qMatrix, 0, 2, gsl_spline_eval( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 1, 2, gsl_spline_eval_deriv( spline, nrTimePeak, acc ) );
  gsl_matrix_set( qMatrix, 2, 2, gsl_spline_eval_deriv2( spline, nrTimePeak, acc ) );

  /* Amplitude */
  gsl_spline_init( spline, timeVec->data, amplitude->data, amplitude->length );
  gsl_interp_accel_reset( acc );
  a     = gsl_spline_eval( spline, nrTimePeak, acc );
  aDot  = gsl_spline_eval_deriv( spline, nrTimePeak, acc );
  aDDot = gsl_spline_eval_deriv2( spline, nrTimePeak, acc );

  nra = GetNRPeakAmplitude( l, m, eta );
  nraDDot = GetNRPeakADDot( l, m, eta );

  if ( XLAL_IS_REAL8_FAIL_NAN( nra ) || XLAL_IS_REAL8_FAIL_NAN( nraDDot ) )
  {
    XLALDestroyREAL8Vector( q1LM );
    XLALDestroyREAL8Vector( q2LM );
    XLALDestroyREAL8Vector( q3LM );
    XLALDestroyREAL8Vector( timeVec );
    XLAL_ERROR( XLAL_EFUNC );
  }

  gsl_vector_set( amps, 0, nra - a );
  gsl_vector_set( amps, 1, - aDot );
  gsl_vector_set( amps, 2, nraDDot - aDDot );

  /* We have now set up all the stuff to calculate the a coefficients */
  /* So let us do it! */
  gsl_linalg_LU_decomp( qMatrix, perm1, &signum );
  gsl_linalg_LU_solve( qMatrix, perm1, amps, aCoeff );

  /* Now we (should) have calculated the a values. Now we can do the b values */

  /* P1 */
  gsl_spline_init( spline, timeVec->data, p1->data, p1->length );
  gsl_interp_accel_reset( acc );
  gsl_matrix_set( pMatrix, 0, 0, - gsl_spline_eval_deriv( spline, nrTimePeak, acc ) );
  gsl_matrix_set( pMatrix, 1, 0, - gsl_spline_eval_deriv2( spline, nrTimePeak, acc ) );

  /* P2 */
  gsl_spline_init( spline, timeVec->data, p2->data, p2->length );
  gsl_interp_accel_reset( acc );
  gsl_matrix_set( pMatrix, 0, 1, - gsl_spline_eval_deriv( spline, nrTimePeak, acc ) );
  gsl_matrix_set( pMatrix, 1, 1, - gsl_spline_eval_deriv2( spline, nrTimePeak, acc ) );

  /* Phase */
  gsl_spline_init( spline, timeVec->data, phase->data, phase->length );
  gsl_interp_accel_reset( acc );
  omega    = gsl_spline_eval_deriv( spline, nrTimePeak, acc );
  omegaDot = gsl_spline_eval_deriv2( spline, nrTimePeak, acc );

  /* Since the phase can be decreasing, we need to take care not to have a -ve frequency */
  if ( omega * omegaDot > 0.0 )
  {
    omega    = fabs( omega );
    omegaDot = fabs( omegaDot );
  }
  else
  {
    omega    = fabs( omega );
    omegaDot = - fabs( omegaDot );
  }

  nromega = GetNRPeakOmega( l, m, eta );
  nromegaDot = GetNRPeakOmegaDot( l, m, eta );

  if ( XLAL_IS_REAL8_FAIL_NAN( nromega ) || XLAL_IS_REAL8_FAIL_NAN( nromegaDot ) )
  {
    XLALDestroyREAL8Vector( q1LM );
    XLALDestroyREAL8Vector( q2LM );
    XLALDestroyREAL8Vector( q3LM );
    XLALDestroyREAL8Vector( timeVec );
    XLAL_ERROR( XLAL_EFUNC );
  }

  gsl_vector_set( omegaVec, 0, nromega - omega );
  gsl_vector_set( omegaVec, 1, nromegaDot - omegaDot );

  /* And now solve for the b coefficients */
  gsl_linalg_LU_decomp( pMatrix, perm2, &signum );
  gsl_linalg_LU_solve( pMatrix, perm2, omegaVec, bCoeff );

  /* We can now populate the coefficients structure */
  coeffs->a1 = gsl_vector_get( aCoeff, 0 );
  coeffs->a2 = gsl_vector_get( aCoeff, 1 );
  coeffs->a3 = gsl_vector_get( aCoeff, 2 );
  coeffs->b1 = gsl_vector_get( bCoeff, 0 );
  coeffs->b2 = gsl_vector_get( bCoeff, 1 );

  /* Free memory and exit */
  gsl_matrix_free( qMatrix );
  gsl_vector_free( amps );
  gsl_vector_free( aCoeff );
  gsl_permutation_free( perm1 );

  gsl_matrix_free( pMatrix );
  gsl_vector_free( omegaVec );
  gsl_vector_free( bCoeff );
  gsl_permutation_free( perm2 );

  gsl_spline_free( spline );
  gsl_interp_accel_free( acc );

  XLALDestroyREAL8Vector( q1LM );
  XLALDestroyREAL8Vector( q2LM );
  XLALDestroyREAL8Vector( q3LM );
  XLALDestroyREAL8Vector( timeVec );

  return XLAL_SUCCESS;
}

/* ------------------------------------------------
 *          Spin (SEOBNRv1 and SEOBNRv2)
 * ------------------------------------------------*/

/**
 * The time difference between the orbital peak and the peak amplitude
 * of the mode in question (currently only 2,2 implemented ).
 * Eq. 33 of Taracchini et al. PRD 86, 024011 (2012).
 */
UNUSED static inline REAL8 XLALSimIMREOBGetNRSpinPeakDeltaT( 
                 INT4 l,           /**<< Mode l */
                 INT4 m,           /**<< Mode m */
                 REAL8 UNUSED eta, /**<< Symmetric mass ratio */
                 REAL8 a           /**<< Dimensionless spin */
                 )
{

  switch ( l )
  {
    case 2:
      switch ( m )
      {
        case 2:
          /* DeltaT22 defined here is a minus sign different from Eq. (33) of Taracchini et al. */
          if ( a <= 0.0 )
          {
            return 2.5;
          }
          else
          {
            return (2.5 + 1.77*a*a*a*a/(0.43655*0.43655*0.43655*0.43655)/(1.0-2.0*eta)/(1.0-2.0*eta)/(1.0-2.0*eta)/(1.0-2.0*eta));
          }
          break;
        default:
          XLAL_ERROR_REAL8( XLAL_EINVAL );
      }
      break;
    default:
      XLAL_ERROR_REAL8( XLAL_EINVAL );
  }

  /* We should never get here, but I expect a compiler whinge without it... */
  XLALPrintError( "XLAL Error %s - We should never get here!!\n", __func__ );
  XLAL_ERROR_REAL8( XLAL_EINVAL );
}

/* FIXME: Add XLALSimIMREOB to these function names */

/**
 * Peak amplitude predicted by fitting NR results (currently only 2,2 available).
 * Tables IV and V and Eq. 42 of Taracchini et al. PRD 86, 024011 (2012).
 */
UNUSED static inline REAL8 GetNRSpinPeakAmplitude( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 UNUSED a )
{
  /* Fit for HOMs missing */
  return 1.3547468629743946*eta + 0.9187885481024214*eta*eta;
}

/**
 * Peak amplitude curvature predicted by fitting NR results (currently only 2,2 available).
 * Tables IV and V and Eq. 42 of Taracchini et al. PRD 86, 024011 (2012).
 */
UNUSED static inline REAL8 GetNRSpinPeakADDot( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 UNUSED a )
{
  /* Fit for HOMs missing */
  return eta*(-0.0024971911410897156 + (-0.006128515435641139 + 0.01732656*a/(2.0-4.0*eta))*eta);
}

/**
 * Peak frequency predicted by fitting NR results (currently only 2,2 available).
 * Tables IV and V and Eq. 42 of Taracchini et al. PRD 86, 024011 (2012).
 */
UNUSED static inline REAL8 GetNRSpinPeakOmega( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 a )
{
  /* Fit for HOMs missing */
  return 0.27581190323955274 + 0.19347381066059993*eta
       - 0.08898338208573725*log(1.0 - a/(1.0-2.0*eta))
       + eta*eta*(1.78832*(0.2690779744133912 + a/(2.0-4.0*eta))*(1.2056469070395925
       + a/(2.0-4.0*eta)) + 1.423734113371796*log(1.0 - a/(1.0-2.0*eta)));
}

/**
 * Peak frequency slope predicted by fitting NR results (currently only 2,2 available).
 * Tables IV and V and Eq. 42 of Taracchini et al. PRD 86, 024011 (2012).
 */
UNUSED static inline REAL8 GetNRSpinPeakOmegaDot( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 UNUSED a )
{
  /* Fit for HOMs missing */
  return 0.006075014646800278 + 0.012040017219351778*eta
       + (0.0007353536801336875 + 0.0015592659912461832*a/(1.0-2.0*eta))*log(1.0-a/(1.0-2.0*eta))
       + eta*eta*(0.03575969677378844 + (-0.011765658882139 - 0.02494825585993893*a/(1.0-2.0*eta))
       * log(1.0 - a/(1.0-2.0*eta)));
}

/**
 * The time difference between the orbital peak and the peak amplitude
 * of the mode in question (currently only 2,2 implemented ).
 */
UNUSED static inline REAL8 XLALSimIMREOBGetNRSpinPeakDeltaTv2(
                 INT4 UNUSED l,    /**<< Mode l */
                 INT4 UNUSED m,    /**<< Mode m */
                 REAL8 UNUSED m1,  /**<< mass 1 */
                 REAL8 UNUSED m2,  /**<< mass 22 */
                 REAL8 chi1,       /**<< Dimensionless spin1 */
                 REAL8 chi2        /**<< Dimensionless spin2 */
                 )
{
  REAL8 chi, chichi;
  REAL8 eta = m1*m2 / ((m1+m2)*(m1+m2));
  chi    = (chi1+chi2)/2. + (chi1-chi2)/2.*((m1-m2)/(m1+m2))/(1-2.*eta);
    
  chichi = chi*chi;
  if ( chi > 0.8 )
  {
    return (0.75*eta*chi + sqrt(1.-4.*eta)) * (57.1755-48.0564*chi);
  }
  else if ( chi > 0.0 )
  {
    return (0.75*eta*chi + sqrt(1.-4.*eta)) * (2.5+10.*chichi+24.*chichi*chichi);
  }
  else
  {
    return 2.5 + (1.+2.5*chi) * (-2.5+2.5*sqrt(1.-4.*eta));
  }
}

/**
 * Peak frequency predicted by fitting NR results (currently only 2,2 available).
 * Take from unpublished SEOBNRv2 results.
 */
UNUSED static inline REAL8 GetNRSpinPeakOmegav2( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 a )
{
  REAL8 chi = a/(1.0 - 2.0*eta);
  REAL8 eta2= eta*eta;
  if ( eta > 50./51./51. )
  {
    return 0.43747541927878864 + (-0.10933208665273314 - 0.007325831113333813*chi)*log(
            4.500844771420863 - 9.681916048928946*eta +
            chi*(-4.254886879579986 + 11.513558950322647*eta));
  }
  else
  {
    return 1.5609526077704716 - 122.25721149839733 * eta +
           3586.2192688666914 * eta2 - 13869.506144441548 * eta*eta2 +
           (eta - 0.25) * (1651.5823693445805 * (-0.01922337588094282 + eta) *
           (-0.01922337536857659 + eta) + 66.87492814925524 * chi *
           (0.0003695381704106058 - 0.03844675124951941 * eta + eta2)) *
           log(5600.67382718678 - 5555.824895398546 * chi) +
           (-1412.8186461833657 + 67.66455403259023 * chi) * (eta - 0.001) *
           (0.0003695381704106056 - 0.038446751249519406 * eta + eta2) *
           log(0.5680439481719505 - 0.36813967358200156 * chi) +
           0.012328326527732041 * log(4.500844771420863 - 9.681916048928946 * eta + 
           chi * (-4.254886879579986 + 11.513558950322647 * eta)) +
           0.0008260634258180991 * chi * log(4.500844771420863 -9.681916048928946 * eta +
           chi * (-4.254886879579986 + 11.513558950322647 * eta)) -
           12.6575493872956 * eta * log(4.500844771420863 -
           9.681916048928946 * eta + chi * (-4.254886879579986 + 11.513558950322647 * eta)) -
           0.8481231078533651 * chi * eta * log(4.500844771420863 - 
           9.681916048928946 * eta + chi * (-4.254886879579986 + 11.513558950322647 * eta)) +
           329.2228595635586 * eta2 * log(4.500844771420863 - 9.681916048928946 * eta +
           chi * (-4.254886879579986 + 11.513558950322647 * eta)) +
           22.05968203526603 * chi * eta2 * log(4.500844771420863 -9.681916048928946 * eta +
           chi * (-4.254886879579986 + 11.513558950322647 * eta));
  }
}

/**
 * Peak frequency slope predicted by fitting NR results (currently only 2,2 available).
 * Take from unpublished SEOBNRv2 results.
 */
UNUSED static inline REAL8 GetNRSpinPeakOmegaDotv2( INT4 UNUSED l, INT4 UNUSED m, REAL8 UNUSED eta, REAL8 UNUSED a )
{
  REAL8 chi = a/(1.0-2.0*eta);
  REAL8 eta2= eta*eta;
  /* Fit for HOMs missing */
  if (chi < 0.8 )
  {
    return -0.07086074186161867 * chi * (-0.26367236731979804 + eta) *
           (-0.0010019969893089581 + eta) + 0.2893863668183948 *
           (-0.16845695144529893 + eta) * (0.23032241797163952 + eta) +
           (0.004086861548547749 - 0.06538978477676398 * eta2 +
           chi * (0.0006334026884930817 - 0.010134443015889307 * eta2)) *
           log(68.47466578101876 - 58.30148755701496 * chi);
  }
  else
  {
    return -0.10069512275335238 * (-0.46107388514323044 + eta) *
           (0.2832795481380979 + eta) + 0.2614619716504706 * chi *
           (-0.24838163750494138 + eta) * (0.320112993649413 + eta) +
           chi * chi * (0.010000160002560042 - 0.16000256004096067 * eta2);
  }
}

/**
 * Function to interpolate known amplitude NQC coeffcients of spin terms,
 * namely a3s, a4 and a5.
 * The a3s, a4 and a5 values were calculated for
 * 11 mass ratios q=1,1.5,2,3,4,5,6,10,20,50 and 100, and
 * 19 spin (\f$\chi\f$ defined in Taracchini et al. PRD 86, 024011 (2012)) values
 * chi = -1, -0.9, -0.8, ......, 0.3, 0.4, 0.5, 0.55, 0.6, 0.65.
 * The calculation was done by Andrea Taracchini using a C++ code of the UMaryland group.
 * In principle, these numbers can be automatically calculated iteratively by the LAL code.
 * However, since such calcualtion increase the cost of each waveform generation by
 * about an order of magnitude, we prepare these numbers in advance reduce cost.
 * These number can be verified by confirming that
 * the peak amplitude and frequency agree well with the NR-fits predicted values,
 * and to get exact NR-fits predicted values, corrections on these numbers are ~1%.
 */
UNUSED static int XLALSimIMRGetEOBCalibratedSpinNQC( EOBNonQCCoeffs * restrict coeffs,                                   INT4 UNUSED l,                                   INT4 UNUSED m,                                   REAL8 eta,                                   REAL8 a )
{
  const unsigned int nsqdim = 100;
  const unsigned int   qdim = 50;
  const unsigned int   adim = 69;
  UINT4 i;
  /* REAL8 eta2 = eta*eta;*/
  REAL8 a3slist[qdim], a4list[qdim], a5list[qdim];

  memset( coeffs, 0, sizeof( *coeffs ) );
  const double nsetalist[100] = {0.0025, 0.005, 0.0075, 0.01, 0.0125, 0.015, 0.0175, 0.02, 0.0225, 0.025, 0.0275, 0.03, 0.0325, 0.035, 0.0375, 0.04, 0.0425, 0.045, 0.0475, 0.05, 0.0525, 0.055, 0.0575, 0.06, 0.0625, 0.065, 0.0675, 0.07, 0.0725, 0.075, 0.0775, 0.08, 0.0825, 0.085, 0.0875, 0.09, 0.0925, 0.095, 0.0975, 0.1, 0.1025, 0.105, 0.1075, 0.11, 0.1125, 0.115, 0.1175, 0.12, 0.1225, 0.125, 0.1275, 0.13, 0.1325, 0.135, 0.1375, 0.14, 0.1425, 0.145, 0.1475, 0.15, 0.1525, 0.155, 0.1575, 0.16, 0.1625, 0.165, 0.1675, 0.17, 0.1725, 0.175, 0.1775, 0.18, 0.1825, 0.185, 0.1875, 0.19, 0.1925, 0.195, 0.1975, 0.2, 0.2025, 0.205, 0.2075, 0.21, 0.2125, 0.215, 0.2175, 0.22, 0.2225, 0.225, 0.2275, 0.23, 0.2325, 0.235, 0.2375, 0.24, 0.2425, 0.245, 0.2475, 0.25};
  const double etalist[50] = {0.005, 0.01, 0.015, 0.02, 0.025, 0.03, 0.035, 0.04, 0.045, 0.05, 0.055, 0.06, 0.065, 0.07, 0.075, 0.08, 0.085, 0.09, 0.095, 0.1, 0.105, 0.11, 0.115, 0.12, 0.125, 0.13, 0.135, 0.14, 0.145, 0.15, 0.155, 0.16, 0.165, 0.17, 0.175, 0.18, 0.185, 0.19, 0.195, 0.2, 0.205, 0.21, 0.215, 0.22, 0.225, 0.23, 0.235, 0.24, 0.245, 0.25};
  const double alist[69]   = {-1., -0.975, -0.95, -0.925, -0.9, -0.875, -0.85, -0.825, -0.8, -0.775, -0.75, -0.725, -0.7, -0.675, -0.65, -0.625, -0.6, -0.575, -0.55, -0.525, -0.5, -0.475, -0.45, -0.425, -0.4, -0.375, -0.35, -0.325, -0.3, -0.275, -0.25, -0.225, -0.2, -0.175, -0.15, -0.125, -0.1, -0.075, -0.05, -0.025, 0., 0.025, 0.05, 0.075, 0.1, 0.125, 0.15, 0.175, 0.2, 0.225, 0.25, 0.275, 0.3, 0.325, 0.35, 0.375, 0.4, 0.425, 0.45, 0.475, 0.5, 0.525, 0.55, 0.575, 0.6, 0.625, 0.65, 0.675, 0.7};

  const double a1list[100] = {-21.648106754277453, -20.47585498319721, -19.530321799953853, -18.760004570030972, -18.079213572570588, -17.419521042795658, -16.859312290615286, -16.28756533894047, -15.772560753656803, -15.303367029222859, -14.847798623544204, -14.404050939670224, -13.976897526988164, -13.578815297784823, -13.200763216702892, -12.841976980076495, -12.501692284239752, -12.179144825526786, -11.865421415437213, -11.561721630442458, -11.267369305715642, -10.980678895252732, -10.699964853049693, -10.42829543384451, -10.167524207712544, -9.914674799750443, -9.669372681081281, -9.43124332282813, -9.199912196114067, -8.97500477206216, -8.756146521795488, -8.542962916437121, -8.335079427110134, -8.130867290476582, -7.9313984345769715, -7.736518055359128, -7.545988962963793, -7.35957396753171, -7.177035879203623, -6.998068318971551, -6.8224288840577945, -6.649961333309703, -6.480441962265538, -6.313647066463564, -6.149352941442041, -5.9873358827392344, -5.827372185893404, -5.669238146442815, -5.512513869477759, -5.347444298469859, -5.183197664196798, -5.019955297249104, -4.857898528217308, -4.697208687691938, -4.538067106263523, -4.380655114522592, -4.225154043059675, -4.071745222465302, -3.9206099833299994, -3.771929656244297, -3.6258855717987246, -3.4826590605838113, -3.342431453190086, -3.2053840802080775, -3.0798495456033685, -2.957143501994997, -2.837141996613968, -2.7197210766912865, -2.6047567894579564, -2.4921251821449832, -2.381702301983373, -2.273364196204128, -2.1669869120382543, -2.0624464967167566, -1.9596189974706397, -1.8495581080328225, -1.7409749049276515, -1.6340876818580403, -1.5291147325269023, -1.4262743506371505, -1.3257848298916983, -1.2278644639934593, -1.1327315466453465, -1.0406043715502744, -0.9517012324111543, -0.8662404229309005, -0.7844402368124264, -0.7065189677586451, -0.6326949094724702, -0.5631863556568149, -0.4982116000145925, -0.4379889362487165, -0.3827366580621, -0.3326730591576565, -0.28801643323829923, -0.24898507400694198, -0.21579727516649713, -0.18867133041987855, -0.1678255334699995, -0.15347817801977337};

  const double a2list[100] = {178.73204288078207, 167.7345170263427, 158.85457776976878, 151.63702661020528, 145.25886103554546, 139.08071349079492, 133.85300186061994, 128.49833153024582, 123.69581397508206, 119.34304449990263, 115.1225036726121, 111.02138715464824, 107.08437843884623, 103.42908442981195, 99.96941370234742, 96.69762600347904, 93.6059810802332, 90.68673867963636, 87.86042720725196, 85.13659365364732, 82.50827361644501, 79.95961751850369, 77.47477578268207, 75.08044751075573, 72.79299640785891, 70.58542621544011, 68.45392194344085, 66.39466860180258, 64.40385120046682, 62.47765474937501, 60.61226425846867, 58.80386473768926, 57.04864119697828, 55.32723154023535, 53.65375061411054, 52.0271070701423, 50.44518833065014, 48.905881817953585, 47.407074954372156, 45.94482445612376, 44.51687981462722, 43.123270309615144, 41.76224017946817, 40.43203366256695, 39.13089499729211, 37.85706842202429, 36.60879817514412, 35.384328495032264, 34.180608559988464, 32.932965816541305, 31.701640999519693, 30.48755199799916, 29.291616701055233, 28.11475299776345, 26.95787877719934, 25.821911928438432, 24.70777034055626, 23.616371902628366, 22.54863450373026, 21.50547603293748, 20.487814379325563, 19.496567431970035, 18.53265307994643, 17.596989212330282, 16.733055464585682, 15.895477726977264, 15.083579856295431, 14.296685709330593, 13.53411914287315, 12.795204013713512, 12.079264178642092, 11.385623494449275, 10.71360581792548, 10.06253500586111, 9.431734915046569, 8.755551062926262, 8.098379005319565, 7.462063310457031, 6.848448546569211, 6.259379281886656, 5.69670008463992, 5.162255523059555, 4.657890165376111, 4.1854485798201475, 3.7467753346222055, 3.3437149980128416, 2.978112138222609, 2.651811323482059, 2.366657122021744, 2.124494102072216, 1.9271668318640272, 1.77651987962773, 1.674397813593876, 1.6226452019930173, 1.6231066130557064, 1.6776266150124943, 1.7880497760939342, 1.956220664530578, 2.1839838485529777, 2.4731838963916855};

  const double a3list[100] = {-198.6684740510964, -185.71560983427335, -175.26102024642407, -166.7798147654891, -159.28986302859136, -152.03810090459336, -145.91734205946045, -139.6355596843493, -134.01653939524905, -128.93945143613453, -124.02034829337536, -119.24645127356649, -114.67037356959001, -110.43118685036107, -106.4264567068743, -102.6466423946065, -99.08220316903453, -95.72359828563523, -92.4798228915885, -89.36096908139594, -86.35837723133453, -83.45329697039512, -80.62697792756852, -77.90968987612793, -75.32016588735331, -72.82709457524146, -70.42579806564999, -68.11159848443651, -65.87981795745866, -63.72577861057402, -61.64480256964021, -59.63221196051487, -57.683328909055604, -55.7731074201788, -53.92035933426217, -52.12397499502319, -50.38150684315194, -48.69050731933857, -47.04852886427319, -45.45070004394427, -43.89439067703587, -42.37998904664536, -40.90552002179141, -39.46900847149271, -38.068479264767944, -36.70195727063581, -35.36746735811498, -34.063034396224154, -32.78536874375959, -31.468633548362497, -30.173781283368523, -28.901550623295833, -27.652680242662605, -26.427908815987003, -25.227975017787198, -24.053617522581362, -22.905575004887663, -21.78458613922428, -20.691389600109368, -19.6267240620611, -18.59132819959765, -17.585940687237187, -16.61130019949788, -15.668145410897901, -14.794309398041404, -13.950185357101319, -13.135122694543167, -12.348470816832473, -11.589579130434757, -10.857797041815543, -10.152473957440362, -9.47295928377472, -8.818602427284151, -8.188752794434174, -7.582759791690314, -6.932160232642741, -6.304213548518645, -5.700899806714059, -5.124199074625014, -4.576091419647542, -4.0585569091776765, -3.573575610611449, -3.123127591344892, -2.709192918774041, -2.3337516602949204, -1.9987838833035663, -1.706269655196011, -1.458189043368287, -1.2565221152164259, -1.1032489381364605, -1.0003495795244224, -0.9498041067763443, -0.953592587288258, -1.0136950884561957, -1.1320916776761898, -1.31076242234427, -1.5516873898564725, -1.8568466476088277, -2.2282202629973678, -2.667788303418125};

  const double b1list[100] = {-0.5693500504085347, -0.576434151837257, -0.5827940588889807, -0.5875005969333106, -0.5915255507494274, -0.5970658827548452, -0.6057016775611604, -0.6053160995270499, -0.6070988602490128, -0.6110941958474475, -0.6140262971912503, -0.6172989788502661, -0.6206089494421659, -0.6234488672149939, -0.6258813681301192, -0.6277498501118401, -0.628897711084455, -0.6291683489722621, -0.6308371388180571, -0.6331613723912005, -0.6359581432020665, -0.6393457874435003, -0.6434426413083475, -0.6478966252652467, -0.6524003236766366, -0.6571425380351175, -0.6620695856777987, -0.6671277839417898, -0.6722634501642004, -0.6774229016821397, -0.6825524558327174, -0.6875984299530429, -0.6925071413802255, -0.6961049747777461, -0.6996298196832709, -0.7032241904115, -0.7069570372346382, -0.7108973104248908, -0.7151139602544623, -0.720064149835206, -0.7258460785616149, -0.7320745022572427, -0.7387427060855165, -0.7458439752098636, -0.7533715947937112, -0.7613188500004865, -0.7696790259936167, -0.778445407936529, -0.7876035918776259, -0.7967733093076456, -0.8063159350249182, -0.8162406283658405, -0.8265565486668099, -0.8372728552642233, -0.8483987074944775, -0.8599432646939698, -0.871915686199097, -0.884325131346256, -0.8971807594718443, -0.9104917299122585, -0.9242672020038958, -0.938516335083153, -0.9532482884864274, -0.9684722215501159, -0.9837886712042127, -0.9996512322126136, -1.0160843677317775, -1.0331125409181625, -1.0507602149282274, -1.0690518529184303, -1.08801191804523, -1.1076648734650847, -1.1280351823344532, -1.1491473078097938, -1.171025713047565, -1.1939017528688474, -1.2175927015073384, -1.2421149961497486, -1.2674850739827876, -1.2937193721931661, -1.320834327967594, -1.3488463784927816, -1.377771960955439, -1.4076275125422764, -1.4384294704400042, -1.4701942718353325, -1.5029383539149717, -1.5366781538656316, -1.5714301088740226, -1.6072106561268549, -1.6440362328108387, -1.6819232761126843, -1.7208882232191016, -1.7609475113168012, -1.802117577592493, -1.8444148592328868, -1.8878557934246936, -1.9324568173546235, -1.9782343682093864, -2.0252048831756926};

  const double b2list[100] = {1.6514745488753086, 1.6733593678301482, 1.687838328174986, 1.7031979370575185, 1.712831020475929, 1.7266279186283089, 1.7581796869631672, 1.7499867318965905, 1.7518398412177276, 1.7634468469918447, 1.770740685014047, 1.779639998248617, 1.788893228893931, 1.7964389585725973, 1.8024779675983216, 1.8063408969988246, 1.8073583878018264, 1.8048610810350476, 1.8077536017385523, 1.8130592620946404, 1.8200051503317694, 1.8290042591854243, 1.8404695813910905, 1.8531136802761718, 1.8658884601302625, 1.8795206835759333, 1.8938451458175296, 1.908696642059398, 1.923909967505884, 1.9393199173613336, 1.9547612868300928, 1.9700688711165077, 1.9850774654249244, 1.9960816492286722, 2.0069990026439246, 2.0182845185376195, 2.0301606467081585, 2.042849836953944, 2.056574539073377, 2.0726094682881886, 2.0912563552368337, 2.111506538675078, 2.1333773752936596, 2.1568862217833127, 2.1820504348347747, 2.208887371138781, 2.2374143873860683, 2.2676488402673725, 2.299578947658098, 2.3318064402049257, 2.3657435940219895, 2.4014679228653333, 2.439056940491002, 2.478588160655038, 2.520139097113486, 2.56378726362239, 2.609610173937794, 2.657685341815741, 2.708090281012276, 2.760902505283443, 2.8161995283852854, 2.874058864073847, 2.9345580261051722, 2.9977745282353045, 3.0615596376827443, 3.128412230361185, 3.1984931979224718, 3.271963432018449, 3.3489838243009618, 3.429715266421855, 3.5143186500329735, 3.602954866786163, 3.6957848083332676, 3.792969366326133, 3.894669432416603, 4.002044979667201, 4.114256382857298, 4.23142577617022, 4.353675293789295, 4.48112706989785, 4.613903238679211, 4.752125934316707, 4.895917290993665, 5.04539944289341, 5.200694524199272, 5.361924669094577, 5.529212011762653, 5.702678686386827, 5.882446827150425, 6.068638568236777, 6.261376043829206, 6.460781388111044, 6.666976735265615, 6.880084219476247, 7.100225974926268, 7.327524135799001, 7.5621008362777795, 7.804078210545929, 8.053578392786775, 8.310723517183646};

  const double a3stab[50][69] = {
{1298.87913, 1261.29637, 1227.72908, 1197.25947, 1168.96973, 1141.94207, 1115.2587, 1088.00181, 1059.25362, 1022.90618, 985.307903, 947.617046, 910.991876, 881.346609, 853.181173, 825.751447, 798.313312, 767.87765, 736.843339, 705.36426, 673.594293, 642.125047, 610.497581, 578.690684, 546.683144, 514.497622, 482.051486, 449.305976, 416.222332, 382.495195, 348.459044, 314.181759, 279.731222, 245.411071, 210.959123, 176.348956, 141.554145, 106.628243, 71.43286, 35.9095836, 0., -36.1519021, -72.887899, -110.347365, -148.669675, -186.733245, -226.44279, -268.442067, -313.374835, -358.91977, -409.871742, -468.06054, -535.315953, -602.768787, -687.227409, -794.8012, -931.599543, -1073.78457, -1269.39181, -1536.50955, -1893.22609, -2326.65745, -2947.80866, -3741.68954, -4974.29064, -6365.10282, -9538.63496, -15643.1414, -25826.8766}, 
{1196.32002, 1167.06104, 1137.27475, 1107.12301, 1076.76769, 1046.37066, 1016.09379, 986.098937, 956.547978, 928.8469, 901.415801, 873.918898, 846.020409, 814.568078, 783.16919, 752.614552, 723.694973, 702.877684, 683.006503, 662.601671, 640.183429, 607.718815, 572.902552, 536.876163, 500.78117, 469.351999, 438.700105, 408.529845, 378.545579, 347.703945, 316.75611, 285.705522, 254.555628, 223.39562, 192.108901, 160.664621, 129.031928, 97.2453174, 65.1824526, 32.7863434, 0., -33.0547348, -66.6852171, -101.01997, -136.187517, -170.971669, -207.383546, -246.089556, -287.756108, -330.313665, -378.258958, -433.352771, -497.35589, -562.191692, -643.393334, -746.656564, -877.67713, -1013.79966, -1200.41146, -1454.54875, -1793.24772, -2204.40576, -2792.4755, -3542.23306, -4707.36116, -6024.93464, -9029.55323, -14806.4354, -24440.7998}, 
{1111.34089, 1088.04208, 1061.93086, 1033.65502, 1003.86233, 973.200551, 942.317471, 911.86086, 882.478492, 858.393797, 835.248631, 812.260507, 788.646935, 759.48951, 729.796029, 700.438372, 672.288418, 651.387883, 631.370875, 611.041341, 589.203225, 559.502697, 527.964588, 495.455951, 462.843844, 433.627925, 404.989603, 376.74289, 348.701797, 320.308767, 291.898012, 263.432172, 234.873889, 206.130982, 177.242843, 148.194039, 118.969142, 89.6533759, 60.0903922, 30.2244979, 0., -30.4005164, -61.3663335, -93.0484555, -125.597886, -157.893074, -191.866601, -228.178495, -267.488782, -307.871972, -353.607815, -406.390546, -467.914398, -530.2754, -608.605274, -708.437534, -835.305696, -967.651586, -1148.93709, -1395.53239, -1723.80769, -2122.58454, -2690.87904, -3414.09589, -4533.73512, -5812.62405, -8672.18555, -14119.1936, -23160.4223}, 
{1039.53251, 1020.2822, 997.603313, 972.198968, 944.772298, 916.02643, 886.664491, 857.389611, 828.904917, 804.925294, 781.93741, 759.439688, 736.930555, 712.636669, 687.836926, 662.538457, 636.748394, 609.788544, 582.625489, 555.540492, 528.814811, 504.169828, 479.870634, 455.622441, 431.13046, 404.996091, 378.469884, 351.698574, 324.8289, 298.520437, 272.201947, 245.815032, 219.301294, 192.43116, 165.385874, 138.17551, 110.810137, 83.4601202, 55.9111218, 28.1090968, 0., -28.1232207, -56.8004007, -86.2243824, -116.588008, -146.981092, -179.140716, -213.700934, -251.2958, -290.060701, -334.127824, -385.130693, -444.702827, -504.716717, -580.471327, -677.504592, -801.354447, -931.385611, -1109.77852, -1352.54038, -1675.67843, -2068.96254, -2627.11195, -3337.00139, -4427.97475, -5693.23265, -8428.0796, -13552.2941, -21985.6547}, 
{976.485655, 959.824144, 940.197989, 918.098323, 894.016279, 868.442993, 841.869596, 814.787222, 787.687004, 761.819814, 736.613152, 712.254257, 688.930369, 670.53532, 652.067116, 632.23036, 609.729651, 574.459963, 537.459377, 500.956344, 467.179318, 448.413342, 432.807639, 418.568025, 403.900314, 381.127188, 356.690849, 331.150363, 305.064801, 280.549729, 255.985115, 231.307428, 206.453135, 181.110154, 155.562924, 129.847331, 103.999264, 78.2732522, 52.3990841, 26.3251899, 0., -26.1568213, -52.856571, -80.3393122, -108.845108, -137.719354, -168.454647, -201.64892, -237.900101, -275.345863, -318.028498, -367.530041, -425.432527, -483.212445, -556.599593, -651.218222, -772.692584, -901.046982, -1077.7456, -1318.65266, -1639.63241, -2031.30849, -2585.26688, -3290.67293, -4364.64226, -5631.82205, -8258.783, -13076.6149, -20916.4077}, 
{917.791081, 902.710621, 885.620781, 866.696557, 846.112943, 824.044936, 800.66753, 776.155722, 750.684506, 722.455778, 694.406874, 667.502029, 642.705478, 629.711227, 617.261835, 602.829629, 583.886937, 541.782439, 496.561572, 452.146121, 412.457876, 398.926373, 390.962551, 385.485097, 379.412698, 359.691907, 337.202401, 312.851722, 287.547412, 264.607418, 241.564715, 218.362682, 194.944699, 170.981969, 146.79892, 122.447804, 97.9808711, 73.7004736, 49.308722, 24.7578271, 0., -24.435292, -49.4039968, -75.1848062, -102.056412, -129.591492, -159.057156, -191.014502, -226.024627, -262.193469, -303.519346, -351.545419, -407.814847, -463.459392, -534.598171, -626.938905, -746.189313, -872.680952, -1047.64817, -1286.94915, -1606.44207, -1997.39115, -2549.4365, -3254.83387, -4318.29988, -5593.45384, -8125.84343, -12663.0342, -19952.5918}, 
{859.039559, 844.98436, 829.777581, 813.337142, 795.580959, 776.426953, 755.793041, 733.597141, 709.757173, 680.211609, 650.449591, 621.980817, 596.314984, 586.690157, 578.196318, 567.651816, 551.875002, 508.136269, 460.621105, 413.967044, 372.811617, 362.402057, 358.522317, 357.566049, 355.926907, 338.360939, 317.554443, 294.556114, 270.414647, 248.904279, 227.257944, 205.434118, 183.391278, 160.860605, 138.11879, 115.215226, 92.1993096, 69.3494866, 46.3944783, 23.2920582, 0., -22.8926065, -46.3118306, -70.5524255, -95.9091441, -122.081137, -150.197001, -180.789729, -214.392317, -249.069531, -288.809882, -335.133657, -389.561139, -443.154361, -512.075161, -602.027124, -718.713836, -842.332775, -1014.29608, -1250.50977, -1566.87987, -1954.97924, -2503.71346, -3209.20757, -4263.50982, -5543.18962, -7990.80852, -12282.43, -19094.1177}, 
{795.821854, 782.688084, 768.57428, 753.36355, 736.939001, 719.18374, 699.980874, 679.213512, 656.764759, 628.465728, 599.872319, 572.488437, 547.817986, 537.997872, 529.6458, 520.012472, 506.348594, 469.901748, 430.32701, 391.276333, 356.401671, 345.533527, 339.673884, 336.003275, 331.702235, 314.804975, 295.296878, 274.017005, 251.804419, 231.651088, 211.382003, 190.97506, 170.40816, 149.560065, 128.547461, 107.387899, 86.098929, 64.8279929, 43.4107958, 21.8129331, 0., -21.4627385, -43.4492248, -66.2337315, -90.0905311, -114.671921, -141.12294, -169.966649, -201.726112, -234.440059, -272.109617, -316.251583, -368.382751, -419.994158, -486.638661, -573.843361, -687.135356, -806.047703, -972.49916, -1202.41444, -1511.71827, -1891.84153, -2432.19041, -3133.5174, -4174.83432, -5446.09099, -7815.22592, -11905.6805, -18340.896}, 
{723.728735, 711.864516, 697.916767, 682.119252, 664.705738, 645.909991, 625.965777, 605.106863, 583.567013, 560.596558, 537.806075, 515.822704, 495.273586, 480.160139, 466.385516, 453.227147, 439.962463, 423.459176, 406.36832, 388.931212, 371.389169, 355.013918, 338.6042, 321.989168, 304.997976, 286.694707, 267.979608, 248.987859, 229.854639, 211.058618, 192.254089, 173.438833, 154.610635, 135.894352, 117.109862, 98.2041202, 79.1240795, 59.7436944, 40.1121174, 20.2055015, 0., -20.0796619, -40.6853319, -62.0202854, -84.2877981, -106.847476, -131.083733, -157.53731, -186.748953, -216.771064, -251.628063, -292.856026, -341.991034, -391.675587, -455.896772, -539.748099, -648.323078, -759.870989, -917.067264, -1135.7431, -1431.72971, -1795.74674, -2318.96003, -3007.48672, -4026.8356, -5267.21955, -7560.6433, -11503.6636, -17692.8373}, 
{670.702392, 658.608735, 645.231211, 630.627242, 614.85425, 597.969656, 580.030884, 561.095353, 541.220486, 518.602027, 495.903746, 473.927738, 453.476095, 438.689549, 425.696101, 413.962387, 402.955046, 391.824136, 380.479507, 368.51443, 355.522175, 339.17332, 321.752905, 303.623275, 285.146778, 267.626973, 250.10851, 232.577251, 215.019058, 197.328079, 179.618576, 161.913096, 144.234187, 126.856941, 109.450342, 91.9359207, 74.2352057, 56.1190544, 37.7199375, 19.0196532, 0., -18.9413868, -38.4048807, -58.5750179, -79.636335, -100.928376, -123.818666, -148.82974, -176.48413, -204.795257, -237.798413, -277.019773, -323.985518, -371.643004, -433.528758, -514.600485, -619.815891, -726.601121, -878.458066, -1091.35706, -1381.26842, -1738.68566, -2256.0096, -2944.21016, -3956.54775, -5195.43966, -7424.38522, -11191.6133, -17045.3526}, 
{623.046755, 610.336599, 597.375254, 583.988785, 570.003256, 555.244732, 539.539276, 522.712953, 504.591828, 481.903784, 458.812338, 436.382826, 415.680585, 401.790876, 390.151139, 380.218743, 371.451052, 363.730969, 355.920111, 347.30563, 337.174678, 321.458633, 304.142732, 285.856435, 267.229206, 250.700686, 234.366085, 218.130789, 201.900189, 185.228258, 168.512366, 151.798468, 135.13252, 118.913825, 102.69365, 86.3766111, 69.8673234, 52.8685906, 35.5675654, 17.9495883, 0., -17.9060924, -36.3290217, -55.439354, -75.4076554, -95.5838756, -117.287444, -141.017173, -167.271875, -194.031478, -225.321234, -262.64751, -307.516673, -353.072705, -412.529312, -490.737815, -592.549535, -694.603535, -841.248297, -1048.62004, -1332.855, -1684.25015, -2196.45943, -2885.12536, -3891.89364, -5131.56784, -7300.31693, -10898.9685, -16428.3503}, 
{578.777874, 565.343763, 552.617764, 540.222136, 527.779136, 514.911024, 501.240056, 486.388491, 469.978587, 447.525103, 424.400795, 401.870923, 381.200742, 368.162933, 357.712364, 349.311322, 342.422098, 337.556737, 332.707869, 326.917882, 319.229161, 304.187697, 287.130832, 268.899514, 250.334688, 234.780471, 219.573371, 204.553063, 189.559225, 173.868813, 158.109315, 142.345496, 126.64212, 111.492196, 96.360951, 81.1418536, 65.7283731, 49.7810319, 33.5194242, 16.9301977, 0., -16.9275463, -34.3655585, -52.4701786, -71.3975492, -90.5102836, -111.075464, -133.566646, -158.457381, -183.715399, -213.322409, -248.754293, -291.486936, -334.842038, -391.711338, -466.832391, -564.942754, -661.929092, -802.920208, -1004.19402, -1282.02842, -1626.52282, -2132.4907, -2819.95411, -3819.61549, -5057.44347, -7167.66153, -10607.4294, -15833.9069}, 
{537.705661, 523.592259, 510.97538, 499.323946, 488.10688, 476.793102, 464.851536, 451.751104, 436.960726, 415.185168, 392.563172, 370.469324, 350.278208, 338.111834, 328.698393, 321.5135, 316.032768, 313.155404, 310.363995, 306.564723, 300.663765, 286.44217, 269.9813, 252.237386, 234.16666, 219.614044, 205.491601, 191.600088, 177.74026, 163.024266, 148.21691, 133.394391, 118.632905, 104.472791, 90.3404513, 76.1264269, 61.7212598, 46.7790037, 31.5212837, 15.9332368, 0., -15.974157, -32.4508828, -49.5726937, -67.4821057, -85.5612143, -105.017124, -126.29652, -149.846085, -173.651974, -201.605614, -235.1379, -275.67973, -316.701541, -370.778872, -442.526802, -536.56041, -628.112094, -762.932689, -957.39035, -1227.85323, -1564.25632, -2062.26726, -2745.81353, -3735.46447, -4967.31563, -7018.71577, -10306.7003, -15248.3048}, 
{499.640027, 485.044119, 472.464742, 461.290869, 450.911477, 440.715539, 430.092032, 418.42993, 405.118208, 384.603164, 363.193521, 342.255325, 323.154622, 311.94369, 303.427849, 297.098651, 292.447647, 290.380935, 288.409705, 285.45969, 280.456626, 267.30371, 251.958228, 235.354929, 218.428561, 204.949119, 191.88201, 179.027883, 166.187391, 152.469136, 138.642637, 124.785364, 110.97479, 97.7363464, 84.5203574, 71.2251094, 57.7488886, 43.7851316, 29.5189138, 14.9304613, 0., -15.0143326, -30.5213867, -46.652101, -63.5374142, -80.5902821, -98.9468193, -119.025158, -141.243429, -163.646157, -189.974524, -221.596107, -259.878479, -298.401753, -349.435951, -417.463636, -506.967366, -592.686844, -720.744632, -907.520433, -1169.39395, -1496.20329, -1983.95294, -2659.82073, -3635.19172, -4855.43337, -6845.77638, -9986.48562, -14657.826}, 
{464.390881, 449.661375, 437.102489, 426.119557, 416.117917, 406.502905, 396.679857, 386.054108, 374.030995, 355.498276, 336.185896, 317.306223, 300.071624, 289.964612, 282.219351, 276.34015, 271.831319, 269.087297, 266.366212, 262.816322, 257.585886, 245.853978, 232.325711, 217.737017, 202.823827, 190.533415, 178.505829, 166.592464, 154.644712, 141.977941, 129.193979, 116.358629, 103.537694, 91.1635985, 78.7888757, 66.3326796, 53.7141645, 40.722041, 27.4580845, 13.8936269, 0., -14.0164816, -28.5134619, -43.6136023, -59.439564, -75.4511011, -92.6989449, -111.57092, -132.45485, -153.502899, -178.232816, -207.926688, -243.866605, -279.69321, -327.386613, -391.285482, -475.728482, -555.187646, -675.814927, -853.895645, -1105.71512, -1421.11637, -1895.71161, -2559.09281, -3514.54842, -4716.04577, -6641.14011, -9636.48968, -14048.7528}, 
{431.768135, 417.406059, 404.905261, 393.806663, 383.651192, 373.979771, 364.333324, 354.252775, 343.279049, 327.589687, 311.434349, 295.699313, 281.270856, 272.480712, 265.391519, 259.511373, 254.348369, 249.128454, 243.754732, 237.848156, 231.029682, 221.17463, 210.347843, 198.868528, 187.055894, 176.114645, 165.124294, 154.049849, 142.856319, 131.325202, 119.678423, 107.954395, 96.1915321, 84.6352843, 73.0342128, 61.3439161, 49.5199927, 37.5123573, 25.2845654, 12.794489, 0., -12.9490123, -26.3635003, -40.3623994, -55.0646445, -69.9972857, -86.1078972, -103.752168, -123.285788, -143.027154, -166.184163, -193.927421, -227.427532, -260.326451, -304.334894, -363.634929, -442.40862, -515.148803, -627.602467, -795.827371, -1035.88127, -1337.74821, -1795.70711, -2440.7469, -3369.28571, -4543.40188, -6397.10368, -9246.41689, -13407.3673}, 
{397.382476, 384.161063, 372.406617, 361.789178, 351.978785, 342.645477, 333.459295, 324.090277, 314.208462, 301.130105, 287.820545, 274.891334, 262.954027, 255.327872, 248.833648, 242.999831, 237.354894, 230.264221, 222.884616, 215.20979, 207.233455, 198.951905, 190.355236, 181.436127, 172.187257, 162.452347, 152.432615, 142.180322, 131.747728, 121.278447, 110.696847, 100.018645, 89.2595619, 78.5197505, 67.6967231, 56.7724257, 45.7288044, 34.6183092, 23.3241805, 11.8001626, 0., -11.943738, -24.3276619, -37.2695573, -50.8872099, -64.7777681, -79.7879094, -96.2436743, -114.471103, -132.961668, -154.609806, -180.475383, -211.618268, -241.71635, -282.164266, -336.974677, -410.160241, -476.397825, -580.770198, -739.024339, -966.907224, -1254.16741, -1694.54713, -2321.20376, -3222.44556, -4362.79135, -6146.7668, -8868.89588, -12823.7025}, 
{359.974686, 348.352, 338.125236, 328.975588, 320.58425, 312.632416, 304.801281, 296.772038, 288.225882, 276.655174, 264.805473, 253.233508, 242.496004, 235.582587, 229.643928, 224.263593, 219.02515, 212.45994, 205.624648, 198.523735, 191.16166, 183.591405, 175.749496, 167.620982, 159.190914, 150.290229, 141.119734, 131.726121, 122.156087, 112.539351, 102.806371, 92.9706295, 83.0456113, 73.124859, 63.1097732, 52.9818134, 42.7224396, 32.3549175, 21.8021786, 11.0289603, 0., -11.1220613, -22.6495511, -34.6968934, -47.3785121, -60.3411594, -74.354, -89.7185264, -106.736232, -124.027317, -144.247083, -168.369539, -197.368694, -225.064687, -262.446946, -313.351026, -381.612486, -442.034343, -539.097709, -688.251154, -904.943253, -1178.0799, -1602.7377, -2215.93586, -3095.8159, -4202.28901, -5927.06542, -8553.90562, -12366.5701}, 
{324.284553, 314.363487, 305.717204, 298.046437, 291.051921, 284.434388, 277.894571, 271.133205, 263.851022, 253.758508, 243.342743, 233.100562, 223.528797, 217.260142, 211.801225, 206.794534, 201.882558, 195.779693, 189.427757, 182.840476, 176.031575, 169.10102, 161.9418, 154.533147, 146.854291, 138.731414, 130.358013, 121.774536, 113.021431, 104.2128, 95.2859787, 86.2519542, 77.1217145, 67.9818174, 58.7374528, 49.3693806, 39.8583605, 30.1989457, 20.3525856, 10.2945228, 0., -10.342512, -21.0562901, -32.2513829, -44.0378392, -56.1058801, -69.1533129, -83.4581171, -99.2982726, -115.426321, -134.255855, -156.675029, -183.572, -208.924082, -243.294606, -290.336061, -353.700939, -408.371479, -498.138522, -638.12266, -843.444484, -1102.14389, -1510.58355, -2109.2871, -2967.15339, -4038.77567, -5707.59445, -8249.83842, -11941.7363}, 
{290.624535, 282.418333, 275.336854, 269.105352, 263.449083, 258.093303, 252.763265, 247.184226, 241.08144, 232.406317, 223.367496, 214.39977, 205.937932, 200.246338, 195.198394, 190.497067, 185.845325, 180.150535, 174.229506, 168.103447, 161.793567, 155.43567, 148.890531, 142.133519, 135.140006, 127.739783, 120.11203, 112.290346, 104.308333, 96.2624774, 88.09834, 79.8243672, 71.4490055, 63.0513927, 54.5410077, 45.8980203, 37.1026006, 28.1249184, 18.959144, 9.58944784, 0., -9.60482061, -19.549136, -29.9368594, -40.8719042, -52.0789138, -64.1927792, -77.4691216, -92.1635624, -107.160693, -124.635576, -145.392245, -170.234732, -193.315202, -224.750304, -268.004818, -326.543524, -375.558951, -458.097032, -588.931449, -782.835883, -1027.0032, -1418.94953, -2002.20341, -2837.66181, -3874.62585, -5491.41154, -7959.08117, -11548.697}, 
{259.307093, 252.739349, 247.138517, 242.255957, 237.843023, 233.651073, 229.431464, 224.935552, 219.914693, 212.564812, 204.814872, 197.038402, 189.608933, 184.426978, 179.72829, 175.275605, 170.831659, 165.499518, 159.965458, 154.256084, 148.398001, 142.550276, 136.554067, 130.382995, 124.010678, 117.279218, 110.346361, 103.238334, 95.9813658, 88.6520651, 81.2061239, 73.6496167, 65.9886178, 58.2943521, 50.4816833, 42.5306256, 34.4211932, 26.1073599, 17.6055963, 8.90633292, 0., -8.90871776, -18.1293461, -27.7571564, -37.8874201, -48.2672439, -59.4793298, -71.758215, -85.3384368, -99.2324482, -115.385705, -134.521577, -157.363436, -178.258714, -206.857097, -246.432332, -300.258165, -343.746479, -419.177631, -540.970111, -723.542412, -953.301677, -1328.70045, -1895.63073, -2708.54494, -3712.21403, -5281.5743, -7684.02071, -11186.9482}, 
{230.644686, 225.549343, 221.276528, 217.601877, 214.301026, 211.14961, 207.923267, 204.397631, 200.348339, 194.200203, 187.620012, 180.923731, 174.427326, 169.687863, 165.283766, 161.034559, 156.759767, 151.753696, 146.571178, 141.241823, 135.795243, 130.399758, 124.890791, 119.24247, 113.428926, 107.313599, 101.025583, 94.5832827, 88.0051028, 81.3452456, 74.5719997, 67.6894509, 60.7016853, 53.6714628, 46.5207255, 39.2300897, 31.7801716, 24.1207947, 16.2756849, 8.23777562, 0., -8.25393396, -16.7981776, -25.7161074, -35.0911002, -44.6778543, -55.0198957, -66.3320723, -78.829232, -91.6435998, -106.505696, -124.063416, -144.964658, -163.775287, -189.658045, -225.69364, -274.962784, -313.083784, -381.584712, -494.531239, -665.989037, -881.683134, -1240.70113, -1790.51499, -2581.00657, -3553.91473, -5081.14038, -7427.04393, -10855.9858}, 
{204.949773, 201.071126, 197.905217, 195.246737, 192.890376, 190.630825, 188.262774, 185.580914, 182.379935, 177.278699, 171.718057, 165.963028, 160.278636, 155.914796, 151.757677, 147.678343, 143.547856, 138.840123, 133.982228, 129.004099, 123.935661, 118.939037, 113.859081, 108.672842, 103.357369, 97.8068086, 92.1142731, 86.2899726, 80.3441169, 74.3057015, 68.1586367, 61.9056181, 55.5493414, 49.1434918, 42.6193799, 35.9593056, 29.1455691, 22.1397473, 14.9531524, 7.57637353, 0., -7.6401998, -15.5568876, -23.817546, -32.4896574, -41.3177284, -50.8214078, -61.1973687, -72.6422837, -84.3961618, -97.9950056, -114.018154, -133.044944, -149.885588, -173.196203, -205.863779, -250.775306, -283.720583, -345.522668, -449.907425, -610.600723, -812.791405, -1155.8164, -1687.80212, -2456.25049, -3402.10245, -4893.16741, -7190.53768, -10555.3056}, 
{182.534813, 179.527507, 177.178917, 175.294163, 173.678361, 172.136629, 170.474086, 168.49585, 166.007037, 161.766511, 157.044147, 152.063566, 147.048389, 142.99358, 139.042878, 135.111368, 131.114134, 126.685851, 122.134173, 117.486347, 112.76962, 108.123033, 103.417319, 98.6350069, 93.7586259, 88.7227275, 83.5770077, 78.3231858, 72.9629811, 67.4971152, 61.9287041, 56.2598663, 50.4927201, 44.6712063, 38.738892, 32.6811664, 26.4834191, 20.1387421, 13.621741, 6.91472423, 0., -7.06724582, -14.4067335, -22.0653056, -30.0898049, -38.19385, -46.8907973, -56.3607792, -66.783928, -77.4921481, -89.8530911, -104.386181, -121.610841, -136.610285, -157.514631, -187.017785, -227.813654, -255.806599, -311.195892, -407.391261, -557.802432, -747.270316, -1074.91108, -1588.43805, -2335.48049, -3259.15168, -4720.71301, -6976.88882, -10284.4035}, 
{165.290507, 162.286376, 160.078255, 158.450848, 157.188853, 156.076972, 154.899907, 153.442357, 151.489025, 147.77082, 143.54775, 139.026033, 134.411887, 130.567225, 126.78029, 122.995023, 119.155364, 115.021951, 110.795345, 106.492805, 102.131591, 97.7874608, 93.395776, 88.9503958, 84.4451802, 79.8803286, 75.2408251, 70.5179937, 65.7031583, 60.7565616, 55.7130412, 50.5763533, 45.3502544, 40.0687128, 34.693188, 29.2153516, 23.6268752, 17.9944365, 12.2046983, 6.21932983, 0., -6.54855977, -13.3849676, -20.5247794, -27.983551, -35.3990376, -43.315716, -51.9002622, -61.3193524, -70.9677412, -82.0927948, -95.1699579, -110.674675, -124.050902, -142.81816800000001, -169.464514, -206.47798, -229.664784, -278.867519, -367.246955, -507.963863, -685.99481, -999.053177, -1492.11906, -2218.17943, -3129.51904, -4573.22466, -6797.68259, -10051.2791}, 
{152.137356, 148.560618, 146.028799, 144.288529, 143.086442, 142.169167, 141.283337, 140.175584, 138.592538, 135.133503, 131.151371, 126.851704, 122.440065, 118.735667, 115.084965, 111.448061, 107.78506, 103.948328, 100.048801, 96.0896769, 92.0741556, 87.9923766, 83.8658206, 79.7029095, 75.5120655, 71.371298, 67.1916066, 62.9535786, 58.6378009, 54.159387, 49.5905871, 44.9381778, 40.2089354, 35.4261284, 30.5734448, 25.6510642, 20.6591665, 15.7676707, 10.7391215, 5.50580328, 0., -6.0717388, -12.4611006, -19.1455073, -26.1023811, -32.8645712, -40.0319015, -47.7596232, -56.2029876, -64.7948193, -74.7017665, -86.3680509, -100.237894, -112.156793, -129.007185, -153.07278, -186.637291, -205.292071, -248.690132, -329.792132, -461.558725, -629.404881, -928.92831, -1400.89323, -2107.06889, -3013.74267, -4448.12437, -6647.19318, -9847.92831}, 
{140.781753, 136.682234, 133.820464, 131.915018, 130.684473, 129.847403, 129.122385, 128.227993, 126.882804, 123.587991, 119.766493, 115.623843, 111.365579, 107.785616, 104.265758, 100.776187, 97.2870864, 93.7156472, 90.1062419, 86.4502507, 82.7390537, 78.8895359, 74.9973708, 71.0837365, 67.1698115, 63.3987972, 59.6210392, 55.8089063, 51.9347676, 47.8783341, 43.7416956, 39.5342838, 35.2655307, 30.9463936, 26.5841688, 22.1876782, 17.7657435, 13.5945717, 9.30864537, 4.80983222, 0., -5.61379701, -11.5769522, -17.8296726, -24.3121648, -30.4556994, -36.9129944, -43.8278312, -51.3439913, -58.9189753, -67.6573579, -77.9774331, -90.2974945, -100.815114, -115.857596, -137.531524, -167.94348, -182.552124, -220.771131, -295.366252, -419.10324, -577.762732, -865.065819, -1317.76997, -2006.19433, -2910.75903, -4337.91853, -6512.61869, -9659.80539}, 
{130.494647, 126.16847, 123.140568, 121.124242, 119.832792, 118.979516, 118.277715, 117.440689, 116.181737, 112.994264, 109.299423, 105.29847, 101.192663, 97.752079, 94.3816266, 91.0550345, 87.7460315, 84.3994753, 81.0295143, 77.6214257, 74.1604867, 70.521654, 66.8446537, 63.1588911, 59.4937719, 56.0305044, 52.5859699, 49.1288527, 45.6278369, 41.946012, 38.1998949, 34.400408, 30.5584734, 26.6696049, 22.7662965, 18.8656338, 14.9847023, 11.5026577, 7.92968738, 4.13804898, 0., -5.16541025, -10.7114327, -16.5445265, -22.5711508, -28.1368435, -33.9333535, -40.0915086, -46.7421363, -53.3523053, -60.9821059, -70.0278696, -80.8859276, -90.0081075, -103.313046, -122.774876, -150.36773, -161.586559, -195.476351, -264.602911, -381.532045, -532.136331, -809.061255, -1246.06057, -1920.26372, -2824.73561, -4245.42361, -6395.05448, -9486.35496}, 
{120.218374, 116.363283, 113.606148, 111.699338, 110.395219, 109.44616, 108.604528, 107.622691, 106.253017, 103.165789, 99.6282923, 95.8257276, 91.9432956, 88.7062903, 85.5437823, 82.4249355, 79.3189135, 76.1411207, 72.935984, 69.6941707, 66.4063484, 62.9681364, 59.5032696, 56.0404345, 52.6083179, 49.374891, 46.1738422, 42.9781445, 39.7607707, 36.4023776, 33.0051808, 29.5790797, 26.1339737, 22.6437028, 19.1686496, 15.7331373, 12.361489, 9.52475412, 6.62183918, 3.4983769, 0., -4.71442103, -9.83778635, -15.2497584, -20.8299995, -25.8728603, -31.0774398, -36.5575249, -42.4269028, -48.1422156, -54.7372534, -62.5886611, -72.0730838, -79.7205999, -91.2930477, -108.705699, -133.873825, -142.639015, -173.419696, -238.560614, -350.406513, -494.295262, -763.592234, -1191.33966, -1857.42283, -2763.52395, -4177.2945, -6299.89362, -9332.48046}, 
{110.75343, 107.590506, 105.231651, 103.484465, 102.156549, 101.055506, 99.9889362, 98.7644409, 97.1896215, 94.1790465, 90.790563, 87.1889854, 83.539128, 80.5146295, 77.5679503, 74.660375, 71.7531881, 68.7101427, 65.6290678, 62.5102607, 59.3540188, 56.1009217, 52.8348715, 49.5800526, 46.3606494, 43.309762, 40.2990927, 37.3092595, 34.3208804, 31.2458379, 28.1609798, 25.0744183, 21.9942657, 18.872276, 15.7954629, 12.7944818, 9.89998836, 7.66367431, 5.38674428, 2.89143919, 0., -4.26469321, -8.96129429, -13.9478179, -19.0822789, -23.6261418, -28.2725914, -33.1182624, -38.2597896, -43.1364487, -48.7651769, -55.5055526, -63.7171541, -69.9228172, -79.8535597, -95.4036569, -118.467384, -125.376221, -153.812357, -215.895186, -323.744101, -462.041787, -725.217765, -1146.38373, -1806.37838, -2714.14396, -4118.48155, -6208.22748, -9172.21812}, 
{102.109544, 99.7569725, 97.8623984, 96.2978131, 94.9352071, 93.6465714, 92.3038964, 90.7791731, 88.9443922, 85.9985892, 82.7558921, 79.3574736, 75.9445064, 73.1306553, 70.3956043, 67.6915293, 64.9706062, 62.0343038, 59.0457882, 56.0175184, 52.9619535, 49.8802388, 46.8006721, 43.7402377, 40.7159199, 37.8110855, 34.9497834, 32.1224447, 29.3195008, 26.4924822, 23.6862811, 20.9068892, 18.1602981, 15.3777876, 12.6699457, 10.0726484, 7.62177209, 5.93500366, 4.23368424, 2.32096572, 0., -3.81327259, -8.07360397, -12.6229576, -17.3032971, -21.3601287, -25.4704559, -29.714825, -34.1737824, -38.2677164, -43.0013945, -48.7194263, -55.7664212, -60.5911985, -68.9924741, -82.8731736, -104.136223, -109.710419, -136.462466, -196.284943, -301.070425, -434.866515, -693.100714, -1109.41568, -1764.10962, -2672.48073, -4063.17178, -6111.48937, -8992.74014}, 
{94.2964493, 92.7695158, 91.3437153, 89.9575739, 88.5496178, 87.0583731, 85.422366, 83.5801226, 81.4701691, 78.5889691, 75.4939364, 72.300422, 69.1237773, 66.5079261, 63.9682178, 61.4485745, 58.8929184, 56.0413661, 53.1231674, 50.1637667, 47.1886082, 44.2663165, 41.3618836, 38.4834818, 35.6392833, 32.8548298, 30.1139763, 27.4179471, 24.7679666, 22.1583995, 19.6000738, 17.0969577, 14.6530196, 12.1827008, 9.81530731, 7.59061801, 5.54841194, 4.35432756, 3.17194063, 1.79068637, 0., -3.35720491, -7.16636284, -11.2594299, -15.4683622, -19.038262, -22.6226808, -26.2883164, -30.1018666, -33.4687308, -37.3814243, -42.1711641, -48.1691674, -51.7021831, -58.7076835, -71.1186729, -90.8681555, -95.5538487, -121.178158, -179.408204, -281.911104, -412.260057, -666.403945, -1078.65842, -1727.59583, -2634.41935, -4005.5522, -6001.11258, -8781.21869}, 
{87.7713906, 86.4960091, 85.1811683, 83.7968101, 82.3128763, 80.6993088, 78.9260494, 76.9630399, 74.7802222, 72.0511526, 69.1607128, 66.1973987, 63.2497064, 60.7494344, 58.3044555, 55.8659448, 53.3850774, 50.6152646, 47.784551, 44.9232173, 42.0615442, 39.301138, 36.5724235, 33.8771513, 31.2170719, 28.5794753, 25.9863566, 23.4452507, 20.9636922, 18.5573668, 16.2223978, 13.9630598, 11.783627, 9.60000198, 7.54017972, 5.64378337, 3.95043606, 3.11669301, 2.31847245, 1.34862468, 0., -2.86170759, -6.14562894, -9.68805125, -13.3252617, -16.3257928, -19.3207882, -22.3736369, -25.547728, -28.2713039, -31.4969588, -35.5421401, -40.7242953, -43.2310953, -49.1616752, -60.4853932, -79.1716077, -83.2430703, -108.194389, -165.573564, -266.928597, -396.429249, -647.75824, -1054.36803, -1693.5088, -2596.01584, -3934.58632, -5846.45193, -8468.84437}, 
{82.0309902, 80.8879567, 79.6096357, 78.1899457, 76.622805, 74.9021323, 73.0218458, 70.9758641, 68.7581056, 66.1932829, 63.5122026, 60.7764656, 58.0476726, 55.637402, 53.257286, 50.8689344, 48.4339568, 45.7340129, 42.9826428, 40.2134361, 37.4599828, 34.8511951, 32.2872114, 29.7634926, 27.2754996, 24.774658, 22.3180779, 19.918834, 17.5900013, 15.3641527, 13.2270657, 11.1840158, 9.24027873, 7.31904842, 5.54051468, 3.94278571, 2.56396973, 2.03191979, 1.55910133, 0.947724641, 0., -2.3603199, -5.10118416, -8.06899945, -11.1101725, -13.5280581, -15.9293355, -18.3776321, -20.9365754, -23.0678086, -25.6777369, -29.0707811, -33.5513625, -35.1556526, -40.1636214, -50.5869898, -68.4374784, -72.203814, -96.8299089, -153.736681, -254.34505, -383.750348, -632.350248, -1032.43624, -1659.18033, -2552.95931, -3850.45028, -5664.25527, -8106.97631}, 
{76.9707195, 75.8621394, 74.5611871, 73.0790334, 71.4268495, 69.6158063, 67.6570748, 65.5618261, 63.3412311, 60.9450827, 58.4704813, 55.9531493, 53.4288089, 51.087234, 48.7484747, 46.3866325, 43.9758091, 41.3365635, 38.6579569, 35.975508, 33.3247354, 30.8558433, 28.4437906, 26.0782215, 23.7487807, 21.3748538, 19.0444472, 16.775309, 14.585187, 12.5188412, 10.5562024, 8.70421353, 6.96981753, 5.28812718, 3.76664768, 2.44105406, 1.34702134, 1.06977331, 0.875616728, 0.580407118, 0., -1.85821402, -4.04652853, -6.42570206, -8.85649313, -10.6846322, -12.4919171, -14.3451174, -16.311003, -17.8933143, -19.9470619, -22.7642269, -26.6367909, -27.4525786, -31.6693904, -41.3408701, -58.5206616, -62.2399847, -86.7838771, -143.414952, -243.395824, -373.020461, -618.457409, -1010.56686, -1621.66818, -2501.0912, -3748.66476, -5451.59296, -7697.07988}, 
{72.4860498, 71.3353381, 69.967892, 68.4061262, 66.6724552, 64.7892935, 62.7790558, 60.6641567, 58.4670106, 56.2362749, 53.9576244, 51.6429764, 49.3042485, 47.0143357, 44.6997869, 42.3481283, 39.9468865, 37.3618686, 34.7510078, 32.1505183, 29.5966139, 27.2544382, 24.9797041, 22.7570538, 20.5711298, 18.3145381, 16.1007718, 13.9512876, 11.8875423, 9.96151604, 8.15193297, 6.46804054, 4.91908621, 3.45552525, 2.16891421, 1.09201745, 0.257599308, 0.200019021, 0.249808109, 0.239092969, 0., -1.36056209, -2.99516214, -4.78158666, -6.59762218, -7.83508972, -9.05212751, -10.3209083, -11.7136048, -12.7828905, -14.3282371, -16.629617, -19.9670027, -20.0985972, -23.6348501, -32.6644414, -49.2760512, -53.1554874, -77.7554513, -134.125772, -233.316278, -363.036691, -604.357163, -986.463671, -1578.03011, -2436.25295, -3624.75043, -5205.53533, -7240.62045}, 
{68.4724527, 67.2243335, 65.7618201, 64.1132768, 62.3070676, 60.3715567, 58.3351084, 56.2260867, 54.0728559, 51.9965824, 49.8957072, 47.7614734, 45.5851245, 43.3341123, 41.0329879, 38.682511, 36.2834413, 33.7488806, 31.2023098, 28.6795519, 26.21643, 23.9863352, 21.8324951, 19.7357053, 17.6767614, 15.5281868, 13.4223587, 11.3833816, 9.43536035, 7.63226104, 5.95638238, 4.41988452, 3.0349276, 1.76952966, 0.697649771, -0.150895095, -0.746287965, -0.607577646, -0.336535071, -0.0837969507, 0., -0.872536286, -1.96058503, -3.16008079, -4.36695815, -5.01900493, -5.65356114, -6.34982007, -7.18697498, -7.77160669, -8.84456592, -10.6740909, -13.5284199, -13.0704323, -16.0158687, -24.475111, -40.558541, -44.7542269, -69.4437895, -125.386536, -223.341773, -352.596143, -588.326948, -957.830483, -1525.32388, -2354.28602, -3474.22797, -4923.15275, -6739.06338}, 
{68.2547617, 64.4030038, 61.0624053, 58.1628557, 55.6342446, 53.4064616, 51.4093962, 49.572938, 47.8269765, 45.957039, 44.0951222, 42.2288605, 40.3458883, 38.4667234, 36.5329635, 34.5190896, 32.3995828, 29.9385344, 27.4049711, 24.8575296, 22.354847, 20.1021213, 17.9528037, 15.9069066, 13.9644425, 12.1438868, 10.4194037, 8.78362046, 7.22916423, 5.71567078, 4.28195543, 2.93384198, 1.67715426, 0.346381608, -0.812783878, -1.72598459, -2.3188629, -1.86839962, -1.20836336, -0.523861146, 0., -0.627400983, -1.46345147, -2.37105288, -3.21310661, -3.25085515, -3.18952241, -3.13267337, -3.18387301, -2.80547233, -2.9987359, -4.1237143, -6.54045811, -6.83371701, -10.6489628, -19.8563666, -36.3260991, -39.4859101, -62.6253602, -116.591589, -212.231736, -342.977808, -571.922343, -920.991834, -1451.388, -2230.85247, -3261.692, -4562.57208, -6152.15819}, 
{68.4539243, 61.4483189, 55.8628302, 51.4992593, 48.1594069, 45.6450741, 43.7580616, 42.3001706, 41.0732018, 39.3851077, 37.7290771, 36.1044501, 34.5105672, 33.1173068, 31.6852556, 30.1455386, 28.4292807, 26.0790926, 23.5700193, 20.9885914, 18.4213396, 16.0987041, 13.9057423, 11.871421, 10.0247072, 8.59280239, 7.32714494, 6.17740787, 5.09326418, 3.90357687, 2.727153, 1.56198962, 0.406083746, -1.08429602, -2.43273282, -3.50453824, -4.16502385, -3.33725149, -2.20568241, -1.01252808, 0., -0.497549335, -1.22525202, -1.99042368, -2.60037993, -2.04190726, -1.27106206, -0.423371592, 0.365636893, 1.81885925, 2.59897586, 2.22709019, 0.224305732, -1.26648705, -6.39468643, -16.6879037, -33.6737502, -35.6379855, -56.6468132, -107.524585, -199.095654, -329.609444, -547.615087, -868.664624, -1349.75899, -2057.6127, -2978.63703, -4119.31905, -5486.14585}, 
{65.4719479, 57.4208681, 51.1223649, 46.3288876, 42.792886, 40.2668095, 38.5031077, 37.2542302, 36.2726266, 34.6627386, 33.0842266, 31.5487434, 30.0679416, 28.9024083, 27.7152878, 26.4186592, 24.9246012, 22.6910859, 20.2659417, 17.7428902, 15.2156527, 12.9020086, 10.7219986, 8.71972156, 6.93927587, 5.70709501, 4.6720087, 3.76518156, 2.9177782, 1.89110459, 0.854127432, -0.194045212, -1.25430528, -2.72905117, -4.05706578, -5.07863845, -5.63405853, -4.49776766, -3.002242, -1.41410996, 0., -0.232086191, -0.689233061, -1.15585079, -1.41634954, -0.326326355, 1.02947019, 2.49510466, 3.91464163, 6.02483426, 7.41998307, 7.58707717, 6.01310568, 4.25526237, -1.09775015, -11.3870246, -27.9536538, -30.037432, -49.9212697, -97.7867791, -183.815572, -309.038656, -511.089458, -801.715921, -1230.32357, -1857.50504, -2661.44699, -3640.93761, -4794.76507}, 
{60.4371914, 52.8944439, 47.0061111, 42.5344545, 39.2417357, 36.8902165, 35.2421584, 34.0598231, 33.1054723, 31.4888045, 29.8856696, 28.3193545, 26.8131459, 25.6686144, 24.5194497, 23.2776252, 21.8551143, 19.7300162, 17.4217281, 15.0157733, 12.5976748, 10.34573, 8.215578, 6.25563199, 4.51430532, 3.33212144, 2.34853948, 1.49512871, 0.703458391, -0.282898639, -1.26917813, -2.24861224, -3.21443316, -4.5363881, -5.68058815, -6.48965946, -6.80622818, -5.41661253, -3.64226975, -1.74834914, 0., 0.144274795, 0.0900209157, 0.0494304766, 0.23469559, 1.80159795, 3.64130425, 5.58857079, 7.47815384, 9.93910759, 11.6941713, 12.260382, 11.1547768, 9.8495151, 5.12446293, -4.2853914, -19.6450596, -22.6973878, -42.0384195, -86.7420329, -165.882106, -281.689148, -463.767142, -721.985092, -1095.87966, -1636.34431, -2319.17767, -3138.9521, -4090.23996}, 
{54.4780137, 48.4428385, 43.6791707, 39.9986734, 37.21301, 35.1338437, 33.5728378, 32.3416556, 31.2519604, 29.5621783, 27.8585046, 26.1738973, 24.5413145, 23.2625116, 21.9941304, 20.6616102, 19.1903901, 17.1513851, 14.9663681, 12.7025882, 10.427294, 8.26356395, 6.20048559, 4.282976, 2.55595225, 1.31323844, 0.251281779, -0.684563488, -1.54894311, -2.57958539, -3.57482051, -4.51606121, -5.38472022, -6.45481104, -7.29810532, -7.77897552, -7.76179404, -6.16045051, -4.16999329, -2.03498794, 0., 0.606819966, 1.05792541, 1.54218478, 2.24846652, 4.24757601, 6.49367, 8.82284217, 11.0711862, 13.6883341, 15.6514259, 16.5511399, 15.9781543, 15.634255, 12.1545697, 4.28533388, -9.22721724, -13.6309911, -32.5879525, -73.754209, -144.785868, -247.984626, -407.069825, -631.311504, -949.22515, -1399.94534, -1960.88484, -2624.88686, -3384.79461}, 
{48.7227738, 44.6398442, 41.3066454, 38.604258, 36.4137625, 34.6162396, 33.0927698, 31.7244338, 30.3923122, 28.5817329, 26.7278303, 24.8699861, 23.0475818, 21.5306864, 20.0357196, 18.509788, 16.8999986, 14.9106943, 12.8288514, 10.6986821, 8.56439891, 6.48920575, 4.49072666, 2.60557732, 0.870373416, -0.504197236, -1.72521994, -2.82570787, -3.83867419, -4.96010825, -5.99485657, -6.91074186, -7.67558681, -8.43282424, -8.90442271, -8.98796086, -8.58101735, -6.79594598, -4.62964023, -2.2937687, 0., 1.13083567, 2.15989592, 3.23917679, 4.5206743, 6.91731822, 9.51579733, 12.1637342, 14.7087513, 17.3991688, 19.5216327, 20.7634863, 20.8120732, 21.7274658, 19.8751872, 13.9934892, 2.82062372, -2.85138016, -21.1595584, -58.1871699, -120.017474, -208.348795, -342.419193, -531.534524, -793.157935, -1154.12298, -1595.62429, -2110.26622, -2690.65314}, 
{44.2998304, 42.0592533, 40.0536373, 38.233922, 36.5510472, 34.9559528, 33.3995785, 31.832864, 30.2067492, 28.2463412, 26.2187454, 24.1652348, 22.1270823, 20.3197254, 18.5406064, 16.7613322, 14.9535099, 12.9634457, 10.9381676, 8.89940251, 6.86887778, 4.85635082, 2.90030642, 1.02725964, -0.73627446, -2.27482884, -3.6864212, -4.98011725, -6.16498272, -7.38561978, -8.46134313, -9.34700391, -9.99745327, -10.418932, -10.4943457, -10.1579897, -9.34415937, -7.38976337, -5.06543819, -2.54443376, 0., 1.69160825, 3.34134796, 5.05717119, 6.94702998, 9.71653493, 12.6369161, 15.5770622, 18.4058619, 21.1982666, 23.5346769, 25.2015565, 25.9853686, 28.2471315, 28.1689322, 24.5074123, 16.0192136, 9.6283068, -7.34292715, -39.4047781, -91.0675361, -163.20536, -271.236932, -424.493521, -630.475917, -904.692043, -1232.45178, -1606.61452, -2020.03966}, 
{31.3611692, 33.4138344, 34.6123875, 35.0543707, 34.837326, 34.0587958, 32.8163219, 31.2074467, 29.3297123, 27.404283, 25.3556298, 23.2318459, 21.0810247, 19.0533799, 17.0540359, 15.0902378, 13.1692306, 11.3600238, 9.58339213, 7.82187479, 6.05801097, 4.17746025, 2.29839333, 0.442101287, -1.37012481, -3.08825048, -4.73122545, -6.289256, -7.75254846, -9.29802506, -10.6544898, -11.7374626, -12.4624633, -12.5589911, -12.202995, -11.3844033, -10.0931443, -7.98083331, -5.51103668, -2.80900778, 0., 2.33855583, 4.71445469, 7.18331415, 9.80075179, 13.1193625, 16.4989955, 19.7964776, 22.8686353, 25.1976694, 27.1648828, 28.7769525, 30.0405554, 34.3745852, 37.0086155, 36.5844368, 31.7438393, 24.3660043, 8.56037482, -18.326215, -58.9469312, -111.659095, -192.003407, -312.956989, -466.821193, -649.230038, -863.162643, -1109.76185, -1390.17051}, 
{37.107078, 37.9494898, 38.1475584, 37.7748672, 36.9049999, 35.61154, 33.9680713, 32.0481773, 29.9254415, 27.7748251, 25.5279832, 23.2179484, 20.8777534, 18.5825868, 16.3064628, 14.0655515, 11.8760232, 9.83055734, 7.83821112, 5.88455095, 3.9551433, 1.96138975, -0.00731245436, -1.93573091, -3.80863323, -5.6288923, -7.35592834, -8.96726683, -10.4404333, -11.8918334, -13.1045604, -14.0005876, -14.5018883, -14.3327386, -13.6918882, -12.5803894, -10.9992944, -8.70169555, -6.03578926, -3.10181195, 0., 2.87088123, 5.82853593, 8.89213931, 12.0808666, 15.681342, 19.3383122, 22.9639726, 26.4705188, 29.5587418, 32.4368037, 35.1014621, 37.5494744, 42.4526156, 46.0626187, 47.3062344, 45.1102132, 39.9921181, 28.651562, 9.37897069, -19.5352305, -57.2665312, -113.12676, -196.55736, -298.235907, -413.117734, -537.811019, -668.510728, -801.411827}, 
{54.2496337, 50.3266662, 46.8385324, 43.7179324, 40.8975662, 38.3101336, 35.8883348, 33.5648695, 31.2724379, 28.7494751, 26.2006517, 23.6363737, 21.0670471, 18.525831, 15.9912767, 13.4646886, 10.9473713, 8.38916945, 5.86343143, 3.39204573, 0.99690084, -1.26416265, -3.41958917, -5.46187108, -7.3835007, -9.22601578, -10.9132451, -12.4180629, -13.7133433, -14.8183369, -15.6409909, -16.1356292, -16.2565754, -15.8913591, -15.0878157, -13.8269866, -12.089913, -9.575889, -6.66040207, -3.43719234, 0., 3.32808289, 6.77440878, 10.3369783, 14.0137919, 17.7915291, 21.6840401, 25.6938539, 29.8234995, 34.4683192, 39.0809036, 43.5066565, 47.5909816, 52.0892308, 55.5728806, 57.5233557, 57.4220807, 57.0603078, 52.6857029, 42.8557596, 26.1279713, 0.306807571, -33.7911656, -77.5102898, -128.97305, -194.828091, -250.841278, -280.499173, -267.288336}, 
{45.9632263, 43.5948047, 41.2944117, 39.0497927, 36.8486933, 34.6788586, 32.5280343, 30.3839656, 28.2343979, 26.0514214, 23.8446989, 21.6082379, 19.3360459, 16.9792869, 14.5919492, 12.1851779, 9.7701179, 7.3618403, 4.9659933, 2.59215129, 0.249888704, -2.11170034, -4.39836902, -6.57635081, -8.61187918, -10.3777734, -11.9710468, -13.3952985, -14.6541278, -15.978402, -17.0535448, -17.792248, -18.1072036, -17.711688, -16.7975747, -15.3573219, -13.3833878, -10.6397615, -7.43875759, -3.86422188, 0., 3.91966746, 8.02148764, 12.2817728, 16.6768352, 21.1786397, 25.7695849, 30.427722, 35.1311022, 39.788373, 44.4747508, 49.1960481, 53.9580778, 59.5390344, 64.8633958, 69.6280218, 73.5297724, 77.145323, 78.9397916, 78.2581119, 74.4452177, 65.9571768, 54.8055203, 38.1237308, 26.1576508, 25.2296282, 44.2436292, 91.4582154, 175.131949}, 
{51.3870855, 48.6794894, 46.1831565, 43.8403542, 41.59335, 39.3844114, 37.155806, 34.8498012, 32.4086645, 29.4555478, 26.3794804, 23.2503763, 20.1381492, 17.3421612, 14.6110986, 11.9230958, 9.25628726, 6.47079096, 3.70996456, 0.999149184, -1.63631406, -4.15046897, -6.54683554, -8.80831871, -10.9178234, -12.8574803, -14.6112782, -16.1624317, -17.4941555, -18.7345028, -19.6639141, -20.2076685, -20.2910453, -19.6603376, -18.491405, -16.7811208, -14.5263586, -11.5733769, -8.12991022, -4.25307808, 0., 4.55743405, 9.3827832, 14.4248361, 19.6323815, 24.7770149, 30.0555953, 35.4877886, 41.0932604, 46.9053614, 52.9245989, 59.1651649, 65.6412515, 72.5154609, 79.5942109, 86.8323292, 94.184644, 100.942288, 107.989262, 115.545873, 123.832425, 131.321797, 143.476579, 160.747982, 191.051251, 237.852606, 314.950336, 433.559716, 604.896017}, 
{52.5791609, 52.189412, 50.9809292, 49.0755733, 46.5952055, 43.6616866, 40.3968776, 36.9226396, 33.3608334, 30.1281937, 26.9337584, 23.7814391, 20.6751474, 17.664573, 14.689538, 11.7356427, 8.78848729, 5.69032192, 2.6274371, -0.357226835, -3.22072955, -5.77288536, -8.1768974, -10.4487235, -12.6043213, -14.8262476, -16.8972218, -18.766562, -20.3835865, -21.7939441, -22.8120902, -23.348811, -23.3148923, -22.3006067, -20.6654593, -18.4484415, -15.6885449, -12.4879459, -8.79717705, -4.62995592, 0., 5.28008011, 10.9150172, 16.810651, 22.8728212, 28.6611796, 34.566229, 40.6322844, 46.9036604, 53.1753873, 59.8407787, 67.0438633, 74.92867, 83.7004277, 93.417485, 104.199391, 116.165694, 127.53876, 141.094195, 157.710418, 178.265852, 202.213911, 234.708038, 276.259409, 335.688216, 421.713219, 535.1499, 678.789558, 855.423494}
};

  const double a4tab[50][69] = {
{-4795.25483, -4657.11547, -4533.57386, -4420.99082, -4315.72713, -4214.14359, -4112.601, -4007.46016, -3895.08186, -3749.86323, -3598.9142, -3447.38103, -3300.40998, -3183.58631, -3073.44169, -2966.94678, -2861.07226, -2743.59299, -2624.35375, -2504.00351, -2383.19127, -2264.35938, -2145.64608, -2026.98299, -1908.30176, -1789.65691, -1670.80798, -1551.63744, -1432.02772, -1310.85768, -1189.41483, -1067.98309, -946.846341, -827.165104, -707.996037, -589.272406, -470.927472, -353.153181, -235.520643, -117.859652, 0., 117.602012, 235.993705, 355.595891, 476.829381, 595.994952, 719.281464, 848.757742, 986.492612, 1125.29678, 1280.20043, 1456.97565, 1661.39451, 1865.75367, 2122.69078, 2451.36808, 2870.9478, 3306.89371, 3909.54591, 4735.54605, 5841.53576, 7186.8075, 9120.05042, 11591.8611, 15444.2924, 19758.4141, 29767.7712, 49167.7902, 81653.8973}, 
{-4386.2486, -4276.4998, -4166.87176, -4057.27842, -3947.63375, -3837.85169, -3727.8462, -3617.53124, -3506.82077, -3396.24985, -3284.86287, -3172.32536, -3058.30283, -2934.12142, -2811.12175, -2692.30507, -2580.67265, -2501.29049, -2426.26919, -2349.78411, -2266.0106, -2143.26267, -2011.92156, -1876.50719, -1741.53944, -1625.69861, -1513.68005, -1404.33951, -1296.53273, -1186.17111, -1076.23248, -966.750311, -857.758075, -749.581249, -641.844515, -534.46455, -427.358033, -320.645434, -213.958122, -107.131257, 0., 107.056887, 214.965285, 324.107473, 434.865731, 543.216635, 655.710448, 774.491733, 901.705051, 1031.01986, 1176.44587, 1343.51767, 1537.76987, 1733.97589, 1980.73599, 2295.88922, 2697.27467, 3113.77504, 3687.76832, 4472.67611, 5521.92004, 6796.91537, 8627.10278, 10961.4315, 14608.3709, 18698.1916, 28214.4395, 46677.342, 77607.1262}, 
{-4089.70007, -3999.83272, -3900.40845, -3793.52861, -3681.29454, -3565.8076, -3449.16915, -3333.48054, -3220.84311, -3124.77905, -3031.40055, -2938.24064, -2842.83236, -2728.08787, -2612.00939, -2497.97828, -2389.37592, -2309.8317, -2234.37973, -2158.30216, -2076.88115, -1964.99854, -1846.4969, -1724.8185, -1603.40559, -1496.11382, -1391.80673, -1289.76124, -1189.25426, -1088.05749, -987.55515, -887.626246, -788.149784, -688.774549, -589.701849, -490.902776, -392.348421, -294.336138, -196.38025, -98.321342, 0., 97.9874303, 196.858427, 297.074709, 399.097994, 499.259521, 603.803679, 714.844378, 834.49553, 956.8933, 1095.32044, 1255.08195, 1441.48285, 1629.7946, 1867.36916, 2171.52496, 2559.58043, 2963.77126, 3520.53168, 4281.21321, 5297.16736, 6532.52802, 8300.29955, 10550.9045, 14057.1258, 18028.6165, 27105.1878, 44569.1112, 73702.6582}, 
{-3869.29607, -3792.59815, -3700.71666, -3596.83571, -3484.13943, -3365.81194, -3245.03736, -3124.99981, -3008.88343, -2913.32449, -2822.67409, -2734.73548, -2647.31191, -2552.51261, -2456.11249, -2358.19241, -2258.83326, -2155.5432, -2052.00489, -1949.32827, -1848.62329, -1756.61105, -1666.5459, -1577.29332, -1487.71881, -1392.40141, -1296.20764, -1199.71758, -1103.51131, -1010.11081, -917.377477, -825.114634, -733.125587, -640.567837, -548.148831, -455.930204, -363.973591, -272.876208, -181.949879, -91.0420086, 0., 90.1866618, 181.269486, 273.857901, 368.561335, 462.482307, 561.139919, 666.546362, 780.71383, 897.948208, 1031.05052, 1185.11547, 1365.23778, 1545.93007, 1775.10198, 2070.08108, 2448.19491, 2844.61617, 3391.68922, 4139.60356, 5138.54867, 6355.55344, 8090.28923, 10297.6099, 13712.8828, 17642.831, 26328.2418, 42776.3866, 69994.5366}, 
{-3688.72344, -3620.28003, -3534.3291, -3434.29407, -3323.59833, -3205.66528, -3083.91832, -2961.78084, -2842.67626, -2739.75984, -2642.83036, -2551.41848, -2465.05485, -2394.42261, -2323.43893, -2247.17347, -2160.69588, -2024.75159, -1882.46418, -1742.633, -1614.05739, -1545.14423, -1489.24234, -1439.30806, -1388.29774, -1306.06025, -1217.90239, -1126.02348, -1032.62284, -945.925062, -859.694111, -773.719216, -687.789602, -600.783951, -513.766253, -426.889953, -340.308496, -254.916558, -169.829861, -84.9053578, 0., 83.447602, 167.794816, 253.817352, 342.290922, 431.243692, 525.297932, 626.328371, 736.209734, 849.215705, 977.862472, 1127.06518, 1301.73898, 1475.10257, 1696.44611, 1983.36334, 2353.44796, 2744.04363, 3285.09418, 4026.29337, 5017.33496, 6227.8996, 7947.72035, 10138.8771, 13497.9681, 17433.977, 25771.8272, 41232.4571, 66536.8049}, 
{-3511.669, -3448.36228, -3367.77851, -3272.99804, -3167.10117, -3053.16822, -2934.27953, -2813.5154, -2693.95617, -2581.95877, -2476.01625, -2377.89827, -2289.37451, -2240.84482, -2193.9966, -2139.14746, -2066.61499, -1903.78345, -1729.07712, -1557.98694, -1406.00384, -1357.64213, -1331.76003, -1316.23912, -1298.96098, -1228.58923, -1147.91061, -1060.49389, -969.90785, -889.094259, -808.499694, -727.943732, -647.245948, -565.245724, -483.134905, -401.125143, -319.42809, -239.108105, -159.183049, -79.523491, 0., 77.5632713, 156.030772, 236.313367, 319.321921, 403.902372, 493.856483, 590.921089, 696.833024, 805.726906, 929.982676, 1074.37806, 1243.69077, 1410.03238, 1623.91323, 1903.17752, 2265.66943, 2649.78746, 3184.59978, 3919.72886, 4904.79718, 6111.47446, 7823.24142, 10012.0358, 13334.7074, 17295.1967, 25324.1697, 39870.6114, 63383.5064}, 
{-3301.81958, -3242.32883, -3167.59761, -3080.04195, -2982.07786, -2876.12137, -2764.5885, -2649.89526, -2534.45768, -2417.79493, -2306.37861, -2203.78347, -2113.58426, -2078.80618, -2047.79337, -2008.34038, -1948.2418, -1778.96537, -1595.16323, -1415.16067, -1257.28299, -1221.1488, -1211.27277, -1213.4629, -1213.52717, -1151.48722, -1077.25191, -994.943767, -908.685307, -833.212394, -757.788871, -682.291925, -606.598741, -529.775993, -452.835581, -375.978895, -299.407326, -224.101762, -149.172293, -74.5085093, 0., 72.3266902, 145.573709, 220.706247, 298.689499, 378.817046, 464.394334, 557.055201, 658.433482, 762.51293, 881.637499, 1020.50106, 1183.79748, 1343.43977, 1550.015, 1821.32939, 2175.18916, 2549.58149, 3074.05924, 3798.35623, 4772.2063, 5968.186, 7667.50096, 9854.41554, 13145.4268, 17119.632, 24873.4951, 38624.1385, 60588.6845}, 
{-3022.862, -2967.66361, -2900.31911, -2822.52015, -2735.95834, -2642.32532, -2543.31273, -2440.61219, -2335.91534, -2225.14199, -2118.06432, -2018.68268, -1930.99743, -1895.33367, -1864.83711, -1828.97824, -1777.2275, -1636.62394, -1484.04202, -1333.92478, -1200.71524, -1162.70827, -1144.95436, -1136.35581, -1125.81492, -1066.2531, -996.945919, -921.188061, -842.274195, -771.873467, -701.556286, -631.267537, -560.952101, -490.197592, -419.449071, -348.794329, -278.321158, -208.548445, -138.960447, -69.472514, 0., 67.5308789, 136.01998, 206.356297, 279.428824, 354.34641, 434.49025, 521.461393, 616.860889, 714.604893, 827.05331, 958.88115, 1114.76342, 1268.04501, 1467.2631, 1729.62474, 2072.337, 2431.15956, 2937.32579, 3640.6217, 4590.83328, 5759.94219, 7431.14749, 9603.34594, 12852.4522, 16800.4251, 24308.029, 37426.3271, 58206.3824}, 
{-2638.48309, -2589.85055, -2532.47575, -2467.52697, -2396.17251, -2319.58066, -2238.91972, -2155.35797, -2070.06371, -1981.87362, -1895.22025, -1812.20453, -1734.92739, -1677.45422, -1625.13573, -1575.28704, -1525.22331, -1463.08574, -1399.03301, -1334.04983, -1269.12095, -1209.3646, -1149.97859, -1090.29423, -1029.64285, -964.385738, -898.012251, -831.041724, -763.993491, -698.671476, -633.796583, -569.374308, -505.410146, -442.333357, -379.556167, -316.914565, -254.24454, -191.09907, -127.710361, -64.0276063, 0., 62.9688578, 126.965942, 192.62382, 260.575062, 328.849163, 401.722993, 480.870349, 567.965028, 657.033913, 760.456481, 882.965295, 1029.29292, 1176.56839, 1368.1692, 1619.86934, 1947.44278, 2282.2555, 2758.25267, 3424.97148, 4331.9491, 5448.65099, 7064.82952, 9196.15657, 12378.1095, 16230.7179, 23515.9972, 36210.4661, 56290.6436}, 
{-2405.71268, -2357.37371, -2305.09333, -2248.92728, -2188.93131, -2125.16116, -2057.67258, -1986.52131, -1911.7631, -1825.78366, -1739.37678, -1655.66624, -1577.77578, -1521.53728, -1472.28315, -1428.05392, -1386.8901, -1345.84384, -1304.33938, -1260.8126, -1213.69937, -1153.95915, -1090.49479, -1024.73273, -958.099404, -895.75918, -833.905408, -772.469359, -711.382305, -650.167044, -589.326712, -528.955972, -469.149485, -410.875751, -353.00606, -295.28554, -237.459321, -178.716175, -119.580127, -60.018845, 0., 59.1806414, 119.414746, 181.265883, 245.297621, 309.449323, 377.958449, 452.438249, 534.501976, 618.058134, 715.506621, 831.542589, 970.861189, 1111.62937, 1295.68177, 1538.3248, 1854.86492, 2173.99376, 2632.27845, 3279.67134, 4166.12475, 5260.24742, 6856.02255, 8984.53783, 12142.789, 15991.1783, 23070.5741, 35201.7283, 54205.3928}, 
{-2204.0302, -2154.47867, -2105.90828, -2057.23597, -2007.37868, -1955.25335, -1899.77692, -1839.86634, -1774.43854, -1688.83889, -1600.98454, -1515.22105, -1435.89401, -1383.22182, -1339.32807, -1302.20921, -1269.86166, -1242.18385, -1214.50944, -1184.07407, -1148.11338, -1090.76646, -1027.60414, -961.10068, -893.730354, -835.115058, -777.722379, -721.167533, -665.065735, -607.620119, -550.422817, -493.653877, -437.493349, -383.360258, -329.700085, -276.19729, -222.536329, -167.664731, -112.298656, -56.4173358, 0., 55.7448171, 112.55891, 170.95477, 231.444888, 291.989405, 356.674102, 427.03241, 504.59776, 583.170483, 675.11035, 785.044031, 917.598198, 1051.62411, 1227.83402, 1461.16475, 1766.55314, 2070.16998, 2511.22455, 3140.16011, 4007.4199, 5080.84751, 6658.68514, 8786.8724, 11925.8417, 15777.5862, 22663.3632, 34249.6139, 52202.7796}, 
{-2019.11349, -1967.86498, -1922.03119, -1879.54631, -1838.34453, -1796.36004, -1751.52704, -1701.77973, -1645.05228, -1560.9017, -1472.99026, -1386.60303, -1307.02508, -1257.73185, -1218.54189, -1187.46414, -1162.50754, -1146.0165, -1129.93027, -1110.52355, -1084.07109, -1029.27314, -967.008674, -900.58221, -833.298258, -778.328451, -725.163331, -673.160561, -621.677803, -567.845774, -514.139863, -460.808512, -408.100162, -357.77398, -307.963395, -258.312561, -208.46563, -157.218748, -105.403279, -53.0025778, 0., 52.510405, 106.100252, 161.230464, 218.361962, 275.485215, 336.519778, 402.914754, 476.119245, 549.893005, 636.450223, 740.315742, 866.014403, 993.013679, 1160.91873, 1384.27734, 1677.6373, 1964.76951, 2387.3094, 2996.11552, 3842.0464, 4892.15254, 6448.71667, 8571.55319, 11686.9318, 15534.6254, 22233.9388, 33309.2935, 50285.1113}, 
{-1852.906, -1799.92433, -1755.7931, -1717.7438, -1683.00798, -1648.81714, -1612.4028, -1570.99649, -1521.82971, -1440.70191, -1354.84952, -1270.0769, -1192.18842, -1146.35433, -1111.26673, -1084.98361, -1065.56296, -1056.86211, -1048.81999, -1037.17485, -1017.66497, -965.979445, -905.925352, -841.260613, -775.743151, -724.511292, -675.390392, -627.586209, -580.304504, -530.041543, -479.796376, -429.858557, -380.517643, -333.704815, -287.411353, -241.270163, -194.914152, -147.113112, -98.7083065, -49.6778856, 0., 49.3678455, 99.8185964, 151.765844, 205.623179, 259.438588, 316.93751, 379.479777, 448.425223, 517.590152, 598.895338, 696.718027, 815.435464, 935.025994, 1094.02532, 1306.57025, 1586.79759, 1856.40165, 2258.93873, 2845.52263, 3667.26715, 4690.50381, 6220.69328, 8329.94908, 11413.3409, 15245.1024, 21759.8548, 32352.1221, 48416.4288}, 
{-1707.35118, -1653.0484, -1609.52501, -1573.71394, -1542.54814, -1512.96053, -1481.88405, -1446.25164, -1402.99624, -1326.96932, -1246.01786, -1165.90739, -1092.40343, -1050.37625, -1018.84473, -995.932496, -979.763174, -974.240986, -969.396721, -961.041762, -944.987495, -897.385648, -841.571127, -781.219183, -720.005065, -672.775514, -627.56569, -583.582244, -540.031828, -493.404962, -446.710883, -400.242696, -354.293506, -310.740658, -267.659322, -224.708907, -181.548824, -137.082706, -92.0280493, -46.3465739, 0., 46.2075789, 93.4937656, 142.233789, 192.802879, 243.351361, 297.36933, 356.121975, 420.874489, 485.626378, 561.814791, 653.61119, 765.18704, 876.888988, 1026.24324, 1226.95119, 1492.71423, 1743.67573, 2124.5183, 2686.36654, 3480.34503, 4472.24266, 5969.19114, 8053.42895, 11092.3504, 14891.8235, 21218.6652, 31349.4551, 46560.7729}, 
{-1584.39249, -1529.62886, -1485.55795, -1449.34221, -1418.14411, -1389.1261, -1359.45065, -1326.28022, -1286.77728, -1218.43376, -1145.95085, -1074.35921, -1008.68952, -971.084549, -942.618, -921.475684, -905.843417, -897.67345, -889.878586, -879.138066, -862.131131, -819.992009, -771.162956, -718.541216, -665.024034, -622.233049, -580.851352, -540.286431, -499.94577, -457.133564, -414.20191, -371.399611, -328.975471, -288.469407, -248.322665, -208.267601, -168.036574, -126.862414, -85.1768186, -42.9119574, 0., 42.9200457, 86.9055837, 132.30718, 179.4754, 226.72537, 277.257271, 332.235845, 392.825836, 453.366138, 524.577678, 610.355535, 714.594789, 817.830593, 956.661925, 1144.32784, 1394.06738, 1625.20104, 1982.45386, 2516.63232, 3278.54291, 4233.71041, 5688.78638, 7733.36167, 10711.2418, 14457.5951, 20587.924, 30272.6475, 44682.1844}, 
{-1485.97338, -1432.0574, -1386.22294, -1346.5141, -1310.975, -1277.64975, -1244.58246, -1209.81724, -1171.39821, -1113.82504, -1054.10405, -995.697115, -942.066106, -909.766207, -883.928673, -862.778061, -844.53893, -826.679817, -808.483705, -788.47756, -765.188346, -730.298793, -691.917793, -651.310007, -609.740092, -571.99583, -534.409508, -496.836536, -459.132322, -420.424886, -381.587985, -342.767985, -304.111253, -266.47896, -229.016746, -191.585055, -154.044331, -116.18712, -77.9689255, -39.2773507, 0., 39.3956861, 79.8338744, 121.658895, 165.21508, 209.062449, 256.043366, 307.215884, 363.638057, 420.173885, 486.553096, 566.311367, 662.984373, 757.078737, 884.370809, 1057.60788, 1289.53726, 1499.58688, 1831.15114, 2334.30506, 3059.12368, 3971.24838, 5374.05517, 7361.11611, 10257.2967, 13925.2237, 19845.1853, 29093.0545, 42744.7041}, 
{-1376.76382, -1326.68783, -1282.07861, -1241.75639, -1204.5414, -1169.25385, -1134.71398, -1099.742, -1063.15815, -1015.4217, -967.058202, -920.232267, -877.1085, -849.472809, -826.019977, -805.066087, -784.927224, -759.752074, -733.691079, -706.727285, -678.843735, -650.016673, -620.238666, -589.495479, -557.772878, -524.522938, -490.478592, -455.839083, -420.803651, -385.914211, -350.890265, -315.793989, -280.687555, -245.903895, -211.126125, -176.308115, -141.403738, -106.570871, -71.4777782, -35.9967311, 0., 36.0990347, 73.185657, 111.604041, 151.698359, 192.269829, 235.822765, 283.318523, 335.718461, 388.430658, 450.191058, 524.182331, 613.587143, 698.997506, 815.223007, 974.482576, 1188.99514, 1378.73642, 1685.06585, 2157.09963, 2843.95399, 3710.28866, 5058.5893, 6988.85511, 9802.94449, 13373.0307, 19088.1684, 27958.192, 40992.9357}, 
{-1237.58523, -1193.72216, -1155.05249, -1120.43609, -1088.73286, -1058.80268, -1029.50544, -999.701022, -968.249314, -926.239712, -883.41079, -841.730634, -803.167327, -778.318849, -757.071429, -737.941191, -719.444261, -696.334781, -672.39565, -647.647786, -622.112106, -595.968306, -569.015013, -541.209634, -512.509574, -482.320899, -451.372892, -419.843492, -387.910642, -356.060285, -324.03916, -291.902006, -259.703565, -227.751645, -195.74669, -163.642216, -131.391734, -99.0699117, -66.4606474, -33.4689929, 0., 33.4513572, 67.8059178, 103.394599, 140.548317, 178.214416, 218.660816, 262.771863, 311.431906, 360.448557, 417.813589, 486.442042, 569.248956, 647.275805, 754.06062, 901.267868, 1100.56201, 1272.16848, 1555.76639, 1999.59584, 2651.8969, 3474.58878, 4774.87422, 6664.71547, 9415.52636, 12887.0161, 18426.2608, 27008.4947, 39608.9522}, 
{-1105.58134, -1068.28477, -1035.7158, -1006.80631, -980.488224, -955.693435, -931.35385, -906.401371, -879.767902, -843.331603, -805.899619, -769.225353, -735.062207, -712.723629, -693.378956, -675.757571, -658.588856, -637.292406, -615.231308, -592.458861, -569.028363, -545.287037, -520.876688, -495.733044, -469.791833, -442.442738, -414.385952, -385.775624, -356.765902, -327.780257, -298.595783, -269.258899, -239.816024, -210.549023, -181.174687, -151.645255, -121.912965, -91.969331, -61.7116068, -31.0763203, 0., 30.9521313, 62.7237173, 95.6297075, 129.985051, 164.863675, 202.31796, 243.159265, 288.198947, 333.654277, 386.768336, 450.190117, 526.568614, 597.431846, 694.998168, 830.364963, 1014.62961, 1168.42203, 1429.49406, 1845.13006, 2462.6144, 3241.06015, 4492.26561, 6339.07397, 9025.31546, 12395.8862, 17770.4317, 26099.91, 38335.2793}, 
{-981.843359, -951.135893, -924.577433, -901.1915, -880.001617, -860.031308, -840.304093, -819.843497, -797.673041, -766.547528, -734.266689, -702.361533, -672.363072, -652.260502, -634.543371, -618.159416, -602.056372, -582.353316, -561.958105, -540.949937, -519.408011, -497.806608, -475.671811, -452.924788, -429.486703, -404.761057, -379.393749, -353.513012, -327.24708, -300.95003, -274.433913, -247.736624, -220.896058, -194.167337, -167.28424, -140.19777, -112.858933, -85.1877245, -57.178561, -28.7948502, 0., 28.5987857, 57.9394136, 88.3159941, 120.022638, 152.2321, 186.808389, 224.494156, 266.032056, 308.046801, 357.046158, 415.419955, 485.558019, 549.519157, 638.156626, 761.992664, 931.549505, 1067.95242, 1306.87941, 1694.61147, 2277.42964, 3011.70205, 4213.44837, 6014.88302, 8636.06863, 11907.0428, 17130.0951, 25239.4523, 37169.3408}, 
{-867.462487, -843.035749, -822.146291, -803.916104, -787.467177, -771.921499, -756.401059, -740.027848, -721.923855, -695.737645, -668.254002, -640.784284, -614.639853, -596.502818, -580.165486, -564.790917, -549.542169, -531.245875, -512.336089, -492.910439, -473.066553, -453.360756, -433.248503, -412.643943, -391.461228, -369.148461, -346.272259, -322.93319, -299.231825, -275.445509, -251.427325, -227.207134, -202.814794, -178.477895, -153.949475, -129.180301, -104.121142, -78.6436884, -52.8094149, -26.6007194, 0., 26.3887492, 53.4533645, 81.4600863, 110.675155, 140.334184, 172.146291, 206.789968, 244.943705, 283.625111, 328.637912, 382.124952, 446.229074, 503.591266, 583.656969, 696.36977, 851.673255, 971.215005, 1188.55301, 1548.94926, 2097.66575, 2788.51376, 3941.10738, 5695.09497, 8251.54269, 11427.8877, 16514.6653, 24434.1361, 36108.5609}, 
{-763.529923, -744.744569, -728.931269, -715.304576, -703.079038, -691.469208, -679.689636, -666.954874, -652.479471, -630.752109, -607.603558, -584.138715, -561.462481, -545.02393, -529.846115, -515.296264, -500.741608, -483.698449, -466.12531, -448.129792, -429.819492, -411.783222, -393.454883, -374.749589, -355.582454, -335.477555, -314.897458, -293.913693, -272.597788, -251.142599, -229.449798, -207.542384, -185.443357, -163.352002, -141.044517, -118.473386, -95.5910942, -72.2558187, -48.5520731, -24.4700646, 0., 24.3194502, 49.2659279, 75.0686116, 101.95668, 129.18442, 158.345858, 190.060128, 224.946366, 260.38819, 301.534457, 350.298506, 408.59368, 459.701698, 531.62017, 633.715083, 775.352427, 878.665118, 1075.14545, 1409.05263, 1924.64589, 2573.49455, 3677.92749, 5382.66222, 7875.49447, 10965.8228, 15933.5562, 23690.976, 35150.3634}, 
{-671.136869, -657.022582, -645.441263, -635.681364, -627.031337, -618.779636, -610.214713, -600.625022, -589.299015, -571.441079, -552.057359, -532.069936, -512.400888, -497.39719, -483.186069, -469.319648, -455.35005, -439.439403, -423.08582, -406.397423, -389.482331, -372.907744, -356.139071, -339.100802, -321.717425, -303.620942, -285.145324, -266.332054, -247.222617, -227.917203, -208.375107, -188.61433, -168.652875, -148.660967, -128.443493, -107.957565, -87.1602927, -65.9427114, -44.3544406, -22.3790227, 0., 22.3883175, 45.3774618, 69.1481977, 93.8812897, 118.797302, 145.421279, 174.318067, 206.052511, 238.33502, 275.726649, 319.934016, 372.66374, 417.903983, 482.167203, 574.247404, 702.938588, 790.758109, 967.287275, 1275.83075, 1759.69318, 2368.6437, 3426.5936, 5080.53715, 7511.68081, 10528.25, 15396.182, 23016.9864, 34292.1725}, 
{-591.374525, -580.630018, -572.185168, -565.370918, -559.518208, -553.957981, -548.021179, -541.038742, -532.341613, -517.654709, -501.357407, -484.223055, -467.025006, -453.195949, -439.786162, -426.505259, -413.062857, -398.197102, -382.977668, -367.502757, -351.870573, -336.568062, -321.149189, -305.556662, -289.733187, -273.451226, -256.891831, -240.065808, -222.983963, -205.645227, -188.07703, -170.294927, -152.314472, -134.276096, -116.020528, -97.5133744, -78.7202402, -59.6229627, -40.164422, -20.3037303, 0., 20.5927796, 41.7883242, 63.7054721, 86.463062, 109.187321, 133.386744, 159.577213, 188.274612, 217.464584, 251.205346, 291.024877, 338.451155, 378.251645, 435.419044, 518.185534, 634.783301, 707.949324, 865.609071, 1150.19281, 1604.13079, 2175.96048, 3189.79057, 4791.67214, 7163.85853, 10122.571, 14911.9569, 22419.182, 33531.4121}, 
{-531.224895, -520.626258, -512.797608, -506.999762, -502.493538, -498.539755, -494.399231, -489.332783, -482.601229, -469.841541, -455.387921, -439.950727, -424.240315, -411.213282, -398.435247, -385.718072, -372.873616, -359.085644, -345.04535, -330.815832, -316.46019, -302.247325, -287.952211, -273.555623, -259.038339, -244.396642, -229.589599, -214.591784, -199.377772, -183.808107, -168.017004, -152.02465, -135.851231, -119.609428, -103.189935, -86.5759385, -69.7506277, -52.91163, -35.741918, -18.1389039, 0., 18.9692432, 38.6028073, 58.9265352, 79.9662703, 100.621877, 122.495568, 146.063579, 171.802143, 197.878892, 228.002105, 263.571461, 305.986636, 341.038107, 391.978429, 466.450961, 572.099059, 631.266927, 771.016734, 1033.1115, 1459.31424, 1998.35806, 2971.09571, 4515.64123, 6831.24812, 9763.79368, 14511.5205, 21934.5798, 32893.1229}, 
{-486.583679, -473.979814, -465.030685, -458.861989, -454.599418, -451.368669, -448.295435, -444.505412, -439.124294, -427.323975, -413.765471, -399.155998, -384.202771, -371.710721, -359.450261, -347.28952, -335.096629, -322.375158, -309.503618, -296.495961, -283.36614, -270.088482, -256.732415, -243.32774, -229.904261, -216.721713, -203.487992, -190.140926, -176.618344, -162.629655, -148.432476, -134.056005, -119.529439, -104.937331, -90.2313794, -75.4186386, -60.5061629, -45.9950208, -31.2006471, -15.9324906, 0., 17.4807597, 35.7296022, 54.6597253, 74.1843266, 92.8924074, 112.55104, 133.603101, 156.491467, 179.486059, 206.071889, 237.561016, 275.265497, 306.105316, 351.541434, 418.642738, 514.478116, 560.661054, 683.898001, 925.440002, 1326.53811, 1837.0297, 2772.40681, 4258.3764, 6521.60732, 9452.61168, 14185.5666, 21543.733, 32350.3718}, 
{-448.814905, -434.351947, -424.23195, -417.480755, -413.1242, -410.188124, -407.698365, -404.680764, -400.161159, -388.964091, -375.997217, -361.966893, -347.579478, -335.550049, -323.772757, -312.150471, -300.586061, -288.808322, -276.963829, -265.025081, -252.964578, -240.504183, -227.967286, -215.426643, -202.955006, -191.03281, -179.162057, -167.252429, -155.213609, -142.637141, -129.878101, -116.973426, -103.960054, -90.8894741, -77.7782523, -64.6575056, -51.5583513, -39.2929482, -26.798955, -13.7950721, 0., 16.0603954, 32.9974454, 50.6153159, 68.7181726, 85.5958004, 103.172498, 121.858185, 142.062778, 162.114943, 185.338352, 212.975426, 246.268582, 273.110677, 313.433519, 373.819356, 460.850433, 495.64119, 604.4288, 827.982629, 1207.07205, 1692.46558, 2594.93511, 4028.40848, 6246.18361, 9183.8677, 13908.4607, 21204.1145, 31854.9808}, 
{-415.107526, -399.828275, -389.110135, -381.960008, -377.384796, -374.391399, -371.986718, -369.177655, -364.971111, -354.165601, -341.659766, -328.143861, -314.308141, -302.780952, -291.539221, -280.497968, -269.57221, -258.587577, -247.584233, -236.512952, -225.324507, -213.595855, -201.801116, -190.040593, -178.414589, -167.532131, -156.781306, -146.058926, -135.261805, -123.926816, -112.454684, -100.886198, -89.2621454, -77.5906511, -65.9582303, -54.4187358, -43.0260202, -32.8911881, -22.5879395, -11.7472262, 0., 14.6788678, 30.3399124, 46.6897497, 63.4349954, 78.6160763, 94.2722731, 110.776677, 128.50238, 145.787851, 165.854652, 189.889726, 219.080012, 241.990545, 277.478935, 331.780884, 411.132096, 436.62174, 533.690667, 742.633193, 1103.74364, 1567.96956, 2443.64553, 3835.9396, 6019.38928, 8970.19375, 13688.5361, 20918.6033, 31404.5825}, 
{-381.404703, -367.810516, -358.064516, -351.312694, -346.701042, -343.375552, -340.482213, -337.167018, -332.575957, -322.140584, -310.207104, -297.407282, -284.372885, -273.56541, -263.055001, -252.741533, -242.524879, -232.132597, -221.705805, -211.213306, -200.6239, -189.585152, -178.515596, -167.512531, -156.673252, -146.561658, -136.621805, -126.76435, -116.899948, -106.625098, -96.2902787, -85.9318111, -75.5860158, -65.1968036, -54.9298686, -44.858495, -35.0559672, -26.8956431, -18.630704, -9.81440464, 0., 13.2974927, 27.6712859, 42.7528299, 58.1735747, 71.8319533, 85.7856399, 100.359291, 115.877565, 130.624358, 147.781391, 168.489624, 193.890017, 212.684812, 243.429178, 292.239562, 365.232414, 384.336086, 473.530361, 672.606928, 1021.35747, 1469.13545, 2327.04725, 3698.40244, 5866.54264, 8835.96394, 13546.5125, 20704.1969, 31015.0258}, 
{-350.693994, -339.567532, -331.244199, -325.066325, -320.376238, -316.516268, -312.828743, -308.655992, -303.340345, -293.184706, -281.786597, -269.704117, -257.495364, -247.430419, -237.670605, -228.089227, -218.55959, -208.634708, -198.636295, -188.565772, -178.424563, -168.013895, -157.615462, -147.310764, -137.1813, -127.67297, -118.357114, -109.169472, -100.045784, -90.6878242, -81.3588831, -72.0882854, -62.9053565, -53.6877426, -44.6771193, -35.9634834, -27.6368316, -21.2984096, -14.9224656, -7.99449665, 0., 11.9307478, 25.0149408, 38.8249936, 52.933321, 65.1521334, 77.5181313, 90.3078109, 103.797669, 116.194025, 130.671622, 148.335027, 170.288808, 185.113374, 211.447112, 255.404249, 323.099016, 337.713944, 421.467639, 613.647007, 953.538957, 1388.60832, 2233.60823, 3592.34408, 5751.29326, 8739.15611, 13433.0691, 20498.0598, 30599.1557}, 
{-322.965118, -314.722469, -308.053627, -302.529432, -297.72072, -293.198329, -288.533096, -283.29586, -277.057457, -267.132496, -256.250535, -244.884904, -233.50893, -224.173201, -215.142885, -206.260408, -197.368196, -187.807977, -178.123154, -168.356433, -158.550519, -148.713216, -138.936092, -129.275811, -119.789041, -110.754011, -101.917197, -93.246636, -84.7103687, -76.143424, -67.7000552, -59.4015055, -51.2690183, -43.1182337, -35.2582393, -27.7925197, -20.8245593, -16.1400159, -11.4873314, -6.29712101, 0., 10.5704871, 22.3478263, 34.8625739, 47.6452865, 58.4725586, 69.3304936, 80.4512328, 92.0669177, 102.300478, 114.336951, 129.252163, 148.121941, 159.204636, 181.520537, 221.27246, 284.663221, 296.452606, 376.86367, 564.67644, 898.670942, 1324.54377, 2160.32525, 3511.71103, 5663.53469, 8666.0039, 13328.6919, 20271.3306, 30113.6518}, 
{-298.207795, -292.898473, -287.897243, -283.010547, -278.044824, -272.806515, -267.102061, -260.737901, -253.520476, -243.818486, -233.451208, -222.800179, -212.246936, -203.590978, -195.228694, -186.974434, -178.64255, -169.366469, -159.913835, -150.371369, -140.825792, -131.514249, -122.312866, -113.248194, -104.346783, -95.6927269, -87.2320163, -78.9681837, -70.9047623, -63.0203273, -55.353353, -47.9173557, -40.7258518, -33.5430427, -26.7314855, -20.4044225, -14.675096, -11.4609901, -8.34940844, -4.73189658, 0., 9.2085648, 19.6468917, 30.8219035, 42.2405231, 51.6891709, 61.0834733, 70.6185542, 80.4895374, 88.7473442, 98.5889822, 111.067257, 127.234972, 134.887001, 153.637255, 189.841712, 249.85635, 260.249361, 339.079625, 524.618236, 855.136285, 1275.09739, 2104.19508, 3450.4498, 5593.16045, 8602.74098, 13213.8667, 19995.1477, 29515.1939}, 
{-277.692073, -273.266665, -268.661563, -263.791764, -258.572268, -252.918071, -246.744171, -239.965567, -232.497256, -223.318657, -213.654579, -203.794253, -194.026909, -185.767148, -177.728681, -169.75059, -161.671955, -152.674565, -143.517713, -134.303398, -125.133621, -116.351293, -107.721136, -99.2487811, -90.9398631, -82.7556789, -74.7639327, -66.9879926, -59.4512265, -52.2008259, -45.2268059, -38.543005, -32.163262, -25.856573, -19.9795559, -14.6439865, -9.96164035, -7.81861146, -5.84262963, -3.43574307, 0., 7.74471995, 16.6413325, 26.2149221, 35.990573, 43.8290239, 51.5854432, 59.4506533, 67.6154769, 74.1745522, 82.25336, 92.8811966, 107.087358, 112.157665, 128.36228, 162.227891, 220.281186, 230.228588, 308.945153, 494.485671, 824.904937, 1246.65773, 2072.59887, 3408.55568, 5528.94543, 8536.44835, 13052.1028, 19567.3577, 28573.6617}, 
{-259.939683, -255.968472, -251.491441, -246.505072, -241.005847, -234.990248, -228.454756, -221.395854, -213.810024, -205.1907, -196.23863, -187.151517, -178.127059, -170.170813, -162.349484, -154.537632, -146.609818, -137.847271, -128.955213, -120.045537, -111.230135, -102.937193, -94.8357926, -86.9113071, -79.1491114, -71.3967003, -63.8324798, -56.4969763, -49.4307162, -42.7343354, -36.3642068, -30.3368131, -24.6686367, -19.1508851, -14.1154261, -9.66885215, -5.91775591, -4.66091588, -3.63586439, -2.27231968, 0., 6.27642281, 13.5921656, 21.5074915, 29.5826637, 35.7843028, 41.9037718, 48.1387909, 54.6870805, 59.7199925, 66.272163, 75.3518594, 87.9673494, 90.8661607, 105.021597, 137.146221, 193.952597, 204.451318, 284.137707, 470.805114, 802.246893, 1227.94069, 2050.62697, 3373.0398, 5462.06578, 8452.38814, 12843.3511, 19043.1335, 27459.9144}, 
{-244.613452, -240.742451, -236.178876, -230.976389, -225.188652, -218.869326, -212.072073, -204.850553, -197.258429, -189.207498, -180.950031, -172.596435, -164.257118, -156.526127, -148.836772, -141.106004, -133.250774, -124.687935, -116.034573, -107.407678, -98.9242389, -91.0767404, -83.4564783, -76.0302437, -68.7648278, -61.4070679, -54.2316906, -47.2934683, -40.6471739, -34.4312784, -28.5833766, -23.1247619, -18.0767276, -13.2663693, -8.98685713, -5.33716324, -2.41625999, -1.89649508, -1.67411532, -1.2187429, 0., 4.81904605, 10.5395963, 16.7694066, 23.1162328, 27.6727934, 32.1678966, 36.8153135, 41.8288147, 45.4844063, 50.7087301, 58.4906628, 69.8190813, 70.9233303, 83.4556316, 114.308675, 170.37515, 182.25048, 263.643527, 451.965888, 784.629157, 1214.99982, 2032.62481, 3336.30428, 5382.70068, 8336.57764, 12572.4318, 18412.3265, 26178.3252}, 
{-231.376208, -227.327158, -222.515866, -217.031636, -210.963774, -204.401586, -197.434378, -190.151455, -182.642124, -175.141937, -167.535452, -159.853477, -152.126817, -144.557247, -136.936217, -129.226146, -121.389454, -112.999904, -104.564031, -96.1997155, -88.0248379, -80.5747267, -73.3828348, -66.400063, -59.5773124, -52.5780585, -45.755598, -39.1758021, -32.9045422, -27.1020775, -21.7021359, -16.7328335, -12.2222863, -8.04341573, -4.44161024, -1.50706368, 0.670030087, 0.566059177, 0.0976149603, -0.252129215, 0., 3.38796234, 7.52383021, 12.0704624, 16.6907177, 19.6122814, 22.5072553, 25.6125674, 29.1651461, 31.5685348, 35.6264003, 42.3090248, 52.5866904, 52.2400158, 63.5048119, 93.4272263, 149.053406, 162.959, 246.448854, 436.357316, 769.518734, 1203.88868, 2012.93783, 3290.75124, 5281.02929, 8175.03416, 12224.1654, 17664.7881, 24733.2674}, 
{-219.890779, -215.461152, -210.294408, -204.496731, -198.174303, -191.433308, -184.37993, -177.120352, -169.760758, -162.766899, -155.741564, -148.647109, -141.445889, -133.988329, -126.393489, -118.6685, -110.820491, -102.586526, -94.3518247, -86.2315432, -78.3408367, -71.2359444, -64.414504, -57.8152372, -51.3768655, -44.7009486, -38.1982351, -31.9423113, -26.0067637, -20.5571553, -15.5383054, -10.9870099, -6.94006466, -3.32241444, -0.327446578, 1.96330266, 3.46829701, 2.81815514, 1.73432386, 0.650404896, 0., 1.99854432, 4.58507283, 7.48045392, 10.405556, 11.7205528, 13.0512852, 14.6628993, 16.8205414, 18.073119, 21.0885124, 26.8183632, 36.2143128, 34.727059, 45.009565, 74.2138497, 129.491933, 145.909806, 231.539927, 422.368724, 754.382627, 1190.66081, 1985.91147, 3228.78282, 5147.23075, 7953.77503, 11783.3722, 16790.3694, 23129.1142}, 
{-226.724899, -212.764992, -200.613432, -190.033165, -180.787138, -172.638298, -165.34959, -158.683961, -152.404358, -145.81283, -139.317579, -132.865911, -126.40513, -119.900384, -113.273999, -106.466146, -99.4169927, -91.5262098, -83.4906648, -75.4667261, -67.6107623, -60.4573783, -53.6334118, -47.1439369, -40.9940276, -35.2433287, -29.8205152, -24.7088327, -19.8915271, -15.2693682, -10.9410679, -6.92286233, -3.23098758, 0.61081133, 3.89681083, 6.41377843, 7.9484816, 6.40498557, 4.20584103, 1.89089637, 0., 1.36847131, 3.32249934, 5.4837441, 7.4738656, 7.10299269, 6.52892902, 6.09794707, 6.15631933, 4.78894311, 5.50801613, 9.56436095, 18.2088001, 19.6826667, 33.4500685, 65.965624, 123.683951, 138.560516, 221.348751, 408.302935, 735.677349, 1179.71831, 1956.70398, 3138.69873, 4942.56472, 7595.34243, 11138.5612, 15660.1283, 21247.9508}, 
{-236.84171, -211.295632, -190.740811, -174.495373, -161.877439, -152.205134, -144.796581, -138.969903, -134.043225, -127.670992, -121.500477, -115.515274, -109.698978, -104.443078, -99.1601186, -93.6705368, -87.7947709, -80.3881138, -72.6222059, -64.7035429, -56.8386203, -49.4919717, -42.5088397, -35.9925048, -30.0462478, -25.3821555, -21.2511794, -17.5130774, -14.0276074, -10.327812, -6.73085042, -3.2271669, 0.192794552, 4.54191817, 8.42510028, 11.4505655, 13.2265385, 10.5952775, 7.03736001, 3.26739713, 0., 1.10351971, 2.87733143, 4.77455029, 6.24829142, 4.25773493, 1.74750499, -0.831709239, -3.02921862, -7.34619724, -9.19934742, -6.95723472, 1.01157529, 6.82986723, 25.4411853, 62.2804239, 122.782477, 135.652615, 213.747207, 393.192996, 710.116727, 1156.14026, 1900.90499, 2996.52076, 4644.78844, 7069.78957, 10256.7919, 14250.7667, 19096.6851}, 
{-232.280967, -202.805662, -179.493136, -161.488734, -147.937801, -137.985681, -130.77772, -125.459263, -121.175653, -114.86966, -108.770235, -102.903752, -97.2965875, -92.6305548, -88.0144139, -83.2123638, -77.9886036, -70.9785561, -63.5267075, -55.8487674, -48.1604456, -40.8346679, -33.8670415, -27.4103897, -21.6175357, -17.5081853, -14.0215263, -10.963629, -8.14056397, -4.8928611, -1.67834712, 1.51069153, 4.68196843, 9.02829042, 12.8982405, 15.8254948, 17.3437298, 13.841766, 9.25607751, 4.37828269, 0., 0.430831866, 1.53721913, 2.70358662, 3.31435919, -0.0847916974, -4.13518649, -8.31689902, -12.1100031, -18.0667268, -21.3661281, -20.2594189, -12.9978114, -5.65089228, 14.4758508, 52.2585556, 112.57336, 127.359326, 203.604497, 375.359842, 676.676327, 1108.31316, 1804.19659, 2804.49135, 4284.70754, 6449.15256, 9251.59936, 12708.4545, 16836.1246}, 
{-217.723575, -189.98152, -168.069873, -151.165799, -138.446465, -129.089038, -122.270683, -117.168567, -112.959856, -106.589909, -100.360423, -94.3412868, -88.602391, -83.9929754, -79.4918381, -74.8571281, -69.8469946, -63.1281994, -55.9868336, -48.6176009, -41.2152054, -34.0498118, -27.2104788, -20.8617257, -15.168072, -11.1893742, -7.83668007, -4.91637443, -2.23484203, 0.921668114, 4.01858121, 7.04145819, 9.97586002, 13.921929, 17.3048121, 19.6642379, 20.5399346, 16.3430655, 10.9933504, 5.28194367, 0., -0.573489477, -0.530341535, -0.4745363, -1.0100539, -5.63864982, -10.9074187, -16.2612305, -21.1449552, -27.7808742, -31.7254815, -31.3126825, -24.8763827, -18.1852808, 0.835428433, 36.825757, 94.4257167, 113.608134, 189.54708, 352.749442, 633.722106, 1037.49814, 1671.00588, 2568.45021, 3871.51835, 5752.88672, 8153.79254, 11073.3328, 14510.6046}, 
{-197.850438, -175.509646, -157.670492, -143.67912, -132.881674, -124.624299, -118.253141, -113.114343, -108.55405, -102.012813, -95.5046078, -89.1378169, -83.0208222, -78.0605024, -73.2473444, -68.3703316, -63.2184475, -56.6677061, -49.7852481, -42.7252446, -35.6418668, -28.701748, -22.0416126, -15.8106468, -10.1580371, -5.99367853, -2.40176533, 0.772799518, 3.68511306, 7.00195942, 10.1620738, 13.1158783, 15.8137953, 19.0748347, 21.6333959, 23.0924661, 23.0550322, 18.2977904, 12.3805353, 6.03677068, 0., -1.83334156, -3.15785451, -4.50520788, -6.40707065, -12.1179021, -18.3574648, -24.5683118, -30.1929959, -36.8968677, -41.0105633, -41.0875167, -35.6811617, -31.1989674, -15.1987182, 16.907766, 69.7086655, 94.3265236, 170.201412, 323.307766, 579.620021, 944.956329, 1505.75997, 2294.23711, 3414.41715, 5000.44737, 6994.1804, 9385.54253, 12164.4601}, 
{-177.34246, -162.076478, -149.494462, -139.181246, -130.721668, -123.700563, -117.702768, -112.313118, -107.11645, -100.319447, -93.4363589, -86.6032825, -79.9563146, -74.3632984, -68.935886, -63.5174757, -57.9514658, -51.4277386, -44.704615, -37.8868996, -31.0793971, -24.3548212, -17.8629043, -11.7212873, -6.04761157, -1.48905451, 2.57809353, 6.24800591, 9.61485598, 13.2341966, 16.5542697, 19.4846971, 21.9351002, 24.3390085, 25.8725726, 26.2358508, 25.1289017, 19.9045551, 13.5489893, 6.70115434, 0., -3.27262161, -6.17782377, -9.13381751, -12.5588139, -19.2366111, -26.2735981, -33.1417511, -39.3130462, -45.8229357, -49.9545293, -50.5544126, -46.4691713, -45.1176209, -33.3452256, -6.56967931, 39.791324, 69.4419812, 144.193951, 284.980784, 512.73603, 831.948853, 1312.88596, 1987.69175, 2922.60026, 4211.28985, 5803.57192, 7685.22462, 9842.0261}, 
{-160.880546, -152.368455, -144.741251, -137.82473, -131.444689, -125.426926, -119.597237, -113.78142, -107.805271, -100.690886, -93.3892444, -86.0476237, -78.8133015, -72.4315259, -66.2124162, -60.0640621, -53.8945533, -47.2389595, -40.5275983, -33.8177671, -27.1667634, -20.5733762, -14.1768151, -8.05778129, -2.29697582, 2.75654163, 7.39777206, 11.6533579, 15.5499414, 19.5045633, 22.9973084, 25.8986598, 28.0791008, 29.5664513, 30.0109229, 29.2200637, 27.0014222, 21.3619739, 14.6300691, 7.33348529, 0., -4.81522689, -9.42275327, -14.1057546, -19.1474064, -26.7088395, -34.4440918, -41.8851566, -48.5640273, -54.9673069, -59.290535, -60.683861, -58.2974344, -60.3669104, -53.3227302, -32.6808411, 6.04281005, 38.8819919, 110.151155, 235.714466, 431.436092, 699.736845, 1096.81095, 1654.65389, 2405.26399, 3404.86948, 4612.77606, 6012.52008, 7587.63786}, 
{-117.464876, -123.918365, -127.492752, -128.518553, -127.326281, -124.246452, -119.609581, -113.746181, -106.986767, -100.040866, -92.7083752, -85.1682045, -77.5992633, -70.6372629, -63.8215906, -57.1484355, -50.6139865, -44.2722108, -38.038408, -31.8856556, -25.7870314, -19.5202853, -13.3319539, -7.27324596, -1.3953703, 4.18790751, 9.51295815, 14.5535955, 19.2836336, 24.2243411, 28.5830951, 32.1147276, 34.5740706, 35.1779179, 34.4343549, 32.3134286, 28.7851863, 22.7771943, 15.718973, 7.99756168, 0., -6.61155838, -13.2352339, -19.9940327, -27.010961, -35.9903708, -44.8413841, -53.054469, -60.1200932, -64.6781795, -67.4099591, -68.1461179, -66.7173418, -74.5060328, -75.1704745, -63.9206668, -35.9666093, 0.717528466, 68.7915847, 178.151228, 338.692126, 545.47663, 852.90036, 1304.26458, 1880.44809, 2572.04849, 3393.21069, 4352.19278, 5457.25286}, 
{-136.527966, -139.159539, -139.607421, -138.121857, -134.953088, -130.351358, -124.566911, -117.849989, -110.450837, -102.93602, -95.112929, -87.1052792, -79.0367843, -71.261443, -63.5805711, -56.0257688, -48.6286365, -41.5501343, -34.6407587, -27.880366, -21.2488126, -14.5876696, -8.07039264, -1.7321522, 4.3918813, 10.3326244, 15.962385, 21.2195577, 26.0425374, 30.7830152, 34.8007711, 37.8688812, 39.7604217, 39.674461, 38.1876865, 35.3027776, 31.0224139, 24.5660901, 17.0329442, 8.73892973, 0., -8.04750415, -16.2361718, -24.5782045, -33.0858038, -42.6581978, -52.0657511, -60.9658549, -69.0159002, -75.50637, -80.6083267, -84.1259248, -85.8633187, -94.7693226, -97.845567, -91.2383424, -71.0939391, -40.7141918, 13.7723711, 99.0816769, 221.929653, 380.179382, 607.105325, 934.522007, 1336.34321, 1800.19873, 2316.75508, 2875.91963, 3467.59973}, 
{-194.664249, -181.230941, -169.302868, -158.648225, -149.035212, -140.232023, -132.006856, -124.127907, -116.363375, -107.812862, -99.1805959, -90.502209, -81.8133354, -73.2086007, -64.6410495, -56.1227188, -47.6656453, -39.1592333, -30.7872051, -22.6106505, -14.6906592, -7.17655082, -0.00589349249, 6.79551487, 13.2018764, 19.3079585, 24.9191718, 29.9614923, 34.3608958, 38.1894679, 41.1686312, 43.1659177, 44.0488595, 43.4935761, 41.6355773, 38.4189602, 33.7878219, 26.7947351, 18.6319311, 9.6001171, 0., -9.24386883, -18.7041645, -28.3297179, -38.0693597, -47.9114983, -57.7495558, -67.516532, -77.1454268, -88.0404396, -98.0748906, -106.593299, -112.940186, -120.019097, -122.191915, -117.379548, -103.502907, -87.2876613, -54.3280535, 0.976913021, 84.2282345, 202.753802, 356.973927, 552.751261, 786.612743, 1083.52895, 1364.03162, 1574.7622, 1662.36213}, 
{-175.36164, -166.624748, -158.159801, -149.925768, -141.881618, -133.986319, -126.198841, -118.478152, -110.783222, -103.030633, -95.2386948, -87.3833331, -79.440472, -71.2174429, -62.9261997, -54.6101037, -46.3125163, -38.121428, -30.017719, -22.0268988, -14.1744769, -6.30376836, 1.30464509, 8.55237609, 15.3410373, 21.2894899, 26.6951987, 31.5728771, 35.9372383, 40.4691856, 44.2507664, 47.0302182, 48.5557784, 48.0057096, 45.9262139, 42.2935185, 37.0838506, 29.5326914, 20.6533127, 10.7182403, 0., -10.8380537, -22.0705555, -33.5813111, -45.2541263, -57.0379543, -68.7253946, -80.1741938, -91.2420989, -101.801961, -111.69038, -120.759063, -128.859713, -138.865711, -146.398417, -150.100866, -148.616093, -144.359406, -130.692659, -104.749978, -63.6654913, -2.1174713, 75.3923952, 179.345261, 277.973065, 352.592929, 374.43045, 317.234105, 154.752373}, 
{-191.788584, -182.390037, -173.740918, -165.639879, -157.885566, -150.276631, -142.611723, -134.68949, -126.308584, -116.144987, -105.569081, -94.8285798, -84.1712001, -74.6575672, -65.3973217, -56.3130147, -47.3271972, -37.9729249, -28.7180425, -19.6408993, -10.8198446, -2.39999017, 5.63378214, 13.2298281, 20.3365033, 26.8791912, 32.8384087, 38.1717007, 42.8366119, 47.2220462, 50.6816452, 53.00041, 53.9633413, 52.8498506, 50.1527642, 45.859319, 39.9567519, 31.9102235, 22.4378773, 11.7357803, 0., -12.5765892, -25.7934503, -39.453239, -53.3586108, -66.8576235, -80.38937, -93.9383452, -107.489044, -121.262378, -134.911858, -148.327413, -161.398972, -174.938597, -187.545228, -198.739941, -208.043809, -213.510736, -216.715838, -217.767058, -216.77234, -208.338342, -209.076867, -219.416934, -264.909434, -355.793203, -538.433422, -850.163683, -1328.31758}, 
{-198.71236, -197.308699, -193.101836, -186.510063, -177.951671, -167.844951, -156.608195, -144.659693, -132.417739, -121.285996, -110.303233, -99.4935919, -88.8812147, -78.707498, -68.6924271, -58.7732421, -48.8871829, -38.459504, -28.1442253, -18.0833812, -8.41900594, 0.229792022, 8.38888204, 16.1070594, 23.4331194, 30.88596, 37.8562326, 44.2046916, 49.7920911, 54.7738453, 58.5981843, 61.0079983, 61.7461775, 59.6423142, 55.7179154, 50.0811906, 42.840349, 34.2000096, 24.1334081, 12.7101898, 0., -14.5670711, -30.0260009, -46.0513217, -62.3175661, -77.5333094, -92.7254243, -107.954826, -123.28243, -138.151262, -153.487283, -169.598565, -186.793178, -205.872273, -226.45361, -248.648032, -272.566377, -292.582224, -316.83858, -347.741193, -387.695808, -434.404278, -504.384028, -598.974796, -746.35905, -974.171705, -1285.39517, -1689.17496, -2194.65663}
};

  const double a5tab[50][69] = {
{4296.48547, 4145.14906, 4028.39347, 3938.25617, 3866.77463, 3805.98633, 3747.92872, 3684.63929, 3608.15549, 3470.8602, 3320.30733, 3164.39619, 3011.02609, 2895.53131, 2787.40219, 2683.56406, 2580.94224, 2467.08826, 2352.05074, 2236.50453, 2121.12447, 2008.42576, 1896.50673, 1785.30605, 1674.76241, 1564.89358, 1455.52754, 1346.57134, 1237.93205, 1128.57981, 1019.73336, 911.67453, 804.685139, 699.870801, 596.360042, 494.10517, 393.058497, 293.389367, 194.746242, 96.9946212, 0., -95.8821865, -191.472354, -287.100981, -383.098545, -476.415091, -572.113704, -671.877035, -777.387735, -883.098866, -1000.81451, -1135.10914, -1290.55726, -1445.5122, -1641.25806, -1892.85779, -2215.37433, -2550.46591, -3015.96209, -3656.28769, -4515.86756, -5562.5241, -7070.48939, -8999.14401, -12015.737, -15367.8837, -23273.8659, -38709.2987, -64649.7973}, 
{3989.50138, 3890.74144, 3791.28858, 3691.26149, 3590.77885, 3489.95935, 3388.92168, 3287.78452, 3186.66657, 3088.65256, 2989.7087, 2888.76729, 2784.76057, 2664.7319, 2544.25806, 2427.02688, 2316.72622, 2240.60956, 2169.37286, 2097.27769, 2018.58566, 1902.04814, 1777.64101, 1649.82996, 1523.08064, 1415.81637, 1312.96214, 1213.40058, 1116.01432, 1016.79356, 918.670313, 821.68419, 725.874783, 631.536102, 538.351566, 446.259007, 355.196258, 265.266634, 176.17629, 87.7968658, 0., -86.9254916, -173.692018, -260.594811, -347.929102, -432.365887, -519.274328, -610.399354, -707.48589, -805.717825, -916.025541, -1042.77838, -1190.34568, -1339.01104, -1526.86385, -1767.90776, -2076.14641, -2395.68728, -2838.38865, -3446.21264, -4261.12134, -5252.34273, -6680.04137, -8501.55957, -11361.8428, -14541.9955, -22090.8829, -36865.43, -61722.5616}, 
{3757.56237, 3688.98233, 3602.74083, 3502.53272, 3392.05283, 3274.99599, 3155.05703, 3035.9308, 2921.31211, 2833.32287, 2749.86003, 2667.24761, 2581.80961, 2470.83397, 2357.29521, 2245.13178, 2138.28212, 2062.17793, 1990.6671, 1919.09078, 1842.79012, 1736.93836, 1625.1117, 1510.71846, 1397.16693, 1298.16665, 1202.70421, 1110.06741, 1019.54408, 928.906343, 839.563964, 751.411028, 664.341619, 578.018704, 492.659931, 408.25183, 324.780933, 242.505048, 161.030918, 80.2365616, 0., -79.1939341, -158.316759, -237.733181, -317.807904, -395.541068, -476.007772, -560.918545, -651.983919, -744.762196, -849.577029, -970.599841, -1112.00206, -1254.41911, -1434.97281, -1667.24897, -1964.8334, -2274.39487, -2703.20307, -3291.61063, -4079.9702, -5039.50056, -6417.95592, -8173.42849, -10924.8947, -14014.4248, -21230.8298, -35251.2358, -58752.7686}, 
{3573.43954, 3518.13048, 3441.64113, 3347.99242, 3241.20528, 3125.30065, 3004.29946, 2882.22265, 2763.09114, 2665.78538, 2573.523, 2484.38113, 2396.4369, 2303.02573, 2208.86315, 2113.92298, 2018.17905, 1918.81705, 1819.71417, 1721.95946, 1626.642, 1540.32011, 1456.42589, 1373.86068, 1291.52585, 1204.16451, 1116.49954, 1029.09559, 942.517315, 859.174238, 777.048176, 695.965823, 615.753875, 535.637321, 456.285245, 377.765026, 300.144041, 223.94418, 148.596504, 73.9865859, 0., -72.5271837, -145.040088, -218.033339, -292.001563, -364.645482, -440.371185, -520.79086, -607.516694, -696.21608, -796.823916, -913.330305, -1049.72535, -1186.00008, -1359.74331, -1584.54477, -1873.9942, -2177.10138, -2597.86799, -3175.71572, -3950.0663, -4894.3482, -6245.96286, -7966.33323, -10645.6449, -13703.476, -20613.072, -33834.2617, -55826.8739}, 
{3409.90395, 3356.44466, 3286.88038, 3203.56312, 3108.8449, 3005.07774, 2894.61363, 2779.80461, 2663.00268, 2546.95431, 2433.45928, 2324.71182, 2222.90615, 2150.49542, 2081.31138, 2009.44471, 1928.98609, 1797.5506, 1660.29474, 1525.89938, 1403.04545, 1339.4171, 1289.09066, 1245.14575, 1200.66196, 1126.02999, 1046.0939, 963.008853, 878.930002, 801.853306, 725.756809, 650.459352, 575.779776, 500.710664, 426.227618, 352.479982, 279.6171, 208.423598, 138.159426, 68.7198163, 0., -66.7649103, -133.555516, -201.012536, -269.776693, -338.38397, -410.421719, -487.372554, -570.71909, -656.063581, -753.121147, -865.726549, -997.714547, -1128.01761, -1295.3337, -1513.4585, -1796.18769, -2094.31954, -2509.84608, -3081.76197, -3849.06182, -4787.23624, -6125.79199, -7831.85622, -10464.8454, -13527.4537, -20156.9751, -32582.0535, -53031.3329}, 
{3239.72667, 3182.18361, 3117.34947, 3045.16737, 2965.58039, 2878.53165, 2783.96423, 2681.82123, 2572.04576, 2437.74389, 2302.43055, 2172.78365, 2055.48109, 2002.43127, 1956.9894, 1907.74119, 1843.27233, 1685.40227, 1516.18949, 1350.92621, 1204.90465, 1161.45301, 1140.61312, 1130.46279, 1119.07984, 1055.98318, 983.233088, 904.330944, 822.778121, 751.19961, 680.323725, 610.002396, 540.087553, 469.557447, 399.487158, 330.078085, 261.53163, 194.782874, 129.006064, 64.1091301, 0., -61.7467835, -123.556551, -186.188021, -250.399908, -315.46138, -384.216524, -458.019878, -538.225981, -620.2888, -713.823669, -822.545353, -950.168616, -1074.73535, -1235.90234, -1447.6535, -1723.97275, -2016.56203, -2426.60002, -2992.98342, -3754.60894, -4688.5153, -6019.17312, -7721.57993, -10323.2482, -13404.6625, -19781.9045, -31462.1566, -50452.6013}, 
{3035.67878, 2973.60609, 2911.93932, 2848.72771, 2782.02045, 2709.86678, 2630.31591, 2541.41704, 2441.2194, 2299.06835, 2153.19849, 2013.14059, 1888.4254, 1848.02151, 1818.24671, 1784.85662, 1733.60687, 1569.39574, 1391.17913, 1217.05561, 1065.12374, 1033.65155, 1028.50034, 1035.70092, 1041.28404, 986.24413, 919.662878, 845.585609, 768.05765, 701.469213, 635.382783, 569.705732, 504.34543, 438.496382, 373.063973, 308.240723, 244.21915, 181.861577, 120.422796, 59.8274049, 0., -57.3124734, -114.736706, -173.077041, -233.137821, -294.582554, -359.81275, -430.089086, -506.672238, -584.87584, -674.286427, -778.543494, -901.286533, -1020.41696, -1175.60758, -1380.79311, -1649.90829, -1934.34159, -2335.59247, -2892.61414, -3644.35983, -4568.53596, -5887.83607, -7587.08678, -10161.6055, -13253.4069, -19407.2257, -30442.1168, -48177.1347}, 
{2770.53136, 2708.97085, 2649.54083, 2590.16668, 2528.77379, 2463.28755, 2391.63335, 2311.73658, 2221.52263, 2091.84191, 1958.52477, 1830.32661, 1716.0028, 1676.45437, 1647.43281, 1616.83524, 1572.5588, 1436.55469, 1289.04433, 1144.30323, 1016.60689, 983.236401, 970.259435, 966.74926, 961.779143, 909.032918, 847.129054, 779.296591, 708.764568, 646.91818, 585.567846, 524.68014, 464.221637, 403.846181, 343.958173, 284.649282, 226.011177, 168.499276, 111.696, 55.5475182, 0., -53.3016495, -106.789491, -161.196848, -217.257045, -274.452337, -335.267548, -400.93643, -472.692737, -545.808803, -629.864368, -728.477751, -845.267274, -959.326113, -1108.60779, -1306.54069, -1566.55318, -1838.17091, -2224.28609, -2763.88818, -3495.96666, -4397.64884, -5693.51063, -7379.95924, -9920.66925, -12991.9915, -18952.3044, -29489.4796, -46291.3886}, 
{2417.05548, 2366.53666, 2309.0449, 2245.40682, 2176.44909, 2102.99835, 2025.88123, 1945.9244, 1863.95449, 1776.9788, 1691.17107, 1608.88567, 1532.477, 1476.91808, 1426.8972, 1379.72126, 1332.69721, 1273.9028, 1213.56578, 1152.68474, 1092.25826, 1037.43127, 983.397481, 929.496943, 875.06971, 816.569606, 757.3774, 697.987635, 638.894853, 581.802574, 525.512772, 470.036397, 415.384401, 361.925557, 309.169865, 256.985149, 205.239231, 153.535543, 102.112055, 50.9423473, 0., -49.5539819, -99.4084146, -150.064689, -202.024195, -253.775575, -308.638068, -367.918164, -432.922351, -499.071792, -575.912436, -667.104903, -776.309814, -885.726459, -1029.06132, -1218.55955, -1466.46631, -1718.56271, -2080.14355, -2590.0396, -3287.08162, -4146.20453, -5397.92663, -7051.77975, -9541.19154, -12538.7207, -18336.5059, -28571.7906, -44881.8187}, 
{2185.86243, 2129.66864, 2072.54176, 2014.18321, 1954.2944, 1892.57674, 1828.73164, 1762.46052, 1693.46479, 1613.81027, 1533.88821, 1456.45424, 1384.26402, 1331.09789, 1284.27693, 1242.1469, 1203.05357, 1164.97587, 1126.77315, 1086.93792, 1043.96269, 989.074843, 930.938071, 870.950935, 810.511996, 754.725473, 699.802008, 645.657899, 592.209444, 538.934767, 486.363612, 434.587547, 383.69814, 334.548323, 286.163756, 238.331466, 190.838476, 142.966997, 95.210792, 47.5588098, 0., -46.4100171, -93.1756315, -140.734562, -189.524529, -237.936684, -289.27394, -344.79264, -405.74913, -467.474595, -539.520605, -625.513568, -729.079894, -833.298458, -970.562217, -1152.71659, -1391.60701, -1630.84359, -1977.80117, -2471.61931, -3151.43754, -3991.40953, -5225.63246, -6875.89452, -9345.32147, -12339.851, -17972.8354, -27755.7736, -43200.1643}, 
{1981.4486, 1917.74747, 1860.59169, 1808.1705, 1758.6731, 1710.28871, 1661.20656, 1609.61586, 1553.70582, 1477.58662, 1399.15813, 1322.2412, 1250.65666, 1201.62972, 1160.21509, 1124.87184, 1094.05904, 1069.0927, 1044.43221, 1017.39389, 985.294068, 932.67365, 874.734515, 813.90314, 752.606002, 700.319407, 649.600069, 600.054533, 551.289343, 501.502803, 452.272994, 403.769755, 356.162928, 310.714537, 266.065364, 221.948375, 178.096535, 133.578698, 89.0575891, 44.5318191, 0., -43.5656426, -87.5295569, -132.282577, -178.215537, -223.723063, -271.99068, -324.207705, -381.563456, -439.300701, -506.933925, -588.031063, -686.16005, -784.990737, -915.94838, -1090.56015, -1320.35322, -1546.9207, -1879.69744, -2358.18424, -3021.88189, -3844.23415, -5062.91291, -6711.52012, -9164.32682, -12162.1323, -17638.9538, -26980.504, -41572.4958}, 
{1794.4152, 1724.40808, 1667.3188, 1620.06974, 1579.58327, 1542.78174, 1506.58753, 1467.923, 1423.71051, 1351.59344, 1275.48475, 1200.01839, 1129.82833, 1084.63602, 1047.95293, 1018.37801, 994.510209, 980.512175, 967.19371, 950.928297, 928.089424, 877.880279, 820.71276, 759.828472, 698.469016, 649.593536, 602.839079, 557.560233, 513.111583, 466.653192, 420.611983, 375.220355, 330.710706, 288.654201, 247.408964, 206.671883, 166.13985, 124.746758, 83.2576926, 41.6747436, 0., -40.8980775, -82.2303199, -124.341187, -167.575137, -210.340302, -255.692002, -304.749227, -358.630969, -412.54525, -475.886418, -552.13785, -644.782925, -738.021909, -862.334537, -1028.91743, -1248.96722, -1462.16565, -1779.83017, -2241.76334, -2887.76773, -3690.53634, -4891.20049, -6534.3606, -8967.55178, -11963.4481, -17291.233, -26220.9625, -40022.6926}, 
{1628.55148, 1554.65838, 1498.11445, 1454.99224, 1421.36432, 1393.30328, 1366.88168, 1338.17208, 1303.24707, 1235.66334, 1163.01568, 1090.38301, 1022.84422, 981.368108, 948.787774, 923.826191, 905.206329, 898.844712, 893.393335, 884.697746, 868.603494, 821.350866, 766.232776, 706.936878, 647.150825, 601.764906, 558.783084, 517.41196, 476.858131, 433.670357, 390.776215, 348.44544, 306.947768, 268.010996, 229.863574, 192.192013, 154.682824, 116.244139, 77.6521993, 38.9048655, 0., -38.3123138, -77.0873241, -116.628058, -157.237541, -197.364371, -239.907775, -285.912554, -336.423505, -386.690844, -445.871786, -517.328963, -604.425008, -691.796217, -809.022092, -966.955801, -1176.45051, -1375.54392, -1677.03085, -2120.89065, -2747.10266, -3527.644, -4706.50075, -6337.96158, -8745.4887, -11730.9564, -16913.4172, -25457.6291, -38528.3499}, 
{1487.64668, 1413.5063, 1358.36997, 1318.04925, 1288.35567, 1265.10079, 1244.09613, 1221.15325, 1192.0837, 1129.6289, 1061.89856, 993.932245, 930.769552, 893.077305, 864.016948, 842.37717, 826.946657, 823.700716, 821.366769, 815.858856, 803.091017, 759.741547, 708.65453, 653.438306, 597.701215, 556.050561, 516.696131, 478.846677, 441.710953, 401.838723, 362.161324, 322.951103, 284.480408, 248.428604, 213.098217, 178.198789, 143.439861, 107.843807, 72.0822061, 36.1394671, 0., -35.7133438, -71.9099732, -108.860855, -146.836957, -184.371235, -224.167871, -267.193037, -314.412902, -361.220084, -416.38373, -483.099433, -564.562788, -645.717902, -755.312447, -903.842611, -1101.80458, -1286.02099, -1570.131, -1994.10022, -2597.89427, -3352.88507, -4504.81926, -6115.86863, -8488.62993, -11451.815, -16489.2504, -24670.9838, -37067.063}, 
{1375.49005, 1305.95973, 1253.47674, 1214.35207, 1184.89673, 1161.42172, 1140.23803, 1117.65665, 1089.9886, 1033.32271, 972.280995, 911.263313, 854.669523, 821.014935, 794.937775, 775.19172, 760.530448, 754.690595, 749.449696, 741.568244, 727.806733, 689.708458, 645.337991, 597.542705, 549.16997, 511.667545, 475.842264, 441.101348, 406.852016, 370.442713, 334.162944, 298.243438, 262.914923, 229.550707, 196.781911, 164.382231, 132.125366, 99.3187258, 66.3888102, 33.2958309, 0., -33.0061597, -66.5076708, -100.757245, -136.007596, -170.936864, -208.00216, -248.086024, -292.070995, -335.615571, -386.915951, -448.944294, -524.672755, -599.191206, -700.507005, -838.745224, -1024.03094, -1192.56234, -1457.96212, -1859.92612, -2438.15015, -3163.58747, -4282.16159, -5861.62737, -8187.4678, -11113.1819, -16002.4765, -23841.5066, -35616.4271}, 
{1295.87085, 1237.02659, 1188.82609, 1149.01198, 1115.32691, 1085.51353, 1057.31449, 1028.47242, 996.729976, 946.577353, 894.310623, 842.97341, 795.609339, 766.432322, 742.847581, 723.430622, 706.756955, 691.424756, 675.977799, 658.982526, 639.005378, 607.907736, 573.643128, 537.46002, 500.606877, 467.832902, 435.48553, 403.412934, 371.463285, 338.766752, 306.176712, 273.828538, 241.857605, 211.020988, 180.583677, 150.432364, 120.453741, 90.4418588, 60.4131083, 30.291239, 0., -30.0957535, -60.6898206, -92.0348943, -124.383668, -156.637225, -190.940513, -228.086866, -268.869621, -309.359907, -356.962153, -414.358576, -484.231398, -551.62037, -643.907167, -770.830997, -942.131067, -1094.13344, -1339.35572, -1716.90239, -2265.87789, -2957.07913, -4034.53329, -5568.78338, -7832.49467, -10702.2149, -15436.8397, -22949.6777, -34154.0375}, 
{1204.65587, 1157.00006, 1114.47906, 1076.03051, 1040.59206, 1007.10134, 974.496, 941.713669, 907.69199, 863.90708, 819.742711, 777.121132, 737.964591, 712.778866, 691.469265, 672.524625, 654.433783, 631.935693, 608.769028, 584.922579, 560.385137, 535.126853, 509.162613, 482.488661, 455.101244, 426.520612, 397.409404, 367.954264, 338.341835, 309.081426, 279.907949, 250.878982, 222.052103, 193.704483, 165.586269, 137.667202, 109.917023, 82.4541578, 55.0401876, 27.5853794, 0., -27.3906496, -55.2573507, -83.8558505, -113.441896, -143.126009, -174.767252, -209.079461, -246.776473, -284.355627, -328.433857, -381.411599, -445.689291, -506.351102, -590.040242, -706.083651, -863.808272, -999.956678, -1225.47393, -1578.72072, -2098.05773, -2753.3297, -3788.44518, -5278.67022, -7480.52598, -10279.4618, -14863.9593, -22096.742, -32840.5336}, 
{1074.63988, 1033.0459, 996.321555, 963.439721, 933.373293, 905.095166, 877.57823, 849.795378, 820.719503, 782.391091, 743.488403, 705.757293, 670.943616, 648.479622, 629.350211, 612.226682, 595.780332, 575.305119, 554.200613, 532.489048, 510.192655, 487.464298, 464.143325, 440.199715, 415.603449, 389.833046, 363.546529, 336.910461, 310.091403, 283.54333, 257.03043, 230.6043, 204.316538, 178.421378, 152.686729, 127.083133, 101.581135, 76.2399296, 50.9059493, 25.5142782, 0., -25.256491, -50.9435458, -77.3042049, -104.581509, -131.993253, -161.217822, -192.908352, -227.717982, -262.453961, -303.153673, -352.008612, -411.210273, -466.199332, -542.618428, -649.359384, -795.314019, -917.313637, -1125.13479, -1456.4935, -1949.1058, -2570.59931, -3568.95528, -5028.68341, -7183.5431, -9910.74615, -14364.4451, -21379.5571, -31790.9996}, 
{952.045299, 916.813429, 886.000482, 858.645986, 833.789466, 810.47045, 787.728463, 764.603033, 740.133686, 707.0761, 673.26719, 640.260019, 609.607655, 589.582905, 572.331196, 556.717696, 541.607574, 522.902498, 503.616539, 483.800265, 463.504248, 443.03075, 422.07797, 400.595797, 378.534124, 355.357107, 331.694667, 307.69099, 283.49026, 259.485015, 235.471748, 211.495303, 187.600525, 164.018334, 140.533068, 117.115138, 93.7349577, 70.3914253, 47.0150752, 23.564927, 0., -23.2517332, -46.8877878, -71.1367257, -96.2271088, -121.468367, -148.375847, -177.545764, -209.574332, -241.585969, -279.037406, -323.913578, -378.199419, -427.713909, -497.074318, -594.73196, -729.138153, -837.32558, -1027.73164, -1337.3751, -1803.27474, -2390.79442, -3351.91758, -4779.57717, -6886.99293, -9541.19849, -13873.6851, -20699.8296, -30835.0089}, 
{837.812697, 808.936927, 783.919108, 761.884766, 741.959432, 723.268633, 704.937899, 686.092758, 665.858739, 637.79203, 608.815237, 580.281624, 553.544456, 535.682276, 520.03296, 505.659659, 491.625529, 474.469091, 456.787981, 438.655203, 420.143763, 401.668489, 382.823831, 363.546065, 343.771465, 322.977172, 301.742249, 280.186624, 258.430227, 236.798139, 215.123074, 193.4429, 171.795483, 150.388204, 129.021612, 107.665771, 86.2907419, 64.8428209, 43.325344, 21.7178806, 0., -21.3729548, -43.0874978, -65.3543693, -88.3843094, -111.556974, -136.246622, -162.996426, -192.349561, -221.745105, -256.071965, -297.114955, -346.658886, -390.927315, -453.490813, -542.358694, -665.540274, -760.3361, -933.747761, -1222.06808, -1661.58988, -2215.46107, -3139.40919, -4533.64928, -6593.803, -9176.58984, -13398.9314, -20062.6987, -29969.7628}, 
{732.882625, 710.050697, 690.4807, 673.391519, 658.002041, 643.53115, 629.197732, 614.220674, 597.81886, 574.368803, 549.868711, 525.47442, 502.341765, 486.371293, 472.076243, 458.714566, 445.544212, 429.746158, 413.486118, 396.852833, 379.935045, 363.21979, 346.238194, 328.919679, 311.19367, 292.577619, 273.577705, 254.288137, 234.803126, 215.37436, 195.875577, 176.337996, 156.792834, 137.423841, 118.04869, 98.6375877, 79.1607395, 59.5282925, 39.7945341, 19.9536936, 0., -19.6167344, -39.5400971, -59.9580924, -81.0587245, -102.2647, -124.83544, -149.265068, -176.047706, -202.924822, -234.244259, -271.6012, -316.590832, -355.872032, -411.950817, -492.396897, -604.779981, -686.688787, -843.66641, -1111.27496, -1525.07654, -2046.14531, -2933.5072, -4293.19749, -6306.90082, -8822.69125, -12947.4358, -19473.3033, -29192.4625}, 
{638.19564, 620.789034, 606.088527, 593.401702, 582.036144, 571.299435, 560.499161, 548.942905, 535.938251, 516.636341, 496.163778, 475.490721, 455.58733, 441.243514, 428.081786, 415.544409, 403.073644, 388.474958, 373.482125, 358.192127, 342.701942, 327.526931, 312.178341, 296.585799, 280.678933, 264.042825, 247.089466, 229.886301, 212.500777, 195.105336, 177.620429, 160.071501, 142.483999, 125.018098, 107.510627, 89.933143, 72.2572021, 54.3820161, 36.3804242, 18.2529207, 0., -17.9796505, -36.2430068, -54.9488516, -74.2559679, -93.5971696, -114.147596, -136.356417, -160.672804, -185.118576, -213.541195, -247.360771, -287.997414, -322.580542, -372.537235, -445.003884, -547.116876, -616.727231, -757.970853, -1005.69828, -1394.76004, -1884.39319, -2736.28873, -4060.51959, -6029.21391, -8485.27376, -12526.4503, -18936.7826, -28500.3094}, 
{554.692301, 541.786237, 531.145858, 522.150773, 514.180592, 506.614925, 498.833382, 490.215574, 480.14111, 464.424568, 447.436604, 429.98284, 412.8689, 399.892497, 387.670329, 375.811181, 363.923841, 350.396751, 336.547181, 322.472055, 308.268298, 294.432189, 280.501557, 266.413583, 252.105452, 237.257168, 222.165962, 206.871889, 191.415003, 175.882728, 160.248799, 144.53432, 128.760398, 113.063829, 97.3037508, 81.4549918, 65.4923811, 49.3381679, 33.0407928, 16.5961166, 0., -16.4582818, -33.193648, -50.3276035, -67.9816533, -85.560007, -104.188383, -124.275204, -146.228893, -168.319821, -193.949684, -224.382126, -260.880791, -291.085326, -335.332971, -400.336967, -492.810558, -550.795024, -677.144355, -906.040578, -1271.66572, -1731.75074, -2549.83088, -3837.91334, -5763.66979, -8170.1084, -12143.2269, -18458.2756, -27890.5049}, 
{483.313163, 473.676603, 466.055961, 459.874189, 454.554236, 449.519054, 444.191593, 437.994804, 430.351638, 417.563406, 403.423355, 388.60309, 373.77422, 361.911802, 350.462611, 339.176876, 327.804822, 315.252799, 302.452463, 289.491591, 276.457961, 263.777843, 251.065126, 238.272192, 225.351423, 212.105025, 198.695625, 185.135674, 171.437623, 157.598192, 143.651858, 129.617363, 115.513453, 101.453888, 87.3243877, 73.1056889, 58.7785281, 44.3309239, 29.7334185, 14.9638358, 0., -15.0492067, -30.3894418, -46.0953047, -62.2413945, -78.1588373, -94.9630951, -113.026156, -132.72001, -152.52201, -175.456634, -202.653723, -235.243119, -261.418867, -300.420927, -358.55346, -442.120629, -489.235757, -601.67018, -813.004394, -1156.81889, -1589.764, -2376.21074, -3627.6765, -5513.19596, -7882.96623, -11805.0174, -18042.9214, -27360.2502}, 
{430.484418, 421.113456, 414.158086, 408.98146, 404.94673, 401.417046, 397.755561, 393.325426, 387.489794, 376.483753, 364.049742, 350.802139, 337.355319, 326.256844, 315.414632, 304.669786, 293.863408, 282.296992, 270.567093, 258.73066, 246.844638, 235.147337, 223.441797, 211.712417, 199.9436, 188.128389, 176.239085, 164.256634, 152.161977, 139.83341, 127.395587, 114.870511, 102.280187, 89.7188522, 77.1073842, 64.4388941, 51.7064931, 39.0573716, 26.2689294, 13.2726458, 0., -13.7764105, -27.9026719, -42.3837523, -57.2246196, -71.5863824, -86.6554121, -102.77422, -120.285319, -137.798375, -158.081883, -182.171493, -211.102857, -233.791427, -268.24113, -320.335691, -395.958842, -432.859693, -532.310436, -727.448645, -1051.41189, -1460.81344, -2218.36381, -3429.83, -5277.77804, -7636.36413, -11536.5847, -17720.2261, -26929.0746}, 
{392.308466, 381.186439, 373.266059, 367.790599, 364.003329, 361.147518, 358.466437, 355.203357, 350.601548, 340.48549, 328.88476, 316.410145, 303.672432, 293.084004, 282.733413, 272.510809, 262.30634, 251.700434, 241.016849, 230.269621, 219.472788, 208.611231, 197.739805, 186.884208, 176.07014, 165.51361, 154.973883, 144.400534, 133.743138, 122.753431, 111.657963, 100.485445, 89.2645885, 78.0710907, 66.8678814, 55.664877, 44.4719937, 33.6607917, 22.7348862, 11.559536, 0., -12.6123, -25.6647331, -39.0785057, -52.774824, -65.6832532, -79.1132972, -93.3828187, -108.809681, -124.072008, -141.783297, -162.917303, -188.447785, -208.078692, -238.561512, -285.377927, -354.009618, -381.598833, -469.302458, -649.937946, -956.32275, -1345.70917, -2077.61012, -3248.67972, -5062.9382, -7430.12351, -11329.7366, -17473.8377, -26574.487}, 
{360.66773, 347.89344, 338.935119, 332.947164, 329.083975, 326.499951, 324.349491, 321.786994, 317.966858, 308.406182, 297.351585, 285.41239, 273.197914, 263.040569, 253.137348, 243.408333, 233.77361, 224.009152, 214.236795, 204.434269, 194.579301, 184.439194, 174.28627, 164.182425, 154.189557, 144.710428, 135.329724, 125.972995, 116.565793, 106.761765, 96.8671266, 86.916191, 76.9432708, 67.0016617, 57.0990999, 47.2623045, 37.5179949, 28.466852, 19.3320488, 9.91071997, 0., -11.5081985, -23.54943, -35.9644712, -48.5940986, -60.1461305, -72.0474852, -84.5921225, -98.0740026, -111.204635, -126.493411, -144.867268, -167.253148, -184.019583, -210.875282, -252.97055, -315.455689, -335.026177, -412.669073, -580.916609, -872.30102, -1244.6194, -1954.6094, -3090.28309, -4876.46181, -7258.87633, -11162.8104, -17268.6597, -26256.8197}, 
{332.848826, 319.339406, 309.84285, 303.496527, 299.437804, 296.804049, 294.732628, 292.360909, 288.82626, 279.6239, 268.990204, 257.519397, 245.805708, 236.102767, 226.681634, 217.472775, 208.406655, 199.343792, 190.312576, 181.27145, 172.178856, 162.676833, 153.16679, 143.733731, 134.462662, 125.865105, 117.42894, 109.068565, 100.698377, 91.9270899, 83.0970604, 74.2449607, 65.4074631, 56.6069349, 47.9000748, 39.3292765, 30.936934, 23.542257, 16.1000972, 8.34212235, 0., -10.4410908, -21.5043985, -32.9596599, -44.576612, -54.88014, -65.3827734, -76.3521899, -88.0560672, -99.2001149, -112.238766, -128.064485, -147.569738, -161.556655, -185.044169, -222.960877, -280.235378, -293.441953, -363.205243, -521.795574, -801.483269, -1160.10306, -1853.23206, -2962.50825, -4729.38986, -7132.2067, -11041.982, -17106.5658, -25973.8078}, 
{304.952152, 292.952218, 284.33093, 278.349426, 274.268842, 271.350315, 268.854983, 266.043982, 262.17845, 253.319746, 243.208695, 232.386348, 221.393753, 212.329699, 203.554397, 194.985801, 186.541861, 178.001582, 169.477443, 160.942978, 152.371717, 143.465933, 134.578921, 125.792715, 117.189349, 109.241916, 101.484971, 93.8441289, 86.2450012, 78.3468177, 70.4481283, 62.5810991, 54.7778966, 47.0124833, 39.3985105, 31.9914262, 24.8466778, 18.9719631, 13.08958, 6.87407634, 0., -9.38029515, -19.4611898, -29.9592584, -40.5910756, -49.7813155, -59.0552134, -68.6461041, -78.7873225, -88.1276716, -99.1188308, -112.627947, -129.522169, -140.631068, -160.874397, -195.134334, -248.293057, -257.393244, -322.292372, -475.008418, -747.559361, -1096.57811, -1780.23784, -2878.9862, -4641.40046, -7069.0396, -10983.4062, -17001.0171, -25738.3888}, 
{279.78566, 269.98824, 262.640456, 257.177906, 253.036189, 249.650905, 246.457651, 242.892026, 238.389629, 229.788609, 220.160994, 209.981364, 199.724297, 191.311845, 183.192125, 175.260728, 167.413243, 159.281274, 151.129992, 142.960581, 134.774227, 126.404438, 118.087145, 109.890602, 101.883063, 94.4378202, 87.1960756, 80.1040686, 73.1080386, 65.9558819, 58.8715185, 51.8805254, 45.0084794, 38.1787473, 31.56, 25.2186982, 19.2213027, 14.7377201, 10.2895871, 5.50198627, 0., -8.3384809, -17.4423055, -26.9875147, -36.6501492, -44.7971021, -52.9372211, -61.2702062, -69.9957574, -77.6869199, -86.8207102, -98.2474902, -112.817622, -121.191888, -138.48606, -169.626333, -219.538899, -226.031281, -287.995812, -437.206154, -705.435969, -1047.95037, -1726.04867, -2820.76157, -4583.33267, -7035.48163, -10946.811, -16899.9525, -25477.5381}, 
{257.302813, 250.074573, 244.204199, 239.330028, 235.090398, 231.123647, 227.068111, 222.562129, 217.244038, 208.851392, 199.683625, 190.139389, 180.617334, 172.839114, 165.351176, 158.02297, 150.723945, 142.906536, 135.024016, 127.112641, 119.208668, 111.321731, 103.525358, 95.8664567, 88.3919317, 81.3337627, 74.4797527, 67.8027781, 61.2757154, 54.7583607, 48.3819031, 42.1644511, 36.1241133, 30.1364421, 24.4191248, 19.0472927, 14.0960769, 10.8651988, 7.71536327, 4.23186531, 0., -7.30986874, -15.4315422, -24.013753, -32.7052334, -39.8530662, -46.9282936, -54.100308, -61.5385019, -67.7347666, -75.206996, -84.795583, -97.3409206, -103.184816, -117.865682, -146.423345, -193.897633, -199.101319, -259.792105, -407.500641, -673.757574, -1012.59923, -1688.03923, -2782.74657, -4546.80967, -7020.11695, -10915.9374, -16779.1949, -25154.8133}, 
{237.457073, 232.838319, 228.454933, 224.153853, 219.782019, 215.186369, 210.213843, 204.71138, 198.525917, 190.328997, 181.613115, 172.69537, 163.892858, 156.701414, 149.787905, 142.997937, 136.177112, 128.601035, 120.91331, 113.187537, 105.497321, 98.0471923, 90.727455, 83.5593415, 76.5640841, 69.8106881, 63.253503, 56.8946518, 50.7362571, 44.7583326, 38.993954, 33.4540878, 28.1497004, 22.9162824, 18.0104665, 13.5134097, 9.5062687, 7.38006997, 5.38215293, 3.0697266, 0., -6.28867938, -13.4126964, -21.0072975, -28.7077289, -34.8747742, -40.9279278, -47.0122213, -53.2726862, -58.1281187, -64.1402798, -72.1446952, -82.9768906, -86.555554, -98.9997838, -125.511841, -171.293985, -176.348612, -237.157795, -385.00374, -651.168655, -988.904083, -1663.58422, -2759.85339, -4523.45463, -7011.52972, -10874.5267, -16614.567, -24733.7721}, 
{221.112345, 217.273009, 213.245771, 208.971399, 204.390662, 199.444326, 194.07316, 188.21793, 181.819406, 174.07591, 165.967632, 157.732316, 149.607708, 142.759168, 136.12578, 129.574241, 122.971252, 115.636065, 108.201803, 100.754142, 93.3787588, 86.3649399, 79.5133073, 72.8280932, 66.31353, 59.9400399, 53.7591891, 47.7887339, 42.0464305, 36.5675385, 31.3453094, 26.3904981, 21.7138595, 17.1544906, 12.9634673, 9.22020788, 6.00413027, 4.68026363, 3.52817058, 2.11302482, 0., -5.20156086, -11.1883212, -17.612775, -24.1274164, -29.1515342, -34.0641099, -39.0109194, -44.1377391, -47.8775291, -52.7740078, -59.6580777, -69.3606412, -71.3464121, -82.3589571, -107.775654, -152.973881, -158.692157, -220.802263, -370.537118, -639.129644, -981.466477, -1657.81939, -2751.51569, -4503.96735, -6999.36377, -10792.7001, -16321.2699, -24022.367}, 
{207.178203, 203.732178, 199.820678, 195.45373, 190.64136, 185.393593, 179.720456, 173.631975, 167.138176, 159.874527, 152.375435, 144.800749, 137.310317, 130.721435, 124.273526, 117.863461, 111.388109, 104.253972, 97.0444365, 89.8525206, 82.7712422, 76.1562566, 69.7328892, 63.4891028, 57.4128605, 51.3843422, 45.5424068, 39.9181301, 34.5425882, 29.4933277, 24.7363656, 20.2841896, 16.1492874, 12.1875237, 8.63065835, 5.55382834, 3.03217056, 2.36462843, 1.91300975, 1.26292884, 0., -4.1202108, -8.95002157, -14.1717987, -19.4679084, -23.3384285, -27.1209295, -30.9706931, -35.0430011, -37.8103107, -41.7838582, -47.7920555, -56.6633142, -57.3923226, -67.3747053, -92.1723635, -137.347198, -144.531429, -208.788512, -361.25222, -633.056326, -981.531677, -1659.22083, -2747.46148, -4481.02972, -6970.9883, -10669.8149, -15944.6335, -23162.5681}, 
{195.375656, 192.004692, 188.015746, 183.466195, 178.413415, 172.914785, 167.027681, 160.809479, 154.317557, 147.537176, 140.626675, 133.672277, 126.760207, 120.360119, 114.021432, 107.676999, 101.25967, 94.2939214, 87.28433, 80.3270978, 73.5184264, 67.2620822, 61.2236764, 55.3763849, 49.6933835, 43.9759018, 38.4378405, 33.1211539, 28.0677965, 23.3846307, 19.0227396, 14.9981141, 11.3267455, 7.89149902, 4.89474193, 2.40571575, 0.493661944, 0.363790214, 0.494986523, 0.502105057, 0., -3.05610252, -6.72786903, -10.7365941, -14.8035724, -17.5234724, -20.1948652, -22.9896958, -26.0799092, -27.9994607, -31.213481, -36.549111, -44.8334916, -44.6143205, -53.9099591, -78.4593255, -124.001338, -133.312004, -200.278317, -355.824359, -630.8742119999999, -985.890452, -1663.18168, -2741.44858, -4446.50068, -6914.71693, -10493.0898, -15475.9586, -22157.6622}, 
{185.425713, 181.879415, 177.667063, 172.87414, 167.58613, 161.888515, 155.86678, 149.606408, 143.192881, 136.876184, 130.511499, 124.11851, 117.716901, 111.447124, 105.159785, 98.8262595, 92.4179225, 85.5950808, 78.7646057, 72.0222999, 65.4639662, 59.5233565, 53.8231446, 48.3239537, 42.9864068, 37.5470253, 32.2801747, 27.2361189, 22.4651217, 18.090378, 14.0600481, 10.3952234, 7.11699527, 4.1425339, 1.63842043, -0.332685234, -1.7081232, -1.39162518, -0.767583015, -0.186780138, 0., -2.0207093, -4.55193494, -7.35938705, -10.2087758, -11.7946811, -13.3823954, -15.166081, -17.3399, -18.5179766, -21.1065264, -25.9317269, -33.8197554, -32.9334411, -41.8276492, -66.3938965, -112.5237, -124.479458, -194.433453, -352.92885, -630.508812, -991.333571, -1665.09509, -2727.23475, -4392.23916, -6818.86329, -10249.7435, -14906.5457, -21010.9359}, 
{177.049383, 173.145211, 168.610719, 163.542914, 158.038804, 152.195397, 146.109701, 139.878725, 133.599477, 127.703878, 121.820057, 115.911057, 109.93992, 103.754355, 97.4788721, 91.1226459, 84.6948522, 77.9966166, 71.3283849, 64.7825529, 58.4515163, 52.7810191, 47.3687694, 42.1658237, 37.1232385, 31.9300195, 26.9040941, 22.1013388, 17.5776303, 13.4595, 9.70390802, 6.33846907, 3.39079796, 0.816745601, -1.25560381, -2.76992995, -3.66991249, -2.97099193, -1.91638277, -0.821060353, 0., -1.02550445, -2.45229068, -4.09240305, -5.75788596, -6.24006989, -6.77999864, -7.59800213, -8.91441027, -9.4388557, -11.5066445, -15.9423856, -23.5706878, -22.2707196, -30.9907063, -55.7334329, -102.501684, -117.479367, -190.415696, -351.241008, -629.885639, -994.651801, -1660.35421, -2698.57779, -4310.10409, -6671.741, -9926.99437, -14227.6955, -19725.6759}, 
{187.82453, 175.380638, 164.522254, 155.048751, 146.759502, 139.453881, 132.931261, 126.991014, 121.432515, 115.682444, 110.061944, 104.519465, 99.0034562, 93.4094581, 87.7599952, 82.0246817, 76.1731321, 69.8497126, 63.4793845, 57.1618613, 50.9968561, 45.3113449, 39.8868738, 34.732251, 29.8562852, 25.3107344, 21.0442775, 17.0485431, 13.3151599, 9.78749463, 6.52474233, 3.53783623, 0.837709548, -1.92123783, -4.22892593, -5.93180815, -6.87633789, -5.53361613, -3.67558966, -1.69885283, 0., -0.623134421, -1.65833802, -2.8432016, -3.91531598, -3.23888229, -2.47423689, -1.90832646, -1.82809767, -0.571915493, -1.15474102, -4.64295363, -12.1029327, -13.5154792, -25.4667823, -53.4574527, -102.988101, -117.986443, -190.155143, -349.623968, -626.522688, -999.557189, -1653.12889, -2646.10523, -4169.23759, -6411.62763, -9437.09199, -13340.7686, -18217.7952}, 
{202.594033, 179.751196, 161.257808, 146.529047, 134.980092, 126.026122, 119.082316, 113.563853, 108.885911, 103.059218, 97.4651852, 92.0807714, 86.8829361, 82.0677256, 77.3053773, 72.4852159, 67.4965658, 61.6813602, 55.6952715, 49.6465805, 43.6435684, 37.8332563, 32.2696889, 27.0456511, 22.2539279, 18.4601648, 15.0951418, 12.0624993, 9.26587801, 6.39380025, 3.65107215, 1.02738158, -1.48758357, -4.64361367, -7.41575128, -9.51851721, -10.6664323, -8.53514926, -5.69360421, -2.67186512, 0., -0.508669667, -1.50711222, -2.60515855, -3.41263955, -1.63513838, 0.451567238, 2.47594729, 4.06647178, 7.34644229, 8.45156461, 6.01237612, -1.34058582, -6.53591649, -22.7602928, -54.7595243, -107.279421, -121.97668, -191.921867, -347.096438, -617.481845, -993.994698, -1623.81099, -2550.28408, -3950.79094, -6012.26715, -8749.39028, -12219.9024, -16481.5456}, 
{202.282333, 175.871048, 154.829792, 138.423698, 125.9179, 116.577528, 109.667717, 104.453599, 100.200306, 94.3003516, 88.640535, 83.2350361, 78.0980346, 73.6586754, 69.3501876, 65.0207652, 60.5186022, 55.0509575, 49.3633342, 43.5603002, 37.7464236, 31.9534243, 26.3878575, 21.1834305, 16.4738504, 13.064521, 10.1487743, 7.59163891, 5.25814349, 2.701185, 0.222776282, -2.18720142, -4.53886688, -7.71996386, -10.5119361, -12.5738524, -13.5647814, -10.8155658, -7.24479091, -3.44281592, 0., -0.08641868, -0.680828277, -1.34170145, -1.62751084, 1.08043447, 4.17563281, 7.22874608, 9.81043621, 14.0900627, 16.0001108, 14.0717634, 6.83620339, -0.0235436439, -17.9888746, -51.3893436, -104.554505, -121.858854, -191.569027, -341.996602, -601.453156, -968.32838, -1560.69951, -2412.60149, -3678.80089, -5530.33367, -7952.42665, -10976.7589, -14635.0096}, 
{191.434042, 166.516391, 146.685199, 131.231755, 119.447344, 110.623254, 104.050772, 99.021184, 94.8257772, 88.8485881, 83.051054, 77.4873619, 72.211699, 67.8168808, 63.6030143, 59.4088351, 55.0730788, 49.8012396, 44.318591, 38.7171651, 33.088994, 27.4028755, 21.9233699, 16.7918031, 12.1495011, 8.82957003, 6.00484342, 3.539935, 1.29945857, -1.20513783, -3.61480841, -5.9236731, -8.12585181, -11.0434715, -13.5114422, -15.1926812, -15.7501056, -12.5223676, -8.42635531, -4.05469173, 0., 0.584774086, 0.691459186, 0.751561269, 1.1965863, 4.6894944, 8.53775573, 12.2802946, 15.4560354, 19.9833366, 22.0699146, 20.30292, 13.2695034, 6.39295062, -11.3101771, -43.9871831, -95.785371, -117.497179, -187.966722, -332.683249, -577.13601, -923.480797, -1467.20723, -2237.65049, -3360.76265, -4981.95802, -7072.18514, -9646.00898, -12717.9945}, 
{174.593769, 154.463423, 138.271025, 125.452267, 115.442846, 107.678454, 101.594787, 96.6275385, 92.2124029, 86.1466711, 80.1598024, 74.3428524, 68.7868768, 64.1769151, 59.7724458, 55.426931, 50.993833, 45.7749414, 40.3960599, 34.9313197, 29.4548519, 23.9126366, 18.5582165, 13.5169832, 8.91432833, 5.46107933, 2.46301766, -0.188639368, -2.60267448, -5.23995495, -7.71634509, -9.99979375, -12.0582498, -14.5073195, -16.4082314, -17.4698714, -17.4011254, -13.8030567, -9.33550295, -4.55047905, 0., 1.44606418, 2.48069555, 3.47902116, 4.81616805, 8.97369958, 13.3777319, 17.5608176, 21.055509, 25.3506551, 27.2399934, 25.4735579, 18.8013827, 13.0848776, -2.88184972, -33.1933156, -81.9440363, -108.755869, -179.985053, -317.515168, -543.229795, -860.374511, -1346.74691, -2030.02411, -3004.17146, -4383.27103, -6134.6498, -8262.3235, -10770.3079}, 
{156.306122, 142.488343, 131.034262, 121.584286, 113.778824, 107.258283, 101.66307, 96.6335931, 91.81026, 85.6373442, 79.4298407, 73.3066107, 67.3865153, 62.3733517, 57.5670705, 52.8525584, 48.1147021, 42.814798, 37.4307592, 32.0169088, 26.6275694, 21.2137341, 15.9743875, 11.0051847, 6.40178052, 2.66481611, -0.677034553, -3.69011116, -6.44075338, -9.31805307, -11.9364969, -14.2333237, -16.1457723, -18.0046911, -19.1962656, -19.5002906, -18.6965613, -14.805135, -10.0694394, -4.97316436, 0., 2.43860714, 4.5578262, 6.6450698, 8.98775053, 13.7147082, 18.5353573, 23.0005397, 26.6610969, 30.516409, 32.0893643, 30.3513891, 24.2739098, 20.4235486, 7.13845799, -19.6480136, -64.0025181, -95.4991373, -166.494121, -294.851148, -498.433899, -779.932084, -1202.73129, -1794.31537, -2616.52254, -3750.40353, -5165.80466, -6860.37338, -8831.75711}, 
{141.115714, 133.367348, 126.421904, 120.126861, 114.329698, 108.877894, 103.618928, 98.4002787, 93.0694261, 86.7633508, 80.3242294, 73.8837403, 67.5735619, 62.0407637, 56.6954766, 51.4632226, 46.2695237, 40.7635444, 35.2577072, 29.788077, 24.3907188, 19.0371946, 13.8538735, 8.90262193, 4.24530605, 0.146547664, -3.61564479, -7.06050731, -10.2072759, -13.3542189, -16.129927, -18.4420232, -20.1981307, -21.4287693, -21.8695065, -21.3788065, -19.8151337, -15.6761045, -10.7253702, -5.36573418, 0., 3.50355851, 6.79379651, 10.0540987, 13.4678499, 18.6941784, 23.8504278, 28.5296855, 32.3250388, 35.8049895, 37.1970445, 35.7041252, 30.5291532, 28.7802748, 18.5930966, -3.99155006, -42.9328337, -77.5911981, -146.364027, -263.049978, -441.44771, -683.076077, -1038.57315, -1535.11731, -2205.31111, -3099.48633, -4191.63378, -5474.82953, -6942.14962}, 
{103.940274, 109.246159, 112.100836, 112.786499, 111.585343, 108.779563, 104.651353, 99.482908, 93.556423, 87.4462621, 81.0255828, 74.4597118, 67.9139761, 62.0289608, 56.3046312, 50.7162112, 45.2389241, 39.780798, 34.4111305, 29.1320235, 23.945579, 18.7904806, 13.7576159, 8.87445415, 4.16846479, -0.307552597, -4.56159118, -8.57631379, -12.3343833, -16.2187059, -19.6516038, -22.4556421, -24.4533865, -25.0813324, -24.702543, -23.294012, -20.8327326, -16.4901945, -11.3710965, -5.77463332, 0., 4.75478168, 9.44887446, 14.1426142, 18.8963367, 25.0263267, 30.8345917, 35.8790878, 39.7177713, 41.4854709, 41.3325214, 38.9861302, 34.1735046, 36.3321689, 31.5948865, 15.8047379, -15.1951963, -53.3876082, -119.973336, -223.978989, -374.43118, -567.576215, -850.781612, -1259.91447, -1784.82538, -2422.06506, -3187.40028, -4091.7938, -5146.20839}, 
{119.696459, 121.904076, 122.249633, 120.94698, 118.209966, 114.25244, 109.288254, 103.531255, 97.1952948, 90.7420274, 84.0383753, 77.1990662, 70.3388277, 63.8365474, 57.4371289, 51.1496357, 44.9831315, 38.9566103, 33.0652331, 27.3140912, 21.708276, 16.2226188, 10.9045753, 5.77134083, 0.84011102, -3.93325859, -8.44569664, -12.6554717, -16.5208523, -20.3073567, -23.5431039, -26.0634625, -27.7038011, -27.8850365, -27.02277, -25.1181511, -22.1723292, -17.5689495, -12.1736673, -6.23463379, 0., 5.72320321, 11.4692564, 17.2135602, 22.9315156, 29.3276159, 35.3565322, 40.7020283, 45.0478681, 48.0136362, 49.3729471, 48.8352361, 46.1099386, 48.6623724, 45.3437374, 32.7611163, 7.52159136, -26.3360594, -82.6091273, -167.663208, -287.863898, -441.843924, -659.167485, -966.431705, -1346.42536, -1792.66723, -2299.56444, -2861.30205, -3472.06512}, 
{169.150239, 157.676836, 147.513519, 138.460033, 130.316123, 122.881534, 115.956011, 109.3393, 102.831146, 95.6550193, 88.4174499, 81.1486922, 73.8790011, 66.6745961, 59.5153815, 52.4172262, 45.3959992, 38.4031129, 31.5446755, 24.8623387, 18.3977542, 12.2411791, 6.36621736, 0.795078651, -4.45002757, -9.41572308, -13.9834346, -18.10342, -21.7259373, -24.9162069, -27.4635394, -29.2722079, -30.2464854, -30.1545278, -29.0911722, -27.0151386, -23.8851471, -18.9590206, -13.1767348, -6.77736834, 0., 6.50069279, 13.0674696, 19.6274911, 26.1079182, 32.5190318, 38.6716244, 44.4596088, 49.7768977, 55.8502074, 60.7075256, 63.7096435, 64.2173524, 64.9421617, 60.5538573, 49.0729433, 28.5199236, 5.13411772, -34.5703119, -95.8603875, -184.003131, -305.004472, -461.914711, -660.672514, -900.983227, -1205.90322, -1511.31808, -1774.90607, -1954.34547}, 
{157.588269, 149.87012, 142.397846, 135.138514, 128.05919, 121.126941, 114.308835, 107.571936, 100.883312, 94.184065, 87.4776118, 80.7414055, 73.952899, 66.9294522, 59.8726476, 52.8239751, 45.8249243, 38.9794848, 32.2416468, 25.6279002, 19.1547349, 12.7006072, 6.47525399, 0.550378712, -5.00231531, -9.89727766, -14.3621909, -18.4108905, -22.0572119, -25.8017927, -28.9769454, -31.4017847, -32.8954252, -32.8713233, -31.7165151, -29.4123786, -25.9402917, -20.6885535, -14.4688524, -7.49979804, 0., 7.57072875, 15.3314634, 23.1600759, 30.9344379, 38.6372245, 45.9995829, 52.8574638, 59.0468175, 64.5920658, 69.0652995, 72.2270807, 73.8379713, 76.5085846, 76.0094109, 70.9609919, 59.9838691, 45.5410236, 20.8735817, -16.934891, -70.8008287, -145.218014, -238.370836, -359.648158, -484.977861, -600.901629, -684.966921, -716.969752, -676.706137}, 
{170.100056, 162.077358, 154.716959, 147.843675, 141.282325, 134.857726, 128.394696, 121.718051, 114.652611, 106.038997, 97.0798998, 87.993815, 78.9992377, 71.0318317, 63.3060567, 55.7535409, 48.3059124, 40.5721135, 32.9355325, 25.4568718, 18.1968339, 11.2698263, 4.66136436, -1.58933151, -7.44304092, -12.8267762, -17.748591, -22.1827721, -26.1036058, -29.8062716, -32.815806, -34.9781382, -36.1391978, -35.7891867, -34.2720525, -31.5760155, -27.6892961, -22.1624404, -15.5964124, -8.15450233, 0., 8.75252273, 17.8715705, 27.174366, 36.478132, 45.3117409, 53.896106, 62.1637903, 70.0473565, 77.8136669, 84.9272654, 91.1869954, 96.3917, 101.4355, 104.58385, 105.197482, 102.63713, 95.6793504, 84.50272, 68.7016409, 47.8705152, 17.3576345, -10.504268, -36.2030578, -39.5162004, -13.1099957, 81.1941944, 273.863774, 595.366146}, 
{177.272638, 176.171898, 172.67048, 167.127495, 159.902055, 151.353269, 141.840249, 131.722107, 121.357952, 111.932907, 102.649669, 93.5369434, 84.6234388, 76.1690857, 67.8788777, 59.6890322, 51.5357667, 42.9013283, 34.3574928, 26.0220658, 18.0128529, 10.8353733, 4.06463329, -2.336647, -8.40574756, -14.5091726, -20.223288, -25.4536843, -30.1059515, -34.3111544, -37.6592191, -39.965546, -41.0455356, -40.0655795, -37.7496906, -34.1728732, -29.4101313, -23.5468726, -16.6435361, -8.77096437, 0., 10.1187145, 20.7860564, 31.7231031, 42.6509319, 52.6140477, 62.2807294, 71.6426832, 80.6916156, 89.0636486, 97.2483064, 105.379529, 113.591258, 122.654121, 131.810694, 140.94024, 149.922024, 154.37936, 160.149841, 168.815111, 181.956813, 197.339586, 227.99609, 274.768336, 359.778501, 504.715443, 711.721124, 987.824231, 1340.05345}
};
  
  /* Stuff for interpolating the NQC data */
  gsl_spline    *spline = NULL;
  gsl_interp_accel *acc = NULL;
  /* Interpolate the spin NQC data in 2-D parameter space -- spin and mass ratio */
  /* First, interpolate in spin dimension for all mass ratios */
  spline = gsl_spline_alloc( gsl_interp_cspline, adim );
  acc    = gsl_interp_accel_alloc();
  for (i = 0; i < qdim; i++)
  {
    gsl_spline_init( spline, alist, a3stab[i], adim );
    gsl_interp_accel_reset( acc );
    a3slist[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, a4tab[i], adim );
    gsl_interp_accel_reset( acc );
    a4list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, a5tab[i], adim );
    gsl_interp_accel_reset( acc );
    a5list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
//printf("%.15f\n",a3slist[i]);
  }
//printf("%.15f\n",a);
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);
  /* Second, interpolate in mass ratio dimension */
  spline = gsl_spline_alloc( gsl_interp_cspline, qdim );
  acc    = gsl_interp_accel_alloc();
  gsl_spline_init( spline, etalist, a3slist, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a3S = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, a4list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a4 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, a5list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a5 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);
 
  /* Interpolate nonspin NQC data in the mass ratio dimension */
  spline = gsl_spline_alloc( gsl_interp_cspline, nsqdim );
  acc    = gsl_interp_accel_alloc();
  gsl_spline_init( spline, nsetalist, a1list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a1 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, a2list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a2 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, a3list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a3 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, b1list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->b1 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, b2list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->b2 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);
  /* Andrea and I have different sign conventions, so I need to put a minus sign in front */
  coeffs->b1 = - coeffs->b1;
  coeffs->b2 = - coeffs->b2;
#if 0
  coeffs->a1 = -8.02798637014;
  coeffs->a2 = 48.7446843797;
  coeffs->a3 = -45.7900277224;
  coeffs->a3S= 0.;
  coeffs->a4 = 0.;
  coeffs->a5 = 0.;
  coeffs->b1 = 0.834742923041;
  coeffs->b2 = -2.33512320852; // q=1
#endif
#if 0
  coeffs->a1 = -7.79667;
  coeffs->a2 = 47.182;
  coeffs->a3 = 2238.85334023;
  coeffs->a3S= 0.;
  coeffs->a4 = -7143.16738899;
  coeffs->a5 = 5596.0086893;
  coeffs->b1 = 0.85069;
  coeffs->b2 = -2.47071; // q=1, chi1=chi2=0.98
#endif
#if 0
  coeffs->a1 = -6.82562365707;
  coeffs->a2 = 41.5711482044;
  coeffs->a3 = -39.4329799959;
  coeffs->a3S= 0.;
  coeffs->a4 = 0.;
  coeffs->a5 = 0.;
  coeffs->b1 = 0.461925688819;
  coeffs->b2 = -1.38733263299; // q=8
#endif
#if 0
  coeffs->a1 = -7.5758;
  coeffs->a2 = 46.9323;
  coeffs->a3 = -118.368375152;
  //coeffs->a3 = -45.0036; // NS part
  coeffs->a3S= 0.;
  coeffs->a4 = 125.555824111;
  coeffs->a5 = -22.0751068073;
  //coeffs->a4 = 0.;
  //coeffs->a5 = 0.;
  coeffs->b1 = 0.51305;
  coeffs->b2 = -1.55133; // q=8, chi1=0.5
#endif

  /* Obsolete polynomial fitting of nonspin NQC coefficients a1, a2, a3, b1 and b2 */
  /*
  coeffs->a1 = -12.67955358602124 + 75.41927959573084 * eta - 106.15933052937714 * eta2;
  coeffs->a2 = 101.45522216901628 - 757.3158549733314 * eta + 1473.314771676588 * eta2;
  coeffs->a3 = -107.6647834845902 + 857.6219519536213 * eta - 1776.2776804623143 * eta2;
  // Andrea and I have different sign conventions, so I need to put a minus sign in front
  coeffs->b1 = - (-1.464129495621165 + 12.81732978488213 * eta - 60.09957767247623 * eta2);
  coeffs->b2 = - ( 7.477426352542122 - 85.26122117590637 * eta + 353.3251639728075 * eta2);
  */

  return XLAL_SUCCESS;

}

/**
 * Function to interpolate known amplitude NQC coeffcients of spin terms for SEOBNRv2,
 * namely a3s, a4 and a5.
 * The a3s, a4 and a5 values were calculated for
 * 11 mass ratios q=1,1.5,2,4,6,8,10,11,12,14 and 20, and
 * 19 spin (\f$\chi\f$ defined in Taracchini et al. PRD 86, 024011 (2012)) values
 * chi = -1, -0.9, -0.8, ......, 0.3, 0.4, 0.5, 0.55, 0.6, 0.65.
 * The calculation was done by Andrea Taracchini using a C++ code of the UMaryland group.
 * In principle, these numbers can be automatically calculated iteratively by the LAL code.
 * However, since such calcualtion increase the cost of each waveform generation by
 * about an order of magnitude, we prepare these numbers in advance reduce cost.
 * These number can be verified by confirming that
 * the peak amplitude and frequency agree well with the NR-fits predicted values,
 * and to get exact NR-fits predicted values, corrections on these numbers are ~1%.
 */
UNUSED static int XLALSimIMRGetEOBCalibratedSpinNQCv2( EOBNonQCCoeffs * restrict coeffs,                                   INT4 UNUSED l,                                   INT4 UNUSED m,                                   REAL8 eta,                                   REAL8 a )
{
  const unsigned int nsqdim = 101;
  const unsigned int   qdim = 21;
  const unsigned int   adim = 81;
  UINT4 i;
  /* REAL8 eta2 = eta*eta;*/
  REAL8 a3slist[qdim], a4list[qdim], a5list[qdim];
  REAL8 b3list[qdim], b4list[qdim];

  memset( coeffs, 0, sizeof( *coeffs ) );
  const double nsetalist[101] = {0.021, 0.0233, 0.0256, 0.0279, 0.0302, 0.0324, 0.0347, 0.037, 0.0393, 0.0416, 0.0439, 0.0462, 0.0485, 0.0508, 0.0531, 0.0553, 0.0576, 0.0599, 0.0622, 0.0645, 0.0668, 0.0691, 0.0714, 0.0737, 0.076, 0.0782, 0.0805, 0.0828, 0.0851, 0.0874, 0.0897, 0.092, 0.0943, 0.0966, 0.0989, 0.1012, 0.1034, 0.1057, 0.108, 0.1103, 0.1126, 0.1149, 0.1172, 0.1195, 0.1218, 0.124, 0.1263, 0.1286, 0.1309, 0.1332, 0.1355, 0.1378, 0.1401, 0.1424, 0.1447, 0.147, 0.1492, 0.1515, 0.1538, 0.1561, 0.1584, 0.1607, 0.163, 0.1653, 0.1676, 0.1698, 0.1721, 0.1744, 0.1767, 0.179, 0.1813, 0.1836, 0.1859, 0.1882, 0.1905, 0.1927, 0.195, 0.1973, 0.1996, 0.2019, 0.2042, 0.2065, 0.2088, 0.2111, 0.2134, 0.2156, 0.2179, 0.2202, 0.2225, 0.2248, 0.2271, 0.2294, 0.2317, 0.234, 0.2363, 0.2385, 0.2408, 0.2431, 0.2454, 0.2477, 0.25};
  const double etalist[21] = {0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.11, 0.12, 0.13, 0.14, 0.15, 0.16, 0.17, 0.18, 0.19, 0.2, 0.21, 0.22, 0.23, 0.24, 0.25};
  const double alist[81]   = {-1., -0.975, -0.95, -0.925, -0.9, -0.875, -0.85, -0.825, -0.8, -0.775, -0.75, -0.725, -0.7, -0.675, -0.65, -0.625, -0.6, -0.575, -0.55, -0.525, -0.5, -0.475, -0.45, -0.425, -0.4, -0.375, -0.35, -0.325, -0.3, -0.275, -0.25, -0.225, -0.2, -0.175, -0.15, -0.125, -0.1, -0.075, -0.05, -0.025, 0., 0.025, 0.05, 0.075, 0.1, 0.125, 0.15, 0.175, 0.2, 0.225, 0.25, 0.275, 0.3, 0.325, 0.35, 0.375, 0.4, 0.425, 0.45, 0.475, 0.5, 0.525, 0.55, 0.575, 0.6, 0.625, 0.65, 0.675, 0.7, 0.725, 0.75, 0.775, 0.8, 0.825, 0.85, 0.875, 0.9, 0.925, 0.95, 0.975, 1.};

  const double a1list[101] = {-9.70365622284, -9.5487796595, -9.40705218072, -9.28242447497, -9.16287060127, -9.03811669547, -8.92394749379, -8.81661027414, -8.7095851328, -8.60723315388, -8.50836995352, -8.40985391052, -8.314500912, -8.22035943611, -8.12770592382, -8.03757606119, -7.94997595927, -7.86421160183, -7.77953168332, -7.69584399308, -7.61324344682, -7.53192249536, -7.45216063215, -7.37374612092, -7.29648777301, -7.22046897906, -7.14581213839, -7.07265232481, -7.00073270258, -6.92991647363, -6.85992514074, -6.79104613524, -6.72328769905, -6.65664446919, -6.59158891131, -6.52783781218, -6.4667874811, -6.41088676175, -6.35876275395, -6.3073217784, -6.25613493225, -6.20587481079, -6.15851113649, -6.1141742151, -6.07327317284, -6.03621329966, -6.0001893395, -5.96655531812, -5.93844433597, -5.91027864135, -5.88037825648, -5.85358373242, -5.83336708498, -5.82744285781, -5.82818819089, -5.8231589417, -5.81812839946, -5.81887431779, -5.82568678472, -5.83621354379, -5.84931204972, -5.86442954488, -5.88109070102, -5.89755497618, -5.91392008194, -5.93177097875, -5.95199881766, -5.97697408745, -6.00401909417, -6.032531535, -6.06148086596, -6.09023163793, -6.11920032326, -6.14820900838, -6.17641365359, -6.20357599269, -6.23098735047, -6.25903214508, -6.28832401322, -6.32033354968, -6.35344345473, -6.38745961465, -6.42231976572, -6.46800466293, -6.50219865357, -6.52990506622, -6.54965342702, -6.58271099648, -6.626558215, -6.67664802777, -6.72810147477, -6.77989789496, -6.83928403282, -6.91446932169, -6.99994425628, -7.08201636107, -7.21472596367, -7.36827339046, -7.55475356656, -7.81665470799, -8.24936799702};

  const double a2list[101] = {71.8355837876, 70.2888163877, 68.858392382, 67.578368761, 66.3483263783, 65.0816770892, 63.9139561883, 62.8042133759, 61.7074812267, 60.6519669001, 59.6324792699, 58.6222094978, 57.6434687679, 56.6791933949, 55.7311388293, 54.8071777364, 53.9062860992, 53.0218701557, 52.1495389357, 51.2895492368, 50.4445863927, 49.6138840037, 48.797074544, 47.9943062521, 47.2056828988, 46.4310888273, 45.6708900349, 44.9258152616, 44.1929201091, 43.4702452949, 42.7537119379, 42.0477065689, 41.3531483593, 40.6709264131, 40.0029858455, 39.3453826346, 38.7105183605, 38.119829408, 37.5632095631, 37.0166865581, 36.4716876499, 35.9345595307, 35.4276417334, 34.9454114602, 34.4897359229, 34.0666726398, 33.6585411373, 33.268152249, 32.9062444867, 32.5494651377, 32.1932422911, 31.8674681969, 31.598287958, 31.4450951286, 31.3504698588, 31.2224679676, 31.1007751184, 31.0231242897, 30.991643584, 30.991534397, 31.0161048282, 31.0616750366, 31.1249606145, 31.1939358163, 31.2696363303, 31.3636652679, 31.4828878696, 31.6456708233, 31.8311325739, 32.034527166, 32.2477681231, 32.4658509298, 32.6919766483, 32.9247097971, 33.1580140019, 33.3900369294, 33.6294910387, 33.8788248397, 34.1421252583, 34.4296046145, 34.7294034713, 35.0398350888, 35.3602204133, 35.7665471711, 36.0881399675, 36.3632831298, 36.5789495976, 36.9023316975, 37.3120873104, 37.7708043813, 38.2382659746, 38.7083475975, 39.2358324951, 39.8798584792, 40.5947669891, 41.2766466634, 42.3250573144, 43.5282809602, 44.9693438982, 46.9606371658, 50.0808316558};

  const double a3list[101] = {-75.3870348213, -73.5563054743, -71.8586610238, -70.3326010764, -68.8669937325, -67.3652081804, -65.9797869993, -64.6591310991, -63.3595036445, -62.1068621881, -60.8985027736, -59.7042086628, -58.5479728539, -57.4103808438, -56.2930281651, -55.2041741738, -54.1420295685, -53.0987898958, -52.0706872299, -51.0586173069, -50.0667002046, -49.0926497207, -48.1344431455, -47.1933388413, -46.2704158358, -45.3650449971, -44.4772042283, -43.6074103996, -42.7520134632, -41.9084025308, -41.0711187682, -40.2460776811, -39.4347613879, -38.6386667672, -37.8587722914, -37.0898368809, -36.3458764907, -35.6510020946, -34.9947343733, -34.3515708492, -33.7091654129, -33.0750000728, -32.4772646605, -31.9060849394, -31.3624600752, -30.8542478078, -30.364610911, -29.8930785724, -29.4449207659, -29.0043606667, -28.5703691139, -28.171109369, -27.8341554058, -27.6229720264, -27.4766951091, -27.2985750163, -27.1291617273, -27.0061494789, -26.9317625784, -26.8914625991, -26.8788122164, -26.8902006455, -26.9223881792, -26.9631792328, -27.013848345, -27.0865681299, -27.1887211003, -27.3400465839, -27.5181167553, -27.7178394005, -27.9305187037, -28.1507835284, -28.3821145422, -28.6229907203, -28.8672132002, -29.1128371338, -29.36865915, -29.6371071707, -29.922311561, -30.2347674515, -30.5620356831, -30.9022473284, -31.254609498, -31.7001465622, -32.0565827322, -32.3648712883, -32.6103616483, -32.9736989432, -33.4312219324, -33.9418551023, -34.4614297518, -34.9838283376, -35.5674613231, -36.2737268484, -37.0519888454, -37.7923554764, -38.9140097, -40.1981922876, -41.7294871636, -43.8332602049, -47.0600574352};

  const double b1list[101] = {-0.454231789035, -0.453740895324, -0.45114951071, -0.445455559729, -0.44030174389, -0.438246415293, -0.435617386371, -0.430628416274, -0.428155263233, -0.424563996869, -0.421496742404, -0.419542314684, -0.417786543725, -0.416452769889, -0.415181542102, -0.413649756645, -0.411562161412, -0.408719678496, -0.40593475615, -0.403713197659, -0.402947057134, -0.402692696086, -0.401882254635, -0.401361543256, -0.401857592043, -0.403135870926, -0.405023096371, -0.407445456424, -0.410163626113, -0.412943367674, -0.415232607237, -0.417748668538, -0.420844132902, -0.424905043844, -0.429142196414, -0.432959586521, -0.436617400517, -0.440401062979, -0.445048680123, -0.451651705843, -0.457760235328, -0.463733230566, -0.472390233324, -0.480892678638, -0.488468389259, -0.496054924285, -0.505287254412, -0.516712248163, -0.530702320051, -0.543402030632, -0.553376187444, -0.564523083803, -0.576716924164, -0.588673429838, -0.599807622576, -0.6083625871, -0.61629220663, -0.626333930393, -0.638492384934, -0.651370663496, -0.663578126675, -0.675399039036, -0.687337552218, -0.699811934302, -0.712809946825, -0.725971047683, -0.73913751946, -0.752013364118, -0.764774460887, -0.777402692471, -0.789886995534, -0.802209798764, -0.814336203683, -0.826241102491, -0.837954892052, -0.849457493421, -0.860601375162, -0.871314573433, -0.881462999336, -0.890680302818, -0.899437045192, -0.907818061982, -0.915897388877, -0.92305068548, -0.931064125926, -0.939248450894, -0.947470287824, -0.953643840407, -0.957725818058, -0.959847025364, -0.960115447844, -0.959051577361, -0.957073291984, -0.952866371063, -0.945294863841, -0.937774627549, -0.921959850725, -0.901024675728, -0.87472187553, -0.834748962688, -0.733155709571};

  const double b2list[101] = {1.24250927283, 1.23820882351, 1.22814744033, 1.20899501569, 1.19187482987, 1.18459068292, 1.17544439367, 1.16014645984, 1.15217431136, 1.14122006693, 1.13199616785, 1.12640121441, 1.121556275, 1.11826795578, 1.11552355075, 1.11238433978, 1.10807035787, 1.10209548232, 1.09668053255, 1.09318287767, 1.09397420715, 1.09653673342, 1.09802015935, 1.10068222122, 1.10643573897, 1.11470254998, 1.12510511039, 1.13755406941, 1.15120517214, 1.16532741744, 1.17824955526, 1.19205557182, 1.20770493667, 1.22624497606, 1.24563734056, 1.26435121321, 1.28310293005, 1.30265970794, 1.32505082997, 1.35327207812, 1.38087484583, 1.40864740792, 1.44367567593, 1.47847338398, 1.51107248182, 1.54412932876, 1.58210883511, 1.62865620041, 1.68826303807, 1.74233039828, 1.78373725125, 1.82868267503, 1.87681370736, 1.92312517747, 1.9656186804, 1.99785596132, 2.02740592217, 2.06495795508, 2.11055684374, 2.15878076242, 2.20440433139, 2.24836632819, 2.29241478185, 2.33795222667, 2.38484419984, 2.43171272272, 2.47786827853, 2.52194361567, 2.56488422632, 2.60668479559, 2.6474159293, 2.6870781889, 2.72548564593, 2.76255774767, 2.79845498187, 2.83311863463, 2.86593650322, 2.89662455284, 2.9246508029, 2.94855630091, 2.97022227107, 2.98999047671, 3.00814380105, 3.02093394891, 3.03817745817, 3.05662290422, 3.07635559396, 3.08574133623, 3.0854061584, 3.07702110074, 3.06226203175, 3.04321477098, 3.01960302068, 2.98621442345, 2.94157099234, 2.89879486315, 2.82337825218, 2.72933462053, 2.61436316871, 2.4455062038, 2.1171325295};

  const double a3stab[21][81] = {
{194.034062, 193.315403, 192.340696, 191.108131, 189.615898, 187.862187, 185.845189, 183.563092, 181.014087, 178.196365, 175.108115, 171.747527, 168.112791, 164.202098, 159.88123, 155.232138, 150.346797, 145.250479, 139.968456, 134.526002, 128.948389, 123.451375, 117.843441, 112.123552, 106.290673, 100.343771, 94.2818106, 88.0767764, 81.7047066, 75.2290824, 68.6598248, 62.0068545, 55.2800926, 48.4894598, 41.7952745, 35.1123156, 28.3439776, 21.4694187, 14.4677972, 7.31827158, 0, -7.06413602, -14.3999825, -22.0896625, -30.2152987, -38.8590141, -48.1029316, -57.9430959, -68.3884858, -79.7270473, -92.0758537, -105.551978, -120.272495, -136.354477, -161.225662, -190.378371, -218.751539, -244.966902, -267.646199, -285.411167, -296.883542, -310.354742, -328.37036, -319.492152, -252.281869, 29.4084482, 249.400555, 588.783155, 1228.64495, 2363.34086, 4134.16095, 7287.44826, 10172.0754, 11510.5972, 12299.3266, 12652.4652, 12433.2947, 11460.0626, 9982.07373, 8101.6811, 5921.23755},
{160.112548, 160.667985, 160.855096, 160.683878, 160.164332, 159.306454, 158.120244, 156.615701, 154.802822, 152.691607, 150.292055, 147.614163, 144.66793, 141.463355, 137.974809, 134.234829, 130.278648, 126.123553, 121.786828, 117.285761, 112.637636, 107.962912, 103.161454, 98.2362991, 93.1904845, 88.0270475, 82.7490251, 77.3472032, 71.8142083, 66.1823723, 60.4597069, 54.6542236, 48.7739343, 42.8268505, 36.9526253, 31.0759936, 25.1117255, 19.0409065, 12.8446223, 6.50395828, 0, -6.31861818, -12.8891206, -19.7811827, -27.06448, -34.8086881, -43.0834823, -52.0954881, -62.0307527, -72.6314885, -83.9117648, -95.8856513, -108.567217, -121.970532, -144.794469, -171.559059, -196.126931, -216.73575, -231.62318, -239.026886, -237.184533, -230.598764, -228.655499, -197.207571, -102.107815, 154.463951, 449.595197, 893.840096, 1597.75282, 2688.58018, 4226.79843, 6614.56539, 8915.92748, 10410.9574, 11588.5513, 12442.7012, 12813.2517, 12518.0513, 11617.0807, 10087.775, 7907.56954},
{135.75872, 136.138893, 136.299307, 136.238612, 135.955459, 135.448498, 134.71638, 133.757755, 132.571273, 131.155584, 129.50934, 127.63119, 125.519785, 123.173775, 120.539911, 117.649675, 114.538477, 111.215586, 107.690265, 103.971782, 100.069403, 95.9633653, 91.6959723, 87.2804996, 82.7302224, 78.058416, 73.2783556, 68.4234731, 63.5241716, 58.5455294, 53.4926375, 48.3705869, 43.1844686, 37.9393735, 32.7564707, 27.5674199, 22.2950938, 16.9208406, 11.4260085, 5.79194558, 0, -5.65721456, -11.5526034, -17.7478063, -24.304463, -31.2842132, -38.7486967, -47.177716, -56.9882396, -67.2420324, -77.8309468, -88.646835, -99.5815494, -110.526942, -130.898858, -154.564236, -174.669094, -189.157228, -195.972438, -193.058522, -178.359277, -95.540229, -64.7497577, -36.3204737, 39.4150121, 182.864142, 515.868654, 1052.98428, 1808.76676, 2783.21579, 4034.5552, 5797.96238, 7575.84812, 8998.68735, 10279.6823, 11358.5028, 12072.546, 12243.6693, 11858.1609, 10846.2528, 9138.17686},
{117.594822, 118.216324, 118.587481, 118.713031, 118.597712, 118.24626, 117.663414, 116.853912, 115.822491, 114.573888, 113.112843, 111.444091, 109.572371, 107.502421, 105.228449, 102.761854, 100.114831, 97.2942708, 94.3070652, 91.1601054, 87.8602826, 84.4097412, 80.820775, 77.1009309, 73.2577557, 69.2987964, 65.2315998, 61.0732898, 56.8395514, 52.5150315, 48.1033884, 43.6082803, 39.0333656, 34.3823025, 29.7493628, 25.0808824, 20.3163379, 15.4408534, 10.439553, 5.29756047, 0, -5.35071899, -10.9029559, -16.6877846, -22.736279, -29.0795131, -35.7485608, -43.2877245, -52.1641879, -61.1818369, -70.1633582, -78.9314385, -87.3087643, -95.1180224, -113.214171, -134.440841, -150.806571, -159.877486, -159.219713, -146.399379, -118.982611, -70.3022292, -22.0562145, 53.8038232, 185.326274, 404.156778, 729.470528, 1193.63258, 1829.00799, 2680.16117, 3742.85913, 5113.35478, 6506.86997, 7696.20069, 8895.67902, 10040.8571, 10932.5801, 11339.6312, 11293.7482, 10737.025, 9611.55528},
{104.536193, 105.151992, 105.52303, 105.656338, 105.558946, 105.237884, 104.700185, 103.952877, 103.002992, 101.85756, 100.523612, 99.0081792, 97.3182909, 95.4609783, 93.4570201, 91.3047497, 89.00146, 86.5513695, 83.9586968, 81.2276605, 78.362479, 75.3512364, 72.216514, 68.9647586, 65.6024172, 62.1359365, 58.5717633, 54.9331837, 51.2409534, 47.4612548, 43.5936975, 39.637891, 35.593445, 31.4599689, 27.2943493, 23.0599623, 18.715848, 14.2499006, 9.65001414, 4.90408258, 0, -5.02324517, -10.2359098, -15.6571562, -21.306147, -27.2020446, -33.3640114, -40.5534775, -49.4203385, -58.20891, -66.6369699, -74.4222961, -81.2826666, -86.9358592, -101.298089, -117.635564, -128.44252, -131.350726, -123.991956, -103.997981, -69.0005725, -33.6912932, 21.7514718, 116.165345, 268.387948, 496.490784, 821.201233, 1260.43758, 1832.11811, 2569.02123, 3444.48487, 4457.74625, 5529.91911, 6593.29966, 7712.26975, 8816.62971, 9754.9376, 10360.7413, 10608.8717, 10425.9949, 9738.77691},
{96.9661604, 97.1499366, 97.1688322, 97.0244892, 96.7185494, 96.2526546, 95.6284466, 94.8475673, 93.9116586, 92.8223622, 91.58132, 90.190174, 88.6505658, 86.9641373, 85.1097322, 83.1034146, 80.9629742, 78.6947161, 76.3049452, 73.7999668, 71.1860856, 68.4994692, 65.712436, 62.8271668, 59.8458427, 56.7706443, 53.6037525, 50.3545028, 47.0311555, 43.6187841, 40.1166645, 36.5240723, 32.8402837, 29.0645742, 25.2349668, 21.3262257, 17.3101827, 13.178188, 8.92159236, 4.53174618, 0, -4.6801893, -9.51986849, -14.5279779, -19.713458, -25.0852492, -30.6522918, -37.2783443, -45.6987185, -53.8783872, -61.4792071, -68.1630351, -73.591728, -77.4271424, -90.8801985, -106.306773, -115.188531, -114.825063, -102.515961, -75.5608158, -31.2592188, 33.1149764, 110.137717, 217.631759, 373.41986, 572.166076, 888.81781, 1313.4076, 1835.96799, 2443.46734, 3135.13071, 3899.77877, 4762.59709, 5753.91388, 6767.53656, 7738.27797, 8606.45078, 9341.27386, 9790.84147, 9844.54054, 9391.75801},
{91.9984129, 92.0724899, 91.9723426, 91.7032712, 91.2705759, 90.6795572, 89.9355152, 89.0437503, 88.0095627, 86.8382527, 85.5351205, 84.1054666, 82.554591, 80.8877943, 79.1289793, 77.2716783, 75.3080151, 73.2394849, 71.067583, 68.7938044, 66.4196446, 63.9380846, 61.3603095, 58.6889905, 55.9267986, 53.0764049, 50.1404805, 47.134602, 44.0724069, 40.9257076, 37.6919349, 34.3685201, 30.9528939, 27.4424876, 23.8547107, 20.1743557, 16.3847026, 12.479096, 8.45088037, 4.29340019, 0, -4.52502743, -9.18698719, -13.9802361, -18.899131, -23.9380288, -29.0912862, -35.37357, -43.6462329, -51.4639546, -58.4068133, -64.0548872, -67.9882546, -69.7869938, -79.8850741, -90.9963626, -95.0131348, -89.2953953, -71.2031485, -38.096399, 12.6648486, 60.4616544, 143.801577, 269.670606, 445.05473, 673.029945, 970.22686, 1338.93951, 1781.46194, 2307.71228, 2897.11226, 3505.21395, 4196.11738, 5020.27394, 5911.38909, 6815.19861, 7663.7143, 8405.30481, 8936.63545, 9171.99775, 9025.68322},
{88.7697502, 88.7243429, 88.5229502, 88.1697385, 87.668874, 87.0245232, 86.2408524, 85.322028, 84.2722164, 83.0955839, 81.796297, 80.378522, 78.8464252, 77.2041731, 75.4608637, 73.6175437, 71.6748862, 69.6360488, 67.504189, 65.2824646, 62.9740331, 60.5905035, 58.1254149, 55.5807576, 52.9585221, 50.2606987, 47.4892778, 44.6635083, 41.8000455, 38.8596132, 35.8371944, 32.7277721, 29.5263291, 26.2278486, 22.8396849, 19.3489948, 15.7419984, 12.0111483, 8.14889689, 4.14769663, 0, -4.57373064, -9.27148902, -14.063259, -18.9190244, -23.8087691, -28.7024769, -34.6285481, -42.4563432, -49.6250515, -55.6749058, -60.1461388, -62.5789832, -62.5136718, -69.857389, -77.5921962, -77.915422, -68.2468249, -46.0061634, -8.61319596, 46.512319, 98.1593023, 187.472996, 318.664443, 495.944685, 718.350267, 1002.85596, 1347.46346, 1750.17445, 2217.72359, 2721.91372, 3194.0735, 3742.39868, 4446.91421, 5256.03088, 6117.02483, 6949.94367, 7676.88355, 8238.97487, 8570.85908, 8607.17759},
{86.991682, 86.9009299, 86.6584244, 86.2679651, 85.7333516, 85.0583835, 84.2468603, 83.3025818, 82.2293474, 81.0309567, 79.7112095, 78.2739051, 76.7228433, 75.0618236, 73.2860615, 71.4047868, 69.4278794, 67.3608948, 65.2093882, 62.9789152, 60.6750311, 58.3371277, 55.932251, 53.4612835, 50.9251075, 48.3246055, 45.6606599, 42.952423, 40.2163017, 37.4094932, 34.5254615, 31.5576709, 28.4995856, 25.3446698, 22.0706172, 18.6808685, 15.1800581, 11.5648762, 7.83201255, 3.97815717, 0, -4.44925282, -8.98199005, -13.5540843, -18.121408, -22.639834, -27.0652346, -32.4829047, -39.8084298, -46.2971051, -51.4462215, -54.7530699, -55.7149412, -53.8291262, -58.6640354, -63.3459328, -60.2387428, -46.7797921, -20.4064072, 21.4440851, 81.3343584, 146.856546, 244.47561, 378.327141, 552.546728, 759.30391, 1032.25046, 1361.16283, 1735.81745, 2148.15637, 2581.45923, 2984.80058, 3447.78767, 4035.66729, 4736.79845, 5514.39717, 6296.19014, 7018.13887, 7618.75482, 8035.68015, 8206.55699},
{87.4052431, 87.2488223, 86.9362164, 86.4718191, 85.8600238, 85.1052242, 84.2118138, 83.184186, 82.0267346, 80.7438529, 79.3399346, 77.8193732, 76.1865622, 74.4458952, 72.5913588, 70.6339301, 68.5853738, 66.4522119, 64.2409669, 61.9581608, 59.6103158, 57.2444895, 54.8210706, 52.3409829, 49.8051505, 47.2144972, 44.5699471, 41.8924106, 39.199795, 36.4452343, 33.6215372, 30.7215125, 27.737969, 24.6637155, 21.4396148, 18.0913369, 14.648484, 11.1144899, 7.49278858, 3.78681398, 0, -4.20404561, -8.43513042, -12.6428882, -16.7769527, -20.7869576, -24.6225368, -29.4472857, -36.2423878, -42.0547553, -46.3411145, -48.5581916, -48.162713, -44.6114048, -47.2420788, -49.2606505, -43.1250808, -26.2710014, 3.86595568, 49.8501587, 114.245976, 188.587294, 293.492819, 432.021615, 607.232747, 806.962399, 1071.81931, 1386.59523, 1736.08191, 2100.7761, 2478.35461, 2845.05278, 3255.50918, 3749.4546, 4345.50524, 5025.01448, 5735.30498, 6443.8395, 7071.55114, 7555.23546, 7831.68799},
{90.6686478, 90.338945, 89.8498427, 89.2065534, 88.4142895, 87.4782634, 86.4036875, 85.1957743, 83.8597362, 82.4007857, 80.8241351, 79.1349969, 77.3385836, 75.4401075, 73.4416099, 71.3503094, 69.1736645, 66.9175363, 64.5877858, 62.1902741, 59.7308624, 57.2362237, 54.6885327, 52.0907763, 49.445941, 46.7570139, 44.0269815, 41.2851226, 38.5567652, 35.7820291, 32.9532257, 30.0626665, 27.1026627, 24.0655259, 20.8727666, 17.5614852, 14.1741411, 10.7175274, 7.19843724, 3.62366374, 0, -4.06090718, -8.10254597, -12.0635508, -15.8825563, -19.4981968, -22.8491068, -27.132479, -33.3563927, -38.4501259, -41.8412979, -42.9575283, -41.2264364, -36.0756416, -36.4842853, -35.8376593, -26.798047, -6.83938301, 26.5643983, 75.9393624, 143.811575, 212.08357, 319.941909, 466.480743, 650.794218, 862.613943, 1124.12827, 1423.1964, 1747.67751, 2084.00816, 2424.31545, 2731.24107, 3082.92956, 3537.96935, 4077.76369, 4684.63584, 5326.20908, 5987.9228, 6598.65193, 7106.5548, 7459.78976},
{95.4194996, 94.9744199, 94.3458563, 93.5410786, 92.5673567, 91.4319604, 90.1421596, 88.7052241, 87.1284237, 85.4190283, 83.5843078, 81.6315319, 79.5679705, 77.4008935, 75.1326394, 72.7735975, 70.3345307, 67.8237172, 65.2494357, 62.6199646, 59.9435824, 57.2613442, 54.5442252, 51.7959772, 49.020352, 46.2211015, 43.4019773, 40.5995521, 37.8454667, 35.0649948, 32.2485619, 29.3865936, 26.4695154, 23.4877529, 20.3632018, 17.1296398, 13.8260322, 10.4568216, 7.02645092, 3.53936279, 0, -4.2462037, -8.44478288, -12.5002812, -16.3172421, -19.8002093, -22.8537263, -26.6855879, -32.3077596, -36.5085628, -38.6633793, -38.1475908, -34.3365791, -26.605726, -22.9025218, -17.179601, -2.74072752, 22.792071, 61.7967668, 116.651332, 189.733739, 263.596099, 376.57103, 526.669187, 711.901228, 926.647159, 1177.87322, 1459.23331, 1764.38136, 2097.49779, 2420.65693, 2632.95065, 2903.44544, 3359.22954, 3869.45585, 4404.85502, 4965.83074, 5566.44504, 6136.45351, 6634.35268, 7018.6391},
{101.846145, 101.147731, 100.262496, 99.1975513, 97.9600035, 96.5569615, 94.9955335, 93.2828279, 91.4259529, 89.432017, 87.3081284, 85.0613955, 82.6989265, 80.2278299, 77.6380801, 74.9476243, 72.1757068, 69.3329406, 66.4299384, 63.4773132, 60.485678, 57.4779833, 54.4508003, 51.4130383, 48.3736061, 45.3414129, 42.3253675, 39.3907795, 36.5944834, 33.8105285, 31.0249232, 28.2236762, 25.3927962, 22.5182917, 19.4346176, 16.2236559, 12.9787608, 9.7169399, 6.45520101, 3.21055181, 0, -3.68378724, -7.22704665, -10.5403555, -13.5342911, -16.1194306, -18.2063515, -21.0997824, -25.8949643, -29.1688988, -30.2660929, -28.5310535, -23.3082876, -13.9423019, -8.20238269, -0.103490751, 16.9771316, 45.4181963, 87.5984154, 145.896501, 222.691165, 302.797812, 421.032832, 575.026689, 762.409844, 975.885659, 1225.23808, 1502.1851, 1798.44471, 2114.12912, 2415.77363, 2618.3873, 2857.47376, 3230.62205, 3675.97438, 4169.32865, 4684.71809, 5189.26697, 5679.50007, 6141.91923, 6563.02631},
{109.957867, 108.923771, 107.696307, 106.283107, 104.691802, 102.930024, 101.005405, 98.9255759, 96.6981698, 94.330818, 91.8311523, 89.2068045, 86.4654063, 83.6145897, 80.6454548, 77.5760913, 74.4258405, 71.2057155, 67.9267295, 64.5998959, 61.2362277, 57.7997181, 54.3548943, 50.9192633, 47.510332, 44.1456075, 40.8425968, 37.7092791, 34.8400388, 32.036054, 29.278097, 26.5469404, 23.8233566, 21.088118, 18.1070639, 14.9969343, 11.8907385, 8.81321199, 5.78908988, 2.84310746, 0, -3.28293647, -6.33516022, -9.05356875, -11.3350596, -13.0765301, -14.174878, -16.0101729, -19.7396202, -21.7136831, -21.2270435, -17.5743834, -10.0503846, 2.05027093, 12.7196716, 26.9099491, 50.0813972, 84.3124696, 131.68162, 194.267302, 274.14797, 350.227635, 466.768801, 620.509952, 808.189573, 1022.38726, 1270.10006, 1543.07583, 1833.06243, 2139.56356, 2431.05959, 2635.46524, 2856.25309, 3163.22021, 3542.38549, 3979.04626, 4441.16822, 4893.91506, 5330.99006, 5736.36707, 6094.01992},
{119.074672, 117.759823, 116.217882, 114.459582, 112.495658, 110.336844, 107.993873, 105.477481, 102.7984, 99.9673644, 96.9951093, 93.8923682, 90.6698751, 87.3383642, 83.9063765, 80.3860334, 76.7896223, 73.1283257, 69.4133263, 65.6558067, 61.8669495, 57.9512264, 54.0412684, 50.1629958, 46.3423287, 42.6051873, 38.9774916, 35.5978285, 32.587855, 29.7040929, 26.9267158, 24.2358979, 21.6118129, 19.0346348, 16.2844216, 13.4679411, 10.7071094, 8.02303235, 5.43681598, 2.96956625, 0.642389138, -2.31001716, -4.9715306, -7.21243689, -8.90302176, -9.91357094, -10.1143701, -10.8729696, -13.3319374, -13.7814292, -11.4837931, -5.70137737, 4.30346968, 19.2683999, 34.8047905, 54.8931933, 83.9021962, 123.617988, 175.826758, 242.314694, 324.867985, 399.313533, 514.196004, 666.062089, 851.458481, 1063.45811, 1307.17626, 1574.99112, 1859.28093, 2159.87331, 2446.79825, 2659.00754, 2872.53587, 3134.43182, 3455.95783, 3831.93732, 4236.88674, 4657.31296, 5052.06814, 5389.4322, 5637.68509},
{128.495326, 127.074107, 125.349104, 123.337657, 121.057102, 118.524777, 115.758021, 112.77417, 109.590562, 106.224536, 102.693428, 99.0145773, 95.2053203, 91.2829951, 87.2836963, 83.2128958, 79.074646, 74.8824479, 70.6498026, 66.3902114, 62.1171752, 57.720481, 53.3544299, 49.049609, 44.836605, 40.746005, 36.8083958, 33.1657559, 29.9433259, 26.9053427, 24.0371649, 21.3241507, 18.7516587, 16.305047, 13.8190975, 11.3744239, 9.06303794, 6.90109728, 4.90475952, 3.09018229, 1.4735232, -0.793048375, -2.71006388, -4.14204299, -4.95350539, -5.00897077, -4.17295882, -3.71878542, -4.70807799, -3.63676357, 0.202654687, 7.51767362, 19.0157901, 35.4045009, 51.4404702, 71.5957135, 100.792698, 140.956113, 194.010651, 261.881001, 346.491854, 436.641211, 558.397569, 710.286632, 890.834104, 1091.36933, 1328.16899, 1591.12322, 1870.12216, 2157.70833, 2435.81478, 2673.27241, 2895.86752, 3111.89571, 3385.10739, 3718.32051, 4076.70075, 4436.6436, 4767.2776, 5040.20866, 5227.04271},
{138.900398, 137.337422, 135.391602, 133.087098, 130.44807, 127.498678, 124.263082, 120.765443, 117.02992, 113.080674, 108.941865, 104.637653, 100.192198, 95.6296595, 91.0129641, 86.3417484, 81.6127149, 76.8420946, 72.0461186, 67.2410178, 62.4430231, 57.528747, 52.6733214, 47.9122593, 43.2810742, 38.8152792, 34.5503876, 30.751528, 27.6493265, 24.73026, 21.9366103, 19.2106597, 16.49469, 13.7309833, 9.83315317, 5.39422106, 1.08507419, -2.94159973, -6.533113, -9.53677793, -11.7999068, -12.1167774, -11.5331679, -10.0418217, -7.63548268, -4.30689424, -0.0488000308, 3.44288832, 4.72818055, 7.88680239, 13.6175512, 22.6192243, 35.5906192, 53.2305331, 70.1560327, 90.9132406, 120.508639, 160.884993, 213.985069, 281.75163, 366.127442, 463.170325, 585.968126, 734.458352, 908.578505, 1100.01084, 1329.05576, 1585.74455, 1860.10848, 2144.0068, 2421.9869, 2675.82075, 2901.51559, 3083.49661, 3307.1074, 3583.65042, 3885.38171, 4198.36692, 4492.47968, 4743.11362, 4925.66238},
{150.492141, 148.740377, 146.534433, 143.904424, 140.880463, 137.492664, 133.771141, 129.746008, 125.447379, 120.905368, 116.150089, 111.211655, 106.120181, 100.905781, 95.6494103, 90.3490205, 84.9987135, 79.6182038, 74.2272058, 68.8454341, 63.4926031, 58.0444393, 52.6845308, 47.4524779, 42.3878808, 37.5303397, 32.9194547, 28.9855324, 26.100171, 23.3687472, 20.672222, 17.8915564, 14.9077111, 11.601647, 5.59600934, -1.79962227, -9.10472355, -15.9764138, -22.0718122, -27.0480381, -30.5622108, -28.5501535, -24.904218, -19.7954601, -13.3949353, -5.87369959, 2.59719142, 9.71708841, 13.505339, 18.8829889, 26.5436657, 37.1809971, 51.4886108, 70.1601344, 88.6285408, 110.915367, 141.440359, 181.973167, 234.283443, 300.140838, 381.315003, 474.099497, 590.161596, 730.114744, 894.572384, 1077.6267, 1295.97687, 1542.41088, 1809.71674, 2095.08978, 2378.09595, 2639.604, 2862.82661, 3020.70131, 3195.47472, 3406.67752, 3645.53151, 3922.39098, 4199.38914, 4455.09829, 4668.09072},
{161.612584, 159.667115, 157.206125, 154.264113, 150.875577, 147.075014, 142.896922, 138.3758, 133.546144, 128.442453, 123.099225, 117.550958, 111.832149, 105.977296, 100.052322, 94.0718457, 88.0481062, 82.0091739, 75.9831191, 69.9980119, 64.0819226, 58.2016273, 52.4549554, 46.878442, 41.5086225, 36.3820321, 31.5352061, 27.2761023, 23.8718937, 20.7101138, 17.7170917, 14.8191564, 11.942637, 9.01386256, 4.4868292, -0.780730804, -5.81963604, -10.4024039, -14.3015518, -17.289597, -19.139057, -17.5768668, -14.7036351, -10.5743881, -5.24415237, 1.23204592, 8.79918041, 15.7981135, 20.8107444, 27.6176602, 36.8151553, 48.9995239, 64.7670605, 84.7140596, 104.239416, 127.227319, 157.955375, 198.082965, 249.269468, 313.174262, 391.456729, 487.744948, 599.674335, 730.018462, 881.550901, 1048.86479, 1254.91206, 1492.64982, 1755.03523, 2037.67667, 2325.57748, 2619.73529, 2861.71536, 2991.38112, 3129.50183, 3302.22954, 3496.20653, 3710.8285, 3933.98728, 4157.49651, 4373.16983},
{177.48227, 175.503869, 172.943216, 169.838793, 166.229083, 162.152567, 157.647727, 152.753046, 147.507005, 141.948087, 136.114773, 130.045546, 123.778887, 117.353279, 110.83074, 104.234863, 97.5874593, 90.9221961, 84.2727413, 77.6727629, 71.1559286, 64.7702854, 58.533136, 52.4761623, 46.6310463, 41.0294699, 35.7031149, 30.7482842, 26.2515706, 22.09014, 18.2694361, 14.7949029, 11.671984, 8.90612311, 6.4000957, 4.2242939, 2.45688145, 1.1243021, 0.252999572, -0.130582374, 0, 0.107665064, 0.86020865, 2.36190121, 4.71701321, 8.02981511, 12.4045774, 17.1481302, 21.687129, 28.0326146, 36.6126438, 47.8552734, 62.1885602, 80.0405611, 97.444392, 117.608365, 144.073471, 178.166715, 221.215102, 274.545638, 339.48533, 421.429031, 513.437617, 619.067665, 741.875754, 879.019633, 1049.83962, 1250.21376, 1476.0201, 1723.93173, 1987.44155, 2289.09571, 2536.59024, 2645.32769, 2757.98307, 2901.89402, 3057.58613, 3212.0205, 3373.22828, 3542.05763, 3719.35668},
{203.880451, 202.208214, 199.864409, 196.892449, 193.33575, 189.237727, 184.641796, 179.59137, 174.129866, 168.300697, 162.14728, 155.71303, 149.041361, 142.175688, 135.235082, 128.215097, 121.105564, 113.934421, 106.72961, 99.5190709, 92.3307445, 85.1683129, 78.087325, 71.1190715, 64.294843, 57.6459303, 51.2036241, 45.0011085, 39.0712832, 33.4409113, 28.1405147, 23.2006152, 18.6517348, 14.5243953, 10.9819919, 7.97099029, 5.42779827, 3.35575953, 1.75821776, 0.638516679, 0, -0.204729396, 0.0854206845, 0.880801607, 2.19176474, 4.02866144, 6.40184309, 8.80951187, 10.8268265, 13.6887456, 17.6135694, 22.8195984, 29.525133, 37.9484735, 47.1793729, 58.1500548, 71.8781689, 88.8128507, 109.403236, 134.098459, 163.347656, 184.145996, 219.852907, 270.187002, 334.866894, 420.243061, 509.675555, 610.841165, 731.41668, 883.008518, 1061.50458, 1288.88822, 1489.49727, 1597.87824, 1694.42859, 1798.16764, 1908.18382, 2037.43503, 2163.3321, 2275.40424, 2363.18065}
};

  const double a4tab[21][81] = {
{-631.969628, -629.931559, -626.93986, -622.993704, -618.092259, -612.234699, -605.420193, -597.647913, -588.91703, -579.226715, -568.576139, -556.964473, -544.390888, -530.854555, -515.846825, -499.688117, -482.737292, -465.097392, -446.871459, -428.162534, -409.073659, -390.431317, -371.515197, -352.328428, -332.874139, -313.155461, -293.175523, -272.835853, -252.053245, -231.073772, -209.941815, -188.701758, -167.397985, -146.074879, -125.285446, -104.752312, -84.159603, -63.4476689, -42.5568579, -21.4275187, 0, 20.492378, 41.5808063, 63.5035042, 86.4986913, 110.804587, 136.659411, 164.137905, 193.339377, 224.892938, 259.103185, 296.274714, 336.71212, 380.720002, 451.856144, 535.71509, 616.131221, 688.652886, 748.828432, 792.206209, 814.334565, 847.454547, 888.473551, 839.16961, 601.32076, -334.087968, -1051.99176, -2143.56025, -4199.96305, -7855.60243, -13571.9501, -23786.9485, -33106.7677, -37360.6537, -39761.0933, -40691.7902, -39750.1263, -36377.5393, -31425.5653, -25265.023, -18266.7309},
{-505.215416, -508.196276, -509.772603, -509.990387, -508.895619, -506.534288, -502.952385, -498.195899, -492.310822, -485.343142, -477.33885, -468.343937, -458.404391, -447.566204, -435.751584, -423.084826, -409.699594, -395.667196, -381.058941, -365.946138, -350.400093, -334.883419, -319.02208, -302.833344, -286.334476, -269.542745, -252.475417, -235.102086, -217.39951, -199.498947, -181.437021, -163.250356, -144.975574, -126.649301, -108.741298, -91.0141828, -73.197789, -55.240145, -37.0892796, -18.6932216, 0, 17.996917, 36.5393224, 55.8235696, 76.0460123, 97.4030037, 120.090898, 144.841731, 172.307048, 201.402322, 232.1064, 264.398133, 298.256369, 333.659957, 398.01185, 473.942308, 542.005708, 596.571523, 632.009229, 642.688299, 622.97821, 589.700061, 568.112915, 450.1228, 127.635743, -717.548639, -1675.26318, -3105.72753, -5369.16134, -8880.19603, -13835.8159, -21538.445, -28962.0227, -33785.1132, -37519.4657, -40137.7205, -41147.9007, -39965.1597, -36840.7475, -31740.8799, -24631.7726},
{-416.815775, -419.057598, -420.469505, -421.053329, -420.810899, -419.744046, -417.854601, -415.144394, -411.615257, -407.26902, -402.107514, -396.13257, -389.346018, -381.749689, -373.154365, -363.682735, -353.471949, -342.562916, -330.996543, -318.813739, -306.055411, -292.666119, -278.796426, -264.500547, -249.832696, -234.847087, -219.597935, -204.212585, -188.80739, -173.261705, -157.600052, -141.846951, -126.026926, -110.164496, -94.647616, -79.2708986, -63.8014475, -48.1894474, -32.385083, -16.3385389, 0, 15.8033661, 32.1194751, 49.1192595, 66.9736521, 85.8535854, 105.929992, 128.799096, 155.842954, 183.824465, 212.335847, 240.969319, 269.317099, 296.971406, 353.48309, 419.49243, 473.371969, 508.586135, 518.599357, 496.876061, 436.880677, 151.559476, 37.8684007, -64.67, -316.533178, -780.861526, -1858.22995, -3592.31237, -6026.7827, -9156.64679, -13171.5827, -18819.4885, -24531.9546, -29153.6948, -33275.8654, -36682.7175, -38862.2881, -39230.2155, -37787.9462, -34337.2185, -28679.7708},
{-352.810903, -355.83479, -357.93123, -359.12342, -359.43456, -358.887848, -357.506483, -355.313665, -352.332593, -348.586464, -344.098478, -338.891834, -332.989731, -326.415368, -319.156081, -311.257756, -302.768993, -293.720326, -284.14229, -274.065418, -263.520245, -252.52262, -241.11979, -229.344318, -217.228767, -204.805698, -192.107676, -179.198016, -166.135417, -152.8789, -139.44854, -125.864412, -112.14659, -98.3151495, -84.6689645, -71.0517401, -57.2860767, -43.3350226, -29.1616263, -14.728936, 0, 14.8074113, 30.0201505, 45.7103479, 61.9501343, 78.81164, 96.3669955, 116.389741, 140.39776, 164.394917, 187.762516, 209.881862, 230.134257, 247.901005, 297.350154, 355.856786, 398.16278, 416.534098, 403.236704, 350.53656, 250.699628, 81.0333436, -86.8683061, -342.335077, -774.696726, -1485.36227, -2533.86603, -4024.03276, -6059.68721, -8785.088, -12182.7583, -16539.1067, -20994.3126, -24867.3396, -28756.8221, -32428.176, -35236.2064, -36409.007, -36092.1278, -34118.3483, -30320.4477},
{-308.350086, -311.217498, -313.195697, -314.31487, -314.605207, -314.096896, -312.820128, -310.805091, -308.081975, -304.680968, -300.632259, -295.966039, -290.712495, -284.901817, -278.614509, -271.848932, -264.599635, -256.886517, -248.729473, -240.148403, -231.163203, -221.738348, -211.956813, -201.846149, -191.433908, -180.747641, -169.8149, -158.71565, -147.52198, -136.136116, -124.564327, -112.812884, -100.888056, -88.7961136, -76.7068069, -64.522988, -52.135134, -39.5160767, -26.6386475, -13.4756781, 0, 13.7532862, 27.8822177, 42.4225625, 57.4100889, 72.8805651, 88.8697592, 107.833319, 131.863276, 155.20919, 176.924281, 196.061768, 211.674871, 222.816808, 260.477512, 303.506922, 328.337507, 327.490095, 293.485516, 218.844597, 96.0881681, -27.8598655, -216.405701, -529.145665, -1025.67608, -1762.42935, -2806.80613, -4214.60606, -6041.62879, -8393.47272, -11176.5412, -14348.9799, -17747.3316, -21230.7092, -24884.6691, -28450.6124, -31439.9222, -33295.7233, -33964.1906, -33220.0041, -30837.8436},
{-284.293575, -285.455423, -286.02854, -286.023086, -285.449223, -284.317112, -282.636914, -280.418789, -277.6729, -274.409407, -270.638472, -266.370255, -261.614918, -256.382622, -250.60771, -244.348306, -237.668273, -230.593279, -223.148995, -215.361088, -207.255227, -198.95188, -190.368826, -181.518641, -172.413902, -163.067186, -153.49107, -143.717927, -133.777153, -123.633993, -113.292987, -102.758672, -92.03559, -81.128279, -70.1406469, -59.0143722, -47.6836118, -36.1325801, -24.3454913, -12.3065598, 0, 12.6663563, 25.6213633, 38.8702577, 52.4182761, 66.2706552, 80.4326318, 97.6599695, 120.295132, 141.766533, 160.962606, 176.771785, 188.082505, 193.783199, 229.149613, 270.051435, 289.603997, 279.253008, 230.444179, 134.623223, -16.7641508, -233.206538, -489.797272, -842.946308, -1349.0636, -1986.97268, -3004.46287, -4364.84115, -6031.4145, -7957.99019, -10136.3744, -12487.2395, -15199.5083, -18463.6459, -21789.2919, -24932.21, -27708.8938, -30015.7269, -31368.8032, -31413.8615, -29796.6404},
{-269.261202, -269.893529, -269.922537, -269.369733, -268.256623, -266.604716, -264.435518, -261.770537, -258.63128, -255.039254, -251.015968, -246.582927, -241.76164, -236.573613, -231.101996, -225.329301, -219.233375, -212.823116, -206.107424, -199.095198, -191.795336, -184.183849, -176.307067, -168.178431, -159.811381, -151.21936, -142.415809, -133.451427, -124.371315, -115.099826, -105.635271, -95.975965, -86.1202194, -76.0663476, -65.8536243, -55.4544499, -44.8381242, -33.9945817, -22.9137572, -11.5855851, 0, 12.2126797, 24.6532377, 37.2820741, 50.0595888, 62.9461818, 75.902253, 92.1509699, 114.42523, 134.883784, 152.162243, 164.896218, 171.721322, 171.273165, 196.398655, 224.091212, 228.754614, 202.026835, 135.545848, 20.9496279, -150.123852, -310.660599, -586.222934, -997.812488, -1566.43089, -2298.62462, -3251.05128, -4427.36643, -5831.22566, -7492.21992, -9336.19869, -11173.0819, -13320.9669, -16036.5623, -18968.5552, -21908.6681, -24641.4471, -26991.1174, -28630.3656, -29283.5396, -28674.987},
{-260.19861, -260.329225, -259.924836, -259.002871, -257.580756, -255.67592, -253.305787, -250.487786, -247.239344, -243.577888, -239.520843, -235.085639, -230.2897, -225.150455, -219.702272, -213.951861, -207.904649, -201.574597, -194.975667, -188.121822, -181.027023, -173.729491, -166.215579, -158.495896, -150.581055, -142.481666, -134.20834, -125.822067, -117.376265, -108.761087, -99.9666886, -90.9832271, -81.8008583, -72.4097385, -62.81914, -53.0071261, -42.9503135, -32.6349483, -22.0472768, -11.1735453, 0, 12.4164605, 25.0318929, 37.7317017, 50.4012913, 62.9260662, 75.1914308, 90.4528171, 111.45927, 130.038073, 144.706292, 153.980991, 156.379236, 150.418091, 167.309568, 184.88779, 178.512015, 140.011827, 61.2168106, -66.0434513, -249.939375, -421.336883, -714.638337, -1141.92747, -1715.28801, -2427.92313, -3338.48847, -4436.41128, -5711.11882, -7182.28864, -8748.59704, -10135.8843, -11813.1906, -14138.1958, -16810.2724, -19624.9526, -22323.0395, -24636.7153, -26387.3746, -27365.9987, -27363.5686},
{-255.977882, -255.881463, -255.267847, -254.15314, -252.553448, -250.484879, -247.963539, -245.005536, -241.626976, -237.843965, -233.672612, -229.129022, -224.229303, -218.989561, -213.39858, -207.489751, -201.298535, -194.846627, -188.155723, -181.247519, -174.14371, -166.976683, -159.642156, -152.146537, -144.496235, -136.697658, -128.757215, -120.733546, -112.677443, -104.470422, -96.0976845, -87.54443, -78.7958597, -69.837174, -60.5898415, -51.07938, -41.3361316, -31.3583329, -21.1442203, -10.6920305, 0, 12.0834578, 24.2514848, 36.3470461, 48.2131066, 59.6926314, 70.6285854, 84.449911, 104.046714, 120.68748, 132.759149, 138.648667, 136.742975, 125.429017, 134.94212, 143.521823, 126.996883, 77.2399175, -13.8764533, -154.479611, -352.696935, -567.399046, -886.363643, -1321.38453, -1884.25553, -2545.61083, -3418.77101, -4466.13874, -5650.11671, -6941.4336, -8277.51408, -9446.43994, -10847.9371, -12784.0016, -15099.507, -17645.0049, -20184.1708, -22494.7141, -24385.0854, -25653.2552, -26097.1941},
{-259.301517, -258.908377, -257.986116, -256.552805, -254.626518, -252.225327, -249.367304, -246.070523, -242.353056, -238.232975, -233.728354, -228.857265, -223.63778, -218.087972, -212.193491, -205.992919, -199.527296, -192.821326, -185.899714, -178.787165, -171.508383, -164.22357, -156.80322, -149.253326, -141.57988, -133.788873, -125.886297, -117.935423, -109.990914, -101.921802, -93.710822, -85.3407098, -76.7941999, -68.0540274, -58.9282556, -49.5101175, -39.9060675, -30.1345682, -20.2140821, -10.1630719, 0, 11.3798946, 22.6798009, 33.7261314, 44.3452987, 54.3637154, 63.6077941, 75.75625, 93.9089438, 108.680992, 118.33465, 121.13217, 115.335806, 99.2078126, 102.39748, 103.311507, 77.9807358, 18.24746, -84.0460249, -237.057424, -448.944442, -690.660789, -1032.33718, -1482.18179, -2048.4028, -2687.14723, -3535.04044, -4537.81771, -5641.21434, -6777.12204, -7932.80681, -8992.50289, -10228.8405, -11850.2394, -13815.456, -16038.211, -18346.2706, -20623.88, -22615.3228, -24112.5969, -24907.7002},
{-272.576592, -271.509326, -269.905439, -267.78581, -265.171318, -262.08284, -258.541257, -254.567445, -250.182285, -245.406655, -240.261433, -234.767498, -228.945729, -222.817004, -216.396127, -209.707821, -202.777266, -195.626582, -188.277893, -180.753317, -173.074977, -165.336801, -157.479185, -149.514334, -141.454452, -133.311744, -125.098412, -116.905882, -108.813673, -100.644566, -92.3785988, -83.9958093, -75.4762357, -66.7999162, -57.7079606, -48.3315539, -38.8199673, -29.2021095, -19.5068898, -9.76321692, 0, 11.0216661, 21.8299644, 32.2208922, 41.9904468, 50.9346252, 58.8494248, 69.5324439, 86.1799924, 99.0197821, 106.223032, 105.960959, 96.4047834, 75.725723, 72.5747565, 65.8414949, 32.1083658, -36.6877954, -148.610153, -311.721872, -534.086117, -754.965378, -1105.54732, -1581.21004, -2177.33164, -2855.74813, -3694.57455, -4648.93882, -5673.96884, -6720.15437, -7756.5378, -8621.99595, -9668.11695, -11166.9082, -12949.962, -14934.2741, -17017.0813, -19145.3929, -21086.8863, -22670.375, -23724.6727},
{-290.803667, -289.274472, -287.128883, -284.394981, -281.100843, -277.274549, -272.944177, -268.137807, -262.883517, -257.209386, -251.143493, -244.713917, -237.948737, -230.876032, -223.51659, -215.903102, -208.068811, -200.043286, -191.856098, -183.536816, -175.115011, -166.733726, -158.293387, -149.807891, -141.291137, -132.757024, -124.219449, -115.792997, -107.577122, -99.3449735, -91.0695665, -82.7239183, -74.2810457, -65.7139654, -56.7644909, -47.551899, -38.2129678, -28.7680048, -19.2373174, -9.64121332, 0, 11.7686104, 23.2513321, 34.1374748, 44.1163481, 52.8772616, 60.109525, 69.661843, 84.7579215, 95.1414814, 98.8129842, 93.7728915, 78.021665, 49.5597661, 33.8139514, 11.434708, -39.0035261, -125.110111, -254.494408, -434.765777, -673.533578, -911.692133, -1278.54529, -1765.94828, -2365.75631, -3055.88021, -3862.5713, -4760.95166, -5726.14334, -6770.29687, -7757.44895, -8304.76651, -9080.37076, -10594.8809, -12284.7376, -14027.3107, -15837.8677, -17770.7606, -19586.105, -21144.3248, -22305.8438},
{-314.860794, -312.399881, -309.315105, -305.634028, -301.38421, -296.593211, -291.288593, -285.497916, -279.248741, -272.568628, -265.485138, -258.025831, -250.218269, -242.090012, -233.621957, -224.871184, -215.898305, -206.740427, -197.434653, -188.018091, -178.527844, -169.058221, -159.581225, -150.12606, -140.721932, -131.398046, -122.183608, -113.282612, -104.872792, -96.565407, -88.3186921, -80.0908819, -71.8402108, -63.5249134, -54.6077698, -45.360441, -36.0920914, -26.8622969, -17.730633, -8.75667548, 0, 10.1361592, 19.7071329, 28.4245933, 36.0002126, 42.145663, 46.5726166, 53.4597832, 66.3146467, 74.1676824, 74.9168018, 66.4599163, 46.6949373, 13.5197764, -8.04792363, -37.2662924, -95.4463037, -190.237163, -329.288076, -520.248248, -770.766884, -1027.86707, -1411.75897, -1912.81564, -2521.41019, -3210.41698, -4013.37242, -4899.65133, -5838.62854, -6829.70559, -7752.17711, -8281.29834, -8959.95466, -10188.4093, -11658.3146, -13267.0828, -14935.01, -16552.2472, -18106.642, -19552.7363, -20845.072},
{-344.679856, -341.022052, -336.721252, -331.8069, -326.308441, -320.255316, -313.67697, -306.602846, -299.062388, -291.085039, -282.700244, -273.937445, -264.826086, -255.395611, -245.632885, -235.594288, -225.33942, -214.906433, -204.33348, -193.658713, -182.920285, -172.031953, -161.173444, -150.40009, -139.767223, -129.330176, -119.144279, -109.546635, -100.832006, -92.3819813, -84.1374849, -76.0394417, -68.0287763, -60.0464131, -51.3358712, -42.2792565, -33.3114654, -24.5160711, -15.9766467, -7.77676527, 0, 9.05698653, 17.2769261, 24.3294609, 29.8842333, 33.6108856, 35.1790599, 39.0222633, 48.8581746, 52.9654922, 49.0795762, 34.9357869, 8.26948448, -33.1839709, -69.8652883, -117.845006, -194.845637, -307.595725, -462.823816, -667.258454, -927.628182, -1172.06384, -1551.03265, -2052.14941, -2663.0289, -3356.3428, -4156.56543, -5033.37999, -5956.46971, -6923.29205, -7820.20756, -8370.28056, -8997.37166, -9998.44616, -11243.2334, -12663.9936, -14157.567, -15604.3705, -16985.934, -18249.317, -19341.579},
{-377.73975, -373.104753, -367.713572, -361.606407, -354.823458, -347.404927, -339.391013, -330.821917, -321.73784, -312.178982, -302.185543, -291.797725, -281.055728, -269.999751, -258.680042, -247.130445, -235.384047, -223.478992, -211.453428, -199.3455, -187.193353, -174.725993, -162.333401, -150.096417, -138.095882, -126.412635, -115.127518, -104.688851, -95.4897345, -86.7323237, -78.3482499, -70.2691445, -62.4266393, -54.752366, -46.39279, -37.7762419, -29.3904859, -21.3277529, -13.6802736, -6.54027905, 0, 8.0344215, 14.9827519, 20.4508465, 24.0445606, 25.3697495, 24.0322684, 24.5292753, 30.6229514, 30.2234886, 20.9507164, 0.424463843, -33.7354397, -83.909165, -135.640282, -201.959857, -297.173412, -427.104905, -597.578294, -814.417538, -1083.44659, -1323.86307, -1698.16163, -2193.72605, -2797.94011, -3485.15224, -4274.9, -5138.92486, -6048.9683, -7003.2791, -7894.07684, -8485.75838, -9099.45099, -9946.79918, -10994.753, -12210.4725, -13512.3499, -14852.487, -16096.9162, -17141.0119, -17880.1487},
{-411.395283, -406.424621, -400.436968, -393.495203, -385.662204, -377.00085, -367.574018, -357.444586, -346.675432, -335.329434, -323.469471, -311.158421, -298.459161, -285.43457, -272.235267, -258.868624, -245.335373, -231.680445, -217.94877, -204.185281, -190.434908, -176.384163, -162.485896, -148.83454, -135.524528, -122.650289, -110.306257, -98.9695798, -89.0598987, -79.7565246, -70.9984936, -62.7248418, -54.8746053, -47.3868203, -39.3994688, -31.3583368, -23.7698461, -16.736882, -10.3623296, -4.74907387, 0, 5.82942382, 10.3061333, 13.044481, 13.6588196, 11.7635017, 6.97287981, 3.62121897, 5.33356082, 0.438408409, -13.3663218, -38.3827132, -76.9128493, -131.258814, -184.276109, -250.568796, -346.212957, -477.488314, -650.674588, -872.051499, -1147.89877, -1439.87244, -1836.72835, -2332.59123, -2921.58585, -3572.41332, -4341.90989, -5193.6922, -6091.37687, -7008.53331, -7878.91978, -8574.4393, -9231.47707, -9918.11911, -10798.793, -11871.3261, -13019.5422, -14158.9815, -15192.3213, -16026.6241, -16568.9523},
{-447.704261, -442.311139, -435.636192, -427.765098, -418.783533, -408.777175, -397.831702, -386.03279, -373.466118, -360.217362, -346.372199, -332.016307, -317.235363, -302.115045, -286.898914, -271.57277, -256.110459, -240.565366, -224.990874, -209.440365, -193.967224, -178.224329, -162.720881, -147.565575, -132.867107, -118.734172, -105.275468, -93.0029713, -82.3680642, -72.5151463, -63.3891673, -54.9350772, -47.097826, -39.8223634, -32.2517941, -24.8383183, -18.0948322, -12.1302966, -7.05367222, -2.97391977, 0, 3.9144123, 6.20741074, 6.47237482, 4.30268407, -0.708281985, -8.96714383, -16.6217659, -20.4599418, -31.0714639, -50.5921446, -81.1577962, -124.904231, -183.967261, -239.909578, -307.881626, -404.591142, -536.382, -709.598075, -930.583241, -1205.68137, -1520.42345, -1920.72148, -2405.59936, -2974.08102, -3596.46265, -4342.62963, -5177.1329, -6064.52341, -6976.43908, -7856.16996, -8624.20503, -9304.38621, -9874.12416, -10584.9548, -11466.2867, -12424.6604, -13408.4535, -14321.5634, -15083.0508, -15611.977},
{-486.985302, -481.063369, -473.621951, -464.766211, -454.601306, -443.232398, -430.764646, -417.30321, -402.95325, -387.819926, -372.008398, -355.623825, -338.771369, -321.556188, -304.281376, -286.926879, -269.457662, -251.938399, -234.433764, -217.008433, -199.727079, -182.251363, -165.104632, -148.40722, -132.27946, -116.841686, -102.214231, -88.9322037, -77.46839, -66.9513455, -57.3329911, -48.5652476, -40.6000359, -33.3892771, -26.1172825, -19.2215655, -13.1977443, -8.15474813, -4.20150593, -1.44694686, 0, 2.46866343, 3.07511633, 1.37368948, -3.08128632, -10.7354803, -22.0345617, -33.804155, -43.4138398, -59.9652257, -85.3738362, -121.555195, -170.424826, -233.898252, -294.951034, -367.480185, -466.816392, -598.74919, -769.068117, -983.56271, -1248.02251, -1549.17979, -1927.42098, -2384.28343, -2921.30444, -3517.40583, -4229.90974, -5033.21508, -5901.72077, -6825.70807, -7731.92885, -8541.16776, -9225.3043, -9715.70976, -10261.4576, -10921.8731, -11668.0294, -12532.7126, -13390.1094, -14168.4748, -14796.0635},
{-523.530987, -517.18179, -509.099572, -499.404296, -488.215926, -475.654428, -461.839764, -446.8919, -430.930799, -414.076427, -396.448746, -378.167723, -359.353319, -340.125501, -320.743696, -301.239643, -281.634523, -262.019777, -242.486841, -223.127154, -204.032154, -185.1605, -166.754747, -148.924671, -131.780048, -115.430653, -99.9862631, -85.8913394, -73.5400536, -62.2419086, -51.970787, -42.7005711, -34.4051436, -27.0583868, -19.9670615, -13.5270739, -8.18482821, -4.05066127, -1.23490977, 0.152089566, 0, 0.63444541, -0.827294392, -4.83197899, -11.826368, -22.2572209, -36.5712973, -52.069956, -66.5271897, -87.9107707, -117.944593, -158.352549, -210.858533, -277.186437, -340.946978, -415.322511, -514.866481, -645.007681, -811.174906, -1018.79695, -1273.30261, -1585.92326, -1950.09153, -2374.72401, -2868.73727, -3413.26677, -4085.75571, -4861.78357, -5716.92985, -6636.49509, -7566.89564, -8502.24447, -9262.53211, -9658.4135, -10079.7924, -10610.8568, -11207.0811, -11865.721, -12545.62, -13218.7166, -13856.9493},
{-573.544328, -567.3913, -559.264478, -549.297849, -537.625402, -524.381123, -509.699, -493.713021, -476.557174, -458.365445, -439.271823, -419.410296, -398.91485, -377.919473, -356.673279, -335.237426, -313.664358, -292.064516, -270.548338, -249.226266, -228.208738, -207.724455, -187.749263, -168.377269, -149.702582, -131.819308, -114.821554, -98.9907231, -84.5800736, -71.2358692, -58.9761693, -47.8190334, -37.782521, -28.8846914, -20.80112, -13.7665228, -8.04154082, -3.71428576, -0.872869407, 0.394596441, 0, -0.352641544, -2.79107424, -7.65091515, -15.2677814, -25.9772899, -40.115058, -55.4388162, -70.0936536, -90.5792079, -118.277801, -154.571754, -200.84339, -258.47503, -314.439123, -379.233729, -464.447671, -574.410692, -713.452534, -885.902939, -1096.09165, -1361.90342, -1660.1381, -2002.53274, -2400.82438, -2844.97238, -3400.43194, -4052.80709, -4787.70187, -5593.56535, -6447.46639, -7421.88042, -8212.53356, -8539.75889, -8877.49863, -9315.97155, -9789.15075, -10251.9446, -10731.7661, -11231.3108, -11753.2741},
{-655.005025, -650.22828, -643.153049, -633.930991, -622.713764, -609.653026, -594.900437, -578.607655, -560.926338, -542.008144, -522.004733, -501.067762, -479.348891, -456.999777, -434.44623, -411.666479, -388.618003, -365.396384, -342.097206, -318.816053, -295.648506, -272.707411, -250.068705, -227.82559, -206.071264, -184.898927, -164.401779, -144.67741, -125.822753, -107.920505, -91.0620844, -75.338907, -60.8423896, -47.6639487, -36.2708463, -26.5167378, -18.2268289, -11.4156602, -6.09777208, -2.28770517, 0, 0.899018952, 0.226125436, -2.05369074, -5.97543976, -11.5741318, -18.8847771, -26.3284314, -32.5686664, -41.4996417, -53.8116848, -70.1951234, -91.3402849, -117.937497, -147.032956, -181.622281, -224.976933, -278.532618, -343.72504, -421.989903, -514.762911, -581.305808, -694.756475, -854.432254, -1059.65049, -1330.79573, -1615.21967, -1937.52009, -2322.2948, -2806.55382, -3377.65823, -4109.87456, -4751.12504, -5083.61292, -5376.10302, -5691.28314, -6024.29668, -6414.84462, -6791.94206, -7122.33809, -7372.7818}
};

  const double a5tab[21][81] = {
{519.523893, 518.131239, 515.8465, 512.673834, 508.617401, 503.681359, 497.869868, 491.187084, 483.637169, 475.224279, 465.952575, 455.826214, 444.849356, 433.026159, 419.876634, 405.711214, 390.876979, 375.477116, 359.614813, 343.393256, 326.915633, 310.97921, 294.897239, 278.677049, 262.325971, 245.851335, 229.260471, 212.468357, 195.403847, 178.295097, 161.186937, 144.124194, 127.151698, 110.314275, 94.0951208, 78.2617512, 62.5485019, 46.9105375, 31.3030226, 15.6811219, 0, -14.8448983, -29.9685477, -45.5456425, -61.7508773, -78.7589465, -96.7445447, -115.834203, -136.161689, -158.016862, -181.593973, -207.087271, -234.691007, -264.599431, -315.508826, -375.908989, -432.889195, -482.859257, -522.228987, -547.408199, -554.806705, -574.962906, -596.142264, -541.649647, -334.789923, 439.372044, 1023.20821, 1899.29847, 3550.22271, 6493.90189, 11106.8927, 19378.4641, 26907.6181, 30293.5484, 32120.0118, 32707.9753, 31765.1304, 28865.1246, 24731.0574, 19694.1679, 14085.6953},
{401.023924, 404.584455, 406.799776, 407.720454, 407.397055, 405.880146, 403.220293, 399.468062, 394.674018, 388.888729, 382.162761, 374.54668, 366.091051, 356.846442, 346.759978, 335.947662, 324.533328, 312.588699, 300.185499, 287.395453, 274.290286, 261.318544, 248.123086, 234.723595, 221.139753, 207.391243, 193.497745, 179.435614, 165.187713, 150.877329, 136.541738, 122.218216, 107.944036, 93.7564757, 80.056375, 66.6510166, 53.320162, 40.0267222, 26.733608, 13.4037303, 0, -12.7748555, -25.7999156, -39.2144432, -53.157701, -67.768952, -83.1874591, -100.053584, -118.932391, -138.764961, -159.487092, -181.034587, -203.343245, -226.348868, -271.660391, -325.50311, -372.461762, -408.039085, -427.737821, -427.060708, -401.510488, -366.642495, -338.58577, -231.772114, 39.3666742, 733.614349, 1509.27518, 2659.77438, 4478.53714, 7303.46344, 11294.5542, 17504.0752, 23491.4724, 27390.0091, 30357.5262, 32364.0939, 33034.5068, 31900.5993, 29211.3213, 24971.1506, 19184.5649},
{320.853928, 323.645506, 325.647365, 326.8671, 327.312305, 326.990576, 325.909508, 324.076696, 321.499735, 318.186219, 314.143744, 309.379905, 303.902297, 297.718514, 290.663545, 282.854178, 274.420263, 265.404701, 255.850392, 245.800236, 235.297134, 224.310145, 212.966208, 201.31842, 189.419881, 177.323689, 165.082942, 152.819472, 140.644784, 128.447624, 116.253182, 104.086648, 91.9732114, 79.938062, 68.2956295, 56.8881296, 45.5358834, 34.2049193, 22.8612652, 11.4709494, 0, -10.9706148, -22.1747344, -33.7312581, -45.7590852, -58.377115, -71.7042469, -87.0660335, -105.606713, -124.55994, -143.554672, -162.21987, -180.184491, -197.077497, -236.11891, -282.013897, -317.681883, -337.92647, -337.551259, -311.359852, -254.155851, -10.8720224, 90.9067418, 180.871687, 388.714058, 763.065479, 1633.56374, 3032.62932, 4992.68271, 7505.62609, 10725.4348, 15243.7119, 19832.0917, 23594.4221, 26919.4491, 29614.2054, 31277.3477, 31430.3293, 30111.7931, 27184.0668, 22509.478},
{264.604952, 268.01221, 270.555054, 272.259659, 273.152196, 273.258839, 272.605761, 271.219136, 269.125135, 266.349933, 262.919703, 258.860616, 254.198848, 248.96057, 243.144059, 236.793135, 229.953732, 222.657727, 214.937001, 206.823432, 198.348899, 189.538801, 180.432392, 171.062448, 161.461742, 151.663049, 141.699143, 131.629487, 121.509533, 111.308241, 101.047548, 90.7493939, 80.4357158, 70.1284518, 60.0684983, 50.1392797, 40.2076459, 30.2507491, 20.2457412, 10.1697742, 0, -10.1716849, -20.49872, -31.0198002, -41.7736206, -52.7988762, -64.1342619, -77.2252061, -93.3057587, -109.050954, -123.92831, -137.40534, -148.949562, -158.028492, -191.573447, -231.678243, -258.357804, -265.462155, -246.841316, -196.345311, -107.824161, 37.7516773, 181.758735, 395.308243, 749.511432, 1325.47786, 2169.65629, 3365.15583, 4995.08563, 7176.1872, 9890.67249, 13346.7352, 16906.2956, 20068.6758, 23231.2194, 26180.9312, 28396.171, 29233.1402, 28845.3183, 27114.3841, 23922.0167},
{226.936794, 230.072047, 232.394903, 233.936573, 234.728264, 234.801186, 234.186548, 232.915558, 231.019425, 228.529359, 225.476568, 221.89226, 217.807645, 213.253932, 208.310342, 202.977713, 197.253243, 191.158322, 184.714336, 177.942674, 170.864725, 163.458199, 155.794193, 147.900128, 139.803422, 131.531496, 123.11177, 114.613821, 106.100893, 97.4996009, 88.8202487, 80.0731387, 71.2685737, 62.4168564, 53.6474839, 44.895356, 36.0863504, 27.2063895, 18.2413957, 9.17729174, 0, -9.32918894, -18.7963188, -28.4120651, -38.1871037, -48.13211, -58.2577598, -70.546693, -86.685238, -101.968879, -115.607611, -126.811428, -134.790326, -138.754299, -162.950269, -190.74978, -203.62919, -195.677397, -160.983299, -93.6357938, 12.2762194, 119.095052, 277.884525, 535.748638, 939.791391, 1533.80335, 2373.06163, 3500.69416, 4959.82882, 6835.43808, 9045.11621, 11519.1146, 14208.7872, 17071.2823, 20064.301, 22950.6412, 25337.2994, 26758.9674, 27195.3575, 26474.4695, 24424.3036},
{208.120791, 209.580113, 210.509836, 210.922517, 210.830712, 210.24698, 209.183878, 207.653963, 205.669793, 203.243925, 200.388916, 197.117324, 193.441706, 189.374619, 184.867629, 179.971877, 174.743121, 169.206393, 163.386727, 157.309156, 150.998712, 144.5579, 137.923582, 131.11009, 124.131758, 117.00292, 109.737907, 102.365653, 94.9128929, 87.3590567, 79.7125504, 71.9817802, 64.1751523, 56.3010729, 48.4310367, 40.53354, 32.5723035, 24.5428292, 16.4406192, 8.26117543, 0, -8.46945348, -17.0127789, -25.6176185, -34.2716146, -42.9624094, -51.6776452, -62.6192298, -77.6563512, -91.4820897, -103.185025, -111.853735, -116.576801, -116.442801, -139.244831, -165.912887, -175.204164, -160.335968, -114.525607, -30.9903895, 97.0523769, 277.669417, 490.299248, 779.530743, 1189.95278, 1700.99006, 2517.96931, 3607.28304, 4935.32375, 6461.28195, 8173.15611, 9967.4643, 12094.3003, 14791.2017, 17529.173, 20080.8976, 22307.089, 24120.9478, 25137.6962, 25074.1402, 23647.0861},
{196.99112, 197.852928, 198.189355, 198.02151, 197.370502, 196.257439, 194.703432, 192.729589, 190.35702, 187.606833, 184.500137, 181.058042, 177.301656, 173.252089, 168.982587, 164.481277, 159.732339, 154.746218, 149.53336, 144.104209, 138.469211, 132.610235, 126.570248, 120.363641, 114.004805, 107.50813, 100.888009, 94.1862905, 87.4406999, 80.5999704, 73.6673442, 66.6460632, 59.5393694, 52.3505049, 45.099863, 37.7798357, 30.3815167, 22.9046398, 15.3489389, 7.71414766, 0, -8.13710569, -16.3066832, -24.4615817, -32.5546505, -40.5387386, -48.3666954, -58.602855, -73.4191447, -86.5240549, -96.8100879, -103.169746, -104.495532, -99.6799483, -114.611839, -131.105722, -128.932611, -101.463129, -42.0679019, 55.8824482, 199.017297, 332.674538, 559.635715, 895.530662, 1355.98921, 1943.25865, 2705.97899, 3643.72114, 4756.05602, 6064.72725, 7502.78762, 8874.38743, 10537.2961, 12783.959, 15205.7867, 17604.9312, 19811.9288, 21677.5237, 22943.2431, 23387.8477, 22790.0979},
{190.832536, 191.182498, 191.070976, 190.515396, 189.533185, 188.141768, 186.358572, 184.201023, 181.686546, 178.832569, 175.656516, 172.175814, 168.40789, 164.370168, 160.095692, 155.592007, 150.865481, 145.930345, 140.800832, 135.491174, 130.015603, 124.4074, 118.659118, 112.782357, 106.788719, 100.689805, 94.4972168, 88.259642, 82.0201968, 75.7018023, 69.3010013, 62.8143362, 56.2383498, 49.5695846, 42.8059539, 35.9431333, 28.9766944, 21.9028994, 14.7180105, 7.41828999, 0, -8.32900583, -16.675172, -24.9333518, -32.9983983, -40.7651645, -48.1285038, -57.6696175, -71.566051, -83.289292, -91.6434499, -95.4326343, -93.4609546, -84.5325202, -93.2601919, -102.121369, -91.6398416, -55.3407527, 13.2507551, 120.609539, 273.210457, 414.532412, 654.724254, 1002.23974, 1465.53261, 2035.81164, 2764.06811, 3638.06196, 4645.55311, 5800.59664, 7012.06701, 8011.00495, 9284.86209, 11212.7157, 13428.9288, 15737.5879, 17929.6157, 19775.3061, 21140.6326, 21858.9509, 21763.6162},
{188.572122, 188.667871, 188.320343, 187.545776, 186.360407, 184.780474, 182.822216, 180.50187, 177.835674, 174.839867, 171.530685, 167.924367, 164.03715, 159.885273, 155.464333, 150.803623, 145.934004, 140.875935, 135.649876, 130.276286, 124.775626, 119.260585, 113.646656, 107.941562, 102.153024, 96.2887649, 90.3565074, 84.4012916, 78.4625502, 72.4587743, 66.3825339, 60.226399, 53.9829398, 47.6447264, 41.1445932, 34.5128992, 27.7845253, 20.9642599, 14.0568915, 7.06720873, 0, -8.0991995, -16.1338685, -23.9667389, -31.460543, -38.4780125, -44.8818797, -53.3854886, -66.2738445, -76.5935407, -83.0498692, -84.3481219, -79.1935907, -66.2915677, -69.5515217, -71.7344903, -53.6936662, -8.97903516, 68.8594169, 186.271704, 349.70784, 524.871315, 784.98631, 1138.2079, 1592.69116, 2121.10369, 2819.1361, 3652.35886, 4586.34238, 5594.64432, 6618.87351, 7444.53231, 8495.24079, 10098.3722, 12019.8806, 14111.2512, 16180.113, 18033.9887, 19524.6826, 20488.903, 20763.3579},
{192.640596, 192.425679, 191.760187, 190.661977, 189.148906, 187.238831, 184.949609, 182.299097, 179.305152, 175.985632, 172.358392, 168.44129, 164.252184, 159.80893, 155.105186, 150.174117, 145.050721, 139.757805, 134.318175, 128.754639, 123.090004, 117.461761, 111.762194, 105.99827, 100.176957, 94.3052222, 88.3900346, 82.4792313, 76.6145088, 70.7051111, 64.7414116, 58.7137839, 52.6126016, 46.4282381, 40.0060334, 33.4281099, 26.7875683, 20.1044475, 13.3987865, 6.69062437, 0, -7.58044865, -14.9752011, -22.0361381, -28.61514, -34.5640876, -39.7348614, -47.0396638, -58.9308472, -67.9427133, -72.6845537, -71.76566, -63.7953237, -47.3828366, -46.0843521, -42.7276859, -18.2522863, 33.8352296, 120.028244, 246.820141, 420.704303, 616.581004, 894.493792, 1259.59357, 1717.03125, 2227.35983, 2905.73808, 3703.79987, 4573.17903, 5454.46437, 6332.42487, 7077.50902, 7998.51352, 9337.6381, 10966.8088, 12792.4299, 14674.4371, 16510.3169, 18093.8205, 19254.0633, 19820.1611},
{205.201042, 204.348048, 203.041196, 201.300742, 199.146942, 196.600052, 193.680328, 190.408026, 186.803402, 182.886711, 178.67821, 174.198155, 169.466801, 164.504404, 159.33076, 153.966415, 148.431952, 142.747722, 136.934076, 131.011362, 124.999932, 118.983059, 112.90948, 106.790855, 100.638845, 94.4651098, 88.2813103, 82.1587966, 76.15995, 70.1537049, 64.1274852, 58.0687151, 51.9648188, 45.8032204, 39.3720536, 32.7848145, 26.1700843, 19.5560505, 12.9709002, 6.44282098, 0, -7.3490753, -14.4156901, -21.0308295, -27.0254786, -32.2306226, -36.4772465, -42.7807182, -53.6779141, -61.3855962, -64.4417882, -61.3845132, -50.7517947, -31.0816562, -25.2471131, -16.391784, 14.1827643, 72.9160298, 166.247511, 300.616705, 482.463112, 660.681848, 945.390279, 1331.37205, 1813.41082, 2356.24576, 3028.77002, 3789.71416, 4597.80879, 5408.08129, 6190.37179, 6777.22915, 7544.01912, 8786.78629, 10268.4158, 11898.0197, 13595.0589, 15311.8408, 16858.9378, 18095.2411, 18879.6421},
{221.690131, 220.398247, 218.58656, 216.281616, 213.509957, 210.298129, 206.672674, 202.660137, 198.287063, 193.579994, 188.565475, 183.270051, 177.720264, 171.94266, 165.966042, 159.815525, 153.516051, 147.093703, 140.574562, 133.984708, 127.350225, 120.795949, 114.235566, 107.681522, 101.146258, 94.6422182, 88.1818448, 81.8547985, 75.7391366, 69.6626667, 63.6064793, 57.5516645, 51.4793127, 45.3705141, 39.013682, 32.5117949, 25.9824161, 19.4460464, 12.9231866, 6.43433754, 0, -8.04107527, -15.7543762, -22.8871409, -29.1866072, -34.4000132, -38.2745968, -43.8795544, -53.7849183, -59.7947447, -60.3074499, -53.7214501, -38.4351615, -12.8470002, 2.69274262, 23.672487, 67.242313, 139.493879, 246.518844, 394.408865, 589.255602, 780.905091, 1078.63779, 1474.37621, 1960.04287, 2514.26833, 3161.76182, 3878.49565, 4640.44212, 5456.18109, 6203.86222, 6522.65532, 7063.54625, 8330.475, 9738.76737, 11163.6008, 12630.3005, 14190.7865, 15640.4513, 16862.1399, 17738.6977},
{243.041167, 240.916442, 238.268739, 235.124196, 231.508948, 227.44913, 222.97088, 218.100331, 212.863622, 207.286886, 201.396261, 195.217881, 188.777884, 182.102404, 175.188008, 168.079538, 160.824073, 153.453798, 146.000897, 138.497553, 130.975952, 123.529313, 116.120354, 108.772831, 101.510497, 94.3571074, 87.336416, 80.6075209, 74.3091827, 68.1415349, 62.0733782, 56.0735135, 50.1107416, 44.1538631, 37.7686538, 31.1788696, 24.6387738, 18.1996027, 11.9125925, 5.82897951, 0, -6.83643511, -13.134385, -18.661234, -23.1843668, -26.4711679, -28.2890216, -31.9863919, -40.3736424, -44.6553882, -43.1449803, -34.15577, -16.0011085, 13.005653, 32.7119582, 58.6200168, 107.88517, 186.661095, 301.10147, 457.359973, 661.59028, 867.621722, 1179.24059, 1586.96251, 2081.30312, 2637.14399, 3283.56464, 3992.32, 4735.16499, 5510.68417, 6210.14351, 6525.55868, 6991.93862, 8010.82803, 9232.1407, 10550.4892, 11905.9158, 13206.7859, 14442.7419, 15575.7181, 16567.6489},
{269.088009, 265.924239, 262.224134, 258.015465, 253.326007, 248.183532, 242.615814, 236.650625, 230.315739, 223.638929, 216.647968, 209.370629, 201.834685, 194.067909, 186.073201, 177.89407, 169.575906, 161.151569, 152.653921, 144.115822, 135.570133, 126.971984, 118.442702, 110.025882, 101.765121, 93.7040138, 85.886157, 78.5745132, 71.9990828, 65.6789304, 59.5685817, 53.6225628, 47.7953994, 42.0416175, 35.7514821, 29.2364724, 22.8507801, 16.6643459, 10.7471106, 5.16901506, 0, -6.10211882, -11.4602514, -15.8094327, -18.884698, -20.4210821, -20.1536201, -21.6460347, -27.8867423, -29.4559488, -24.5341191, -11.3017181, 12.0607894, 47.3729384, 78.7156359, 119.129938, 182.999637, 275.772557, 402.896523, 569.81936, 781.988894, 978.133378, 1286.266, 1694.78546, 2192.09049, 2753.06761, 3399.44548, 4103.40834, 4837.14042, 5597.6513, 6282.64914, 6628.25706, 7059.49464, 7882.44393, 8911.69889, 10073.7181, 11286.1547, 12446.5037, 13541.6988, 14528.1828, 15362.399},
{297.538563, 293.556225, 288.943238, 283.736585, 277.973249, 271.690214, 264.924463, 257.712979, 250.092744, 242.100743, 233.773957, 225.149371, 216.263967, 207.154729, 197.88078, 188.465097, 178.928981, 169.304887, 159.625269, 149.922583, 140.229283, 130.357193, 120.589868, 110.990234, 101.621217, 92.545741, 83.8267321, 75.8132842, 68.8114909, 62.2000155, 55.9255896, 49.9349446, 44.1748124, 38.5919243, 32.4983618, 26.2423386, 20.2201084, 14.5082154, 9.18320368, 4.32161718, 0, -5.43618466, -9.94023776, -13.1965411, -14.8894766, -14.7034259, -12.322771, -11.3742724, -14.8923013, -13.1345613, -4.18470192, 13.8736276, 42.956778, 84.9810999, 128.019953, 182.746814, 260.883015, 367.176375, 506.374714, 683.225853, 902.477611, 1096.46704, 1401.4294, 1805.90828, 2298.44724, 2855.37901, 3495.36723, 4192.30257, 4920.07567, 5676.13651, 6363.69775, 6757.39355, 7186.55806, 7877.68362, 8738.18483, 9726.41403, 10777.6599, 11849.6135, 12833.4199, 13642.9101, 14191.9155},
{325.967927, 321.759463, 316.699275, 310.843686, 304.249017, 296.971589, 289.067725, 280.593744, 271.605969, 262.160722, 252.314323, 242.123094, 231.643357, 220.931433, 210.135985, 199.25492, 188.279152, 177.246116, 166.193245, 155.157973, 144.177733, 133.034222, 122.055929, 111.315608, 100.886012, 90.8398934, 81.2500054, 72.4853239, 64.8703136, 57.7694243, 51.135133, 44.9199163, 39.076251, 33.5566141, 27.6663412, 21.767293, 16.2703157, 11.2602536, 6.82195087, 3.0402516, 0, -3.83572388, -6.53633503, -7.79301253, -7.29693548, -4.73928297, 0.188765894, 3.97288046, 3.67320435, 8.74015049, 21.0348622, 42.4184827, 74.7521554, 119.897023, 163.757313, 218.288572, 296.654215, 403.979242, 545.38865, 726.007439, 950.960608, 1187.36036, 1510.91855, 1915.91188, 2396.61706, 2924.84282, 3550.2867, 4240.26406, 4962.09027, 5692.37005, 6370.55074, 6867.86278, 7344.32832, 7893.98241, 8608.93485, 9476.692, 10400.6984, 11305.6616, 12115.2992, 12753.6661, 13144.8175},
{355.924553, 351.463197, 345.92638, 339.389563, 331.928211, 323.617786, 314.533751, 304.75157, 294.346705, 283.394619, 271.970776, 260.150639, 248.009671, 235.623334, 223.220949, 210.780647, 198.268917, 185.72975, 173.207141, 160.745081, 148.387565, 135.896053, 123.636089, 111.690687, 100.142859, 89.0756176, 78.5719749, 69.0240383, 60.7774699, 53.1762019, 46.1777443, 39.7396075, 33.8193017, 28.3743371, 22.7181984, 17.2158088, 12.2808414, 8.00253643, 4.47013431, 1.77287539, 0, -2.47767474, -3.61515922, -3.08588707, -0.563291916, 4.27919262, 11.7681329, 18.7689569, 22.6671699, 32.0653158, 48.6952942, 74.2890049, 110.578348, 159.295223, 205.2998, 260.997097, 340.069247, 447.699122, 589.069595, 769.363538, 993.763824, 1249.0487, 1575.51435, 1971.68525, 2436.08585, 2942.24165, 3550.34111, 4228.91015, 4946.47472, 5678.36192, 6372.69412, 6946.10806, 7452.8416, 7901.32133, 8470.01808, 9175.75662, 9940.06053, 10715.5055, 11425.7601, 12004.3475, 12384.791},
{387.430025, 382.686339, 376.665071, 369.457889, 361.156459, 351.852449, 341.637525, 330.603354, 318.841603, 306.44394, 293.50203, 280.107541, 266.352139, 252.327492, 238.313641, 224.283086, 210.194068, 196.099725, 182.053193, 168.107609, 154.316108, 140.455871, 126.894102, 113.72205, 101.030964, 88.9120906, 77.4566788, 67.070536, 58.1122035, 49.9207821, 42.4597989, 35.6927807, 29.5832546, 24.0947477, 18.578586, 13.385577, 8.91286911, 5.24920986, 2.48334686, 0.704027705, 0, -1.49811338, -1.47300968, 0.434489138, 4.58356111, 11.3333843, 21.0431367, 31.1274123, 39.4432642, 53.390707, 74.5245146, 104.399461, 144.570319, 196.591864, 246.451664, 305.552364, 386.474976, 493.958418, 632.741605, 807.563455, 1023.16289, 1267.6883, 1576.12367, 1949.31661, 2388.11474, 2873.75188, 3455.4562, 4110.53896, 4816.31143, 5564.37504, 6289.17058, 6910.67669, 7431.21712, 7813.68352, 8241.97583, 8760.16318, 9344.65662, 10021.3633, 10685.864, 11278.1692, 11738.2894},
{415.644597, 410.80384, 404.496223, 396.826118, 387.897897, 377.815933, 366.684597, 354.608262, 341.6913, 328.038083, 313.752984, 298.940374, 283.704625, 268.15011, 252.526896, 236.847188, 221.112163, 205.396391, 189.774444, 174.320894, 159.110312, 144.161103, 129.611761, 115.544616, 102.041996, 89.1862286, 77.0596429, 65.993561, 56.2818907, 47.411588, 39.3638818, 32.1200012, 25.6611749, 19.9686321, 14.4900332, 9.54414462, 5.49012131, 2.41832923, 0.419134351, -0.417097379, 0, -0.19189763, 1.31994565, 4.89557566, 10.8950383, 19.6783793, 31.6056445, 44.4793832, 56.4864388, 74.1021239, 98.7249103, 131.75327, 174.585674, 228.620594, 280.369375, 340.648141, 421.399902, 527.068169, 662.096454, 830.928265, 1038.00711, 1291.98541, 1588.48038, 1934.65271, 2337.66309, 2781.05892, 3330.24684, 3964.0518, 4661.29879, 5409.67817, 6161.41866, 6903.2878, 7499.09516, 7801.54674, 8123.45042, 8532.17509, 8990.76641, 9496.34206, 10014.0548, 10519.9545, 10990.0912},
{453.17581, 448.837173, 442.811713, 435.216307, 426.167836, 415.783177, 404.179211, 391.472816, 377.78087, 363.220254, 347.907847, 331.960526, 315.495171, 298.628662, 281.605983, 264.462973, 247.225773, 229.98506, 212.831509, 195.855796, 179.148598, 162.951874, 147.184124, 131.915128, 117.214671, 103.152535, 89.7985016, 77.3576024, 66.0145451, 55.5157189, 45.8759913, 37.1102299, 29.2333022, 22.2600757, 15.9192601, 10.4067475, 5.94009081, 2.59268896, 0.43794067, -0.450755246, 0, 0.43664773, 2.55661682, 6.63037865, 12.9284046, 21.7211662, 33.2791347, 45.7879692, 57.7465969, 74.4105134, 96.8966909, 126.322102, 163.803718, 210.458512, 255.584784, 307.776041, 376.520412, 465.35228, 577.806029, 717.416043, 887.716706, 1103.5243, 1345.46792, 1623.23514, 1946.51352, 2306.45731, 2758.46946, 3289.99774, 3888.48991, 4543.93457, 5236.15697, 6023.08053, 6654.59158, 6900.07118, 7152.52932, 7486.30119, 7845.51724, 8191.2698, 8547.00849, 8914.92465, 9297.20964},
{514.044111, 511.244973, 506.460749, 499.824339, 491.468637, 481.526543, 470.130952, 457.414762, 443.51087, 428.552173, 412.671569, 396.001954, 378.676226, 360.827281, 342.838575, 324.6845, 306.320485, 287.828178, 269.289225, 250.785274, 232.397974, 214.298224, 196.466094, 178.970904, 161.881976, 145.268632, 129.200193, 113.748283, 98.9841805, 84.9717004, 71.779229, 59.4751529, 48.1278582, 37.8057315, 28.8385816, 21.1294181, 14.5614618, 9.14962737, 4.90882923, 1.85398193, 0, -0.744376298, -0.243394776, 1.53252249, 4.61295341, 9.02747592, 14.8056679, 20.7043685, 25.671661, 32.7803872, 42.576898, 55.6075441, 72.4186764, 93.5566457, 116.626628, 144.035572, 178.41306, 220.907037, 272.665445, 334.836229, 408.567332, 461.903817, 552.18437, 679.028025, 842.053812, 1057.64344, 1284.06812, 1541.06193, 1848.35894, 2235.51207, 2692.79891, 3282.76076, 3795.71593, 4050.6964, 4271.96945, 4511.18603, 4762.95519, 5057.70467, 5339.62931, 5582.32165, 5759.37424}
};

  const double b3tab[21][81] = {
{43.0068882, 42.0198432, 41.0202058, 40.0082996, 38.9844479, 37.9489742, 36.902202, 35.8444546, 34.7760554, 33.697328, 32.6085957, 31.510182, 30.4024104, 29.2856042, 28.1351348, 26.9671105, 25.7995281, 24.6378149, 23.487398, 22.3537045, 21.2421617, 20.2361252, 19.2523312, 18.2854444, 17.3301295, 16.3810513, 15.4328744, 14.4741566, 13.494373, 12.5027911, 11.4965554, 10.4728101, 9.42869978, 8.36136873, 7.43944692, 6.55159635, 5.5734983, 4.46722121, 3.19483351, 1.71840363, 0, -1.82864689, -3.99856215, -6.57110885, -9.60765004, -13.1695488, -17.3181681, -23.1599594, -31.6443369, -40.3337383, -48.8651873, -56.8757074, -64.0023224, -69.8820558, -135.302016, -221.215459, -283.946851, -310.625471, -288.380599, -204.341513, -45.6374928, 1094.39585, 1322.02807, 1275.54747, 1593.24234, 5319.21786, 7157.42945, 10633.1226, 19271.5426, 33460.8266, 66137.5454, 158302.586, 233321.855, 232304.72, 214245.066, 191781.421, 164045.903, 133506.452, 102794.137, 74484.4135, 51152.7378},
{34.8479771, 34.0828232, 33.2979234, 32.4946199, 31.6742548, 30.8381703, 29.9877087, 29.124212, 28.2490225, 27.3634824, 26.4689337, 25.5667189, 24.6581798, 23.7446589, 22.8204984, 21.8914687, 20.9638698, 20.0404758, 19.1240605, 18.2173979, 17.323262, 16.48684, 15.6626349, 14.8475632, 14.0385411, 13.2324851, 12.4263115, 11.6119553, 10.7820998, 9.94557265, 9.10131288, 8.24825948, 7.38535144, 6.51152776, 5.87655893, 5.32070703, 4.66524731, 3.85781326, 2.84603836, 1.57755611, 0, -1.69213164, -3.83216388, -6.50655697, -9.80177113, -13.8042666, -18.6005036, -25.4550677, -35.4555167, -45.8712791, -56.3104583, -66.381158, -75.6914817, -83.8495329, -163.679532, -268.472799, -345.978518, -380.829042, -357.656722, -261.093911, -75.7729602, 1382.32395, 1669.1229, 1567.52846, 1860.4452, 5398.05478, 7863.99151, 12521.8726, 22635.3153, 38866.4163, 72283.3546, 159204.733, 230487.863, 232792.773, 220036.698, 203067.024, 179727.633, 149951.728, 118318.723, 87824.0906, 61463.3019},
{27.3016278, 26.3201609, 25.4316053, 24.6262982, 23.8945769, 23.2267788, 22.613241, 22.044301, 21.510296, 21.0015632, 20.5084399, 20.0212635, 19.5303712, 19.0261003, 18.4401786, 17.80002, 17.1374748, 16.4548684, 15.7545261, 15.0387734, 14.3099356, 13.5774014, 12.8354574, 12.0854534, 11.3287394, 10.5666652, 9.80058079, 9.02693396, 8.2429092, 7.46157766, 6.68627963, 5.92035536, 5.16714512, 4.42998917, 4.08113488, 3.89055052, 3.60027976, 3.13820596, 2.43221249, 1.41018271, 0, -1.63570072, -3.83620844, -6.7060606, -10.3497946, -14.8719479, -20.377058, -28.3310151, -39.9951502, -52.2188502, -64.5538995, -76.5520826, -87.7651839, -97.7449876, -189.784157, -310.45969, -400.009772, -440.857681, -415.426698, -306.140102, -95.4211725, 1828.26796, 2199.66409, 1975.70095, 2113.31228, 4971.62859, 8048.84049, 13984.5044, 25418.1767, 42730.9598, 75337.7724, 154867.656, 220221.658, 224088.91, 215027.204, 202272.431, 182777.578, 154477.671, 123662.487, 93773.5817, 68252.5085},
{19.3140409, 18.8127435, 18.3268192, 17.8544265, 17.3937237, 16.9428693, 16.5000216, 16.0633391, 15.6309802, 15.2011032, 14.7718666, 14.3414288, 13.9079482, 13.4695833, 13.0132296, 12.5441705, 12.0685417, 11.5868053, 11.0994235, 10.6068582, 10.1095716, 9.60921725, 9.10490122, 8.59692112, 8.08557452, 7.57115897, 7.05397204, 6.53172619, 6.00252231, 5.47283926, 4.94402423, 4.41742444, 3.89438708, 3.37625936, 3.26677856, 3.31273845, 3.23047363, 2.93902574, 2.35743642, 1.40474729, 0, -1.64066316, -3.93633285, -7.00899908, -10.9806518, -15.9732812, -22.108877, -31.1013368, -44.4253548, -58.3964885, -72.4903626, -86.1826014, -98.9488294, -110.264671, -213.913405, -349.711289, -450.335908, -495.973035, -466.808444, -343.027907, -104.817199, 1622.64393, 2013.60032, 1968.70522, 2388.61189, 5741.17106, 9061.29224, 15130.2506, 26729.3214, 45277.9892, 77642.9008, 149564.093, 208953.624, 215315.125, 207096.358, 192638.252, 171985.692, 147259.988, 120572.795, 94039.0621, 69773.735},
{12.7881556, 12.5733124, 12.333449, 12.0708547, 11.7878189, 11.4866307, 11.1695797, 10.838955, 10.497046, 10.146142, 9.78853229, 9.42650626, 9.06235319, 8.69836239, 8.35563103, 8.02455051, 7.69408857, 7.36268754, 7.02878975, 6.69083752, 6.34727319, 5.96573091, 5.57971601, 5.19192562, 4.8050569, 4.42180699, 4.04487306, 3.68477709, 3.35086534, 3.0271248, 2.71307548, 2.40823739, 2.11213051, 1.82427487, 2.01413104, 2.38393289, 2.60034188, 2.56675554, 2.18657142, 1.36318706, 0, -1.7361858, -4.24175745, -7.64969563, -12.092981, -17.7045944, -24.6175163, -34.7775333, -49.8580361, -65.6573797, -81.5724874, -97.0002825, -111.337688, -123.981628, -235.651602, -381.647502, -489.599547, -538.179962, -506.060977, -371.914819, -114.413714, 1427.64716, 1823.98063, 1923.64426, 2575.69561, 6217.63839, 9780.37322, 16019.0778, 27688.9297, 46640.1621, 78342.7864, 145553.488, 200330.975, 204985.626, 196733.976, 183520.692, 164854.785, 142255.175, 117636.798, 92779.5835, 69463.4617},
{8.67660924, 8.32370542, 8.00786559, 7.72441883, 7.46869424, 7.23602091, 7.02172791, 6.82114435, 6.6295993, 6.44242186, 6.25494111, 6.06248614, 5.86038604, 5.64396989, 5.36030655, 5.0352547, 4.69832613, 4.35472116, 4.0096401, 3.66828328, 3.33585102, 3.07039037, 2.81695643, 2.57345101, 2.33777595, 2.10783307, 1.88152419, 1.63946016, 1.36484999, 1.09694022, 0.840653377, 0.600911977, 0.382638539, 0.190755583, 0.61184877, 1.28287839, 1.79677541, 2.03948795, 1.89696414, 1.25515211, 0, -1.78730835, -4.44302392, -8.10816197, -12.9237377, -19.0307665, -26.5702635, -37.9782779, -55.3460025, -73.3267645, -91.1297216, -107.964032, -123.038852, -135.563341, -253.40089, -407.025591, -518.686891, -565.369623, -524.05862, -371.738712, -85.3947335, 1642.39205, 2114.71184, 2223.45834, 2860.52525, 5983.2463, 9855.43663, 16647.5077, 28529.8711, 46469.7646, 76247.1207, 140366.291, 192371.438, 195823.168, 187334.435, 174699.255, 157267.568, 136599.006, 113858.512, 90345.5841, 67359.7194},
{6.43126977, 6.27531737, 6.08715296, 5.87043508, 5.62882228, 5.3659731, 5.08554609, 4.7911998, 4.48659277, 4.17538354, 3.86123066, 3.54779269, 3.23872816, 2.93769562, 2.6778981, 2.44430419, 2.21964611, 2.00153933, 1.78759931, 1.57544153, 1.36268145, 1.11185147, 0.860495329, 0.611073717, 0.366047309, 0.127876785, -0.100977178, -0.325939738, -0.551251114, -0.755594685, -0.933307868, -1.07872808, -1.18619273, -1.25003925, -0.612569755, 0.319398433, 1.08969179, 1.57060448, 1.63443064, 1.15346443, 0, -1.83248891, -4.60962345, -8.47584522, -13.5755958, -20.0533167, -28.0534496, -40.4072577, -59.5422772, -79.1784521, -98.3692881, -116.168291, -131.628966, -143.804819, -261.626797, -414.639536, -524.070431, -566.498467, -518.502627, -356.661892, -57.5552467, 1284.42393, 1733.93985, 2047.08474, 2979.95078, 6611.84084, 10434.936, 16793.1685, 28030.4706, 45936.7797, 74518.013, 131486.776, 178779.249, 186637.764, 180981.383, 168083.622, 149637.232, 129364.855, 107450.345, 85283.9669, 64255.9838},
{5.29448937, 5.04323248, 4.77966155, 4.50506682, 4.22073848, 3.92796675, 3.62804184, 3.32225395, 3.0118933, 2.69825009, 2.38261454, 2.06627686, 1.75052725, 1.43665593, 1.13009179, 0.829506878, 0.53326, 0.241794828, -0.0444449594, -0.325015685, -0.599473671, -0.889773659, -1.16998018, -1.43655617, -1.68596459, -1.91466838, -2.11913049, -2.29959983, -2.45575642, -2.57501055, -2.65228792, -2.68251428, -2.66061534, -2.58151683, -1.71284295, -0.509613024, 0.518100112, 1.22660722, 1.47221906, 1.1112464, 0, -2.14884821, -5.31551114, -9.6238405, -15.197688, -22.1609055, -30.6373446, -43.7251104, -64.075391, -84.7002585, -104.515922, -122.438591, -137.384475, -148.269783, -263.962577, -413.823124, -518.888817, -555.586124, -500.34151, -329.581443, -19.7323883, 1253.13033, 1722.01745, 2095.46885, 3082.02438, 6471.78476, 10305.4467, 16589.4126, 27330.0851, 43794.8627, 70207.1597, 123503.677, 168501.148, 178068.16, 173279.753, 159892.662, 141050.456, 122301.905, 102188.135, 81639.254, 61585.3689},
{4.84351675, 4.51636659, 4.18454238, 3.84828129, 3.50782051, 3.16339724, 2.81524864, 2.46361192, 2.10872426, 1.75082284, 1.39014485, 1.02692748, 0.661407912, 0.293823333, -0.0856447143, -0.470397806, -0.853076352, -1.23138637, -1.60303387, -1.96572487, -2.31716538, -2.67235893, -3.00932511, -3.32338105, -3.60984386, -3.86403063, -4.0812585, -4.26199265, -4.40592472, -4.49606216, -4.52563179, -4.48786044, -4.37597493, -4.18320209, -3.08069225, -1.58172083, -0.261790799, 0.717722075, 1.19544202, 1.00999325, 0, -2.44350491, -5.97249041, -10.6865168, -16.6851445, -24.0679338, -32.934445, -46.6456857, -68.0725919, -89.5162313, -109.751912, -127.554941, -141.700627, -150.964279, -263.324215, -408.47352, -507.839091, -537.859663, -474.973969, -295.620744, 23.7612774, 1419.05114, 1935.38659, 2283.16973, 3172.80264, 6234.3338, 9909.7101, 16012.9004, 26357.8738, 42320.9255, 67029.0451, 114093.299, 154438.174, 165687.996, 163355.177, 151948.899, 134753.464, 116800.69, 97357.2159, 77655.0931, 58926.3714},
{4.82260682, 4.44699608, 4.06271369, 3.67030988, 3.27033492, 2.86333905, 2.44987252, 2.0304856, 1.60572854, 1.17615158, 0.742304977, 0.304738993, -0.135996125, -0.579350123, -1.0338924, -1.49330357, -1.95057395, -2.40328793, -2.84902991, -3.2853843, -3.70993548, -4.15038129, -4.57003383, -4.96231861, -5.32066118, -5.63848706, -5.90922177, -6.13521168, -6.31746272, -6.42806964, -6.45683582, -6.39356464, -6.22805947, -5.9501237, -4.63367847, -2.8479176, -1.23136316, 0.0388453389, 0.785568377, 0.831666437, 0, -2.65228347, -6.43086025, -11.4071197, -17.652451, -25.2382436, -34.2358866, -48.1992661, -70.1589809, -91.8593735, -111.957828, -129.111727, -141.978456, -149.215398, -255.278804, -391.897309, -482.791038, -504.977092, -435.47257, -251.294572, 70.5398011, 1522.76762, 2063.48009, 2388.61223, 3194.09905, 6026.91803, 9483.77153, 15281.8373, 25138.2933, 40758.8599, 63895.0867, 104532.188, 139701.079, 151597.138, 151500.514, 142684.992, 127836.263, 110418.605, 91370.037, 72489.8072, 55577.1629},
{5.12537602, 4.67830998, 4.22600324, 3.7684454, 3.30562605, 2.8375348, 2.36416125, 1.88549499, 1.40152563, 0.91224276, 0.417635984, -0.0823050996, -0.587590892, -1.09823179, -1.62810779, -2.16845532, -2.70946099, -3.24829829, -3.78214069, -4.30816171, -4.82353482, -5.36894761, -5.89204989, -6.38400556, -6.83597852, -7.23913267, -7.58463191, -7.87544036, -8.11274905, -8.25950615, -8.30208429, -8.22685609, -8.02019418, -7.66847118, -6.16973237, -4.13556985, -2.25238776, -0.708712917, 0.306927866, 0.606007777, 0, -2.81622665, -6.7603839, -11.8667879, -18.1697546, -25.7036003, -34.5026409, -48.1725756, -69.7824605, -90.8270269, -109.890495, -125.557084, -136.411015, -141.036506, -236.673829, -359.497024, -438.212308, -451.224619, -376.938899, -193.760087, 119.906876, 1347.48589, 1851.26156, 2237.22252, 3111.35744, 5837.1687, 9152.11593, 14571.197, 23609.4097, 37622.2754, 58603.2218, 96635.7552, 129169.689, 138671.701, 138033.642, 130473.028, 117553.059, 101287.315, 83542.1718, 66234.1913, 51279.9336},
{6.07046285, 5.585625, 5.08344396, 4.56462091, 4.02985707, 3.47985361, 2.91531175, 2.33693266, 1.74541756, 1.14146763, 0.52578407, -0.100931924, -0.737979157, -1.38465643, -2.06000494, -2.75083438, -3.44246004, -4.13014258, -4.80914267, -5.47472097, -6.12213813, -6.78464611, -7.41426743, -8.00101589, -8.53490532, -9.0059495, -9.40416227, -9.73401699, -9.99781433, -10.1509936, -10.1776975, -10.0620689, -9.78825047, -9.34038497, -7.63854184, -5.34000051, -3.18258604, -1.3680886, -0.0982983435, 0.424994576, 0, -3.08931915, -7.32137822, -12.6888395, -19.1843653, -26.8006179, -35.5302595, -49.0355546, -70.427366, -90.9239181, -109.027897, -123.241988, -132.068878, -134.011252, -220.744836, -331.830711, -399.777169, -404.029121, -324.031477, -139.229147, 170.932957, 1189.77782, 1682.44204, 2145.68871, 3076.28091, 5592.05626, 8657.798, 13515.5527, 21407.3669, 32906.4806, 51261.3599, 89151.949, 120385.793, 124416.287, 122483.423, 118297.419, 109902.396, 95063.6149, 78327.8777, 61930.0868, 48105.1446},
{7.30722818, 6.78165834, 6.2222456, 5.63092945, 5.00964936, 4.36034482, 3.6849553, 2.98542031, 2.26367931, 1.52167178, 0.761337221, -0.0153848974, -0.80655509, -1.61023387, -2.44139153, -3.2873914, -4.13431683, -4.97676958, -5.80935141, -6.62666409, -7.42330939, -8.25801351, -9.05239776, -9.79220786, -10.4631895, -11.0510886, -11.5416507, -11.9223401, -12.1803631, -12.2973559, -12.2583665, -12.0484429, -11.652633, -11.0559848, -9.17566822, -6.67227475, -4.28722941, -2.22400572, -0.686077233, 0.123082519, 0, -2.92215445, -6.95377737, -12.0686214, -18.2404393, -25.4429835, -33.6500069, -46.3563417, -66.5277358, -85.7186384, -102.473131, -115.335294, -122.849209, -123.558957, -201.015913, -299.988234, -358.809399, -358.63595, -280.62443, -105.93138, 184.286659, 1171.02488, 1639.01505, 2052.88883, 2877.27786, 5016.97965, 7850.88622, 12371.824, 19572.6195, 29795.4396, 45985.0894, 79044.5536, 106928.722, 112598.551, 111337.789, 106287.883, 97341.2319, 84708.7183, 70576.344, 56456.772, 43862.6649},
{9.20514842, 8.63722588, 8.01374915, 7.3382654, 6.6143218, 5.8454655, 5.03524369, 4.18720353, 3.30489219, 2.39185683, 1.45164463, 0.487802757, -0.496121627, -1.49658135, -2.52715272, -3.5734562, -4.61981607, -5.65918269, -6.68450644, -7.68873768, -8.66482677, -9.67964643, -10.6420155, -11.5346751, -12.3403666, -13.0418309, -13.6218094, -14.0573956, -14.3265315, -14.4254627, -14.3392234, -14.0528478, -13.5513704, -12.8198253, -10.7519076, -8.02303678, -5.39124208, -3.06478227, -1.25191614, -0.160902459, 0, -2.89490659, -6.87163049, -11.873619, -17.8443194, -24.727179, -32.4656451, -44.1354082, -62.3415014, -79.5378213, -94.3960238, -105.587765, -111.784701, -111.658487, -179.78201, -266.811531, -317.657921, -315.467867, -243.388055, -84.5651731, 177.854093, 1055.49709, 1480.67478, 1860.11891, 2600.56127, 4472.5578, 6985.40964, 10982.4341, 17306.9484, 26267.2918, 40311.7159, 68486.5005, 92528.7394, 98412.2179, 97620.7655, 92766.4715, 84466.186, 73852.4758, 61990.28, 49967.5925, 38872.4072},
{11.9539984, 11.396919, 10.7419251, 9.9961151, 9.16658737, 8.26044025, 7.28477208, 6.24668123, 5.15326603, 4.01162485, 2.82885603, 1.61205793, 0.368328902, -0.895232707, -2.18239559, -3.47918686, -4.77081105, -6.04794703, -7.3012737, -8.52146993, -9.69921461, -10.888297, -12.0075697, -13.0389955, -13.9645373, -14.7661581, -15.4258207, -15.9226181, -16.2360743, -16.3550149, -16.262568, -15.9418619, -15.3760248, -14.548185, -12.3130815, -9.36766573, -6.49430298, -3.90692415, -1.81946011, -0.445841767, 0, -2.83192104, -6.72447656, -11.5965936, -17.367199, -23.95522, -31.2795835, -42.0007638, -58.3672847, -73.7427208, -86.9328428, -96.7434215, -101.980228, -101.449032, -160.189052, -235.106538, -278.094161, -274.410187, -209.312884, -68.0605165, 164.088647, 884.061076, 1259.43462, 1620.23501, 2296.48794, 3967.08945, 6101.52072, 9448.44848, 14756.5395, 22360.0286, 34250.8782, 58000.5016, 78152.0784, 82774.3965, 82064.3976, 78248.386, 71601.9274, 62799.6613, 52870.6706, 42723.2044, 33265.5121},
{15.6671109, 15.2502665, 14.6526298, 13.8888203, 12.9734576, 11.921161, 10.7465502, 9.46424457, 8.0888636, 6.63502681, 5.11735366, 3.55046366, 1.94897629, 0.327511038, -1.28246908, -2.87699986, -4.45339206, -6.0004714, -7.5070636, -8.96199437, -10.3540894, -11.7138662, -12.9827007, -14.1436609, -15.1798145, -16.0742294, -16.8099733, -17.3781596, -17.7686925, -17.9454023, -17.8880901, -17.576557, -16.9906041, -16.1100324, -13.7676545, -10.6688606, -7.60586307, -4.79307008, -2.4448898, -0.775730382, 0, -2.42321699, -5.8798827, -10.2956087, -15.5960064, -21.7066874, -28.5532632, -38.631874, -54.0524075, -68.5940473, -81.1386877, -90.568223, -95.7645474, -95.6095552, -144.935686, -207.230314, -241.745606, -235.919234, -177.18887, -52.9921887, 149.233139, 762.447176, 1093.2023, 1412.08142, 1989.66745, 3399.62048, 5164.93501, 7919.88365, 12298.739, 18650.8127, 28465.2603, 47623.0852, 63949.4113, 68060.681, 67420.7648, 63840.1625, 57976.2463, 51008.8496, 43181.0877, 35055.4132, 27194.2787},
{20.7028796, 20.5861279, 20.1539088, 19.4333935, 18.4517532, 17.2361592, 15.8137827, 14.2117952, 12.4573677, 10.5776715, 8.59987803, 6.55115842, 4.45868398, 2.34962596, 0.323808252, -1.63755819, -3.5587619, -5.42749212, -7.23143807, -8.95828898, -10.5957341, -12.1542692, -13.5956273, -14.9043477, -16.06497, -17.0620336, -17.8800781, -18.5192514, -18.9773562, -19.2016102, -19.1702154, -18.8613735, -18.2532866, -17.3241566, -14.8846365, -11.6515239, -8.42999473, -5.43706336, -2.88974419, -1.00505161, 0, -2.21275942, -5.44624247, -9.62451741, -14.6716525, -20.511716, -27.0687761, -36.3999129, -50.2416955, -63.4179179, -74.9865771, -84.00567, -89.5331937, -90.6271451, -131.051248, -181.582493, -209.613846, -205.059116, -157.832113, -57.8466478, 104.98347, 594.229003, 862.120608, 1121.81634, 1586.47426, 2713.72533, 4113.69602, 6292.90951, 9757.88901, 14790.8997, 22571.2387, 37853.8867, 50706.6617, 53429.424, 52491.7453, 49449.0084, 44759.2656, 39356.2846, 33368.3027, 27198.171, 21248.7405},
{26.2786764, 26.8149455, 26.8307674, 26.3716265, 25.4830075, 24.2103949, 22.5992733, 20.6951271, 18.5434409, 16.1896994, 13.679387, 11.0579882, 8.37098771, 5.66386998, 3.13542106, 0.734146418, -1.60305272, -3.86204835, -6.02871249, -8.08891713, -10.0285343, -11.8312131, -13.4853554, -14.9771402, -16.2927466, -17.4183534, -18.3401396, -19.061775, -19.584301, -19.8520745, -19.8441726, -19.5396726, -18.9176517, -17.9571871, -15.4599763, -12.1499111, -8.84000616, -5.75016203, -3.10027925, -1.11025839, 0, -2.18440409, -5.3852273, -9.51922572, -14.5031554, -20.2537726, -26.6878332, -35.1634238, -46.8220536, -58.1340934, -68.4310726, -77.0445209, -83.305968, -86.5469435, -119.067102, -159.342218, -183.352521, -183.686185, -152.931381, -83.6762811, 31.4909424, 378.75362, 568.315242, 756.471386, 1099.51763, 1936.39386, 2972.8742, 4584.42565, 7146.5152, 10821.9368, 16624.1765, 28601.6773, 38298.484, 39062.8219, 37650.5301, 35460.6162, 32288.2609, 28204.785, 23780.7761, 19416.6161, 15512.6874},
{30.7209326, 32.3181309, 33.1233752, 33.2056257, 32.6338428, 31.4769868, 29.804018, 27.6838966, 25.1855831, 22.3780376, 19.3302205, 16.1110921, 12.7896127, 9.43474254, 6.36402724, 3.48917116, 0.703061833, -1.97618648, -4.53045955, -6.94164312, -9.19162296, -11.192986, -13.0064874, -14.6235837, -16.0357312, -17.2343863, -18.2110054, -18.9885104, -19.5850951, -19.9169783, -19.9628406, -19.7013622, -19.1112237, -18.1711055, -15.6711163, -12.3418315, -9.00379556, -5.87880157, -3.18864237, -1.15511088, 0, -2.05100706, -5.1331807, -9.17747401, -14.1148401, -19.8762319, -26.3926027, -34.3899995, -44.4749973, -54.6773882, -64.6052914, -73.8668264, -82.0701122, -88.8232684, -115.820738, -148.698752, -171.421702, -179.080132, -166.764588, -129.565612, -62.5737494, 153.460398, 267.996353, 383.268615, 601.511685, 1178.28366, 1837.62182, 2865.74751, 4548.88207, 7038.95973, 11025.0632, 19420.5602, 26011.4796, 25852.1266, 24394.1761, 22732.7513, 20557.3096, 17767.5076, 14827.8707, 12010.3547, 9586.91506},
{33.2118217, 37.2674517, 40.093284, 41.7904447, 42.4600599, 42.2032557, 41.1211581, 39.3148933, 36.8855873, 33.9343662, 30.5623561, 26.870683, 22.9604731, 18.9328525, 15.2375346, 11.7551279, 8.33985421, 5.02153894, 1.83000748, -1.20491475, -4.05340236, -6.52948658, -8.78104985, -10.7998312, -12.5775698, -14.1060045, -15.3768746, -16.3943759, -17.1608327, -17.6381981, -17.8131533, -17.6723794, -17.2025577, -16.3903692, -14.1693483, -11.1924007, -8.19215066, -5.37069192, -2.93011813, -1.07252293, 0, -1.59714616, -4.1511867, -7.63185005, -12.0088647, -17.251959, -23.3308615, -30.4691984, -38.8524451, -47.8432309, -57.3081937, -67.1139712, -77.1272012, -87.2145214, -108.962979, -134.824838, -156.365179, -171.053327, -176.358606, -169.750341, -148.697858, -43.8984837, 2.44346809, 43.0311626, 130.567765, 428.429069, 716.326467, 1179.76921, 2004.26657, 3330.98416, 5478.46212, 9981.34654, 13433.8445, 13098.0778, 12052.9428, 10942.9317, 9628.42492, 8131.29836, 6611.55752, 5182.11805, 3955.89562},
{32.9833959, 41.9638812, 49.0019939, 54.2435725, 57.834455, 59.9204799, 60.6474854, 60.1613096, 58.6077911, 56.1327678, 52.8820783, 49.0015607, 44.6370532, 39.9343943, 35.4826326, 31.14723, 26.7600988, 22.3764224, 18.0513838, 13.8401663, 9.79795311, 6.2160221, 2.8808555, -0.184969778, -2.95887684, -5.41828877, -7.54062865, -9.21369604, -10.3387575, -11.1075362, -11.5338454, -11.6314981, -11.4143075, -10.8960868, -9.48107211, -7.56869726, -5.60453816, -3.72709184, -2.07485534, -0.786325716, 0, -0.605736375, -1.88690262, -3.87822766, -6.6144404, -10.1302698, -14.4604447, -19.5041537, -25.1809518, -31.8496605, -39.6000426, -48.5218608, -58.7048776, -70.2388559, -84.6194615, -101.047078, -118.615667, -137.127426, -156.384551, -176.18924, -196.343689, -174.409178, -186.787085, -223.983128, -276.503022, -332.503198, -388.284267, -431.532825, -449.935465, -441.679614, -362.949371, -45.2891724, 163.197081, -19.4728738, -202.899705, -322.640033, -440.529833, -632.727459, -791.822389, -872.984671, -831.384353}
};

  const double b4tab[21][81] = {
{-76.9329587, -74.9120768, -72.8831959, -70.8465369, -68.8023206, -66.750768, -64.6921, -62.6265373, -60.554301, -58.4756119, -56.3906909, -54.2997588, -52.2030367, -50.1007452, -47.9461158, -45.769095, -43.6031866, -41.4582228, -39.3440357, -37.2704577, -35.247321, -33.4197451, -31.6435908, -29.9100064, -28.2101399, -26.5351394, -24.8761532, -23.2108204, -21.5188101, -19.8235718, -18.1217387, -16.4099441, -14.6848212, -12.943003, -11.4512563, -10.0353269, -8.50051257, -6.78819309, -4.83974807, -2.59655716, 0, 2.76386468, 6.03212818, 9.89720271, 14.4515005, 19.7874337, 25.9974146, 34.7489177, 47.4727453, 60.495156, 73.2690363, 85.2472724, 95.8827507, 104.628358, 205.529346, 338.199126, 435.090883, 476.309414, 441.959521, 312.146002, 66.9736571, -1739.1593, -2089.86491, -1995.19665, -2465.208, -8373.90658, -11200.2837, -16591.1008, -30193.1191, -52541.1941, -104617.805, -252880.445, -373123.584, -369854.392, -339140.269, -301672.616, -256155.659, -206854.116, -157777.913, -113023.84, -76688.683},
{-61.0665593, -59.5405326, -57.9862446, -56.4062639, -54.803159, -53.1794984, -51.5378507, -49.8807845, -48.2108684, -46.5306708, -44.8427603, -43.1497056, -41.4540751, -39.7584374, -38.0534102, -36.3491224, -34.6566071, -32.9808773, -31.326946, -29.6998262, -28.1045309, -26.6183587, -25.1640537, -23.736646, -22.3311654, -20.9426417, -19.5661048, -18.1863601, -16.7897493, -15.3957502, -14.0035442, -12.6123126, -11.2212368, -9.82949807, -8.82586569, -7.96306628, -6.96433642, -5.74917049, -4.23706286, -2.3475079, 0, 2.5232123, 5.71548762, 9.70743045, 14.6296452, 20.6127364, 27.7873085, 38.0516951, 53.0386074, 68.6518075, 84.304145, 99.4084695, 113.377631, 125.624478, 248.967192, 411.077997, 531.044831, 585.084421, 549.413493, 400.248775, 113.806991, -2210.93178, -2654.26793, -2462.11744, -2880.39627, -8492.70369, -12312.0249, -19589.4638, -35576.1242, -61261.328, -114681.525, -254898.409, -369567.039, -372005.489, -350069.387, -321533.332, -283016.134, -234619.989, -183669.342, -135026.499, -93553.7637},
{-46.6827849, -44.8401848, -43.166788, -41.64566, -40.2598663, -38.9924727, -37.8265445, -36.7451475, -35.7313471, -34.768209, -33.8387986, -32.9261816, -32.0134235, -31.0835898, -30.014852, -28.8566321, -27.666292, -26.4483527, -25.207335, -23.9477595, -22.6741473, -21.4036985, -20.1265034, -18.8453316, -17.5629529, -16.2821369, -15.0056532, -13.72913, -12.4492681, -11.1859141, -9.94473733, -8.7314073, -7.55159344, -6.41096523, -5.88317934, -5.61459444, -5.20857505, -4.55461355, -3.5422023, -2.06083366, 0, 2.40324205, 5.65496541, 9.91367872, 15.3378906, 22.0861097, 30.3168448, 42.2279784, 59.7109529, 78.0479007, 96.569279, 114.605545, 131.487156, 146.54457, 288.992582, 475.996345, 614.889334, 678.435238, 639.397747, 470.54055, 144.627335, -2932.79777, -3509.49445, -3113.86505, -3274.31192, -7801.70177, -12594.3728, -21919.6626, -40044.909, -67533.3141, -119764.622, -248260.16, -353625.192, -358898.889, -343194.683, -321641.463, -289396.778, -243276.737, -193434.545, -145476.652, -105009.505},
{-31.8683333, -30.9374697, -30.0361184, -29.1612377, -28.3097862, -27.4787223, -26.6650044, -25.865591, -25.0774406, -24.2975115, -23.5227623, -22.7501514, -21.9766372, -21.1991781, -20.3947106, -19.5728592, -18.7447638, -17.9114784, -17.0740568, -16.2335527, -15.3910199, -14.5491341, -13.7071032, -12.8657571, -12.0259258, -11.188439, -10.3541265, -9.51942752, -8.68144093, -7.85149525, -7.03220309, -6.22617708, -5.43602981, -4.66437391, -4.52962731, -4.64484193, -4.57645592, -4.201124, -3.3955009, -2.03624133, 0, 2.38547115, 5.75691422, 10.2999741, 16.2002957, 23.643524, 32.8153038, 46.2870767, 66.2717898, 87.2503757, 108.439769, 129.056905, 148.318717, 165.442141, 326.167472, 537.020431, 693.43557, 764.659679, 719.939552, 528.521982, 159.653759, -2608.91273, -3214.82827, -3099.76368, -3705.3898, -9026.72777, -14207.2007, -23752.4752, -42168.218, -71709.0263, -123633.775, -239923.275, -335811.755, -345417.838, -331331.384, -307187.809, -273154.191, -232810.415, -189553.475, -146842.712, -108137.47},
{-20.0302572, -19.6381658, -19.2053708, -18.7358815, -18.2337072, -17.7028571, -17.1473405, -16.5711667, -15.978345, -15.3728846, -14.7587948, -14.1400848, -13.520764, -12.9048416, -12.3275698, -11.7731935, -11.2235926, -10.6763859, -10.1291922, -9.57963022, -9.02531876, -8.41389128, -7.79985521, -7.18773265, -6.58204571, -5.98731649, -5.40806709, -4.86092485, -4.36069817, -3.88296412, -3.42732968, -2.99340183, -2.58078756, -2.18909385, -2.53214569, -3.15773313, -3.5595831, -3.59121517, -3.10614896, -1.95790403, 0, 2.50229889, 6.16408247, 11.1886959, 17.7794844, 26.139793, 36.472967, 51.7037552, 74.3479831, 98.1018087, 122.0636, 145.331724, 167.00455, 186.180444, 359.580345, 586.590006, 754.66385, 830.659851, 781.435983, 573.850219, 174.760534, -2303.86234, -2917.28492, -3028.24439, -3999.47794, -9783.43757, -15354.9138, -25184.2768, -43741.8967, -73989.5527, -124923.387, -233695.183, -322248.095, -329244.738, -315251.626, -293243.744, -262504.614, -225610.988, -185658.156, -145564.631, -108248.927},
{-12.6902541, -12.0962501, -11.5645809, -11.0876955, -10.6580429, -10.2680723, -9.91023253, -9.5769728, -9.26074213, -8.95398956, -8.64916416, -8.33871498, -8.01509107, -7.67074149, -7.21941031, -6.70333566, -6.17071324, -5.63009056, -5.09001513, -4.55903445, -4.04569602, -3.6434085, -3.26413833, -2.90471312, -2.56196047, -2.23270796, -1.91378321, -1.57505508, -1.19044332, -0.821236911, -0.475209569, -0.160134995, 0.116213103, 0.34606102, -0.359519246, -1.45087998, -2.31167396, -2.76944208, -2.65172521, -1.78606422, 0, 2.55805603, 6.42574797, 11.8198493, 18.9571337, 28.0543746, 39.3283455, 56.4541114, 82.5870872, 109.674877, 136.530078, 161.965285, 184.793094, 203.826103, 387.349059, 626.965857, 801.452096, 874.956242, 811.626764, 575.612131, 131.060809, -2646.10171, -3379.52842, -3502.81122, -4449.542, -9401.98794, -15480.3531, -26216.6228, -45142.7826, -73801.6903, -121692.714, -225579.161, -309758.77, -314876.602, -300584.785, -279611.063, -250946.073, -217184.806, -180241.945, -142264.098, -105397.871},
{-8.67512051, -8.41984217, -8.11450304, -7.76504885, -7.37742535, -6.95757828, -6.51145338, -6.04499641, -5.56415309, -5.07486918, -4.58309042, -4.09476254, -3.6158313, -3.15224244, -2.75719541, -2.4067431, -2.07336145, -1.75333087, -1.44293172, -1.1384444, -0.836149293, -0.476906124, -0.120069905, 0.230425007, 0.570644256, 0.896653486, 1.20451834, 1.50199322, 1.79507616, 2.05188364, 2.26373529, 2.42195074, 2.51784962, 2.54275157, 1.50268045, 0.0122575862, -1.23931314, -2.059178, -2.25448326, -1.63237517, 0, 2.61669024, 6.65957361, 12.3517224, 19.916209, 29.5761055, 41.5544845, 60.1388366, 89.0059252, 118.660334, 147.67485, 174.622262, 198.075357, 216.606923, 400.555509, 639.834256, 811.354392, 878.555439, 804.876918, 553.758348, 88.6392516, -2073.4964, -2770.75087, -3221.23308, -4643.05199, -10416.5871, -16426.3604, -26485.1848, -44405.8732, -73070.8069, -119084.093, -211386.946, -287977.184, -300412.52, -290833.052, -269475.859, -239172.615, -206071.305, -170460.078, -134624.763, -100851.19},
{-6.56467003, -6.16903661, -5.75562568, -5.32659038, -4.88408385, -4.43025922, -3.96726963, -3.49726822, -3.02240813, -2.54484249, -2.06672443, -1.5902071, -1.11744363, -0.650587158, -0.199108732, 0.23946785, 0.668172719, 1.08634956, 1.49334205, 1.88849387, 2.2711487, 2.67469139, 3.05972314, 3.42088631, 3.75282326, 4.05017636, 4.30758797, 4.52566205, 4.70410679, 4.82330965, 4.87549239, 4.85287677, 4.74768455, 4.55213748, 3.1606537, 1.25992955, -0.380239742, -1.54308576, -2.01184009, -1.5697343, 0, 3.08044718, 7.69909034, 14.0497285, 22.3261608, 32.7221862, 45.4316038, 65.1716839, 95.9799915, 127.234189, 157.2914, 184.508749, 207.243361, 223.85236, 404.978382, 639.990109, 805.214965, 863.770499, 778.774256, 513.343783, 30.596626, -2021.49672, -2750.28202, -3299.07912, -4811.2079, -10199.2299, -16242.3468, -26206.9516, -43359.4371, -69736.4892, -112293.622, -198712.494, -271662.186, -286963.163, -278845.454, -256704.337, -225749.505, -195124.667, -162423.052, -129175.57, -96913.1306},
{-5.56124504, -5.05499527, -4.54410271, -4.02898833, -3.51007309, -2.98777796, -2.46252392, -1.93473193, -1.40482298, -0.873218021, -0.340338033, 0.193396014, 0.727563151, 1.26174241, 1.81049577, 2.36392399, 2.91099369, 3.44821924, 3.97211504, 4.47919548, 4.96597494, 5.4534668, 5.91030297, 6.32961438, 6.70453194, 7.02818655, 7.29370913, 7.50304539, 7.65681649, 7.72707617, 7.70337626, 7.57526857, 7.33230493, 6.96403715, 5.21864841, 2.87098488, 0.789858834, -0.781266194, -1.59892669, -1.41965913, 0, 3.51847295, 8.6793345, 15.6411182, 24.5623574, 35.6015858, 48.9173368, 69.6540832, 102.207101, 134.812973, 165.605781, 192.719607, 214.288534, 228.446643, 404.925127, 633.30511, 790.178568, 838.580815, 741.547159, 462.112912, -36.6866138, -2284.05963, -3089.80009, -3601.01998, -4964.8313, -9839.3587, -15644.5593, -25340.7457, -41888.2307, -67517.4983, -107378.347, -183709.23, -249142.889, -267254.155, -263181.338, -244282.62, -216004.911, -186655.952, -155011.43, -123096.606, -92936.7449},
{-5.27365126, -4.70375745, -4.12317741, -3.53283143, -2.93363981, -2.32652285, -1.71240084, -1.09219407, -0.466822831, 0.162792571, 0.795731847, 1.4310747, 2.06790084, 2.70528998, 3.35530245, 4.00880637, 4.65568726, 5.29236974, 5.91527844, 6.52083796, 7.10547294, 7.70873069, 8.27795759, 8.80362274, 9.27619522, 9.6861441, 10.0239385, 10.2953378, 10.5038039, 10.603245, 10.5779216, 10.4120945, 10.0900245, 9.59597211, 7.53027008, 4.75265703, 2.22762887, 0.222517333, -0.995345823, -1.15862885, 0, 3.83543963, 9.37725417, 16.7425749, 26.048533, 37.4122599, 50.9508867, 72.1330057, 105.623088, 138.74231, 169.43494, 195.645242, 215.317483, 226.395929, 393.567701, 609.294928, 753.418037, 789.775422, 682.205474, 394.546589, -109.362842, -2448.65246, -3295.332, -3774.37078, -5010.73815, -9531.90674, -15003.3467, -24233.018, -40028.8803, -65171.9598, -102551.017, -168472.081, -225525.938, -244736.343, -244336.067, -229668.948, -205209.544, -176713.735, -145681.396, -115063.321, -87810.301},
{-5.56216828, -4.89659609, -4.22510258, -3.54780906, -2.86483687, -2.17630733, -1.48234175, -0.783061461, -0.0785877887, 0.630957947, 1.34545442, 2.06478032, 2.78881431, 3.51743507, 4.26997829, 5.03401407, 5.7956397, 6.55075409, 7.29525615, 8.02504478, 8.7360189, 9.48666324, 10.2016473, 10.8682266, 11.4736563, 12.005192, 12.4500888, 12.8159221, 13.1072135, 13.2626317, 13.2611816, 13.081868, 12.7036959, 12.1056701, 9.77307445, 6.62984725, 3.71299499, 1.30705091, -0.303451771, -0.833979815, 0, 4.0926515, 9.89657631, 17.4740066, 26.8871747, 38.1983126, 51.4696528, 72.2842673, 105.395656, 137.665093, 166.91317, 190.960483, 207.627624, 214.735188, 366.046031, 560.731102, 686.157209, 708.248243, 592.928098, 306.120667, -186.250159, -2169.11327, -2959.58379, -3540.04735, -4892.88955, -9245.41646, -14509.2179, -23160.5721, -37675.7573, -60261.3122, -94202.7332, -155991.43, -208832.763, -224134.522, -222840.103, -210216.716, -188897.934, -162270.56, -133342.115, -105249.421, -81129.3001},
{-6.81819222, -6.11133366, -5.37916772, -4.62300956, -3.84417433, -3.04397719, -2.22373331, -1.38475784, -0.528365933, 0.34412724, 1.23140652, 2.13215676, 3.04506279, 3.96880946, 4.92891572, 5.90709105, 6.88301372, 7.84987985, 8.8008856, 9.72922711, 10.6281005, 11.5449591, 12.4112486, 13.2126719, 13.9349318, 14.5637312, 15.0847729, 15.5091816, 15.8442621, 16.0149308, 15.9965684, 15.7645555, 15.2942731, 14.5611019, 11.9296181, 8.39420326, 5.07116982, 2.26537526, 0.281677096, -0.575067161, 0, 4.52295874, 10.7820589, 18.7767731, 28.5065737, 39.9709333, 53.1693243, 73.8052084, 106.724954, 138.289145, 166.181252, 188.084742, 201.683088, 204.659757, 342.470831, 519.318287, 628.218366, 636.62887, 512.007604, 221.812371, -266.499026, -1911.56176, -2686.45663, -3397.68549, -4851.75019, -8873.56267, -13757.553, -21532.3054, -34226.4041, -52759.5347, -82486.9772, -144214.754, -195045.015, -201335.676, -197880.8, -190780.071, -176865.566, -152524.57, -125191.137, -98533.3628, -76219.3402},
{-8.50563467, -7.75712742, -6.95769135, -6.11052505, -5.21882713, -4.28579619, -3.31463085, -2.30852971, -1.27069137, -0.204314439, 0.887402479, 2.00126078, 3.13406185, 4.2826071, 5.46637267, 6.66781582, 7.86767744, 9.05812097, 10.2313099, 11.3794076, 12.4945777, 13.6589526, 14.7623014, 15.7843622, 16.7048731, 17.5035721, 18.1601975, 18.6635354, 19.0010128, 19.1307322, 19.0287581, 18.6711545, 18.0339858, 17.0933162, 14.2022937, 10.3636453, 6.70294558, 3.52821283, 1.14746542, -0.131278303, 0, 4.29175776, 10.2722567, 17.9121975, 27.182281, 38.0532079, 50.495679, 69.9784769, 101.144231, 130.817102, 156.735391, 176.637403, 188.261439, 189.345803, 312.838843, 470.944493, 565.631885, 566.983223, 445.080713, 170.006561, -288.157028, -1883.52972, -2621.96049, -3258.96568, -4550.06164, -7974.90776, -12502.8055, -19758.2359, -31365.6801, -47873.2169, -74134.5337, -128055.042, -173488.059, -182501.555, -180156.964, -171629.173, -156782.909, -136018.94, -112904.031, -89925.6229, -69571.1584},
{-11.1687064, -10.3817552, -9.50984147, -8.55865164, -7.53387206, -6.44118915, -5.28628928, -4.07485884, -2.81258425, -1.50515187, -0.158248107, 1.22244065, 2.631228, 4.06242757, 5.53237265, 7.02144711, 8.50836803, 9.98294507, 11.4349879, 12.8543061, 14.2307095, 15.6582985, 17.0081887, 18.2557864, 19.3764979, 20.3457297, 21.138888, 21.7312377, 22.0980647, 22.2151141, 22.0578497, 21.6017354, 20.8222351, 19.6948126, 16.5310094, 12.3588927, 8.33248143, 4.76758106, 1.97999707, 0.28553489, 0, 4.27317723, 10.1955007, 17.6913839, 26.6852403, 37.1014835, 48.8645268, 66.8206199, 95.0764473, 121.787751, 144.88052, 162.280745, 171.914416, 171.707522, 280.53568, 419.811704, 501.793056, 499.666481, 386.61872, 135.836519, -279.493379, -1700.41279, -2372.90976, -2959.46396, -4122.55508, -7123.68991, -11147.7518, -17576.0471, -27789.8824, -42285.6309, -65099.3987, -111099.244, -150308.439, -159708.128, -158151.438, -149949.174, -136152.92, -118673.915, -99244.0018, -79656.1808, -61703.4528},
{-15.0034096, -14.2724615, -13.3902868, -12.3680307, -11.2168388, -9.94785629, -8.57222864, -7.10110119, -5.54561934, -3.91692846, -2.22617391, -0.484501089, 1.29694464, 3.10701791, 4.94583587, 6.79512844, 8.63577297, 10.4543204, 12.2373218, 13.971328, 15.6428901, 17.328053, 18.911514, 20.3674642, 21.6700948, 22.793597, 23.712162, 24.4049746, 24.8504692, 25.0108968, 24.8584212, 24.3652061, 23.5034152, 22.2452121, 18.8383676, 14.3459026, 9.96136703, 6.00963498, 2.81558045, 0.704077447, 0, 4.20045209, 10.0199711, 17.3453243, 26.0632791, 36.0606028, 47.2240627, 63.77577, 89.2863988, 113.276414, 133.872294, 149.200516, 157.387561, 156.559905, 250.571061, 370.6642, 440.079422, 435.317615, 332.879666, 109.266463, -259.021108, -1425.77493, -2020.73533, -2582.30767, -3648.89731, -6332.28436, -9756.55346, -15148.1542, -23733.536, -36053.0643, -55391.4414, -94200.9791, -127087.449, -134450.543, -133052.038, -126571.412, -115494.953, -100976.185, -84692.8819, -68144.0964, -52828.88},
{-20.0462326, -19.591834, -18.8574763, -17.8657248, -16.639145, -15.200302, -13.5717613, -11.7760882, -9.8358481, -7.77360624, -5.611928, -3.37337873, -1.08052375, 1.24407159, 3.54735161, 5.82603928, 8.07916554, 10.2904016, 12.4434186, 14.5218878, 16.5094805, 18.4470624, 20.2532111, 21.9036989, 23.3742981, 24.6407807, 25.6789191, 26.486672, 27.0586643, 27.3176179, 27.2302965, 26.7634639, 25.8838841, 24.5583208, 20.9964813, 16.2766499, 11.6101121, 7.32302301, 3.74153784, 1.19181178, 0, 3.60418307, 8.78881258, 15.4502655, 23.4849188, 32.7891494, 43.2593342, 58.8619025, 82.9517093, 105.693172, 125.330096, 140.106285, 148.265546, 148.051682, 227.311307, 327.605162, 383.711452, 375.546468, 283.026502, 86.0678438, -235.413214, -1228.99871, -1754.62657, -2253.85336, -3168.23562, -5438.45292, -8274.89496, -12718.0611, -19808.4507, -30115.6316, -46092.8977, -77411.6011, -104063.578, -110627.87, -109379.491, -103312.972, -93537.7931, -82031.5276, -69184.4152, -55927.8351, -43193.1661},
{-26.6172289, -26.7224939, -26.3390842, -25.5085305, -24.272363, -22.6721124, -20.7493091, -18.5454836, -16.1021664, -13.460888, -10.6631789, -7.7505696, -4.76459053, -1.74677221, 1.14811707, 3.95018146, 6.69809673, 9.37349424, 11.9580054, 14.4332615, 16.7808939, 19.0100211, 21.0709912, 22.9416395, 24.5998012, 26.0233114, 27.1900054, 28.1121784, 28.7969475, 29.1397503, 29.1044301, 28.6548302, 27.754794, 26.3681648, 22.6649347, 17.7437439, 12.8390194, 8.2815219, 4.40201229, 1.53125138, 0, 3.30355828, 8.17110317, 14.4963509, 22.1730178, 31.0948201, 41.155474, 55.6343922, 77.3022458, 97.9571645, 116.118163, 130.304256, 139.034459, 140.827786, 205.93302, 287.492664, 333.21959, 326.961083, 252.564425, 93.876899, -165.25421, -957.708558, -1384.02209, -1791.95534, -2529.26882, -4346.7656, -6598.03722, -10114.4914, -15727.5357, -23898.7166, -36569.0259, -61556.1781, -82536.9717, -86855.432, -85156.2204, -80011.5735, -72196.0449, -63274.5668, -53448.24, -43383.1097, -33745.2212},
{-33.3425809, -34.5797467, -35.0142483, -34.7149696, -33.7507949, -32.1906082, -30.1032937, -27.5577354, -24.6228174, -21.3674239, -17.8604389, -14.1707467, -10.3672312, -6.51877659, -2.92507814, 0.49099202, 3.82403315, 7.05237166, 10.1543339, 13.1082464, 15.8924355, 18.472886, 20.8419704, 22.9797195, 24.8661641, 26.4813351, 27.8052633, 28.8560246, 29.6459781, 30.0641847, 30.0752278, 29.6436907, 28.7341567, 27.3112092, 23.5246129, 18.4870122, 13.4484249, 8.74464055, 4.71144862, 1.68463861, 0, 3.27487408, 8.11029377, 14.3888438, 21.9931088, 30.8056735, 40.7091226, 53.8891774, 72.1854853, 89.9735684, 106.200864, 119.81481, 129.762842, 134.992399, 187.316708, 252.240115, 291.265256, 292.52631, 244.157451, 134.292858, -48.9332942, -611.195059, -912.713132, -1208.30045, -1752.76996, -3101.57027, -4767.94823, -7367.47672, -11515.7286, -17477.6065, -26920.694, -46499.002, -62319.7035, -63452.8615, -61014.9319, -57316.892, -52034.8045, -45306.6837, -38057.6528, -30944.5013, -24624.0188},
{-37.4574626, -40.5138186, -42.3515766, -43.0745254, -42.7864534, -41.5911492, -39.5924013, -36.8939983, -33.5997288, -29.8133813, -25.6387443, -21.1796065, -16.5397563, -11.8229823, -7.50072163, -3.444187, 0.503238223, 4.31296495, 7.9564041, 11.4049666, 14.6300634, 17.4882773, 20.0817059, 22.3976185, 24.4232848, 26.1459741, 27.5529558, 28.6894543, 29.5919855, 30.1092422, 30.2049623, 29.8428838, 28.9867447, 27.6002828, 23.8125351, 18.747878, 13.6695669, 8.91661274, 4.82802626, 1.74281839, 0, 3.08152086, 7.74939156, 13.9085616, 21.4639804, 30.3205975, 40.3833624, 52.8391957, 68.6823865, 84.7525449, 100.434099, 115.111476, 128.169105, 138.991414, 182.3011, 235.127472, 271.82489, 284.54963, 265.45797, 206.706186, 100.450556, -248.727491, -430.444526, -610.947882, -956.48489, -1884.44788, -2941.59479, -4595.54455, -7313.91607, -11342.7906, -17814.4, -31506.4558, -42234.9855, -41893.8713, -39432.2948, -36652.8, -33051.8461, -28475.7232, -23677.4838, -19101.0555, -15190.3661},
{-37.6999664, -44.6073811, -49.6529908, -52.9865426, -54.757784, -55.1164621, -54.2123245, -52.1951182, -49.2145908, -45.4204894, -40.9625614, -35.9905541, -30.6542148, -25.1032908, -19.9894335, -15.1448838, -10.3638909, -5.69354162, -1.18092301, 3.12687795, 7.18277429, 10.6916503, 13.8887022, 16.7610974, 19.2960035, 21.480588, 23.3020182, 24.7764267, 25.9155933, 26.6374274, 26.9173356, 26.7307249, 26.053002, 24.8595739, 21.493007, 16.9616489, 12.3974458, 8.10978864, 4.40806809, 1.60167494, 0, 2.41668115, 6.31100838, 11.6465188, 18.3867495, 26.4952375, 35.9355201, 47.0895956, 60.2765839, 74.4594328, 89.4317708, 104.987226, 120.919427, 137.022002, 171.755359, 213.104453, 247.641248, 271.341234, 280.179905, 270.132754, 237.175274, 68.9326989, -3.94063173, -66.6118504, -204.248089, -682.503503, -1141.34506, -1882.52258, -3207.78589, -5345.72218, -8817.56921, -16127.4291, -21722.5291, -21138.8662, -19403.4913, -17570.8586, -15415.6481, -12978.8405, -10515.9876, -8209.49146, -6241.75435},
{-32.9009557, -47.1404618, -58.5117179, -67.2253254, -73.4918858, -77.5220004, -79.5262706, -79.7152978, -78.2996834, -75.4900288, -71.4969354, -66.5310046, -60.8028378, -54.5230364, -48.5130929, -42.5971569, -36.5531358, -30.4666784, -24.4234333, -18.5090492, -12.8091746, -7.78310138, -3.0912324, 1.23238631, 5.15370873, 8.63868887, 11.6532807, 14.0420896, 15.667955, 16.7869894, 17.4144182, 17.5654669, 17.255361, 16.4993258, 14.343992, 11.4173216, 8.42059515, 5.56715624, 3.07034838, 1.14351512, 0, 0.972835271, 3.00103908, 6.14331829, 10.4583798, 16.0049303, 22.8416769, 30.8168166, 39.8101783, 50.3838214, 62.6819266, 76.8486743, 93.0282452, 111.36482, 134.169525, 160.215722, 188.112747, 217.561549, 248.263077, 279.918281, 312.228111, 279.105702, 299.57109, 358.81613, 442.032678, 530.434992, 619.026309, 688.225405, 718.451057, 706.837009, 583.657133, 81.7638928, -247.651353, 41.7735923, 332.221774, 521.477762, 706.957138, 1007.80801, 1256.61738, 1383.66396, 1319.22649}
};

  /* Stuff for interpolating the NQC data */
  gsl_spline    *spline = NULL;
  gsl_interp_accel *acc = NULL;
  /* Interpolate the spin NQC data in 2-D parameter space -- spin and mass ratio */
  /* First, interpolate in spin dimension for all mass ratios */
  spline = gsl_spline_alloc( gsl_interp_cspline, adim );
  acc    = gsl_interp_accel_alloc();
  for (i = 0; i < qdim; i++)
  {
    gsl_spline_init( spline, alist, a3stab[i], adim );
    gsl_interp_accel_reset( acc );
    a3slist[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, a4tab[i], adim );
    gsl_interp_accel_reset( acc );
    a4list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, a5tab[i], adim );
    gsl_interp_accel_reset( acc );
    a5list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, b3tab[i], adim );
    gsl_interp_accel_reset( acc );
    b3list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
    gsl_spline_init( spline, alist, b4tab[i], adim );
    gsl_interp_accel_reset( acc );
    b4list[i] = gsl_spline_eval( spline, a/(1.0-2.0*eta), acc );
//  printf("%.15f\n",a3slist[i]);
  }
//printf("%.15f\n",a);
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);
  /* Second, interpolate in mass ratio dimension */
  spline = gsl_spline_alloc( gsl_interp_cspline, qdim );
  acc    = gsl_interp_accel_alloc();
  gsl_spline_init( spline, etalist, a3slist, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a3S = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, a4list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a4 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, a5list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->a5 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, b3list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->b3 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, etalist, b4list, qdim );
  gsl_interp_accel_reset( acc );
  coeffs->b4 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);

  /* Interpolate nonspin NQC data in the mass ratio dimension */
  spline = gsl_spline_alloc( gsl_interp_cspline, nsqdim );
  acc    = gsl_interp_accel_alloc();
  gsl_spline_init( spline, nsetalist, a1list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a1 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, a2list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a2 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, a3list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->a3 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, b1list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->b1 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_init( spline, nsetalist, b2list, nsqdim );
  gsl_interp_accel_reset( acc );
  coeffs->b2 = gsl_spline_eval( spline, eta, acc );
  gsl_spline_free(spline);
  gsl_interp_accel_free(acc);
  /* Andrea and I have different sign conventions, so I need to put a minus sign in front */
  coeffs->b1 = - coeffs->b1;
  coeffs->b2 = - coeffs->b2;
#if 0
  coeffs->a1 = -8.02798637014;
  coeffs->a2 = 48.7446843797;
  coeffs->a3 = -45.7900277224;
  coeffs->a3S= 0.;
  coeffs->a4 = 0.;
  coeffs->a5 = 0.;
  coeffs->b1 = 0.834742923041;
  coeffs->b2 = -2.33512320852; // q=1
#endif
#if 0
  coeffs->a1 = -7.79667;
  coeffs->a2 = 47.182;
  coeffs->a3 = 2238.85334023;
  coeffs->a3S= 0.;
  coeffs->a4 = -7143.16738899;
  coeffs->a5 = 5596.0086893;
  coeffs->b1 = 0.85069;
  coeffs->b2 = -2.47071; // q=1, chi1=chi2=0.98
#endif
#if 0
  coeffs->a1 = -6.82562365707;
  coeffs->a2 = 41.5711482044;
  coeffs->a3 = -39.4329799959;
  coeffs->a3S= 0.;
  coeffs->a4 = 0.;
  coeffs->a5 = 0.;
  coeffs->b1 = 0.461925688819;
  coeffs->b2 = -1.38733263299; // q=8
#endif
#if 0
  coeffs->a1 = -7.5758;
  coeffs->a2 = 46.9323;
  coeffs->a3 = -118.368375152;
  //coeffs->a3 = -45.0036; // NS part
  coeffs->a3S= 0.;
  coeffs->a4 = 125.555824111;
  coeffs->a5 = -22.0751068073;
  //coeffs->a4 = 0.;
  //coeffs->a5 = 0.;
  coeffs->b1 = 0.51305;
  coeffs->b2 = -1.55133; // q=8, chi1=0.5
#endif
#if 0
  coeffs->a1 = -5.79216;
  coeffs->a2 = 30.3902;
  coeffs->a3 = 3329.40898295;
  coeffs->a3S= 0.0;
  coeffs->a4 = -11082.7670039;
  coeffs->a5 = 9152.28860893;
  coeffs->b1 = 0.974779;
  coeffs->b2 = -3.34626;
#endif

  /* Obsolete polynomial fitting of nonspin NQC coefficients a1, a2, a3, b1 and b2 */
  /*
  coeffs->a1 = -12.67955358602124 + 75.41927959573084 * eta - 106.15933052937714 * eta2;
  coeffs->a2 = 101.45522216901628 - 757.3158549733314 * eta + 1473.314771676588 * eta2;
  coeffs->a3 = -107.6647834845902 + 857.6219519536213 * eta - 1776.2776804623143 * eta2;
  // Andrea and I have different sign conventions, so I need to put a minus sign in front 
  coeffs->b1 = - (-1.464129495621165 + 12.81732978488213 * eta - 60.09957767247623 * eta2);
  coeffs->b2 = - ( 7.477426352542122 - 85.26122117590637 * eta + 353.3251639728075 * eta2);
  */

  return XLAL_SUCCESS;

}









/**
 * ==================================================
 *
 *   Below are three interpolation functions for
 *   max, med and min possible chiA values with given
 *   eta and chi values.
 *   They provide 2-D interpolation results for the
 *   3-D interpolation function of SEOBNRv2
 *
 * ==================================================
 */

UNUSED static int XLALSimIMRGetEOBCalibratedSpinNQCv2chiAmax( EOBNonQCCoeffs * restrict coeffs,                                                             INT4 UNUSED l,                                                             INT4 UNUSED m,                                                             REAL8 eta,                                                             REAL8 a )
{
  const unsigned int   qdim = 98;
  const unsigned int   adim = 200;
  UINT4 i;
  /* REAL8 eta2 = eta*eta;*/
  REAL8 a3slist[qdim], a4list[qdim], a5list[qdim];
  REAL8 b3list[qdim], b4list[qdim];
  REAL8 chiA;

  chiA = a/(1.0-2.0*eta);
  if (chiA < -1.){
      chiA = -1.;
  }
  if (chiA > 0.99){
      chiA = 0.99;
  }

  memset( coeffs, 0, sizeof( *coeffs ) );
  const double alist[200] = {-1., -0.99, -0.98, -0.97, -0.96, -0.95, -0.94, -0.93, -0.92, -0.91, -0.9, -0.89, -0.88, -0.87, -0.86, -0.85, -0.84, -0.83, -0.82, -0.81, -0.8, -0.79, -0.78, -0.77, -0.76, -0.75, -0.74, -0.73, -0.72, -0.71, -0.7, -0.69, -0.68, -0.67, -0.66, -0.65, -0.64, -0.63, -0.62, -0.61, -0.6, -0.59, -0.58, -0.57, -0.56, -0.55, -0.54, -0.53, -0.52, -0.51, -0.5, -0.49, -0.48, -0.47, -0.46, -0.45, -0.44, -0.43, -0.42, -0.41, -0.4, -0.39, -0.38, -0.37, -0.36, -0.35, -0.34, -0.33, -0.32, -0.31, -0.3, -0.29, -0.28, -0.27, -0.26, -0.25, -0.24, -0.23, -0.22, -0.21, -0.2, -0.19, -0.18, -0.17, -0.16, -0.15, -0.14, -0.13, -0.12, -0.11, -0.1, -0.09, -0.08, -0.07, -0.06, -0.05, -0.04, -0.03, -0.02, -0.01, 0, 0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.11, 0.12, 0.13, 0.14, 0.15, 0.16, 0.17, 0.18, 0.19, 0.2, 0.21, 0.22, 0.23, 0.24, 0.25, 0.26, 0.27, 0.28, 0.29, 0.3, 0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4, 0.41, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5, 0.51, 0.52, 0.53, 0.54, 0.55, 0.56, 0.57, 0.58, 0.59, 0.6, 0.61, 0.62, 0.63, 0.64, 0.65, 0.66, 0.67, 0.68, 0.69, 0.7, 0.71, 0.72, 0.73, 0.74, 0.75, 0.76, 0.77, 0.78, 0.79, 0.8, 0.81, 0.82, 0.83, 0.84, 0.85, 0.86, 0.87, 0.88, 0.89, 0.9, 0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.97, 0.98, 0.99}; 
  const double etalist[98]    = {0.0075, 0.01, 0.0125, 0.015, 0.0175, 0.02, 0.0225, 0.025, 0.0275, 0.03, 0.0325, 0.035, 0.0375, 0.04, 0.0425, 0.045, 0.0475, 0.05, 0.0525, 0.055, 0.0575, 0.06, 0.0625, 0.065, 0.0675, 0.07, 0.0725, 0.075, 0.0775, 0.08, 0.0825, 0.085, 0.0875, 0.09, 0.0925, 0.095, 0.0975, 0.1, 0.1025, 0.105, 0.1075, 0.11, 0.1125, 0.115, 0.1175, 0.12, 0.1225, 0.125, 0.1275, 0.13, 0.1325, 0.135, 0.1375, 0.14, 0.1425, 0.145, 0.1475, 0.15, 0.1525, 0.155, 0.1575, 0.16, 0.1625, 0.165, 0.1675, 0.17, 0.1725, 0.175, 0.1775, 0.18, 0.1825, 0.185, 0.1875, 0.19, 0.1925, 0.195, 0.1975, 0.2, 0.2025, 0.205, 0.2075, 0.21, 0.2125, 0.215, 0.2175, 0.22, 0.2225, 0.225, 0.2275, 0.23, 0.2325, 0.235, 0.2375, 0.24, 0.2425, 0.245, 0.2475, 0.25};

  /* YPtag: chiAmax tables below */
  const double a3stab[98][200] = {{486.38171, 481.380476, 476.404729, 471.453495, 466.525797, 461.620659, 456.737106, 451.874162, 447.03085, 442.206196, 437.399224, 432.608957, 427.834419, 423.074636, 418.32863, 413.595427, 408.87405, 404.163524, 399.462873, 394.771121, 390.087292, 385.41041, 380.739499, 376.073585, 371.41169, 366.752839, 362.096057, 357.440367, 352.784793, 348.12836, 343.470092, 338.809014, 334.144148, 329.47452, 324.760718, 320.021792, 315.277256, 310.527379, 305.772434, 301.012693, 296.248427, 291.479906, 286.707404, 281.931191, 277.151539, 272.368718, 267.583002, 262.794661, 258.003967, 253.211191, 248.416604, 243.63398, 238.849796, 234.064031, 229.276664, 224.487673, 219.697037, 214.904735, 210.110747, 205.31505, 200.517623, 195.718446, 190.917497, 186.114756, 181.310199, 176.503808, 171.69556, 166.908623, 162.165606, 157.418664, 152.666274, 147.906911, 143.139051, 138.361168, 133.571739, 128.76924, 123.952146, 119.118932, 114.268074, 109.398049, 104.507331, 99.5943955, 94.6577192, 89.6957773, 84.6479933, 79.5436111, 74.412584, 69.255304, 64.0721631, 58.8635533, 53.6298665, 48.3714947, 43.0888299, 37.7822641, 32.4521892, 27.0989972, 21.7230801, 16.3248299, 10.9046385, 5.46289784, 0, -5.02236294, -10.0652441, -15.1402674, -20.2590568, -25.4332361, -30.6744295, -35.9942607, -41.4043538, -46.9163327, -52.5418212, -58.2924434, -64.1798232, -70.2155844, -76.4113512, -82.7787473, -89.3293967, -96.2039592, -103.432492, -110.734753, -118.150017, -125.717557, -133.476649, -141.466567, -149.726584, -158.295977, -167.214019, -176.519984, -186.253147, -196.452783, -207.203609, -218.482794, -230.315979, -242.728807, -255.430166, -268.604073, -282.519016, -297.295662, -313.054678, -329.916733, -348.002494, -367.774213, -388.907013, -411.454731, -435.471203, -461.010266, -487.34786, -515.510191, -545.745569, -578.302302, -613.428701, -651.338492, -692.356838, -735.965331, -782.347584, -833.1205, -889.257464, -951.73186, -1023.26493, -1102.68497, -1186.91349, -1271.49453, -1363.04314, -1467.55843, -1587.86861, -1721.42903, -1868.16125, -2025.72243, -2193.44705, -2378.97259, -2585.23989, -2813.96745, -3069.50649, -3355.22098, -3674.47486, -4030.6321, -4546.27556, -5075.74556, -5592.60132, -6070.40207, -6482.70704, -6760.47171, -6948.26154, -7037.8945, -6938.24518, -6673.96329, -6304.25833, -5852.05204, -5340.26615, -4791.82242, -4229.64258, -3723.43883, -3248.96725, -2806.12957, -2394.82752, -2014.96283, -1666.43723, -1349.15247, -1071.01933, -797.885891}, 
{459.284623, 454.253762, 449.282376, 444.368215, 439.509034, 434.702585, 429.946621, 425.238895, 420.57716, 415.959168, 411.382673, 406.845427, 402.345183, 397.879695, 393.446715, 389.043995, 384.669289, 380.32035, 375.99493, 371.690782, 367.40566, 363.137315, 358.883502, 354.641972, 350.410479, 346.186775, 341.968614, 337.753748, 333.53993, 329.324914, 325.106451, 320.882294, 316.650198, 312.407913, 308.0665, 303.668882, 299.259024, 294.837492, 290.404852, 285.961672, 281.508516, 277.045952, 272.574545, 268.094862, 263.607469, 259.112932, 254.611818, 250.104693, 245.592123, 241.074674, 236.552912, 232.055218, 227.553742, 223.048445, 218.539293, 214.026247, 209.509272, 204.988331, 200.463386, 195.934402, 191.401341, 186.864168, 182.322844, 177.777334, 173.227601, 168.673608, 164.115319, 159.573738, 155.069365, 150.558768, 146.040544, 141.513293, 136.975614, 132.426108, 127.863372, 123.286006, 118.69261, 114.081783, 109.452123, 104.802231, 100.130705, 95.4361452, 90.7171501, 85.9723192, 81.1339324, 76.2351437, 71.309904, 66.3589647, 61.3830773, 56.3829932, 51.3594639, 46.3132407, 41.2450752, 36.1557188, 31.045923, 25.9164391, 20.7680186, 15.6014131, 10.4173738, 5.2166523, 0, -4.79856953, -9.6042007, -14.4289622, -19.2849229, -24.1841512, -29.1387161, -34.1606862, -39.2621302, -44.4551168, -49.7517147, -55.1639927, -60.7040194, -66.3838636, -72.2155939, -78.2112791, -84.3829879, -90.8101349, -97.5144097, -104.355551, -111.360059, -118.554433, -125.965173, -133.61878, -141.541752, -149.760592, -158.301797, -167.191869, -176.457307, -186.124611, -196.185579, -206.714138, -217.747197, -229.321668, -241.280348, -253.757203, -266.937965, -280.91778, -295.791793, -311.655148, -328.60299, -346.76356, -366.188834, -386.967483, -409.188176, -432.939583, -458.590821, -485.880001, -514.82568, -545.446415, -577.760764, -611.490337, -647.313576, -684.096495, -721.905562, -763.258166, -809.571282, -862.261886, -925.862718, -997.966861, -1072.76838, -1144.14936, -1218.99033, -1304.23846, -1402.25636, -1507.63834, -1618.54448, -1730.2445, -1840.14912, -1956.2671, -2080.61233, -2218.47154, -2368.08232, -2530.32567, -2706.0826, -2896.23412, -3101.33424, -3322.67273, -3561.21234, -3817.91584, -4093.74598, -4455.02698, -4793.78584, -5082.97327, -5169.40169, -5075.95154, -4878.06091, -4595.98235, -4249.96844, -3860.27176, -3447.14486, -3064.62474, -2701.69608, -2359.20272, -2037.98848, -1738.89719, -1462.77268, -1210.45878, -996.8879, -799.369153}, 
{434.112712, 429.650054, 425.21708, 420.812653, 416.435632, 412.084878, 407.759251, 403.457613, 399.178823, 394.921743, 390.685232, 386.468151, 382.269361, 378.087722, 373.922095, 369.77134, 365.634318, 361.509889, 357.396914, 353.294253, 349.200767, 345.115316, 341.036761, 336.963963, 332.895781, 328.831077, 324.768711, 320.707544, 316.646435, 312.584246, 308.519837, 304.452068, 300.379801, 296.301895, 292.179441, 288.03098, 283.875504, 279.713101, 275.543856, 271.367856, 267.185187, 262.995934, 258.800186, 254.598026, 250.389543, 246.174821, 241.953948, 237.72701, 233.494092, 229.255281, 225.010663, 220.753673, 216.491193, 212.223453, 207.950684, 203.673116, 199.390979, 195.104504, 190.813922, 186.519462, 182.221356, 177.919833, 173.615124, 169.30746, 164.99707, 160.684186, 156.369037, 152.085015, 147.864712, 143.63997, 139.40887, 135.169492, 130.919917, 126.658226, 122.3825, 118.09082, 113.781266, 109.45192, 105.100861, 100.726171, 96.3259314, 91.8982218, 87.4411235, 82.9527173, 78.3265578, 73.6151875, 68.8724052, 64.0996836, 59.2984958, 54.4703146, 49.6166129, 44.7388635, 39.8385393, 34.9171132, 29.9760582, 25.016847, 20.0409526, 15.0498478, 10.0450055, 5.02789861, 0, -4.78250344, -9.52437487, -14.2407085, -18.9465985, -23.6571392, -28.3874246, -33.1525491, -37.9676069, -42.8476921, -47.807899, -52.8633217, -58.0290545, -63.3201916, -68.7518272, -74.3390555, -80.0969707, -86.0394014, -92.1812609, -98.5405063, -105.131961, -111.970447, -119.070788, -126.447807, -134.116326, -142.09117, -150.38716, -159.019121, -168.001874, -177.350243, -186.97904, -197.039769, -207.577258, -218.636331, -230.268822, -242.516054, -255.41748, -269.015826, -283.353813, -298.474168, -314.419613, -330.542814, -347.786571, -366.32862, -386.346696, -408.018533, -433.562858, -460.606168, -488.815951, -517.859694, -547.404885, -575.706973, -605.57131, -637.925839, -673.414864, -712.103587, -754.317214, -800.380952, -851.282321, -906.533689, -964.924903, -1024.71467, -1088.38563, -1159.99052, -1240.4031, -1325.11042, -1412.07213, -1498.64422, -1582.62985, -1664.04544, -1741.65545, -1812.44563, -1877.23281, -1935.39716, -1986.31885, -2029.37806, -1966.7829, -1919.37863, -1910.83842, -1964.83547, -2105.04296, -2505.7788, -2939.64165, -3365.74267, -3591.49968, -3636.06913, -3581.81302, -3444.72753, -3240.80883, -2986.05309, -2696.45649, -2408.38342, -2116.37128, -1827.32267, -1548.1402, -1285.72647, -1046.98409, -838.81567, -648.023968, -481.654895}, 
{408.975106, 404.985559, 401.008521, 397.04356, 393.090241, 389.148133, 385.2168, 381.295811, 377.384731, 373.483128, 369.590568, 365.706617, 361.830843, 357.962812, 354.10209, 350.248245, 346.400843, 342.559451, 338.723635, 334.892962, 331.066998, 327.245311, 323.427467, 319.613033, 315.801575, 311.992661, 308.185855, 304.380727, 300.576841, 296.773765, 292.971065, 289.168309, 285.365062, 281.560892, 277.767947, 273.979239, 270.187627, 266.392269, 262.592324, 258.786949, 254.975304, 251.156546, 247.329833, 243.494325, 239.64918, 235.793555, 231.926609, 228.0475, 224.155388, 220.249429, 216.328783, 212.332441, 208.321031, 204.295018, 200.254864, 196.201031, 192.133981, 188.054179, 183.962085, 179.858163, 175.742876, 171.616685, 167.480055, 163.333446, 159.177323, 155.012147, 150.838381, 146.680221, 142.561291, 138.433108, 134.294597, 130.14468, 125.982282, 121.806328, 117.615741, 113.409446, 109.186367, 104.945427, 100.685552, 96.4056654, 92.1046909, 87.781553, 83.4351759, 79.0644836, 74.6035898, 70.085187, 65.542747, 60.977297, 56.3898645, 51.7814768, 47.1531613, 42.5059454, 37.8408564, 33.1589218, 28.4611688, 23.748625, 19.0223176, 14.283274, 9.53252162, 4.77108783, 0, -4.44877624, -8.87861571, -13.3025604, -17.7336522, -22.1849332, -26.6694454, -31.2002306, -35.790331, -40.4527883, -45.2006448, -50.0469422, -55.0047226, -60.087028, -65.3069003, -70.6773815, -76.2115136, -81.9965942, -88.056273, -94.2356332, -100.563629, -107.069213, -113.78134, -120.728965, -127.94104, -135.446519, -143.274357, -151.453507, -160.012924, -168.981561, -178.44686, -188.357841, -198.725912, -209.562479, -220.6587, -232.136107, -244.174965, -256.852758, -270.246967, -284.435074, -299.494563, -315.568962, -332.649605, -350.801054, -370.087868, -390.574608, -412.663888, -435.997701, -460.556093, -486.319112, -513.266803, -541.083953, -570.406743, -601.687646, -635.134169, -670.453687, -707.578124, -746.439403, -786.107851, -827.572808, -872.763539, -924.275341, -979.286962, -1037.08167, -1096.52702, -1155.78615, -1213.34554, -1268.72741, -1320.68676, -1364.18089, -1396.31527, -1419.37522, -1424.48952, -1406.97064, -1362.13106, -1285.28326, -966.597996, -657.814899, -405.53188, -256.34685, -256.857717, -691.291436, -1210.19751, -1758.33283, -2094.71786, -2237.63017, -2282.73748, -2244.44785, -2137.16933, -1975.30997, -1773.27782, -1566.24636, -1344.07188, -1114.56605, -885.540522, -664.806968, -460.177048, -279.462428, -103.363394, 51.3919587}, 
{385.836919, 382.594083, 379.328095, 376.039662, 372.72949, 369.398284, 366.046749, 362.675593, 359.28552, 355.877237, 352.451449, 349.008862, 345.550182, 342.076115, 338.587366, 335.084641, 331.568647, 328.040088, 324.499671, 320.948102, 317.386086, 313.814329, 310.233537, 306.644416, 303.047671, 299.444009, 295.834135, 292.218755, 288.598575, 284.9743, 281.346637, 277.716291, 274.083968, 270.450374, 266.878819, 263.33739, 259.793421, 256.245586, 252.69256, 249.133016, 245.56563, 241.989075, 238.402024, 234.803154, 231.191137, 227.564648, 223.922361, 220.26295, 216.585089, 212.887453, 209.168716, 205.329318, 201.468297, 197.586457, 193.6846, 189.763531, 185.824053, 181.86697, 177.893086, 173.903204, 169.898128, 165.878662, 161.845609, 157.799773, 153.741958, 149.672967, 145.593604, 141.526206, 137.49259, 133.449152, 129.395299, 125.330441, 121.253984, 117.165336, 113.063905, 108.9491, 104.820327, 100.676995, 96.5185109, 92.3442836, 88.1537205, 83.9462293, 79.721218, 75.4780944, 71.1764429, 66.8364207, 62.4786636, 58.1038718, 53.7127454, 49.3059846, 44.8842897, 40.4483608, 35.998898, 31.5366016, 27.0621718, 22.5763086, 18.0797124, 13.5730832, 9.0571213, 4.53252684, 0, -4.01001769, -8.04263381, -12.1076946, -16.2150461, -20.3745348, -24.5960067, -28.8893081, -33.2642853, -37.7307843, -42.2986514, -46.9777329, -51.777875, -56.7089238, -61.7807256, -67.0031266, -72.385973, -78.0639739, -84.0648132, -90.1159089, -96.2538637, -102.51528, -108.936761, -115.554908, -122.406324, -129.527613, -136.955375, -144.726215, -152.876734, -161.443535, -170.594144, -180.186234, -190.217133, -200.684165, -211.263568, -222.113212, -233.476591, -245.447357, -258.119162, -271.585657, -285.940497, -301.939279, -318.812247, -336.523541, -355.037303, -374.317673, -393.17015, -413.007177, -434.082556, -456.650089, -480.963577, -507.969469, -536.38235, -565.40954, -594.778108, -625.276269, -657.215805, -690.908498, -727.414323, -766.126826, -805.623337, -845.475757, -885.016093, -925.047482, -964.851461, -1001.13131, -1031.77376, -1055.73562, -1071.18106, -1072.35067, -1055.70445, -1037.98114, -992.242162, -907.927943, -774.478902, -581.335462, 16.0848818, 610.813051, 1129.90289, 1500.40825, 1649.38298, 1162.48788, 535.765197, -157.419908, -607.522181, -834.664429, -957.377837, -990.864791, -950.327681, -850.968894, -707.990819, -549.66023, -383.275639, -212.34673, -40.383187, 129.105306, 292.609066, 446.618408, 589.86229, 726.643015}, 
{365.912682, 363.744983, 361.490085, 359.150581, 356.729068, 354.22814, 351.650393, 348.998422, 346.274821, 343.482187, 340.623115, 337.700198, 334.716034, 331.673217, 328.574342, 325.422004, 322.218798, 318.967321, 315.670166, 312.32993, 308.949206, 305.530592, 302.076681, 298.590069, 295.07335, 291.529122, 287.959977, 284.368512, 280.757322, 277.129003, 273.486148, 269.831353, 266.167215, 262.496327, 258.931166, 255.417075, 251.898077, 248.373202, 244.841479, 241.301937, 237.753606, 234.195516, 230.626694, 227.046171, 223.452976, 219.846139, 216.224688, 212.587654, 208.934065, 205.26295, 201.57334, 197.758823, 193.926154, 190.076649, 186.211622, 182.33239, 178.440266, 174.536567, 170.622607, 166.699701, 162.769164, 158.832312, 154.89046, 150.944922, 146.997015, 143.048052, 139.099349, 135.200162, 131.398595, 127.59709, 123.793854, 119.987095, 116.175018, 112.355833, 108.527745, 104.688963, 100.837694, 96.972144, 93.0905217, 89.191034, 85.2718882, 81.3312915, 77.3674512, 73.3785747, 69.2579166, 65.0583681, 60.8340818, 56.5866708, 52.3177484, 48.0289278, 43.7218223, 39.398045, 35.0592091, 30.706928, 26.3428146, 21.9684824, 17.5855445, 13.195614, 8.80030435, 4.4012286, 0, -3.83440188, -7.67924906, -11.5453874, -15.4436627, -19.3849207, -23.3800074, -27.4397685, -31.5750499, -35.7966974, -40.1155568, -44.542474, -49.0882947, -53.7638648, -58.5800302, -63.5476366, -68.6775299, -73.9989029, -79.5252225, -85.2258353, -91.1155185, -97.2090496, -103.521206, -110.066765, -116.860503, -123.917199, -131.251629, -138.878571, -146.812803, -155.0691, -163.579717, -172.472214, -181.786126, -191.560988, -201.93223, -212.891439, -224.404631, -236.482572, -249.136028, -262.375766, -276.212551, -290.396536, -305.278419, -320.919957, -337.382905, -354.729019, -373.649022, -393.418463, -413.941855, -435.123714, -456.868554, -478.611292, -501.299992, -524.861263, -549.480354, -575.870577, -604.508156, -635.869314, -673.771219, -714.589845, -751.056497, -775.293263, -794.223421, -815.566468, -841.035697, -862.167215, -875.168627, -877.214619, -864.763519, -830.727682, -770.025264, -710.743517, -609.528568, -449.816964, -215.045253, 111.350017, 1023.44582, 1940.91371, 2760.93875, 3380.70601, 3697.40056, 3148.27271, 2397.06547, 1538.07881, 959.311907, 638.808791, 432.238834, 323.764805, 297.549475, 337.755613, 428.545989, 565.188595, 699.5487, 831.428743, 960.63116, 1086.95839, 1210.21287, 1330.19703, 1426.82895, 1528.14677}, 
{346.268512, 344.320946, 342.299719, 340.206661, 338.043602, 335.812371, 333.5148, 331.152718, 328.727956, 326.242343, 323.697709, 321.095886, 318.438702, 315.727989, 312.965575, 310.153292, 307.292969, 304.386437, 301.435525, 298.442065, 295.407885, 292.334816, 289.224689, 286.079333, 282.900578, 279.690255, 276.450193, 273.182223, 269.888176, 266.56988, 263.229167, 259.867866, 256.487807, 253.090821, 249.745437, 246.418732, 243.076983, 239.719855, 236.347013, 232.958124, 229.552853, 226.130865, 222.691826, 219.235403, 215.761259, 212.269062, 208.758476, 205.229168, 201.680803, 198.113046, 194.525563, 190.830217, 187.116379, 183.38562, 179.639507, 175.87961, 172.107498, 168.32474, 164.532904, 160.733561, 156.928278, 153.118625, 149.306171, 145.492484, 141.679135, 137.867692, 134.059723, 130.316305, 126.697084, 123.080902, 119.46547, 115.848499, 112.227701, 108.600788, 104.96547, 101.319459, 97.6604674, 93.9862054, 90.294385, 86.5827175, 82.8489144, 79.0906872, 75.3057473, 71.4918061, 67.5080891, 63.4244637, 59.3124615, 55.1742882, 51.0121494, 46.8282506, 42.6247973, 38.4039953, 34.16805, 29.919167, 25.6595519, 21.3914104, 17.1169478, 12.83837, 8.55788232, 4.27769047, 0, -3.76442042, -7.5164676, -11.2683326, -15.0322065, -18.8202805, -22.6447455, -26.5177926, -30.451613, -34.4583978, -38.5503379, -42.7396244, -47.0384486, -51.4590013, -56.0134738, -60.7140571, -65.5729422, -70.5463537, -75.6384873, -80.9881254, -86.5954661, -92.4607077, -98.5840484, -104.965686, -111.60582, -118.504648, -125.662367, -133.079177, -140.755275, -148.69086, -156.670776, -164.989539, -173.711952, -182.90282, -192.945669, -203.745941, -215.12409, -227.049303, -239.490769, -252.417676, -265.799212, -278.981408, -292.746267, -307.184898, -322.388412, -338.447918, -356.641011, -375.575696, -395.046462, -414.847799, -434.774197, -454.029044, -473.720387, -493.765921, -514.32411, -536.044971, -559.357828, -584.691998, -615.632875, -648.736418, -677.115599, -694.263363, -704.651876, -713.71344, -722.021966, -721.919103, -709.525243, -682.984759, -638.942778, -566.623171, -459.44769, -361.44604, -207.646184, 26.0482623, 363.733683, 829.506461, 2084.01239, 3355.66109, 4509.4116, 5410.22295, 5923.05416, 5328.54282, 4465.51705, 3449.35935, 2745.38025, 2330.44908, 2035.63185, 1845.08805, 1742.97718, 1713.45876, 1740.69228, 1824.24003, 1909.81087, 1996.91077, 2085.04572, 2173.72167, 2262.44459, 2350.72047, 2412.16589, 2463.52166}, 
{326.197572, 323.555743, 320.934535, 318.332357, 315.747622, 313.178738, 310.624118, 308.082173, 305.551312, 303.029947, 300.516489, 298.009348, 295.506936, 293.007663, 290.509939, 288.012177, 285.512786, 283.010177, 280.502762, 277.98895, 275.467154, 272.935783, 270.393249, 267.837962, 265.268333, 262.682773, 260.079693, 257.457504, 254.814616, 252.14944, 249.460387, 246.745869, 244.004295, 241.234076, 238.377931, 235.463289, 232.518246, 229.543023, 226.537835, 223.502902, 220.438443, 217.344674, 214.221814, 211.070081, 207.889694, 204.680871, 201.443829, 198.178788, 194.885964, 191.565577, 188.217844, 184.786741, 181.329947, 177.848902, 174.345041, 170.819801, 167.274622, 163.710939, 160.13019, 156.533813, 152.923245, 149.299923, 145.665285, 142.020768, 138.36781, 134.707847, 131.042318, 127.417191, 123.877367, 120.33244, 116.780959, 113.221476, 109.652541, 106.072704, 102.480516, 98.8745278, 95.2532895, 91.6153517, 87.9592651, 84.2835803, 80.5868476, 76.8676177, 73.1244412, 69.3558685, 65.4528234, 61.4699337, 57.4631206, 53.4344274, 49.3858974, 45.3195737, 41.2374997, 37.1417184, 33.0342733, 28.9172075, 24.7925642, 20.6623868, 16.5287184, 12.3936023, 8.25908168, 4.12719984, 0, -3.6889738, -7.35474513, -11.0094906, -14.6653868, -18.3346103, -22.0293377, -25.7617457, -29.5440108, -33.3883095, -37.3068186, -41.3117147, -45.4151742, -49.6293738, -53.9664901, -58.4386998, -63.0581793, -67.799791, -72.6703808, -77.7665268, -83.0924097, -88.6522102, -94.4501089, -100.490286, -106.776924, -113.314201, -120.106299, -127.157399, -134.471681, -142.053326, -149.753532, -157.785555, -166.199472, -175.045358, -184.512954, -194.582503, -205.197003, -216.364629, -228.093559, -240.391966, -253.268027, -266.854795, -280.997562, -295.680072, -310.886066, -326.59929, -342.327363, -358.649182, -375.66752, -393.48515, -412.204846, -432.804075, -453.441848, -472.974776, -490.848834, -507.713278, -523.677115, -538.849354, -553.003007, -566.659438, -580.706546, -600.418643, -615.988203, -619.66566, -606.691417, -577.372217, -529.689272, -465.922891, -385.168865, -270.759621, -114.944237, 19.4044312, 221.526282, 523.61893, 957.879993, 1556.50709, 3138.402, 4752.38213, 6233.96904, 7418.68432, 8142.04954, 7547.37844, 6623.87229, 5503.71317, 4711.04941, 4222.90977, 3853.00369, 3586.52589, 3408.67107, 3304.63392, 3259.60917, 3266.86238, 3286.26722, 3316.23313, 3355.16955, 3401.48592, 3453.5917, 3509.89631, 3544.32202, 3533.28461}, 
{308.114675, 305.49012, 302.91027, 300.372586, 297.87453, 295.413563, 292.987146, 290.59274, 288.227806, 285.889804, 283.576197, 281.284446, 279.012011, 276.756353, 274.514934, 272.285215, 270.064657, 267.850721, 265.640867, 263.432558, 261.223255, 259.010417, 256.791507, 254.563986, 252.325314, 250.072954, 247.804365, 245.517009, 243.208347, 240.875841, 238.516951, 236.129138, 233.709864, 231.256589, 228.673016, 226.005457, 223.301351, 220.561204, 217.785519, 214.974798, 212.129548, 209.25027, 206.337468, 203.391648, 200.413311, 197.402963, 194.361107, 191.288246, 188.184885, 185.051526, 181.888675, 178.663712, 175.410982, 172.131706, 168.827107, 165.498405, 162.146823, 158.773583, 155.379906, 151.967015, 148.536131, 145.088475, 141.625271, 138.147738, 134.657101, 131.154579, 127.641395, 124.154, 120.728, 117.291958, 113.844812, 110.3855, 106.912961, 103.426133, 99.9239536, 96.4053607, 92.8692925, 89.314687, 85.7404824, 82.1456167, 78.5290279, 74.889654, 71.2264332, 67.5383034, 63.7235088, 59.8334523, 55.9207472, 51.9875993, 48.0362143, 44.0687979, 40.0875561, 36.0946944, 32.0924187, 28.0829348, 24.0684484, 20.0511652, 16.0332911, 12.0170317, 8.0045929, 3.99818041, 0, -3.51270233, -7.00628527, -10.492272, -13.9821857, -17.4875495, -21.0198866, -24.5907201, -28.2115733, -31.8939693, -35.6494312, -39.4894823, -43.4256457, -47.4694446, -51.6324021, -55.9260415, -60.3618858, -64.9164092, -69.5961278, -74.4918426, -79.607566, -84.9473108, -90.5150897, -96.3149153, -102.3508, -108.626757, -115.146799, -121.914937, -128.935186, -136.211558, -143.62718, -151.351275, -159.42412, -167.885992, -176.814604, -186.231515, -196.148306, -206.594024, -217.597719, -229.188437, -241.395227, -254.705665, -268.550719, -282.869725, -297.602018, -312.686936, -327.546431, -342.766568, -358.416029, -374.563495, -391.277648, -409.411937, -427.291117, -444.587889, -460.861889, -475.441919, -487.76042, -497.249834, -500.648426, -500.695128, -503.067976, -519.267776, -532.103367, -527.28947, -497.329947, -446.232487, -372.724335, -281.404686, -172.523142, -24.7988187, 170.870306, 344.79381, 596.988077, 963.829436, 1481.69422, 2186.95876, 3978.12552, 5813.91317, 7515.16649, 8902.73028, 9797.44933, 9282.47393, 8408.13971, 7311.44628, 6528.36449, 6037.96779, 5653.0916, 5361.12129, 5149.4422, 5005.4397, 4916.49912, 4934.71915, 4929.30355, 4908.7493, 4881.55336, 4856.21271, 4841.22431, 4845.08515, 4814.21138, 4731.18224}, 
{291.464229, 289.590747, 287.711549, 285.825864, 283.932921, 282.031951, 280.122181, 278.202843, 276.273165, 274.332376, 272.379707, 270.414386, 268.435643, 266.442707, 264.434808, 262.411175, 260.371037, 258.313625, 256.238167, 254.143893, 252.030032, 249.895814, 247.740468, 245.563223, 243.363309, 241.139956, 238.892393, 236.619848, 234.321553, 231.996735, 229.644625, 227.264452, 224.855445, 222.416834, 219.918973, 217.376136, 214.802946, 212.199567, 209.566167, 206.902913, 204.209969, 201.487504, 198.735684, 195.954674, 193.144642, 190.305754, 187.438176, 184.542075, 181.617618, 178.66497, 175.684298, 172.645848, 169.580356, 166.488637, 163.371506, 160.229778, 157.064268, 153.875791, 150.665162, 147.433197, 144.180711, 140.908518, 137.617434, 134.308274, 130.981853, 127.638985, 124.280487, 120.9292, 117.607439, 114.270588, 110.918035, 107.549167, 104.16337, 100.760033, 97.338541, 93.8982827, 90.4386447, 86.9590143, 83.4587785, 79.9373245, 76.3940394, 72.8283103, 69.2395245, 65.627069, 61.8995717, 58.1037084, 54.2872458, 50.4525161, 46.601852, 42.7375858, 38.8620499, 34.9775769, 31.0864991, 27.191149, 23.2938591, 19.3969618, 15.5027896, 11.6136748, 7.73195, 3.85994758, 0, -3.1252691, -6.27211294, -9.4491745, -12.6650967, -15.9285226, -19.2480951, -22.6324573, -26.0902519, -29.6301221, -33.2607109, -36.9906611, -40.8286157, -44.7832178, -48.8631102, -53.0769361, -57.4333382, -61.9443102, -66.6189738, -71.4583931, -76.4719292, -81.6689429, -87.0587953, -92.6508473, -98.4544598, -104.478994, -110.73381, -117.22827, -123.971734, -130.973564, -138.222167, -145.75554, -153.589329, -161.739183, -170.058303, -178.643559, -187.635138, -197.097419, -207.094782, -217.691606, -228.952273, -241.768953, -255.126298, -268.926729, -283.072665, -297.466529, -312.317498, -327.144545, -341.773402, -356.029799, -369.739468, -382.235048, -394.438028, -408.548014, -424.946701, -440.304227, -452.957143, -461.242, -459.288131, -450.595492, -443.253738, -450.17458, -452.88684, -436.456366, -392.983666, -326.48909, -235.521583, -125.3351, 3.78307966, 176.130694, 402.098803, 619.539404, 922.542477, 1346.60876, 1927.23899, 2699.9339, 4536.57608, 6427.18895, 8198.1778, 9675.9479, 10686.9045, 10360.1579, 9684.27172, 8784.49201, 8143.72269, 7745.63684, 7424.88344, 7172.58677, 6979.87114, 6837.86085, 6737.6802, 6867.28986, 6880.66221, 6810.8189, 6690.78158, 6553.57188, 6432.21147, 6359.72199, 6219.91783, 6057.40346}, 
{276.767662, 275.730022, 274.623853, 273.450616, 272.211769, 270.908773, 269.543086, 268.116168, 266.629478, 265.084475, 263.482619, 261.825369, 260.114184, 258.350524, 256.535847, 254.671614, 252.759284, 250.800315, 248.796167, 246.7483, 244.658172, 242.527244, 240.356974, 238.148821, 235.904246, 233.624707, 231.311664, 228.966575, 226.590901, 224.1861, 221.753632, 219.294956, 216.811532, 214.304819, 211.829733, 209.359881, 206.868227, 204.354495, 201.818409, 199.259694, 196.678074, 194.073274, 191.445019, 188.793032, 186.117039, 183.416763, 180.69193, 177.942264, 175.167489, 172.36733, 169.541511, 166.653007, 163.739088, 160.800276, 157.837093, 154.850059, 151.839697, 148.806526, 145.751069, 142.673846, 139.575378, 136.456188, 133.316795, 130.157722, 126.979489, 123.782617, 120.567629, 117.349037, 114.141021, 110.915242, 107.671313, 104.40885, 101.127465, 97.8267743, 94.5063905, 91.1659281, 87.8050011, 84.4232236, 81.0202098, 77.5955736, 74.1489292, 70.6798907, 67.1880721, 63.6730875, 60.0456475, 56.3516878, 52.6382129, 48.9077219, 45.162714, 41.4056884, 37.6391442, 33.8655807, 30.0874969, 26.3073921, 22.5277654, 18.7511161, 14.9799432, 11.216746, 7.46402358, 3.7242752, 0, -2.71017731, -5.48868496, -8.34087076, -11.2720825, -14.2876681, -17.3929754, -20.593352, -23.894146, -27.3007051, -30.8183771, -34.4525099, -38.2084513, -42.0915491, -46.1071511, -50.2606052, -54.5572592, -59.0476355, -63.7435354, -68.5481262, -73.4764359, -78.5434927, -83.7643246, -89.1539597, -94.7274262, -100.499752, -106.485965, -112.701095, -119.160167, -125.878212, -132.939408, -140.264277, -147.8471, -155.682161, -163.446221, -171.292323, -179.458181, -188.033336, -197.107326, -206.769689, -217.109964, -229.336671, -242.079808, -255.209984, -268.597806, -282.113881, -296.716176, -310.9161, -324.312422, -336.50391, -347.089332, -353.966323, -360.513947, -370.92978, -386.331649, -401.549553, -414.2366, -422.045895, -418.646212, -406.580521, -392.738336, -386.828858, -374.33128, -344.559455, -291.054386, -214.519743, -112.371413, 11.6324911, 158.431802, 352.221174, 604.041019, 864.384938, 1214.15505, 1685.35027, 2309.96952, 3120.01173, 4910.4328, 6759.53543, 8508.57928, 9998.82403, 11071.5294, 10945.0051, 10498.7607, 9841.03425, 9383.92154, 9115.38379, 8896.7748, 8722.38612, 8586.50931, 8483.43589, 8407.45742, 8675.81367, 8736.46045, 8645.60269, 8459.44534, 8234.19335, 8026.05166, 7891.22522, 7662.38801, 7425.75215}, 
{263.768306, 262.768937, 261.711772, 260.59786, 259.428249, 258.203989, 256.926126, 255.595711, 254.213792, 252.781417, 251.299635, 249.769494, 248.192044, 246.568332, 244.899407, 243.186319, 241.430114, 239.631843, 237.792554, 235.913295, 233.995114, 232.039061, 230.046185, 228.017532, 225.954153, 223.857096, 221.727409, 219.566141, 217.37434, 215.153056, 212.903337, 210.626231, 208.322786, 205.994053, 203.677984, 201.356399, 199.010675, 196.640661, 194.24621, 191.827172, 189.383397, 186.914738, 184.421044, 181.902167, 179.357958, 176.788267, 174.192945, 171.571844, 168.924815, 166.251708, 163.552373, 160.795217, 158.012218, 155.203907, 152.370819, 149.513486, 146.63244, 143.728214, 140.80134, 137.852351, 134.88178, 131.89016, 128.878023, 125.845901, 122.794328, 119.723835, 116.634956, 113.543138, 110.463469, 107.365722, 104.249464, 101.114259, 97.9596741, 94.7852746, 91.5906264, 88.3752952, 85.1388468, 81.8808471, 78.6008619, 75.2984568, 71.9731978, 68.6246506, 65.252381, 61.8559548, 58.3377983, 54.7480902, 51.1381769, 47.5107764, 43.8686069, 40.2143865, 36.5508335, 32.880666, 29.2066021, 25.53136, 21.8576578, 18.1882137, 14.5257458, 10.8729724, 7.23261151, 3.60738132, 0, -2.50115979, -5.08526884, -7.75622804, -10.5179383, -13.3743004, -16.3292154, -19.3865841, -22.5503073, -25.8242861, -29.2124212, -32.7186136, -36.3467641, -40.1007736, -43.9845431, -48.0019733, -52.1569653, -56.5122655, -61.0801814, -65.7315106, -70.4827637, -75.3504514, -80.3510843, -85.5011733, -90.8172289, -96.3157617, -102.013283, -107.926302, -114.071331, -120.46488, -127.211262, -134.206991, -141.442237, -148.90717, -156.255988, -163.646846, -171.327493, -179.388892, -187.922003, -197.01779, -206.767213, -218.584719, -230.834985, -243.35003, -255.961874, -268.502534, -281.562142, -294.025077, -305.53383, -315.730891, -324.258751, -329.336232, -333.769533, -341.128284, -352.334978, -362.907548, -370.790465, -373.928199, -367.56547, -352.960079, -334.315018, -318.211212, -293.342736, -252.398213, -189.917237, -103.575767, 10.3898795, 150.461777, 319.034303, 537.867738, 817.767912, 1114.04489, 1501.53808, 2010.26338, 2670.23669, 3511.47392, 5288.24215, 7125.2433, 8871.43049, 10375.7568, 11487.1754, 11467.8649, 11144.7358, 10618.216, 10269.6219, 10088.1963, 9946.14527, 9838.56345, 9760.54557, 9707.18633, 9673.58046, 10006.9833, 10108.8884, 10039.7507, 9860.02498, 9630.16616, 9410.6291, 9261.86865, 9022.93639, 8729.6314}, 
{251.80006, 250.779002, 249.715024, 248.608616, 247.460266, 246.270463, 245.039696, 243.768455, 242.457227, 241.106503, 239.71677, 238.288519, 236.822237, 235.318415, 233.77754, 232.200102, 230.586589, 228.937491, 227.253297, 225.534496, 223.781575, 221.995026, 220.175336, 218.322994, 216.438489, 214.522311, 212.574948, 210.59689, 208.588624, 206.550641, 204.483428, 202.387476, 200.263272, 198.111307, 195.946443, 193.76168, 191.549844, 189.310959, 187.045045, 184.752126, 182.432224, 180.085362, 177.711562, 175.310846, 172.883237, 170.428758, 167.947431, 165.439278, 162.904321, 160.342584, 157.754089, 155.112156, 152.444088, 149.750487, 147.031954, 144.28909, 141.522498, 138.732777, 135.92053, 133.086357, 130.230861, 127.354642, 124.458301, 121.542441, 118.607661, 115.654565, 112.683752, 109.713912, 106.763299, 103.795211, 100.809076, 97.8043229, 94.7803807, 91.736678, 88.6726437, 85.5877064, 82.4812951, 79.3528384, 76.2017651, 73.0275041, 69.829484, 66.6071337, 63.3598819, 60.0871574, 56.679948, 53.1941839, 49.6870986, 46.1616401, 42.6207561, 39.0673946, 35.5045035, 31.9350307, 28.3619241, 24.7881315, 21.2166009, 17.6502801, 14.092117, 10.5450596, 7.01205569, 3.49605319, 0, -2.31217964, -4.71973805, -7.22524719, -9.83127901, -12.5404054, -15.3551985, -18.27823, -21.312072, -24.4592964, -27.7224752, -31.1041803, -34.6069836, -38.2334572, -41.9861729, -45.8677027, -49.8806186, -54.0987389, -58.5348136, -63.030262, -67.6029233, -72.2706365, -77.0512405, -81.9625744, -87.0224773, -92.2487882, -97.6593461, -103.27199, -109.104559, -115.174892, -121.599797, -128.261856, -135.149217, -142.250028, -149.227107, -156.231267, -163.500077, -171.119286, -179.174641, -187.75189, -196.93678, -208.271449, -219.942006, -231.749254, -243.493993, -254.977027, -266.341584, -276.960433, -286.548768, -294.821786, -301.494683, -305.315356, -308.148549, -312.512861, -319.00819, -324.317081, -326.880889, -325.140972, -316.620273, -300.88729, -278.512429, -251.960051, -214.199632, -162.154971, -91.7318057, 2.88917623, 126.735461, 280.910571, 469.42467, 710.675784, 1014.92335, 1343.38289, 1762.90298, 2300.71978, 2984.06947, 3840.18824, 5568.01895, 7355.16441, 9060.93411, 10544.6376, 11665.5843, 11740.2078, 11532.6108, 11134.764, 10896.3766, 10807.3978, 10750.3853, 10720.6578, 10713.5338, 10724.3319, 10748.3707, 11135.911, 11285.7595, 11257.3333, 11110.0496, 10903.3255, 10696.5781, 10549.2247, 10320.5479, 9984.66346}, 
{240.493549, 239.453152, 238.383991, 237.286012, 236.159163, 235.003389, 233.818636, 232.604851, 231.361979, 230.089968, 228.788763, 227.458311, 226.098558, 224.709449, 223.290932, 221.842952, 220.365457, 218.858391, 217.321701, 215.755334, 214.159236, 212.533352, 210.87763, 209.192015, 207.476453, 205.730892, 203.955276, 202.149553, 200.313669, 198.447569, 196.5512, 194.624509, 192.667441, 190.679942, 188.653612, 186.592745, 184.501738, 182.380808, 180.230173, 178.050049, 175.840653, 173.602203, 171.334916, 169.039009, 166.714699, 164.362203, 161.981739, 159.573523, 157.137774, 154.674707, 152.18454, 149.645715, 147.080696, 144.490173, 141.874835, 139.235371, 136.57247, 133.886821, 131.179114, 128.450037, 125.70028, 122.930533, 120.141483, 117.333821, 114.508235, 111.665416, 108.806051, 105.953035, 103.12873, 100.288027, 97.4301779, 94.5544312, 91.660037, 88.746245, 85.8123049, 82.8574665, 79.8809794, 76.8820934, 73.8600582, 70.8141236, 67.7435393, 64.647555, 61.5254205, 58.3763855, 55.0785882, 51.6943815, 48.287574, 44.8613457, 41.4188767, 37.9633469, 34.4979365, 31.0258253, 27.5501935, 24.0742211, 20.6010882, 17.1339747, 13.6760606, 10.2305261, 6.80055117, 3.38931578, 0, -2.13372714, -4.37428686, -6.7229933, -9.18116057, -11.7501028, -14.4311342, -17.2255687, -20.1347207, -23.1599041, -26.3024331, -29.5636219, -32.9447846, -36.4472353, -40.0722881, -43.8212572, -47.6954567, -51.7796519, -56.0870783, -60.4302918, -64.8284889, -69.300866, -73.8666197, -78.5449464, -83.3550428, -88.3161052, -93.4473302, -98.7679143, -104.297054, -110.053946, -116.166912, -122.506011, -129.057701, -135.80844, -142.44661, -149.107709, -156.00672, -163.219528, -170.822013, -178.890058, -187.499545, -198.23292, -209.200981, -220.184849, -230.965645, -241.324486, -251.013449, -259.849958, -267.622397, -274.119146, -279.128586, -281.97057, -283.474654, -284.980972, -286.72797, -286.704807, -283.910526, -277.344168, -266.987266, -250.633079, -225.003064, -188.240498, -137.809674, -74.8926527, 3.14857217, 105.424897, 238.076056, 404.672735, 610.68862, 871.017233, 1195.2237, 1551.40651, 1996.97692, 2555.8539, 3251.95643, 4109.20347, 5761.35749, 7470.03309, 9106.68837, 10542.7814, 11649.7704, 11805.0373, 11703.5004, 11428.3644, 11297.7013, 11301.748, 11332.8798, 11386.2268, 11456.9187, 11540.0853, 11630.8566, 12056.7754, 12256.1706, 12282.3533, 12188.6348, 12028.3263, 11854.7391, 11721.1845, 11516.7881, 11155.9431}, 
{229.479402, 228.484324, 227.469055, 226.433179, 225.376285, 224.297958, 223.197785, 222.075351, 220.930243, 219.762047, 218.57035, 217.354737, 216.114796, 214.850112, 213.560272, 212.244861, 210.903467, 209.535674, 208.141071, 206.719242, 205.269775, 203.792255, 202.286269, 200.751403, 199.187243, 197.593376, 195.969387, 194.314864, 192.629392, 190.912558, 189.163947, 187.383147, 185.569743, 183.723321, 181.817992, 179.866615, 177.882357, 175.865632, 173.816853, 171.736432, 169.624783, 167.482318, 165.309452, 163.106596, 160.874164, 158.612569, 156.322224, 154.003542, 151.656937, 149.28282, 146.881606, 144.437787, 141.968041, 139.473127, 136.953802, 134.410826, 131.844956, 129.25695, 126.647567, 124.017564, 121.3677, 118.698733, 116.011421, 113.306523, 110.584796, 107.846998, 105.093889, 102.352184, 99.6479787, 96.9284915, 94.1927984, 91.4399745, 88.6690954, 85.8792363, 83.0694727, 80.2388799, 77.3865332, 74.511508, 71.6128798, 68.6897237, 65.7411153, 62.7661298, 59.7638426, 56.7333291, 53.5402103, 50.2530956, 46.9421988, 43.610926, 40.2626836, 36.9008777, 33.5289145, 30.1502002, 26.7681411, 23.3861434, 20.0076133, 16.635957, 13.2745807, 9.9268906, 6.59629299, 3.28619405, 0, -1.95629261, -4.03110957, -6.22453144, -8.53663879, -10.9675122, -13.5172321, -16.1858792, -18.9735341, -21.8802772, -24.9061891, -28.0513504, -31.3158416, -34.6997434, -38.2031362, -41.8261007, -45.5687173, -49.5276008, -53.7166222, -57.9175113, -62.1510344, -66.437958, -70.7990486, -75.2550726, -79.8267964, -84.5349864, -89.400409, -94.4438308, -99.6860181, -105.147737, -110.974503, -117.016593, -123.257348, -129.680111, -136.00153, -142.338293, -148.878209, -155.684625, -162.820894, -170.350365, -178.336388, -188.305189, -198.412019, -208.434014, -218.148307, -227.332035, -235.536682, -242.821447, -249.019876, -253.965517, -257.491918, -259.368746, -259.571506, -258.430074, -255.871002, -250.997383, -243.282027, -232.197742, -219.843093, -202.468372, -173.459415, -127.217674, -65.0805684, 8.32581781, 94.370561, 204.581205, 345.822389, 523.44213, 743.911869, 1019.26401, 1358.38535, 1737.12332, 2202.4871, 2774.80019, 3474.38612, 4321.56841, 5879.85202, 7490.58366, 9038.29148, 10407.5036, 11482.7483, 11705.3571, 11698.519, 11536.7035, 11507.1117, 11600.0068, 11717.0139, 11852.8283, 12002.1451, 12159.6594, 12320.0664, 12763.755, 13009.2184, 13098.8133, 13074.8962, 12979.8237, 12855.9525, 12745.6391, 12573.2225, 12208.5652}, 
{218.388245, 217.565456, 216.720598, 215.853247, 214.962977, 214.049364, 213.111983, 212.150407, 211.164213, 210.152975, 209.116267, 208.053666, 206.964745, 205.84908, 204.706246, 203.535817, 202.337368, 201.110474, 199.854711, 198.569653, 197.254874, 195.909951, 194.534457, 193.127968, 191.690058, 190.220302, 188.718276, 187.183554, 185.615712, 184.014323, 182.378963, 180.709207, 179.004629, 177.264805, 175.458084, 173.60031, 171.707703, 169.780851, 167.820343, 165.826767, 163.800712, 161.742764, 159.653513, 157.533547, 155.383454, 153.203822, 150.995239, 148.758295, 146.493576, 144.201671, 141.883169, 139.530264, 137.15212, 134.749506, 132.323194, 129.873954, 127.402555, 124.909768, 122.396363, 119.86311, 117.31078, 114.740143, 112.151968, 109.547026, 106.926088, 104.289923, 101.639302, 99.0030363, 96.4092649, 93.8009239, 91.1769659, 88.536343, 85.8780078, 83.2009125, 80.5040095, 77.7862513, 75.0465901, 72.2839783, 69.4973684, 66.6857126, 63.8479633, 60.983073, 58.089994, 55.1676786, 52.0713058, 48.8747387, 45.6535687, 42.4114137, 39.1518918, 35.8786211, 32.5952198, 29.305306, 26.0124976, 22.7204128, 19.4326697, 16.1528864, 12.884681, 9.63167157, 6.3974762, 3.18571297, 0, -1.77036634, -3.67240045, -5.70492672, -7.86676949, -10.1567531, -12.573702, -15.1164405, -17.7837929, -20.5745836, -23.487637, -26.5217775, -29.6758294, -32.948617, -36.3389648, -39.845697, -43.4676382, -47.3151821, -51.4030916, -55.4778315, -59.5621337, -63.6787305, -67.8503539, -72.0997362, -76.4496093, -80.9227055, -85.5417568, -90.3294955, -95.3086536, -100.501963, -106.084467, -111.87074, -117.83782, -123.962747, -129.978898, -135.985144, -142.145328, -148.509587, -155.128059, -162.05088, -169.328187, -178.324316, -187.375231, -196.273942, -204.813461, -212.786798, -219.870233, -226.00269, -231.006364, -234.703449, -236.916139, -237.576758, -236.262765, -232.757626, -226.813969, -218.121463, -206.398044, -191.36165, -176.364399, -156.664095, -123.553974, -69.0566995, 3.07997934, 86.437516, 181.580825, 300.907908, 451.385186, 638.912618, 870.180135, 1155.78802, 1504.12466, 1899.54089, 2378.16069, 2956.69308, 3651.84712, 4480.33185, 5935.09684, 7437.55048, 8885.34168, 10176.1193, 11207.5324, 11484.1704, 11558.7812, 11497.4675, 11558.1233, 11730.9341, 11926.1726, 12138.02, 12360.6577, 12588.2669, 12815.0287, 13251.0282, 13533.9997, 13690.7159, 13747.9493, 13732.4727, 13671.0586, 13590.4798, 13451.4166, 13107.6244}, 
{205.34996, 205.123404, 204.840503, 204.501892, 204.108202, 203.660066, 203.158118, 202.60299, 201.995315, 201.335726, 200.624856, 199.863338, 199.051804, 198.190888, 197.281222, 196.32344, 195.318174, 194.266057, 193.167722, 192.023802, 190.83493, 189.601738, 188.32486, 187.004929, 185.642576, 184.238436, 182.793141, 181.307325, 179.781619, 178.216656, 176.613071, 174.971495, 173.292562, 171.576903, 169.822937, 168.032451, 166.207258, 164.348064, 162.455574, 160.530492, 158.573522, 156.585371, 154.566743, 152.518342, 150.440874, 148.335043, 146.201554, 144.041111, 141.85442, 139.642186, 137.405113, 135.149537, 132.870409, 130.568313, 128.243832, 125.897547, 123.530042, 121.1419, 118.733703, 116.306034, 113.859476, 111.394611, 108.912023, 106.412293, 103.896006, 101.363743, 98.8160866, 96.2776041, 93.7722853, 91.251249, 88.7135232, 86.1581358, 83.584115, 80.9904886, 78.3762847, 75.7405313, 73.0822564, 70.4004879, 67.694254, 64.9625825, 62.2045015, 59.4190389, 56.6052229, 53.7620813, 50.7410957, 47.6181709, 44.4709847, 41.3033535, 38.1190934, 34.9220205, 31.7159511, 28.5047012, 25.292087, 22.0819247, 18.8780306, 15.6842206, 12.504311, 9.34211801, 6.20145773, 3.08614634, 0, -1.49017289, -3.13985657, -4.94589407, -6.90512843, -9.01440268, -11.2705598, -13.670443, -16.2108951, -18.8887592, -21.7008784, -24.6440957, -27.7152542, -30.9111968, -34.2287665, -37.6648066, -41.2161598, -45.0239809, -49.1057288, -53.1318275, -57.1300441, -61.1281455, -65.1538986, -69.2350703, -73.3994276, -77.6747373, -82.0887665, -86.669282, -91.4440508, -96.4408398, -101.927074, -107.602987, -113.42445, -119.347331, -125.008968, -130.524497, -136.093995, -141.768891, -147.600614, -153.640595, -159.940261, -167.634581, -175.361673, -182.96097, -190.271902, -197.133904, -204.254843, -210.388607, -215.15752, -218.183905, -219.090087, -216.901052, -212.566539, -206.815237, -199.746377, -190.174046, -177.489334, -161.08333, -144.028901, -121.198594, -83.4484745, -22.2326226, 58.6732198, 155.038018, 267.40436, 404.403476, 570.953776, 768.712759, 1001.75341, 1286.1054, 1631.03503, 2035.47241, 2520.61018, 3099.29256, 3784.36373, 4588.66789, 5933.39336, 7320.95418, 8662.10855, 9867.61462, 10848.2306, 11172.1823, 11321.1151, 11352.5866, 11494.1797, 11736.1503, 11997.9109, 12273.2599, 12555.9956, 12839.9164, 13118.8205, 13432.269, 13709.4578, 13940.1438, 14114.0837, 14221.0344, 14250.7526, 14192.9952, 14087.9459, 13825.7986}, 
{194.679794, 194.802502, 194.850728, 194.825666, 194.728506, 194.560441, 194.32266, 194.016357, 193.642722, 193.202947, 192.698223, 192.129742, 191.498696, 190.806275, 190.053671, 189.242076, 188.372682, 187.446678, 186.465259, 185.429613, 184.340934, 183.200413, 182.00924, 180.768608, 179.479708, 178.143731, 176.76187, 175.335314, 173.865257, 172.352889, 170.799402, 169.205987, 167.573836, 165.90414, 164.210435, 162.487479, 160.729886, 158.938445, 157.113949, 155.257187, 153.368951, 151.450032, 149.50122, 147.523306, 145.517082, 143.483339, 141.422866, 139.336456, 137.224898, 135.088985, 132.929506, 130.761393, 128.570991, 126.358784, 124.125255, 121.870889, 119.596171, 117.301584, 114.987615, 112.654745, 110.303461, 107.934246, 105.547585, 103.143962, 100.723862, 98.2877681, 95.8361657, 93.3919045, 90.9772973, 88.5467013, 86.099151, 83.6336808, 81.1493253, 78.645119, 76.1200964, 73.573292, 71.0037402, 68.4104756, 65.7925327, 63.1489459, 60.4787499, 57.780979, 55.0546677, 52.2988507, 49.3606364, 46.3182179, 43.2516142, 40.1647901, 37.0617104, 33.9463399, 30.8226434, 27.6945857, 24.5661314, 21.4412456, 18.3238928, 15.218038, 12.1276458, 9.05668116, 6.00910878, 2.98889346, 0, -1.34403908, -2.85381904, -4.52547418, -6.35513879, -8.33894718, -10.4730336, -12.7535325, -15.176578, -17.7383045, -20.4348463, -23.2623376, -26.2169128, -29.2947062, -32.4918521, -35.8044847, -39.2287384, -42.9107103, -46.867958, -50.7574141, -54.6073479, -58.4460284, -62.301725, -66.2027067, -70.1772428, -74.2536026, -78.460055, -82.8248695, -87.3763151, -92.142661, -97.4162184, -102.864399, -108.43626, -114.080857, -119.483688, -124.725588, -129.957678, -135.208081, -140.504923, -145.876327, -151.350419, -157.611072, -163.831085, -169.910286, -175.748499, -181.24555, -187.785954, -193.413676, -197.65737, -200.045692, -200.107295, -196.652816, -190.806502, -183.740554, -175.591732, -164.382781, -149.085598, -128.672077, -104.36574, -72.3751234, -26.4524395, 38.9185956, 122.868424, 223.489674, 342.223708, 486.18613, 659.887977, 864.872344, 1104.8808, 1394.53736, 1742.3103, 2150.77783, 2635.20449, 3206.09742, 3873.96379, 4649.31074, 5902.70641, 7194.58174, 8445.42864, 9575.73905, 10506.0049, 10850.0597, 11039.4927, 11126.2206, 11317.7368, 11604.3759, 11911.6487, 12233.1311, 12562.3988, 12893.0274, 13218.5927, 13553.0437, 13868.4903, 14149.414, 14380.2963, 14545.6186, 14629.8624, 14617.5093, 14523.5929, 14305.9194}, 
{186.065989, 186.304952, 186.466927, 186.553178, 186.56497, 186.503565, 186.370228, 186.166222, 185.892813, 185.551264, 185.142838, 184.6688, 184.130414, 183.528943, 182.865652, 182.141805, 181.358666, 180.517498, 179.619565, 178.666133, 177.658463, 176.597821, 175.485471, 174.322676, 173.110701, 171.850809, 170.544264, 169.192331, 167.796272, 166.357354, 164.876838, 163.35599, 161.796073, 160.198351, 158.577074, 156.92674, 155.241691, 153.52277, 151.770818, 149.986678, 148.171194, 146.325206, 144.449559, 142.545095, 140.612656, 138.653084, 136.667223, 134.655914, 132.620001, 130.560325, 128.47773, 126.390101, 124.280868, 122.150503, 119.999482, 117.828275, 115.637357, 113.427201, 111.198279, 108.951066, 106.686033, 104.403654, 102.104402, 99.7887507, 97.4571725, 95.1101408, 92.7481285, 90.394614, 88.0725229, 85.7348823, 83.3806738, 81.0088793, 78.6184805, 76.2084593, 73.7777973, 71.3254764, 68.8504783, 66.3517849, 63.8283779, 61.279239, 58.7033501, 56.0996929, 53.4672492, 50.8050009, 47.9549584, 44.997891, 42.0164542, 39.0147238, 35.9967757, 32.9666856, 29.9285292, 26.8863825, 23.8443212, 20.8064211, 17.776758, 14.7594076, 11.7584458, 8.77794832, 5.82199101, 2.89464964, 0, -1.31264276, -2.77856369, -4.39432988, -6.15650848, -8.06166659, -10.1063713, -12.2871899, -14.6006893, -17.0434367, -19.6119992, -22.3029439, -25.1128381, -28.0382487, -31.0757429, -34.2218879, -37.4732507, -40.9527289, -44.6749368, -48.3506941, -52.0036389, -55.6574089, -59.3356422, -63.0619766, -66.8600502, -70.7535007, -74.7659662, -78.9210845, -83.2424936, -87.7538314, -92.7043021, -97.8092697, -103.024702, -108.306568, -113.502558, -118.622779, -123.706211, -128.741304, -133.716511, -138.620283, -143.441069, -148.327123, -153.05846, -157.592264, -161.885722, -165.896018, -171.308418, -175.920008, -179.255954, -180.841423, -180.201579, -176.453366, -170.029113, -162.11136, -152.651335, -139.086345, -119.982397, -93.905498, -59.5035933, -15.2421276, 40.5028438, 108.407041, 191.810265, 290.562455, 407.39595, 549.927152, 723.530598, 932.729352, 1182.6772, 1481.64999, 1836.15754, 2244.6906, 2723.01179, 3280.4068, 3926.16131, 4669.56098, 5846.42472, 7057.87167, 8231.55421, 9295.12472, 10176.2356, 10520.8604, 10726.1162, 10840.3746, 11058.0594, 11369.73, 11705.0913, 12057.7457, 12421.2955, 12789.3432, 13155.491, 13615.6403, 14001.8311, 14315.7799, 14559.203, 14733.8169, 14841.3379, 14883.4825, 14788.1318, 14591.022}, 
{178.669995, 178.886664, 179.033134, 179.110457, 179.119684, 179.06187, 178.938065, 178.749322, 178.496693, 178.18123, 177.803987, 177.366014, 176.868364, 176.312089, 175.698242, 175.027875, 174.30204, 173.52179, 172.688175, 171.80225, 170.865066, 169.877674, 168.841129, 167.756481, 166.624783, 165.447087, 164.224445, 162.95791, 161.648534, 160.29737, 158.905468, 157.473882, 156.003664, 154.495866, 152.957742, 151.387113, 149.781725, 148.142429, 146.470077, 144.765519, 143.029606, 141.26319, 139.467121, 137.64225, 135.789428, 133.909507, 132.003337, 130.071769, 128.115655, 126.135844, 124.133189, 122.124183, 120.093696, 118.042239, 115.970323, 113.878461, 111.767164, 109.636945, 107.488315, 105.321786, 103.13787, 100.937079, 98.7199242, 96.4869181, 94.2385724, 91.9753989, 89.6979096, 87.431273, 85.2000659, 82.9539473, 80.6918307, 78.4126294, 76.115257, 73.798627, 71.4616527, 69.1032477, 66.7223254, 64.3177993, 61.8885828, 59.4335894, 56.9517325, 54.4419257, 51.9030823, 49.3341158, 46.5721116, 43.7003009, 40.8038602, 37.8869547, 34.9537494, 32.0084095, 29.0551, 26.097986, 23.1412326, 20.189005, 17.2454682, 14.3147873, 11.4011274, 8.50865364, 5.64153109, 2.80392484, 0, -1.34928305, -2.82793657, -4.43373566, -6.16445541, -8.01787094, -9.99175733, -12.0838897, -14.2920431, -16.6139928, -19.0475137, -21.5903809, -24.2403697, -26.995255, -29.8528121, -32.8108159, -35.8670416, -39.0955091, -42.5048857, -45.9204879, -49.3564289, -52.8268221, -56.3457807, -59.927418, -63.5858473, -67.3351819, -71.189535, -75.16302, -79.2697501, -83.5238386, -88.0780262, -92.7569688, -97.5331913, -102.379219, -107.35735, -112.395223, -117.396537, -122.306884, -127.071858, -131.637052, -135.948059, -139.693876, -143.154786, -146.326586, -149.205074, -151.786045, -155.808815, -159.089782, -161.188865, -161.665981, -160.081046, -155.98728, -148.959486, -140.073409, -129.061363, -112.912817, -89.8487221, -58.0900293, -13.8399127, 42.2043153, 107.144132, 177.032573, 259.000627, 353.839309, 465.410729, 602.780794, 772.626783, 983.418274, 1242.29721, 1549.83379, 1910.31556, 2315.68062, 2784.46587, 3325.23367, 3946.54639, 4656.96638, 5765.67766, 6905.46551, 8009.73685, 9011.89863, 9845.35778, 10181.3955, 10390.2949, 10517.8027, 10747.7119, 11071.1256, 11421.6276, 11793.197, 12179.8129, 12575.4545, 12974.1007, 13594.3279, 14061.4246, 14401.1648, 14639.3225, 14801.6714, 14913.9857, 15002.0391, 14905.7752, 14730.5381}, 
{170.921216, 171.128672, 171.272048, 171.352225, 171.370085, 171.326511, 171.222383, 171.058585, 170.835996, 170.555501, 170.217979, 169.824314, 169.375387, 168.872079, 168.315273, 167.705851, 167.044694, 166.332684, 165.570703, 164.759632, 163.900355, 162.993752, 162.040705, 161.042096, 159.998807, 158.911721, 157.781717, 156.60968, 155.396489, 154.143028, 152.850178, 151.518821, 150.149838, 148.744112, 147.305916, 145.834365, 144.328533, 142.789191, 141.217111, 139.613065, 137.977824, 136.312161, 134.616847, 132.892654, 131.140353, 129.360717, 127.554517, 125.722524, 123.865512, 121.984251, 120.079513, 118.164485, 116.227256, 114.268326, 112.2882, 110.287378, 108.266365, 106.225662, 104.165772, 102.087198, 99.9904422, 97.8760075, 95.7443962, 93.5961111, 91.4316548, 89.2515298, 87.0562388, 84.8695455, 82.7146551, 80.5440959, 78.3568625, 76.1519498, 73.9283522, 71.6850646, 69.4210816, 67.1353978, 64.827008, 62.4949068, 60.1380889, 57.755549, 55.3462818, 52.9092819, 50.443544, 47.9480628, 45.2606981, 42.4644013, 39.6440605, 36.8038994, 33.948142, 31.0810119, 28.206733, 25.3295292, 22.4536242, 19.5832418, 16.722606, 13.8759404, 11.0474688, 8.24141524, 5.46200336, 2.713457, 0, -1.42620271, -2.94702471, -4.56256943, -6.27294029, -8.07824071, -9.9785741, -11.9740439, -14.0647535, -16.2508063, -18.5323058, -20.9093553, -23.3820584, -25.9505183, -28.6148386, -31.3751226, -34.2314738, -37.1531204, -40.1357551, -43.2493181, -46.4872965, -49.8431777, -53.3104492, -56.882598, -60.5531116, -64.3154773, -68.1631824, -72.0897142, -76.08856, -80.1532071, -84.2538386, -88.4157909, -92.6395426, -96.9255724, -101.530379, -106.326431, -111.117925, -115.828534, -120.381929, -124.701784, -128.71177, -132.11023, -135.114746, -137.693078, -139.812984, -141.442224, -144.440568, -146.410762, -146.847561, -145.245722, -141.100001, -133.442628, -122.796195, -110.88638, -97.7299068, -79.8543924, -55.3539911, -22.3228577, 22.8391052, 79.9584334, 147.013388, 221.861847, 309.275006, 407.824091, 520.444267, 657.465145, 825.822163, 1034.82548, 1292.02619, 1596.26812, 1951.32041, 2351.8642, 2812.15482, 3339.10724, 3939.63648, 4620.65753, 5655.39825, 6717.88256, 7748.44724, 8687.42908, 9475.16486, 9816.17982, 10043.83, 10199.5142, 10447.7345, 10781.1614, 11141.1725, 11523.0029, 11921.8878, 12333.0624, 12751.7619, 13453.9979, 13982.4998, 14363.8439, 14624.6062, 14791.3631, 14890.6908, 14949.1654, 14872.6005, 14863.9416}, 
{163.185019, 163.396772, 163.549537, 163.644065, 163.681109, 163.661417, 163.585743, 163.454837, 163.26945, 163.030333, 162.738238, 162.393916, 161.998117, 161.551594, 161.055096, 160.509376, 159.915184, 159.273272, 158.584391, 157.849291, 157.068724, 156.243442, 155.374194, 154.461734, 153.506811, 152.510176, 151.472582, 150.394778, 149.277517, 148.121549, 146.927626, 145.696498, 144.428917, 143.125634, 141.790367, 140.422322, 139.020668, 137.58606, 136.119152, 134.6206, 133.091058, 131.531181, 129.941624, 128.323042, 126.67609, 125.001422, 123.299693, 121.571559, 119.817674, 118.038692, 116.23527, 114.414813, 112.571078, 110.704574, 108.815809, 106.905291, 104.973529, 103.021032, 101.048308, 99.0558649, 97.0442119, 95.0138574, 92.9653096, 90.8990773, 88.8156687, 86.7155925, 84.599357, 82.4896723, 80.4087155, 78.3112059, 76.1962128, 74.0628053, 71.9100528, 69.7370245, 67.5427895, 65.3264173, 63.0869769, 60.8235377, 58.5351688, 56.2209396, 53.8799192, 51.5111769, 49.113782, 46.6868036, 44.0678591, 41.3401395, 38.5887766, 35.8180791, 33.0323556, 30.2359147, 27.4330652, 24.6281156, 21.8253748, 19.0291512, 16.2437537, 13.4734909, 10.7226714, 7.99560385, 5.29659701, 2.6299595, 0, -1.46554655, -2.99829602, -4.60026387, -6.27346552, -8.01991642, -9.84163201, -11.7406277, -13.718919, -15.7785213, -17.9214501, -20.1497208, -22.4653488, -24.8703495, -27.3667385, -29.9565312, -32.6417429, -35.2871385, -37.8751262, -40.7181694, -43.7888726, -47.0598404, -50.5036774, -54.0929882, -57.8003774, -61.5984495, -65.4598092, -69.357061, -73.2628095, -77.1496593, -80.8116888, -84.4654883, -88.1372203, -91.8530473, -96.0000528, -100.423939, -104.874161, -109.268607, -113.525161, -117.56171, -121.296141, -124.685164, -127.596025, -129.939012, -131.624416, -132.562527, -134.626711, -135.273413, -133.922152, -129.992449, -122.903825, -110.965115, -96.0640288, -80.7315622, -65.3911389, -46.163224, -21.1002231, 11.745458, 54.9982182, 109.775028, 176.454526, 256.280049, 350.387841, 455.163557, 572.212486, 711.984289, 880.868915, 1086.98146, 1337.15914, 1631.91362, 1975.33463, 2368.73859, 2819.41526, 3332.4144, 3912.78577, 4565.57912, 5526.90631, 6512.98948, 7471.11284, 8348.56064, 9092.6171, 9442.24641, 9691.93286, 9878.24072, 10143.8616, 10483.3511, 10847.0448, 11231.7093, 11634.1109, 12051.0162, 12479.1916, 13184.342, 13744.4911, 14171.7406, 14478.1921, 14675.9469, 14777.1067, 14793.7731, 14759.0911, 14892.5826}, 
{155.621618, 155.872754, 156.067182, 156.20561, 156.288742, 156.317285, 156.291945, 156.213428, 156.082441, 155.899688, 155.665876, 155.381711, 155.0479, 154.665147, 154.23416, 153.755644, 153.230305, 152.65885, 152.041983, 151.380413, 150.674843, 149.925981, 149.134532, 148.301203, 147.426699, 146.511727, 145.556992, 144.563201, 143.53106, 142.461274, 141.35455, 140.211593, 139.033111, 137.819808, 136.577557, 135.304373, 133.998208, 132.659599, 131.289085, 129.887205, 128.454497, 126.991499, 125.498749, 123.976787, 122.42615, 120.847376, 119.241004, 117.607573, 115.947621, 114.261686, 112.550306, 110.811552, 109.048484, 107.261694, 105.451773, 103.619313, 101.764906, 99.8891444, 97.9926196, 96.0759235, 94.139648, 92.1843849, 90.2107261, 88.2192635, 86.2105889, 84.1852941, 82.1439711, 80.1119619, 78.1140143, 76.0996748, 74.0679308, 72.0177695, 69.9481785, 67.8581452, 65.7466569, 63.612701, 61.455265, 59.2733362, 57.0659021, 54.83195, 52.5704674, 50.2804416, 47.9608601, 45.6107103, 43.0596143, 40.3948045, 37.705549, 34.9963313, 32.2716352, 29.5359442, 26.7937419, 24.0495121, 21.3077383, 18.5729041, 15.8494932, 13.1419892, 10.4548758, 7.79263649, 5.15975502, 2.56071498, 0, -1.37537004, -2.81832477, -4.3307171, -5.91439996, -7.57122626, -9.30304893, -11.1117209, -12.9990951, -14.9670244, -17.0173617, -19.15196, -21.3726722, -23.6813513, -26.07985, -28.5700214, -31.1537184, -33.6474542, -36.0266046, -38.7122433, -41.6665075, -44.8515346, -48.2294618, -51.7624264, -55.4125656, -59.1420167, -62.9129171, -66.6874038, -70.4276143, -74.0956858, -77.4099297, -80.6657119, -83.8983175, -87.1430314, -90.7946026, -94.7085841, -98.644672, -102.530312, -106.29295, -109.860031, -113.159003, -116.489382, -119.293302, -121.425414, -122.740365, -123.092805, -124.126249, -123.459264, -120.49928, -114.653731, -105.330048, -90.5331967, -72.7872016, -55.1252852, -38.1888363, -17.7359171, 8.28289641, 41.917028, 85.7286675, 141.137935, 209.00557, 291.212379, 388.411054, 497.996306, 621.343294, 766.571822, 938.705793, 1143.03091, 1384.63894, 1667.66174, 1996.77413, 2380.07503, 2818.28841, 3314.90386, 3873.41094, 4497.29925, 5390.39785, 6305.77194, 7196.82624, 8016.96545, 8719.59427, 9074.45774, 9341.06001, 9551.51705, 9828.67458, 10168.7347, 10530.0956, 10910.8496, 11309.0889, 11722.9061, 12150.3933, 12758.1224, 13303.8715, 13773.328, 14152.1796, 14426.1137, 14580.8177, 14601.9793, 14638.9387, 14711.9123}, 
{148.24968, 148.611409, 148.913131, 149.155669, 149.339849, 149.466496, 149.536436, 149.550492, 149.509489, 149.414254, 149.265611, 149.064384, 148.811399, 148.507482, 148.153455, 147.750146, 147.298378, 146.798978, 146.252768, 145.660576, 145.023225, 144.341541, 143.616349, 142.848473, 142.038739, 141.187972, 140.296996, 139.366637, 138.397719, 137.391068, 136.347508, 135.267865, 134.152963, 133.003628, 131.832171, 130.633433, 129.402114, 128.138668, 126.843547, 125.517202, 124.160086, 122.772652, 121.355351, 119.908636, 118.432958, 116.928771, 115.396526, 113.836676, 112.249673, 110.635968, 108.996015, 107.313556, 105.606116, 103.874507, 102.119545, 100.342045, 98.5428196, 96.7226847, 94.8824544, 93.0229429, 91.1449649, 89.2493346, 87.3368667, 85.4083754, 83.4646752, 81.5065807, 79.5349061, 77.5845622, 75.6896409, 73.7806358, 71.8561511, 69.9147909, 67.9551593, 65.9758603, 63.9754982, 61.952677, 59.9060008, 57.8340739, 55.7355002, 53.608884, 51.4528292, 49.2659402, 47.0468209, 44.7940755, 42.3148424, 39.7074857, 37.0726711, 34.4152159, 31.7399378, 29.0516543, 26.355183, 23.6553414, 20.9569471, 18.2648176, 15.5837704, 12.9186232, 10.2741934, 7.65529868, 5.06675649, 2.51338441, 0, -1.05418016, -2.22612794, -3.51371992, -4.91483263, -6.42734264, -8.0491265, -9.77806077, -11.612022, -13.5488868, -15.5865316, -17.7228331, -19.9556678, -22.2829122, -24.7024429, -27.2121365, -29.8098695, -32.3756736, -34.8905905, -37.6190504, -40.5336775, -43.6070959, -46.8119296, -50.1208027, -53.5063393, -56.9411636, -60.3978996, -63.8491714, -67.2676031, -70.6258187, -73.7613784, -76.8314938, -79.8493081, -82.8279647, -85.9766664, -89.2105268, -92.3923768, -95.476542, -98.4173477, -101.169119, -103.686183, -106.568731, -108.928654, -110.59391, -111.392459, -111.152261, -110.932386, -109.021906, -104.941005, -98.2098635, -88.3486654, -74.0077076, -56.6402512, -39.1188619, -21.8516967, -0.215412429, 28.1542932, 65.6217226, 117.17287, 181.954507, 256.253375, 335.874539, 428.249151, 538.107199, 669.089895, 822.993573, 1002.4614, 1208.25616, 1442.53354, 1714.34385, 2028.83739, 2398.44356, 2819.91374, 3295.87855, 3828.96862, 4421.81458, 5255.28808, 6109.21848, 6941.67616, 7710.73148, 8374.45483, 8726.16771, 8998.52195, 9220.1946, 9499.48851, 9833.56217, 10186.3967, 10556.8093, 10943.6176, 11345.6387, 11761.6901, 12136.5486, 12601.4344, 13103.8228, 13591.1893, 14011.0092, 14310.7578, 14437.9105, 14587.95, 14213.5886}, 
{141.68755, 142.08308, 142.421239, 142.702748, 142.928326, 143.098696, 143.214577, 143.276691, 143.285757, 143.242497, 143.147632, 143.001882, 142.805967, 142.560609, 142.266528, 141.924445, 141.535081, 141.099156, 140.61739, 140.090505, 139.519222, 138.904261, 138.246342, 137.546187, 136.804515, 136.022049, 135.199508, 134.337613, 133.437085, 132.498644, 131.523012, 130.510909, 129.463055, 128.380171, 127.268074, 126.124828, 124.948439, 123.739463, 122.498454, 121.225967, 119.922559, 118.588785, 117.2252, 115.832358, 114.410817, 112.96113, 111.483854, 109.979544, 108.448755, 106.892042, 105.309961, 103.690292, 102.046644, 100.379847, 98.6907337, 96.9801369, 95.2488886, 93.497821, 91.7277665, 89.9395573, 88.1340256, 86.3120038, 84.4743242, 82.6218189, 80.7553203, 78.8756606, 76.9836721, 75.117715, 73.3152481, 71.4997057, 69.669487, 67.8229916, 65.9586191, 64.0747688, 62.1698404, 60.2422332, 58.2903469, 56.3125808, 54.3073345, 52.2730075, 50.2079992, 48.1107092, 45.979537, 43.8128821, 41.4050737, 38.8608388, 36.287357, 33.6896505, 31.0727411, 28.4416508, 25.8014015, 23.1570153, 20.513514, 17.8759198, 15.2492544, 12.63854, 10.0487984, 7.48505164, 4.95232168, 2.45563047, 0, -0.902925863, -1.94062149, -3.10938633, -4.40551984, -5.82532148, -7.36509072, -9.021127, -10.7897298, -12.6671985, -14.6498327, -16.7339317, -18.9157951, -21.1917223, -23.5580127, -26.0109658, -28.5468811, -31.0740127, -33.5760823, -36.2485393, -39.0688162, -42.0143457, -45.0625605, -48.190893, -51.376776, -54.5976421, -57.8309239, -61.054054, -64.244465, -67.3795896, -70.3681989, -73.2815628, -76.1177125, -78.8746789, -81.6576509, -84.4110808, -87.0508452, -89.5428278, -91.8529123, -93.9469825, -95.7909222, -98.1544905, -99.9550378, -101.001168, -101.101484, -100.064591, -98.1105283, -94.5336041, -89.0395626, -81.3341482, -71.1231055, -58.2586746, -42.121054, -24.0050533, -3.70616743, 22.0412093, 55.1283111, 97.446372, 153.901281, 222.684468, 298.698649, 374.556406, 461.695838, 569.01417, 701.646014, 858.350444, 1040.81301, 1248.82011, 1483.56492, 1753.20419, 2061.95567, 2418.66634, 2822.21255, 3275.20964, 3780.27293, 4340.01777, 5121.35234, 5921.52591, 6702.08058, 7424.55848, 8050.50172, 8388.58038, 8655.12332, 8877.18923, 9160.20703, 9500.21387, 9860.59269, 10239.0134, 10633.1458, 11040.6599, 11459.2255, 11683.379, 12086.4459, 12595.1407, 13136.178, 13636.2724, 14022.1384, 14220.4905, 14419.2486, 14083.4789}, 
{135.976907, 136.32744, 136.628954, 136.881862, 137.086577, 137.243511, 137.353077, 137.41569, 137.43176, 137.401702, 137.325929, 137.204853, 137.038887, 136.828445, 136.573939, 136.275782, 135.934388, 135.550168, 135.123537, 134.654907, 134.144691, 133.593302, 133.001153, 132.368657, 131.696227, 130.984276, 130.233217, 129.443463, 128.615426, 127.74952, 126.846158, 125.905753, 124.928717, 123.915464, 122.854201, 121.7517, 120.614881, 119.444553, 118.241524, 117.006605, 115.740603, 114.444328, 113.11859, 111.764196, 110.381957, 108.972681, 107.537177, 106.076255, 104.590723, 103.081391, 101.549067, 100.003039, 98.4354537, 96.8469365, 95.2381128, 93.609608, 91.9620472, 90.2960558, 88.6122591, 86.9112823, 85.1937508, 83.4602898, 81.7115247, 79.9480806, 78.170583, 76.379657, 74.575928, 72.7936607, 71.0663123, 69.325129, 67.5685556, 65.7950366, 64.0030164, 62.1909398, 60.3572513, 58.5003955, 56.6188168, 54.7109601, 52.7752697, 50.8101902, 48.8141664, 46.7856426, 44.7230635, 42.6248737, 40.2856225, 37.8099907, 35.3051099, 32.7760415, 30.2278468, 27.6655871, 25.094324, 22.5191187, 19.9450326, 17.3771271, 14.8204636, 12.2801033, 9.76110781, 7.26853833, 4.8074563, 2.38292306, 0, -0.959574658, -2.02947038, -3.20742643, -4.49118208, -5.87847659, -7.36704923, -8.95463928, -10.638986, -12.4178286, -14.2889065, -16.2499588, -18.2987248, -20.4329439, -22.6503552, -24.9486981, -27.3257117, -29.6660136, -31.9511826, -34.434829, -37.0904516, -39.8915494, -42.8116212, -45.8241658, -48.9026821, -52.0206689, -55.1516252, -58.2690498, -61.3464415, -64.3572992, -67.2012951, -69.9528246, -72.6075343, -75.1610712, -77.7366695, -80.2661822, -82.6474393, -84.8378114, -86.794669, -88.4753829, -89.8373236, -91.6861858, -92.8728304, -93.1886519, -92.4250443, -90.3734019, -86.2092542, -80.4938263, -73.1724785, -64.1905712, -53.4934648, -42.5083583, -27.8876374, -8.14909921, 17.7986406, 50.2490452, 89.853828, 137.264703, 194.450282, 260.446085, 332.851011, 405.359521, 489.016641, 592.244746, 720.304773, 873.100788, 1052.99575, 1262.58159, 1504.28062, 1779.6756, 2090.82419, 2436.49809, 2822.54643, 3252.16431, 3728.54684, 4254.8891, 4991.01968, 5744.34182, 6478.89226, 7158.70772, 7747.82492, 8063.14311, 8313.92817, 8527.27573, 8816.01558, 9173.30694, 9556.58649, 9960.79957, 10380.8916, 10811.8078, 11248.4936, 11441.5637, 11820.8458, 12312.899, 12844.2821, 13341.5544, 13731.2746, 13940.0019, 14111.8755, 14373.7146}, 
{130.320679, 130.732602, 131.093539, 131.403994, 131.664471, 131.875474, 132.037509, 132.151078, 132.216688, 132.234842, 132.206045, 132.1308, 132.009613, 131.842989, 131.63143, 131.375442, 131.075529, 130.732196, 130.345947, 129.917286, 129.446718, 128.934747, 128.381878, 127.788614, 127.155462, 126.482924, 125.771505, 125.02171, 124.234043, 123.409009, 122.547112, 121.648856, 120.714745, 119.745285, 118.731873, 117.679758, 116.594299, 115.476296, 114.326549, 113.145857, 111.935022, 110.694841, 109.426117, 108.129648, 106.806235, 105.456677, 104.081774, 102.682327, 101.259135, 99.8129987, 98.3447172, 96.8673639, 95.3691996, 93.8507583, 92.3125736, 90.7551797, 89.1791102, 87.5848992, 85.9730805, 84.344188, 82.6987556, 81.0373172, 79.3604066, 77.6685577, 75.9623045, 74.2421808, 72.5087205, 70.7939373, 69.1290892, 67.4497853, 65.7545186, 64.0417823, 62.3100697, 60.5578739, 58.7836881, 56.9860054, 55.1633191, 53.3141222, 51.4369081, 49.5301698, 47.5924005, 45.6220934, 43.6177418, 41.5778386, 39.296444, 36.8785676, 34.4316764, 31.9608977, 29.4713591, 26.9681879, 24.4565116, 21.9414576, 19.4281533, 16.9217262, 14.4273036, 11.950013, 9.49498184, 7.06733754, 4.67220751, 2.31471919, 0, -0.898710535, -1.90693952, -3.02203662, -4.24135149, -5.56223376, -6.98203312, -8.49809919, -10.1077816, -11.8084301, -13.5973943, -15.4720238, -17.4296684, -19.4676775, -21.583401, -23.7741884, -26.0373895, -28.2833007, -30.4968357, -32.8722497, -35.3882383, -38.0234968, -40.7567207, -43.5666054, -46.4318464, -49.331139, -52.2431787, -55.1466609, -58.020281, -60.8427345, -63.56985, -66.2115742, -68.7534625, -71.1810704, -73.5831297, -75.8936076, -78.0189577, -79.9137823, -81.5326841, -82.8302654, -83.7611287, -85.2856605, -86.0465711, -85.8016793, -84.3088038, -81.3257631, -74.7190637, -66.6106647, -57.2312126, -46.8113544, -35.5817366, -26.5298565, -13.7600264, 6.22939017, 35.2561338, 71.699991, 115.484319, 166.532477, 225.088882, 290.682864, 362.493499, 435.116366, 518.790712, 621.508612, 748.28357, 899.082062, 1076.18723, 1282.40103, 1520.14111, 1789.92282, 2093.33757, 2428.51952, 2801.0492, 3213.71476, 3669.30432, 4170.60605, 4865.51338, 5575.43283, 6266.87621, 6906.35534, 7460.38203, 7755.22788, 7991.13841, 8194.72839, 8480.02796, 8839.04999, 9227.38425, 9638.86475, 10067.3255, 10506.6005, 10950.5237, 11105.774, 11472.9088, 11969.7715, 12514.2055, 13024.0542, 13417.161, 13611.3693, 13764.7342, 14443.3569}, 
{124.827823, 125.383696, 125.878335, 126.312661, 126.687597, 127.004064, 127.262984, 127.465278, 127.611869, 127.703678, 127.741627, 127.726637, 127.65963, 127.541529, 127.373254, 127.155728, 126.889871, 126.576607, 126.216857, 125.811542, 125.361584, 124.867905, 124.331427, 123.753071, 123.13376, 122.474414, 121.775956, 121.039307, 120.265389, 119.455124, 118.609434, 117.72924, 116.815464, 115.869027, 114.902405, 113.9105, 112.888074, 111.835676, 110.753852, 109.643148, 108.504111, 107.337289, 106.143228, 104.922475, 103.675577, 102.40308, 101.105532, 99.7834791, 98.4374683, 97.0680466, 95.6757606, 94.2594216, 92.8213496, 91.3621292, 89.8823448, 88.3825809, 86.8634219, 85.3254522, 83.7692564, 82.1954189, 80.6045242, 78.9971567, 77.3739009, 75.7353412, 74.0820621, 72.4146481, 70.7336835, 69.0720447, 67.4618324, 65.8370319, 64.1961343, 62.5376307, 60.8600122, 59.16177, 57.4413951, 55.6973788, 53.928212, 52.1323859, 50.3083917, 48.4547205, 46.5698633, 44.6523114, 42.7005557, 40.7130875, 38.4800384, 36.1084619, 33.7079155, 31.2836522, 28.8409247, 26.3849858, 23.9210882, 21.4544848, 18.9904283, 16.5341715, 14.0909673, 11.6660683, 9.26472735, 6.89219728, 4.55373084, 2.25458082, 0, -0.707706466, -1.55002823, -2.52200358, -3.6186708, -4.83506818, -6.166234, -7.60720655, -9.15302411, -10.798725, -12.5393474, -14.3699297, -16.2855102, -18.2811271, -20.3518188, -22.4926234, -24.6985794, -26.9463553, -29.2283651, -31.5811976, -33.9959548, -36.4637385, -38.9756507, -41.5227933, -44.0962682, -46.6871774, -49.2866228, -51.8857063, -54.4755298, -57.0471953, -59.6638684, -62.218164, -64.6795648, -67.0175537, -69.2396167, -71.2962352, -73.1277559, -74.6922606, -75.947831, -76.8525491, -77.3644966, -78.6743291, -79.1324233, -78.4557054, -76.3611013, -72.5655371, -63.7014192, -53.3413231, -41.9733048, -30.08542, -18.1657249, -10.3667005, 0.96676457, 20.1451703, 49.385145, 86.6283292, 131.735746, 184.568417, 245.306842, 313.419959, 388.028191, 463.941037, 550.748871, 656.605528, 786.101425, 937.788974, 1112.99213, 1311.97855, 1535.79835, 1789.3747, 2075.43992, 2400.26248, 2762.49459, 3163.64456, 3605.22072, 4088.73137, 4745.44479, 5414.66933, 6065.47331, 6666.92505, 7188.09287, 7466.72775, 7690.76025, 7885.53753, 8158.63197, 8502.84327, 8876.71063, 9274.74679, 9691.46446, 10121.3764, 10558.9953, 10659.4186, 11020.5375, 11542.6093, 12125.891, 12670.6397, 13077.1127, 13245.567, 13409.1244, 14233.8833}, 
{120.365703, 120.961423, 121.493393, 121.962686, 122.370374, 122.717532, 123.005231, 123.234545, 123.406547, 123.522311, 123.582908, 123.589413, 123.542898, 123.444437, 123.295102, 123.095966, 122.848103, 122.552586, 122.210487, 121.82288, 121.390837, 120.915433, 120.397739, 119.838829, 119.239777, 118.601654, 117.925534, 117.212491, 116.463597, 115.679925, 114.862549, 114.012541, 113.130974, 112.218922, 111.299221, 110.361605, 109.395545, 108.401408, 107.379562, 106.330372, 105.254207, 104.151432, 103.022414, 101.867521, 100.687119, 99.4815743, 98.2512544, 96.9965258, 95.7177555, 94.4153101, 93.0895566, 91.728487, 90.3451112, 88.940064, 87.5139805, 86.0674959, 84.6012451, 83.1158632, 81.6119852, 80.0902463, 78.5512815, 76.9957257, 75.4242142, 73.8373818, 72.2358638, 70.6202951, 68.9913108, 67.3833918, 65.8302065, 64.2625854, 62.6789692, 61.0777991, 59.457516, 57.8165608, 56.1533745, 54.4663981, 52.7540725, 51.0148386, 49.2471375, 47.4494101, 45.6200973, 43.7576402, 41.8604797, 39.9270566, 37.7410495, 35.4127978, 33.0552222, 30.6737333, 28.2737414, 25.8606569, 23.4398904, 21.0168521, 18.5969525, 16.185602, 13.7882111, 11.4101902, 9.05694971, 6.73390001, 4.44645154, 2.20001473, 0, -0.55620183, -1.26305054, -2.11418287, -3.1032356, -4.22384546, -5.46964921, -6.83428362, -8.31138542, -9.89459139, -11.5775383, -13.3538628, -15.2172018, -17.1611919, -19.17947, -21.2656727, -23.4134369, -25.6165628, -27.8687104, -30.1631468, -32.4935438, -34.8535731, -37.2369065, -39.6372158, -42.0481728, -44.4634493, -46.8767171, -49.2816479, -51.6719135, -54.0411857, -56.4759401, -58.8430166, -61.1082459, -63.2374587, -65.2440464, -67.0700592, -68.6448647, -69.9200253, -70.8471036, -71.377662, -71.463263, -72.3035652, -72.2221796, -70.926476, -68.1238241, -63.5215939, -53.599115, -42.0988073, -29.5350504, -16.422224, -3.27470792, 5.82549011, 18.2920525, 37.9781966, 66.9689809, 103.739474, 148.385556, 201.003108, 262.03531, 331.151811, 407.643386, 487.305416, 578.258393, 687.946771, 820.274362, 973.14556, 1147.21859, 1341.13169, 1555.01938, 1796.4228, 2068.6935, 2382.78235, 2733.27242, 3120.88471, 3546.3402, 4010.3599, 4631.38251, 5262.98188, 5876.44959, 6443.07721, 6934.15631, 7197.90124, 7410.73226, 7596.68832, 7856.23674, 8183.16358, 8539.40974, 8920.3535, 9321.3732, 9737.84713, 10165.1536, 10245.4575, 10607.7982, 11143.713, 11744.7394, 12302.4149, 12708.2769, 12853.8629, 13056.9916, 13991.0943}, 
{117.105437, 117.588896, 118.018115, 118.393889, 118.717014, 118.988286, 119.208502, 119.378456, 119.498946, 119.570766, 119.594713, 119.571583, 119.502172, 119.387276, 119.227691, 119.024212, 118.777636, 118.488759, 118.158376, 117.787284, 117.376279, 116.926156, 116.437711, 115.911741, 115.349041, 114.750408, 114.116637, 113.448524, 112.746866, 112.012457, 111.246095, 110.448575, 109.620693, 108.763245, 107.89052, 106.996282, 106.074137, 105.124442, 104.147555, 103.143835, 102.11364, 101.057327, 99.9752556, 98.8677829, 97.7352673, 96.578067, 95.39654, 94.1910445, 92.9619386, 91.7095803, 90.4343279, 89.1234208, 87.79062, 86.4365683, 85.0619079, 83.6672815, 82.2533316, 80.8207006, 79.370031, 77.9019654, 76.4171463, 74.9162162, 73.3998175, 71.8685928, 70.3231846, 68.7642354, 67.1923876, 65.6438111, 64.1538224, 62.649792, 61.1300592, 59.5929634, 58.0368442, 56.4600408, 54.8608927, 53.2377394, 51.5889203, 49.9127747, 48.2076421, 46.471862, 44.7037737, 42.9017166, 41.0640302, 39.1890539, 37.0516718, 34.7666511, 32.4514447, 30.1116435, 27.7528386, 25.3806208, 23.0005812, 20.6183108, 18.2394006, 15.8694414, 13.5140243, 11.1787403, 8.86918025, 6.59093525, 4.34959622, 2.15075415, 0, -0.489813758, -1.12828895, -1.90927231, -2.82661057, -3.87415046, -5.04573871, -6.33522204, -7.7364472, -9.24326091, -10.8495099, -12.5490409, -14.3357006, -16.2033359, -18.1457933, -20.1569196, -22.2305617, -24.3088638, -26.3782866, -28.5496227, -30.8056396, -33.1291051, -35.5027868, -37.9094524, -40.3318695, -42.7528058, -45.1550289, -47.5213066, -49.8344064, -52.077096, -54.2487802, -56.3094892, -58.2369997, -60.0090883, -61.7545082, -63.3755477, -64.7342343, -65.7630517, -66.3944833, -66.5610126, -66.1951232, -66.2205322, -65.2768096, -63.102502, -59.4361558, -54.0163176, -44.4513515, -33.1425318, -20.3609506, -6.37769991, 8.53612833, 21.7129675, 38.2072246, 60.3062799, 89.4434492, 125.308333, 168.373424, 219.111216, 278.151764, 345.774189, 422.085727, 504.794949, 599.756823, 712.662845, 846.555169, 999.986788, 1173.57129, 1366.26819, 1578.26224, 1815.80316, 2081.71, 2384.55205, 2720.51344, 3090.42271, 3495.10844, 3935.39919, 4523.14627, 5120.3998, 5700.23266, 6235.71773, 6699.92788, 6949.69598, 7151.82826, 7329.21473, 7576.67361, 7888.62335, 8229.5123, 8595.25741, 8981.77562, 9384.98387, 9800.79912, 9903.938, 10271.8447, 10800.7093, 11386.7221, 11926.0732, 12314.9529, 12449.5513, 12723.2639, 13785.008}, 
{114.315772, 114.655332, 114.952446, 115.207551, 115.421082, 115.593477, 115.725172, 115.816602, 115.868205, 115.880416, 115.853672, 115.788409, 115.685064, 115.544072, 115.36587, 115.150895, 114.899582, 114.612368, 114.28969, 113.931983, 113.539684, 113.113229, 112.653054, 112.159597, 111.633292, 111.074577, 110.483888, 109.861661, 109.208332, 108.524338, 107.810114, 107.066098, 106.292726, 105.490434, 104.661007, 103.804179, 102.919667, 102.007864, 101.069161, 100.103951, 99.1126278, 98.0955826, 97.0532083, 95.9858974, 94.8940424, 93.7780357, 92.63827, 91.4751377, 90.2890314, 89.0803435, 87.8494665, 86.5856593, 85.3006894, 83.9951907, 82.6697971, 81.3251424, 79.9618606, 78.5805853, 77.1819507, 75.7665904, 74.3351384, 72.8882285, 71.4264946, 69.9505706, 68.4610904, 66.9586877, 65.4439966, 63.9548313, 62.5281137, 61.0877955, 59.6321003, 58.1592516, 56.6674732, 55.1549885, 53.6200213, 52.0607951, 50.4755334, 48.86246, 47.2197983, 45.5457721, 43.8386049, 42.0965203, 40.3177419, 38.5004934, 36.4104734, 34.1670589, 31.8924215, 29.5923307, 27.2725559, 24.9388667, 22.5970324, 20.2528226, 17.9120068, 15.5803544, 13.2636349, 10.9676178, 8.69807257, 6.4607687, 4.26147567, 2.10596294, 0, -0.438318446, -1.02160362, -1.74403966, -2.59981073, -3.58310096, -4.6880945, -5.90897551, -7.23992812, -8.67513648, -10.2087847, -11.8350571, -13.5481376, -15.3422104, -17.2114597, -19.1500697, -21.1522245, -23.1068854, -24.9932048, -27.0433707, -29.2290195, -31.5217876, -33.8933115, -36.3152276, -38.7591722, -41.1967817, -43.5996927, -45.9395415, -48.1879644, -50.316598, -52.2396845, -54.0072989, -55.608296, -57.0315303, -58.5144814, -59.9216915, -61.0514029, -61.8178829, -62.1353986, -61.9182172, -61.080606, -60.2242144, -58.3667241, -55.2879145, -50.767565, -44.5854549, -35.7187708, -24.9505331, -12.2611694, 2.36889264, 18.9592256, 36.4397628, 57.2514966, 82.040537, 111.541784, 146.671415, 188.264219, 237.154984, 294.159856, 360.136502, 435.96292, 521.172816, 619.452923, 734.623066, 868.916334, 1021.87717, 1194.28416, 1385.95637, 1597.42363, 1832.73392, 2093.94512, 2385.65702, 2706.9946, 3059.1359, 3443.25895, 3860.54178, 4418.76817, 4985.85898, 5536.34079, 6044.74021, 6485.58382, 6723.11156, 6915.66112, 7085.02482, 7320.54268, 7617.26937, 7942.11468, 8291.54052, 8662.00881, 9049.98147, 9451.92041, 9586.6895, 9958.577, 10472.6155, 11033.8378, 11547.2763, 11917.9638, 12050.933, 12411.4064, 13565.2269}, 
{111.077693, 111.372077, 111.627942, 111.845632, 112.025492, 112.167868, 112.273103, 112.341544, 112.373536, 112.369422, 112.329548, 112.25426, 112.143901, 111.998818, 111.819354, 111.605856, 111.358667, 111.078134, 110.7646, 110.418411, 110.039912, 109.629447, 109.187363, 108.714003, 108.209712, 107.674837, 107.109721, 106.51471, 105.890148, 105.236381, 104.553753, 103.84261, 103.103296, 102.336156, 101.540421, 100.717015, 99.8668783, 98.9903928, 98.0879394, 97.1598992, 96.2066534, 95.2285832, 94.2260698, 93.1994943, 92.149238, 91.0756819, 89.9792074, 88.8601955, 87.7190275, 86.5560846, 85.3717478, 84.1562079, 82.9202575, 81.6644987, 80.3895336, 79.0959642, 77.7843928, 76.4554213, 75.1096518, 73.7476866, 72.3701275, 70.9775768, 69.5706366, 68.1499089, 66.7159958, 65.2694994, 63.8110218, 62.3791264, 61.011465, 59.6303475, 58.2339152, 56.8203093, 55.3876709, 53.9341414, 52.4578619, 50.9569736, 49.4296179, 47.8739358, 46.2880687, 44.6701577, 43.0183441, 41.3307691, 39.6055739, 37.8408997, 35.7959593, 33.5933837, 31.3586756, 29.09773, 26.8164416, 24.5207055, 22.2164166, 19.9094697, 17.6057597, 15.3111817, 13.0316304, 10.7730008, 8.5411878, 6.34208632, 4.18159125, 2.06559751, 0, -0.330167432, -0.818760399, -1.45860488, -2.24252686, -3.16335232, -4.21390723, -5.38701758, -6.67550935, -8.07220853, -9.56994108, -11.161533, -12.8398103, -14.5975988, -16.4277247, -18.3230139, -20.2762923, -22.2215515, -24.1432123, -26.1671791, -28.2736703, -30.4429045, -32.6551003, -34.8904763, -37.1292511, -39.3516431, -41.5378711, -43.6681536, -45.7227092, -47.6817564, -49.5552633, -51.2827908, -52.8356328, -54.185083, -55.4793904, -56.6013713, -57.3866535, -57.7534443, -57.6199508, -56.9043802, -55.5249396, -54.0225438, -51.5031727, -47.7631993, -42.5989966, -35.8069378, -26.5920914, -15.4899611, -2.444746, 12.5993547, 29.6981418, 47.8424044, 69.4546361, 95.3266296, 126.227081, 162.877529, 206.020684, 256.399258, 314.660605, 381.664465, 458.374608, 544.644695, 643.449489, 757.708905, 889.319334, 1038.44286, 1206.03764, 1392.41884, 1598.37792, 1827.06397, 2080.29246, 2361.55415, 2670.73149, 3009.06007, 3377.77546, 3778.11322, 4313.78556, 4857.93227, 5386.16977, 5874.1145, 6297.38285, 6526.30427, 6711.9735, 6875.12995, 7099.6061, 7381.16702, 7689.28905, 8021.05172, 8373.53456, 8743.8171, 9128.97887, 9255.48416, 9615.30308, 10116.5474, 10667.3291, 11175.7598, 11549.9514, 11698.0157, 12147.7656, 13315.8052}, 
{107.896638, 108.176168, 108.418544, 108.624107, 108.793195, 108.926148, 109.023306, 109.085008, 109.111593, 109.103402, 109.060773, 108.984045, 108.87356, 108.729655, 108.55267, 108.342946, 108.10082, 107.826634, 107.520726, 107.183436, 106.815103, 106.416067, 105.986667, 105.527243, 105.038134, 104.519679, 103.972219, 103.396093, 102.791639, 102.159199, 101.49911, 100.811713, 100.097347, 99.3563514, 98.5883817, 97.7941339, 96.974312, 96.1292778, 95.2593929, 94.3650191, 93.446518, 92.5042513, 91.5385807, 90.5498679, 89.5384745, 88.5047622, 87.4490928, 86.3718279, 85.2733291, 84.1539583, 83.014077, 81.8449279, 80.6561893, 79.4484208, 78.2221816, 76.9780311, 75.7165287, 74.4382337, 73.1437056, 71.8335036, 70.5081872, 69.1683157, 67.8144485, 66.447145, 65.0669644, 63.6744663, 62.2702099, 60.89315, 59.5813195, 58.2560901, 56.9155323, 55.5577164, 54.1807127, 52.7825915, 51.3614233, 49.9152784, 48.4422271, 46.9403397, 45.4076868, 43.8423385, 42.2423652, 40.6058374, 38.9308252, 37.2153992, 35.2132242, 33.049714, 30.85322, 28.629744, 26.3852877, 24.1258529, 21.8574415, 19.5860551, 17.3176956, 15.0583648, 12.8140643, 10.5907961, 8.3945618, 6.23136328, 4.10720229, 2.0280806, 0, -0.215853063, -0.606944492, -1.16436451, -1.87920334, -2.74255121, -3.74549834, -4.87913495, -6.13455128, -7.50283754, -8.97508396, -10.5423808, -12.1958182, -13.9264864, -15.7254757, -17.5838763, -19.4927783, -21.4536302, -23.4590019, -25.4765546, -27.499598, -29.521442, -31.5353965, -33.5347712, -35.5128759, -37.4630205, -39.3785148, -41.2526686, -43.0787917, -44.850194, -46.7125389, -48.4509196, -50.0129398, -51.3462031, -52.4628943, -53.2783265, -53.6905914, -53.6279184, -53.0185369, -51.7906764, -49.8725662, -47.8204521, -44.7434117, -40.4468018, -34.7359789, -27.4162997, -17.574603, -5.91439285, 7.57934473, 22.9216236, 40.1274575, 57.7827604, 79.078324, 105.491117, 138.045467, 176.841558, 222.394498, 275.219393, 335.743199, 404.589207, 482.476864, 568.927822, 666.888242, 778.875655, 906.686517, 1050.89546, 1212.63796, 1392.61139, 1591.83765, 1812.94495, 2057.6529, 2329.09884, 2627.3665, 2953.68471, 3309.28229, 3695.3881, 4212.22567, 4737.28045, 5247.03258, 5717.96221, 6126.54949, 6348.26836, 6527.94266, 6685.34094, 6899.12817, 7165.75969, 7457.1507, 7770.98356, 8104.94063, 8456.70429, 8823.95691, 8935.79229, 9278.9234, 9764.88485, 10305.2113, 10811.4372, 11195.0974, 11367.7264, 11909.3196, 13051.9927}, 
{104.915139, 105.192371, 105.432627, 105.636279, 105.8037, 105.935263, 106.031342, 106.09231, 106.118541, 106.110407, 106.068282, 105.992539, 105.883551, 105.741692, 105.567335, 105.360852, 105.122618, 104.853006, 104.552388, 104.221139, 103.85963, 103.468237, 103.047331, 102.597286, 102.118476, 101.611273, 101.076051, 100.513183, 99.923043, 99.3060032, 98.6624372, 97.9927184, 97.2972199, 96.5763151, 95.831421, 95.0623671, 94.26897, 93.4515693, 92.6105043, 91.7461143, 90.8587388, 89.948717, 89.0163885, 88.0620925, 87.0861685, 86.0889558, 85.0707939, 84.032022, 82.9729795, 81.8940059, 80.7954405, 79.6696148, 78.5250497, 77.3622581, 76.1817531, 74.9840475, 73.7696544, 72.5390867, 71.2928574, 70.0314794, 68.7554658, 67.4653296, 66.1615836, 64.8447409, 63.5153144, 62.1738172, 60.8207621, 59.4953221, 58.2357418, 56.962801, 55.6745066, 54.3688654, 53.0438842, 51.6975697, 50.3279288, 48.9329683, 47.510695, 46.0591158, 44.5762373, 43.0600665, 41.5086101, 39.919875, 38.291868, 36.6225958, 34.6605886, 32.5338389, 30.3733379, 28.1851883, 25.9754929, 23.7503545, 21.5158761, 19.2781602, 17.0433098, 14.8174277, 12.6066167, 10.4169795, 8.25461901, 6.12563801, 4.0361393, 1.9922257, 0, -0.114775475, -0.419478007, -0.903602231, -1.55664278, -2.3680943, -3.32745141, -4.42420876, -5.64786098, -6.9879027, -8.43382856, -9.9751332, -11.6013112, -13.3018573, -15.0662661, -16.8840322, -18.7446503, -20.7083848, -22.7748405, -24.7734371, -26.7088343, -28.5856917, -30.4086689, -32.1824256, -33.9116212, -35.6009156, -37.2549682, -38.8784387, -40.4759867, -42.0522718, -43.8692047, -45.5798685, -47.1117476, -48.3923263, -49.3147694, -49.823721, -49.8729769, -49.4003172, -48.3435219, -46.6403713, -44.2286455, -41.6962508, -38.1329765, -33.3494041, -27.1561151, -19.363691, -8.83519289, 3.43439728, 17.397618, 33.0070076, 50.2151046, 67.0971669, 87.7774674, 114.502474, 148.59207, 189.474356, 237.426806, 292.726891, 355.66538, 426.503429, 505.48769, 591.430635, 687.703443, 796.843185, 920.714506, 1060.04267, 1216.07594, 1389.73627, 1582.18734, 1795.78921, 2032.22517, 2294.59946, 2582.95588, 2898.48587, 3242.3809, 3615.83243, 4115.95281, 4624.03237, 5117.28236, 5572.91399, 5968.13849, 6182.87996, 6356.49486, 6507.88893, 6711.12403, 6963.27323, 7238.42791, 7534.81024, 7850.64238, 8184.14651, 8533.54481, 8636.85587, 8963.51301, 9430.83261, 9956.13104, 10456.7247, 10849.93, 11053.0632, 11685.6242, 12781.0914}, 
{102.481655, 102.731473, 102.946102, 103.125894, 103.271201, 103.382377, 103.459772, 103.503739, 103.514632, 103.492801, 103.438599, 103.352379, 103.234493, 103.085292, 102.905131, 102.694359, 102.453331, 102.182398, 101.881913, 101.552228, 101.193694, 100.806666, 100.391494, 99.9485306, 99.4781289, 98.9806408, 98.4564187, 97.9058148, 97.3291814, 96.7268708, 96.0992353, 95.4466272, 94.7693987, 94.0679023, 93.3435175, 92.5960615, 91.8253387, 91.0316681, 90.2153688, 89.3767596, 88.5161594, 87.6338874, 86.7302623, 85.8056032, 84.860229, 83.8944586, 82.9086111, 81.9030053, 80.8779602, 79.8337948, 78.770828, 77.6818194, 76.5748112, 75.4502862, 74.3087272, 73.1506171, 71.9764387, 70.7866748, 69.5818082, 68.3623218, 67.1286983, 65.8814207, 64.6209717, 63.3478342, 62.0624909, 60.7654247, 59.4571185, 58.1771534, 56.9641721, 55.7380197, 54.4966445, 53.2379947, 51.9600185, 50.6606641, 49.3378799, 47.989614, 46.6138147, 45.2084301, 43.7714086, 42.3006984, 40.7942476, 39.2500045, 37.6659174, 36.0399344, 34.11496, 32.0218287, 29.8944408, 27.7390211, 25.5617944, 23.3689858, 21.1668201, 18.9615222, 16.759317, 14.5664294, 12.3890842, 10.2335065, 8.10592103, 6.01255275, 3.95962656, 1.95336734, 0, -0.116939789, -0.413941807, -0.880994376, -1.50808582, -2.28520445, -3.20233859, -4.24947656, -5.41660669, -6.6937173, -8.07079669, -9.5378332, -11.0848152, -12.7017309, -14.3785686, -16.1053168, -17.8719637, -19.6793726, -21.5190854, -23.3564922, -25.1839118, -26.9936629, -28.778064, -30.5294339, -32.2400912, -33.9023546, -35.5085428, -37.0509744, -38.5219681, -39.9138426, -41.3729586, -42.6811107, -43.7844047, -44.6289468, -45.2363102, -45.5148674, -45.352866, -44.673772, -43.4010511, -41.4581692, -38.7685921, -35.7671732, -31.7103511, -26.4215376, -19.7241444, -11.4415833, -0.735818369, 11.7429291, 26.0058854, 42.0642769, 59.92933, 78.0144905, 99.8816081, 127.351979, 161.584782, 202.38737, 250.174037, 305.359079, 368.785795, 440.341974, 519.447398, 604.046377, 698.375156, 805.81133, 928.485226, 1066.41374, 1220.58388, 1391.5709, 1580.25504, 1789.02628, 2019.42062, 2274.09109, 2553.28478, 2858.15101, 3189.8391, 3549.49837, 4030.44391, 4518.61783, 4992.128, 5429.0823, 5807.5886, 6012.19908, 6176.94778, 6320.03788, 6511.941, 6750.17074, 7010.46214, 7291.41771, 7591.63997, 7909.73141, 8244.29455, 8374.69743, 8697.53037, 9142.25976, 9638.35198, 10115.2734, 10502.4905, 10729.4695, 11451.4968, 12521.4244}, 
{100.344301, 100.565685, 100.753566, 100.908279, 101.030156, 101.119531, 101.176738, 101.202108, 101.195977, 101.158677, 101.090542, 100.991904, 100.863098, 100.704456, 100.516312, 100.299, 100.052852, 99.7782026, 99.4753844, 99.1447308, 98.7865753, 98.4012513, 97.989092, 97.5504308, 97.0856012, 96.5949363, 96.0787696, 95.5374344, 94.9712641, 94.380592, 93.7657514, 93.1270758, 92.4648984, 91.7795526, 91.0724284, 90.3433085, 89.5919631, 88.8186913, 88.0237922, 87.2075647, 86.370308, 85.5123211, 84.633903, 83.735353, 82.8169699, 81.8790528, 80.9219009, 79.9458132, 78.9510887, 77.9380266, 76.9069258, 75.8508379, 74.7774667, 73.6872683, 72.5806988, 71.4582144, 70.3202713, 69.1673256, 67.9998336, 66.8182514, 65.6230351, 64.414641, 63.1935253, 61.960144, 60.7149534, 59.4584096, 58.1909688, 56.9526425, 55.7824923, 54.5993948, 53.4012418, 52.1859253, 50.9513373, 49.6953697, 48.4159145, 47.1108637, 45.7781091, 44.4155429, 43.0210569, 41.5925431, 40.1278934, 38.6249999, 37.0817545, 35.4960492, 33.6050083, 31.5423924, 29.4450965, 27.319475, 25.1718822, 23.0086724, 20.8362002, 18.6608197, 16.4888855, 14.3267518, 12.1807731, 10.0573038, 7.96269813, 5.90331055, 3.88549541, 1.91560711, 0, -0.128875335, -0.425220215, -0.879794788, -1.4833592, -2.2266736, -3.10049814, -4.09559297, -5.20271822, -6.41263406, -7.71610062, -9.10387806, -10.5667265, -12.0954062, -13.6806771, -15.3132995, -16.9840336, -18.6129935, -20.1808473, -21.8381492, -23.5605209, -25.3235843, -27.102961, -28.8742729, -30.6131417, -32.2951891, -33.8960369, -35.3913068, -36.7566206, -37.9676001, -39.01171, -39.8483867, -40.4496989, -40.7877157, -41.0542119, -41.1114034, -40.7629177, -39.9149117, -38.4735425, -36.3449671, -33.4353424, -30.0230822, -25.5287912, -19.7857937, -12.6274139, -3.88697584, 6.92302245, 19.568224, 34.135098, 50.7101137, 69.3797405, 89.0461622, 112.427593, 140.690203, 174.742226, 214.965279, 261.977421, 316.39671, 379.70353, 451.457628, 530.278029, 613.302662, 705.650449, 811.671672, 933.784865, 1071.13533, 1224.37083, 1393.58697, 1579.28838, 1784.00424, 2009.11853, 2256.83214, 2527.58646, 2822.48268, 3142.622, 3489.10562, 3951.65093, 4420.58888, 4874.86663, 5293.43131, 5655.23009, 5848.91436, 6003.92419, 6137.90489, 6318.60552, 6543.83868, 6790.54043, 7057.56721, 7343.77544, 7648.02156, 7969.16202, 8144.54701, 8468.10616, 8885.88205, 9343.91727, 9788.25439, 10164.936, 10420.0047, 11221.5125, 12261.9341}, 
{98.4644962, 98.6585766, 98.8206317, 98.950981, 99.0499443, 99.1178411, 99.1549909, 99.1617135, 99.1383284, 99.0851553, 99.0025136, 98.8907232, 98.7501034, 98.580974, 98.3836546, 98.1584648, 97.9057241, 97.6257522, 97.3188687, 96.9853932, 96.6256453, 96.2399446, 95.8286108, 95.3919633, 94.9303219, 94.4440062, 93.9333356, 93.39863, 92.8402088, 92.2583916, 91.6534981, 91.0258479, 90.3757606, 89.7035558, 89.010758, 88.2970591, 87.5621362, 86.80627, 86.029741, 85.2328297, 84.4158165, 83.5789819, 82.7226066, 81.8469709, 80.9523554, 80.0390406, 79.107307, 78.1574351, 77.1897054, 76.2043985, 75.2017947, 74.1751922, 73.1320052, 72.0726658, 70.9976056, 69.9072567, 68.8020509, 67.68242, 66.5487959, 65.4016106, 64.2412958, 63.0682834, 61.8830054, 60.6858935, 59.4773797, 58.2578958, 57.0278738, 55.8277975, 54.6971896, 53.5538771, 52.3956956, 51.2204806, 50.0260674, 48.8102916, 47.5709886, 46.3059938, 45.0131427, 43.6902707, 42.3352133, 40.945806, 39.5198842, 38.0552833, 36.5498389, 35.0013863, 33.1410117, 31.1055361, 29.0349862, 26.935854, 24.8146316, 22.6778108, 20.5318838, 18.3833423, 16.2386785, 14.1043843, 11.9869517, 9.89287266, 7.82863914, 5.80074314, 3.81567663, 1.87993159, 0, -0.122301393, -0.402853393, -0.832991019, -1.40404929, -2.10736322, -2.93426783, -3.87609814, -4.92418917, -6.06987594, -7.30449345, -8.61937674, -10.0058608, -11.4552807, -12.9589714, -14.508268, -16.0945054, -17.5592195, -18.8723455, -20.3640498, -21.9935676, -23.7201342, -25.5029846, -27.3013542, -29.074478, -30.7815914, -32.3819294, -33.8347274, -35.0992204, -36.1346437, -36.7722643, -37.1462072, -37.2540981, -37.0935628, -37.017656, -36.8462889, -36.3045913, -35.28356, -33.6741921, -31.3674842, -28.2544334, -24.4990957, -19.6363038, -13.5036298, -5.93864611, 3.22107518, 14.1413419, 26.9803567, 41.8997024, 59.0609615, 78.6257169, 100.024438, 125.043404, 154.131214, 187.914596, 227.383947, 273.366386, 326.68903, 389.379188, 460.791015, 538.969369, 620.527432, 711.207244, 816.161648, 938.017553, 1075.12181, 1227.7859, 1395.62847, 1578.7814, 1779.91679, 2000.26985, 2241.59493, 2504.5282, 2790.12518, 3099.44134, 3433.5322, 3878.91009, 4329.80945, 4765.92156, 5166.93771, 5512.54916, 5694.83194, 5839.50278, 5963.80235, 6133.6905, 6347.12325, 6581.75697, 6836.56266, 7110.51133, 7402.57399, 7711.72164, 7949.02233, 8277.05504, 8663.03723, 9074.18634, 9477.71985, 9840.8552, 10130.8099, 10996.12, 12002.0021}, 
{96.7807852, 96.9550648, 97.0981577, 97.2103887, 97.2920826, 97.3435645, 97.3651591, 97.3571914, 97.3199862, 97.2538685, 97.1591632, 97.036195, 96.885289, 96.70677, 96.5009628, 96.2681924, 96.0087837, 95.7230616, 95.4113509, 95.0739765, 94.7112634, 94.3235364, 93.9111203, 93.4743402, 93.0135208, 92.5289871, 92.021064, 91.4900763, 90.9363489, 90.3602067, 89.7619747, 89.1419776, 88.5005405, 87.8379881, 87.1564535, 86.4553195, 85.7339471, 84.9926026, 84.2315522, 83.451062, 82.6513984, 81.8328274, 80.9956154, 80.1400285, 79.2663329, 78.374795, 77.4656807, 76.5392565, 75.5957884, 74.6355428, 73.6587857, 72.6594951, 71.6443619, 70.6137885, 69.5681775, 68.5079316, 67.4334531, 66.3451446, 65.2434087, 64.1286479, 63.0012647, 61.8616616, 60.7102413, 59.5474061, 58.3735587, 57.1891016, 55.9944374, 54.8305411, 53.7374138, 52.6317801, 51.5114123, 50.3740827, 49.2175639, 48.039628, 46.8380476, 45.610595, 44.3550425, 43.0691625, 41.7507274, 40.3975096, 39.0072815, 37.5778153, 36.1068836, 34.5922586, 32.7585262, 30.7457981, 28.6974695, 26.6201784, 24.5205625, 22.4052597, 20.2809076, 18.1541442, 16.0316073, 13.9199346, 11.8257639, 9.75573307, 7.71647992, 5.71464223, 3.75685781, 1.84976446, 0, 0.0553848848, -0.0735256236, -0.376032411, -0.841436364, -1.45903837, -2.21813931, -3.10804008, -4.11804155, -5.23744462, -6.45555017, -7.76165909, -9.14507226, -10.5950906, -12.1010149, -13.6521462, -15.2377852, -16.7168939, -18.0601533, -19.5516787, -21.1528413, -22.825012, -24.5295621, -26.2278626, -27.8812846, -29.4511992, -30.8989775, -32.1859906, -33.2736096, -34.1232055, -34.5868073, -34.7752203, -34.6826185, -34.3031756, -33.9422851, -33.4385109, -32.547425, -31.1698355, -29.2065501, -26.5583768, -23.1261235, -19.2368923, -14.2354552, -7.93921461, -0.165573139, 9.26806672, 20.5254647, 33.8127654, 49.3172759, 67.2263031, 87.7271539, 110.527315, 136.880361, 166.90443, 201.063482, 240.527526, 286.14957, 338.78262, 400.252036, 470.217485, 547.277886, 628.880496, 719.842585, 824.64639, 945.581453, 1081.22192, 1231.84735, 1397.11264, 1577.13539, 1774.32377, 1989.79029, 2224.90172, 2480.47718, 2757.54116, 3057.11815, 3380.23262, 3811.04644, 4246.66238, 4667.32061, 5053.26126, 5384.7245, 5556.02902, 5690.61738, 5805.55319, 5966.69454, 6171.79057, 6398.25939, 6644.86671, 6910.37821, 7193.55959, 7493.17655, 7842.83432, 8184.94559, 8522.36974, 8857.96616, 9194.59425, 9535.11339, 9882.38296, 10773.1851, 11742.8931}, 
{95.2597827, 95.4229555, 95.5551053, 95.6565823, 95.7277363, 95.7689173, 95.7804753, 95.7627602, 95.7161221, 95.6409109, 95.5374765, 95.4061689, 95.2473382, 95.0613342, 94.8485069, 94.6092064, 94.3437825, 94.0525852, 93.7359645, 93.3942704, 93.0278529, 92.6370618, 92.2222472, 91.7837591, 91.3219474, 90.8371621, 90.3297531, 89.8000704, 89.248464, 88.6752839, 88.0808799, 87.4656022, 86.8298006, 86.1738252, 85.5009061, 84.8098926, 84.0995994, 83.370283, 82.6221998, 81.8556063, 81.070759, 80.2679145, 79.4473292, 78.6092596, 77.7539622, 76.8816936, 75.9927102, 75.0872684, 74.1656249, 73.2280361, 72.2747585, 71.3008853, 70.3119481, 69.3083155, 68.2903559, 67.2584378, 66.2129294, 65.1541994, 64.082616, 62.9985478, 61.9023631, 60.7944304, 59.6751182, 58.5447947, 57.4038286, 56.2525882, 55.0914419, 53.9619052, 52.9045057, 51.8347492, 50.7503365, 49.6489688, 48.5283471, 47.3861724, 46.2201458, 45.0279683, 43.8073408, 42.5559645, 41.2715404, 39.9517695, 38.5943529, 37.1969915, 35.7573864, 34.2732387, 32.4620899, 30.4676136, 28.4368506, 26.3765936, 24.2936354, 22.1947687, 20.0867863, 17.9764809, 15.8706452, 13.776072, 11.6995541, 9.64788411, 7.62785486, 5.64625909, 3.70988952, 1.82553891, 0, 0.404232736, 0.562616075, 0.490527516, 0.203344559, -0.283555297, -0.954794553, -1.79499571, -2.78878127, -3.92077373, -5.17559559, -6.53786935, -7.99221752, -9.52326259, -11.1156271, -12.7539335, -14.4228042, -16.1021824, -17.776022, -19.4395301, -21.0763264, -22.6700308, -24.2042629, -25.6626425, -27.0287893, -28.2863231, -29.4188637, -30.4100308, -31.2434441, -31.9027233, -32.4545475, -32.769022, -32.8048491, -32.5207308, -31.9486468, -31.0106603, -29.6092936, -27.6812655, -25.1632945, -21.9920993, -18.1043986, -14.2701004, -9.3391547, -3.08526486, 4.71786573, 14.2965337, 26.1648029, 40.189261, 56.5242626, 75.3241626, 96.7433157, 120.444788, 147.674687, 178.711267, 214.013784, 254.411038, 300.565912, 353.14129, 412.946898, 480.4654, 556.019267, 639.273808, 732.526094, 838.039548, 957.200796, 1089.91078, 1236.75239, 1397.98037, 1574.09253, 1766.78993, 1977.09298, 2206.03579, 2454.62301, 2723.87034, 3014.7935, 3328.40817, 3747.51229, 4170.89376, 4579.12274, 4952.76937, 5272.40378, 5433.34185, 5558.24417, 5664.21843, 5818.61785, 6018.66125, 6240.63182, 6482.79518, 6743.41692, 7020.76266, 7313.098, 7816.77121, 8180.25043, 8454.47475, 8690.38328, 8938.91509, 9251.00929, 9677.60496, 10553.7433, 11483.561}, 
{93.9037119, 94.0590911, 94.1833976, 94.2770104, 94.3403087, 94.3736716, 94.377478, 94.3521071, 94.2979378, 94.2153494, 94.1047208, 93.9664312, 93.8008595, 93.6083848, 93.3893863, 93.144243, 92.8733339, 92.5770381, 92.2557347, 91.9098028, 91.5396214, 91.1455695, 90.7280263, 90.2873708, 89.8239821, 89.3382393, 88.8305213, 88.3012074, 87.7506765, 87.1793077, 86.5874801, 85.9755727, 85.3439646, 84.6930349, 84.0271904, 83.3447115, 82.6438304, 81.9247953, 81.1878547, 80.4332569, 79.6612503, 78.8720832, 78.066004, 77.2432611, 76.4041028, 75.5487775, 74.6775335, 73.7906192, 72.8882829, 71.9707731, 71.0383381, 70.087019, 69.1213627, 68.1417087, 67.1483965, 66.1417656, 65.1221557, 64.0899063, 63.045357, 61.9888473, 60.9207168, 59.8413051, 58.7509517, 57.6499961, 56.538778, 55.4176369, 54.2869124, 53.1888093, 52.1645272, 51.1280617, 50.0770385, 49.0090831, 47.9218213, 46.8128784, 45.6798803, 44.5204525, 43.3322205, 42.11281, 40.8598467, 39.5709561, 38.2437637, 36.8758954, 35.4649765, 34.0086328, 32.2168927, 30.2372296, 28.2205736, 26.1738833, 24.1041176, 22.018235, 19.9231942, 17.825954, 15.733473, 13.65271, 11.5906235, 9.55417237, 7.55031518, 5.58601067, 3.6682175, 1.80389439, 0, 0.730763287, 1.15851236, 1.30296743, 1.18384871, 0.820876387, 0.233770687, -0.55774819, -1.53396004, -2.67514464, -3.9615818, -5.37355131, -6.89133296, -8.49520653, -10.1654518, -11.8823487, -13.6261768, -15.4995079, -17.5000919, -19.3315962, -21.0005061, -22.5133066, -23.876483, -25.0965205, -26.1799042, -27.1331192, -27.9626509, -28.6749844, -29.2766049, -29.7739975, -30.4597175, -30.9492876, -31.1633721, -31.0226351, -30.2754975, -28.9287451, -27.0350952, -24.5668847, -21.496451, -17.7961309, -13.4382617, -9.62213638, -4.71971471, 1.5367228, 9.41489566, 19.1825234, 31.7616097, 46.6020189, 63.8078992, 83.4833991, 105.732667, 130.062646, 157.904607, 189.851922, 226.416996, 267.946932, 314.863049, 367.586668, 425.786807, 490.808065, 564.815732, 649.767717, 745.371963, 851.607669, 968.939099, 1098.65243, 1241.65657, 1398.83686, 1571.09611, 1759.42348, 1964.75929, 2187.78293, 2429.73578, 2691.64847, 2974.55166, 3279.47597, 3688.55569, 4101.4419, 4498.88932, 4861.65267, 5170.48668, 5320.95799, 5435.80146, 5532.28099, 5679.67968, 5874.6705, 6092.25175, 6330.18015, 6586.21239, 6858.10518, 7143.61522, 7780.52944, 8159.98193, 8376.30666, 8523.83763, 8696.90884, 8989.85427, 9497.00792, 10342.8514, 11220.0284}, 
{92.6925216, 92.8421508, 92.9605937, 93.0482572, 93.1055479, 93.1328726, 93.1306381, 93.0992511, 93.0391185, 92.9506469, 92.8342431, 92.6903139, 92.519266, 92.3215062, 92.0974412, 91.8474779, 91.572023, 91.2714832, 90.9462653, 90.596776, 90.2234221, 89.8266104, 89.4067477, 88.9642406, 88.499496, 88.0129206, 87.5049212, 86.9759045, 86.4262773, 85.8564463, 85.2668183, 84.6578001, 84.0297985, 83.3832201, 82.7235835, 82.048632, 81.3560476, 80.6460712, 79.9189436, 79.1749057, 78.4141984, 77.6370626, 76.8437391, 76.0344688, 75.2094927, 74.3690514, 73.513386, 72.6427374, 71.7573463, 70.8574537, 69.9433004, 69.0114843, 68.0659682, 67.1070721, 66.1351157, 65.150419, 64.1533016, 63.1440836, 62.1230848, 61.0906249, 60.0470238, 58.9926015, 57.9276777, 56.8525723, 55.7676051, 54.673096, 53.5693649, 52.4995114, 51.5056081, 50.4997446, 49.4794675, 48.4423235, 47.3858591, 46.3076209, 45.2051556, 44.0760097, 42.9177299, 41.7278627, 40.5039548, 39.2435527, 37.9442031, 36.6034526, 35.2188478, 33.7879352, 32.0127382, 30.044781, 28.0391397, 26.0029513, 23.9433526, 21.8674805, 19.7824719, 17.6954638, 15.6135931, 13.5439966, 11.4938112, 9.47017395, 7.48022162, 5.53109116, 3.62991946, 1.78384344, 0, 0.962556337, 1.58410981, 1.88729384, 1.89474182, 1.62908717, 1.1129633, 0.369003619, -0.580158469, -1.71188955, -3.00355622, -4.43252506, -5.97616266, -7.61183562, -9.31691053, -11.068754, -12.8447325, -14.8382564, -17.0575554, -18.9913264, -20.6632308, -22.0969304, -23.3160868, -24.3443616, -25.2054165, -25.9229131, -26.520513, -27.0218779, -27.4506694, -27.8305492, -28.6333643, -29.2702563, -29.6304311, -29.6030946, -28.7131852, -27.0320426, -24.7281448, -21.7999779, -18.2460282, -14.0647821, -9.2547256, -5.32280459, -0.299949484, 6.11048684, 14.2051515, 24.2806916, 37.6347845, 53.3127896, 71.3610962, 91.826094, 114.754173, 139.419096, 167.584199, 200.252035, 238.033086, 280.737356, 328.534246, 381.593159, 438.534648, 501.428975, 574.036057, 660.206351, 757.531627, 864.149762, 979.775318, 1106.79446, 1246.36703, 1399.87178, 1568.5253, 1752.74162, 1953.38876, 2170.73799, 2406.34587, 2661.28695, 2936.63582, 3233.46705, 3633.92694, 4037.75039, 4425.74403, 4778.71451, 5077.46845, 5217.22398, 5321.43526, 5407.5897, 5546.84317, 5735.43934, 5947.26003, 6179.66702, 6430.02206, 6695.68693, 6974.02338, 7694.53124, 8078.36407, 8250.88475, 8337.45618, 8463.44126, 8754.20287, 9335.10392, 10143.0506, 10950.1515}, 
{91.5947241, 91.739401, 91.8529126, 91.9356844, 91.9881421, 92.0107111, 92.0038169, 91.9678852, 91.9033413, 91.810611, 91.6901196, 91.5422928, 91.367556, 91.1663349, 90.9390549, 90.6861415, 90.4080204, 90.105117, 89.7778569, 89.4266656, 89.0519686, 88.6541915, 88.2337599, 87.7910991, 87.3266349, 86.8407927, 86.333998, 85.8066765, 85.2592535, 84.6921547, 84.1058056, 83.5006317, 82.8770586, 82.2355117, 81.5824321, 80.9151109, 80.2307673, 79.5296315, 78.811934, 78.077905, 77.3277748, 76.5617738, 75.7801323, 74.9830806, 74.1708491, 73.3436681, 72.5017678, 71.6453787, 70.7747311, 69.8900552, 68.9915814, 68.0753565, 67.145885, 66.203488, 65.2484865, 64.2812015, 63.301954, 62.311065, 61.3088556, 60.2956467, 59.2717594, 58.2375146, 57.1932335, 56.139237, 55.0758462, 54.003382, 52.9221654, 51.8766507, 50.9102322, 49.9322098, 48.9400436, 47.9311935, 46.9031197, 45.8532822, 44.779141, 43.6781562, 42.5477878, 41.3854959, 40.1887405, 38.9549817, 37.6816795, 36.366294, 35.0062852, 33.5991132, 31.8378602, 29.8785774, 27.8809743, 25.8523885, 23.8001577, 21.7316197, 19.654112, 17.5749725, 15.5015387, 13.4411484, 11.4011393, 9.38884908, 7.4116154, 5.47677595, 3.59166845, 1.76363056, 0, 0.951050035, 1.57286065, 1.88727715, 1.91614483, 1.68130899, 1.20461492, 0.507907928, -0.386966691, -1.45816364, -2.68383762, -4.04214333, -5.51123547, -7.06926875, -8.69439787, -10.3647775, -12.0585624, -13.9635181, -16.0877437, -17.9352742, -19.5291807, -20.8925347, -22.0484074, -23.0198702, -23.8299944, -24.5018513, -25.0585122, -25.5230485, -25.9185313, -26.2680322, -27.0684938, -27.6953631, -28.02955, -27.9519643, -26.9553083, -25.1145957, -22.6083622, -19.4339799, -15.588821, -11.0702577, -5.87566209, -1.49203787, 4.02624045, 10.9732504, 19.6430694, 30.329775, 44.5832464, 61.1278089, 79.9435897, 101.010716, 124.309314, 148.76113, 176.69825, 209.65063, 248.397787, 292.187106, 340.953866, 394.633348, 451.171566, 512.945172, 584.50093, 670.365831, 767.456116, 873.452198, 987.842599, 1113.17957, 1250.60907, 1401.54112, 1567.19013, 1747.80224, 1944.17129, 2155.97528, 2385.29556, 2633.31223, 2901.2054, 3190.15514, 3583.41812, 3979.57874, 4359.29796, 4703.23674, 4992.05604, 5119.77967, 5211.46383, 5284.8997, 5412.58078, 5590.64522, 5792.43854, 6015.23853, 6256.32297, 6512.96965, 6782.45637, 7485.649, 7850.81447, 8008.85929, 8090.68995, 8227.21296, 8549.33481, 9187.962, 9966.03689, 10671.4956}, 
{90.5861288, 90.7253899, 90.8338087, 90.9118144, 90.9598362, 90.9783034, 90.9676452, 90.9282907, 90.8606692, 90.7652098, 90.6423419, 90.4924946, 90.316097, 90.1135785, 89.8853683, 89.6318955, 89.3535894, 89.0508791, 88.7241939, 88.373963, 88.0006157, 87.604581, 87.1862882, 86.7461666, 86.2846454, 85.8021537, 85.2991208, 84.7759759, 84.2331481, 83.6710668, 83.090161, 82.4908601, 81.8735933, 81.2387897, 80.5934863, 79.9346695, 79.2592462, 78.5674311, 77.8594391, 77.1354849, 76.3957833, 75.6405491, 74.869997, 74.0843418, 73.2837983, 72.4685813, 71.6389056, 70.7949859, 69.9370369, 69.0652736, 68.1799106, 67.2749337, 66.3569217, 65.4262244, 64.4831917, 63.5281734, 62.5615193, 61.5835791, 60.5947028, 59.59524, 58.5855407, 57.5659546, 56.5368316, 55.4985214, 54.4513739, 53.3957388, 52.331966, 51.3064516, 50.3644859, 49.411452, 48.4447147, 47.4616387, 46.4595888, 45.43593, 44.3880271, 43.3132448, 42.2089479, 41.0725014, 39.90127, 38.6926185, 37.4439117, 36.1525146, 34.8157918, 33.4311083, 31.6814941, 29.7280931, 27.7358271, 25.7122616, 23.6649622, 21.6014945, 19.5294241, 17.4563166, 15.3897376, 13.3372526, 11.3064272, 9.30482705, 7.3400177, 5.41956474, 3.55103374, 1.7419903, 0, 0.596264668, 0.945307251, 1.06296342, 0.965068862, 0.66745924, 0.185970233, -0.463562484, -1.26530324, -2.20341635, -3.26206614, -4.42541695, -5.67763309, -7.00287889, -8.38531868, -9.80911677, -11.2584375, -12.7741683, -14.3485767, -15.8375249, -17.2373322, -18.5443177, -19.7548009, -20.865101, -21.8715372, -22.7704288, -23.5580951, -24.2308553, -24.7850287, -25.2169346, -25.8185788, -26.1821756, -26.2153383, -25.8256804, -24.7571964, -22.999309, -20.5850725, -17.4711858, -13.6143478, -8.97125731, -3.49861333, 1.80773735, 8.34650615, 16.3643057, 26.1077487, 37.8234477, 53.0996381, 70.5059041, 89.9534528, 111.353491, 134.617226, 158.202524, 185.250237, 217.9058, 257.189906, 301.826899, 351.572137, 406.180978, 463.539444, 525.697077, 596.742698, 680.06567, 774.043353, 877.947461, 991.813755, 1116.97262, 1254.16019, 1404.13093, 1567.62575, 1745.31825, 1937.9201, 2144.26316, 2367.22812, 2608.17736, 2868.47324, 3149.47815, 3536.79466, 3926.4849, 4298.8512, 4634.1959, 4912.82131, 5026.7158, 5103.37166, 5160.94007, 5272.23024, 5433.7579, 5619.26455, 5826.40463, 6052.83254, 6296.2027, 6554.16952, 7102.16155, 7417.51114, 7601.53359, 7755.5442, 7980.85824, 8378.79101, 9050.6578, 9817.66513, 10381.8117}, 
{89.7128469, 89.8448223, 89.94645, 90.0181545, 90.0603603, 90.0734917, 90.0579732, 90.0142293, 89.9426845, 89.843763, 89.7178895, 89.5654883, 89.3869839, 89.1828008, 88.9533633, 88.699096, 88.4204232, 88.1177694, 87.7915591, 87.4422167, 87.0701667, 86.6758334, 86.2596415, 85.8220151, 85.363379, 84.8841574, 84.3847748, 83.8656557, 83.3272245, 82.7699056, 82.1941236, 81.6003028, 80.9888677, 80.3602428, 79.7214778, 79.0695453, 78.4013379, 77.717065, 77.0169362, 76.3011608, 75.5699482, 74.823508, 74.0620495, 73.2857822, 72.4949156, 71.6896589, 70.8702218, 70.0368137, 69.1896439, 68.3289219, 67.4548571, 66.5604849, 65.6533444, 64.7338004, 63.802218, 62.8589621, 61.9043977, 60.9388897, 59.9628031, 58.9765029, 57.9803539, 56.9747212, 55.9599698, 54.9364645, 53.9045703, 52.8646522, 51.8170752, 50.8096791, 49.8891641, 48.9579814, 48.0134186, 47.0527629, 46.0733017, 45.0723223, 44.0471122, 42.9949586, 41.9131489, 40.7989705, 39.6497107, 38.4626568, 37.2350962, 35.9643163, 34.6476044, 33.2822479, 31.5425261, 29.5940245, 27.6062106, 25.5868541, 23.5437246, 21.4845916, 19.4172248, 17.3493936, 15.2888677, 13.2434166, 11.22081, 9.22881742, 7.27520842, 5.3677526, 3.51421953, 1.72237881, 0, 0.266282478, 0.361057238, 0.294649958, 0.0773863184, -0.280408003, -0.768407328, -1.37628598, -2.09371827, -2.91037853, -3.81594107, -4.80008023, -5.85247031, -6.96278564, -8.12070055, -9.31588935, -10.5380264, -11.6872713, -12.7405108, -13.8898925, -15.1059091, -16.3590532, -17.6198174, -18.8586943, -20.0461767, -21.152757, -22.148928, -23.0051822, -23.6920124, -24.179911, -24.5562163, -24.6317321, -24.3418973, -23.6221508, -22.4689221, -20.7871549, -18.4655289, -15.421186, -11.5712681, -6.83291696, -1.12327453, 5.06813688, 12.5707586, 21.5794363, 32.2890156, 44.8943421, 60.7258008, 78.558813, 98.3043397, 119.873341, 143.176779, 166.57359, 193.423676, 226.02281, 265.466649, 310.400612, 360.570225, 415.721013, 474.138463, 537.359966, 608.515684, 689.543441, 780.496825, 882.292985, 995.561037, 1120.44204, 1257.30107, 1406.29003, 1567.71875, 1742.67869, 1931.81916, 2133.42266, 2350.86945, 2585.62815, 2839.16735, 3112.95565, 3493.56502, 3876.08487, 4240.70794, 4567.62702, 4837.03485, 4940.46744, 5005.87882, 5051.45753, 5148.73454, 5294.78366, 5464.45265, 5655.92663, 5867.3907, 6097.02995, 6343.02949, 6698.58699, 6957.18462, 7179.19156, 7424.97696, 7754.91002, 8229.3599, 8908.69576, 9647.51447, 10080.6392}, 
{89.0085334, 89.1255284, 89.2131763, 89.2718777, 89.3020328, 89.3040422, 89.2783061, 89.225225, 89.1451993, 89.0386294, 88.9059156, 88.7474585, 88.5636584, 88.3549156, 88.1216307, 87.8642039, 87.5830358, 87.2785266, 86.9510769, 86.601087, 86.2289572, 85.8350881, 85.41988, 84.9837333, 84.5270484, 84.0502258, 83.5536657, 83.0377687, 82.502935, 81.9495652, 81.3780596, 80.7888187, 80.1822427, 79.5587322, 78.9237268, 78.2750011, 77.6102696, 76.929769, 76.2337362, 75.5224081, 74.7960215, 74.0548134, 73.2990205, 72.5288798, 71.7446282, 70.9465024, 70.1347394, 69.3095761, 68.4712493, 67.6199958, 66.7560526, 65.8758651, 64.9835438, 64.0794077, 63.163776, 62.2369675, 61.2993015, 60.351097, 59.392673, 58.4243485, 57.4464427, 56.4592746, 55.4631633, 54.4584278, 53.4453872, 52.4243605, 51.3956668, 50.4067273, 49.5038323, 48.5901562, 47.6629646, 46.7195232, 45.7570975, 44.7729533, 43.7643561, 42.7285715, 41.6628652, 40.5645027, 39.4307498, 38.2588719, 37.0461348, 35.789804, 34.4871453, 33.135424, 31.4045199, 29.4622776, 27.480477, 25.4670079, 23.4297603, 21.3766242, 19.3154894, 17.2542458, 15.2007835, 13.1629923, 11.1487622, 9.16598309, 7.2225449, 5.32633757, 3.48525102, 1.70717519, 0, 0.210694432, 0.266424265, 0.176484462, -0.0498300118, -0.403224194, -0.87440312, -1.45407183, -2.13293535, -2.90169873, -3.751067, -4.67174519, -5.65443835, -6.6898515, -7.76868969, -8.88165795, -10.0194613, -11.0427831, -11.9237539, -12.9471754, -14.0758908, -15.2727434, -16.5005763, -17.722233, -18.9005565, -19.9983902, -20.9785773, -21.8039611, -22.4373847, -22.8416915, -23.0406385, -22.9138201, -22.4058053, -21.4611632, -20.1507642, -18.3560267, -15.9246888, -12.7634292, -8.77892673, -3.87786002, 2.0330922, 8.70072113, 16.6703438, 26.103081, 37.1600532, 50.0023812, 64.9829388, 82.0231552, 101.236213, 122.735294, 146.63358, 172.346031, 201.537438, 235.204479, 273.972266, 317.707292, 366.616652, 420.907442, 480.238642, 545.490038, 618.139365, 698.882894, 788.758753, 889.244205, 1001.21879, 1124.54926, 1259.56725, 1406.36999, 1565.22832, 1737.27264, 1923.14712, 2121.61458, 2335.49002, 2566.06884, 2814.64649, 3082.51838, 3452.12969, 3823.33866, 4177.15325, 4494.58147, 4756.63128, 4859.81008, 4925.96019, 4972.30412, 5065.09471, 5202.14666, 5360.84538, 5540.02704, 5738.52779, 5955.18381, 6188.83127, 6333.15413, 6532.76025, 6798.44239, 7140.99333, 7571.20584, 8099.87268, 8737.78664, 9395.21545, 9761.14333}, 
{88.4407418, 88.5405767, 88.612147, 88.6558255, 88.6719844, 88.6609966, 88.6232343, 88.5590703, 88.4688769, 88.3530268, 88.2118926, 88.0458466, 87.8552615, 87.6405098, 87.4019641, 87.1399968, 86.8549805, 86.5472878, 86.2172912, 85.8653632, 85.4918763, 85.0972031, 84.6817162, 84.245788, 83.7897911, 83.314098, 82.8190813, 82.3051136, 81.7725672, 81.2218149, 80.653229, 80.0671822, 79.464047, 78.8441959, 78.2109354, 77.5631094, 76.8995265, 76.2204639, 75.526199, 74.8170091, 74.0931715, 73.3549636, 72.6026627, 71.8365461, 71.0568911, 70.2639751, 69.4580753, 68.6394691, 67.8084338, 66.9652468, 66.1101854, 65.2449368, 64.368338, 63.4806355, 62.5820762, 61.6729068, 60.753374, 59.8237247, 58.8842056, 57.9350633, 56.9765448, 56.0088967, 55.0323658, 54.0471989, 53.0536426, 52.0519439, 51.0423493, 50.0710206, 49.1830171, 48.2838893, 47.3709078, 46.4413426, 45.4924641, 44.5215426, 43.5258483, 42.5026516, 41.4492227, 40.3628318, 39.2407494, 38.0802456, 36.8785907, 35.6330551, 34.3409089, 32.9994225, 31.2762809, 29.3404794, 27.3649747, 25.3577326, 23.3267192, 21.2799004, 19.2252422, 17.1707104, 15.1242711, 13.0938903, 11.0875338, 9.11316759, 7.1787577, 5.29227004, 3.46167056, 1.69492523, 0, 0.328635235, 0.481704786, 0.470278843, 0.305427601, -0.00177874853, -0.440270012, -0.998975997, -1.66682651, -2.43275136, -3.28568035, -4.2145433, -5.20827, -6.25579026, -7.3460339, -8.46793072, -9.61041052, -10.6563596, -11.5795585, -12.6087977, -13.7102833, -14.8502215, -15.9948184, -17.1102803, -18.1628133, -19.1186237, -19.9439175, -20.6049011, -21.0677805, -21.298762, -21.3409207, -21.0554086, -20.3853711, -19.2739538, -17.7686864, -15.7605222, -13.1125789, -9.7366868, -5.54467605, -0.448376938, 5.6403803, 12.550454, 20.7082462, 30.2526615, 41.3226045, 54.0569797, 67.5367846, 83.2233077, 101.51993, 122.830034, 147.557001, 176.675621, 209.319476, 244.564581, 282.328218, 324.24526, 371.179417, 423.994398, 484.071942, 551.639997, 626.361394, 707.975737, 797.979445, 897.717846, 1008.09129, 1129.24614, 1261.67485, 1405.74238, 1561.90815, 1731.099, 1913.97736, 2110.23086, 2321.6467, 2549.22475, 2793.96488, 3056.86693, 3414.03867, 3772.59509, 4114.7591, 4422.75358, 4678.80142, 4786.71945, 4859.40734, 4912.69059, 5005.87936, 5137.58646, 5288.30611, 5457.55706, 5644.85805, 5849.72784, 6071.68518, 6020.79174, 6168.24533, 6476.07125, 6906.29482, 7420.94135, 7982.03615, 8551.60453, 9103.5276, 9430.07774}, 
{87.976596, 88.0627374, 88.1213411, 88.152764, 88.1573628, 88.1354945, 88.0875159, 88.0137837, 87.914655, 87.7904864, 87.6416349, 87.4684572, 87.2713103, 87.0505509, 86.8065359, 86.5396221, 86.2501664, 85.9385256, 85.6050565, 85.2501161, 84.874061, 84.4772482, 84.0600346, 83.6227768, 83.1658319, 82.6895566, 82.1943077, 81.6804421, 81.1483167, 80.5982882, 80.0307136, 79.4459496, 78.8443531, 78.2262809, 77.5935127, 76.9456638, 76.2823329, 75.6038305, 74.9104674, 74.2025542, 73.4804015, 72.7443201, 71.9946207, 71.2316138, 70.4556101, 69.6669205, 68.8658554, 68.0527255, 67.2278416, 66.3915143, 65.5440543, 64.6916613, 63.8286293, 62.9551412, 62.0713801, 61.1775289, 60.2737707, 59.3602885, 58.4372653, 57.5048841, 56.5633279, 55.6127797, 54.6534225, 53.6854394, 52.7090132, 51.7243271, 50.731564, 49.7758173, 48.9011021, 48.014977, 47.1147139, 46.1975844, 45.26086, 44.3018125, 43.3177136, 42.3058348, 41.2634478, 40.1878242, 39.0762358, 37.9259542, 36.734251, 35.4983978, 34.2156664, 32.8833284, 31.166917, 29.2365508, 27.2663355, 25.2643084, 23.2385065, 21.1969672, 19.1477276, 17.0988249, 15.0582963, 13.034179, 11.0345103, 9.06732727, 7.14066717, 5.2625672, 3.44106454, 1.68419641, 0, 0.521206106, 0.830734193, 0.942591459, 0.870785104, 0.629322327, 0.232210329, -0.306543691, -0.972932535, -1.752949, -2.63258589, -3.59783601, -4.63469214, -5.72914711, -6.8671937, -8.03482471, -9.21803295, -10.3445736, -11.3921197, -12.4783921, -13.5769042, -14.6611693, -15.7047006, -16.6810114, -17.5636151, -18.3260249, -18.9417542, -19.3843163, -19.6272244, -19.6439919, -19.5223593, -19.0797294, -18.2553472, -16.9884579, -15.285244, -13.051482, -10.1750982, -6.57525648, -2.17112081, 3.11814496, 9.37337695, 16.4658881, 24.7498056, 34.3469592, 45.3791787, 57.968294, 70.0599981, 84.4962917, 101.943039, 123.066103, 148.531348, 180.719452, 216.485582, 253.200671, 290.148293, 330.517001, 375.742089, 427.258856, 487.884034, 557.35752, 633.912192, 716.705921, 807.31785, 906.651332, 1015.50651, 1134.50861, 1264.36317, 1405.80244, 1559.53888, 1726.18688, 1906.41632, 2100.69128, 2309.91906, 2534.84073, 2776.19738, 3034.73011, 3380.92398, 3728.34009, 4060.28354, 4360.05942, 4610.97282, 4723.35864, 4802.13895, 4861.89177, 4955.69612, 5082.89006, 5226.76954, 5387.47473, 5565.14576, 5759.9228, 5971.94599, 5778.28546, 5889.95735, 6230.83088, 6724.77522, 7295.65958, 7867.35315, 8363.72512, 8815.59539, 9094.57904}, 
{87.5465505, 87.6368163, 87.6986998, 87.7326, 87.7389162, 87.7180472, 87.6703924, 87.5963507, 87.4963212, 87.370703, 87.2198953, 87.0442971, 86.8443074, 86.6203255, 86.3727503, 86.101981, 85.8084167, 85.4924564, 85.1544993, 84.7949443, 84.4141908, 84.0126376, 83.5906839, 83.1487288, 82.6871714, 82.2064108, 81.706846, 81.1888762, 80.6529005, 80.0993179, 79.5285275, 78.9409284, 78.3369198, 77.7169006, 77.0846631, 76.4388384, 75.778017, 75.1024878, 74.4125398, 73.708462, 72.9905434, 72.2590729, 71.5143396, 70.7566324, 69.9862402, 69.2034522, 68.4085572, 67.6018442, 66.7836022, 65.9541202, 65.1136872, 64.2667356, 63.409321, 62.5416428, 61.6639, 60.7762917, 59.8790172, 58.9722756, 58.056266, 57.1311876, 56.1972396, 55.2546211, 54.3035313, 53.3441693, 52.3767343, 51.4014254, 50.4184419, 49.4741395, 48.6137658, 47.7423253, 46.8570251, 45.9550722, 45.0336735, 44.0900361, 43.121367, 42.1248732, 41.0977617, 40.0372395, 38.9405137, 37.8047911, 36.6272789, 35.4051841, 34.1357136, 32.8160745, 31.1053253, 29.1768597, 27.2081358, 25.2073341, 23.1826353, 21.1422198, 19.0942684, 17.0469615, 15.0084798, 12.9870037, 10.990714, 9.02779112, 7.10641569, 5.23476829, 3.42102949, 1.67337987, 0, 0.672794714, 1.10858825, 1.32346605, 1.33351355, 1.15481621, 0.803459467, 0.295528764, -0.352890455, -1.12571275, -2.00685266, -2.98022476, -4.0297436, -5.13932374, -6.29287972, -7.47432612, -8.66757747, -9.81423923, -10.8921818, -11.9810189, -13.0555988, -14.09077, -15.0613806, -15.9422791, -16.7083137, -17.3343327, -17.7951846, -18.0657174, -18.1207797, -17.9352197, -17.5796277, -16.8980046, -15.8364762, -14.3411682, -12.4936796, -10.1723994, -7.21959056, -3.54073665, 0.958678596, 6.37317147, 12.7972583, 20.2856871, 28.9848458, 38.9970315, 50.4245411, 63.3696717, 75.6863188, 90.2872814, 107.836957, 128.999743, 154.440036, 186.448253, 222.075417, 258.941141, 296.407946, 337.396796, 383.230986, 435.233812, 495.896618, 565.109186, 641.487067, 724.578586, 815.505813, 914.76381, 1022.98539, 1141.03677, 1269.67705, 1409.78698, 1562.1572, 1727.13216, 1905.30869, 2096.49511, 2302.19809, 2523.28743, 2760.63293, 3015.10437, 3358.74096, 3703.45072, 4032.31112, 4328.39958, 4574.79356, 4680.63982, 4752.23359, 4804.33688, 4889.92716, 5008.63949, 5144.18584, 5296.99743, 5467.50548, 5656.14122, 5863.33586, 5687.91024, 5809.91656, 6152.58065, 6639.12836, 7192.78552, 7736.77796, 8194.33153, 8619.8157, 8781.65296}, 
{87.2064897, 87.3047229, 87.373389, 87.4129405, 87.4238296, 87.4065085, 87.3614297, 87.2890454, 87.1898079, 87.0641695, 86.9125825, 86.7354992, 86.5333719, 86.306653, 86.0557946, 85.7812492, 85.483469, 85.1629064, 84.8200135, 84.4552428, 84.0690465, 83.661877, 83.2341865, 82.7864273, 82.3190518, 81.8325122, 81.3272608, 80.80375, 80.262432, 79.7037592, 79.1281839, 78.5361582, 77.9281347, 77.3045655, 76.6719267, 76.027532, 75.3686228, 74.6954559, 74.0082881, 73.3073762, 72.592977, 71.8653474, 71.1247441, 70.3714241, 69.605644, 68.8276608, 68.0377313, 67.2361122, 66.4230604, 65.5988327, 64.7636859, 63.9188092, 63.0635069, 62.1980156, 61.3225718, 60.4374122, 59.5427734, 58.638892, 57.7260046, 56.8043478, 55.8741582, 54.9356725, 53.9891271, 53.0347588, 52.0728042, 51.1034998, 50.1270823, 49.1917222, 48.3444393, 47.48661, 46.6153634, 45.7278288, 44.8211355, 43.8924127, 42.9387896, 41.9573956, 40.9453597, 39.8998113, 38.8178797, 37.696694, 36.5333835, 35.3250775, 34.0689051, 32.7619957, 31.0561867, 29.1285159, 27.1601715, 25.159488, 23.1348, 21.0944422, 19.0467491, 17.0000555, 14.962696, 12.9430051, 10.9493175, 8.98996781, 7.07329067, 5.20762069, 3.4012925, 1.66264073, 0, 0.831649043, 1.39995517, 1.72314703, 1.81945327, 1.70710252, 1.40432344, 0.929344659, 0.300394835, -0.464297394, -1.34650338, -2.32799449, -3.39054206, -4.51591746, -5.68589205, -6.88223717, -8.08672419, -9.25633613, -10.3693031, -11.4634655, -12.5152826, -13.5012142, -14.3977198, -15.1812589, -15.8282912, -16.3152762, -16.6186735, -16.7149426, -16.5805432, -16.1919347, -15.6033698, -14.6849909, -13.3899198, -11.6712779, -9.69613953, -7.31065004, -4.3039009, -0.564828056, 4.01763256, 9.55454502, 16.1569734, 24.0651172, 33.221603, 43.7122291, 55.622794, 69.0390961, 82.0485937, 97.2350104, 115.18373, 136.480134, 161.709609, 192.64419, 227.232252, 263.889425, 302.256807, 344.477489, 391.571222, 444.557752, 505.106009, 573.439018, 649.071609, 732.23115, 823.409581, 922.668554, 1030.44729, 1147.82478, 1275.58635, 1414.73327, 1566.10686, 1729.7566, 1906.17985, 2094.20098, 2296.24778, 2513.39675, 2746.72438, 2997.30718, 3340.20349, 3684.0125, 4011.31525, 4304.69279, 4546.72614, 4643.73551, 4705.40333, 4746.99387, 4822.77258, 4932.56576, 5059.94915, 5205.56158, 5370.04183, 5554.0287, 5758.161, 5661.672, 5811.40943, 6143.91146, 6595.71622, 7103.36187, 7603.38656, 8032.32845, 8442.61716, 8457.74511}, 
{86.969027, 87.0748677, 87.1500037, 87.1949366, 87.2101679, 87.1961991, 87.1535316, 87.082667, 86.9841067, 86.8583523, 86.7059051, 86.5272667, 86.3229387, 86.0934224, 85.8392193, 85.560831, 85.2587589, 84.9335046, 84.5855695, 84.215455, 83.8236628, 83.4106942, 82.9770508, 82.5232341, 82.0497454, 81.5570864, 81.0457586, 80.5162633, 79.9691021, 79.4047765, 78.823788, 78.226638, 77.613828, 76.9858596, 76.3516972, 75.7074329, 75.0490568, 74.3767958, 73.6908767, 72.9915265, 72.2789719, 71.5534398, 70.815157, 70.0643504, 69.3012468, 68.5260731, 67.7390561, 66.9404227, 66.1303997, 65.3092139, 64.4770923, 63.6317365, 62.7759532, 61.910024, 61.0342306, 60.1488544, 59.2541771, 58.3504803, 57.4380456, 56.5171545, 55.5880886, 54.6511295, 53.7065588, 52.7546582, 51.795709, 50.8299931, 49.8577919, 48.9292162, 48.0931806, 47.2471975, 46.3883183, 45.5135943, 44.620077, 43.7048176, 42.7648676, 41.7972783, 40.799101, 39.7673872, 38.6991881, 37.5915551, 36.4415397, 35.2461931, 34.0025668, 32.707712, 31.0063356, 29.0790724, 27.1107919, 25.1099798, 23.0851217, 21.0447032, 18.99721, 16.9511275, 14.9149415, 12.8971376, 10.9062014, 8.95061845, 7.03887445, 5.17945498, 3.38084568, 1.65153214, 0, 0.989488414, 1.68961245, 2.1207169, 2.30314658, 2.25724629, 2.00336085, 1.56183505, 0.953013696, 0.1972416, -0.685136434, -1.6737756, -2.7483311, -3.88845811, -5.07381185, -6.28404749, -7.49882024, -8.69481604, -9.85126593, -10.9545412, -11.9836609, -12.9176438, -13.7355089, -14.4162751, -14.9389613, -15.2825866, -15.4261698, -15.3487298, -15.0292855, -14.446856, -13.6511216, -12.5245306, -11.0249033, -9.11006031, -7.01920832, -4.56947467, -1.50295031, 2.30696018, 6.98685223, 12.6633213, 19.4629627, 27.784721, 37.3999537, 48.3819706, 60.8040815, 74.7395961, 88.6773723, 104.671285, 123.190756, 144.705209, 169.684067, 199.189422, 232.373652, 268.67415, 308.094456, 351.793997, 400.413231, 454.592615, 515.001139, 582.244245, 656.896246, 739.910418, 831.159072, 930.325403, 1037.67412, 1154.4553, 1281.46671, 1419.81731, 1570.38556, 1732.90888, 1907.77012, 2092.76193, 2291.25587, 2504.53143, 2733.86807, 2980.54527, 3322.5935, 3665.3535, 3990.91696, 4281.37562, 4518.82121, 4606.6979, 4658.17664, 4689.05523, 4755.04105, 4856.13436, 4975.70645, 5114.58903, 5273.61381, 5453.61251, 5655.41684, 5670.31618, 5856.63121, 6172.0182, 6574.13341, 7020.63311, 7469.17356, 7877.41102, 8266.30437, 8110.19314}, 
{86.8415902, 86.9414277, 87.0107719, 87.0501181, 87.0599618, 87.0407983, 86.9931232, 86.9174319, 86.8142196, 86.683982, 86.5272144, 86.3444121, 86.1360708, 85.9026856, 85.6447522, 85.3627659, 85.0572221, 84.7286162, 84.3774437, 84.0042, 83.6093805, 83.1934807, 82.7569959, 82.3004215, 81.8242531, 81.328986, 80.8151156, 80.2831374, 79.7335467, 79.1668391, 78.5835099, 77.9840545, 77.3689684, 76.7387469, 76.1016864, 75.4542178, 74.7926782, 74.1173099, 73.4283551, 72.7260562, 72.0106554, 71.282395, 70.5415172, 69.7882643, 69.0228786, 68.2456024, 67.4566779, 66.6563474, 65.8448531, 65.0224374, 64.1893425, 63.3435784, 62.487668, 61.621902, 60.746571, 59.8619658, 58.968377, 58.0660951, 57.155411, 56.2366153, 55.3099985, 54.3758515, 53.4344648, 52.4861291, 51.531135, 50.5697733, 49.6023346, 48.6796523, 47.8513465, 47.013467, 46.163028, 45.2970437, 44.4125283, 43.5064958, 42.5759606, 41.6179367, 40.6294384, 39.6074798, 38.5490752, 37.4512385, 36.3109842, 35.1253263, 33.8912789, 32.6058564, 30.9089917, 28.9837502, 27.0174894, 25.0188112, 22.996318, 20.9586118, 18.9142947, 16.871969, 14.8402367, 12.8277, 10.842961, 8.89462194, 6.99128491, 5.14155205, 3.35402552, 1.63730745, 0, 1.15690964, 1.99638663, 2.54105976, 2.8135578, 2.83650953, 2.63254373, 2.22428917, 1.63437464, 0.885428906, 0.000080751784, -0.999041045, -2.08930771, -3.24809046, -4.45276052, -5.68068911, -6.90924746, -8.16776826, -9.44104582, -10.5889194, -11.5998949, -12.4624781, -13.1651748, -13.6964908, -14.044932, -14.1990043, -14.1472133, -13.878065, -13.3800652, -12.6417197, -11.7902222, -10.6245385, -9.09156791, -7.13820993, -4.95797399, -2.37445454, 0.854516619, 4.85602304, 9.75714827, 15.6849759, 22.7665893, 31.4363747, 41.4205862, 52.7861831, 65.6001244, 79.9293695, 94.3940049, 110.86958, 129.784773, 151.568261, 176.648721, 205.836544, 238.712154, 275.070782, 315.044522, 359.453229, 408.807965, 463.619794, 524.318685, 591.515228, 665.908479, 748.415957, 838.955789, 937.204266, 1043.4484, 1159.01003, 1284.73599, 1421.91933, 1571.52256, 1732.87213, 1906.21995, 2089.07958, 2284.86255, 2494.76865, 2719.99766, 2961.74939, 3295.46112, 3629.53576, 3946.61374, 4229.33546, 4460.34134, 4546.06972, 4596.16447, 4625.92404, 4687.94875, 4782.77172, 4895.38361, 5027.09467, 5179.21512, 5353.05522, 5549.92519, 5641.34827, 5852.08707, 6154.67998, 6521.66535, 6925.58155, 7338.96694, 7734.35991, 8042.84947, 7674.33986}, 
{86.8441654, 86.9330495, 86.9920873, 87.0217493, 87.0225063, 86.9948288, 86.9391875, 86.8560531, 86.7458961, 86.6091873, 86.4463973, 86.2579966, 86.0444561, 85.8062462, 85.5438376, 85.2577011, 84.9483072, 84.6161265, 84.2616297, 83.8852875, 83.4875705, 83.0689493, 82.6298946, 82.170877, 81.6923672, 81.1948358, 80.6787534, 80.1445907, 79.5928184, 79.023907, 78.4383272, 77.8365497, 77.2190451, 76.586284, 75.9448487, 75.2920254, 74.625027, 73.9441259, 73.2495943, 72.5417045, 71.8207289, 71.0869398, 70.3406094, 69.5820102, 68.8114143, 68.0290941, 67.2353219, 66.4303701, 65.6145109, 64.7880166, 63.9511596, 63.1032933, 62.2456288, 61.3784583, 60.5020741, 59.6167683, 58.7228333, 57.8205612, 56.9102442, 55.9921747, 55.0666448, 54.1339467, 53.1943727, 52.248215, 51.2957659, 50.3373175, 49.3731622, 48.4545141, 47.631365, 46.7989841, 45.9543626, 45.0944917, 44.2163626, 43.3169663, 42.3932941, 41.4423371, 40.4610865, 39.4465335, 38.3956691, 37.3054846, 36.1729711, 34.9951198, 33.7689218, 32.4913683, 30.7987795, 28.8760684, 26.9124826, 24.9167179, 22.8974698, 20.8634341, 18.8233062, 16.7857819, 14.7595567, 12.7533263, 10.7757864, 8.83563242, 6.94156013, 5.10226512, 3.32644301, 1.62278943, 0, 1.29879373, 2.25664732, 2.8981415, 3.24785705, 3.33037472, 3.17027527, 2.79213945, 2.22054803, 1.48008175, 0.595321382, -0.409152323, -1.50875861, -2.67891671, -3.89504588, -5.13256536, -6.36689439, -7.67904511, -9.05952147, -10.2449016, -11.2332319, -12.0225585, -12.6109278, -12.996386, -13.1769795, -13.1507546, -12.9157575, -12.4700345, -11.811632, -10.9385962, -10.0642705, -8.89246177, -7.35662732, -5.3902243, -3.1151159, -0.370556286, 3.05444196, 7.28294346, 12.4380128, 18.6427147, 26.0201137, 34.9845074, 45.2790913, 56.9699497, 70.1231669, 84.8048271, 99.7373157, 116.666341, 135.993911, 158.122035, 183.452723, 212.651917, 245.535106, 282.077475, 322.477202, 367.387744, 417.258146, 472.537454, 533.617073, 601.016789, 675.31927, 757.208701, 846.827881, 943.961515, 1048.97287, 1163.20558, 1287.55324, 1423.47923, 1572.02487, 1732.14247, 1903.96602, 2085.02344, 2278.45569, 2485.29882, 2706.58891, 2943.36203, 3267.08065, 3590.74782, 3897.79299, 4171.64564, 4395.7352, 4480.79655, 4531.41682, 4562.18029, 4621.72495, 4711.15551, 4817.38764, 4942.23551, 5087.51329, 5255.03514, 5446.61522, 5606.98839, 5839.14732, 6129.34663, 6463.84094, 6828.88488, 7210.73306, 7595.64013, 7797.68094, 7211.44455}, 
{86.9930403, 87.0682282, 87.1143916, 87.1319685, 87.1213968, 87.0831142, 87.0175587, 86.925168, 86.8063799, 86.6616325, 86.4913634, 86.2960105, 86.0760117, 85.8318048, 85.5638277, 85.2725181, 84.9583141, 84.6216532, 84.2629736, 83.8827129, 83.481309, 83.0591997, 82.616823, 82.1546166, 81.6730184, 81.1724663, 80.653398, 80.1162514, 79.5614644, 78.9894748, 78.4007204, 77.7956392, 77.1746688, 76.5382473, 75.8908141, 75.2307221, 74.5562759, 73.8677836, 73.1655531, 72.4498925, 71.7211096, 70.9795124, 70.225409, 69.4591074, 68.6809154, 67.8911411, 67.0900924, 66.2780774, 65.455404, 64.6223802, 63.779314, 62.9271797, 62.0656045, 61.1948819, 60.3153055, 59.4271687, 58.5307652, 57.6263884, 56.7143319, 55.7948892, 54.8683539, 53.9350194, 52.9951793, 52.0491272, 51.0971566, 50.139561, 49.1766339, 48.2598508, 47.4394583, 46.6101911, 45.7690248, 44.9129349, 44.0388972, 43.1438872, 42.2248805, 41.2788527, 40.3027796, 39.2936366, 38.2483995, 37.1640437, 36.037545, 34.865879, 33.6460213, 32.3749474, 30.6863497, 28.7664867, 26.8059831, 24.8136039, 22.7981139, 20.768278, 18.7328609, 16.7006277, 14.6803431, 12.6807719, 10.7106791, 8.77882957, 6.89398808, 5.06491952, 3.30038874, 1.60916062, 0, 1.39348525, 2.43104808, 3.13859399, 3.54202852, 3.66725717, 3.54018547, 3.18671894, 2.6327631, 1.90422346, 1.02700554, 0.0270148762, -1.06984303, -2.23766265, -3.45053847, -4.68256497, -5.90783662, -7.24258187, -8.68120003, -9.87629214, -10.8324099, -11.5541052, -12.0459297, -12.3124351, -12.3581733, -12.1876959, -11.8055548, -11.2163017, -10.4244883, -9.4346664, -8.52385677, -7.32423684, -5.75861766, -3.74981029, -1.35613231, 1.56735851, 5.20173944, 9.66485138, 15.0745353, 21.548632, 29.2049824, 38.4016314, 48.9431105, 60.900264, 74.3439365, 89.3449724, 104.740324, 122.143201, 141.932921, 164.488801, 190.19016, 219.609498, 252.696836, 289.510315, 330.279087, 375.581451, 425.838945, 481.473108, 542.918213, 610.58017, 684.850998, 766.143822, 854.85455, 950.904101, 1054.73754, 1167.68807, 1290.68122, 1425.29959, 1572.63905, 1731.38627, 1901.59073, 2080.99995, 2272.31946, 2476.39552, 2694.07443, 2926.20244, 3240.21205, 3553.71678, 3850.91634, 4116.01048, 4333.1989, 4417.33612, 4468.19723, 4499.70135, 4556.8896, 4641.36195, 4741.75088, 4860.30474, 4999.27189, 5160.90066, 5347.43942, 5568.61769, 5819.46378, 6098.22434, 6403.14599, 6732.47536, 7084.45906, 7457.34372, 7534.59402, 6745.68346}, 
{87.3325968, 87.3942783, 87.4275588, 87.4328506, 87.4105662, 87.3611178, 87.2849179, 87.1823788, 87.053913, 86.8999327, 86.7208504, 86.5170784, 86.2890292, 86.037115, 85.7617483, 85.4633415, 85.1423069, 84.7990568, 84.4340037, 84.04756, 83.640138, 83.2121501, 82.7640087, 82.2961261, 81.8089147, 81.3027869, 80.7781551, 80.2354317, 79.675029, 79.0973593, 78.5028352, 77.8918689, 77.2648728, 76.6222594, 75.9667655, 75.2975924, 74.6139133, 73.9160651, 73.2043849, 72.4792094, 71.7408757, 70.9897206, 70.2260812, 69.4502944, 68.662697, 67.8636261, 67.0534186, 66.2324113, 65.4009414, 64.5593456, 63.7079609, 62.8486498, 61.9801907, 61.1028874, 60.2170437, 59.3229635, 58.4209508, 57.5113093, 56.5943429, 55.6703556, 54.7396511, 53.8025333, 52.8593061, 51.9102734, 50.955739, 49.9960067, 49.0313806, 48.1137831, 47.2938974, 46.4655677, 45.6257517, 44.771407, 43.8994913, 43.0069621, 42.0907771, 41.1478939, 40.1752702, 39.1698636, 38.1286318, 37.0485324, 35.9265229, 34.7595612, 33.5446047, 32.2786112, 30.5938931, 28.6774003, 26.7204658, 24.7318807, 22.7204359, 20.6949226, 18.6641318, 16.6368546, 14.6218821, 12.6280054, 10.6640155, 8.73870357, 6.86086059, 5.03927766, 3.28274587, 1.60005629, 0, 1.35593845, 2.36494585, 3.05247487, 3.44397817, 3.56490841, 3.44071824, 3.09686033, 2.55878734, 1.85195193, 1.00180676, 0.0338044884, -1.02660222, -2.15396072, -3.32281833, -4.50772241, -5.68322028, -6.93503977, -8.25361114, -9.36200208, -10.2585843, -10.9417296, -11.4098096, -11.6611961, -11.6942608, -11.5073755, -11.0989118, -10.4672415, -9.61073634, -8.52776802, -7.44454482, -6.04806185, -4.26833989, -2.0353997, 0.558343127, 3.66406523, 7.47624857, 12.1135908, 17.6947897, 24.3385428, 32.1635479, 41.4562284, 52.1165093, 64.2302724, 77.8833993, 93.1617719, 109.123876, 127.139839, 147.55239, 170.704261, 196.938182, 226.707946, 260.10948, 297.279702, 338.472659, 384.181543, 434.792177, 490.690381, 552.242931, 619.859023, 693.968632, 775.000355, 863.313652, 958.806685, 1061.90485, 1173.92691, 1295.78154, 1429.00466, 1574.66755, 1731.54176, 1899.69971, 2077.31603, 2266.6529, 2468.43964, 2683.40558, 2912.28008, 3221.19024, 3529.11817, 3820.44379, 4079.54699, 4290.80766, 4369.72757, 4415.4835, 4441.97456, 4493.79064, 4572.68239, 4667.61264, 4780.97144, 4915.14882, 5072.53484, 5255.51953, 5529.9522, 5797.14126, 6065.36951, 6342.91977, 6638.07482, 6959.11747, 7314.33052, 7248.6562, 6319.78599}, 
{87.8762463, 87.9259711, 87.9475478, 87.9413779, 87.9078626, 87.8474035, 87.7604018, 87.647259, 87.5083764, 87.3441555, 87.1549976, 86.9413041, 86.7034763, 86.4419157, 86.1570237, 85.8492016, 85.5188508, 85.1663727, 84.7921687, 84.3966402, 83.9801885, 83.543215, 83.0861212, 82.6093084, 82.1131779, 81.5981312, 81.0645697, 80.5128947, 79.9435077, 79.3568099, 78.7532029, 78.1330879, 77.4968664, 76.8449397, 76.1791959, 75.4992619, 74.8047464, 74.0960026, 73.3733837, 72.6372428, 71.8879331, 71.1258076, 70.3512195, 69.5645219, 68.766068, 67.956211, 67.1353039, 66.3036999, 65.4617521, 64.6098137, 63.7482378, 62.87855, 61.9999055, 61.1126321, 60.2170576, 59.3135095, 58.4023157, 57.4838039, 56.5583018, 55.626137, 54.6876375, 53.7431308, 52.7929446, 51.8374068, 50.876845, 49.911587, 48.9419604, 48.0206564, 47.199109, 46.3696502, 45.5292133, 44.6747314, 43.8031377, 42.9113656, 41.9963481, 41.0550186, 40.0843101, 39.081156, 38.0424895, 36.9652437, 35.8463519, 34.6827473, 33.471363, 32.2091324, 30.5281461, 28.6154324, 26.6624051, 24.6778372, 22.6705018, 20.6491718, 18.6226204, 16.5996206, 14.5889455, 12.5993682, 10.6396616, 8.71859886, 6.84495304, 5.02749719, 3.27500436, 1.59624761, 0, 1.17161014, 2.03190546, 2.60391166, 2.91065444, 2.9751595, 2.82045254, 2.46955924, 1.94550532, 1.27131646, 0.470018359, -0.435363278, -1.42180276, -2.46627438, -3.54575245, -4.63721127, -5.71762515, -6.76616457, -7.76011759, -8.67149092, -9.47772949, -10.1562782, -10.684582, -11.0400857, -11.2002344, -11.1424729, -10.8442461, -10.2829989, -9.43617628, -8.28122314, -6.85834896, -5.05922041, -2.84245302, -0.166662325, 2.7217744, 6.00812249, 9.95438371, 14.6882711, 20.3374975, 27.029776, 34.8928196, 44.1394129, 54.7863058, 66.9445669, 80.7252645, 96.2394673, 112.886397, 131.666931, 152.870098, 176.78493, 203.700456, 233.907839, 267.69137, 305.293112, 346.995614, 393.164077, 444.126592, 500.211252, 561.547945, 628.708011, 702.479016, 783.688018, 872.273447, 967.893312, 1070.81999, 1182.36684, 1303.36942, 1435.12679, 1578.59468, 1733.02915, 1898.64771, 2074.20869, 2261.61322, 2461.58388, 2674.84324, 2902.11389, 3211.7954, 3520.0141, 3810.57333, 4067.27642, 4273.92671, 4342.25431, 4376.18457, 4390.40935, 4432.99821, 4505.431, 4595.28017, 4704.72093, 4835.92849, 4991.07806, 5172.34485, 5488.41864, 5768.78642, 6028.53976, 6282.77025, 6546.56945, 6835.02896, 7163.24033, 6941.7838, 5951.81363}, 
{88.5777614, 88.6154085, 88.6250075, 88.6069546, 88.5616462, 88.4894783, 88.3908472, 88.2661491, 88.1157802, 87.9401367, 87.7396148, 87.5146108, 87.2655208, 86.992741, 86.6966678, 86.3776972, 86.0362254, 85.6726488, 85.2873635, 84.8807656, 84.4532515, 84.0052174, 83.5370593, 83.0491737, 82.5419565, 82.0158042, 81.4711128, 80.9082786, 80.3276978, 79.7297666, 79.1148812, 78.4834379, 77.8358328, 77.1724621, 76.4947564, 75.8025729, 75.0957566, 74.3746701, 73.639676, 72.8911369, 72.1294156, 71.3548746, 70.5678766, 69.7687842, 68.9579602, 68.135767, 67.3025674, 66.4587241, 65.6045996, 64.7405566, 63.8669577, 62.9842718, 62.092753, 61.1927617, 60.2846582, 59.3688029, 58.4455561, 57.5152781, 56.5783293, 55.6350701, 54.6858607, 53.7310615, 52.7710329, 51.8061353, 50.8367288, 49.863174, 48.8858311, 47.9582895, 47.1328608, 46.3001243, 45.4569895, 44.6003663, 43.7271644, 42.8342935, 41.9186633, 40.9771836, 40.006764, 39.0043142, 37.966744, 36.8909631, 35.7738813, 34.6124081, 33.4034534, 32.1439269, 30.466085, 28.5568342, 26.6074634, 24.6267163, 22.6233361, 20.6060662, 18.5836502, 16.5648314, 14.5583532, 12.5729591, 10.6173924, 8.70039657, 6.83071504, 5.0170912, 3.26826847, 1.59299026, 0, 0.961723705, 1.65238122, 2.09223765, 2.3015581, 2.3006077, 2.10965153, 1.74895472, 1.23878237, 0.59939959, -0.148928508, -0.985936817, -1.89136023, -2.84493363, -3.82639192, -4.81546998, -5.7919027, -6.61684314, -7.25308592, -7.94858676, -8.65767017, -9.33466064, -9.93388267, -10.4096608, -10.7163194, -10.8081831, -10.6395763, -10.1648235, -9.33824928, -8.11417808, -6.32828149, -4.09704298, -1.41038292, 1.74177832, 4.94666189, 8.42577666, 12.5133938, 17.3464504, 23.0618838, 29.796631, 37.6876293, 46.8710232, 57.4847838, 69.6660034, 83.5517741, 99.2791883, 116.560843, 136.064449, 158.033223, 182.710382, 210.33914, 241.096421, 275.372761, 313.452394, 355.622379, 402.175543, 453.402123, 509.592356, 570.666512, 637.368878, 710.847343, 792.368184, 881.398581, 977.327701, 1080.2543, 1191.47948, 1311.75245, 1442.09567, 1583.3292, 1735.27115, 1898.3064, 2071.78024, 2257.27705, 2455.54156, 2667.3185, 2893.35258, 3205.03676, 3514.80547, 3805.74141, 4060.92725, 4263.44564, 4320.41314, 4341.52264, 4342.41373, 4375.51034, 4441.8916, 4527.3098, 4633.61039, 4762.63885, 4916.24061, 5096.26114, 5429.33233, 5716.5668, 5973.36318, 6215.12011, 6457.23626, 6715.11024, 7004.14071, 6629.08194, 5612.32502}, 
{89.4250614, 89.4505741, 89.4479804, 89.4176776, 89.3600628, 89.2755334, 89.1644865, 89.0273195, 88.8644295, 88.6762139, 88.4630697, 88.2253944, 87.9635851, 87.678039, 87.3691535, 87.0373257, 86.6829529, 86.3064324, 85.9081613, 85.488537, 85.0479566, 84.5868174, 84.1055167, 83.6044517, 83.0840196, 82.5446177, 81.9866432, 81.4104934, 80.8165655, 80.2052567, 79.5769643, 78.9320856, 78.2710177, 77.5941579, 76.9029098, 76.1971462, 75.4767281, 74.74202, 73.9933864, 73.2311921, 72.4558015, 71.6675792, 70.8668899, 70.0540982, 69.2295685, 68.3936655, 67.5467539, 66.6891981, 65.8213627, 64.9436125, 64.0563118, 63.1580872, 62.251079, 61.3356896, 60.4123212, 59.4813761, 58.5432566, 57.5983649, 56.6471033, 55.6898741, 54.7270796, 53.759122, 52.7864035, 51.8093265, 50.8282933, 49.843706, 48.855967, 47.9196893, 47.0881845, 46.2500491, 45.402171, 44.5414384, 43.6647391, 42.7689612, 41.8509927, 40.9077216, 39.9360359, 38.9328236, 37.8949726, 36.8193711, 35.7029069, 34.5424682, 33.3349428, 32.0772189, 30.4017257, 28.4952464, 26.5489448, 24.5715363, 22.5717364, 20.5582603, 18.5398234, 16.5251413, 14.5229291, 12.5419024, 10.5907765, 8.67826685, 6.81308875, 5.00395763, 3.25958887, 1.58869787, 0, 0.769768346, 1.30546897, 1.62486147, 1.74570547, 1.68576057, 1.46278638, 1.0945425, 0.59878854, -0.00671588557, -0.704211173, -1.47593771, -2.3041359, -3.17104612, -4.05890878, -4.94996426, -5.82645295, -6.4417415, -6.74537404, -7.2372816, -7.8506602, -8.51870585, -9.17461457, -9.75158238, -10.1828053, -10.4014793, -10.3408005, -9.93396481, -9.11416829, -7.81460697, -5.68588543, -3.04740797, 0.0828519873, 3.68692099, 7.20637385, 10.893463, 15.1445614, 20.103831, 25.915434, 32.7235325, 40.6722885, 49.8254287, 60.4320311, 72.6519949, 86.6452196, 102.571605, 120.374119, 140.483825, 163.114855, 188.481342, 216.797417, 248.202924, 283.077081, 321.662616, 364.22086, 411.051127, 462.435674, 518.656763, 579.502578, 645.861742, 719.161865, 801.058137, 890.584791, 986.880666, 1089.88024, 1200.86615, 1320.50221, 1449.54486, 1588.68196, 1738.26159, 1898.82405, 2070.21011, 2253.76715, 2450.27758, 2660.52381, 2885.28824, 3198.94542, 3510.28758, 3801.69911, 4055.56438, 4254.26777, 4300.4573, 4309.41126, 4297.68675, 4322.08006, 4383.22028, 4464.97014, 4568.79705, 4696.1684, 4848.55159, 5027.41402, 5344.94306, 5630.91717, 5892.41271, 6136.50605, 6370.27357, 6600.79164, 6835.13664, 6318.1044, 5299.07691}, 
{90.3986141, 90.4132684, 90.3994482, 90.357565, 90.2880302, 90.1912553, 90.0676518, 89.9176312, 89.7416049, 89.5399844, 89.3131811, 89.0616066, 88.7856723, 88.4857896, 88.1623701, 87.8158252, 87.4465664, 87.0550051, 86.6415529, 86.2066212, 85.7506214, 85.2739651, 84.7770636, 84.2603286, 83.7241714, 83.1690035, 82.5952364, 82.0032816, 81.3935504, 80.7664545, 80.1224053, 79.4618142, 78.7850927, 78.0926523, 77.3868201, 76.6670094, 75.9326106, 75.183973, 74.4214459, 73.6453786, 72.8561204, 72.0540206, 71.2394284, 70.4126933, 69.5741645, 68.7241913, 67.863123, 66.991309, 66.1090984, 65.2168407, 64.3148851, 63.3984152, 62.4730579, 61.5392747, 60.5975267, 59.6482754, 58.6919819, 57.7291076, 56.7601137, 55.7854616, 54.8056125, 53.8210278, 52.8321686, 51.8394964, 50.8434723, 49.8445577, 48.8432138, 47.8953656, 47.0556062, 46.2100074, 45.3554351, 44.488755, 43.6068331, 42.7065351, 41.7847267, 40.838274, 39.8640425, 38.8588982, 37.8197069, 36.7433343, 35.6266463, 34.4665087, 33.2597874, 32.003348, 30.3282201, 28.4218825, 26.4762112, 24.4999441, 22.5018191, 20.4905741, 18.4749469, 16.4636756, 14.465498, 12.4891521, 10.5433757, 8.63690667, 6.77848302, 4.97684262, 3.24072338, 1.57886321, 0, 0.74587896, 1.26400744, 1.57195121, 1.68727606, 1.62754778, 1.41033214, 1.05319494, 0.57370194, -0.0105810631, -0.682088289, -1.42325396, -2.21651228, -3.04429748, -3.88904377, -4.73318538, -5.55915651, -6.09753897, -6.29478808, -6.7030042, -7.25065318, -7.86620087, -8.47811311, -9.01485574, -9.40489462, -9.57669557, -9.45872445, -8.9794471, -8.06732936, -6.65083708, -4.34648123, -1.50906597, 1.83935637, 5.6767335, 9.43765363, 13.3717442, 17.8735285, 23.083962, 29.1440003, 36.194599, 44.3767135, 53.7106782, 64.4947806, 76.8935761, 91.0716201, 107.193468, 125.092107, 145.346552, 168.204252, 193.912652, 222.719201, 254.983085, 290.703443, 329.832585, 372.518948, 419.311402, 470.57903, 526.690915, 587.623265, 654.227329, 727.782945, 809.951153, 899.679556, 996.027865, 1098.87535, 1209.48995, 1328.50219, 1456.59236, 1594.40388, 1742.39768, 1901.13795, 2070.49327, 2251.83038, 2445.95421, 2653.66969, 2875.78175, 3187.37132, 3496.39834, 3785.09874, 4035.70844, 4230.46338, 4270.8144, 4273.63925, 4255.79601, 4275.78716, 4333.98658, 4413.41634, 4515.3179, 4640.93273, 4791.50231, 4968.26812, 5184.27819, 5448.66871, 5737.07657, 6025.13868, 6288.49192, 6502.7732, 6643.6194, 6028.0876, 5007.09056}, 
{91.488041, 91.4928875, 91.4686096, 91.4156451, 91.3344318, 91.2254075, 91.0890098, 90.9256767, 90.7358458, 90.519955, 90.2784421, 90.0117447, 89.7203007, 89.4045478, 89.0649239, 88.7018667, 88.315814, 87.9072035, 87.4764731, 87.0240605, 86.5504034, 86.0559398, 85.5411072, 85.0063436, 84.4520867, 83.8787742, 83.286844, 82.6767338, 82.0488814, 81.4037245, 80.741701, 80.0632486, 79.3688052, 78.6588083, 77.937389, 77.2030606, 76.4542925, 75.6914026, 74.9147088, 74.1245291, 73.3211814, 72.5049836, 71.6762536, 70.8353094, 69.982469, 69.1180501, 68.2423709, 67.3557492, 66.4585029, 65.55095, 64.6334084, 63.6961235, 62.7497042, 61.7946866, 60.8316071, 59.8610021, 58.8834076, 57.8993602, 56.9093959, 55.9140512, 54.9138622, 53.9093654, 52.9010969, 51.889593, 50.8753901, 49.8590244, 48.8410322, 47.8788957, 47.0287298, 46.1736241, 45.3104234, 44.4359724, 43.5471158, 42.6406983, 41.7135647, 40.7625596, 39.7845278, 38.7763139, 37.7347627, 36.656719, 35.5390273, 34.3785325, 33.1720793, 31.9165123, 30.2397814, 28.330938, 26.3834673, 24.4061853, 22.407908, 20.3974516, 18.3836321, 16.3752655, 14.381168, 12.4101556, 10.4710443, 8.57265017, 6.72378935, 4.93327786, 3.20993176, 1.56256712, 0, 0.909182859, 1.56283431, 1.98089138, 2.1832911, 2.1899705, 2.0208666, 1.69591642, 1.23505701, 0.658225387, -0.0146414254, -0.763606396, -1.5687325, -2.4100827, -3.26771998, -4.12170731, -4.95210765, -5.56128542, -5.90391808, -6.36200584, -6.8775334, -7.39248547, -7.84884675, -8.18860197, -8.35373582, -8.28623301, -7.92807825, -7.22125625, -6.10775172, -4.52954937, -2.23669276, 0.566513837, 3.88050338, 7.70570884, 11.622236, 15.8406814, 20.6837287, 26.2779092, 32.7497539, 40.2257939, 48.8325604, 58.5792284, 69.7454021, 82.4805734, 96.9342345, 113.255877, 130.81859, 150.74237, 173.370809, 199.047501, 228.116038, 261.405769, 298.18083, 337.866517, 380.416536, 426.863424, 477.755347, 533.640469, 595.009463, 662.48105, 736.736672, 819.024315, 908.589638, 1004.60754, 1107.02513, 1217.09801, 1335.48117, 1462.98951, 1600.31949, 1747.58112, 1905.21611, 2072.62286, 2251.44694, 2442.49125, 2646.55868, 2864.45212, 3169.30071, 3471.49952, 3753.76986, 3998.83306, 4189.41044, 4229.52534, 4233.06238, 4216.47059, 4236.85316, 4294.56684, 4373.02916, 4473.44385, 4597.01462, 4744.9452, 4918.4393, 4949.46112, 5172.50126, 5509.33586, 5881.74104, 6211.49294, 6420.36768, 6430.14138, 5763.91657, 4734.74761}, 
{92.6936721, 92.6877135, 92.6519367, 92.5868072, 92.4927907, 92.3703527, 92.2199587, 92.0420743, 91.8371652, 91.6056967, 91.3481346, 91.0649442, 90.7565913, 90.4235414, 90.06626, 89.6852127, 89.280865, 88.8536825, 88.4041308, 87.9326755, 87.4397819, 86.9259159, 86.3915428, 85.8371283, 85.2631379, 84.6700372, 84.0582917, 83.428367, 82.7807287, 82.1158423, 81.4341734, 80.7361874, 80.0223501, 79.2931269, 78.5545706, 77.8042356, 77.0396092, 76.2609757, 75.4686194, 74.6628244, 73.843875, 73.0120555, 72.1676501, 71.3109431, 70.4422187, 69.5617611, 68.6698546, 67.7667834, 66.8528318, 65.928284, 64.9934244, 64.0332461, 63.0636559, 62.0852694, 61.0987025, 60.1045707, 59.1034899, 58.0960757, 57.0829439, 56.0647102, 55.0419903, 54.0153999, 52.9855548, 51.9530707, 50.9185632, 49.8826482, 48.8459414, 47.867445, 47.0047598, 46.1380832, 45.264246, 44.3800791, 43.4824131, 42.5680787, 41.6339069, 40.6767282, 39.6933734, 38.6806734, 37.6354588, 36.5545604, 35.434809, 34.2730353, 33.0660701, 31.810744, 30.1318636, 28.2200966, 26.2705827, 24.2922254, 22.2939284, 20.2845953, 18.2731299, 16.2684357, 14.2794165, 12.3149759, 10.3840176, 8.49544518, 6.65816234, 4.88107274, 3.17308005, 1.54308791, 0, 1.12554858, 1.95836259, 2.52145932, 2.83785603, 2.93057001, 2.82261853, 2.53701886, 2.09678827, 1.52494405, 0.844503468, 0.0784837987, -0.750097683, -1.6182237, -2.50287698, -3.38104025, -4.22969623, -4.95084573, -5.51075979, -6.05602306, -6.54755072, -6.94625794, -7.21305989, -7.30887175, -7.19460868, -6.83118587, -6.17951848, -5.20052169, -3.85511067, -2.10420059, 0.118709491, 2.81523571, 6.01623806, 9.75257653, 13.7944888, 18.303146, 23.5092186, 29.5217532, 36.4497966, 44.4023955, 53.4885965, 63.7165944, 75.3269819, 88.4485375, 103.21004, 119.740268, 136.882801, 156.372916, 178.660692, 204.196206, 233.429537, 267.689168, 305.473166, 345.663261, 388.014733, 434.066371, 484.556204, 540.222262, 602.1261, 670.608695, 745.658089, 828.001794, 917.291152, 1012.8821, 1114.82638, 1224.35474, 1342.15675, 1469.22864, 1606.33948, 1753.13393, 1909.89268, 2075.42, 2251.70014, 2439.52487, 2639.68596, 2852.97518, 3149.72854, 3443.8075, 3718.61774, 3957.56496, 4144.05488, 4185.877, 4192.464, 4179.62849, 4202.15583, 4260.43194, 4338.60413, 4437.87025, 4559.42817, 4704.47575, 4874.21085, 4704.72951, 4882.1275, 5270.35753, 5733.37225, 6135.12436, 6339.56652, 6210.65142, 5513.68738, 4484.36091}, 
{94.0084985, 93.9899385, 93.9409166, 93.8619236, 93.7534501, 93.6159868, 93.4500246, 93.2560541, 93.034566, 92.786051, 92.5109999, 92.2099034, 91.8832522, 91.531537, 91.1552486, 90.7548776, 90.3309148, 89.8838509, 89.4141766, 88.9223826, 88.4089596, 87.8743985, 87.3191898, 86.7438243, 86.1487928, 85.5345858, 84.9016943, 84.2506088, 83.5818201, 82.895819, 82.193096, 81.474142, 80.7394477, 79.9895038, 79.2321152, 78.4639617, 77.6816351, 76.8853888, 76.0754762, 75.2521506, 74.4156654, 73.5662739, 72.7042295, 71.8297855, 70.9431954, 70.0447125, 69.13459, 68.2130815, 67.2804402, 66.3369195, 65.3827727, 64.397871, 63.4032919, 62.3997304, 61.3878819, 60.3684416, 59.3421046, 58.3095661, 57.2715215, 56.2286657, 55.1816942, 54.131302, 53.0781845, 52.0230367, 50.9665539, 49.9094313, 48.8523642, 47.8557198, 46.978433, 46.0981294, 45.211636, 44.3157797, 43.4073874, 42.483286, 41.5403024, 40.5752635, 39.5849962, 38.5663273, 37.5160839, 36.4310928, 35.3081808, 34.144175, 32.9359021, 31.6801892, 29.99907, 28.0846513, 26.1335445, 24.1547392, 22.157225, 20.1499917, 18.1420288, 16.1423261, 14.1598731, 12.2036596, 10.2826751, 8.40590945, 6.58235215, 4.8209929, 3.13082136, 1.52082717, 0, 1.36587957, 2.39782185, 3.12222301, 3.56547922, 3.75398664, 3.71414143, 3.47233975, 3.05497776, 2.48845164, 1.79915754, 1.01349162, 0.157850048, -0.741371011, -1.6577754, -2.56496694, -3.43654949, -4.28757811, -5.09757822, -5.74639495, -6.21651454, -6.49042324, -6.55060727, -6.37955289, -5.95974634, -5.27367386, -4.3038217, -3.03267609, -1.44272328, 0.483550485, 2.60489098, 5.15579488, 8.20140633, 11.8068695, 15.9527081, 20.7463766, 26.3178947, 32.7577927, 40.1566008, 48.6048492, 58.1930683, 68.9339098, 81.0194847, 94.5555602, 109.647904, 126.402283, 143.144804, 162.205811, 184.135986, 209.48601, 238.806566, 273.880598, 312.52043, 353.125344, 395.276749, 440.96956, 491.115079, 546.624606, 609.106371, 678.616352, 754.450239, 836.781093, 925.712681, 1020.82325, 1122.2884, 1231.29588, 1348.56886, 1475.29426, 1612.31548, 1758.77543, 1914.7789, 2078.50904, 2252.28795, 2436.88561, 2633.07201, 2841.61713, 3129.53174, 3414.78486, 3681.58628, 3914.1458, 4096.67321, 4141.44891, 4152.56502, 4145.05823, 4170.86455, 4230.41468, 4308.76456, 4407.11362, 4526.66127, 4668.60693, 4834.15002, 4471.75142, 4604.46617, 5040.74588, 5589.04216, 6057.80661, 6255.49085, 5990.54647, 5277.00231, 4255.5491}, 
{95.4255111, 95.3917547, 95.3270368, 95.2318667, 95.1067533, 94.952206, 94.7687338, 94.5568459, 94.3170515, 94.0498596, 93.7557796, 93.4353204, 93.0889913, 92.7173014, 92.3207599, 91.8998759, 91.4551586, 90.9871171, 90.4962606, 89.9830982, 89.4481392, 88.8918925, 88.3148675, 87.7175732, 87.1005188, 86.4642134, 85.8091663, 85.1358865, 84.4448832, 83.7366656, 83.0117427, 82.2706239, 81.5138182, 80.7418347, 79.9637729, 79.1756661, 78.3734444, 77.5573382, 76.7275778, 75.8843938, 75.0280164, 74.158676, 73.2766031, 72.382028, 71.4751811, 70.5562928, 69.6255934, 68.6833134, 67.7296831, 66.7649329, 65.7892933, 64.7780864, 63.7569909, 62.7267768, 61.6882147, 60.6420749, 59.5891277, 58.5301434, 57.4658924, 56.397145, 55.3246716, 54.2492425, 53.171628, 52.0925986, 51.0129244, 49.933376, 48.8547236, 47.8384263, 46.9444862, 46.0485076, 45.1473265, 44.2377791, 43.3167016, 42.38093, 41.4273006, 40.4526494, 39.4538126, 38.4276264, 37.3709268, 36.28055, 35.1533321, 33.9861094, 32.7757178, 31.5189935, 29.8360043, 27.9198952, 25.9683399, 23.9904015, 21.995143, 19.9916273, 17.9889174, 15.9960764, 14.0221672, 12.0762529, 10.1673964, 8.30466076, 6.49710894, 4.75380396, 3.08380881, 1.49618649, 0, 1.60107928, 2.82844165, 3.71175047, 4.28066904, 4.5648607, 4.59398878, 4.39771662, 4.00570753, 3.44762486, 2.75313192, 1.95189205, 1.07356858, 0.147824843, -0.795675839, -1.72727013, -2.61729471, -3.59284075, -4.64663842, -5.39446063, -5.84023429, -5.98788632, -5.84134366, -5.40453322, -4.68138192, -3.67581669, -2.39176445, -0.833152123, 0.996093371, 3.09204511, 5.10701715, 7.50688656, 10.3908541, 13.8581206, 18.0951902, 23.1576119, 29.077654, 35.9283263, 43.7826386, 52.7136006, 62.7942221, 74.0423083, 86.6028751, 100.559733, 115.996693, 132.997565, 149.464667, 168.208676, 189.858777, 215.044154, 244.393991, 280.027379, 319.262598, 360.155293, 402.16579, 447.622312, 497.565455, 553.035816, 616.083472, 686.510111, 763.016166, 845.259715, 933.78281, 1028.40271, 1129.42052, 1237.95721, 1354.75741, 1481.17088, 1618.09914, 1764.22491, 1919.486, 2081.51435, 2252.90837, 2434.40402, 2626.7373, 2830.64419, 3109.58723, 3385.89419, 3644.6194, 3870.81721, 4049.54195, 4097.82057, 4114.08638, 4112.54833, 4142.14872, 4203.34785, 4282.1338, 4379.69048, 4497.2018, 4635.85169, 4796.82405, 4272.19491, 4366.43597, 4841.10521, 5457.76061, 5977.96014, 6163.26179, 5775.22353, 5053.46362, 4047.93082}, 
{96.8658079, 96.8078819, 96.7195602, 96.6013221, 96.4536472, 96.2770149, 96.0719047, 95.838796, 95.5781683, 95.2905011, 94.9762737, 94.6359658, 94.2700567, 93.8790258, 93.4633528, 93.0235169, 92.5599978, 92.0732748, 91.5638274, 91.032135, 90.4786772, 89.9039333, 89.308383, 88.6925055, 88.0567804, 87.4016871, 86.7277051, 86.0353138, 85.3249928, 84.5972215, 83.8524792, 83.0912456, 82.314, 81.521222, 80.7201646, 79.907778, 79.0809309, 78.2398831, 77.384894, 76.5162234, 75.6341309, 74.7388762, 73.8307188, 72.9099185, 71.9767349, 71.0314276, 70.0742562, 69.1054804, 68.1253599, 67.1341543, 66.1321232, 65.0940516, 64.046226, 62.9894584, 61.9245606, 60.8523445, 59.773622, 58.689205, 57.5999054, 56.506535, 55.4099057, 54.3108295, 53.2101181, 52.1085835, 51.0070376, 49.9062923, 48.8071594, 47.7710695, 46.8579978, 45.9437347, 45.0251624, 44.099163, 43.1626188, 42.212412, 41.2454248, 40.2585395, 39.2486383, 38.2126034, 37.1473169, 36.0496612, 34.9165185, 33.7447709, 32.5313007, 31.2729901, 29.5910678, 27.6781441, 25.7313445, 23.7597418, 21.772409, 19.778419, 17.7868446, 15.8067588, 13.8472344, 11.9173444, 10.0261617, 8.1827591, 6.39620957, 4.675586, 3.02996128, 1.46840831, 0, 1.73603829, 3.07870517, 4.05929383, 4.70909749, 5.05940937, 5.14152266, 4.98673058, 4.62632635, 4.09160316, 3.41385423, 2.62437277, 1.754452, 0.835385108, -0.101534681, -1.02501416, -1.90376012, -2.91403814, -4.05420627, -4.82349308, -5.23508224, -5.30215744, -5.03790233, -4.4555006, -3.56813592, -2.38899196, -0.931252395, 0.791899096, 2.76727884, 4.98170317, 6.99387283, 9.37569543, 12.242422, 15.7093035, 20.0694896, 25.3492819, 31.5275426, 38.666153, 46.8269945, 56.0719486, 66.4628967, 77.9007251, 90.6573088, 104.826028, 120.500264, 137.773395, 154.998252, 174.443904, 196.63887, 222.111668, 251.390818, 286.048291, 324.293819, 364.842529, 407.403109, 453.712767, 504.597949, 560.8851, 624.1676, 694.330656, 770.422813, 852.131163, 940.094128, 1034.37788, 1135.34267, 1243.84696, 1360.52055, 1486.51998, 1622.6116, 1767.63029, 1921.50351, 2081.91077, 2251.37324, 2430.59649, 2620.28607, 2821.14753, 3095.06844, 3366.27684, 3620.18278, 3842.19632, 4017.72751, 4065.70479, 4082.3409, 4081.53803, 4111.82984, 4173.41021, 4252.04322, 4348.64422, 4464.12858, 4599.41166, 4755.40881, 4259.80408, 4355.71086, 4811.69346, 5396.31619, 5878.14335, 6025.73924, 5607.66818, 4841.58002, 3860.68833}, 
{98.4092025, 98.3240728, 98.2092734, 98.065247, 97.892436, 97.6912831, 97.4622309, 97.2057219, 96.9221988, 96.6121042, 96.2758805, 95.9139704, 95.5268166, 95.1148615, 94.6785478, 94.218318, 93.7346148, 93.2278807, 92.6985583, 92.1470903, 91.5739191, 90.9794874, 90.3642378, 89.7286128, 89.0730551, 88.3980072, 87.7039117, 86.9912112, 86.2603483, 85.5117656, 84.7459056, 83.963211, 83.1641244, 82.3490882, 81.5229729, 80.683914, 79.8299939, 78.9615115, 78.0787657, 77.1820554, 76.2716796, 75.347937, 74.4111267, 73.4615474, 72.4994982, 71.5252779, 70.5391855, 69.5415197, 68.5325795, 67.5126639, 66.4820717, 65.4158897, 64.3401755, 63.2557745, 62.1635321, 61.0642939, 59.9589052, 58.8482115, 57.7330583, 56.614291, 55.4927551, 54.3692959, 53.244759, 52.1199899, 50.9958338, 49.8731364, 48.752743, 47.6956976, 46.7615988, 45.8271372, 44.8892558, 43.9448976, 42.9910057, 42.0245229, 41.0423925, 40.0415574, 39.0189606, 37.9715452, 36.8962541, 35.7900305, 34.6498173, 33.4725575, 32.2551942, 30.9946704, 29.3146604, 27.4065929, 25.4663827, 23.5030862, 21.5257597, 19.5434596, 17.5652421, 15.6001635, 13.6572802, 11.7456484, 9.87432451, 8.05236481, 6.28882561, 4.59276321, 2.97323394, 1.4392941, 0, 1.82138133, 3.23924754, 4.28582086, 4.99332352, 5.39397776, 5.5200058, 5.40362987, 5.07707221, 4.57255506, 3.92230063, 3.15853117, 2.31346889, 1.41933605, 0.508354856, -0.387252448, -1.23526363, -2.24107338, -3.40640475, -4.16156865, -4.52526077, -4.51617679, -4.1530124, -3.45446326, -2.43922506, -1.12599348, 0.466535802, 2.31966711, 4.41470476, 6.73295307, 8.77423693, 11.1778826, 14.0696381, 17.5752517, 22.0559472, 27.5197349, 33.9118315, 41.2873427, 49.7013739, 59.2090305, 69.865418, 81.4702282, 94.4117148, 108.794956, 124.725028, 142.30701, 160.456064, 180.764662, 203.63536, 229.470716, 258.673285, 292.145879, 329.179378, 369.279201, 412.451775, 459.726423, 511.673206, 568.862188, 632.312061, 702.042297, 777.582953, 858.746964, 946.188649, 1040.18695, 1141.13825, 1249.63797, 1366.19017, 1491.71617, 1626.82822, 1770.60859, 1923.00498, 2081.91852, 2249.65837, 2426.88065, 2614.24151, 2812.39708, 3081.66357, 3348.12203, 3597.5136, 3815.57941, 3988.06056, 4035.65115, 4052.50404, 4052.23767, 4083.1204, 4145.05507, 4223.50705, 4319.10107, 4432.46184, 4564.21409, 4714.98252, 4296.74371, 4405.07151, 4828.61372, 5356.01817, 5775.93267, 5877.00501, 5447.88302, 4642.27445, 3693.96725}, 
{100.067663, 99.9529698, 99.8093687, 99.6372643, 99.437061, 99.209163, 98.9539747, 98.6719002, 98.3633438, 98.0287099, 97.6684027, 97.2828265, 96.8723856, 96.4374842, 95.9785267, 95.4959172, 94.9900601, 94.4613596, 93.9102201, 93.3370457, 92.7422409, 92.1262098, 91.4893567, 90.8320859, 90.1548018, 89.4579084, 88.7418103, 88.0069115, 87.2536164, 86.4823293, 85.6934544, 84.8873961, 84.0645585, 83.225346, 82.3720534, 81.5041, 80.6208823, 79.7227431, 78.8100254, 77.8830721, 76.9422262, 75.9878305, 75.020228, 74.0397616, 73.0467743, 72.0416089, 71.0246084, 69.9961157, 68.9564738, 67.9060255, 66.8451137, 65.74962, 64.644879, 63.5317638, 62.4111478, 61.2839041, 60.1509059, 59.0130264, 57.8711388, 56.7261163, 55.5788321, 54.4301595, 53.2809715, 52.1321415, 50.9845426, 49.839048, 48.6965309, 47.6173649, 46.6604947, 45.7040788, 44.7451332, 43.7806739, 42.807717, 41.8232785, 40.8243744, 39.8080207, 38.7712336, 37.7110291, 36.6244231, 35.5084318, 34.3600711, 33.1763572, 31.954306, 30.6909336, 29.0135817, 27.1118793, 25.1798981, 23.2266509, 21.2611504, 19.2924092, 17.32944, 15.3812556, 13.4568686, 11.5652916, 9.71553741, 7.91661863, 6.17754796, 4.50733808, 2.91500167, 1.40955142, 0, 1.84543728, 3.28864388, 4.36195421, 5.09770269, 5.52822372, 5.68585172, 5.60292111, 5.31176629, 4.84472168, 4.23412169, 3.51230074, 2.71159324, 1.86433359, 1.00285623, 0.159495548, -0.63341403, -1.58842924, -2.70820011, -3.40647179, -3.70338654, -3.61908662, -3.17371429, -2.38741182, -1.28032145, 0.127414539, 1.8156539, 3.76425437, 5.95307368, 8.36196959, 10.4695841, 12.9407698, 15.9057491, 19.4947444, 24.0926504, 29.7023535, 36.2588274, 43.815893, 52.427371, 62.1470825, 73.028848, 84.8133034, 97.9627718, 112.592349, 128.817132, 146.752216, 165.884058, 187.113554, 210.71296, 236.954532, 266.110526, 298.401391, 334.21451, 373.84644, 417.602837, 465.807938, 518.777638, 576.827833, 640.392149, 709.641992, 784.620334, 865.322443, 952.347621, 1046.12214, 1147.05137, 1255.50525, 1371.87, 1496.82923, 1630.84628, 1773.2939, 1924.16172, 2081.72937, 2247.9596, 2423.43394, 2608.73399, 2804.44129, 3069.11539, 3330.8654, 3575.77836, 3789.94137, 3959.44148, 4006.82822, 4024.08457, 4024.528, 4056.07354, 4118.34376, 4196.54557, 4291.02231, 4402.11733, 4530.17398, 4675.53561, 4369.81771, 4499.00202, 4880.88529, 5333.26426, 5673.93567, 5720.69625, 5291.34275, 4455.43783, 3547.50139}, 
{101.85316, 101.707216, 101.533038, 101.330997, 101.101464, 100.844807, 100.561398, 100.251607, 99.9158041, 99.5543595, 99.1676435, 98.7560263, 98.3198784, 97.85957, 97.3754712, 96.8679525, 96.3373841, 95.7841362, 95.2085791, 94.6110831, 93.9920185, 93.3517555, 92.6906645, 92.0091157, 91.3074793, 90.5861256, 89.845425, 89.0857477, 88.3074639, 87.5109439, 86.6965581, 85.8646766, 85.0156698, 84.1499079, 83.2672619, 82.3683622, 81.4538453, 80.5240976, 79.5795056, 78.6204557, 77.6473346, 76.6605286, 75.6604242, 74.6474079, 73.6218661, 72.5841854, 71.5347523, 70.4739531, 69.4021745, 68.3198028, 67.2272245, 66.1012618, 64.9663764, 63.8234654, 62.6734263, 61.5171564, 60.3555529, 59.1895131, 58.0199344, 56.8477141, 55.6737494, 54.4989377, 53.3241764, 52.1503626, 50.9783937, 49.809167, 48.6435798, 47.5411256, 46.5598911, 45.5799233, 44.5983209, 43.6121828, 42.6186075, 41.614694, 40.597541, 39.5642472, 38.5119113, 37.4376322, 36.3385085, 35.2116391, 34.0541226, 32.8630579, 31.6355436, 30.3686787, 28.6946311, 26.8006409, 24.878334, 22.9366519, 20.984536, 19.0309278, 17.0847687, 15.1550003, 13.2505638, 11.3804009, 9.55345289, 7.77866129, 6.06496754, 4.42131308, 2.85663937, 1.37988786, 0, 1.79653504, 3.2054693, 4.25831655, 4.98659056, 5.42180511, 5.59547397, 5.53911091, 5.28422971, 4.86234415, 4.30496799, 3.64361501, 2.90979898, 2.13503369, 1.35083289, 0.588710374, -0.119820095, -0.970588458, -1.96455861, -2.55598695, -2.76207621, -2.60002913, -2.08704844, -1.24033688, -0.0770971827, 1.38546791, 3.13015565, 5.13976332, 7.39708817, 9.88492747, 12.101389, 14.6916789, 17.7840015, 21.5065607, 26.2176867, 31.9305206, 38.5968367, 46.2758012, 55.0265804, 64.9083408, 75.9802486, 87.9924367, 101.407159, 116.344044, 132.922721, 151.26282, 171.328189, 193.433184, 217.73638, 244.396351, 273.571673, 304.896074, 339.694453, 378.925381, 423.14734, 472.10197, 525.897657, 584.642789, 648.283159, 717.126698, 791.658707, 872.072922, 958.852292, 1052.47572, 1153.32613, 1261.62383, 1377.66377, 1501.92894, 1634.76304, 1775.82033, 1925.14504, 2081.53509, 2246.4728, 2420.43384, 2603.89388, 2797.32859, 3057.16669, 3313.94254, 3554.14354, 3764.2571, 3930.77064, 3978.4046, 3996.5913, 3998.28972, 4030.74236, 4093.33762, 4171.17903, 4264.36921, 4373.01078, 4497.20633, 4637.0585, 4465.82996, 4621.9865, 4957.52744, 5324.45212, 5574.7599, 5560.45009, 5133.52205, 4280.96113, 3421.02457}, 
{103.777659, 103.599453, 103.393474, 103.160069, 102.899585, 102.612368, 102.298764, 101.959119, 101.593781, 101.203094, 100.787406, 100.347062, 99.8824096, 99.3937946, 98.8815633, 98.3460621, 97.7876372, 97.206635, 96.6034019, 95.9782841, 95.3316279, 94.6637798, 93.975086, 93.2658928, 92.5365465, 91.7873935, 91.0187802, 90.2310527, 89.4245576, 88.5996409, 87.7566492, 86.8959287, 86.0178257, 85.1226866, 84.208454, 83.2767266, 82.329132, 81.3660946, 80.3880387, 79.3953886, 78.3885686, 77.3680031, 76.3341165, 75.2873329, 74.2280768, 73.1567725, 72.0738443, 70.9797165, 69.8748135, 68.7595596, 67.6343792, 66.4768345, 65.3107075, 64.1369181, 62.9563862, 61.7700318, 60.5787748, 59.3835351, 58.1852327, 56.9847875, 55.7831194, 54.5811483, 53.3797942, 52.179977, 50.9826166, 49.788633, 48.5989461, 47.4720341, 46.4649935, 45.4600343, 44.4543453, 43.445115, 42.4295321, 41.4047852, 40.3680631, 39.3165544, 38.2474477, 37.1579317, 36.045195, 34.9064263, 33.7388143, 32.5395476, 31.3058148, 30.0348047, 28.364608, 26.4795155, 24.5681339, 22.6393053, 20.7018718, 18.7646754, 16.8365583, 14.9263625, 13.04293, 11.195103, 9.39172348, 7.64163354, 5.95367525, 4.33669068, 2.79952191, 1.35101099, 0, 1.66300351, 2.96829891, 3.94553053, 4.62434271, 5.03437978, 5.20528607, 5.16670591, 4.94828364, 4.57966359, 4.0904901, 3.51040749, 2.8690601, 2.19609226, 1.52114831, 0.873872574, 0.283909388, -0.40203381, -1.18044653, -1.6078986, -1.69394646, -1.44814653, -0.88005525, 0.000770956926, 1.18477566, 2.66240242, 4.42409482, 6.46029642, 8.76145079, 11.3180015, 13.6911262, 16.4579318, 19.7376419, 23.6494796, 28.4691434, 34.2376191, 40.954166, 48.6910648, 57.5205965, 67.5150418, 78.7466816, 91.070114, 104.841554, 120.175872, 137.18794, 155.992629, 176.834411, 199.666155, 224.570332, 251.62941, 280.925861, 311.711175, 345.914443, 384.897157, 429.376331, 478.753177, 533.019676, 592.167809, 655.860387, 724.493372, 798.82182, 879.213725, 965.983909, 1059.53993, 1160.20662, 1268.16871, 1383.67523, 1507.08507, 1638.67577, 1778.32199, 1926.12626, 2081.52743, 2245.39384, 2418.05779, 2599.85156, 2791.10743, 3045.56024, 3296.78908, 3531.77563, 3737.50152, 3900.94841, 3949.54887, 3969.53303, 3973.40356, 4007.18001, 4070.09797, 4147.42769, 4239.10305, 4345.05791, 4465.22614, 4599.5416, 4571.58438, 4758.50904, 5047.55945, 5325.97946, 5481.01292, 5399.90368, 4969.8956, 4118.73526, 3314.2706}, 
{105.942093, 105.731493, 105.492907, 105.226688, 104.933194, 104.61278, 104.265802, 103.892616, 103.493578, 103.069044, 102.619369, 102.14491, 101.646023, 101.123063, 100.576386, 100.006349, 99.4133062, 98.7976148, 98.1596304, 97.4997089, 96.8182062, 96.1154781, 95.3918806, 94.6477696, 93.883501, 93.0994307, 92.2959146, 91.4733086, 90.6319687, 89.7722507, 88.8945106, 87.9991042, 87.0863874, 86.1567163, 85.2080004, 84.2418702, 83.2599857, 82.262782, 81.2506945, 80.2241585, 79.1836093, 78.1294821, 77.0622122, 75.982235, 74.8899858, 73.7858997, 72.6704122, 71.5439585, 70.4069738, 69.2598936, 68.103153, 66.9138103, 65.7161845, 64.5112179, 63.2998524, 62.0830301, 60.8616931, 59.6367835, 58.4092433, 57.1800146, 55.9500395, 54.7202599, 53.491618, 52.2650559, 51.0415156, 49.8219391, 48.6072685, 47.4549361, 46.4210178, 45.3899489, 44.3590095, 43.3254796, 42.2866391, 41.2397683, 40.182147, 39.1110554, 38.0237734, 36.9175811, 35.7897586, 34.6375858, 33.4583429, 32.2493098, 31.0077666, 29.7309933, 28.0646135, 26.1891321, 24.2892805, 22.3737509, 20.451235, 18.530425, 16.6200128, 14.7286903, 12.8651495, 11.0380823, 9.2561808, 7.52813688, 5.86264252, 4.26838969, 2.75407035, 1.32837646, 0, 1.24890629, 2.22045636, 2.93900544, 3.42890872, 3.71452142, 3.82019874, 3.77029589, 3.58916808, 3.30117052, 2.93065842, 2.50198697, 2.0395114, 1.56758691, 1.1105687, 0.692811994, 0.338671988, 0.01225811, -0.270680973, -0.349519828, -0.212813056, 0.150884738, 0.753018953, 1.60503498, 2.71837823, 4.10449409, 5.77482796, 7.74082523, 10.0139313, 12.6055916, 15.2580942, 18.3507328, 21.9757, 26.2251883, 31.1385842, 36.8344832, 43.4455628, 51.0798573, 59.8454011, 69.8502286, 81.2023743, 94.1757961, 108.662106, 124.736876, 142.475677, 161.95408, 183.008816, 206.014006, 231.104932, 258.416874, 288.085113, 319.66072, 354.577222, 393.738541, 437.711561, 486.375053, 539.916737, 598.524333, 662.311653, 731.557121, 806.619792, 887.871042, 975.358441, 1069.03149, 1169.05907, 1275.9818, 1390.16964, 1512.15794, 1642.35955, 1780.58098, 1926.97173, 2081.93009, 2245.31847, 2417.20022, 2597.6387, 2786.69727, 3033.17165, 3276.20976, 3503.69185, 3703.49819, 3863.50907, 3916.24583, 3941.8536, 3951.93789, 3988.8124, 4051.92731, 4127.93776, 4216.93369, 4319.00505, 4434.2418, 4562.73388, 4673.63302, 4897.63786, 5150.637, 5348.51905, 5407.17259, 5242.48622, 4770.34853, 3962.03329, 3230.76808}, 
{108.201916, 107.959102, 107.687881, 107.388628, 107.061716, 106.707518, 106.326409, 105.918762, 105.48495, 105.025347, 104.540326, 104.030262, 103.495527, 102.936495, 102.35354, 101.747036, 101.117355, 100.464872, 99.7899596, 99.092992, 98.3743426, 97.6343848, 96.8734924, 96.0920388, 95.2903977, 94.4689425, 93.6280469, 92.7680843, 91.8894285, 90.9924529, 90.0775311, 89.1450367, 88.1953432, 87.2288242, 86.2437386, 85.2415618, 84.2237945, 83.190879, 82.1432575, 81.0813721, 80.0056651, 78.9165785, 77.8145547, 76.7000359, 75.5734641, 74.4352816, 73.2859306, 72.1258533, 70.9554919, 69.7752885, 68.5856854, 67.3628723, 66.1320696, 64.8942452, 63.650367, 62.4014031, 61.1483213, 59.8920895, 58.6336757, 57.3740479, 56.1141739, 54.8550217, 53.5975593, 52.3427545, 51.0915753, 49.8449897, 48.6039656, 47.4248796, 46.3627783, 45.304353, 44.2469797, 43.1880345, 42.1248935, 41.0549325, 39.9755276, 38.8840548, 37.7778901, 36.6544096, 35.5109892, 34.345005, 33.1538329, 31.934849, 30.6854292, 29.4029497, 27.7410992, 25.8763264, 23.9892106, 22.0882812, 20.1820676, 18.2790989, 16.3879046, 14.5170139, 12.6749562, 10.8702608, 9.11145705, 7.4070742, 5.76564159, 4.19568855, 2.70574439, 1.30433844, 0, 0.86675525, 1.53033385, 2.01023333, 2.32595122, 2.49698504, 2.54283231, 2.48299058, 2.33695736, 2.12423019, 1.86430659, 1.57668408, 1.2808602, 0.99633247, 0.742598422, 0.539155581, 0.405501475, 0.417012835, 0.601169932, 0.851076856, 1.19820525, 1.67402676, 2.31001302, 3.13763568, 4.18836638, 5.49367676, 7.08503847, 8.99392314, 11.2518024, 13.890148, 16.8275962, 20.2498269, 24.2221622, 28.8099244, 33.8181486, 39.4423006, 45.9472558, 53.4764225, 62.1732091, 72.1810238, 83.6432749, 97.2541379, 112.438629, 129.232398, 147.671094, 167.790366, 188.911894, 211.963789, 237.160195, 264.715252, 294.843104, 327.536758, 363.501763, 403.023414, 446.454799, 494.287421, 546.950639, 604.87381, 668.721745, 738.634837, 814.496624, 896.657682, 984.908922, 1078.73654, 1178.1534, 1284.06484, 1396.96653, 1517.58583, 1646.47847, 1783.35069, 1928.38927, 2082.87864, 2245.73902, 2416.7772, 2595.79998, 2782.61415, 3021.26932, 3256.26877, 3476.35859, 3670.28488, 3826.79374, 3883.26542, 3914.05573, 3929.92467, 3969.7588, 4033.1847, 4108.10959, 4194.74606, 4293.30666, 4404.00397, 4527.05056, 4759.22283, 5015.79212, 5237.06649, 5363.354, 5334.96271, 5092.20067, 4575.37596, 3822.38919, 3163.57458}, 
{110.497277, 110.22292, 109.919687, 109.587968, 109.228154, 108.840635, 108.425804, 107.984049, 107.515762, 107.021334, 106.501155, 105.955615, 105.385106, 104.790019, 104.170743, 103.52767, 102.86119, 102.171694, 101.459573, 100.725217, 99.9690167, 99.1913633, 98.3926473, 97.5732592, 96.7335899, 95.87403, 94.9949701, 94.0968009, 93.1799132, 92.2446976, 91.2915448, 90.3208454, 89.3329902, 88.3283698, 87.3054671, 86.2656668, 85.2103765, 84.1400489, 83.0551365, 81.956092, 80.8433679, 79.7174169, 78.5786915, 77.4276443, 76.264728, 75.090395, 73.9050981, 72.7092898, 71.5034228, 70.2879495, 69.0633226, 67.8047055, 66.5383883, 65.2653717, 63.9866565, 62.7032436, 61.4161337, 60.1263277, 58.8348264, 57.5426306, 56.2507411, 54.9601587, 53.6718843, 52.3869186, 51.1062626, 49.8309169, 48.5618824, 47.3545238, 46.2629005, 45.1758919, 44.0909747, 43.0056254, 41.9173208, 40.8235374, 39.7217518, 38.6094407, 37.4840808, 36.3431485, 35.1841206, 34.0044737, 32.8016844, 31.5732294, 30.3165852, 29.0292285, 27.3728942, 25.5199918, 23.6469658, 21.7621883, 19.8740316, 17.9908675, 16.1210683, 14.2730059, 12.4550526, 10.6755803, 8.94296119, 7.26556734, 5.65177084, 4.10994377, 2.64845821, 1.27568626, 0, 0.65939231, 1.15760122, 1.51147613, 1.73786646, 1.85362159, 1.87559094, 1.82062391, 1.7055699, 1.54727832, 1.36259857, 1.16838006, 0.981472181, 0.818724348, 0.696985961, 0.633106425, 0.643935142, 0.872310209, 1.35307941, 1.81811774, 2.31127216, 2.87638967, 3.55731724, 4.39790186, 5.44199049, 6.73343014, 8.31606776, 10.2337504, 12.5303249, 15.2496384, 18.4249918, 22.114645, 26.3656086, 31.2248935, 36.3354352, 41.9462821, 48.4142358, 55.9075297, 64.5943972, 74.6430715, 86.2217859, 100.31395, 116.024523, 133.362249, 152.335869, 172.954126, 194.125457, 217.233986, 242.563532, 270.397913, 301.020949, 334.824768, 371.852501, 411.857186, 454.999483, 502.272283, 554.294822, 611.686336, 675.529882, 745.875463, 822.267098, 905.058352, 993.84752, 1087.81008, 1186.84493, 1292.07674, 1404.06754, 1523.64415, 1651.43724, 1787.10656, 1930.86108, 2084.59704, 2246.57653, 2416.42443, 2593.76565, 2778.22508, 3010.35025, 3238.61275, 3452.40683, 3641.12672, 3794.16665, 3852.42773, 3886.12607, 3905.43868, 3947.28227, 4011.24938, 4085.79099, 4171.05671, 4267.19616, 4374.35893, 4492.69465, 4814.39788, 5092.49869, 5286.1988, 5354.69993, 5257.20377, 4952.91206, 4401.0265, 3705.2248, 3109.46146}, 
{112.72494, 112.419836, 112.085681, 111.722864, 111.331775, 110.912802, 110.466333, 109.992759, 109.492467, 108.965846, 108.413287, 107.835176, 107.231904, 106.603859, 105.951429, 105.275005, 104.574974, 103.851726, 103.105649, 102.337133, 101.546565, 100.734336, 99.9008337, 99.0464472, 98.1715654, 97.2765772, 96.3618713, 95.4278368, 94.4748625, 93.5033372, 92.5136498, 91.5061892, 90.4813443, 89.439504, 88.378256, 87.2994487, 86.2049641, 85.0952819, 83.970882, 82.832244, 81.6798478, 80.5141731, 79.3356997, 78.1449074, 76.9422759, 75.7282851, 74.5034146, 73.2681443, 72.022954, 70.7683234, 69.5047322, 68.2063951, 66.9006264, 65.5884753, 64.2709909, 62.9492224, 61.6242188, 60.2970295, 58.9687034, 57.6402898, 56.3128379, 54.9873967, 53.6650154, 52.3467432, 51.0336293, 49.7267227, 48.4270726, 47.1891987, 46.0663366, 44.9492568, 43.8355422, 42.7227756, 41.6085399, 40.4904181, 39.365993, 38.2328474, 37.0885643, 35.9307266, 34.7569171, 33.5647186, 32.3517142, 31.1154866, 29.8536188, 28.5636936, 26.9139539, 25.0735798, 23.2157099, 21.3486147, 19.4805648, 17.6198309, 15.7746835, 13.9533932, 12.1642306, 10.4154664, 8.71537106, 7.07221526, 5.49426958, 3.98980461, 2.56709096, 1.23439923, 0, 1.02843507, 1.83199263, 2.43212471, 2.85028334, 3.10792055, 3.22648836, 3.22743882, 3.13222394, 2.96229577, 2.73910632, 2.48410763, 2.21875173, 1.96449064, 1.74277641, 1.57506105, 1.4827966, 1.52652879, 1.73329444, 2.03611939, 2.46483289, 3.04926417, 3.8192425, 4.80459712, 6.03515728, 7.54075224, 9.35121124, 11.4963635, 14.0060384, 16.910065, 20.1392483, 23.8587509, 28.1281092, 33.0068598, 38.2226157, 44.0008751, 50.6556495, 58.3460523, 67.2311974, 77.4701982, 89.2221685, 103.343023, 119.083004, 136.464895, 155.511479, 176.245538, 197.6079, 220.973792, 246.636487, 274.889257, 306.025374, 340.530473, 378.270353, 418.934866, 462.636246, 510.349086, 562.6608, 620.158798, 683.771895, 753.668509, 829.644619, 912.253705, 1000.98537, 1095.08924, 1194.446, 1300.00568, 1412.22746, 1531.78363, 1659.18865, 1794.17544, 1936.91899, 2088.99826, 2248.96828, 2416.51804, 2591.33654, 2773.11279, 3002.92609, 3228.72755, 3439.85862, 3625.66072, 3775.47528, 3829.48852, 3858.9672, 3874.31929, 3915.03098, 3980.00007, 4056.00841, 4142.5596, 4239.15723, 4345.30489, 4460.50618, 4797.01656, 5068.12454, 5240.421, 5280.4968, 5154.9428, 4830.34989, 4273.30892, 3624.71422, 3066.32564}, 
{114.960181, 114.626463, 114.263392, 113.871356, 113.450743, 113.001944, 112.525345, 112.021336, 111.490305, 110.932641, 110.348732, 109.738968, 109.103737, 108.443427, 107.758427, 107.049126, 106.315912, 105.559174, 104.779301, 103.976681, 103.151703, 102.304755, 101.436227, 100.546506, 99.6359815, 98.705042, 97.7540761, 96.7834725, 95.7936197, 94.7849065, 93.7577214, 92.712453, 91.64949, 90.5692211, 89.4682615, 88.348966, 87.2137345, 86.063078, 84.8975076, 83.7175345, 82.5236696, 81.3164242, 80.0963091, 78.8638356, 77.6195146, 76.3638573, 75.0973748, 73.8205781, 72.5339783, 71.2380865, 69.9334137, 68.5930489, 67.2455197, 65.8919318, 64.5333905, 63.1710016, 61.8058704, 60.4391026, 59.0718037, 57.7050792, 56.3400346, 54.9777755, 53.6194074, 52.2660358, 50.9187663, 49.5787045, 48.2469558, 46.9772959, 45.8222357, 44.6742528, 43.5310383, 42.3902834, 41.2496791, 40.1069167, 38.9596872, 37.8056819, 36.6425919, 35.4681082, 34.2799222, 33.0757249, 31.8532074, 30.610061, 29.3439768, 28.0526458, 26.4109285, 24.5847216, 22.7438023, 20.8963383, 19.050497, 17.2144462, 15.3963535, 13.6043865, 11.8467128, 10.1315002, 8.46691612, 6.86112831, 5.32230438, 3.85861195, 2.47821866, 1.18929213, 0, 1.51091858, 2.71242468, 3.63200244, 4.29713603, 4.73530961, 4.97400733, 5.04071336, 4.96291185, 4.76808696, 4.48372286, 4.13730369, 3.75631362, 3.3682368, 3.00055741, 2.68075958, 2.43632749, 2.2170697, 2.03937383, 2.10641888, 2.42904421, 3.01808923, 3.88439331, 5.03879585, 6.49213623, 8.25525384, 10.3389881, 12.7541783, 15.5116639, 18.6222843, 21.8717426, 25.5785644, 29.82113, 34.6778196, 40.0238215, 46.0391119, 52.9578517, 60.9193789, 70.0630313, 80.5281468, 92.4540634, 106.42944, 122.007543, 139.239801, 158.17764, 178.872489, 200.46768, 224.149758, 250.197173, 278.888378, 310.501822, 345.48861, 383.743518, 425.076152, 469.610557, 518.112748, 571.060508, 628.931622, 692.338462, 761.593634, 836.862919, 918.841671, 1007.08, 1101.1659, 1201.04354, 1307.26071, 1420.08811, 1539.89091, 1666.9643, 1801.25718, 1942.91433, 2093.06596, 2250.71983, 2415.67962, 2587.749, 2766.73167, 2995.10553, 3219.33147, 3428.5446, 3611.88005, 3758.47291, 3806.83385, 3830.47211, 3840.21899, 3879.19247, 3945.44374, 4023.57121, 4112.29871, 4210.35005, 4316.44906, 4429.31956, 4747.35868, 5000.40161, 5156.55085, 5183.90892, 5050.57833, 4724.66158, 4174.2612, 3565.56686, 3030.96067}, 
{117.215076, 116.855786, 116.46656, 116.047795, 115.599887, 115.123232, 114.618229, 114.085272, 113.524759, 112.937086, 112.322651, 111.681849, 111.015078, 110.322734, 109.605214, 108.862915, 108.096232, 107.305564, 106.491306, 105.653855, 104.793608, 103.910961, 103.006311, 102.080056, 101.132591, 100.164313, 99.1756187, 98.1669051, 97.1385687, 96.0910062, 95.0246142, 93.9397894, 92.8369284, 91.716428, 90.5741745, 89.4129147, 88.2354494, 87.0423216, 85.8340743, 84.6112506, 83.3743935, 82.1240461, 80.8607513, 79.5850523, 78.297492, 76.9986135, 75.6889598, 74.3690739, 73.0394989, 71.7007779, 70.3534537, 68.9692323, 67.578119, 66.1812821, 64.7798896, 63.3751098, 61.9681108, 60.5600608, 59.152128, 57.7454805, 56.3412866, 54.9407143, 53.5449319, 52.1551075, 50.7724092, 49.3980054, 48.033064, 46.7307193, 45.5428574, 44.3634709, 43.1903593, 42.0213219, 40.8541582, 39.6866676, 38.5166496, 37.3419036, 36.1602291, 34.9694254, 33.767292, 32.5516283, 31.3202338, 30.070908, 28.8014501, 27.5096598, 25.8776507, 24.0677052, 22.2458998, 20.4202785, 18.5988855, 16.7897647, 15.0009602, 13.240516, 11.5164762, 9.83688478, 8.20978584, 6.64322339, 5.14524148, 3.72388414, 2.38719542, 1.14321936, 0, 1.94997226, 3.51394175, 4.72487874, 5.61575347, 6.21953619, 6.56919716, 6.69770662, 6.63803483, 6.42315203, 6.08602849, 5.65963444, 5.17694014, 4.67091584, 4.17453179, 3.72075825, 3.34256546, 2.89143164, 2.37439959, 2.23495729, 2.46718383, 3.06515831, 4.02295983, 5.3346675, 6.9943604, 8.99611764, 11.3340183, 14.0021415, 16.9945664, 20.305372, 23.5902654, 27.3057674, 31.5474689, 36.4109605, 41.9249128, 48.2183766, 55.438248, 63.7001938, 73.1198806, 83.8129753, 95.8951446, 109.625775, 124.933073, 141.904586, 160.627861, 181.190446, 203.052051, 227.085021, 253.533862, 282.64308, 314.657183, 349.88638, 388.429171, 430.348523, 475.829837, 525.308482, 579.107597, 637.550319, 700.879728, 769.517216, 843.971515, 924.918409, 1012.19986, 1106.0107, 1206.45024, 1313.4559, 1427.03942, 1547.11443, 1673.66725, 1807.04379, 1947.38657, 2095.52897, 2250.81631, 2413.15195, 2582.43924, 2758.58151, 2985.26417, 3207.663, 3414.73586, 3595.44058, 3738.735, 3781.58318, 3799.59926, 3804.02414, 3841.5149, 3909.29353, 3989.82524, 4081.06986, 4180.98719, 4287.53705, 4398.67926, 4676.52381, 4905.51027, 5050.89341, 5077.92803, 4951.86891, 4637.97084, 4101.4886, 3522.92373, 3000.55072}, 
{119.53678, 119.161667, 118.754891, 118.3169, 117.848141, 117.349061, 116.820109, 116.261732, 115.674377, 115.058492, 114.414524, 113.742922, 113.044132, 112.318601, 111.566779, 110.789112, 109.986047, 109.158033, 108.305516, 107.428945, 106.528767, 105.605429, 104.659379, 103.691065, 102.700933, 101.689432, 100.65701, 99.6041126, 98.5311889, 97.4386859, 96.3270511, 95.1967323, 94.0481768, 92.8818323, 91.6946833, 90.4889817, 89.267021, 88.0293611, 86.7765619, 85.5091833, 84.2277852, 82.9329277, 81.6251705, 80.3050737, 78.9731971, 77.6301007, 76.2763443, 74.912488, 73.5390916, 72.1567151, 70.7659183, 69.3356174, 67.8987021, 66.4564183, 65.0100119, 63.5607287, 62.1098148, 60.658516, 59.2080781, 57.7597472, 56.314769, 54.8743895, 53.4398546, 52.0124102, 50.5933022, 49.1837764, 47.7850789, 46.4503274, 45.2313947, 44.0225443, 42.8216594, 41.6266234, 40.4353195, 39.2456309, 38.0554408, 36.8626325, 35.6650893, 34.4606944, 33.247331, 32.0228823, 30.7852317, 29.5322624, 28.2618576, 26.9719005, 25.3531692, 23.563559, 21.7646566, 19.9643113, 18.1703726, 16.3906898, 14.6331124, 12.9054897, 11.2156712, 9.57150619, 7.98084413, 6.45153442, 4.99142645, 3.60836963, 2.31021334, 1.104807, 0, 1.81844633, 3.28004121, 4.41591395, 5.25719383, 5.83501016, 6.18049223, 6.32476934, 6.29897079, 6.13422587, 5.86166388, 5.51241412, 5.11760589, 4.70836847, 4.31583118, 3.97112331, 3.70537414, 3.43382756, 3.17105778, 3.21031593, 3.55789871, 4.22010282, 5.20322497, 6.51356187, 8.1574102, 10.1410667, 12.470828, 15.1529909, 18.1938521, 21.5997082, 25.126656, 29.1229322, 33.6698934, 38.8488964, 44.6521114, 51.2054883, 58.65876, 67.1200393, 76.6974388, 87.499071, 99.6330486, 113.173928, 128.273591, 145.046715, 163.607979, 184.07206, 206.227988, 230.597501, 257.37669, 286.761645, 318.948456, 353.991258, 392.401597, 434.605484, 480.908675, 531.37142, 586.164194, 645.457475, 709.349038, 778.098585, 852.045125, 931.132311, 1016.26593, 1108.96301, 1209.73981, 1317.41683, 1431.59296, 1551.35034, 1676.15391, 1807.36326, 1945.26624, 2091.48519, 2244.76821, 2404.94121, 2571.83015, 2745.26095, 2967.41251, 3185.16955, 3387.76978, 3564.45087, 3704.45054, 3745.90226, 3763.2174, 3767.53545, 3806.10805, 3875.78029, 3958.35057, 4051.41633, 4152.57501, 4259.42405, 4369.5609, 4607.51193, 4818.35988, 4961.82349, 4997.62154, 4885.47279, 4585.096, 4056.20993, 3481.1609, 2962.64006}, 
{121.8803, 121.490585, 121.067347, 120.611085, 120.122305, 119.601508, 119.049197, 118.465874, 117.852043, 117.208206, 116.534866, 115.832526, 115.101687, 114.342854, 113.556529, 112.743213, 111.903411, 111.037625, 110.146358, 109.230111, 108.289389, 107.324693, 106.336527, 105.325392, 104.291793, 103.236231, 102.159209, 101.06123, 99.9427973, 98.8044126, 97.6465789, 96.4697991, 95.2745757, 94.0614116, 92.8284832, 91.5775054, 90.3102207, 89.0272074, 87.7290437, 86.4163077, 85.0895776, 83.7494318, 82.3964483, 81.0312053, 79.6542811, 78.2662539, 76.8677018, 75.4592031, 74.0413359, 72.6146785, 71.179809, 69.7031043, 68.2200854, 66.7320718, 65.2403833, 63.7463394, 62.2512597, 60.7564639, 59.2632715, 57.7730023, 56.2869758, 54.8065116, 53.3329293, 51.8675487, 50.4116892, 48.9666705, 47.5338122, 46.1659939, 44.9148571, 43.675383, 42.4455489, 41.2233319, 40.0067091, 38.7936579, 37.5821555, 36.3701789, 35.1557055, 33.9367124, 32.7111768, 31.477076, 30.232387, 28.9750872, 27.7031537, 26.4145637, 24.8102933, 23.0430673, 21.2691801, 19.4962473, 17.7318843, 15.9837065, 14.2593295, 12.5663687, 10.9124395, 9.30515738, 7.75213784, 6.26099631, 4.83934826, 3.49480915, 2.23499443, 1.06751956, 0, 1.56187369, 2.81830062, 3.79708452, 4.52602911, 5.03293812, 5.34561529, 5.49186433, 5.49948899, 5.39629297, 5.21008003, 4.96865388, 4.69981825, 4.43137688, 4.19113348, 4.0068918, 3.90645555, 3.8947151, 3.99620085, 4.29054437, 4.80063937, 5.54937959, 6.55965874, 7.85437055, 9.45640874, 11.388667, 13.6740391, 16.3354188, 19.3956997, 22.8777756, 26.6698834, 30.9789478, 35.8682596, 41.4011096, 47.5121683, 54.3290367, 62.0136142, 70.6677778, 80.3934041, 91.2923701, 103.466553, 116.818466, 131.710025, 148.282112, 166.67561, 187.0314, 209.49722, 234.205385, 261.295063, 290.905423, 323.175634, 357.85984, 395.952823, 438.262626, 485.230901, 536.551253, 592.253145, 652.36604, 716.903255, 785.914069, 859.465377, 936.681121, 1019.58653, 1111.02048, 1211.92717, 1320.04039, 1434.56755, 1553.78448, 1676.65704, 1805.56707, 1940.96411, 2085.57054, 2237.21347, 2395.55575, 2560.26022, 2730.98972, 2947.0276, 3158.51106, 3355.19779, 3526.84551, 3663.21192, 3704.9696, 3723.68481, 3730.15174, 3770.87794, 3842.52107, 3926.85822, 4021.29107, 4123.22129, 4230.05058, 4339.18061, 4532.06194, 4724.66082, 4869.74106, 4920.06646, 4828.40082, 4547.50795, 4030.15163, 3452.93274, 2930.09592}, 
{124.234194, 123.828122, 123.386855, 122.910942, 122.400935, 121.857384, 121.280838, 120.671848, 120.030965, 119.358737, 118.655717, 117.922453, 117.159497, 116.367398, 115.546707, 114.697974, 113.821749, 112.918582, 111.989024, 111.033625, 110.052935, 109.047505, 108.017884, 106.964623, 105.888273, 104.789382, 103.668503, 102.526184, 101.362977, 100.179431, 98.9760964, 97.7535239, 96.5122636, 95.2528657, 93.9742671, 92.6778586, 91.3650505, 90.0364454, 88.692646, 87.3342547, 85.9618743, 84.5761073, 83.1775563, 81.766824, 80.3445129, 78.9112256, 77.4675647, 76.014133, 74.5515328, 73.0803669, 71.6012379, 70.0788688, 68.5505196, 67.0175706, 65.4814024, 63.9433951, 62.4049294, 60.8673855, 59.3321439, 57.8005849, 56.274089, 54.7540366, 53.241808, 51.7387838, 50.2463441, 48.7658696, 47.2987406, 45.897027, 44.6115819, 43.3392422, 42.0781022, 40.8262564, 39.581799, 38.3428247, 37.1074276, 35.8737023, 34.6397432, 33.4036445, 32.1635008, 30.9174064, 29.6634558, 28.3997433, 27.1243633, 25.8354102, 24.2459523, 22.5025922, 20.7553583, 19.0115946, 17.2786452, 15.5638542, 13.8745655, 12.2181233, 10.6018715, 9.0331543, 7.51931562, 6.06769956, 4.68565015, 3.38051145, 2.15962749, 1.03034233, 0, 1.2716044, 2.2939432, 3.09125579, 3.6877816, 4.10776001, 4.37543044, 4.51503228, 4.55080495, 4.50698785, 4.40782038, 4.27754194, 4.14039195, 4.0206098, 3.94243491, 3.93010666, 4.00786448, 4.26401725, 4.73195716, 5.29100454, 5.97912796, 6.83429601, 7.89447726, 9.1976403, 10.7817537, 12.6847861, 14.944706, 17.5994819, 20.6870826, 24.2454766, 28.278875, 32.8713815, 38.071092, 43.9261023, 50.3287063, 57.4049008, 65.3222297, 74.1755292, 84.0596359, 95.0693861, 107.299616, 120.48505, 135.190237, 151.580467, 169.821036, 190.077235, 212.858606, 237.900131, 265.281042, 295.080571, 327.377948, 361.625511, 399.295588, 441.530868, 488.900238, 540.801066, 597.156708, 657.890519, 722.989263, 792.298479, 865.594533, 941.290805, 1022.32892, 1112.57691, 1213.32039, 1321.46925, 1435.9419, 1554.40478, 1675.45172, 1802.26718, 1935.4389, 2078.94783, 2229.46649, 2386.40803, 2549.18563, 2717.21247, 2925.4112, 3128.80813, 3317.93907, 3483.33982, 3615.54619, 3659.22539, 3681.36086, 3692.1464, 3735.98957, 3809.54119, 3895.1963, 3990.31849, 4092.27136, 4198.41849, 4306.12348, 4453.29581, 4628.0383, 4776.15646, 4843.45585, 4775.74198, 4518.82039, 4018.49661, 3440.59663, 2906.56469}, 
{126.503785, 126.064996, 125.591248, 125.083065, 124.540973, 123.965498, 123.357165, 122.716501, 122.04403, 121.340279, 120.605773, 119.841037, 119.046597, 118.22298, 117.370709, 116.490312, 115.582313, 114.647239, 113.685615, 112.697966, 111.684819, 110.646698, 109.58413, 108.49764, 107.387753, 106.254996, 105.099893, 103.922972, 102.724756, 101.505772, 100.266546, 99.0076022, 97.7294674, 96.4326669, 95.1135668, 93.7748599, 92.4192891, 91.0475151, 89.6601984, 88.2579996, 86.8415793, 85.4115982, 83.9687169, 82.5135959, 81.0468959, 79.5692774, 78.0814011, 76.5839276, 75.0775176, 73.5628315, 72.04053, 70.477382, 68.9086746, 67.3358031, 65.7601626, 64.1831487, 62.6061565, 61.0305813, 59.4578186, 57.8892636, 56.3263115, 54.7703578, 53.2227978, 51.6850266, 50.1584398, 48.6444325, 47.1444, 45.7077881, 44.3828888, 43.071997, 41.7733932, 40.4853578, 39.2061712, 37.9341138, 36.6674661, 35.4045084, 34.143521, 32.8827846, 31.6205793, 30.3551857, 29.0848841, 27.8079549, 26.5226786, 25.2273355, 23.6501621, 21.9301409, 20.2094998, 18.4952827, 16.7945336, 15.1142961, 13.4616143, 11.8435319, 10.2670928, 8.73934083, 7.26731983, 5.85807367, 4.51864619, 3.25608123, 2.07742264, 0.989714286, 0, 1.2194166, 2.19837856, 2.96109699, 3.53178299, 3.93464766, 4.19390209, 4.3337574, 4.37842467, 4.35211503, 4.27903955, 4.18340936, 4.08943554, 4.0213292, 4.00330145, 4.05956337, 4.21432609, 4.52845981, 5.03141266, 5.6644763, 6.45971734, 7.44920243, 8.66499817, 10.1391712, 11.9037882, 13.9909157, 16.4326204, 19.2609689, 22.5080278, 26.2058639, 30.2960894, 34.9343918, 40.1799739, 46.0920386, 52.6712919, 60.0061852, 68.2007688, 77.3317947, 87.4760149, 98.7101812, 111.111046, 124.176799, 138.738838, 154.987111, 173.111569, 193.302158, 216.420396, 241.816771, 269.513342, 299.532166, 331.895298, 365.843859, 403.135324, 444.992971, 492.035104, 543.644873, 599.759987, 660.318152, 725.354234, 794.686704, 868.028052, 943.843331, 1024.89284, 1114.73827, 1214.57615, 1321.59539, 1434.82444, 1552.11441, 1672.18853, 1798.08701, 1930.40817, 2074.43446, 2225.35936, 2382.15969, 2543.81231, 2709.29404, 2906.10747, 3097.57226, 3275.53383, 3431.83759, 3558.32895, 3606.60732, 3635.59478, 3654.385, 3702.83469, 3777.86052, 3863.56123, 3957.49437, 4057.21752, 4160.28824, 4264.26408, 4372.46237, 4529.68358, 4677.6027, 4757.89478, 4712.23482, 4482.29785, 4009.75888, 3453.65015, 2895.23277}, 
{128.77986, 128.303291, 127.792525, 127.248045, 126.670332, 126.059867, 125.417131, 124.742607, 124.036776, 123.300119, 122.533118, 121.736255, 120.91001, 120.054866, 119.171303, 118.259804, 117.32085, 116.354922, 115.362502, 114.344072, 113.300113, 112.231106, 111.137533, 110.019875, 108.878615, 107.714233, 106.527211, 105.318031, 104.087173, 102.835121, 101.562354, 100.269355, 98.9566047, 97.6245853, 96.2661508, 94.8857567, 93.4879503, 92.0734605, 90.6430166, 89.1973473, 87.7371819, 86.2632493, 84.7762785, 83.2769987, 81.7661387, 80.2444277, 78.7125947, 77.1713687, 75.6214788, 74.0636539, 72.4986231, 70.8968601, 69.2900052, 67.6794432, 66.066559, 64.4527376, 62.8393639, 61.2278228, 59.6194991, 58.0157778, 56.4180439, 54.8276821, 53.2460774, 51.6746147, 50.114679, 48.5676551, 47.0349279, 45.5622546, 44.1943269, 42.8410154, 41.5008285, 40.1722746, 38.8538622, 37.5440997, 36.2414955, 34.9445581, 33.651796, 32.3617174, 31.072831, 29.783645, 28.4926681, 27.1984085, 25.8993747, 24.5940752, 23.0283421, 21.3323366, 19.6392571, 17.9558119, 16.2887091, 14.6446571, 13.0303641, 11.4525385, 9.91788834, 8.43312206, 7.00494788, 5.64007408, 4.34520891, 3.12706065, 1.99233757, 0.947747928, 0, 1.27507146, 2.29623111, 3.08933779, 3.68025033, 4.09482759, 4.35892841, 4.49841163, 4.53913609, 4.50696064, 4.42774411, 4.32734536, 4.23162322, 4.16643654, 4.15764416, 4.23110493, 4.41267768, 4.6818923, 5.05798922, 5.67161896, 6.53871274, 7.67520176, 9.09701724, 10.8200904, 12.8603524, 15.2337345, 17.9561679, 21.0435837, 24.5119133, 28.3770878, 32.4322411, 36.997794, 42.1565171, 47.9911808, 54.6755617, 62.2469271, 70.7182761, 80.1450773, 90.5827991, 102.08691, 114.712879, 127.724245, 142.209428, 158.378839, 176.44289, 196.611991, 220.047096, 245.770438, 273.754794, 303.972939, 336.397649, 370.129747, 407.079682, 448.403185, 494.721183, 545.562716, 600.947061, 660.893492, 725.532521, 794.746906, 868.298059, 945.101206, 1027.16006, 1117.02376, 1215.57439, 1320.86935, 1432.26264, 1548.23361, 1667.90947, 1793.62445, 1925.8987, 2071.16126, 2223.11436, 2380.23276, 2540.99126, 2703.86462, 2887.66416, 3065.44398, 3230.59474, 3376.50709, 3496.57167, 3550.62895, 3587.98654, 3616.41371, 3670.07169, 3746.28295, 3831.37335, 3923.2152, 4019.68081, 4118.6425, 4217.97258, 4294.98411, 4436.21104, 4581.70294, 4671.50937, 4645.67991, 4444.26412, 4007.31157, 3485.68162, 2900.31146}, 
{131.136568, 130.626126, 130.08168, 129.503691, 128.892618, 128.248921, 127.573061, 126.865497, 126.12669, 125.3571, 124.557186, 123.72741, 122.86823, 121.980107, 121.063502, 120.118873, 119.146682, 118.147389, 117.121452, 116.069333, 114.991492, 113.888388, 112.760482, 111.608234, 110.432104, 109.232552, 108.010037, 106.765021, 105.497963, 104.209323, 102.899562, 101.569139, 100.218514, 98.8481479, 97.4483572, 96.024887, 94.5836042, 93.1252981, 91.6507578, 90.1607726, 88.6561318, 87.1376246, 85.6060403, 84.062168, 82.5067971, 80.9407168, 79.3647163, 77.7795848, 76.1861116, 74.585086, 72.9772972, 71.3369774, 69.6920485, 68.0438756, 66.3938236, 64.7432573, 63.0935419, 61.4460422, 59.8021231, 58.1631496, 56.5304868, 54.9054994, 53.2895525, 51.6840109, 50.0902398, 48.5096039, 46.9434683, 45.4334092, 44.0200375, 42.6217852, 41.2374106, 39.8656719, 38.5053272, 37.1551348, 35.8138529, 34.4802396, 33.1530533, 31.8310521, 30.5129943, 29.1976379, 27.8837414, 26.5700628, 25.2553603, 23.9383922, 22.3845294, 20.7141327, 19.0503681, 17.3995677, 15.7680636, 14.1621879, 12.5882726, 11.0526497, 9.56165151, 8.12160994, 6.73885711, 5.41972509, 4.17054597, 2.99765183, 1.90737473, 0.906046764, 0, 1.3443958, 2.41732638, 3.24661991, 3.86010454, 4.28560843, 4.55095973, 4.6839866, 4.7125172, 4.66437968, 4.56740219, 4.4494129, 4.33823997, 4.26171154, 4.24765577, 4.32390082, 4.51827484, 4.71752518, 4.92932557, 5.50061557, 6.42899172, 7.71205058, 9.3473887, 11.3326026, 13.6652889, 16.3430441, 19.3634648, 22.7241474, 26.4226887, 30.456685, 34.4547191, 38.9187065, 43.9569479, 49.6777442, 56.4263688, 64.192636, 72.9031675, 82.5951722, 93.305859, 105.072437, 117.932114, 130.973407, 145.47095, 161.647567, 179.72608, 199.929313, 223.625079, 249.604963, 277.805539, 308.163381, 340.615065, 374.182431, 410.834796, 451.590087, 497.047092, 546.948868, 601.422678, 660.595784, 724.718723, 793.767469, 867.583514, 945.658233, 1029.05886, 1119.09495, 1216.26472, 1319.69159, 1429.13002, 1543.86576, 1663.53174, 1789.47965, 1922.08896, 2068.5898, 2221.45703, 2378.69874, 2538.32304, 2698.33799, 2869.07725, 3033.14196, 3185.45885, 3320.9546, 3434.55593, 3494.30358, 3539.9342, 3577.89705, 3636.60414, 3713.81117, 3798.12264, 3887.75178, 3980.91183, 4075.81602, 4170.67758, 4225.9091, 4353.74858, 4495.08542, 4590.80905, 4581.80888, 4408.97432, 4013.1948, 3531.70662, 2925.92011}, 
{133.721218, 133.201864, 132.6455, 132.05269, 131.423999, 130.759992, 130.061234, 129.32829, 128.561724, 127.762101, 126.929987, 126.065945, 125.170542, 124.244341, 123.287908, 122.301807, 121.286604, 120.242862, 119.171148, 118.072026, 116.946061, 115.793817, 114.615859, 113.412753, 112.185063, 110.933354, 109.658191, 108.360139, 107.039763, 105.697626, 104.334295, 102.950335, 101.546309, 100.122783, 98.6733689, 97.2022538, 95.7137092, 94.2085256, 92.6874935, 91.1514035, 89.601046, 88.0372115, 86.4606906, 84.8722738, 83.2727515, 81.6629142, 80.0435525, 78.4154569, 76.7794178, 75.1362258, 73.4866714, 71.8054806, 70.1200735, 68.4318055, 66.7420321, 65.0521089, 63.3633914, 61.6772352, 59.9949958, 58.3180286, 56.6476893, 54.9853334, 53.3323163, 51.6899937, 50.059721, 48.4428538, 46.8407476, 45.2908449, 43.8297216, 42.3843068, 40.9536165, 39.536667, 38.1324745, 36.740055, 35.3584247, 33.9865998, 32.6235965, 31.2684309, 29.9201192, 28.5776775, 27.240122, 25.9064689, 24.5757343, 23.2469343, 21.7076481, 20.0670576, 18.4368861, 16.8230114, 15.2313112, 13.6676632, 12.137945, 10.6480343, 9.20380897, 7.81114659, 6.47592487, 5.20402153, 4.00131426, 2.87368078, 1.82699877, 0.867145943, 0, 1.30317398, 2.3356967, 3.12607611, 3.70282013, 4.09443669, 4.32943374, 4.4363192, 4.44360101, 4.3797871, 4.2733854, 4.15290385, 4.04685038, 3.98373292, 3.9920594, 4.10033777, 4.33707594, 4.53070661, 4.68115554, 5.2694818, 6.28132007, 7.70230502, 9.51807131, 11.7142536, 14.2764866, 17.1904049, 20.4416432, 24.0158361, 27.8986184, 32.0756247, 36.0731119, 40.5045309, 45.4933298, 51.1629565, 57.9328893, 65.7785612, 74.5964635, 84.4212352, 95.2875152, 107.229942, 120.283156, 133.570388, 148.315068, 164.730155, 183.028605, 203.423375, 227.248276, 253.315199, 281.556889, 311.906092, 344.295552, 377.767531, 414.233624, 454.557462, 499.302914, 548.421842, 602.140884, 660.686683, 724.471539, 793.494236, 867.551016, 946.446284, 1030.67491, 1120.69462, 1216.76397, 1318.80417, 1426.89127, 1541.05031, 1661.34413, 1788.02259, 1921.22976, 2067.97302, 2220.47724, 2376.51071, 2533.84175, 2690.23867, 2850.33847, 3003.32358, 3145.24514, 3272.15428, 3380.10213, 3442.99836, 3493.79655, 3538.037, 3600.3633, 3678.75407, 3763.22826, 3852.16247, 3943.93331, 4036.9174, 4129.49134, 4181.26721, 4301.20767, 4435.25009, 4529.33184, 4529.39027, 4381.36275, 4031.18664, 3585.97283, 2982.41138}, 
{136.633402, 136.143578, 135.608855, 135.030096, 134.408161, 133.743913, 133.038214, 132.291925, 131.505909, 130.681027, 129.818141, 128.918113, 127.981804, 127.010077, 126.003794, 124.963815, 123.891004, 122.786222, 121.650331, 120.484192, 119.288667, 118.064619, 116.812909, 115.534399, 114.229951, 112.900427, 111.546688, 110.169596, 108.770014, 107.348803, 105.906824, 104.444941, 102.964014, 101.464905, 99.9534323, 98.4278753, 96.8864542, 95.3298701, 93.7588239, 92.1740167, 90.5761495, 88.9659234, 87.3440393, 85.7111982, 84.0681013, 82.4154495, 80.7539438, 79.0842852, 77.4071749, 75.7233137, 74.0334028, 72.3072297, 70.577079, 68.8443219, 67.1103295, 65.376473, 63.6441236, 61.9146524, 60.1894306, 58.4698294, 56.7572199, 55.0529734, 53.3584608, 51.6750535, 50.0041226, 48.3470392, 46.7051746, 45.1123103, 43.6014498, 42.1071201, 40.6285913, 39.1651337, 37.7160173, 36.2805122, 34.8578885, 33.4474164, 32.0483661, 30.6600076, 29.281611, 27.9124466, 26.5517843, 25.1988944, 23.853047, 22.5135122, 20.9930005, 19.3879859, 17.7971847, 16.2259212, 14.6795196, 13.1633045, 11.6826, 10.2427306, 8.84902062, 7.50679436, 6.22137618, 4.99809042, 3.84226143, 2.75921355, 1.75427112, 0.83275849, 0, 1.0678891, 1.8996743, 2.52213285, 2.962042, 3.246179, 3.4013211, 3.45424556, 3.43172963, 3.36055054, 3.26748557, 3.17931195, 3.12280694, 3.12474779, 3.21191174, 3.41107606, 3.74901799, 4.06408446, 4.35613408, 5.07815559, 6.21654829, 7.75771151, 9.68804458, 11.9939468, 14.6618175, 17.6780561, 21.0290618, 24.7012339, 28.6809718, 32.9546749, 37.0661323, 41.6066438, 46.6953918, 52.4515585, 59.2045717, 66.968491, 75.7013107, 85.4591394, 96.2980859, 108.274259, 121.443767, 135.270035, 150.582238, 167.552444, 186.352722, 207.155141, 230.925104, 256.843011, 284.882598, 315.0176, 347.221751, 380.679207, 417.118326, 457.277243, 501.682527, 550.428793, 603.804595, 662.098484, 725.926692, 795.175623, 869.374215, 948.110444, 1032.05063, 1121.62866, 1217.15769, 1318.74629, 1426.59696, 1541.21331, 1662.87591, 1790.76115, 1924.67001, 2069.92837, 2219.96232, 2372.66037, 2525.91104, 2677.60283, 2831.27718, 2977.75645, 3113.51592, 3235.03088, 3338.77661, 3400.57055, 3451.31772, 3496.34676, 3559.93198, 3639.90269, 3726.19421, 3817.03053, 3910.63569, 4005.2337, 4099.04859, 4171.45839, 4290.90963, 4413.6728, 4496.01838, 4494.21688, 4364.53879, 4063.25461, 3644.13305, 3077.02945}, 
{139.66004, 139.20204, 138.690888, 138.127756, 137.513815, 136.850235, 136.138188, 135.378845, 134.573377, 133.722954, 132.828749, 131.891931, 130.913671, 129.895142, 128.837513, 127.741957, 126.609643, 125.441743, 124.239427, 123.003868, 121.736236, 120.437701, 119.109435, 117.752609, 116.368395, 114.957962, 113.522482, 112.063125, 110.581064, 109.077469, 107.553511, 106.010361, 104.449189, 102.871168, 101.294657, 99.7118717, 98.1148194, 96.5041137, 94.8803676, 93.2441943, 91.5962071, 89.937019, 88.2672434, 86.5874932, 84.8983818, 83.2005223, 81.4945279, 79.7810117, 78.060587, 76.3338669, 74.6014645, 72.8281677, 71.0511917, 69.2719262, 67.4917611, 65.7120862, 63.9342913, 62.1597661, 60.3899005, 58.6260843, 56.8697073, 55.1221593, 53.3848301, 51.6591094, 49.9463872, 48.2480531, 46.5654971, 44.9289962, 43.3681342, 41.8247228, 40.298279, 38.78832, 37.294363, 35.8159252, 34.3525236, 32.9036755, 31.468898, 30.0477082, 28.6396232, 27.2441603, 25.8608366, 24.4891692, 23.1286753, 21.778872, 20.277684, 18.7088237, 17.1579453, 15.6298102, 14.1291793, 12.660814, 11.2294752, 9.83992416, 8.49692199, 7.20522981, 5.96960877, 4.79482001, 3.68562465, 2.64678383, 1.6830587, 0.799210374, 0, 0.82170609, 1.44444785, 1.89295243, 2.19194702, 2.36615876, 2.44031483, 2.43914238, 2.38736859, 2.30972061, 2.23092562, 2.17571077, 2.16880324, 2.23493018, 2.39881875, 2.68519614, 3.11878949, 3.5702416, 4.0422676, 4.9081188, 6.15950429, 7.78813316, 9.78571448, 12.1439574, 14.8545709, 17.9092641, 21.2997461, 25.0177261, 29.054913, 33.4030159, 37.665436, 42.3645699, 47.6086192, 53.5057854, 60.252647, 67.913217, 76.5279412, 86.1785079, 96.9466056, 108.913923, 122.162147, 136.539749, 152.432615, 169.968063, 189.273412, 210.475979, 234.103668, 259.783066, 287.541344, 317.405674, 349.403228, 382.89861, 419.391364, 459.512669, 503.74944, 552.294061, 605.471153, 663.605339, 727.44834, 796.80061, 870.996681, 949.416607, 1032.94474, 1122.11667, 1217.32629, 1318.72747, 1426.58429, 1541.76169, 1664.67952, 1793.55432, 1927.84894, 2070.90902, 2217.71728, 2366.39255, 2515.05363, 2661.81936, 2810.65877, 2952.3779, 3083.63301, 3201.08037, 3301.37624, 3360.98372, 3410.21434, 3454.37858, 3518.16252, 3599.20712, 3687.0802, 3779.82637, 3875.49026, 3972.11651, 4067.74974, 4169.99219, 4291.99296, 4402.35578, 4469.68439, 4462.58254, 4349.65394, 4099.50235, 3711.04734, 3196.64969}, 
{142.743311, 142.307924, 141.812207, 141.257595, 140.645522, 139.977423, 139.254732, 138.478884, 137.651313, 136.773454, 135.846742, 134.87261, 133.852493, 132.787826, 131.680043, 130.530579, 129.340868, 128.112345, 126.846443, 125.544599, 124.208246, 122.838818, 121.43775, 120.006477, 118.546433, 117.059053, 115.545771, 114.008021, 112.447238, 110.864857, 109.262312, 107.641038, 106.002468, 104.348038, 102.706201, 101.064313, 99.409406, 97.7420387, 96.0627685, 94.372153, 92.6707499, 90.9591169, 89.2378117, 87.507392, 85.7684154, 84.0214397, 82.2670226, 80.5057217, 78.7380948, 76.9646994, 75.1860935, 73.363631, 71.5379231, 69.7103773, 67.8824012, 66.0554022, 64.230788, 62.4099661, 60.594344, 58.7853293, 56.9843295, 55.1927522, 53.412005, 51.6434953, 49.8886307, 48.1488187, 46.425467, 44.7455568, 43.1354555, 41.5438265, 39.9704197, 38.4149846, 36.877271, 35.3570286, 33.8540071, 32.3679562, 30.8986255, 29.4457649, 28.0091239, 26.5884524, 25.1834999, 23.7940162, 22.4197511, 21.0604541, 19.5780427, 18.0440818, 16.531793, 15.0453991, 13.5891226, 12.1671862, 10.7838124, 9.44322398, 8.14964346, 6.9072935, 5.72039672, 4.59317575, 3.52985322, 2.53465176, 1.61179398, 0.76550252, 0, 0.615368177, 1.06345593, 1.36711707, 1.5492054, 1.63257475, 1.64007891, 1.59457171, 1.51890694, 1.43593843, 1.36851998, 1.33950541, 1.37174852, 1.48810314, 1.71142305, 2.06456209, 2.57037406, 3.14786174, 3.80504308, 4.79967304, 6.13235162, 7.80367885, 9.81425476, 12.1646793, 14.8555526, 17.8874747, 21.2610454, 24.9768649, 29.0355333, 33.4376504, 37.8777685, 42.7747531, 48.2210183, 54.3089785, 61.087461, 68.6706737, 77.1844476, 86.7342735, 97.4256416, 109.364043, 122.654967, 137.49311, 153.867608, 171.866499, 191.577819, 213.089607, 236.510298, 261.902431, 289.348945, 318.932779, 350.736868, 384.314037, 420.925258, 461.130605, 505.380174, 553.899522, 607.015018, 665.053033, 728.776869, 797.976661, 871.965893, 950.063786, 1033.25684, 1122.17279, 1217.25204, 1318.6175, 1426.5378, 1542.09808, 1665.77867, 1795.06606, 1929.14025, 2069.43592, 2212.53146, 2356.82637, 2500.72017, 2642.61235, 2787.77695, 2926.02031, 3054.02332, 3168.46683, 3266.03173, 3323.11419, 3370.20292, 3412.67223, 3475.78723, 3557.04881, 3645.67001, 3739.56609, 3836.65234, 3934.84402, 4032.05641, 4168.69618, 4295.03871, 4392.90203, 4444.10422, 4430.46333, 4333.79741, 4135.92453, 3787.00386, 3338.26168}, 
{145.807159, 145.372063, 144.872239, 144.309268, 143.684733, 143.000218, 142.257304, 141.457575, 140.602613, 139.694001, 138.733322, 137.722158, 136.662092, 135.554706, 134.401584, 133.204309, 131.964462, 130.683626, 129.363385, 128.005321, 126.611016, 125.182054, 123.720016, 122.226487, 120.703048, 119.151282, 117.572771, 115.969099, 114.341849, 112.692602, 111.022942, 109.334452, 107.628713, 105.907309, 104.20292, 102.500926, 100.786333, 99.0597138, 97.3216429, 95.5726933, 93.8134387, 92.0444525, 90.2663084, 88.4795797, 86.68484, 84.8826628, 83.0736217, 81.2582901, 79.4372417, 77.6110499, 75.7802881, 73.9064251, 72.0299868, 70.1523948, 68.2750701, 66.399434, 64.5269078, 62.6589127, 60.7968699, 58.9422007, 57.0963264, 55.260668, 53.4366471, 51.6256846, 49.8292019, 48.0486203, 46.2853609, 44.5634659, 42.9064341, 41.2690331, 39.6512176, 38.0529426, 36.4741629, 34.9148332, 33.3749084, 31.8543434, 30.353093, 28.8711119, 27.408355, 25.9647772, 24.5403333, 23.134978, 21.7486663, 20.3813529, 18.9157812, 17.4129772, 15.9353756, 14.4867349, 13.0708138, 11.6913707, 10.3521643, 9.05695308, 7.80949566, 6.61355057, 5.4728764, 4.3912317, 3.37237503, 2.42006496, 1.53806006, 0.730118882, 0, 0.524622521, 0.896160654, 1.13650768, 1.2675569, 1.31120158, 1.28933501, 1.22385048, 1.13664127, 1.04960066, 0.984621948, 0.96359841, 1.00842333, 1.14099, 1.3831917, 1.75692171, 2.28407333, 2.94089493, 3.74275911, 4.81480356, 6.1691405, 7.81788219, 9.77314084, 12.0470287, 14.651658, 17.599141, 20.9015899, 24.571117, 28.6198344, 33.0598545, 37.6984365, 42.8276584, 48.5210882, 54.852294, 61.7390257, 69.3327606, 77.8265274, 87.3406397, 97.9954112, 109.911155, 123.208186, 138.283653, 154.896779, 173.113714, 193.000607, 214.62361, 237.802191, 262.910851, 290.077409, 319.429686, 351.095501, 384.781202, 421.551213, 461.962807, 506.441283, 555.142485, 608.34324, 666.320371, 729.630028, 798.206227, 871.678268, 949.662218, 1032.89313, 1121.89162, 1217.02384, 1318.39412, 1426.22699, 1541.6539, 1665.13451, 1793.80295, 1926.67445, 2063.78483, 2202.97174, 2342.89712, 2482.22293, 2619.61114, 2761.78214, 2897.32483, 3022.88657, 3135.11469, 3230.65655, 3285.76506, 3331.07829, 3371.98404, 3433.74725, 3513.8877, 3601.64306, 3694.94723, 3791.7341, 3889.93757, 3987.49155, 4158.592, 4289.87861, 4376.18638, 4412.35029, 4393.20536, 4313.58656, 4168.3289, 3874.0066, 3500.67456}, 
{148.599376, 148.113621, 147.565791, 146.957321, 146.289643, 145.564189, 144.782394, 143.945689, 143.055507, 142.113282, 141.120447, 140.078433, 138.988674, 137.852603, 136.671653, 135.447256, 134.180845, 132.873854, 131.527715, 130.143861, 128.723725, 127.268739, 125.780337, 124.259952, 122.709015, 121.128961, 119.521222, 117.887231, 116.22842, 114.546223, 112.842073, 111.117401, 109.373642, 107.612228, 105.854736, 104.092102, 102.315022, 100.524276, 98.7206418, 96.9048995, 95.0778279, 93.2402064, 91.3928139, 89.5364298, 87.6718331, 85.799803, 83.9211187, 82.0365594, 80.1469042, 78.2529323, 76.3554228, 74.4249068, 72.4930673, 70.5613392, 68.6311574, 66.7039568, 64.7811722, 62.8642385, 60.9545907, 59.0536636, 57.162892, 55.283711, 53.4175553, 51.5658599, 49.7300596, 47.9115894, 46.111884, 44.3534771, 42.6583955, 40.9845598, 39.3320372, 37.7008948, 36.0911998, 34.5030193, 32.9364205, 31.3914705, 29.8682365, 28.3667856, 26.8871849, 25.4295017, 23.9938031, 22.5801562, 21.1886281, 19.8192861, 18.3643943, 16.8801888, 15.4242015, 13.9999979, 12.6111436, 11.2612041, 9.95374504, 8.69233193, 7.48053035, 6.32190585, 5.220024, 4.17845036, 3.2007505, 2.29048997, 1.45123433, 0.686549155, 0, 0.86676665, 1.52663418, 2.00450238, 2.32527101, 2.51383987, 2.59510872, 2.59397735, 2.53534554, 2.44411306, 2.34517969, 2.26344522, 2.22380941, 2.25117206, 2.37043293, 2.60649181, 2.98424848, 3.53058695, 4.27069047, 5.22497065, 6.41875247, 7.87736089, 9.6261209, 11.6903575, 14.0953956, 16.8665601, 20.0291762, 23.6085687, 27.6300627, 32.118983, 37.0122411, 42.4559939, 48.5020904, 55.2023796, 62.4273366, 70.3194974, 79.0697645, 88.7843991, 99.5696624, 111.531816, 124.77712, 139.450896, 155.608458, 173.348426, 192.769418, 213.970053, 236.968693, 261.96428, 289.075497, 318.421029, 350.11956, 383.84102, 420.701322, 461.501839, 506.703035, 556.069357, 609.677749, 667.605157, 729.506864, 795.977307, 868.070918, 946.964527, 1031.82387, 1122.14457, 1217.76044, 1319.07835, 1426.24213, 1540.13852, 1661.10393, 1786.75088, 1916.23265, 2049.86288, 2185.45605, 2321.7638, 2457.53781, 2591.52971, 2730.44052, 2863.0852, 2986.22809, 3096.63349, 3191.06573, 3247.03219, 3293.39208, 3335.16271, 3395.0011, 3470.94215, 3553.67086, 3641.58963, 3733.10085, 3826.6069, 3920.51017, 4122.91524, 4259.10561, 4336.04929, 4360.71428, 4340.06859, 4281.08024, 4190.71721, 3990.63517, 3700.27556}, 
{151.484769, 150.92313, 150.3044, 149.629775, 148.900454, 148.117637, 147.282522, 146.396308, 145.460193, 144.475377, 143.443057, 142.364432, 141.240702, 140.073065, 138.862719, 137.610864, 136.318697, 134.987418, 133.618226, 132.212318, 130.770894, 129.295153, 127.786292, 126.245512, 124.67401, 123.072985, 121.443636, 119.787161, 118.10476, 116.397631, 114.666972, 112.913983, 111.139862, 109.345807, 107.538693, 105.716763, 103.878187, 102.023983, 100.155163, 98.2727419, 96.3777349, 94.4711563, 92.5540206, 90.6273421, 88.6921355, 86.7494151, 84.8001957, 82.8454915, 80.8863172, 78.9236872, 76.958616, 74.9730703, 72.9875254, 71.0034087, 69.0221476, 67.0451696, 65.073902, 63.1097722, 61.1542078, 59.2086361, 57.2744846, 55.3531806, 53.4461516, 51.554825, 49.6806283, 47.8249888, 45.9893341, 44.1939497, 42.4586688, 40.7460248, 39.0562227, 37.3894674, 35.7459637, 34.1259167, 32.5295311, 30.957012, 29.4085643, 27.8843928, 26.3847024, 24.9096982, 23.459585, 22.0345677, 20.6348512, 19.2606405, 17.8159706, 16.3511543, 14.9176599, 13.5188133, 12.1579403, 10.8383667, 9.56341835, 8.33642102, 7.16070053, 6.03958269, 4.97639331, 3.9744582, 3.03710317, 2.16765402, 1.36943657, 0.645776629, 0, 1.11300895, 1.98159341, 2.63256116, 3.09271996, 3.3888776, 3.54784185, 3.5964205, 3.56142131, 3.46965207, 3.34792056, 3.22303454, 3.12180181, 3.07103013, 3.09752728, 3.22810104, 3.4895592, 3.96644858, 4.69382542, 5.56789721, 6.62784438, 7.91284738, 9.46208663, 11.3147426, 13.5099957, 16.0870263, 19.0850149, 22.543142, 26.5005879, 30.9965332, 36.0910557, 41.794776, 48.1406051, 55.1614544, 62.7069251, 70.9015834, 79.9188778, 89.8467125, 100.772991, 112.785619, 125.972499, 140.182357, 155.815069, 173.005335, 191.887855, 212.597328, 235.338465, 260.158453, 287.17449, 316.503772, 348.263498, 382.137062, 419.205667, 460.428366, 506.272962, 556.204302, 610.137539, 667.987826, 728.67248, 793.331273, 864.192531, 943.727339, 1029.79501, 1121.082, 1216.97319, 1318.03698, 1424.29853, 1536.28768, 1654.16041, 1776.2222, 1901.82529, 2031.87395, 2163.92959, 2296.8072, 2429.32177, 2560.28829, 2696.3669, 2826.56614, 2947.73971, 3056.74132, 3150.42466, 3207.48875, 3255.04509, 3297.72796, 3355.44087, 3426.77403, 3503.95545, 3585.89345, 3671.49632, 3759.67239, 3849.32996, 4072.17394, 4210.6683, 4280.97078, 4299.23914, 4281.63114, 4244.30453, 4203.41708, 4097.40899, 3891.88039}, 
{154.592035, 153.922288, 153.202676, 152.434097, 151.617448, 150.753625, 149.843525, 148.888046, 147.888084, 146.844536, 145.7583, 144.630272, 143.461349, 142.252428, 141.004406, 139.718181, 138.394648, 137.034705, 135.639249, 134.209177, 132.745386, 131.248772, 129.720233, 128.160666, 126.570967, 124.952034, 123.304764, 121.630052, 119.928798, 118.201896, 116.450245, 114.674741, 112.876281, 111.055763, 109.203422, 107.325711, 105.429208, 103.515157, 101.5848, 99.6393801, 97.6801406, 95.7083241, 93.7251736, 91.731932, 89.729842, 87.7201467, 85.7040889, 83.6829115, 81.6578575, 79.6301696, 77.6010908, 75.5651136, 73.5303777, 71.4982722, 69.4701864, 67.4475094, 65.4316306, 63.4239392, 61.4258244, 59.4386754, 57.4638814, 55.5028317, 53.5569155, 51.627522, 49.7160406, 47.8238603, 45.9523704, 44.1181261, 42.3373185, 40.5800582, 38.8467512, 37.1378036, 35.4536215, 33.7946109, 32.161178, 30.5537288, 28.9726695, 27.418406, 25.8913445, 24.3918911, 22.9204518, 21.4774327, 20.06324, 18.6782796, 17.244988, 15.8043961, 14.398472, 13.0301519, 11.7023723, 10.4180694, 9.1801797, 7.99163942, 6.85538496, 5.77435265, 4.75147885, 3.7896999, 2.89195214, 2.06117192, 1.3002956, 0.612259509, 0, 1.05880507, 1.88523268, 2.50487214, 2.94331276, 3.22614384, 3.37895468, 3.42733458, 3.39687286, 3.31315881, 3.20178174, 3.08833096, 2.99839576, 2.95756545, 2.99142934, 3.12557673, 3.38559692, 3.92448318, 4.78602539, 5.7076372, 6.74220879, 7.94263029, 9.36179186, 11.0525837, 13.0678958, 15.4606186, 18.283642, 21.5898562, 25.4321514, 29.8634178, 35.0412246, 40.8754005, 47.3874319, 54.5988054, 62.3729792, 70.8104538, 80.0538705, 90.1721243, 101.23411, 113.308722, 126.464856, 140.299092, 155.496387, 172.218045, 190.625372, 210.87967, 233.20675, 257.687284, 284.46645, 313.689426, 345.501389, 379.743461, 417.236501, 458.860627, 505.022858, 555.164303, 609.159743, 666.88396, 727.034388, 790.930737, 861.177099, 940.691489, 1026.86411, 1118.18116, 1213.85197, 1314.31107, 1419.43061, 1529.31848, 1643.90794, 1762.26786, 1883.95606, 2010.55483, 2139.28606, 2269.00678, 2398.57404, 2526.84487, 2660.40345, 2788.44789, 2907.90344, 3015.69536, 3108.74889, 3166.81611, 3215.44424, 3258.91843, 3314.66809, 3381.77549, 3453.84612, 3530.23607, 3610.30145, 3693.39835, 3778.88287, 4003.08807, 4138.95052, 4206.78041, 4226.8879, 4219.58316, 4205.17636, 4203.97767, 4179.55883, 4057.26003}, 
{158.269136, 157.416606, 156.52594, 155.597615, 154.632109, 153.629901, 152.591467, 151.517285, 150.407833, 149.263589, 148.085031, 146.872636, 145.626882, 144.348247, 143.037209, 141.694245, 140.319832, 138.91445, 137.478575, 136.012685, 134.517258, 132.992771, 131.439703, 129.858532, 128.249734, 126.613787, 124.95117, 123.26236, 121.547835, 119.808073, 118.04355, 116.254746, 114.442137, 112.606202, 110.718811, 108.795346, 106.851536, 104.888785, 102.908501, 100.912089, 98.9009549, 96.8765056, 94.8401469, 92.7932849, 90.7373257, 88.6736753, 86.60374, 84.5289257, 82.4506386, 80.3702848, 78.2892704, 76.2134229, 74.1396311, 72.0692054, 70.0034559, 67.9436929, 65.8912268, 63.8473676, 61.8134257, 59.7907114, 57.7805349, 55.7842064, 53.8030363, 51.8383347, 49.891412, 47.9635783, 46.056144, 44.179403, 42.3434337, 40.5310179, 38.7428837, 36.9797588, 35.2423711, 33.5314485, 31.8477189, 30.1919101, 28.5647501, 26.9669666, 25.3992876, 23.862441, 22.3571546, 20.8841562, 19.4441739, 18.0379354, 16.6209898, 15.2176047, 13.8525906, 12.5281415, 11.2464515, 10.0097146, 8.82012483, 7.67987613, 6.59116255, 5.5561781, 4.57711681, 3.65617268, 2.79553973, 1.99741197, 1.26398342, 0.597448094, 0, 0.201162701, 0.31539633, 0.358712714, 0.347123681, 0.29664106, 0.223276678, 0.143042364, 0.0719499454, 0.0260112507, 0.0212381079, 0.0736423451, 0.19923579, 0.414030272, 0.734037617, 1.17526965, 1.75373821, 2.72159061, 4.12857231, 5.45657916, 6.77222345, 8.14211746, 9.6328735, 11.3111038, 13.2434208, 15.4964366, 18.1367636, 21.2310141, 24.8458003, 29.0477346, 34.0267367, 39.6808988, 46.0398407, 53.1331825, 60.9200085, 69.4652067, 78.8524744, 89.1325924, 100.356342, 112.574502, 125.837856, 139.674889, 154.817634, 171.419062, 189.632141, 209.60984, 231.065619, 254.701833, 280.781327, 309.566948, 341.32154, 376.449342, 414.898995, 456.844423, 502.454804, 551.889623, 605.312717, 662.887922, 724.476541, 790.613703, 862.164577, 940.278951, 1024.04517, 1113.26342, 1207.67286, 1306.90933, 1410.65609, 1518.60131, 1630.42952, 1745.80729, 1864.41135, 1988.24954, 2114.30883, 2241.45902, 2368.56991, 2494.51129, 2625.49497, 2751.21324, 2868.7004, 2974.99076, 3067.11861, 3125.83359, 3175.3111, 3219.5646, 3274.38696, 3339.12893, 3408.23317, 3481.29583, 3557.91307, 3637.68103, 3720.19588, 3906.68642, 4029.62278, 4103.27817, 4141.9258, 4159.83887, 4171.29059, 4190.55416, 4213.1286, 4171.88208}, 
{161.951839, 160.946857, 159.912523, 158.849006, 157.756475, 156.635097, 155.485042, 154.306477, 153.099572, 151.864494, 150.601413, 149.310497, 147.991914, 146.645832, 145.272421, 143.871849, 142.444284, 140.989895, 139.50885, 138.001319, 136.467468, 134.907467, 133.321485, 131.709689, 130.072249, 128.409333, 126.721109, 125.007745, 123.269412, 121.506276, 119.718506, 117.906271, 116.06974, 114.209081, 112.28233, 110.311609, 108.319544, 106.30767, 104.277525, 102.230642, 100.168559, 98.092811, 96.0049335, 93.9064623, 91.7989333, 89.6838822, 87.5628448, 85.4373568, 83.3089539, 81.1791719, 79.0495467, 76.9379218, 74.8291717, 72.7244784, 70.6250242, 68.5319913, 66.446562, 64.3699184, 62.3032429, 60.2477176, 58.2045247, 56.1748465, 54.1598652, 52.160763, 50.1787222, 48.214925, 46.2705536, 44.3473279, 42.4469549, 40.569508, 38.7161344, 36.8879816, 35.0861969, 33.3119277, 31.5663213, 29.8505252, 28.1656867, 26.5129531, 24.8934718, 23.3083903, 21.7588558, 20.2460158, 18.7710176, 17.3350085, 15.9340472, 14.5719325, 13.2525243, 11.9771352, 10.7470775, 9.56366398, 8.42820697, 7.34201904, 6.30641269, 5.32270042, 4.39219476, 3.51620822, 2.69605329, 1.93304249, 1.22848834, 0.58370334, 0, -0.72127124, -1.37707989, -1.96125116, -2.46761025, -2.88998237, -3.22219272, -3.45806651, -3.59142896, -3.61610525, -3.52592061, -3.31470023, -2.97626932, -2.50445309, -1.89307675, -1.1359655, -0.226944541, 1.16865659, 3.10394063, 4.84205646, 6.4595708, 8.03305036, 9.63906188, 11.3541721, 13.2549477, 15.4179554, 17.919762, 20.8369341, 24.2460386, 28.2236421, 32.9717612, 38.3955145, 44.5338337, 51.4256506, 59.0934404, 67.5843632, 76.9499673, 87.2341213, 98.4806941, 110.733554, 124.036571, 137.964163, 153.172524, 169.797372, 187.974424, 207.839398, 228.588273, 251.531438, 277.039547, 305.48325, 337.233199, 373.229573, 412.577409, 454.726926, 499.609006, 548.135892, 600.779213, 658.0106, 720.980895, 789.328154, 861.949475, 937.917617, 1018.37308, 1104.71868, 1197.35335, 1294.97495, 1397.06218, 1502.86888, 1611.81545, 1724.14668, 1839.64106, 1960.6156, 2083.91969, 2208.45305, 2333.11543, 2456.80657, 2585.83862, 2709.84582, 2825.87479, 2930.97218, 3022.18463, 3080.62619, 3129.8978, 3173.87435, 3227.55388, 3290.40834, 3357.27508, 3427.8578, 3501.86019, 3578.98597, 3658.93883, 3797.67478, 3904.60586, 3985.34901, 4045.52113, 4090.73915, 4126.61997, 4158.78053, 4212.38249, 4237.43807}, 
{165.33743, 164.276512, 163.187006, 162.069043, 160.922759, 159.748286, 158.54576, 157.315314, 156.057083, 154.771199, 153.457797, 152.117012, 150.748977, 149.353825, 147.931692, 146.482711, 145.007016, 143.504741, 141.97602, 140.420987, 138.839776, 137.232521, 135.599355, 133.940414, 132.255831, 130.545739, 128.810274, 127.049568, 125.263757, 123.452973, 121.617351, 119.757024, 117.872128, 115.962796, 113.982821, 111.956484, 109.908619, 107.840864, 105.754856, 103.652234, 101.534635, 99.4036973, 97.2610582, 95.1083557, 92.9472275, 90.7793115, 88.6062453, 86.4296669, 84.251214, 82.0725244, 79.8952358, 77.7518528, 75.6124773, 73.478078, 71.3496235, 69.2280824, 67.1144234, 65.0096152, 62.9146264, 60.8304257, 58.7579817, 56.6982631, 54.6522385, 52.6208765, 50.6051459, 48.6060153, 46.6244534, 44.6501694, 42.6731432, 40.7175648, 38.7851328, 36.8775456, 34.9965018, 33.1437, 31.3208387, 29.5296164, 27.7717318, 26.0488833, 24.3627696, 22.715089, 21.1075403, 19.5418218, 18.0196323, 16.5426702, 15.1536989, 13.833021, 12.560445, 11.3363369, 10.1610627, 9.03498812, 7.95847922, 6.93190186, 5.95562195, 5.0300054, 4.15541812, 3.33222602, 2.56079501, 1.84149099, 1.17467988, 0.560727577, 0, -1.34793107, -2.53370547, -3.55713416, -4.41802812, -5.1161983, -5.65145566, -6.02361116, -6.23247578, -6.27786047, -6.1595762, -5.87743392, -5.43124461, -4.82081922, -4.04596872, -3.10650407, -2.00223623, -0.364452217, 1.85968329, 3.83679324, 5.64603609, 7.36657031, 9.07755434, 10.8581466, 12.7875057, 14.9447899, 17.4091577, 20.2597676, 23.5757781, 27.4363476, 32.0539098, 37.3254804, 43.2902351, 49.9873498, 57.384406, 65.5563769, 74.597327, 84.5679107, 95.5287821, 107.540595, 120.664004, 134.666676, 149.991421, 166.756219, 185.079047, 205.077882, 225.676065, 248.48487, 273.920935, 302.400898, 334.341395, 370.970665, 410.901788, 453.138992, 497.461032, 545.227983, 597.089939, 653.696994, 717.040623, 786.12468, 858.489696, 931.628423, 1008.35767, 1091.51878, 1182.07832, 1277.82611, 1378.01015, 1481.40101, 1587.12294, 1696.05064, 1808.06867, 1925.66013, 2045.71121, 2167.20695, 2289.13239, 2410.47257, 2538.76811, 2662.30956, 2777.94305, 2882.51472, 2972.87071, 3029.20678, 3076.1197, 3117.59146, 3169.19718, 3230.32994, 3295.55236, 3364.49906, 3436.80464, 3512.1037, 3590.03086, 3681.22728, 3771.34063, 3857.48261, 3936.76494, 4006.29931, 4063.19742, 4104.57097, 4176.53959, 4241.61291}, 
{168.065254, 167.12278, 166.138483, 165.11293, 164.04669, 162.940329, 161.794416, 160.609519, 159.386205, 158.125042, 156.826598, 155.49144, 154.120137, 152.713256, 151.271364, 149.79503, 148.284821, 146.741305, 145.16505, 143.556624, 141.916594, 140.245528, 138.543993, 136.812559, 135.051791, 133.262259, 131.444529, 129.59917, 127.726749, 125.827834, 123.902993, 121.952794, 119.977803, 117.97859, 115.920095, 113.821449, 111.70221, 109.564102, 107.40885, 105.238176, 103.053805, 100.85746, 98.6508664, 96.4357472, 94.2138264, 91.9868279, 89.7564756, 87.5244935, 85.2926053, 83.0625351, 80.8360068, 78.6659139, 76.5017014, 74.3439838, 72.193376, 70.0504924, 67.9159477, 65.7903566, 63.6743337, 61.5684936, 59.473451, 57.3898205, 55.3182168, 53.2592545, 51.2135483, 49.1817127, 47.1643624, 45.1340625, 43.0640514, 41.012794, 38.9827233, 36.9762722, 34.9958738, 33.043961, 31.1229669, 29.2353243, 27.3834663, 25.5698258, 23.7968359, 22.0669295, 20.3825396, 18.7460992, 17.1600412, 15.6267987, 14.2409923, 12.9570221, 11.7281801, 10.5539078, 9.43364632, 8.36683713, 7.35292149, 6.39134071, 5.48153612, 4.62294903, 3.81502076, 3.05719262, 2.34890593, 1.68960201, 1.07872217, 0.515707727, 0, -1.25416717, -2.37881959, -3.3710563, -4.22797632, -4.9466787, -5.52426246, -5.95782663, -6.24447025, -6.38129234, -6.36539194, -6.19386809, -5.8638198, -5.37234613, -4.71654608, -3.89351871, -2.90036304, -1.42146429, 0.590751892, 2.43185764, 4.17176403, 5.88038213, 7.627623, 9.48339771, 11.5176173, 13.8001929, 16.4010356, 19.3900564, 22.8371663, 26.8122765, 31.5590665, 36.9099638, 42.8827489, 49.4952024, 56.4332668, 63.8806418, 72.1095169, 81.2372241, 91.3810954, 102.658463, 115.186658, 129.122775, 144.532283, 161.524733, 180.209678, 200.69667, 222.173915, 245.902648, 272.222759, 301.474137, 333.996671, 370.802689, 410.737771, 452.97811, 497.335813, 544.921309, 596.262107, 651.885715, 713.515771, 780.211808, 849.728484, 919.381166, 992.479508, 1072.65933, 1161.11998, 1254.93231, 1353.0622, 1453.68057, 1555.5472, 1660.33683, 1768.07534, 1881.25477, 1997.05542, 2114.64951, 2233.20925, 2351.90684, 2481.48184, 2606.64728, 2723.68355, 2828.87102, 2918.49007, 2969.66903, 3010.60836, 3045.79645, 3093.49801, 3152.74831, 3216.84251, 3285.10291, 3356.85179, 3431.41145, 3508.10417, 3562.1873, 3636.83735, 3723.61979, 3814.10014, 3899.84389, 3972.41653, 4023.38356, 4104.51333, 4169.41828}, 
{170.892876, 170.096424, 169.242107, 168.331034, 167.364316, 166.343063, 165.268387, 164.141398, 162.963205, 161.734921, 160.457654, 159.132517, 157.760619, 156.343071, 154.880983, 153.375466, 151.827631, 150.238588, 148.609448, 146.94132, 145.235317, 143.492547, 141.714123, 139.901154, 138.05475, 136.176023, 134.266083, 132.326041, 130.357006, 128.36009, 126.336403, 124.287056, 122.213159, 120.115822, 117.978275, 115.810944, 113.624472, 111.420551, 109.200871, 106.967124, 104.720999, 102.464188, 100.198382, 97.925272, 95.646548, 93.3639013, 91.0790228, 88.7936034, 86.5093339, 84.2279051, 81.951008, 79.7387223, 77.5330842, 75.3345189, 73.1434513, 70.9603067, 68.7855101, 66.6194867, 64.4626615, 62.3154597, 60.1783063, 58.0516265, 55.9358454, 53.831388, 51.7386795, 49.658145, 47.5902096, 45.4946884, 43.3323716, 41.1874393, 39.0629492, 36.961959, 34.8875264, 32.8427091, 30.8305647, 28.8541509, 26.9165255, 25.0207461, 23.1698704, 21.3669561, 19.6150609, 17.9172424, 16.2765584, 14.6960666, 13.3230988, 12.0855411, 10.9095453, 9.7934871, 8.73574223, 7.73468637, 6.78869525, 5.89614456, 5.05541002, 4.26486734, 3.52289222, 2.82786038, 2.17814751, 1.57212933, 1.00818155, 0.484679867, 0, -1.25722132, -2.40737465, -3.44556416, -4.366894, -5.16646835, -5.83939138, -6.38076724, -6.78570011, -7.04929416, -7.16665354, -7.13288244, -6.943085, -6.59236541, -6.07582782, -5.38857641, -4.52571534, -3.21889646, -1.4255879, 0.26320214, 1.90882355, 3.57262624, 5.31596011, 7.20017505, 9.28662097, 11.6366478, 14.3116053, 17.3728436, 20.8817124, 24.8995618, 29.6672236, 35.0007556, 40.907663, 47.3954511, 53.9688491, 60.8867503, 68.5421215, 77.0933008, 86.6986262, 97.5164359, 109.705068, 123.753565, 139.388912, 156.704742, 175.79469, 196.752393, 219.038857, 243.538502, 270.503117, 300.184493, 332.834422, 369.169188, 408.408376, 449.992485, 493.807909, 540.651407, 590.911002, 644.974717, 704.167947, 767.728305, 833.870817, 900.123642, 969.789238, 1046.68438, 1132.06047, 1222.82202, 1317.86884, 1415.24013, 1513.61257, 1614.81842, 1718.90494, 1827.94348, 1939.64588, 2053.35877, 2168.42882, 2284.20267, 2413.80326, 2539.35687, 2656.76609, 2761.93348, 2850.76164, 2898.51243, 2935.4896, 2966.44175, 3011.08188, 3068.23518, 3130.49165, 3196.98822, 3266.86185, 3339.24946, 3413.288, 3452.24532, 3517.01193, 3598.77482, 3688.72098, 3778.03739, 3857.91105, 3919.52895, 4001.14824, 4057.53694}, 
{174.307038, 173.619954, 172.859481, 172.027313, 171.125145, 170.154672, 169.117587, 168.015586, 166.850362, 165.623612, 164.337029, 162.992307, 161.591142, 160.135229, 158.62626, 157.065933, 155.455939, 153.797976, 152.093736, 150.344915, 148.553208, 146.720308, 144.84791, 142.93771, 140.991401, 139.010678, 136.997236, 134.95277, 132.878973, 130.777541, 128.650168, 126.498549, 124.324378, 122.12935, 119.927087, 117.713068, 115.482631, 113.237084, 110.977733, 108.705886, 106.422851, 104.129935, 101.828446, 99.5196913, 97.2049785, 94.8856152, 92.5629088, 90.2381669, 87.912697, 85.5878067, 83.2648035, 80.968172, 78.6755403, 76.3877134, 74.1054965, 71.8296946, 69.5611127, 67.3005562, 65.0488299, 62.806739, 60.5750887, 58.354684, 56.14633, 53.9508318, 51.7689945, 49.6016233, 47.4495231, 45.2772232, 43.0501235, 40.8438457, 38.6615465, 36.5063827, 34.381511, 32.2900881, 30.2352707, 28.2202156, 26.2480795, 24.3220192, 22.4451912, 20.6207525, 18.8518597, 17.1416694, 15.4933386, 13.9100238, 12.5808412, 11.4120565, 10.3077001, 9.26489402, 8.28076031, 7.3524209, 6.4769978, 5.65161297, 4.87338842, 4.13944612, 3.44690807, 2.79289624, 2.17453263, 1.58893921, 1.03323797, 0.504550909, 0, -2.08829274, -3.95507728, -5.60287051, -7.0341893, -8.25155055, -9.25747111, -10.0544679, -10.6450577, -11.0317576, -11.2170842, -11.2035546, -10.9936857, -10.5899941, -9.99499701, -9.21121112, -8.24115335, -6.78842862, -4.81285208, -2.99305105, -1.26748983, 0.425367248, 2.14705589, 3.95911177, 5.92307057, 8.100468, 10.5528397, 13.3417214, 16.5286488, 20.1751576, 24.386227, 29.1640198, 34.5570386, 40.6137861, 47.1102367, 54.231157, 62.233988, 71.2489906, 81.4064258, 92.8365548, 105.669638, 120.412537, 136.704295, 154.601492, 174.160705, 195.438514, 217.915269, 242.367834, 268.996845, 298.002936, 329.586743, 364.226704, 401.506118, 441.02403, 482.74961, 527.407697, 575.38385, 627.063628, 683.719534, 744.648605, 808.180304, 872.196635, 939.40697, 1012.55478, 1092.67073, 1177.8832, 1267.65275, 1361.27314, 1458.16168, 1558.34739, 1661.51326, 1768.05409, 1876.83159, 1987.28237, 2098.84305, 2210.95023, 2335.51914, 2456.38814, 2569.87419, 2672.29427, 2759.96534, 2811.1795, 2852.29516, 2887.3542, 2932.50767, 2987.00426, 3045.04722, 3106.13753, 3169.77617, 3235.46413, 3302.70238, 3380.75419, 3450.4368, 3515.29151, 3578.85963, 3644.68245, 3716.30128, 3797.2574, 3871.18625, 3933.55122}, 
{178.020724, 177.379316, 176.656519, 175.854305, 174.974647, 174.019517, 172.990889, 171.890734, 170.721026, 169.483738, 168.180841, 166.814309, 165.386114, 163.898229, 162.352626, 160.751279, 159.09616, 157.389241, 155.632495, 153.827895, 151.977414, 150.083024, 148.146697, 146.170407, 144.156127, 142.105828, 140.021483, 137.905066, 135.758548, 133.583903, 131.383103, 129.158121, 126.910929, 124.643501, 122.385424, 120.124282, 117.847329, 115.555639, 113.250289, 110.932356, 108.602917, 106.263047, 103.913823, 101.556322, 99.1916207, 96.8207946, 94.4449206, 92.0650752, 89.6823348, 87.2977759, 84.912475, 82.5253121, 80.1396077, 77.7564859, 75.3770707, 73.0024863, 70.6338568, 68.2723062, 65.9189585, 63.574938, 61.2413686, 58.9193745, 56.6100797, 54.3146083, 52.0340844, 49.769632, 47.5223753, 45.2580292, 42.9431588, 40.6519169, 38.3877229, 36.1539964, 33.954157, 31.791624, 29.6698171, 27.5921558, 25.5620595, 23.5829478, 21.6582402, 19.7913561, 17.9857152, 16.2447369, 14.5718408, 12.9704463, 11.6857417, 10.5951769, 9.57329582, 8.61567217, 7.71787958, 6.87549169, 6.08408216, 5.33922465, 4.63649279, 3.97146025, 3.33970068, 2.73678772, 2.15829503, 1.59979627, 1.05686507, 0.525075098, 0, -2.71730287, -5.14550606, -7.29130362, -9.16138958, -10.762458, -12.1012028, -13.1843182, -14.0184982, -14.6104367, -14.9668278, -15.0943656, -14.9997441, -14.6896574, -14.1707994, -13.4498642, -12.5335459, -11.07893, -9.04276627, -7.22253017, -5.54999956, -3.95695237, -2.37516651, -0.736419909, 1.02750952, 2.98484384, 5.20380516, 7.75261554, 10.6994971, 14.1126718, 18.0464713, 22.5881015, 27.8099514, 33.7844104, 40.5141627, 48.10645, 56.6871021, 66.3494197, 77.1867035, 89.2922542, 102.759373, 118.071616, 134.813255, 153.001235, 172.652503, 193.784005, 215.659564, 239.23753, 264.72313, 292.321589, 322.238136, 354.946947, 390.055581, 427.12204, 466.120258, 507.873365, 552.82322, 601.411681, 654.990226, 712.884362, 773.427287, 834.818965, 899.186621, 968.252461, 1042.81735, 1122.12098, 1206.11966, 1295.14954, 1389.26542, 1487.12923, 1588.19078, 1691.85397, 1797.62157, 1904.9593, 2013.33286, 2122.20794, 2242.08348, 2358.63364, 2468.56583, 2568.58746, 2655.40593, 2710.27728, 2755.66117, 2794.88698, 2840.50685, 2892.19951, 2945.80091, 3001.18192, 3058.21342, 3116.76628, 3176.7114, 3291.17495, 3365.2969, 3413.79607, 3451.39129, 3492.8014, 3552.74523, 3645.94162, 3709.6625, 3774.95039}, 
{182.127512, 181.042207, 179.925705, 178.777867, 177.598551, 176.387619, 175.144929, 173.870342, 172.563718, 171.224916, 169.853796, 168.450219, 167.014043, 165.54513, 164.043339, 162.508529, 160.940561, 159.339294, 157.704589, 156.036305, 154.334302, 152.59844, 150.828579, 149.024579, 147.186299, 145.3136, 143.406341, 141.464383, 139.487584, 137.475806, 135.428908, 133.346749, 131.22919, 129.076091, 126.807067, 124.463789, 122.08889, 119.684834, 117.254086, 114.799108, 112.322366, 109.826322, 107.313442, 104.786188, 102.247024, 99.6984159, 97.142826, 94.5827185, 92.0205575, 89.4588069, 86.8999307, 84.4366436, 81.9792022, 79.528114, 77.0838865, 74.6470271, 72.2180434, 69.7974426, 67.3857325, 64.9834203, 62.5910136, 60.2090199, 57.8379465, 55.4783011, 53.130591, 50.7953238, 48.4730068, 46.1003164, 43.6154615, 41.1505965, 38.7103668, 36.2994178, 33.9223949, 31.5839434, 29.2887088, 27.0413364, 24.8464717, 22.7087599, 20.6328465, 18.6233768, 16.6849964, 14.8223504, 13.0400844, 11.3428436, 10.0332267, 8.96063539, 7.97089025, 7.05896755, 6.21984352, 5.44849439, 4.7398964, 4.08902577, 3.49085874, 2.94037154, 2.43254041, 1.96234158, 1.52475127, 1.11474573, 0.727301179, 0.357393859, 0, 0.665886977, 1.04867918, 1.17523846, 1.07242666, 0.767105627, 0.286137221, -0.343616715, -1.09529433, -1.94203378, -2.85697321, -3.81325077, -4.78400462, -5.7423729, -6.66149376, -7.51450537, -8.27454586, -8.40996037, -7.82177378, -7.62491795, -7.68436109, -7.86507143, -8.03201719, -8.05016658, -7.78448784, -7.09994917, -5.86151879, -3.93416493, -1.18285581, 2.52744035, 8.10385892, 14.6262235, 21.9979346, 30.1223932, 38.11136, 46.2640556, 55.0908047, 64.7324998, 75.3300337, 87.024299, 99.9561884, 115.331012, 131.901291, 149.599664, 168.358768, 188.111239, 205.868123, 225.214046, 246.812045, 271.325153, 299.416406, 333.695602, 370.499637, 407.441852, 443.692791, 481.602287, 522.09274, 566.086547, 616.462458, 671.741894, 728.312074, 782.143434, 838.062228, 900.926898, 972.486851, 1049.2234, 1130.036, 1212.65905, 1295.68995, 1381.99798, 1472.036, 1570.041, 1672.09953, 1777.35457, 1884.94906, 1994.02599, 2115.24603, 2233.35501, 2344.61644, 2445.29388, 2531.65085, 2582.48341, 2623.16756, 2657.45289, 2700.81776, 2752.36913, 2806.32697, 2862.04258, 2918.86729, 2976.15238, 3033.24917, 3116.97587, 3174.09269, 3215.34169, 3251.4649, 3293.20439, 3351.30219, 3436.50036, 3495.07044, 3531.30989}, 
{189.757548, 189.648749, 189.400061, 189.01538, 188.4986, 187.853615, 187.084322, 186.194614, 185.188387, 184.069535, 182.841954, 181.509538, 180.076182, 178.545781, 176.922231, 175.209425, 173.411258, 171.531627, 169.574425, 167.543547, 165.442888, 163.276344, 161.047809, 158.761178, 156.420345, 154.029206, 151.591656, 149.111589, 146.592901, 144.039486, 141.455239, 138.844055, 136.209829, 133.556456, 130.993478, 128.469744, 125.932833, 123.383212, 120.821347, 118.247705, 115.662752, 113.066955, 110.460779, 107.844692, 105.21916, 102.584648, 99.9416246, 97.2905549, 94.6319057, 91.9661433, 89.2937343, 86.5866663, 83.8745018, 81.1583246, 78.4392185, 75.7182673, 72.9965548, 70.2751646, 67.5551807, 64.8376869, 62.1237668, 59.4145043, 56.7109832, 54.0142872, 51.3255002, 48.6457059, 45.9759881, 43.2139365, 40.2596256, 37.3275881, 34.4256168, 31.5615046, 28.7430445, 25.9780292, 23.2742517, 20.6395049, 18.0815816, 15.6082748, 13.2273773, 10.9466821, 8.77398191, 6.71706977, 4.78373851, 2.98178102, 1.98445157, 1.45281526, 1.03993889, 0.732019843, 0.515255511, 0.375843278, 0.299980531, 0.273864659, 0.283693047, 0.315663083, 0.355972155, 0.390817649, 0.406396952, 0.388907451, 0.324546534, 0.199511588, 0, -4.64903933, -8.83734084, -12.5799476, -15.8919027, -18.7882493, -21.2840304, -23.3942892, -25.1340687, -26.518412, -27.5623622, -28.2809625, -28.6892558, -28.8022854, -28.6350942, -28.2027254, -27.5202221, -26.0326385, -23.6735907, -21.7473824, -20.146916, -18.7650944, -17.49482, -16.2289955, -14.8605236, -13.2823068, -11.3872479, -9.06824935, -6.21821386, -2.73004405, 1.93412659, 7.29267761, 13.3234756, 20.0043873, 26.998163, 34.4402273, 42.5500362, 51.3999913, 61.062494, 71.6099459, 83.1147485, 95.6462289, 109.280799, 124.091461, 140.15122, 157.533076, 178.533713, 200.446535, 222.788625, 245.077068, 266.828947, 285.141836, 304.909506, 329.379907, 359.918448, 394.047011, 431.013144, 470.064393, 508.407677, 547.794951, 592.204314, 645.73879, 704.237779, 764.625893, 825.512241, 888.360222, 953.323067, 1020.40409, 1089.71766, 1161.92784, 1237.38776, 1318.06813, 1402.45563, 1490.34348, 1581.52487, 1675.793, 1787.35325, 1897.98359, 2003.87419, 2101.21519, 2186.19675, 2240.04693, 2283.89271, 2320.33657, 2355.85554, 2391.51843, 2426.77965, 2462.78875, 2500.69526, 2541.6487, 2586.79862, 2665.46258, 2738.03981, 2804.1781, 2863.52523, 2915.72898, 2960.43713, 2997.29746, 3107.20523, 3128.99753}, 
{192.62829, 192.49083, 192.217802, 191.812901, 191.279817, 190.622246, 189.84388, 188.948411, 187.939534, 186.82094, 185.596325, 184.269379, 182.843797, 181.323272, 179.711497, 178.012165, 176.228968, 174.365601, 172.425756, 170.413127, 168.331406, 166.184286, 163.975461, 161.708625, 159.387469, 157.015687, 154.596973, 152.135018, 149.633518, 147.096164, 144.526649, 141.928667, 139.305911, 136.662075, 134.097805, 131.566278, 129.019506, 126.458035, 123.882412, 121.293184, 118.690897, 116.076098, 113.449334, 110.811152, 108.162098, 105.50272, 102.833563, 100.155175, 97.4681019, 94.7728912, 92.0700894, 89.3380613, 86.6000164, 83.8569823, 81.1099866, 78.360057, 75.608221, 72.8555063, 70.1029405, 67.3515511, 64.6023658, 61.8564121, 59.1147178, 56.3783103, 53.6482174, 50.9254665, 48.2110854, 45.4002746, 42.3907758, 39.4018766, 36.441465, 33.5174289, 30.6376562, 27.8100349, 25.0424529, 22.3427982, 19.7189587, 17.1788223, 14.730277, 12.3812106, 10.1395112, 8.01306666, 6.00976492, 4.13749392, 3.13154623, 2.62080748, 2.22542047, 1.92966748, 1.71783075, 1.57419253, 1.4830351, 1.4286407, 1.39529159, 1.36727002, 1.32885826, 1.26433857, 1.15799319, 0.994104386, 0.756954416, 0.430825535, 0, -6.87835047, -13.0519482, -18.5432954, -23.3748943, -27.5692472, -31.1488564, -34.1362239, -36.5538522, -38.4242434, -39.7698997, -40.6133235, -40.9770169, -40.8834822, -40.3552216, -39.4147374, -38.0845318, -36.2288395, -33.8475529, -31.3211606, -28.6382505, -25.7874102, -22.7572275, -19.5362902, -16.1131859, -12.4765025, -8.6148276, -4.51674907, -0.170854594, 4.43426806, 9.07939199, 14.0911363, 19.550105, 25.5369022, 32.3650855, 39.9987822, 48.3399983, 57.3994517, 67.1878602, 77.7159418, 88.9944142, 100.686655, 113.256435, 126.78243, 141.343314, 157.017765, 175.268931, 194.444896, 214.278217, 234.501451, 254.847153, 273.612385, 293.719698, 317.003128, 344.279395, 374.2882, 406.701783, 441.192383, 476.118868, 512.765344, 553.84869, 602.574171, 655.546252, 709.617433, 763.660818, 819.973313, 879.280243, 942.931593, 1011.81464, 1084.52623, 1160.95883, 1240.86171, 1324.29251, 1411.19325, 1501.50591, 1595.17252, 1708.53075, 1821.028, 1928.50738, 2026.81198, 2111.78488, 2162.58609, 2202.86383, 2235.61108, 2270.37211, 2307.54666, 2344.80483, 2382.68285, 2421.71698, 2462.44348, 2505.39859, 2587.15934, 2649.29956, 2697.25676, 2736.46849, 2772.37227, 2810.40563, 2856.00612, 2913.89874, 2963.61026}, 
{202.644418, 202.43914, 202.104309, 201.642974, 201.058186, 200.352995, 199.530452, 198.593605, 197.545507, 196.389205, 195.127752, 193.764197, 192.30159, 190.742981, 189.091421, 187.34996, 185.521647, 183.609534, 181.61667, 179.546105, 177.40089, 175.184075, 172.89871, 170.547845, 168.13453, 165.661816, 163.132752, 160.550389, 157.917778, 155.237967, 152.514008, 149.74895, 146.945844, 144.10774, 141.296455, 138.48442, 135.643358, 132.774413, 129.878729, 126.957447, 124.011712, 121.042666, 118.051453, 115.039215, 112.007096, 108.956239, 105.887786, 102.802882, 99.7026678, 96.5882881, 93.4608855, 90.295565, 87.1200723, 83.9361149, 80.7454003, 77.549636, 74.3505297, 71.1497888, 67.9491209, 64.7502335, 61.5548342, 58.3646304, 55.1813298, 52.0066399, 48.8422681, 45.6899222, 42.5513095, 39.2822013, 35.7418729, 32.2330144, 28.7667936, 25.3543787, 22.0069376, 18.7356383, 15.5516488, 12.4661371, 9.49027115, 6.63521898, 3.91214858, 1.33222792, -1.09337499, -3.35349217, -5.43695561, -7.33259734, -8.05386084, -8.09778835, -7.97314556, -7.7004177, -7.30009003, -6.79264778, -6.19857619, -5.5383605, -4.83248597, -4.10143782, -3.3657013, -2.64576166, -1.96210413, -1.33521396, -0.785576384, -0.333676651, 0, -5.49097175, -10.4293973, -14.8328726, -18.7189938, -22.1053569, -25.0095581, -27.4491933, -29.4418586, -31.0051502, -32.1566639, -32.913996, -33.2947425, -33.3164994, -32.9968628, -32.3534288, -31.4037933, -29.996337, -28.1244822, -26.1885745, -24.1699495, -22.0499427, -19.8098898, -17.4311263, -14.8949877, -12.1828098, -9.27592804, -6.155678, -2.8033953, 0.799584505, 4.76257641, 8.9803557, 13.4443916, 18.1461534, 22.88521, 27.7489809, 32.8760588, 38.3154829, 44.1162927, 50.3275274, 56.9982263, 64.1202615, 71.8172382, 80.1493807, 89.1769132, 98.9600599, 110.693127, 123.018735, 135.71359, 148.554395, 161.317853, 172.569519, 184.777539, 199.806525, 218.382888, 238.909212, 260.831434, 283.595489, 305.147597, 326.774255, 351.39801, 381.687067, 415.324382, 450.873075, 487.666503, 526.343153, 566.942433, 609.018087, 652.483614, 699.033286, 749.354071, 806.135912, 867.754647, 934.2039, 1005.4773, 1081.56847, 1179.16534, 1277.39366, 1372.07348, 1459.02485, 1534.06783, 1576.36727, 1609.50187, 1636.42963, 1667.55378, 1702.88524, 1739.33272, 1777.06224, 1816.23981, 1857.03145, 1899.60317, 2019.13008, 2086.42187, 2118.48724, 2132.33492, 2144.9736, 2173.41199, 2234.65881, 2303.53122, 2315.11357}
}; 

  const double a4tab[98][200] = {{-1766.7572, -1746.59454, -1726.56578, -1706.66718, -1686.89499, -1667.24545, -1647.71482, -1628.29933, -1608.99525, -1589.79882, -1570.70629, -1551.71391, -1532.81792, -1514.01458, -1495.30014, -1476.67084, -1458.12293, -1439.65266, -1421.25629, -1402.93006, -1384.67021, -1366.473, -1348.33468, -1330.2515, -1312.2197, -1294.23553, -1276.29525, -1258.3951, -1240.53133, -1222.70019, -1204.89792, -1187.12079, -1169.36503, -1151.62689, -1133.74537, -1115.79865, -1097.8668, -1079.95118, -1062.05315, -1044.17405, -1026.31524, -1008.47807, -990.663897, -972.874065, -955.10993, -937.372845, -919.664161, -901.985233, -884.337411, -866.722049, -849.140499, -831.642108, -814.179193, -796.752066, -779.36104, -762.006426, -744.688536, -727.407682, -710.164177, -692.958333, -675.79046, -658.660872, -641.56988, -624.517796, -607.504933, -590.531602, -573.598115, -556.784068, -540.167153, -523.584166, -507.030278, -490.500662, -473.990491, -457.494937, -441.009172, -424.528369, -408.0477, -391.562337, -375.067453, -358.55822, -342.029811, -325.477397, -308.896152, -292.281247, -275.414886, -258.403205, -241.354901, -224.272057, -207.156756, -190.011083, -172.83712, -155.636953, -138.412663, -121.166336, -103.900054, -86.6159006, -69.3159605, -52.0023167, -34.6770529, -17.3422528, 0, 15.5874605, 31.2260846, 46.9499571, 62.7931629, 78.7897866, 94.973913, 111.379627, 128.041013, 144.992156, 162.267141, 179.900052, 197.924975, 216.375994, 235.287193, 254.692659, 274.626474, 295.552045, 317.564787, 339.727705, 362.166879, 385.008392, 408.378325, 432.402762, 457.207783, 482.91947, 509.663906, 537.567173, 566.755352, 597.354525, 629.677491, 663.595153, 699.177577, 736.494832, 774.577317, 814.014934, 855.674826, 899.93896, 947.189304, 997.807825, 1052.17649, 1111.90581, 1175.77531, 1243.92657, 1316.50121, 1393.64081, 1472.96419, 1557.76644, 1648.81985, 1746.89671, 1852.76932, 1967.16894, 2090.95905, 2222.38732, 2361.96984, 2514.8541, 2684.10817, 2872.80013, 3089.68725, 3330.85544, 3586.18425, 3841.24649, 4117.09438, 4433.27118, 4798.65112, 5204.0304, 5648.50333, 6123.68137, 6626.71883, 7182.20718, 7799.21788, 8485.94375, 9252.47006, 10108.1724, 11062.4264, 12124.6076, 13681.8839, 15271.3905, 16808.055, 18206.8049, 19382.5677, 20091.1032, 20512.6184, 20630.2556, 20190.7467, 19270.8129, 18045.6799, 16587.1438, 14967.0007, 13257.0468, 11529.078, 10020.3607, 8629.63832, 7353.5544, 6188.75257, 5131.87645, 4179.56964, 3328.47574, 2598.30101, 1896.87465}, 
{-1661.23266, -1640.96932, -1620.97775, -1601.24897, -1581.77402, -1562.54394, -1543.54977, -1524.78253, -1506.23326, -1487.89301, -1469.75279, -1451.80366, -1434.03664, -1416.44278, -1399.01309, -1381.73864, -1364.61043, -1347.61952, -1330.75694, -1314.01372, -1297.3809, -1280.84951, -1264.4106, -1248.05518, -1231.77431, -1215.55902, -1199.40033, -1183.2893, -1167.21694, -1151.1743, -1135.15242, -1119.14232, -1103.13505, -1087.12164, -1070.73828, -1054.1609, -1037.56968, -1020.96717, -1004.35591, -987.738468, -971.117383, -954.495208, -937.874494, -921.257792, -904.647651, -888.046623, -871.457258, -854.882107, -838.323721, -821.784649, -805.267442, -788.880905, -772.519031, -756.182067, -739.870261, -723.58386, -707.32311, -691.08826, -674.879556, -658.697245, -642.541574, -626.412792, -610.311145, -594.236879, -578.190243, -562.171484, -546.180848, -530.288547, -514.563114, -498.860499, -483.176413, -467.506569, -451.846677, -436.192451, -420.5396, -404.883837, -389.220874, -373.546423, -357.856194, -342.1459, -326.411252, -310.647962, -294.851742, -279.018303, -262.912158, -246.649482, -230.349226, -214.014606, -197.648836, -181.255131, -164.836705, -148.396772, -131.938549, -115.465248, -98.9800839, -82.4862721, -65.9870267, -49.4855623, -32.9850933, -16.4888344, 0, 14.8569198, 29.7147625, 44.6096261, 59.5776086, 74.6548082, 89.8773229, 105.281251, 120.90269, 136.777738, 152.942494, 169.433055, 186.285519, 203.535985, 221.220551, 239.375314, 258.036373, 277.452256, 297.68941, 318.303434, 339.375945, 360.988564, 383.222909, 406.1606, 429.883255, 454.472493, 480.009934, 506.577196, 534.255899, 563.127661, 593.173693, 624.612839, 657.556841, 692.117441, 727.790178, 764.994896, 804.315757, 846.049363, 890.492315, 937.941216, 988.692668, 1043.21002, 1101.57238, 1164.04372, 1230.88802, 1302.36926, 1379.75529, 1462.05525, 1549.28215, 1641.44899, 1738.5688, 1839.69945, 1946.97645, 2056.64946, 2168.87932, 2291.81232, 2430.00943, 2588.03164, 2780.69681, 2999.97795, 3226.65873, 3440.43982, 3663.9879, 3920.42426, 4217.46043, 4536.69692, 4871.71963, 5206.52967, 5532.228, 5875.0598, 6241.3907, 6649.49656, 7092.00043, 7571.14516, 8089.17358, 8648.32856, 9255.30047, 9906.77275, 10603.8764, 11347.7423, 12139.5014, 13168.1639, 14121.7288, 14920.8076, 15102.6546, 14739.6266, 14063.8127, 13139.5829, 12031.3076, 10803.357, 9520.10122, 8369.80381, 7294.9199, 6295.895, 5373.17462, 4527.20425, 3758.42941, 3067.2956, 2491.81541, 1969.09585}, 
{-1563.20972, -1545.28891, -1527.51745, -1509.89087, -1492.40469, -1475.05445, -1457.83566, -1440.74385, -1423.77455, -1406.92328, -1390.18557, -1373.55695, -1357.03293, -1340.60906, -1324.28084, -1308.04382, -1291.8935, -1275.82543, -1259.83512, -1243.91811, -1228.06991, -1212.28605, -1196.56206, -1180.89347, -1165.27579, -1149.70456, -1134.17531, -1118.68355, -1103.22481, -1087.79463, -1072.38851, -1057.002, -1041.63062, -1026.26988, -1010.7578, -995.171968, -979.591885, -964.018191, -948.451522, -932.892516, -917.341812, -901.800047, -886.267859, -870.745886, -855.234766, -839.735137, -824.247636, -808.772903, -793.311573, -777.864287, -762.43168, -746.988135, -731.561115, -716.151828, -700.76148, -685.391279, -670.042432, -654.716146, -639.413627, -624.136084, -608.884724, -593.660752, -578.465377, -563.299806, -548.165245, -533.062902, -517.993984, -503.07351, -488.413768, -473.781234, -459.169738, -444.57311, -429.985177, -415.39977, -400.810718, -386.211849, -371.596993, -356.959979, -342.294636, -327.594793, -312.85428, -298.066925, -283.226557, -268.327006, -252.994623, -237.414705, -221.776968, -206.087164, -190.351048, -174.574376, -158.762901, -142.922379, -127.058562, -111.177207, -95.2840667, -79.3848965, -63.4854506, -47.5914835, -31.7087498, -15.8430038, 0, 14.8505086, 29.5312302, 44.0893059, 58.5718764, 73.0260825, 87.4990651, 102.037965, 116.689923, 131.50208, 146.521577, 161.795555, 177.371154, 193.295515, 209.61578, 226.379088, 243.632582, 261.39395, 279.706127, 298.682868, 318.365003, 338.793362, 360.008775, 382.052073, 404.964084, 428.785639, 453.557568, 479.3207, 506.115867, 533.983897, 562.63634, 592.564042, 623.90662, 656.803686, 691.447945, 727.952467, 766.416164, 806.962724, 849.715834, 894.799182, 942.336455, 990.198043, 1041.44672, 1096.64703, 1156.36353, 1221.16077, 1298.43436, 1380.21002, 1465.34454, 1552.69471, 1641.1173, 1724.8122, 1812.98489, 1908.67802, 2013.97314, 2128.98948, 2254.72731, 2392.1869, 2544.54794, 2710.13595, 2884.89886, 3063.00941, 3252.62597, 3467.07994, 3709.20491, 3964.04645, 4224.81512, 4482.57319, 4729.97426, 4967.54896, 5191.37224, 5391.45539, 5570.86986, 5727.78953, 5860.38829, 5966.84003, 5745.49967, 5569.3148, 5511.41406, 5644.92607, 6042.97947, 7242.7316, 8543.92989, 9820.83353, 10473.9987, 10565.192, 10353.5562, 9890.98934, 9229.38905, 8420.65319, 7516.67954, 6644.99579, 5777.79864, 4933.24222, 4129.48061, 3384.66795, 2716.95833, 2144.50586, 1627.53437, 1184.91263}, 
{-1464.60418, -1448.64276, -1432.76005, -1416.95439, -1401.22412, -1385.56758, -1369.98311, -1354.46905, -1339.02374, -1323.64551, -1308.3327, -1293.08366, -1277.89672, -1262.77022, -1247.70251, -1232.69191, -1217.73676, -1202.83542, -1187.98621, -1173.18748, -1158.43755, -1143.73479, -1129.07751, -1114.46406, -1099.89278, -1085.36201, -1070.87009, -1056.41535, -1041.99614, -1027.61079, -1013.25764, -998.935039, -984.641314, -970.374809, -956.174813, -942.018327, -927.881863, -913.76243, -899.657038, -885.562697, -871.476417, -857.395209, -843.316082, -829.236046, -815.152111, -801.061287, -786.960585, -772.847014, -758.717584, -744.569305, -730.399187, -715.963577, -701.505366, -687.026779, -672.530045, -658.017391, -643.491044, -628.95323, -614.406178, -599.852113, -585.293264, -570.731857, -556.170119, -541.610278, -527.05456, -512.505194, -497.964405, -483.516331, -469.243145, -454.978137, -440.718226, -426.460329, -412.201363, -397.938244, -383.667891, -369.38722, -355.093149, -340.782595, -326.452475, -312.099705, -297.721204, -283.313889, -268.874676, -254.400484, -239.669857, -224.793492, -209.88471, -194.947514, -179.985908, -165.003895, -150.00548, -134.994666, -119.975456, -104.951855, -89.9278661, -74.9074933, -59.8947403, -44.8936107, -29.9081083, -14.9422368, 0, 13.7319382, 27.3661114, 40.942923, 54.5027762, 68.086074, 81.7332199, 95.4846169, 109.380668, 123.461777, 137.768347, 152.340781, 167.219482, 182.444853, 198.057299, 214.097221, 230.605023, 247.847818, 265.898395, 284.284364, 303.094708, 322.41841, 342.344456, 362.961827, 384.359509, 406.626484, 429.851737, 454.124252, 479.533012, 506.167001, 534.32142, 563.803423, 594.640127, 626.858652, 659.771479, 693.763044, 729.408355, 766.948921, 806.626252, 848.681857, 893.357247, 941.143216, 991.956119, 1045.98869, 1103.43367, 1164.48379, 1230.50685, 1300.22676, 1373.5425, 1450.35302, 1530.55731, 1613.08508, 1699.9892, 1792.72752, 1891.94659, 1996.6366, 2106.53141, 2221.36488, 2338.01489, 2459.72037, 2592.83587, 2745.89213, 2909.56452, 3081.33939, 3257.47066, 3432.124, 3600.42359, 3760.69897, 3908.90535, 4029.24508, 4112.56858, 4165.15931, 4160.06038, 4082.78001, 3918.82644, 3653.70791, 2627.74217, 1632.92556, 816.063929, 323.96313, 303.429019, 1648.95758, 3261.20445, 4966.53686, 6002.07698, 6427.43489, 6547.7394, 6410.32463, 6062.5247, 5551.67373, 4925.10585, 4304.14647, 3651.03516, 2987.21362, 2334.12358, 1713.20674, 1145.90481, 653.659527, 180.949463, -226.115744}, 
{-1373.5761, -1360.68588, -1347.72756, -1334.70411, -1321.61851, -1308.47374, -1295.27277, -1282.01859, -1268.71417, -1255.36249, -1241.96653, -1228.52926, -1215.05366, -1201.54271, -1187.99939, -1174.42667, -1160.82753, -1147.20496, -1133.56191, -1119.90139, -1106.22635, -1092.53979, -1078.84467, -1065.14398, -1051.44069, -1037.73777, -1024.03822, -1010.345, -996.661091, -982.989473, -969.333122, -955.695016, -942.078133, -928.485449, -915.161209, -901.982682, -888.824239, -875.681027, -862.548194, -849.420888, -836.294258, -823.16345, -810.023614, -796.869898, -783.697448, -770.501414, -757.276942, -744.019182, -730.723281, -717.384388, -703.997649, -690.168774, -676.290793, -662.367295, -648.401871, -634.398112, -620.359607, -606.289946, -592.192721, -578.071521, -563.929936, -549.771557, -535.599975, -521.418778, -507.231558, -493.041905, -478.85341, -464.745523, -450.795875, -436.851598, -422.911365, -408.973847, -395.037718, -381.101649, -367.164314, -353.224385, -339.280533, -325.331433, -311.375755, -297.412173, -283.43936, -269.455987, -255.460727, -241.452252, -227.306741, -213.086691, -198.856068, -184.617522, -170.373699, -156.127247, -141.880814, -127.637048, -113.398595, -99.1681045, -84.9482232, -70.7415988, -56.5508792, -42.3787118, -28.2277444, -14.1006246, 0, 12.2170632, 24.4832114, 36.827677, 49.2796927, 61.8684911, 74.6233046, 87.573366, 100.747908, 114.176162, 127.887363, 141.910741, 156.27553, 171.010962, 186.14627, 201.710687, 217.733444, 234.651903, 252.553601, 270.544622, 288.741656, 307.261391, 326.220516, 345.735719, 365.923688, 386.901114, 408.784683, 431.691085, 455.737009, 481.039143, 508.173191, 536.62852, 566.384116, 597.41896, 628.650595, 660.588725, 694.026108, 729.260162, 766.588303, 806.307948, 848.716516, 896.336869, 946.563669, 999.258919, 1054.28462, 1111.50278, 1166.97415, 1225.3123, 1287.32953, 1353.83818, 1425.65056, 1505.86857, 1590.21655, 1676.04437, 1762.42284, 1851.93634, 1945.59172, 2044.39585, 2151.79585, 2265.8037, 2381.76928, 2498.24025, 2613.12658, 2728.98687, 2843.52984, 2946.24639, 3030.39954, 3092.56874, 3126.87681, 3115.28638, 3046.63856, 2973.49309, 2811.16962, 2526.45282, 2086.12738, 1456.97797, -455.640717, -2358.15651, -4018.42723, -5204.31069, -5683.66473, -4137.80084, -2145.48737, 59.755175, 1488.21783, 2205.70467, 2593.09826, 2700.44875, 2577.80628, 2275.22098, 1842.743, 1383.10257, 907.74269, 426.300494, -51.5868882, -516.282327, -958.148691, -1367.54885, -1741.14511, -2092.20259}, 
{-1295.44714, -1286.9622, -1278.14521, -1269.00696, -1259.55822, -1249.80977, -1239.77239, -1229.45687, -1218.87398, -1208.0345, -1196.94923, -1185.62892, -1174.08438, -1162.32637, -1150.36569, -1138.2131, -1125.87939, -1113.37534, -1100.71174, -1087.89936, -1074.94897, -1061.87138, -1048.67734, -1035.37766, -1021.98309, -1008.50444, -994.952467, -981.337965, -967.671712, -953.964487, -940.227072, -926.470247, -912.704793, -898.941492, -885.628885, -872.549665, -859.481262, -846.420251, -833.363204, -820.306698, -807.247306, -794.181603, -781.106164, -768.017562, -754.912372, -741.787169, -728.638527, -715.46302, -702.257224, -689.017711, -675.741058, -662.018918, -648.261563, -634.474346, -620.662619, -606.831735, -592.987045, -579.133902, -565.277659, -551.423668, -537.57728, -523.743849, -509.928727, -496.137265, -482.374817, -468.646734, -454.95837, -441.486345, -428.403171, -415.360968, -402.353986, -389.376474, -376.422682, -363.486859, -350.563256, -337.646121, -324.729704, -311.808254, -298.876022, -285.927257, -272.956208, -259.957125, -246.924258, -233.851855, -220.377137, -206.680378, -192.946144, -179.180269, -165.38859, -151.576944, -137.751165, -123.91709, -110.080554, -96.2473942, -82.4234455, -68.6145439, -54.8265254, -41.0652259, -27.3364813, -13.6461273, 0, 11.5903495, 23.1944077, 34.8444744, 46.5728497, 58.4118335, 70.3937257, 82.5508262, 94.9154349, 107.519852, 120.396377, 133.57731, 147.094951, 160.9816, 175.269556, 189.991121, 205.178593, 220.921496, 237.260306, 254.097887, 271.478802, 289.447611, 308.048879, 327.327166, 347.327036, 368.09305, 389.66977, 412.101759, 435.43358, 459.709793, 484.722598, 510.861454, 538.246632, 566.998403, 597.560455, 629.891349, 663.863408, 699.499877, 736.824004, 775.859036, 816.62822, 858.314577, 902.0373, 947.98403, 996.342404, 1047.30006, 1103.14662, 1161.44224, 1221.84908, 1284.02927, 1347.64496, 1410.79403, 1476.61477, 1544.84042, 1616.06799, 1692.65807, 1776.17944, 1868.20088, 1981.27795, 2103.49565, 2210.95335, 2277.63902, 2326.55417, 2383.03849, 2452.86794, 2508.66889, 2538.28391, 2532.51696, 2479.97838, 2358.42009, 2151.73613, 1949.04892, 1612.83518, 1090.56376, 329.703502, -722.276739, -3646.11166, -6584.07797, -9206.65595, -11184.3258, -12187.5679, -10411.399, -7985.40514, -5212.40798, -3342.65843, -2303.95418, -1629.3308, -1266.32411, -1162.46993, -1265.30405, -1522.3623, -1900.20494, -2265.40003, -2617.91446, -2957.71511, -3284.76887, -3599.04261, -3900.50323, -4131.64011, -4373.70656}, 
{-1218.35287, -1210.79363, -1202.95486, -1194.8443, -1186.46968, -1177.83873, -1168.95919, -1159.83879, -1150.48526, -1140.90633, -1131.10974, -1121.10322, -1110.8945, -1100.49132, -1089.9014, -1079.13248, -1068.19229, -1057.08857, -1045.82904, -1034.42145, -1022.87351, -1011.19298, -999.387565, -987.465014, -975.433057, -963.299425, -951.071852, -938.758069, -926.365809, -913.902805, -901.37679, -888.795496, -876.166655, -863.498001, -851.060942, -838.725827, -826.359834, -813.962141, -801.531921, -789.068352, -776.570609, -764.037868, -751.469304, -738.864094, -726.221413, -713.540437, -700.820342, -688.060303, -675.259497, -662.4171, -649.532286, -636.278337, -622.987389, -609.665682, -596.319456, -582.954953, -569.578412, -556.196075, -542.814181, -529.438971, -516.076687, -502.733568, -489.415855, -476.129789, -462.881609, -449.677558, -436.523875, -423.639467, -411.238135, -398.887511, -386.58005, -374.308206, -362.064433, -349.841185, -337.630917, -325.426082, -313.219136, -301.002533, -288.768726, -276.51017, -264.219319, -251.888628, -239.510551, -227.077542, -214.106778, -200.83835, -187.518058, -174.153781, -160.753397, -147.324784, -133.87582, -120.414382, -106.948349, -93.4855987, -80.0340085, -66.6014566, -53.1958211, -39.8249797, -26.4968104, -13.2191912, 0, 11.3459487, 22.6296822, 33.8880574, 45.1579314, 56.4761613, 67.8796039, 79.4051165, 91.0895559, 102.969779, 115.082644, 127.465006, 140.153723, 153.185652, 166.59765, 180.426575, 194.709282, 209.289991, 224.178039, 239.846012, 256.289489, 273.504046, 291.485261, 310.228711, 329.729974, 349.984627, 370.988247, 392.736411, 415.224697, 438.448683, 461.702515, 485.940392, 511.36832, 538.192305, 567.687104, 599.524349, 633.090668, 668.271442, 704.952051, 743.017875, 782.354295, 820.792765, 860.8977, 902.956334, 947.255904, 994.083645, 1047.66584, 1103.36591, 1160.48634, 1218.32959, 1276.19816, 1331.41588, 1387.68217, 1444.70206, 1502.99526, 1564.74483, 1631.38699, 1704.358, 1795.47795, 1893.43926, 1975.60649, 2020.47381, 2043.03721, 2061.13483, 2076.83422, 2065.48062, 2014.68524, 1918.46596, 1766.09497, 1523.35339, 1169.31021, 845.435077, 343.411627, -413.906754, -1503.66668, -3003.01476, -7041.83271, -11131.3483, -14835.5243, -17718.3235, -19343.7089, -17386.0052, -14558.572, -11235.2168, -8927.9098, -7559.93602, -6580.34611, -5936.13681, -5574.30486, -5441.84701, -5485.75998, -5685.17929, -5886.99515, -6090.10267, -6293.39695, -6495.77312, -6696.12627, -6893.35153, -7010.901, -7097.41562}, 
{-1139.48068, -1129.12466, -1118.87162, -1108.71535, -1098.64965, -1088.6683, -1078.7651, -1068.93384, -1059.16831, -1049.46232, -1039.80965, -1030.20409, -1020.63943, -1011.10948, -1001.60802, -992.128851, -982.665757, -973.212535, -963.762979, -954.31088, -944.850033, -935.374229, -925.877262, -916.352926, -906.795012, -897.197313, -887.553624, -877.857736, -868.103443, -858.284538, -848.394814, -838.428063, -828.378078, -818.238654, -807.76608, -797.077898, -786.294499, -775.417385, -764.448055, -753.388011, -742.238752, -731.001779, -719.678592, -708.270691, -696.779577, -685.206751, -673.553711, -661.82196, -650.012997, -638.128322, -626.169436, -613.93677, -601.637252, -589.276741, -576.861097, -564.396178, -551.887845, -539.341955, -526.764369, -514.160944, -501.537541, -488.900019, -476.254236, -463.606051, -450.961325, -438.325916, -425.705682, -413.264694, -401.163219, -389.080823, -377.013109, -364.95568, -352.904139, -340.854089, -328.801134, -316.740875, -304.668917, -292.580862, -280.472314, -268.338875, -256.176148, -243.979738, -231.745246, -219.468276, -206.787732, -193.885069, -180.94603, -167.977793, -154.987538, -141.982441, -128.969683, -115.956442, -102.949896, -89.9572242, -76.985605, -64.0422172, -51.1342393, -38.26885, -25.453228, -12.6945517, 0, 11.1156629, 22.1283913, 33.0753746, 43.9938022, 54.9208636, 65.8937482, 76.9496453, 88.1257446, 99.4592353, 110.987307, 122.747149, 134.775951, 147.110902, 159.789191, 172.848009, 186.324544, 200.119179, 214.249558, 229.062318, 244.565331, 260.766471, 277.673612, 295.294627, 313.637389, 332.709773, 352.519651, 373.074898, 394.383386, 416.452989, 438.788645, 462.085573, 486.502527, 512.198263, 539.8121, 569.26251, 600.339813, 633.058593, 667.433436, 703.478924, 741.209643, 781.052073, 822.483543, 865.438048, 909.849584, 955.652147, 1001.20959, 1048.41858, 1097.60567, 1149.09737, 1203.22022, 1263.15759, 1322.88749, 1378.60381, 1428.48096, 1474.73695, 1517.77427, 1557.99545, 1594.68085, 1629.61015, 1665.78716, 1720.5243, 1761.83581, 1764.21893, 1712.2103, 1607.35777, 1443.40075, 1227.93566, 958.294463, 581.000186, 71.316261, -366.711757, -1021.7998, -1997.87824, -3398.87745, -5328.72781, -10446.4597, -15662.1285, -20440.8896, -24247.8982, -26548.3099, -24551.5771, -21482.3598, -17772.5431, -15139.5095, -13506.3014, -12260.3372, -11351.4405, -10729.4349, -10344.144, -10145.3913, -10096.0479, -10080.5049, -10094.4472, -10133.5595, -10193.5268, -10270.0338, -10358.7656, -10381.0559, -10270.3949}, 
{-1068.67232, -1058.40742, -1048.34206, -1038.46617, -1028.7697, -1019.24258, -1009.87476, -1000.65617, -991.576764, -982.626473, -973.795239, -965.073002, -956.449703, -947.915282, -939.45968, -931.072837, -922.744692, -914.465188, -906.224262, -898.011857, -889.817912, -881.632368, -873.445165, -865.246243, -857.025542, -848.773003, -840.478567, -832.132173, -823.723762, -815.243273, -806.680649, -798.025828, -789.26875, -780.399358, -771.015568, -761.311577, -751.486295, -741.542384, -731.482505, -721.309321, -711.025494, -700.633687, -690.13656, -679.536777, -668.837, -658.03989, -647.148111, -636.164324, -625.091191, -613.931374, -602.687536, -591.250725, -579.737636, -568.153351, -556.502953, -544.791522, -533.024141, -521.205891, -509.341855, -497.437113, -485.496748, -473.525841, -461.529474, -449.51273, -437.480689, -425.438433, -413.391044, -401.46692, -389.791498, -378.115529, -366.436101, -354.750303, -343.055221, -331.347944, -319.625559, -307.885154, -296.123816, -284.338634, -272.526694, -260.685086, -248.810895, -236.901211, -224.95312, -212.963711, -200.605075, -188.043632, -175.449712, -162.830949, -150.194978, -137.549432, -124.901947, -112.260157, -99.6316956, -87.0241984, -74.4452995, -61.9026333, -49.4038343, -36.956537, -24.5683759, -12.2469854, 0, 10.5526227, 21.010986, 31.4104946, 41.7865532, 52.1745666, 62.6099394, 73.1280765, 83.7643826, 94.5542623, 105.53312, 116.736362, 128.199391, 139.957613, 152.046432, 164.501253, 177.357481, 190.514695, 203.988897, 218.122715, 232.922448, 248.394395, 264.544857, 281.380132, 298.906519, 317.130319, 336.057829, 355.69535, 376.049181, 397.125621, 418.52582, 440.809782, 464.10535, 488.540367, 514.388966, 541.705848, 570.5067, 600.87548, 632.896144, 666.65265, 702.228954, 741.213649, 781.728124, 823.561951, 866.5047, 910.345944, 953.159772, 996.880109, 1041.7254, 1087.91408, 1135.66459, 1187.7515, 1238.71298, 1287.43595, 1332.47615, 1371.7133, 1403.33064, 1425.51141, 1427.52983, 1418.50294, 1417.26674, 1461.68315, 1495.45496, 1471.58326, 1365.60887, 1191.26497, 944.753834, 641.312509, 282.07821, -201.606097, -838.917874, -1402.39128, -2217.33067, -3401.75156, -5073.66948, -7351.09996, -13165.6048, -19118.2667, -24623.7146, -29096.5775, -31951.4842, -30187.0645, -27244.6127, -23574.186, -20941.0446, -19274.6445, -17956.4404, -16943.2885, -16192.0448, -15659.5653, -15302.7063, -15270.6821, -15161.2774, -15001.8821, -14819.886, -14642.6791, -14497.6512, -14412.1924, -14221.0705, -13880.1115}, 
{-1003.72827, -996.531557, -989.327998, -982.11475, -974.888964, -967.647793, -960.38839, -953.107908, -945.803499, -938.472316, -931.111511, -923.718238, -916.289649, -908.822896, -901.315132, -893.763511, -886.165184, -878.517304, -870.817024, -863.061497, -855.247875, -847.373311, -839.434957, -831.429967, -823.355492, -815.208686, -806.986702, -798.686691, -790.305807, -781.841202, -773.290029, -764.64944, -755.916589, -747.088627, -738.036852, -728.82399, -719.514284, -710.10897, -700.609284, -691.016465, -681.331748, -671.556371, -661.691571, -651.738584, -641.698648, -631.573, -621.362877, -611.069515, -600.694151, -590.238023, -579.702367, -568.980611, -558.184139, -547.316524, -536.381341, -525.382163, -514.322565, -503.20612, -492.036403, -480.816988, -469.551449, -458.24336, -446.896294, -435.513826, -424.099531, -412.656981, -401.189752, -389.775181, -378.488835, -367.182156, -355.853935, -344.502966, -333.12804, -321.727948, -310.301485, -298.847441, -287.364609, -275.851782, -264.307751, -252.731308, -241.121246, -229.476358, -217.795434, -206.077268, -194.039493, -181.827394, -169.589636, -157.334136, -145.068809, -132.801572, -120.540342, -108.293035, -96.0675669, -83.8718546, -71.7138142, -59.6013622, -47.5424149, -35.5448886, -23.6166998, -11.7657648, 0, 9.28966353, 18.6159067, 28.004897, 37.4828019, 47.0757888, 56.8100253, 66.7116788, 76.8069168, 87.1219067, 97.6828161, 108.515812, 119.647063, 131.102736, 142.908997, 155.092016, 167.677959, 180.680569, 194.12424, 208.063241, 222.521077, 237.521252, 253.087273, 269.242645, 286.010872, 303.41546, 321.479914, 340.227739, 359.682441, 379.867524, 400.728754, 422.39588, 444.915731, 468.335132, 492.183366, 516.766033, 542.526745, 569.667593, 598.390667, 628.89806, 661.391861, 698.792657, 737.756676, 777.954129, 819.055227, 860.730182, 903.646881, 946.228441, 987.895655, 1028.06931, 1066.17021, 1099.95773, 1132.54468, 1171.16499, 1217.09209, 1259.4302, 1292.74724, 1311.61115, 1296.65286, 1259.54483, 1227.16359, 1242.0961, 1243.79652, 1183.12188, 1034.28788, 811.090152, 509.156496, 145.945457, -277.255349, -839.204392, -1573.38155, -2275.7357, -3253.82095, -4623.33943, -6499.9933, -8999.48468, -14978.7934, -21127.0244, -26874.5606, -31651.7847, -34889.0793, -33723.751, -31407.9762, -28354.8849, -26160.4764, -24769.6613, -23634.4877, -22724.5093, -22009.2796, -21458.3522, -21041.2809, -21333.9704, -21262.0719, -20929.5148, -20440.2281, -19898.1412, -19407.1832, -19071.2833, -18531.9506, -17932.6812}, 
{-946.797945, -943.00765, -938.953851, -934.642798, -930.08074, -925.273926, -920.228606, -914.951029, -909.447445, -903.724101, -897.787249, -891.643137, -885.298015, -878.758131, -872.029736, -865.119077, -858.032406, -850.775971, -843.356021, -835.778805, -828.050574, -820.177576, -812.16606, -804.022276, -795.752474, -787.362901, -778.859809, -770.249446, -761.538061, -752.731903, -743.837223, -734.860269, -725.807291, -716.684537, -707.709977, -698.779547, -689.786639, -680.730631, -671.610902, -662.42683, -653.177795, -643.863174, -634.482346, -625.03469, -615.519584, -605.936408, -596.284539, -586.563357, -576.772239, -566.910565, -556.977713, -546.830791, -536.614533, -526.3314, -515.983857, -505.574365, -495.105388, -484.579388, -473.998827, -463.36617, -452.683877, -441.954413, -431.18024, -420.363821, -409.507617, -398.614093, -387.685711, -376.76813, -365.905972, -355.01261, -344.087708, -333.130927, -322.14193, -311.120379, -300.065937, -288.978266, -277.85703, -266.70189, -255.512508, -244.288548, -233.029672, -221.735542, -210.405821, -199.040172, -187.370422, -175.535786, -163.678694, -151.807502, -139.930563, -128.056231, -116.192862, -104.348809, -92.5324257, -80.7520673, -69.0160876, -57.332841, -45.7106815, -34.1579635, -22.6830411, -11.2942685, 0, 7.92416365, 16.0380738, 24.3571314, 32.8967375, 41.6722928, 50.6991984, 59.9928551, 69.568664, 79.4420259, 89.6283418, 100.143013, 111.001439, 122.219023, 133.811164, 145.793263, 158.180723, 171.114718, 184.628619, 198.453333, 212.631213, 227.204612, 242.215883, 257.707379, 273.721453, 290.300457, 307.486744, 325.322668, 343.850581, 363.112836, 383.372887, 404.370916, 426.082945, 448.484996, 470.516973, 492.672957, 515.723331, 539.954954, 565.654685, 593.109384, 622.605909, 658.106624, 695.104251, 733.166529, 771.861198, 810.755996, 852.98889, 893.664834, 931.459008, 965.046594, 993.102775, 1008.65384, 1022.92806, 1049.7607, 1092.85887, 1135.25143, 1169.25134, 1187.17159, 1168.0916, 1120.56551, 1067.58405, 1041.23783, 993.974136, 890.647813, 710.222975, 455.571076, 118.58971, -288.121161, -767.296661, -1397.58062, -2214.6786, -3054.91449, -4184.04965, -5706.80641, -7727.90707, -10352.0739, -16194.0548, -22221.0401, -27910.2459, -32738.8881, -36184.1826, -35667.9016, -34093.0007, -31817.6005, -30206.7, -29219.8635, -28397.1226, -27719.0186, -27166.0923, -26718.8848, -26357.9371, -27067.1292, -27125.878, -26710.8402, -25998.6729, -25166.0328, -24389.5769, -23845.9621, -23019.1395, -22173.9482}, 
{-896.802199, -893.170384, -889.317809, -885.249068, -880.968756, -876.481471, -871.791805, -866.904356, -861.823717, -856.554485, -851.101255, -845.468622, -839.661181, -833.683527, -827.540257, -821.235965, -814.775247, -808.162697, -801.402912, -794.500486, -787.460015, -780.286094, -772.983318, -765.556283, -758.009584, -750.347816, -742.575575, -734.697456, -726.718055, -718.641965, -710.473784, -702.218105, -693.879525, -685.462639, -677.117447, -668.772782, -660.355733, -651.866176, -643.303987, -634.669044, -625.961222, -617.180399, -608.326451, -599.399254, -590.398686, -581.324623, -572.176942, -562.955519, -553.660232, -544.290956, -534.847568, -525.209572, -515.499826, -505.720818, -495.875034, -485.964958, -475.993079, -465.96188, -455.87385, -445.731473, -435.537236, -425.293626, -415.003127, -404.668226, -394.29141, -383.875165, -373.421976, -362.980254, -352.597308, -342.180908, -331.730562, -321.24578, -310.72607, -300.170941, -289.579902, -278.952463, -268.288132, -257.586418, -246.84683, -236.068877, -225.252069, -214.395914, -203.49992, -192.563598, -181.292515, -169.839334, -158.360249, -146.864307, -135.360558, -123.858049, -112.365828, -100.892943, -89.448442, -78.0413724, -66.6807825, -55.3757202, -44.1352335, -32.9683701, -21.8841781, -10.8917055, 0, 7.23333339, 14.704329, 22.4236738, 30.402055, 38.6501594, 47.1786741, 55.9982862, 65.1196826, 74.5535505, 84.3105768, 94.4014486, 104.836853, 115.627477, 126.784007, 138.317131, 150.237535, 162.727511, 175.822259, 189.144317, 202.741143, 216.660197, 230.948937, 245.654824, 260.825316, 276.507872, 292.749952, 309.599014, 327.102519, 345.307925, 364.545384, 384.476008, 405.06245, 426.26736, 446.954025, 467.634778, 489.115118, 511.687505, 535.644401, 561.278267, 588.881563, 623.091294, 658.533126, 694.649499, 730.882856, 766.675638, 803.951616, 839.05157, 870.797611, 898.011846, 919.516386, 929.395838, 937.000094, 954.090677, 983.734817, 1011.2489, 1029.91907, 1033.03149, 1004.88457, 949.794859, 881.883706, 822.91774, 735.968591, 597.071711, 388.493672, 103.059337, -271.252859, -729.227661, -1278.41783, -1989.5778, -2897.71157, -3854.34688, -5106.1911, -6751.75923, -8889.5663, -11618.1273, -17424.8515, -23422.1361, -29111.2725, -33993.5523, -37570.2668, -37402.0481, -36224.62, -34370.9852, -33106.5277, -32395.1826, -31812.3926, -31341.5174, -30965.9167, -30668.9502, -30433.9775, -31345.9176, -31528.0457, -31171.4668, -30467.2861, -29606.6087, -28780.5396, -28180.184, -27307.728, -26269.5885}, 
{-851.022377, -847.29979, -843.41625, -839.374093, -835.17565, -830.823257, -826.319245, -821.66595, -816.865704, -811.920841, -806.833695, -801.606598, -796.241886, -790.74189, -785.108946, -779.345385, -773.453543, -767.435752, -761.294346, -755.031658, -748.650023, -742.151773, -735.539242, -728.814764, -721.980673, -715.039301, -707.992982, -700.844051, -693.59484, -686.247683, -678.804914, -671.268865, -663.641872, -655.926267, -648.179167, -640.374362, -632.484983, -624.511586, -616.454726, -608.314959, -600.09284, -591.788927, -583.403773, -574.937935, -566.391968, -557.766429, -549.061873, -540.278856, -531.417933, -522.47966, -513.464593, -504.273565, -495.009016, -485.673663, -476.270225, -466.801418, -457.26996, -447.678569, -438.029962, -428.326857, -418.571971, -408.768021, -398.917725, -389.023801, -379.088966, -369.115938, -359.107434, -349.123193, -339.221583, -329.287721, -319.320629, -309.319327, -299.282836, -289.210177, -279.100372, -268.952441, -258.765406, -248.538287, -238.270106, -227.959884, -217.606642, -207.2094, -196.76718, -186.279004, -175.411953, -164.337999, -153.233106, -142.107065, -130.96967, -119.830713, -108.699989, -97.5872905, -86.50241, -75.455141, -64.4552765, -53.5126097, -42.6369337, -31.8380416, -21.1257265, -10.5097816, 0, 6.60563213, 13.4903855, 20.6605705, 28.1224975, 35.882477, 43.9468195, 52.3218352, 61.0138347, 70.0291284, 79.3740267, 89.05484, 99.0778788, 109.449453, 120.175874, 131.263452, 142.718497, 154.761731, 167.430097, 180.244923, 193.258466, 206.522981, 220.090725, 234.013954, 248.344923, 263.135888, 278.439106, 294.306831, 310.791321, 327.944831, 346.140229, 364.991602, 384.45502, 404.486556, 423.975395, 443.411053, 463.567548, 484.721019, 507.147602, 531.123437, 556.924662, 589.605636, 623.210035, 657.079125, 690.554177, 722.976457, 754.794912, 783.966213, 809.55471, 830.62475, 846.240685, 852.230628, 854.850559, 862.392667, 876.864383, 887.441764, 889.06189, 876.661845, 842.081279, 784.058666, 704.711498, 612.213675, 483.757432, 309.395799, 75.9058131, -235.486469, -641.007, -1144.0878, -1757.63787, -2541.47732, -3528.89066, -4590.38857, -5947.07158, -7688.56899, -9904.51015, -12684.5244, -18338.8052, -24181.2766, -29746.427, -34568.7447, -38182.7178, -38324.0713, -37525.2325, -36091.5881, -35184.8752, -34771.5087, -34461.091, -34237.8775, -34086.1237, -33990.0853, -33934.0177, -35019.3094, -35349.7461, -35114.3536, -34502.1579, -33702.1851, -32903.461, -32295.0116, -31440.31, -30252.6469}, 
{-807.945011, -804.135934, -800.223006, -796.206365, -792.086149, -787.862495, -783.535542, -779.105427, -774.572288, -769.936263, -765.19749, -760.356106, -755.41225, -750.366059, -745.21767, -739.967223, -734.614854, -729.160701, -723.604903, -717.947597, -712.18892, -706.329011, -700.368008, -694.306049, -688.14327, -681.87981, -675.515807, -669.051399, -662.486723, -655.821918, -649.05712, -642.192468, -635.228101, -628.164154, -620.964274, -613.647613, -606.23376, -598.724038, -591.119769, -583.422275, -575.632878, -567.752901, -559.783664, -551.726491, -543.582703, -535.353623, -527.040572, -518.644873, -510.167848, -501.610819, -492.975107, -484.183759, -475.31807, -466.381059, -457.375745, -448.305147, -439.172284, -429.980174, -420.731836, -411.430291, -402.078556, -392.67965, -383.236593, -373.752403, -364.2301, -354.672702, -345.083228, -335.536373, -326.105111, -316.644636, -307.153318, -297.629531, -288.071648, -278.478041, -268.847081, -259.177143, -249.466598, -239.713819, -229.917178, -220.075048, -210.185801, -200.24781, -190.259447, -180.219085, -169.749741, -159.04536, -148.3044, -137.537413, -126.754952, -115.967573, -105.185827, -94.420268, -83.6814503, -72.979927, -62.3262516, -51.7309776, -41.2046584, -30.7578475, -20.4010985, -10.1449648, 0, 6.0114793, 12.3407776, 18.9900326, 25.9613816, 33.2569624, 40.8789125, 48.8293694, 57.1104707, 65.7243539, 74.6731566, 83.9590164, 93.5840708, 103.550457, 113.860314, 124.515777, 135.518986, 147.129378, 159.38585, 171.708546, 184.154741, 196.781708, 209.64672, 222.807051, 236.319975, 250.242765, 264.632696, 279.547041, 295.043072, 311.178066, 328.365094, 346.175171, 364.558829, 383.466602, 401.869228, 420.207137, 439.182041, 459.038411, 480.020719, 502.373437, 526.341039, 557.108198, 588.475644, 619.721289, 650.123042, 678.958814, 705.386122, 728.833369, 748.608566, 764.019722, 774.374846, 777.38652, 775.90815, 774.344308, 773.514728, 766.920098, 751.347211, 723.582856, 683.602876, 624.280226, 535.008893, 409.693026, 240.362615, 31.1567854, -226.344769, -561.843617, -995.272879, -1538.20372, -2208.40136, -3054.29148, -4106.95649, -5259.29196, -6701.8266, -8513.47668, -10773.1585, -13559.7883, -18972.3444, -24564.6655, -29910.6526, -34584.2063, -38159.2275, -38570.2723, -38123.4826, -37095.3357, -36542.8744, -36433.6019, -36409.7856, -36455.1905, -36553.5817, -36688.724, -36844.3825, -38056.6834, -38543.764, -38476.475, -38025.6672, -37362.1915, -36656.8984, -36080.6389, -35288.4824, -34007.1298}, 
{-766.056631, -762.418883, -758.711904, -754.934375, -751.084979, -747.162397, -743.16531, -739.092402, -734.942353, -730.713846, -726.405562, -722.016184, -717.544392, -712.98887, -708.348299, -703.621361, -698.806738, -693.903111, -688.909162, -683.823575, -678.645029, -673.372207, -668.003792, -662.538464, -656.974906, -651.3118, -645.547828, -639.68167, -633.712011, -627.63753, -621.45691, -615.168833, -608.771981, -602.265036, -595.541904, -588.655859, -581.661434, -574.560709, -567.355768, -560.048692, -552.641564, -545.136465, -537.535477, -529.840683, -522.054164, -514.178003, -506.214281, -498.165081, -490.032484, -481.818573, -473.52543, -465.101144, -456.602961, -448.034132, -439.397911, -430.697548, -421.936298, -413.117412, -404.244144, -395.319744, -386.347466, -377.330563, -368.272287, -359.175889, -350.044624, -340.881742, -331.690498, -322.559223, -313.57421, -304.563237, -295.524041, -286.454358, -277.351927, -268.214484, -259.039767, -249.825511, -240.569454, -231.269333, -221.922885, -212.527848, -203.081957, -193.582951, -184.028565, -174.416538, -164.326887, -153.974995, -143.581266, -133.156991, -122.713464, -112.261976, -101.813822, -91.3802936, -80.9726837, -70.6022849, -60.2803902, -50.0182922, -39.8272837, -29.7186575, -19.7037063, -9.79372288, 0, 5.42129432, 11.2000398, 17.3342714, 23.822024, 30.6613323, 37.8502313, 45.3867559, 53.2689409, 61.4948212, 70.0624317, 78.9698072, 88.2149826, 97.7959928, 107.710873, 117.957657, 128.534381, 139.74245, 151.623239, 163.488584, 175.40153, 187.425119, 199.622394, 212.056397, 224.790173, 237.886764, 251.409214, 265.420564, 279.983859, 295.16214, 311.427648, 328.284191, 345.672053, 363.531517, 380.923666, 398.228385, 416.060014, 434.622598, 454.120182, 474.756811, 496.736531, 525.05753, 553.670622, 581.841869, 608.837332, 633.923072, 655.59259, 674.077647, 688.837441, 699.331174, 705.018045, 705.09182, 699.601555, 689.62324, 674.953015, 652.773983, 621.442449, 579.314717, 533.3705, 471.382826, 371.717361, 215.923183, 8.80609479, -234.108621, -517.062241, -877.78938, -1338.65066, -1917.07759, -2634.15285, -3529.03262, -4630.6497, -5857.30944, -7365.59166, -9222.7233, -11495.9312, -14252.4424, -19361.8977, -24638.5066, -29698.8924, -34159.6784, -37637.4882, -38276.952, -38148.015, -37498.1547, -37281.6573, -37466.2219, -37725.0442, -38040.5485, -38395.1591, -38771.3004, -39151.3968, -40427.4183, -41062.8842, -41194.8055, -40960.1929, -40496.0573, -39939.4093, -39427.2599, -38723.8419, -37417.0438}, 
{-723.843768, -720.888706, -717.856775, -714.746616, -711.556868, -708.286171, -704.933165, -701.496488, -697.97478, -694.366682, -690.670831, -686.885868, -683.010433, -679.043164, -674.982702, -670.827685, -666.576754, -662.228547, -657.781705, -653.234866, -648.586671, -643.835759, -638.980768, -634.02034, -628.953113, -623.777727, -618.492821, -613.097035, -607.589008, -601.96738, -596.230791, -590.377879, -584.407285, -578.317647, -571.981193, -565.462428, -558.827375, -552.078776, -545.219374, -538.251909, -531.179125, -524.003762, -516.728563, -509.35627, -501.889623, -494.331366, -486.68424, -478.950986, -471.134348, -463.237065, -455.261881, -447.186709, -439.039658, -430.824006, -422.543035, -414.200025, -405.798254, -397.341005, -388.831555, -380.273186, -371.669178, -363.022811, -354.337364, -345.616118, -336.862353, -328.079348, -319.270385, -310.531171, -301.955195, -293.355111, -284.728207, -276.071772, -267.383095, -258.659462, -249.898163, -241.096486, -232.25172, -223.361152, -214.42207, -205.431764, -196.387522, -187.286631, -178.126381, -168.904058, -159.164397, -149.140482, -139.070838, -128.967439, -118.842259, -108.707274, -98.5744579, -88.4557852, -78.3632306, -68.3087688, -58.3043743, -48.3620218, -38.4936859, -28.7113411, -19.0269622, -9.45252358, 0, 4.80549664, 10.0127065, 15.6154983, 21.607741, 27.9833033, 34.736054, 41.8598619, 49.3485958, 57.1961246, 65.396317, 73.9430418, 82.8301678, 92.0515638, 101.601099, 111.472641, 121.66006, 132.512945, 144.075981, 155.538433, 166.970393, 178.441957, 190.02322, 201.784276, 213.795219, 226.126146, 238.847149, 252.028323, 265.739765, 280.051567, 295.535562, 311.576137, 328.092865, 345.005319, 361.426854, 377.680151, 394.302886, 411.456497, 429.302422, 448.002102, 467.716975, 492.91218, 518.135634, 542.706747, 565.944929, 587.16959, 605.281662, 620.123654, 631.119597, 637.693521, 639.269457, 635.574837, 625.359463, 607.907103, 582.446407, 548.093497, 504.015022, 449.377625, 395.305294, 326.289757, 213.778371, 31.4715327, -207.890173, -482.863711, -795.05077, -1185.10103, -1675.74052, -2286.21153, -3038.3369, -3966.71306, -5098.71094, -6380.6934, -7933.50228, -9812.54982, -12073.2482, -14771.0098, -19543.8938, -24469.0036, -29206.0896, -33414.9022, -36755.1919, -37580.4115, -37727.474, -37415.9716, -37502.3558, -37954.1288, -38473.4345, -39041.0431, -39637.7244, -40244.2485, -40841.3854, -42100.8927, -42859.8917, -43206.3196, -43228.114, -43013.212, -42649.5511, -42225.0687, -41617.9854, -40366.3952}, 
{-673.679382, -673.124023, -672.35271, -671.368386, -670.173992, -668.772469, -667.166758, -665.3598, -663.354537, -661.15391, -658.76086, -656.178329, -653.409257, -650.456586, -647.323257, -644.012212, -640.526392, -636.868737, -633.04219, -629.049691, -624.894181, -620.578603, -616.105897, -611.479004, -606.700866, -601.774424, -596.702619, -591.488393, -586.134686, -580.64444, -575.020596, -569.266096, -563.38388, -557.376891, -551.242548, -544.986671, -538.615141, -532.13108, -525.537608, -518.837846, -512.034913, -505.13193, -498.132018, -491.038297, -483.853887, -476.581908, -469.225482, -461.787728, -454.271767, -446.680719, -439.017704, -431.310356, -423.536751, -415.699478, -407.801125, -399.844283, -391.831539, -383.765484, -375.648705, -367.483792, -359.273334, -351.01992, -342.726138, -334.394579, -326.02783, -317.628481, -309.199121, -300.81974, -292.56847, -284.288266, -275.976699, -267.63134, -259.249761, -250.829533, -242.368228, -233.863417, -225.312672, -216.713563, -208.063662, -199.360541, -190.601771, -181.784923, -172.907568, -163.967279, -154.499314, -144.742125, -134.93972, -125.104673, -115.249559, -105.386952, -95.5294266, -85.6895569, -75.8799172, -66.1130819, -56.4016253, -46.7581218, -37.1951455, -27.7252709, -18.3610722, -9.1151238, 0, 3.88460888, 8.26305334, 13.1226793, 18.4508328, 24.2348598, 30.4621062, 37.119918, 44.1956411, 51.6766216, 59.5502053, 67.8037383, 76.4245665, 85.4000359, 94.7174924, 104.364282, 114.327751, 125.061088, 136.618188, 147.932705, 159.091807, 170.182664, 181.292446, 192.508322, 203.917462, 215.607035, 227.664211, 240.176159, 253.230049, 266.913051, 282.102941, 297.806392, 313.873392, 330.153929, 345.441923, 360.115394, 374.833982, 389.764496, 405.073742, 420.928529, 437.495664, 458.486978, 479.445337, 499.843955, 519.156049, 536.854835, 555.246551, 570.263135, 580.669549, 585.230755, 582.711713, 569.828298, 549.898995, 525.365637, 496.616, 459.844157, 413.12707, 354.5417, 294.278368, 215.547638, 88.3454955, -115.508305, -383.202883, -700.558688, -1069.11759, -1517.06053, -2060.34067, -2704.10137, -3461.49326, -4385.30295, -5505.89669, -6818.13771, -8393.3573, -10274.2112, -12503.3553, -15123.4452, -19535.0278, -24083.395, -28471.7299, -32403.2156, -35581.0351, -36570.0917, -36970.7012, -36973.8808, -37334.706, -38020.9207, -38765.2448, -39547.1734, -40346.2018, -41141.8252, -41913.5389, -42785.2659, -43530.5277, -44118.1515, -44516.9645, -44695.7941, -44623.4675, -44268.812, -43767.5339, -42768.6632}, 
{-633.134032, -633.973973, -634.524619, -634.791176, -634.77885, -634.492849, -633.938379, -633.120646, -632.044858, -630.71622, -629.13994, -627.321225, -625.26528, -622.977312, -620.462529, -617.726137, -614.773342, -611.609351, -608.239371, -604.668609, -600.90227, -596.945562, -592.803692, -588.481866, -583.98529, -579.319171, -574.488717, -569.499133, -564.355627, -559.063404, -553.627672, -548.053638, -542.346507, -536.511487, -530.60951, -524.616748, -518.508703, -512.288772, -505.960354, -499.526847, -492.99165, -486.35816, -479.629776, -472.809896, -465.901918, -458.909241, -451.835262, -444.683381, -437.456994, -430.159501, -422.794299, -415.419451, -407.982506, -400.485677, -392.931178, -385.321221, -377.65802, -369.943789, -362.180739, -354.371086, -346.517041, -338.620818, -330.68463, -322.71069, -314.701213, -306.65841, -298.584495, -290.55286, -282.635189, -274.686892, -266.705571, -258.688822, -250.634246, -242.539442, -234.402008, -226.219543, -217.989646, -209.709917, -201.377954, -192.991357, -184.547724, -176.044654, -167.479746, -158.850599, -149.679593, -140.211931, -130.69853, -121.152411, -111.586595, -102.014103, -92.4479548, -82.9011723, -73.3867758, -63.9177862, -54.5072244, -45.168111, -35.9134669, -26.756313, -17.70967, -8.78655873, 0, 3.40832316, 7.33088624, 11.7527569, 16.6590028, 22.0346916, 27.8648909, 34.1346685, 40.829092, 47.9332289, 55.4321471, 63.3109142, 71.5545977, 80.1482655, 89.076985, 98.3258241, 107.87985, 118.209762, 129.370001, 140.247188, 150.930452, 161.508926, 172.071741, 182.708027, 193.506917, 204.557541, 215.949031, 227.770518, 240.111133, 253.060008, 267.577468, 282.562012, 297.841413, 313.243444, 327.718058, 341.531939, 355.185853, 368.77092, 382.378256, 396.098982, 410.024215, 426.384061, 442.479655, 457.983619, 472.568575, 485.907144, 502.517349, 516.015059, 524.861545, 527.518078, 522.445928, 505.667859, 481.064046, 452.544351, 420.620486, 378.866269, 323.97049, 252.621942, 168.885943, 60.3982713, -93.2524138, -310.193905, -587.278542, -918.017747, -1306.95245, -1777.31246, -2343.7491, -3011.08373, -3791.41915, -4732.90131, -5863.28814, -7189.4954, -8763.32963, -10619.7633, -12793.769, -15320.3191, -19432.849, -23668.2527, -27761.8869, -31449.1085, -34465.2743, -35526.2706, -36066.5716, -36258.4501, -36790.5245, -37630.6806, -38531.6605, -39472.1071, -40430.6635, -41385.973, -42316.6787, -43265.8343, -44142.1618, -44897.6094, -45484.1252, -45853.6573, -45958.154, -45749.5634, -45281.9998, -44420.3356}, 
{-600.947329, -602.236046, -603.225762, -603.921964, -604.330141, -604.455781, -604.304373, -603.881404, -603.192362, -602.242736, -601.038014, -599.583684, -597.885235, -595.948154, -593.777929, -591.38005, -588.760003, -585.923278, -582.875362, -579.621744, -576.167911, -572.519352, -568.681556, -564.66001, -560.460202, -556.087621, -551.547754, -546.846091, -541.988119, -536.979326, -531.825201, -526.531231, -521.102905, -515.545712, -509.924402, -504.213586, -498.387161, -492.448693, -486.401746, -480.249885, -473.996675, -467.645681, -461.200468, -454.664601, -448.041645, -441.335165, -434.548726, -427.685893, -420.750231, -413.745305, -406.674679, -399.60721, -392.479756, -385.294466, -378.053492, -370.758981, -363.413085, -356.017951, -348.575731, -341.088573, -333.558627, -325.988042, -318.378969, -310.733557, -303.053955, -295.342314, -287.600782, -279.904653, -272.327465, -264.720514, -257.081207, -249.406952, -241.695157, -233.94323, -226.148579, -218.308612, -210.420737, -202.482362, -194.490895, -186.443744, -178.338316, -170.17202, -161.942264, -153.646455, -144.790706, -135.628407, -126.418852, -117.175392, -107.911379, -98.6401645, -89.3750995, -80.1295356, -70.9168243, -61.7503171, -52.6433653, -43.6093204, -34.6615339, -25.8133571, -17.0781416, -8.46923875, 0, 3.31287574, 7.0984024, 11.3431687, 16.0337632, 21.1567748, 26.698792, 32.6464036, 38.9861983, 45.7047647, 52.7886916, 60.2245676, 67.9989814, 76.0985218, 84.5097773, 93.2193368, 102.213789, 111.888037, 122.286999, 132.473697, 142.522215, 152.506638, 162.50105, 172.579537, 182.816181, 193.285069, 204.060285, 215.215913, 226.826039, 238.964746, 252.450374, 266.33986, 280.484011, 294.733636, 308.570878, 322.030878, 335.247086, 348.18091, 360.793756, 373.047032, 384.902145, 396.829137, 408.125978, 418.654561, 428.276779, 436.854523, 449.865389, 460.151639, 466.17124, 466.382157, 459.242355, 441.819928, 415.661449, 384.689474, 348.792774, 299.62448, 232.535359, 142.876176, 26.1929658, -122.403154, -308.014085, -532.716319, -807.341604, -1131.1943, -1513.11507, -1978.10338, -2543.7415, -3224.84234, -4038.27019, -5011.04389, -6164.43831, -7491.66897, -9046.38688, -10859.5798, -12962.2353, -15385.3412, -19247.9894, -23221.0373, -27063.4467, -30534.1791, -33392.1964, -34459.303, -35056.3891, -35344.0551, -35968.3397, -36897.6907, -37898.29, -38948.6986, -40027.4776, -41113.1881, -42184.3913, -43550.5423, -44664.7974, -45534.7812, -46168.1187, -46572.4345, -46755.3535, -46724.5004, -46251.3528, -45453.4}, 
{-573.713799, -574.88926, -575.793894, -576.432306, -576.809102, -576.928886, -576.796263, -576.415838, -575.792215, -574.930001, -573.8338, -572.508217, -570.957856, -569.187323, -567.201223, -565.004161, -562.600741, -559.995569, -557.193249, -554.198387, -551.015588, -547.649456, -544.104596, -540.385613, -536.497113, -532.4437, -528.229979, -523.860555, -519.340034, -514.673019, -509.864116, -504.917931, -499.839067, -494.63213, -489.333206, -483.930498, -478.411828, -472.78078, -467.040939, -461.195886, -455.249206, -449.204481, -443.065296, -436.835232, -430.517875, -424.116806, -417.63561, -411.077869, -404.447168, -397.747088, -390.981215, -384.21375, -377.386344, -370.501266, -363.560784, -356.567167, -349.522686, -342.429609, -335.290205, -328.106744, -320.881495, -313.616727, -306.314708, -298.97771, -291.608, -284.207847, -276.779522, -269.404113, -262.160819, -254.889346, -247.586855, -240.250504, -232.877454, -225.464864, -218.009894, -210.509704, -202.961452, -195.362298, -187.709403, -179.999925, -172.231025, -164.399861, -156.503594, -148.539383, -139.997434, -131.139048, -122.231615, -113.288748, -104.324057, -95.3511539, -86.3836495, -77.4351551, -68.5192819, -59.649641, -50.8398437, -42.103501, -33.4542243, -24.9056247, -16.4713133, -8.16490132, 0, 3.44469515, 7.28242127, 11.5038652, 16.099714, 21.0606543, 26.3773732, 32.0405574, 38.0408939, 44.3690697, 51.0157714, 57.9716861, 65.2275006, 72.7739019, 80.6015767, 88.701212, 97.0634947, 105.922846, 115.304773, 124.648657, 133.997414, 143.39396, 152.88121, 162.502081, 172.299488, 182.316346, 192.595573, 203.180082, 214.11279, 225.436613, 237.652117, 250.176762, 262.916169, 275.775958, 288.943614, 302.183826, 315.186117, 327.771551, 339.761186, 350.976085, 361.23731, 369.504555, 376.722403, 382.880441, 387.968259, 391.975446, 400.527331, 406.568828, 408.680591, 405.443273, 395.437529, 377.208937, 349.416096, 315.59792, 274.906765, 217.539103, 137.712413, 29.6441763, -119.193353, -306.304462, -521.83501, -752.551366, -1021.85845, -1332.13144, -1695.96955, -2143.29601, -2696.08206, -3382.34127, -4225.61139, -5227.27495, -6401.24672, -7718.9249, -9243.28185, -11002.9496, -13026.5603, -15342.7458, -18982.8863, -22722.1788, -26338.5684, -29610.0002, -32314.4193, -33356.9965, -33970.3005, -34306.3225, -34976.6485, -35951.3466, -37008.9872, -38129.2131, -39291.6674, -40475.9931, -41661.8333, -43557.0045, -44943.2772, -45905.9066, -46530.1478, -46901.256, -47104.4863, -47225.094, -46749.5328, -46022.0824}, 
{-545.055603, -546.174588, -547.04835, -547.680757, -548.075678, -548.236981, -548.168534, -547.874206, -547.357864, -546.623378, -545.674616, -544.515445, -543.149735, -541.581353, -539.814168, -537.852049, -535.698863, -533.358478, -530.834765, -528.131589, -525.252821, -522.202328, -518.983978, -515.60164, -512.059183, -508.360474, -504.509382, -500.509775, -496.365521, -492.08049, -487.658549, -483.103566, -478.41941, -473.609949, -468.697081, -463.67528, -458.538804, -453.290937, -447.934961, -442.47416, -436.911816, -431.251214, -425.495637, -419.648367, -413.712688, -407.691884, -401.589237, -395.40803, -389.151548, -382.823073, -376.425888, -370.011447, -363.533818, -356.995242, -350.397956, -343.7442, -337.036212, -330.276233, -323.4665, -316.609253, -309.70673, -302.761172, -295.774816, -288.749903, -281.68867, -274.593357, -267.466203, -260.383364, -253.419222, -246.423567, -239.393846, -232.327506, -225.221995, -218.07476, -210.883248, -203.644907, -196.357184, -189.017525, -181.623379, -174.172192, -166.661412, -159.088486, -151.450861, -143.745984, -135.469096, -126.87931, -118.241776, -109.570241, -100.878448, -92.1801433, -83.4890707, -74.8189754, -66.1836022, -57.5966961, -49.072002, -40.6232646, -32.2642289, -24.0086397, -15.8702419, -7.86278039, 0, 3.7140203, 7.70565423, 11.9733519, 16.5155634, 21.3307388, 26.4173283, 31.7737819, 37.3985497, 43.2900819, 49.4468285, 55.8672396, 62.5497653, 69.4928557, 76.694961, 84.1545312, 91.8700164, 99.7331501, 107.727137, 116.09181, 124.802752, 133.835545, 143.165771, 152.769012, 162.62085, 172.696869, 182.972649, 193.423774, 204.025825, 214.754385, 225.511372, 236.373043, 247.337079, 258.40116, 270.390316, 282.888553, 295.259256, 307.251899, 318.615962, 329.100921, 338.456254, 345.667833, 351.481141, 355.795058, 358.508462, 359.520233, 364.858561, 366.760684, 363.593152, 353.722513, 335.515319, 305.801975, 266.362685, 222.865237, 175.397738, 112.656444, 28.4523196, -83.4036709, -234.719811, -424.788786, -646.773194, -893.53624, -1180.55632, -1502.7498, -1869.61444, -2315.3554, -2862.83673, -3542.86738, -4380.37122, -5371.14073, -6527.44676, -7830.23882, -9327.92773, -11043.6264, -13000.4475, -15221.5041, -18618.0124, -22102.9558, -25477.4214, -28542.4961, -31099.2669, -32164.1091, -32845.9623, -33282.2183, -34019.8112, -35034.0225, -36129.491, -37290.03, -38499.4531, -39741.5738, -41000.2056, -43177.2961, -44774.7152, -45881.0187, -46584.7629, -46974.5038, -47138.7974, -47166.1999, -46756.2412, -46575.1051}, 
{-516.455553, -517.573981, -518.468629, -519.142786, -519.59974, -519.842778, -519.875189, -519.70026, -519.32128, -518.741536, -517.964316, -516.992908, -515.8306, -514.480679, -512.946435, -511.231154, -509.338124, -507.270634, -505.031971, -502.625424, -500.054279, -497.321825, -494.43135, -491.386142, -488.189489, -484.844678, -481.354997, -477.723735, -473.954179, -470.049618, -466.013338, -461.848628, -457.558776, -453.14707, -448.630419, -444.005013, -439.266881, -434.418867, -429.463817, -424.404577, -419.243994, -413.984912, -408.630178, -403.182638, -397.645137, -392.020521, -386.311636, -380.521328, -374.652444, -368.707827, -362.690326, -356.629542, -350.500985, -344.30692, -338.049612, -331.731328, -325.354333, -318.920893, -312.433274, -305.893742, -299.304562, -292.668, -285.986322, -279.261794, -272.496681, -265.69325, -258.853766, -252.050592, -245.354408, -238.622909, -231.853818, -225.044856, -218.193745, -211.298206, -204.355961, -197.364731, -190.322239, -183.226206, -176.074354, -168.864404, -161.594079, -154.261099, -146.863186, -139.398063, -131.363003, -123.016479, -114.622976, -106.196458, -97.7508846, -89.3002195, -80.8584245, -72.4394618, -64.0572934, -55.7258815, -47.4591883, -39.2711759, -31.1758063, -23.1870418, -15.3188445, -7.58517654, 0, 3.86233851, 7.91077311, 12.150161, 16.5853592, 21.2212249, 26.0626153, 31.1143875, 36.3813987, 41.8685059, 47.5805663, 53.5224371, 59.6989754, 66.1150384, 72.7754831, 79.6851668, 86.8489465, 93.8168271, 100.528687, 108.018215, 116.192802, 124.959834, 134.226702, 143.900794, 153.889499, 164.100206, 174.440302, 184.817178, 195.138222, 205.310823, 214.659119, 223.887608, 233.078653, 242.314619, 252.849676, 264.180284, 275.490423, 286.510914, 296.972581, 306.606244, 315.142725, 322.410833, 328.013582, 331.662624, 333.069609, 331.946189, 334.38844, 332.127481, 323.278858, 305.958115, 278.280796, 234.676212, 181.45154, 126.932751, 72.5543152, 5.68056515, -80.0069618, -190.826728, -335.292788, -517.029013, -737.264088, -1000.20304, -1309.16538, -1651.73971, -2032.983, -2487.55135, -3036.51436, -3706.67313, -4520.58322, -5479.7847, -6597.70521, -7877.91584, -9345.21712, -11016.4485, -12908.4496, -15038.0598, -18191.9276, -21424.6312, -24560.558, -27424.0953, -29839.6302, -30939.247, -31701.1714, -32246.4919, -33048.6696, -34089.3122, -35203.2057, -36379.299, -37606.5409, -38873.8804, -40170.2664, -42371.0334, -44082.4486, -45346.5642, -46205.4322, -46701.1048, -46875.6342, -46771.0723, -46500.4007, -46782.3717}, 
{-488.567494, -489.825979, -490.870626, -491.704499, -492.33066, -492.752172, -492.9721, -492.993504, -492.819449, -492.452997, -491.897211, -491.155154, -490.22989, -489.124481, -487.841989, -486.385479, -484.758012, -482.962652, -481.002463, -478.880505, -476.599844, -474.163541, -471.57466, -468.836263, -465.951414, -462.923175, -459.75461, -456.448781, -453.008751, -449.437583, -445.738341, -441.914087, -437.967884, -433.902794, -429.742072, -425.478259, -421.103657, -416.620674, -412.031717, -407.339195, -402.545514, -397.653083, -392.66431, -387.581601, -382.407366, -377.144012, -371.793945, -366.359576, -360.84331, -355.247556, -349.574721, -343.820426, -337.994014, -332.098038, -326.135054, -320.107618, -314.018283, -307.869605, -301.664138, -295.404438, -289.09306, -282.732558, -276.325487, -269.874403, -263.381859, -256.850412, -250.282616, -243.759633, -237.36074, -230.926369, -224.453978, -217.941026, -211.384972, -204.783276, -198.133396, -191.432792, -184.678922, -177.869247, -171.001224, -164.072312, -157.079972, -150.021661, -142.89484, -135.696966, -127.900558, -119.776587, -111.602333, -103.392291, -95.1609558, -86.9228211, -78.6923817, -70.4841319, -62.3125662, -54.192179, -46.1374648, -38.162918, -30.2830331, -22.5123043, -14.8652263, -7.35629338, 0, 3.58478179, 7.35529311, 11.3160163, 15.4714338, 19.8260278, 24.3842807, 29.150675, 34.1296928, 39.3258167, 44.7435289, 50.3873117, 56.2616476, 62.3710189, 68.7199079, 75.312797, 82.1541686, 88.635326, 94.6731547, 101.659089, 109.466216, 117.967623, 127.036397, 136.545623, 146.36839, 156.377784, 166.446892, 176.448801, 186.256598, 195.743369, 203.98251, 211.940019, 219.728892, 227.462122, 236.421204, 246.134882, 255.820301, 265.239906, 274.156139, 282.331446, 289.528271, 296.695506, 302.048056, 305.116232, 305.430349, 302.520721, 301.75203, 295.361628, 281.421234, 258.00257, 223.177357, 170.35051, 107.964427, 46.0262687, -13.3005715, -83.8670565, -172.33403, -285.362336, -431.251698, -614.651607, -838.423683, -1108.92133, -1427.84517, -1786.13486, -2188.08432, -2660.61547, -3220.26362, -3884.38443, -4669.7259, -5590.02797, -6660.7321, -7908.36067, -9335.56948, -10953.9643, -12775.151, -14810.7352, -17739.1068, -20738.3916, -23653.4992, -26329.3394, -28610.822, -29733.3348, -30557.6574, -31189.9234, -32037.0203, -33086.0329, -34198.7323, -35368.469, -36588.5938, -37852.4574, -39153.4106, -41042.7934, -42715.1116, -44125.5767, -45229.4003, -45981.7938, -46337.9689, -46253.137, -46220.7832, -46294.2315}, 
{-461.473317, -463.156022, -464.612031, -465.844868, -466.858055, -467.655116, -468.239574, -468.614952, -468.784774, -468.752562, -468.52184, -468.096131, -467.478958, -466.673845, -465.684314, -464.51389, -463.166094, -461.64445, -459.952482, -458.093712, -456.071665, -453.889862, -451.551827, -449.061085, -446.421156, -443.635566, -440.707837, -437.641491, -434.440054, -431.107047, -427.645994, -424.060418, -420.353842, -416.52979, -412.636412, -408.653979, -404.562225, -400.363225, -396.059055, -391.651789, -387.143502, -382.536269, -377.832166, -373.033267, -368.141647, -363.159381, -358.088544, -352.931211, -347.689456, -342.365356, -336.960985, -331.42143, -325.80699, -320.120974, -314.366693, -308.547457, -302.666577, -296.727363, -290.733125, -284.687173, -278.592819, -272.453371, -266.272142, -260.05244, -253.797577, -247.510862, -241.195606, -234.965511, -228.93163, -222.869596, -216.775565, -210.645691, -204.476127, -198.263027, -192.002547, -185.690839, -179.324058, -172.898358, -166.409893, -159.854818, -153.229285, -146.52945, -139.751466, -132.891488, -125.346822, -117.425643, -109.44332, -101.415443, -93.3575986, -85.2853749, -77.2143598, -69.1601412, -61.1383066, -53.1644442, -45.2541415, -37.4229865, -29.686567, -22.0604708, -14.5602856, -7.20159944, 0, 2.54507037, 5.43899654, 8.67340884, 12.2399376, 16.1302132, 20.335866, 24.8485262, 29.6598242, 34.7613905, 40.1448552, 45.8018488, 51.7240015, 57.9029438, 64.330306, 70.9977183, 77.8968112, 84.6242705, 91.1153059, 98.2548745, 105.949976, 114.107609, 122.634774, 131.438469, 140.425694, 149.503449, 158.578733, 167.558545, 176.349885, 184.859752, 192.546637, 199.930501, 207.052895, 213.955372, 221.315453, 228.856705, 236.133102, 242.995407, 249.294379, 254.880779, 259.605369, 265.399237, 269.399672, 271.050417, 269.79521, 265.077794, 260.355125, 250.054423, 232.616122, 206.480658, 170.088468, 118.951181, 58.0176903, -3.17753732, -63.2258074, -137.308543, -233.158544, -358.508608, -529.781031, -744.04423, -988.887175, -1250.38817, -1552.88995, -1911.8155, -2339.03657, -2840.40753, -3424.5448, -4093.74755, -4854.99442, -5738.42737, -6761.08586, -7963.87186, -9336.29119, -10887.1232, -12625.1474, -14559.143, -17291.1881, -20088.4387, -22811.3497, -25320.3757, -27475.9715, -28591.9585, -29439.8468, -30114.3628, -30974.5318, -32010.4615, -33102.1347, -34245.1653, -35435.1672, -36667.7542, -37938.54, -39059.6054, -40470.3429, -41997.8876, -43469.3744, -44711.9381, -45552.7136, -45818.836, -46162.6346, -44748.4485}, 
{-437.574084, -439.388162, -440.985139, -442.368158, -443.540362, -444.504892, -445.264891, -445.823501, -446.183866, -446.349126, -446.322426, -446.106907, -445.705711, -445.121982, -444.358861, -443.419491, -442.307014, -441.024573, -439.57531, -437.962369, -436.18889, -434.258017, -432.172892, -429.936657, -427.552455, -425.023428, -422.35272, -419.543471, -416.598825, -413.521924, -410.315911, -406.983927, -403.529116, -399.95462, -396.285263, -392.512891, -388.629088, -384.636293, -380.536946, -376.333484, -372.028347, -367.623973, -363.122801, -358.527271, -353.83982, -349.062889, -344.198914, -339.250336, -334.219593, -329.109124, -323.921367, -318.617448, -313.242015, -307.798402, -302.289944, -296.719975, -291.09183, -285.408842, -279.674348, -273.89168, -268.064173, -262.195162, -256.287982, -250.345965, -244.372448, -238.370765, -232.344249, -226.417771, -220.713284, -214.983463, -209.223764, -203.429642, -197.596554, -191.719955, -185.795302, -179.818048, -173.783652, -167.687567, -161.52525, -155.292157, -148.983744, -142.595466, -136.122778, -129.561137, -122.26531, -114.564572, -106.795901, -98.9755432, -91.1197471, -83.2447595, -75.3668278, -67.5021993, -59.6671214, -51.8778415, -44.1506068, -36.5016647, -28.9472625, -21.5036476, -14.1870674, -7.01376903, 0, 2.07218869, 4.54629374, 7.40902556, 10.6470946, 14.2472112, 18.1960859, 22.480429, 27.086951, 32.0023622, 37.2133732, 42.7066943, 48.469036, 54.4871086, 60.7476226, 67.2372884, 73.9428164, 80.5519288, 87.0086234, 93.9758935, 101.376381, 109.132727, 117.167573, 125.40356, 133.763331, 142.169526, 150.544787, 158.811756, 166.893075, 174.711383, 181.95654, 188.869325, 195.442214, 201.667684, 207.881951, 213.905622, 219.467639, 224.457359, 228.764135, 232.277324, 234.886279, 239.081103, 241.358872, 241.100099, 237.685297, 230.494978, 220.222758, 204.607772, 182.702259, 153.558455, 116.228599, 70.1883439, 13.5490032, -49.3468511, -119.11008, -206.4648, -317.594483, -458.682602, -645.95015, -873.261821, -1123.53229, -1372.14871, -1656.87229, -2007.0279, -2439.49317, -2950.05467, -3544.18013, -4220.99785, -4984.332, -5861.25124, -6865.67585, -8026.45228, -9340.27826, -10815.9073, -12462.0928, -14287.5884, -16848.045, -19468.5943, -22021.2653, -24378.0872, -26411.089, -27485.1743, -28317.5814, -28997.6791, -29875.2012, -30936.4551, -32059.2625, -33235.3551, -34456.4644, -35714.3221, -37000.6597, -37631.1091, -38844.9883, -40400.5921, -42056.2153, -43570.1527, -44700.6991, -45206.1494, -45713.5798, -44381.6453}, 
{-417.028928, -418.669942, -420.125465, -421.397477, -422.487958, -423.398887, -424.132245, -424.690012, -425.074168, -425.286693, -425.329566, -425.204767, -424.914277, -424.460076, -423.844143, -423.068459, -422.135003, -421.045756, -419.802697, -418.407806, -416.863063, -415.170449, -413.331943, -411.349526, -409.225176, -406.960875, -404.558601, -402.020336, -399.348059, -396.54375, -393.609389, -390.546955, -387.35843, -384.045792, -380.568576, -376.950877, -373.217301, -369.371206, -365.41595, -361.35489, -357.191383, -352.928787, -348.570459, -344.119757, -339.580038, -334.954659, -330.246979, -325.460354, -320.598143, -315.663701, -310.660388, -305.628124, -300.53291, -295.37731, -290.163891, -284.895216, -279.573851, -274.202359, -268.783307, -263.319259, -257.812779, -252.266433, -246.682785, -241.0644, -235.413843, -229.733679, -224.026472, -218.402092, -212.967829, -207.504944, -202.009045, -196.47574, -190.900639, -185.279351, -179.607483, -173.880645, -168.094446, -162.244494, -156.326398, -150.335767, -144.26821, -138.119335, -131.884751, -125.560067, -118.501837, -111.03864, -103.506733, -95.9224636, -88.3021792, -80.6622269, -73.0189542, -65.3887082, -57.7878362, -50.2326856, -42.7396035, -35.3249374, -28.0050344, -20.7962419, -13.7149072, -6.77737742, 0, 2.28780742, 4.8940391, 7.81039329, 11.0285682, 14.540262, 18.337173, 22.4109994, 26.7534394, 31.3561912, 36.2109531, 41.3094232, 46.6432999, 52.2042812, 57.9840655, 63.9743509, 70.1668358, 76.1725182, 81.9287107, 88.2882188, 95.1611619, 102.45766, 110.087832, 117.961799, 125.989679, 134.081592, 142.147659, 150.097998, 157.84273, 165.291974, 172.108196, 178.539976, 184.571729, 190.187869, 195.779056, 201.126583, 205.903412, 209.972088, 213.195151, 215.435146, 216.554614, 219.170198, 219.552136, 217.024127, 210.909866, 200.533051, 183.129653, 160.633026, 132.888801, 99.7426062, 61.040071, 21.4751605, -29.8805768, -97.8858208, -186.090732, -295.373908, -427.813427, -585.487362, -774.896302, -992.690694, -1230.69642, -1467.84414, -1740.65065, -2076.84931, -2493.76683, -2991.09641, -3576.62531, -4258.95455, -5046.08236, -5943.02327, -6956.47956, -8081.64973, -9338.50966, -10737.7437, -12290.0363, -14006.0718, -16417.792, -18884.3095, -21285.9942, -23503.2162, -25416.3455, -26417.8123, -27201.2195, -27856.054, -28756.5453, -29879.4699, -31082.9615, -32349.7285, -33662.4795, -35003.9231, -36356.7678, -36898.4413, -38043.0873, -39549.9005, -41178.0753, -42686.8065, -43835.2887, -44382.7163, -44804.0947, -45367.8811}, 
{-396.733209, -398.600028, -400.274243, -401.758166, -403.054109, -404.164383, -405.0913, -405.837172, -406.404309, -406.795024, -407.011628, -407.056432, -406.931748, -406.639888, -406.183164, -405.563886, -404.784366, -403.846916, -402.753847, -401.507471, -400.1101, -398.564045, -396.871617, -395.035128, -393.05689, -390.939214, -388.684411, -386.294794, -383.772674, -381.120362, -378.34017, -375.434409, -372.405392, -369.255428, -365.95619, -362.525953, -358.983362, -355.331724, -351.574343, -347.714527, -343.755582, -339.700812, -335.553524, -331.317024, -326.994617, -322.589611, -318.10531, -313.54502, -308.912048, -304.209699, -299.44128, -294.659839, -289.817861, -284.917573, -279.961204, -274.950979, -269.889128, -264.777878, -259.619455, -254.416089, -249.170005, -243.883433, -238.558599, -233.197732, -227.803058, -222.376805, -216.921201, -211.538006, -206.326589, -201.083901, -195.805716, -190.487811, -185.12596, -179.715939, -174.253523, -168.734488, -163.154608, -157.509659, -151.795416, -146.007655, -140.142151, -134.19468, -128.161015, -122.036934, -115.178667, -107.915213, -100.583258, -93.1993331, -85.779967, -78.3416895, -70.9010306, -63.4745198, -56.078687, -48.7300619, -41.4451742, -34.2405537, -27.1327302, -20.1382333, -13.2735928, -6.55533844, 0, 2.0932669, 4.50337142, 7.22071899, 10.235715, 13.5387649, 17.1202741, 20.9706479, 25.0802918, 29.4396113, 34.0390116, 38.8688983, 43.9196767, 49.1817522, 54.6455303, 60.3014164, 66.1398158, 71.8587776, 77.406942, 83.4359967, 89.8736994, 96.6478076, 103.686079, 110.916271, 118.266142, 125.663448, 133.035948, 140.3114, 147.41756, 154.282187, 160.753538, 166.86802, 172.577242, 177.83281, 182.912821, 187.605637, 191.612559, 194.787247, 196.983361, 198.054562, 197.854511, 199.508398, 198.602672, 194.350912, 185.966697, 172.663605, 147.346773, 117.115334, 82.7599777, 45.0713935, 4.84027163, -28.0665382, -73.0275903, -141.597313, -239.735162, -361.97752, -507.989499, -677.436208, -871.105301, -1087.28422, -1323.03582, -1560.25013, -1832.83299, -2167.15594, -2579.75468, -3070.49857, -3646.90713, -4318.25398, -5092.51339, -5971.22779, -6959.5778, -8050.81386, -9263.88223, -10608.0932, -12092.7572, -13727.1845, -16003.4279, -18327.3697, -20587.6344, -22672.8469, -24471.6318, -25407.5697, -26144.3588, -26769.9284, -27662.6676, -28795.532, -30020.4523, -31316.4352, -32662.4875, -34037.6162, -35420.828, -35841.4969, -36952.8107, -38484.0445, -40164.4733, -41723.3723, -42890.0165, -43393.6812, -43757.4607, -45638.9499}, 
{-377.094882, -379.496162, -381.666689, -383.610327, -385.330937, -386.832381, -388.118521, -389.19322, -390.060338, -390.723738, -391.187283, -391.454834, -391.530253, -391.417402, -391.120143, -390.642338, -389.98785, -389.16054, -388.16427, -387.002902, -385.680298, -384.200321, -382.566832, -380.783693, -378.854766, -376.783914, -374.574998, -372.23188, -369.758422, -367.158487, -364.435936, -361.594631, -358.638435, -355.571208, -352.443279, -349.234298, -345.923361, -342.51282, -339.005032, -335.402349, -331.707126, -327.921718, -324.048477, -320.089759, -316.047917, -311.925306, -307.72428, -303.447193, -299.096398, -294.674252, -290.183106, -285.62267, -280.998001, -276.311511, -271.565611, -266.762713, -261.905228, -256.995567, -252.036144, -247.029367, -241.977651, -236.883405, -231.749042, -226.576973, -221.369609, -216.129362, -210.858644, -205.662298, -200.64271, -195.591031, -190.503033, -185.374487, -180.201164, -174.978835, -169.703272, -164.370246, -158.975528, -153.51489, -147.984102, -142.378936, -136.695164, -130.928555, -125.074883, -119.129917, -112.438178, -105.33476, -98.1625874, -90.9385647, -83.6795978, -76.4025924, -69.1244542, -61.8620887, -54.6324015, -47.4522983, -40.3386848, -33.3084664, -26.3785489, -19.5658379, -12.887239, -6.35965784, 0, 1.45281493, 3.30909871, 5.55145621, 8.1624923, 11.1248118, 14.4210197, 18.0337207, 21.9455199, 26.1390219, 30.5968317, 35.3015542, 40.2357942, 45.3821566, 50.7232463, 56.2416681, 61.920027, 67.6777425, 73.4883932, 79.4775027, 85.6141362, 91.8673588, 98.2062358, 104.599832, 111.017214, 117.427445, 123.799591, 130.102718, 136.305891, 142.378174, 148.519633, 154.383633, 159.86994, 164.878318, 169.426174, 173.354452, 176.472726, 178.645468, 179.73715, 179.612244, 178.135224, 179.180114, 177.381536, 171.819484, 161.573952, 145.724936, 113.099501, 75.5938031, 34.8510689, -7.48547455, -49.7726005, -78.2805007, -118.225018, -183.844508, -282.422783, -407.044795, -557.170493, -732.259829, -932.882491, -1157.13648, -1401.90916, -1649.78649, -1932.58217, -2277.33807, -2699.21436, -3193.23846, -3763.67907, -4411.21523, -5139.185, -5964.08839, -6894.98028, -7952.57416, -9132.47263, -10439.6949, -11879.2601, -13456.1876, -15607.0618, -17797.4453, -19924.4658, -21885.2511, -23576.9292, -24461.0628, -25160.7214, -25759.7035, -26615.3826, -27703.267, -28884.8759, -30141.4172, -31454.0988, -32804.1287, -34172.715, -34407.841, -35503.8044, -37128.5602, -38950.0637, -40636.2701, -41855.1345, -42274.6122, -42676.7178, -45006.7268}, 
{-361.453756, -363.990002, -366.28682, -368.348611, -370.179778, -371.784724, -373.167849, -374.333558, -375.286252, -376.030333, -376.570204, -376.910266, -377.054923, -377.008577, -376.775629, -376.360482, -375.767539, -375.001202, -374.065872, -372.965953, -371.705847, -370.289955, -368.72268, -367.008425, -365.151592, -363.156583, -361.0278, -358.769646, -356.386523, -353.882833, -351.262979, -348.531362, -345.692386, -342.750452, -339.794074, -336.78383, -333.679286, -330.482115, -327.193989, -323.816582, -320.351565, -316.800611, -313.165394, -309.447586, -305.64886, -301.770888, -297.815343, -293.783898, -289.678226, -285.499998, -281.250889, -276.89038, -272.46325, -267.972084, -263.419471, -258.807998, -254.140253, -249.418821, -244.646292, -239.825252, -234.958288, -230.047988, -225.096939, -220.107729, -215.082944, -210.025173, -204.937001, -199.928724, -195.108048, -190.255425, -185.366459, -180.436757, -175.461923, -170.437562, -165.35928, -160.222681, -155.023372, -149.756956, -144.419039, -139.005227, -133.511124, -127.932336, -122.264468, -116.503125, -109.972652, -103.018404, -95.9938025, -88.9162363, -81.8030943, -74.6717652, -67.5396379, -60.4241011, -53.3425437, -46.3123546, -39.3509224, -32.475636, -25.7038843, -19.053056, -12.54054, -6.18372506, 0, 0.958456258, 2.37442933, 4.22589843, 6.49084279, 9.14724163, 12.1730742, 15.5463197, 19.2449573, 23.2469663, 27.5303259, 32.0730153, 36.8530137, 41.8483005, 47.0368547, 52.3966556, 57.9056825, 63.5410702, 69.2806775, 75.1043932, 80.9900155, 86.9153428, 92.8581734, 98.7963057, 104.707538, 110.569668, 116.360495, 122.057817, 127.639432, 133.083138, 138.664137, 143.953777, 148.840634, 153.213287, 157.112738, 160.351353, 162.700848, 164.004074, 164.103879, 162.843115, 160.06463, 159.665599, 156.200622, 148.71931, 136.271277, 117.906134, 81.9537721, 40.8634554, -3.63527459, -49.8128754, -95.9398049, -128.517586, -171.969865, -239.027784, -336.542728, -459.361571, -607.720969, -781.857575, -983.187474, -1210.49984, -1461.29719, -1721.46733, -2017.9507, -2375.48769, -2806.96905, -3305.2071, -3872.23048, -4503.2219, -5198.43513, -5983.22573, -6868.75026, -7891.52339, -9033.44234, -10296.8859, -11684.2328, -13197.8619, -15231.5991, -17298.5142, -19303.1241, -21149.9459, -22743.4964, -23579.9208, -24244.3558, -24816.335, -25630.0109, -26664.1303, -27791.3488, -28995.7175, -30261.288, -31572.1116, -32912.2396, -33083.9949, -34190.2067, -35868.7411, -37757.4643, -39494.2427, -40716.9422, -41063.4293, -41605.2793, -44264.6257}, 
{-350.461653, -352.552208, -354.440298, -356.129245, -357.622372, -358.923002, -360.034457, -360.960061, -361.703137, -362.267006, -362.654992, -362.870418, -362.916606, -362.796879, -362.514561, -362.072973, -361.475439, -360.725281, -359.825822, -358.780386, -357.592294, -356.264869, -354.801435, -353.205315, -351.47983, -349.628304, -347.654059, -345.560419, -343.350706, -341.028243, -338.596352, -336.058357, -333.417581, -330.677345, -327.893311, -325.041532, -322.097431, -319.062634, -315.938763, -312.727445, -309.430303, -306.048961, -302.585043, -299.040175, -295.415981, -291.714084, -287.936109, -284.08368, -280.158423, -276.16196, -272.095917, -267.917208, -263.673136, -259.366296, -254.999279, -250.574681, -246.095094, -241.563113, -236.98133, -232.352339, -227.678734, -222.963109, -218.208056, -213.41617, -208.590044, -203.732272, -198.845447, -194.0452, -189.44445, -184.812657, -180.145088, -175.437007, -170.683681, -165.880376, -161.022357, -156.10489, -151.12324, -146.072675, -140.948459, -135.745857, -130.460137, -125.086564, -119.620402, -114.05692, -107.692356, -100.886193, -94.006313, -87.0706669, -80.0972051, -73.103878, -66.108636, -59.1294296, -52.1842093, -45.2909255, -38.4675287, -31.7319693, -25.1021978, -18.5961647, -12.2318204, -6.02711531, 0, 0.770106128, 1.98803411, 3.63273103, 5.683144, 8.1182201, 10.9169064, 14.0581501, 17.5208981, 21.2840977, 25.3266958, 29.6276397, 34.1658763, 38.9203528, 43.8700162, 48.9938138, 54.2706925, 59.5067797, 64.6563341, 70.0892043, 75.7473048, 81.5725498, 87.5068536, 93.4921305, 99.4702948, 105.383261, 111.172943, 116.781255, 122.150112, 127.221428, 131.982581, 136.313351, 140.142014, 143.396845, 146.497857, 149.127457, 150.83692, 151.407, 150.618452, 148.252028, 144.088484, 141.116683, 134.932887, 124.690175, 109.541628, 88.6403254, 54.0630552, 13.808262, -31.2019012, -80.0452816, -131.799726, -177.651539, -234.21533, -309.045564, -406.838405, -526.454319, -669.373883, -837.077672, -1031.56275, -1253.67581, -1503.70015, -1773.94451, -2083.77257, -2452.04253, -2888.77304, -3389.00478, -3954.65356, -4582.04766, -5271.65429, -6044.42824, -6909.73536, -7896.09126, -8990.76591, -10196.4964, -11516.0198, -12952.0733, -14876.598, -16830.8259, -18725.193, -20470.1353, -21976.0888, -22767.6866, -23398.3696, -23943.9181, -24719.9282, -25707.2528, -26786.8217, -27944.484, -29166.0888, -30437.4853, -31744.5227, -32005.8743, -33139.1305, -34799.163, -36640.8432, -38319.0426, -39488.6327, -39804.485, -40591.0575, -43638.8317}, 
{-341.290358, -342.814612, -344.182012, -345.39449, -346.453976, -347.362404, -348.121706, -348.733813, -349.200657, -349.524172, -349.706288, -349.748937, -349.654053, -349.423566, -349.05941, -348.563516, -347.937815, -347.184241, -346.304725, -345.301199, -344.175596, -342.929847, -341.565884, -340.08564, -338.491047, -336.784036, -334.96654, -333.040491, -331.007821, -328.870461, -326.630345, -324.289403, -321.849569, -319.312773, -316.687023, -313.971086, -311.163655, -308.266466, -305.281254, -302.209753, -299.0537, -295.814827, -292.494871, -289.095567, -285.618649, -282.065852, -278.438912, -274.739563, -270.96954, -267.130578, -263.224413, -259.215688, -255.144034, -251.011988, -246.82209, -242.57688, -238.278895, -233.930676, -229.53476, -225.093688, -220.609997, -216.086228, -211.524918, -206.928608, -202.299835, -197.641139, -192.955059, -188.362297, -183.980719, -179.569161, -175.122501, -170.635618, -166.103391, -161.5207, -156.882423, -152.183439, -147.418627, -142.582866, -137.671035, -132.678013, -127.598679, -122.427911, -117.160589, -111.791592, -105.587681, -98.9231076, -92.1808774, -85.3794977, -78.5374765, -71.6733213, -64.80554, -57.9526404, -51.1331302, -44.3655172, -37.6683092, -31.0600139, -24.559139, -18.1841925, -11.9536819, -5.88611519, 0, 0.63719018, 1.70573567, 3.18605001, 5.05854675, 7.30363944, 9.90174161, 12.8332668, 16.0786286, 19.6182405, 23.4325161, 27.5018689, 31.8067125, 36.3274603, 41.044526, 45.9383231, 50.9892652, 55.8264361, 60.3800595, 65.4252231, 70.8670554, 76.6106851, 82.5612408, 88.6238512, 94.7036448, 100.70575, 106.535297, 112.097412, 117.297225, 122.039864, 126.030598, 129.447698, 132.256251, 134.421343, 136.719332, 138.709669, 139.735468, 139.518433, 137.78027, 134.242685, 128.627383, 122.866577, 113.798702, 100.712974, 82.8986089, 59.6448211, 27.5617335, -10.7125729, -55.2191096, -105.998888, -163.092921, -222.973967, -293.612486, -377.136752, -475.934215, -592.921471, -730.777541, -892.181443, -1079.71688, -1296.17985, -1544.47035, -1823.02104, -2143.93712, -2519.7792, -2957.86523, -3456.62984, -4018.58523, -4643.0254, -5331.62822, -6097.8717, -6948.559, -7898.85278, -8945.91068, -10093.6422, -11345.9569, -12706.7643, -14534.5665, -16390.0324, -18188.423, -19844.9999, -21275.0242, -22027.6203, -22628.278, -23149.1745, -23887.7917, -24827.1238, -25856.2879, -26962.9718, -28134.8636, -29359.6511, -30625.0223, -31004.4718, -32158.7667, -33773.8058, -35535.4876, -37129.7109, -38242.3743, -38559.3764, -39635.7087, -42966.5106}, 
{-330.401283, -331.738324, -332.934065, -333.990067, -334.907895, -335.689109, -336.335272, -336.847947, -337.228697, -337.479083, -337.600668, -337.595014, -337.463685, -337.208241, -336.830247, -336.331264, -335.712854, -334.97658, -334.124004, -333.15669, -332.076199, -330.884093, -329.581936, -328.171289, -326.653715, -325.030776, -323.304035, -321.475055, -319.545397, -317.516624, -315.390299, -313.167984, -310.851241, -308.441632, -305.93725, -303.341464, -300.657689, -297.887598, -295.032867, -292.095171, -289.076185, -285.977585, -282.801045, -279.54824, -276.220845, -272.820536, -269.348988, -265.807876, -262.198874, -258.523658, -254.783903, -250.947417, -247.050477, -243.095492, -239.08487, -235.021021, -230.906355, -226.743279, -222.534204, -218.281539, -213.987694, -209.655076, -205.286096, -200.883162, -196.448685, -191.985073, -187.494735, -183.100444, -178.922082, -174.713819, -170.470261, -166.186015, -161.855687, -157.473884, -153.035213, -148.534279, -143.96569, -139.324052, -134.603972, -129.800056, -124.90691, -119.919143, -114.831359, -109.638165, -103.58572, -97.0585963, -90.4503638, -83.7799179, -77.0661536, -70.3279661, -63.5842507, -56.8539025, -50.1558167, -43.5088886, -36.9320132, -30.4440858, -24.0640015, -17.8106556, -11.7029433, -5.75975969, 0, 0.307590643, 1.08807832, 2.31767558, 3.97259497, 6.02904905, 8.46325037, 11.2514115, 14.3697449, 17.7944633, 21.5017791, 25.4679048, 29.6690532, 34.0814366, 38.6812677, 43.4447589, 48.348123, 53.1657348, 57.8449731, 62.8185876, 68.0195399, 73.3807917, 78.8353046, 84.3160403, 89.7559605, 95.0880268, 100.245201, 105.160445, 109.766719, 113.996987, 117.866525, 121.195797, 123.893069, 125.866608, 127.601708, 128.718122, 128.681729, 127.227688, 124.091158, 119.007297, 111.711264, 103.9432, 92.8230686, 77.6937495, 57.8981221, 32.7790656, -0.32672054, -39.568632, -85.1022444, -137.083133, -195.666875, -257.500268, -330.536169, -417.395966, -520.588489, -642.392756, -785.190965, -951.365314, -1142.96144, -1362.7746, -1613.96719, -1896.01977, -2218.68625, -2591.53424, -3020.76861, -3506.8963, -4053.0397, -4660.15674, -5330.80866, -6075.49297, -6900.21796, -7816.46146, -8823.92079, -9926.71095, -11128.947, -12434.7438, -14189.2, -15970.2012, -17696.6166, -19287.3151, -20661.166, -21387.5057, -21967.7578, -22470.5911, -23175.036, -24066.4933, -25042.713, -26093.4785, -27208.573, -28377.7799, -29590.8824, -29940.0299, -31056.6368, -32635.3061, -34370.641, -35957.2445, -37089.7198, -37462.6701, -38805.1868, -42196.5757}, 
{-319.712611, -320.979408, -322.110652, -323.107862, -323.972559, -324.706264, -325.310497, -325.786779, -326.13663, -326.36157, -326.463121, -326.442802, -326.302134, -326.042638, -325.665833, -325.173241, -324.566382, -323.846777, -323.015945, -322.075408, -321.026685, -319.871298, -318.610767, -317.246612, -315.780353, -314.213512, -312.547609, -310.784163, -308.924696, -306.970729, -304.923781, -302.785373, -300.557026, -298.240259, -295.8346, -293.342607, -290.766865, -288.108958, -285.370472, -282.552992, -279.658103, -276.68739, -273.642439, -270.524835, -267.336162, -264.078007, -260.751954, -257.359588, -253.902495, -250.38226, -246.800468, -243.128394, -239.39859, -235.613299, -231.774763, -227.885224, -223.946926, -219.962109, -215.933016, -211.861891, -207.750974, -203.602508, -199.418735, -195.201898, -190.95424, -186.678001, -182.375425, -178.170129, -174.182815, -170.165399, -166.112256, -162.017759, -157.876283, -153.682201, -149.429889, -145.113719, -140.728066, -136.267303, -131.725806, -127.097948, -122.378103, -117.560645, -112.639948, -107.610387, -101.70061, -95.303959, -88.8229813, -82.2768997, -75.6849377, -69.0663185, -62.4402654, -55.8260017, -49.2427508, -42.7097359, -36.2461804, -29.8713076, -23.6043408, -17.4645034, -11.4710185, -5.64310964, 0, -0.0467632914, 0.432694108, 1.40909645, 2.85316797, 4.73563292, 7.02721556, 9.69864012, 12.7206309, 16.063912, 19.6992078, 23.5972426, 27.7287405, 32.0644258, 36.5750227, 41.2312556, 46.0038486, 50.8866223, 55.8536003, 60.8232654, 65.771291, 70.6733506, 75.5051177, 80.2422655, 84.8604677, 89.3353976, 93.6427288, 97.7581345, 101.657288, 105.315864, 109.190894, 112.600193, 115.375029, 117.346665, 118.555322, 118.727788, 117.535131, 114.74593, 110.128766, 103.452217, 94.4848631, 85.0270551, 72.197236, 55.3664648, 33.9058005, 7.18630228, -27.8811454, -68.8502655, -115.734955, -168.549112, -227.306633, -287.282093, -359.021218, -447.422099, -555.849811, -684.539509, -835.13162, -1009.26657, -1208.30116, -1434.2239, -1689.3327, -1971.95947, -2291.80674, -2657.13177, -3073.82165, -3543.74727, -4070.62304, -4656.66611, -5305.20271, -6025.04889, -6821.91534, -7706.14045, -8678.09527, -9741.8886, -10901.6292, -12161.426, -13854.9364, -15574.3333, -17241.3383, -18777.6731, -20105.0594, -20809.9363, -21372.8294, -21859.1528, -22530.5228, -23374.6901, -24297.6543, -25291.2536, -26347.3262, -27457.7101, -28614.2437, -28906.6251, -29968.3029, -31502.7146, -33213.2983, -34803.4916, -35976.7324, -36436.4584, -38041.0861, -41377.5298}, 
{-309.752308, -310.997801, -312.108907, -313.087257, -313.93448, -314.652203, -315.242056, -315.705668, -316.044668, -316.260685, -316.355347, -316.330284, -316.187124, -315.927497, -315.553031, -315.065355, -314.466098, -313.756889, -312.939357, -312.015131, -310.985839, -309.853111, -308.618576, -307.283862, -305.850599, -304.320415, -302.694939, -300.9758, -299.164627, -297.263049, -295.272694, -293.195193, -291.032173, -288.785264, -286.460496, -284.057205, -281.574673, -279.014386, -276.377831, -273.666493, -270.881858, -268.025413, -265.098643, -262.103035, -259.040075, -255.911248, -252.718041, -249.46194, -246.14443, -242.766999, -239.331132, -235.811709, -232.237399, -228.610264, -224.932368, -221.205773, -217.432543, -213.614739, -209.754425, -205.853664, -201.914518, -197.939051, -193.929324, -189.887401, -185.815345, -181.715219, -177.589084, -173.560805, -169.751317, -165.911483, -162.035469, -158.117442, -154.15157, -150.132019, -146.052956, -141.908548, -137.692963, -133.400367, -129.024928, -124.560813, -120.002188, -115.343221, -110.578079, -105.700929, -99.9243092, -93.6496423, -87.287744, -80.8581466, -74.3803825, -67.8739841, -61.3584839, -54.853414, -48.3783071, -41.9526953, -35.5961112, -29.3280871, -23.1681554, -17.1358484, -11.2506986, -5.53223835, 0, -0.361976439, -0.150571994, 0.599855268, 1.85494728, 3.58034597, 5.74169327, 8.30463111, 11.2348014, 14.4978462, 18.0594072, 21.8851265, 25.9406461, 30.1916077, 34.6036535, 39.1424252, 43.7735649, 48.6831996, 53.8684693, 58.7962992, 63.4795781, 67.9311948, 72.1640381, 76.1909966, 80.0249593, 83.6788147, 87.1654518, 90.4977593, 93.688626, 96.7509406, 100.521549, 103.887265, 106.61379, 108.466828, 109.09604, 108.325147, 106.008815, 101.947557, 95.941887, 87.7923178, 77.2993631, 66.3776182, 52.0700985, 33.7636928, 10.8452899, -17.2982215, -54.5262512, -97.3940373, -145.703116, -199.255025, -257.8513, -315.049858, -384.526946, -473.730505, -587.011593, -722.398471, -880.758205, -1062.95786, -1269.95244, -1502.5011, -1761.26705, -2042.1653, -2356.373, -2712.25878, -3115.95751, -3569.81911, -4077.93084, -4643.2375, -5269.53016, -5964.78897, -6734.62449, -7589.30012, -8529.05779, -9557.89745, -10679.8191, -11898.8227, -13538.3811, -15203.1532, -16817.2707, -18304.8654, -19590.069, -20274.1142, -20819.2979, -21288.1372, -21926.6986, -22724.8442, -23595.9573, -24533.7241, -25531.8308, -26583.9633, -27683.8079, -27938.7601, -28945.7523, -30424.6603, -32095.3598, -33677.7266, -34891.6365, -35456.9652, -37316.6838, -40532.5031}, 
{-301.815533, -302.944605, -303.946577, -304.82298, -305.575342, -306.205194, -306.714065, -307.103486, -307.374985, -307.530094, -307.570342, -307.497258, -307.312373, -307.017216, -306.613317, -306.102206, -305.485414, -304.764468, -303.940901, -303.01624, -301.992017, -300.869761, -299.651001, -298.337268, -296.930092, -295.431002, -293.841528, -292.1632, -290.397548, -288.546102, -286.610391, -284.591946, -282.492296, -280.31297, -278.059552, -275.73146, -273.328062, -270.850759, -268.300947, -265.680026, -262.989393, -260.230448, -257.404587, -254.513211, -251.557716, -248.539502, -245.459966, -242.320507, -239.122523, -235.867413, -232.556575, -229.166509, -225.724051, -222.23114, -218.689713, -215.101708, -211.469064, -207.793719, -204.077611, -200.322678, -196.530857, -192.704088, -188.844307, -184.953454, -181.033466, -177.086281, -173.113838, -169.240891, -165.589246, -161.907541, -158.189752, -154.429856, -150.621829, -146.759648, -142.837289, -138.848727, -134.787941, -130.648905, -126.425597, -122.111992, -117.702068, -113.189799, -108.569164, -103.834138, -98.1795797, -92.0158329, -85.7628459, -79.440528, -73.0687884, -66.6675365, -60.2566814, -53.8561325, -47.4857991, -41.1655903, -34.9154156, -28.755184, -22.7048051, -16.7841879, -11.0132418, -5.4118761, 0, -0.348100344, -0.154253078, 0.54881337, 1.72837057, 3.3516901, 5.38604353, 7.79870244, 10.5569384, 13.6280229, 16.9792277, 20.5778242, 24.3910841, 28.3862788, 32.5306801, 36.7915594, 41.1361883, 45.5597656, 50.0335526, 54.4616519, 58.8173198, 63.073812, 67.2043846, 71.1822935, 74.9807948, 78.5731444, 81.9325983, 85.0324125, 87.8458429, 90.3461456, 92.9972966, 95.101901, 96.4859989, 96.9756301, 96.6352709, 95.1717432, 92.2282855, 87.5594069, 80.9196165, 72.0634232, 60.7453362, 48.38286, 32.5613789, 12.7100333, -11.7420364, -41.3656901, -79.0178074, -122.411723, -171.546791, -226.422367, -287.037805, -348.085796, -421.358282, -512.872872, -626.442307, -761.377716, -919.01136, -1100.6755, -1309.19359, -1544.06778, -1803.1735, -2079.48132, -2387.12874, -2737.35058, -3137.20104, -3586.65035, -4088.92042, -4645.8174, -5260.19613, -5940.10251, -6690.64605, -7520.53645, -8430.72923, -9425.08772, -10507.4752, -11681.7551, -13258.7924, -14858.6982, -16408.5856, -17835.5675, -19066.7568, -19718.7666, -20236.2095, -20679.27, -21281.9573, -22035.6499, -22859.2995, -23747.9059, -24696.4691, -25699.9891, -26753.4658, -27105.6424, -28104.766, -29511.9744, -31088.4058, -32595.1981, -33793.4893, -34444.4174, -36566.9857, -39718.7795}, 
{-294.951689, -295.961205, -296.850493, -297.620992, -298.274142, -298.811381, -299.234149, -299.543885, -299.742028, -299.830017, -299.809292, -299.681292, -299.447456, -299.109223, -298.668033, -298.125324, -297.482536, -296.741108, -295.90248, -294.96809, -293.939377, -292.817782, -291.604742, -290.301698, -288.910088, -287.431353, -285.86693, -284.218259, -282.486779, -280.67393, -278.781151, -276.809881, -274.761559, -272.637624, -270.443368, -268.178224, -265.841576, -263.434739, -260.959027, -258.415754, -255.806234, -253.131782, -250.393712, -247.593337, -244.731973, -241.810933, -238.831531, -235.795082, -232.7029, -229.556299, -226.356593, -223.081488, -219.75642, -216.383212, -212.963692, -209.499684, -205.993016, -202.445513, -198.859001, -195.235305, -191.576253, -187.883669, -184.159381, -180.405213, -176.622991, -172.814542, -168.981692, -165.25016, -161.742693, -158.205594, -154.632656, -151.017675, -147.354445, -143.63676, -139.858414, -136.013202, -132.094919, -128.097358, -124.014314, -119.839582, -115.566956, -111.19023, -106.703198, -102.099655, -96.5563922, -90.4928892, -84.3384473, -78.1133741, -71.8379775, -65.532565, -59.2174446, -52.9129239, -46.6393106, -40.4169124, -34.2660372, -28.2069926, -22.2600864, -16.4456263, -10.7839201, -5.29527539, 0, -0.304647195, -0.107191383, 0.562142938, 1.67313127, 3.19554911, 5.09917196, 7.35377532, 9.9291347, 12.7950256, 15.9212235, 19.2775039, 22.8336423, 26.5594143, 30.4245953, 34.3989608, 38.4522863, 42.3199567, 45.9382632, 49.8071505, 53.8461675, 57.9748631, 62.1127862, 66.1794859, 70.0945111, 73.7774108, 77.1477338, 80.1250291, 82.6288457, 84.5787326, 85.9254779, 86.5459369, 86.3502868, 85.2487047, 83.8563799, 81.7309836, 78.2421836, 73.0886534, 65.9690667, 56.5820969, 44.6264176, 31.0095336, 13.853382, -7.37987412, -33.2280716, -64.2290474, -102.053847, -145.823797, -195.793431, -252.217285, -315.349893, -381.528236, -459.712524, -553.723809, -666.53442, -799.383874, -954.289684, -1133.26936, -1341.28696, -1576.7438, -1834.82677, -2105.77546, -2406.74343, -2752.27135, -3150.40826, -3598.20351, -4097.75548, -4649.2967, -5254.44179, -5921.64695, -6655.49838, -7463.13002, -8346.18842, -9308.3779, -10353.4028, -11484.9674, -13002.0077, -14539.1885, -16026.406, -17393.5569, -18570.5373, -19187.5083, -19673.9253, -20088.1431, -20655.1025, -21367.2177, -22147.4915, -22991.8085, -23896.0531, -24856.1098, -25867.8633, -26375.6845, -27382.0585, -28703.5394, -30156.6816, -31558.0393, -32724.1667, -33471.6181, -35830.0032, -38902.4158}, 
{-289.008112, -289.903051, -290.683809, -291.351755, -291.908259, -292.354689, -292.692415, -292.922806, -293.047231, -293.067058, -292.983658, -292.798398, -292.512649, -292.127779, -291.645158, -291.066153, -290.392136, -289.624474, -288.764537, -287.813694, -286.773314, -285.644765, -284.429418, -283.128641, -281.743804, -280.276275, -278.727423, -277.098618, -275.391229, -273.606624, -271.746174, -269.811246, -267.803211, -265.723436, -263.577396, -261.364321, -259.083392, -256.735846, -254.322918, -251.845844, -249.30586, -246.704203, -244.042107, -241.32081, -238.541546, -235.705552, -232.814064, -229.868317, -226.869548, -223.818992, -220.717886, -217.544988, -214.324499, -211.058141, -207.747638, -204.394712, -201.001088, -197.568489, -194.098636, -190.593255, -187.054067, -183.482796, -179.881166, -176.250898, -172.593717, -168.911346, -165.205508, -161.603034, -158.227752, -154.823358, -151.383466, -147.901689, -144.371639, -140.786929, -137.141173, -133.427983, -129.640972, -125.773752, -121.819938, -117.773142, -113.626976, -109.375054, -105.010988, -100.528392, -95.0853071, -89.1107067, -83.0436077, -76.9047389, -70.714829, -64.494607, -58.2648016, -52.0461416, -45.8593559, -39.7251732, -33.6643223, -27.6975322, -21.8455315, -16.1290492, -10.5688139, -5.18555459, 0, -0.321690602, -0.170022857, 0.426633982, 1.43991066, 2.84143794, 4.60284655, 6.69576725, 9.09183078, 11.7626679, 14.6799093, 17.8151859, 21.1401282, 24.6263671, 28.2455334, 31.9692577, 35.7691709, 39.1280948, 41.9478306, 45.3056477, 49.068432, 53.1030698, 57.2764468, 61.4554493, 65.5069632, 69.2978744, 72.6950692, 75.5654334, 77.7758531, 79.1932144, 79.2641626, 78.4299126, 76.6834226, 74.0176508, 71.5695765, 68.7601474, 64.7052387, 59.0546024, 51.4579902, 41.565154, 29.0258456, 14.3740531, -3.89382329, -26.3010344, -53.3708311, -85.6264643, -123.642307, -167.877708, -218.843137, -277.049064, -343.005959, -414.830301, -498.352544, -594.991742, -706.777983, -836.988876, -988.34192, -1163.55461, -1369.42022, -1603.65417, -1859.52555, -2125.49719, -2420.85825, -2762.87593, -3160.32316, -3607.57985, -4105.63253, -4653.15358, -5250.52957, -5906.63236, -6625.53404, -7412.80626, -8270.79085, -9203.04086, -10213.1093, -11304.5492, -12765.7392, -14244.2002, -15672.279, -16982.3222, -18106.6764, -18686.6868, -19139.7025, -19522.7849, -20055.0498, -20729.4123, -21471.2649, -22276.8988, -23142.6052, -24064.6754, -25039.4005, -25753.1118, -26777.9676, -27999.332, -29302.5691, -30573.0432, -31696.1185, -32557.159, -35107.1494, -38081.8825}, 
{-283.726184, -284.535746, -285.234737, -285.824529, -286.306493, -286.682, -286.952422, -287.119131, -287.183497, -287.146892, -287.010688, -286.776255, -286.444966, -286.018191, -285.497303, -284.883671, -284.178669, -283.383666, -282.500036, -281.529148, -280.472374, -279.331086, -278.106656, -276.800454, -275.413851, -273.94822, -272.404932, -270.785357, -269.090868, -267.322836, -265.482632, -263.571628, -261.591195, -259.542704, -257.433659, -255.262237, -253.02654, -250.72774, -248.367009, -245.945521, -243.464446, -240.924959, -238.32823, -235.675433, -232.96774, -230.206323, -227.392355, -224.527007, -221.611453, -218.646864, -215.634414, -212.555276, -209.431054, -206.263355, -203.053784, -199.803946, -196.515449, -193.189897, -189.828896, -186.434054, -183.006974, -179.549264, -176.062528, -172.548374, -169.008407, -165.444232, -161.857456, -158.376051, -155.124956, -151.845155, -148.530061, -145.17309, -141.767654, -138.307169, -134.785047, -131.194703, -127.529551, -123.783006, -119.94848, -116.019389, -111.989145, -107.851164, -103.598859, -99.2256443, -93.8697714, -87.9702227, -81.9762592, -75.9090465, -69.7897503, -63.6395363, -57.4795703, -51.3310178, -45.2150448, -39.1528167, -33.1654994, -27.2742585, -21.5002597, -15.8646688, -10.3886514, -5.09337323, 0, -0.890426295, -1.2222853, -1.03052733, -0.350102682, 0.784038319, 2.33694537, 4.27366816, 6.55925637, 9.1587597, 12.0372278, 15.1597105, 18.4912573, 21.996918, 25.6417422, 29.3907797, 33.2090802, 36.6386013, 39.5839511, 42.967173, 46.662654, 50.5447813, 54.4879421, 58.3665234, 62.0549124, 65.4274963, 68.3586621, 70.7227969, 72.394288, 73.2475225, 72.800239, 71.4142441, 69.0709196, 65.7516471, 62.4348846, 58.6034756, 53.4743766, 46.7298461, 38.052143, 27.1235259, 13.6262537, -1.36930305, -19.9914664, -42.8295647, -70.4729264, -103.51088, -142.487097, -188.047976, -240.794261, -301.326695, -370.246019, -446.603306, -534.443012, -634.083921, -747.013772, -877.106915, -1027.16616, -1199.99432, -1401.69556, -1631.02102, -1883.12038, -2149.26461, -2445.6487, -2787.29384, -3181.85044, -3624.47942, -4116.0746, -4655.45392, -5242.97298, -5886.59889, -6589.99326, -7357.41322, -8191.83323, -9096.70876, -10075.4953, -11131.6482, -12546.0864, -13975.4361, -15353.787, -16615.2288, -17693.851, -18237.9184, -18657.8955, -19010.3682, -19514.5814, -20162.7352, -20879.2555, -21659.7312, -22499.7514, -23394.9048, -24340.7807, -25387.0715, -26452.6268, -27528.6196, -28606.2227, -29676.6091, -30730.9517, -31760.4235, -34391.5133, -37261.8464}, 
{-278.9773, -279.735142, -280.383449, -280.92367, -281.357254, -281.685649, -281.910306, -282.032673, -282.0542, -281.976334, -281.800527, -281.528226, -281.160881, -280.699941, -280.146855, -279.503072, -278.770042, -277.949212, -277.042033, -276.049954, -274.974424, -273.816891, -272.578805, -271.261615, -269.86677, -268.39572, -266.849913, -265.230798, -263.539825, -261.778443, -259.9481, -258.050246, -256.08633, -254.057802, -251.976116, -249.83751, -247.638096, -245.379, -243.061346, -240.686258, -238.254861, -235.768278, -233.227635, -230.634056, -227.988664, -225.292586, -222.546943, -219.752862, -216.911466, -214.023881, -211.091229, -208.098438, -205.063181, -201.986933, -198.87117, -195.717368, -192.527002, -189.301547, -186.04248, -182.751274, -179.429407, -176.078353, -172.699589, -169.294588, -165.864828, -162.411783, -158.936929, -155.569525, -152.435764, -149.273574, -146.07615, -142.836681, -139.548359, -136.204378, -132.797928, -129.322201, -125.770389, -122.135683, -118.411276, -114.59036, -110.666125, -106.631764, -102.480469, -98.2054312, -92.9237942, -87.0852003, -81.1498216, -75.1392793, -69.0751945, -62.9791887, -56.8728828, -50.7778983, -44.7158563, -38.7083781, -32.7770849, -26.943598, -21.2295385, -15.6565277, -10.2461869, -5.02013724, 0, -2.00983973, -3.26136403, -3.80464093, -3.68973849, -2.96672474, -1.68566774, 0.103364486, 2.35030388, 5.0050824, 8.017632, 11.3378846, 14.9157723, 18.7012268, 22.6441803, 26.6945647, 30.8023118, 34.9045047, 38.9492396, 42.9155119, 46.7505002, 50.4013831, 53.8153392, 56.9395469, 59.721185, 62.107432, 64.0454665, 65.4824671, 66.3656123, 66.6420807, 66.5306497, 65.6073128, 63.7377691, 60.7877176, 56.8424524, 51.6578748, 44.9313276, 36.4626312, 26.0516059, 13.4980723, -1.39814926, -16.1127191, -34.3995375, -56.9918423, -84.622871, -118.025862, -158.892275, -206.757569, -262.115427, -325.459529, -397.28356, -476.491536, -567.109725, -670.004248, -786.652193, -919.778034, -1071.54619, -1244.12109, -1440.1846, -1661.26608, -1908.33047, -2180.12642, -2484.36378, -2828.59259, -3217.42675, -3650.50503, -4129.74987, -4655.99659, -5230.88578, -5860.04416, -6546.84369, -7294.46516, -8106.50107, -8986.38954, -9937.56866, -10963.4766, -12341.1276, -13731.9896, -15071.1066, -16293.5225, -17334.2814, -17844.0833, -18231.879, -18554.5742, -19037.1929, -19670.094, -20373.5873, -21141.5341, -21967.7959, -22846.2341, -23770.7101, -25248.8797, -26370.087, -27262.0192, -28052.3635, -28868.807, -29839.0369, -31090.7404, -33686.3982, -36439.3707}, 
{-274.797779, -275.516946, -276.126773, -276.628801, -277.024576, -277.31564, -277.503535, -277.589807, -277.575997, -277.46365, -277.254308, -276.949515, -276.550814, -276.059748, -275.477861, -274.806696, -274.047796, -273.202705, -272.272965, -271.260121, -270.165715, -268.99129, -267.738391, -266.408559, -265.00334, -263.524275, -261.972908, -260.350782, -258.659442, -256.900429, -255.075287, -253.185561, -251.232791, -249.218524, -247.158499, -245.046863, -242.87759, -240.651762, -238.370463, -236.034774, -233.645778, -231.204558, -228.712196, -226.169774, -223.578376, -220.939083, -218.252978, -215.521143, -212.744662, -209.924617, -207.062089, -204.145106, -201.188089, -198.192404, -195.159417, -192.090491, -188.986995, -185.850292, -182.681748, -179.482729, -176.254601, -172.998729, -169.716478, -166.409214, -163.078303, -159.72511, -156.351001, -153.087075, -150.061319, -147.007529, -143.918661, -140.787671, -137.607513, -134.371144, -131.071518, -127.701591, -124.254319, -120.722655, -117.099557, -113.37798, -109.550878, -105.611207, -101.551922, -97.3659794, -92.1479244, -86.3588141, -80.4705036, -74.5051034, -68.4847238, -62.4314755, -56.3674688, -50.3148143, -44.2956224, -38.3320037, -32.4460685, -26.6599275, -20.9956909, -15.4754695, -10.1213735, -4.95551351, 0, -3.05450294, -5.16555382, -6.39722348, -6.81358276, -6.4787025, -5.45665356, -3.81150677, -1.60733297, 1.09179699, 4.22181226, 7.71864201, 11.5182154, 15.5564616, 19.7693097, 24.0926889, 28.4625284, 33.2159121, 38.346077, 42.8815783, 46.8443069, 50.2561537, 53.1390096, 55.5147655, 57.4053123, 58.8325409, 59.8183422, 60.3846071, 60.5532265, 60.3460914, 60.718811, 60.4171944, 59.183017, 56.7580541, 52.302884, 45.8498805, 37.5864036, 27.4285879, 15.292568, 1.09447844, -15.2495463, -29.8073215, -47.9010389, -70.4003127, -98.1747572, -132.093987, -175.191836, -225.632644, -283.744971, -349.857374, -424.298415, -505.449693, -597.966341, -703.782118, -824.55769, -961.396168, -1115.65099, -1288.67561, -1479.30867, -1691.98994, -1933.90263, -2211.52875, -2523.85411, -2870.73219, -3253.67192, -3676.98732, -4143.70482, -4656.77897, -5219.21758, -5834.29203, -6505.12461, -7233.73882, -8024.52478, -8880.98513, -9806.6225, -10804.9395, -12151.451, -13509.6444, -14815.0193, -16003.0753, -17009.312, -17484.2943, -17838.413, -18129.2828, -18589.2269, -19206.7042, -19897.3786, -20653.349, -21466.714, -22329.5725, -23234.0233, -25084.258, -26245.6505, -26968.8853, -27504.6471, -28103.6203, -29016.4896, -30493.9395, -33008.0647, -35601.8727}, 
{-271.120774, -271.80938, -272.388562, -272.859954, -273.225189, -273.485901, -273.643725, -273.700292, -273.657239, -273.516197, -273.278801, -272.946685, -272.521482, -272.004826, -271.398351, -270.70369, -269.922477, -269.056347, -268.106932, -267.075867, -265.964784, -264.775319, -263.509104, -262.167774, -260.752962, -259.266302, -257.709427, -256.083972, -254.391569, -252.633854, -250.812458, -248.929018, -246.985165, -244.982534, -242.940936, -240.852534, -238.709272, -236.512194, -234.262342, -231.960763, -229.608498, -227.206592, -224.756088, -222.258032, -219.713465, -217.123433, -214.488978, -211.811146, -209.090979, -206.329522, -203.527817, -200.67558, -197.785429, -194.858655, -191.896545, -188.900391, -185.871481, -182.811104, -179.720551, -176.601111, -173.454072, -170.280725, -167.082359, -163.860264, -160.615728, -157.350042, -154.064494, -150.892805, -147.965515, -145.010785, -142.02132, -138.989825, -135.909003, -132.771561, -129.570201, -126.297629, -122.946549, -119.509666, -115.979684, -112.349307, -108.611241, -104.758189, -100.782857, -96.6779477, -91.5136844, -85.7634158, -79.9115783, -73.9808127, -67.9937601, -61.9730611, -55.9413567, -49.9212877, -43.9354949, -38.0066193, -32.1573015, -26.4101826, -20.7879034, -15.3131047, -10.0084273, -4.89651211, 0, -3.78958061, -6.51318011, -8.24420583, -9.05606505, -9.02216508, -8.21591323, -6.71071681, -4.57998311, -1.89711944, 1.26446689, 4.83136858, 8.73017832, 12.8874888, 17.2298928, 21.6839828, 26.1763518, 31.340165, 37.2029543, 42.0931118, 46.0886389, 49.2675365, 51.7078059, 53.4874482, 54.6844646, 55.3768562, 55.6426243, 55.5597698, 55.206294, 54.6601981, 55.4614506, 55.6900305, 54.9853489, 52.9868165, 48.12804, 40.6513324, 31.1205549, 19.5368597, 5.90139905, -9.78467469, -27.5202093, -42.3673665, -60.7641496, -83.6752797, -112.065478, -146.899464, -192.44998, -245.546723, -306.327407, -374.929751, -451.491469, -533.612767, -627.070276, -735.179908, -859.936225, -1000.63602, -1157.78734, -1331.89821, -1518.27464, -1723.80897, -1961.06855, -2242.92447, -2561.22807, -2909.62506, -3287.09976, -3701.68252, -4157.2514, -4658.42319, -5209.26747, -5811.14524, -6466.94962, -7177.34031, -7947.78747, -8781.95734, -9683.51615, -10656.1301, -11976.0746, -13306.2545, -14582.1837, -15739.3762, -16713.346, -17152.5314, -17470.9059, -17727.0157, -18160.2665, -18757.6144, -19430.658, -20170.1216, -20966.7294, -21811.2057, -22694.2748, -24780.0455, -25950.2208, -26544.4693, -26902.4599, -27363.8613, -28268.3423, -29955.5716, -32364.6679, -34742.8035}, 
{-267.841247, -268.502437, -269.054591, -269.4994, -269.838558, -270.073756, -270.206687, -270.239043, -270.172516, -270.008799, -269.749584, -269.396563, -268.951428, -268.415873, -267.791588, -267.080267, -266.283602, -265.403285, -264.441008, -263.398463, -262.277343, -261.079341, -259.806148, -258.459457, -257.04096, -255.552349, -253.995317, -252.371555, -250.682757, -248.930615, -247.11682, -245.243065, -243.311043, -241.322446, -239.300414, -237.235465, -235.117858, -232.948589, -230.728653, -228.459047, -226.140768, -223.774812, -221.362174, -218.903851, -216.40084, -213.854136, -211.264736, -208.633636, -205.961832, -203.250321, -200.500098, -197.698722, -194.860918, -191.987974, -189.081178, -186.141817, -183.171179, -180.17055, -177.141219, -174.084474, -171.001601, -167.893889, -164.762625, -161.609095, -158.434589, -155.240394, -152.027796, -148.934841, -146.096288, -143.231375, -140.332525, -137.392159, -134.4027, -131.35657, -128.24619, -125.063984, -121.802374, -118.453781, -115.010628, -111.465336, -107.810329, -104.038028, -100.140856, -96.1112342, -90.9915862, -85.2693155, -79.4432442, -73.5366237, -67.5727054, -61.5747409, -55.5659816, -49.569679, -43.6090846, -37.7074498, -31.8880262, -26.1740651, -20.5888181, -15.1555366, -9.89747207, -4.83787606, 0, -3.73213075, -6.43779499, -8.18768217, -9.05248171, -9.10288306, -8.40957565, -7.04324892, -5.07459229, -2.57429521, 0.386952893, 3.73846258, 7.40954441, 11.329509, 15.4276668, 19.6333285, 23.8758045, 28.7698241, 34.3426145, 38.9731351, 42.7375719, 45.7121108, 47.9729378, 49.5962388, 50.6581997, 51.2350065, 51.4028452, 51.2379017, 50.8163619, 50.2144118, 51.0532578, 51.2975578, 50.5599918, 48.4532395, 43.310705, 35.3847059, 25.2687001, 12.9591503, -1.54748109, -18.2547314, -37.1661382, -53.4184692, -73.3632223, -97.956129, -128.152921, -164.909329, -213.331699, -269.187495, -332.394796, -402.87168, -480.536226, -561.810439, -654.381449, -763.338954, -891.247406, -1035.51554, -1195.8669, -1372.02503, -1557.02381, -1758.79684, -1992.57554, -2273.52443, -2591.21526, -2937.74633, -3311.34314, -3720.61338, -4169.39914, -4662.44453, -5203.82549, -5794.31059, -6436.53921, -7129.10473, -7879.31505, -8691.21015, -9568.82999, -10516.2145, -11814.0108, -13120.5, -14370.5702, -15499.1095, -16441.0062, -16839.8545, -17116.0325, -17329.2686, -17724.1089, -18287.125, -18927.8808, -19636.7981, -20404.2988, -21220.8046, -22076.7374, -24129.6387, -25248.8783, -25795.2504, -26129.549, -26612.5683, -27605.1024, -29467.9454, -31793.8094, -33854.9813}, 
{-264.87853, -265.510504, -266.03493, -266.453505, -266.767924, -266.979883, -267.091078, -267.103205, -267.01796, -266.837038, -266.562136, -266.194949, -265.737174, -265.190506, -264.556641, -263.837275, -263.034104, -262.148823, -261.18313, -260.138719, -259.017286, -257.820528, -256.55014, -255.207818, -253.795258, -252.314157, -250.766209, -249.15311, -247.476558, -245.738246, -243.939873, -242.083132, -240.169721, -238.201335, -236.203164, -234.164664, -232.075006, -229.935123, -227.74595, -225.508418, -223.223462, -220.892016, -218.515011, -216.093383, -213.628065, -211.119989, -208.57009, -205.9793, -203.348553, -200.678784, -197.970924, -195.205168, -192.40364, -189.56772, -186.698793, -183.798242, -180.86745, -177.907798, -174.920672, -171.907453, -168.869524, -165.80827, -162.725072, -159.621313, -156.498377, -153.357647, -150.200505, -147.171504, -144.411756, -141.627372, -138.810452, -135.953099, -133.047416, -130.085503, -127.059464, -123.961399, -120.783412, -117.517603, -114.156076, -110.690931, -107.114271, -103.418199, -99.5948152, -95.6362225, -90.5527863, -84.8481261, -79.0376617, -73.1453542, -67.1951649, -61.2110549, -55.2169856, -49.2369182, -43.2948139, -37.414634, -31.6203396, -25.9358921, -20.3852526, -14.9923824, -9.78124276, -4.77579488, 0, -2.5575157, -4.35673792, -5.44864386, -5.88421067, -5.71441555, -4.99023566, -3.76264818, -2.08263029, -0.00115917075, 2.43078801, 5.16223407, 8.14220183, 11.3197141, 14.6437938, 18.0634636, 21.5277464, 25.1727736, 28.9742977, 32.45812, 35.6133579, 38.4291289, 40.8945505, 42.9987403, 44.7308157, 46.0798943, 47.0350936, 47.585531, 47.7203242, 47.4285906, 47.6626124, 47.0951821, 45.4264677, 42.3566372, 37.0472498, 29.467778, 19.7313235, 7.699637, -6.7655306, -23.8024286, -43.5493061, -62.7704119, -86.0048654, -114.051046, -147.707335, -187.77211, -239.480598, -298.085121, -363.274848, -434.738946, -512.166585, -590.429141, -679.921986, -787.796846, -917.42705, -1064.4717, -1228.05344, -1407.29489, -1595.0289, -1798.0974, -2030.20386, -2302.69389, -2610.06304, -2949.76429, -3321.8731, -3730.90869, -4179.35053, -4669.7884, -5204.73037, -5786.27998, -6416.76959, -7091.76399, -7821.40427, -8610.3654, -9463.32237, -10384.9501, -11664.2758, -12950.5341, -14177.3122, -15278.1969, -16186.7754, -16539.1867, -16764.7641, -16924.5721, -17264.6237, -17772.7747, -18359.8191, -19017.4659, -19737.4239, -20511.4022, -21331.1097, -22986.055, -23974.2239, -24584.3489, -25105.1623, -25825.3966, -27033.7842, -29019.0575, -31314.3024, -32931.628}, 
{-262.386706, -262.982958, -263.473745, -263.860735, -264.145599, -264.330007, -264.415631, -264.40414, -264.297205, -264.096496, -263.803684, -263.420439, -262.948431, -262.389331, -261.744809, -261.016536, -260.206181, -259.315416, -258.345911, -257.299336, -256.177362, -254.981658, -253.713896, -252.375746, -250.968878, -249.494962, -247.95567, -246.35267, -244.687635, -242.962234, -241.178137, -239.337016, -237.440539, -235.490379, -233.511791, -231.494156, -229.426572, -227.309945, -225.145179, -222.933179, -220.674851, -218.371098, -216.022827, -213.630943, -211.196349, -208.719952, -206.202655, -203.645365, -201.048987, -198.414424, -195.742582, -193.009976, -190.242429, -187.441375, -184.608249, -181.744484, -178.851514, -175.930773, -172.983694, -170.011711, -167.016259, -163.998769, -160.960677, -157.903417, -154.828421, -151.737124, -148.630959, -145.659341, -142.968133, -140.253568, -137.507486, -134.721729, -131.888136, -128.99855, -126.044811, -123.018758, -119.912234, -116.717079, -113.425133, -110.028238, -106.518234, -102.886961, -99.1262609, -95.2279741, -90.1746519, -84.4840088, -78.6858901, -72.8048899, -66.8656024, -60.8926217, -54.910542, -48.9439572, -43.0174616, -37.1556492, -31.3831143, -25.7244508, -20.2042529, -14.8471148, -9.67763051, -4.72039421, 0, -1.46627225, -2.42148276, -2.89854648, -2.9303784, -2.54989346, -1.79000664, -0.683632909, 0.736312779, 2.43691545, 4.38526015, 6.54843191, 8.89351575, 11.3875967, 13.9977599, 16.6910902, 19.4346727, 21.9060453, 24.0309289, 26.4313371, 29.012309, 31.6788838, 34.3361008, 36.888999, 39.2426177, 41.3019961, 42.9721733, 44.1581885, 44.765081, 44.6978898, 44.240975, 42.7809707, 40.1091198, 36.0166653, 30.4954866, 23.2365084, 13.8771523, 2.14847044, -12.2184853, -29.4926629, -49.9430103, -72.0073257, -98.3430133, -129.57729, -166.337372, -209.250475, -262.699804, -322.617592, -388.692056, -460.611418, -538.063895, -615.584875, -704.313319, -811.908876, -941.999012, -1089.97881, -1254.93954, -1435.97244, -1627.24775, -1833.89616, -2066.4168, -2331.28589, -2628.64191, -2961.4772, -3331.87066, -3740.3535, -4188.16734, -4675.90982, -5204.65558, -5777.84063, -6397.56524, -7057.4216, -7769.32705, -8538.32696, -9369.46674, -10267.7918, -11525.9202, -12789.4316, -13991.478, -15065.2117, -15943.7848, -16261.4043, -16449.4645, -16569.2373, -16862.7197, -17319.6462, -17854.4491, -18460.6281, -19131.6827, -19861.1127, -20642.4177, -21776.7266, -22609.378, -23320.5886, -24090.5755, -25099.5553, -26527.7451, -28555.3616, -30763.9469, -31969.8577}, 
{-260.487583, -261.02029, -261.451403, -261.7825, -262.015159, -262.150959, -262.191476, -262.13829, -261.992978, -261.757119, -261.43229, -261.020071, -260.522038, -259.93977, -259.274845, -258.528841, -257.703336, -256.799909, -255.820137, -254.765599, -253.637873, -252.438536, -251.169167, -249.831344, -248.426645, -246.956648, -245.422932, -243.827074, -242.170652, -240.455245, -238.682431, -236.853788, -234.970893, -233.035325, -231.066945, -229.057804, -226.999736, -224.893725, -222.740756, -220.541814, -218.297884, -216.009951, -213.678998, -211.306012, -208.891977, -206.437878, -203.944699, -201.413426, -198.845043, -196.240535, -193.600887, -190.913528, -188.193293, -185.441459, -182.659307, -179.848114, -177.009158, -174.14372, -171.253076, -168.338507, -165.40129, -162.442704, -159.464029, -156.466542, -153.451522, -150.420248, -147.373999, -144.46073, -141.824877, -139.165205, -136.473487, -133.741491, -130.960988, -128.123749, -125.221543, -122.246141, -119.189313, -116.042828, -112.798458, -109.447971, -105.98314, -102.395733, -98.6775202, -94.8202728, -89.7932611, -84.1210119, -78.3403416, -72.4762025, -66.5535468, -60.5973267, -54.6324945, -48.6840024, -42.7768028, -36.9358477, -31.1860895, -25.5524804, -20.0599727, -14.7335185, -9.59807018, -4.67857993, 0, -1.2767387, -2.0956972, -2.4863853, -2.47831283, -2.1009896, -1.38392544, -0.356630142, 0.95138646, 2.51061456, 4.29154433, 6.26466596, 8.40046964, 10.6694455, 13.0420839, 15.4888748, 17.9803085, 20.0654529, 21.654595, 23.671442, 25.9961794, 28.5089924, 31.0900666, 33.6195873, 35.9777401, 38.0447102, 39.7006831, 40.8258443, 41.3003791, 41.0044729, 40.0153648, 37.9439335, 34.6112487, 29.8383795, 23.8600194, 16.2904258, 6.63355626, -5.41360704, -20.154082, -37.8908863, -58.927038, -82.4731351, -110.257091, -142.795657, -180.605586, -224.20363, -274.733633, -331.928481, -396.148156, -467.752635, -547.101897, -632.240703, -728.673965, -839.730274, -967.471437, -1111.37291, -1272.07137, -1450.20348, -1644.52955, -1857.98907, -2095.56852, -2359.60005, -2653.52045, -2982.21565, -3348.60729, -3752.28855, -4194.38046, -4675.27011, -5195.88825, -5759.85654, -6369.2745, -7019.45871, -7720.33515, -8476.35129, -9291.95459, -10171.5925, -11393.9366, -12620.6543, -13787.6369, -14830.7758, -15685.9625, -16004.171, -16196.155, -16319.9137, -16599.7072, -17027.7227, -17526.8717, -18092.8051, -18721.1739, -19407.6288, -20147.8209, -20642.7833, -21294.6081, -22134.4315, -23193.3896, -24502.6187, -26093.255, -27996.4346, -29947.806, -30945.564}, 
{-259.058751, -259.520319, -259.884429, -260.152556, -260.326173, -260.406753, -260.395771, -260.2947, -260.105014, -259.828187, -259.465691, -259.019002, -258.489592, -257.878936, -257.188506, -256.419777, -255.574223, -254.653317, -253.658532, -252.591343, -251.453223, -250.245646, -248.970085, -247.628015, -246.220908, -244.750239, -243.217482, -241.624109, -239.971595, -238.261413, -236.495038, -234.673942, -232.7996, -230.873485, -228.90834, -226.899768, -224.843235, -222.739849, -220.590717, -218.396947, -216.159648, -213.879928, -211.558894, -209.197654, -206.797317, -204.358989, -201.88378, -199.372798, -196.827149, -194.247943, -191.636286, -188.996711, -186.326828, -183.62767, -180.900271, -178.145665, -175.364887, -172.558968, -169.728944, -166.875848, -164.000714, -161.104575, -158.188466, -155.253419, -152.300469, -149.33065, -146.344995, -143.487119, -140.897214, -138.28225, -135.634019, -132.944311, -130.204916, -127.407627, -124.544233, -121.606526, -118.586295, -115.475333, -112.26543, -108.948376, -105.515962, -101.95998, -98.272219, -94.4444714, -89.4401644, -83.7869006, -78.0246426, -72.1785535, -66.2737964, -60.3355346, -54.3889311, -48.4591492, -42.5713519, -36.7507024, -31.022364, -25.4114996, -19.9432725, -14.6428459, -9.53538279, -4.64604646, 0, -1.65545421, -2.78493465, -3.42371181, -3.60705619, -3.37023828, -2.74852857, -1.77719755, -0.491515711, 1.07324645, 2.88181845, 4.89892978, 7.08930996, 9.41768851, 11.8487949, 14.3473587, 16.8781093, 19.0626148, 20.8165813, 22.8809375, 25.1468782, 27.5055983, 29.8482925, 32.0661559, 34.0503833, 35.6921695, 36.8827094, 37.5131979, 37.4748299, 36.6588002, 35.2059057, 32.6662186, 28.856053, 23.5917234, 17.0280483, 8.81208997, -1.49935759, -14.1915314, -29.5496686, -47.8590063, -69.4047814, -93.6564348, -121.963286, -154.770184, -192.521978, -235.663518, -281.122545, -333.740293, -394.84089, -465.748462, -547.787135, -644.188852, -752.038149, -868.256975, -992.549767, -1130.30188, -1284.34807, -1457.52305, -1654.34271, -1875.57859, -2120.16828, -2387.26769, -2681.83092, -3008.30918, -3369.68966, -3766.47895, -4200.32225, -4672.4503, -5184.40086, -5739.23151, -6339.14013, -6982.77635, -7676.43219, -8423.53379, -9227.50726, -10091.7787, -11273.5002, -12458.9405, -13588.0942, -14600.956, -15437.5206, -15773.5588, -15989.4382, -16138.3923, -16421.2996, -16832.9505, -17306.6169, -17840.1653, -18431.4621, -19078.3739, -19778.7672, -19650.4321, -20132.2661, -21101.2192, -22434.2414, -24008.2829, -25700.2935, -27387.2235, -29004.5191, -29882.7371}, 
{-257.975999, -258.379547, -258.688463, -258.904161, -259.028055, -259.061558, -259.006084, -258.863048, -258.633863, -258.319942, -257.922699, -257.443549, -256.883905, -256.24518, -255.528789, -254.736145, -253.868662, -252.927754, -251.914835, -250.831318, -249.678617, -248.458146, -247.171318, -245.819548, -244.40425, -242.926836, -241.388721, -239.791319, -238.136043, -236.424307, -234.657525, -232.837111, -230.964478, -229.04104, -227.074546, -225.063108, -223.004764, -220.900723, -218.752191, -216.560379, -214.326493, -212.051742, -209.737334, -207.384477, -204.99438, -202.568251, -200.107298, -197.61273, -195.085753, -192.527577, -189.939411, -187.340552, -184.713727, -182.05975, -179.379439, -176.673609, -173.943076, -171.188657, -168.411166, -165.611421, -162.790238, -159.948431, -157.086819, -154.206215, -151.307437, -148.391301, -145.458622, -142.649265, -140.099755, -137.524133, -134.9142, -132.261758, -129.558609, -126.796556, -123.9674, -121.062943, -118.074989, -114.995337, -111.815792, -108.528155, -105.124228, -101.595813, -97.9347116, -94.1327268, -89.1477341, -83.5102458, -77.7632017, -71.9319565, -66.0418653, -60.1182832, -54.186565, -48.2720658, -42.4001406, -36.5961443, -30.8854319, -25.2933585, -19.845279, -14.5665485, -9.48252176, -4.61855394, 0, -2.2752771, -3.90611652, -4.93731821, -5.41368211, -5.38000817, -4.88109632, -3.96174651, -2.66675868, -1.04093277, 0.870931267, 3.02403349, 5.37357397, 7.87475274, 10.4827699, 13.1528254, 15.8401195, 18.3123453, 20.4979163, 22.7761549, 25.0620811, 27.2707151, 29.3170768, 31.1161863, 32.5830637, 33.6327288, 34.1802019, 34.140503, 33.428652, 31.959669, 30.0205312, 27.0179172, 22.75526, 17.0359925, 9.87576239, 0.971895175, -10.0348742, -23.4047771, -39.3980452, -58.2749098, -80.2956026, -105.05754, -133.685495, -166.56938, -204.099107, -246.66459, -287.430043, -335.817501, -394.023302, -464.243783, -548.675281, -655.225081, -773.398526, -894.429824, -1015.91303, -1148.4269, -1296.72032, -1465.54217, -1664.20099, -1891.84974, -2142.66725, -2413.89452, -2710.73572, -3036.15826, -3392.80559, -3782.77208, -4208.40414, -4672.11149, -5176.25693, -5722.97119, -6314.51628, -6952.36876, -7639.69693, -8379.03468, -9172.91592, -10023.8745, -11170.0757, -12319.5142, -13415.8162, -14402.6077, -15223.5148, -15576.3117, -15814.3777, -15986.7039, -16273.3606, -16671.4535, -17123.4722, -17629.2534, -18188.6338, -18801.4499, -19467.5385, -18871.575, -19231.7653, -20302.3185, -21837.4438, -23591.3501, -25318.2467, -26772.3426, -28073.9986, -28806.6574}, 
{-256.970622, -257.382003, -257.695879, -257.91381, -258.037359, -258.068087, -258.007554, -257.857321, -257.618951, -257.294003, -256.884039, -256.390621, -255.815309, -255.159665, -254.42525, -253.613624, -252.726349, -251.764987, -250.731097, -249.626243, -248.451984, -247.209881, -245.901497, -244.528391, -243.092126, -241.594262, -240.036361, -238.419983, -236.746691, -235.018044, -233.235604, -231.400933, -229.51559, -227.581139, -225.612518, -223.604318, -221.550968, -219.453595, -217.313327, -215.13129, -212.908612, -210.646419, -208.345838, -206.007996, -203.634021, -201.225039, -198.782177, -196.306562, -193.799322, -191.261582, -188.694471, -186.111288, -183.500724, -180.86364, -178.200901, -175.51337, -172.801909, -170.067382, -167.310652, -164.532581, -161.734034, -158.915872, -156.07896, -153.22416, -150.352335, -147.464349, -144.561065, -141.786318, -139.280514, -136.749643, -134.1853, -131.579079, -128.922574, -126.207382, -123.425096, -120.567311, -117.625622, -114.591623, -111.456909, -108.213076, -104.851716, -101.364426, -97.7427999, -93.9784323, -89.0097185, -83.3770054, -77.633299, -71.8043721, -65.9159979, -59.9939494, -54.0639996, -48.1519216, -42.2834884, -36.484473, -30.7806484, -25.1977879, -19.7616642, -14.4980506, -9.43271996, -4.59144542, 0, -2.7481119, -4.77044878, -6.11842728, -6.84346403, -6.99697568, -6.63037885, -5.79509019, -4.54252633, -2.9241039, -0.991239541, 1.20465011, 3.61214842, 6.17983875, 8.85630446, 11.5901289, 14.3298955, 16.8873402, 19.1914969, 21.5004844, 23.7335617, 25.8099878, 27.6490215, 29.1699219, 30.2919479, 30.9343584, 31.0164123, 30.4573686, 29.1764862, 27.093024, 24.4375119, 20.7038052, 15.7177817, 9.30531914, 1.72876494, -7.4042381, -18.6024391, -32.1709014, -48.4146883, -67.6388629, -90.1484885, -116.143106, -146.065416, -180.241129, -218.995953, -262.655596, -304.082323, -352.931149, -411.393645, -481.661381, -565.925927, -671.787881, -789.418753, -910.888967, -1034.06017, -1168.61776, -1318.93852, -1489.39921, -1688.21182, -1915.04627, -2165.38865, -2437.8238, -2735.99052, -3061.46088, -3416.30729, -3803.44998, -4225.42, -4685.12636, -5185.1981, -5726.8785, -6312.19471, -6940.38596, -7616.69624, -8344.11774, -9125.64265, -9964.26318, -11104.1205, -12246.7706, -13334.9184, -14311.2689, -15118.527, -15449.6788, -15663.6274, -15810.1003, -16069.7329, -16440.4349, -16865.5708, -17345.7775, -17881.6919, -18473.9508, -19123.1911, -18595.3122, -18987.3805, -20055.7095, -21556.6126, -23246.4035, -24881.3955, -26217.9023, -27443.6947, -27810.2081}, 
{-256.262747, -256.69641, -257.028446, -257.260603, -257.39463, -257.432274, -257.375285, -257.225411, -256.984399, -256.653999, -256.235958, -255.732025, -255.143949, -254.473477, -253.722358, -252.892341, -251.985173, -251.002604, -249.946381, -248.818253, -247.619967, -246.353274, -245.01992, -243.621655, -242.160226, -240.637382, -239.054871, -237.414441, -235.717842, -233.966821, -232.163127, -230.308507, -228.404711, -226.453487, -224.47926, -222.471962, -220.421255, -218.32815, -216.19366, -214.018798, -211.804576, -209.552007, -207.262102, -204.935875, -202.574338, -200.178503, -197.749382, -195.287989, -192.795336, -190.272434, -187.720297, -185.141425, -182.535311, -179.902933, -177.245274, -174.563312, -171.858028, -169.130402, -166.381414, -163.612044, -160.823274, -158.016082, -155.191449, -152.350356, -149.493782, -146.622707, -143.738113, -140.989638, -138.523355, -136.033643, -133.511844, -130.949304, -128.337363, -125.667366, -122.930656, -120.118576, -117.222469, -114.233678, -111.143546, -107.943417, -104.624633, -101.178539, -97.5964757, -93.869788, -88.9149765, -83.2834133, -77.5393892, -71.7091279, -65.8188533, -59.8947892, -53.9631596, -48.0501882, -42.1820989, -36.3851155, -30.6854618, -25.1093618, -19.6830392, -14.4327179, -9.38462166, -4.56497443, 0, -3.241389, -5.67276198, -7.3523416, -8.3383505, -8.68901133, -8.46254675, -7.71717939, -6.51113191, -4.90262695, -2.94988718, -0.711135225, 1.75540625, 4.39151461, 7.13896719, 9.93954136, 12.7350144, 15.3860765, 17.822921, 20.170737, 22.353926, 24.2968897, 25.9240293, 27.1597466, 27.9284428, 28.1545196, 27.7623785, 26.6764208, 24.8210482, 22.1206621, 18.7520353, 14.2946619, 8.59723193, 1.50843555, -6.42888026, -15.7131039, -27.0277321, -40.7323218, -57.1864301, -76.7496139, -99.7814302, -127.092663, -158.454312, -194.137651, -234.413953, -279.554492, -323.196933, -373.904559, -433.607046, -504.234069, -587.715303, -689.922102, -804.024856, -924.754455, -1050.9049, -1189.56571, -1344.10189, -1517.87845, -1716.35201, -1940.32057, -2188.30034, -2461.18163, -2760.46101, -3086.22648, -3439.89222, -3825.11955, -4244.53831, -4701.47936, -5198.75431, -5736.60434, -6316.72464, -6934.96057, -7599.75705, -8314.83418, -9083.91204, -9910.71073, -11050.4185, -12192.4203, -13277.569, -14246.7179, -15040.7197, -15342.0792, -15522.2302, -15632.7466, -15859.727, -16201.4959, -16600.3257, -17057.3437, -17573.6768, -18150.4522, -18788.7969, -18523.1626, -19002.6579, -20030.4417, -21409.6724, -22943.5089, -24435.1097, -25687.6335, -26873.8728, -26779.6023}, 
{-255.903863, -256.358779, -256.708099, -256.953743, -257.097635, -257.141695, -257.087846, -256.938009, -256.694107, -256.35806, -255.931792, -255.417223, -254.816276, -254.130872, -253.362934, -252.514383, -251.587141, -250.58313, -249.504271, -248.352488, -247.1297, -245.837831, -244.478802, -243.054535, -241.566951, -240.017974, -238.409524, -236.743523, -235.021893, -233.246556, -231.419435, -229.54245, -227.617523, -225.646577, -223.662807, -221.65184, -219.598928, -217.504977, -215.370895, -213.197588, -210.985963, -208.736927, -206.451387, -204.130251, -201.774425, -199.384815, -196.96233, -194.507876, -192.022359, -189.506688, -186.961768, -184.3786, -181.768213, -179.131727, -176.470266, -173.78495, -171.076902, -168.347242, -165.597094, -162.827577, -160.039815, -157.234929, -154.414041, -151.578271, -148.728743, -145.866577, -142.992896, -140.263576, -137.830778, -135.376452, -132.89169, -130.367581, -127.795214, -125.165678, -122.470064, -119.699461, -116.844959, -113.897646, -110.848613, -107.68895, -104.409746, -101.00209, -97.4570718, -93.7657817, -88.8230502, -83.1911546, -77.4455483, -71.6128966, -65.7198647, -59.7931181, -53.8593219, -47.9451415, -42.0772421, -36.2822891, -30.5869477, -25.0178832, -19.601761, -14.3652462, -9.3350043, -4.53770046, 0, -3.73089304, -6.5685551, -8.57793107, -9.82396588, -10.3716044, -10.2857916, -9.63147224, -8.47359135, -6.8770938, -4.90692448, -2.6280283, -0.105350148, 2.59616506, 5.41157243, 8.27592705, 11.124284, 13.8806658, 16.4785513, 18.8779506, 21.0115545, 22.8120542, 24.2121404, 25.1445041, 25.5418364, 25.3368282, 24.4621703, 22.8505538, 20.4346697, 17.1472088, 13.1499588, 8.06251192, 1.74882998, -5.92712508, -14.1852465, -23.6196447, -35.0687356, -48.9434006, -65.6545213, -85.6129792, -109.229656, -137.839813, -170.648619, -207.886098, -249.782275, -296.567174, -343.207553, -396.51252, -458.027917, -529.299585, -611.873365, -709.258199, -818.637484, -938.151051, -1067.80298, -1211.40351, -1371.05401, -1548.85582, -1746.92298, -1967.34129, -2212.1827, -2484.78911, -2784.56008, -3110.28721, -3462.79375, -3846.34549, -4263.62803, -4718.36098, -5213.49802, -5748.2015, -6323.77836, -6932.51074, -7586.11881, -8289.03347, -9045.68564, -9860.50622, -10999.5102, -12140.1484, -13221.4557, -14182.4669, -14962.2171, -15232.5767, -15377.8545, -15451.5844, -15645.9194, -15959.5232, -16333.3017, -16768.8065, -17267.5893, -17831.2016, -18461.1952, -18559.3492, -19153.93, -20119.4333, -21330.3548, -22661.1902, -23986.4354, -25180.5859, -26306.667, -25672.3501}, 
{-255.940798, -256.367339, -256.68915, -256.908132, -257.026183, -257.045204, -256.967092, -256.793749, -256.527072, -256.168962, -255.721318, -255.186039, -254.565025, -253.860175, -253.073388, -252.206564, -251.261602, -250.240401, -249.144861, -247.976882, -246.738362, -245.431201, -244.057298, -242.618553, -241.116866, -239.554134, -237.932259, -236.253138, -234.518672, -232.730761, -230.891302, -229.002196, -227.065342, -225.08264, -223.085069, -221.059377, -218.991962, -216.883778, -214.735783, -212.548932, -210.324179, -208.062482, -205.764795, -203.432075, -201.065276, -198.665355, -196.233267, -193.769968, -191.276414, -188.75356, -186.202361, -183.615067, -181.001528, -178.36289, -175.700297, -173.014892, -170.307821, -167.580228, -164.833258, -162.068054, -159.285761, -156.487523, -153.674485, -150.847792, -148.008588, -145.158016, -142.297222, -139.584351, -137.173776, -134.742842, -132.282515, -129.783762, -127.23755, -124.634845, -121.966614, -119.223824, -116.397442, -113.478434, -110.457767, -107.326407, -104.075322, -100.695478, -97.1778421, -93.5133806, -88.582728, -82.9553714, -77.2141437, -71.3860438, -65.4980711, -59.5772246, -53.6505035, -47.7449069, -41.8874339, -36.1050838, -30.4248556, -24.8737486, -19.4787618, -14.2668944, -9.2651455, -4.50051433, 0, -4.2590713, -7.53368175, -9.89613625, -11.4187397, -12.1737969, -12.2336129, -11.6704924, -10.5567405, -8.96466193, -6.96656163, -4.6347445, -2.04151541, 0.740820739, 3.63995906, 6.58359466, 9.49942265, 12.4842037, 15.4897852, 18.061452, 20.1631275, 21.7587352, 22.8121986, 23.287441, 23.1483859, 22.3589569, 20.8830773, 18.6846705, 15.7276601, 11.9759694, 7.84675447, 2.68452095, -3.68277742, -11.4271869, -19.9181071, -29.7289077, -41.6469974, -56.0852165, -73.4564054, -94.1734043, -118.649053, -148.335801, -182.290477, -220.72252, -263.84137, -311.856465, -360.155218, -414.9746, -477.729557, -549.835033, -632.705974, -729.031696, -837.395208, -957.110227, -1088.60305, -1234.57147, -1396.69343, -1576.64688, -1775.76872, -1996.15543, -2240.27555, -2511.33655, -2808.90806, -3131.75972, -3480.79659, -3860.54219, -4273.85912, -4725.10992, -5217.54607, -5748.91918, -6320.09193, -6922.46037, -7567.81013, -8260.28076, -9004.01179, -9803.14276, -10914.9154, -12027.0916, -13080.5352, -14016.1102, -14774.6807, -15037.9966, -15178.7785, -15248.9392, -15430.9604, -15725.0714, -16077.0978, -16489.9891, -16966.6946, -17510.1639, -18123.3464, -18463.6604, -19131.8092, -20051.2851, -21145.5806, -22338.1881, -23552.6, -24712.3088, -25584.5459, -24273.8252}, 
{-256.431669, -256.811816, -257.089671, -257.267045, -257.345749, -257.327593, -257.214388, -257.007943, -256.71007, -256.322578, -255.847278, -255.285981, -254.640497, -253.912636, -253.104209, -252.217027, -251.252898, -250.213635, -249.101048, -247.916946, -246.66314, -245.341441, -243.95366, -242.501606, -240.98709, -239.411922, -237.777913, -236.086873, -234.340613, -232.540943, -230.689673, -228.788614, -226.839577, -224.844371, -222.828094, -220.780423, -218.690755, -216.560146, -214.389649, -212.18032, -209.933214, -207.649385, -205.329887, -202.975776, -200.588107, -198.167933, -195.71631, -193.234293, -190.722936, -188.183294, -185.616421, -183.01937, -180.397285, -177.751307, -175.082578, -172.392239, -169.681433, -166.951299, -164.202981, -161.437619, -158.656356, -155.860331, -153.050688, -150.228567, -147.39511, -144.551459, -141.698754, -138.996311, -136.599644, -134.183677, -131.739297, -129.257394, -126.728854, -124.144566, -121.495416, -118.772294, -115.966087, -113.067682, -110.067968, -106.957833, -103.728164, -100.369849, -96.8737761, -93.2308332, -88.3110107, -82.6892217, -76.9538517, -71.1321636, -65.2514207, -59.3388859, -53.4218226, -47.5274937, -41.6831624, -35.9160918, -30.2535451, -24.7227853, -19.3510756, -14.1656791, -9.19385895, -4.4628782, 0, -4.7069335, -8.35253154, -11.0154097, -12.7741834, -13.7074682, -13.8938798, -13.4120336, -12.3405451, -10.7580299, -8.74310349, -6.37438147, -3.73047934, -0.890012645, 2.06840309, 5.06615232, 8.02461952, 11.2105347, 14.5946176, 17.3171136, 19.3734098, 20.7588935, 21.4689516, 21.4989712, 20.8443395, 19.5004436, 17.4626705, 14.7264073, 11.2870412, 7.13995914, 2.98617605, -2.14327889, -8.4647069, -16.1944092, -24.9362299, -35.2126987, -47.7096674, -62.8271743, -80.9652578, -102.523956, -127.903308, -158.489146, -193.395689, -232.830104, -276.999556, -326.11121, -375.875254, -432.120077, -496.177088, -569.377699, -653.053318, -749.434528, -857.85458, -978.202538, -1111.11671, -1258.7651, -1422.62893, -1604.18941, -1804.68441, -2025.89377, -2269.86274, -2538.98756, -2833.62288, -3152.91501, -3498.03711, -3873.59663, -4282.62462, -4730.07961, -5219.49239, -5747.32626, -6314.04232, -6911.17142, -7549.47872, -8232.51654, -8963.83717, -9746.99291, -10825.5899, -11903.1132, -12923.1015, -13829.0939, -14564.629, -14825.8877, -14968.6723, -15042.5323, -15217.322, -15494.97, -15827.2785, -16218.6826, -16673.6175, -17196.5186, -17791.8211, -18345.6638, -19076.6677, -19952.5269, -20940.9352, -22009.5866, -23126.1751, -24258.3946, -24791.3387, -22786.2659}, 
{-257.431268, -257.75499, -257.979504, -258.106503, -258.137678, -258.074722, -257.919329, -257.673191, -257.338, -256.91545, -256.407232, -255.81504, -255.140566, -254.385503, -253.551544, -252.64038, -251.653705, -250.593212, -249.460593, -248.257541, -246.985748, -245.646907, -244.242711, -242.774852, -241.245024, -239.654918, -238.006228, -236.300645, -234.539863, -232.725575, -230.859473, -228.943249, -226.978597, -224.967208, -222.926768, -220.850638, -218.731984, -216.571981, -214.371802, -212.132622, -209.855613, -207.541949, -205.192805, -202.809354, -200.392769, -197.944224, -195.464893, -192.955949, -190.418567, -187.85392, -185.263181, -182.649235, -180.011508, -177.351136, -174.669257, -171.967006, -169.245521, -166.505937, -163.749392, -160.977023, -158.189965, -155.389355, -152.576331, -149.752028, -146.917583, -144.074134, -141.222815, -138.52374, -136.133203, -133.724466, -131.288361, -128.815718, -126.297367, -123.724141, -121.08687, -118.376384, -115.583515, -112.699093, -109.71395, -106.618917, -103.404823, -100.062501, -96.5827811, -92.9564939, -88.0459093, -82.4298765, -76.7008094, -70.8861628, -65.0133917, -59.1099508, -53.2032952, -47.3208797, -41.4901591, -35.7385884, -30.0936224, -24.5827161, -19.2333243, -14.0729019, -9.12890376, -4.42878484, 0, -5.00303355, -8.8953317, -11.7597936, -13.6793182, -14.7368048, -15.0151523, -14.59726, -13.5660268, -12.004352, -9.99513451, -7.62127355, -4.96566818, -2.11121751, 0.859179363, 3.86262335, 6.81621534, 10.1027332, 13.7058032, 16.4892092, 18.46984, 19.6645844, 20.0903312, 19.7639694, 18.7023876, 16.9224748, 14.4411198, 11.2752114, 7.44163839, 2.95728967, -1.26690126, -6.43590664, -12.8010511, -20.6136591, -29.6851895, -40.4868999, -53.6073451, -69.4298095, -88.3375773, -110.713933, -136.94216, -168.22095, -203.870013, -244.113099, -289.173955, -339.27633, -390.49654, -448.242624, -513.775189, -588.354839, -673.242182, -770.376963, -879.510591, -1000.78483, -1134.9335, -1283.89929, -1449.08211, -1631.88188, -1833.71259, -2055.95691, -2299.98218, -2567.23889, -2858.9704, -3174.80064, -3516.19208, -3887.72252, -4292.53977, -4736.02355, -5221.90031, -5745.71287, -6307.63319, -6900.01751, -7532.05589, -8206.62571, -8926.60434, -9694.86916, -10741.218, -11784.8777, -12771.9955, -13648.7186, -14361.1943, -14619.4101, -14763.1129, -14839.6611, -15007.5285, -15270.0906, -15584.4248, -15956.2217, -16391.1715, -16894.9647, -17473.2917, -18211.9151, -18996.4212, -19832.4961, -20725.8257, -21682.0961, -22706.9932, -23806.2031, -23939.7925, -21289.1416}, 
{-259.094316, -259.361922, -259.532746, -259.608387, -259.590445, -259.480519, -259.280209, -258.991114, -258.614834, -258.152968, -257.607115, -256.978875, -256.269847, -255.481631, -254.615826, -253.674031, -252.657847, -251.568872, -250.408706, -249.178949, -247.881199, -246.517056, -245.08812, -243.59599, -242.042266, -240.428546, -238.756431, -237.02752, -235.243411, -233.405706, -231.516003, -229.575901, -227.587, -225.550899, -223.479332, -221.368617, -219.214952, -217.019609, -214.783857, -212.508968, -210.196213, -207.84686, -205.462182, -203.043449, -200.591932, -198.1089, -195.595625, -193.053378, -190.483428, -187.887047, -185.265504, -182.62519, -179.962144, -177.277528, -174.572501, -171.848221, -169.10585, -166.346546, -163.57147, -160.78178, -157.978637, -155.163201, -152.33663, -149.500085, -146.654725, -143.80171, -140.942199, -138.237735, -135.846009, -133.437404, -131.002683, -128.532608, -126.017942, -123.449448, -120.817889, -118.114026, -115.328624, -112.452444, -109.476249, -106.390803, -103.186867, -99.8552052, -96.3865792, -92.7717522, -87.86886, -82.2588019, -76.5360907, -70.7282452, -64.8627838, -58.9672253, -53.0690883, -47.1958914, -41.3751533, -35.6343925, -30.0011278, -24.5028778, -19.1671611, -14.0214963, -9.09340211, -4.41039713, 0, -4.8693068, -8.65662365, -11.4433272, -13.3107943, -14.3404014, -14.6135254, -14.2115429, -13.2158306, -11.7077651, -9.76872325, -7.48008164, -4.92321697, -2.17950594, 0.669674757, 3.54294844, 6.35893842, 9.40035707, 12.6378404, 15.166477, 16.9829092, 18.0837794, 18.4657301, 18.1254035, 17.0594422, 15.2644884, 12.7371847, 9.47417331, 5.47209676, 0.727597392, -4.01523629, -9.78127155, -16.7980998, -25.2933126, -34.9652543, -46.30614, -59.9493169, -76.2811505, -95.6880064, -118.55625, -145.272247, -176.792222, -212.759246, -253.44819, -299.133925, -350.091322, -403.133739, -462.862938, -530.419171, -606.942686, -693.573732, -791.86048, -902.036691, -1024.49795, -1160.06003, -1310.39662, -1476.79621, -1660.54729, -1862.86519, -2085.12822, -2328.7945, -2595.32904, -2885.89302, -3200.05954, -3539.23962, -3907.95856, -4309.31214, -4748.53166, -5229.26654, -5747.33607, -6302.9887, -6890.03802, -7516.15733, -8183.82254, -8895.50955, -9653.69427, -10684.0611, -11710.0753, -12678.4107, -13535.7411, -14228.7403, -14469.0101, -14595.011, -14654.1618, -14804.4715, -15049.5839, -15347.0633, -15702.8575, -16122.914, -16613.1806, -17179.6048, -18079.5328, -18910.2012, -19709.1709, -20514.0028, -21362.2579, -22291.4971, -23339.2814, -23014.4422, -19922.9324}, 
{-261.466571, -261.683279, -261.804367, -261.831391, -261.765907, -261.60947, -261.363637, -261.029964, -260.610006, -260.10532, -259.51746, -258.847984, -258.098447, -257.270405, -256.365414, -255.385029, -254.330807, -253.204304, -252.007075, -250.740676, -249.406663, -248.006593, -246.542021, -245.014502, -243.425593, -241.77685, -240.069829, -238.306085, -236.487174, -234.614653, -232.690077, -230.715002, -228.690984, -226.619579, -224.50946, -222.358476, -220.164388, -217.92852, -215.652197, -213.336745, -210.983487, -208.59375, -206.168857, -203.710135, -201.218906, -198.696498, -196.144234, -193.563439, -190.955439, -188.321557, -185.66312, -182.98605, -180.286974, -177.567117, -174.827705, -172.069963, -169.295115, -166.504388, -163.699006, -160.880195, -158.04918, -155.207185, -152.355437, -149.49516, -146.62758, -143.753922, -140.875411, -138.156096, -135.756115, -133.340884, -130.901072, -128.427349, -125.910385, -123.34085, -120.709415, -118.006749, -115.223523, -112.350406, -109.378069, -106.297181, -103.098412, -99.7724333, -96.3099141, -92.7015244, -87.8045116, -82.200097, -76.4831251, -70.6810476, -64.8213164, -58.9313832, -53.0386999, -47.1707181, -41.3548898, -35.6186666, -29.9895004, -24.494843, -19.1621462, -14.0188616, -9.09244124, -4.41033677, 0, -4.25793476, -7.5495215, -9.94815898, -11.527246, -12.3601813, -12.5203636, -12.0811918, -11.1160645, -9.69838064, -7.90153889, -5.79893804, -3.46397687, -0.970054135, 1.60943139, 4.20108093, 6.73149572, 9.13307086, 11.3332353, 13.245485, 14.7976629, 15.9176116, 16.533174, 16.5721928, 15.9625108, 14.6319707, 12.5084154, 9.51968765, 5.59363016, 0.658085729, -5.154625, -12.1961123, -20.5998769, -30.4994192, -41.0863286, -52.9650614, -66.99125, -83.5809685, -103.150291, -126.115291, -152.892043, -184.182596, -220.030013, -260.794416, -306.835927, -358.514669, -413.79565, -476.032884, -546.185272, -625.211713, -714.07111, -813.753194, -925.14835, -1049.01329, -1186.26618, -1338.15481, -1505.77897, -1690.23846, -1891.9741, -2112.89443, -2355.62683, -2622.93927, -2914.61619, -3229.4526, -3568.35848, -3935.82755, -4334.70856, -4769.43104, -5243.25359, -5753.63842, -6301.32649, -6882.02839, -7502.28936, -8164.59166, -8871.41752, -9625.24918, -10660.3723, -11689.5149, -12657.2083, -13507.984, -14186.3733, -14390.0897, -14475.0279, -14491.4984, -14610.6502, -14834.962, -15116.563, -15460.4493, -15871.6171, -16355.0624, -16915.7814, -17941.6251, -18808.8361, -19576.8816, -20305.2286, -21053.3443, -21880.6959, -22846.7506, -22021.821, -18746.8997}, 
{-264.383961, -264.54945, -264.619945, -264.596979, -264.482085, -264.276795, -263.982643, -263.601161, -263.133882, -262.58234, -261.948067, -261.232596, -260.43746, -259.564192, -258.614325, -257.589391, -256.490924, -255.320457, -254.079522, -252.769652, -251.39238, -249.94924, -248.441764, -246.871485, -245.239936, -243.548649, -241.799159, -239.992997, -238.131696, -236.21679, -234.249812, -232.232293, -230.165768, -228.051769, -225.897247, -223.700913, -221.461411, -219.180097, -216.858329, -214.497464, -212.09886, -209.663872, -207.193858, -204.690176, -202.154182, -199.587233, -196.990687, -194.365901, -191.71423, -189.037034, -186.335668, -183.613282, -180.869401, -178.105345, -175.322431, -172.521978, -169.705303, -166.873726, -164.028565, -161.171137, -158.302761, -155.424754, -152.538437, -149.645126, -146.746139, -143.842796, -140.936414, -138.193991, -135.778545, -133.349695, -130.898018, -128.414093, -125.888497, -123.311809, -120.674606, -117.967466, -115.180967, -112.305688, -109.332205, -106.251098, -103.052943, -99.7283194, -96.2678046, -92.6619765, -87.7681573, -82.1673894, -76.4543359, -70.6563512, -64.8007895, -58.9150051, -53.0263523, -47.1621853, -41.3498584, -35.6167259, -29.990142, -24.4974611, -19.1660374, -14.0232252, -9.09637867, -4.41285219, 0, -3.56448324, -6.29318339, -8.25044246, -9.50060244, -10.1080053, -10.1369931, -9.65190779, -8.71709136, -7.39688583, -5.75563318, -3.85767541, -1.76735453, 0.450987487, 2.73300863, 5.0143669, 7.23072031, 8.92634855, 9.98099844, 11.2155884, 12.4819098, 13.6317537, 14.5169114, 14.9891741, 14.900333, 14.1021793, 12.4465044, 9.78509934, 5.9697554, 0.85226381, -6.10801221, -14.5229632, -24.4230695, -35.8388115, -47.4140249, -59.8725124, -74.3078487, -91.1665075, -110.894963, -133.939688, -160.747157, -191.75556, -227.422176, -268.195099, -314.522422, -366.852241, -424.212908, -488.827194, -561.49813, -643.028744, -734.222066, -835.666401, -948.641947, -1074.06101, -1212.866, -1366.06082, -1534.62176, -1719.52511, -1920.52452, -2140.09681, -2382.05258, -2650.60353, -2943.98486, -3260.11642, -3599.33658, -3966.08171, -4362.90715, -4793.30277, -5260.06619, -5762.56835, -6302.11849, -6876.31434, -7490.74771, -8148.01262, -8850.7031, -9601.41318, -10645.5789, -11682.2417, -12653.2852, -13500.5929, -14166.0484, -14330.5669, -14370.979, -14341.0119, -14427.808, -14632.2841, -14899.7674, -15233.9541, -15638.5406, -16117.2233, -16673.6984, -17745.0637, -18628.6515, -19384.4201, -20072.3277, -20752.3323, -21484.3924, -22328.466, -21010.8255, -17663.9018}, 
{-267.802551, -267.916946, -267.936399, -267.862443, -267.69661, -267.440433, -267.095444, -266.663176, -266.145161, -265.542931, -264.85802, -264.091958, -263.24628, -262.322516, -261.3222, -260.246864, -259.098041, -257.877262, -256.586061, -255.225969, -253.79852, -252.305245, -250.747677, -249.127349, -247.445792, -245.70454, -243.905124, -242.049077, -240.137932, -238.173221, -236.156476, -234.08923, -231.973015, -229.809364, -227.604995, -225.358739, -223.069364, -220.738233, -218.36671, -215.95616, -213.507948, -211.023436, -208.503989, -205.950971, -203.365747, -200.749679, -198.104134, -195.430474, -192.730064, -190.004268, -187.254449, -184.478443, -181.681219, -178.864219, -176.028882, -173.176651, -170.308964, -167.427264, -164.53299, -161.627583, -158.712483, -155.789133, -152.858971, -149.923439, -146.983978, -144.042027, -141.099028, -138.325351, -135.887313, -133.437946, -130.967742, -128.467188, -125.926775, -123.336993, -120.68833, -117.971277, -115.176323, -112.293957, -109.31467, -106.22895, -103.027288, -99.7001725, -96.2380934, -92.6315404, -87.7377029, -82.137558, -76.4257041, -70.6294084, -64.7759383, -58.8925612, -53.0065442, -47.1451549, -41.3356604, -35.6053282, -29.9814255, -24.4912197, -19.161978, -14.0209679, -9.09545667, -4.41271158, 0, -2.93065494, -5.14526076, -6.69995014, -7.65085576, -8.05411031, -7.96584647, -7.44219693, -6.53929435, -5.31327144, -3.82026086, -2.11639531, -0.257807469, 1.69936999, 3.69900437, 5.684963, 7.60111319, 8.63708658, 8.62900253, 9.22673764, 10.2125373, 11.3686469, 12.4773118, 13.3207774, 13.681289, 13.3410922, 12.0824322, 9.68755443, 5.9387043, 0.618127185, -7.42396188, -17.1315427, -28.4427608, -41.2957616, -53.8546172, -66.9445097, -81.8637078, -99.0825791, -119.071491, -142.300812, -169.240908, -200.082266, -235.660317, -276.500187, -323.127004, -376.065895, -435.130299, -501.734952, -576.582904, -660.377202, -753.820896, -857.35844, -972.267536, -1099.33761, -1239.43453, -1393.58032, -1562.72688, -1747.82613, -1948.20015, -2166.80108, -2408.35903, -2678.36753, -2973.63438, -3291.26273, -3631.04841, -3997.34981, -4392.43104, -4818.87669, -5279.03396, -5774.07487, -6305.83615, -6873.49053, -7481.94867, -8133.96947, -8832.31188, -9579.73482, -10632.7576, -11676.9387, -12651.5969, -13496.051, -14149.6198, -14276.9981, -14275.2115, -14201.2748, -14258.1211, -14445.006, -14700.3856, -15026.5552, -15425.8103, -15900.4462, -16452.7585, -17463.1933, -18337.1791, -19106.6519, -19803.5479, -20459.8034, -21107.3543, -21778.137, -20006.0965, -16666.347}, 
{-271.647706, -271.716623, -271.689473, -271.567838, -271.353299, -271.047438, -270.651835, -270.168072, -269.59773, -268.942391, -268.203635, -267.383044, -266.482199, -265.502681, -264.446072, -263.313953, -262.107905, -260.829509, -259.480347, -258.061999, -256.576048, -255.024074, -253.407659, -251.728384, -249.987829, -248.187577, -246.329209, -244.414306, -242.444448, -240.421218, -238.346197, -236.220965, -234.047105, -231.826197, -229.568075, -227.27002, -224.929215, -222.546974, -220.124609, -217.663434, -215.164763, -212.629908, -210.060185, -207.456905, -204.821382, -202.154931, -199.458863, -196.734494, -193.983135, -191.206102, -188.404706, -185.56621, -182.706284, -179.826545, -176.928612, -174.014103, -171.084635, -168.141828, -165.187298, -162.222665, -159.249546, -156.269559, -153.284322, -150.295454, -147.304572, -144.313294, -141.323239, -138.509094, -136.041393, -133.564809, -131.069742, -128.546589, -125.985749, -123.377622, -120.712605, -117.981098, -115.173499, -112.280206, -109.291619, -106.198135, -102.990155, -99.6580751, -96.1922955, -92.5832144, -87.6827451, -82.0744753, -76.3556767, -70.5536941, -64.695872, -58.8095552, -52.9220884, -47.0608162, -41.2530834, -35.5262346, -29.9076145, -24.4245679, -19.1044393, -13.9745736, -9.06231533, -4.39500924, 0, -2.84630297, -4.99575848, -6.50386958, -7.42613927, -7.81807058, -7.73516654, -7.23293016, -6.36686447, -5.19247249, -3.76525725, -2.14072176, -0.374369045, 1.47829787, 3.36177596, 5.2205622, 6.99915358, 7.81148788, 7.48341078, 7.83806501, 8.6419706, 9.66164755, 10.6636159, 11.4143956, 11.6805067, 11.2284691, 9.82480304, 7.23602835, 3.2286651, -2.43076672, -11.0052467, -21.3212721, -33.3034729, -46.8764792, -60.1793142, -74.0294114, -89.7203663, -107.712491, -128.466098, -152.441499, -180.099006, -211.487724, -247.604322, -288.989568, -336.184226, -389.729064, -449.070189, -516.11669, -591.682995, -676.583536, -771.632741, -878.002301, -995.712736, -1124.58766, -1265.11972, -1419.16742, -1587.97606, -1772.79089, -1973.56914, -2193.13687, -2435.72525, -2706.84391, -3003.01635, -3321.08807, -3660.67245, -4026.07285, -4419.43986, -4843.09927, -5299.24711, -5789.43691, -6315.58562, -6876.90905, -7478.44087, -8123.03305, -8813.53757, -9552.80638, -10600.3099, -11638.127, -12604.9552, -13439.4917, -14080.434, -14187.8559, -14164.8274, -14069.4557, -14110.6793, -14286.9017, -14533.6802, -14852.5606, -15245.0886, -15712.8102, -16257.2712, -16923.2746, -17721.8723, -18580.2181, -19425.466, -20184.7698, -20785.2832, -21154.1602, -19070.5803, -15737.4698}, 
{-275.882505, -275.910852, -275.840923, -275.674394, -275.412939, -275.058236, -274.611959, -274.075784, -273.451387, -272.740444, -271.94463, -271.06562, -270.105092, -269.064719, -267.946178, -266.751144, -265.481294, -264.138303, -262.723846, -261.239599, -259.687238, -258.068438, -256.384875, -254.638225, -252.830164, -250.962366, -249.036508, -247.054266, -245.017315, -242.92733, -240.785988, -238.594964, -236.355933, -234.070572, -231.754941, -229.403222, -227.009421, -224.574746, -222.100408, -219.587614, -217.037574, -214.451496, -211.830589, -209.176063, -206.489125, -203.770984, -201.02285, -198.245931, -195.441437, -192.610575, -189.754554, -186.845145, -183.913633, -180.961865, -177.991689, -175.004951, -172.003499, -168.98918, -165.96384, -162.929327, -159.887487, -156.840168, -153.789216, -150.736479, -147.683804, -144.633038, -141.586027, -138.722581, -136.218235, -133.707805, -131.181599, -128.629929, -126.043105, -123.411438, -120.725237, -117.974814, -115.150479, -112.242542, -109.241314, -106.137105, -102.920226, -99.5809871, -96.1096989, -92.4966718, -87.5830875, -81.9579913, -76.22424, -70.4094354, -64.5411796, -58.6470744, -52.7547217, -46.8917234, -41.0856814, -35.3641976, -29.7548738, -24.2853121, -18.9831143, -13.8758823, -8.99121795, -4.35672323, 0, -3.37359941, -5.95787474, -7.81610745, -9.01157904, -9.60757099, -9.66736479, -9.25424193, -8.43148388, -7.26237214, -5.8101882, -4.13821352, -2.30972961, -0.388017947, 1.56363998, 3.4819627, 5.30366871, 6.37742294, 6.55593627, 7.10604839, 7.83846635, 8.56389718, 9.0930479, 9.23662556, 8.80533718, 7.60988979, 5.46099043, 2.16934614, -2.45433607, -8.59934915, -17.0910787, -27.249491, -39.0730514, -52.5602251, -66.325244, -81.0586899, -97.8202735, -117.02373, -139.082794, -164.411202, -193.422688, -226.142437, -263.49099, -305.958101, -354.033526, -408.207021, -466.371298, -532.262418, -607.019395, -691.781248, -787.686993, -897.490039, -1018.74186, -1149.49671, -1289.59416, -1442.51888, -1610.11895, -1794.24249, -1996.57003, -2219.15531, -2464.2348, -2735.9489, -3031.80811, -3349.03605, -3687.4706, -4051.37935, -4442.99662, -4865.10776, -5320.08996, -5808.2997, -6331.23649, -6886.53467, -7480.15353, -8114.93051, -8793.70306, -9519.30862, -10544.6649, -11560.0091, -12505.6584, -13321.9302, -13949.1418, -14056.1122, -14035.6562, -13944.4476, -13986.0794, -14159.0698, -14400.7211, -14712.6124, -15096.3229, -15553.4318, -16085.5183, -16132.0233, -16791.3993, -17811.5597, -18940.4178, -19925.887, -20515.8805, -20458.3117, -18220.1678, -14871.7741}, 
{-280.514156, -280.498876, -280.382934, -280.168106, -279.856168, -279.448896, -278.948067, -278.355457, -277.672841, -276.901997, -276.0447, -275.102726, -274.077852, -272.971854, -271.786508, -270.52359, -269.184876, -267.772143, -266.287166, -264.731723, -263.107588, -261.416538, -259.66035, -257.840799, -255.959663, -254.018716, -252.019735, -249.964496, -247.854776, -245.69235, -243.478995, -241.216487, -238.906603, -236.551117, -234.172754, -231.762375, -229.31059, -226.818498, -224.287193, -221.717772, -219.111331, -216.468968, -213.791778, -211.080858, -208.337304, -205.562213, -202.756681, -199.921805, -197.05868, -194.168404, -191.252073, -188.264956, -185.25497, -182.224205, -179.174751, -176.108698, -173.028135, -169.935152, -166.831839, -163.720287, -160.602585, -157.480822, -154.357089, -151.233475, -148.112071, -144.994966, -141.88425, -138.964749, -136.416903, -133.865921, -131.302047, -128.715524, -126.096596, -123.435508, -120.722502, -117.947822, -115.101713, -112.174418, -109.15618, -106.037245, -102.807854, -99.4582531, -95.9786849, -92.3593934, -87.4304742, -81.7866475, -76.0365771, -70.2081561, -64.3292772, -58.4278332, -52.5317169, -46.6688212, -40.8670389, -35.1542627, -29.5583856, -24.1073003, -18.8288997, -13.7510765, -8.90172365, -4.30873386, 0, -4.07439526, -7.23606303, -9.55839341, -11.1147765, -11.9786024, -12.2232611, -11.9221429, -11.1486377, -9.97613567, -8.47802691, -6.72770149, -4.79854952, -2.76396108, -0.697326271, 1.32796482, 3.23852209, 4.70925468, 5.63081524, 6.49913687, 7.18689359, 7.56675942, 7.5114084, 6.89351454, 5.58575188, 3.46079445, 0.391316264, -3.75000865, -9.09050626, -15.7575025, -23.9745586, -33.7374791, -45.1447194, -58.294735, -72.4236351, -88.0610482, -105.966562, -126.497335, -150.010527, -176.863298, -207.412806, -241.696472, -280.488507, -324.208625, -373.276544, -428.111981, -484.817809, -549.209798, -622.786877, -707.047973, -803.492014, -916.554907, -1041.20896, -1173.68117, -1313.14427, -1464.78517, -1631.09052, -1814.54698, -2018.74447, -2244.81565, -2492.68969, -2764.78406, -3059.95112, -3376.00917, -3713.14207, -4075.54187, -4465.56079, -4886.60221, -5341.29089, -5828.39728, -6348.87205, -6898.3695, -7483.95148, -8108.40202, -8774.50513, -9485.04483, -10483.3424, -11471.5102, -12392.1981, -13188.0558, -13801.733, -13915.0982, -13905.4365, -13827.1606, -13874.9827, -14047.7807, -14285.9748, -14591.3658, -14965.7547, -15410.9424, -15928.7298, -15306.0904, -15813.2156, -17005.757, -18439.3661, -19669.6945, -20252.3937, -19743.1152, -17415.702, -14076.9462}, 
{-285.519624, -285.454667, -285.286836, -285.017996, -284.650018, -284.184767, -283.624113, -282.969923, -282.224065, -281.388406, -280.464815, -279.455159, -278.361306, -277.185124, -275.928481, -274.593244, -273.181282, -271.694462, -270.134652, -268.50372, -266.803534, -265.035961, -263.202869, -261.306126, -259.347601, -257.32916, -255.252672, -253.120004, -250.933025, -248.693601, -246.403602, -244.064894, -241.679345, -239.248825, -236.802145, -234.327135, -231.811298, -229.255628, -226.661118, -224.028762, -221.359552, -218.654483, -215.914547, -213.140738, -210.334049, -207.495475, -204.626007, -201.726639, -198.798365, -195.842179, -192.859072, -189.788273, -186.69388, -183.578226, -180.443642, -177.292461, -174.127017, -170.94964, -167.762663, -164.568419, -161.36924, -158.167458, -154.965406, -151.765416, -148.56982, -145.380951, -142.201141, -139.219751, -136.621648, -134.023436, -131.415322, -128.787516, -126.130224, -123.433655, -120.688016, -117.883518, -115.010366, -112.05877, -109.018937, -105.881076, -102.635394, -99.2721005, -95.7814028, -92.1535092, -87.2084661, -81.5461753, -75.7806028, -69.9399309, -64.0523415, -58.1460169, -52.2491391, -46.3898902, -40.5964524, -34.8970077, -29.3197383, -23.8928262, -18.6444536, -13.6028025, -8.79605519, -4.25239365, 0, -4.85343334, -8.6575792, -11.4969217, -13.4559451, -14.6191335, -15.070971, -14.8959417, -14.17853, -13.0032198, -11.4544955, -9.616841, -7.57474061, -5.41267846, -3.21513869, -1.06660547, 0.948437062, 2.8753453, 4.64818375, 5.88878126, 6.54047665, 6.54660874, 5.85051636, 4.39553834, 2.12501348, -1.01771938, -5.08932142, -10.1464538, -16.2457778, -23.4439544, -31.2825423, -40.5221762, -51.3740482, -64.0493501, -78.4726904, -94.9985528, -114.057843, -135.947729, -160.965377, -189.407955, -221.572629, -257.527261, -297.868111, -342.937213, -393.076596, -448.628292, -503.944637, -566.854782, -639.198181, -722.814289, -819.542562, -935.353648, -1062.90657, -1196.80484, -1335.63581, -1486.12049, -1651.32803, -1834.32762, -2040.5354, -2270.14013, -2520.77022, -2793.0093, -3087.20581, -3401.91009, -3737.71569, -4098.67821, -4487.26409, -4907.52626, -5362.34247, -5848.77348, -6367.17048, -6911.13143, -7488.79971, -8102.86415, -8756.01361, -9450.93692, -10419.4076, -11377.7587, -12271.4079, -13045.7727, -13646.2709, -13770.4852, -13776.8915, -13717.1224, -13774.86, -13949.4673, -14185.2868, -14484.4327, -14849.0188, -15281.1594, -15782.9686, -14518.249, -14877.4183, -16231.754, -17952.5338, -19411.0354, -19978.5365, -19026.3147, -16655.7527, -13351.638}, 
{-290.875875, -290.752199, -290.523958, -290.193087, -289.761522, -289.231198, -288.604051, -287.882016, -287.06703, -286.161027, -285.165943, -284.083715, -282.916277, -281.665565, -280.333515, -278.922063, -277.433144, -275.868693, -274.230647, -272.52094, -270.741509, -268.894289, -266.981216, -265.004226, -262.965253, -260.866234, -258.709104, -256.4958, -254.228255, -251.908407, -249.53819, -247.119541, -244.654394, -242.144686, -239.623749, -237.077159, -234.490117, -231.863538, -229.198341, -226.495441, -223.755755, -220.9802, -218.169693, -215.32515, -212.447489, -209.537625, -206.596476, -203.624959, -200.62399, -197.594486, -194.537364, -191.377726, -188.193946, -184.988586, -181.764204, -178.523363, -175.26862, -172.002538, -168.727676, -165.446595, -162.161854, -158.876014, -155.591635, -152.311278, -149.037502, -145.772868, -142.519937, -139.471736, -136.81672, -134.164624, -131.505662, -128.830045, -126.127987, -123.3897, -120.605399, -117.765296, -114.859603, -111.878534, -108.812302, -105.65112, -102.385201, -99.0047569, -95.5000021, -91.861149, -86.9006238, -81.2223061, -75.4442316, -69.5948346, -63.7025496, -57.7958111, -51.9030534, -46.052711, -40.2732184, -34.5930099, -29.04052, -23.6441832, -18.4324339, -13.4337065, -8.67643538, -4.18905508, 0, -5.61545651, -10.0496791, -13.3978867, -15.7552983, -17.217133, -17.8786096, -17.8349473, -17.1813649, -16.0130814, -14.425316, -12.5132874, -10.3722148, -8.09731702, -5.78381316, -3.52692218, -1.42186306, 0.94405699, 3.54817782, 5.14643236, 5.75243988, 5.37981961, 4.04219082, 1.75317276, -1.47361533, -5.62455418, -10.6860246, -16.6444072, -23.4860828, -31.1974322, -38.6418852, -47.3405221, -57.6166089, -69.7934118, -84.4706129, -101.83327, -121.992731, -145.189335, -171.663425, -201.655342, -235.405426, -273.012234, -314.901044, -361.339938, -412.596996, -468.940302, -523.286702, -585.09332, -656.466049, -739.510778, -836.333397, -954.043006, -1083.62726, -1218.53151, -1356.93452, -1506.67901, -1671.26879, -1854.20765, -2062.38578, -2295.15096, -2548.15671, -2820.28451, -3113.33263, -3426.64146, -3761.22027, -4120.90618, -4508.23826, -4927.82355, -5382.73729, -5868.47217, -6384.81001, -6923.53837, -7493.66319, -8097.7335, -8738.29835, -9417.90679, -10355.9263, -11283.8829, -12150.1209, -12902.9849, -13490.8192, -13627.9444, -13652.7447, -13613.8606, -13683.1818, -13860.5625, -14094.5028, -14387.4247, -14741.7498, -15159.9002, -15644.2975, -13841.2725, -14074.1046, -15558.4948, -17510.1439, -19144.7529, -19678.0227, -18325.6544, -15938.8896, -12694.501}, 
{-296.320558, -296.105001, -295.787093, -295.368665, -294.851545, -294.237564, -293.528552, -292.726338, -291.832753, -290.849627, -289.778788, -288.622068, -287.381295, -286.0583, -284.654913, -283.172963, -281.614281, -279.980695, -278.274037, -276.496136, -274.648821, -272.733924, -270.753272, -268.708697, -266.602029, -264.435096, -262.209729, -259.927758, -257.591013, -255.201323, -252.760519, -250.27043, -247.732886, -245.149717, -242.547791, -239.915894, -237.242507, -234.528647, -231.775332, -228.983579, -226.154406, -223.288829, -220.387866, -217.452535, -214.483853, -211.482836, -208.450502, -205.387869, -202.295954, -199.175774, -196.028346, -192.778108, -189.504317, -186.20965, -182.896785, -179.568399, -176.22717, -172.875775, -169.516892, -166.153198, -162.78737, -159.422085, -156.060022, -152.703858, -149.356269, -146.019934, -142.697529, -139.582128, -136.86223, -134.147838, -131.429288, -128.696914, -125.941051, -123.152033, -120.320197, -117.435876, -114.489405, -111.47112, -108.371356, -105.180446, -101.888727, -98.4865326, -94.9641982, -91.3120586, -86.3448478, -80.6654282, -74.8907923, -69.0494255, -63.1698132, -57.2804407, -51.4097935, -45.5863569, -39.8386165, -34.1950575, -28.6841654, -23.3344256, -18.1743235, -13.2323445, -8.53697393, -4.11669731, 0, -6.04555386, -10.8443651, -14.4969958, -17.1040083, -18.7659647, -19.5834271, -19.6569577, -19.0871188, -17.9744725, -16.4195809, -14.5230063, -12.3853107, -10.1070565, -7.78880577, -5.53112065, -3.43456335, -0.925151875, 1.99291506, 3.6933206, 4.22004775, 3.61707951, 1.92839889, -0.802011114, -4.53016749, -9.21208723, -14.8037873, -21.2612848, -28.5405966, -36.5977398, -43.9873126, -52.5806043, -62.7540574, -74.8841146, -89.9237073, -107.961034, -128.930561, -153.035787, -180.480205, -211.467314, -246.200607, -284.385465, -326.875101, -373.970469, -425.972521, -483.182212, -540.022721, -604.142219, -677.311103, -761.299771, -857.87862, -972.327238, -1098.61782, -1232.40806, -1372.70964, -1525.38198, -1693.21003, -1878.9787, -2088.04103, -2320.03018, -2571.77778, -2842.26766, -3133.62504, -3446.03886, -3780.71996, -4140.61078, -4527.85893, -4946.41624, -5398.89807, -5881.30435, -6393.37706, -6927.08385, -7491.13711, -8087.96994, -8720.01547, -9389.7068, -10310.6522, -11221.3159, -12071.3371, -12810.3551, -13388.0093, -13525.9879, -13553.8487, -13518.9243, -13589.5633, -13765.1962, -13994.8302, -14280.4167, -14623.9074, -15027.2539, -15492.4078, -13792.7207, -14034.5509, -15458.3434, -17304.5431, -18813.5948, -19225.9437, -17782.0345, -15260.3225, -12103.734}, 
{-302.119652, -301.801575, -301.383952, -300.868484, -300.256868, -299.550802, -298.751985, -297.862114, -296.882889, -295.816008, -294.663168, -293.426069, -292.106408, -290.705883, -289.226193, -287.669037, -286.036112, -284.329116, -282.549749, -280.699708, -278.780691, -276.794397, -274.742524, -272.62677, -270.448834, -268.210414, -265.913208, -263.558915, -261.149232, -258.685858, -256.170491, -253.60483, -250.990573, -248.329417, -245.639935, -242.915032, -240.147413, -237.33823, -234.488632, -231.599771, -228.672796, -225.70886, -222.709113, -219.674705, -216.606787, -213.50651, -210.375024, -207.213481, -204.023031, -200.804825, -197.560013, -194.21512, -190.847541, -187.460044, -184.055398, -180.636371, -177.205731, -173.766248, -170.320689, -166.871824, -163.422421, -159.975247, -156.533073, -153.098665, -149.674793, -146.264225, -142.86973, -139.683291, -136.892348, -134.109429, -131.325035, -128.52967, -125.713835, -122.868034, -119.982769, -117.048542, -114.055855, -110.995212, -107.857114, -104.632065, -101.310566, -97.8831195, -94.340229, -90.6723964, -85.7006073, -80.024722, -74.2586571, -68.4308682, -62.5698109, -56.7039407, -50.8617131, -45.0715837, -39.3620079, -33.7614414, -28.2983397, -23.0011583, -17.8983528, -13.0183787, -8.38969147, -4.04074673, 0, -6.31373581, -11.3462257, -15.2010923, -17.9819581, -19.7924456, -20.7361774, -20.9167762, -20.4378644, -19.4030645, -17.9159992, -16.0802911, -13.9995625, -11.7774362, -9.51753472, -7.32348052, -5.29889621, -2.77393253, 0.25828393, 1.94461104, 2.34717022, 1.52808284, -0.450529666, -3.52654591, -7.63784449, -12.722304, -18.717803, -25.5622202, -33.193434, -41.5493232, -48.9892024, -57.6083209, -67.8181263, -80.0300666, -95.4242238, -114.027728, -135.662742, -160.510122, -188.750726, -220.565411, -256.135034, -294.820504, -337.872177, -385.631336, -438.439264, -496.637244, -556.533104, -623.509944, -698.91741, -784.10515, -880.422809, -990.909538, -1113.16053, -1245.48542, -1387.89488, -1543.87257, -1715.34284, -1904.23005, -2113.95584, -2344.60699, -2594.63685, -2863.45162, -3153.23811, -3464.91661, -3799.82391, -4160.00249, -4547.17092, -4964.48157, -5414.02475, -5892.63338, -6400.11428, -6929.21034, -7487.879, -8078.37228, -8702.94215, -9363.84062, -10268.9123, -11163.4185, -11998.2129, -12724.1494, -13292.0818, -13430.6264, -13461.033, -13429.4, -13500.9618, -13674.6591, -13899.7922, -14177.7838, -14510.0562, -14898.032, -15343.1337, -13908.2147, -14195.2928, -15513.0487, -17170.1634, -18475.3177, -18737.1924, -17264.4683, -14622.6787, -11578.5359}, 
{-308.313295, -307.884642, -307.359391, -306.739104, -306.025342, -305.219666, -304.323638, -303.33882, -302.266772, -301.109056, -299.867234, -298.542867, -297.137516, -295.652743, -294.09011, -292.451177, -290.737506, -288.950659, -287.092198, -285.163682, -283.166675, -281.102737, -278.973429, -276.780314, -274.524953, -272.208907, -269.833737, -267.401005, -264.912273, -262.369101, -259.773052, -257.125686, -254.428566, -251.683252, -248.899397, -246.074347, -243.205349, -240.293704, -237.34071, -234.347666, -231.315871, -228.246623, -225.141222, -222.000967, -218.827156, -215.621088, -212.384062, -209.117378, -205.822333, -202.500227, -199.152358, -195.708895, -192.243808, -188.75994, -185.260132, -181.747223, -178.224055, -174.693469, -171.158306, -167.621406, -164.085611, -160.553761, -157.028698, -153.513262, -150.010293, -146.522634, -143.053124, -139.791773, -136.924095, -134.066912, -131.210931, -128.34686, -125.465406, -122.557277, -119.61318, -116.623824, -113.579915, -110.472161, -107.291269, -104.027948, -100.672905, -97.2168469, -93.6504818, -89.964517, -84.9899133, -79.3216454, -73.5686273, -67.7592026, -61.9217147, -56.0845074, -50.2759241, -44.5243084, -38.858004, -33.3053543, -27.8947029, -22.6543935, -17.6127695, -12.7981746, -8.23895237, -3.9634463, 0, -6.38272729, -11.486852, -15.416403, -18.2754093, -20.1678996, -21.1979029, -21.4694481, -21.0865641, -20.1532797, -18.7736238, -17.0516254, -15.0913133, -12.9967164, -10.8718636, -8.82078376, -6.94750582, -4.55796949, -1.64219092, -0.109400662, 0.107391502, -0.924824213, -3.13905759, -6.46831842, -10.8456165, -16.2039616, -22.4763634, -29.5958319, -37.4953768, -46.1080078, -53.7210824, -62.5166686, -72.9214719, -85.3621977, -101.100212, -120.14489, -142.283032, -167.690943, -196.544932, -229.021305, -265.296371, -304.524536, -348.215021, -396.74407, -450.487927, -509.822836, -572.974734, -643.00775, -720.835706, -807.372423, -903.531723, -1010.06609, -1128.24787, -1259.04206, -1403.46376, -1562.63042, -1737.61354, -1929.48462, -2139.70145, -2368.86143, -2617.14061, -2884.55268, -3173.11143, -3484.25096, -3819.34352, -4179.65653, -4566.50552, -4982.23284, -5428.42024, -5902.88445, -6405.57191, -6930.54276, -7484.5353, -8069.53353, -8687.52143, -9340.48299, -10229.8949, -11108.3752, -11928.0349, -12640.9847, -13199.3355, -13338.8621, -13372.2354, -13344.2816, -13416.9215, -13588.6314, -13809.0986, -14079.239, -14399.9687, -14772.2036, -15196.8597, -14143.1173, -14503.5722, -15685.0639, -17094.432, -18138.5161, -18224.1557, -16758.1903, -14025.4148, -11117.6778}, 
{-314.941622, -314.396926, -313.758267, -313.027085, -312.204821, -311.292914, -310.292804, -309.205931, -308.033735, -306.777656, -305.439134, -304.019609, -302.520521, -300.943309, -299.289414, -297.560276, -295.757334, -293.882028, -291.935799, -289.920087, -287.83633, -285.68597, -283.470446, -281.191198, -278.849666, -276.44729, -273.985509, -271.465765, -268.889496, -266.258143, -263.573146, -260.835944, -258.047977, -255.210686, -252.325398, -249.393612, -246.416828, -243.396488, -240.334036, -237.230916, -234.088573, -230.908448, -227.691986, -224.44063, -221.155824, -217.839011, -214.491635, -211.11514, -207.710969, -204.280566, -200.825374, -197.279562, -193.713307, -190.129511, -186.531075, -182.920902, -179.301894, -175.676952, -172.048979, -168.420876, -164.795546, -161.17589, -157.564811, -153.965209, -150.379988, -146.81205, -143.264295, -139.92412, -136.97449, -134.037804, -131.105004, -128.16703, -125.214826, -122.239334, -119.231495, -116.182251, -113.082545, -109.923318, -106.695512, -103.39007, -99.9979333, -96.5100439, -92.9173438, -89.2107752, -84.2347765, -78.5776566, -72.8415045, -67.0544684, -61.2446965, -55.4403371, -49.6695383, -43.9604485, -38.3412158, -32.8399885, -27.4849148, -22.304143, -17.3258213, -12.5780979, -8.08912104, -3.88703901, 0, -6.21525322, -11.197835, -15.0491551, -17.8706237, -19.7636506, -20.8296457, -21.1700191, -20.8861805, -20.07954, -18.8515075, -17.3034929, -15.5369061, -13.6531572, -11.7536559, -9.93981235, -8.31303635, -6.23294723, -3.69498482, -2.47841886, -2.52570415, -3.77929548, -6.18164764, -9.67521543, -14.2024536, -19.7058171, -26.1277605, -33.4107388, -41.4972066, -50.3296189, -58.2564805, -67.3986444, -78.1767502, -91.0114379, -107.079722, -126.424055, -148.885189, -174.656851, -203.932768, -236.906668, -273.772278, -313.704748, -358.226384, -407.7302, -462.609209, -523.256426, -589.504495, -662.446893, -742.616724, -830.547097, -926.771116, -1030.07309, -1144.87234, -1274.35645, -1420.38978, -1582.13519, -1759.96845, -1954.26536, -2164.84904, -2392.77353, -2639.69572, -2906.28711, -3194.18459, -3505.01815, -3840.09022, -4200.14813, -4586.19403, -4999.88332, -5442.38749, -5912.48273, -6410.30014, -6931.70602, -7481.75241, -8062.04672, -8674.19634, -9319.80866, -10192.788, -11054.3706, -11858.0896, -12557.4782, -13106.0695, -13247.6972, -13285.3937, -13262.5634, -13336.9865, -13506.7935, -13722.4587, -13984.4955, -14293.4173, -14649.7377, -15053.9701, -14452.7911, -14906.6311, -15936.842, -17064.7758, -17811.7847, -17699.2205, -16248.4353, -13467.9872, -10719.9313}, 
{-322.044773, -321.381149, -320.625435, -319.778986, -318.843156, -317.819299, -316.70877, -315.512923, -314.233113, -312.870693, -311.427019, -309.903444, -308.301323, -306.62201, -304.866859, -303.037226, -301.134464, -299.159927, -297.114971, -295.000948, -292.819215, -290.571125, -288.258032, -285.88129, -283.442255, -280.942281, -278.382721, -275.76493, -273.090263, -270.360074, -267.575717, -264.738546, -261.849917, -258.911183, -255.917157, -252.872603, -249.782361, -246.647997, -243.471079, -240.253173, -236.995845, -233.700663, -230.369193, -227.003001, -223.603655, -220.17272, -216.711763, -213.222352, -209.706052, -206.16443, -202.599054, -198.947255, -195.276226, -191.588927, -187.888317, -184.177355, -180.459, -176.736211, -173.011946, -169.289166, -165.57083, -161.859896, -158.159323, -154.47207, -150.801098, -147.149364, -143.519828, -140.09688, -137.060555, -134.039623, -131.025281, -128.008728, -124.981161, -121.933778, -118.857776, -115.744353, -112.584706, -109.370034, -106.091534, -102.740403, -99.3078387, -95.7850394, -92.1632025, -88.4335256, -83.4572076, -77.8142136, -72.0980902, -66.3367056, -60.5579281, -54.789626, -49.0596676, -43.395921, -37.8262546, -32.3785366, -27.0806352, -21.9604188, -17.0457556, -12.3645139, -7.94456186, -3.81376781, 0, -5.77403853, -10.4107656, -14.0055756, -16.6538631, -18.4510224, -19.4924481, -19.8735345, -19.6896762, -19.0362676, -18.008703, -16.7023771, -15.2126842, -13.6350188, -12.0647753, -10.5973482, -9.32813199, -7.75455026, -5.88657311, -5.17214791, -5.57853251, -7.07298477, -9.62276255, -13.1951237, -17.7573261, -23.2766276, -29.7202861, -37.0555594, -45.2497053, -54.2699818, -62.6689246, -72.347245, -83.6966176, -97.108717, -113.490803, -132.976758, -155.562973, -181.486448, -210.984181, -244.293171, -281.650418, -322.568326, -368.229017, -419.011257, -475.293813, -537.455452, -606.279271, -681.638626, -763.8112, -853.074679, -949.706746, -1051.20671, -1164.02643, -1292.70705, -1439.64644, -1602.86651, -1782.35388, -1978.09518, -2188.96984, -2416.32331, -2662.70887, -2929.37121, -3217.3972, -3528.19443, -3862.87539, -4222.05252, -4606.56777, -5017.6463, -5456.22942, -5921.85338, -6414.8492, -6933.32503, -7480.17676, -8056.5049, -8663.40994, -9301.99236, -10156.7799, -10999.5891, -11785.6634, -12470.2467, -13008.5826, -13154.1341, -13198.4459, -13183.2393, -13260.701, -13428.826, -13639.5822, -13893.2661, -14190.1743, -14530.6033, -14914.8495, -14792.5991, -15351.7116, -16230.8361, -17068.6219, -17503.7181, -17174.7736, -15720.4378, -12949.8526, -10384.0674}, 
{-329.973393, -329.191013, -328.316047, -327.349883, -326.293905, -325.149499, -323.918051, -322.600946, -321.19957, -319.71531, -318.149549, -316.503675, -314.779073, -312.977129, -311.099227, -309.146755, -307.121097, -305.023639, -302.855768, -300.618867, -298.314325, -295.943525, -293.507854, -291.008697, -288.44744, -285.825469, -283.144169, -280.404927, -277.609127, -274.758155, -271.853398, -268.89624, -265.888068, -262.830267, -259.717797, -256.555393, -253.347864, -250.096806, -246.803811, -243.470475, -240.09839, -236.689153, -233.244356, -229.765594, -226.254462, -222.712552, -219.14146, -215.54278, -211.918105, -208.269031, -204.597151, -200.838345, -197.061346, -193.269174, -189.464846, -185.651381, -181.831798, -178.009116, -174.186353, -170.366529, -166.552662, -162.74777, -158.954873, -155.176988, -151.417136, -147.678335, -143.963602, -140.454346, -137.327688, -134.218737, -131.118947, -128.019774, -124.91267, -121.789092, -118.640493, -115.458328, -112.234053, -108.95912, -105.624986, -102.223104, -98.7449291, -95.1819161, -91.5255194, -87.7671936, -82.7897476, -77.1622777, -71.4672052, -65.7319625, -59.9839819, -54.2506955, -48.5595358, -42.9379348, -37.413325, -32.0131386, -26.7648079, -21.6957651, -16.8334426, -12.2052725, -7.8386872, -3.76111894, 0, -4.42420863, -7.97005689, -10.7158349, -12.7398329, -14.1203411, -14.9356495, -15.2640484, -15.1838278, -14.7732781, -14.1106893, -13.2743516, -12.3425552, -11.3935903, -10.505747, -9.7573154, -9.22658579, -8.79269911, -8.50549565, -8.89372204, -9.99299361, -11.8389257, -14.4671337, -17.9132329, -22.2128386, -27.4015663, -33.5150312, -40.5888486, -48.6586341, -57.7600028, -67.0361731, -77.7423697, -90.1819269, -104.658179, -121.29951, -140.496729, -162.687301, -188.227044, -217.471778, -250.777322, -288.499498, -331.540522, -379.54352, -432.757409, -491.431104, -555.813521, -625.336789, -701.270807, -784.068686, -874.183538, -972.068476, -1076.21994, -1191.4392, -1320.73063, -1465.99033, -1626.85166, -1803.96392, -1997.97639, -2209.30282, -2438.88156, -2687.90793, -2957.61402, -3248.20778, -3559.57083, -3892.29629, -4248.18283, -4628.47576, -5034.97717, -5469.07672, -5930.12253, -6418.61751, -6936.07908, -7481.83962, -8056.05087, -8658.86461, -9290.43259, -10118.2994, -10933.3758, -11693.9653, -12358.3714, -12884.8977, -13044.4452, -13107.6551, -13113.1464, -13199.5952, -13366.0146, -13569.727, -13811.83, -14093.4209, -14415.5973, -14779.4567, -15111.1513, -15793.5989, -16558.4946, -17137.5335, -17262.4108, -16664.8217, -15076.4614, -12448.2526, -10119.639}, 
{-338.221334, -337.322127, -336.32905, -335.243553, -334.067084, -332.801092, -331.447023, -330.006327, -328.480451, -326.870845, -325.178956, -323.406232, -321.554122, -319.624074, -317.617535, -315.535956, -313.380782, -311.153464, -308.855448, -306.488184, -304.053119, -301.551702, -298.98538, -296.355603, -293.663818, -290.911474, -288.100018, -285.2309, -282.305567, -279.325467, -276.292049, -273.20676, -270.07105, -266.886367, -263.649295, -260.363816, -257.033972, -253.661368, -250.24761, -246.794304, -243.303056, -239.775473, -236.21316, -232.617723, -228.990769, -225.333902, -221.648731, -217.93686, -214.199895, -210.439443, -206.657109, -202.786051, -198.897807, -194.995467, -191.082121, -187.16086, -183.234773, -179.30695, -175.38048, -171.458454, -167.543962, -163.640094, -159.749938, -155.876586, -152.023127, -148.192652, -144.388249, -140.788318, -137.567048, -134.365969, -131.176804, -127.99128, -124.801122, -121.598055, -118.373804, -115.120095, -111.828653, -108.491204, -105.099472, -101.645183, -98.1200628, -94.5158361, -90.8242284, -87.036965, -82.0599486, -76.4506548, -70.7796096, -65.0737753, -59.3601145, -53.6655893, -48.0171622, -42.4417954, -36.9664514, -31.6180925, -26.423681, -21.4101794, -16.6045498, -12.0337548, -7.72475654, -3.7045175, 0, -3.17986169, -5.72000203, -7.68302697, -9.13154243, -10.1281544, -10.7354687, -11.0160914, -11.0326283, -10.8476854, -10.5238687, -10.1237841, -9.71003749, -9.34523482, -9.09198205, -9.0128851, -9.17054991, -9.80583067, -11.0067974, -12.4328753, -14.1848666, -16.3635731, -19.0697968, -22.4043398, -26.4680041, -31.3615916, -37.1859043, -44.0417442, -52.0299132, -61.2512135, -71.4243109, -83.1722601, -96.7105037, -112.254485, -129.159567, -148.071233, -169.864319, -195.012292, -223.988617, -257.266761, -295.320191, -340.444873, -390.737099, -446.313759, -507.291745, -573.787946, -643.524913, -719.61246, -802.766064, -893.7012, -993.133343, -1101.03147, -1219.76994, -1350.27866, -1493.73928, -1651.84752, -1826.0683, -2017.86654, -2229.52235, -2461.50017, -2713.37516, -2986.29064, -3279.60565, -3591.65281, -3922.5038, -4275.17707, -4651.33845, -5053.4226, -5483.29468, -5940.0008, -6424.18167, -6940.49741, -7484.95098, -8056.79176, -8655.26912, -9279.63247, -10081.3895, -10869.4667, -11605.0489, -12249.3208, -12763.4673, -12934.5099, -13013.9055, -13037.4803, -13132.126, -13297.6219, -13495.8039, -13728.3484, -13996.9323, -14303.2323, -14648.925, -15375.6905, -16166.2639, -16831.2585, -17181.2876, -17026.9643, -16178.9021, -14447.7141, -12001.7271, -9906.44932}, 
{-346.578053, -345.566176, -344.458837, -343.25755, -341.963827, -340.579184, -339.105134, -337.543191, -335.894869, -334.161682, -332.345143, -330.446768, -328.468069, -326.41056, -324.275756, -322.065171, -319.780318, -317.422711, -314.993864, -312.495291, -309.928507, -307.295024, -304.596357, -301.83402, -299.009526, -296.12439, -293.180126, -290.178247, -287.120267, -284.007701, -280.842062, -277.624864, -274.35762, -271.041846, -267.67552, -264.261996, -260.804675, -257.305184, -253.765152, -250.186209, -246.569981, -242.918099, -239.232191, -235.513886, -231.764811, -227.986596, -224.18087, -220.349261, -216.493398, -212.614909, -208.715423, -204.724953, -200.718295, -196.698632, -192.669143, -188.633012, -184.593417, -180.553541, -176.516566, -172.485671, -168.464038, -164.454848, -160.461283, -156.486523, -152.53375, -148.606145, -144.706889, -141.011472, -137.691247, -134.39402, -131.111804, -127.836608, -124.560446, -121.275326, -117.973261, -114.646262, -111.286339, -107.885504, -104.435768, -100.929141, -97.3576358, -93.7132625, -89.9880322, -86.1739563, -81.1997754, -75.6115002, -69.9679211, -64.2955543, -58.6209161, -52.9705228, -47.3708904, -41.8485354, -36.429974, -31.1417223, -26.0102967, -21.0622134, -16.3239887, -11.8221387, -7.58317978, -3.63362813, 0, -2.50498158, -4.50402156, -6.05124187, -7.20076441, -8.0067111, -8.52320384, -8.80436454, -8.90431511, -8.87717747, -8.77707353, -8.65812519, -8.57445436, -8.58018296, -8.72943289, -9.07632608, -9.67498441, -10.9851906, -13.119018, -15.2130226, -17.4082536, -19.8457602, -22.6665917, -26.0117972, -30.022426, -34.8395272, -40.6041502, -47.457344, -55.540158, -64.9936412, -75.910444, -88.4977608, -102.911161, -119.306212, -136.503385, -155.325797, -176.952594, -201.939876, -230.843741, -264.220289, -302.625618, -349.313765, -401.32178, -458.677905, -521.410382, -589.547454, -659.44224, -735.716885, -819.318409, -911.193836, -1012.29019, -1123.90417, -1246.20573, -1378.34562, -1520.86408, -1677.13812, -1849.2711, -2039.36638, -2251.11287, -2484.66802, -2738.45963, -3013.63599, -3308.9109, -3621.57326, -3951.31143, -4301.87959, -4675.16563, -5073.93276, -5500.29582, -5953.16015, -6433.24656, -6947.35985, -7489.20085, -8057.38436, -8650.5252, -9267.23817, -10047.6874, -10813.5511, -11528.0566, -12154.4316, -12655.9034, -12830.3168, -12916.5372, -12948.6705, -13047.8005, -13213.6041, -13409.5177, -13637.0305, -13897.6316, -14192.8102, -14524.0553, -15543.8983, -16406.7614, -16984.6374, -17149.5191, -16773.3992, -15728.2705, -13886.1257, -11628.2999, -9734.89912}, 
{-354.681743, -353.564079, -352.350017, -351.041085, -349.638811, -348.144724, -346.560352, -344.887224, -343.126867, -341.28081, -339.350582, -337.33771, -335.243724, -333.070151, -330.818519, -328.490358, -326.087195, -323.610559, -321.061977, -318.442979, -315.755093, -312.999847, -310.178769, -307.293388, -304.345232, -301.33583, -298.266709, -295.139398, -291.955425, -288.716319, -285.423608, -282.078821, -278.683485, -275.23913, -271.741793, -268.195863, -264.605796, -260.973297, -257.300073, -253.58783, -249.838275, -246.053115, -242.234055, -238.382802, -234.501063, -230.590545, -226.652953, -222.689994, -218.703375, -214.694801, -210.665981, -206.544406, -202.407605, -198.258894, -194.101587, -189.939001, -185.77445, -181.61125, -177.452716, -173.302164, -169.162908, -165.038263, -160.931546, -156.846072, -152.785155, -148.752111, -144.750255, -140.952553, -137.527896, -134.129709, -130.75031, -127.382017, -124.017148, -120.648019, -117.266948, -113.866254, -110.438253, -106.975263, -103.469602, -99.9135874, -96.2995365, -92.619767, -88.8665966, -85.0323428, -80.0634785, -74.4972415, -68.88344, -63.2483312, -57.6181724, -52.0192207, -46.4777335, -41.019968, -35.6721814, -30.460631, -25.411574, -20.5512677, -15.9059693, -11.5019361, -7.36542531, -3.5226942, 0, -3.70722268, -6.69797508, -9.04152246, -10.8071301, -12.0640631, -12.8815869, -13.3289667, -13.4754677, -13.3903551, -13.1428943, -12.8023504, -12.4379888, -12.1190746, -11.9148732, -11.8946497, -12.1276694, -12.803949, -14.0100042, -15.5419716, -17.4949918, -19.9642054, -23.0447531, -26.8317753, -31.4204126, -36.9058056, -43.3830949, -50.9474211, -59.6939247, -69.7177462, -80.7751711, -93.424442, -107.862356, -124.285711, -141.793938, -161.132516, -183.339558, -208.941069, -238.463057, -272.431528, -311.37249, -358.110049, -410.17269, -467.636792, -530.578733, -599.074893, -669.600526, -746.733415, -831.450219, -924.727598, -1027.54221, -1141.49038, -1266.17173, -1400.54825, -1545.00554, -1702.83575, -1876.02607, -2066.56368, -2277.58674, -2509.66986, -2762.13204, -3036.83343, -3332.08717, -3645.40611, -3976.40732, -4328.27339, -4702.55027, -5101.50197, -5526.86062, -5977.72544, -6454.68496, -6963.26342, -7498.35431, -8058.83766, -8643.5935, -9251.50184, -10026.6721, -10786.4475, -11493.4009, -12110.1048, -12599.132, -12752.6078, -12817.1838, -12828.4818, -12919.7005, -13087.8372, -13289.232, -13522.9151, -13787.9165, -14083.2665, -14407.9951, -15484.6796, -16328.0494, -16836.1148, -16906.8857, -16438.3725, -15328.5851, -13475.5338, -11375.4012, -9599.94046}, 
{-362.793212, -361.578193, -360.265363, -358.856271, -357.352467, -355.7555, -354.066919, -352.288272, -350.42111, -348.466981, -346.427435, -344.30402, -342.098286, -339.811782, -337.446057, -335.00266, -332.483141, -329.889048, -327.221931, -324.483339, -321.67482, -318.797925, -315.854202, -312.845201, -309.77247, -306.637559, -303.442016, -300.187392, -296.875235, -293.507094, -290.084519, -286.609059, -283.082262, -279.505678, -275.873155, -272.190253, -268.462628, -264.69208, -260.880407, -257.029408, -253.140882, -249.216629, -245.258447, -241.268136, -237.247495, -233.198323, -229.122419, -225.021582, -220.897611, -216.752306, -212.587465, -208.327671, -204.053614, -199.768766, -195.476601, -191.180591, -186.88421, -182.59093, -178.304225, -174.027567, -169.76443, -165.518286, -161.292608, -157.090871, -152.916545, -148.773105, -144.664024, -140.760026, -137.227821, -133.725937, -130.247005, -126.783656, -123.328519, -119.874227, -116.41341, -112.938697, -109.442721, -105.918112, -102.357501, -98.7535171, -95.0987927, -91.385958, -87.6076437, -83.7564805, -78.7968622, -73.2571654, -67.6781257, -62.0857421, -56.5060136, -50.9649392, -45.4885178, -40.1027485, -34.8336301, -29.7071619, -24.7493426, -19.9861714, -15.4436471, -11.1477689, -7.12453561, -3.39994632, 0, -5.2773147, -9.56009029, -12.9373682, -15.4981898, -17.3315966, -18.5266299, -19.1723312, -19.3577418, -19.1719033, -18.703857, -18.0426443, -17.2773067, -16.4968855, -15.7904223, -15.2469583, -14.955535, -14.7442039, -14.6647222, -15.3964656, -16.9725491, -19.4260877, -22.7901964, -27.0979902, -32.3825841, -38.6770932, -46.0146324, -54.4283168, -63.9512613, -74.616581, -85.7054254, -98.2785957, -112.594796, -128.912732, -146.818521, -166.907161, -189.953006, -216.416537, -246.758234, -281.438579, -320.918051, -367.126333, -418.607556, -475.534748, -538.080936, -606.41915, -677.712052, -755.895626, -841.895493, -936.637271, -1041.04658, -1156.59214, -1282.99268, -1419.63701, -1566.95816, -1727.52112, -1902.93366, -2094.80357, -2305.16743, -2535.10676, -2785.23527, -3057.93463, -3351.68409, -3665.1109, -3998.03541, -4352.2996, -4728.81747, -5128.83952, -5553.36698, -6002.1674, -6475.70617, -6977.79571, -7505.03946, -8056.74425, -8632.21689, -9230.76418, -10003.2552, -10759.544, -11461.0467, -12069.1796, -12545.3589, -12672.704, -12709.7925, -12694.4538, -12775.9539, -12948.0115, -13157.9134, -13401.7, -13675.4117, -13975.0888, -14296.7718, -15323.0984, -16110.9758, -16565.6814, -16592.493, -16096.6882, -14983.5446, -13158.3397, -11191.4749, -9489.42555}, 
{-370.95284, -369.650863, -368.248795, -366.748232, -365.150768, -363.457996, -361.671512, -359.792909, -357.823782, -355.765726, -353.620334, -351.389202, -349.073923, -346.676091, -344.197302, -341.63915, -339.003229, -336.291133, -333.504456, -330.644794, -327.71374, -324.712889, -321.643836, -318.508173, -315.307497, -312.043401, -308.71748, -305.331327, -301.886539, -298.384708, -294.827429, -291.216296, -287.552905, -283.838849, -280.066422, -276.242065, -272.37233, -268.459112, -264.504307, -260.509813, -256.477525, -252.409339, -248.307151, -244.172858, -240.008356, -235.81554, -231.596308, -227.352555, -223.086177, -218.799071, -214.493133, -210.08936, -205.672301, -201.245605, -196.812923, -192.377905, -187.9442, -183.515458, -179.09533, -174.687465, -170.295513, -165.923124, -161.573947, -157.251634, -152.959833, -148.702195, -144.482369, -140.469163, -136.827418, -133.22016, -129.640332, -126.080879, -122.534742, -118.994866, -115.454194, -111.905669, -108.342235, -104.756836, -101.142414, -97.4919132, -93.7982771, -90.0544491, -86.2533725, -82.3879908, -77.4425099, -71.9354594, -66.3974931, -60.85429, -55.331529, -49.854889, -44.4500489, -39.1426876, -33.9584839, -28.9231169, -24.0622654, -19.4016082, -14.9668244, -10.7835927, -6.87759217, -3.27450163, 0, -6.70328335, -12.1604267, -16.4784267, -19.7642796, -22.1249819, -23.6675302, -24.4989209, -24.7261504, -24.4562153, -23.796112, -22.852837, -21.7333867, -20.5447576, -19.3939462, -18.387949, -17.6337624, -16.6389294, -15.4248103, -15.4543083, -16.7059654, -19.1583238, -22.7899256, -27.5793132, -33.5050285, -40.5456137, -48.679611, -57.8855627, -68.1420107, -79.4274973, -90.5983198, -103.166755, -117.448018, -133.757326, -152.187496, -173.164943, -197.175386, -224.600761, -255.823001, -291.224042, -331.185818, -376.538481, -427.079334, -483.102618, -544.902572, -612.773436, -684.939846, -764.283046, -851.614679, -947.746384, -1053.4898, -1169.83426, -1297.19654, -1435.83675, -1586.39995, -1750.31719, -1928.66662, -2122.5264, -2332.65893, -2560.52989, -2807.9494, -3077.2677, -3367.9196, -3680.56345, -4015.5348, -4372.62224, -4751.86545, -5153.00405, -5575.99991, -6021.91504, -6491.18916, -6986.56629, -7505.82733, -8048.61433, -8614.56937, -9203.33451, -9971.04638, -10721.7288, -11415.9003, -12014.0792, -12476.784, -12579.4233, -12591.0318, -12551.4607, -12625.2, -12802.7648, -13022.7371, -13278.2286, -13562.3507, -13868.2152, -14188.9337, -15093.529, -15806.1416, -16224.9872, -16248.2818, -15774.2413, -14701.0816, -12927.0185, -11060.0099, -9393.72023}, 
{-379.327632, -377.966969, -376.500657, -374.930459, -373.258133, -371.485442, -369.614147, -367.646008, -365.582786, -363.426243, -361.17814, -358.840238, -356.414297, -353.902078, -351.305344, -348.625854, -345.86537, -343.025652, -340.108462, -337.115561, -334.04871, -330.90967, -327.700201, -324.422065, -321.077023, -317.666836, -314.193265, -310.65807, -307.063014, -303.409856, -299.700359, -295.936282, -292.119388, -288.251436, -284.328307, -280.354825, -276.335887, -272.273446, -268.169452, -264.025859, -259.844617, -255.627679, -251.376997, -247.094522, -242.782206, -238.442002, -234.075861, -229.685735, -225.273576, -220.841336, -216.390966, -211.835627, -207.267986, -202.691922, -198.111311, -193.53003, -188.951955, -184.380963, -179.820932, -175.275737, -170.749256, -166.245365, -161.767941, -157.320861, -152.908002, -148.53324, -144.200453, -140.078584, -136.332617, -132.625987, -128.95187, -125.303442, -121.673879, -118.056356, -114.444051, -110.830139, -107.207796, -103.570198, -99.910521, -96.2219414, -92.4976351, -88.7307781, -84.9145464, -81.0421162, -76.1219616, -70.6603216, -65.1752702, -59.6919389, -54.2354593, -48.8309628, -43.503581, -38.2784453, -33.1806873, -28.2354385, -23.4678303, -18.9029944, -14.5660621, -10.482165, -6.67643462, -3.17400245, 0, -6.26190938, -11.3724423, -15.4325388, -18.5431392, -20.8051837, -22.3196124, -23.1873656, -23.5093833, -23.3866059, -22.9199735, -22.2104263, -21.3589044, -20.4663481, -19.6336975, -18.9618929, -18.5518744, -18.1225271, -17.7202118, -18.3100412, -19.9110863, -22.5424184, -26.2231088, -30.9722288, -36.8088494, -43.752042, -51.8208778, -61.034428, -71.4117639, -82.9719567, -94.9021788, -108.35843, -123.609351, -140.923582, -160.276951, -182.084504, -206.839375, -234.898048, -266.61701, -302.352744, -342.461736, -387.15824, -436.984261, -492.324112, -553.562107, -621.082559, -694.19587, -774.628743, -863.033968, -960.064338, -1066.37264, -1182.09915, -1309.03559, -1448.65997, -1602.02047, -1769.28782, -1951.02668, -2147.80174, -2359.8944, -2588.21701, -2833.99098, -3097.07163, -3380.51496, -3689.48746, -4025.74229, -4385.24889, -4766.6315, -5166.76909, -5583.83343, -6022.39574, -6483.40735, -6972.89748, -7485.95837, -8021.78363, -8579.56686, -9158.50168, -9907.13419, -10637.9674, -11312.8567, -11893.6577, -12342.2257, -12439.4627, -12448.8134, -12409.6851, -12488.4364, -12674.2226, -12903.3025, -13167.4765, -13458.5446, -13768.3071, -14088.5642, -14867.5771, -15523.205, -15936.4975, -15988.5043, -15560.2749, -14532.8588, -12787.3056, -10927.8184, -9270.6994}, 
{-387.751961, -386.336967, -384.810387, -383.174157, -381.430218, -379.580507, -377.626964, -375.571525, -373.416131, -371.162718, -368.813227, -366.369596, -363.833762, -361.207665, -358.493242, -355.692434, -352.807177, -349.83941, -346.791073, -343.664103, -340.460439, -337.18202, -333.830784, -330.408669, -326.917614, -323.359558, -319.736439, -316.050195, -312.302766, -308.496089, -304.632103, -300.712747, -296.739958, -292.715677, -288.639527, -284.514654, -280.344228, -276.130263, -271.874772, -267.579769, -263.247267, -258.87928, -254.477822, -250.044905, -245.582545, -241.092753, -236.577545, -232.038932, -227.47893, -222.899551, -218.302809, -213.59484, -208.875614, -204.149222, -199.419757, -194.691311, -189.967974, -185.25384, -180.553001, -175.869548, -171.207573, -166.571168, -161.964426, -157.391438, -152.856296, -148.363093, -143.91592, -139.683192, -135.82938, -132.01967, -128.247504, -124.506319, -120.789556, -117.090654, -113.403053, -109.720192, -106.035511, -102.342449, -98.6344469, -94.9049429, -91.1473771, -87.355189, -83.5218182, -79.6407042, -74.7493472, -69.3392635, -63.9134587, -58.4963946, -53.1125326, -47.7863346, -42.5422621, -37.4047767, -32.3983402, -27.547414, -22.8764599, -18.4099395, -14.1723144, -10.1880463, -6.48159675, -3.07742744, 0, -5.4117809, -9.83982594, -13.3741549, -16.1047878, -18.1217442, -19.5150442, -20.3747074, -20.7907538, -20.8532032, -20.6520754, -20.2773903, -19.8191677, -19.3674274, -19.0121892, -18.843473, -18.9512987, -19.3510272, -20.1220127, -21.5231464, -23.6284498, -26.5119444, -30.2476516, -34.9095931, -40.5717903, -47.3082647, -55.1930378, -64.3001311, -74.7035661, -86.4773643, -99.2442445, -113.695009, -130.039069, -148.485838, -168.824605, -191.474845, -216.968063, -245.639707, -277.825227, -313.860071, -354.079687, -398.126957, -447.240678, -501.891804, -562.551284, -629.690071, -703.797829, -785.322117, -874.72921, -972.485381, -1079.0569, -1193.59255, -1319.48637, -1459.49044, -1615.10574, -1785.27895, -1970.10359, -2169.67316, -2384.00352, -2613.28348, -2857.78657, -3114.5657, -3390.49618, -3695.26002, -4032.06794, -4393.16535, -4775.83089, -5174.1647, -5584.62137, -6015.3101, -6467.74758, -6952.38887, -7460.43716, -7990.31508, -8540.44527, -9109.25038, -9832.0056, -10536.0679, -11185.6467, -11744.9515, -12178.1919, -12277.6283, -12294.0519, -12265.3646, -12354.1974, -12549.0804, -12786.5728, -13057.8032, -13353.8999, -13665.9914, -13985.2061, -14619.0451, -15216.6879, -15636.653, -15737.4586, -15377.6232, -14415.6653, -12710.1031, -10838.058, -9163.58804}, 
{-396.18132, -394.707652, -393.117043, -391.411585, -389.59337, -387.664489, -385.627035, -383.4831, -381.234774, -378.88415, -376.43332, -373.884376, -371.239409, -368.500512, -365.669776, -362.749293, -359.741154, -356.647453, -353.47028, -350.211728, -346.873888, -343.458852, -339.968712, -336.40556, -332.771487, -329.068586, -325.298948, -321.464666, -317.567831, -313.610534, -309.594869, -305.522925, -301.396797, -297.218574, -292.990268, -288.714012, -284.391942, -280.026153, -275.618739, -271.171795, -266.687415, -262.167695, -257.614727, -253.030608, -248.417431, -243.777291, -239.112283, -234.424501, -229.716039, -224.988992, -220.245455, -215.38728, -210.518976, -205.644812, -200.769054, -195.895971, -191.02983, -186.174899, -181.335444, -176.515735, -171.720038, -166.95262, -162.217751, -157.519696, -152.862724, -148.251102, -143.689098, -139.342829, -135.374524, -131.454651, -127.576984, -123.7353, -119.923373, -116.134981, -112.363897, -108.603898, -104.848759, -101.092256, -97.3281647, -93.5502602, -89.7523182, -85.9281144, -82.0714242, -78.1760233, -73.3143208, -67.960155, -62.5984388, -57.2528479, -51.9470578, -46.7047443, -41.5495832, -36.50525, -31.5954205, -26.8437704, -22.2739753, -17.909711, -13.7746531, -9.89247743, -6.28685953, -2.98147516, 0, -4.4528948, -8.10515314, -11.0351055, -13.3210823, -15.041414, -16.2744311, -17.098464, -17.5918432, -17.8328992, -17.8999624, -17.8713634, -17.8254324, -17.8405001, -17.9948968, -18.366953, -19.0349993, -20.2905194, -22.2422942, -24.4865217, -27.1472079, -30.348359, -34.2139813, -38.8680809, -44.434664, -51.0377369, -58.8013056, -67.8493764, -78.3059554, -90.295049, -103.82248, -119.173771, -136.508385, -155.985782, -177.257148, -200.736083, -226.971724, -256.276016, -288.960901, -325.338323, -365.720226, -409.194219, -457.669203, -511.696666, -571.828094, -638.614977, -713.736331, -796.33423, -886.678278, -985.038082, -1091.68324, -1204.76661, -1329.2617, -1469.03055, -1625.99632, -1798.12351, -1985.15396, -2186.82954, -2403.10037, -2633.45269, -2877.14549, -3128.79703, -3398.41416, -3699.19964, -4035.52827, -4396.81644, -4779.34667, -5175.12207, -5579.31568, -6002.79187, -6447.53909, -6928.88693, -7433.43026, -7958.53945, -8501.58488, -9059.93694, -9750.34492, -10420.9556, -11039.2945, -11572.8874, -11989.26, -12097.5672, -12129.2857, -12119.8144, -12222.8385, -12426.935, -12671.395, -12947.2599, -13245.5709, -13557.3694, -13873.6969, -14357.3489, -14897.6022, -15329.6313, -15488.6107, -15209.7151, -14328.1191, -12678.9971, -10798.8036, -9084.53262}, 
{-404.288181, -402.707269, -401.00991, -399.198129, -397.273948, -395.23939, -393.096478, -390.847237, -388.493687, -386.037853, -383.481758, -380.827425, -378.076877, -375.232137, -372.295228, -369.268173, -366.152995, -362.951718, -359.666364, -356.298956, -352.851519, -349.326073, -345.724644, -342.049254, -338.301925, -334.484682, -330.599547, -326.648543, -322.633693, -318.557021, -314.420549, -310.226301, -305.976299, -301.672567, -297.309288, -292.892569, -288.428613, -283.919698, -279.368101, -274.7761, -270.145971, -265.479994, -260.780445, -256.049601, -251.289741, -246.503141, -241.69208, -236.858835, -232.005683, -227.134902, -222.248769, -217.256249, -212.254956, -207.249189, -202.243249, -197.241437, -192.248052, -187.267396, -182.303768, -177.361469, -172.444799, -167.558059, -162.705549, -157.89157, -153.120421, -148.396403, -143.723817, -139.260776, -135.161699, -131.11366, -127.110988, -123.148013, -119.219064, -115.31847, -111.44056, -107.579664, -103.730111, -99.8862307, -96.042352, -92.1928044, -88.3319172, -84.4540197, -80.5534411, -76.6245109, -71.7841609, -66.483874, -61.1856787, -55.9123776, -50.6867734, -45.5316684, -40.4698654, -35.5241667, -30.7173751, -26.072293, -21.611723, -17.3584677, -13.3353295, -9.56511112, -6.07061503, -2.87464381, 0, -4.27673842, -7.78022516, -10.58873, -12.7805229, -14.4338735, -15.6270516, -16.4383272, -16.9459699, -17.2282497, -17.3634363, -17.4297995, -17.5056092, -17.6691351, -17.9986471, -18.572415, -19.4687086, -20.8916109, -22.9373649, -25.3996632, -28.3837355, -31.9948115, -36.3381211, -41.518894, -47.6423601, -54.813749, -63.1382905, -72.7212145, -83.6677506, -96.0831287, -109.772277, -125.250838, -142.714131, -162.357475, -184.18495, -208.487497, -235.607053, -265.796313, -299.307967, -336.39471, -377.309233, -420.359078, -468.334091, -521.867539, -581.592686, -648.142799, -724.364602, -808.124537, -899.502506, -998.57841, -1105.43215, -1217.51893, -1340.75132, -1479.24664, -1635.08879, -1806.21, -1992.40651, -2193.47457, -2409.54164, -2639.99746, -2883.87039, -3135.94806, -3405.62377, -3705.05389, -4038.3108, -4395.76896, -4774.09041, -5165.96414, -5567.02213, -5987.46416, -6429.24965, -6912.05775, -7417.402, -7941.10807, -8479.00161, -9026.90828, -9674.06868, -10299.5398, -10875.7935, -11375.3017, -11770.5364, -11896.7612, -11955.1285, -11976.9135, -12098.7927, -12310.3203, -12557.1334, -12830.9819, -13123.6159, -13426.7853, -13732.2402, -14082.1429, -14564.8506, -14999.8877, -15206.7784, -15005.0471, -14214.218, -12653.8155, -10841.4647, -9044.1751}, 
{-412.375627, -410.673832, -408.85762, -406.928888, -404.889534, -402.741455, -400.486548, -398.126711, -395.663841, -393.099835, -390.436591, -387.676007, -384.819979, -381.870406, -378.829183, -375.69821, -372.479382, -369.174598, -365.785755, -362.314751, -358.763482, -355.133846, -351.42774, -347.647062, -343.79371, -339.86958, -335.87657, -331.816577, -327.691499, -323.503233, -319.253677, -314.944727, -310.578281, -306.156238, -301.662207, -297.107614, -292.504104, -287.854168, -283.160296, -278.424979, -273.650708, -268.839974, -263.995267, -259.119079, -254.213899, -249.28222, -244.326531, -239.349324, -234.353089, -229.340316, -224.313498, -219.193752, -214.066706, -208.936614, -203.807731, -198.684312, -193.570611, -188.470883, -183.389383, -178.330366, -173.298087, -168.2968, -163.33076, -158.404222, -153.52144, -148.68667, -143.904167, -139.320715, -135.079679, -130.891355, -126.750756, -122.652898, -118.592796, -114.565465, -110.565921, -106.589177, -102.63025, -98.6841541, -94.7459045, -90.8105162, -86.8730045, -82.9283842, -78.9716704, -74.9978783, -70.1760016, -64.9314403, -59.6995224, -54.5020689, -49.3609009, -44.2978392, -39.334705, -34.4933194, -29.7955032, -25.2630776, -20.9178636, -16.7816822, -12.8763544, -9.22370126, -5.8455438, -2.76370303, 0, -4.45688243, -8.09447055, -10.9965005, -13.2467085, -14.9288307, -16.1266032, -16.9237621, -17.4040437, -17.6511842, -17.7489196, -17.7809861, -17.8311199, -17.9830572, -18.320534, -18.9272867, -19.8870512, -21.1394058, -22.7474924, -25.1211193, -28.3131314, -32.3763741, -37.3636923, -43.3279312, -50.321936, -58.3985518, -67.6106238, -78.010997, -89.6525167, -102.588028, -116.140316, -131.359975, -148.518867, -167.888856, -190.037243, -215.088173, -243.087004, -274.216969, -308.661298, -346.603223, -388.225974, -431.064751, -478.756741, -532.003267, -591.505655, -657.965228, -735.227914, -820.064285, -912.389515, -1012.11878, -1119.16725, -1230.52643, -1352.60854, -1489.2888, -1642.65494, -1811.14523, -1994.8365, -2193.80556, -2408.51227, -2638.56334, -2883.14757, -3138.58407, -3411.68857, -3711.15811, -4039.99394, -4391.54607, -4763.59458, -5151.05354, -5550.96001, -5970.85969, -6412.35401, -6898.60056, -7406.32824, -7929.67672, -8462.78566, -8999.79474, -9598.97915, -10174.3091, -10703.8905, -11165.8289, -11538.2303, -11686.4286, -11776.4826, -11834.8287, -11977.4071, -12195.2257, -12441.9996, -12710.6015, -12993.904, -13284.7799, -13576.1019, -13814.8306, -14244.316, -14675.8218, -14920.6116, -14789.9492, -14095.0981, -12647.3221, -10945.4592, -9056.50968}, 
{-420.690164, -418.881681, -416.958887, -414.92363, -412.777758, -410.523117, -408.161555, -405.694919, -403.125057, -400.453816, -397.683043, -394.814586, -391.850291, -388.792006, -385.641579, -382.400856, -379.071685, -375.655914, -372.155389, -368.571958, -364.907468, -361.163766, -357.3427, -353.446118, -349.475865, -345.43379, -341.32174, -337.141563, -332.895105, -328.584213, -324.210736, -319.776521, -315.283414, -310.733263, -306.102366, -301.405884, -296.659281, -291.865234, -287.02642, -282.145514, -277.225194, -272.268136, -267.277017, -262.254511, -257.203297, -252.12605, -247.025447, -241.904164, -236.764878, -231.610265, -226.443001, -221.196395, -215.943996, -210.689982, -205.438535, -200.193834, -194.96006, -189.741392, -184.542012, -179.366099, -174.217833, -169.101395, -164.020964, -158.980721, -153.984846, -149.03752, -144.142921, -139.435001, -135.044592, -130.708234, -126.421697, -122.180748, -117.981155, -113.818684, -109.689105, -105.588185, -101.511691, -97.4553918, -93.4150546, -89.3864473, -85.3653376, -81.3474933, -77.3286822, -73.304672, -68.5025042, -63.3184788, -58.1581353, -53.0421882, -47.9913518, -43.0263406, -38.167869, -33.4366514, -28.8534022, -24.4388359, -20.2136667, -16.1986091, -12.4143776, -8.88168653, -5.6212503, -2.65378332, 0, -4.6851301, -8.49104464, -11.508031, -13.8263766, -15.5363688, -16.7282951, -17.4924427, -17.9190991, -18.0985517, -18.1210879, -18.076995, -18.0565605, -18.1500718, -18.4478162, -19.0400811, -20.0171539, -21.0159132, -22.0618737, -24.2708903, -27.6360915, -32.1506056, -37.8075609, -44.6000857, -52.5213086, -61.5643577, -71.7223615, -82.9884483, -95.3557464, -108.817384, -122.160657, -137.026665, -153.770286, -172.7464, -195.080083, -220.741115, -249.493888, -281.462219, -316.769929, -355.540834, -397.898755, -440.804322, -488.507249, -541.750241, -601.276, -667.827228, -745.944024, -831.622347, -924.655551, -1024.83699, -1131.96002, -1242.75495, -1363.8219, -1498.55979, -1648.97775, -1814.24728, -1994.81394, -2191.12327, -2404.04634, -2633.50645, -2878.96268, -3138.69774, -3416.33283, -3716.33317, -4040.39019, -4385.49541, -4750.79767, -5134.03166, -5533.97959, -5954.60899, -6396.95426, -6886.37789, -7395.84345, -7917.96412, -8445.35306, -8970.62345, -9522.32423, -10048.1488, -10531.7265, -10956.6865, -11306.658, -11476.5552, -11597.8658, -11692.2396, -11854.8438, -12078.2278, -12324.3088, -12587.1678, -12860.8859, -13139.5439, -13417.223, -13574.8647, -13959.4609, -14381.8909, -14653.0339, -14583.7688, -13984.9751, -12667.5317, -11094.8697, -9135.22986}, 
{-429.7245, -427.890143, -425.931247, -423.850019, -421.648662, -419.329381, -416.894382, -414.34587, -411.68605, -408.917126, -406.041304, -403.060789, -399.977785, -396.794498, -393.513132, -390.135894, -386.664986, -383.102616, -379.450987, -375.712305, -371.888774, -367.9826, -363.995988, -359.931142, -355.790268, -351.57557, -347.289254, -342.933525, -338.510587, -334.022646, -329.471906, -324.860573, -320.190851, -315.464947, -310.670685, -305.817762, -300.916049, -295.968217, -290.976937, -285.944882, -280.874722, -275.769129, -270.630776, -265.462333, -260.266472, -255.045864, -249.803182, -244.541098, -239.262281, -233.969405, -228.665141, -223.284398, -217.899079, -212.513325, -207.131275, -201.757072, -196.394854, -191.048764, -185.72294, -180.421525, -175.148658, -169.90848, -164.705132, -159.542754, -154.425486, -149.35747, -144.342845, -139.502964, -134.956363, -130.465444, -126.026751, -121.636824, -117.292206, -112.98944, -108.725067, -104.495631, -100.297672, -96.1277343, -91.9823591, -87.8580888, -83.7514658, -79.6590323, -75.5773305, -71.5029027, -66.7290988, -61.6188478, -56.5435309, -51.5225102, -46.5751479, -41.7208063, -36.9788475, -32.3686337, -27.9095272, -23.6208901, -19.5220847, -15.6324731, -11.9714175, -8.55828025, -5.41242344, -2.55320928, 0, -4.5571159, -8.23440655, -11.1245411, -13.3201889, -14.9140189, -15.9987003, -16.6669024, -17.0112944, -17.1245453, -17.0993245, -17.0283009, -17.0041439, -17.1195227, -17.4671063, -18.139564, -19.2295649, -20.1824699, -20.9984756, -23.234401, -26.8442061, -31.7818514, -38.001297, -45.4565032, -54.1014302, -63.8900384, -74.776288, -86.7141392, -99.6575523, -113.560488, -126.875818, -141.608989, -158.164289, -176.946003, -199.319681, -225.208978, -254.281213, -286.652292, -322.438123, -361.754614, -404.717672, -448.406117, -496.897275, -550.901266, -611.128212, -678.288232, -756.815916, -842.765798, -935.91688, -1036.04817, -1142.93866, -1253.38536, -1373.79395, -1507.03877, -1655.00985, -1817.58764, -1995.55486, -2189.6942, -2401.42241, -2630.74407, -2876.97214, -3139.47037, -3419.88834, -3719.7918, -4040.04454, -4380.32087, -4740.8411, -5121.83848, -5523.53667, -5946.11182, -6389.76696, -6878.79265, -7385.59859, -7902.04983, -8420.01139, -8931.3483, -9444.85778, -9931.23964, -10378.1258, -10773.1483, -11103.9392, -11286.4247, -11427.7463, -11546.2667, -11723.5863, -11953.0365, -12201.5999, -12463.9254, -12734.6619, -13008.4582, -13279.9633, -13418.9009, -13776.9953, -14179.2077, -14450.4993, -14415.8312, -13900.1647, -12728.4608, -11271.5178, -9314.26466}, 
{-439.810344, -438.073492, -436.186165, -434.151545, -431.972814, -429.653154, -427.195746, -424.603771, -421.880413, -419.028851, -416.052269, -412.953847, -409.736767, -406.404212, -402.959362, -399.4054, -395.745508, -391.982866, -388.120656, -384.162061, -380.110263, -375.968441, -371.73978, -367.427459, -363.034661, -358.564568, -354.020361, -349.405222, -344.722333, -339.974875, -335.16603, -330.29898, -325.376907, -320.402991, -315.40528, -310.373998, -305.299076, -300.182887, -295.027806, -289.836208, -284.610467, -279.35296, -274.066059, -268.752142, -263.413581, -258.052752, -252.67203, -247.27379, -241.860406, -236.434253, -230.997706, -225.469778, -219.938012, -214.406592, -208.879698, -203.361513, -197.856218, -192.367997, -186.90103, -181.459499, -176.047587, -170.669476, -165.329347, -160.031382, -154.779764, -149.578674, -144.432295, -139.450989, -134.742572, -130.092233, -125.497271, -120.954985, -116.462673, -112.017634, -107.617167, -103.25857, -98.9391416, -94.6561811, -90.4069867, -86.1888572, -81.9990912, -77.8349872, -73.693844, -69.5729601, -64.8410374, -59.8228946, -54.8509058, -49.9427686, -45.116181, -40.3888408, -35.7784458, -31.3026937, -26.9792825, -22.8259099, -18.8602738, -15.1000719, -11.5630021, -8.2667622, -5.22905, -2.46756333, 0, -3.80076397, -6.83052176, -9.17649507, -10.9259056, -12.1659751, -12.9839252, -13.4669777, -13.7023543, -13.7772766, -13.7789664, -13.7946454, -13.9115353, -14.2168579, -14.7978347, -15.7416876, -17.1356382, -18.4546805, -19.6985754, -22.3393449, -26.3330193, -31.6356286, -38.2032028, -45.991772, -54.9573664, -65.0560159, -76.2437506, -88.4766007, -101.710596, -115.901767, -129.557074, -144.612942, -161.460122, -180.489365, -202.771263, -228.356648, -257.115057, -289.23329, -324.898146, -364.296422, -407.614917, -453.058893, -503.399763, -559.212016, -621.070144, -689.548637, -767.86448, -853.289046, -945.736203, -1045.11982, -1151.35376, -1261.70217, -1381.96719, -1514.60596, -1661.38515, -1822.66168, -1999.42544, -2192.66629, -2404.48084, -2634.50071, -2881.15018, -3143.08979, -3422.50638, -3720.90603, -4039.31346, -4377.93774, -4737.36235, -5119.28343, -5524.57292, -5950.02295, -6394.73343, -6877.38361, -7374.48184, -7878.35055, -8381.31219, -8875.6892, -9366.66628, -9830.48804, -10256.2614, -10633.0931, -10950.0901, -11129.8936, -11272.3866, -11395.1614, -11578.5019, -11815.1849, -12071.8656, -12342.6267, -12621.5512, -12902.7217, -13180.2212, -13384.0295, -13740.775, -14108.1492, -14343.844, -14305.551, -13850.962, -12837.7686, -11461.6957, -9617.45018}, 
{-450.224459, -448.593411, -446.784789, -444.80279, -442.65161, -440.335445, -437.858492, -435.224947, -432.439006, -429.504866, -426.426723, -423.208773, -419.855213, -416.370239, -412.758047, -409.022834, -405.168796, -401.20013, -397.121031, -392.935696, -388.648321, -384.263104, -379.784239, -375.215923, -370.562354, -365.827726, -361.016237, -356.132082, -351.179459, -346.162563, -341.08559, -335.952738, -330.768202, -325.536179, -320.325916, -315.107715, -309.851097, -304.558148, -299.230953, -293.871598, -288.482167, -283.064746, -277.62142, -272.154275, -266.665396, -261.156869, -255.630777, -250.089208, -244.534246, -238.967977, -233.392485, -227.710808, -222.026226, -216.342973, -210.66528, -204.997381, -199.343507, -193.707892, -188.094767, -182.508366, -176.95292, -171.432662, -165.951825, -160.514641, -155.125343, -149.788163, -144.507334, -139.382743, -134.51198, -129.701998, -124.950828, -120.256501, -115.617049, -111.030504, -106.494897, -102.008261, -97.5686256, -93.1740236, -88.8224864, -84.5120454, -80.2407324, -76.006579, -71.8076166, -67.641877, -62.9534577, -58.0290084, -53.1618371, -48.3679515, -43.6633595, -39.0640688, -34.5860873, -30.2454229, -26.0580832, -22.0400763, -18.2074099, -14.5760918, -11.1621298, -7.98153186, -5.05030567, -2.38445911, 0, -3.01083785, -5.36784663, -7.15175978, -8.44331068, -9.32323276, -9.87225943, -10.1711241, -10.3005601, -10.341301, -10.3740801, -10.4796308, -10.7386866, -11.2319808, -12.0402469, -13.2442182, -14.9246282, -16.6590033, -18.4561902, -21.5351286, -25.8687218, -31.4298732, -38.1914862, -46.126464, -55.2077099, -65.4081275, -76.7006198, -89.0580904, -102.453443, -116.85958, -130.974651, -146.513724, -163.832128, -183.285191, -205.509153, -230.718887, -259.05436, -290.786629, -326.18675, -365.52578, -409.074775, -456.313137, -508.544517, -566.19486, -629.690112, -699.456219, -777.262158, -861.855085, -953.325188, -1051.76265, -1157.25767, -1267.66912, -1388.04565, -1520.49345, -1666.64898, -1827.1896, -2003.22329, -2195.85801, -2407.63825, -2637.909, -2884.44815, -3145.23384, -3423.15568, -3720.09153, -4037.31817, -4375.09385, -4734.14435, -5117.2153, -5525.55614, -5953.01016, -6397.60999, -6871.4745, -7356.38296, -7845.49201, -8331.95826, -8808.93834, -9283.4628, -9731.34586, -10142.2757, -10505.9403, -10812.0281, -10985.2545, -11123.5953, -11244.5838, -11430.0781, -11672.1498, -11936.0688, -12215.2468, -12503.0954, -12793.0266, -13078.452, -13379.137, -13744.9031, -14073.222, -14261.5651, -14207.4041, -13808.2103, -12961.4554, -11680.9892, -10002.4877}, 
{-460.767447, -459.2137, -457.458773, -455.507726, -453.365621, -451.037519, -448.528481, -445.843569, -442.987844, -439.966367, -436.7842, -433.446402, -429.958037, -426.324164, -422.549846, -418.640143, -414.600117, -410.434829, -406.14934, -401.748711, -397.238004, -392.622279, -387.906599, -383.096024, -378.195616, -373.210435, -368.145543, -363.006002, -357.796872, -352.523214, -347.190091, -341.802563, -336.365691, -330.884536, -325.461471, -320.050855, -314.605876, -309.128439, -303.620447, -298.083803, -292.52041, -286.932172, -281.320992, -275.688773, -270.037419, -264.368833, -258.684918, -252.987578, -247.278716, -241.560235, -235.834038, -229.992448, -224.149324, -218.308946, -212.475592, -206.653541, -200.847073, -195.060466, -189.297998, -183.563949, -177.862598, -172.198223, -166.575103, -160.997517, -155.469744, -149.996063, -144.580752, -139.313863, -134.283384, -129.316699, -124.412526, -119.569583, -114.786591, -110.062266, -105.395328, -100.784496, -96.2284876, -91.7260221, -87.2758179, -82.8765937, -78.5270682, -74.22596, -69.9719878, -65.7638702, -61.117002, -56.2819298, -51.5150096, -46.8306446, -42.2432384, -37.7671945, -33.4169162, -29.2068069, -25.1512702, -21.2647095, -17.5615281, -14.0561296, -10.7629174, -7.69629485, -4.87066549, -2.30043273, 0, -2.35166923, -4.14904674, -5.46693713, -6.380145, -6.96347495, -7.29173159, -7.43971951, -7.48224333, -7.49410763, -7.55011703, -7.72507613, -8.09378952, -8.73106181, -9.71169761, -11.1105015, -13.0022781, -15.1199539, -17.4894939, -20.9589982, -25.5300119, -31.2040799, -37.9827474, -45.8675593, -54.8600606, -64.9617964, -76.1743116, -88.4991513, -101.93786, -116.491984, -131.154468, -147.305276, -165.248534, -185.288365, -207.578327, -232.49783, -260.466431, -291.833427, -326.948111, -366.159779, -409.817726, -458.54661, -512.336556, -571.482983, -636.281309, -707.026955, -784.099246, -867.688718, -958.069812, -1055.51697, -1160.30464, -1270.9127, -1391.60349, -1524.25516, -1670.38378, -1830.76618, -2006.5111, -2198.72729, -2409.99064, -2639.60928, -2885.29084, -3144.81291, -3421.39216, -3717.24574, -4033.79326, -4371.10295, -4729.86334, -5113.41509, -5523.1343, -5950.67272, -6393.18278, -6856.45447, -7327.67373, -7801.00304, -8270.60485, -8730.64165, -9193.13204, -9629.9183, -10030.6989, -10385.1722, -10683.0367, -10848.4597, -10980.3581, -11096.5152, -11281.046, -11525.4774, -11793.6977, -12078.6156, -12373.1397, -12670.1788, -12962.6415, -13376.7696, -13757.8431, -14046.3996, -14182.9767, -14108.1119, -13762.3429, -13086.2073, -11930.4381, -10459.9179}, 
{-471.178999, -469.632186, -467.86959, -465.896762, -463.719253, -461.342612, -458.772393, -456.014144, -453.073417, -449.955763, -446.666732, -443.211875, -439.596744, -435.826888, -431.90786, -427.845208, -423.644485, -419.311242, -414.851028, -410.269395, -405.571893, -400.764074, -395.851489, -390.839687, -385.73422, -380.540639, -375.264494, -369.911336, -364.486717, -358.996186, -353.445295, -347.839595, -342.184636, -336.485969, -330.859916, -325.254313, -319.615663, -313.945923, -308.247048, -302.520994, -296.769718, -290.995176, -285.199324, -279.384117, -273.551512, -267.703465, -261.841933, -255.96887, -250.086233, -244.195979, -238.300062, -232.291881, -226.284303, -220.281638, -214.288196, -208.308286, -202.346217, -196.406299, -190.492842, -184.610154, -178.762546, -172.954326, -167.189804, -161.47329, -155.809093, -150.201522, -144.654886, -139.25045, -134.067628, -128.952019, -123.902943, -118.919722, -114.001675, -109.148124, -104.35839, -99.631794, -94.967656, -90.3652971, -85.8240383, -81.3432003, -76.9221039, -72.5600701, -68.2564195, -64.0104731, -59.3986635, -54.6406505, -49.9611766, -45.3732863, -40.8900239, -36.5244338, -32.2895603, -28.1984478, -24.2641407, -20.4996833, -16.9181199, -13.5324951, -10.355853, -7.40123808, -4.68169471, -2.21026722, 0, -2.06848166, -3.6256955, -4.74345145, -5.49355946, -5.94782946, -6.1780714, -6.25609523, -6.25371089, -6.24272831, -6.29495745, -6.48220823, -6.87629061, -7.54901454, -8.57218994, -10.0176268, -11.957135, -14.3092961, -17.1240305, -20.8197358, -25.4353874, -31.0099606, -37.5824307, -45.191773, -53.8769627, -63.6769753, -74.6307859, -86.77737, -100.155703, -114.804759, -130.084511, -146.961905, -165.679619, -186.48033, -209.089691, -234.008891, -261.87699, -293.091772, -328.051019, -367.152516, -410.794044, -460.269589, -514.807976, -574.631646, -639.963038, -711.024592, -787.237115, -869.825088, -959.21136, -1055.81878, -1160.07019, -1270.95257, -1392.07961, -1525.33214, -1672.14592, -1833.04791, -2008.97311, -2200.8565, -2410.57261, -2637.90337, -2881.60578, -3140.43853, -3416.77891, -3712.5013, -4028.77267, -4365.56128, -4723.38564, -5105.66639, -5513.67442, -5938.03942, -6375.41034, -6826.89815, -7284.00333, -7741.8303, -8195.48347, -8640.06726, -9093.08939, -9521.65014, -9915.25309, -10263.4018, -10555.5999, -10715.1799, -10841.9304, -10953.7142, -11134.9062, -11377.0412, -11643.9406, -11928.5249, -12223.7146, -12522.4302, -12817.5922, -13346.7747, -13745.5755, -13997.2575, -14085.0836, -13992.3167, -13702.2195, -13198.055, -12216.6853, -10986.229}, 
{-480.610388, -478.909371, -477.00077, -474.889656, -472.581104, -470.080186, -467.391976, -464.521546, -461.473971, -458.254322, -454.867674, -451.319099, -447.613671, -443.756462, -439.752547, -435.606997, -431.324887, -426.911289, -422.371277, -417.709924, -412.932302, -408.043486, -403.048548, -397.952561, -392.760599, -387.477735, -382.109042, -376.659593, -371.134461, -365.53872, -359.877442, -354.155701, -348.378571, -342.551123, -336.753662, -330.952063, -325.111299, -319.233999, -313.322797, -307.380324, -301.409212, -295.412093, -289.391598, -283.350359, -277.291008, -271.216177, -265.128498, -259.030602, -252.925122, -246.814688, -240.701933, -234.511159, -228.325026, -222.147863, -215.984, -209.837766, -203.713493, -197.615509, -191.548145, -185.515729, -179.522593, -173.573066, -167.671477, -161.822157, -156.029435, -150.297641, -144.631105, -139.106503, -133.798773, -128.563036, -123.398934, -118.306107, -113.284193, -108.332834, -103.451669, -98.6403376, -93.8984804, -89.2257372, -84.621748, -80.0861526, -75.6185911, -71.2187033, -66.8861293, -62.620509, -58.0233913, -53.3018351, -48.6676837, -44.1334961, -39.711831, -35.4152472, -31.2563034, -27.2475585, -23.4015712, -19.7309003, -16.2481045, -12.9657427, -9.8963736, -7.05255601, -4.44684869, -2.09181043, 0, -3.18795368, -5.68789001, -7.58141353, -8.95012879, -9.87564034, -10.4395527, -10.7234705, -10.8089982, -10.7777404, -10.7113016, -10.6912865, -10.7992994, -11.116945, -11.7258278, -12.7075524, -14.1437234, -16.1212822, -18.7225958, -22.0171971, -26.0878341, -31.0172552, -36.8882084, -43.783442, -51.7857041, -60.9777431, -71.4423069, -83.2621439, -96.5200022, -111.29863, -127.386339, -145.268274, -165.115514, -187.099139, -210.791024, -236.66185, -265.342086, -297.18257, -332.534144, -371.747647, -415.173919, -463.272507, -516.25246, -574.443348, -638.174741, -707.776212, -783.325612, -865.467161, -954.593359, -1051.09671, -1155.36971, -1266.27673, -1387.60612, -1522.07521, -1671.24038, -1834.2877, -2011.46744, -2203.02991, -2407.82551, -2627.82255, -2866.51649, -3127.8352, -3408.94799, -3708.26273, -4025.18207, -4360.79426, -4715.41382, -5091.77809, -5490.82974, -5904.62776, -6330.25625, -6769.51168, -7214.04081, -7659.29655, -8100.73181, -8533.79949, -8976.2167, -9395.60609, -9781.85454, -10124.8489, -10414.476, -10578.369, -10710.171, -10826.449, -11002.5906, -11231.8798, -11483.0919, -11750.7362, -12029.3221, -12313.359, -12597.3562, -13232.9237, -13650.0955, -13872.2052, -13922.5867, -13824.5736, -13601.4998, -13276.699, -12600.5031, -11635.3644}, 
{-490.273963, -488.338142, -486.210514, -483.8954, -481.397118, -478.719989, -475.868331, -472.846465, -469.65871, -466.309385, -462.802811, -459.143306, -455.33519, -451.382783, -447.290404, -443.062373, -438.70301, -434.216633, -429.607563, -424.880119, -420.03862, -415.087387, -410.030738, -404.872993, -399.618473, -394.271495, -388.836381, -383.317449, -377.719018, -372.04541, -366.300942, -360.489935, -354.616709, -348.685582, -342.72907, -336.736801, -330.698066, -324.616269, -318.494814, -312.337106, -306.14655, -299.92655, -293.680511, -287.411837, -281.123933, -274.820203, -268.504052, -262.178885, -255.848105, -249.515118, -243.183328, -236.815498, -230.456555, -224.110785, -217.782473, -211.475905, -205.195367, -198.945144, -192.729522, -186.552786, -180.419222, -174.333116, -168.298752, -162.320418, -156.402398, -150.548977, -144.764442, -139.11864, -133.679842, -128.31712, -123.030508, -117.820044, -112.685761, -107.627696, -102.645884, -97.7403607, -92.9111616, -88.1583222, -83.481878, -78.8818646, -74.3583174, -69.911272, -65.5407639, -61.2468287, -56.6628556, -51.9799152, -47.3934848, -42.9154984, -38.5578898, -34.332593, -30.2515418, -26.3266702, -22.5699121, -18.9932013, -15.6084718, -12.4276575, -9.46269223, -6.72550995, -4.22804455, -1.98222994, 0, -3.99952687, -7.18694698, -9.65010173, -11.4768325, -12.7549807, -13.5723877, -14.016895, -14.1763439, -14.1385757, -13.991432, -13.8227541, -13.7203834, -13.7721613, -14.0659292, -14.6895284, -15.7308005, -17.4675508, -20.0147584, -23.0305842, -26.6435762, -30.9822822, -36.1752504, -42.3510286, -49.6381648, -58.1652071, -68.0607034, -79.4532017, -92.4712499, -107.243396, -123.965014, -142.673323, -163.476823, -186.484015, -211.199294, -238.033213, -267.557419, -300.061643, -335.835618, -375.169074, -418.351745, -464.861734, -516.047417, -572.357322, -634.239979, -702.143914, -776.760984, -858.235557, -946.955331, -1043.308, -1147.68126, -1258.98693, -1380.89244, -1516.65406, -1667.85282, -1832.64944, -2010.7403, -2201.82177, -2402.26868, -2615.85385, -2849.97362, -3112.84001, -3397.2898, -3698.97663, -4015.81808, -4349.56777, -4700.21853, -5069.47054, -5457.75927, -5859.25979, -6271.6884, -6698.56256, -7130.90492, -7564.42369, -7994.8271, -8417.82335, -8851.27575, -9263.19863, -9643.76144, -9983.1336, -10271.4846, -10440.4974, -10577.8189, -10698.6838, -10869.0007, -11083.9462, -11317.5833, -11566.1444, -11825.8621, -12092.9688, -12363.697, -13070.3247, -13497.4126, -13698.7968, -13728.3131, -13639.7974, -13487.0856, -13324.0135, -12953.0632, -12260.0165}, 
{-500.59357, -498.318345, -495.874497, -493.265366, -490.494297, -487.56463, -484.479709, -481.242875, -477.857471, -474.32684, -470.654323, -466.843264, -462.897004, -458.818885, -454.612251, -450.280443, -445.826804, -441.254675, -436.567401, -431.768322, -426.860781, -421.848121, -416.733683, -411.520811, -406.212846, -400.813131, -395.325009, -389.75182, -384.096909, -378.363617, -372.555286, -366.67526, -360.726879, -354.713487, -348.613438, -342.443092, -336.219114, -329.945656, -323.626872, -317.266915, -310.869938, -304.440095, -297.981539, -291.498423, -284.994901, -278.475125, -271.943249, -265.403427, -258.859811, -252.316555, -245.777812, -239.247871, -232.730748, -226.230591, -219.751552, -213.297781, -206.873428, -200.482643, -194.129577, -187.81838, -181.553203, -175.338195, -169.177508, -163.075291, -157.035695, -151.06287, -145.160966, -139.388539, -133.802837, -128.295805, -122.868066, -117.520245, -112.252964, -107.066848, -101.962518, -96.9406002, -92.0017167, -87.1464913, -82.3755474, -77.6895087, -73.0889985, -68.5746405, -64.147058, -59.8068747, -55.2398599, -50.6106868, -46.0878126, -41.6820786, -37.4043258, -33.2653954, -29.2761285, -25.4473663, -21.7899498, -18.3147201, -15.0325185, -11.954186, -9.0905638, -6.45249296, -4.05081463, -1.89636993, 0, -3.84176715, -6.90754351, -9.28130531, -11.0470288, -12.2886901, -13.0902655, -13.5357313, -13.7090635, -13.6942386, -13.5752326, -13.4360217, -13.3605823, -13.4328905, -13.7369226, -14.3566547, -15.3760632, -17.2969116, -20.2628603, -23.4128898, -26.9205022, -30.9591996, -35.7024841, -41.3238579, -47.9968231, -55.8948818, -65.1915361, -76.0602883, -88.6746403, -103.208094, -120.177189, -139.28661, -160.606949, -184.208795, -209.642456, -237.238664, -267.466896, -300.553829, -336.726139, -376.210503, -419.233599, -464.441225, -514.122072, -568.812118, -629.04734, -695.363718, -768.518626, -848.771298, -936.602361, -1032.49245, -1136.92218, -1249.33763, -1372.51845, -1509.44895, -1661.50101, -1826.75453, -2004.76743, -2195.09765, -2393.38586, -2603.99754, -2835.57154, -3097.79475, -3381.95077, -3682.97236, -3998.19928, -4329.04981, -4675.07011, -5036.6827, -5413.66092, -5802.56492, -6201.77241, -6616.85758, -7037.89452, -7460.76986, -7881.37024, -8295.58228, -8721.32969, -9126.95275, -9502.82883, -9839.33528, -10126.8495, -10300.548, -10442.8094, -10567.6687, -10732.5645, -10934.3511, -11151.7974, -11382.662, -11624.7032, -11875.6794, -12133.3489, -12848.1447, -13268.7918, -13463.0702, -13498.7597, -13443.6401, -13365.4914, -13332.0932, -13226.2905, -12801.0824}, 
{-512.726318, -509.865085, -506.873401, -503.75324, -500.506572, -497.13537, -493.641603, -490.027244, -486.294264, -482.444635, -478.480328, -474.403314, -470.215565, -465.919053, -461.515748, -457.007622, -452.396647, -447.684794, -442.874034, -437.966339, -432.963681, -427.86803, -422.681359, -417.405638, -412.042839, -406.594934, -401.063894, -395.451689, -389.760293, -383.991676, -378.14781, -372.230665, -366.242214, -360.184428, -353.975666, -347.661464, -341.288364, -334.86105, -328.384207, -321.86252, -315.300674, -308.703354, -302.075244, -295.421029, -288.745394, -282.053025, -275.348605, -268.63682, -261.922354, -255.209892, -248.504119, -241.845934, -235.203023, -228.579286, -221.978621, -215.40493, -208.862111, -202.354064, -195.88469, -189.457887, -183.077557, -176.747598, -170.47191, -164.254393, -158.098947, -152.009472, -145.989867, -140.079819, -134.318157, -128.634971, -123.031842, -117.510348, -112.07207, -106.718588, -101.451482, -96.2723303, -91.1827141, -86.1842129, -81.2784065, -76.4668746, -71.7511972, -67.132954, -62.6137249, -58.1950897, -53.6614769, -49.1276095, -44.7108204, -40.4197364, -36.2629843, -32.2491909, -28.3869829, -24.6849872, -21.1518306, -17.7961399, -14.6265418, -11.6516632, -8.8801309, -6.32057161, -3.98161216, -1.87187936, 0, -1.08831546, -1.86775062, -2.3914323, -2.71248732, -2.8840425, -2.95922467, -2.99116065, -3.03297726, -3.13780133, -3.35875967, -3.74897911, -4.36158647, -5.24970857, -6.46647223, -8.06500428, -10.0984315, -13.3865506, -18.0920125, -22.5338101, -26.9293566, -31.4960652, -36.4513492, -42.0126216, -48.3972958, -55.8227848, -64.506502, -74.6658604, -86.5182734, -100.281154, -116.57284, -135.062814, -155.848213, -179.026172, -204.459969, -232.36367, -263.013701, -296.577356, -333.221929, -373.114711, -416.422998, -461.573209, -511.003342, -565.221295, -624.734968, -690.052258, -760.231258, -837.592126, -923.005211, -1017.34087, -1121.46944, -1236.73605, -1362.95602, -1500.69153, -1650.49738, -1812.91311, -1988.48515, -2177.75986, -2380.2958, -2597.85174, -2833.26407, -3090.35298, -3366.03657, -3659.5949, -3970.12714, -4296.42585, -4637.42435, -4992.16991, -5359.62538, -5738.33571, -6127.08225, -6532.80294, -6944.86766, -7359.23431, -7771.86079, -8178.70501, -8596.8193, -8995.79353, -9366.312, -9699.059, -9984.71885, -10161.5912, -10307.6681, -10436.0843, -10598.8247, -10793.6459, -11002.1172, -11222.8047, -11454.2746, -11695.093, -11943.8259, -12537.5743, -12917.5502, -13131.8715, -13228.6563, -13256.0227, -13262.0888, -13294.9727, -13342.3676, -13179.314}, 
{-524.778692, -521.433797, -517.9869, -514.438961, -510.790944, -507.043811, -503.198524, -499.256046, -495.21734, -491.083366, -486.855088, -482.533469, -478.11947, -473.614054, -469.018184, -464.332821, -459.558928, -454.697468, -449.749402, -444.715694, -439.597306, -434.395199, -429.110337, -423.743682, -418.296196, -412.768841, -407.16258, -401.478376, -395.71719, -389.879985, -383.967724, -377.981369, -371.921881, -365.790224, -359.459675, -352.997724, -346.473396, -339.891799, -333.258037, -326.577217, -319.854445, -313.094827, -306.303468, -299.485476, -292.645956, -285.790013, -278.922755, -272.049287, -265.174715, -258.304145, -251.442683, -244.669478, -237.913987, -231.179713, -224.470156, -217.788816, -211.139196, -204.524795, -197.949115, -191.415656, -184.92792, -178.489407, -172.103619, -165.774056, -159.504219, -153.29761, -147.157728, -141.09841, -135.133243, -129.244414, -123.434753, -117.707092, -112.064261, -106.509092, -101.044416, -95.6730625, -90.3978639, -85.2216508, -80.1472542, -75.177505, -70.3152342, -65.5632728, -60.9244518, -56.4016022, -51.9002247, -47.4738625, -43.1772289, -39.0163135, -34.9971057, -31.125595, -27.4077711, -23.8496235, -20.4571416, -17.2363151, -14.1931335, -11.3335862, -8.66366293, -6.18935313, -3.91664635, -1.85153213, 0, 1.87278947, 3.564712, 5.05434748, 6.32027579, 7.34107684, 8.0953305, 8.56161667, 8.71851524, 8.5446061, 8.01846914, 7.11868425, 5.82383133, 4.11249025, 1.96324092, -0.645336777, -3.73466295, -8.38798477, -14.7784119, -20.5256128, -25.878542, -31.0861539, -36.397403, -42.0612437, -48.3266307, -55.4425182, -63.6578609, -73.2216132, -84.3827296, -97.3901645, -112.897664, -130.600967, -150.627591, -173.105054, -198.103739, -225.779731, -256.304347, -289.822245, -326.478081, -366.416512, -409.782196, -455.161457, -504.731559, -558.942049, -618.242474, -683.082382, -750.796933, -825.72866, -909.105706, -1002.15622, -1106.10833, -1224.11417, -1353.12639, -1491.23171, -1638.16909, -1797.05087, -1969.47483, -2157.03871, -2363.6357, -2588.04648, -2826.54759, -3076.0697, -3340.34153, -3623.96379, -3928.22628, -4248.80863, -4583.96562, -4931.29118, -5288.86874, -5657.20495, -6035.43576, -6431.50955, -6834.39406, -7240.17479, -7644.93723, -8044.76685, -8457.49205, -8852.01969, -9218.99952, -9549.08131, -9832.91481, -10009.6907, -10155.8239, -10284.018, -10443.7261, -10633.1053, -10835.0002, -11048.3336, -11272.0281, -11505.0063, -11746.1909, -12184.5927, -12513.2763, -12751.8902, -12920.0828, -13037.5025, -13123.7977, -13198.6168, -13347.2147, -13399.2748}, 
{-535.733965, -532.227073, -528.620091, -524.913877, -521.109293, -517.207197, -513.20845, -509.113912, -504.924442, -500.640902, -496.26415, -491.795048, -487.234454, -482.583229, -477.842232, -473.012325, -468.094366, -463.089216, -457.997735, -452.820783, -447.559219, -442.213904, -436.785698, -431.275461, -425.684052, -420.012332, -414.261161, -408.431399, -402.523906, -396.539541, -390.479165, -384.343637, -378.133818, -371.850568, -365.353742, -358.717667, -352.018367, -345.261278, -338.451837, -331.595478, -324.697638, -317.763752, -310.799257, -303.809589, -296.800183, -289.776475, -282.743901, -275.707897, -268.673899, -261.647343, -254.633665, -247.758107, -240.903701, -234.073286, -227.2697, -220.495784, -213.754374, -207.04831, -200.380431, -193.753575, -187.170582, -180.634288, -174.147535, -167.713159, -161.334001, -155.012898, -148.752689, -142.530911, -136.325707, -130.1921, -124.13457, -118.157595, -112.265655, -106.463228, -100.754794, -95.1448311, -89.6378186, -84.2382354, -78.9505606, -73.7792732, -68.7288522, -63.8037764, -59.008525, -54.3475769, -49.8655241, -45.5459455, -41.3719375, -37.3466775, -33.4733426, -29.75511, -26.1951572, -22.7966611, -19.5627992, -16.4967486, -13.6016866, -10.8807904, -8.33723729, -5.97420444, -3.79486911, -1.80240856, 0, 3.87749146, 7.26375453, 10.1566859, 12.5541822, 14.45414, 15.8544561, 16.7530271, 17.1477497, 17.0365204, 16.417236, 15.2877931, 13.6460884, 11.4900185, 8.81748002, 5.6263697, 1.91458417, -3.50902254, -10.816417, -17.3202014, -23.277274, -28.9445329, -34.5788763, -40.4372024, -46.7764094, -53.8533955, -61.9250589, -71.2482978, -82.0800104, -94.6770948, -109.725951, -126.896491, -146.316764, -168.114816, -192.180591, -218.76119, -248.167205, -280.598117, -316.253405, -355.332546, -398.035021, -443.598588, -493.477144, -548.05833, -607.729788, -672.87916, -739.912611, -814.194628, -897.108222, -990.036404, -1094.36218, -1214.20885, -1344.86991, -1482.9222, -1627.59083, -1783.50786, -1952.87774, -2137.90492, -2345.29709, -2571.732, -2808.97475, -3048.7154, -3300.24955, -3572.96356, -3870.03376, -4184.11395, -4512.68793, -4851.72193, -5198.3063, -5555.09593, -5921.59812, -6306.28927, -6698.32765, -7094.11585, -7490.05644, -7882.55199, -8294.08925, -8688.46558, -9055.56253, -9385.26163, -9667.44441, -9837.76926, -9976.48964, -10096.7256, -10250.1035, -10434.5013, -10631.7525, -10840.5375, -11059.5362, -11287.4289, -11522.8957, -11804.4034, -12078.3149, -12335.9477, -12568.6191, -12767.6466, -12924.3474, -13030.0391, -13237.9415, -13420.5099}, 
{-544.3831, -541.293812, -538.060051, -534.684121, -531.168328, -527.514975, -523.726367, -519.804809, -515.752604, -511.572057, -507.265473, -502.835155, -498.283409, -493.612539, -488.824849, -483.922644, -478.908228, -473.783905, -468.55198, -463.214757, -457.774541, -452.233636, -446.594346, -440.858977, -435.029831, -429.109215, -423.099431, -417.002785, -410.821582, -404.558124, -398.214717, -391.793666, -385.297274, -378.727846, -371.982706, -365.118856, -358.194562, -351.215535, -344.187485, -337.116124, -330.007163, -322.866314, -315.699288, -308.511795, -301.309548, -294.098258, -286.883635, -279.671391, -272.467237, -265.276884, -258.106044, -251.143579, -244.208079, -237.301285, -230.424937, -223.580776, -216.770544, -209.995981, -203.258828, -196.560826, -189.903716, -183.289239, -176.719135, -170.195146, -163.719012, -157.292474, -150.917274, -144.518968, -138.024945, -131.594067, -125.233012, -118.948462, -112.747094, -106.63559, -100.620627, -94.7088862, -88.9070466, -83.2217877, -77.6597891, -72.2277303, -66.9322908, -61.7801501, -56.7779877, -51.9324831, -47.4413058, -43.2116231, -39.1483069, -35.2518385, -31.5226995, -27.9613715, -24.568336, -21.3440744, -18.2890683, -15.4037992, -12.6887487, -10.1443982, -7.77122934, -5.56972355, -3.54036239, -1.68362736, 0, 3.55560619, 6.72639506, 9.5015781, 11.8703668, 13.8219727, 15.3456072, 16.4304818, 17.065808, 17.2407974, 16.9446614, 16.1666114, 14.8958591, 13.1216158, 10.8330931, 8.0195024, 4.67005527, -0.236153465, -6.85421469, -12.9101293, -18.6311393, -24.2444869, -29.977414, -36.0571629, -42.7109754, -50.1660938, -58.6497601, -68.3892164, -79.6117047, -92.5444671, -107.980806, -125.374348, -144.782517, -166.262736, -188.781886, -212.942663, -239.638573, -269.254204, -302.174143, -338.782976, -379.46529, -424.767335, -474.862833, -530.104743, -590.846022, -657.439627, -727.1115, -804.123369, -889.609944, -984.705938, -1090.54606, -1210.52618, -1340.75621, -1478.44413, -1622.94839, -1778.01806, -1945.43094, -2126.9648, -2328.39787, -2546.59831, -2774.0701, -3001.91403, -3241.07136, -3503.58057, -3793.39639, -4100.77606, -4422.26416, -4751.81978, -5085.31717, -5428.11094, -5780.19307, -6149.99227, -6527.76584, -6910.58536, -7295.52241, -7679.64856, -8096.86991, -8498.21488, -8871.54644, -9204.72753, -9485.62109, -9638.99594, -9757.87193, -9857.15236, -9997.84396, -10176.6008, -10370.867, -10578.2572, -10796.3859, -11022.8675, -11255.3166, -11410.7193, -11633.095, -11894.5961, -12167.3744, -12423.5823, -12635.3718, -12774.895, -13010.6002, -13193.8509}, 
{-553.228513, -550.651218, -547.875733, -544.906167, -541.746628, -538.401223, -534.874062, -531.169252, -527.290902, -523.243119, -519.030012, -514.655689, -510.124259, -505.439828, -500.606506, -495.628401, -490.509621, -485.254273, -479.866467, -474.350311, -468.709911, -462.949378, -457.072819, -451.084341, -444.988054, -438.788065, -432.488483, -426.093416, -419.606971, -413.033258, -406.376384, -399.640457, -392.829586, -385.947879, -378.95307, -371.873429, -364.737784, -357.551749, -350.320937, -343.050961, -335.747434, -328.415969, -321.062179, -313.691678, -306.310078, -298.922993, -291.536035, -284.154818, -276.784955, -269.432059, -262.101743, -255.005415, -247.938433, -240.901948, -233.897111, -226.925075, -219.986992, -213.084014, -206.217291, -199.387977, -192.597223, -185.846181, -179.136003, -172.467841, -165.842846, -159.262171, -152.726967, -146.124465, -139.34463, -132.623569, -125.969819, -119.391916, -112.898398, -106.497801, -100.198663, -94.0095202, -87.9389097, -81.9953685, -76.1874335, -70.5236416, -65.0125298, -59.6626351, -54.4824943, -49.4806444, -45.0102341, -40.900246, -36.9755208, -33.2334121, -29.6712736, -26.2864587, -23.0763213, -20.0382148, -17.169493, -14.4675094, -11.9296177, -9.5531715, -7.33552443, -5.27403012, -3.3660422, -1.60891428, 0, 3.53152951, 6.75334498, 9.64820481, 12.1988674, 14.3880911, 16.1986343, 17.6132554, 18.6147129, 19.185765, 19.3091702, 18.9676868, 18.1440733, 16.8210881, 14.9814895, 12.6080359, 9.68348572, 5.33995489, -0.561318562, -6.11350402, -11.5161236, -16.9686995, -22.6707538, -28.8218086, -35.6213862, -43.2690085, -51.9641979, -61.9064763, -73.295366, -86.330389, -101.800374, -119.099457, -138.25037, -159.275841, -180.54842, -202.915926, -227.666231, -255.317121, -286.386379, -321.391793, -360.851146, -406.406705, -457.109542, -513.257433, -575.148158, -643.079494, -715.158255, -794.420925, -881.713023, -977.88007, -1083.76758, -1201.77062, -1329.29129, -1464.47742, -1606.95654, -1759.37722, -1923.03165, -2099.21201, -2292.32833, -2499.84635, -2715.83047, -2932.0875, -3159.54562, -3410.85936, -3690.16278, -3987.15253, -4298.15225, -4616.6699, -4938.29914, -5268.95786, -5608.724, -5964.56986, -6328.6183, -6698.56073, -7072.08854, -7446.89314, -7865.44058, -8269.45395, -8645.43099, -8979.86944, -9259.26704, -9401.66221, -9507.65156, -9593.07254, -9724.03126, -9896.46714, -10085.3744, -10287.7397, -10500.5497, -10720.7912, -10945.4509, -11047.5684, -11236.9991, -11483.7817, -11757.9548, -12029.5568, -12268.6264, -12445.2021, -12681.7708, -12838.5435}, 
{-563.878466, -561.692931, -559.257475, -556.57813, -553.66093, -550.511907, -547.137094, -543.542523, -539.734229, -535.718243, -531.500598, -527.087328, -522.484465, -517.698042, -512.734092, -507.598647, -502.297741, -496.837407, -491.223677, -485.462584, -479.560161, -473.52244, -467.355456, -461.06524, -454.657825, -448.139245, -441.515532, -434.792719, -427.976839, -421.073924, -414.090008, -407.031123, -399.903303, -392.712579, -385.514819, -378.29009, -371.017861, -363.702549, -356.348568, -348.960334, -341.542264, -334.098772, -326.634275, -319.153188, -311.659926, -304.158906, -296.654544, -289.151254, -281.653452, -274.165555, -266.691977, -259.333644, -251.99637, -244.682478, -237.394292, -230.134137, -222.904334, -215.707209, -208.545084, -201.420283, -194.335129, -187.291947, -180.293059, -173.34079, -166.437462, -159.5854, -152.786926, -145.944787, -138.964117, -132.052614, -125.219057, -118.472223, -111.820893, -105.273845, -98.8398579, -92.5277102, -86.3461811, -80.3040492, -74.4100936, -68.6730929, -63.1018261, -57.7050719, -52.4916092, -47.4702169, -43.1173813, -39.1981899, -35.4721087, -31.9327384, -28.57368, -25.3885342, -22.3709018, -19.5143838, -16.8125808, -14.2590938, -11.8475235, -9.57147084, -7.42453656, -5.40032151, -3.49242652, -1.6944524, 0, 6.15265964, 11.6329807, 16.4473796, 20.6022725, 24.1040758, 26.9592057, 29.1740786, 30.7551107, 31.7087185, 32.041318, 31.7593258, 30.869158, 29.377231, 27.2899611, 24.6137646, 21.3550577, 16.5565974, 10.0871342, 4.13279039, -1.50651611, -7.03086748, -12.6403459, -18.5350335, -24.9150125, -31.980365, -39.9311732, -48.9675192, -59.2894852, -71.0971534, -84.7480446, -100.227075, -117.687094, -137.280954, -158.260714, -181.229621, -207.031129, -236.088328, -268.824303, -305.662144, -347.024936, -394.542129, -447.063295, -504.775496, -567.865792, -636.521244, -709.003006, -787.905521, -873.897325, -967.646953, -1069.82294, -1182.01435, -1302.84411, -1430.99298, -1566.36313, -1711.3505, -1867.2314, -2035.28212, -2219.71761, -2418.20763, -2625.21613, -2833.72819, -3052.7116, -3291.26718, -3552.80294, -3831.08069, -4124.29002, -4430.10223, -4746.57251, -5073.65647, -5410.23471, -5757.71462, -6112.06133, -6471.28083, -6833.3791, -7196.36213, -7598.79561, -7987.98588, -8351.79899, -8678.10099, -8954.75792, -9109.94839, -9231.01754, -9331.41175, -9465.13593, -9629.58205, -9805.24272, -9990.35489, -10183.1555, -10381.8815, -10584.7699, -10815.672, -11022.332, -11215.058, -11404.1581, -11599.9403, -11812.7127, -12052.7832, -12267.7852, -12443.9962}, 
{-575.353228, -573.358249, -571.085667, -568.54246, -565.735606, -562.672082, -559.358866, -555.802936, -552.011268, -547.990841, -543.748632, -539.291618, -534.626777, -529.761087, -524.701525, -519.455068, -514.028695, -508.429382, -502.664107, -496.739848, -490.663582, -484.442287, -478.082941, -471.59252, -464.978002, -458.246366, -451.404588, -444.459645, -437.418517, -430.288179, -423.07561, -415.787787, -408.431687, -401.014289, -393.64325, -386.27309, -378.857118, -371.399045, -363.90258, -356.371435, -348.809319, -341.219942, -333.607015, -325.974248, -318.325351, -310.664034, -302.994008, -295.318983, -287.64267, -279.968777, -272.301017, -264.661662, -257.035456, -249.425709, -241.835726, -234.268817, -226.728289, -219.217449, -211.739607, -204.298068, -196.896143, -189.537137, -182.224359, -174.961117, -167.750718, -160.596471, -153.501683, -146.37395, -139.123166, -131.950038, -124.864077, -117.874797, -110.99171, -104.224327, -97.582163, -91.0747285, -84.7115366, -78.5020995, -72.4559299, -66.58254, -60.8914424, -55.3921494, -50.0941734, -45.007027, -40.7731923, -37.0723832, -33.5767057, -30.2751313, -27.1566312, -24.2101768, -21.4247395, -18.7892905, -16.2928013, -13.9242432, -11.6725875, -9.52680564, -7.47586887, -5.50874858, -3.61441611, -1.7818428, 0, 8.10219764, 15.3219223, 21.6787685, 27.1923309, 31.8822038, 35.767982, 38.8692598, 41.2056319, 42.7966927, 43.6620368, 43.8212588, 43.2939531, 42.0997144, 40.2581371, 37.7888157, 34.7113449, 29.9310917, 23.3084754, 17.3833667, 11.9365971, 6.74899811, 1.6014013, -3.72536182, -9.45045971, -15.7930609, -22.9723337, -31.2074468, -40.7175685, -51.7218674, -64.398497, -79.0226794, -95.8258877, -115.039595, -136.666394, -161.052198, -188.603954, -219.621799, -254.405869, -293.256302, -336.473235, -385.549972, -439.230344, -497.58104, -560.668753, -628.560172, -698.892766, -774.769753, -856.865129, -945.852891, -1042.40703, -1148.08275, -1261.59582, -1381.48819, -1507.67477, -1642.87362, -1788.54425, -1946.14617, -2120.14662, -2308.31379, -2505.13473, -2704.65077, -2913.9053, -3138.61444, -3381.42914, -3639.80653, -3913.5878, -4203.90496, -4510.93383, -4830.11724, -5159.57525, -5497.44767, -5841.83182, -6190.84095, -6542.58831, -6895.18716, -7282.52372, -7657.99502, -8010.77107, -8330.02186, -8604.91741, -8773.64004, -8910.33923, -9026.03681, -9162.4416, -9318.42854, -9480.08636, -9646.92934, -9818.47175, -9994.22783, -10173.7119, -10525.8029, -10748.5169, -10888.8193, -10993.6754, -11110.0507, -11284.9103, -11565.2198, -11750.9308, -11937.0854}, 
{-587.760262, -584.400687, -580.921294, -577.322384, -573.604258, -569.767217, -565.811563, -561.737598, -557.545622, -553.235936, -548.808843, -544.264643, -539.603638, -534.826129, -529.932418, -524.922805, -519.797592, -514.557081, -509.201572, -503.731368, -498.146768, -492.448076, -486.635591, -480.709616, -474.670451, -468.518398, -462.253759, -455.876834, -449.387925, -442.787334, -436.07536, -429.252307, -422.318475, -415.274166, -407.878133, -400.256531, -392.538421, -384.731942, -376.845234, -368.886436, -360.86369, -352.785135, -344.658911, -336.493157, -328.296015, -320.075624, -311.840124, -303.597655, -295.356357, -287.12437, -278.909834, -271.032209, -263.181566, -255.359297, -247.566793, -239.805444, -232.076642, -224.381779, -216.722245, -209.099432, -201.514731, -193.969533, -186.465229, -179.00321, -171.584868, -164.211594, -156.884779, -149.423719, -141.65208, -133.946813, -126.321114, -118.788179, -111.361203, -104.053381, -96.8779109, -89.8479868, -82.9768049, -76.2775608, -69.7634505, -63.4476695, -57.3434136, -51.4638786, -45.8222602, -40.4317541, -36.1040812, -32.4363788, -29.0161865, -25.8307862, -22.8674602, -20.1134903, -17.5561587, -15.1827475, -12.9805386, -10.9368142, -9.03885623, -7.27394684, -5.62936804, -4.09240191, -2.65033049, -1.29043583, 0, -2.55751112, -4.18707911, -4.97580059, -5.01077219, -4.37909051, -3.16785217, -1.4641538, 0.644907995, 3.07223659, 5.73073537, 8.53330772, 11.392857, 14.2222866, 16.9345, 19.4424004, 21.6588913, 21.9723425, 20.0778668, 19.3367174, 19.335112, 19.6592681, 19.8954034, 19.6297354, 18.4484818, 15.9378603, 11.6840883, 5.27338347, -3.70803652, -15.6739541, -33.3808686, -54.0408493, -77.3648634, -103.063878, -128.363254, -154.216763, -182.241002, -212.892622, -246.628271, -283.9046, -325.178257, -374.347772, -427.380385, -484.059335, -544.167859, -607.489196, -664.2598, -726.196388, -795.468897, -874.247259, -964.701411, -1075.37571, -1194.2747, -1313.55537, -1430.49768, -1552.84096, -1683.6285, -1825.9036, -1989.17771, -2168.55591, -2352.08708, -2526.43635, -2707.56647, -2911.55087, -3144.17283, -3393.77949, -3656.72143, -3925.50918, -4195.49773, -4476.12253, -4768.85431, -5087.9322, -5420.09422, -5762.39128, -6111.87431, -6465.59424, -6858.11378, -7239.59412, -7597.70826, -7920.12917, -8194.52983, -8350.88275, -8473.02837, -8573.36845, -8703.19399, -8859.51308, -9023.12999, -9191.86301, -9363.53044, -9535.95057, -9706.94171, -9956.45526, -10123.6249, -10241.7539, -10344.1454, -10464.1029, -10634.9294, -10889.9283, -11060.8824, -11155.0355}, 
{-612.03065, -611.823335, -611.147064, -610.015078, -608.440622, -606.436935, -604.017261, -601.194843, -597.982921, -594.394739, -590.443539, -586.142563, -581.505053, -576.544252, -571.273402, -565.705745, -559.854524, -553.73298, -547.354356, -540.731894, -533.878837, -526.808426, -519.533905, -512.068515, -504.425499, -496.618098, -488.659555, -480.563113, -472.342014, -464.009499, -455.578812, -447.063194, -438.475888, -429.830135, -421.487489, -413.279709, -405.034371, -396.753414, -388.438777, -380.092399, -371.716219, -363.312175, -354.882207, -346.428254, -337.952255, -329.456148, -320.941873, -312.411368, -303.866573, -295.309426, -286.741867, -278.1112, -269.475183, -260.836938, -252.19959, -243.566262, -234.940077, -226.324159, -217.72163, -209.135614, -200.569235, -192.025615, -183.507879, -175.019149, -166.562549, -158.141202, -149.758231, -141.116453, -131.925895, -122.80904, -113.788479, -104.886803, -96.1266045, -87.530473, -79.121, -70.9207765, -62.9523934, -55.2384419, -47.8015129, -40.6641975, -33.8490867, -27.3787714, -21.2758428, -15.5628918, -12.1720098, -10.1308738, -8.44421027, -7.07264321, -5.9767967, -5.1172948, -4.45476158, -3.94982109, -3.56309741, -3.25521458, -2.98679667, -2.71846774, -2.41085186, -2.02457308, -1.52025547, -0.858523091, 0, 14.0945819, 26.7717266, 38.0776457, 48.0585508, 56.7606537, 64.2301661, 70.5132995, 75.6562657, 79.7052764, 82.7065432, 84.7062777, 85.7506918, 85.8859969, 85.1584049, 83.6141273, 81.2993759, 76.4860194, 68.9667921, 62.8013, 57.6555387, 53.1955033, 49.0871892, 44.9965919, 40.5897066, 35.5325286, 29.4910535, 22.1312764, 13.1191927, 2.1207978, -12.5014089, -29.2999877, -48.2178943, -69.1980846, -91.2065999, -114.674853, -140.294768, -168.302376, -198.933706, -232.424789, -269.011655, -308.945309, -352.44225, -399.735577, -451.05839, -506.64379, -573.862589, -644.025748, -715.581937, -786.97983, -856.668099, -915.291305, -978.640587, -1057.19563, -1155.36655, -1265.17139, -1384.192, -1510.01021, -1633.60804, -1760.66711, -1904.06877, -2077.1232, -2266.33657, -2461.75755, -2658.87412, -2862.39094, -3072.78205, -3290.01885, -3514.44506, -3748.2475, -3992.57044, -4254.07935, -4527.54785, -4812.20901, -5107.2959, -5412.04158, -5772.92272, -6130.11788, -6471.04923, -6783.13893, -7053.80914, -7221.05691, -7354.67961, -7463.2816, -7568.38881, -7673.49082, -7776.69323, -7881.75483, -7992.43441, -8112.49075, -8245.68263, -8486.62777, -8708.05048, -8908.45562, -9086.34806, -9240.23268, -9368.61433, -9469.99788, -9807.88063, -9859.41027}, 
{-620.114449, -619.883032, -619.193969, -618.059909, -616.493505, -614.507406, -612.114264, -609.32673, -606.157454, -602.619088, -598.724282, -594.485688, -589.915956, -585.027737, -579.833682, -574.346442, -568.578668, -562.543011, -556.252122, -549.718651, -542.95525, -535.97457, -528.789261, -521.411974, -513.85536, -506.132071, -498.254757, -490.236069, -482.088658, -473.825175, -465.45827, -457.000596, -448.464801, -439.863539, -431.531194, -423.312782, -415.049453, -406.743417, -398.396884, -390.012063, -381.591165, -373.1364, -364.649978, -356.134108, -347.591002, -339.022868, -330.431917, -321.820358, -313.190403, -304.54426, -295.88414, -287.176402, -278.459884, -269.737572, -261.012456, -252.28752, -243.565754, -234.850143, -226.143675, -217.449338, -208.770118, -200.109003, -191.46898, -182.853036, -174.264158, -165.705333, -157.179549, -148.384217, -139.024085, -129.732368, -120.531861, -111.445362, -102.495666, -93.705571, -85.0978723, -76.6953666, -68.5208503, -60.59712, -52.9469719, -45.5932025, -38.5586083, -31.8659857, -25.5381311, -19.5978409, -16.1530782, -14.1401973, -12.4704908, -11.0990872, -9.98111544, -9.07170419, -8.32598222, -7.69907829, -7.14612117, -6.62223961, -6.08256236, -5.48221819, -4.77633585, -3.92004411, -2.86847171, -1.57674742, 0, 20.5891464, 39.0629862, 55.4878825, 69.9301986, 82.4562977, 93.1325428, 102.025297, 109.200924, 114.725786, 118.666247, 121.08867, 122.059418, 121.644854, 119.911342, 116.925244, 112.752924, 106.974675, 99.5874236, 91.7569734, 83.4455303, 74.6153, 65.2284877, 55.2472992, 44.6339398, 33.3506151, 21.3595306, 8.62289173, -4.89709597, -19.238227, -33.6945936, -49.3203833, -66.3765014, -85.123853, -106.574516, -130.613809, -156.926738, -185.548858, -216.515721, -249.86288, -285.62589, -322.729892, -362.658803, -405.665431, -452.002582, -501.923065, -560.159315, -621.364606, -684.67184, -749.213917, -814.123739, -873.873872, -937.953518, -1012.3189, -1099.62276, -1195.77317, -1299.70642, -1410.3588, -1522.37802, -1639.96364, -1771.99363, -1928.9534, -2099.73482, -2274.03265, -2448.16105, -2629.65045, -2820.88282, -3026.30487, -3248.83388, -3483.81649, -3730.88173, -3989.22577, -4258.97708, -4539.91455, -4831.81702, -5134.46336, -5502.00047, -5866.24718, -6213.39033, -6529.61677, -6801.11338, -6958.65303, -7080.77919, -7177.42753, -7280.36296, -7390.88045, -7501.17944, -7613.00966, -7728.12086, -7848.26276, -7975.18509, -8230.05336, -8419.08392, -8560.43623, -8672.26976, -8772.74397, -8880.01834, -9012.25233, -9182.1552, -9326.19517}, 
{-650.766465, -650.401968, -649.593433, -648.351663, -646.687464, -644.611642, -642.135001, -639.268348, -636.022487, -632.408223, -628.436362, -624.117708, -619.463069, -614.483247, -609.18905, -603.591281, -597.700747, -591.528252, -585.084603, -578.380603, -571.427059, -564.234775, -556.814557, -549.17721, -541.33354, -533.294351, -525.070449, -516.672639, -508.111727, -499.398517, -490.543816, -481.558427, -472.453157, -463.238811, -454.129609, -445.030369, -435.843467, -426.573106, -417.223491, -407.798826, -398.303315, -388.741161, -379.11657, -369.433744, -359.696888, -349.910207, -340.077903, -330.204182, -320.293247, -310.349302, -300.376552, -290.343268, -280.290365, -270.222826, -260.145635, -250.063775, -239.982228, -229.905978, -219.840007, -209.789299, -199.758836, -189.753602, -179.77858, -169.838752, -159.939102, -150.084613, -140.280267, -130.105707, -119.150788, -108.297726, -97.5790768, -87.0273967, -76.6752418, -66.5551683, -56.6997323, -47.1414898, -37.9129971, -29.0468102, -20.5754853, -12.5315784, -4.94764573, 2.14375661, 8.71007251, 14.7187459, 17.302208, 17.9050397, 18.0058966, 17.664224, 16.9394674, 15.8910725, 14.5784845, 13.0611491, 11.3985118, 9.6500179, 7.87511302, 6.13324263, 4.48385221, 2.98638725, 1.70029324, 0.685015657, 0, 16.3593768, 31.0869257, 44.2328571, 55.8473814, 65.9807092, 74.6830507, 82.0046164, 87.9956168, 92.7062623, 96.1867632, 98.48733, 99.6581732, 99.7495031, 98.8115302, 96.8944649, 94.0485175, 89.8151667, 84.1719467, 78.3197713, 72.1998369, 65.7533399, 58.9214769, 51.6454441, 43.8664381, 35.5256552, 26.5642918, 16.9235445, 6.54460957, -4.6313165, -16.9509838, -30.0796689, -43.9897917, -58.6537717, -73.4315142, -88.6016959, -104.606331, -121.601593, -139.743657, -159.188696, -180.092884, -202.427957, -226.59066, -252.773254, -281.167999, -311.967155, -349.012194, -387.93386, -428.012111, -468.526904, -508.758195, -544.084096, -582.455331, -629.884234, -688.711008, -753.778591, -823.29604, -895.472413, -963.666501, -1032.03996, -1110.04565, -1206.31788, -1313.35536, -1426.50906, -1543.61917, -1666.74382, -1796.00503, -1929.93782, -2068.25275, -2216.47951, -2376.85615, -2558.13982, -2755.04657, -2967.55785, -3195.65513, -3439.31986, -3753.47327, -4069.4221, -4373.41288, -4651.6921, -4890.5063, -5021.34864, -5121.72121, -5201.33621, -5294.40102, -5400.93187, -5510.73868, -5624.34801, -5742.28642, -5865.08044, -5993.25662, -6371.14744, -6578.84146, -6671.59434, -6704.66175, -6733.29936, -6812.76285, -6998.30787, -7207.89873, -7231.63886}
};

  const double a5tab[98][200] = {{1623.72041, 1603.26038, 1582.9671, 1562.83685, 1542.86587, 1523.05045, 1503.38684, 1483.87131, 1464.50013, 1445.26955, 1426.17585, 1407.21528, 1388.38412, 1369.67863, 1351.09507, 1332.6297, 1314.2788, 1296.03863, 1277.90545, 1259.87553, 1241.94513, 1224.11051, 1206.36795, 1188.7137, 1171.14404, 1153.65522, 1136.24351, 1118.90518, 1101.63649, 1084.43371, 1067.29309, 1050.21091, 1033.18343, 1016.20692, 999.118104, 981.99637, 964.923024, 947.899508, 930.927265, 914.007736, 897.142363, 880.332589, 863.579855, 846.885603, 830.251275, 813.678314, 797.168161, 780.722259, 764.342049, 748.028973, 731.784474, 715.655146, 699.596299, 683.608397, 667.691903, 651.847281, 636.074993, 620.375503, 604.749273, 589.196768, 573.71845, 558.314783, 542.986229, 527.733252, 512.556316, 497.455882, 482.432416, 467.555474, 452.892958, 438.302825, 423.78106, 409.323648, 394.926571, 380.585814, 366.297362, 352.057197, 337.861305, 323.705669, 309.586273, 295.499101, 281.440137, 267.405366, 253.390771, 239.392336, 225.216873, 210.958931, 196.715333, 182.488202, 168.279662, 154.091834, 139.926843, 125.786812, 111.673862, 97.5901183, 83.537703, 69.5187392, 55.5353499, 41.5896584, 27.6837875, 13.8198603, 0, -12.1186564, -24.2674317, -36.4713762, -48.7555398, -61.1449727, -73.6647251, -86.3398469, -99.1953884, -112.2564, -125.547931, -139.095031, -152.922752, -167.056143, -181.520255, -196.340136, -211.540838, -227.50695, -244.314885, -261.176449, -278.193736, -295.468841, -313.103857, -331.20088, -349.862004, -369.189322, -389.284928, -410.250918, -432.189385, -455.202424, -479.575975, -505.160876, -532.004066, -560.152486, -588.79689, -618.412311, -649.702096, -682.970043, -718.519944, -756.655596, -797.680794, -842.990751, -891.465673, -943.195819, -998.271446, -1056.78281, -1116.76687, -1180.8805, -1249.7273, -1323.91084, -1404.0347, -1490.72445, -1584.53482, -1683.98873, -1789.44553, -1905.01407, -2033.11973, -2176.18793, -2341.27756, -2525.12745, -2719.42172, -2912.47091, -3121.06027, -3361.10615, -3639.589, -3948.34786, -4286.17191, -4645.67738, -5024.05306, -5441.12197, -5903.90393, -6420.91603, -6997.45046, -7640.00877, -8355.09248, -9149.20315, -10327.923, -11523.9026, -12668.8735, -13694.567, -14532.7144, -14971.0169, -15181.2562, -15156.8911, -14725.3653, -13945.8518, -12946.6962, -11783.6127, -10512.3157, -9188.51922, -7867.93762, -6749.89481, -5735.29053, -4819.2389, -3996.85405, -3263.2501, -2613.54118, -2042.8414, -1562.95934, -1111.20735}, 
{1520.58131, 1500.0443, 1479.81369, 1459.88041, 1440.23538, 1420.86953, 1401.77377, 1382.93904, 1364.35626, 1346.01635, 1327.91024, 1310.02885, 1292.36311, 1274.90394, 1257.64227, 1240.56901, 1223.67511, 1206.95147, 1190.38903, 1173.97871, 1157.71143, 1141.57813, 1125.56971, 1109.67711, 1093.89126, 1078.20307, 1062.60348, 1047.0834, 1031.63376, 1016.24549, 1000.90951, 985.616738, 970.358109, 955.124545, 939.545162, 923.799399, 908.071043, 892.36276, 876.677215, 861.017074, 845.385004, 829.78367, 814.215737, 798.683873, 783.190743, 767.739013, 752.331348, 736.970415, 721.65888, 706.399408, 691.194666, 676.151615, 661.166364, 646.239319, 631.370884, 616.561464, 601.811465, 587.121291, 572.491347, 557.922038, 543.41377, 528.966946, 514.581973, 500.259256, 485.999198, 471.802205, 457.668683, 443.658296, 429.829288, 416.059842, 402.346523, 388.685893, 375.074517, 361.508956, 347.985776, 334.501539, 321.052808, 307.636147, 294.24812, 280.885289, 267.544218, 254.221471, 240.91361, 227.6172, 214.129976, 200.552098, 186.986114, 173.435039, 159.90189, 146.389682, 132.901432, 119.440153, 106.008863, 92.6105767, 79.2483101, 65.9250787, 52.6438983, 39.4077845, 26.2197531, 13.0828197, 0, -11.525326, -23.0344724, -34.5546084, -46.1129033, -57.736526, -69.4526459, -81.2884319, -93.2710534, -105.427679, -117.785479, -130.371622, -143.213276, -156.337612, -169.771798, -183.543004, -197.678398, -212.374085, -227.681368, -243.245301, -259.129253, -275.396593, -292.110691, -309.334916, -327.132639, -345.567229, -364.702055, -384.600488, -405.325896, -426.94165, -449.438621, -472.97926, -497.648684, -523.532013, -550.224084, -578.055158, -607.486233, -638.749515, -672.077205, -707.701507, -745.854624, -786.948105, -830.980224, -878.148093, -928.648825, -982.679534, -1041.32383, -1103.6707, -1169.69564, -1239.37415, -1312.6817, -1388.82261, -1469.48611, -1551.57472, -1635.17949, -1726.90139, -1830.41858, -1949.40921, -2095.98975, -2263.48225, -2436.00328, -2596.74012, -2764.34856, -2958.01933, -3184.04821, -3426.88122, -3680.99374, -3932.93077, -4175.11167, -4429.03408, -4699.74733, -5002.86271, -5331.24267, -5686.28049, -6069.36942, -6481.90274, -6933.16819, -7414.69094, -7925.89065, -8466.18698, -9034.99957, -9765.84752, -10434.6514, -10983.3597, -11063.6267, -10732.3089, -10167.2189, -9419.1655, -8538.95753, -7577.40381, -6585.31314, -5724.15795, -4930.84252, -4204.0848, -3542.60272, -2945.11422, -2410.33722, -1936.98966, -1548.9227, -1202.39183}, 
{1424.81636, 1406.6982, 1388.76052, 1370.9988, 1353.40854, 1335.98521, 1318.72428, 1301.62125, 1284.6716, 1267.87079, 1251.21433, 1234.69768, 1218.31633, 1202.06575, 1185.94144, 1169.93888, 1154.05353, 1138.28089, 1122.61644, 1107.05565, 1091.59401, 1076.227, 1060.9501, 1045.75879, 1030.64856, 1015.61488, 1000.65324, 985.75911, 970.927981, 956.155332, 941.436643, 926.767396, 912.143072, 897.559152, 882.848309, 868.090852, 853.369049, 838.683664, 824.035462, 809.425209, 794.853667, 780.321604, 765.829782, 751.378967, 736.969923, 722.603415, 708.280209, 694.001067, 679.766756, 665.57804, 651.435684, 637.317148, 623.247006, 609.226529, 595.256985, 581.339646, 567.47578, 553.666658, 539.91355, 526.217725, 512.580453, 499.003005, 485.48665, 472.032658, 458.642299, 445.316843, 432.05756, 418.964619, 406.135816, 393.368446, 380.65737, 367.997444, 355.383529, 342.810482, 330.273163, 317.766429, 305.28514, 292.824154, 280.37833, 267.942526, 255.511601, 243.080414, 230.643824, 218.196688, 205.412935, 192.45464, 179.487734, 166.51749, 153.549182, 140.588083, 127.639466, 114.708606, 101.800775, 88.9212468, 76.075295, 63.268193, 50.5052142, 37.7916321, 25.1327201, 12.5337516, 0, -11.56261, -22.9568568, -34.2199259, -45.3890025, -56.5012718, -67.5939192, -78.70413, -89.8690894, -101.125983, -112.511995, -124.064313, -135.82012, -147.816602, -160.090944, -172.680333, -185.621952, -198.908768, -212.571648, -226.7478, -241.464932, -256.750754, -272.632976, -289.139308, -306.297459, -324.135138, -342.680056, -361.959922, -382.002445, -402.835335, -424.213426, -446.537357, -469.916702, -494.461034, -520.346942, -547.65049, -576.42987, -606.77455, -638.773995, -672.517673, -708.095051, -743.75089, -781.980795, -823.235154, -867.964354, -916.618785, -975.359217, -1037.49806, -1102.0581, -1168.06213, -1234.53295, -1296.65046, -1361.97719, -1433.01117, -1511.4386, -1597.28815, -1691.33271, -1794.34515, -1908.88958, -2033.54054, -2164.9185, -2298.16295, -2439.97692, -2601.31508, -2784.48016, -2977.1146, -3173.59016, -3366.38684, -3549.38594, -3723.40516, -3885.33858, -4026.91504, -4150.98851, -4256.24103, -4341.35465, -4405.01141, -4214.88475, -4058.41748, -3992.04378, -4072.19787, -4355.31393, -5255.02389, -6232.43241, -7190.88893, -7664.48683, -7702.73943, -7509.01676, -7125.1393, -6592.92755, -5954.202, -5250.78314, -4593.25484, -3950.42195, -3333.97552, -2755.6066, -2227.00625, -1759.86552, -1365.87547, -1015.11083, -720.103069}, 
{1327.80903, 1311.72413, 1295.74686, 1279.87552, 1264.1084, 1248.44379, 1232.87998, 1217.41525, 1202.0479, 1186.77622, 1171.5985, 1156.51304, 1141.5181, 1126.612, 1111.79302, 1097.05945, 1082.40958, 1067.84171, 1053.35411, 1038.94508, 1024.61291, 1010.3559, 996.172328, 982.060488, 968.018669, 954.045163, 940.138259, 926.296248, 912.517421, 898.800066, 885.142475, 871.542937, 857.999743, 844.511183, 831.108371, 817.772494, 804.484347, 791.241154, 778.040141, 764.878532, 751.753553, 738.662427, 725.602382, 712.570641, 699.564429, 686.580972, 673.617495, 660.671222, 647.739379, 634.819191, 621.907882, 608.765078, 595.630754, 582.507286, 569.39705, 556.302422, 543.225778, 530.169492, 517.135941, 504.127501, 491.146547, 478.195456, 465.276602, 452.392361, 439.54511, 426.737224, 413.971078, 401.321422, 388.861266, 376.443723, 364.066476, 351.72721, 339.423608, 327.153355, 314.914135, 302.703632, 290.51953, 278.359512, 266.221264, 254.10247, 242.000812, 229.913976, 217.839646, 205.775505, 193.537248, 181.217381, 168.910599, 156.620493, 144.350651, 132.104665, 119.886124, 107.698618, 95.5457371, 83.4310715, 71.3582112, 59.3307461, 47.3522663, 35.4263617, 23.5566225, 11.7466386, 0, -10.6266262, -21.1449064, -31.5864849, -41.9830058, -52.3661131, -62.7674512, -73.218664, -83.7513958, -94.3972906, -105.187993, -116.155146, -127.330395, -138.745384, -150.431756, -162.421157, -174.745229, -187.608561, -201.067503, -214.762518, -228.762309, -243.135579, -257.951033, -273.277373, -289.183303, -305.737526, -323.008746, -341.065666, -359.97699, -379.81142, -400.817297, -422.817821, -445.827805, -469.862061, -494.35408, -519.609336, -546.08832, -573.980242, -603.47431, -634.759733, -668.02572, -703.688666, -741.641451, -782.028832, -824.995571, -870.686426, -920.261125, -972.595717, -1027.58122, -1085.10865, -1145.06902, -1206.55577, -1271.23233, -1340.27593, -1414.19316, -1492.12132, -1573.8125, -1659.0188, -1745.12558, -1834.78954, -1933.2492, -2047.51931, -2169.90355, -2298.2179, -2429.37124, -2558.73547, -2682.38798, -2798.88314, -2904.94051, -2988.19749, -3041.42884, -3068.86663, -3050.0657, -2973.83487, -2828.98295, -2604.31873, -1781.5205, -983.310226, -325.279336, 76.980733, 107.878545, -936.301575, -2190.944, -3519.58691, -4318.80246, -4636.96955, -4717.0363, -4597.65317, -4317.47063, -3915.13913, -3429.30913, -2964.09608, -2484.03346, -2003.69125, -1537.63941, -1100.44789, -706.686661, -370.925674, -53.1529409, 214.943335}, 
{1238.0202, 1225.09738, 1212.13096, 1199.12394, 1186.07935, 1173.00021, 1159.88953, 1146.75033, 1133.58563, 1120.39844, 1107.19178, 1093.96868, 1080.73215, 1067.4852, 1054.23085, 1040.97213, 1027.71205, 1014.45362, 1001.19986, 987.953803, 974.718451, 961.496826, 948.291946, 935.106828, 921.944491, 908.807951, 895.700226, 882.624335, 869.583294, 856.580121, 843.617834, 830.699451, 817.827989, 805.006465, 792.471419, 780.104195, 767.783333, 755.504271, 743.26245, 731.053309, 718.872286, 706.714823, 694.576357, 682.452329, 670.338177, 658.229342, 646.121263, 634.009379, 621.88913, 609.755955, 597.605293, 585.049753, 572.475904, 559.887486, 547.288236, 534.681892, 522.072194, 509.462881, 496.857689, 484.260358, 471.674627, 459.104234, 446.552916, 434.024414, 421.522465, 409.050808, 396.61318, 384.28221, 372.128868, 360.014818, 347.939332, 335.901682, 323.901143, 311.936986, 300.008484, 288.11491, 276.255537, 264.429637, 252.636484, 240.87535, 229.145507, 217.446229, 205.776789, 194.136459, 182.433315, 170.714146, 159.026839, 147.373625, 135.756736, 124.178405, 112.640864, 101.146345, 89.6970803, 78.2953027, 66.9432442, 55.643137, 44.3972133, 33.2077056, 22.0768459, 11.0068666, 0, -9.32002674, -18.6616475, -28.0467313, -37.497147, -47.0347635, -56.6814499, -66.459075, -76.3895078, -86.4946172, -96.7962722, -107.316342, -118.076695, -129.0992, -140.405727, -152.018144, -163.95832, -176.583243, -189.962655, -203.360419, -216.870217, -230.585728, -244.600631, -259.008607, -273.903337, -289.378498, -305.527773, -322.44484, -340.22338, -358.957072, -379.139881, -400.318432, -422.466319, -445.557137, -468.68514, -492.263593, -516.94025, -542.95251, -570.537767, -599.933419, -631.376862, -666.972164, -704.521933, -743.898346, -784.973582, -827.619817, -868.588343, -911.652446, -957.464527, -1006.67699, -1059.94222, -1119.80369, -1182.71145, -1246.45396, -1310.24478, -1376.20702, -1445.15748, -1517.91293, -1597.28015, -1681.63366, -1767.17712, -1852.68589, -1936.52847, -2020.74576, -2103.51363, -2176.45883, -2234.21427, -2273.98265, -2291.06313, -2271.33221, -2205.99637, -2136.31815, -1995.74704, -1758.16244, -1397.44373, -887.470304, 640.901641, 2160.01374, 3485.23081, 4431.91767, 4815.43914, 3586.69221, 2001.8214, 246.677769, -887.893237, -1455.51754, -1761.90209, -1848.06875, -1755.03941, -1523.83595, -1195.48023, -861.747293, -522.029963, -182.849448, 149.273045, 467.816312, 766.259146, 1038.08034, 1281.16222, 1505.72281}, 
{1161.2229, 1152.81576, 1144.09307, 1135.06593, 1125.74538, 1116.14252, 1106.26841, 1096.13412, 1085.75073, 1075.12931, 1064.28093, 1053.21666, 1041.94759, 1030.48477, 1018.83928, 1007.0222, 995.044591, 982.917535, 970.652101, 958.259362, 945.75039, 933.136257, 920.428037, 907.636802, 894.773624, 881.849577, 868.875731, 855.863161, 842.822938, 829.766135, 816.703825, 803.64708, 790.606972, 777.594575, 765.058622, 752.782149, 740.542933, 728.337842, 716.163747, 704.017517, 691.896021, 679.79613, 667.714712, 655.648638, 643.594778, 631.549999, 619.511173, 607.475169, 595.438856, 583.399104, 571.352783, 558.910623, 546.464003, 534.018164, 521.578347, 509.14979, 496.737736, 484.347424, 471.984095, 459.652988, 447.359345, 435.108405, 422.90541, 410.755598, 398.664211, 386.636489, 374.677673, 362.947915, 351.603651, 340.330622, 329.124029, 317.979067, 306.890936, 295.854833, 284.865956, 273.919504, 263.010673, 252.134662, 241.28667, 230.461893, 219.65553, 208.862778, 198.078837, 187.298903, 176.215123, 164.980594, 153.752058, 142.534547, 131.333094, 120.152731, 108.99849, 97.875405, 86.7885077, 75.7428309, 64.743407, 53.7952687, 42.9034485, 32.072979, 21.3088927, 10.6162222, 0, -8.7629998, -17.5228605, -26.3037598, -35.1298753, -44.0253846, -53.0144653, -62.1212951, -71.3700516, -80.7849123, -90.390055, -100.209657, -110.267897, -120.588951, -131.196998, -142.116214, -153.370779, -165.031192, -177.128248, -189.581348, -202.424594, -215.69209, -229.417942, -243.636252, -258.381125, -273.686665, -289.586976, -306.116161, -323.308326, -341.197573, -359.62576, -378.889728, -399.081257, -420.292125, -442.884856, -466.815857, -491.969332, -518.355838, -545.985929, -574.870159, -605.019083, -635.763653, -668.000862, -701.87423, -737.527278, -775.103526, -816.500119, -859.668547, -904.313924, -950.141365, -996.855984, -1042.85975, -1090.75366, -1140.30052, -1191.98409, -1247.76007, -1308.92331, -1376.76862, -1461.62503, -1553.69997, -1633.34536, -1679.09246, -1710.04086, -1746.99066, -1794.8053, -1831.34436, -1846.86262, -1833.87997, -1783.23846, -1677.47474, -1503.82345, -1332.98202, -1055.8819, -630.866789, -16.2803793, 829.533635, 3170.65522, 5520.71111, 7615.75168, 9191.82735, 9984.98848, 8548.10943, 6589.20067, 4351.38806, 2841.58375, 2000.58151, 1450.93082, 1149.33765, 1052.50799, 1117.14781, 1299.96312, 1560.00147, 1806.7442, 2040.60955, 2262.01576, 2471.38109, 2669.12378, 2855.66208, 2989.84669, 3130.55645}, 
{1085.38834, 1077.95688, 1070.26061, 1062.30756, 1054.10578, 1045.6633, 1036.98817, 1028.08842, 1018.97211, 1009.64726, 1000.12192, 990.404135, 980.501939, 970.423374, 960.176482, 949.769304, 939.20988, 928.506251, 917.666457, 906.69854, 895.61054, 884.410498, 873.106455, 861.706452, 850.218529, 838.650726, 827.011086, 815.307648, 803.548454, 791.741543, 779.894958, 768.016738, 756.114924, 744.197558, 732.535004, 720.998625, 709.456626, 697.908537, 686.353884, 674.792196, 663.223, 651.645824, 640.060196, 628.465644, 616.861696, 605.247879, 593.623721, 581.988751, 570.342495, 558.684481, 547.014238, 535.032074, 523.043223, 511.053698, 499.069515, 487.096686, 475.141228, 463.209153, 451.306476, 439.439212, 427.613375, 415.834979, 404.110038, 392.444567, 380.84458, 369.316091, 357.865114, 346.689567, 335.982759, 325.35492, 314.799622, 304.310442, 293.880951, 283.504725, 273.175337, 262.886362, 252.631373, 242.403944, 232.19765, 222.006065, 211.822762, 201.641315, 191.455299, 181.258287, 170.636126, 159.794828, 148.94531, 138.094376, 127.248833, 116.415486, 105.60114, 94.8126013, 84.0566755, 73.3401681, 62.6698847, 52.0526307, 41.4952119, 31.0044339, 20.5871021, 10.2500223, 0, -8.55111025, -17.0361809, -25.4832345, -33.9202934, -42.3753804, -50.8765178, -59.4517282, -68.1290341, -76.936458, -85.9020224, -95.05375, -104.419663, -114.027784, -123.906136, -134.082741, -144.585622, -155.279042, -166.16763, -177.649816, -189.718529, -202.366702, -215.587265, -229.373152, -243.717292, -258.612617, -274.052059, -290.028549, -306.535018, -323.564399, -340.537862, -358.229745, -376.804505, -396.426603, -418.154871, -441.706581, -466.560506, -492.612794, -519.75959, -547.897042, -576.921298, -605.033622, -634.340877, -665.070817, -697.451197, -731.70977, -771.342042, -812.491079, -854.567699, -896.982717, -939.146952, -978.815692, -1019.07871, -1059.67646, -1101.03744, -1144.99489, -1192.75137, -1245.50939, -1313.01468, -1385.98497, -1445.81818, -1474.74759, -1485.33329, -1492.22249, -1497.25888, -1480.59125, -1432.32668, -1347.64489, -1217.96812, -1016.11895, -725.440983, -459.576371, -51.5242279, 560.475194, 1438.18164, 2643.35486, 5892.35818, 9178.69687, 12150.4798, 14455.8158, 15742.8137, 14132.0541, 11817.5263, 9101.99427, 7213.43077, 6088.3711, 5277.84678, 4737.66828, 4423.64605, 4291.59056, 4297.31226, 4408.80546, 4519.95418, 4630.21957, 4739.06277, 4845.94492, 4950.32716, 5051.67064, 5094.44178, 5113.97782}, 
{1007.70859, 997.464839, 987.344033, 977.340011, 967.446613, 957.657681, 947.967057, 938.368581, 928.856094, 919.423438, 910.064453, 900.772981, 891.542863, 882.36794, 873.242053, 864.159043, 855.112752, 846.09702, 837.105689, 828.1326, 819.171593, 810.216511, 801.261194, 792.299483, 783.325219, 774.332245, 765.314399, 756.265525, 747.179462, 738.050053, 728.871138, 719.636558, 710.340154, 700.975768, 691.2867, 681.397332, 671.435055, 661.401842, 651.299664, 641.130493, 630.8963, 620.599058, 610.240737, 599.823309, 589.348747, 578.819021, 568.236104, 557.601967, 546.918581, 536.187919, 525.411951, 514.417361, 503.385209, 492.321267, 481.231306, 470.121098, 458.996415, 447.863028, 436.726708, 425.593228, 414.468359, 403.357872, 392.267539, 381.203132, 370.170422, 359.175181, 348.22318, 337.462775, 327.038898, 316.663251, 306.332364, 296.042765, 285.790983, 275.573545, 265.38698, 255.227817, 245.092584, 234.977809, 224.880022, 214.79575, 204.721522, 194.653866, 184.589311, 174.524386, 164.161037, 153.64928, 143.142671, 132.647298, 122.169248, 111.71461, 101.289472, 90.8999232, 80.5520503, 70.2519421, 60.0056867, 49.8193722, 39.6990869, 29.6509188, 19.6809562, 9.79528722, 0, -8.37939856, -16.6550084, -24.855446, -33.0093277, -41.1452701, -49.2918897, -57.4778028, -65.7316259, -74.0819756, -82.5574682, -91.1867202, -99.9983481, -109.020968, -118.283197, -127.813652, -137.640948, -147.670918, -157.914637, -168.67845, -179.96466, -191.775574, -204.113498, -216.980736, -230.379594, -244.312378, -258.781394, -273.788946, -289.337341, -305.428884, -321.651525, -338.573855, -356.322487, -375.024033, -395.216843, -416.821662, -439.649434, -463.703248, -488.986197, -515.501369, -543.251856, -572.578397, -603.043669, -634.584702, -667.138523, -700.642161, -733.737164, -767.97991, -803.631298, -840.952224, -880.203589, -923.979343, -967.35582, -1007.16856, -1041.91532, -1073.48753, -1102.25292, -1128.57923, -1151.89921, -1173.72809, -1196.60112, -1234.72882, -1262.02155, -1257.49556, -1208.58552, -1116.99064, -977.862483, -797.524279, -574.023943, -264.446559, 150.951897, 506.624755, 1036.11299, 1823.29971, 2952.06802, 4506.30103, 8644.11745, 12856.6059, 16709.0905, 19766.8956, 21595.3454, 19922.4262, 17375.6919, 14307.8973, 12125.4592, 10764.0578, 9720.34716, 8951.95872, 8416.52386, 8071.67396, 7875.04041, 7784.20116, 7717.38392, 7671.70392, 7644.27635, 7632.21643, 7632.63935, 7642.66032, 7602.94526, 7466.69955}, 
{938.236429, 928.1099, 918.202939, 908.505482, 899.007469, 889.698837, 880.569525, 871.609471, 862.808614, 854.156892, 845.644243, 837.260606, 828.995919, 820.840119, 812.783147, 804.814939, 796.925434, 789.10457, 781.342287, 773.628521, 765.953212, 758.306298, 750.677716, 743.057406, 735.435306, 727.801353, 720.145487, 712.457646, 704.727767, 696.94579, 689.101652, 681.185293, 673.186649, 665.09566, 656.494929, 647.586631, 638.577833, 629.471693, 620.271367, 610.980014, 601.600791, 592.136853, 582.591359, 572.967465, 563.268329, 553.497107, 543.656957, 533.751036, 523.782501, 513.754509, 503.670218, 493.444201, 483.17012, 472.85305, 462.49807, 452.110256, 441.694687, 431.256439, 420.80059, 410.332217, 399.856398, 389.378209, 378.902729, 368.435035, 357.980204, 347.543314, 337.129441, 326.853639, 316.828321, 306.831746, 296.861864, 286.916623, 276.99397, 267.091854, 257.208222, 247.341024, 237.488207, 227.647719, 217.817508, 207.995523, 198.179711, 188.368021, 178.5584, 168.748798, 158.676699, 148.475761, 138.282774, 128.104136, 117.94625, 107.815515, 97.7183316, 87.6611013, 77.6502243, 67.6921012, 57.7931326, 47.9597191, 38.1982614, 28.51516, 18.9168156, 9.40962872, 0, -7.93297828, -15.7653053, -23.5244386, -31.237836, -38.9329549, -46.637253, -54.378188, -62.1832175, -70.0797991, -78.0953904, -86.2574491, -94.5934327, -103.130799, -111.897005, -120.91951, -130.22577, -139.715555, -149.398081, -159.589625, -170.290282, -181.500148, -193.219319, -205.447892, -218.185961, -231.433623, -245.190974, -259.458109, -274.235125, -289.522117, -304.978516, -321.069994, -337.898846, -355.567367, -374.316737, -394.17981, -415.152402, -437.295144, -460.668667, -485.3336, -511.350574, -540.01287, -569.773313, -600.45136, -631.866473, -663.838109, -694.763552, -726.239981, -758.4424, -791.545812, -825.72522, -863.237624, -899.631369, -933.963688, -965.051614, -991.221761, -1011.02093, -1022.99593, -1018.33018, -1004.60735, -997.443886, -1028.00528, -1050.18696, -1025.32675, -933.464519, -786.329699, -581.1057, -330.407623, -35.4196825, 359.256785, 877.013967, 1332.26333, 1989.52867, 2944.54163, 4293.03389, 6130.73712, 10848.3559, 15672.9058, 20126.3751, 23730.7524, 26008.026, 24502.4878, 22032.2872, 18966.9971, 16759.7236, 15351.1408, 14230.4504, 13360.9001, 12705.7375, 12228.2102, 11891.566, 11800.7793, 11647.7477, 11454.4786, 11242.9792, 11035.2566, 10853.3184, 10719.1715, 10505.4857, 10185.0648}, 
{874.787044, 867.796736, 860.814736, 853.838339, 846.86484, 839.891535, 832.915717, 825.934682, 818.945726, 811.946144, 804.933229, 797.904279, 790.856586, 783.787448, 776.694158, 769.574012, 762.424305, 755.242331, 748.025387, 740.770767, 733.475765, 726.137678, 718.753801, 711.321427, 703.837853, 696.300374, 688.706284, 681.052878, 673.337453, 665.557302, 657.70972, 649.792004, 641.801448, 633.735346, 625.456459, 617.032178, 608.531515, 599.956129, 591.307682, 582.587834, 573.798248, 564.940584, 556.016502, 547.027665, 537.975733, 528.862366, 519.689227, 510.457976, 501.170273, 491.827781, 482.432161, 472.891739, 463.303534, 453.67123, 443.998511, 434.289062, 424.546566, 414.774708, 404.977172, 395.157642, 385.319804, 375.467341, 365.603937, 355.733276, 345.859044, 335.984924, 326.1146, 316.31536, 306.652964, 296.999919, 287.355787, 277.72013, 268.092508, 258.472484, 248.859617, 239.253469, 229.653602, 220.059576, 210.470953, 200.887294, 191.30816, 181.733112, 172.161712, 162.59352, 152.813628, 142.933343, 133.066551, 123.219771, 113.399527, 103.612338, 93.8647258, 84.1632121, 74.5143177, 64.9245638, 55.4004716, 45.9485623, 36.575357, 27.287377, 18.0911434, 8.9931773, 0, -6.9064089, -13.817204, -20.7524485, -27.732206, -34.7765397, -41.9055132, -49.1391896, -56.4976324, -64.0009051, -71.6690709, -79.5221932, -87.5803355, -95.863561, -104.391933, -113.185516, -122.264371, -131.6197, -141.267442, -151.292947, -161.710094, -172.532761, -183.774826, -195.450169, -207.572666, -220.156196, -233.214638, -246.761869, -260.811769, -275.378214, -290.402527, -305.997747, -322.199519, -339.04349, -356.153164, -373.770256, -392.246387, -411.740846, -432.41292, -454.421898, -477.927067, -505.31838, -533.845563, -563.23147, -593.198953, -623.470867, -654.581798, -685.239934, -714.965196, -743.277503, -769.696777, -792.345005, -813.848625, -840.158261, -872.336762, -901.467762, -923.115361, -932.843657, -914.676435, -878.340905, -846.153716, -853.234608, -849.952386, -795.87225, -669.944648, -483.967256, -234.66033, 63.4811688, 409.128726, 866.127732, 1461.45753, 2027.65789, 2816.06151, 3920.95374, 5436.61993, 7457.34543, 12323.0404, 17320.9591, 21983.9808, 25844.9847, 28436.8498, 27407.6574, 25431.6165, 22849.3767, 20979.3564, 19775.1716, 18783.0352, 17976.9334, 17330.8527, 16818.7796, 16414.7003, 16558.8057, 16418.0453, 16074.1359, 15608.7943, 15103.7372, 14640.6812, 14301.3432, 13809.1951, 13279.09}, 
{819.559987, 816.042758, 812.271591, 808.25306, 803.993739, 799.500202, 794.779022, 789.836772, 784.680026, 779.315358, 773.749341, 767.988549, 762.039555, 755.908933, 749.603256, 743.129099, 736.493034, 729.701635, 722.761477, 715.679131, 708.461172, 701.114173, 693.644709, 686.059352, 678.364676, 670.567255, 662.673662, 654.69047, 646.624254, 638.481587, 630.269043, 621.993194, 613.660615, 605.277879, 597.062876, 588.912076, 580.71941, 572.484595, 564.207346, 555.88738, 547.524412, 539.118158, 530.668335, 522.174657, 513.636842, 505.054605, 496.427662, 487.75573, 479.038523, 470.275758, 461.467151, 452.47769, 443.44474, 434.370937, 425.258917, 416.111318, 406.930776, 397.719927, 388.481408, 379.217856, 369.931907, 360.626198, 351.303366, 341.966046, 332.616877, 323.258493, 313.893533, 304.559274, 295.292164, 286.023392, 276.753349, 267.482427, 258.211015, 248.939506, 239.668289, 230.397756, 221.128298, 211.860305, 202.594169, 193.33028, 184.069029, 174.810807, 165.556006, 156.305015, 146.860925, 137.326928, 127.808586, 118.312692, 108.84604, 99.4154242, 90.0276378, 80.6894745, 71.407728, 62.1891919, 53.04066, 43.9689258, 34.9807831, 26.0830255, 17.2824467, 8.5858403, 0, -5.78508753, -11.7000624, -17.7561911, -23.9647403, -30.3369766, -36.8841666, -43.6175767, -50.5484737, -57.688124, -65.0477944, -72.6387512, -80.4722613, -88.559591, -96.912007, -105.540776, -114.457164, -123.757658, -133.465697, -143.399791, -153.589468, -164.064256, -174.853682, -185.987275, -197.494562, -209.405071, -221.748329, -234.553865, -247.851206, -261.66988, -276.214752, -291.275722, -306.829717, -322.853662, -338.478906, -354.105165, -370.357641, -387.466934, -405.663645, -425.178374, -446.24172, -472.101785, -499.053298, -526.736476, -554.791539, -582.858706, -613.510226, -642.721279, -669.399078, -692.450834, -710.783757, -718.616738, -725.27548, -742.163395, -772.354425, -801.916511, -824.552941, -833.967004, -812.875863, -768.465778, -719.907872, -693.720374, -650.955554, -562.708709, -411.842829, -201.160606, 75.6879179, 408.15415, 798.206942, 1309.7765, 1971.74419, 2648.8664, 3559.39946, 4789.03721, 6423.47351, 8548.4022, 13313.4151, 18224.3336, 22850.8771, 26762.7648, 29529.7163, 29026.2878, 27647.4703, 25689.4817, 24282.5423, 23392.7438, 22638.677, 22003.8085, 21471.6052, 21025.5338, 20649.0611, 21104.5835, 21051.8662, 20629.6832, 19976.8083, 19232.0154, 18534.0786, 18021.7716, 17297.0979, 16563.3361}, 
{771.4088, 768.057881, 764.495505, 760.726583, 756.756024, 752.588738, 748.229636, 743.683627, 738.955622, 734.050531, 728.973264, 723.728731, 718.321842, 712.757507, 707.040637, 701.176141, 695.168929, 689.023912, 682.746, 676.340103, 669.811131, 663.163993, 656.403601, 649.534864, 642.562692, 635.491996, 628.327685, 621.07467, 613.737861, 606.322167, 598.8325, 591.273768, 583.650882, 575.968753, 568.377183, 560.805589, 553.181644, 545.505557, 537.777535, 529.997788, 522.166523, 514.283947, 506.35027, 498.365698, 490.330441, 482.244705, 474.1087, 465.922633, 457.686712, 449.401146, 441.066142, 432.569557, 424.026386, 415.439273, 406.810861, 398.143793, 389.440714, 380.704266, 371.937095, 363.141842, 354.321152, 345.477668, 336.614034, 327.732893, 318.836889, 309.928666, 301.010867, 292.122732, 283.302623, 274.477706, 265.648251, 256.814531, 247.976815, 239.135377, 230.290486, 221.442414, 212.591432, 203.737812, 194.881823, 186.023739, 177.16383, 168.302366, 159.43962, 150.575862, 141.493151, 132.305454, 123.129361, 113.972225, 104.841399, 95.7442352, 86.6880864, 77.6803052, 68.7282442, 59.8392561, 51.0206934, 42.2799089, 33.6242551, 25.0610847, 16.5977503, 8.2416045, 0, -5.21525898, -10.5998448, -16.1611745, -21.9066653, -27.8437342, -33.9797985, -40.3222753, -46.8785816, -53.6561347, -60.6623515, -67.9046494, -75.3904454, -83.1271566, -91.1222001, -99.3829932, -107.916953, -116.855078, -126.222439, -135.746926, -145.462435, -155.402867, -165.60212, -176.094093, -186.912684, -198.091794, -209.66532, -221.667161, -234.131217, -247.091386, -260.807827, -275.005218, -289.649577, -304.706927, -319.243703, -333.675718, -348.658675, -364.428468, -381.220994, -399.272148, -418.817826, -443.659051, -469.381553, -495.523704, -521.623873, -547.220433, -573.88384, -598.612358, -620.436336, -638.386125, -651.492073, -654.845011, -656.229778, -665.337134, -684.716451, -702.336762, -712.713374, -710.361594, -682.319221, -632.278504, -572.088974, -519.750086, -445.091148, -328.465191, -155.465037, 79.4379588, 385.847833, 759.3014, 1205.75564, 1782.74734, 2518.63685, 3290.16714, 4300.64165, 5630.90364, 7361.79639, 9574.16315, 14318.1783, 19213.0211, 23847.2023, 27809.2323, 30687.6215, 30466.6551, 29408.5525, 27789.3496, 26658.7057, 25986.4304, 25419.9903, 24945.312, 24548.3225, 24214.9485, 23931.1168, 24544.0084, 24582.5811, 24197.8495, 23540.8282, 22762.5318, 22013.975, 21446.1723, 20674.0552, 19776.4636}, 
{727.568456, 724.124736, 720.529622, 716.78574, 712.895716, 708.862175, 704.687744, 700.375049, 695.926715, 691.345368, 686.633635, 681.794141, 676.829511, 671.742374, 666.535352, 661.211074, 655.772165, 650.22125, 644.560956, 638.793909, 632.922734, 626.950057, 620.878505, 614.710702, 608.449276, 602.096852, 595.656056, 589.129514, 582.519852, 575.829695, 569.06167, 562.218403, 555.302519, 548.316644, 541.317157, 534.278677, 527.175177, 520.007539, 512.776643, 505.483372, 498.128606, 490.713229, 483.23812, 475.704162, 468.112236, 460.463224, 452.758007, 444.997467, 437.182486, 429.313944, 421.392724, 413.329407, 405.217132, 397.058738, 388.857064, 380.61495, 372.335233, 364.020755, 355.674352, 347.298865, 338.897133, 330.471995, 322.026289, 313.562854, 305.084531, 296.594158, 288.094573, 279.634928, 271.263262, 262.886899, 254.505676, 246.11943, 237.727997, 229.331216, 220.928922, 212.520952, 204.107144, 195.687334, 187.261359, 178.829056, 170.390262, 161.944813, 153.492547, 145.033301, 136.316326, 127.472023, 118.633803, 109.809635, 101.007489, 92.2353315, 83.5011328, 74.8128614, 66.1784861, 57.6059757, 49.103299, 40.6784248, 32.3393219, 24.0939592, 15.9503055, 7.91632944, 0, -4.6947176, -9.59355123, -14.7003003, -20.0187641, -25.5527422, -31.3060338, -37.2824383, -43.4857551, -49.9197836, -56.5883232, -63.4951733, -70.6441332, -78.0390023, -85.68358, -93.5816656, -101.737059, -110.314035, -119.33932, -128.453729, -137.695449, -147.102669, -156.713574, -166.566353, -176.699193, -187.15028, -197.957803, -209.159948, -220.794903, -232.900854, -245.77454, -259.100796, -272.840243, -286.953505, -300.526318, -313.956747, -327.876158, -342.507639, -358.074277, -374.799158, -392.90537, -416.535335, -440.799965, -465.15552, -489.058259, -511.964446, -534.227869, -554.182877, -571.061349, -584.095161, -592.516192, -592.851832, -590.343934, -591.819601, -598.957288, -602.923615, -599.604804, -584.887075, -552.04877, -500.176686, -431.202575, -351.904205, -243.770054, -98.7498833, 93.8407544, 349.149821, 680.276453, 1089.8878, 1588.37839, 2224.39342, 3024.94134, 3882.01642, 4978.41271, 6387.87434, 8184.14543, 10440.9701, 15066.6985, 19841.8172, 24381.4189, 28300.5963, 31214.442, 31248.441, 30500.3653, 29223.7109, 28383.1703, 27950.724, 27602.0208, 27323.8579, 27103.0325, 26926.3418, 26780.583, 27531.518, 27683.9477, 27388.2187, 26794.6773, 26053.6701, 25315.5434, 24730.6438, 23960.2895, 22932.1602}, 
{686.487617, 682.949948, 679.318433, 675.593476, 671.775484, 667.864861, 663.862013, 659.767346, 655.581265, 651.304174, 646.936481, 642.478589, 637.930905, 633.293833, 628.56778, 623.75315, 618.850349, 613.859782, 608.781855, 603.616974, 598.365542, 593.027967, 587.604652, 582.096005, 576.502429, 570.824331, 565.062116, 559.216189, 553.286955, 547.27482, 541.18019, 535.00347, 528.745064, 522.40538, 515.94686, 509.389689, 502.754508, 496.042954, 489.256665, 482.397278, 475.466429, 468.465757, 461.396899, 454.261492, 447.061173, 439.797579, 432.472348, 425.087118, 417.643525, 410.143207, 402.5878, 394.911552, 387.184952, 379.411097, 371.593087, 363.734018, 355.83699, 347.9051, 339.941448, 331.94913, 323.931245, 315.890892, 307.831169, 299.755173, 291.666004, 283.566758, 275.460536, 267.409805, 259.47561, 251.5386, 243.598026, 235.653135, 227.703178, 219.747404, 211.785063, 203.815403, 195.837674, 187.851126, 179.855008, 171.848569, 163.83106, 155.801728, 147.759824, 139.704597, 131.347037, 122.836585, 114.32615, 105.824335, 97.3397448, 88.8809823, 80.4566521, 72.075358, 63.7457042, 55.4762946, 47.2757331, 39.1526238, 31.1155707, 23.1731778, 15.3340491, 7.60678845, 0, -4.20069216, -8.63840868, -13.3134716, -18.226203, -23.376925, -28.7659595, -34.3936288, -40.2602547, -46.3661595, -52.7116651, -59.2970936, -66.1227671, -73.1890076, -80.4961373, -88.0444781, -95.8343521, -104.064154, -112.762502, -121.481697, -130.264505, -139.153692, -148.192024, -157.422267, -166.887188, -176.629553, -186.692129, -197.11768, -207.948974, -219.228777, -231.288969, -243.777194, -256.649483, -269.861869, -282.565061, -295.117754, -308.093391, -321.689601, -336.104013, -351.534254, -368.177954, -390.283197, -412.764408, -435.026735, -456.475324, -476.515326, -494.434644, -509.78498, -522.000792, -530.51654, -534.766681, -532.830985, -527.154332, -521.356965, -516.141209, -506.251975, -489.108931, -462.131744, -425.330017, -372.944863, -296.392015, -190.667123, -49.5044117, 123.498695, 335.066778, 609.414481, 962.728223, 1404.38729, 1948.8133, 2635.38732, 3489.37194, 4420.95964, 5588.18623, 7056.14625, 8889.93427, 11154.6448, 15587.5691, 20163.5559, 24529.6506, 28332.8987, 31220.3455, 31479.5989, 31023.4335, 30081.5108, 29531.9041, 29347.6118, 29231.4606, 29169.9573, 29149.6089, 29156.9223, 29178.4044, 30033.4589, 30308.8548, 30141.4918, 29668.2694, 29026.0872, 28351.8449, 27782.442, 27049.4914, 25934.9294}, 
{646.614948, 643.240145, 639.80643, 636.312736, 632.757996, 629.141143, 625.46111, 621.716829, 617.907233, 614.031256, 610.087829, 606.075887, 601.994362, 597.842187, 593.618294, 589.321617, 584.951088, 580.505641, 575.984208, 571.385722, 566.709117, 561.953324, 557.117277, 552.199908, 547.200152, 542.116939, 536.949205, 531.69588, 526.355898, 520.928193, 515.411696, 509.805341, 504.108061, 498.318788, 492.330353, 486.196971, 479.974134, 473.664218, 467.2696, 460.792655, 454.235759, 447.601289, 440.891621, 434.10913, 427.256194, 420.335188, 413.348488, 406.298471, 399.187512, 392.017988, 384.792275, 377.470307, 370.097822, 362.678117, 355.214487, 347.710229, 340.168639, 332.593013, 324.986647, 317.352838, 309.694882, 302.016075, 294.319713, 286.609092, 278.887509, 271.158259, 263.42464, 255.761305, 248.241197, 240.72044, 233.197704, 225.671658, 218.140975, 210.604323, 203.060373, 195.507796, 187.945261, 180.37144, 172.785002, 165.184617, 157.568956, 149.936689, 142.286487, 134.617019, 126.60187, 118.409091, 110.21064, 102.015737, 93.8336025, 85.6734566, 77.5445191, 69.4560099, 61.417149, 53.4371565, 45.5252523, 37.6906563, 29.9425886, 22.2902691, 14.7429179, 7.30975484, 0, -3.71041147, -7.69164402, -11.9405916, -16.454148, -21.2292072, -26.2626632, -31.5514098, -37.092341, -42.8823506, -48.9183326, -55.1971809, -61.7157894, -68.471052, -75.4598626, -82.6791152, -90.1257036, -98.0350591, -106.438152, -114.792329, -123.145597, -131.545966, -140.041446, -148.680044, -157.509771, -166.578635, -175.934646, -185.625811, -195.700142, -206.205646, -217.525191, -229.249146, -241.325063, -253.700492, -265.598245, -277.328242, -289.393671, -301.959506, -315.19072, -329.252288, -344.309183, -364.455195, -384.730756, -404.532159, -423.255695, -440.297657, -454.396496, -465.770807, -473.981343, -478.588859, -479.154107, -474.976254, -466.197357, -453.695395, -437.332493, -414.896748, -385.107453, -346.683901, -305.428466, -251.36757, -166.800636, -36.5232375, 135.193639, 335.342026, 567.205268, 861.671439, 1236.95759, 1707.27385, 2289.79844, 3016.40161, 3910.64298, 4903.54221, 6125.43578, 7631.91633, 9478.57654, 11721.0091, 15909.3837, 20231.0717, 24368.0215, 28002.1813, 30815.4998, 31268.0824, 31078.2821, 30451.6942, 30180.8749, 30239.0809, 30355.0013, 30514.1212, 30701.9253, 30903.8985, 31105.5257, 32016.1779, 32410.1913, 32398.37, 32091.5184, 31600.4405, 31035.9406, 30508.8229, 29835.351, 28689.2748}, 
{606.399111, 603.701953, 600.938107, 598.106466, 595.205922, 592.235369, 589.193699, 586.079806, 582.892581, 579.630917, 576.293708, 572.879846, 569.388224, 565.817734, 562.167271, 558.435725, 554.62199, 550.72496, 546.743525, 542.676581, 538.523018, 534.28173, 529.95161, 525.531551, 521.020445, 516.417185, 511.720664, 506.929774, 502.043409, 497.060461, 491.979824, 486.800388, 481.521049, 476.140698, 470.531699, 464.758871, 458.888553, 452.923744, 446.867444, 440.72265, 434.492362, 428.179578, 421.787298, 415.318521, 408.776244, 402.163468, 395.483191, 388.738412, 381.932129, 375.067343, 368.147051, 361.159983, 354.123717, 347.041562, 339.916824, 332.752812, 325.552835, 318.320201, 311.058218, 303.770194, 296.459437, 289.129256, 281.782958, 274.423853, 267.055248, 259.680451, 252.30277, 245.003372, 237.861553, 230.720046, 223.577111, 216.43101, 209.280004, 202.122353, 194.956319, 187.780163, 180.592146, 173.390528, 166.173571, 158.939536, 151.686684, 144.413276, 137.117572, 129.797835, 122.097413, 114.199488, 106.291508, 98.3832518, 90.4844974, 82.6050235, 74.7546089, 66.9430319, 59.1800712, 51.4755053, 43.8391129, 36.2806724, 28.8099625, 21.4367618, 14.1708487, 7.02200194, 0, -3.20110431, -6.71048416, -10.5215632, -14.6277651, -19.0225135, -23.6992321, -28.6513446, -33.8722745, -39.3554456, -45.0942815, -51.082206, -57.3126425, -63.7790149, -70.4747468, -77.3932617, -84.5279835, -92.1563744, -100.312435, -108.347121, -116.31472, -124.269523, -132.265818, -140.357895, -148.600042, -157.046548, -165.751704, -174.769796, -184.155116, -193.961951, -204.657283, -215.731388, -227.114749, -238.737848, -249.864178, -260.757715, -271.860299, -283.302507, -295.214917, -307.728108, -320.972658, -338.603887, -356.154883, -373.066602, -388.779997, -402.736023, -414.005757, -422.492501, -427.729679, -429.250717, -426.589038, -419.481422, -407.009392, -388.581059, -363.595415, -331.432841, -291.482071, -243.131844, -195.609624, -136.229344, -41.5717776, 110.043058, 307.811958, 533.841589, 789.248458, 1107.36019, 1506.71899, 2003.02142, 2614.0719, 3368.10888, 4287.46875, 5326.30954, 6585.63494, 8111.38153, 9949.48587, 12145.8845, 16060.7356, 20097.1987, 23972.6552, 27404.4861, 30110.0727, 30721.8448, 30765.4359, 30423.2064, 30406.0509, 30687.1184, 31019.335, 31386.8606, 31773.8553, 32164.4789, 32542.8915, 33446.0217, 33940.846, 34099.5547, 33994.3382, 33697.3871, 33280.8915, 32817.0421, 32211.5587, 31099.6997}, 
{558.061705, 557.780689, 557.291903, 556.598608, 555.70407, 554.61155, 553.324314, 551.845623, 550.178741, 548.326932, 546.293459, 544.081585, 541.694573, 539.135688, 536.408192, 533.515349, 530.460422, 527.246674, 523.877369, 520.35577, 516.685141, 512.868744, 508.909844, 504.811703, 500.577586, 496.210754, 491.714472, 487.092003, 482.346611, 477.481558, 472.500108, 467.405525, 462.201072, 456.890011, 451.474562, 445.958532, 440.345742, 434.639489, 428.843071, 422.959783, 416.992925, 410.945793, 404.821684, 398.623896, 392.355725, 386.02047, 379.621428, 373.161895, 366.645169, 360.074547, 353.453327, 346.813561, 340.129167, 333.40282, 326.637193, 319.834961, 312.998798, 306.131377, 299.235372, 292.313458, 285.368308, 278.402596, 271.418996, 264.420182, 257.408828, 250.387608, 243.359195, 236.390258, 229.545927, 222.696894, 215.841685, 208.978825, 202.10684, 195.224256, 188.329597, 181.42139, 174.49816, 167.558432, 160.600732, 153.623585, 146.625517, 139.605054, 132.56072, 125.491042, 118.033777, 110.375423, 102.706813, 95.0381814, 87.3797604, 79.7417832, 72.134483, 64.568093, 57.0528461, 49.5989756, 42.2167145, 34.916296, 27.7079531, 20.6019189, 13.6084266, 6.73770926, 0, -2.4457295, -5.2760342, -8.47895805, -12.042545, -15.954839, -20.203884, -24.7777239, -29.6644028, -34.8519644, -40.3284529, -46.0819121, -52.1003859, -58.3719184, -64.8845535, -71.6263351, -78.5853071, -86.1261422, -94.2906882, -102.215091, -109.968101, -117.618469, -125.234943, -132.886275, -140.641214, -148.568511, -156.736915, -165.215177, -174.072046, -183.376273, -193.848327, -204.666275, -215.703351, -226.832791, -237.052416, -246.673167, -256.239427, -265.887053, -275.751903, -285.969834, -296.676704, -310.909705, -325.020346, -338.576831, -351.147366, -362.300156, -373.916822, -382.673799, -387.560939, -387.568093, -381.685113, -367.149308, -346.845066, -322.818544, -295.437727, -261.644326, -219.91838, -168.739928, -116.555308, -49.6131907, 56.7100827, 225.651432, 446.364652, 706.981834, 1008.58016, 1374.17156, 1816.69954, 2340.15196, 2955.15047, 3705.15426, 4615.04774, 5679.31296, 6957.914, 8486.25919, 10299.7569, 12433.8153, 16052.0419, 19779.5962, 23369.8367, 26576.122, 29151.8107, 29905.0186, 30164.5088, 30088.7439, 30303.8174, 30783.088, 31305.0682, 31852.859, 32409.5615, 32958.2767, 33482.1056, 34077.9515, 34564.9212, 34919.4067, 35117.7997, 35136.4922, 34951.8759, 34540.3428, 34016.1238, 33098.6953}, 
{519.490016, 520.604075, 521.436213, 521.991988, 522.27696, 522.296687, 522.056728, 521.562642, 520.819986, 519.83432, 518.611203, 517.156194, 515.47485, 513.572731, 511.455395, 509.128402, 506.597309, 503.867675, 500.94506, 497.835022, 494.543119, 491.074911, 487.435955, 483.631811, 479.668038, 475.550193, 471.283837, 466.874526, 462.327821, 457.64928, 452.844462, 447.918924, 442.878227, 437.727928, 432.536584, 427.276929, 421.920942, 416.472137, 410.934027, 405.310128, 399.603954, 393.819019, 387.958837, 382.026923, 376.02679, 369.961954, 363.835928, 357.652228, 351.414366, 345.125858, 338.790218, 332.466153, 326.100788, 319.69644, 313.255429, 306.78007, 300.272683, 293.735585, 287.171094, 280.581528, 273.969205, 267.336442, 260.685557, 254.018868, 247.338693, 240.64735, 233.947157, 227.298487, 220.760322, 214.215242, 207.661802, 201.098554, 194.524055, 187.936858, 181.335517, 174.718586, 168.084621, 161.432174, 154.759801, 148.066056, 141.349493, 134.608666, 127.842129, 121.048437, 113.85579, 106.45571, 99.0441678, 91.6317376, 84.2289919, 76.8465039, 69.4948463, 62.1845923, 54.9263146, 47.7305864, 40.6079805, 33.5690699, 26.6244275, 19.7846264, 13.0602395, 6.46183969, 0, -2.05847428, -4.51836185, -7.36586397, -10.5871819, -14.1685169, -18.0960702, -22.3560431, -26.9346368, -31.8180527, -36.9924918, -42.4441556, -48.1592453, -54.1239621, -60.3245073, -66.7470822, -73.3778879, -80.5969973, -88.446879, -96.0228344, -103.395466, -110.635376, -117.813166, -124.999439, -132.264798, -139.679844, -147.315179, -155.241407, -163.529129, -172.248948, -182.189868, -192.440675, -202.856451, -213.292277, -222.872386, -231.817285, -240.542373, -249.121984, -257.630454, -266.142121, -274.73132, -285.218556, -295.400553, -305.010005, -313.779608, -321.442054, -331.686577, -339.300199, -343.026478, -341.608976, -333.791251, -316.25026, -292.322015, -265.267344, -235.55638, -197.961582, -149.813551, -88.442885, -17.2242582, 73.9294582, 201.698929, 380.984315, 608.968786, 880.145136, 1198.09081, 1581.77671, 2043.10786, 2585.85569, 3219.81651, 3984.61, 4902.98608, 5979.3725, 7257.66011, 8766.94108, 10536.3076, 12594.8519, 15968.2563, 19439.8753, 22789.6534, 25797.5353, 28243.4656, 29060.5302, 29440.1048, 29525.0734, 29886.547, 30497.9031, 31154.0911, 31837.4099, 32530.1582, 33214.6348, 33873.1385, 34538.7407, 35136.9701, 35630.7367, 35982.9507, 36156.5222, 36114.3612, 35819.378, 35319.6369, 34506.9535}, 
{489.408433, 490.956453, 492.21316, 493.184383, 493.875954, 494.293704, 494.443464, 494.331065, 493.962337, 493.343113, 492.479222, 491.376495, 490.040765, 488.477861, 486.693615, 484.693857, 482.48442, 480.071132, 477.459827, 474.656333, 471.666484, 468.496109, 465.151039, 461.637106, 457.96014, 454.125972, 450.140434, 446.009356, 441.73857, 437.333906, 432.801195, 428.146268, 423.374957, 418.493092, 413.573866, 408.588013, 403.505455, 398.329838, 393.064807, 387.714006, 382.281081, 376.769676, 371.183437, 365.526008, 359.801034, 354.01216, 348.163032, 342.257293, 336.29859, 330.290567, 324.236868, 318.20591, 312.135163, 306.026868, 299.883265, 293.706595, 287.499099, 281.263018, 275.000592, 268.714063, 262.405672, 256.077658, 249.732263, 243.371729, 236.998294, 230.614202, 224.221691, 217.882538, 211.657087, 205.424795, 199.184044, 192.933214, 186.670688, 180.394847, 174.104073, 167.796746, 161.471249, 155.125962, 148.759268, 142.369548, 135.955183, 129.514555, 123.046046, 116.548036, 109.635405, 102.506352, 95.3636851, 88.2182321, 81.0808194, 73.962274, 66.8734226, 59.8250923, 52.8281099, 45.8933022, 39.0314962, 32.2535187, 25.5701966, 18.9923568, 12.5308262, 6.19643162, 0, -1.98681623, -4.34051091, -7.04860201, -10.0986075, -13.4780453, -17.1744334, -21.1752897, -25.4681323, -30.040479, -34.8798478, -39.9737567, -45.3097236, -50.8752665, -56.6579034, -62.6451522, -68.8245308, -75.5141456, -82.7487412, -89.7680767, -96.6305101, -103.3944, -110.118104, -116.859982, -123.67839, -130.631688, -137.778234, -145.176386, -152.884503, -160.960942, -170.077792, -179.454647, -188.965746, -198.48533, -207.574336, -216.263654, -224.667721, -232.754767, -240.493019, -247.850709, -254.796065, -261.703566, -268.011552, -273.608768, -278.383961, -282.225876, -289.58857, -294.654151, -296.170038, -292.88365, -283.542404, -265.714086, -240.767523, -211.955771, -179.225917, -135.775359, -77.8319702, -1.62361969, 96.5199791, 220.495943, 374.312489, 559.564145, 785.038501, 1050.0007, 1361.60724, 1740.38611, 2200.72754, 2754.77777, 3416.34525, 4207.46656, 5145.52398, 6223.46125, 7486.85898, 8961.55892, 10673.4029, 12648.2326, 15817.6765, 19075.343, 22220.6273, 25052.9246, 27371.6298, 28197.0907, 28627.1149, 28794.9748, 29237.1072, 29927.199, 30670.9517, 31450.4515, 32247.7843, 33045.0361, 33824.293, 34835.1013, 35632.5503, 36224.5323, 36618.9394, 36823.6641, 36846.5983, 36695.6345, 36189.6973, 35425.4022}, 
{464.357294, 465.76943, 466.919421, 467.812177, 468.45261, 468.845633, 468.996156, 468.909091, 468.58935, 468.041844, 467.271485, 466.283184, 465.081853, 463.672404, 462.059748, 460.248797, 458.244462, 456.051654, 453.675287, 451.12027, 448.391516, 445.493937, 442.432443, 439.211946, 435.837358, 432.313591, 428.645556, 424.838165, 420.896329, 416.82496, 412.628969, 408.313269, 403.882769, 399.342384, 394.735895, 390.047961, 385.262773, 380.38398, 375.415233, 370.360182, 365.222476, 360.005766, 354.713703, 349.349935, 343.918114, 338.421889, 332.86491, 327.250828, 321.583292, 315.865953, 310.10246, 304.357293, 298.571953, 292.748774, 286.890086, 280.99822, 275.075509, 269.124283, 263.146874, 257.145614, 251.122833, 245.080864, 239.022038, 232.948687, 226.863141, 220.767733, 214.664793, 208.621062, 202.701734, 196.776303, 190.842923, 184.899751, 178.944944, 172.976657, 166.993047, 160.992269, 154.97248, 148.931836, 142.868492, 136.780605, 130.666332, 124.523828, 118.351249, 112.146751, 105.511974, 98.6516711, 91.7753646, 84.8940779, 78.0188348, 71.1606591, 64.3305748, 57.5396056, 50.7987754, 44.1191081, 37.5116274, 30.9873574, 24.5573217, 18.2325443, 12.0240489, 5.94285958, 0, -2.1047361, -4.51022423, -7.20744046, -10.1873608, -13.4409613, -16.9592181, -20.7331071, -24.7536044, -29.011686, -33.498328, -38.2045065, -43.1211975, -48.2393769, -53.5500209, -59.0441056, -64.7126069, -70.7418229, -77.150633, -83.4882131, -89.7873941, -96.0810066, -102.401882, -108.78285, -115.256742, -121.856388, -128.614621, -135.564269, -142.738165, -150.169138, -158.267542, -166.550261, -174.936868, -183.346939, -191.921151, -200.468526, -208.739128, -216.586198, -223.862981, -230.42272, -236.11866, -240.083337, -243.110045, -245.193037, -246.326563, -246.504874, -250.278918, -251.947075, -250.364422, -244.386035, -232.866991, -214.625147, -188.598292, -157.696708, -121.248107, -71.2657855, -3.04706414, 88.1107372, 212.544497, 368.042197, 546.24542, 736.071031, 956.731203, 1209.99618, 1506.14805, 1869.8917, 2319.31173, 2877.59121, 3564.13655, 4379.65977, 5335.44749, 6406.28128, 7645.56899, 9077.23361, 10725.1981, 12613.3856, 15601.2175, 18668.2437, 21629.5125, 24300.0722, 26494.9711, 27303.2112, 27749.9181, 27961.303, 28445.8995, 29178.5827, 29974.9266, 30817.7714, 31689.9573, 32574.3244, 33453.713, 34901.0846, 35926.4035, 36600.2035, 36993.0185, 37175.3823, 37217.8289, 37190.8922, 36682.5593, 35974.2226}, 
{437.858147, 439.195971, 440.298293, 441.169237, 441.812925, 442.233481, 442.435028, 442.421689, 442.197587, 441.766845, 441.133586, 440.301934, 439.276011, 438.05994, 436.657845, 435.073849, 433.312075, 431.376645, 429.271684, 427.001314, 424.569658, 421.980839, 419.238981, 416.348206, 413.312638, 410.136399, 406.823614, 403.378404, 399.804894, 396.107205, 392.289462, 388.355787, 384.310304, 380.157135, 375.923669, 371.601906, 367.183569, 362.672026, 358.070644, 353.382792, 348.611838, 343.761151, 338.834097, 333.834046, 328.764365, 323.628423, 318.429587, 313.171226, 307.856708, 302.489401, 297.072672, 291.658753, 286.20109, 280.701991, 275.163766, 269.588724, 263.979174, 258.337424, 252.665784, 246.966562, 241.242067, 235.494609, 229.726495, 223.940035, 218.137538, 212.321313, 206.493668, 200.717037, 195.052411, 189.378095, 183.6925, 177.994039, 172.281122, 166.552159, 160.805564, 155.039746, 149.253118, 143.44409, 137.611073, 131.752479, 125.86672, 119.952205, 114.007347, 108.030557, 101.629459, 95.006078, 88.3671344, 81.7237218, 75.0869331, 68.4678613, 61.8775996, 55.3272409, 48.8278782, 42.3906047, 36.0265133, 29.7466971, 23.5622491, 17.4842624, 11.5238299, 5.69204479, 0, -2.3402798, -4.8855765, -7.63332381, -10.5809554, -13.725905, -17.0656063, -20.5974929, -24.3189987, -28.2275572, -32.3206022, -36.5955674, -41.0498864, -45.680993, -50.4863209, -55.4633037, -60.6093752, -65.830628, -71.1114472, -76.655871, -82.4417602, -88.4469752, -94.6493768, -101.026826, -107.557182, -114.218307, -120.988061, -127.844305, -134.764899, -141.727704, -148.652355, -155.596288, -162.554831, -169.52331, -177.165694, -185.142991, -192.937906, -200.345176, -207.159537, -213.175726, -218.18848, -221.348209, -223.290075, -223.934878, -223.203417, -221.016493, -222.262164, -220.652158, -214.86546, -203.581056, -185.477931, -157.961595, -122.54098, -83.8708373, -42.0487781, 12.1317463, 83.6999065, 177.684873, 303.776742, 461.284458, 644.432319, 847.275094, 1082.37822, 1345.25473, 1643.5951, 2005.72375, 2450.49377, 3003.40945, 3685.04825, 4491.60002, 5433.04963, 6492.47796, 7710.91293, 9107.65235, 10701.994, 12513.2357, 15300.3743, 18158.0837, 20920.7369, 23422.7071, 25498.3671, 26329.7547, 26838.4685, 27138.4537, 27684.0408, 28454.4392, 29287.0312, 30168.1177, 31083.9995, 32020.9773, 32965.352, 34654.0494, 35859.927, 36656.7915, 37118.4495, 37318.7076, 37331.3724, 37230.2505, 36775.6961, 36514.2609}, 
{411.415967, 412.737372, 413.845687, 414.744402, 415.437004, 415.926984, 416.21783, 416.313032, 416.216078, 415.930458, 415.459661, 414.807176, 413.976491, 412.971097, 411.794483, 410.450136, 408.941547, 407.272205, 405.445598, 403.465217, 401.334549, 399.057084, 396.636312, 394.07572, 391.3788, 388.549038, 385.589926, 382.504951, 379.297603, 375.971371, 372.529744, 368.976211, 365.314262, 361.547385, 357.695326, 353.753105, 349.715544, 345.585606, 341.36625, 337.06044, 332.671137, 328.201302, 323.653897, 319.031882, 314.338221, 309.575874, 304.747804, 299.85697, 294.906336, 289.898862, 284.83751, 279.753904, 274.621722, 269.443304, 264.220991, 258.957121, 253.654037, 248.314077, 242.939583, 237.532893, 232.09635, 226.632292, 221.14306, 215.630994, 210.098435, 204.547722, 198.981196, 193.457906, 188.035539, 182.599479, 177.148388, 171.680931, 166.195771, 160.691574, 155.167002, 149.62072, 144.051391, 138.457681, 132.838251, 127.191768, 121.516894, 115.812294, 110.076632, 104.308571, 98.1196331, 91.7101958, 85.2852913, 78.8561473, 72.4339912, 66.0300505, 59.6555529, 53.3217258, 47.0397968, 40.8209934, 34.6765432, 28.6176737, 22.6556123, 16.8015868, 11.0668245, 5.46255307, 0, -2.47851376, -5.08575286, -7.82450469, -10.6975566, -13.7076961, -16.8577104, -20.1503871, -23.5885133, -27.1748767, -30.9122644, -34.803464, -38.8512628, -43.0584482, -47.4278076, -51.9621284, -56.6641979, -61.1605981, -65.4003729, -70.2372503, -75.5934023, -81.3910009, -87.552218, -93.9992256, -100.654196, -107.4393, -114.27671, -121.088599, -127.797138, -134.324499, -140.116486, -145.746308, -151.279046, -156.779782, -163.265049, -170.324203, -177.292884, -183.950738, -190.077414, -195.452559, -199.855821, -203.125303, -204.964406, -205.141342, -203.424321, -199.581553, -198.576239, -193.682852, -183.370857, -166.109717, -140.368895, -101.560775, -54.9483307, -7.38135403, 39.9279411, 97.2773971, 169.797076, 262.61704, 382.653226, 532.84394, 714.179262, 930.202385, 1183.3208, 1462.91431, 1772.96895, 2142.22021, 2588.04318, 3132.57881, 3794.43778, 4574.75592, 5484.55381, 6526.18394, 7720.66856, 9082.02879, 10624.2857, 12361.4604, 14948.0004, 17597.3938, 20163.555, 22500.3986, 24461.8391, 25327.1472, 25907.977, 26304.5187, 26909.1892, 27706.5058, 28559.2248, 29457.9517, 30393.2918, 31355.8506, 32336.2334, 34055.4998, 35362.7, 36294.3134, 36886.8194, 37176.6975, 37200.4271, 36994.4875, 36654.9242, 36769.9049}, 
{385.697795, 387.143572, 388.386933, 389.431103, 390.279304, 390.934759, 391.400691, 391.680322, 391.776877, 391.693578, 391.433648, 391.000309, 390.396786, 389.626301, 388.692076, 387.597336, 386.345302, 384.939198, 383.382247, 381.677672, 379.828696, 377.838542, 375.710433, 373.447592, 371.053241, 368.530605, 365.882905, 363.113365, 360.225208, 357.221657, 354.105934, 350.881264, 347.550868, 344.11797, 340.60649, 337.008868, 333.317292, 329.534316, 325.662489, 321.704364, 317.662491, 313.539423, 309.337709, 305.059902, 300.708553, 296.286213, 291.795434, 287.238766, 282.618762, 277.937972, 273.198948, 268.402332, 263.552625, 258.652421, 253.704312, 248.710892, 243.674752, 238.598485, 233.484684, 228.335942, 223.154852, 217.944006, 212.705996, 207.443417, 202.158859, 196.854917, 191.534182, 186.263055, 181.106403, 175.935222, 170.747968, 165.543097, 160.319066, 155.074332, 149.80735, 144.516577, 139.200469, 133.857482, 128.486074, 123.0847, 117.651817, 112.185881, 106.685348, 101.148675, 95.1681838, 88.9539403, 82.720892, 76.4806753, 70.2449263, 64.0252812, 57.8333762, 51.6808476, 45.5793315, 39.5404642, 33.5758819, 27.6972207, 21.916117, 16.2442068, 10.6931265, 5.27451212, 0, -2.26642939, -4.6600261, -7.18338767, -9.83911159, -12.6297954, -15.5580366, -18.6264326, -21.8375811, -25.1940795, -28.6985253, -32.3535161, -36.1616494, -40.1255226, -44.2477333, -48.530879, -52.9775572, -57.0837952, -60.7798233, -65.2140527, -70.2805301, -75.8733022, -81.8864159, -88.2139178, -94.7498547, -101.388273, -108.023221, -114.548743, -120.858887, -126.847701, -131.753582, -136.366629, -140.777584, -145.077187, -150.306189, -156.080325, -161.761998, -167.156944, -172.070903, -176.309612, -179.678811, -182.931706, -184.638206, -184.418676, -181.893479, -176.68298, -173.169087, -165.020232, -150.666396, -128.537556, -97.0636905, -50.793947, 3.21762929, 56.7141831, 107.87411, 168.014443, 242.553234, 336.908537, 457.813714, 609.072574, 793.054043, 1015.10772, 1276.28871, 1568.76211, 1895.88186, 2279.96082, 2734.60937, 3274.08299, 3912.15933, 4660.25077, 5531.10759, 6546.49251, 7708.7569, 9027.53141, 10512.4467, 12173.1333, 14573.2567, 17029.904, 19414.1972, 21597.2583, 23450.2091, 24339.0305, 24976.7464, 25450.9676, 26098.2487, 26907.6605, 27764.5475, 28663.1668, 29597.7757, 30562.6312, 31551.9906, 33022.2098, 34303.5459, 35361.1082, 36160.0057, 36665.3477, 36842.2433, 36655.8017, 36513.6104, 36449.3586}, 
{360.792502, 362.645892, 364.284319, 365.711451, 366.930956, 367.946504, 368.761763, 369.380401, 369.806087, 370.04249, 370.093278, 369.96212, 369.652685, 369.16864, 368.513655, 367.691399, 366.705539, 365.559744, 364.257684, 362.803026, 361.199439, 359.450592, 357.560154, 355.531792, 353.369176, 351.075974, 348.655855, 346.112487, 343.449539, 340.670679, 337.779577, 334.7799, 331.675318, 328.469498, 325.210449, 321.878737, 318.454396, 314.939654, 311.336743, 307.647891, 303.875329, 300.021285, 296.087991, 292.077674, 287.992566, 283.834896, 279.606893, 275.310787, 270.948809, 266.523187, 262.036151, 257.44363, 252.795159, 248.093971, 243.343299, 238.546377, 233.706437, 228.826715, 223.910442, 218.960853, 213.981181, 208.974658, 203.94452, 198.893998, 193.826327, 188.74474, 183.652471, 178.643534, 173.809768, 168.967173, 164.113096, 159.244888, 154.359895, 149.455466, 144.52895, 139.577694, 134.599048, 129.59036, 124.548977, 119.472249, 114.357523, 109.202148, 104.003472, 98.7588443, 92.9975336, 86.9607732, 80.8954223, 74.8140191, 68.7291022, 62.6532098, 56.5988805, 50.5786526, 44.6050644, 38.6906545, 32.8479612, 27.0895228, 21.4278779, 15.8755647, 10.4451218, 5.14908739, 0, -1.42522145, -3.11027967, -5.04737492, -7.22870745, -9.64647752, -12.2928854, -15.1601313, -18.2404155, -21.5259383, -25.0088998, -28.6815005, -32.5359405, -36.56442, -40.7591394, -45.1122988, -49.6160986, -53.932449, -58.006366, -62.5771343, -67.5661778, -72.8949202, -78.4847854, -84.2571973, -90.1335797, -96.0353563, -101.883951, -107.600788, -113.10729, -118.324883, -122.802721, -126.970995, -130.862808, -134.511264, -138.465766, -142.501269, -146.255048, -149.605315, -152.430286, -154.608175, -156.017195, -158.212727, -158.885377, -157.585219, -153.862325, -147.266768, -140.622952, -129.388037, -112.293513, -88.0708719, -55.4516033, -10.7047396, 41.9660987, 94.6485637, 146.131835, 208.887377, 289.243588, 393.528867, 535.275075, 711.969995, 913.243084, 1127.54014, 1374.81183, 1667.64469, 2015.72326, 2423.81483, 2898.94377, 3442.83401, 4061.13575, 4778.93416, 5610.32089, 6588.99019, 7706.41533, 8969.90247, 10386.7577, 11964.2873, 14202.7403, 16493.2441, 18719.8692, 20766.6858, 22517.7644, 23404.3329, 24065.1991, 24578.5088, 25241.4311, 26045.415, 26890.6496, 27773.147, 28688.919, 29633.9776, 30604.3348, 31440.4484, 32509.8279, 33670.3188, 34779.7667, 35696.0171, 36276.9157, 36380.3078, 36550.0677, 35250.402}, 
{339.029877, 341.011679, 342.78719, 344.359732, 345.732627, 346.909198, 347.892767, 348.686656, 349.294188, 349.718685, 349.963469, 350.031863, 349.927189, 349.652769, 349.211926, 348.607982, 347.844259, 346.92408, 345.850767, 344.627642, 343.258028, 341.745248, 340.092622, 338.303474, 336.381127, 334.328901, 332.150121, 329.848107, 327.426183, 324.88767, 322.235892, 319.47417, 316.605827, 313.634185, 310.586385, 307.453339, 304.225673, 300.905936, 297.496678, 294.000447, 290.419794, 286.757268, 283.015418, 279.196793, 275.303944, 271.339418, 267.305766, 263.205538, 259.041281, 254.815547, 250.530883, 246.158877, 241.733712, 237.258609, 232.736788, 228.17147, 223.565875, 218.923225, 214.246739, 209.539639, 204.805145, 200.046477, 195.266857, 190.469504, 185.657639, 180.834484, 176.003258, 171.266925, 166.726053, 162.178151, 157.619975, 153.048277, 148.459814, 143.85134, 139.219609, 134.561377, 129.873398, 125.152427, 120.395217, 115.598525, 110.759105, 105.873711, 100.939099, 95.9520219, 90.4062341, 84.5605706, 78.6799127, 72.7773384, 66.8659258, 60.9587529, 55.0688979, 49.2094387, 43.3934534, 37.6340201, 31.9442168, 26.3371217, 20.8258128, 15.423368, 10.1428656, 4.99738361, 0, -1.05682854, -2.41483982, -4.06239508, -5.98785556, -8.17958248, -10.6259371, -13.3152806, -16.2359743, -19.3763793, -22.7248569, -26.2697684, -29.999475, -33.9023379, -37.9667184, -42.1809776, -46.5334768, -50.7594109, -54.8109746, -59.2491671, -64.0080999, -69.0218842, -74.2246313, -79.5504526, -84.9334592, -90.3077626, -95.6074741, -100.766705, -105.719566, -110.400169, -114.545546, -118.35915, -121.834216, -124.96398, -128.017911, -130.851126, -133.245082, -135.110143, -136.356674, -136.895039, -136.635605, -137.594589, -136.93559, -134.156958, -128.757042, -120.234192, -109.136714, -93.6505132, -73.011449, -46.4553821, -13.2181731, 27.1620079, 76.1210997, 130.020007, 189.312767, 262.810232, 355.571212, 472.654522, 627.474304, 814.8351, 1020.42655, 1223.75052, 1455.98384, 1741.30257, 2093.58319, 2509.25026, 2992.73906, 3543.1994, 4163.69626, 4876.67464, 5693.61709, 6638.07482, 7707.60503, 8909.51268, 10251.1027, 11739.6801, 13836.8937, 15982.1185, 18069.0737, 19991.4781, 21643.0508, 22497.5413, 23148.6177, 23669.986, 24351.2998, 25180.7858, 26056.6924, 26971.7454, 27918.6702, 28890.1925, 29879.0379, 30315.4016, 31227.7763, 32416.9161, 33683.5751, 34828.5074, 35652.4671, 35956.2083, 36262.5984, 34995.6824}, 
{320.562039, 322.378136, 324.017819, 325.48331, 326.776836, 327.900622, 328.856891, 329.64787, 330.275783, 330.742855, 331.051311, 331.203377, 331.201276, 331.047234, 330.743476, 330.292227, 329.695712, 328.956156, 328.075783, 327.05682, 325.901489, 324.612018, 323.19063, 321.63955, 319.961004, 318.157217, 316.230413, 314.182817, 312.016654, 309.73415, 307.337529, 304.829017, 302.210837, 299.485216, 296.618757, 293.632245, 290.546893, 287.366082, 284.093192, 280.731604, 277.284699, 273.755858, 270.148463, 266.465893, 262.71153, 258.888754, 255.000947, 251.05149, 247.043763, 242.981148, 238.867024, 234.745083, 230.577522, 226.366848, 222.115568, 217.82619, 213.50122, 209.143166, 204.754535, 200.337833, 195.895568, 191.430247, 186.944378, 182.440466, 177.921019, 173.388545, 168.845549, 164.381254, 160.082797, 155.773846, 151.451286, 147.112004, 142.752884, 138.370813, 133.962676, 129.52536, 125.055749, 120.550731, 116.00719, 111.422013, 106.792085, 102.114292, 97.38552, 92.6026546, 87.2615773, 81.620214, 75.9425123, 70.2416167, 64.5306714, 58.8228208, 53.1312093, 47.4689812, 41.8492808, 36.2852526, 30.7900408, 25.3767899, 20.0586442, 14.8487479, 9.7602456, 4.80628151, 0, -1.25865788, -2.74733074, -4.45867733, -6.38535641, -8.52002674, -10.8553471, -13.3839761, -16.0985727, -18.9917956, -22.0563034, -25.284755, -28.6698092, -32.2041246, -35.8803601, -39.6911743, -43.6292261, -47.3673679, -50.8525718, -54.8008681, -59.1363854, -63.7832526, -68.6655985, -73.7075521, -78.8332421, -83.9667974, -89.0323468, -93.9540192, -98.6559434, -103.062248, -106.889545, -110.34557, -113.416707, -116.08934, -118.673648, -120.994117, -122.78889, -123.947213, -124.358333, -123.911495, -122.495946, -122.238953, -120.110604, -115.56227, -108.045326, -97.0111416, -80.1459977, -59.1076322, -33.7886906, -4.0818185, 30.1203387, 64.9566073, 109.353293, 167.294534, 241.670158, 333.13578, 443.350131, 573.971945, 730.372521, 909.654256, 1104.86947, 1298.41997, 1520.50051, 1793.95468, 2133.05623, 2537.55777, 3013.87736, 3569.15741, 4210.00376, 4940.36632, 5765.69733, 6681.45652, 7704.62599, 8844.11694, 10108.8406, 11507.7081, 13482.4975, 15501.0364, 17464.0196, 19272.1416, 20826.0968, 21622.6815, 22235.7542, 22739.1087, 23442.6505, 24326.7126, 25273.2482, 26267.509, 27294.7471, 28340.2146, 29389.1632, 29763.017, 30625.4363, 31779.0066, 33026.3136, 34169.9429, 35012.4801, 35356.5108, 35593.5176, 35830.0846}, 
{302.36832, 304.392341, 306.233512, 307.894358, 309.377403, 310.685174, 311.820196, 312.784993, 313.582091, 314.214016, 314.683292, 314.992445, 315.144, 315.140482, 314.984417, 314.67833, 314.224746, 313.626191, 312.885189, 312.004266, 310.985947, 309.832758, 308.547224, 307.131869, 305.589219, 303.921801, 302.132137, 300.222755, 298.196179, 296.054935, 293.801547, 291.438542, 288.968443, 286.393778, 283.69266, 280.880333, 277.972335, 274.971983, 271.882595, 268.707487, 265.449978, 262.113384, 258.701023, 255.216213, 251.662271, 248.042515, 244.360261, 240.618827, 236.821531, 232.97169, 229.072621, 225.17931, 221.242287, 217.263748, 213.24589, 209.190913, 205.101011, 200.978384, 196.825229, 192.643742, 188.436121, 184.204564, 179.951267, 175.678429, 171.388247, 167.082917, 162.764638, 158.515332, 154.415008, 150.301435, 146.171643, 142.02266, 137.851516, 133.65524, 129.43086, 125.175406, 120.885907, 116.559392, 112.19289, 107.78343, 103.328042, 98.8237532, 94.2675942, 89.6565936, 84.4872112, 79.0172895, 73.5106876, 67.9806789, 62.4405369, 56.9035353, 51.3829474, 45.8920468, 40.444107, 35.0524016, 29.730204, 24.4907878, 19.3474264, 14.3133934, 9.40196237, 4.62640672, 0, -1.103136, -2.43594238, -3.98998886, -5.7568452, -7.72808112, -9.89526638, -12.2499707, -14.7837638, -17.4882155, -20.3548954, -23.3753734, -26.5412191, -29.8440023, -33.2752928, -36.8266602, -40.4896743, -44.0108794, -47.3468415, -51.0433547, -55.0394834, -59.2742917, -63.686844, -68.2162045, -72.8014376, -77.3816074, -81.8957782, -86.2830143, -90.4823801, -94.4329397, -98.0050626, -101.231696, -104.072513, -106.487185, -108.694025, -110.523386, -111.73665, -112.2159, -111.843215, -110.500677, -108.070367, -107.097657, -103.99077, -98.1107094, -88.8184761, -75.4750728, -52.1840902, -24.878295, 5.76695728, 39.0763116, 74.3744126, 103.51234, 142.422661, 200.640423, 283.053538, 385.062022, 506.325764, 646.504651, 806.237369, 983.982553, 1177.13107, 1370.51428, 1592.2116, 1863.99003, 2199.49113, 2598.58827, 3067.47465, 3613.83098, 4244.23609, 4959.81466, 5764.77652, 6653.09953, 7640.7997, 8735.62874, 9945.33835, 11277.6803, 13141.0972, 15042.477, 16889.3988, 18589.4413, 20050.1836, 20794.3224, 21370.907, 21852.466, 22552.496, 23448.1424, 24416.147, 25438.6752, 26497.8923, 27575.964, 28655.0555, 28931.9225, 29772.0578, 30952.4142, 32249.9443, 33441.601, 34304.337, 34615.1051, 34806.9226, 36084.8979}, 
{284.82981, 287.349992, 289.651822, 291.739263, 293.616282, 295.286842, 296.754909, 298.024448, 299.099424, 299.983801, 300.681546, 301.196622, 301.532994, 301.694629, 301.685489, 301.509542, 301.170751, 300.673081, 300.020499, 299.216967, 298.266452, 297.172918, 295.940331, 294.572655, 293.073856, 291.447897, 289.698745, 287.830364, 285.84672, 283.751776, 281.549499, 279.243853, 276.838802, 274.338313, 271.793846, 269.184618, 266.489277, 263.710246, 260.849951, 257.910813, 254.895257, 251.805706, 248.644584, 245.414315, 242.117323, 238.75603, 235.332861, 231.850239, 228.310588, 224.716332, 221.069894, 217.376358, 213.63543, 209.849477, 206.020862, 202.151954, 198.245118, 194.302719, 190.327124, 186.320699, 182.28581, 178.224822, 174.140102, 170.034016, 165.908929, 161.767208, 157.611219, 153.525664, 149.59327, 145.646589, 141.682649, 137.698479, 133.691107, 129.657561, 125.59487, 121.500062, 117.370166, 113.20221, 108.993222, 104.740231, 100.440265, 96.0903535, 91.6875237, 87.2288042, 82.2018232, 76.8690238, 71.4989717, 66.1052265, 60.7013475, 55.3008942, 49.917426, 44.5645024, 39.2556827, 34.0045265, 28.824593, 23.7294419, 18.7326324, 13.847724, 9.08827622, 4.4678484, 0, -0.565700319, -1.43582217, -2.59533698, -4.02921618, -5.7224312, -7.65995348, -9.82675444, -12.2078055, -14.7880781, -17.5525438, -20.4861738, -23.5739396, -26.8008128, -30.1517646, -33.6117666, -37.1657901, -40.7449625, -44.326563, -48.0173537, -51.7907678, -55.6202387, -59.4791999, -63.3410848, -67.1793266, -70.9673589, -74.678615, -78.2865283, -81.7645322, -85.0860601, -88.4098803, -91.4561351, -94.1426574, -96.3872801, -98.1992593, -99.4507164, -99.9893929, -99.7056948, -98.4900279, -96.2327982, -92.8244114, -91.4190588, -87.650035, -80.7691792, -70.0283308, -54.6793288, -25.4561947, 7.74195997, 43.5338417, 80.5381568, 117.373612, 142.688939, 177.258342, 232.843359, 315.431144, 419.219481, 543.699341, 688.361695, 853.659247, 1037.90266, 1238.35342, 1440.41069, 1670.46481, 1950.94535, 2294.37438, 2696.48632, 3160.72141, 3687.47836, 4279.40886, 4950.3168, 5707.69775, 6568.69402, 7529.67032, 8594.78296, 9768.18822, 11054.0424, 12814.5294, 14606.271, 16343.9167, 17942.116, 19315.5183, 20018.2436, 20565.8238, 21027.4209, 21699.445, 22561.1059, 23496.9267, 24490.8516, 25526.8247, 26588.7901, 27660.6921, 27780.7101, 28611.6454, 29877.2318, 31301.2034, 32607.294, 33519.2376, 33760.7681, 33987.5563, 35608.8858}, 
{271.159829, 273.792753, 276.199922, 278.385782, 280.354777, 282.111353, 283.659955, 285.005028, 286.151017, 287.102367, 287.863524, 288.438933, 288.833038, 289.050285, 289.09512, 288.971986, 288.68533, 288.239596, 287.63923, 286.888677, 285.992382, 284.954789, 283.780345, 282.473494, 281.038682, 279.480353, 277.802953, 276.010927, 274.10872, 272.100777, 269.991544, 267.785465, 265.486985, 263.100551, 260.712741, 258.285207, 255.77861, 253.194731, 250.535348, 247.802243, 244.997195, 242.121982, 239.178386, 236.168185, 233.093159, 229.955089, 226.755753, 223.496932, 220.180405, 216.807951, 213.381351, 209.868231, 206.305264, 202.69497, 199.03987, 195.342483, 191.60533, 187.83093, 184.021805, 180.180473, 176.309455, 172.411271, 168.488442, 164.543486, 160.578925, 156.597278, 152.601066, 148.679629, 144.920224, 141.14631, 137.354778, 133.54252, 129.706428, 125.843394, 121.950309, 118.024065, 114.061555, 110.059669, 106.015301, 101.925341, 97.7866811, 93.5962138, 89.3508307, 85.0474234, 80.1577339, 74.9522748, 69.7078707, 64.4384555, 59.1579632, 53.8803277, 48.619483, 43.389363, 38.2039018, 33.0770332, 28.0226913, 23.0548099, 18.187323, 13.4341646, 8.80926872, 4.32656918, 0, -0.161909691, -0.673867322, -1.51700983, -2.67247413, -4.12139718, -5.84491589, -7.8241672, -10.040288, -12.4744154, -15.1076861, -17.9212371, -20.8962054, -24.0137279, -27.2549415, -30.6009831, -34.0329897, -37.5306372, -41.0748536, -44.6500809, -48.2371429, -51.8168634, -55.3700662, -58.8775752, -62.3202142, -65.678807, -68.9341776, -72.0671496, -75.0585471, -77.8891937, -80.7784084, -83.3810717, -85.6064592, -87.363846, -88.6853536, -89.4188344, -89.3793816, -88.4394168, -86.4713615, -83.3476373, -78.9406655, -76.4182392, -71.3544693, -62.9770309, -50.5135988, -33.191848, -1.34202777, 34.6864048, 73.4414184, 113.470982, 153.323063, 181.836478, 219.135092, 275.738326, 357.277885, 459.406373, 582.256808, 725.962206, 891.656161, 1078.24371, 1283.53835, 1495.70511, 1737.09214, 2028.2605, 2379.8556, 2785.72124, 3247.41346, 3760.69745, 4325.62789, 4963.49251, 5683.56844, 6516.32026, 7446.57802, 8476.28472, 9607.38336, 10841.8169, 12506.1446, 14196.5392, 15833.7897, 17338.6849, 18632.0139, 19296.4722, 19816.3377, 20257.2965, 20896.7989, 21716.7076, 22610.6503, 23564.9014, 24565.7355, 25599.427, 26652.2505, 26722.4366, 27566.725, 28882.9907, 30369.1087, 31722.9538, 32642.4012, 32825.3257, 33174.344, 35042.4127}, 
{261.977625, 264.169001, 266.170536, 267.985625, 269.617661, 271.070037, 272.346149, 273.449389, 274.383151, 275.150829, 275.755817, 276.201509, 276.491298, 276.628578, 276.616744, 276.459188, 276.159305, 275.720489, 275.146133, 274.439631, 273.604377, 272.643764, 271.561187, 270.360039, 269.043715, 267.615607, 266.07911, 264.437617, 262.694523, 260.85322, 258.917104, 256.889567, 254.774004, 252.573808, 250.343943, 248.060934, 245.700684, 243.264914, 240.755344, 238.173694, 235.521685, 232.801036, 230.013467, 227.1607, 224.244454, 221.26645, 218.228407, 215.132047, 211.979088, 208.771253, 205.510259, 202.16144, 198.763692, 195.319525, 191.831449, 188.301972, 184.733604, 181.128852, 177.490228, 173.820239, 170.121395, 166.396205, 162.647178, 158.876823, 155.087649, 151.282165, 147.462881, 143.723304, 140.154759, 136.572075, 132.971863, 129.350731, 125.70529, 122.03215, 118.327922, 114.589214, 110.812637, 106.994801, 103.132316, 99.2217917, 95.2598383, 91.2430657, 87.1680838, 83.0315027, 78.2825054, 73.2029296, 68.081193, 62.9316705, 57.7687371, 52.6067679, 47.4601379, 42.3432221, 37.2703956, 32.2560333, 27.3145104, 22.4602018, 17.7074826, 13.0707278, 8.56431237, 4.20261144, 0, -0.0316074413, -0.402696696, -1.09541604, -2.09191375, -3.37433811, -4.92483739, -6.72555988, -8.75865384, -11.0062676, -13.4505493, -16.0736474, -18.85771, -21.7848855, -24.8373222, -27.9971682, -31.246572, -34.4235534, -37.4896709, -40.7530774, -44.1650366, -47.6768121, -51.2396675, -54.8048664, -58.3236725, -61.7473494, -65.0271607, -68.1143701, -70.9602412, -73.5160376, -75.7631197, -77.6116189, -79.0037698, -79.881807, -80.5893009, -80.8678181, -80.3519026, -78.8633883, -76.2241091, -72.2558989, -66.7805915, -62.2181573, -55.0015567, -44.4442923, -29.8598671, -10.561784, 20.0108439, 55.2015266, 94.2281639, 136.308656, 180.660902, 220.003387, 267.997155, 330.882828, 412.510114, 511.847264, 630.054206, 768.290866, 928.139942, 1110.2425, 1314.77841, 1535.29984, 1787.81953, 2087.96457, 2443.98654, 2851.63963, 3312.41455, 3823.09126, 4383.93916, 5012.50037, 5716.54657, 6519.82948, 7411.68232, 8394.34523, 9470.05836, 10641.0619, 12215.7301, 13813.6355, 15360.4846, 16781.9839, 18003.8402, 18632.2926, 19125.4932, 19546.0634, 20156.224, 20939.588, 21796.6017, 22715.0364, 23682.6632, 24687.2534, 25716.5783, 25872.1793, 26745.6933, 28050.3704, 29499.4612, 30806.2159, 31683.8849, 31845.7186, 32405.7134, 34568.1633}, 
{254.546663, 256.181573, 257.670736, 259.016201, 260.220018, 261.284235, 262.210902, 263.002067, 263.659781, 264.186092, 264.583049, 264.852701, 264.997098, 265.018289, 264.918323, 264.699249, 264.363116, 263.911973, 263.34787, 262.672856, 261.88898, 260.99829, 260.002837, 258.904668, 257.705835, 256.408385, 255.014367, 253.525832, 251.944827, 250.273403, 248.513608, 246.667492, 244.737103, 242.724491, 240.639151, 238.479252, 236.242875, 233.931827, 231.547916, 229.092948, 226.568732, 223.977076, 221.319786, 218.59867, 215.815536, 212.972192, 210.070444, 207.1121, 204.098969, 201.032856, 197.915571, 194.719704, 191.476912, 188.189637, 184.860318, 181.491398, 178.085316, 174.644514, 171.171432, 167.668511, 164.138193, 160.582918, 157.005126, 153.40726, 149.791759, 146.161065, 142.517618, 138.958756, 135.579538, 132.186688, 128.776495, 125.345247, 121.889234, 118.404746, 114.88807, 111.335497, 107.743315, 104.107813, 100.425281, 96.6920081, 92.9042824, 89.0583935, 85.1506306, 81.1772826, 76.5638013, 71.6039011, 66.5981579, 61.5613855, 56.5083978, 51.4540086, 46.4130319, 41.4002815, 36.4305712, 31.5187149, 26.6795266, 21.9278201, 17.2784093, 12.746108, 8.34573006, 4.09208945, 0, 0.047753574, -0.226868078, -0.807500993, -1.6777812, -2.82134475, -4.22182766, -5.86286597, -7.72809572, -9.80115295, -12.0656737, -14.505294, -17.1036498, -19.8443773, -22.7111124, -25.6874912, -28.7571497, -31.6106484, -34.1897554, -37.1410157, -40.3852635, -43.8433329, -47.4360581, -51.0842731, -54.7088121, -58.2305092, -61.5701985, -64.6487142, -67.3868903, -69.7055611, -71.3523578, -72.4848251, -73.0757578, -73.097951, -73.1870857, -72.9845137, -71.9548174, -69.8719259, -66.5097685, -61.6422741, -55.043372, -48.2659648, -38.7635814, -25.9620909, -9.2873625, 11.8347347, 40.2113274, 73.6253017, 112.09254, 155.628923, 204.250335, 255.049333, 314.538073, 384.47205, 466.792776, 563.821545, 677.709136, 810.606329, 964.556589, 1141.8424, 1344.86331, 1572.30762, 1834.11367, 2140.61305, 2497.80338, 2904.3386, 3362.2435, 3870.84993, 4431.48451, 5055.34811, 5748.05616, 6522.13931, 7375.2339, 8310.56166, 9331.34429, 10440.8035, 11936.2009, 13453.2084, 14921.5376, 16270.9003, 17431.0082, 18028.3506, 18498.0099, 18899.6504, 19480.5214, 20226.0346, 21043.4413, 21922.0611, 22851.2137, 23820.2188, 24818.3962, 25082.4407, 25979.5936, 27250.1902, 28634.5659, 29873.056, 30705.9958, 30873.7208, 31675.3242, 34054.5795}, 
{245.454061, 246.896676, 248.208917, 249.392463, 250.448989, 251.380172, 252.18769, 252.873218, 253.438434, 253.885014, 254.214635, 254.428973, 254.529705, 254.518509, 254.397059, 254.167034, 253.83011, 253.387964, 252.842272, 252.194711, 251.446958, 250.600689, 249.657581, 248.619311, 247.487555, 246.26399, 244.950294, 243.548142, 242.059211, 240.485178, 238.827719, 237.088512, 235.269233, 233.371558, 231.395327, 229.343172, 227.217752, 225.020802, 222.75406, 220.41926, 218.01814, 215.552436, 213.023885, 210.434222, 207.785184, 205.078507, 202.315928, 199.499183, 196.630009, 193.710142, 190.741317, 187.698711, 184.611197, 181.481087, 178.310693, 175.102327, 171.858301, 168.580927, 165.272518, 161.935385, 158.571842, 155.184199, 151.774769, 148.345864, 144.899797, 141.438879, 137.965423, 134.578089, 131.373226, 128.154434, 124.917777, 121.659323, 118.375138, 115.061288, 111.71384, 108.328859, 104.902411, 101.430564, 97.909383, 94.3349344, 90.7032844, 87.0104993, 83.2526453, 79.4257888, 74.9389844, 70.0941511, 65.2002633, 60.2724368, 55.3257876, 50.3754315, 45.4364843, 40.524062, 35.6532805, 30.8392555, 26.0971031, 21.441939, 16.8888792, 12.4530395, 8.14953584, 3.99348404, 0, 0.29762241, 0.240935189, -0.150447554, -0.856911713, -1.85884318, -3.13662784, -4.6706516, -6.44130033, -8.42895994, -10.6140163, -12.9768554, -15.4978629, -18.157425, -20.9359273, -23.8137559, -26.7712966, -29.6162352, -32.3042862, -35.2064666, -38.2661551, -41.4267306, -44.6315716, -47.824057, -50.9475655, -53.9454757, -56.7611665, -59.3380165, -61.6194045, -63.5487092, -65.124572, -66.214846, -66.746331, -66.645827, -66.3115854, -65.4346803, -63.5804656, -60.5343055, -56.0815647, -50.0076076, -42.0977985, -33.7532939, -22.6519035, -8.2628588, 9.94460909, 32.5012688, 61.6358441, 95.75666, 134.969996, 179.382133, 229.099351, 281.335707, 342.624641, 415.138833, 500.928072, 601.791245, 719.639887, 856.385538, 1013.64367, 1193.68918, 1399.1199, 1629.48137, 1892.7702, 2196.82972, 2546.73706, 2942.88212, 3387.80633, 3882.23293, 4428.23198, 5034.54014, 5706.12304, 6452.46105, 7273.31052, 8172.07431, 9152.15528, 10216.9563, 11652.6924, 13109.2513, 14519.3326, 15815.6361, 16930.8618, 17508.2897, 17962.9857, 18351.6864, 18906.1696, 19613.885, 20389.3484, 21223.6562, 22107.9046, 23033.1899, 23990.6085, 24226.8433, 25094.4282, 26339.6418, 27708.7632, 28948.0713, 29803.8447, 30022.3626, 31021.0588, 33460.7348}, 
{236.525754, 237.889014, 239.127872, 240.243946, 241.238852, 242.114208, 242.871632, 243.512739, 244.039148, 244.452476, 244.75434, 244.946358, 245.030146, 245.007322, 244.879504, 244.648307, 244.315351, 243.882251, 243.350626, 242.722092, 241.998268, 241.180769, 240.271213, 239.271218, 238.182401, 237.006379, 235.744769, 234.399189, 232.971256, 231.462587, 229.874799, 228.209511, 226.468338, 224.652898, 222.764176, 220.804118, 218.774679, 216.677498, 214.514211, 212.286457, 209.995874, 207.644099, 205.23277, 202.763524, 200.238001, 197.657837, 195.02467, 192.340139, 189.60588, 186.823532, 183.994733, 181.097402, 178.157409, 175.176906, 172.158045, 169.102979, 166.013859, 162.892837, 159.742066, 156.563696, 153.359881, 150.132772, 146.884521, 143.61728, 140.333202, 137.034437, 133.723139, 130.498241, 127.456355, 124.400025, 121.32513, 118.227548, 115.103156, 111.947833, 108.757455, 105.527903, 102.255052, 98.934782, 95.5629699, 92.135494, 88.6482322, 85.0970624, 81.4778627, 77.7865108, 73.4182409, 68.6818865, 63.8935007, 59.0684534, 54.2221147, 49.3698546, 44.5270431, 39.7090501, 34.9312457, 30.2089999, 25.5576827, 20.992664, 16.529314, 12.1830026, 7.96909978, 3.90297558, 0, 0.571658978, 0.746766781, 0.549284214, 0.00317208194, -0.867608809, -2.03909765, -3.48733365, -5.18835598, -7.11820385, -9.25291646, -11.568533, -14.0410926, -16.6466346, -19.3611981, -22.1608223, -25.0215463, -27.9295075, -30.8621875, -33.7727848, -36.6395026, -39.4405438, -42.1541115, -44.7584088, -47.2316387, -49.5520042, -51.6977084, -53.6469543, -55.3779451, -56.8688837, -58.4780274, -59.6641717, -60.2915036, -60.2242097, -59.4964365, -57.8873908, -55.1309572, -51.0403344, -45.4287214, -38.109317, -28.8953202, -19.2453723, -6.82644286, 8.87020365, 28.3533028, 52.13159, 82.814653, 118.285162, 158.52664, 203.522609, 253.256591, 303.780603, 363.814848, 437.423416, 527.373006, 633.781484, 757.955988, 901.203656, 1064.60403, 1249.74358, 1458.45702, 1689.29705, 1950.25353, 2248.10375, 2587.67207, 2970.47385, 3399.54331, 3876.63714, 4404.45837, 4990.39439, 5639.18286, 6359.39681, 7151.34845, 8018.44778, 8964.10481, 9991.72955, 11378.1425, 12784.9906, 14147.3311, 15400.2216, 16478.7194, 17040.3419, 17482.7131, 17859.836, 18388.795, 19059.0402, 19792.0708, 20580.7345, 21417.8788, 22296.3512, 23208.9992, 23391.2286, 24213.3887, 25426.9879, 26783.5346, 28034.537, 28931.5035, 29225.9424, 30408.5193, 32825.1774}, 
{228.252986, 229.584667, 230.793508, 231.881211, 232.849477, 233.700009, 234.434509, 235.054678, 235.562219, 235.958834, 236.246225, 236.426094, 236.500144, 236.470075, 236.337591, 236.104393, 235.772183, 235.342664, 234.817537, 234.198505, 233.487269, 232.685533, 231.794997, 230.817364, 229.754335, 228.607614, 227.378902, 226.069901, 224.682313, 223.217841, 221.678186, 220.06505, 218.380135, 216.625145, 214.807026, 212.924748, 210.977216, 208.965962, 206.892517, 204.758414, 202.565184, 200.314359, 198.007471, 195.646051, 193.231632, 190.765745, 188.249922, 185.685694, 183.074595, 180.418155, 177.717906, 174.954649, 172.151096, 169.309229, 166.431028, 163.518476, 160.573552, 157.598239, 154.594518, 151.564369, 148.509774, 145.432714, 142.33517, 139.219124, 136.086557, 132.93945, 129.779784, 126.706251, 123.815218, 120.909211, 117.983942, 115.035123, 112.058465, 109.04968, 106.004481, 102.918579, 99.7876864, 96.6075149, 93.3737763, 90.0821827, 86.728446, 83.308278, 79.8173907, 76.251496, 71.9932038, 67.3576728, 62.6674307, 57.9380858, 53.1852463, 48.4245204, 43.6715165, 38.9418428, 34.2511075, 29.614919, 25.0488854, 20.568615, 16.189716, 11.9277968, 7.79846555, 3.81733053, 0, 0.81722445, 1.20005428, 1.17650835, 0.774605507, 0.0223646175, -1.05219546, -2.42105588, -4.05619778, -5.92960229, -8.01325057, -10.2791238, -12.699203, -15.2454694, -17.8899042, -20.6044884, -23.3612033, -26.3033752, -29.4274633, -32.3178819, -34.9833289, -37.4325024, -39.6741004, -41.7168208, -43.5693617, -45.2404211, -46.7386969, -48.0728871, -49.2516899, -50.283803, -51.8378183, -53.020579, -53.642815, -53.5152563, -52.3515126, -50.0108737, -46.3785286, -41.2943432, -34.5981835, -26.1299155, -15.7294051, -4.95747825, 8.59072936, 25.4120613, 46.003361, 70.8614721, 103.257896, 140.221153, 181.554424, 227.060887, 276.543722, 324.612594, 382.611824, 456.717944, 550.51992, 662.323779, 792.807479, 942.648979, 1112.63642, 1303.31254, 1515.09987, 1744.49196, 2000.75555, 2290.79918, 2619.6703, 2989.26278, 3402.91801, 3862.9789, 4372.52798, 4938.30897, 5564.99459, 6261.13165, 7026.92282, 7865.69981, 8780.79435, 9775.53815, 11118.4059, 12481.3005, 13801.2682, 15015.3547, 16060.6062, 16606.8562, 17036.5047, 17401.1173, 17904.6226, 18538.2759, 19229.8476, 19973.767, 20764.4634, 21596.3663, 22463.9048, 22607.371, 23383.9523, 24556.526, 25887.9697, 27141.1605, 28078.9761, 28464.2938, 29821.7223, 32166.208}, 
{221.842616, 223.052753, 224.147412, 225.128185, 225.996662, 226.754433, 227.403087, 227.944216, 228.37941, 228.710259, 228.938352, 229.065281, 229.092636, 229.022007, 228.854984, 228.593157, 228.238117, 227.791454, 227.254759, 226.62962, 225.91763, 225.120377, 224.239453, 223.276448, 222.232951, 221.110553, 219.910845, 218.635416, 217.285857, 215.863758, 214.370709, 212.808301, 211.178124, 209.481768, 207.725446, 205.90834, 204.029575, 202.090593, 200.092833, 198.037736, 195.926742, 193.761291, 191.542823, 189.272779, 186.952599, 184.583722, 182.16759, 179.705643, 177.199321, 174.650063, 172.059311, 169.409304, 166.721098, 163.996552, 161.23752, 158.44586, 155.623428, 152.77208, 149.893673, 146.990063, 144.063108, 141.114663, 138.146584, 135.160729, 132.158954, 129.143115, 126.115068, 123.173811, 120.416005, 117.643165, 114.850851, 112.034621, 109.190035, 106.312652, 103.398033, 100.441735, 97.4393185, 94.3863428, 91.2783671, 88.1109507, 84.879653, 81.5800334, 78.2076511, 74.7580655, 70.6000068, 66.0556333, 61.4546376, 56.8129208, 52.146384, 47.4709283, 42.8024549, 38.156865, 33.5500597, 28.9979401, 24.5164074, 20.1213627, 15.8287072, 11.6543421, 7.61416838, 3.72408731, 0, 0.800786563, 1.19221611, 1.20097552, 0.853751658, 0.177231405, -0.801898366, -2.05695078, -3.56123897, -5.28807605, -7.21077515, -9.3026494, -11.5370119, -13.8871758, -16.3264543, -18.8281604, -21.3656073, -23.9286862, -26.4930788, -28.9945996, -31.4101145, -33.7164889, -35.8905884, -37.9092787, -39.7494253, -41.3878939, -42.80155, -43.9672593, -44.8618873, -45.4622996, -46.1362935, -46.3264613, -45.8923892, -44.693663, -42.7788991, -39.9131684, -35.811134, -30.275673, -23.1096625, -14.1159797, -3.09750155, 8.78881802, 23.506289, 41.5169625, 63.2828893, 89.2661206, 121.897576, 159.17822, 201.077887, 247.566411, 298.613626, 349.781326, 410.834988, 486.748087, 580.658588, 691.956963, 821.716231, 971.009414, 1142.20414, 1334.78459, 1546.82264, 1772.31472, 2023.05931, 2308.40736, 2634.20063, 3000.33471, 3409.4346, 3862.91018, 4363.0714, 4916.68355, 5527.99174, 6204.19694, 6946.13362, 7757.02368, 8640.08905, 9598.55163, 10890.4194, 12200.4317, 13468.1139, 14632.9914, 15634.5895, 16155.5342, 16563.5167, 16908.1621, 17383.2616, 17981.3746, 18634.9916, 19339.6757, 20090.9898, 20884.4968, 21715.7599, 21945.7398, 22719.5884, 23835.1831, 25090.4013, 26283.1205, 27211.2179, 27672.5709, 29221.2304, 31528.5291}, 
{216.397646, 217.484043, 218.461901, 219.332709, 220.097953, 220.759121, 221.3177, 221.775178, 222.133042, 222.392779, 222.555877, 222.623823, 222.598104, 222.480208, 222.271623, 221.973834, 221.588331, 221.1166, 220.560129, 219.920405, 219.198915, 218.397147, 217.516588, 216.558725, 215.525047, 214.417039, 213.23619, 211.983987, 210.661917, 209.271468, 207.814127, 206.291381, 204.704719, 203.055626, 201.349716, 199.586327, 197.764747, 195.88633, 193.952429, 191.964397, 189.923588, 187.831356, 185.689054, 183.498035, 181.259653, 178.975261, 176.646214, 174.273863, 171.859564, 169.404669, 166.910531, 164.360529, 161.774382, 159.153832, 156.500623, 153.816498, 151.1032, 148.362472, 145.596058, 142.8057, 139.993142, 137.160127, 134.308398, 131.439698, 128.55577, 125.658358, 122.749205, 119.927674, 117.290788, 114.638951, 111.96758, 109.272089, 106.547892, 103.790405, 100.995042, 98.1572175, 95.2723471, 92.3358453, 89.3431267, 86.2896063, 83.1706987, 79.9818187, 76.7183812, 73.3758008, 69.3087059, 64.8463826, 60.3258187, 55.7632237, 51.1748073, 46.5767788, 41.9853479, 37.4167241, 32.8871169, 28.4127358, 24.0097903, 19.69449, 15.4830445, 11.3916632, 7.43655565, 3.63393141, 0, 0.762250868, 1.14664653, 1.17785587, 0.88054779, 0.279391178, -0.600945073, -1.73579207, -3.10048092, -4.67034274, -6.42070862, -8.32690969, -10.364277, -12.5081418, -14.733835, -17.0166879, -19.3320314, -21.4607578, -23.3504211, -25.416155, -27.591625, -29.8104968, -32.0064359, -34.113108, -36.0641787, -37.7933135, -39.2341782, -40.3204382, -40.9857593, -41.163807, -40.8080157, -39.8250336, -38.1425958, -35.6884372, -32.9566786, -29.591862, -25.0874933, -19.2013916, -11.6913761, -2.31526607, 9.16911939, 22.0207812, 37.7643087, 56.8342439, 79.6651289, 106.691506, 139.341813, 176.808222, 219.276802, 266.933618, 319.964741, 375.315024, 440.373231, 518.271427, 611.443731, 720.899294, 848.28705, 995.25593, 1165.97176, 1358.99456, 1570.13865, 1791.08741, 2036.21521, 2317.68083, 2642.18384, 3007.17346, 3414.34499, 3863.82064, 4356.88788, 4900.60202, 5498.75578, 6157.1662, 6877.29018, 7662.21966, 8515.04661, 9438.86298, 10681.7938, 11940.6397, 13157.2343, 14273.4113, 15231.0044, 15723.8401, 16107.0973, 16428.8987, 16875.4748, 17440.2804, 18059.2251, 18728.6409, 19444.8598, 20204.2137, 21003.0347, 21367.3813, 22151.1006, 23198.4194, 24353.5646, 25460.7633, 26364.2423, 26908.2287, 28630.4919, 30886.169}, 
{211.768591, 212.736698, 213.602393, 214.367083, 215.032172, 215.599064, 216.069164, 216.443878, 216.724609, 216.912764, 217.009745, 217.016959, 216.93581, 216.767703, 216.514043, 216.176233, 215.75568, 215.253788, 214.671962, 214.011606, 213.274126, 212.460925, 211.573409, 210.612983, 209.581052, 208.479019, 207.308291, 206.070271, 204.766365, 203.397977, 201.966512, 200.473375, 198.919971, 197.307704, 195.642038, 193.922262, 192.147619, 190.319381, 188.438822, 186.507214, 184.525831, 182.495945, 180.41883, 178.295759, 176.128005, 173.91684, 171.663539, 169.369373, 167.035617, 164.663543, 162.254424, 159.792651, 157.296745, 154.768346, 152.209092, 149.620622, 147.004577, 144.362594, 141.696313, 139.007373, 136.297412, 133.568071, 130.820987, 128.057801, 125.280151, 122.489676, 119.688016, 116.975032, 114.448231, 111.906671, 109.345624, 106.760362, 104.146156, 101.498278, 98.8119996, 96.0825925, 93.3053281, 90.4754782, 87.5883142, 84.6391079, 81.6231308, 78.5356544, 75.3719506, 72.1272907, 68.141774, 63.7520389, 59.3025987, 54.8099866, 50.2907358, 45.7613795, 41.238451, 36.7384835, 32.2780101, 27.8735642, 23.5416788, 19.2988873, 15.1617229, 11.1467187, 7.27040802, 3.54932405, 0, 0.773307878, 1.19113423, 1.27666409, 1.05308248, 0.543574419, -0.228675057, -1.24048093, -2.46865816, -3.89002173, -5.48138662, -7.2195678, -9.08138023, -11.0436389, -13.0831588, -15.1767548, -17.3012421, -19.0345732, -20.296583, -21.9662747, -23.9349927, -26.0940815, -28.3348857, -30.5487498, -32.6270182, -34.4610356, -35.9421464, -36.961695, -37.4110261, -37.1814841, -35.819719, -33.6881578, -30.7815535, -27.0946588, -23.5436485, -19.6625644, -14.7397363, -8.49349037, -0.642153064, 9.09594926, 21.0024902, 34.6417331, 51.2311042, 71.1926401, 94.9483772, 122.920352, 155.603195, 193.3282, 236.499257, 285.520251, 340.795073, 400.767049, 470.196869, 550.206232, 642.440596, 749.614767, 873.963439, 1017.7213, 1186.58378, 1378.5383, 1587.79697, 1804.53848, 2044.96644, 2323.55425, 2647.62936, 3012.41081, 3418.6756, 3865.27355, 4352.48202, 4887.64471, 5474.10822, 6116.31182, 6816.34129, 7577.16496, 8401.75117, 9293.06826, 10490.5594, 11701.5994, 12870.0379, 13939.7245, 14854.5088, 15317.3449, 15673.575, 15970.2898, 16388.9786, 16923.5327, 17511.8433, 18150.6017, 18836.4987, 19566.2254, 20336.4728, 20872.2206, 21674.1947, 22642.6181, 23677.714, 24679.7054, 25548.8157, 26185.2678, 28050.6117, 30238.2671}, 
{207.691201, 208.569457, 209.3491, 210.031523, 210.618116, 211.110271, 211.509378, 211.816828, 212.034014, 212.162327, 212.203156, 212.157895, 212.027934, 211.814663, 211.519475, 211.143761, 210.688911, 210.156318, 209.547372, 208.863464, 208.105986, 207.276329, 206.375884, 205.406043, 204.368196, 203.263735, 202.094051, 200.860535, 199.564579, 198.207573, 196.790909, 195.315979, 193.784173, 192.196882, 190.561221, 188.8756, 187.138358, 185.350701, 183.513835, 181.628966, 179.697297, 177.720036, 175.698388, 173.633558, 171.526751, 169.379173, 167.19203, 164.966527, 162.703869, 160.405263, 158.071913, 155.690282, 153.276638, 150.832506, 148.359412, 145.85888, 143.332435, 140.781603, 138.207909, 135.612877, 132.998034, 130.364903, 127.71501, 125.049881, 122.371039, 119.680011, 116.978321, 114.366344, 111.942078, 109.503182, 107.044773, 104.561968, 102.049884, 99.5036386, 96.9183487, 94.2891315, 91.6111041, 88.8793838, 86.0890875, 83.2353326, 80.3132362, 77.3179153, 74.2444872, 71.0880691, 67.1737359, 62.8457193, 58.4563044, 54.0223524, 49.5607246, 45.0882825, 40.6218872, 36.1784, 31.7746824, 27.4275956, 23.1540008, 18.9707595, 14.8947329, 10.9427824, 7.13176917, 3.47855461, 0, 1.2293059, 2.03343422, 2.44090663, 2.4802448, 2.17997041, 1.56860511, 0.674670587, -0.473311502, -1.84681949, -3.41733169, -5.15632646, -7.03528211, -9.02567697, -11.0989894, -13.2266977, -15.3802802, -17.1878026, -18.5716843, -20.2801703, -22.2111505, -24.2625148, -26.3321532, -28.3179554, -30.1178115, -31.6296114, -32.7512449, -33.380602, -33.4155726, -32.7540466, -31.0034739, -28.458679, -25.1046839, -20.9265105, -16.7084289, -12.0358368, -6.28099923, 0.810836557, 9.49442314, 20.0245131, 32.6558591, 46.5114835, 63.3223088, 83.564513, 107.714274, 136.247771, 169.614869, 208.324636, 252.859829, 303.703203, 361.337513, 424.994203, 497.936725, 580.370752, 673.489772, 780.504065, 903.718415, 1045.4376, 1210.76929, 1398.57837, 1604.67202, 1821.59251, 2062.95443, 2341.34961, 2663.17126, 3024.30938, 3425.47513, 3865.66089, 4345.13223, 4870.45675, 5444.63724, 6070.9943, 6752.16406, 7491.03931, 8290.51285, 9153.47749, 10313.4922, 11485.1171, 12613.5784, 13644.1023, 14521.9151, 14955.2555, 15284.3289, 15555.9279, 15952.0537, 16465.9786, 17034.6374, 17654.1191, 18320.5127, 19029.907, 19778.3911, 20558.1421, 21389.0702, 22249.3108, 23116.999, 23970.2701, 24787.2593, 25546.1019, 27475.9822, 29589.067}, 
{204.04205, 204.863056, 205.58679, 206.214701, 206.748239, 207.188854, 207.537996, 207.797113, 207.967656, 208.051074, 208.048817, 207.962334, 207.793075, 207.54249, 207.212027, 206.803138, 206.317271, 205.755876, 205.120402, 204.4123, 203.633019, 202.784008, 201.866717, 200.882595, 199.833093, 198.71966, 197.543745, 196.306798, 195.010269, 193.655607, 192.244262, 190.777683, 189.25732, 187.684623, 186.070238, 184.410822, 182.702923, 180.947693, 179.146281, 177.299841, 175.409522, 173.476477, 171.501855, 169.486809, 167.43249, 165.340048, 163.210635, 161.045402, 158.8455, 156.612081, 154.346295, 152.037683, 149.699258, 147.332423, 144.938582, 142.519138, 140.075492, 137.609049, 135.121211, 132.613381, 130.086961, 127.543356, 124.983967, 122.410197, 119.82345, 117.225128, 114.616634, 112.098973, 109.770756, 107.427967, 105.065552, 102.678458, 100.261629, 97.8100137, 95.3185567, 92.7822046, 90.1959035, 87.5545995, 84.8532389, 82.0867678, 79.2501324, 76.3382788, 73.3461531, 70.2687017, 66.4152442, 62.1379613, 57.7972742, 53.4103789, 48.9944715, 44.566748, 40.1444045, 35.744637, 31.3846417, 27.0816144, 22.8527514, 18.7152486, 14.6863021, 10.7831079, 7.02286214, 3.42276082, 0, 2.12843815, 3.66963451, 4.66433755, 5.15329573, 5.17725753, 4.77697142, 3.99318585, 2.8666493, 1.43811024, -0.251682862, -2.16198154, -4.25203734, -6.48110177, -8.80842637, -11.1932627, -13.5948622, -15.9639858, -18.2586718, -20.4573773, -22.5175343, -24.3965749, -26.0519312, -27.4410352, -28.521319, -29.2502147, -29.5851544, -29.4835701, -28.902894, -27.800558, -26.3565944, -24.2242151, -21.2940722, -17.4568177, -12.7669616, -7.033227, -0.0206415301, 8.42930016, 18.4751034, 30.2752735, 43.9883159, 57.543235, 74.0065807, 93.9730651, 118.0374, 146.794298, 181.634544, 222.157759, 268.759635, 321.835867, 381.782147, 447.708023, 522.867292, 607.933051, 704.093979, 813.591391, 938.193988, 1079.67047, 1240.24527, 1421.12778, 1623.03028, 1844.798, 2092.90173, 2373.64279, 2690.86046, 3044.22231, 3435.31077, 3864.8119, 4334.07574, 4847.73916, 5408.57983, 6019.0556, 6682.31313, 7401.2408, 8178.72698, 9017.66003, 10148.9075, 11290.3838, 12387.9825, 13387.5971, 14235.1212, 14640.0465, 14942.2697, 15188.9995, 15567.7466, 16070.1835, 16629.5234, 17240.3618, 17897.2941, 18594.9156, 19327.8219, 20402.8256, 21267.7042, 21995.545, 22659.4356, 23332.4632, 24087.7154, 24998.2795, 26909.2325, 28936.5489}, 
{200.87636, 201.652992, 202.332864, 202.917503, 203.408432, 203.807176, 204.115261, 204.33421, 204.465549, 204.510803, 204.471496, 204.349154, 204.1453, 203.861459, 203.499157, 203.059919, 202.545268, 201.95673, 201.295829, 200.564091, 199.76304, 198.8942, 197.959097, 196.959256, 195.896201, 194.771457, 193.586548, 192.343, 191.042337, 189.686085, 188.275767, 186.812909, 185.299035, 183.73567, 182.137338, 180.498791, 178.814624, 177.08594, 175.313843, 173.499435, 171.643819, 169.748098, 167.813376, 165.840754, 163.831337, 161.786226, 159.706525, 157.593337, 155.447765, 153.270912, 151.06388, 148.818663, 146.545672, 144.246206, 141.921566, 139.573052, 137.201965, 134.809604, 132.397272, 129.966267, 127.51789, 125.053442, 122.574223, 120.081533, 117.576674, 115.060945, 112.535646, 110.102928, 107.862517, 105.607721, 103.333303, 101.034025, 98.7046518, 96.3399444, 93.9346663, 91.48358, 88.9814486, 86.4230347, 83.8031011, 81.1164107, 78.3577262, 75.5218105, 72.6034264, 69.5973366, 65.7959239, 61.5599147, 57.2584776, 52.909169, 48.5295452, 44.1371626, 39.7495776, 35.3843466, 31.0590259, 26.791172, 22.5983413, 18.49809, 14.5079747, 10.6455516, 6.92837722, 3.37400789, 0, 2.96507057, 5.19300895, 6.73586372, 7.64568342, 7.97451664, 7.77441194, 7.09741787, 5.995583, 4.5209559, 2.72558513, 0.661519264, -1.61919315, -4.06450353, -6.62236333, -9.24072397, -11.8675369, -14.7798134, -17.9725134, -20.6492864, -22.8285966, -24.5289082, -25.7686856, -26.566393, -26.9404947, -26.9094548, -26.4917377, -25.7058077, -24.5701289, -23.1031657, -22.0860214, -20.4948869, -18.1194345, -14.7493368, -9.68489807, -2.96047486, 5.25907792, 15.0372773, 26.4376402, 39.5236836, 54.3589247, 67.7179237, 83.9541411, 103.774585, 127.886265, 156.996188, 193.600257, 236.169364, 284.963293, 340.241829, 402.264759, 469.704355, 546.348206, 633.773555, 733.32732, 845.886169, 972.537904, 1114.37033, 1270.37033, 1444.2041, 1641.82973, 1868.61483, 2123.67933, 2406.84004, 2719.32216, 3064.73731, 3445.60172, 3864.37903, 4323.57183, 4825.87545, 5373.87619, 5969.09714, 6615.35138, 7315.59325, 8072.77708, 8889.85722, 9997.0733, 11113.273, 12184.5894, 13157.1555, 13977.1043, 14352.9662, 14626.8786, 14846.8041, 15207.092, 15697.7604, 16247.8327, 16850.3766, 17498.4599, 18185.1503, 18903.5153, 20231.4611, 21119.4117, 21725.821, 22209.1427, 22727.8306, 23440.3385, 24505.1203, 26363.4399, 28270.7474}, 
{198.137604, 198.877953, 199.521759, 200.070621, 200.526135, 200.889901, 201.163515, 201.348576, 201.446682, 201.45943, 201.388419, 201.235246, 201.00151, 200.688807, 200.298737, 199.832897, 199.292885, 198.680298, 197.996736, 197.243794, 196.423073, 195.536168, 194.58468, 193.570204, 192.494339, 191.358684, 190.164835, 188.914391, 187.60895, 186.25011, 184.839468, 183.378622, 181.869171, 180.312712, 178.727469, 177.106377, 175.442171, 173.735911, 171.988654, 170.201459, 168.375384, 166.511488, 164.610829, 162.674465, 160.703455, 158.698857, 156.661729, 154.593131, 152.49412, 150.365754, 148.209092, 146.017316, 143.799532, 141.556968, 139.290854, 137.002419, 134.692892, 132.363502, 130.015478, 127.65005, 125.268447, 122.871897, 120.46163, 118.038874, 115.60486, 113.160816, 110.707972, 108.35035, 106.189508, 104.014668, 101.820395, 99.6012544, 97.3518113, 95.0666313, 92.7402798, 90.367322, 87.9423233, 85.4598491, 82.9144647, 80.3007354, 77.6132267, 74.8465039, 71.9951322, 69.0536772, 65.2960856, 61.0923892, 56.8212813, 52.5007125, 48.1486337, 43.7829957, 39.4217493, 35.0828453, 30.7842345, 26.5438677, 22.3796958, 18.3096694, 14.3517394, 10.5238567, 6.84397205, 3.33003621, 0, 3.54873086, 6.26157356, 8.1980665, 9.41774805, 9.9801566, 9.94483055, 9.37130829, 8.31912819, 6.84782865, 5.01694806, 2.8860248, 0.514597261, -2.03779617, -4.7116171, -7.44732714, -10.1853879, -13.4440924, -17.2464497, -20.2259216, -22.4467908, -23.97334, -24.8698518, -25.2006088, -25.0298937, -24.421989, -23.4411774, -22.1517415, -20.617964, -18.9041275, -18.2675668, -17.1420598, -15.2339737, -12.2496752, -6.89807942, 0.61572065, 9.82064493, 20.7110909, 33.281456, 47.5261376, 63.4395333, 76.974116, 93.3963587, 113.49147, 138.044658, 167.841133, 206.39894, 251.087242, 302.008036, 359.263323, 422.955102, 491.100769, 568.434774, 657.701547, 760.53201, 876.283675, 1005.33477, 1148.06351, 1300.48046, 1468.32417, 1662.09794, 1892.56258, 2152.82285, 2437.4623, 2745.5798, 3083.92518, 3455.73633, 3864.87423, 4314.73832, 4806.44347, 5342.39682, 5922.98993, 6552.95563, 7235.39879, 7973.42424, 8770.13686, 9857.03612, 10951.7336, 12000.2355, 12948.5479, 13742.6772, 14088.5409, 14332.2929, 14522.8344, 14861.1492, 15335.9394, 15872.5391, 16462.8188, 17098.6487, 17771.8993, 18474.4409, 19964.6264, 20855.2587, 21367.5518, 21722.7196, 22141.9761, 22846.5352, 24057.6108, 25845.1459, 27586.6672}, 
{195.736755, 196.443977, 197.055282, 197.572314, 197.996714, 198.330125, 198.57419, 198.730551, 198.800851, 198.786732, 198.689838, 198.51181, 198.254292, 197.918925, 197.507352, 197.021217, 196.462161, 195.831828, 195.131859, 194.363897, 193.529586, 192.630566, 191.668482, 190.644976, 189.56169, 188.420267, 187.222349, 185.969579, 184.6636, 183.306054, 181.898583, 180.442831, 178.94044, 177.393053, 175.821875, 174.218357, 172.573712, 170.888947, 169.165071, 167.403092, 165.604016, 163.768852, 161.898607, 159.99429, 158.056907, 156.087467, 154.086977, 152.056444, 149.996878, 147.909284, 145.794672, 143.644086, 141.468713, 139.269775, 137.048497, 134.806103, 132.543815, 130.262859, 127.964456, 125.649832, 123.320209, 120.976811, 118.620863, 116.253587, 113.876207, 111.489947, 109.096031, 106.80192, 104.712527, 102.609965, 100.488572, 98.3426842, 96.1666382, 93.9547708, 91.7014186, 89.4009183, 87.0476065, 84.6358199, 82.1598951, 79.6141689, 76.9929779, 74.2906587, 71.5015479, 68.6199824, 64.8983191, 60.717491, 56.4673492, 52.166309, 47.8327856, 43.4851945, 39.1419509, 34.8214701, 30.5421675, 26.3224584, 22.1807582, 18.1354821, 14.2050454, 10.4078636, 6.76235183, 3.28692553, 0, 3.48671732, 6.17080113, 8.10946072, 9.35990536, 9.97934435, 10.024987, 9.5540425, 8.62372025, 7.29122948, 5.6137795, 3.64857958, 1.45283901, -0.916232924, -3.40142693, -5.94553373, -8.49134402, -11.5420082, -15.1203682, -17.9017348, -19.9489757, -21.3249588, -22.0925516, -22.3146222, -22.0540381, -21.3736672, -20.3363773, -19.0050362, -17.4425116, -15.7116712, -15.1354817, -14.0546758, -12.1540917, -9.11856758, -3.57942454, 4.25174075, 13.8823938, 25.3116414, 38.5385903, 53.5623473, 70.3820191, 85.01899, 102.660701, 124.084507, 150.067767, 181.387837, 222.252042, 269.150278, 322.00241, 380.728304, 445.247825, 512.592223, 589.100505, 679.008923, 784.428713, 903.13253, 1034.84097, 1179.27461, 1330.52956, 1495.2292, 1686.13268, 1915.94531, 2175.84722, 2459.11261, 2764.20279, 3098.36838, 3464.82556, 3867.55948, 4309.98568, 4792.7002, 5317.89394, 5884.16454, 6497.84898, 7162.382, 7881.19831, 8657.73264, 9727.70453, 10804.1927, 11832.5607, 12758.172, 13526.3901, 13838.3512, 14046.4645, 14200.7624, 14507.1068, 14953.8475, 15464.4057, 16030.3781, 16643.361, 17294.9509, 17976.7443, 19459.6589, 20316.8716, 20789.4166, 21118.3282, 21544.6407, 22309.3884, 23653.6054, 25384.5761, 26879.046}, 
{193.605523, 194.277934, 194.85606, 195.341536, 195.735999, 196.041088, 196.25844, 196.389691, 196.436479, 196.400441, 196.283215, 196.086438, 195.811746, 195.460778, 195.03517, 194.53656, 193.966585, 193.326883, 192.61909, 191.844843, 191.005781, 190.10354, 189.139757, 188.116071, 187.034117, 185.895534, 184.701958, 183.455028, 182.156379, 180.80765, 179.410477, 177.966498, 176.477351, 174.944671, 173.391404, 171.808084, 170.184991, 168.523073, 166.823273, 165.08654, 163.313817, 161.506051, 159.664189, 157.789175, 155.881955, 153.943476, 151.974684, 149.976523, 147.94994, 145.895881, 143.815292, 141.692603, 139.545633, 137.375686, 135.184066, 132.972078, 130.741024, 128.492209, 126.226936, 123.94651, 121.652234, 119.345412, 117.027348, 114.699346, 112.362709, 110.018743, 107.668749, 105.425631, 103.399611, 101.36183, 99.3063574, 97.2272628, 95.1186156, 92.9744853, 90.7889412, 88.5560528, 86.2698895, 83.9245207, 81.5140158, 79.0324443, 76.4738757, 73.8323792, 71.1020243, 68.2768806, 64.5837601, 60.4164991, 56.1781557, 51.8876984, 47.5640955, 43.2263153, 38.8933264, 34.5840971, 30.3175957, 26.1127907, 21.9886506, 17.9641436, 14.0582381, 10.2899027, 6.67810555, 3.2418152, 0, 2.51536062, 4.44748496, 5.83741959, 6.72621107, 7.15490597, 7.16455086, 6.79619228, 6.09087683, 5.08965104, 3.8335615, 2.36365477, 0.720977406, -1.05342402, -2.91850295, -4.83321281, -6.75650704, -8.80142751, -10.9489403, -12.8094654, -14.3749754, -15.6374424, -16.5888391, -17.2211376, -17.5263105, -17.49633, -17.1231686, -16.3987987, -15.3151926, -13.8643228, -12.8229253, -11.1104621, -8.48347646, -4.6985116, 0.930390292, 8.42543542, 17.6908297, 28.8372795, 41.9754913, 57.2161714, 74.6700263, 91.707361, 112.013318, 136.234771, 165.01859, 199.011647, 242.529523, 291.633204, 346.052386, 405.516763, 469.756031, 534.505271, 608.371097, 697.317699, 804.134672, 925.130149, 1059.5223, 1206.52932, 1360.07728, 1525.8792, 1715.42125, 1938.20458, 2189.55669, 2467.25578, 2771.32632, 3105.599, 3472.15737, 3873.21821, 4310.89961, 4786.83127, 5302.91914, 5855.05646, 6452.08696, 7097.99793, 7796.77667, 8552.41047, 9608.07512, 10668.7723, 11678.6922, 12582.0249, 13322.9607, 13595.857, 13761.2918, 13870.5264, 14131.0034, 14532.1623, 14998.3673, 15522.3116, 16096.6887, 16714.1918, 17367.5144, 18614.2465, 19390.1824, 19897.5696, 20338.6553, 20915.6868, 21830.9115, 23286.5766, 24996.8446, 26142.792}, 
{191.875385, 192.507405, 193.047279, 193.496612, 193.857013, 194.130088, 194.317443, 194.420686, 194.441423, 194.381262, 194.241809, 194.024672, 193.731456, 193.36377, 192.923219, 192.411411, 191.829952, 191.180451, 190.464512, 189.683744, 188.839753, 187.934146, 186.96853, 185.944512, 184.863698, 183.727697, 182.538113, 181.296555, 180.00463, 178.663943, 177.276103, 175.842716, 174.365388, 172.845727, 171.306722, 169.73884, 168.132289, 166.487981, 164.80683, 163.089749, 161.337651, 159.551448, 157.732054, 155.880382, 153.997344, 152.083854, 150.140825, 148.169169, 146.1698, 144.14363, 142.091573, 139.994498, 137.873796, 135.730814, 133.5669, 131.383402, 129.181666, 126.963041, 124.728873, 122.480511, 120.219301, 117.946592, 115.66373, 113.372064, 111.07294, 108.767706, 106.45771, 104.259936, 102.288593, 100.306535, 98.3076131, 96.2856785, 94.2345822, 92.1481755, 90.0203095, 87.8448354, 85.6156043, 83.3264674, 80.9712759, 78.5438809, 76.0381337, 73.4478853, 70.7669869, 67.9892898, 64.3195331, 60.1627219, 55.9333458, 51.650864, 47.334736, 43.0044211, 38.6793786, 34.379068, 30.1229486, 25.9304798, 21.8211209, 17.8143312, 13.9295702, 10.1862973, 6.60397165, 3.20205278, 0, 1.61414669, 2.84706317, 3.72499563, 4.27419025, 4.52089323, 4.49135077, 4.21180906, 3.70851429, 3.00771266, 2.13565035, 1.11857356, -0.0172715113, -1.24563868, -2.54028175, -3.87495452, -5.22341081, -6.32487685, -7.11960251, -8.11182063, -9.2250291, -10.3827258, -11.5084087, -12.5255756, -13.3577245, -13.9283532, -14.1609597, -13.9790418, -13.3060974, -12.0656245, -10.4890902, -8.07910106, -4.66676427, -0.0831869452, 5.6744393, 12.8580496, 21.7638681, 32.6106132, 45.617003, 61.0017556, 78.9835894, 98.3164468, 121.129623, 147.928423, 179.218151, 215.504112, 260.397738, 310.521675, 365.604696, 425.375571, 489.563076, 553.617147, 626.775079, 715.430393, 822.587995, 944.334364, 1079.86221, 1228.36425, 1384.88483, 1553.70783, 1743.64264, 1960.10701, 2203.17724, 2475.29354, 2778.17137, 3112.29819, 3478.72511, 3878.02328, 4311.11939, 4780.70009, 5288.45652, 5828.51404, 6411.29349, 7041.10487, 7722.25821, 8459.06354, 9497.99252, 10540.6531, 11530.815, 12412.2477, 13128.7209, 13372.6254, 13506.6954, 13582.4343, 13804.1445, 14162.8464, 14585.894, 15067.4911, 15601.8413, 16183.1484, 16805.616, 17714.8319, 18387.9859, 18959.2195, 19562.674, 20332.4907, 21402.8112, 22907.7767, 24551.4731, 25374.4558}, 
{190.658088, 191.224616, 191.702726, 192.093935, 192.399763, 192.621727, 192.761347, 192.82014, 192.799627, 192.701324, 192.52675, 192.277425, 191.954866, 191.560593, 191.096123, 190.562975, 189.962668, 189.296721, 188.566651, 187.773977, 186.920218, 186.006893, 185.03552, 184.007617, 182.924703, 181.788296, 180.599916, 179.36108, 178.073307, 176.738116, 175.357025, 173.931553, 172.463218, 170.953539, 169.420954, 167.858166, 166.257674, 164.620447, 162.947456, 161.239669, 159.498057, 157.723588, 155.917232, 154.079959, 152.212737, 150.316538, 148.392328, 146.44108, 144.463761, 142.461341, 140.43479, 138.373632, 136.29053, 134.186701, 132.063363, 129.921734, 127.76303, 125.58847, 123.399272, 121.196652, 118.981828, 116.756018, 114.52044, 112.276311, 110.024849, 107.76727, 105.504794, 103.353123, 101.425213, 99.4861612, 97.5297646, 95.5498187, 93.5401194, 91.4944626, 89.4066442, 87.2704601, 85.0797062, 82.8281784, 80.5096726, 78.1179847, 75.6469105, 73.090246, 70.4417871, 67.6953296, 64.0456732, 59.9029178, 55.6867665, 51.4169441, 47.1131755, 42.7951857, 38.4826993, 34.1954414, 29.9531367, 25.7755101, 21.6822866, 17.6931908, 13.8279478, 10.1062824, 6.54791933, 3.17258359, 0, 1.45426717, 2.57030847, 3.3715648, 3.88147704, 4.12348609, 4.12103284, 3.89755818, 3.476503, 2.88130819, 2.13541466, 1.26226328, 0.285294955, -0.772049427, -1.88632898, -3.0341028, -4.19193, -4.99447396, -5.36945153, -6.06675742, -6.98968735, -8.04153707, -9.12560231, -10.1451788, -11.0035623, -11.6040485, -11.8499332, -11.6445121, -10.8910809, -9.49293544, -7.51281967, -4.63611668, -0.718287708, 4.385206, 10.4791835, 17.8780434, 26.9867763, 38.0518367, 51.3196793, 67.0367589, 85.4495301, 105.944032, 129.889001, 157.699232, 189.789524, 226.574673, 268.981425, 316.78464, 370.271127, 429.727696, 495.441156, 565.778179, 645.292546, 736.766661, 841.902481, 960.196051, 1092.13383, 1238.20226, 1397.28126, 1571.82895, 1766.05606, 1981.92147, 2222.26463, 2491.12699, 2790.93308, 3121.3676, 3483.37286, 3877.31935, 4304.00111, 4766.30899, 5265.94778, 5798.5863, 6372.89391, 6992.66477, 7661.69303, 8383.77285, 9393.59299, 10406.3293, 11368.0524, 12224.8328, 12922.7409, 13168.3094, 13306.8386, 13387.058, 13598.3495, 13933.7677, 14327.7623, 14776.3182, 15275.4204, 15821.0538, 16409.2034, 16836.6614, 17371.488, 18036.3819, 18854.0422, 19847.1675, 21038.4569, 22450.6093, 23891.038, 24552.9213}, 
{189.838058, 190.332549, 190.742561, 191.069515, 191.314831, 191.479929, 191.566231, 191.575157, 191.508128, 191.366565, 191.151888, 190.865518, 190.508875, 190.083381, 189.590456, 189.031521, 188.407996, 187.721303, 186.972861, 186.164092, 185.296416, 184.371254, 183.390027, 182.354155, 181.265059, 180.12416, 178.932879, 177.692636, 176.404851, 175.070946, 173.692341, 172.270457, 170.806715, 169.302535, 167.770392, 166.205945, 164.604726, 162.967795, 161.296215, 159.591047, 157.853355, 156.084198, 154.284641, 152.455744, 150.598569, 148.71418, 146.803637, 144.868003, 142.908339, 140.925708, 138.921172, 136.898133, 134.855262, 132.79357, 130.714069, 128.617769, 126.505683, 124.378821, 122.238194, 120.084815, 117.919694, 115.743842, 113.558271, 111.363992, 109.162016, 106.953355, 104.73902, 102.630964, 100.738478, 98.8337622, 96.9106362, 94.9629193, 92.9844308, 90.9689901, 88.9104165, 86.8025294, 84.6391481, 82.4140919, 80.1211801, 77.7542321, 75.3070672, 72.7735048, 70.1473642, 67.4224647, 63.7899377, 59.6618299, 55.4598359, 51.2038237, 46.9136615, 42.6092173, 38.3103593, 34.0369555, 29.8088741, 25.6459832, 21.5681508, 17.595245, 13.7471341, 10.043686, 6.50476889, 3.15025085, 0, 1.75994538, 3.12531453, 4.12423081, 4.78481759, 5.13519823, 5.20349612, 5.0178346, 4.60633706, 3.99712686, 3.21832737, 2.29806195, 1.26445398, 0.145626822, -1.03029616, -2.23519159, -3.4409361, -4.34143225, -4.86884609, -5.62380514, -6.51862016, -7.4656019, -8.37706112, -9.16530859, -9.74265506, -10.0214113, -9.91388803, -9.33239604, -8.18924609, -6.39674893, -4.07004585, -0.844245884, 3.42918938, 8.89879833, 15.4378399, 23.3324923, 32.9423414, 44.4985095, 58.2321189, 74.3742917, 93.1561502, 114.167487, 138.475941, 166.438113, 198.410601, 234.750007, 272.887921, 316.837204, 367.685706, 426.52128, 494.431777, 574.097936, 663.067859, 758.779336, 860.970722, 974.077363, 1100.42575, 1242.34237, 1403.51586, 1584.60099, 1784.7666, 2003.33384, 2244.39902, 2511.64685, 2807.55356, 3132.54766, 3487.99757, 3874.93336, 4294.63575, 4749.62613, 5241.72411, 5769.59202, 6338.69245, 6951.93791, 7612.24091, 8322.51394, 9299.18506, 10278.2723, 11209.3094, 12041.8298, 12725.3672, 12987.4833, 13147.6649, 13250.5765, 13465.8845, 13788.7735, 14162.3438, 14584.3841, 15052.6831, 15565.0295, 16119.2119, 16048.2179, 16444.1042, 17207.5934, 18239.4079, 19440.2702, 20710.903, 21952.0286, 23127.9659, 23699.3306}, 
{189.297894, 189.732807, 190.08597, 190.358748, 190.552503, 190.668599, 190.708401, 190.673271, 190.564573, 190.38367, 190.131927, 189.810707, 189.421373, 188.96529, 188.44382, 187.858327, 187.210175, 186.500728, 185.731349, 184.903401, 184.018249, 183.077255, 182.081784, 181.033199, 179.932864, 178.782142, 177.582397, 176.334993, 175.041292, 173.70266, 172.320458, 170.896052, 169.430803, 167.926077, 166.390243, 164.821013, 163.216017, 161.576392, 159.903272, 158.197796, 156.461098, 154.694316, 152.898585, 151.075042, 149.224823, 147.349064, 145.448901, 143.525472, 141.579911, 139.613355, 137.626941, 135.636093, 133.627349, 131.601536, 129.55948, 127.502008, 125.429946, 123.34412, 121.245358, 119.134486, 117.01233, 114.879716, 112.737472, 110.586423, 108.427397, 106.261219, 104.088717, 102.018545, 100.15677, 98.2818294, 96.3875599, 94.4677978, 92.5163795, 90.5271413, 88.4939197, 86.410551, 84.2708717, 82.0687181, 79.7979266, 77.4523337, 75.0257757, 72.512089, 69.90511, 67.1986752, 63.5806196, 59.4647334, 55.2744944, 51.0298952, 46.7509284, 42.4575869, 38.1698632, 33.9077501, 29.6912402, 25.5403262, 21.4750008, 17.5152568, 13.6810867, 9.99248331, 6.46943926, 3.13194727, 0, 2.26048133, 4.02930452, 5.34233193, 6.23542594, 6.74444891, 6.90526322, 6.75373124, 6.32571533, 5.65707786, 4.78368122, 3.74138775, 2.56605985, 1.29355986, -0.0402498221, -1.39950684, -2.74834883, -3.89973534, -4.79620713, -5.74385224, -6.67441299, -7.51963172, -8.21125076, -8.68101243, -8.86065906, -8.68193298, -8.07657652, -6.976332, -5.31294176, -3.01814812, -0.326861946, 3.24350476, 7.85216024, 13.6583127, 20.6525234, 29.0783241, 39.224219, 51.3000104, 65.5155007, 82.0804921, 101.204787, 122.573754, 147.08124, 175.039653, 206.761402, 242.558895, 276.743717, 317.129308, 365.528282, 423.753256, 493.616844, 581.688216, 679.209866, 778.874455, 878.662569, 987.350984, 1108.86921, 1247.14675, 1409.87541, 1596.36733, 1801.83033, 2024.00878, 2267.18677, 2533.8047, 2826.04428, 3145.64897, 3494.56338, 3874.76019, 4288.19131, 4736.70577, 5222.21078, 5745.92799, 6310.55568, 6918.23836, 7571.12058, 8271.34687, 9219.33264, 10169.3838, 11074.0772, 11885.9896, 12557.6979, 12835.3232, 13015.535, 13139.3913, 13359.159, 13671.8757, 14028.2417, 14427.6116, 14869.3395, 15352.7801, 15877.2876, 15422.3402, 15714.6312, 16555.9907, 17748.2482, 19093.2338, 20392.7771, 21448.708, 22375.7364, 22835.9121}, 
{188.77964, 189.215877, 189.56792, 189.837261, 190.025393, 190.133808, 190.163999, 190.117457, 189.995675, 189.800145, 189.532359, 189.193811, 188.785991, 188.310393, 187.768509, 187.16183, 186.49185, 185.76006, 184.967953, 184.117021, 183.208756, 182.244652, 181.226199, 180.15489, 179.032218, 177.859675, 176.638753, 175.370945, 174.057742, 172.700637, 171.301123, 169.860691, 168.380834, 166.863044, 165.3221, 163.75257, 162.148866, 160.512049, 158.843179, 157.143319, 155.413528, 153.654869, 151.868402, 150.055189, 148.21629, 146.352767, 144.465681, 142.556093, 140.625065, 138.673656, 136.70293, 134.723179, 132.726031, 130.712348, 128.682991, 126.638821, 124.580698, 122.509484, 120.42604, 118.331227, 116.225905, 114.110935, 111.98718, 109.855499, 107.716754, 105.571805, 103.421515, 101.377633, 99.54925, 97.7085228, 95.8491238, 93.9647256, 92.0490005, 90.095621, 88.0982595, 86.0505884, 83.94628, 81.7790069, 79.5424414, 77.230256, 74.836123, 72.3537149, 69.7767041, 67.0987631, 63.4932333, 59.3812074, 55.1936372, 50.9508165, 46.6730391, 42.3805987, 38.0937889, 33.8329036, 29.6182365, 25.4700812, 21.4087316, 17.4544813, 13.627624, 9.94845351, 6.43726354, 3.11434779, 0, 2.63009139, 4.70334812, 6.26090174, 7.34388379, 7.99342582, 8.25065938, 8.15671601, 7.75272726, 7.07982467, 6.17913978, 5.09180415, 3.85894932, 2.52170684, 1.12120824, -0.30141492, -1.7050311, -2.93760041, -3.94214727, -4.92840421, -5.83147364, -6.58645791, -7.12845943, -7.39258057, -7.31392371, -6.82759123, -5.86868553, -4.37230897, -2.27356394, 0.492447176, 3.73724158, 7.87200346, 13.0376446, 19.3750766, 26.6731666, 35.2488486, 45.5129354, 57.7119522, 72.0924242, 88.9008765, 108.383834, 130.720856, 156.245814, 185.218338, 217.898053, 254.544588, 289.221532, 329.933559, 378.489308, 436.697416, 506.36652, 593.805155, 690.822184, 790.813288, 891.985725, 1002.37039, 1125.58757, 1265.25755, 1428.14814, 1614.01675, 1819.18711, 2042.55889, 2287.07648, 2553.98764, 2844.98706, 3162.52665, 3508.71075, 3885.93373, 4296.37515, 4741.151, 5221.97891, 5738.15948, 6294.19915, 6892.65218, 7536.07284, 8227.01537, 9171.5971, 10117.9185, 11017.6429, 11822.434, 12483.9552, 12743.5976, 12903.4786, 13005.3787, 13202.7941, 13493.2842, 13827.8598, 14206.4076, 14628.8144, 15094.9667, 15604.7514, 15212.284, 15529.0233, 16361.9405, 17518.0068, 18804.1935, 20027.4716, 20994.8126, 21868.1458, 22042.2963}, 
{188.497448, 188.948644, 189.312053, 189.589334, 189.782143, 189.892137, 189.920974, 189.870312, 189.741806, 189.537115, 189.257897, 188.905807, 188.482503, 187.989643, 187.428884, 186.801883, 186.110298, 185.355785, 184.540002, 183.664606, 182.731254, 181.741604, 180.697313, 179.600038, 178.451436, 177.253165, 176.006882, 174.714244, 173.376908, 171.996532, 170.574773, 169.113288, 167.613735, 166.07777, 164.528544, 162.956516, 161.351885, 159.715612, 158.048657, 156.351979, 154.626538, 152.873293, 151.093206, 149.287234, 147.456338, 145.601478, 143.723614, 141.823705, 139.902711, 137.961592, 136.001307, 134.023128, 132.027696, 130.015965, 127.988887, 125.947415, 123.892503, 121.825103, 119.746169, 117.656654, 115.55751, 113.44969, 111.334148, 109.211837, 107.083709, 104.950718, 102.813817, 100.789369, 98.9909678, 97.1815397, 95.3545564, 93.5034891, 91.6218092, 89.7029883, 87.7404976, 85.7278087, 83.6583928, 81.5257215, 79.3232661, 77.044498, 74.6828886, 72.2319094, 69.6850317, 67.0357269, 63.4409043, 59.3299124, 55.1421543, 50.8982496, 46.6188178, 42.3244782, 38.0358504, 33.7735538, 29.5582079, 25.4104322, 21.350846, 17.4000688, 13.5787202, 9.90741957, 6.40678634, 3.09744, 0, 3.01374059, 5.40345901, 7.2156908, 8.49697151, 9.29383667, 9.65282184, 9.62046257, 9.24329439, 8.56785285, 7.6406735, 6.50829189, 5.21724355, 3.81406404, 2.34528889, 0.857453662, -0.602906108, -1.92269166, -3.04585843, -4.0764305, -4.95360878, -5.61659419, -6.00458764, -6.05679003, -5.71240227, -4.91062528, -3.59065996, -1.69170723, 0.847032014, 4.08635686, 7.88200702, 12.5750294, 18.2877409, 25.1424584, 32.6979335, 41.3582657, 51.6778387, 63.9480391, 78.4602532, 95.5058677, 115.376269, 138.754775, 165.421558, 195.591321, 229.478769, 267.298606, 303.758569, 345.957072, 395.485559, 453.935476, 522.898269, 607.23968, 701.27494, 800.62331, 904.272333, 1018.08648, 1144.84262, 1287.31762, 1449.97083, 1633.51427, 1836.8245, 2060.75797, 2306.43824, 2573.84188, 2864.104, 3180.32284, 3524.69554, 3899.98777, 4308.54404, 4750.62433, 5227.66775, 5736.0352, 6283.02528, 6871.83496, 7505.6612, 8187.70096, 9134.16284, 10081.9793, 10981.0943, 11781.452, 12432.9964, 12667.4414, 12798.448, 12869.4875, 13039.5976, 13306.5045, 13619.7817, 13979.563, 14385.9824, 14839.1735, 15339.2702, 15164.7517, 15550.3516, 16344.0859, 17393.9708, 18548.0221, 19654.2558, 20560.688, 21409.7985, 21222.1545}, 
{188.502602, 188.968143, 189.342427, 189.627264, 189.824466, 189.935842, 189.963204, 189.908362, 189.773126, 189.559307, 189.268716, 188.903163, 188.464459, 187.954415, 187.37484, 186.727547, 186.014344, 185.237043, 184.397454, 183.497389, 182.538657, 181.523069, 180.452435, 179.328567, 178.153275, 176.928369, 175.655661, 174.33696, 172.974077, 171.568823, 170.123008, 168.638443, 167.116939, 165.560305, 163.999415, 162.420936, 160.811188, 159.171039, 157.501356, 155.803006, 154.076857, 152.323777, 150.544633, 148.740292, 146.911623, 145.059492, 143.184766, 141.288315, 139.371004, 137.433702, 135.477276, 133.493489, 131.492511, 129.475406, 127.44324, 125.397076, 123.337979, 121.267015, 119.185249, 117.093745, 114.993567, 112.885782, 110.771454, 108.651647, 106.527426, 104.399857, 102.270003, 100.259213, 98.485945, 96.7031909, 94.9042183, 93.0822947, 91.2306876, 89.3426643, 87.4114924, 85.4304393, 83.3927724, 81.2917592, 79.1206672, 76.8727637, 74.5413162, 72.1195922, 69.6008591, 66.9783844, 63.392779, 59.281573, 55.092545, 50.8466327, 46.5647739, 42.2679064, 37.9769681, 33.7128967, 29.4966299, 25.3491057, 21.2912617, 17.3440358, 13.5283657, 9.86518934, 6.37544439, 3.08006868, 0, 3.39399473, 6.09761122, 8.16272344, 9.64120539, 10.584931, 11.0457743, 11.0756093, 10.7263099, 10.0497502, 9.09780399, 7.92234539, 6.57524834, 5.10838683, 3.57363483, 2.02286632, 0.507955287, -0.908887079, -2.17431005, -3.25982145, -4.11133218, -4.67475313, -4.89599523, -4.72096935, -4.09558642, -2.96575733, -1.27739298, 1.02359572, 3.99129787, 7.67980256, 11.9571364, 17.1316738, 23.3133228, 30.6119912, 38.3913224, 47.1343563, 57.5231372, 69.8914521, 84.5730882, 101.901833, 122.211473, 146.618662, 174.436058, 205.844278, 241.02394, 280.155662, 319.047995, 363.34664, 414.325232, 473.257405, 541.416792, 621.703052, 711.776435, 810.100386, 916.672726, 1034.62397, 1165.6781, 1311.55913, 1473.93212, 1654.59339, 1855.40354, 2079.28924, 2325.59936, 2593.19932, 2882.72383, 3197.80557, 3540.70111, 3914.52777, 4321.76529, 4761.73724, 5235.5525, 5736.48519, 6274.68429, 6853.97072, 7478.16537, 8151.08916, 9098.79856, 10047.32, 10944.9156, 11739.8472, 12380.377, 12588.4266, 12689.4919, 12728.8379, 12871.6051, 13115.608, 13408.7122, 13751.2293, 14143.4709, 14585.7486, 15078.3738, 15201.1967, 15677.2689, 16414.854, 17322.2156, 18307.6174, 19279.3231, 20145.5961, 20953.3619, 20339.8609}, 
{188.855419, 189.290113, 189.634423, 189.890139, 190.059049, 190.142945, 190.143615, 190.06285, 189.90244, 189.664174, 189.349843, 188.961235, 188.500142, 187.968353, 187.367657, 186.699845, 185.966706, 185.170031, 184.311608, 183.393229, 182.416683, 181.38376, 180.296249, 179.15594, 177.964624, 176.724091, 175.436129, 174.102529, 172.725081, 171.305575, 169.8458, 168.347546, 166.812604, 165.242762, 163.66704, 162.07304, 160.448038, 158.79294, 157.108652, 155.396082, 153.656134, 151.889716, 150.097733, 148.281092, 146.440699, 144.57746, 142.692282, 140.786071, 138.859732, 136.914173, 134.9503, 132.96108, 130.95553, 128.934729, 126.899754, 124.851685, 122.7916, 120.720576, 118.639693, 116.550029, 114.452661, 112.348669, 110.23913, 108.125124, 106.007728, 103.888021, 101.767081, 99.7680079, 98.0109713, 96.2453899, 94.4644322, 92.6612662, 90.8290602, 88.9609824, 87.050201, 85.0898843, 83.0732004, 80.9933176, 78.8434041, 76.616628, 74.3061577, 71.9051613, 69.406807, 66.8042631, 63.2277425, 59.1200421, 54.934331, 50.6917819, 46.4135676, 42.1208609, 37.8348345, 33.5766613, 29.3675139, 25.2285652, 21.180988, 17.2459549, 13.4446389, 9.79821261, 6.32784885, 3.05472039, 0, 3.81210897, 6.85984072, 9.20100192, 10.8933993, 11.9948394, 12.5631291, 12.656075, 12.3314837, 11.647162, 10.6609166, 9.43055404, 8.01388113, 6.46870451, 4.85283088, 3.22406691, 1.6402193, 0.0217593359, -1.59312563, -2.83598936, -3.67845419, -4.09214243, -4.04867643, -3.51967849, -2.47677096, -0.891576143, 1.26428362, 4.01918601, 7.4015087, 11.4396294, 15.7911872, 20.9912359, 27.1793745, 34.4952022, 42.4246977, 51.4342706, 62.164629, 74.9514584, 90.1304441, 108.037272, 129.007626, 154.255298, 182.970618, 215.317469, 251.459733, 291.561292, 331.767026, 377.26457, 429.222557, 488.809622, 557.194397, 636.609661, 725.859278, 824.369317, 932.484188, 1052.42324, 1185.56402, 1333.28406, 1496.6168, 1677.36209, 1877.69517, 2100.41818, 2345.03575, 2610.39537, 2897.206, 3209.33079, 3549.18508, 3920.44596, 4325.85587, 4763.53085, 5234.20392, 5730.45634, 6262.42696, 6833.67053, 7447.74181, 8108.19555, 9033.66457, 9958.85605, 10833.5552, 11607.5473, 12230.6176, 12432.8256, 12530.166, 12566.5923, 12699.7574, 12928.6085, 13204.7184, 13529.4032, 13903.9789, 14329.7618, 14808.0679, 15122.2136, 15653.2448, 16349.7475, 17160.3078, 18033.5116, 18917.945, 19762.1941, 20369.6008, 19217.301}, 
{189.594339, 189.981866, 190.281297, 190.494342, 190.62271, 190.668111, 190.632256, 190.516854, 190.323616, 190.054251, 189.71047, 189.293983, 188.806499, 188.249728, 187.625381, 186.935167, 186.180797, 185.363981, 184.486428, 183.549849, 182.555953, 181.506451, 180.403052, 179.247467, 178.041405, 176.786577, 175.484693, 174.137462, 172.746595, 171.313802, 169.840792, 168.329275, 166.780963, 165.197564, 163.603023, 161.987465, 160.340748, 158.663861, 156.957791, 155.223528, 153.46206, 151.674376, 149.861463, 148.024311, 146.163907, 144.28124, 142.377299, 140.453072, 138.509548, 136.547715, 134.568561, 132.569309, 130.554794, 128.526088, 126.484261, 124.430381, 122.36552, 120.290748, 118.207134, 116.115749, 114.017663, 111.913946, 109.805668, 107.693899, 105.579709, 103.464169, 101.348348, 99.3562143, 97.608785, 95.8536629, 94.0839512, 92.2927529, 90.4731712, 88.6183092, 86.7212702, 84.7751572, 82.7730734, 80.708122, 78.573406, 76.3620287, 74.0670931, 71.6817025, 69.1989599, 66.6119686, 63.0432665, 58.9395506, 54.7579812, 50.519913, 46.2467006, 41.9596987, 37.6802618, 33.4297448, 29.2295022, 25.1008887, 21.065259, 17.1439676, 13.3583694, 9.7298188, 6.27967062, 3.02927947, 0, 4.16720985, 7.50733226, 10.0832775, 11.9579559, 13.1942776, 13.8551531, 14.0034925, 13.7022061, 13.0142043, 12.0023972, 10.7296952, 9.25900862, 7.65324762, 5.97532254, 4.28814364, 2.65462121, 0.85542769, -1.08684633, -2.47089546, -3.29428897, -3.5545961, -3.24938612, -2.37622829, -0.932691854, 1.08365392, 3.67523978, 6.84449647, 10.5938547, 14.9257453, 19.2639228, 24.4016753, 30.5150365, 37.7800398, 45.8745053, 55.2235734, 66.3852413, 79.6850069, 95.4483677, 114.000821, 135.667866, 161.608352, 191.060795, 224.187645, 261.151351, 302.114365, 343.474917, 390.100732, 443.095313, 503.562165, 572.604794, 652.092837, 741.42728, 840.493199, 949.804713, 1071.16024, 1205.78157, 1354.8905, 1519.4624, 1701.0215, 1901.36083, 2122.57855, 2364.86341, 2627.38954, 2911.10069, 3219.93922, 3556.47108, 3924.89214, 4328.19094, 4763.37956, 5230.85069, 5723.38495, 6250.15821, 6814.19809, 7418.53225, 8066.18833, 8964.09383, 9860.40156, 10707.1642, 11456.4345, 12060.265, 12260.8304, 12359.9802, 12399.7147, 12527.6469, 12743.9722, 13004.881, 13312.7691, 13670.0321, 14079.0659, 14542.2662, 15021.0823, 15597.5574, 16255.8596, 16980.1569, 17754.6173, 18563.4088, 19390.6994, 19727.6243, 18021.4256}, 
{190.764245, 191.095524, 191.341588, 191.504041, 191.584488, 191.584533, 191.50578, 191.349833, 191.118296, 190.812775, 190.434873, 189.986194, 189.468342, 188.882923, 188.23154, 187.515797, 186.737299, 185.89765, 184.998455, 184.041316, 183.02784, 181.95963, 180.83829, 179.665424, 178.442637, 177.171534, 175.853718, 174.490793, 173.084365, 171.636036, 170.147412, 168.620097, 167.055694, 165.455809, 163.837948, 162.195429, 160.52138, 158.81689, 157.083045, 155.320935, 153.531648, 151.716272, 149.875894, 148.011605, 146.12449, 144.21564, 142.286141, 140.337082, 138.369552, 136.384638, 134.383429, 132.368346, 130.339114, 128.296794, 126.242446, 124.177127, 122.101898, 120.017817, 117.925945, 115.82734, 113.723062, 111.61417, 109.501724, 107.386782, 105.270404, 103.15365, 101.037578, 99.0467353, 97.3027025, 95.5518567, 93.7872522, 92.0019431, 90.1889837, 88.3414281, 86.4523304, 84.5147449, 82.5217256, 80.4663269, 78.3416027, 76.1406074, 73.8563951, 71.48202, 69.0105361, 66.4349978, 62.872448, 58.7723722, 54.5947848, 50.3611682, 46.0930049, 41.8117773, 37.5389678, 33.2960588, 29.1045327, 24.985872, 20.9615591, 17.0530763, 13.2819062, 9.66953104, 6.23743336, 3.00709554, 0, 4.40028149, 7.93291458, 10.6642742, 12.6607352, 13.9886724, 14.7144609, 14.9044756, 14.6250912, 13.9426827, 12.923625, 11.6342931, 10.1410618, 8.51030598, 6.80840061, 5.10172056, 3.45664074, 1.5582028, -0.581694533, -2.03413687, -2.81446357, -2.938014, -2.42012752, -1.27614351, 0.478598668, 2.82875966, 5.75900009, 9.2539806, 13.2983618, 17.8768044, 22.2400258, 27.3757423, 33.4887975, 40.7840349, 49.1088821, 58.8468907, 70.4769232, 84.3110481, 100.661334, 119.839849, 142.158661, 168.621234, 198.637818, 232.385206, 270.040196, 311.779581, 354.293141, 402.116443, 456.298036, 517.88647, 587.930296, 668.074176, 758.041964, 857.909341, 968.263511, 1090.73602, 1226.48953, 1376.6867, 1542.47732, 1725.03984, 1925.56674, 2145.33256, 2385.30433, 2645.07402, 2925.84041, 3231.52489, 3564.7991, 3930.22819, 4330.97471, 4763.25851, 5227.22664, 5716.44013, 6238.64526, 6796.26886, 7391.73772, 8027.47868, 8898.58164, 9766.64453, 10585.9284, 11310.6943, 11895.2032, 12093.1932, 12193.1304, 12235.2141, 12358.0853, 12562.977, 12810.1052, 13102.754, 13444.2072, 13837.749, 14286.6634, 14904.8852, 15518.8072, 16142.6283, 16790.5476, 17476.7638, 18215.4761, 19020.8832, 19038.1282, 16817.5622}, 
{192.497459, 192.772543, 192.964753, 193.075606, 193.106622, 193.05932, 192.935219, 192.735838, 192.462696, 192.117312, 191.701205, 191.215894, 190.662898, 190.043737, 189.359929, 188.612994, 187.80445, 186.935816, 186.008612, 185.024356, 183.984568, 182.890767, 181.744472, 180.547201, 179.300474, 178.00581, 176.664728, 175.278747, 173.849386, 172.378165, 170.866601, 169.316214, 167.728524, 166.105049, 164.45807, 162.783498, 161.077117, 159.340096, 157.573605, 155.778814, 153.956893, 152.109011, 150.236339, 148.340046, 146.421302, 144.481277, 142.52114, 140.542062, 138.545212, 136.531761, 134.502878, 132.464331, 130.412592, 128.348731, 126.273818, 124.188924, 122.095118, 119.993471, 117.885053, 115.770934, 113.652183, 111.529872, 109.405071, 107.278848, 105.152276, 103.026423, 100.90236, 98.9057409, 97.1592275, 95.4069461, 93.6418907, 91.857055, 90.045433, 88.2000186, 86.3138057, 84.3797881, 82.3909599, 80.3403148, 78.2208469, 76.0255498, 73.7474177, 71.3794443, 68.9146236, 66.3459495, 62.7875763, 58.6904028, 54.515891, 50.2855548, 46.0209081, 41.7434648, 37.4747389, 33.2362443, 29.049495, 24.9360048, 20.9172876, 17.0148576, 13.2502284, 9.64491416, 6.22042871, 2.99828602, 0, 4.28380754, 7.72312602, 10.3830516, 12.3286803, 13.6251084, 14.3374319, 14.5307471, 14.2701499, 13.6207367, 12.6476034, 11.4158463, 9.99056148, 8.43684504, 6.81979315, 5.20450193, 3.65606751, 1.94163293, 0.0837295975, -1.17860482, -1.84412127, -1.91157072, -1.37970412, -0.247272447, 1.48697335, 3.82428231, 6.76590348, 10.3130859, 14.4670786, 19.2291306, 23.9869909, 29.5803587, 36.1945329, 44.0148127, 52.7947774, 62.935586, 74.9535224, 89.1634017, 105.880039, 125.418248, 148.092846, 174.702301, 204.930574, 238.997852, 277.124322, 319.530169, 363.518228, 412.955376, 468.79114, 531.975045, 603.456615, 684.557536, 775.400312, 876.264771, 987.805489, 1111.44215, 1248.25092, 1399.30796, 1565.61975, 1748.34798, 1948.73038, 2168.02186, 2407.15448, 2665.69909, 2944.82356, 3248.40186, 3579.06584, 3941.26283, 4338.09534, 4766.00908, 5225.21523, 5710.5194, 6228.36995, 6780.84921, 7370.03949, 7998.02312, 8856.68411, 9710.85269, 10515.1607, 11224.2402, 11792.7229, 11975.201, 12059.706, 12086.6014, 12195.0291, 12386.2525, 12620.3772, 12900.7246, 13230.616, 13613.3728, 14052.3162, 14791.7842, 15437.4308, 16028.7197, 16605.1146, 17206.0789, 17871.0765, 18639.571, 18288.9917, 15721.2087}, 
{194.832096, 195.055485, 195.197257, 195.258889, 195.241859, 195.147641, 194.977714, 194.733552, 194.416634, 194.028435, 193.570432, 193.044102, 192.450921, 191.792365, 191.069911, 190.285036, 189.439217, 188.533929, 187.570649, 186.550854, 185.47602, 184.347624, 183.167142, 181.936051, 180.655828, 179.327949, 177.95389, 176.535128, 175.07314, 173.569402, 172.02539, 170.442582, 168.822453, 167.166481, 165.484175, 163.772828, 162.029631, 160.255802, 158.452556, 156.621108, 154.762674, 152.878471, 150.969714, 149.037619, 147.083401, 145.108277, 143.113462, 141.100172, 139.069622, 137.02303, 134.961609, 132.891255, 130.808404, 128.714169, 126.609666, 124.496008, 122.374311, 120.245688, 118.111254, 115.972123, 113.82941, 111.684229, 109.537695, 107.390921, 105.245023, 103.101114, 100.960309, 98.9502612, 97.1955835, 95.4364157, 93.6656694, 91.8762558, 90.0610866, 88.213073, 86.3251267, 84.3901589, 82.4010813, 80.3508051, 78.2322419, 76.0383031, 73.7619002, 71.3959446, 68.9333477, 66.367021, 62.81053, 58.714955, 54.5419453, 50.3129489, 46.0494136, 41.7727874, 37.5045181, 33.2660537, 29.0788421, 24.9643313, 20.943969, 17.0392033, 13.2714821, 9.66225324, 6.23296465, 3.00506427, 0, 3.77841618, 6.80644929, 9.14263622, 10.8455139, 11.9736192, 12.5854891, 12.7396605, 12.4946703, 11.9090553, 11.0413526, 9.95009901, 8.69383144, 7.33108681, 5.92040203, 4.52031402, 3.18935969, 1.98270208, 0.958396132, 0.182610103, -0.28684206, -0.392146418, -0.0754890297, 0.720944044, 2.05496674, 3.98439301, 6.56703678, 9.86071201, 13.9232326, 18.8124126, 24.4192344, 31.029515, 38.7511176, 47.6919056, 57.1883309, 67.7339626, 80.0280794, 94.4099681, 111.218915, 130.794208, 153.475133, 179.841132, 209.918246, 243.998775, 282.375017, 325.339275, 371.167919, 422.673159, 480.651276, 545.898551, 619.211267, 701.431646, 793.253876, 895.267606, 1008.21893, 1133.17337, 1271.05301, 1422.77997, 1588.72848, 1770.49302, 1970.26571, 2190.36501, 2430.5992, 2689.90978, 2969.05515, 3271.87226, 3600.78496, 3959.56399, 4350.98236, 4772.87459, 5225.8687, 5706.29605, 6219.74232, 6768.32283, 7354.15289, 7979.34779, 8843.89247, 9702.56523, 10508.014, 11212.8866, 11769.8311, 11920.6876, 11969.4502, 11959.1129, 12041.0959, 12215.4747, 12437.1469, 12708.4469, 13031.7089, 13409.2672, 13843.456, 14677.4158, 15347.5385, 15910.8914, 16424.5418, 16945.557, 17531.0043, 18237.951, 17485.7726, 14780.9934}, 
{197.624867, 197.796465, 197.887235, 197.898629, 197.832099, 197.689098, 197.471078, 197.179491, 196.815789, 196.381424, 195.877849, 195.306515, 194.668875, 193.966381, 193.200486, 192.37264, 191.484298, 190.53691, 189.531929, 188.470807, 187.354996, 186.185949, 184.965117, 183.693953, 182.37391, 181.006438, 179.592991, 178.13502, 176.633978, 175.091317, 173.508489, 171.886947, 170.228141, 168.533526, 166.810984, 165.058616, 163.274447, 161.45972, 159.615677, 157.743564, 155.844622, 153.920097, 151.97123, 149.999266, 148.005447, 145.991018, 143.957222, 141.905303, 139.836503, 137.752066, 135.653237, 133.544177, 131.423147, 129.291328, 127.1499, 125.000042, 122.842936, 120.67976, 118.511696, 116.339923, 114.165622, 111.989973, 109.814156, 107.639351, 105.466738, 103.297498, 101.13281, 99.1026985, 97.3340941, 95.5624463, 93.7805827, 91.9813312, 90.1575195, 88.3019753, 86.4075264, 84.4670006, 82.4732255, 80.4190289, 78.2972386, 76.1006823, 73.8221878, 71.4545828, 68.990695, 66.4233523, 62.8661125, 58.7699325, 54.5963373, 50.3666889, 46.1023489, 41.824679, 37.555041, 33.3147967, 29.1253078, 25.007936, 20.9840431, 17.0749908, 13.3021409, 9.68685512, 6.25049521, 3.01442293, 0, 3.20704487, 5.7698482, 7.73951411, 9.16714671, 10.1038501, 10.6007285, 10.7088859, 10.4794265, 9.96345437, 9.21207367, 8.27638851, 7.20750299, 6.05652126, 4.87454741, 3.71268558, 2.62203988, 1.97697321, 1.87476952, 1.63535236, 1.37909557, 1.22637303, 1.29755859, 1.7130261, 2.59314943, 4.05830243, 6.22885897, 9.22519289, 13.1676781, 18.1766883, 24.6969471, 32.4055505, 41.3255676, 51.4800673, 61.7545882, 72.7409644, 85.3343717, 99.8991382, 116.799592, 136.400061, 159.064874, 185.146566, 215.024847, 249.066352, 287.637718, 331.105579, 378.648794, 432.118719, 492.178935, 559.493022, 634.724558, 718.363535, 811.459286, 914.743603, 1028.99051, 1155.06027, 1293.77442, 1445.9545, 1611.41241, 1792.20879, 1991.50576, 2212.80655, 2454.64789, 2715.25996, 2994.93127, 3297.43802, 3624.9555, 3980.45882, 4366.33065, 4782.02095, 5228.6386, 5704.01147, 6213.03419, 6757.95132, 7341.00748, 7964.44723, 8838.6132, 9705.62743, 10515.71, 11219.081, 11765.9606, 11882.9183, 11892.9249, 11842.0508, 11896.3254, 12054.3351, 12264.6833, 12528.4555, 12846.7372, 13220.6139, 13651.1711, 14514.0391, 15192.5429, 15743.7116, 16224.5739, 16692.1589, 17203.4953, 17815.6121, 16668.0716, 13916.7747}, 
{200.836347, 200.956641, 200.996377, 200.957003, 200.839968, 200.646718, 200.378704, 200.037372, 199.624172, 199.140551, 198.587957, 197.967839, 197.281645, 196.530823, 195.716821, 194.841088, 193.905072, 192.910221, 191.857983, 190.749806, 189.587138, 188.371429, 187.104125, 185.786676, 184.420529, 183.007132, 181.547934, 180.044384, 178.497928, 176.910016, 175.282095, 173.615615, 171.912022, 170.172765, 168.405395, 166.60818, 164.779316, 162.920052, 161.03164, 159.115329, 157.17237, 155.204012, 153.211507, 151.196103, 149.159052, 147.101603, 145.025007, 142.930514, 140.819374, 138.692837, 136.552154, 134.3977, 132.231619, 130.05518, 127.869652, 125.676304, 123.476406, 121.271226, 119.062033, 116.850097, 114.636687, 112.423073, 110.210522, 108.000304, 105.793689, 103.591946, 101.396343, 99.3396352, 97.5514187, 95.7617854, 93.963481, 92.1492509, 90.3118407, 88.4439958, 86.5384617, 84.5879841, 82.5853083, 80.52318, 78.3943445, 76.1915476, 73.9075345, 71.535051, 69.0668425, 66.4956544, 62.9347081, 58.8350109, 54.6581411, 50.4253866, 46.1580351, 41.8773744, 37.6046922, 33.3612764, 29.1684146, 25.0473947, 21.0195044, 17.1060316, 13.3282638, 9.70748906, 6.26499498, 3.02206937, 0, 2.68518809, 4.82326813, 6.45861407, 7.63559985, 8.39859944, 8.79198677, 8.8601358, 8.64742047, 8.19821473, 7.55689255, 6.76782785, 5.87539459, 4.92396673, 3.9579182, 3.02162297, 2.15945498, 2.03760443, 2.78927618, 3.05235501, 3.00446121, 2.82321507, 2.68623687, 2.77114692, 3.25556551, 4.31711292, 6.13340944, 8.88207538, 12.740731, 17.8869966, 25.267042, 34.0081364, 44.0573355, 55.3616945, 66.4115418, 77.8822965, 90.8378381, 105.662241, 122.739577, 142.453922, 165.189348, 191.086602, 220.847141, 254.902646, 293.6848, 337.625282, 386.57224, 441.686774, 503.546448, 572.728827, 649.811477, 735.147168, 829.813008, 934.451164, 1049.77959, 1176.67098, 1315.92853, 1468.35548, 1633.41204, 1813.54965, 2012.68477, 2235.37536, 2478.98325, 2741.07337, 3021.48689, 3323.92201, 3650.30648, 4002.8477, 4383.54592, 4793.37605, 5233.89297, 5704.15434, 6208.59649, 6749.63874, 7329.70039, 7951.20078, 8834.76034, 9710.047, 10524.9298, 11227.2778, 11764.9599, 11849.8373, 11823.1256, 11733.8404, 11762.2331, 11905.3714, 12105.6414, 12362.845, 12676.7838, 13047.2599, 13474.075, 14278.7489, 14944.8838, 15505.9041, 15995.2346, 16446.2999, 16892.5246, 17367.3334, 15855.9979, 13122.1393}, 
{204.39771, 204.47268, 204.466252, 204.379917, 204.215164, 203.973483, 203.656365, 203.2653, 202.801777, 202.267287, 201.66332, 200.991366, 200.252914, 199.449455, 198.58248, 197.653477, 196.663938, 195.615352, 194.509209, 193.346999, 192.130212, 190.860339, 189.538869, 188.167293, 186.7471, 185.279781, 183.766826, 182.209724, 180.609966, 178.969042, 177.288441, 175.569655, 173.814172, 172.023484, 170.207757, 168.36396, 166.488959, 164.58396, 162.650174, 160.688807, 158.70107, 156.688169, 154.651314, 152.591713, 150.510574, 148.409106, 146.288517, 144.150017, 141.994812, 139.824112, 137.639126, 135.432112, 133.213423, 130.98446, 128.746625, 126.501323, 124.249954, 121.993921, 119.734626, 117.473473, 115.211863, 112.9512, 110.692884, 108.438319, 106.188908, 103.946052, 101.711155, 99.6204738, 97.8070308, 95.9940969, 94.1743324, 92.3403975, 90.4849526, 88.6006578, 86.6801734, 84.7161596, 82.7012767, 80.628185, 78.4895446, 76.2780159, 73.9862591, 71.6069344, 69.1327021, 66.5562224, 62.9861655, 58.8757802, 54.6889402, 50.446993, 46.171286, 41.8831666, 37.6039822, 33.3550803, 29.1578083, 25.0335137, 21.0035438, 17.089246, 13.3119679, 9.69305682, 6.25386019, 3.01572544, 0, 2.61258858, 4.69284531, 6.28462698, 7.43179036, 8.17819223, 8.56768938, 8.64413857, 8.45139659, 8.03332021, 7.43376622, 6.6965914, 5.86565252, 4.98480636, 4.09790971, 3.24881933, 2.48139202, 2.52472152, 3.52055562, 3.96281031, 4.04217885, 3.94935453, 3.87503062, 4.00990039, 4.54465712, 5.66999408, 7.57660456, 10.4551818, 14.4964191, 19.8910098, 27.6791302, 36.8905133, 47.4610075, 59.326461, 70.9549175, 83.0161275, 96.5712565, 111.996494, 129.668031, 149.962056, 173.254759, 199.571537, 229.746136, 264.223379, 303.448089, 347.86509, 397.016162, 452.474932, 514.911981, 584.997894, 663.403254, 751.083563, 848.07625, 954.220128, 1069.92452, 1196.76354, 1335.78833, 1488.05003, 1653.54576, 1834.62025, 2034.76812, 2258.55697, 2503.11655, 2765.80293, 3046.30703, 3348.27607, 3673.54126, 4024.09107, 4401.79751, 4807.95585, 5244.1876, 5709.53549, 6208.59652, 6743.88484, 7317.9146, 7933.19993, 8813.3693, 9684.54397, 10493.9595, 11188.8515, 11716.4555, 11784.2, 11738.999, 11630.8625, 11645.3473, 11778.8094, 11971.1272, 12221.4806, 12529.0497, 12893.0144, 13312.5548, 13825.1807, 14425.6943, 15060.1585, 15674.6365, 16215.1914, 16627.8863, 16858.7843, 15101.0813, 12382.9869}, 
{208.276248, 208.311333, 208.263147, 208.133265, 207.923263, 207.634714, 207.269193, 206.828275, 206.313534, 205.726546, 205.068884, 204.342124, 203.54784, 202.687607, 201.762998, 200.77559, 199.726956, 198.618671, 197.45231, 196.229448, 194.951658, 193.620516, 192.237597, 190.804474, 189.322723, 187.793918, 186.219634, 184.601445, 182.940927, 181.239653, 179.499198, 177.721138, 175.907046, 174.058497, 172.191035, 170.298956, 168.376389, 166.424455, 164.444274, 162.43697, 160.403662, 158.345472, 156.263522, 154.158932, 152.032824, 149.886319, 147.720539, 145.536605, 143.335639, 141.11876, 138.887092, 136.62072, 134.342257, 132.053279, 129.755364, 127.45009, 125.139033, 122.823771, 120.505881, 118.186941, 115.868527, 113.552218, 111.239589, 108.93222, 106.631686, 104.339565, 102.057435, 99.9257212, 98.0815155, 96.2400308, 94.3938435, 92.5355299, 90.6576662, 88.7528289, 86.8135941, 84.8325382, 82.8022376, 80.7152685, 78.5642073, 76.3416302, 74.0401136, 71.6522337, 69.170567, 66.5876897, 63.0032758, 58.8751338, 54.6718041, 50.4148395, 46.1257926, 41.8262162, 37.537663, 33.2816856, 29.0798368, 24.9536692, 20.9247357, 17.0145889, 13.2447815, 9.63686615, 6.21239567, 2.99292272, 0, 3.03980786, 5.47060778, 7.34262983, 8.70610409, 9.61126066, 10.1083296, 10.247541, 10.079125, 9.65331152, 9.02033081, 8.23041288, 7.33378783, 6.38068573, 5.42133667, 4.50597073, 3.68481799, 3.49506592, 4.05650995, 4.31792891, 4.43390092, 4.55900406, 4.84781641, 5.45491609, 6.53488118, 8.24228977, 10.73172, 14.1577499, 18.6749575, 24.4379211, 32.128227, 41.1802086, 51.5903414, 63.355101, 75.3306381, 88.0835906, 102.484683, 118.872489, 137.585581, 158.962533, 183.341916, 210.739895, 241.915577, 277.270616, 317.206665, 362.125377, 410.256943, 464.717344, 526.451101, 596.402733, 675.516758, 766.079293, 866.053533, 973.792576, 1089.15799, 1215.09111, 1353.15054, 1504.89486, 1671.76473, 1855.46348, 2057.82307, 2282.27593, 2526.76962, 2788.97131, 3068.75788, 3369.75049, 3693.85161, 4043.44108, 4420.54514, 4825.4397, 5259.39076, 5720.11312, 6212.97032, 6740.45762, 7305.07027, 7909.30355, 8771.2961, 9623.98893, 10415.9665, 11095.8131, 11612.1133, 11679.7048, 11636.7497, 11532.0096, 11546.0386, 11675.4304, 11861.865, 12104.6819, 12403.2202, 12756.8195, 13164.8189, 13158.6324, 13641.9784, 14411.7053, 15264.6618, 15997.6964, 16407.6575, 16291.3936, 14416.2533, 11694.6601}, 
{212.481508, 212.474547, 212.382267, 212.206332, 211.94841, 211.610165, 211.193263, 210.699368, 210.130147, 209.487265, 208.772387, 207.987179, 207.133305, 206.212433, 205.226226, 204.176351, 203.064473, 201.892256, 200.661368, 199.373473, 198.030236, 196.633323, 195.184399, 193.68513, 192.137182, 190.542219, 188.901907, 187.217912, 185.491898, 183.725532, 181.920478, 180.078403, 178.200972, 176.289849, 174.366355, 172.421913, 170.447715, 168.444786, 166.414155, 164.356848, 162.273895, 160.166322, 158.035157, 155.881428, 153.706162, 151.510386, 149.29513, 147.061419, 144.810282, 142.542747, 140.25984, 137.928667, 135.584914, 133.230343, 130.866718, 128.495801, 126.119357, 123.739146, 121.356934, 118.974482, 116.593554, 114.215912, 111.84332, 109.477541, 107.120338, 104.773473, 102.438711, 100.260572, 98.3801549, 96.5047887, 94.6269822, 92.739244, 90.8340825, 88.9040062, 86.9415234, 84.9391427, 82.8893724, 80.7847211, 78.6176972, 76.3808091, 74.0665653, 71.6674742, 69.1760443, 66.5847841, 62.9839718, 58.8362083, 54.6149382, 50.3419491, 46.0390287, 41.7279643, 37.4305436, 33.1685542, 28.9637835, 24.8380192, 20.8130487, 16.9106596, 13.1526396, 9.56077602, 6.15685654, 2.96266869, 0, 3.60891903, 6.50677425, 8.75208325, 10.4033636, 11.5191329, 12.1579086, 12.3782085, 12.2385499, 11.7974506, 11.1134281, 10.245, 9.25068377, 8.18899707, 7.11845747, 6.09758253, 5.18488982, 4.65046592, 4.58305255, 4.56261835, 4.69301712, 5.07810263, 5.82172868, 7.02774906, 8.80001756, 11.242388, 14.458714, 18.5528496, 23.6286484, 29.7899643, 37.2246787, 45.9218076, 55.9599965, 67.4178908, 79.6603347, 93.1228744, 108.431403, 125.878706, 145.75757, 168.360781, 193.981123, 222.657495, 255.013841, 291.392622, 332.136298, 377.587328, 424.463462, 477.638048, 538.359724, 607.877128, 687.438897, 780.749339, 883.600046, 992.807935, 1107.67402, 1232.571, 1369.5945, 1520.84015, 1689.34537, 1876.04992, 2080.86615, 2305.80061, 2549.9107, 2811.35261, 3090.28933, 3390.28439, 3713.34041, 4062.36281, 4439.58829, 4843.94329, 5276.22674, 5732.50831, 6219.04372, 6738.26893, 7292.61989, 7884.53257, 8723.90446, 9553.84459, 10324.9235, 10987.7118, 11492.78, 11566.2487, 11532.7717, 11439.1126, 11457.4743, 11584.967, 11766.4346, 12001.6008, 12290.189, 12631.9231, 13026.5266, 12461.8239, 12818.1804, 13732.5268, 14841.7936, 15782.9115, 16192.8111, 15708.4229, 13769.2912, 11063.5486}, 
{216.994077, 216.940148, 216.798998, 216.572374, 216.262025, 215.869701, 215.39715, 214.84612, 214.218361, 213.515622, 212.739652, 211.892198, 210.97501, 209.989838, 208.938429, 207.822532, 206.643897, 205.404272, 204.105406, 202.749048, 201.336947, 199.870851, 198.352509, 196.78367, 195.166084, 193.501498, 191.791662, 190.038324, 188.243234, 186.408139, 184.53479, 182.624934, 180.680321, 178.7027, 176.718689, 174.717079, 172.686362, 170.62748, 168.541374, 166.428985, 164.291257, 162.129129, 159.943545, 157.735446, 155.505773, 153.255469, 150.985474, 148.696731, 146.390182, 144.066768, 141.727431, 139.326692, 136.91292, 134.488063, 132.054068, 129.612884, 127.166459, 124.716741, 122.265678, 119.815218, 117.367308, 114.923898, 112.486935, 110.058367, 107.640142, 105.234209, 102.842514, 100.613303, 98.6913052, 96.7767458, 94.8620891, 92.9397994, 91.0023408, 89.0421778, 87.0517745, 85.0235954, 82.9501046, 80.8237665, 78.6370454, 76.3824055, 74.0523112, 71.6392268, 69.1356164, 66.5339445, 62.9158092, 58.7482667, 54.509321, 50.2209959, 45.9053151, 41.584302, 37.2799805, 33.0143741, 28.8095064, 24.6874013, 20.6700822, 16.7795728, 13.0378969, 9.46707799, 6.08913977, 2.92610589, 0, 4.24195007, 7.65995769, 10.3216406, 12.2946165, 13.6465031, 14.4449183, 14.7574797, 14.651805, 14.195512, 13.4562185, 12.5015421, 11.3991006, 10.2165118, 9.02139338, 7.88136305, 6.86403858, 5.93598727, 5.15039107, 4.80343418, 4.94108067, 5.60929462, 6.85404009, 8.72128116, 11.2569819, 14.5071064, 18.5176187, 23.3344829, 29.003663, 35.5711232, 42.6648743, 50.9020831, 60.4540995, 71.4922738, 83.945218, 98.1056512, 114.331022, 132.865503, 153.953264, 177.838477, 204.765313, 234.81067, 268.436902, 305.920909, 347.53959, 393.569844, 439.248964, 491.153357, 550.819821, 619.785157, 699.586164, 795.223177, 900.536253, 1010.97598, 1125.35046, 1249.32266, 1385.47786, 1536.40132, 1706.65583, 1896.39972, 2103.634, 2328.84908, 2572.33916, 2832.86325, 3110.92322, 3409.97411, 3732.11541, 4080.80471, 4458.49359, 4862.65397, 5293.57454, 5745.62922, 6225.93146, 6736.81738, 7280.62312, 7859.68479, 8673.87399, 9478.60752, 10226.8376, 10871.5166, 11365.5966, 11448.9201, 11429.6225, 11351.9887, 11377.7583, 11504.724, 11681.739, 11909.0428, 12186.8752, 12515.4757, 12895.0839, 11795.8801, 12029.7234, 13080.323, 14431.3878, 15566.6268, 15969.7492, 15124.4639, 13158.9183, 10488.4711}, 
{221.794543, 221.685964, 221.488728, 221.204645, 220.835526, 220.383184, 219.849428, 219.236072, 218.544924, 217.777798, 216.936503, 216.022852, 215.038656, 213.985725, 212.865871, 211.680905, 210.432639, 209.122884, 207.75345, 206.32615, 204.842793, 203.305193, 201.715159, 200.074503, 198.385037, 196.648571, 194.866917, 193.041885, 191.175288, 189.268936, 187.324641, 185.344214, 183.329466, 181.282208, 179.23301, 177.1687, 175.075758, 172.955062, 170.807491, 168.633922, 166.435234, 164.212304, 161.966011, 159.697231, 157.406844, 155.095728, 152.764759, 150.414816, 148.046778, 145.661521, 143.259925, 140.785534, 138.297801, 135.798848, 133.290796, 130.775765, 128.255876, 125.733249, 123.210007, 120.688269, 118.170157, 115.65779, 113.153291, 110.65878, 108.176377, 105.708203, 103.25638, 100.972187, 99.0033224, 97.0442777, 95.0875048, 93.1254552, 91.1505804, 89.1553317, 87.1321607, 85.0735189, 82.9718577, 80.8196285, 78.609283, 76.3332724, 73.9840484, 71.5540624, 69.0357657, 66.4216101, 62.7863436, 58.6005722, 54.3459314, 50.0446541, 45.7189729, 41.3911205, 37.0833297, 32.8178331, 28.6168634, 24.5026533, 20.4974356, 16.623443, 12.9029081, 9.35806363, 6.01114233, 2.88437689, 0, 4.86092902, 8.78877106, 11.8599552, 14.1509104, 15.7380658, 16.6978504, 17.1066933, 17.0410235, 16.5772701, 15.7918622, 14.7612287, 13.5617988, 12.2700014, 10.9622657, 9.71502064, 8.60469535, 7.29669569, 5.80873313, 5.14693195, 5.29964481, 6.25522438, 8.00202331, 10.5283943, 13.8226899, 17.8732629, 22.6684659, 28.1966516, 34.4461726, 41.4053816, 48.1452027, 55.9078077, 64.9567776, 75.5556936, 88.1864987, 103.003593, 120.103148, 139.683235, 161.941927, 187.077298, 215.287419, 246.685754, 281.580736, 320.186991, 362.719145, 409.391826, 454.226694, 505.179583, 564.013361, 632.490895, 712.375055, 809.630287, 916.682616, 1028.00651, 1142.06518, 1265.46558, 1401.15825, 1552.09379, 1724.0643, 1916.533, 2125.86325, 2351.13944, 2593.85437, 2853.41967, 3130.68139, 3428.91601, 3750.2844, 4098.71523, 4476.82769, 4880.75906, 5310.31315, 5758.384, 6232.74825, 6735.60156, 7269.13959, 7835.55802, 8623.88428, 9402.77432, 10127.7158, 10754.1965, 11237.7042, 11332.8077, 11329.8594, 11270.455, 11304.9948, 11432.0061, 11604.681, 11823.8135, 12090.1977, 12404.6275, 12767.8971, 11221.9259, 11352.0304, 12512.794, 14058.8001, 15344.6322, 15724.8738, 14554.1084, 12583.858, 9968.24625}, 
{226.661734, 226.468407, 226.188535, 225.823835, 225.376026, 224.846825, 224.237951, 223.551122, 222.788055, 221.950468, 221.040081, 220.058609, 219.007772, 217.889288, 216.704874, 215.456249, 214.14513, 212.773235, 211.342283, 209.853992, 208.310079, 206.712262, 205.06226, 203.36179, 201.612571, 199.81632, 197.974756, 196.089596, 194.162558, 192.195361, 190.189722, 188.14736, 186.069992, 183.959336, 181.840325, 179.702581, 177.535448, 175.33989, 173.116872, 170.867359, 168.592316, 166.292706, 163.969496, 161.623648, 159.256129, 156.867902, 154.459933, 152.033186, 149.588625, 147.127216, 144.649922, 142.100617, 139.538595, 136.966058, 134.385209, 131.798249, 129.207382, 126.614809, 124.022732, 121.433355, 118.848879, 116.271507, 113.703441, 111.146883, 108.604035, 106.077101, 103.568281, 101.228958, 99.206929, 97.1967281, 95.1908868, 93.1819369, 91.1624098, 89.1248375, 87.0617514, 84.9656834, 82.8291651, 80.6447281, 78.4049042, 76.102225, 73.7292222, 71.2784275, 68.7423726, 66.1135891, 62.4678782, 58.2754191, 54.0174092, 49.71613, 45.3938627, 41.0728888, 36.7754895, 32.5239462, 28.3405403, 24.2475531, 20.267266, 16.4219603, 12.7339173, 9.22541845, 5.91874505, 2.83617845, 0, 5.20537319, 9.42330587, 12.7346016, 15.2200639, 16.9604964, 18.0367025, 18.5294859, 18.5196501, 18.0879985, 17.3153349, 16.2824627, 15.0701855, 13.7593067, 12.4306301, 11.1649591, 10.0430972, 8.59814118, 6.83265072, 6.06629013, 6.2624972, 7.38470972, 9.39636548, 12.2609023, 15.9417579, 20.4023701, 25.6061768, 31.5166156, 38.0971245, 45.3111411, 51.9747625, 59.6194589, 68.5528704, 79.0826368, 91.9833515, 107.329178, 125.069758, 145.372647, 168.405398, 194.335566, 223.330704, 255.173869, 290.534134, 329.65428, 372.77709, 420.145346, 467.039841, 519.905843, 580.226631, 649.485484, 729.165683, 823.701801, 928.018684, 1038.44694, 1154.1484, 1280.06537, 1418.54484, 1571.93379, 1744.72991, 1936.6407, 2145.02746, 2369.06633, 2610.52518, 2869.53519, 3147.11341, 3445.77791, 3767.35778, 4115.23032, 4491.626, 4893.09847, 5319.41254, 5763.62528, 6233.24114, 6730.34679, 7257.02891, 7815.37417, 8589.17121, 9353.37926, 10064.6595, 10679.6732, 11155.0815, 11252.4737, 11253.6307, 11198.8161, 11233.6409, 11356.6684, 11522.5671, 11732.0948, 11986.0097, 12285.0697, 12630.0328, 11175.63, 11316.0848, 12427.7303, 13886.8997, 15069.9262, 15353.1429, 14112.8831, 12040.256, 9502.11885}, 
{231.819896, 231.532695, 231.161601, 230.708213, 230.174135, 229.560967, 228.870311, 228.103769, 227.262943, 226.349434, 225.364843, 224.310774, 223.188826, 222.000602, 220.747704, 219.431733, 218.05429, 216.616979, 215.121399, 213.569153, 211.961843, 210.30107, 208.588435, 206.825542, 205.01399, 203.155382, 201.25132, 199.303405, 197.313238, 195.282422, 193.212558, 191.105248, 188.962094, 186.784697, 184.590781, 182.373547, 180.126004, 177.849229, 175.544302, 173.212299, 170.854301, 168.471385, 166.06463, 163.635115, 161.183917, 158.712115, 156.220788, 153.711014, 151.183871, 148.640439, 146.081795, 143.454401, 140.815135, 138.166262, 135.510043, 132.848741, 130.184617, 127.519936, 124.856959, 122.197948, 119.545167, 116.900877, 114.267341, 111.646822, 109.041581, 106.453882, 103.885987, 101.488481, 99.4083897, 97.3420692, 95.282167, 93.2213303, 91.1522063, 89.0674424, 86.9596857, 84.8215836, 82.6457833, 80.424932, 78.1516771, 75.8186658, 73.4185453, 70.943963, 68.387566, 65.7420016, 62.0871549, 57.8912528, 53.6334933, 49.3361445, 45.0214747, 40.7117519, 36.4292443, 32.1962201, 28.0349475, 23.9676945, 20.0167293, 16.2043202, 12.5527352, 9.08424255, 5.82111035, 2.78560678, 0, 5.41772189, 9.81891811, 13.2869075, 15.9050087, 17.7565407, 18.9248221, 19.4931719, 19.5449088, 19.1633515, 18.431819, 17.4336299, 16.2521031, 14.9705573, 13.6723114, 12.4406842, 11.3589944, 9.88141765, 8.00139508, 7.22530758, 7.50165754, 8.77894734, 11.0056794, 14.130356, 18.1014797, 22.8675527, 28.3770776, 34.5785566, 41.4204921, 48.8513866, 55.5253214, 63.1598409, 72.0917739, 82.6579492, 95.8225107, 111.60863, 129.872193, 150.761835, 174.426189, 201.01389, 230.673572, 262.895488, 298.687031, 338.325647, 382.088785, 430.253892, 479.680809, 534.918993, 597.100294, 667.356563, 746.819647, 838.048263, 939.003449, 1048.24142, 1165.76231, 1294.51418, 1436.12156, 1592.20895, 1765.65028, 1956.53668, 2163.5963, 2386.36144, 2626.65618, 2885.23606, 3163.22679, 3462.38133, 3784.16473, 4131.27966, 4505.51201, 4904.10976, 5326.88781, 5767.57426, 6233.00721, 6725.1085, 7245.8, 7797.00354, 8557.25061, 9307.70102, 10006.1242, 10610.2897, 11077.9668, 11177.4121, 11182.25, 11131.4697, 11166.2048, 11285.0125, 11443.8898, 11643.5219, 11884.5941, 12167.7915, 12493.7993, 11265.7949, 11447.1463, 12472.1176, 13774.9729, 14789.9764, 14951.3922, 13693.4843, 11529.9488, 9088.39306}, 
{237.302875, 236.915102, 236.446219, 235.897705, 235.271041, 234.567707, 233.789185, 232.936955, 232.012497, 231.017292, 229.952821, 228.820564, 227.622001, 226.358614, 225.031883, 223.643288, 222.19431, 220.686429, 219.121127, 217.499883, 215.824179, 214.095494, 212.31531, 210.485107, 208.606365, 206.680565, 204.709189, 202.693715, 200.635626, 198.5364, 196.39752, 194.220465, 192.006717, 189.757755, 187.483596, 185.181272, 182.847715, 180.484128, 178.091716, 175.671681, 173.225227, 170.753558, 168.257877, 165.739389, 163.199296, 160.638802, 158.059111, 155.461427, 152.846952, 150.216891, 147.572448, 144.863899, 142.144479, 139.416495, 136.682255, 133.944065, 131.204234, 128.465069, 125.728878, 122.997969, 120.274648, 117.561223, 114.860003, 112.173294, 109.503404, 106.85264, 104.223311, 101.764517, 99.6218329, 97.4948179, 95.3762634, 93.258961, 91.1357022, 88.9992787, 86.842482, 84.6581038, 82.4389355, 80.1777687, 77.8673951, 75.5006063, 73.0701937, 70.568949, 67.9896638, 65.3251297, 61.6621525, 57.4655729, 53.2111208, 48.9209886, 44.617369, 40.3224544, 36.0584374, 31.8475107, 27.7118666, 23.6736979, 19.7551971, 15.9785566, 12.3659691, 8.93962719, 5.72172331, 2.73445006, 0, 5.46818055, 9.92094066, 13.4419578, 16.1149093, 18.0234728, 19.2513256, 19.8821452, 19.999609, 19.6873945, 19.0291792, 18.1086404, 17.0094557, 15.8153025, 14.6098581, 13.4768002, 12.499806, 11.1125806, 9.3059483, 8.63376227, 9.04042019, 10.4703197, 12.8678584, 16.1774339, 20.3434439, 25.3102861, 31.0223579, 37.4240571, 44.4597812, 52.073928, 58.8594718, 66.6077554, 75.6686032, 86.39184, 99.8113129, 115.934835, 134.587814, 155.914868, 180.060614, 207.16967, 237.386654, 270.022004, 306.308399, 346.553666, 391.065632, 440.152123, 492.282878, 550.063334, 614.260844, 685.642754, 764.976416, 852.904302, 950.471265, 1058.46246, 1177.72123, 1309.20966, 1453.83652, 1612.51059, 1786.45685, 1976.196, 2181.90378, 2403.62014, 2643.03148, 2901.33703, 3179.69598, 3479.1997, 3800.97071, 4147.02617, 4518.72053, 4914.12786, 5333.17779, 5770.73758, 6232.57809, 6720.38133, 7235.82931, 7780.60405, 8527.48759, 9264.2869, 9949.90897, 10543.2608, 11003.2494, 11104.9314, 11113.6311, 11067.0484, 11101.7532, 11216.3095, 11368.0854, 11557.7047, 11785.7911, 12052.9683, 12359.86, 11454.6772, 11700.3836, 12613.8822, 13712.0762, 14511.8684, 14530.162, 13283.86, 11052.3401, 8725.77511}, 
{243.144518, 242.6519, 242.080681, 241.432233, 240.70793, 239.909145, 239.037251, 238.09362, 237.079627, 235.996643, 234.846043, 233.629199, 232.347483, 231.002271, 229.594933, 228.126843, 226.599375, 225.013902, 223.371796, 221.67443, 219.923178, 218.119413, 216.264507, 214.359834, 212.406767, 210.406679, 208.360943, 206.270931, 204.138018, 201.963576, 199.748978, 197.495597, 195.204806, 192.877979, 190.517989, 188.125429, 185.700869, 183.245635, 180.761051, 178.248441, 175.70913, 173.144441, 170.5557, 167.94423, 165.311356, 162.658402, 159.986692, 157.297551, 154.592303, 151.872273, 149.138784, 146.346129, 143.543685, 140.733794, 137.918801, 135.101049, 132.282883, 129.466646, 126.654682, 123.849335, 121.052949, 118.267868, 115.496436, 112.740996, 110.003892, 107.287469, 104.59407, 102.070825, 99.8613872, 97.669491, 95.488094, 93.3101543, 91.1286298, 88.9364787, 86.7266587, 84.4921281, 82.2258447, 79.9207665, 77.5698515, 75.1660578, 72.7023433, 70.171666, 67.5669839, 64.881255, 61.2108502, 57.0158794, 52.767229, 48.4869532, 44.1971058, 39.9197407, 35.6769121, 31.4906738, 27.3830798, 23.3761841, 19.4920406, 15.7527034, 12.1802264, 8.79666361, 5.62406896, 2.68449643, 0, 5.3269546, 9.67470642, 13.1248376, 15.7589304, 17.6585668, 18.9053292, 19.5807995, 19.7665601, 19.544193, 18.9952804, 18.2014045, 17.2441475, 16.2050914, 15.1658186, 14.207911, 13.4129509, 12.2576857, 10.7372925, 10.3014322, 10.9020795, 12.4912092, 15.020796, 18.4428145, 22.7092394, 27.7720455, 33.5832074, 40.0946998, 47.2584975, 55.0265751, 62.0398066, 70.0420043, 79.3784733, 90.3945185, 104.057094, 120.40068, 139.293985, 160.895817, 185.364987, 212.860306, 243.540583, 276.72481, 313.667213, 354.690914, 400.119035, 450.2747, 504.979326, 565.183167, 631.33477, 703.882684, 783.275457, 868.504547, 963.256483, 1070.18255, 1190.83949, 1324.54944, 1471.6378, 1632.43001, 1806.78105, 1995.59369, 2200.28393, 2421.4378, 2660.4352, 2918.65286, 3197.19542, 3496.70645, 3818.04117, 4162.63279, 4531.48637, 4923.48768, 5338.72128, 5773.62187, 6232.48541, 6716.65989, 7227.49328, 7766.33359, 8499.24726, 9221.6842, 9893.81276, 10475.8013, 10927.8183, 11032.3399, 11045.6879, 11004.1846, 11039.3527, 11149.8308, 11294.5903, 11474.253, 11689.4407, 11940.7753, 12228.8785, 11704.5333, 12030.9651, 12820.9505, 13687.2659, 14242.6879, 14099.9931, 12871.958, 10606.8339, 8412.97125}, 
{249.378672, 248.779362, 248.10328, 247.351724, 246.525991, 245.627379, 244.657185, 243.616706, 242.507241, 241.330086, 240.086538, 238.777896, 237.405457, 235.970518, 234.474377, 232.91833, 231.303676, 229.631712, 227.903736, 226.121044, 224.284935, 222.396705, 220.457652, 218.469074, 216.432268, 214.348532, 212.219162, 210.045457, 207.828713, 205.570229, 203.271301, 200.933228, 198.557306, 196.144833, 193.69318, 191.205691, 188.685755, 186.134798, 183.554245, 180.945519, 178.310047, 175.649252, 172.96456, 170.257396, 167.529185, 164.78135, 162.015318, 159.232513, 156.43436, 153.622284, 150.797709, 147.918107, 145.02981, 142.135194, 139.236638, 136.336519, 133.437214, 130.541103, 127.650561, 124.767968, 121.895699, 119.036135, 116.191651, 113.364625, 110.557436, 107.772461, 105.012077, 102.421165, 100.141181, 97.8806052, 95.6325767, 93.3902355, 91.1467215, 88.8951744, 86.6287342, 84.3405408, 82.023734, 79.6714536, 77.2768396, 74.8330317, 72.33317, 69.7703941, 67.1378441, 64.4286597, 60.7512267, 56.5596717, 52.3187553, 48.0503292, 43.7762453, 39.5183555, 35.2985115, 31.1385653, 27.0603687, 23.0857734, 19.2366314, 15.5347944, 12.0021144, 8.66044308, 5.53163236, 2.63753405, 0, 4.96424945, 9.02554828, 12.2606321, 14.7462364, 16.5590969, 17.7759492, 18.4735287, 18.7285713, 18.6178123, 18.2179875, 17.6058323, 16.8580825, 16.0514736, 15.2627412, 14.5686209, 14.0458482, 13.2827884, 12.2864097, 12.2380953, 13.10993, 14.8739984, 17.5023856, 20.9671761, 25.2404547, 30.2943063, 36.1008156, 42.6320675, 49.8601466, 57.7571377, 65.1289182, 73.5413897, 83.3164993, 94.7761939, 108.667191, 125.099052, 144.068067, 165.768753, 190.395624, 218.143198, 249.20599, 283.175297, 321.032444, 363.089965, 409.660399, 461.056283, 517.903435, 580.12279, 647.948564, 721.614975, 801.356239, 885.083626, 978.193454, 1084.47416, 1205.93142, 1340.93117, 1489.47351, 1651.55854, 1826.25432, 2014.70482, 2219.07078, 2440.4098, 2679.65146, 2937.99831, 3216.39956, 3515.37501, 3835.64157, 4178.26245, 4544.04433, 4932.52415, 5343.95709, 5776.73378, 6233.26082, 6714.43881, 7221.16836, 7754.35008, 8471.89473, 9178.44023, 9835.63465, 10405.1261, 10848.5626, 10956.9457, 10976.3344, 10941.5109, 10978.0701, 11084.8477, 11222.8407, 11392.7766, 11595.3831, 11831.3878, 12101.5181, 11977.6198, 12394.0595, 13061.2486, 13689.5985, 13989.5205, 13671.426, 12445.7263, 10192.834, 8148.68769}, 
{256.310456, 255.605356, 254.823231, 253.965406, 253.033206, 252.027953, 250.950972, 249.803588, 248.587124, 247.302904, 245.952254, 244.536496, 243.056955, 241.514956, 239.911821, 238.248877, 236.527445, 234.748852, 232.91442, 231.025475, 229.083339, 227.089338, 225.044795, 222.951035, 220.809382, 218.621159, 216.387692, 214.110303, 211.790318, 209.429061, 207.027855, 204.588024, 202.110894, 199.597787, 197.046437, 194.460039, 191.841831, 189.193255, 186.515751, 183.810759, 181.079722, 178.324078, 175.545269, 172.744736, 169.92392, 167.08426, 164.227198, 161.354174, 158.46663, 155.566006, 152.653742, 149.686254, 146.710984, 143.730348, 140.746765, 137.762649, 134.780419, 131.80249, 128.831281, 125.869206, 122.918684, 119.98213, 117.061962, 114.160597, 111.28045, 108.42394, 105.593482, 102.931821, 100.578331, 98.2460149, 95.9281926, 93.6181839, 91.309309, 88.9948877, 86.6682399, 84.3226855, 81.9515446, 79.5481369, 77.1057826, 74.6178014, 72.0775133, 69.4782383, 66.8132962, 64.0760071, 60.3904082, 56.202745, 51.969711, 47.7128359, 43.4536498, 39.2136822, 35.0144632, 30.8775225, 26.8243898, 22.8765952, 19.0556683, 15.3831389, 11.880537, 8.56939236, 5.47123475, 2.60759403, 0, 3.86551936, 7.03699394, 9.57731911, 11.5493902, 13.0161026, 14.0403517, 14.6850328, 15.0130414, 15.0872726, 14.970622, 14.7259849, 14.4162566, 14.1043325, 13.853108, 13.7254785, 13.7843392, 13.9284324, 14.197203, 15.0248452, 16.4390787, 18.4676231, 21.1381982, 24.4785236, 28.516319, 33.2793041, 38.7951985, 45.0917221, 52.1965944, 60.1375351, 68.2022111, 77.4297475, 88.0698801, 100.372344, 114.441764, 130.600431, 149.209332, 170.561737, 194.950915, 222.670136, 254.012668, 289.721751, 329.503735, 373.563854, 422.107336, 475.339415, 532.766307, 595.467011, 663.821512, 738.209794, 819.011842, 904.969154, 1000.1028, 1106.90994, 1226.97678, 1360.02955, 1506.62958, 1667.3382, 1842.52961, 2032.99478, 2239.72876, 2463.7565, 2705.28736, 2964.25486, 3241.17654, 3537.55944, 3854.45639, 4193.40054, 4555.56925, 4940.37872, 5348.24135, 5780.57591, 6236.63362, 6716.47853, 7220.1747, 7747.78616, 8442.62472, 9125.69476, 9761.24837, 10313.5377, 10746.8148, 10867.0056, 10902.2393, 10884.6221, 10927.5436, 11030.9353, 11160.8607, 11318.945, 11506.813, 11726.0897, 11978.4002, 12225.7586, 12744.8377, 13321.6309, 13742.1318, 13792.3336, 13258.2297, 11925.8135, 9791.08948, 7939.13674}, 
{263.496459, 262.688329, 261.802186, 260.83941, 259.801381, 258.689479, 257.505083, 256.249574, 254.924331, 253.530734, 252.070163, 250.543998, 248.953618, 247.300405, 245.585737, 243.810994, 241.977557, 240.086805, 238.140117, 236.138875, 234.084458, 231.978245, 229.821617, 227.615953, 225.362633, 223.063038, 220.718547, 218.330539, 215.900396, 213.429496, 210.919219, 208.370946, 205.786057, 203.16593, 200.510037, 197.820753, 195.100474, 192.350644, 189.572703, 186.768094, 183.938258, 181.084638, 178.208674, 175.31181, 172.395486, 169.461145, 166.510228, 163.544178, 160.564436, 157.572444, 154.569643, 151.510292, 148.444039, 145.373349, 142.300686, 139.228517, 136.159304, 133.095514, 130.03961, 126.994057, 123.96132, 120.943864, 117.944154, 114.964653, 112.007828, 109.076142, 106.17206, 103.436179, 101.005776, 98.5984326, 96.2076584, 93.8269639, 91.4498593, 89.0698548, 86.6804605, 84.2751869, 81.8475441, 79.3910424, 76.899192, 74.3655032, 71.7834862, 69.1466512, 66.4485085, 63.6825684, 59.9895348, 55.8072587, 51.5838413, 47.3404678, 43.0983239, 38.878595, 34.7024664, 30.5911236, 26.5657522, 22.6475374, 18.8576648, 15.2173199, 11.747688, 8.46995462, 5.40530519, 2.57492517, 0, 2.85380583, 5.20576041, 7.10608714, 8.60500941, 9.7527506, 10.5995341, 11.1955833, 11.5911216, 11.8363724, 11.981559, 12.0769049, 12.1726334, 12.318968, 12.566132, 12.9643488, 13.5638418, 14.5574224, 16.0156837, 17.6663286, 19.5901351, 21.867881, 24.5803444, 27.8083032, 31.6325354, 36.1338189, 41.3929318, 47.490652, 54.5077575, 62.5250262, 71.2997377, 81.3547845, 92.867994, 106.017194, 120.269244, 136.157457, 154.404733, 175.402191, 199.540948, 227.212122, 258.80683, 296.224353, 337.888639, 383.895732, 434.341671, 489.322501, 546.925312, 609.757336, 678.416851, 753.502139, 835.611478, 924.713514, 1022.80572, 1130.64515, 1249.22085, 1379.99548, 1524.21905, 1683.14159, 1858.71916, 2051.33525, 2260.60311, 2487.45507, 2731.40112, 2991.08257, 3266.58206, 3560.42453, 3874.01388, 4209.40199, 4568.16077, 4949.48604, 5353.91766, 5785.662, 6241.02834, 6719.28737, 7219.70977, 7741.56622, 8414.62075, 9075.02737, 9689.43343, 10224.4863, 10646.8333, 10775.7527, 10823.507, 10819.9018, 10868.2409, 10969.2369, 11092.9778, 11241.7072, 11417.6687, 11623.106, 11860.2626, 12429.564, 13038.4639, 13536.7233, 13774.1027, 13600.363, 12865.2648, 11418.5688, 9433.82399, 7769.81901}, 
{270.751476, 269.845297, 268.859774, 267.796346, 266.656456, 265.441545, 264.153053, 262.792423, 261.361094, 259.860508, 258.292107, 256.657332, 254.957623, 253.194422, 251.36917, 249.483309, 247.538279, 245.535521, 243.476477, 241.362588, 239.195295, 236.976039, 234.706261, 232.387403, 230.020905, 227.60821, 225.150757, 222.649989, 220.107346, 217.524269, 214.9022, 212.24258, 209.54685, 206.816451, 204.052504, 201.256619, 198.430405, 195.575315, 192.692799, 189.78431, 186.851299, 183.895218, 180.917518, 177.91965, 174.903066, 171.869218, 168.819557, 165.755535, 162.678603, 159.590213, 156.491817, 153.335258, 150.172672, 147.006584, 143.839522, 140.674012, 137.512581, 134.357756, 131.212065, 128.078033, 124.958189, 121.855058, 118.771167, 115.709044, 112.671216, 109.660209, 106.67855, 103.864733, 101.353986, 98.8684167, 96.4017364, 93.9476582, 91.4998948, 89.052159, 86.5981634, 84.1316208, 81.6462439, 79.1357455, 76.5938383, 74.0142349, 71.3906482, 68.7167908, 65.9863755, 63.193115, 59.4940251, 55.3187658, 51.1070535, 46.8797534, 42.6577306, 38.4618502, 34.3129775, 30.2319774, 26.2397153, 22.3570562, 18.6048653, 15.0040077, 11.5753486, 8.33975315, 5.31808646, 2.53121369, 0, 2.30603887, 4.21698317, 5.7762519, 7.02726407, 8.01343867, 8.77819473, 9.36495123, 9.81712718, 10.1781416, 10.4914135, 10.8003618, 11.1484056, 11.5789638, 12.1354556, 12.8612998, 13.7999155, 15.3217927, 17.5170748, 19.6893774, 21.9522061, 24.4190665, 27.2034644, 30.4189053, 34.1788949, 38.5969389, 43.7865428, 49.8612124, 56.9344533, 65.1197712, 74.4794936, 85.1970695, 97.401358, 111.221218, 125.682043, 141.464426, 159.543214, 180.378307, 204.429604, 232.157003, 264.020403, 302.712666, 345.78113, 393.248809, 445.138716, 501.473867, 559.206653, 622.198366, 691.239676, 767.121252, 850.633764, 942.850315, 1043.93394, 1153.15436, 1270.96586, 1400.24131, 1542.76769, 1700.33197, 1876.07751, 2070.12663, 2281.12198, 2509.99422, 2755.71479, 3016.0504, 3290.76206, 3582.99003, 3894.32345, 4227.08381, 4583.05014, 4961.31603, 5362.49412, 5792.67232, 6246.14561, 6721.63133, 7217.84681, 7733.50938, 8389.21919, 9031.34005, 9628.1186, 10147.8015, 10558.6353, 10688.1199, 10739.0799, 10740.082, 10790.2772, 10890.2925, 11011.3448, 11155.5317, 11324.9506, 11521.699, 11747.8746, 12557.1678, 13226.7619, 13656.2793, 13745.3421, 13393.5727, 12500.5935, 10966.0266, 9136.03362, 7633.65619}, 
{277.756863, 276.761284, 275.685397, 274.530667, 273.298561, 271.990543, 270.608079, 269.152637, 267.625681, 266.028677, 264.363092, 262.630391, 260.832039, 258.969504, 257.04425, 255.057744, 253.011451, 250.906838, 248.74537, 246.528513, 244.257733, 241.934495, 239.560266, 237.136512, 234.664698, 232.146291, 229.582755, 226.975558, 224.326164, 221.636041, 218.906652, 216.139466, 213.335946, 210.49756, 207.624481, 204.718847, 201.782815, 198.817891, 195.825583, 192.807399, 189.764848, 186.699435, 183.612671, 180.506061, 177.381115, 174.23934, 171.082243, 167.911332, 164.728116, 161.534102, 158.330798, 155.068493, 151.801025, 148.53101, 145.261066, 141.993813, 138.731869, 135.47785, 132.234376, 129.004064, 125.789533, 122.5934, 119.418285, 116.266804, 113.141576, 110.045219, 106.980352, 104.083552, 101.488181, 98.9205755, 96.3746685, 93.8443942, 91.3236866, 88.8064797, 86.2867075, 83.758304, 81.2152032, 78.6513391, 76.0606458, 73.4370571, 70.7745071, 68.0669299, 65.3082594, 62.4924295, 58.7885438, 54.6203006, 50.4213123, 46.2122804, 42.0139061, 37.8468908, 33.7319358, 29.6897425, 25.7410122, 21.9064463, 18.2067461, 14.662613, 11.2947482, 8.12385319, 5.17062925, 2.45577774, 0, 3.28707139, 6.00526952, 8.21046446, 9.95852628, 11.3053251, 12.3067309, 13.0186138, 13.4968439, 13.7972913, 13.975826, 14.0883181, 14.1906377, 14.3386548, 14.5882396, 14.9952621, 15.6155923, 16.5985543, 18.0133687, 19.7045218, 21.7479093, 24.2194273, 27.1949717, 30.7504383, 34.9617231, 39.904722, 45.6553309, 52.2894458, 59.8829624, 68.5117769, 77.9621157, 88.7057561, 100.905495, 114.724129, 129.416925, 145.600444, 164.133257, 185.450419, 209.986985, 238.178011, 270.458552, 309.158851, 352.241981, 399.772201, 451.813768, 508.430942, 566.690455, 630.403472, 700.383633, 777.444576, 862.399941, 956.561633, 1059.63603, 1170.7547, 1290.24912, 1420.90057, 1564.39044, 1722.40011, 1897.58215, 2090.42601, 2300.36166, 2528.95264, 2774.80301, 3035.84558, 3311.77268, 3605.25813, 3917.60717, 4250.73979, 4606.12063, 4982.96035, 5381.74459, 5807.2968, 6255.09746, 6724.13066, 7213.38048, 7721.83101, 8375.25246, 9014.14627, 9605.80001, 10117.5013, 10516.5376, 10622.7689, 10649.1954, 10626.2881, 10666.3599, 10767.4858, 10893.6398, 11044.6021, 11220.1526, 11420.0714, 11644.1385, 12506.8736, 13163.2984, 13535.8419, 13546.9329, 13119.0003, 12174.473, 10635.7799, 8937.43128, 7528.00504}, 
{284.741703, 283.664986, 282.506566, 281.267939, 279.950602, 278.556051, 277.085784, 275.541296, 273.924084, 272.235646, 270.477477, 268.651074, 266.757934, 264.799553, 262.777428, 260.693056, 258.547933, 256.343555, 254.081421, 251.763025, 249.389865, 246.963437, 244.485238, 241.956765, 239.379513, 236.754981, 234.084663, 231.370058, 228.612661, 225.81397, 222.97548, 220.098688, 217.185092, 214.236187, 211.250985, 208.232277, 205.182885, 202.104386, 198.998358, 195.866377, 192.710022, 189.530868, 186.330494, 183.110477, 179.872395, 176.617823, 173.34834, 170.065523, 166.770949, 163.466196, 160.152841, 156.77945, 153.40176, 150.022498, 146.644391, 143.270165, 139.902545, 136.54426, 133.198034, 129.866595, 126.552669, 123.258983, 119.988262, 116.743234, 113.526625, 110.341161, 107.189568, 104.206707, 101.524265, 98.8724523, 96.2454309, 93.6373611, 91.0424038, 88.4547196, 85.8684695, 83.2778141, 80.6769143, 78.0599308, 75.4210243, 72.7543558, 70.0540859, 67.3143754, 64.5293851, 61.6932758, 57.9872646, 53.8290833, 49.6462347, 45.459261, 41.2887046, 37.155108, 33.0790134, 29.0809632, 25.1814998, 21.4011655, 17.7605028, 14.2800539, 10.9803612, 7.88196713, 5.00541399, 2.37124416, 0, 4.56631707, 8.33524647, 11.3788636, 13.7692439, 15.5784628, 16.8785956, 17.7417178, 18.2399048, 18.445232, 18.4297748, 18.2656086, 18.0248089, 17.779451, 17.6016103, 17.5633623, 17.7367824, 17.9753017, 18.3197606, 19.3367872, 21.0516046, 23.4894358, 26.6755038, 30.6350319, 35.3932428, 40.9753599, 47.406606, 54.7122043, 62.9173778, 72.0473495, 81.4992055, 92.1566228, 104.233266, 117.942798, 132.94195, 149.722853, 168.926154, 190.932597, 216.122927, 244.877888, 277.578225, 315.805484, 358.374146, 405.430014, 457.118894, 513.586591, 572.483947, 637.07547, 708.130702, 786.419188, 872.710473, 968.198697, 1072.70986, 1185.75556, 1307.71397, 1440.73247, 1586.1641, 1745.3619, 1920.01962, 2111.07215, 2319.08272, 2546.09232, 2790.81169, 3052.08936, 3329.7795, 3625.44024, 3939.84755, 4274.08709, 4629.01512, 5004.35247, 5400.46222, 5820.54778, 6261.69497, 6723.28369, 7204.69388, 7705.30543, 8358.33939, 8995.87427, 9583.82969, 10088.1253, 10474.6807, 10552.9118, 10549.5778, 10497.6714, 10526.3011, 10630.4138, 10764.9079, 10926.7329, 11112.8378, 11320.1719, 11545.6844, 12375.5076, 12989.4748, 13317.8768, 13291.004, 12839.1473, 11892.5972, 10381.6444, 8794.53997, 7441.56955}, 
{291.741164, 290.592618, 289.360279, 288.045693, 286.650406, 285.175963, 283.623912, 281.995797, 280.293165, 278.517562, 276.670534, 274.753627, 272.768387, 270.716359, 268.599091, 266.418128, 264.175015, 261.8713, 259.508528, 257.088244, 254.611996, 252.081329, 249.497789, 246.862922, 244.178274, 241.445391, 238.665819, 235.841104, 232.972793, 230.062431, 227.111563, 224.121738, 221.094499, 218.031393, 214.930705, 211.79568, 208.629602, 205.434124, 202.210897, 198.961574, 195.687806, 192.391245, 189.073542, 185.736351, 182.381322, 179.010108, 175.62436, 172.22573, 168.81587, 165.396433, 161.969069, 158.480186, 154.987879, 151.494996, 148.004388, 144.518903, 141.041391, 137.574702, 134.121684, 130.685188, 127.268063, 123.873158, 120.503323, 117.161407, 113.850259, 110.57273, 107.331669, 104.260469, 101.489389, 98.7520422, 96.0428145, 93.3560928, 90.686264, 88.0277149, 85.3748322, 82.7220029, 80.0636137, 77.3940513, 74.7077027, 71.9989546, 69.2621938, 66.4918072, 63.6821815, 60.8277036, 57.1231066, 52.9794238, 48.8172997, 44.6570714, 40.5190759, 36.4236504, 32.391132, 28.4418578, 24.5961648, 20.8743903, 17.2968712, 13.8839447, 10.655948, 7.63321802, 4.83609196, 2.28490692, 0, 5.72586038, 10.447869, 14.2527876, 17.2273776, 19.4584008, 21.0326188, 22.0367931, 22.5576854, 22.6820573, 22.4966705, 22.0882865, 21.543667, 20.9495736, 20.392768, 19.9600116, 19.7380663, 19.3191119, 18.7192557, 19.1439556, 20.5739916, 22.9901434, 26.3731909, 30.7039139, 35.9630925, 42.1315063, 49.1899352, 57.1191592, 65.899958, 75.5131115, 85.0084207, 95.6390032, 107.664933, 121.346283, 136.758199, 154.25322, 174.233196, 197.013679, 222.91022, 252.238372, 285.313686, 322.799404, 364.557569, 410.835707, 461.881343, 517.942003, 577.560541, 643.115321, 715.280038, 794.728383, 882.134051, 978.282665, 1083.59918, 1198.34484, 1323.08116, 1458.98281, 1606.94918, 1767.87964, 1942.36829, 2131.68918, 2337.44941, 2561.69421, 2803.91428, 3064.65615, 3344.20125, 3642.38177, 3959.23613, 4294.58739, 4648.41812, 5021.5005, 5414.16006, 5828.63401, 6263.0421, 6717.0481, 7190.31583, 7682.50906, 8332.31731, 8965.62222, 9547.33117, 10042.3515, 10415.5906, 10467.8516, 10437.549, 10359.9203, 10379.6956, 10488.7719, 10633.4247, 10807.832, 11006.172, 11222.623, 11451.3631, 12189.6119, 12744.7886, 13043.2683, 13011.4261, 12575.6368, 11662.2755, 10197.7173, 8693.335, 7366.11709}, 
{298.907735, 297.70907, 296.42206, 295.048393, 293.589752, 292.047825, 290.424295, 288.72085, 286.939174, 285.080953, 283.147872, 281.141617, 279.063874, 276.916328, 274.700664, 272.418569, 270.071727, 267.661824, 265.190546, 262.659579, 260.070607, 257.425316, 254.725393, 251.972522, 249.168388, 246.314678, 243.413078, 240.465271, 237.472945, 234.437784, 231.361475, 228.245702, 225.092152, 221.902509, 218.678115, 215.420835, 212.132538, 208.81492, 205.469679, 202.098511, 198.703113, 195.285182, 191.846413, 188.388505, 184.913153, 181.422055, 177.916907, 174.399406, 170.871248, 167.33413, 163.78975, 160.17905, 156.565798, 152.953007, 149.343691, 145.740864, 142.147538, 138.566729, 135.001449, 131.454712, 127.929532, 124.428923, 120.955899, 117.513473, 114.104658, 110.732469, 107.399919, 104.240625, 101.385067, 98.5669007, 95.7806733, 93.0209324, 90.282225, 87.5590985, 84.8461002, 82.1377774, 79.4286772, 76.713347, 73.9863341, 71.2421857, 68.4754492, 65.6806717, 62.8524005, 59.985183, 56.2907903, 52.1716224, 48.0394876, 43.9143357, 39.8161167, 35.7647804, 31.7802766, 27.8825553, 24.0915662, 20.4272593, 16.9095844, 13.5584914, 10.3939302, 7.43585057, 4.70420242, 2.21893561, 0, 5.35677648, 9.78700298, 13.3724461, 16.1948725, 18.3360488, 19.8777415, 20.9017174, 21.4897431, 21.7235851, 21.68501, 21.4557847, 21.1176755, 20.7524492, 20.4418724, 20.2677116, 20.3117336, 20.3413228, 20.3933341, 21.2606365, 22.9576292, 25.4987111, 28.8982812, 33.1707386, 38.3304823, 44.3919113, 51.3694246, 59.2774212, 68.1303002, 77.9424605, 88.0364222, 99.3721522, 112.171613, 126.656768, 142.809087, 160.970778, 181.548183, 204.835412, 231.126576, 260.715786, 293.897152, 330.820707, 371.968489, 417.662799, 468.225935, 523.980198, 584.363231, 650.803155, 723.843432, 804.027527, 891.898904, 987.539545, 1092.51843, 1208.11809, 1335.2376, 1473.99304, 1624.85203, 1788.28221, 1964.47963, 2154.24521, 2358.67609, 2577.69252, 2813.86618, 3071.58581, 3352.33759, 3652.68944, 3971.46684, 4306.03316, 4654.83482, 5021.67934, 5407.34173, 5817.35641, 6247.0067, 6695.42008, 7161.72403, 7645.04603, 8275.53798, 8888.54684, 9450.444, 9927.60084, 10286.3887, 10333.5843, 10301.5502, 10225.0555, 10250.2653, 10367.8544, 10522.083, 10705.9467, 10912.4412, 11134.5623, 11365.3056, 12004.8492, 12515.2934, 12809.7559, 12801.3542, 12403.2061, 11528.4291, 10090.1411, 8588.56663, 7266.04088}, 
{306.085713, 304.841218, 303.503545, 302.074528, 300.555998, 298.949787, 297.257728, 295.481652, 293.623391, 291.684777, 289.667643, 287.573821, 285.405142, 283.163439, 280.850544, 278.468288, 276.018504, 273.503024, 270.923679, 268.282303, 265.580727, 262.820783, 260.004302, 257.133118, 254.209062, 251.233967, 248.209663, 245.137984, 242.020762, 238.859827, 235.657014, 232.414153, 229.133076, 225.815616, 222.466217, 219.085351, 215.673456, 212.23228, 208.763569, 205.269073, 201.750536, 198.209708, 194.648334, 191.068163, 187.470941, 183.858416, 180.232336, 176.594447, 172.946496, 169.290231, 165.6274, 161.89414, 158.159229, 154.425838, 150.697136, 146.976293, 143.266478, 139.570861, 135.892611, 132.234898, 128.600891, 124.993761, 121.416675, 117.872805, 114.36532, 110.897388, 107.47218, 104.223035, 101.280164, 98.3782754, 95.5120988, 92.6763659, 89.865808, 87.0751562, 84.2991418, 81.532496, 78.7699501, 76.0062352, 73.2360827, 70.4542237, 67.6553895, 64.8343113, 61.9857203, 59.1043478, 55.4226905, 51.3326295, 47.2352072, 43.1498909, 39.0961478, 35.0934451, 31.1612499, 27.3190295, 23.5862512, 19.982382, 16.5268892, 13.23924, 10.1389016, 7.24534116, 4.57802596, 2.15642317, 0, 4.65341302, 8.51726987, 11.6643561, 14.1674571, 16.0993585, 17.5328458, 18.5407045, 19.1957201, 19.5706781, 19.738364, 19.7715634, 19.7430617, 19.7256445, 19.7920973, 20.0152055, 20.4677547, 21.1623826, 22.163282, 23.6792873, 25.7702952, 28.4962023, 31.9169053, 36.0923009, 41.0822857, 46.9467565, 53.74561, 61.5387428, 70.3860515, 80.347433, 91.1044381, 103.234036, 116.909627, 132.304612, 149.249175, 168.088325, 189.258594, 213.036346, 239.697948, 269.519763, 302.778158, 339.148682, 379.691372, 424.800143, 474.868911, 530.291591, 591.474576, 658.796184, 732.647214, 813.418462, 901.500723, 996.157224, 1100.28448, 1216.22688, 1345.2605, 1486.48016, 1639.96004, 1805.77429, 1983.91158, 2174.55099, 2377.96487, 2591.67852, 2821.52104, 3075.73741, 3357.0548, 3658.85079, 3978.79456, 4311.85548, 4655.00279, 5015.10509, 5393.43136, 5799.85615, 6225.71889, 6669.3097, 7128.91868, 7602.83594, 8207.03198, 8792.69642, 9328.69926, 9783.91053, 10127.2002, 10177.3309, 10153.3517, 10088.4643, 10124.6221, 10251.9794, 10415.3772, 10607.2355, 10819.9741, 11046.013, 11277.7721, 11800.385, 12264.7445, 12565.727, 12598.2091, 12257.067, 11437.1773, 10033.4162, 8517.00994, 7177.64308}, 
{313.231616, 311.939019, 310.548872, 309.063133, 307.483762, 305.812716, 304.051956, 302.203438, 300.269123, 298.250968, 296.150933, 293.970976, 291.713056, 289.379132, 286.971163, 284.491106, 281.940921, 279.322567, 276.638003, 273.889186, 271.078076, 268.206632, 265.276812, 262.290575, 259.24988, 256.156685, 253.012949, 249.820631, 246.581689, 243.298083, 239.971771, 236.604712, 233.198864, 229.756187, 226.283117, 222.77928, 219.244247, 215.679832, 212.087848, 208.470108, 204.828425, 201.164614, 197.480487, 193.777858, 190.05854, 186.324347, 182.577092, 178.818588, 175.050648, 171.275087, 167.493717, 163.640001, 159.785585, 155.933764, 152.087833, 148.251087, 144.426822, 140.618332, 136.828913, 133.061859, 129.320466, 125.608029, 121.927842, 118.283202, 114.677403, 111.11374, 107.595509, 104.2544, 101.219022, 98.2278632, 95.2758947, 92.3580883, 89.4694158, 86.6048488, 83.759359, 80.9279181, 78.1054979, 75.2870699, 72.467606, 69.6420777, 66.8054568, 63.952715, 61.078824, 58.1787555, 54.5102967, 50.4525173, 46.3933471, 42.3516811, 38.3464138, 34.3964401, 30.5206546, 26.7379522, 23.0672276, 19.5273755, 16.1372906, 12.9158678, 9.88200173, 7.05458717, 4.45251887, 2.09469156, 0, 3.86218828, 7.08432293, 9.72958893, 11.8611712, 13.5422549, 14.8360248, 15.8056659, 16.5143633, 17.0253018, 17.4016665, 17.7066424, 18.0034144, 18.3551674, 18.8250866, 19.4763567, 20.3721629, 21.75377, 23.709803, 25.9006469, 28.4276469, 31.3921478, 34.8954946, 39.0390323, 43.9241058, 49.65206, 56.32424, 64.0419906, 72.9066569, 83.0195837, 94.3788274, 107.226894, 121.696115, 137.918823, 155.61266, 175.117303, 196.88301, 221.16652, 248.224569, 278.313896, 311.691239, 347.572697, 387.572362, 432.150576, 481.767679, 536.884013, 598.882833, 667.070836, 741.677636, 822.932844, 911.066072, 1004.51953, 1107.49484, 1223.25617, 1353.42952, 1496.2962, 1651.63925, 1819.24168, 1999.05722, 2190.65937, 2393.43534, 2602.82486, 2827.3326, 3078.2152, 3359.1922, 3661.21199, 3981.08512, 4311.9757, 4649.74891, 5003.64027, 5375.32226, 5779.31897, 6202.47414, 6642.01165, 7095.15536, 7559.12916, 8131.01751, 8683.21853, 9187.99094, 9617.59346, 9944.2848, 10004.0187, 9996.22884, 9951.5904, 10002.8907, 10140.3854, 10311.7836, 10509.4684, 10725.823, 10953.2303, 11184.0736, 11583.2394, 12001.4294, 12313.953, 12396.1191, 12123.2371, 11370.616, 10013.5651, 8485.57233, 7111.04167}, 
{320.057931, 318.680458, 317.205692, 315.635542, 313.971916, 312.216724, 310.371873, 308.439273, 306.420832, 304.318459, 302.134063, 299.869552, 297.526835, 295.10782, 292.614417, 290.048534, 287.41208, 284.706962, 281.935091, 279.098375, 276.198722, 273.238041, 270.218241, 267.14123, 264.008917, 260.823211, 257.586021, 254.299255, 250.964822, 247.58463, 244.160588, 240.694606, 237.18859, 233.644452, 230.062541, 226.445578, 222.796302, 219.116671, 215.408646, 211.674184, 207.915246, 204.13379, 200.331776, 196.511163, 192.67391, 188.821977, 184.957321, 181.081904, 177.197683, 173.306619, 169.41067, 165.449503, 161.48872, 157.531631, 153.581545, 149.641772, 145.715621, 141.806403, 137.917425, 134.051999, 130.213434, 126.405039, 122.630123, 118.891997, 115.19397, 111.539352, 107.931452, 104.495832, 101.355119, 98.2604861, 95.2073178, 92.1909983, 89.2069121, 86.2504436, 83.3169772, 80.4018974, 77.5005886, 74.6084352, 71.7208217, 68.8331325, 65.940752, 63.0390647, 60.123455, 57.1893072, 53.527203, 49.4997527, 45.4780381, 41.4803158, 37.5248425, 33.6298748, 29.8136693, 26.0944827, 22.4905717, 19.0201928, 15.7016028, 12.5530582, 9.59281579, 6.8391321, 4.3102638, 2.02446756, 0, 3.71452767, 6.81047496, 9.35099143, 11.3992266, 13.0183301, 14.2714513, 15.2217399, 15.9323455, 16.4664174, 16.8871054, 17.2575589, 17.6409275, 18.1003607, 18.6990081, 19.5000192, 20.5665436, 22.0703496, 24.0901038, 26.4432703, 29.216274, 32.49554, 36.3674933, 40.918559, 46.2351619, 52.4037271, 59.5106797, 67.6424446, 76.8854469, 87.3261115, 98.80139, 111.738654, 126.299172, 142.644209, 160.778626, 180.941893, 203.41519, 228.406707, 256.124632, 286.777156, 320.572467, 356.083352, 395.651135, 439.803976, 489.070035, 543.97747, 606.877902, 676.020165, 751.476555, 833.319367, 921.620894, 1014.24757, 1116.17361, 1230.86432, 1360.07619, 1502.07679, 1656.70014, 1823.78026, 2003.43361, 2195.14759, 2398.10149, 2607.87264, 2832.44825, 3082.19417, 3360.5738, 3659.35471, 3975.67303, 4303.32054, 4638.56631, 4989.94216, 5359.04341, 5764.04267, 6187.40763, 6624.99531, 7072.66276, 7526.26699, 8057.55271, 8567.51736, 9033.04606, 9431.02392, 9738.33605, 9815.3136, 9833.09829, 9818.52694, 9888.58938, 10034.4368, 10209.6568, 10407.2787, 10620.332, 10841.846, 11064.8502, 11349.3961, 11720.2615, 12038.1943, 12163.9425, 11958.2541, 11281.877, 9995.55924, 8520.24782, 7075.12605}, 
{326.820878, 325.349509, 323.782267, 322.120966, 320.36742, 318.523446, 316.590858, 314.57147, 312.467097, 310.279554, 308.010656, 305.662218, 303.236054, 300.733979, 298.157809, 295.509357, 292.790439, 290.002869, 287.148462, 284.229034, 281.246398, 278.20237, 275.098764, 271.937395, 268.720078, 265.448628, 262.12486, 258.750588, 255.327627, 251.857792, 248.342898, 244.784759, 241.18519, 237.546007, 233.859375, 230.132136, 226.371249, 222.57884, 218.757039, 214.907973, 211.03377, 207.136558, 203.218465, 199.281618, 195.328146, 191.360176, 187.379837, 183.389256, 179.390561, 175.38588, 171.377341, 167.314692, 163.253576, 159.197257, 155.148998, 151.112062, 147.089714, 143.085215, 139.101831, 135.142824, 131.211457, 127.310994, 123.444698, 119.615833, 115.827663, 112.083449, 108.386457, 104.853524, 101.59876, 98.3911898, 95.2267153, 92.101237, 89.0106557, 85.9508722, 82.9177873, 79.907302, 76.9153171, 73.9377334, 70.9704518, 68.0093731, 65.0503982, 62.089428, 59.1223631, 56.1451047, 52.4868772, 48.4909423, 44.5085745, 40.5573096, 36.6546832, 32.8182313, 29.0654895, 25.4139936, 21.8812795, 18.4848828, 15.2423393, 12.1711849, 9.28895525, 6.61318617, 4.16141342, 1.95117277, 0, 3.86119844, 7.06472862, 9.67826325, 11.769475, 13.4060367, 14.655621, 15.5859006, 16.2645482, 16.7592366, 17.1376385, 17.4674265, 17.8162734, 18.2518519, 18.8418347, 19.6538946, 20.7557042, 22.1040048, 23.7506217, 26.0141442, 28.9384738, 32.5675124, 36.9451615, 42.1153229, 48.1218982, 55.0087891, 62.8198973, 71.5991245, 81.3903723, 92.2375424, 103.586144, 116.297882, 130.596175, 146.704444, 165.085833, 185.843898, 209.01827, 234.760451, 263.221941, 294.554244, 328.90886, 364.222869, 403.536151, 447.433462, 496.499562, 551.319209, 615.077588, 685.108923, 761.347868, 843.729074, 932.187193, 1024.20565, 1125.16627, 1238.31804, 1365.41646, 1505.16778, 1657.6473, 1822.93031, 2001.42223, 2192.79321, 2396.35328, 2608.98081, 2836.46984, 3086.23543, 3360.82955, 3654.564, 3965.53294, 4289.48919, 4623.9199, 4974.89721, 5343.63711, 5750.93478, 6175.41484, 6611.51584, 7053.67638, 7496.335, 7983.6836, 8447.96911, 8871.19179, 9235.35189, 9522.44966, 9620.14807, 9667.67617, 9687.46445, 9777.67526, 9930.76191, 10107.6228, 10302.2891, 10508.7917, 10721.1618, 10933.4305, 11119.4299, 11445.954, 11764.5693, 11926.8424, 11784.3399, 11188.6282, 9991.27395, 8604.50298, 7081.52303}, 
{333.727898, 332.175783, 330.527593, 328.785111, 326.950121, 325.024408, 323.009755, 320.907946, 318.720765, 316.449996, 314.097423, 311.664831, 309.154002, 306.56672, 303.904771, 301.169937, 298.364003, 295.488753, 292.54597, 289.537438, 286.464942, 283.330265, 280.135192, 276.881505, 273.57099, 270.20543, 266.786609, 263.316311, 259.79632, 256.22842, 252.614395, 248.956028, 245.255104, 241.513407, 237.717634, 233.87743, 230.00262, 226.095478, 222.158278, 218.193292, 214.202795, 210.189061, 206.154361, 202.100971, 198.031163, 193.947212, 189.85139, 185.745971, 181.633229, 177.515436, 173.394868, 169.231278, 165.07038, 160.91537, 156.769443, 152.635794, 148.517618, 144.418111, 140.340469, 136.287885, 132.263556, 128.270677, 124.312442, 120.392048, 116.512689, 112.677561, 108.88986, 105.256539, 101.882065, 98.5556336, 95.2737145, 92.0327766, 88.8292885, 85.6597193, 82.5205379, 79.408213, 76.3192137, 73.2500087, 70.1970671, 67.1568576, 64.1258492, 61.1005108, 58.0773112, 55.0527194, 51.39921, 47.4383739, 43.4993019, 39.5987127, 35.7533251, 31.9798577, 28.2950292, 24.7155583, 21.2581638, 17.9395643, 14.7764785, 11.7856252, 8.98372308, 6.38749077, 4.01364702, 1.87891053, 0, 4.04991556, 7.39132138, 10.0973282, 12.241047, 13.8955883, 15.134063, 16.029582, 16.6552559, 17.0841956, 17.3895118, 17.6443154, 17.9217172, 18.2948278, 18.8367581, 19.620619, 20.7195211, 21.8432979, 23.0131636, 25.1239275, 28.1708553, 32.1492126, 37.0542651, 42.8812784, 49.6255182, 57.2822502, 65.8467399, 75.3142531, 85.6800554, 96.9394124, 108.10211, 120.510237, 134.454701, 150.226413, 168.742091, 189.979739, 213.750481, 240.157484, 269.303913, 301.292936, 336.227721, 371.573694, 410.874552, 454.74954, 503.817904, 558.698889, 623.15999, 693.885142, 770.70653, 853.456337, 941.966749, 1033.5054, 1133.60347, 1245.0986, 1369.67665, 1506.67192, 1656.47451, 1819.47452, 1996.42989, 2187.27929, 2391.56009, 2607.82225, 2839.1379, 3089.31402, 3359.78624, 3647.9721, 3953.13349, 4273.48223, 4608.008, 4959.55073, 5328.77244, 5737.92467, 6162.75842, 6596.46242, 7032.22537, 7463.23597, 7907.62355, 8327.40105, 8709.52204, 9040.94007, 9308.60871, 9426.80263, 9503.60686, 9557.12306, 9666.78369, 9826.41364, 10004.2833, 10195.4952, 10395.1519, 10598.3561, 10800.2104, 10911.7093, 11200.5469, 11515.3964, 11704.9309, 11617.8237, 11102.7478, 10008.3766, 8725.61959, 7141.61384}, 
{341.193481, 339.627413, 337.956607, 336.183151, 334.309131, 332.336633, 330.267745, 328.104551, 325.84914, 323.503597, 321.070009, 318.550462, 315.947043, 313.261839, 310.496936, 307.65442, 304.736378, 301.744896, 298.682061, 295.54996, 292.350678, 289.086303, 285.758921, 282.370619, 278.923482, 275.419597, 271.861052, 268.249932, 264.588323, 260.878313, 257.121988, 253.321435, 249.478739, 245.595988, 241.669795, 237.705098, 233.706901, 229.677468, 225.619063, 221.533951, 217.424394, 213.292658, 209.141006, 204.971703, 200.787012, 196.589198, 192.380525, 188.163256, 183.939657, 179.711991, 175.482521, 171.212371, 166.945837, 162.686076, 158.436245, 154.199499, 149.978994, 145.777886, 141.599332, 137.446487, 133.322506, 129.230547, 125.173765, 121.155316, 117.178356, 113.246041, 109.361528, 105.622145, 102.122962, 98.6729078, 95.269034, 91.9083916, 88.5880315, 85.305005, 82.056363, 78.8391567, 75.6504371, 72.4872553, 69.3466624, 66.2257095, 63.1214477, 60.030928, 56.9512015, 53.8793193, 50.2373976, 46.3220477, 42.4367917, 38.5973383, 34.8193963, 31.1186746, 27.5108821, 24.0117275, 20.6369196, 17.4021674, 14.3231795, 11.415665, 8.69533249, 6.17789093, 3.8790491, 1.81451585, 0, 3.95148948, 7.19151186, 9.79524529, 11.8378679, 13.3945578, 14.5404933, 15.3508523, 15.9008131, 16.2655538, 16.5202525, 16.7400874, 17.0002366, 17.3758783, 17.9421906, 18.7743517, 19.9475396, 21.0143336, 21.9752549, 24.0875506, 27.3144133, 31.6190358, 36.9646107, 43.3143306, 50.6313882, 58.8789761, 68.0202871, 78.0185137, 88.8368485, 100.438484, 111.559906, 123.840806, 137.612391, 153.205865, 171.73275, 193.134092, 217.142854, 243.856149, 273.371085, 305.784774, 341.194327, 377.165757, 417.097606, 461.582197, 511.211856, 566.578907, 631.36921, 702.308173, 779.214736, 861.90784, 950.206428, 1041.43275, 1140.95395, 1251.1616, 1373.64002, 1508.32534, 1655.89369, 1817.02122, 1992.92622, 2183.61945, 2388.52017, 2607.11021, 2840.76608, 3090.82994, 3358.02375, 3642.01867, 3942.9682, 4261.19309, 4596.89017, 4949.64312, 5319.38243, 5727.19419, 6148.59865, 6576.20325, 7002.61542, 7420.44261, 7830.68808, 8215.46448, 8565.28029, 8870.64398, 9122.06403, 9252.58132, 9348.48366, 9424.90852, 9549.09965, 9715.56909, 9897.11792, 10089.3437, 10287.8441, 10488.2166, 10686.0589, 10776.0966, 11042.6388, 11343.8794, 11538.0124, 11483.2318, 11037.7314, 10059.7054, 8869.23083, 7283.2224}, 
{349.495856, 348.016876, 346.411673, 344.683142, 342.834178, 340.867673, 338.786524, 336.593623, 334.291866, 331.884146, 329.373357, 326.762395, 324.054154, 321.251526, 318.357408, 315.374693, 312.306276, 309.15505, 305.92391, 302.61575, 299.233465, 295.779949, 292.258096, 288.6708, 285.020956, 281.311458, 277.5452, 273.725077, 269.853982, 265.934811, 261.970457, 257.963814, 253.917778, 249.835241, 245.746118, 241.639225, 237.503053, 233.33962, 229.150944, 224.939042, 220.705932, 216.45363, 212.184155, 207.899524, 203.601754, 199.292864, 194.974869, 190.649788, 186.319639, 181.986438, 177.652203, 173.265231, 168.882425, 164.506967, 160.142038, 155.790821, 151.456498, 147.14225, 142.851261, 138.586711, 134.351784, 130.14966, 125.983522, 121.856553, 117.771933, 113.732845, 109.742472, 105.889875, 102.262054, 98.6850707, 95.1565392, 91.6740745, 88.2352912, 84.8378042, 81.4792282, 78.157178, 74.8692684, 71.6131141, 68.3863298, 65.1865305, 62.0113307, 58.8583454, 55.7251892, 52.6094769, 48.9899623, 45.1346041, 41.3175976, 37.5533955, 33.8564507, 30.2412159, 26.722144, 23.3136877, 20.0303, 16.8864335, 13.8965411, 11.0750756, 8.43648985, 5.99523659, 3.76576866, 1.76253886, 0, 3.34425244, 6.06284509, 8.22666426, 9.90659625, 11.1735274, 12.0983439, 12.7519321, 13.2051784, 13.528969, 13.7941902, 14.0717283, 14.4324697, 14.9473005, 15.6871072, 16.7227761, 18.1251933, 19.4689097, 20.7539063, 23.1737334, 26.6929198, 31.2759941, 36.8874849, 43.491921, 51.0538309, 59.5377433, 68.9081869, 79.1296904, 90.1667823, 101.983991, 113.359355, 125.878939, 139.863221, 155.632678, 174.057194, 195.182541, 218.907982, 245.389169, 274.781757, 307.241398, 342.923746, 380.327679, 421.769861, 467.730098, 518.688193, 575.123953, 639.716646, 710.196747, 786.494194, 868.538926, 956.260881, 1047.36712, 1146.72731, 1256.38605, 1377.82549, 1511.37949, 1657.89748, 1818.22887, 1994.1585, 2185.38777, 2390.59766, 2608.69929, 2841.48843, 3090.2746, 3355.8965, 3638.39498, 3938.17711, 4256.6712, 4594.54907, 4948.73806, 5318.28359, 5719.58752, 6131.66864, 6547.56436, 6960.31214, 7362.94939, 7753.52147, 8118.80592, 8450.58819, 8740.65372, 8980.78797, 9109.9137, 9207.92086, 9289.25032, 9419.98174, 9594.11566, 9784.13651, 9985.12064, 10192.1444, 10400.2842, 10604.6164, 10745.4911, 11011.056, 11285.4055, 11452.6341, 11396.836, 11002.1056, 10152.5374, 9024.52875, 7525.97059}, 
{358.020359, 356.636692, 355.104338, 353.42703, 351.608501, 349.652485, 347.562714, 345.342922, 342.996843, 340.528209, 337.940754, 335.238212, 332.424315, 329.502797, 326.477391, 323.35183, 320.129849, 316.815179, 313.411554, 309.922709, 306.352375, 302.704286, 298.982176, 295.189777, 291.330824, 287.409049, 283.428185, 279.391967, 275.304127, 271.168398, 266.988515, 262.768209, 258.511215, 254.221266, 249.962353, 245.706814, 241.42626, 237.122469, 232.79722, 228.452289, 224.089455, 219.710495, 215.317187, 210.911309, 206.494639, 202.068955, 197.636034, 193.197653, 188.755592, 184.311628, 179.867538, 175.358799, 170.854928, 166.35914, 161.874649, 157.40467, 152.952418, 148.52111, 144.113958, 139.73418, 135.384989, 131.069601, 126.79123, 122.553092, 118.358403, 114.210376, 110.112227, 106.145166, 102.388532, 98.6846796, 95.0317675, 91.4279548, 87.8714006, 84.3602636, 80.8927028, 77.4668772, 74.0809456, 70.733067, 67.4214003, 64.1441044, 60.8993382, 57.6852607, 54.5000308, 51.3418074, 47.7458515, 43.9515074, 40.2036832, 36.5155596, 32.9003175, 29.3711378, 25.9412013, 22.6236889, 19.4317814, 16.3786596, 13.4775045, 10.7414968, 8.18381741, 5.81764719, 3.65616699, 1.71255764, 0, 2.71108226, 4.88904952, 6.59964817, 7.90862458, 8.88172513, 9.58469621, 10.0832842, 10.4432355, 10.7302964, 11.0102134, 11.3487328, 11.811601, 12.4645644, 13.3733694, 14.6037623, 16.2214895, 17.8824396, 19.5938078, 22.3483999, 26.1241356, 30.8989347, 36.6507168, 43.3574016, 50.9969089, 59.5471582, 68.9860693, 79.291562, 90.4415558, 102.41397, 114.140084, 127.028227, 141.37031, 157.458247, 175.806766, 196.596371, 219.948148, 246.087164, 275.238487, 307.627183, 343.47832, 382.34528, 425.329241, 472.786686, 525.0741, 582.547965, 646.688871, 716.448171, 791.90132, 873.123777, 960.190998, 1051.30041, 1150.70087, 1260.149, 1381.01997, 1513.91013, 1659.7655, 1819.53211, 1995.36462, 2186.72572, 2391.75956, 2608.81796, 2840.26197, 3087.73844, 3352.28171, 3633.88833, 3933.03121, 4251.88324, 4591.35808, 4946.1361, 5314.42357, 5707.18332, 6107.90251, 6510.37168, 6908.38136, 7295.72209, 7672.36952, 8024.38279, 8344.00615, 8623.48387, 8855.0602, 8979.28892, 9074.56508, 9155.28599, 9289.03169, 9469.14801, 9666.82128, 9876.5053, 10092.6539, 10309.7207, 10522.1597, 10741.6412, 11015.0658, 11258.6157, 11388.4727, 11320.8192, 10971.837, 10257.7083, 9204.02541, 7835.73577}, 
{366.595695, 365.285845, 363.80766, 362.165589, 360.364082, 358.40759, 356.300562, 354.04745, 351.652702, 349.12077, 346.456104, 343.663154, 340.74637, 337.710203, 334.559102, 331.297519, 327.929902, 324.460703, 320.894371, 317.235358, 313.488112, 309.657085, 305.746726, 301.761487, 297.705816, 293.584165, 289.400983, 285.160722, 280.86783, 276.526758, 272.141958, 267.717877, 263.258968, 258.76968, 254.341456, 249.933421, 245.503652, 241.053777, 236.585423, 232.100218, 227.599788, 223.085761, 218.559764, 214.023424, 209.478368, 204.926223, 200.368617, 195.807177, 191.243529, 186.679301, 182.116121, 177.481042, 172.851882, 168.231883, 163.62429, 159.032347, 154.459296, 149.908383, 145.38285, 140.885941, 136.420901, 131.990973, 127.599401, 123.249427, 118.944298, 114.687255, 110.481543, 106.401009, 102.517805, 98.6895609, 94.914943, 91.1926181, 87.521253, 83.8995147, 80.3260699, 76.7995853, 73.3187279, 69.8821644, 66.4885617, 63.1365865, 59.8249057, 56.5521861, 53.3170944, 50.1182976, 46.5441461, 42.8070954, 39.1245325, 35.5084398, 31.9707996, 28.5235944, 25.1788065, 21.9484183, 18.8444121, 15.8787703, 13.0634753, 10.4105095, 7.9318551, 5.63949461, 3.54541035, 1.66158469, 0, 2.18501245, 3.91519521, 5.25159726, 6.2552676, 6.98725523, 7.50860915, 7.88037834, 8.16361181, 8.41935854, 8.70866754, 9.0925878, 9.63216832, 10.3884581, 11.4225061, 12.7953614, 14.5680729, 16.521368, 18.6762498, 21.7278281, 25.6770828, 30.5249942, 36.2725423, 42.9207071, 50.4704688, 58.9228074, 68.2787031, 78.5391358, 89.7050857, 101.777533, 113.926094, 127.287945, 142.113477, 158.653077, 177.027147, 197.551069, 220.574892, 246.388002, 275.279785, 307.539626, 343.456913, 383.532528, 427.779991, 476.447309, 529.782488, 588.033534, 651.529615, 720.417286, 794.92426, 875.278254, 961.706984, 1052.92037, 1152.51901, 1262.07678, 1382.87042, 1515.56991, 1661.11765, 1820.45603, 1995.75959, 2186.45855, 2390.63904, 2606.48833, 2836.62819, 3083.01444, 3346.79012, 3627.72111, 3926.20544, 4244.79564, 4584.44829, 4938.22017, 5303.63663, 5686.41277, 6074.62424, 6462.9615, 6846.11501, 7218.77521, 7585.66209, 7928.92919, 8240.75955, 8513.33625, 8738.84234, 8857.05624, 8947.5021, 9024.82213, 9158.77876, 9342.19577, 9544.93017, 9760.9541, 9984.23965, 10208.759, 10428.4841, 10741.4879, 11028.2687, 11240.1215, 11328.3411, 11244.2222, 10939.0598, 10364.1486, 9408.65324, 8205.08232}, 
{374.999505, 373.708224, 372.236267, 370.588493, 368.769764, 366.784941, 364.638884, 362.336456, 359.882515, 357.281924, 354.539543, 351.660234, 348.648857, 345.510273, 342.249342, 338.870927, 335.379888, 331.781086, 328.079381, 324.279635, 320.386709, 316.405463, 312.340759, 308.197457, 303.980418, 299.694504, 295.344575, 290.935491, 286.472115, 281.959307, 277.401928, 272.804838, 268.172899, 263.510972, 258.922247, 254.360351, 249.777732, 245.176059, 240.557002, 235.922231, 231.273416, 226.612227, 221.940333, 217.259405, 212.571113, 207.877126, 203.179115, 198.478749, 193.777699, 189.077634, 184.380224, 179.61389, 174.855139, 170.107229, 165.373417, 160.656962, 155.961122, 151.289154, 146.644316, 142.029866, 137.449063, 132.905162, 128.401424, 123.941105, 119.527463, 115.163756, 110.853242, 106.663176, 102.659275, 98.7128091, 94.8228873, 90.988619, 87.2091134, 83.4834796, 79.810827, 76.1902646, 72.6209017, 69.1018476, 65.6322114, 62.2111023, 58.8376296, 55.5109025, 52.2300302, 48.9941218, 45.4363231, 41.7464226, 38.1186074, 34.563863, 31.0931748, 27.7175283, 24.4479088, 21.2953019, 18.270693, 15.3850674, 12.6494106, 10.0747082, 7.67194539, 5.45210775, 3.42618068, 1.60514962, 0, 1.96451446, 3.50682287, 4.68563487, 5.55966008, 6.18760814, 6.62818868, 6.94011134, 7.18208574, 7.41282151, 7.69102829, 8.07541571, 8.6246934, 9.39757099, 10.4527581, 11.8489644, 13.6448995, 15.7718446, 18.2703054, 21.4872223, 25.4539989, 30.2020386, 35.7627451, 42.1675219, 49.4477726, 57.6349007, 66.7603098, 76.8554034, 87.9515851, 100.080258, 112.709861, 126.641184, 142.074519, 159.21016, 177.819191, 198.316511, 221.231472, 246.89313, 275.63054, 307.772759, 343.648842, 384.31317, 429.148723, 478.342644, 532.082078, 590.554167, 653.292913, 721.301889, 794.931526, 874.532254, 960.454502, 1051.82707, 1151.71512, 1261.70473, 1383.00756, 1516.07083, 1661.68498, 1820.64046, 1994.51794, 2183.13803, 2385.45925, 2600.48057, 2830.12193, 3076.06432, 3339.2355, 3619.28696, 3916.45599, 4233.29978, 4570.65708, 4920.86051, 5281.05423, 5653.0316, 6028.57234, 6403.21132, 6772.48338, 7131.92335, 7491.44332, 7828.6066, 8135.35372, 8403.62521, 8625.36162, 8739.2969, 8826.04923, 8900.36196, 9032.45804, 9215.12016, 9418.01451, 9635.07962, 9860.25399, 10087.4762, 10310.6846, 10719.7118, 11022.209, 11204.5611, 11253.1529, 11154.3696, 10894.5959, 10460.2169, 9643.92192, 8631.43485}, 
{382.516148, 381.115486, 379.540329, 377.795159, 375.884458, 373.812711, 371.584399, 369.204006, 366.676015, 364.004909, 361.195169, 358.251281, 355.177725, 351.978986, 348.659545, 345.223887, 341.676494, 338.021848, 334.264433, 330.408732, 326.459227, 322.420402, 318.296738, 314.092721, 309.812831, 305.461552, 301.043367, 296.562759, 292.024211, 287.432205, 282.791225, 278.105753, 273.380273, 268.619267, 263.896795, 259.181087, 254.43954, 249.674377, 244.887824, 240.082106, 235.259448, 230.422074, 225.572211, 220.712082, 215.843913, 210.96993, 206.092356, 201.213417, 196.335337, 191.460343, 186.590659, 181.680931, 176.781995, 171.897107, 167.029522, 162.182499, 157.359292, 152.563159, 147.797355, 143.065138, 138.369764, 133.714488, 129.102568, 124.53726, 120.02182, 115.559505, 111.153571, 106.867748, 102.764314, 98.7218031, 94.739552, 90.8168965, 86.9531728, 83.147717, 79.3998654, 75.708954, 72.0743189, 68.4952965, 64.9712227, 61.5014338, 58.0852659, 54.7220552, 51.4111379, 48.15185, 44.5936101, 40.9180734, 37.3111004, 33.7833828, 30.3456123, 27.0084805, 23.7826792, 20.6788998, 17.7078342, 14.8801739, 12.2066105, 9.69783586, 7.36454148, 5.21741905, 3.26716023, 1.52445666, 0, 2.88022254, 5.1932796, 7.00585151, 8.38461859, 9.39626118, 10.1074596, 10.5848942, 10.8952453, 11.1051932, 11.2814183, 11.4906008, 11.7994212, 12.2745596, 12.9826966, 13.9905123, 15.3646872, 17.176431, 19.493071, 22.3710425, 25.8779964, 30.0815836, 35.049455, 40.8492616, 47.5486543, 55.2152839, 63.9168015, 73.7208579, 84.695104, 96.9071908, 110.179357, 124.91465, 141.254345, 159.339716, 178.816764, 200.074401, 223.633611, 249.784248, 278.816171, 311.019235, 346.683295, 386.171493, 429.678097, 477.478626, 529.848596, 587.063528, 649.200255, 716.78265, 790.135902, 869.5852, 955.455734, 1046.7731, 1146.75047, 1257.68959, 1380.90541, 1515.69812, 1662.2725, 1820.83334, 1990.42356, 2172.67385, 2370.48238, 2587.13043, 2820.21381, 3068.47991, 3331.39332, 3609.63397, 3903.32393, 4214.56246, 4543.98426, 4884.97424, 5235.01819, 5596.27018, 5960.82951, 6324.56605, 6683.34964, 7033.05013, 7384.03082, 7714.04474, 8015.33837, 8280.15818, 8500.75067, 8619.50312, 8711.76067, 8791.13802, 8920.12321, 9092.98526, 9283.62928, 9487.34249, 9699.41209, 9915.12528, 10129.7693, 10629.1616, 10948.5701, 11107.506, 11125.4801, 11022.0035, 10816.5871, 10528.7419, 9959.55953, 9159.17045}, 
{390.157831, 388.583562, 386.847167, 384.952525, 382.903514, 380.704013, 378.357901, 375.869055, 373.241355, 370.478678, 367.584903, 364.56391, 361.419575, 358.155778, 354.776397, 351.28531, 347.686396, 343.983534, 340.180602, 336.281478, 332.29004, 328.210169, 324.04574, 319.800634, 315.478729, 311.083903, 306.620035, 302.091002, 297.500684, 292.85296, 288.151706, 283.400803, 278.604128, 273.765559, 268.92037, 264.056081, 259.159836, 254.234496, 249.282919, 244.307965, 239.312494, 234.299366, 229.27144, 224.231577, 219.182635, 214.127474, 209.068954, 204.009936, 198.953277, 193.901839, 188.858481, 183.810104, 178.775873, 173.758992, 168.762667, 163.790104, 158.844509, 153.929087, 149.047044, 144.201585, 139.395916, 134.633243, 129.916771, 125.249706, 120.635253, 116.076618, 111.577006, 107.194927, 102.987559, 98.844051, 94.7640244, 90.7470994, 86.7928966, 82.9010366, 79.0711399, 75.3028272, 71.595719, 67.9494358, 64.3635984, 60.8378273, 57.371743, 53.9649661, 50.6171173, 47.3278172, 43.7677785, 40.1079939, 36.5234092, 33.0243186, 29.6210164, 26.3237968, 23.1429542, 20.0887829, 17.171577, 14.4016308, 11.7892387, 9.34469493, 7.0782937, 5.00032932, 3.12109607, 1.4508882, 0, 3.54846632, 6.42739653, 8.70855927, 10.4637231, 11.7646568, 12.6831288, 13.2909078, 13.6597624, 13.8614613, 13.967773, 14.0504663, 14.1813096, 14.4320716, 14.874521, 15.5804263, 16.6215562, 18.2270743, 20.4912341, 23.129792, 26.2482439, 29.952086, 34.3468145, 39.5379254, 45.6309149, 52.731279, 60.9445139, 70.3761156, 81.1315804, 93.3164042, 107.089166, 122.482815, 139.586924, 158.491062, 178.786732, 200.81254, 225.03991, 251.707835, 281.055307, 313.321318, 348.744861, 386.87699, 428.854008, 475.049503, 525.837063, 581.590278, 642.892065, 709.854352, 782.798393, 862.045446, 947.916766, 1039.47941, 1139.84175, 1251.76568, 1376.58689, 1512.7291, 1659.92346, 1817.90108, 1983.62874, 2160.2302, 2353.84491, 2571.3007, 2806.59936, 3056.14321, 3318.16783, 3594.01511, 3883.60108, 4188.28863, 4508.36891, 4838.82795, 5177.65266, 5528.21553, 5882.28934, 6235.99674, 6585.46039, 6926.80295, 7270.79522, 7595.2497, 7892.62702, 8155.38781, 8375.99269, 8500.23587, 8598.35536, 8682.8117, 8807.94415, 8969.63625, 9146.32204, 9334.75751, 9531.69865, 9733.90146, 9938.12194, 10499.1588, 10828.6035, 10971.3124, 10972.1422, 10875.9492, 10727.5898, 10571.9207, 10250.4084, 9668.20376}, 
{398.27571, 396.443157, 394.466948, 392.350171, 390.095915, 387.70727, 385.187323, 382.539164, 379.765881, 376.870564, 373.8563, 370.72618, 367.48329, 364.130722, 360.671562, 357.108901, 353.445826, 349.685426, 345.830791, 341.885009, 337.851169, 333.73236, 329.53167, 325.252189, 320.897005, 316.469206, 311.971882, 307.408122, 302.781014, 298.093647, 293.349109, 288.550491, 283.70088, 278.803365, 273.849165, 268.847553, 263.807947, 258.733818, 253.628643, 248.495894, 243.339045, 238.161571, 232.966946, 227.758643, 222.540137, 217.314901, 212.086409, 206.858137, 201.633556, 196.416142, 191.209369, 186.034531, 180.876895, 175.739548, 170.625579, 165.538076, 160.480125, 155.454815, 150.465233, 145.514467, 140.605604, 135.741733, 130.92594, 126.161315, 121.450943, 116.797914, 112.205314, 107.723075, 103.400002, 99.1425737, 94.9508399, 90.8248519, 86.7646608, 82.7703176, 78.8418733, 74.979379, 71.1828857, 67.4524444, 63.7881061, 60.1899218, 56.6579427, 53.1922196, 49.7928036, 46.4597458, 42.9009695, 39.2687341, 35.7188101, 32.2607286, 28.9040207, 25.6582176, 22.5328504, 19.5374502, 16.6815482, 13.9746754, 11.426363, 9.04614211, 6.84354389, 4.82809944, 3.00933989, 1.39679637, 0, 3.43437573, 6.22633736, 8.44457816, 10.1577914, 11.4346702, 12.343908, 12.954198, 13.3342333, 13.5527074, 13.6783133, 13.7797444, 13.925694, 14.1848552, 14.6259213, 15.3175856, 16.3285413, 18.071215, 20.6634048, 23.3963118, 26.4122863, 29.8536788, 33.8628396, 38.5821192, 44.153868, 50.7204364, 58.4241747, 67.4074334, 77.8125629, 89.7819136, 103.738564, 119.441202, 136.947961, 156.316973, 177.177902, 199.803113, 224.579232, 251.692929, 281.330877, 313.679749, 348.926217, 385.934154, 426.615622, 471.416102, 520.781073, 575.156016, 635.174814, 701.047444, 773.172287, 851.947724, 937.772134, 1030.16444, 1131.47737, 1244.23959, 1369.60733, 1505.93481, 1652.83429, 1809.91804, 1973.5397, 2147.31075, 2338.39757, 2554.84415, 2789.36721, 3037.72412, 3297.67539, 3570.37579, 3855.42205, 4153.21962, 4463.57487, 4783.32909, 5111.00077, 5451.49994, 5795.97029, 6140.71788, 6482.04877, 6816.26904, 7154.51287, 7474.55117, 7768.98295, 8030.40723, 8251.42305, 8380.68281, 8484.02991, 8572.89763, 8694.38929, 8845.8188, 9009.63756, 9183.9026, 9366.67096, 9555.99968, 9749.9458, 10320.7513, 10646.6689, 10784.2824, 10790.176, 10720.9335, 10633.139, 10583.3765, 10477.3235, 10110.609}, 
{407.835532, 405.541344, 403.134444, 400.616802, 397.990387, 395.257169, 392.419118, 389.478203, 386.436393, 383.295659, 380.05797, 376.725295, 373.299604, 369.782867, 366.177052, 362.48413, 358.706071, 354.844843, 350.902417, 346.880761, 342.781846, 338.607641, 334.360116, 330.04124, 325.652982, 321.197313, 316.676202, 312.091618, 307.445532, 302.739912, 297.976728, 293.157949, 288.285547, 283.361488, 278.328013, 273.218211, 268.065894, 262.874969, 257.649344, 252.392928, 247.109628, 241.803352, 236.478008, 231.137504, 225.785749, 220.426649, 215.064114, 209.702051, 204.344368, 198.994973, 193.657774, 188.383684, 183.128587, 177.895372, 172.686928, 167.506145, 162.35591, 157.239114, 152.158645, 147.117392, 142.118245, 137.164092, 132.257822, 127.402325, 122.600489, 117.855204, 113.169358, 108.578643, 104.117963, 99.7225543, 95.3931795, 91.130601, 86.9355815, 82.8088836, 78.7512697, 74.7635025, 70.8463446, 67.0005584, 63.2269066, 59.5261517, 55.8990564, 52.3463831, 48.8688944, 45.467353, 41.9233007, 38.3517214, 34.8702339, 31.4867149, 28.2090411, 25.0450892, 22.0027359, 19.0898579, 16.3143319, 13.6840346, 11.2068427, 8.89063288, 6.7432819, 4.77266642, 2.98666314, 1.39314877, 0, 1.22297946, 2.17890016, 2.91159544, 3.46489866, 3.88264318, 4.20866234, 4.4867895, 4.76085801, 5.07470123, 5.47215251, 5.99704519, 6.69321265, 7.60448821, 8.77470525, 10.2476971, 12.0672972, 14.9009569, 18.8815981, 22.642489, 26.3610956, 30.2148835, 34.3813187, 39.037867, 44.3619942, 50.5311661, 57.7228485, 66.1145073, 75.8836083, 87.2076173, 100.589442, 115.761778, 132.804458, 151.797315, 172.62655, 195.46881, 220.552384, 248.015193, 277.995162, 310.630213, 346.058269, 382.966048, 423.38435, 467.735032, 516.439949, 569.920955, 627.402759, 690.803649, 760.84477, 838.247262, 923.732269, 1018.41875, 1122.14381, 1235.36394, 1358.53514, 1492.11236, 1636.55104, 1792.30659, 1959.02824, 2138.16085, 2332.02874, 2543.80421, 2770.88492, 3012.63064, 3268.26507, 3536.78143, 3817.27872, 4109.03373, 4411.19156, 4722.24547, 5041.05741, 5373.62691, 5710.58031, 6048.30891, 6383.20401, 6711.6569, 7044.43687, 7359.96272, 7651.03121, 7910.43913, 8130.98329, 8263.69912, 8370.98563, 8463.53733, 8583.9373, 8730.25747, 8887.28286, 9053.74974, 9228.39439, 9409.95307, 9597.16206, 10070.8116, 10364.715, 10519.4504, 10575.5964, 10573.7312, 10554.4334, 10558.2813, 10577.0695, 10422.3577}, 
{417.246149, 414.576951, 411.817936, 408.970252, 406.035046, 403.013467, 399.906663, 396.715783, 393.441974, 390.086385, 386.650163, 383.134459, 379.540419, 375.869191, 372.121925, 368.299768, 364.403868, 360.435374, 356.395434, 352.285196, 348.105808, 343.85842, 339.544178, 335.164231, 330.719727, 326.211815, 321.641643, 317.010359, 312.319111, 307.569048, 302.761317, 297.897068, 292.977447, 288.003604, 282.881066, 277.660802, 272.394933, 267.087708, 261.74338, 256.3662, 250.96042, 245.53029, 240.080062, 234.613987, 229.136316, 223.651302, 218.163194, 212.676245, 207.194706, 201.722829, 196.264863, 190.902221, 185.560321, 180.241743, 174.949064, 169.684864, 164.451721, 159.252213, 154.08892, 148.96442, 143.881292, 138.842114, 133.849465, 128.905924, 124.014069, 119.176479, 114.395733, 109.688017, 105.06919, 100.513768, 96.0234448, 91.5999184, 87.2448851, 82.9600413, 78.7470836, 74.6077085, 70.5436125, 66.5564921, 62.6480439, 58.8199644, 55.0739501, 51.4116975, 47.8349031, 44.3452635, 40.8149125, 37.313053, 33.9104454, 30.6129909, 27.4265905, 24.3571452, 21.4105563, 18.5927247, 15.9095515, 13.3669378, 10.9707848, 8.72699345, 6.64146487, 4.72010016, 2.96880039, 1.39346664, 0, -1.15522203, -2.1836572, -3.06704198, -3.78711285, -4.32560627, -4.66425872, -4.78480667, -4.66898659, -4.29853495, -3.65518821, -2.72068286, -1.47675535, 0.0948578267, 2.01242021, 4.29419534, 6.95844673, 10.8829107, 16.2086327, 21.0098305, 25.4889405, 29.8483988, 34.2906418, 39.0181057, 44.2332269, 50.1384416, 56.936186, 64.8288965, 74.0190094, 84.7089608, 97.427157, 111.930542, 128.323761, 146.71146, 167.149337, 189.76651, 214.705152, 242.08459, 272.024155, 304.643176, 340.060982, 377.10244, 417.575309, 461.852182, 510.305652, 563.308314, 618.651177, 679.933815, 748.174217, 824.390373, 909.600271, 1006.44494, 1112.33562, 1225.61582, 1346.04457, 1476.27092, 1617.64635, 1771.52234, 1941.18958, 2125.61962, 2321.66856, 2526.78966, 2744.03906, 2977.20005, 3227.31784, 3490.79823, 3766.17658, 4051.51113, 4345.21355, 4647.44496, 4957.3769, 5281.83401, 5611.15731, 5941.86918, 6270.49201, 6593.54816, 6922.87494, 7235.85108, 7525.17023, 7783.52604, 8003.61216, 8136.88194, 8244.76286, 8337.62299, 8456.06224, 8598.48192, 8750.68671, 8911.70771, 9080.57606, 9256.32287, 9437.97926, 9785.18601, 10037.9282, 10213.3685, 10328.6696, 10400.9941, 10447.5047, 10485.364, 10586.6351, 10606.4136}, 
{425.6531, 422.877324, 420.012663, 417.060191, 414.020985, 410.896119, 407.686671, 404.393714, 401.018325, 397.56158, 394.024554, 390.408322, 386.713961, 382.942546, 379.095152, 375.172856, 371.176732, 367.107857, 362.967305, 358.756154, 354.475477, 350.126352, 345.709853, 341.227056, 336.679036, 332.06687, 327.391633, 322.654401, 317.856248, 312.998252, 308.081487, 303.107029, 298.075953, 292.989336, 287.742151, 282.390755, 276.992784, 271.552755, 266.075187, 260.5646, 255.025512, 249.462442, 243.879909, 238.282431, 232.674529, 227.060719, 221.445522, 215.833455, 210.229039, 204.636791, 199.061231, 193.620027, 188.202096, 182.809503, 177.444314, 172.108595, 166.804413, 161.533833, 156.298921, 151.101743, 145.944365, 140.828853, 135.757273, 130.731691, 125.754173, 120.826784, 115.951592, 111.117342, 106.3131, 101.568403, 96.8861805, 92.2693618, 87.7208764, 83.2436537, 78.8406229, 74.5147137, 70.2688553, 66.1059773, 62.0290089, 58.0408797, 54.144519, 50.3428563, 46.638821, 43.0353425, 39.5081888, 36.0744415, 32.7515083, 29.5432, 26.4533275, 23.4857017, 20.6441333, 17.9324333, 15.3544126, 12.9138819, 10.6146521, 8.46053402, 6.45533858, 4.60287661, 2.90695896, 1.37139647, 0, -2.76057923, -5.14494685, -7.15044515, -8.77441641, -10.0142029, -10.8671469, -11.3305908, -11.4018767, -11.0783471, -10.3573441, -9.23621001, -7.7122872, -5.78291791, -3.44544442, -0.697209025, 2.464446, 7.00287879, 13.05799, 18.4594244, 23.4157042, 28.1353514, 32.8268881, 37.6988365, 42.9597185, 48.8180564, 55.4823721, 63.1611879, 72.0630257, 82.3964077, 94.7154596, 108.756379, 124.624006, 142.423