%{

/* lhtml.l: source code for libhtmlparse parser
   Mooneer Salem <mooneer@translator.cx> */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include "lhtml.h"

#define yyterminate() return YY_NULL

static lhtml_tree *working_tree = NULL, *current_level = NULL;
static lhtml_tagnameval html_tags[] = 
        {
                { "a", TAG_A },
                { "base", TAG_BASE },
                { "h1", TAG_H1 },
                { "h2", TAG_H2 },
                { "h3", TAG_H3 },
                { "h4", TAG_H4 },
                { "h5", TAG_H5 },
                { "h6", TAG_H6 },
                { "hr", TAG_HR },
                { "p", TAG_P },
                { "style", TAG_STYLE },
                { "div", TAG_DIV },
                { "br", TAG_BR },
                { "font", TAG_FONT },
                { "img", TAG_IMG },
                { "class", TAG_CLASS },
                { "applet", TAG_APPLET },
                { "embed", TAG_EMBED },
                { "html", TAG_HTML },
                { "head", TAG_HEAD },
                { "body", TAG_BODY },
                { "meta", TAG_META },
                { NULL, TAG_OTHER }
        };

static char *tempparamname, *tempparamvalue;
static lhtml_tree *temptagstruct;
static int godownonelevel=0;

size_t lhtml_input(char *buf, size_t max_size);

int lhtml_text (char *text) {
        lhtml_tree *temp, *tmp;

        /* allocate a temporary struct */
        temp = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temp == NULL) {
                /* terminate if RAM problem */
                return -1;
        }

        temp->other_tag = lhtml_strdup(text);
        temp->type = TYPE_TEXT;
        temp->end = 0;
        temp->params = NULL;
        temp->parent = current_level;
        temp->children = NULL; /* text objects shouldn't have kids */
        temp->next = NULL;

        if (working_tree == NULL) {
                /* first object */
                working_tree = temp;
                current_level = working_tree;
                godownonelevel = 1;
        } else {
                if (godownonelevel == 1) {
                        current_level->children = temp;
                        temp->parent = current_level;
                        current_level = temp;
                        godownonelevel = 0;
                } else {
                        tmp = current_level;
                        while(tmp->next != NULL) {
                                /* find last element */
                                tmp = tmp->next;
                        }

                        tmp->next = temp;
                }
        }

	return 0;
}

int lhtml_entity (char *text) {
        lhtml_tree *temp, *tmp;

        /* allocate a temporary struct */
        temp = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temp == NULL) {
                /* terminate if RAM problem */
                return -1;
        }

        temp->other_tag = lhtml_strdup(text);
        temp->type = TYPE_ENTITY;
        temp->end = 0;
        temp->params = NULL;
        temp->parent = current_level;
        temp->children = NULL; /* text objects shouldn't have kids */
        temp->next = NULL;

        if (working_tree == NULL) {
                /* first object */
                working_tree = temp;
                current_level = working_tree;
                godownonelevel = 1;
        } else {
                if (godownonelevel == 1) {
                        current_level->children = temp;
                        temp->parent = current_level;
                        current_level = temp;
                        godownonelevel = 0;
                } else {
                        tmp = current_level;
                        while(tmp->next != NULL) {
                                /* find last element */
                                tmp = tmp->next;
                        }

                        tmp->next = temp;
                }
        }

	return 0;
}

int lhtml_comment (char *text) {
        lhtml_tree *temp, *tmp;

        /* allocate a temporary struct */
        temp = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temp == NULL) {
                /* terminate if RAM problem */
                return -1;
        }

        temp->other_tag = lhtml_strdup(text);
        temp->type = TYPE_COMMENT;
        temp->end = 0;
        temp->params = NULL;
        temp->parent = current_level;
        temp->children = NULL; /* text objects shouldn't have kids */
        temp->next = NULL;

        if (working_tree == NULL) {
                /* first object */
                working_tree = temp;
                current_level = working_tree;
                godownonelevel = 0;
        } else {
                if (godownonelevel == 1) {
                        current_level->children = temp;
                        temp->parent = current_level;
                        current_level = temp;
                        godownonelevel = 0;
                } else {
                        tmp = current_level;
                        while(tmp->next != NULL) {
                                /* find last element */
                                tmp = tmp->next;
                        }

                        tmp->next = temp;
                }
        }

	return 0;
}

enum lhtml_tag lhtml_tagis (char *tag) {
        lhtml_tagnameval *tmp;

        tmp = html_tags;
        while(tmp->name != NULL) {
                if (lhtml_strcasecmp(tag, tmp->name) == 0) {
                        return tmp->tag;
                }
                tmp++;
        }

        return TAG_OTHER;
}

char *lhtml_tagtext (enum lhtml_tag tag) {
        lhtml_tagnameval *tmp;

        tmp = html_tags;
        while(tmp->name != NULL) {
                if (tag == tmp->tag) {
                        return tmp->name;
                }
                tmp++;
        }

        return NULL;
}

int lhtml_endtag (char *tag) {
        lhtml_tree *temp, *tmp;
        char *tmp2 = lhtml_strdup(tag), *tmp3;

	if (tmp2 == NULL) {
		return -1;
	}

        /* allocate a temporary struct */
        temp = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temp == NULL) {
                /* terminate if RAM problem */
                return -1;
        }

        /* chop off any extra chars from tag */
        tmp3 = tmp2;
        while(*tmp3 != '\0' && *tmp3 != '>') {
                tmp3++;
        }
        *tmp3 = '\0';

        /* figure out correct constant */
        temp->tag = lhtml_tagis(tmp2);
        if (temp->tag == TAG_OTHER) {
                temp->other_tag = tmp2;
        } else {
                temp->other_tag = NULL;
                free(tmp2);
		tmp2 = NULL;
        }

        temp->type = TYPE_HTMLTAG;
        temp->end = 1;
        temp->params = NULL;
        temp->parent = current_level;
        temp->children = NULL;
        temp->next = NULL;

        if (current_level == NULL) {
		current_level = temp;
        } else {
                tmp = current_level;
                while(tmp->next != NULL) {
                        /* find proper location */
                        tmp = tmp->next;
                }

                tmp->next = temp;

                /* move up 1 level */
		if (current_level->parent) current_level = current_level->parent;
        }

	godownonelevel = 0;

	return 0;
}

int lhtml_add_start_tag (lhtml_tree *tree) {
        lhtml_tree *tmp;

        if (current_level == NULL) {
                /* first entry */
                current_level = tree;
                working_tree  = tree;
                godownonelevel = 1;
        } else {
                if (godownonelevel == 1) {
                        current_level->children = tree;
                        tree->parent = current_level;
                        current_level = tree;
                } else {
                        tmp = current_level;
                        while(tmp->next != NULL) {
                                /* find right place */
                                tmp = tmp->next;
                        }
                        tmp->next = tree;
                        current_level = tree;
                        godownonelevel = 1;
                }
        }

	return 0;
}

int lhtml_add_decl_tag (lhtml_tree *tree) {
        lhtml_tree *tmp;

        if (current_level == NULL) {
                /* first entry */
                current_level = tree;
                working_tree  = tree;
        } else {
                if (godownonelevel == 1) {
                        current_level->children = tree;
                        tree->parent = current_level;
                        current_level = tree;
                } else {
                        tmp = current_level;
                        while(tmp->next != NULL) {
                                /* find right place */
                                tmp = tmp->next;
                        }
                        tmp->next = tree;
                        current_level = tree;
                        godownonelevel = 1;
                }
        }

	return 0;
}

lhtml_tree *lhtml_end_parse (void) {
        lhtml_tree *temp = working_tree;

        working_tree = NULL;
        current_level = NULL;

        return temp;
}

#define YY_INPUT(buf, result, max_size) \
	{ \
		result = lhtml_input(buf, max_size); \
	}

#define FALSE 0
#define TRUE 1

int lhtml_fp = 0;
char *lhtml_stringin = NULL;

/* lhtml_input(): allows for string input. */
size_t lhtml_input(char *buf, size_t max_size) {
	size_t amtread = 0;

	if (lhtml_fp == FALSE) {
		if (feof(yyin)) {
			return YY_NULL;
		}
/*
		while(!feof(yyin) && amtread < max_size) {
			*buf++ = fgetc(yyin);
			amtread++;
		}
		if (feof(yyin)) {
			*buf = '\0';
			amtread--;
		} */
		amtread = fread((void*)buf, 1, max_size, yyin);
		if (amtread <= 0) {
			return YY_NULL;
		}

		return amtread;
	}

	if (*lhtml_stringin == '\0') {
		return YY_NULL;
	}

	amtread = strlen(lhtml_stringin);
	amtread = (amtread < max_size ? amtread : max_size);
	strncpy(buf, lhtml_stringin, amtread);
	lhtml_stringin += amtread;
	return amtread;
}

void lhtml_parse_filehandle (FILE *fp) {
        yyin = fp;
	lhtml_fp = FALSE;
        yylex();
}

void lhtml_parse_string (char *string) {
	lhtml_fp = TRUE;
	lhtml_stringin = string;
	yylex();
}

int lhtml_add_param (lhtml_tree *tree, char *param, char *val) {
        lhtml_param_list *tmp, *tmp2 = tree->params;

        tmp = (lhtml_param_list*)malloc((size_t)sizeof(lhtml_param_list));
	if (tmp == NULL) {
		return -1;
        }

        tmp->name = param;
        tmp->val = val;
        tmp->next = NULL;

        if (tmp2 == NULL) {
                tree->params = tmp;
        } else {
                while(tmp2->next != NULL) {
                        tmp2 = tmp2->next;
                }

                tmp2->next = tmp;
        }

	return 0;
}

/* this is a debug function that just dumps the tree to stdout */
int lhtml_dump (lhtml_tree *temp, int tab_count) {
        lhtml_tree *tree = temp;
        int i;
	lhtml_param_list *tmp2;

        while(tree != NULL) {
		if (tree->end == 1) {
			tab_count--;
		}

                for(i=0; i<tab_count; i++) {
                        fprintf(yyout, "  ");
                }

                if (tree->type == TYPE_HTMLTAG) {
                        fprintf(yyout, "tag: %s, kind: %s\n", lhtml_tagtext(tree->tag) ?
                                lhtml_tagtext(tree->tag) : tree->other_tag, 
                                tree->end == 0 ? "start" : "end");
			if (tree->params != NULL) {
				tmp2 = tree->params;
				while(tmp2) {
			                for(i=0; i<=tab_count; i++) {
                			        fprintf(yyout, "  ");
                			}

					printf("%s=\"%s\"\n", tmp2->name, tmp2->val ? tmp2->val : "");
					tmp2 = tmp2->next;
				}
			}
			if (tree->children == NULL && tree->end == 0) {
				tab_count++;
			}
                } else if (tree->type == TYPE_TEXT) {
                        fprintf(yyout, "text: %s\n", tree->other_tag);
                } else if (tree->type == TYPE_COMMENT) {
                        fprintf(yyout, "comment: %s\n", tree->other_tag);
                } else if (tree->type == TYPE_ENTITY) {
			fprintf(yyout, "entity: %s\n", tree->other_tag);
		} else if (tree->type == TYPE_DECL) {
			fprintf(yyout, "decltag: %s\n", tree->other_tag);
			if (tree->params != NULL) {
				tmp2 = tree->params;
				while(tmp2) {
			                for(i=0; i<=tab_count; i++) {
                			        fprintf(yyout, "  ");
                			}

					printf("%s=\"%s\"\n", tmp2->name, tmp2->val ? tmp2->val : "");
					tmp2 = tmp2->next;
				}
			}
		}

                if (tree->children != NULL) lhtml_dump (tree->children, tab_count + 1);

                tree = tree->next;
        }

	return 0;
}

int lhtmlwrap (void) {
        return 1;
}

void lhtml_free_param_list (lhtml_param_list *params) {
        lhtml_param_list *tmp = params, *tmp2;

        while(tmp->next != NULL) {
                free(tmp->name);
                free(tmp->val);
		tmp->name = NULL;
		tmp->val = NULL;

                tmp2 = tmp;
                tmp = tmp->next;
                free(tmp2);
		tmp2 = NULL;
        }
}

void lhtml_free (lhtml_tree *temp) {
	lhtml_tree *tree = temp, *tmp;
	lhtml_tree **parents = NULL;
	int num_parents = 0;

start_again:
	while (tree->next != NULL) {
                free(tree->other_tag);
		tree->other_tag = NULL;

                if (tree->children != NULL) {
			num_parents++;
			parents = realloc(parents, sizeof(lhtml_tree*)*num_parents);
			parents[num_parents-1] = tree;
			tree = tree->children;
			continue;
		}

                if (tree->params != NULL) lhtml_free_param_list(tree->params);
		tree->children = NULL;
		tree->params = NULL;

                tmp = tree;
                tree = tree->next;
                free(tmp);
		tmp = NULL;
        }

	if (num_parents > 0) {
		tree = parents[num_parents-1];
		if (num_parents == 1) {
			free(parents);
			parents = NULL;
			num_parents = 0;
		} else {
			num_parents--;
			parents = realloc(parents, sizeof(lhtml_tree*)*num_parents);
		}

		tree->children = NULL;
		goto start_again;
	}			
}		

%}

%option 8bit
%option align
%option fast

%x COMMENT
%x STARTTAG
%x PARAM
%x PARAM2
%x PARAM3
%x PARAM4
%x DECLTAG
%x DPARAM
%x DPARAM2
%x DPARAM3
%x DPARAM4
%x ENTITY
%x ENDTAG

%%

<INITIAL><<EOF>>        {
        return LHTML_EOF;
}

<INITIAL>\&[^;\r\n\t <]+/[; \r\n\t<]	{
	if (lhtml_entity(&yytext[1]) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
	BEGIN(ENTITY);
}

<ENTITY>[; \r\n\t<]+	{
	char *tmp = lhtml_strdup(yytext), *tmp2;
	int i;

	tmp2 = tmp + strlen(tmp) -1;
	for (i = strlen(tmp) - 1; i >= 0; i--) {
		if (*(tmp+i) != ';') {
			unput(*(tmp+i));
		}
	}

	free(tmp);
	tmp = NULL;

	BEGIN(INITIAL);
}

<INITIAL><\/[^>]+/>    {
        if (lhtml_endtag(&yytext[2]) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
	BEGIN(ENDTAG);
}

<ENDTAG>">"	{
	BEGIN(INITIAL);
}

<INITIAL>"<!--"   {
        BEGIN(COMMENT);
}

<INITIAL>"<!"	{
	BEGIN(DECLTAG);
}

<INITIAL><\?.*\?>	{
	/* ignore */
}

<INITIAL>"<"    {
        BEGIN(STARTTAG);
}

<INITIAL>[\n\t ]+ {
	/* Ignore plain whitespace before text */
}

<INITIAL>[^<]*  {
        if (lhtml_text(yytext) < 0) {
		fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
		yyterminate();
	}
}

<COMMENT>[^>]+       {
	char *x;
	int i = strlen(yytext)-1;

	x = yytext;
	while(*(x+i) == '>' || *(x+i) == '-') {
		*(x+i) = '\0';
		i--;
	}
	i = input();
        if (lhtml_comment(yytext) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
	BEGIN(INITIAL);
}

<STARTTAG>[^ >\/\r\n\t]+  {
        char *tmp2 = lhtml_strdup(yytext), *tmp3;

	if (tmp2 == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        /* allocate a temporary struct */
        temptagstruct = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temptagstruct == NULL) {
                /* terminate if RAM problem */
                yyterminate();
        }

        /* chop off any extra chars from tag */
        tmp3 = tmp2;
        while(*tmp3 != '\0' && *tmp3 != '>' && *tmp3 != '\n' && *tmp3 != '\r'
                && *tmp3 != '\t' && *tmp3 != ' ') {
                tmp3++;
        }
        if (*tmp3 == '>') {
                unput(*tmp3);
        }
        *tmp3 = '\0';

        /* figure out correct constant */
        temptagstruct->tag = lhtml_tagis(tmp2);
        if (temptagstruct->tag == TAG_OTHER) {
                temptagstruct->other_tag = tmp2;
        } else {
                temptagstruct->other_tag = NULL;
                free(tmp2);
		tmp2 = NULL;
        }

        temptagstruct->type = TYPE_HTMLTAG;
        temptagstruct->end = 0;
        temptagstruct->params = NULL;
        temptagstruct->parent = current_level;
        temptagstruct->children = NULL;
        temptagstruct->next = NULL;
}

<STARTTAG>([ \r\n\t]+\/)?>   {
        if (lhtml_add_start_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<STARTTAG>[ \/\r\n\t]+    {
        BEGIN(PARAM);
}

<PARAM>[^ \/\r\n\t=>]+    {
        tempparamname = lhtml_strdup(yytext);
	if (tempparamname == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<PARAM>\/?>      {
        if (lhtml_add_param (temptagstruct, tempparamname, NULL) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        if (lhtml_add_start_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<PARAM>[ \/\r\n\t]+       {
        if (lhtml_add_param (temptagstruct, tempparamname, NULL) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<PARAM>"="      {
        BEGIN(PARAM2);
}

<PARAM2>\"([^"]|\\\")*\" {
        tempparamvalue = lhtml_strdup(yytext+1);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

	tempparamvalue[strlen(tempparamvalue)-1] = '\0';

        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<PARAM3>\" {
	BEGIN(PARAM2);
}

<PARAM2>\'/([^']|\\')* {
	BEGIN(PARAM4);
}

<PARAM4>([^']|\\')*       {
        tempparamvalue = lhtml_strdup(yytext);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<PARAM4>\' {
	BEGIN(PARAM2);
}

<PARAM2>[ \r\n\t]+       {
        BEGIN(PARAM);
}

<PARAM2>\/?>     {
        if (lhtml_add_start_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<PARAM2>[^ \r\n\t\/>]+    {
        tempparamvalue = lhtml_strdup(yytext);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<DECLTAG>[^ >\r\n\t]+  {
        char *tmp2 = lhtml_strdup(yytext), *tmp3;

	if (tmp2 == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        /* allocate a temporary struct */
        temptagstruct = (lhtml_tree*)malloc((size_t)sizeof(lhtml_tree));
        if (temptagstruct == NULL) {
                /* terminate if RAM problem */
                yyterminate();
        }

        /* chop off any extra chars from tag */
        tmp3 = tmp2;
        while(*tmp3 != '\0' && *tmp3 != '>' && *tmp3 != '\n' && *tmp3 != '\r'
                && *tmp3 != '\t' && *tmp3 != ' ') {
                tmp3++;
        }
        if (*tmp3 == '>') {
                unput(*tmp3);
        }
        *tmp3 = '\0';

        temptagstruct->other_tag = tmp2;

        temptagstruct->type = TYPE_DECL;
        temptagstruct->end = 0;
        temptagstruct->params = NULL;
        temptagstruct->parent = current_level;
        temptagstruct->children = NULL;
        temptagstruct->next = NULL;
}

<DECLTAG>">"   {
        if (lhtml_add_decl_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<DECLTAG>[ \r\n\t]+    {
        BEGIN(DPARAM);
}

<DPARAM>\"([^"]|\\\")+\"       {
        tempparamname = lhtml_strdup(&yytext[1]);
	if (tempparamname == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        tempparamname[strlen(tempparamname)-1] = '\0';
}

<DPARAM>[^ \r\n\t=>]+    {
        tempparamname = lhtml_strdup(yytext);
	if (tempparamname == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<DPARAM>">"      {
        if (lhtml_add_param (temptagstruct, tempparamname, NULL) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
        if (lhtml_add_start_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<DPARAM>[ \r\n\t]+       {
        if (lhtml_add_param (temptagstruct, tempparamname, NULL) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<DPARAM>"="      {
        BEGIN(DPARAM2);
}

<DPARAM2>\"/([^"]|\\\")* {
	BEGIN(DPARAM3);
}

<DPARAM3>([^"]|\\\")*       {
        tempparamvalue = lhtml_strdup(yytext);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<DPARAM3>\" {
	BEGIN(DPARAM2);
}

<DPARAM2>\'/([^']|\\\')*       {
	BEGIN(DPARAM4);
}

<DPARAM4>([^']|\\\')* {
        tempparamvalue = lhtml_strdup(yytext);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}

<DPARAM4>\' {
	BEGIN(DPARAM2);
}

<DPARAM2>[ \r\n\t]       {
        BEGIN(DPARAM);
}

<DPARAM2>">"     {
        if (lhtml_add_decl_tag(temptagstruct) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        temptagstruct = NULL;

        BEGIN(INITIAL);
}

<DPARAM2>[^ \r\n\t>]+    {
        tempparamvalue = lhtml_strdup(yytext);
	if (tempparamvalue == NULL) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }

        if (lhtml_add_param (temptagstruct, tempparamname, tempparamvalue) < 0) {
                fprintf(stderr, "libhtmlparse: Eek! Memory error, terminating parse.\n");
                yyterminate();
        }
}
