/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#ifndef CANVAS5_PARSER
#  define CANVAS5_PARSER

#include <functional>
#include <map>
#include <string>
#include <utility>
#include <vector>

#include <librevenge/librevenge.h>

#include "MWAWDebug.hxx"
#include "MWAWInputStream.hxx"
#include "MWAWPosition.hxx"

#include "MWAWParser.hxx"

namespace Canvas5ParserInternal
{
struct Layer;
struct Slide;

struct State;
}

class Canvas5Graph;
class Canvas5StyleManager;

namespace Canvas5Structure
{
struct Stream;
}

/** \brief the main class to read a Canvas 5 files
 *
 */
class Canvas5Parser final : public MWAWGraphicParser
{
  friend class Canvas5Graph;
  friend class Canvas5StyleManager;
public:
  //! constructor
  Canvas5Parser(MWAWInputStreamPtr const &input, MWAWRSRCParserPtr const &rsrcParser, MWAWHeader *header);
  //! destructor
  ~Canvas5Parser() final;

  //! checks if the document header is correct (or not)
  bool checkHeader(MWAWHeader *header, bool strict=false) final;

  // the main parse function
  void parse(librevenge::RVNGDrawingInterface *documentInterface) final;

  //! very low level: debug print some uint32_t either at char4 or sample int
  static std::string getString(unsigned val);

protected:
  //! creates the listener which will be associated to the document
  void createDocument(librevenge::RVNGDrawingInterface *documentInterface);

  //
  // interface
  //

  //! returns true if the file is a windows file
  bool isWindowsFile() const;

protected:
  //! finds the different objects zones
  bool createZones();
  //! try to read the first big block
  bool readMainBlock(std::shared_ptr<Canvas5Structure::Stream> stream);
  //! try to read the third big block: a list of resource?, font, ...
  bool readFileRSRCs(Canvas5Structure::Stream &stream);
  //! try to read the last block: some pathes, ...
  bool readFileDesc(Canvas5Structure::Stream &stream);

  // Intermediate level

  //
  // first block
  //

  //! try to read the file header
  bool readFileHeader(Canvas5Structure::Stream &stream);

  //! read the document settings
  bool readDocumentSettings(Canvas5Structure::Stream &stream);
  //! try to read the different layers
  bool readLayers(Canvas5Structure::Stream &stream);
  //! try to read the preview picture
  bool readPreview(Canvas5Structure::Stream &stream);
  //! try to read the list of slides
  bool readSlides(Canvas5Structure::Stream &stream);
  //! try to read the text links
  bool readTextLinks(Canvas5Structure::Stream &stream);

  //
  // second block
  //

  //! try to read a bitmap
  bool readBitmap(Canvas5Structure::Stream &stream, MWAWEmbeddedObject &object, MWAWColor *avgColor=nullptr);

  //
  // third block
  //

  //! try to read a printer rsrc
  bool readPrinterRsrc(Canvas5Structure::Stream &stream);

  //
  // Mac RSRC
  //
  //! read the RSRC 0 pnot zone
  bool readPnot(Canvas5Structure::Stream &stream, MWAWEntry const &entry);
  //! read the RSRC Pict zone
  bool readPicture(Canvas5Structure::Stream &stream, MWAWEntry const &entry);

  //
  // Windows RSRC
  //

  //
  // send data to the listener
  //

  //! try to send a page/slide
  bool send(Canvas5ParserInternal::Slide const &slide);
  //! try to send a layer
  bool send(Canvas5ParserInternal::Layer const &layer);

  //
  // low level
  //

  //! try to read a pascal string in the data fork or a Pascal/C string depending on the file type
  bool readString(Canvas5Structure::Stream &stream, librevenge::RVNGString &string, int maxSize, bool canBeCString=false);
  //! try to read a double 8
  bool readDouble(Canvas5Structure::Stream &stream, double &val, bool &isNaN) const;
  //! a function used to parse the data of a index map/a extended header
  typedef std::function<void(Canvas5Structure::Stream &, int, std::string const &, long)> DataFunction;
  //! the default function to parse the data of a index map/a extended header
  static void defDataFunction(Canvas5Structure::Stream &, int, std::string const &, long) {}
  //! the default function to parse the data of a index map/a extended header
  static void stringDataFunction(Canvas5Structure::Stream &stream, int id, std::string const &what, long len);

  //! try to read a data header, ie. N fields with a given size
  bool readDataHeader(Canvas5Structure::Stream &stream, int expectedSize, int &N);
  /** try to read an extended data header, ie. N0 is expected to be value

      \note the function func is called on each entry excepted the first one
   */
  bool readExtendedHeader(Canvas5Structure::Stream &stream, int expectedValue, std::string const &what, DataFunction const &func);
  /** try to read an unknown structure which defined some position and two entries */
  bool readUnknownHeader(Canvas5Structure::Stream &stream, MWAWEntry(&entries)[2], long len, std::string const &what);

  //! try to read the used list
  bool readUsed(Canvas5Structure::Stream &stream, std::string const &what);
  //! try to read the defined list
  bool readDefined(Canvas5Structure::Stream &stream, std::vector<bool> &defined, std::string const &what);

  //! try to read a index map
  bool readIndexMap(Canvas5Structure::Stream &stream, std::string const &what, DataFunction const &func=&Canvas5Parser::defDataFunction);

  //! try to read a special shape data (internal helper to understand vkfl structure)
  bool readSpecialData(long len, unsigned type, std::string &extra);

  //! try to decode the input stream
  static MWAWInputStreamPtr decode(MWAWInputStreamPtr input);

  //
  // data
  //
  //! the state
  std::shared_ptr<Canvas5ParserInternal::State> m_state;
  //! the graph parser
  std::shared_ptr<Canvas5Graph> m_graphParser;
  //! the style manager
  std::shared_ptr<Canvas5StyleManager> m_styleManager;
};
#endif
// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
