/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_802_11.h>
#include <protocols/snap/libnd_snap.h>


static LND_Protocol *ieee_802_11;


static gboolean
ieee_802_11_header_complete(const LND_Packet *packet, guchar *data)
{
  if (!data)
    return FALSE;

  return (data + LND_802_11_LEN <= libnd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("IEEE 802.11 Plugin");
}


const char *
description(void)
{
  return ("A plugin providing support for IEEE 802.11.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  ieee_802_11 = libnd_proto_new("801.11", LND_PROTO_LAYER_LINK, DLT_IEEE802_11);
			     		
  ieee_802_11->init_packet     = libnd_802_11_init_packet;
  ieee_802_11->header_complete = libnd_802_11_header_complete;
	     
  return ieee_802_11;
}


/* Protocol method implementations: ------------------------------------ */

guchar *
libnd_802_11_init_packet(LND_Packet *packet, guchar *data, guchar *data_end)
{
  LND_Protocol *payload_proto;
  struct lnd_802_11_header *hdr;

  D_ENTER;

  if (!ieee_802_11_header_complete(packet, data))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      D_RETURN_(data_end);
    }

  libnd_packet_add_proto_data(packet, ieee_802_11, data, data_end);
  hdr = (struct lnd_802_11_header *) data;

  /* Demux to higher layers if this is a data frame containing
   * data. Everything else is ignored right now.
   */
  if (hdr->ctrl.type == 2 && hdr->ctrl.subtype < 3)
    {
      payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, LND_PROTO_SNAP);

      if (! payload_proto)
	payload_proto = libnd_raw_proto_get();

      payload_proto->init_packet(packet, data + LND_802_11_LEN, data_end);
    }
  else
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
    }

  D_RETURN_(data_end);
}


gboolean
libnd_802_11_header_complete(const LND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = libnd_packet_get_data(packet, ieee_802_11, nesting);

  return ieee_802_11_header_complete(packet, data);
  TOUCH(nesting);
}


LND_Protocol *
libnd_802_11_get(void)
{
  return ieee_802_11;
}

