/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif

#include <pcapnav.h>

#include <libnd_debug.h>
#include <libnd_macros.h>
#include <libnd_globals.h>
#include <libnd_misc.h>
#include <libnd_packet.h>
#include <libnd_dumper.h>

struct lnd_dumper {
  pcap_t         *pcap;

  LND_DumperArgs  args;
  char           *base_name;
  char           *curr_name;
  pcap_dumper_t  *dumper;

  guint           file_num;
  guint64         total_size;
  guint64         curr_size;
};


static const char*
dumper_get_nth_name(LND_Dumper *dumper, int file_num)
{
  static char buf[MAXPATHLEN];
  
  if (file_num == 0)
    return dumper->base_name;
  
  g_snprintf(buf, MAXPATHLEN, "%s.%05i", dumper->base_name, file_num);
  return buf;
}

static int
dumper_find_curr(LND_Dumper *dumper, guint64 *size)
{
  int file_num = 0;
  const char *name = dumper_get_nth_name(dumper, file_num);

  *size = 0;
  
  while (libnd_misc_exists(name))
    {
      *size += libnd_misc_get_size(name);
      name = dumper_get_nth_name(dumper, ++file_num);
    }
  
  return file_num > 0 ? file_num - 1 : 0;
}


void
libnd_dumper_args_init(LND_DumperArgs *args)
{
  if (! args)
    return;

  memset(args, 0, sizeof(LND_DumperArgs));
}


LND_Dumper *
libnd_dumper_new(pcap_t *pcap, const char *dumper_name,
		 LND_DumperArgs *args)
{
  LND_Dumper *dumper;

  if (! pcap || ! dumper_name)
    return NULL;

  dumper = calloc(1, sizeof(LND_Dumper));
  dumper->pcap = pcap;
  dumper->args.open_mode = PCAPNAV_DUMP_TRUNC;
  dumper->args.file_limit = LND_DUMPER_1GB;
  dumper->base_name = g_strdup(dumper_name);
  
  if (args)
    {
      /* Selectively overwrite defaults with given values. */
      
      if (args->open_mode != PCAPNAV_DUMP_TRUNC)
	dumper->args.open_mode = args->open_mode;
      if (args->file_limit > 0)
	dumper->args.file_limit = args->file_limit;
      if (args->hard_limit > 0)
	dumper->args.hard_limit = args->hard_limit;
    }
  
  if (dumper->args.open_mode == PCAPNAV_DUMP_TRUNC)
    {
      libnd_dumper_delete(dumper);
    }
  else
    {
      dumper->file_num = dumper_find_curr(dumper, &dumper->total_size);
      dumper->curr_size = libnd_misc_get_size(dumper_get_nth_name(dumper, dumper->file_num));
    }
  
  dumper->curr_name = g_strdup(dumper_get_nth_name(dumper, dumper->file_num));

  dumper->dumper = pcapnav_dump_open(pcap, dumper->curr_name,
				     dumper->args.open_mode);
  
  if (! dumper->dumper)
    {
      D(("Cannot open dump '%s': %s\n", dumper->curr_name, pcap_geterr(pcap)));
      libnd_dumper_free(dumper);
      return NULL;	
    }
  
  return dumper;
}


void        
libnd_dumper_free(LND_Dumper *dumper)
{
  if (! dumper)
    return;

  if (dumper->dumper)
    pcap_dump_close(dumper->dumper);
  
  g_free(dumper->base_name);
  g_free(dumper->curr_name);
  g_free(dumper);
}


gboolean    
libnd_dumper_write(LND_Dumper *dumper, LND_Packet *packet)
{
  if (! dumper || ! dumper->dumper || ! packet)
    return FALSE;

  /* Need to check whether we are hitting the hard dumper
   * size limit, and also the per-file size limit. In the
   * former case we do nothing, in the latter we start 
   * a new file.
   */

  if (dumper->args.hard_limit > 0 &&
      dumper->total_size + packet->ph.caplen + sizeof(packet->ph) > dumper->args.hard_limit)
    {
      D(("Hard size limit reached on dumper '%s'\n", dumper->base_name));
      return FALSE;
    }
  
  if (dumper->args.file_limit > 0 &&
      dumper->curr_size + packet->ph.caplen + sizeof(packet->ph) > dumper->args.file_limit)
    {
      D(("File size limit reached on dumper '%s'\n", dumper->base_name));
      pcap_dump_close(dumper->dumper);
      dumper->file_num++;

      g_free(dumper->curr_name);
      dumper->curr_name = g_strdup(dumper_get_nth_name(dumper, dumper->file_num));
      
      dumper->dumper = pcapnav_dump_open(dumper->pcap, dumper->curr_name,
					 dumper->args.open_mode);
    
      dumper->curr_size = 0;
      
      if (! dumper->dumper)
	{
	  D(("Cannot open dump '%s': %s\n", dumper->curr_name, pcap_geterr(dumper->pcap)));
	  return FALSE;
	}
    }
  
  
  libnd_packet_dump(packet, dumper->dumper);
  
  dumper->total_size += packet->ph.caplen + sizeof(packet->ph);
  dumper->curr_size += packet->ph.caplen + sizeof(packet->ph);
  
  return TRUE;
}


void
libnd_dumper_delete(LND_Dumper *dumper)
{
  int file_num = 0;
  const char *name;
  
  if (! dumper)
    return;
  
  name = dumper_get_nth_name(dumper, file_num);
  
  while (libnd_misc_exists(name))
    {
      unlink(name);
      file_num++;
      name = dumper_get_nth_name(dumper, file_num);
    }
  
  dumper->file_num = 0;
  dumper->curr_size = 0;
  dumper->total_size = 0;
}


guint64        
libnd_dumper_get_size(LND_Dumper *dumper)
{
  if (! dumper)
    return 0;
  
  return dumper->total_size;    
}
