/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_filter_factory_h
#define __libnd_filter_factory_h

#include <libnd_types.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef void (*LND_FilterCreateFunc)(void);
typedef void (*LND_FilterModifyFunc)(LND_Filter *filter);

struct lnd_filter_factory
{
  const char            *name;

  LND_FilterCreateFunc   create_func;
  LND_FilterModifyFunc   modify_func;
};


/**
 * LND_FilterFactoryFunc - signature of callbacks for factory iteration.
 * @factory: currently iterated factory.
 * @user_data: arbitrary data.
 *
 * This is the signature of callbacks passed to libnd_filter_factory_foreach().
 */
typedef void (*LND_FilterFactoryFunc)(LND_FilterFactory *factory, void *user_data);


/**
 * libnd_filter_factory_new - creates a new filter factory.
 * @name: name of the new factory.
 * @create_func: implementation of the filter creation function.
 * @modify_func: implementation of the filter modification function.
 *
 * Returns: new filter factory.
 */
LND_FilterFactory *libnd_filter_factory_new(const char *name,
					    LND_FilterCreateFunc create_func,
					    LND_FilterModifyFunc modify_func);

/**
 * libnd_filter_factory_free - releases a factory.
 * @factory: factory to free.
 *
 * The factory releases the memory occupied by a filter factory.
 */
void               libnd_filter_factory_free(LND_FilterFactory *factory);

/**
 * libnd_filter_factory_register - registers a new filter factory.
 * @factory: filter factory to register.
 *
 * This function adds @factory to the registry of known factories.
 *
 * Returns: %TRUE if the factory was registered, %FALSE if the
 * factory was already known or in case of an error.
 */
gboolean           libnd_filter_factory_register(LND_FilterFactory *factory);

/**
 * libnd_filter_factory_foreach - filter factory iteration.
 * @callback: callback function to call for each registered factory.
 * @user_data: arbitrary user data passed to @callback.
 *
 * The function iterates over all registered filter factories,
 * calling @callback for each of them.
 */
void               libnd_filter_factory_foreach(LND_FilterFactoryFunc callback, void *user_data);

/**
 * libnd_filter_factory_find - factory lookup.
 * @name: name of the factory to find.
 *
 * Returns: filter factory with matching name, %NULL otherwise.
 */
LND_FilterFactory *libnd_filter_factory_find(const char *name);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
