/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_tcpdump_h
#define __libnd_tcpdump_h

#include <libnd_types.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* We keep this global to speed things up -- would be too expensive to
 * query the config system every time it's needed in libnd_tcpdump_get_packet_line().
 */
extern int libnd_tcpdump_print_timestamp;


/**
 * libnd_tcpdump_init - initializes the tcpdump connection system.
 *
 * This function initializes the communication system with
 * tcpdump processes. Calling this is the job of the libnetdude
 * bootstrapper, you do not need this function.
 *
 * Returns: %TRUE if initialization was successful, %FALSE otherwise.
 */
gboolean    libnd_tcpdump_init(void);

/**
 * libnd_tcpdump_open - forks new tcpdump process for a trace.
 * @trace: trace to create tcpdump process for.
 *
 * The function forks a new tcpdump process and send the pcap file header,
 * if available. The tcpdump options you desire can be set using
 * libnd_tcpdump_options_reset() and libnd_tcpdump_options_add()
 * before calling this function. The -l and -r options are always
 * passed; -nnn, -N, -q, and -e are set depending on the values of 
 * the tcpdump_resolve, tcpdump_domains, tcpdump_quick, and
 * tcpdump_print_link preferences settings in the %LND_DOM_NETDUDE
 * domain. You can call this function repeatedly
 * to kill an existing tcpdump connection and create a new one.
 *
 * Returns: %TRUE if operation was successful, %FALSE otherwise.
 */
gboolean    libnd_tcpdump_open(LND_Trace *trace);


/**
 * libnd_tcpdump_send_header - sends a pcap file header to the connected tcpdump process.
 * @trace: trace whose header to send to the connected tcpdump process.
 *
 * If the trace's file header was not available at the time you opened
 * the connection to tcpdump (using libnd_tcpdump_open()), you can
 * use this function to now send it. NOTE: tcpdump expects the
 * trace file header to be the first input data it receives,
 * so you cannot use this function once you've sent packet data,
 * and you cannot send packet data unless you've previously
 * sent a trace file header.
 */
void        libnd_tcpdump_send_header(LND_Trace *trace);

/**
 * libnd_tcpdump_close - kills a tcpdump connection
 * @trace: trace whose tcpdump connection to close.
 *
 * The function kills @trace's existing tcpdump connection, if any.
 */
void        libnd_tcpdump_close(LND_Trace *trace);

/**
 * libnd_tcpdump_get_packet_line - returns a tcpdump line for a packet.
 * @packet: packet to retrieve tcpdump output of.
 * @output: string buffer that receives the tcpdump output.
 * @output_len: length of @output.
 * @find_context: whether to look up context or not.
 *
 * The function sends @packet's raw data to the connected tcpdump
 * process and writes at most @output_len bytes of the returned output
 * into @output.
 * libnetdude can maintain state for TCP connections independently
 * of tcpdump, so you can request contextual output (relative SYN
 * and ACK numbers) using the @find_context flag).
 *
 * Returns: %TRUE on success, %FALSE otherwise.
 */
gboolean    libnd_tcpdump_get_packet_line(const LND_Packet *packet,
					  char *output, int output_len,
					  gboolean find_context);

/**
 * libnd_tcpdump_options_reset - resets tcpdump options to default.
 * 
 * The function resets any tcpdump options set using
 * libnd_tcpdump_options_add().
 */
void        libnd_tcpdump_options_reset(void);

/**
 * libnd_tcpdump_options_add - adds an arbitrary tcpdump command line option.
 * @option: new command line option.
 * 
 * The function registers @option as an additional command line
 * option to pass to future forks of tcpdump processes. The
 * flags you have requested can be reset to the default (see
 * libnd_tcpdump_open()) using libnd_tcpdump_options_reset().
 */
void        libnd_tcpdump_options_add(const char *option);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
