//===-- Utilities for trigonometric functions with FMA ----------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_FMA_H
#define LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_FMA_H

#include "src/__support/FPUtil/FMA.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/nearest_integer.h"

namespace __llvm_libc {

namespace fma {

static constexpr uint32_t FAST_PASS_BOUND = 0x5680'0000U; // 2^46

// Digits of 1/pi, generated by Sollya with:
// > a0 = D(16/pi);
// > a1 = D(16/pi - a0);
// > a2 = D(16/pi - a0 - a1);
// > a3 = D(16/pi - a0 - a1 - a2);
static constexpr double SIXTEEN_OVER_PI[5] = {
    0x1.45f306dc9c883p+2, -0x1.6b01ec5417056p-52, -0x1.6447e493ad4cep-106,
    0x1.e21c820ff28b2p-160, -0x1.508510ea79237p-215};

// Return k and y, where
//   k = round(x * 16 / pi) and y = (x * 16 / pi) - k.
// Assume x is non-negative.
static inline int64_t small_range_reduction(double x, double &y) {
  double kd = fputil::nearest_integer(x * SIXTEEN_OVER_PI[0]);
  y = fputil::fma(x, SIXTEEN_OVER_PI[0], -kd);
  y = fputil::fma(x, SIXTEEN_OVER_PI[1], y);
  return static_cast<int64_t>(kd);
}

// Return k and y, where
//   k = round(x * 16 / pi) and y = (x * 16 / pi) - k.
static inline int64_t large_range_reduction(double x, int x_exp, double &y) {
  // 2^46 <= |x| < 2^99
  if (x_exp < 99) {
    // - When x < 2^99, the full exact product of x * SIXTEEN_OVER_PI[0]
    // contains at least one integral bit <= 2^4.
    // - When 2^46 <= |x| < 2^56, the lowest 5 unit bits are contained
    // in the last 10 bits of double(x * SIXTEEN_OVER_PI[0]).
    // - When |x| >= 2^56, the LSB of double(x * SIXTEEN_OVER_PI[0]) is at least
    // 32.
    fputil::FPBits<double> prod_hi(x * SIXTEEN_OVER_PI[0]);
    prod_hi.bits &= (x_exp < 56) ? (~0xfffULL) : (~0ULL); // |x| < 2^56
    double k_hi = fputil::nearest_integer(static_cast<double>(prod_hi));
    double truncated_prod = fputil::fma(x, SIXTEEN_OVER_PI[0], -k_hi);
    double prod_lo = fputil::fma(x, SIXTEEN_OVER_PI[1], truncated_prod);
    double k_lo = fputil::nearest_integer(prod_lo);
    y = fputil::fma(x, SIXTEEN_OVER_PI[1], truncated_prod - k_lo);
    y = fputil::fma(x, SIXTEEN_OVER_PI[2], y);
    y = fputil::fma(x, SIXTEEN_OVER_PI[3], y);

    return static_cast<int64_t>(k_lo);
  }

  // - When x >= 2^110, the full exact product of x * SIXTEEN_OVER_PI[0] does
  // not contain any of the lowest 5 unit bits, so we can ignore it completely.
  // - When 2^99 <= |x| < 2^110, the lowest 5 unit bits are contained
  // in the last 12 bits of double(x * SIXTEEN_OVER_PI[1]).
  // - When |x| >= 2^110, the LSB of double(x * SIXTEEN_OVER_PI[1]) is at
  // least 32.
  fputil::FPBits<double> prod_hi(x * SIXTEEN_OVER_PI[1]);
  prod_hi.bits &= (x_exp < 110) ? (~0xfffULL) : (~0ULL); // |x| < 2^110
  double k_hi = fputil::nearest_integer(static_cast<double>(prod_hi));
  double truncated_prod = fputil::fma(x, SIXTEEN_OVER_PI[1], -k_hi);
  double prod_lo = fputil::fma(x, SIXTEEN_OVER_PI[2], truncated_prod);
  double k_lo = fputil::nearest_integer(prod_lo);
  y = fputil::fma(x, SIXTEEN_OVER_PI[2], truncated_prod - k_lo);
  y = fputil::fma(x, SIXTEEN_OVER_PI[3], y);
  y = fputil::fma(x, SIXTEEN_OVER_PI[4], y);

  return static_cast<int64_t>(k_lo);
}

// Exceptional cases.
static constexpr int N_EXCEPTS = 2;

static constexpr fputil::ExceptionalValues<float, N_EXCEPTS> SinfExcepts{
    /* inputs */ {
        0x3fa7832a, // x = 0x1.4f0654p0
        0x55cafb2a, // x = 0x1.95f654p44
    },
    /* outputs (RZ, RU offset, RD offset, RN offset) */
    {
        {0x3f7741b5, 1, 0, 1}, // x = 0x1.4f0654p0, sin(x) = 0x1.ee836ap-1 (RZ)
        {0xbf7e7a16, 0, 1,
         1}, // x = 0x1.95f654p44, sin(x) = -0x1.fcf42cp-1 (RZ)
    }};

} // namespace fma

} // namespace __llvm_libc

#endif // LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_FMA_H
