
//===-- Unittests for user defined integer literals -----------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "src/__support/integer_literals.h"
#include "test/UnitTest/Test.h"

using LIBC_NAMESPACE::operator""_u8;
using LIBC_NAMESPACE::operator""_u16;
using LIBC_NAMESPACE::operator""_u32;
using LIBC_NAMESPACE::operator""_u64;
using LIBC_NAMESPACE::operator""_u128;
using LIBC_NAMESPACE::operator""_u256;

TEST(LlvmLibcIntegerLiteralTest, u8) {
  EXPECT_EQ(uint8_t(0), 0_u8);
  EXPECT_EQ(uint8_t(UINT8_MAX), 255_u8);
  EXPECT_EQ(uint8_t(UINT8_MAX), 0xFF_u8);
  EXPECT_EQ(uint8_t(UINT8_MAX), 0b11111111_u8);
}

TEST(LlvmLibcIntegerLiteralTest, u16) {
  EXPECT_EQ(uint16_t(0), 0_u16);
  EXPECT_EQ(uint16_t(UINT8_MAX), 255_u16);
  EXPECT_EQ(uint16_t(UINT8_MAX), 0xFF_u16);
  EXPECT_EQ(uint16_t(UINT8_MAX), 0b11111111_u16);
  EXPECT_EQ(uint16_t(UINT16_MAX), 65535_u16);
  EXPECT_EQ(uint16_t(UINT16_MAX), 0xFFFF_u16);
  EXPECT_EQ(uint16_t(UINT16_MAX), 0b11111111'11111111_u16);
}

TEST(LlvmLibcIntegerLiteralTest, u32) {
  EXPECT_EQ(uint32_t(0), 0_u32);
  EXPECT_EQ(uint32_t(UINT8_MAX), 255_u32);
  EXPECT_EQ(uint32_t(UINT8_MAX), 0xFF_u32);
  EXPECT_EQ(uint32_t(UINT8_MAX), 0b11111111_u32);
  EXPECT_EQ(uint32_t(UINT16_MAX), 65535_u32);
  EXPECT_EQ(uint32_t(UINT16_MAX), 0xFFFF_u32);
  EXPECT_EQ(uint32_t(UINT16_MAX), 0b11111111'11111111_u32);
  EXPECT_EQ(uint32_t(UINT32_MAX), 4294967295_u32);
  EXPECT_EQ(uint32_t(UINT32_MAX), 0xFFFFFFFF_u32);
  EXPECT_EQ(uint32_t(UINT32_MAX), 0b1111111111111111'1111111111111111_u32);
}

TEST(LlvmLibcIntegerLiteralTest, u64) {
  EXPECT_EQ(uint64_t(0), 0_u64);
  EXPECT_EQ(uint64_t(UINT8_MAX), 255_u64);
  EXPECT_EQ(uint64_t(UINT8_MAX), 0xFF_u64);
  EXPECT_EQ(uint64_t(UINT8_MAX), 0b11111111_u64);
  EXPECT_EQ(uint64_t(UINT16_MAX), 65535_u64);
  EXPECT_EQ(uint64_t(UINT16_MAX), 0xFFFF_u64);
  EXPECT_EQ(uint64_t(UINT16_MAX), 0b11111111'11111111_u64);
  EXPECT_EQ(uint64_t(UINT32_MAX), 4294967295_u64);
  EXPECT_EQ(uint64_t(UINT32_MAX), 0xFFFFFFFF_u64);
  EXPECT_EQ(uint64_t(UINT32_MAX), 0b1111111111111111'1111111111111111_u64);
  EXPECT_EQ(uint64_t(UINT64_MAX), 18446744073709551615_u64);
  EXPECT_EQ(uint64_t(UINT64_MAX), 0xFFFFFFFF'FFFFFFFF_u64);
  EXPECT_EQ(
      uint64_t(UINT64_MAX),
      0b1111111111111111'1111111111111111'1111111111111111'1111111111111111_u64);
}

TEST(LlvmLibcIntegerLiteralTest, u128) {
#if defined(__SIZEOF_INT128__)
  const __uint128_t ZERO = 0;
  const __uint128_t U8_MAX = UINT8_MAX;
  const __uint128_t U16_MAX = UINT16_MAX;
  const __uint128_t U32_MAX = UINT32_MAX;
  const __uint128_t U64_MAX = UINT64_MAX;
  const __uint128_t U128_MAX = (U64_MAX << 64) | U64_MAX;
#else
  const UInt128 ZERO = 0;
  const UInt128 U8_MAX = UINT8_MAX;
  const UInt128 U16_MAX = UINT16_MAX;
  const UInt128 U32_MAX = UINT32_MAX;
  const UInt128 U64_MAX = UINT64_MAX;
  const UInt128 U128_MAX = (U64_MAX << 64) | U64_MAX;
#endif
  EXPECT_EQ(ZERO, 0_u128);
  EXPECT_EQ(U8_MAX, 255_u128);
  EXPECT_EQ(U8_MAX, 0xFF_u128);
  EXPECT_EQ(U8_MAX, 0b11111111_u128);
  EXPECT_EQ(U16_MAX, 65535_u128);
  EXPECT_EQ(U16_MAX, 0xFFFF_u128);
  EXPECT_EQ(U16_MAX, 0b11111111'11111111_u128);
  EXPECT_EQ(U32_MAX, 4294967295_u128);
  EXPECT_EQ(U32_MAX, 0xFFFFFFFF_u128);
  EXPECT_EQ(U32_MAX, 0b1111111111111111'1111111111111111_u128);
  EXPECT_EQ(U64_MAX, 18446744073709551615_u128);
  EXPECT_EQ(U64_MAX, 0xFFFFFFFF'FFFFFFFF_u128);
  EXPECT_EQ(
      U64_MAX,
      0b1111111111111111'1111111111111111'1111111111111111'1111111111111111_u128);
  EXPECT_EQ(U128_MAX, 340282366920938463463374607431768211455_u128);
  EXPECT_EQ(U128_MAX, 0xFFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF_u128);
  EXPECT_EQ(
      U128_MAX,
      0b1111111111111111'1111111111111111'1111111111111111'1111111111111111'1111111111111111'1111111111111111'1111111111111111'1111111111111111_u128);
}

TEST(LlvmLibcIntegerLiteralTest, u256) {
  using UInt256 = LIBC_NAMESPACE::UInt<256>;
  const UInt256 ZERO = 0;
  const UInt256 U8_MAX = UINT8_MAX;
  const UInt256 U16_MAX = UINT16_MAX;
  const UInt256 U32_MAX = UINT32_MAX;
  const UInt256 U64_MAX = UINT64_MAX;
  const UInt256 U128_MAX = (U64_MAX << 64) | U64_MAX;
  const UInt256 U256_MAX = (U128_MAX << 128) | U128_MAX;
  EXPECT_EQ(ZERO, 0_u256);
  EXPECT_EQ(U8_MAX, 255_u256);
  EXPECT_EQ(U8_MAX, 0xFF_u256);
  EXPECT_EQ(U8_MAX, 0b11111111_u256);
  EXPECT_EQ(U16_MAX, 65535_u256);
  EXPECT_EQ(U16_MAX, 0xFFFF_u256);
  EXPECT_EQ(U16_MAX, 0b11111111'11111111_u256);
  EXPECT_EQ(U32_MAX, 4294967295_u256);
  EXPECT_EQ(U32_MAX, 0xFFFFFFFF_u256);
  EXPECT_EQ(U32_MAX, 0b1111111111111111'1111111111111111_u256);
  EXPECT_EQ(U64_MAX, 18446744073709551615_u256);
  EXPECT_EQ(U64_MAX, 0xFFFFFFFF'FFFFFFFF_u256);
  EXPECT_EQ(
      U64_MAX,
      0b1111111111111111'1111111111111111'1111111111111111'1111111111111111_u256);
  EXPECT_EQ(U128_MAX, 0xFFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF_u256);
  EXPECT_EQ(
      U256_MAX,
      0xFFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF'FFFFFFFF_u256);
}
