#include "mdtree.h"

int main(int argc, char *argv[])
{
    string dir;
    vector<string> ignoredFolders;

    for (int i=0; i < argc; i++)
    {
        string arg = string(argv[i]);
        if (MDTree::CheckArgument(arg, "-h", "--help"))
        {
            MDTree::PrintHelp();
            return 0;
        }

        if (MDTree::CheckArgument(arg, "-d", "--dir"))
        {
            dir = string(argv[++i]);
            continue;
        }

        if (MDTree::CheckArgument(arg, "-i", "--ignore"))
        {
            for (int x=i+1; x < argc; x++)
            {
                string ignoredDir(argv[x]);

                if (ignoredDir.at(0) != '-')
                {
                    ignoredFolders.push_back(ignoredDir);
                }
                else
                {
                    i = x-1;
                    break;
                }
            }
            continue;
        }
    }

    MDTree(dir, ignoredFolders);
    return 0;
}

MDTree::MDTree(string startDir, const vector<string>& ignoredDirectories)
{
    vector<tuple<string,int>> filepaths;
    filepaths = TraverseFolders(startDir, ignoredDirectories);
    PrintMarkdown(filepaths);
}

vector<tuple<string,int>> MDTree::TraverseFolders(const string dir, const vector<string>& ignoredDirectories)
{
    vector<tuple<string,int>> filepaths;

    for(auto entry = std::filesystem::recursive_directory_iterator(dir);
        entry != std::filesystem::recursive_directory_iterator();
        ++entry)
    {
        bool ignored = false;
        string entryPath = entry->path().string();
        if (!entry->is_symlink())
        {
            for (const string& name : ignoredDirectories)
            {
                if (entryPath.find(name) != string::npos)
                {
                    ignored = true;
                    break;
                }
            }
            if (!ignored)
            {
                tuple<string,int> fileInfo(entry->path().filename(), entry.depth());
                filepaths.push_back(fileInfo);
            }

        }
    }

    return filepaths;
}

void MDTree::PrintMarkdown(const vector<tuple<string,int>>& filePaths)
{
    for (auto filepath : filePaths)
    {
        string path = std::get<0>(filepath);
        int depth = std::get<1>(filepath);

        string line = string(depth,'\t') + "- " + path;

        std::cout << line << std::endl;
    }
}


bool MDTree::CheckArgument(const string& argv, const string& arg1, const string& arg2)
{
    if (argv == arg1 || argv == arg2)
    {
        return true;
    }
    return false;
}

void MDTree::PrintHelp()
{
    std::cout << "mdtree, a simple cli application to print filetree in markdown friendly format." << std::endl;
    std::cout << "The filetree is printed in nested bullet point list." << std::endl;
    std::cout << "For example:" << std::endl;
    std::cout << "mdtree -d . -i dir3 dir4" << std::endl;
    std::cout << "- dir0" << std::endl;
    std::cout << "\t- dir1" << std::endl;
    std::cout << "\t\t- dir1dir0" << std::endl;
    std::cout << "\t- dir2" << std::endl;
    std::cout << "\n----Commands----" << std::endl;
    std::cout << "-d, --dir : Where to start to build the tree from. If left empty, using the current folder."
              << std::endl;
    std::cout << "-i, --ignore : Ignore directories with these names. Optional." << std::endl;
    std::cout << "-h, --help : Show this help info." << std::endl;
}

