/* -*- mode: c; c-basic-offset: 4; -*- */
/*******************************************************
  Matlab mex wrapper for NDS.
******************************************************/

#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>

#include "daqc.h"
#include "nds_os.h"
#include "nds_log_matlab.h"
#include "nds_logging.h"
#include "nds_mex_utils.h"

mxArray*
mlwrapper_chanlist_to_mxarray(int *err,daq_channel_t* channels,int num_channels);
int
mlwrapper_get_channel_list(mxArray **, char*, chantype_t, time_t);

/*   NDS_GetChannels function */
/*   ------------------------ */
/*   This function returns a list of available channels */
/*   The syntax is:                                     */
/*   NDS_GetChannels("server:port");                    */
void
mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) {
    char hpbuf[MAX_HOST_LENGTH];
    int err;
    chantype_t ctype = cUnknown;
    time_t gps = 0;

    /*** Input Validation ***/
    if(nlhs != 1) {
	mexErrMsgTxt("This function only returns one paramter.");
    }

    if(nrhs < 1 || nrhs > 3) {
	mexErrMsgTxt("This function takes 1-3 arguments.");
    }

    /*--- Get the channel type if a second argument was specified */
    if (nrhs > 1) {
        if (mxGetString(prhs[1], hpbuf, sizeof(hpbuf)))
            mexErrMsgTxt("Channel type should be a string"
                         " and shorter than " STRINGIFY(MAX_HOST_LENGTH) " characters.");

	    ctype = cvt_str_chantype(hpbuf);
    }

    /*--- Get the host name                                      */
    /* to make input string an actual C string */
    if (mxGetString(prhs[0], hpbuf, sizeof(hpbuf)))
        mexErrMsgTxt("Hostname should be of the form \"host\" or \"host:port\""
                     " and shorter than " STRINGIFY(MAX_HOST_LENGTH) " characters.");

    /*--- Get the gps time if it was specified                  */
    if (nrhs > 2) {
	if((mxGetM(prhs[2]) * mxGetN(prhs[2])) != 1) {
	    mexErrMsgTxt("Start time must be a scalar.");
	}
	gps = mxGetScalar(prhs[2]);
    }

    err = mlwrapper_get_channel_list(plhs, hpbuf, ctype, gps);

    if(err) {
	mexErrMsgTxt("Failed to get channel list.");
    }
}

int
mlwrapper_get_channel_list(mxArray **plhs, char* host,
			   chantype_t chant, time_t gps) {
    /* Local variable declaration */
    short port = 31200;
    daq_t daq;
    int err;
    int num_alloc    = 0;
    daq_channel_t* channels = NULL;
    int num_channels = 0;

    /* Initialization processing */
    daq_startup();

    /* Get the host ip and port number */
    parse_host_name(host, &port);

    /* Open up a client connection */
    err = daq_connect(&daq, host, port, nds_try);
    /* printf("daq_connect: return code %i\n", err); */
    if (err) {
	mexWarnMsgTxt("Failed to connect to nds2.");
	return 13;
    }

    /*---  Get the number of channels */
    err = daq_recv_channel_list(&daq, channels, 0, &num_alloc, gps, chant);
    if (err) {
	daq_recv_shutdown(&daq);
	mexWarnMsgTxt("get_channel_list() failed.");
	return 54;
    }

    /*---  Read in the channel list */
    if (num_alloc > 0) {
	channels = mxCalloc(num_alloc, sizeof(daq_channel_t));
	if (!channels) {
	    mexWarnMsgTxt("Channel list mxCalloc() failed.");
            return 54;
	}
	err = daq_recv_channel_list(&daq, channels, num_alloc, 
				    &num_channels, gps, chant);
	if (err) {
	    mexWarnMsgTxt("get_channel_list() failed.");
	    return 54;
	} else if (!num_channels) {
	    mxFree(channels);
	    channels = NULL;
	}
    }

    /*---  Close the connection */
    daq_disconnect(&daq);
    daq_recv_shutdown(&daq);

    /*---  Convert the channel list to a cell-array */
    plhs[0] = mlwrapper_chanlist_to_mxarray(&err, channels, num_channels);
    if (err) {
	mexWarnMsgTxt("mlwrapper_chanlist_to_mxarray() failed.");
	if (channels != NULL) {
	    mxFree(channels);
	}
	return 83;
    }

    if (channels != NULL) {
	mxFree(channels);
    }

    return 0;
}

mxArray*
mlwrapper_chanlist_to_mxarray(int *err,daq_channel_t* channels,int num_channels)
{
    /* Retrieves the channel list from the NDS server,
       and converts it into a mxArray of Matlab structs */
    mxArray *ml_list;
    mwSize dims[1];
    char* fields[] = {"name", "chan_type", "rate", "data_type", "signal_gain", 
		      "signal_offset", "signal_slope", "signal_units"};

    dims[0] = (mwSize) num_channels;

    ml_list = mxCreateStructArray(1,dims,8,(const char**)fields);
    if(ml_list == NULL) {
	*err = 2;
	return NULL;
    }

    /*---  Bail out if no channels are available. */
    if (num_channels == 0) {
	return ml_list;
    }

    {
	mwSize i;

	int field_name = mxGetFieldNumber(ml_list,"name");
	int field_chan_type = mxGetFieldNumber(ml_list,"chan_type");
	int field_rate = mxGetFieldNumber(ml_list,"rate");
	int field_data_type = mxGetFieldNumber(ml_list,"data_type");
	int field_signal_gain = mxGetFieldNumber(ml_list,"signal_gain");
	int field_signal_offset = mxGetFieldNumber(ml_list,"signal_offset");
	int field_signal_slope = mxGetFieldNumber(ml_list,"signal_slope");
	int field_signal_units = mxGetFieldNumber(ml_list,"signal_units");

	*err = 0;
	for(i=0; i < num_channels; i++) {
	    /* Channel Name */
	    if (put_mxarray_str(ml_list, field_name, i, channels[i].name)) {
		*err = 3;
		break;
	    }
	
	    /* Channel type (formerly group number) */
	    if (put_mxarray_str(ml_list, field_chan_type, i, 
				cvt_chantype_str(channels[i].type))) {
		*err = 3;
		break;
	    }

	    /* Sample Rate */
	    if (put_mxarray_double(ml_list, field_rate, i, channels[i].rate)){
		*err = 3;
		break;
	    }

	    /* Data Type */
	    if(put_mxarray_str(ml_list, field_data_type, i, 
			       data_type_name(channels[i].data_type))) {
		*err = 3;
		break;
	    }

	    /* Signal Gain */
	    if (put_mxarray_double(ml_list, field_signal_gain, i, 
				   channels[i].s.signal_gain)){
		*err = 3;
		break;
	    }

	    /* Signal Offset */
	    if (put_mxarray_double(ml_list, field_signal_offset, i, 
				   channels[i].s.signal_offset)){
		*err = 3;
		break;
	    }

	    /* Signal Slope */
	    if (put_mxarray_double(ml_list, field_signal_slope, i, 
				   channels[i].s.signal_slope)){
		*err = 3;
		break;
	    }

	    /* Signal Units */
	    if(put_mxarray_str(ml_list, field_signal_units, i, 
			       channels[i].s.signal_units)) {
		*err = 3;
		break;
	    }
	}
    }
    if (*err) {
	mxDestroyArray(ml_list);
	ml_list = NULL;
    }
    return ml_list;
}
