/*
 * Support string arrays as char ** input arguments.
 */
%typemap(in, numinputs=1) (const char **channel_names, size_t count_channels) {
	if (!$input.is_cellstr())
		SWIG_exception(SWIG_TypeError, "channel_names must be a cell array of strings");
	Array<std::string> cellstr = $input.cellstr_value();

	$2 = cellstr.nelem();
	$1 = (char **) calloc($2, sizeof(char *));
	if (!$1)
		SWIG_exception(SWIG_MemoryError, "out of memory");

	for (octave_idx_type i = 0; i < cellstr.nelem(); i ++)
	{
		std::string str = cellstr(i);
		size_t len = str.size() + 1;
		$1[i] = (char *) calloc(len, sizeof(char));
		if (!$1[i])
			SWIG_exception(SWIG_MemoryError, "out of memory");
		memcpy($1[i], str.c_str(), len);
	}
}

%typemap(freearg, noblock=1) (const char **channel_names, size_t count_channels) {
	if ($1)
	{
		for (size_t i = 0; i < $2; i ++)
		{
			free($1[i]);
		}
		free($1);
	}
}

%typemap(freearg, noblock=1) (const char *channels, size_t count_channels) {
	if ($1)
	{
		for (size_t i = 0; i < $2; i ++)
		{
			free($1[i]);
		}
		free($1);
	}
}

/* Override default type checking behavior for resolving overloaded functions.
 * By default, swig would check for a wrapped const char **, but we want it
 * to check for a cell string. */
%typemap(typecheck, numinputs=1, noblock=1) (const char **channel_names, size_t count_channels) {
	$1 = (*$input).is_cellstr();
}

/**
 * Support nds2_channel and nds2_buffer arrays double-pointer return values.
 */

%typemap(in, numinputs=0, noblock=1) size_t *count_channels_ptr (size_t count_channels = 1) {
	$1 = &count_channels;
}

%typemap(out, noblock=1) SWIGTYPE ** (octave_value_list outlist) {
	for (size_t i = 0; i < *arg2; i ++)
		outlist.append(SWIG_NewPointerObj($1[i], $*1_descriptor, 1));
	$result = outlist.cell_value();
}

%typemap(newfree, noblock=1) SWIGTYPE ** {
	free($1);
}

/* Use methods named 'toString' to supply string representations of objects. */
%rename toString __str;

%apply SWIGTYPE ** {nds2_channel **, nds2_buffer **};

/**
 * Make buffer.getChannel return a copy.
 */

%typemap(out, noblock=1) nds2_channel * (nds2_channel *ptr) {
	ptr = (nds2_channel *) calloc(1, sizeof(nds2_channel));
	if (!$1)
		SWIG_exception(SWIG_MemoryError, "out of memory");
	memcpy(ptr, $1, sizeof(nds2_channel));
	$result = SWIG_NewPointerObj(ptr, $1_descriptor, 1);
}

/**
 * Support getting numeric arrays from unsigned char[] using polymorphism.
 */

%typemap(out) unsigned char [] {
	dim_vector dims(arg1->length, 1);
	switch (arg1->channel.data_type)
	{
		case NDS2_DATA_TYPE_INT16:
		{
			int16NDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(octave_int16));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_INT32:
		{
			int32NDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(octave_int32));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_INT64:
		{
			int64NDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(octave_int64));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_FLOAT32:
		{
			FloatNDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(float));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_FLOAT64:
		{
			NDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(double));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_COMPLEX32:
		{
			FloatComplexNDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(double));
			$result = arr;
		} break;
		case NDS2_DATA_TYPE_UINT32:
		{
			uint32NDArray arr(dims);
			memcpy((void *) arr.data(), $1, arg1->length * sizeof(octave_uint32));
			$result = arr;
		} break;
		default:
			SWIG_exception(SWIG_TypeError, "Unknown NDS data type");
			break;
	}
}
