open! Stdune
open Result.O

(* Errors *)

(* The current module never raises. It returns all errors as
   [Result.Error (User_error.E _)] values instead. Errors are later
   inserted into [Build.t] values so that they are only raised during
   the actual build rather than while generating the rules. *)

module Error = struct
  (* This sub-module construct the error values generated by functions
     in this module.

     When a location is not available, for instance because the error
     is attached to transitive dependency of a library written by the
     user in a [dune] file, a dependency path should be used to
     explain how dune came to consider the library that triggered the
     error. *)

  let make ?loc ?hints paragraphs =
    Error
      (User_error.E
         (User_error.make ?loc ?hints paragraphs))

  let pp_lib info =
    let name = Lib_info.name info in
    let src_dir = Lib_info.src_dir info in
    Pp.textf "%S in %s"
      (Lib_name.to_string name)
      (Path.to_string_maybe_quoted src_dir)

  let pp_lib_and_dep_path (info, dp) =
    let info = Pp.box (pp_lib info) in
    match dp with
    | [] -> info
    | _ ->
      Pp.vbox
        (Pp.concat ~sep:Pp.cut
           [ info
           ; Dep_path.Entries.pp dp
           ])

  let external_lib_deps_hint () =
    match !Clflags.external_lib_deps_hint with
    | [] -> (* during bootstrap *) []
    | l ->
      [ Pp.box ~indent:2
          (Pp.concat ~sep:Pp.space
             (Pp.text "try:"
              :: List.map l ~f:(fun x -> Pp.verbatim (String.quote_for_shell x))))
      ]

  let not_found ~loc ~name =
    make ~loc
      [ Pp.textf "Library %S not found." (Lib_name.to_string name) ]
      ~hints:(external_lib_deps_hint ())

  let hidden ~loc ~name ~dir ~reason =
    make ~loc
      [ Pp.textf "Library %S in %s is hidden (%s)." (Lib_name.to_string name)
          (Path.to_string_maybe_quoted dir) reason
      ]
      ~hints:(external_lib_deps_hint ())

  (* diml: it is not very clear what a  "default implementation cycle" is *)
  let default_implementation_cycle cycle =
    make
      [ Pp.text "Default implementation cycle detected between the \
                 following libraries:"
      ; Pp.chain cycle ~f:(fun info ->
          let name = Lib_info.name info in
          Pp.textf "%S" (Lib_name.to_string name))
      ]

  let multiple_implementations_for_virtual_lib ~loc ~lib
        ~given_variants ~conflict =
    let name = Lib_info.name lib in
    let default_implementation = Lib_info.default_implementation lib in
    make ~loc
      [ Pp.textf "Multiple solutions for the implementation of %S%s%s:"
          (Lib_name.to_string name)
          (match default_implementation with
           | None -> ""
           | Some (_, x) ->
             sprintf " (default implementation %S)" (Lib_name.to_string x))
          (match Variant.Set.to_list given_variants
                 |> List.map ~f:Variant.to_string
           with
           | [] -> ""
           | [v] -> sprintf "with variant %s" v
           | vs -> sprintf " with variants %s" (String.enumerate_and vs))
      ; Pp.enumerate conflict ~f:(fun lib ->
          let variant = Lib_info.variant lib in
          Pp.seq
            (pp_lib lib)
            (match variant with
             | Some v -> Pp.textf " (variant %s)" (Variant.to_string v)
             | None -> Pp.nop))
      ]

  let double_implementation impl1 impl2 ~vlib =
    make
      [ Pp.concat
          [ Pp.text "Conflicting implementations for virtual library "
          ; pp_lib vlib
          ; Pp.char ':'
          ]
      ; Pp.enumerate [ impl1; impl2 ] ~f:pp_lib_and_dep_path
      ; Pp.text "This cannot work."
      ]

  let no_implementation (info, dp) =
    make
      (Pp.concat
         [ Pp.text "No implementation found for virtual library "
         ; pp_lib info
         ; Pp.char '.'
         ]
       :: match dp with
       | [] -> []
       | _ -> [ Dep_path.Entries.pp dp ])

  let conflict lib1 lib2 =
    make
      [ Pp.text" Conflict between the following libraries:"
      ; Pp.enumerate [ lib1; lib2 ] ~f:pp_lib_and_dep_path
      ; Pp.text "This cannot work."
      ]

  let overlap ~in_workspace ~installed =
    make
      [ Pp.text "Conflict between the following libraries:"
      ; Pp.enumerate [ (in_workspace, []); installed ]
          ~f:pp_lib_and_dep_path
      ]

  let no_solution_found_for_select ~loc =
    make ~loc
      [ Pp.text "No solution found for this select form."
      ]


  let not_an_implementation_of ~vlib ~impl =
    make
      [ Pp.textf "%S is not an implementation of %S."
          (Lib_name.to_string (Lib_info.name impl))
          (Lib_name.to_string (Lib_info.name vlib))
      ]

  let dependency_cycle cycle =
    make
      [ Pp.text "Dependency cycle detected between the following libraries:"
      ; Pp.chain cycle ~f:(fun (dir, name) ->
          Pp.textf "%S in %s"
            (Lib_name.to_string name)
            (Path.to_string_maybe_quoted dir))
      ]

  let private_deps_not_allowed ~loc private_dep =
    let name = Lib_info.name private_dep in
    make ~loc
      [ Pp.textf "Library %S is private, it cannot be a dependency of \
                  a public library."
          (Lib_name.to_string name)
      ; Pp.textf "You need to give %S a public name."
          (Lib_name.to_string name)
      ]

  let not_virtual_lib ~loc ~impl ~not_vlib =
    let impl = Lib_info.name impl in
    let not_vlib = Lib_info.name not_vlib in
    make ~loc
      [ Pp.textf "Library %S is not virtual. It cannot be implemented by %S."
          (Lib_name.to_string not_vlib)
          (Lib_name.to_string impl)
      ]

  let vlib_known_implementation_mismatch ~loc ~name ~variant ~vlib_name =
    make ~loc
      [ Pp.textf "Virtual library %S does not know about implementation %S with \
                  variant %S. Instead of using (variant %s) here, you need to \
                  reference it in the virtual library project, using the \
                  external_variant stanza:"
          (Lib_name.to_string vlib_name)
          (Lib_name.to_string name)
          (Variant.to_string variant)
          (Variant.to_string variant)
      ; Pp.textf
          "(external_variant\n\
          \  (virtual_library %s)\n\
          \  (variant %s)\n\
          \  (implementation %s))"
          (Lib_name.to_string vlib_name)
          (Variant.to_string variant)
          (Lib_name.to_string name)
      ]

  let vlib_variant_conflict ~loc ~name ~known_impl_name ~variant ~vlib_name =
    make ~loc
      [ Pp.textf "Implementation %S cannot have variant %S for virtual library \
      %S as it is already defined for implementation %S."
          (Lib_name.to_string name)
          (Variant.to_string variant)
          (Lib_name.to_string vlib_name)
          (Lib_name.to_string known_impl_name)
      ]


end


(* Types *)

module Resolved_select = struct
  type t =
    { src_fn : string Or_exn.t
    ; dst_fn : string
    }
end

type sub_system = ..

module Sub_system0 = struct
  module type S = sig
    type t
    type sub_system += T of t
    val encode : (t -> Syntax.Version.t * Dune_lang.t list) option
  end

  type 'a s = (module S with type t = 'a)

  module Instance = struct
    type t = T : 'a s * 'a -> t
  end
end

module Id : sig
  type t =
    { unique_id : int
    ; path      : Path.t
    ; name      : Lib_name.t
    }

  val hash : t -> int

  val compare : t -> t -> Ordering.t

  include Comparator.OPS with type t := t

  val make : path:Path.t -> name:Lib_name.t -> t

  module Set : Set.S with type elt = t

  module Top_closure : Top_closure.S
    with type key := t
     and type 'a monad := 'a Monad.Id.t
end = struct
  module T = struct
    type t =
      { unique_id : int
      ; path      : Path.t
      ; name      : Lib_name.t
      }

    let compare t1 t2 = Int.compare t1.unique_id t2.unique_id
    let to_dyn _ = Dyn.opaque
  end

  include T

  include (
    Comparator.Operators(T)
    : Comparator.OPS with type t := T.t
  )

  let gen_unique_id =
    let next = ref 0 in
    fun () ->
      let n = !next in
      next := n + 1;
      n

  let hash t = t.unique_id

  let make ~path ~name =
    { unique_id = gen_unique_id ()
    ; path
    ; name
    }

  module O = Comparable.Make(T)
  module Set = O.Set
  module Top_closure = Top_closure.Make(Set)(Monad.Id)
end

module T = struct
  type t =
    { info              : Lib_info.external_
    ; name              : Lib_name.t
    ; unique_id         : Id.t
    ; requires          : t list Or_exn.t
    ; ppx_runtime_deps  : t list Or_exn.t
    ; pps               : t list Or_exn.t
    ; resolved_selects  : Resolved_select.t list
    ; user_written_deps : Dune_file.Lib_deps.t
    ; implements        : t Or_exn.t option
    ; stdlib_dir        : Path.t
    ; (* these fields cannot be forced until the library is instantiated *)
      default_implementation     : t Or_exn.t Lazy.t option
    ; (* if this is a virtual library, this library contains all known
         implementations that are associated with a variant *)
      resolved_implementations   : t Or_exn.t Variant.Map.t Lazy.t option
    ; (* This is mutable to avoid this error:

         {[
           This kind of expression is not allowed as right-hand side of `let rec'
         }]
         *)
      mutable sub_systems : Sub_system0.Instance.t Lazy.t Sub_system_name.Map.t
    }

  let compare (x : t) (y : t) = Id.compare x.unique_id y.unique_id

  let to_dyn t = Lib_name.to_dyn t.name
end

include T

include (Comparator.Operators(T) : Comparator.OPS with type t := t)

type status =
  | St_initializing of Id.t (* To detect cycles *)
  | St_found        of t
  | St_not_found
  | St_hidden       of t * Path.t * string (* reason *)

type db =
  { parent               : db option
  ; resolve              : Lib_name.t -> resolve_result
  ; table                : (Lib_name.t, status) Hashtbl.t
  ; all                  : Lib_name.t list Lazy.t
  ; stdlib_dir           : Path.t
  }

and resolve_result =
  | Not_found
  | Found    of Lib_info.external_
  | Hidden   of Lib_info.external_ * string
  | Redirect of db option * Lib_name.t

type lib = t

(* Generals *)

let name t = t.name
let info t = t.info
let implements t = t.implements

let unique_id    t = t.unique_id

let is_impl      t = Option.is_some t.implements

let obj_dir t = Lib_info.obj_dir t.info

let is_local t =
  let obj_dir = obj_dir t in
  Path.is_managed (Obj_dir.byte_dir obj_dir)

let main_module_name t =
  let main_module_name = Lib_info.main_module_name t.info in
  match main_module_name with
  | This mmn -> Ok mmn
  | From _ ->
    let+ vlib = Option.value_exn t.implements in
    let main_module_name = Lib_info.main_module_name vlib.info in
    match main_module_name with
    | This x -> x
    | From _ -> assert false

let wrapped t =
  let wrapped = Lib_info.wrapped t.info in
  match wrapped with
  | None -> Ok None
  | Some (This wrapped) -> Ok (Some wrapped)
  | Some (From _) ->
    let+ vlib = Option.value_exn t.implements in
    let wrapped = Lib_info.wrapped vlib.info in
    match wrapped with
    | Some (From _) (* can't inherit this value in virtual libs *)
    | None -> assert false (* will always be specified in dune package *)
    | Some (This x) -> Some x

let package t =
  let status = Lib_info.status t.info in
  match status with
  | Installed -> Some (Lib_name.package_name t.name)
  | Public (_, p) -> Some p.name
  | Private _ -> None

let to_id t : Id.t = t.unique_id
let equal l1 l2 = Id.equal (to_id l1) (to_id l2)
let hash t = Id.hash (to_id t)

include Comparable.Make(T)

module L = struct
  type nonrec t = t list

  let to_iflags dirs =
    Command.Args.S
      (Path.Set.fold dirs ~init:[] ~f:(fun dir acc ->
         Command.Args.Path dir :: A "-I" :: acc)
       |> List.rev)

  let include_paths ts =
    let dirs =
      List.fold_left ts ~init:Path.Set.empty ~f:(fun acc t ->
        let obj_dir = Lib_info.obj_dir t.info in
        let public_cmi_dir = Obj_dir.public_cmi_dir obj_dir in
        let native_dir = Obj_dir.native_dir obj_dir in
        List.fold_left ~f:Path.Set.add ~init:acc
          [public_cmi_dir ; native_dir])
    in
    match ts with
    | [] -> dirs
    | x :: _ -> Path.Set.remove dirs x.stdlib_dir

  let include_flags ts = to_iflags (include_paths ts)

  let c_include_paths ts =
    let dirs =
      List.fold_left ts ~init:Path.Set.empty ~f:(fun acc t ->
        let src_dir = Lib_info.src_dir t.info in
        Path.Set.add acc src_dir)
    in
    match ts with
    | [] -> dirs
    | x :: _ -> Path.Set.remove dirs x.stdlib_dir

  let c_include_flags ts = to_iflags (c_include_paths ts)

  let link_flags ts ~mode =
    Command.Args.S
      (c_include_flags ts ::
       List.map ts ~f:(fun t ->
         let archives = Lib_info.archives t.info in
         Command.Args.Deps (Mode.Dict.get archives mode)))

  let compile_and_link_flags ~compile ~link ~mode =
    let dirs =
      Path.Set.union
        (  include_paths compile)
        (c_include_paths link)
    in
    Command.Args.S
      (to_iflags dirs ::
       List.map link ~f:(fun t ->
         let archives = Lib_info.archives t.info in
         Command.Args.Deps (Mode.Dict.get archives mode)))

  let jsoo_runtime_files ts =
    List.concat_map ts ~f:(fun t -> Lib_info.jsoo_runtime t.info)

  let archive_files ts ~mode =
    List.concat_map ts ~f:(fun t ->
      let archives = Lib_info.archives t.info in
      let foreign_archives = Lib_info.foreign_archives t.info in
      Mode.Dict.get archives mode @
      Mode.Dict.get foreign_archives mode)

  let remove_dups l =
    let rec loop acc l seen =
      match l with
      | [] -> acc
      | x :: l ->
        if Id.Set.mem seen x.unique_id then
          loop acc l seen
        else
          loop (x :: acc) l (Id.Set.add seen x.unique_id)
    in
    loop [] l Id.Set.empty

  let top_closure l ~key ~deps =
    Id.Top_closure.top_closure l
      ~key:(fun t -> unique_id (key t))
      ~deps
end

module Lib_and_module = struct
  type t =
    | Lib of lib
    | Module of Path.t Obj_dir.t * Module.t

  module L = struct
    type nonrec t = t list

    let link_flags ts ~mode =
      let libs = List.filter_map ts ~f:(function
        | Lib lib -> Some lib
        | Module _ -> None) in
      Command.Args.S
        (L.c_include_flags libs ::
         List.map ts ~f:(function
           | Lib t ->
             let archives = Lib_info.archives t.info in
             Command.Args.Deps (Mode.Dict.get archives mode)
           | Module (obj_dir, m) ->
             Dep (Obj_dir.Module.cm_file_unsafe obj_dir m
                    ~kind:(Mode.cm_kind mode))
         ))

    let of_libs l = List.map l ~f:(fun x -> Lib x)
  end
end

(* Sub-systems *)

module Sub_system = struct
  type t = sub_system = ..

  module type S = sig
    module Info : Sub_system_info.S
    type t
    type sub_system += T of t
    val instantiate
      :  resolve:(Loc.t * Lib_name.t -> lib Or_exn.t)
      -> get:(loc:Loc.t -> lib -> t option)
      -> lib
      -> Info.t
      -> t
    val encode : (t -> Syntax.Version.t * Dune_lang.t list) option
  end

  module type S' = sig
    include S
    val for_instance : t Sub_system0.s
    val get : lib -> t option
  end

  let all = Sub_system_name.Table.create ~default_value:None

  module Register(M : S) = struct
    let get lib =
      Option.map (Sub_system_name.Map.find lib.sub_systems M.Info.name)
        ~f:(fun (lazy (Sub_system0.Instance.T ((module X), t))) ->
          match X.T t with
          | M.T t -> t
          | _   -> assert false)

    let () =
      let module M = struct
        include M
        let for_instance = (module M : Sub_system0.S with type t = t)
        let get = get
      end in
      Sub_system_name.Table.set all ~key:M.Info.name
        ~data:(Some (module M : S'))
  end

  let instantiate name info lib ~resolve =
    let impl = Option.value_exn (Sub_system_name.Table.get all name) in
    let (module M : S') = impl in
    match info with
    | M.Info.T info ->
      let get ~loc lib' =
        if lib = lib' then
          User_error.raise ~loc
            [ Pp.textf "Library %S depends on itself"
                (Lib_name.to_string lib.name)
            ]
        else
          M.get lib'
      in
      Sub_system0.Instance.T
        (M.for_instance, M.instantiate ~resolve ~get lib info)
    | _ -> assert false

  let dump_config lib =
    Sub_system_name.Map.filter_map lib.sub_systems ~f:(fun (lazy inst) ->
      let (Sub_system0.Instance.T ((module M), t)) = inst in
      Option.map ~f:(fun f -> f t) M.encode)
end

(* Library name resolution and transitive closure *)

(* Dependency stack used while resolving the dependencies of a library
   that was just returned by the [resolve] callback *)
module Dep_stack = struct
  type t =
    { stack : Id.t list
    ; seen  : Id.Set.t
    }

  let empty =
    { stack = []
    ; seen  = Id.Set.empty
    }

  let to_required_by t ~stop_at =
    let stop_at = stop_at.stack in
    let rec loop acc l =
      if List.physically_equal l stop_at then
        List.rev acc
      else
        match l with
        | [] -> assert false
        | { Id.path; name; _ } :: l ->
          loop (Dep_path.Entry.Library (path, name) :: acc) l
    in
    loop [] t.stack

  let dependency_cycle t (last : Id.t) =
    assert (Id.Set.mem t.seen last);
    let rec build_loop acc stack =
      match stack with
      | [] -> assert false
      | (x : Id.t) :: stack ->
        let acc = (x.path, x.name) :: acc in
        if Id.equal x last then
          acc
        else
          build_loop acc stack
    in
    let loop = build_loop [last.path, last.name] t.stack in
    Error.dependency_cycle loop

  let create_and_push t name path =
    let init = Id.make ~path ~name in
    (init,
     { stack = init :: t.stack
     ; seen  = Id.Set.add t.seen init
     })

  let push t (x : Id.t) : (_, _) result =
    if Id.Set.mem t.seen x then
      dependency_cycle t x
    else
      Ok { stack = x :: t.stack
         ; seen  = Id.Set.add t.seen x
         }
end

let check_private_deps lib ~loc ~allow_private_deps =
  let status = Lib_info.status lib.info in
  if (not allow_private_deps) && Lib_info.Status.is_private status
  then
    Error.private_deps_not_allowed ~loc lib.info
  else
    Ok lib

let already_in_table info name x =
  let to_dyn = Dyn.Encoder.(pair Path.to_dyn Lib_name.to_dyn) in
  let dyn =
    let open Dyn.Encoder in
    match x with
    | St_initializing x -> constr "St_initializing" [Path.to_dyn x.path]
    | St_found t ->
      let src_dir = Lib_info.src_dir t.info in
      constr "St_found" [Path.to_dyn src_dir]
    | St_not_found -> constr "Not_found" []
    | St_hidden (_, path, reason) ->
      constr "Hidden" [Path.to_dyn path; string reason]
  in
  let src_dir = Lib_info.src_dir info in
  Code_error.raise
    "Lib_db.DB: resolver returned name that's already in the table"
    [ "name"            , Lib_name.to_dyn name
    ; "returned_lib"    , to_dyn (src_dir, name)
    ; "conflicting_with", dyn
    ]

module Vlib : sig
  (** Make sure that for every virtual library in the list there is at
      most one corresponding implementation.

      Additionally, if linking is [true], ensures that every virtual
      library as an implementation and re-arrange the list so that
      implementations replaces virtual libraries. *)
  val associate
    :  (t * Dep_stack.t) list
    -> orig_stack:Dep_stack.t
    -> linking:bool
    -> t list Or_exn.t

  module Unimplemented : sig
    (** set of unimplemented libraries*)
    type t

    val empty : t

    val add : t -> lib -> t Or_exn.t

    val fold
      :  t
      -> init:'acc
      -> f:(lib -> 'acc -> 'acc Or_exn.t)
      -> 'acc Or_exn.t
  end
end = struct
  module Unimplemented = struct
    type status = Implemented | Not_implemented
    type t = status Map.t

    let empty = Map.empty

    let add t lib =
      let virtual_ = Lib_info.virtual_ lib.info in
      match lib.implements, virtual_ with
      | None, None -> Ok t
      | Some _, Some _ ->
        assert false (* can't be virtual and implement *)
      | None, Some _ ->
        Ok (Map.update t lib ~f:(function
          | None -> Some Not_implemented
          | Some _ as x -> x))
      | Some vlib, None ->
        let+ vlib = vlib in
        Map.set t vlib Implemented

    let fold =
      let rec loop ~f ~acc = function
        | [] -> Ok acc
        | (_, Implemented) :: libs -> loop ~f ~acc libs
        | (lib, Not_implemented) :: libs ->
          let* acc = f lib acc in
          loop ~f ~acc libs
      in
      fun t ~init ~f -> loop (Map.to_list t) ~acc:init ~f
  end
  module Table = struct
    module Partial = struct
      type vlib_status =
        | No_impl of Dep_stack.t
        | Impl of lib * Dep_stack.t
      type t = vlib_status Map.t

      let is_empty = Map.is_empty

      let make closure ~orig_stack : t Or_exn.t =
        let rec loop acc = function
          | [] -> Ok acc
          | (lib, stack) :: libs ->
            let virtual_ = Lib_info.virtual_ lib.info in
            match lib.implements, virtual_ with
            | None, None -> loop acc libs
            | Some _, Some _ ->
              assert false (* can't be virtual and implement *)
            | None, Some _ ->
              loop (Map.set acc lib (No_impl stack)) libs
            | Some vlib, None ->
              let* vlib = vlib in
              begin match Map.find acc vlib with
              | None ->
                (* we've already traversed the virtual library because
                   it must have occured earlier in the closure *)
                assert false
              | Some (No_impl _) ->
                loop (Map.set acc vlib (Impl (lib, stack))) libs
              | Some (Impl (lib', stack')) ->
                let req_by' =
                  Dep_stack.to_required_by stack' ~stop_at:orig_stack
                in
                let req_by =
                  Dep_stack.to_required_by stack ~stop_at:orig_stack
                in
                Error.double_implementation
                  (lib'.info, req_by')
                  (lib.info, req_by)
                  ~vlib:vlib.info
              end
        in
        loop Map.empty closure
    end

    type t = lib Map.t

    let make impls ~orig_stack : t Or_exn.t =
      let rec loop acc = function
        | [] -> Ok acc
        | (vlib, Partial.No_impl stack) :: _ ->
          let rb = Dep_stack.to_required_by stack ~stop_at:orig_stack in
          Error.no_implementation (vlib.info, rb)
        | (vlib, (Impl (impl, _stack))) :: libs ->
          loop (Map.set acc vlib impl) libs
      in
      loop Map.empty (Map.to_list impls)
  end

  let second_step_closure ts impls =
    let visited = ref Id.Set.empty in
    let res = ref [] in
    let rec loop t =
      let t = Option.value ~default:t (Map.find impls t) in
      if Id.Set.mem !visited t.unique_id then
        Ok ()
      else begin
        visited := Id.Set.add !visited t.unique_id;
        let* deps = t.requires in
        let+ () = Result.List.iter deps ~f:loop in
        res := t :: !res
      end
    in
    let+ () = Result.List.iter ts ~f:loop in
    List.rev !res

  let associate closure ~orig_stack ~linking =
    let* impls = Table.Partial.make closure ~orig_stack in
    let closure = List.map closure ~f:fst in
    if linking && not (Table.Partial.is_empty impls) then
      let* impls = Table.make impls ~orig_stack in
      second_step_closure closure impls
    else
      Ok closure
end

module Vlib_visit : sig
  type t

  val create : unit -> t

  val visit
    :  t
    -> lib
    -> stack:Lib_info.external_ list
    -> f:(lib -> unit Or_exn.t)
    -> unit Or_exn.t
end = struct
  module Status = struct
    type t =
      | Visiting
      | Visited
  end

  type t = Status.t Map.t ref

  let create () = ref Map.empty

  let visit t lib ~stack ~f =
    match Map.find !t lib with
    | Some Status.Visited -> Ok ()
    | Some Visiting ->
      Error.default_implementation_cycle (lib.info :: stack)
    | None ->
      t := Map.set !t lib Visiting;
      let res = f lib in
      t := Map.set !t lib Visited;
      res
end

(* Find implementation that matches given variants *)
let find_implementation_for lib ~variants =
  assert (
    let virtual_ = Lib_info.virtual_ lib.info in
    Option.is_some virtual_
  );
  match variants with
  | None -> Ok None
  | Some (loc, variants_set) ->
    let available_implementations =
      Lazy.force (Option.value_exn lib.resolved_implementations)
    in
    let* candidates =
      Variant.Set.fold variants_set
        ~init:[]
        ~f:(fun variant acc ->
          match Variant.Map.find available_implementations variant with
          | Some res -> res :: acc
          | None -> acc)
      |> Result.List.all
    in
    (* TODO once we find one conflict, there's no need to search for more *)
    match candidates with
    | [] -> Ok None
    | [elem] -> Ok (Some elem)
    | conflict ->
      let conflict = List.map conflict ~f:(fun lib -> lib.info) in
      Error.multiple_implementations_for_virtual_lib ~loc
        ~lib:lib.info
        ~given_variants:variants_set
        ~conflict

module rec Resolve : sig
  val find_internal : db -> Lib_name.t -> stack:Dep_stack.t -> status

  val resolve_dep
    : db
    -> Lib_name.t
    -> allow_private_deps:bool
    -> loc:Loc.t
    -> stack:Dep_stack.t
    -> lib Or_exn.t

  val resolve_name : db -> Lib_name.t -> stack:Dep_stack.t -> status

  val available_internal : db -> Lib_name.t -> stack:Dep_stack.t -> bool

  val resolve_simple_deps
    : db ->
    (Loc.t * Lib_name.t) list
    -> allow_private_deps:bool
    -> stack:Dep_stack.t
    -> (t list, exn) Result.t


  val resolve_user_deps
    :  db
    -> Lib_info.Deps.t
    -> allow_private_deps:bool
    -> pps:(Loc.t * Lib_name.t) list
    -> stack:Dep_stack.t
    -> lib list Or_exn.t
       * lib list Or_exn.t
       * Resolved_select.t list

  val closure_with_overlap_checks
    :  db option
    -> lib list
    -> stack:Dep_stack.t
    -> linking:bool
    -> variants:(Loc.t * Variant.Set.t) option
    -> lib list Or_exn.t
end = struct

  open Resolve

  let instantiate db name info ~stack ~hidden =
    let unique_id, stack =
      let src_dir = Lib_info.src_dir info in
      Dep_stack.create_and_push stack name src_dir
    in
    Option.iter (Hashtbl.find db.table name) ~f:(fun x ->
      already_in_table info name x);
    (* Add [id] to the table, to detect loops *)
    Hashtbl.add_exn db.table name (St_initializing unique_id);

    let status = Lib_info.status info in
    let allow_private_deps = Lib_info.Status.is_private status in

    let resolve (loc, name) =
      resolve_dep db (name : Lib_name.t) ~allow_private_deps ~loc ~stack in

    let implements =
      let open Option.O in
      let+ ((loc, _) as name) = Lib_info.implements info in
      let open Result.O in
      let* vlib = resolve name in
      let virtual_ = Lib_info.virtual_ vlib.info in
      match virtual_ with
      | None ->
        Error.not_virtual_lib ~loc ~impl:info ~not_vlib:vlib.info
      | Some _ ->
        let variant = Lib_info.variant info in
        begin match variant with
        | None -> Ok vlib
        | Some variant ->
          (* If the library is an implementation tagged with a variant, we must
              make sure that that it's correctly part of the virtual library's
              known implementations. *)
          let name = Lib_info.name info in
          let vlib_name = Lib_info.name vlib.info in
          let vlib_impls = Lib_info.known_implementations vlib.info in
          let* (_, impl_name) =
            match Variant.Map.find vlib_impls variant with
            | None -> Error.vlib_known_implementation_mismatch
                        ~loc ~name ~variant ~vlib_name
            | Some impl_name -> Ok impl_name
          in
          if Lib_name.equal impl_name name then
            Ok vlib
          else
            Error.vlib_variant_conflict
              ~loc ~name ~known_impl_name:impl_name ~variant ~vlib_name
        end
    in
    let resolve_impl impl_name =
      let* impl = resolve impl_name in
      let* vlib =
        match impl.implements with
        | Some vlib -> vlib
        | None -> Error.not_an_implementation_of
                    ~vlib:info ~impl:impl.info
      in
      if Id.equal vlib.unique_id unique_id then
        Ok impl
      else
        Error.not_an_implementation_of
          ~vlib:info ~impl:impl.info
    in
    let default_implementation =
      Lib_info.default_implementation info
      |> Option.map ~f:(fun l -> lazy (resolve_impl l)) in
    let resolved_implementations =
      Lib_info.virtual_ info
      |> Option.map ~f:(fun _ -> lazy (
        (* TODO this can be made even lazier as we don't need to resolve all
           variants at once *)
        Lib_info.known_implementations info
        |> Variant.Map.map ~f:resolve_impl))
    in
    let requires, pps, resolved_selects =
      let pps = Lib_info.pps info in
      Lib_info.requires info
      |> resolve_user_deps db ~allow_private_deps ~pps ~stack
    in
    let requires =
      match implements with
      | None -> requires
      | Some impl ->
        let* impl = impl in
        let+ requires = requires in
        impl :: requires
    in
    let ppx_runtime_deps =
      Lib_info.ppx_runtime_deps info
      |> resolve_simple_deps db ~allow_private_deps ~stack
    in
    let src_dir = Lib_info.src_dir info in
    let map_error x =
      Result.map_error x ~f:(fun e ->
        Dep_path.prepend_exn e (Library (src_dir, name)))
    in
    let requires         = map_error requires         in
    let ppx_runtime_deps = map_error ppx_runtime_deps in
    let t =
      { info
      ; name
      ; unique_id
      ; requires
      ; ppx_runtime_deps
      ; pps
      ; resolved_selects
      ; user_written_deps = Lib_info.user_written_deps info
      ; sub_systems       = Sub_system_name.Map.empty
      ; implements
      ; default_implementation
      ; resolved_implementations
      ; stdlib_dir = db.stdlib_dir
      }
    in
    t.sub_systems <-
      Lib_info.sub_systems info
      |> Sub_system_name.Map.mapi ~f:(fun name info ->
        lazy (Sub_system.instantiate name info t ~resolve));

    let res =
      let hidden =
        match hidden with
        | Some _ -> hidden
        | None ->
          let enabled = Lib_info.enabled info in
          match enabled with
          | Normal -> None
          | Optional ->
            Option.some_if
              (not (Result.is_ok t.requires && Result.is_ok t.ppx_runtime_deps))
              "optional with unavailable dependencies"
          | Disabled_because_of_enabled_if ->
            Some "unsatisfied 'enabled_if'"
      in
      match hidden with
      | None -> St_found t
      | Some reason ->
        St_hidden (t, src_dir, reason)
    in
    Hashtbl.replace db.table ~key:name ~data:res;
    res

  let find_internal db (name : Lib_name.t) ~stack : status =
    match Hashtbl.find db.table name with
    | Some x -> x
    | None   -> resolve_name db name ~stack

  let resolve_dep db (name : Lib_name.t) ~allow_private_deps
        ~loc ~stack : t Or_exn.t =
    match find_internal db name ~stack with
    | St_initializing id ->
      Dep_stack.dependency_cycle stack id
    | St_found lib -> check_private_deps lib ~loc ~allow_private_deps
    | St_not_found -> Error.not_found ~loc ~name
    | St_hidden (_, dir, reason) ->
      Error.hidden ~loc ~name ~dir ~reason

  let resolve_name db name ~stack =
    match db.resolve name with
    | Redirect (db', name') -> begin
        let db' = Option.value db' ~default:db in
        match find_internal db' name' ~stack with
        | St_initializing _ as x -> x
        | x ->
          Hashtbl.add_exn db.table name x;
          x
      end
    | Found info ->
      instantiate db name info ~stack ~hidden:None
    | Not_found ->
      let res =
        match db.parent with
        | None    -> St_not_found
        | Some db -> find_internal db name ~stack
      in
      Hashtbl.add_exn db.table name res;
      res
    | Hidden (info, hidden) ->
      match
        match db.parent with
        | None    -> St_not_found
        | Some db -> find_internal db name ~stack
      with
      | St_found _ as x ->
        Hashtbl.add_exn db.table name x;
        x
      | _ ->
        instantiate db name info ~stack ~hidden:(Some hidden)

  let available_internal db (name : Lib_name.t) ~stack =
    resolve_dep db name ~allow_private_deps:true ~loc:Loc.none ~stack
    |> Result.is_ok

  let resolve_simple_deps db names ~allow_private_deps ~stack =
    Result.List.map names ~f:(fun (loc, name) ->
      resolve_dep db name ~allow_private_deps ~loc ~stack)

  let resolve_complex_deps db deps ~allow_private_deps ~stack =
    let res, resolved_selects =
      List.fold_left deps ~init:(Ok [], []) ~f:(fun (acc_res, acc_selects) dep ->
        let res, acc_selects =
          match (dep : Dune_file.Lib_dep.t) with
          | Direct (loc, name) ->
            let res =
              resolve_dep db name ~allow_private_deps ~loc ~stack
              >>| List.singleton
            in
            (res, acc_selects)
          | Select { result_fn; choices; loc } ->
            let res, src_fn =
              match
                List.find_map choices ~f:(fun { required; forbidden; file } ->
                  if Lib_name.Set.exists forbidden
                       ~f:(available_internal db ~stack) then
                    None
                  else
                    match
                      let deps =
                        Lib_name.Set.fold required ~init:[] ~f:(fun x acc ->
                          (loc, x) :: acc)
                      in
                      resolve_simple_deps ~allow_private_deps db deps ~stack
                    with
                    | Ok ts -> Some (ts, file)
                    | Error _ -> None)
              with
              | Some (ts, file) ->
                (Ok ts, Ok file)
              | None ->
                let e () = Error.no_solution_found_for_select ~loc in
                (e (), e ())
            in
            (res, { Resolved_select. src_fn; dst_fn = result_fn }
                  :: acc_selects)
        in
        let res =
          match res, acc_res with
          | Ok l, Ok acc -> Ok (List.rev_append l acc)
          | (Error _ as res), _
          | _, (Error _ as res) -> res
        in
        (res, acc_selects))
    in
    let res =
      match res with
      | Ok    l -> Ok (List.rev l)
      | Error _ -> res
    in
    (res, resolved_selects)


  let resolve_deps db deps ~allow_private_deps ~stack =
    (* Compute transitive closure *)
    let libs, selects = match (deps : Lib_info.Deps.t) with
      | Simple  names ->
        (resolve_simple_deps db names ~allow_private_deps ~stack, [])
      | Complex names ->
        resolve_complex_deps db names ~allow_private_deps ~stack
    in
    (* Find implementations for virtual libraries. *)
    libs, selects


  let resolve_user_deps db deps ~allow_private_deps ~pps ~stack =
    let deps, resolved_selects =
      resolve_deps db deps ~allow_private_deps ~stack in
    let deps, pps =
      match pps with
      | [] -> (deps, Ok [])
      | first :: others as pps ->
        (* Location of the list of ppx rewriters *)
        let loc : Loc.t =
          let (last, _) : (Loc.t * _) =
            Option.value (List.last others) ~default:first in
          { (fst first) with stop = last.stop }
        in
        let pps =
          let* pps =
            resolve_simple_deps db pps ~allow_private_deps:true ~stack in
          closure_with_overlap_checks None pps ~stack ~linking:true
            ~variants:None
        in
        let deps =
          let* init = deps in
          pps >>=
          Result.List.fold_left ~init ~f:(fun init pp ->
            pp.ppx_runtime_deps >>=
            Result.List.fold_left ~init ~f:(fun acc rt ->
              let+ rt = check_private_deps rt ~loc ~allow_private_deps in
              rt :: acc))
        in
        (deps, pps)
    in
    (deps, pps, resolved_selects)

  (* Compute transitive closure of libraries to figure which ones will trigger
     their default implementation.

     Assertion: libraries is a list of virtual libraries with no implementation.
     The goal is to find which libraries can safely be defaulted. *)
  let resolve_default_libraries libraries ~variants =
    (* Map from a vlib to vlibs that are implemented in the transitive closure
       of its default impl. *)
    let vlib_status = Vlib_visit.create () in
    (* Reverse map *)
    let vlib_default_parent = ref Map.empty in
    let avoid_direct_parent vlib (impl : lib) =
      match impl.implements with
      | None -> Ok true
      | Some x -> let+ x = x in x <> vlib
    in
    (* Either by variants or by default. *)
    let impl_for vlib =
      find_implementation_for vlib ~variants >>= function
      | Some impl -> Ok (Some impl)
      | None ->
        begin match vlib.default_implementation with
        | None -> Ok None
        | Some d -> Result.map ~f:Option.some (Lazy.force d)
        end
    in
    let impl_different_from_vlib_default vlib (impl : lib) =
      impl_for vlib >>| function
      | None -> true
      | Some lib -> lib <> impl
    in
    let library_is_default lib =
      match Map.find !vlib_default_parent lib with
      | Some (_ :: _) -> None
      | None
      | Some [] ->
        Option.bind lib.default_implementation ~f:(fun lib ->
          Result.to_option (Lazy.force lib))
    in
    (* Gather vlibs that are transitively implemented by another
       vlib's default implementation. *)
    let rec visit ~stack ancestor_vlib =
      Vlib_visit.visit vlib_status ~stack ~f:(fun lib ->
        (* Visit direct dependencies *)
        let* deps = lib.requires in
        let* () =
          List.filter deps ~f:(fun x ->
            match avoid_direct_parent x lib with
            | Ok x -> x
            | Error _ -> false)
          |> Result.List.iter
               ~f:(visit ~stack:(lib.info :: stack) ancestor_vlib)
        in
        (* If the library is an implementation of some virtual library that
           overrides default, add a link in the graph. *)
        let* () =
          Result.Option.iter lib.implements ~f:(fun vlib ->
            let* res = impl_different_from_vlib_default vlib lib in
            match res, ancestor_vlib with
            | true, None ->
              (* Recursion: no ancestor, vlib is explored *)
              visit ~stack:(lib.info::stack) None vlib
            | true, Some ancestor ->
              vlib_default_parent := Map.Multi.cons
                                       !vlib_default_parent
                                       lib
                                       ancestor;
              visit ~stack:(lib.info :: stack) None vlib
            | false, _ ->
              (* If lib is the default implementation, we'll manage it when
                 handling virtual lib. *)
              Ok ())
        in
        (* If the library has an implementation according to variants or default
           impl. *)
        let virtual_ = Lib_info.virtual_ lib.info in
        if Option.is_none virtual_ then
          Ok ()
        else
          let* impl = impl_for lib in
          begin match impl with
          | None -> Ok ()
          | Some impl -> visit ~stack:(lib.info :: stack) (Some lib) impl
          end
      )
    in
    (* For each virtual library we know which vlibs will be implemented when
       enabling its default implementation. *)
    let+ () = Result.List.iter ~f:(visit ~stack:[] None) libraries in
    List.filter_map ~f:library_is_default libraries

  let closure_with_overlap_checks db ts ~stack:orig_stack ~linking ~variants =
    let visited = ref Map.empty in
    let unimplemented = ref Vlib.Unimplemented.empty in
    let res = ref [] in
    let rec loop t ~stack =
      match Map.find !visited t with
      | Some (t', stack') ->
        if t = t' then
          Ok ()
        else
          let req_by = Dep_stack.to_required_by ~stop_at:orig_stack in
          Error.conflict
            (t'.info, req_by stack')
            (t.info, req_by stack)
      | None ->
        visited := Map.set !visited t (t, stack);
        let* () =
          match db with
          | None -> Ok ()
          | Some db ->
            match find_internal db t.name ~stack with
            | St_found t' ->
              if t = t' then
                Ok ()
              else begin
                let req_by =
                  Dep_stack.to_required_by stack ~stop_at:orig_stack in
                Error.overlap ~in_workspace:t'.info ~installed:(t.info, req_by)
              end
            | _ -> assert false
        in
        let* new_stack = Dep_stack.push stack (to_id t) in
        let* deps = t.requires in
        let* unimplemented' = Vlib.Unimplemented.add !unimplemented t in
        unimplemented := unimplemented';
        let+ () = Result.List.iter deps ~f:(loop ~stack:new_stack) in
        res := (t, stack) :: !res
    in
    (* Closure loop with virtual libraries/variants selection*)
    let rec handle ts ~stack =
      let* () = Result.List.iter ts ~f:(loop ~stack) in
      if not linking then
        Ok ()
      else begin
        (* Virtual libraries: find implementations according to variants. *)
        let* (lst, with_default_impl) =
          !unimplemented
          |> Vlib.Unimplemented.fold ~init:([], []) ~f:(fun lib (lst, def) ->
            let* impl =
              find_implementation_for lib ~variants in
            match impl, lib.default_implementation with
            | None, Some _ ->
              Ok (lst, (lib :: def))
            | None, None ->
              Ok (lst, def)
            | Some (impl : lib), _ ->
              Ok (impl :: lst, def))
        in
        (* Manage unimplemented libraries that have a default implementation. *)
        match lst, with_default_impl with
        | [], [] ->
          Ok ()
        | [], def ->
          resolve_default_libraries def ~variants
          >>= handle ~stack
        | lst, _ ->
          handle lst ~stack
      end
    in
    let* () = handle ts ~stack:orig_stack in
    Vlib.associate (List.rev !res) ~linking ~orig_stack

end

let closure_with_overlap_checks db l ~variants =
  Resolve.closure_with_overlap_checks db l ~stack:Dep_stack.empty ~variants

let closure l = closure_with_overlap_checks None l ~variants:None

let requires_exn         t = Result.ok_exn t.requires
let ppx_runtime_deps_exn t = Result.ok_exn t.ppx_runtime_deps
let closure_exn          l ~linking = Result.ok_exn (closure l ~linking)

module Compile = struct
  module Resolved_select = Resolved_select

  type nonrec t =
    { direct_requires   : t list Or_exn.t
    ; requires_link     : t list Or_exn.t Lazy.t
    ; pps               : t list Or_exn.t
    ; resolved_selects  : Resolved_select.t list
    ; lib_deps_info     : Lib_deps_info.t
    ; sub_systems       : Sub_system0.Instance.t Lazy.t Sub_system_name.Map.t
    }

  let make_lib_deps_info ~user_written_deps ~pps ~kind =
    Lib_deps_info.merge
      (Dune_file.Lib_deps.info user_written_deps ~kind)
      (List.map pps ~f:(fun (_, pp) -> (pp, kind))
       |> Lib_name.Map.of_list_reduce ~f:Lib_deps_info.Kind.merge)

  let for_lib db (t : lib) =
    let lib_deps_info =
      let pps = Lib_info.pps t.info in
      let user_written_deps = Lib_info.user_written_deps t.info in
      let kind : Lib_deps_info.Kind.t =
        let enabled = Lib_info.enabled t.info in
        match enabled with
        | Normal -> Required
        | _ -> Optional
      in
      make_lib_deps_info ~user_written_deps ~pps ~kind
    in
    let requires_link = lazy (
      t.requires >>= closure_with_overlap_checks
                       db
                       ~linking:false
                       ~variants:None
    ) in
    { direct_requires   = t.requires
    ; requires_link
    ; resolved_selects  = t.resolved_selects
    ; pps               = t.pps
    ; lib_deps_info
    ; sub_systems       = t.sub_systems
    }

  let direct_requires   t = t.direct_requires
  let requires_link     t = t.requires_link
  let resolved_selects  t = t.resolved_selects
  let pps               t = t.pps
  let lib_deps_info     t = t.lib_deps_info
  let sub_systems t =
    Sub_system_name.Map.values t.sub_systems
    |> List.map ~f:(fun (lazy (Sub_system0.Instance.T ((module M), t))) ->
      M.T t)
end

(* Databases *)

module DB = struct
  module Resolve_result = struct
    type t = resolve_result =
      | Not_found
      | Found    of Lib_info.external_
      | Hidden   of Lib_info.external_ * string
      | Redirect of db option * Lib_name.t
  end

  type t = db

  let create ?parent ~stdlib_dir ~resolve ~all () =
    { parent
    ; resolve
    ; table  = Hashtbl.create 1024
    ; all    = Lazy.from_fun all
    ; stdlib_dir
    }

  let check_valid_external_variants libmap external_variants =
    List.iter external_variants ~f:(fun (ev : Dune_file.External_variant.t) ->
      match
        Option.map (Lib_name.Map.find libmap ev.virtual_lib) ~f:(fun res ->
          (* [res] is created by the code in
             [create_from_library_stanzas] bellow.  We know that it is
             either [Found] or [Redirect (_, name)] where [name] is in
             [libmap] for sure and maps to [Found _]. *)
          match res with
          | Not_found | Hidden _ -> assert false
          | Found x -> x
          | Redirect (_, name') ->
            match Lib_name.Map.find libmap name' with
            | Some (Found x) -> x
            | _ -> assert false)
      with
      | None ->
        User_error.raise ~loc:ev.loc
          [ Pp.textf "Virtual library %s hasn't been found in the project."
              (Lib_name.to_string ev.virtual_lib)
          ]
      | Some info ->
        begin match Lib_info.virtual_ info with
        | Some _ -> ()
        | None ->
          User_error.raise ~loc:ev.loc
            [ Pp.textf "Library %s isn't a virtual library."
                (Lib_name.to_string ev.virtual_lib)
            ]
        end)

  let error_two_impl_for_variant name variant (loc1, impl1) (loc2, impl2) =
    User_error.raise
      [ Pp.textf "Two implementations of %s have the same variant %S:"
          (Lib_name.Local.to_string name)
          (Variant.to_string variant)
      ; Pp.textf "- %s (%s)"
          (Lib_name.to_string impl1)
          (Loc.to_file_colon_line loc1)
      ; Pp.textf "- %s (%s)"
          (Lib_name.to_string impl2)
          (Loc.to_file_colon_line loc2)
      ]

  let create_from_library_stanzas ?parent ~lib_config lib_stanzas
        external_variant_stanzas =
    (* Construct a mapping from virtual library name to a list of
       [(variant, implementation_for_this_variant)]. We check a bit
       later that there is duplicate in the inner lists. *)
    let variant_map = Lib_name.Map.empty in
    let variant_map =
      (* Add entries from library stanzas *)
      List.fold_left lib_stanzas ~init:variant_map ~f:(fun acc (_, lib) ->
        match (lib : Dune_file.Library.t) with
        | { implements = Some (_, vlib)
          ; variant = Some variant
          ; buildable = { loc; _ }
          ; _ } ->
          Lib_name.Map.Multi.cons acc vlib
            (variant, (loc, Dune_file.Library.best_name lib))
        | _ -> acc)
    in
    let variant_map =
      (* Add entries from external_variant stanzas *)
      List.fold_left external_variant_stanzas ~init:variant_map
        ~f:(fun acc (ev : Dune_file.External_variant.t) ->
          Lib_name.Map.Multi.cons acc ev.virtual_lib
            (ev.variant, (ev.loc, ev.implementation)))
    in
    let map =
      List.concat_map lib_stanzas ~f:(fun (dir, (conf : Dune_file.Library.t)) ->
        (* In the [implements] field of library stanzas, the user
           might use either public or private library names. As a
           result, we have to lookup for implementations via both the
           public and private names. *)
        let variants_private =
          Lib_name.Map.find variant_map (Lib_name.of_local conf.name)
          |> Option.value ~default:[]
        in
        let variants =
          match conf.public with
          | None -> variants_private
          | Some { name = (_loc, name); _ } ->
            if Lib_name.equal name (Lib_name.of_local conf.name) then
              variants_private
            else
              match Lib_name.Map.find variant_map name with
              | None -> variants_private
              | Some variants_public ->
                List.rev_append variants_private variants_public
        in
        let known_implementations =
          match Variant.Map.of_list variants with
          | Ok x -> x
          | Error (variant, x, y) ->
            error_two_impl_for_variant (snd conf.name) variant x y
        in
        let info =
          Lib_info.of_library_stanza ~dir ~lib_config
            ~known_implementations conf
          |> Lib_info.of_local
        in
        match conf.public with
        | None ->
          [Dune_file.Library.best_name conf, Resolve_result.Found info]
        | Some p ->
          let name = Dune_file.Public_lib.name p in
          if Lib_name.equal name (Lib_name.of_local conf.name) then
            [name, Found info]
          else
            [ name                       , Found info
            ; Lib_name.of_local conf.name, Redirect (None, name)
            ])
      |> Lib_name.Map.of_list
      |> function
      | Ok x -> x
      | Error (name, _, _) ->
        match
          List.filter_map lib_stanzas
            ~f:(fun (_, (conf : Dune_file.Library.t)) ->
              if Lib_name.equal name (Lib_name.of_local conf.name) ||
                 match conf.public with
                 | None -> false
                 | Some p -> Lib_name.equal name (Dune_file.Public_lib.name p)
              then Some conf.buildable.loc
              else None)
        with
        | [] | [_] -> assert false
        | loc1 :: loc2 :: _ ->
          User_error.raise
            [ Pp.textf "Library %s is defined twice:" (Lib_name.to_string name)
            ; Pp.textf "- %s" (Loc.to_file_colon_line loc1)
            ; Pp.textf "- %s" (Loc.to_file_colon_line loc2)
            ]
    in
    (* We need to check that [external_variant] stanzas are correct,
       i.e. contain valid [virtual_library] fields now since this is
       the last time we analyse them. *)
    check_valid_external_variants map external_variant_stanzas;
    create () ?parent ~stdlib_dir:lib_config.stdlib_dir
      ~resolve:(fun name ->
        Lib_name.Map.find map name
        |> Option.value ~default:Not_found)
      ~all:(fun () -> Lib_name.Map.keys map)

  let create_from_findlib ?(external_lib_deps_mode=false)
        ~stdlib_dir findlib =
    create ()
      ~stdlib_dir
      ~resolve:(fun name ->
        match Findlib.find findlib name with
        | Ok pkg -> Found (Lib_info.of_dune_lib pkg)
        | Error e ->
          match e with
          | Not_found ->
            if external_lib_deps_mode then
              let pkg = Findlib.dummy_package findlib ~name in
              Found (Lib_info.of_dune_lib pkg)
            else
              Not_found
          | Hidden pkg ->
            Hidden (Lib_info.of_dune_lib pkg, "unsatisfied 'exist_if'"))
      ~all:(fun () ->
        Findlib.all_packages findlib
        |> List.map ~f:Dune_package.Lib.name)

  let find t name =
    match Resolve.find_internal t name ~stack:Dep_stack.empty with
    | St_initializing _ -> assert false
    | St_found t -> Some t
    | St_not_found | St_hidden _ -> None

  let find_even_when_hidden t name =
    match Resolve.find_internal t name ~stack:Dep_stack.empty with
    | St_initializing _ -> assert false
    | St_found t | St_hidden (t, _, _) -> Some t
    | St_not_found -> None

  let resolve t (loc, name) =
    match Resolve.find_internal t name ~stack:Dep_stack.empty with
    | St_initializing _ -> assert false
    | St_found t  -> Ok t
    | St_not_found -> Error.not_found ~loc ~name
    | St_hidden (_, dir, reason) -> Error.hidden ~loc ~name ~dir ~reason

  let available t name =
    Resolve.available_internal t name ~stack:Dep_stack.empty

  let get_compile_info t ?(allow_overlaps=false) name =
    match find_even_when_hidden t name with
    | None ->
      Code_error.raise "Lib.DB.get_compile_info got library that doesn't exist"
        [ "name", Lib_name.to_dyn name ]
    | Some lib ->
      let t = Option.some_if (not allow_overlaps) t in
      Compile.for_lib t lib

  let resolve_user_written_deps_for_exes t exes
        ?(allow_overlaps=false) deps ~pps ~variants =
    let lib_deps_info =
      Compile.make_lib_deps_info
        ~user_written_deps:deps
        ~pps
        ~kind:Required
    in
    let res, pps, resolved_selects =
      Resolve.resolve_user_deps t (Lib_info.Deps.of_lib_deps deps) ~pps
        ~stack:Dep_stack.empty ~allow_private_deps:true
    in
    let requires_link = lazy (
      res
      >>=
      closure_with_overlap_checks (Option.some_if (not allow_overlaps) t)
        ~linking:true
        ~variants
      |> Result.map_error ~f:(fun e ->
        Dep_path.prepend_exn e (Executables exes))
    ) in
    { Compile.
      direct_requires = res
    ; requires_link
    ; pps
    ; resolved_selects
    ; lib_deps_info
    ; sub_systems = Sub_system_name.Map.empty
    }

  let resolve_pps t pps =
    Resolve.resolve_simple_deps t ~allow_private_deps:true pps
      ~stack:Dep_stack.empty

  let rec all ?(recursive=false) t =
    let l =
      List.fold_left (Lazy.force t.all) ~f:(fun libs name ->
        match find t name with
        | Some x -> Set.add libs x
        | None -> libs) ~init:Set.empty
    in
    match recursive, t.parent with
    | true, Some t -> Set.union (all ~recursive t) l
    | _ -> l
end

(* META files *)

module Meta = struct
  let to_names ts =
    List.fold_left ts ~init:Lib_name.Set.empty ~f:(fun acc t ->
      Lib_name.Set.add acc t.name)

  (* For the deprecated method, we need to put all the runtime
     dependencies of the transitive closure.

     We need to do this because [ocamlfind ocamlc -package ppx_foo]
     will not look for the transitive dependencies of [foo], and the
     runtime dependencies might be attached to a dependency of [foo]
     rather than [foo] itself.

     Sigh... *)
  let ppx_runtime_deps_for_deprecated_method t =
    closure_exn [t] ~linking:false
    |> List.concat_map ~f:ppx_runtime_deps_exn
    |> to_names

  let requires         t = to_names (requires_exn         t)
  let ppx_runtime_deps t = to_names (ppx_runtime_deps_exn t)
end


let to_dune_lib ({ name ; info ; _ } as lib) ~modules ~foreign_objects
      ~dir =
  let add_loc =
    let loc = Lib_info.loc info in
    List.map ~f:(fun x -> (loc, x.name)) in
  let virtual_ = Option.is_some (Lib_info.virtual_ info) in
  let obj_dir =
    match Obj_dir.to_local (obj_dir lib) with
    | None -> assert false
    | Some obj_dir -> Obj_dir.convert_to_external ~dir obj_dir
  in
  let modules =
    let install_dir = Obj_dir.dir obj_dir in
    Modules.version_installed modules ~install_dir
  in
  let orig_src_dir =
    if !Clflags.store_orig_src_dir
    then Some (
      let orig_src_dir = Lib_info.orig_src_dir info in
      match orig_src_dir with
      | Some src_dir -> src_dir
      | None ->
        let src_dir = Lib_info.src_dir info in
        match Path.drop_build_context src_dir with
        | None -> src_dir
        | Some src_dir ->
          Path.(of_string (to_absolute_filename (Path.source src_dir)))
    )
    else None
  in
  let foreign_objects =
    match Lib_info.foreign_objects info with
    | External f -> f
    | Local -> foreign_objects
  in
  let loc = Lib_info.loc info in
  let synopsis = Lib_info.synopsis info in
  let archives = Lib_info.archives info in
  let plugins = Lib_info.plugins info in
  let modes = Lib_info.modes info in
  let kind = Lib_info.kind info in
  let version = Lib_info.version info in
  let jsoo_runtime = Lib_info.jsoo_runtime info in
  let special_builtin_support = Lib_info.special_builtin_support info in
  let known_implementations = Lib_info.known_implementations info in
  let foreign_archives = Lib_info.foreign_archives info in
  let use_public_name ~lib_field ~info_field =
    match (info_field, lib_field) with
    | Some _, None
     |None, Some _ ->
      assert false
    | None, None -> Ok None
    | Some (loc, _), Some field ->
      let open Result.O in
      let+ field = field in
      Some (loc, field.name)
  in
  let open Result.O in
  let* implements =
    use_public_name ~info_field:(Lib_info.implements info)
      ~lib_field:(implements lib)
  in
  let+ default_implementation =
    use_public_name
      ~info_field:(Lib_info.default_implementation info)
      ~lib_field:(Option.map ~f:Lazy.force lib.default_implementation)
  in
  Dune_package.Lib.make ~obj_dir ~orig_src_dir ~name ~loc ~kind ~synopsis
    ~version ~archives ~plugins ~foreign_archives ~foreign_objects
    ~jsoo_runtime
    ~requires:(add_loc (requires_exn lib))
    ~ppx_runtime_deps:(add_loc (ppx_runtime_deps_exn lib))
    ~modes ~implements ~known_implementations ~default_implementation ~virtual_
    ~modules:(Some modules)
    ~main_module_name:(Result.ok_exn (main_module_name lib))
    ~sub_systems:(Sub_system.dump_config lib)
    ~special_builtin_support

module Local : sig
  type t = private lib
  val of_lib : lib -> t option
  val of_lib_exn : lib -> t
  val to_lib : t -> lib
  val obj_dir : t -> Path.Build.t Obj_dir.t
  val info : t -> Path.Build.t Lib_info.t
  val to_dyn : t -> Dyn.t
  val equal : t -> t -> bool
  val hash : t -> int

  module Set : Stdune.Set.S with type elt = t
  module Map : Stdune.Map.S with type key = t

end = struct
  type nonrec t = t

  let to_lib t = t

  let of_lib (t : lib) = Option.some_if (is_local t) t

  let of_lib_exn t =
    match of_lib t with
    | Some l -> l
    | None -> Code_error.raise "Lib.Local.of_lib_exn"
                ["l", to_dyn t]

  let obj_dir t = Obj_dir.as_local_exn (Lib_info.obj_dir t.info)
  let info t = Lib_info.as_local_exn t.info

  module Set = Set
  module Map = Map

  let to_dyn = to_dyn
  let equal = equal
  let hash = hash
end
