<?php
/**
 * +-----------------------------------------------------------------------+
 * | Copyright (c) 2010, Till Klampaeckel                                  |
 * | All rights reserved.                                                  |
 * |                                                                       |
 * | Redistribution and use in source and binary forms, with or without    |
 * | modification, are permitted provided that the following conditions    |
 * | are met:                                                              |
 * |                                                                       |
 * | o Redistributions of source code must retain the above copyright      |
 * |   notice, this list of conditions and the following disclaimer.       |
 * | o Redistributions in binary form must reproduce the above copyright   |
 * |   notice, this list of conditions and the following disclaimer in the |
 * |   documentation and/or other materials provided with the distribution.|
 * | o The names of the authors may not be used to endorse or promote      |
 * |   products derived from this software without specific prior written  |
 * |   permission.                                                         |
 * |                                                                       |
 * | THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS   |
 * | "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT     |
 * | LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR |
 * | A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT  |
 * | OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, |
 * | SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT      |
 * | LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, |
 * | DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY |
 * | THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT   |
 * | (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE |
 * | OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.  |
 * |                                                                       |
 * +-----------------------------------------------------------------------+
 * | Author: Till Klampaeckel <till@php.net>                               |
 * +-----------------------------------------------------------------------+
 *
 * PHP version 5
 *
 * @category Services
 * @package  Services_UseKetchup
 * @author   Till Klampaeckel <till@php.net>
 * @license  http://www.opensource.org/licenses/bsd-license.php The BSD License
 * @version  GIT: $Id$
 * @link     http://github.com/till/Services_UseKetchup
 */

/**
 * Services_UseKetchup_Meetings
 *
 * @category Services
 * @package  Services_UseKetchup
 * @author   Till Klampaeckel <till@php.net>
 * @license  http://www.opensource.org/licenses/bsd-license.php The BSD License
 * @version  Release: 0.1.0
 * @link     http://github.com/till/Services_UseKetchup
 */
class Services_UseKetchup_Meetings extends Services_UseKetchup_Common
{
    /**
     * @var stdClass $lastCreated
     * @see self::add()
     * @see self::getLastCreated()
     */
    protected $lastCreated;

    /**
     * Add a new meeting.
     *
     * @param stdClass $meeting The meeting object.
     *
     * @return true
     */
    public function add(stdClass $meeting)
    {
        $data = json_encode($meeting);
        $resp = $this->makeRequest(
            '/meetings.json',
            HTTP_Request2::METHOD_POST,
            $data
        );
        $data = $this->parseResponse($resp);
        if ($data !== null) {
            $this->lastCreated = $data;
            return true;
        }
        return false;
    }

    /**
     * Delete a meeting.
     *
     * @param mixed $meeting Either a string, or stdClass.
     *                       If stdClass, it must contain a shortcode_url attribute.
     *
     * @return boolean
     */
    public function delete($meeting)
    {
        $id = $this->guessId($meeting);

        $resp = $this->makeRequest(
            "/meetings/{$id}.json",
            HTTP_Request2::METHOD_DELETE
        );
        $data = $this->parseResponse($resp);
        if ($data === 'Meeting Deleted Successfully') {
            return true;
        }
        return false;
    }

    /**
     * Get an ICS calendar of your meetings.
     *
     * @param boolean $lean Lean = true, is faster, but less verbose.
     *
     * @return string
     */
    public function ics($lean = false)
    {
        return $this->show($lean, true);
    }

    /**
     * Get all previous meetings.
     *
     * @param boolean $lean Lean = true, is faster, but less verbose.
     *
     * @return array An array stacked with stdClass.
     */
    public function previous($lean = false)
    {
        return $this->show($lean, false, '/meetings/previous.');
    }

    /**
     * Get meetings.
     *
     * @param boolean $lean Lean = true, is faster, but less verbose.
     * @param boolean $ics  If you'd like an ICS instead (see {@link self::ics()}.
     * @param string  $url  Optional, for previous, upcoming, todays, etc..
     *
     * @return string|array String, if $ics = true, otherwise an array stacked with
     *                      stdClass.
     */
    public function show($lean = false, $ics = false, $url = '/meetings.')
    {
        if ($ics === false) {
            $url .= 'json';
        } else {
            $url .= 'ics';
        }
        if ($lean === true) {
            $url .= '?lean=true';
        }

        $resp = $this->makeRequest($url);
        $data = $this->parseResponse($resp);
        return $data;
    }

    /**
     * Get todays meetings.
     *
     * @param boolean $lean Lean = true, is faster, but less verbose.
     *
     * @return array An array stacked with stdClass.
     */
    public function todays($lean = false)
    {
        return $this->show($lean, false, '/meetings/todays.');
    }

    /**
     * Get all upcoming meetings.
     *
     * @param boolean $lean Lean = true, is faster, but less verbose.
     *
     * @return array An array stacked with stdClass.
     */
    public function upcoming($lean = false)
    {
        return $this->show($lean, false, '/meetings/upcoming.');
    }

    /**
     * Update a meeting.
     *
     * @param stdClass $meeting The meeting object.
     *
     * @return boolean
     */
    public function update(stdClass $meeting)
    {
        if (!isset($meeting->shortcode_url)) {
            throw InvalidArgumentException(
                "Meeting object must contain attribute shortcode_url"
            );
        }
        $id = $meeting->shortcode_url;

        $data = json_encode($meeting);
        $resp = $this->makeRequest(
            "/meetings/{$id}.json",
            HTTP_Request2::METHOD_PUT,
            $data
        );
        $data = $this->parseResponse($resp);
        if ($data !== null) {
            return true;
        }
        return false;
    }

    /**
     * Get last created meeting.
     *
     * @return stdClass
     * @uses   self::$lastCreated
     * @see    self::add()
     * @throws RuntimeException When called prior to add().
     */
    public function getLastCreated()
    {
        if ($this->lastCreated === null) {
            throw new RuntimeException("You need to call add() first.");
        }
        return $this->lastCreated;
    }
}