/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.font.TextLayout;
import java.beans.PropertyEditorSupport;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * A PropertyEditor that brings up a JFileChooser panel to select a
 * file.
 */
public abstract class EditorSpawner extends PropertyEditorSupport {
	
	private static String nameOfNull = "null";
	
	private Box panel;
	private JButton button;
	JButton nullButton;
	private Object storedValue;
	private boolean nullAllowed;
	
	private static String nameOfCreation = "create ...";
	
	public EditorSpawner(boolean nullAllowed) {
		this.nullAllowed = nullAllowed;
		panel = new Box(BoxLayout.X_AXIS);
		button = new JButton();
		button.setMaximumSize(new Dimension(32000,25));
		button.setPreferredSize(new Dimension(50,25));
		button.setMargin(new Insets(2,2,2,2));
		button.setIcon(icon);
		button.setFocusable(false);
		button.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				spawnDialog();
			}
		});
		panel.add(button);
		if (nullAllowed) {
			nullButton = new JButton(nameOfNull);
			nullButton.setMargin(new Insets(2,2,2,2));
			nullButton.setMaximumSize(new Dimension(60,25));
			nullButton.setForeground(Color.red);
			nullButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent arg0) {
					setValue(null);
				}
			});
		}
	}
	
	private void spawnDialog() {
		storedValue = getValue();
		AbstractDialog dialog = getDialog();
		dialog.setDefaultCloseOperation(AbstractDialog.HIDE_ON_CLOSE);
		dialog.setChangeListener(
				new ChangeListener() {
					public void stateChanged(ChangeEvent arg0) {
						setValueFromDialog();
					}
				});
		dialog.setOkListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				setValue(getDialog().getValue());
			}
		});
		dialog.setCancelListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				setValue(storedValue);
			}
		});
		dialog.setValue(getValue());
		dialog.show(button);
	}
	
	private void setValueFromDialog() {
		Object o = getDialog().getValue();
		if (nullAllowed) {
			if (o != null && getValue() == null) {
				panel.add(nullButton);
				panel.validate();
			}
			if (o == null && getValue() != null) {
				panel.remove(nullButton);
				panel.validate();
			}
		}
		button.repaint();
		super.setValue(getDialog().getValue());
	}
	
	public void setValue(Object o) {
		if (nullAllowed) {
			if (o != null && getValue() == null) {
				panel.add(nullButton);
				panel.validate();
			}
			if (o == null && getValue() != null) {
				panel.remove(nullButton);
				panel.validate();
			}
		}
		button.repaint();
		super.setValue(o);
	}
	
	/**
	 * PropertyEditor interface.
	 * 
	 * @return true
	 */
	public boolean supportsCustomEditor() {
		return true;
	}
	
	/**
	 * Returns a JButton that will bring up a dialog.
	 * 
	 * @return JButton button
	 */
	public Component getCustomEditor() {
		return panel;
	}
	
	public abstract AbstractDialog getDialog();

	
	public int getIconWidth() {
		return 32;
	}
	
	public int getIconHeight() {
		return 16;
	}
	
	protected abstract void paintEditedObject(
			Component cmp,
			Graphics g,
			int x,
			int y
	);
	
	protected void paintText(Font font, String text, Component cmp, Graphics g, int x, int y) {
		Graphics2D g2d = (Graphics2D)g;
		g.setColor(Color.black);
		if (font == null) font = button.getFont();
		TextLayout tl = new TextLayout(text,font,g2d.getFontRenderContext());
		Rectangle r = tl.getBounds().getBounds();
		
		int height = r.height;
		int width = r.width;
		
		int w=(getIconWidth()-width)/2;
		int h=(getIconHeight()-height)/2;
		g2d.setFont(font);
		g2d.drawString(text,x+w,y+height+h);
	}
	
	private Icon icon = new Icon() {
		
		public void paintIcon(Component cmp, Graphics g, int x, int y) {
			if (getValue() != null) {
				
				paintEditedObject(cmp, g, x, y);
			} else {
				paintText(null, nameOfCreation, cmp, g, x, y);
			}
		}
		
		public int getIconWidth() {
			return EditorSpawner.this.getIconWidth();
		}
		
		public int getIconHeight() {
			return EditorSpawner.this.getIconHeight();
		}
	};
	
	protected void repaintIcon() {
		button.repaint();
	}
	
	public static String getNameOfNull() {
		return nameOfNull;
	}

	public static void setNameOfNull(String nameOfNull) {
		EditorSpawner.nameOfNull = nameOfNull;
	}

	public static String getNameOfCreation() {
		return nameOfCreation;
	}

	public static void setNameOfCreation(String nameOfCreation) {
		EditorSpawner.nameOfCreation = nameOfCreation;
	}
}