/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.StringTokenizer;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class StrokeDialog extends AbstractDialog implements PropertyChangeListener, ActionListener, ChangeListener {
	
	private static final long serialVersionUID = -5046164637730760157L;
	public static StrokeDialog instance = null;
	boolean updating = false;
	private BasicStroke stroke;
	
	private StrokeDialog(String title, BasicStroke stroke) {
		super();
		setTitle(title);
		Box chooserPanel = new Box(BoxLayout.Y_AXIS);
		JPanel chooser = new JPanel(new GridBagLayout());
		chooser.setPreferredSize(new Dimension(300,150));
		if (stroke == null) stroke=new BasicStroke();
		this.stroke=stroke;
		
		GridBagConstraints label=new GridBagConstraints();
		label.anchor=GridBagConstraints.EAST;
		label.ipadx=3;
		GridBagConstraints editor=new GridBagConstraints();
		editor.fill=GridBagConstraints.BOTH;
		editor.anchor=GridBagConstraints.CENTER;
		editor.weightx=1;
		editor.gridwidth=GridBagConstraints.REMAINDER;
		GridBagConstraints preview=new GridBagConstraints();
		preview.gridwidth=GridBagConstraints.REMAINDER;
		preview.gridheight=GridBagConstraints.REMAINDER;
		preview.fill = GridBagConstraints.BOTH;
		
		
		// lineWidth
		widthSP.addChangeListener(this);
		chooser.add(new JLabel("line width  "),label);
		chooser.add(widthSP,editor);
		
		// endCap
		chooser.add(new JLabel("end cap  "),label);
		cap.addPropertyChangeListener(this);
		chooser.add(cap.getCustomEditor(),editor);
		
		// lineJoin
		chooser.add(new JLabel("line join  "),label);
		join.addPropertyChangeListener(this);
		chooser.add(join.getCustomEditor(),editor);
		
		// miterLimit
		miterSP.addChangeListener(this);
		chooser.add(new JLabel("miter limit  "),label);
		chooser.add(miterSP,editor);
		
		// dashArray
		chooser.add(new JLabel("dash array  "),label);
		dashArrayTF = new JTextField();
		dashArrayTF.addActionListener(this);
		chooser.add(dashArrayTF, editor);
		
		// dashPhase
		chooser.add(new JLabel("dash phase  "),label);
		dashPhaseSP.addChangeListener(this);
		dashPhaseSP.setEnabled(stroke.getDashArray()!=null && stroke.getDashArray().length>0);
		chooser.add(dashPhaseSP,editor);
		
		JPanel previewPanel = new JPanel() {
			private static final long serialVersionUID = 1L;
			public void paintComponent(Graphics g) {
				super.paintComponent(g);
				paintStroke((Graphics2D)g);
			}
		};
		previewPanel.setPreferredSize(new Dimension(300,200));
		previewPanel.setBackground(Color.white);
		
		chooserPanel.add(chooser);
		chooserPanel.add(previewPanel,preview);
		setMainComponent(chooserPanel);
		setValue(stroke);
		pack();
	}
	
	public static StrokeDialog sharedInstance() {
		if( instance == null ) {
			instance = new StrokeDialog("Choose Stroke",new BasicStroke());
		}
		return instance; 
	}
	
	public BasicStroke getValue() {
		return stroke;
	}
	
	
	private JSpinner widthSP= new JSpinner(
			new SpinnerNumberModel(
					0.,
					0.,
					Double.MAX_VALUE,
					1.
			)
	);
	
	private JSpinner miterSP= new JSpinner(
			new SpinnerNumberModel(
					0.,
					0.,
					Double.MAX_VALUE,
					1.
			)
	);
	
	private JSpinner dashPhaseSP= new JSpinner(
			new SpinnerNumberModel(
					0.,
					0.,
					Double.MAX_VALUE,
					1.
			)
	);
	
	private ChoiceEditor join = new ChoiceEditor() {
		protected void defineValuesToStrings() {
			valuesToStrings.put(new Integer(BasicStroke.JOIN_BEVEL),"bevel");
			valuesToStrings.put(new Integer(BasicStroke.JOIN_MITER),"miter");
			valuesToStrings.put(new Integer(BasicStroke.JOIN_ROUND),"round");
		}
	};
	
	ChoiceEditor cap = new ChoiceEditor() {
		protected void defineValuesToStrings() {
			valuesToStrings.put(new Integer(BasicStroke.CAP_BUTT),"butt");
			valuesToStrings.put(new Integer(BasicStroke.CAP_ROUND),"round");
			valuesToStrings.put(new Integer(BasicStroke.CAP_SQUARE),"square");
		}
	};
	
	private JTextField dashArrayTF;
	
	public void paintStroke(Graphics2D g2) {
		g2.setStroke(stroke);
		g2.drawPolyline(
				new int[]{50, 250, 50, 250},
				new int[]{50, 50, 150, 150},
				4
		);
	}
	
	private String dashArrayToString(float [] dashArray) {
		String s = "";
		if (dashArray==null) return s;
		for (int i=0;i<dashArray.length-1;i++) s+=dashArray[i]+",";
		if (dashArray.length>0) s+=dashArray[dashArray.length-1];
		return s;
	}
	
	public void setValue(Object s) {
		updating = true;
		if (s instanceof BasicStroke) {
			stroke = (BasicStroke)s;
		} else {
			stroke = new BasicStroke();
		}
		widthSP.setValue(new Double(stroke.getLineWidth()));
		miterSP.setValue(new Double(stroke.getMiterLimit()));
		dashArrayTF.setText(dashArrayToString(stroke.getDashArray()));
		cap.setValue(new Integer(stroke.getEndCap()));
		join.setValue(new Integer(stroke.getLineJoin()));
		updating = false;
	}
	
	public void stateChanged(ChangeEvent e) {
		if (!updating) {
			Object s = e.getSource();
			if (s==widthSP) {
				float w = ((Double)widthSP.getValue()).floatValue();
				stroke = new BasicStroke(w, stroke.getEndCap(), stroke.getLineJoin(), stroke.getMiterLimit(),
						stroke.getDashArray(), stroke.getDashPhase());
			} else if (s==miterSP) {
				float w = ((Double)miterSP.getValue()).floatValue();
				stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(), stroke.getLineJoin(), w,
						stroke.getDashArray(), stroke.getDashPhase());
			} else if (s==dashPhaseSP) {
				float w = ((Double)dashPhaseSP.getValue()).floatValue();
				stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(), stroke.getLineJoin(),
						stroke.getMiterLimit(), stroke.getDashArray(), w);
			}
			repaint();
			fireStateChanged();
		}
	}
	
	public void actionPerformed(ActionEvent e) {
		if (!updating) {
			String daS = dashArrayTF.getText().trim();
			String cont = "";
			float [] da = null;
			if (daS.length()>0)
			{
				StringTokenizer st = new StringTokenizer(daS, ",");
				if (st.countTokens()>0)
				{
					da = new float[st.countTokens()];
					for (int i=0;i<da.length;i++) da[i]=Float.parseFloat(st.nextToken());
					for (int i=0;i<da.length-1;i++) cont+=da[i]+",";
					cont+=da[da.length-1];
				}
			}
			dashArrayTF.setText(cont);
			stroke = new BasicStroke(stroke.getLineWidth(), stroke.getEndCap(), stroke.getLineJoin(),
					stroke.getMiterLimit(), da, stroke.getDashPhase());
			dashPhaseSP.setEnabled(cont.length()!=0);
			repaint();
			fireStateChanged();
		}
	}
	
	public void propertyChange(PropertyChangeEvent e) {
		if (!updating) {
			Object source = e.getSource();
			if (source==cap) {
				stroke = new BasicStroke(
						stroke.getLineWidth(),
						((Integer)cap.getValue()).intValue(),
						stroke.getLineJoin(),
						stroke.getMiterLimit(),
						stroke.getDashArray(),
						stroke.getDashPhase()
				);
			} else if (source == join) {
				stroke = new BasicStroke(
						stroke.getLineWidth(),
						stroke.getEndCap(),
						((Integer)join.getValue()).intValue(),
						stroke.getMiterLimit(),
						stroke.getDashArray(),
						stroke.getDashPhase()
				);
			}
			repaint();
			fireStateChanged();
		}
	}
}