/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class SimpleColorChooser extends JPanel implements ChangeListener {
	
	private static final long serialVersionUID = -6452509893745135407L;
	//private static final String text="Black";
	//private static final Font font=new Font("Sans Serif", Font.PLAIN, 30);

	private JSlider redSlider;
	private JSlider greenSlider;
	private JSlider blueSlider;
	private JSlider alphaSlider;
	//private JPanel previewPanel;
	private Color color;
	private List<ChangeListener> changeListeners=new CopyOnWriteArrayList<ChangeListener>();
	private boolean updating = false;
	private boolean withAlpha;
	private JButton closeButton;

	public SimpleColorChooser() {
		this(Color.white, false);
	}
	
	public SimpleColorChooser(Color color, boolean withAlpha) {
		super(new BorderLayout());
		this.color = color;
		this.withAlpha = withAlpha;
		// sliders
		Box sliderBox = new Box(BoxLayout.Y_AXIS);
		
		redSlider = makeSlider(color.getRed(), "red");
		sliderBox.add(makeSliderBox(redSlider, color.getRed(), "red"));
		greenSlider = makeSlider(color.getGreen(), "green");
		sliderBox.add(makeSliderBox(greenSlider, color.getGreen(), "green"));
		blueSlider = makeSlider(color.getBlue(), "blue");
		sliderBox.add(makeSliderBox(blueSlider, color.getBlue(), "blue"));
		if (withAlpha) {
			alphaSlider = makeSlider(color.getAlpha(), "alpha");
			sliderBox.add(makeSliderBox(alphaSlider, color.getAlpha(), "alpha"));
		}
		add(sliderBox, BorderLayout.CENTER);
		JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		buttonPanel.setBorder(new EmptyBorder(10, 0, 0, 0));
		
		closeButton = new JButton("Close");
		buttonPanel.add(closeButton);
		add("South", buttonPanel);

//		previewPanel = new JPanel() {
//			public void paint(Graphics g) {
//				Graphics2D g2d = (Graphics2D) g;
//				int w0 = previewPanel.getWidth();
//				int h0 = previewPanel.getHeight();
//				Color col = getColor();
//				if (col != null) {
//					g.setColor(Color.black);
//
//					TextLayout tl = new TextLayout(text,font,g2d.getFontRenderContext());
//					Rectangle r = tl.getBounds().getBounds();
//
//					int height = r.height;
//					int width = r.width;
//					
//					g2d.setFont(font);
//
//					int w=(w0-width)/2;
//					int h=(h0-height)/2;
//					g2d.drawString(text,w,height+h);
//
//					g.setColor(col);
//					g.fillRect(0,0,w0, h0);
//				}
//			}
//
//		};
//		previewPanel.setPreferredSize(new Dimension(350,50));
//		add(previewPanel, BorderLayout.SOUTH);
	}
	
	public void addActionListener(ActionListener l) {
		closeButton.addActionListener(l);
	}
	
	public void removeActionListener(ActionListener l) {
		closeButton.removeActionListener(l);
	}

	private JSlider makeSlider(int value, String text) {
		final JSlider slider = new JSlider(JSlider.HORIZONTAL, 0, 255, value);
		slider.setMajorTickSpacing(64);
		slider.setMinorTickSpacing(8);
		slider.setPaintTicks(true);
		slider.setPaintLabels(false);
		slider.addChangeListener(this);
		return slider;
	}

	private JComponent makeSliderBox(final JSlider slider, int value, String text) {
		Box sliderBox = new Box(BoxLayout.X_AXIS);
		sliderBox.add(slider);
		final JLabel label = new JLabel(String.valueOf(value));
		label.setForeground(Color.red);
		label.setPreferredSize(new Dimension(40,20));
		label.setMinimumSize(new Dimension(40,20));
		label.setBorder(new EmptyBorder(0,10,0,0));
		slider.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				label.setText(String.valueOf(slider.getValue()));
			}
		});
		sliderBox.add(label);
		sliderBox.setBorder(BorderFactory.createTitledBorder(text));
		return sliderBox;
	}
	
	public void stateChanged(ChangeEvent e) {
		
		if (!updating) {
			color = new Color(
					redSlider.getValue(),
					greenSlider.getValue(),
					blueSlider.getValue(),
					withAlpha ? alphaSlider.getValue() : 255
			);
			//previewPanel.repaint();
			for (ChangeListener changeListener : changeListeners) {
				changeListener.stateChanged(new ChangeEvent(this));
			}
		}
	}

	public void addChangeListener(ChangeListener listener) {
		changeListeners.add(listener);
	}

	public void removeChangeListener(ChangeListener listener) {
		changeListeners.remove(listener);
	}

	public void setColor(Color c) {
		if (!color.equals(c)) {
			updating = true;
			redSlider.setValue( c.getRed() );
			greenSlider.setValue( c.getGreen() );
			blueSlider.setValue( c.getBlue() );
			if (withAlpha) alphaSlider.setValue( c.getAlpha() );
			updating = false;
			color = c;
			//previewPanel.repaint();
		}
	}

	public Color getColor() {
		return color;
	}
	
	public static void main(String[] args) {
		JFrame f = new JFrame("SimpleColorChooser");
		SimpleColorChooser scc = new SimpleColorChooser();
		f.getContentPane().add(scc);
		f.pack();
		f.setVisible(true);
	}
}
