# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cleo',
 'cleo.commands',
 'cleo.commands.completions',
 'cleo.config',
 'cleo.io',
 'cleo.testers']

package_data = \
{'': ['*']}

install_requires = \
['clikit>=0.4.0,<0.5.0']

setup_kwargs = {
    'name': 'cleo',
    'version': '0.7.6',
    'description': 'Cleo allows you to create beautiful and testable command-line interfaces.',
    'long_description': 'Cleo\n####\n\n.. image:: https://travis-ci.org/sdispater/cleo.png\n   :alt: Cleo Build status\n   :target: https://travis-ci.org/sdispater/cleo\n\nCreate beautiful and testable command-line interfaces.\n\nCleo is mostly a higher level wrapper for `CliKit <https://github.com/sdispater/clikit>`_, so\na lot of the components and utilities comes from it. Refer to its documentation for more\ninformation.\n\nResources\n=========\n\n* `Documentation <http://cleo.readthedocs.io>`_\n* `Issue Tracker <https://github.com/sdispater/cleo/issues>`_\n\n\nUsage\n=====\n\nTo make a command that greets you from the command line,\ncreate ``greet_command.py`` and add the following to it:\n\n.. code-block:: python\n\n    from cleo import Command\n\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {name? : Who do you want to greet?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n        """\n\n        def handle(self):\n            name = self.argument(\'name\')\n\n            if name:\n                text = \'Hello {}\'.format(name)\n            else:\n                text = \'Hello\'\n\n            if self.option(\'yell\'):\n                text = text.upper()\n\n            self.line(text)\n\n\nYou also need to create the file to run at the command line which creates\nan ``Application`` and adds commands to it:\n\n.. code-block:: python\n\n    #!/usr/bin/env python\n\n    from greet_command import GreetCommand\n    from cleo import Application\n\n    application = Application()\n    application.add(GreetCommand())\n\n    if __name__ == \'__main__\':\n        application.run()\n\nTest the new command by running the following\n\n.. code-block:: bash\n\n    $ python application.py greet John\n\nThis will print the following to the command line:\n\n.. code-block:: text\n\n    Hello John\n\nYou can also use the ``--yell`` option to make everything uppercase:\n\n.. code-block:: bash\n\n    $ python application.py greet John --yell\n\nThis prints:\n\n.. code-block:: text\n\n    HELLO JOHN\n\nAs you may have already seen, Cleo uses the command docstring to determine\nthe command definition.\nThe docstring must be in the following form :\n\n.. code-block:: python\n\n    """\n    Command description\n\n    Command signature\n    """\n\nThe signature being in the following form:\n\n.. code-block:: python\n\n    """\n    command:name {argument : Argument description} {--option : Option description}\n    """\n\nThe signature can span multiple lines.\n\n.. code-block:: python\n\n    """\n    command:name\n        {argument : Argument description}\n        {--option : Option description}\n    """\n\nColoring the Output\n-------------------\n\nWhenever you output text, you can surround the text with tags to color its\noutput. For example:\n\n.. code-block:: python\n\n    # green text\n    self.line(\'<info>foo</info>\')\n\n    # yellow text\n    self.line(\'<comment>foo</comment>\')\n\n    # black text on a cyan background\n    self.line(\'<question>foo</question>\')\n\n    # white text on a red background\n    self.line(\'<error>foo</error>\')\n\nThe closing tag can be replaced by ``</>``, which revokes all formatting options established by the last opened tag.\n\nIt is possible to define your own styles using the ``add_style()`` method:\n\n.. code-block:: python\n\n    self.add_style(\'fire\', fg=\'red\', bg=\'yellow\', options=[\'bold\', \'blink\'])\n    self.line(\'<fire>foo</fire>\')\n\nAvailable foreground and background colors are: ``black``, ``red``, ``green``,\n``yellow``, ``blue``, ``magenta``, ``cyan`` and ``white``.\n\nAnd available options are: ``bold``, ``underscore``, ``blink``, ``reverse`` and ``conceal``.\n\nYou can also set these colors and options inside the tag name:\n\n.. code-block:: python\n\n    # green text\n    self.line(\'<fg=green>foo</>\')\n\n    # black text on a cyan background\n    self.line(\'<fg=black;bg=cyan>foo</>\')\n\n    # bold text on a yellow background\n    self.line(\'<bg=yellow;options=bold>foo</>\')\n\n\nVerbosity Levels\n----------------\n\nCleo has four verbosity levels. These are defined in the ``Output`` class:\n\n=======================================  ================================== ======================\nMode                                     Meaning                            Console option\n=======================================  ================================== ======================\n``NA``                                   Do not output any messages         ``-q`` or ``--quiet``\n``clikit.VERBOSITY_NORMAL``              The default verbosity level        (none)\n``clikit.VERBOSITY_VERBOSE``             Increased verbosity of messages    ``-v``\n``clikit.VERBOSITY_VERY_VERBOSE``        Informative non essential messages ``-vv``\n``clikit.VERBOSITY_DEBUG``               Debug messages                     ``-vvv``\n=======================================  ================================== ======================\n\nIt is possible to print a message in a command for only a specific verbosity\nlevel. For example:\n\n.. code-block:: python\n\n    if clikit.VERBOSITY_VERBOSE <= self.io.verbosity:\n        self.line(...)\n\nThere are also more semantic methods you can use to test for each of the\nverbosity levels:\n\n.. code-block:: python\n\n    if self.output.is_quiet():\n        # ...\n\n    if self.output.is_verbose():\n        # ...\n\nYou can also pass the verbosity flag directly to `line()`.\n\n.. code-block:: python\n\n    self.line("", verbosity=clikit.VERBOSITY_VERBOSE)\n\nWhen the quiet level is used, all output is suppressed.\n\n\nUsing Arguments\n---------------\n\nThe most interesting part of the commands are the arguments and options that\nyou can make available. Arguments are the strings - separated by spaces - that\ncome after the command name itself. They are ordered, and can be optional\nor required. For example, add an optional ``last_name`` argument to the command\nand make the ``name`` argument required:\n\n.. code-block:: python\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {name : Who do you want to greet?}\n            {last_name? : Your last name?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n        """\n\nYou now have access to a ``last_name`` argument in your command:\n\n.. code-block:: python\n\n    last_name = self.argument(\'last_name\')\n    if last_name:\n        text += \' {}\'.format(last_name)\n\nThe command can now be used in either of the following ways:\n\n.. code-block:: bash\n\n    $ python application.py greet John\n    $ python application.py greet John Doe\n\nIt is also possible to let an argument take a list of values (imagine you want\nto greet all your friends). For this it must be specified at the end of the\nargument list:\n\n.. code-block:: python\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {names* : Who do you want to greet?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n        """\n\nTo use this, just specify as many names as you want:\n\n.. code-block:: bash\n\n    $ python application.py demo:greet John Jane\n\nYou can access the ``names`` argument as a list:\n\n.. code-block:: python\n\n    names = self.argument(\'names\')\n    if names:\n        text += \' {}\'.format(\', \'.join(names))\n\nThere are 3 argument variants you can use:\n\n================================ ==================================== ===============================================================================================================\nMode                             Notation                             Value\n================================ ==================================== ===============================================================================================================\n``clikit.ARGUMENT_REQUIRED``     none (just write the argument name)  The argument is required\n``clikit.ARGUMENT_OPTIONAL``     ``argument?``                        The argument is optional and therefore can be omitted\n``clikit.ARGUMENT_MULTI_VALUED`` ``argument*``                        The argument can contain an indefinite number of arguments and must be used at the end of the argument list\n================================ ==================================== ===============================================================================================================\n\nYou can combine them like this:\n\n.. code-block:: python\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {names?* : Who do you want to greet?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n        """\n\nIf you want to set a default value, you can it like so:\n\n.. code-block:: text\n\n    argument=default\n\nThe argument will then be considered optional.\n\n\nUsing Options\n-------------\n\nUnlike arguments, options are not ordered (meaning you can specify them in any\norder) and are specified with two dashes (e.g. ``--yell`` - you can also\ndeclare a one-letter shortcut that you can call with a single dash like\n``-y``). Options are *always* optional, and can be setup to accept a value\n(e.g. ``--dir=src``) or simply as a boolean flag without a value (e.g.\n``--yell``).\n\n.. tip::\n\n    It is also possible to make an option *optionally* accept a value (so that\n    ``--yell`` or ``--yell=loud`` work). Options can also be configured to\n    accept a list of values.\n\nFor example, add a new option to the command that can be used to specify\nhow many times in a row the message should be printed:\n\n.. code-block:: python\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {name? : Who do you want to greet?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n            {--iterations=1 : How many times should the message be printed?}\n        """\n\n\nNext, use this in the command to print the message multiple times:\n\n.. code-block:: python\n\n    for _ in range(0, self.option(\'iterations\')):\n        self.line(text)\n\nNow, when you run the task, you can optionally specify a ``--iterations``\nflag:\n\n.. code-block:: bash\n\n    $ python application.py demo:greet John\n    $ python application.py demo:greet John --iterations=5\n\nThe first example will only print once, since ``iterations`` is empty and\ndefaults to ``1``. The second example will print five times.\n\nRecall that options don\'t care about their order. So, either of the following\nwill work:\n\n.. code-block:: bash\n\n    $ python application.py demo:greet John --iterations=5 --yell\n    $ python application.py demo:greet John --yell --iterations=5\n\nThere are 4 option variants you can use:\n\n================================  =================================== ======================================================================================\nOption                            Notation                            Value\n================================  =================================== ======================================================================================\n``clikit.OPTION_MULTI_VALUED``    ``--option=*``                      This option accepts multiple values (e.g. ``--dir=/foo --dir=/bar``)\n``clikit.OPTION_NO_VALUE``        ``--option``                        Do not accept input for this option (e.g. ``--yell``)\n``clikit.OPTION_REQUIRED_VALUE``  ``--option=``                       This value is required (e.g. ``--iterations=5``), the option itself is still optional\n``clikit.OPTION_OPTIONAL_VALUE``  ``--option=?``                      This option may or may not have a value (e.g. ``--yell`` or ``--yell=loud``)\n================================  =================================== ======================================================================================\n\nYou can combine them like this:\n\n.. code-block:: python\n\n    class GreetCommand(Command):\n        """\n        Greets someone\n\n        greet\n            {name? : Who do you want to greet?}\n            {--y|yell : If set, the task will yell in uppercase letters}\n            {--iterations=?*1 : How many times should the message be printed?}\n        """\n\n\nTesting Commands\n----------------\n\nCleo provides several tools to help you test your commands. The most\nuseful one is the ``CommandTester`` class.\nIt uses a special IO class to ease testing without a real\nconsole:\n\n.. code-block:: python\n\n    import pytest\n\n    from cleo import Application\n    from cleo import CommandTester\n\n    def test_execute(self):\n        application = Application()\n        application.add(GreetCommand())\n\n        command = application.find(\'demo:greet\')\n        command_tester = CommandTester(command)\n        command_tester.execute()\n\n        assert "..." == tester.io.fetch_output()\n\nThe ``CommandTester.io.fetch_output()`` method returns what would have been displayed\nduring a normal call from the console. ``CommandTester.io.fetch_error()`` is also available\nto get what you have been written to the stderr.\n\nYou can test sending arguments and options to the command by passing them\nas a string to the ``CommandTester.execute()`` method:\n\n.. code-block:: python\n\n    import pytest\n\n    from cleo import Application\n    from cleo import CommandTester\n\n    def test_execute(self):\n        application = Application()\n        application.add(GreetCommand())\n\n        command = application.find(\'demo:greet\')\n        command_tester = CommandTester(command)\n        command_tester.execute("John")\n\n        assert "John" in tester.io.fetch_output()\n\nYou can also test a whole console application by using the ``ApplicationTester`` class.\n\n\nCalling an existing Command\n---------------------------\n\nIf a command depends on another one being run before it, instead of asking the\nuser to remember the order of execution, you can call it directly yourself.\nThis is also useful if you want to create a "meta" command that just runs a\nbunch of other commands.\n\nCalling a command from another one is straightforward:\n\n.. code-block:: python\n\n    def handle(self):\n        return_code = self.call(\'demo:greet\', "John --yell")\n\n        # ...\n\nIf you want to suppress the output of the executed command,\nyou can use the ``call_silent()`` method instead.\n\n\n\nAutocompletion\n--------------\n\nCleo supports automatic (tab) completion in ``bash``, ``zsh`` and ``fish``.\n\nTo activate support for autocompletion, pass a ``complete`` keyword when initializing\nyour application:\n\n.. code-block:: python\n\n    application = Application(\'My Application\', \'0.1\', complete=True)\n\nNow, register completion for your application by running one of the following in a terminal,\nreplacing ``[program]`` with the command you use to run your application:\n\n.. code-block:: bash\n\n    # BASH - Ubuntu / Debian\n    [program] completions bash | sudo tee /etc/bash_completion.d/[program].bash-completion\n\n    # BASH - Mac OSX (with Homebrew "bash-completion")\n    [program] completions bash > $(brew --prefix)/etc/bash_completion.d/[program].bash-completion\n\n    # ZSH - Config file\n    mkdir ~/.zfunc\n    echo "fpath+=~/.zfunc" >> ~/.zshrc\n    [program] completions zsh > ~/.zfunc/_test\n\n    # FISH\n    [program] completions fish > ~/.config/fish/completions/[program].fish\n',
    'author': 'Sébastien Eustace',
    'author_email': 'sebastien@eustace.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/sdispater/cleo',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*',
}


setup(**setup_kwargs)
