/**********************************************************************
 * $fill_mem example -- C source code using TF/ACC PLI routines
 *
 * C source to illustrate writing 4-state logic values to all
 * addresses of a Verilog memory array.
 *
 * The logic values are represented as characters in the C language. 
 * Every eight bits of a Verilog memory word are passed to/from 
 * Verilog as a C structure containing a pair of C bytes (of type char),
 * using an aval/bval encoding, where:
 * 0/0 = logic 0, 1/0 = logic 1, 0/1 = logic Z, 1/1 = logic X.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage:
 * ------
 *   Syntax:  $fill_mem(<memory_word_select>, <word_select_address>);
 *
 *   Example:
 *    reg [23:0] RAM [0:3];
 *    initial $fill_mem(RAM[2], 2);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_FillMem_checktf(),
 *              PLIbook_FillMem_calltf();
 *  /* table entries -/
 *   {usertask,                         /* type of PLI routine -/
 *     0,                               /* user_data value -/
 *     PLIbook_FillMem_checktf,         /* checktf routine -/
 *     0,                               /* sizetf routine -/
 *     PLIbook_FillMem_calltf,          /* calltf routine -/
 *     PLIbook_FillMem_misctf,          /* misctf routine -/
 *     "$fill_mem",                     /* system task/function name -/
 *     1                                /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_FillMem_checktf()
{
  if (tf_nump() != 2)
    tf_error("Usage: $fill_mem(mem_word_select, word_select_address);");
  return(0);
}

/**********************************************************************
 * misctf routine
 *
 * The misctf routine is used to call tf_nodeinfo() at the 
 * beginning of simulation, so that the memory allocated by 
 * tf_nodeinfo() and tf_exprinfo() is only allocated one time for each
 * instance of $fill_mem.
 *********************************************************************/
typedef struct PLIbook_my_data {
  p_tfnodeinfo node_info;  /* pointer to structure for tf_nodeinfo() */
  p_tfexprinfo expr_info;  /* pointer to structure for tf_exprinfo() */
} PLIbook_my_data_s, *PLIbook_my_data_p;

int PLIbook_FillMem_misctf(int user_data, int reason)
{
  PLIbook_my_data_p info;  /* pointer to info structures */

  if (reason != REASON_ENDOFCOMPILE)
    return(0);  /* exit now if this is not the start of simulation */
  
  /* allocate memory for structure to store info structure */
  info = (PLIbook_my_data_p)malloc(sizeof(PLIbook_my_data_s));
  
  /* allocate memory for an s_nodeinfo and an s_tfexprinfo structure */
  info->node_info = (p_tfnodeinfo)malloc(sizeof(s_tfnodeinfo));
  info->expr_info = (p_tfexprinfo)malloc(sizeof(s_tfexprinfo));

  /* Get the nodeinfo structure for tfarg 1 */
  if (!tf_nodeinfo(1, info->node_info)) {
    tf_error("Error: $fill_mem could not get tf_nodeinfo for tfarg 1");
    tf_dofinish(); /* about simulation */
    return(0);
  }
  else if (info->node_info->node_type != TF_MEMORY_NODE) {
    tf_error("Error: $fill_mem arg is not a memory word -- aborting");
    tf_dofinish(); /* about simulation */
    return(0);
  }

  /* Get the exprinfo structure for tfarg 1 */
  if (!tf_exprinfo(1, info->expr_info)) {
    tf_error("Error: $fill_mem could not get tf_exprinfo for tfarg 1");
    tf_dofinish(); /* about simulation */
    return(0);
  }

  tf_setworkarea((char *)info);  /* put info pointer in work area */

  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_FillMem_calltf()
{
  int   depth, width, ngroups, word_increment, mem_address, i;
  char *mem_ptr, *aval_ptr, *bval_ptr;
  PLIbook_my_data_p info;  /* pointer to info structures */

  info = (PLIbook_my_data_p)tf_getworkarea();

  mem_ptr    = info->node_info->node_value.memoryval_p;
  width      = info->node_info->node_vec_size;
  depth      = info->node_info->node_mem_size;
  ngroups    = info->node_info->node_ngroups;

  /* Modify current memory values: set aval bits to memory address,
     set bval bits to 0 (2-state logic) */
  word_increment = ngroups * 2; /* 1 word = aval/bval group set */
  for (mem_address = 0;
       mem_address < 4; /* node_info->node_mem_size; */
       mem_address++) {
    aval_ptr = mem_ptr + (mem_address * word_increment);
    bval_ptr = aval_ptr + ngroups;
    aval_ptr[0] = mem_address;
    bval_ptr[0] = 0x0;
    for (i=1; i<ngroups; i++) {
      aval_ptr[i] = 0x0;
      bval_ptr[i] = 0x0;
    }
  }

  io_printf("  Memory contents have been modified by PLI\n");
  
  return(0);
}

/*********************************************************************/
