# BSD 3-Clause License; see https://github.com/scikit-hep/uproot4/blob/main/LICENSE

from __future__ import absolute_import

import numpy
import pytest
import skhep_testdata

import uproot


def test_numpy_1d():
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        values, edges = f["hpx"].to_numpy(flow=True)
        assert values.tolist() == [
            2.0,
            2.0,
            3.0,
            1.0,
            1.0,
            2.0,
            4.0,
            6.0,
            12.0,
            8.0,
            9.0,
            15.0,
            15.0,
            31.0,
            35.0,
            40.0,
            64.0,
            64.0,
            81.0,
            108.0,
            124.0,
            156.0,
            165.0,
            209.0,
            262.0,
            297.0,
            392.0,
            432.0,
            466.0,
            521.0,
            604.0,
            657.0,
            788.0,
            903.0,
            1079.0,
            1135.0,
            1160.0,
            1383.0,
            1458.0,
            1612.0,
            1770.0,
            1868.0,
            1861.0,
            1946.0,
            2114.0,
            2175.0,
            2207.0,
            2273.0,
            2276.0,
            2329.0,
            2325.0,
            2381.0,
            2417.0,
            2364.0,
            2284.0,
            2188.0,
            2164.0,
            2130.0,
            1940.0,
            1859.0,
            1763.0,
            1700.0,
            1611.0,
            1459.0,
            1390.0,
            1237.0,
            1083.0,
            1046.0,
            888.0,
            752.0,
            742.0,
            673.0,
            555.0,
            533.0,
            366.0,
            378.0,
            272.0,
            256.0,
            200.0,
            174.0,
            132.0,
            118.0,
            100.0,
            89.0,
            86.0,
            39.0,
            37.0,
            25.0,
            23.0,
            20.0,
            16.0,
            14.0,
            9.0,
            13.0,
            8.0,
            2.0,
            2.0,
            6.0,
            1.0,
            0.0,
            1.0,
            4.0,
        ]

        assert edges.tolist() == [
            -numpy.inf,
            -4.0,
            -3.92,
            -3.84,
            -3.76,
            -3.68,
            -3.6,
            -3.52,
            -3.44,
            -3.36,
            -3.2800000000000002,
            -3.2,
            -3.12,
            -3.04,
            -2.96,
            -2.88,
            -2.8,
            -2.7199999999999998,
            -2.6399999999999997,
            -2.56,
            -2.48,
            -2.4,
            -2.3200000000000003,
            -2.24,
            -2.16,
            -2.08,
            -2.0,
            -1.92,
            -1.8399999999999999,
            -1.7599999999999998,
            -1.6800000000000002,
            -1.6,
            -1.52,
            -1.44,
            -1.3599999999999999,
            -1.2799999999999998,
            -1.1999999999999997,
            -1.12,
            -1.04,
            -0.96,
            -0.8799999999999999,
            -0.7999999999999998,
            -0.7199999999999998,
            -0.6400000000000001,
            -0.56,
            -0.48,
            -0.3999999999999999,
            -0.31999999999999984,
            -0.23999999999999977,
            -0.16000000000000014,
            -0.08000000000000007,
            0.0,
            0.08000000000000007,
            0.16000000000000014,
            0.2400000000000002,
            0.3200000000000003,
            0.40000000000000036,
            0.4800000000000004,
            0.5600000000000005,
            0.6399999999999997,
            0.7199999999999998,
            0.7999999999999998,
            0.8799999999999999,
            0.96,
            1.04,
            1.12,
            1.2000000000000002,
            1.2800000000000002,
            1.3600000000000003,
            1.4400000000000004,
            1.5200000000000005,
            1.6000000000000005,
            1.6799999999999997,
            1.7599999999999998,
            1.8399999999999999,
            1.92,
            2.0,
            2.08,
            2.16,
            2.24,
            2.3200000000000003,
            2.4000000000000004,
            2.4800000000000004,
            2.5600000000000005,
            2.6400000000000006,
            2.7199999999999998,
            2.8,
            2.88,
            2.96,
            3.04,
            3.12,
            3.2,
            3.2800000000000002,
            3.3600000000000003,
            3.4400000000000004,
            3.5200000000000005,
            3.6000000000000005,
            3.6799999999999997,
            3.76,
            3.84,
            3.92,
            4.0,
            numpy.inf,
        ]

        f["hpx"].errors()


def test_numpy_2d():
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        values, xedges, yedges = f["hpxpy"].to_numpy(flow=True)

        assert values.tolist() == [
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                4.0,
                1.0,
                0.0,
                2.0,
                1.0,
                1.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                1.0,
                1.0,
                2.0,
                0.0,
                2.0,
                2.0,
                0.0,
                1.0,
                1.0,
                2.0,
                2.0,
                0.0,
                1.0,
                5.0,
                2.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                0.0,
                2.0,
                1.0,
                3.0,
                4.0,
                3.0,
                4.0,
                4.0,
                3.0,
                3.0,
                6.0,
                1.0,
                0.0,
                1.0,
                1.0,
                1.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                4.0,
                1.0,
                4.0,
                5.0,
                2.0,
                7.0,
                7.0,
                9.0,
                13.0,
                10.0,
                4.0,
                3.0,
                3.0,
                4.0,
                6.0,
                3.0,
                1.0,
                1.0,
                0.0,
                3.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                3.0,
                3.0,
                2.0,
                9.0,
                4.0,
                8.0,
                7.0,
                8.0,
                10.0,
                17.0,
                10.0,
                13.0,
                17.0,
                17.0,
                9.0,
                12.0,
                1.0,
                6.0,
                7.0,
                2.0,
                1.0,
                1.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                1.0,
                0.0,
                2.0,
                2.0,
                7.0,
                7.0,
                11.0,
                12.0,
                13.0,
                16.0,
                25.0,
                16.0,
                18.0,
                21.0,
                22.0,
                20.0,
                19.0,
                9.0,
                9.0,
                16.0,
                7.0,
                3.0,
                4.0,
                6.0,
                2.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                5.0,
                4.0,
                7.0,
                5.0,
                12.0,
                5.0,
                16.0,
                23.0,
                28.0,
                28.0,
                25.0,
                37.0,
                41.0,
                41.0,
                27.0,
                24.0,
                21.0,
                19.0,
                16.0,
                15.0,
                11.0,
                4.0,
                4.0,
                2.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                4.0,
                1.0,
                6.0,
                6.0,
                14.0,
                14.0,
                21.0,
                26.0,
                46.0,
                42.0,
                47.0,
                52.0,
                44.0,
                51.0,
                53.0,
                41.0,
                56.0,
                30.0,
                24.0,
                19.0,
                20.0,
                21.0,
                12.0,
                8.0,
                1.0,
                2.0,
                3.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                3.0,
                2.0,
                3.0,
                3.0,
                4.0,
                6.0,
                11.0,
                8.0,
                20.0,
                36.0,
                47.0,
                40.0,
                49.0,
                61.0,
                61.0,
                70.0,
                87.0,
                95.0,
                90.0,
                74.0,
                62.0,
                66.0,
                50.0,
                42.0,
                24.0,
                14.0,
                16.0,
                7.0,
                7.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                4.0,
                5.0,
                9.0,
                10.0,
                21.0,
                28.0,
                31.0,
                39.0,
                48.0,
                88.0,
                87.0,
                80.0,
                102.0,
                92.0,
                108.0,
                100.0,
                97.0,
                100.0,
                71.0,
                76.0,
                35.0,
                32.0,
                26.0,
                31.0,
                12.0,
                9.0,
                4.0,
                4.0,
                2.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                2.0,
                6.0,
                5.0,
                11.0,
                9.0,
                18.0,
                23.0,
                32.0,
                54.0,
                69.0,
                81.0,
                106.0,
                105.0,
                126.0,
                132.0,
                140.0,
                148.0,
                137.0,
                130.0,
                121.0,
                104.0,
                88.0,
                68.0,
                53.0,
                35.0,
                30.0,
                16.0,
                9.0,
                6.0,
                3.0,
                8.0,
                2.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                4.0,
                1.0,
                5.0,
                7.0,
                22.0,
                20.0,
                44.0,
                57.0,
                60.0,
                100.0,
                149.0,
                148.0,
                155.0,
                201.0,
                198.0,
                198.0,
                216.0,
                207.0,
                182.0,
                159.0,
                153.0,
                102.0,
                104.0,
                66.0,
                44.0,
                28.0,
                21.0,
                8.0,
                11.0,
                4.0,
                4.0,
                1.0,
                1.0,
                0.0,
                1.0,
                0.0,
                1.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                2.0,
                2.0,
                3.0,
                6.0,
                8.0,
                16.0,
                34.0,
                53.0,
                58.0,
                88.0,
                106.0,
                131.0,
                179.0,
                215.0,
                206.0,
                274.0,
                236.0,
                261.0,
                243.0,
                240.0,
                207.0,
                162.0,
                138.0,
                115.0,
                85.0,
                65.0,
                59.0,
                27.0,
                22.0,
                13.0,
                7.0,
                5.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                1.0,
                2.0,
                0.0,
                2.0,
                1.0,
                5.0,
                6.0,
                9.0,
                13.0,
                20.0,
                39.0,
                60.0,
                74.0,
                94.0,
                145.0,
                171.0,
                211.0,
                253.0,
                281.0,
                321.0,
                311.0,
                354.0,
                317.0,
                289.0,
                269.0,
                221.0,
                199.0,
                139.0,
                97.0,
                73.0,
                50.0,
                31.0,
                29.0,
                9.0,
                11.0,
                4.0,
                3.0,
                2.0,
                0.0,
                0.0,
                1.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                3.0,
                17.0,
                17.0,
                29.0,
                42.0,
                73.0,
                93.0,
                104.0,
                169.0,
                222.0,
                232.0,
                250.0,
                361.0,
                346.0,
                375.0,
                363.0,
                349.0,
                333.0,
                312.0,
                247.0,
                195.0,
                176.0,
                109.0,
                92.0,
                51.0,
                43.0,
                26.0,
                17.0,
                7.0,
                6.0,
                2.0,
                2.0,
                2.0,
                0.0,
                1.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                2.0,
                1.0,
                2.0,
                6.0,
                8.0,
                16.0,
                33.0,
                51.0,
                95.0,
                93.0,
                134.0,
                164.0,
                231.0,
                298.0,
                353.0,
                341.0,
                420.0,
                432.0,
                425.0,
                404.0,
                360.0,
                326.0,
                301.0,
                211.0,
                175.0,
                139.0,
                93.0,
                62.0,
                56.0,
                26.0,
                11.0,
                11.0,
                11.0,
                1.0,
                0.0,
                2.0,
                1.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                2.0,
                1.0,
                1.0,
                9.0,
                13.0,
                28.0,
                21.0,
                47.0,
                82.0,
                106.0,
                150.0,
                199.0,
                241.0,
                284.0,
                334.0,
                403.0,
                479.0,
                445.0,
                438.0,
                408.0,
                386.0,
                316.0,
                300.0,
                218.0,
                231.0,
                135.0,
                111.0,
                77.0,
                68.0,
                27.0,
                27.0,
                12.0,
                3.0,
                6.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                5.0,
                6.0,
                13.0,
                16.0,
                35.0,
                68.0,
                68.0,
                95.0,
                142.0,
                190.0,
                260.0,
                287.0,
                363.0,
                403.0,
                448.0,
                478.0,
                446.0,
                439.0,
                401.0,
                396.0,
                314.0,
                245.0,
                226.0,
                134.0,
                114.0,
                66.0,
                44.0,
                29.0,
                23.0,
                14.0,
                8.0,
                12.0,
                6.0,
                3.0,
                0.0,
                2.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                1.0,
                2.0,
                3.0,
                9.0,
                14.0,
                22.0,
                34.0,
                60.0,
                86.0,
                129.0,
                179.0,
                210.0,
                270.0,
                275.0,
                370.0,
                416.0,
                445.0,
                497.0,
                449.0,
                440.0,
                426.0,
                385.0,
                278.0,
                273.0,
                210.0,
                141.0,
                115.0,
                77.0,
                50.0,
                32.0,
                25.0,
                15.0,
                8.0,
                5.0,
                3.0,
                3.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                1.0,
                0.0,
                0.0,
                0.0,
                1.0,
                4.0,
                5.0,
                11.0,
                24.0,
                19.0,
                41.0,
                88.0,
                126.0,
                120.0,
                197.0,
                260.0,
                281.0,
                344.0,
                398.0,
                411.0,
                476.0,
                436.0,
                488.0,
                393.0,
                331.0,
                302.0,
                236.0,
                205.0,
                171.0,
                115.0,
                61.0,
                65.0,
                23.0,
                19.0,
                11.0,
                4.0,
                5.0,
                2.0,
                0.0,
                3.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                2.0,
                2.0,
                4.0,
                2.0,
                13.0,
                22.0,
                32.0,
                47.0,
                72.0,
                103.0,
                135.0,
                209.0,
                200.0,
                284.0,
                341.0,
                360.0,
                391.0,
                412.0,
                424.0,
                443.0,
                370.0,
                323.0,
                262.0,
                221.0,
                180.0,
                159.0,
                91.0,
                75.0,
                38.0,
                28.0,
                24.0,
                10.0,
                6.0,
                1.0,
                2.0,
                0.0,
                1.0,
                0.0,
                0.0,
            ],
            [
                1.0,
                0.0,
                0.0,
                0.0,
                3.0,
                1.0,
                4.0,
                6.0,
                18.0,
                30.0,
                37.0,
                66.0,
                98.0,
                119.0,
                141.0,
                203.0,
                233.0,
                303.0,
                345.0,
                348.0,
                360.0,
                367.0,
                350.0,
                302.0,
                280.0,
                251.0,
                203.0,
                155.0,
                121.0,
                64.0,
                49.0,
                43.0,
                28.0,
                21.0,
                8.0,
                4.0,
                2.0,
                1.0,
                1.0,
                1.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                4.0,
                4.0,
                10.0,
                17.0,
                28.0,
                43.0,
                52.0,
                75.0,
                108.0,
                162.0,
                155.0,
                211.0,
                268.0,
                278.0,
                339.0,
                331.0,
                339.0,
                305.0,
                239.0,
                241.0,
                223.0,
                161.0,
                136.0,
                93.0,
                86.0,
                63.0,
                32.0,
                25.0,
                15.0,
                10.0,
                0.0,
                2.0,
                1.0,
                0.0,
                0.0,
                0.0,
                1.0,
            ],
            [
                2.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                1.0,
                5.0,
                10.0,
                17.0,
                27.0,
                40.0,
                86.0,
                91.0,
                123.0,
                150.0,
                172.0,
                197.0,
                247.0,
                237.0,
                255.0,
                279.0,
                271.0,
                218.0,
                189.0,
                194.0,
                152.0,
                108.0,
                92.0,
                52.0,
                41.0,
                32.0,
                16.0,
                22.0,
                5.0,
                1.0,
                4.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                1.0,
                1.0,
                0.0,
                0.0,
                1.0,
                2.0,
                6.0,
                4.0,
                6.0,
                14.0,
                22.0,
                28.0,
                57.0,
                56.0,
                87.0,
                111.0,
                142.0,
                169.0,
                206.0,
                202.0,
                211.0,
                209.0,
                181.0,
                174.0,
                158.0,
                157.0,
                105.0,
                89.0,
                62.0,
                44.0,
                34.0,
                20.0,
                15.0,
                12.0,
                9.0,
                7.0,
                1.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                1.0,
                1.0,
                5.0,
                4.0,
                8.0,
                15.0,
                27.0,
                33.0,
                38.0,
                64.0,
                67.0,
                84.0,
                119.0,
                131.0,
                153.0,
                165.0,
                151.0,
                151.0,
                129.0,
                126.0,
                125.0,
                92.0,
                70.0,
                46.0,
                33.0,
                23.0,
                22.0,
                10.0,
                7.0,
                2.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                2.0,
                2.0,
                7.0,
                8.0,
                11.0,
                16.0,
                15.0,
                35.0,
                43.0,
                39.0,
                61.0,
                86.0,
                99.0,
                83.0,
                131.0,
                131.0,
                107.0,
                101.0,
                112.0,
                86.0,
                76.0,
                69.0,
                57.0,
                39.0,
                32.0,
                17.0,
                11.0,
                8.0,
                1.0,
                3.0,
                3.0,
                0.0,
                1.0,
                0.0,
                1.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                6.0,
                4.0,
                11.0,
                17.0,
                22.0,
                20.0,
                34.0,
                27.0,
                46.0,
                80.0,
                69.0,
                71.0,
                76.0,
                79.0,
                66.0,
                82.0,
                67.0,
                58.0,
                49.0,
                32.0,
                21.0,
                22.0,
                21.0,
                9.0,
                5.0,
                4.0,
                5.0,
                2.0,
                3.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                1.0,
                2.0,
                4.0,
                8.0,
                19.0,
                15.0,
                16.0,
                26.0,
                26.0,
                49.0,
                54.0,
                51.0,
                45.0,
                46.0,
                55.0,
                39.0,
                33.0,
                40.0,
                24.0,
                22.0,
                20.0,
                15.0,
                8.0,
                11.0,
                4.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                1.0,
                2.0,
                1.0,
                6.0,
                8.0,
                12.0,
                15.0,
                28.0,
                24.0,
                25.0,
                30.0,
                39.0,
                34.0,
                28.0,
                27.0,
                27.0,
                22.0,
                18.0,
                10.0,
                11.0,
                6.0,
                4.0,
                9.0,
                1.0,
                2.0,
                2.0,
                1.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                0.0,
                1.0,
                2.0,
                0.0,
                4.0,
                5.0,
                5.0,
                9.0,
                12.0,
                13.0,
                22.0,
                22.0,
                19.0,
                23.0,
                21.0,
                20.0,
                20.0,
                10.0,
                20.0,
                11.0,
                8.0,
                5.0,
                5.0,
                4.0,
                0.0,
                2.0,
                2.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                2.0,
                1.0,
                1.0,
                0.0,
                1.0,
                3.0,
                2.0,
                3.0,
                1.0,
                4.0,
                4.0,
                10.0,
                11.0,
                13.0,
                16.0,
                12.0,
                9.0,
                18.0,
                19.0,
                6.0,
                8.0,
                5.0,
                5.0,
                1.0,
                4.0,
                0.0,
                2.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                1.0,
                3.0,
                5.0,
                3.0,
                1.0,
                5.0,
                11.0,
                2.0,
                5.0,
                3.0,
                8.0,
                4.0,
                3.0,
                6.0,
                4.0,
                1.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                1.0,
                4.0,
                0.0,
                3.0,
                2.0,
                3.0,
                4.0,
                4.0,
                8.0,
                3.0,
                6.0,
                2.0,
                2.0,
                4.0,
                1.0,
                1.0,
                2.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                2.0,
                4.0,
                2.0,
                1.0,
                2.0,
                4.0,
                1.0,
                1.0,
                1.0,
                1.0,
                2.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                2.0,
                3.0,
                1.0,
                0.0,
                2.0,
                3.0,
                0.0,
                1.0,
                1.0,
                1.0,
                1.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                2.0,
                3.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
            [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                2.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                1.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ],
        ]

        assert xedges.tolist() == [
            -numpy.inf,
            -4.0,
            -3.8,
            -3.6,
            -3.4,
            -3.2,
            -3.0,
            -2.8,
            -2.5999999999999996,
            -2.4,
            -2.2,
            -2.0,
            -1.7999999999999998,
            -1.5999999999999996,
            -1.4,
            -1.1999999999999997,
            -1.0,
            -0.7999999999999998,
            -0.5999999999999996,
            -0.3999999999999999,
            -0.19999999999999973,
            0.0,
            0.20000000000000018,
            0.40000000000000036,
            0.6000000000000005,
            0.8000000000000007,
            1.0,
            1.2000000000000002,
            1.4000000000000004,
            1.6000000000000005,
            1.8000000000000007,
            2.0,
            2.2,
            2.4000000000000004,
            2.6000000000000005,
            2.8000000000000007,
            3.0,
            3.2,
            3.4000000000000004,
            3.6000000000000005,
            3.8000000000000007,
            4.0,
            numpy.inf,
        ]

        assert yedges.tolist() == [
            -numpy.inf,
            -4.0,
            -3.8,
            -3.6,
            -3.4,
            -3.2,
            -3.0,
            -2.8,
            -2.5999999999999996,
            -2.4,
            -2.2,
            -2.0,
            -1.7999999999999998,
            -1.5999999999999996,
            -1.4,
            -1.1999999999999997,
            -1.0,
            -0.7999999999999998,
            -0.5999999999999996,
            -0.3999999999999999,
            -0.19999999999999973,
            0.0,
            0.20000000000000018,
            0.40000000000000036,
            0.6000000000000005,
            0.8000000000000007,
            1.0,
            1.2000000000000002,
            1.4000000000000004,
            1.6000000000000005,
            1.8000000000000007,
            2.0,
            2.2,
            2.4000000000000004,
            2.6000000000000005,
            2.8000000000000007,
            3.0,
            3.2,
            3.4000000000000004,
            3.6000000000000005,
            3.8000000000000007,
            4.0,
            numpy.inf,
        ]

        f["hpxpy"].errors()


def test_numpy_profile():
    # python -c 'import ROOT, skhep_testdata; f = ROOT.TFile(skhep_testdata.data_path("uproot-hepdata-example.root")); h = f.Get("hprof"); h.SetErrorOption("g"); print(repr(h.GetErrorOption())); print([h.GetBinError(i) for i in range(102)])'

    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        obj = f["hprof"]

        assert obj.axis().edges(flow=True).tolist() == [
            -numpy.inf,
            -4.0,
            -3.92,
            -3.84,
            -3.76,
            -3.68,
            -3.6,
            -3.52,
            -3.44,
            -3.36,
            -3.2800000000000002,
            -3.2,
            -3.12,
            -3.04,
            -2.96,
            -2.88,
            -2.8,
            -2.7199999999999998,
            -2.6399999999999997,
            -2.56,
            -2.48,
            -2.4,
            -2.3200000000000003,
            -2.24,
            -2.16,
            -2.08,
            -2.0,
            -1.92,
            -1.8399999999999999,
            -1.7599999999999998,
            -1.6800000000000002,
            -1.6,
            -1.52,
            -1.44,
            -1.3599999999999999,
            -1.2799999999999998,
            -1.1999999999999997,
            -1.12,
            -1.04,
            -0.96,
            -0.8799999999999999,
            -0.7999999999999998,
            -0.7199999999999998,
            -0.6400000000000001,
            -0.56,
            -0.48,
            -0.3999999999999999,
            -0.31999999999999984,
            -0.23999999999999977,
            -0.16000000000000014,
            -0.08000000000000007,
            0.0,
            0.08000000000000007,
            0.16000000000000014,
            0.2400000000000002,
            0.3200000000000003,
            0.40000000000000036,
            0.4800000000000004,
            0.5600000000000005,
            0.6399999999999997,
            0.7199999999999998,
            0.7999999999999998,
            0.8799999999999999,
            0.96,
            1.04,
            1.12,
            1.2000000000000002,
            1.2800000000000002,
            1.3600000000000003,
            1.4400000000000004,
            1.5200000000000005,
            1.6000000000000005,
            1.6799999999999997,
            1.7599999999999998,
            1.8399999999999999,
            1.92,
            2.0,
            2.08,
            2.16,
            2.24,
            2.3200000000000003,
            2.4000000000000004,
            2.4800000000000004,
            2.5600000000000005,
            2.6400000000000006,
            2.7199999999999998,
            2.8,
            2.88,
            2.96,
            3.04,
            3.12,
            3.2,
            3.2800000000000002,
            3.3600000000000003,
            3.4400000000000004,
            3.5200000000000005,
            3.6000000000000005,
            3.6799999999999997,
            3.76,
            3.84,
            3.92,
            4.0,
            numpy.inf,
        ]

        assert obj.values(flow=True).tolist() == [
            17.99833583831787,
            17.05295467376709,
            16.96826426188151,
            15.189482688903809,
            13.73788833618164,
            13.375219821929932,
            13.510369300842285,
            12.646300633748373,
            12.66011929512024,
            11.824836373329163,
            11.623446782430014,
            11.472076733907064,
            10.052986780802408,
            10.030597317603327,
            9.614417321341378,
            8.776622557640076,
            8.620806604623795,
            8.179968640208244,
            7.4127079410317505,
            7.497226472254153,
            6.980819525257234,
            6.505285000189756,
            6.251851732080633,
            5.813575813074431,
            5.584403858840011,
            5.011047506171846,
            4.91228925087014,
            4.524659741255972,
            4.24002511460382,
            4.077462992146468,
            3.638793389923525,
            3.5221418274773493,
            3.255871357954093,
            2.961020285108953,
            2.706199676046999,
            2.5841911697177635,
            2.3627997641933374,
            2.1493446517490598,
            2.0077903614940302,
            1.8382392522714865,
            1.712551970266353,
            1.6131308919867815,
            1.449079261311019,
            1.3471352570103472,
            1.245844892917823,
            1.1707659457058741,
            1.1247396327430272,
            1.1198479739799145,
            1.0281285326813325,
            1.0417602170529079,
            1.0197545518784679,
            1.0003131686022901,
            1.0794705348466953,
            1.02964734215157,
            1.0603044479791786,
            1.1542847645715888,
            1.1745855332784314,
            1.317462644113901,
            1.2909844154549628,
            1.4553258675057892,
            1.5839730073833629,
            1.7274112791524214,
            1.8171250952244693,
            1.999616364569922,
            2.1976474514968105,
            2.332895248766955,
            2.573682461088714,
            2.7457328102556744,
            2.9121971759978718,
            3.157701852473807,
            3.3310595230272195,
            3.685565097902363,
            4.011118740219254,
            4.3144918141177175,
            4.548257073418039,
            4.93563452094951,
            5.191882547210245,
            5.4767660945653915,
            5.7347985672950745,
            6.18110868574559,
            6.4068912520553125,
            7.048662836268797,
            7.238576850891113,
            7.555341683077009,
            8.169158785842185,
            9.019065893613375,
            8.789572896184149,
            9.365243797302247,
            9.570246945256772,
            10.279665088653564,
            11.086111783981323,
            11.118131773812431,
            12.656685405307346,
            12.176475048065186,
            12.393176078796387,
            16.518978118896484,
            13.303139686584473,
            14.635026613871256,
            14.96741771697998,
            0.0,
            18.32199478149414,
            17.8403746287028,
        ]

        assert obj.errors(flow=True).tolist() == [
            0.2425426377130359,
            0.7421210342302459,
            0.4940066334987832,
            0.0,
            0.0,
            0.2464980351520863,
            0.5555373736396868,
            0.24357921956140027,
            0.224616129931814,
            0.34906168361481404,
            0.4356334723283742,
            0.5128651082538828,
            0.2086307384620165,
            0.28308077003120913,
            0.2891541406820913,
            0.16769727425722117,
            0.1725773236590863,
            0.12765099099147656,
            0.10176558165942572,
            0.15209837443095275,
            0.11509671433352467,
            0.10149120489291587,
            0.11432069747168126,
            0.09759737443630617,
            0.0925726825400381,
            0.06761852807106097,
            0.07883833461255244,
            0.06391971743421765,
            0.07016808339801081,
            0.0679063456384074,
            0.05330254783019173,
            0.056304893803072076,
            0.055238305812566516,
            0.047974962128087315,
            0.042558147198316985,
            0.04422411577185198,
            0.0408986879854767,
            0.03453675368752007,
            0.039438577439864786,
            0.03461426584130604,
            0.036187944978430614,
            0.034085467706933194,
            0.03170797279308202,
            0.031219377450826796,
            0.03011256422687173,
            0.02926608780683337,
            0.0301281364334744,
            0.029773650810830235,
            0.029748389712173053,
            0.03081957669527989,
            0.03132949553456636,
            0.02939420318612115,
            0.029258470846132534,
            0.02930430026995912,
            0.02804401796249436,
            0.031175984988258274,
            0.030108329759273612,
            0.03149116682767534,
            0.029094905772258012,
            0.03256760040302268,
            0.034455467521643364,
            0.03480207320474039,
            0.032712202513451534,
            0.03860859020725239,
            0.03885261043325975,
            0.03856340740992072,
            0.04624045482680718,
            0.04543317885660241,
            0.04864621055120345,
            0.05203738725490573,
            0.043244016740287015,
            0.05850656051444226,
            0.059709748394490884,
            0.06594229969906718,
            0.07220151434675717,
            0.08170131663135467,
            0.08712811029061408,
            0.08092332833341198,
            0.09191356506835095,
            0.10837656197125221,
            0.10509032780349721,
            0.1549338147492931,
            0.12013956272890565,
            0.11435861802671626,
            0.18394299511064918,
            0.36368702093446753,
            0.13346262669376094,
            0.18325723104438668,
            0.17988975869975438,
            0.1926530171606879,
            0.352473088726965,
            0.18420322865597596,
            0.5959353241264886,
            0.21540243485684468,
            0.11755951260322403,
            1.6619844323502102,
            0.1352812684763272,
            0.4534391377411209,
            0.0,
            0.0,
            0.0,
            0.16817919583370047,
        ]

        assert obj.errors(
            flow=True, error_mode=uproot.behaviors.TProfile._kERRORSPREAD
        ).tolist() == [
            0.34300708770751953,
            1.0495176315307617,
            0.8556445884959498,
            0.0,
            0.0,
            0.3486008644104004,
            1.1110747472793736,
            0.5966447998707816,
            0.7780930984827886,
            0.9872955341457128,
            1.3069004169851226,
            1.9863180231181519,
            0.8080233755703451,
            1.5761270231822468,
            1.7106589658888625,
            1.0606106881094808,
            1.3806185892726903,
            1.0212079279318125,
            0.9158902349348315,
            1.5806526735782713,
            1.281662768690052,
            1.2676247428226026,
            1.4684759475789604,
            1.4109488746385728,
            1.4984197698897908,
            1.1653166117127,
            1.560919388615718,
            1.3285463784181335,
            1.5147207420285738,
            1.549991160077581,
            1.3099853470686935,
            1.443207670599461,
            1.5506131361772943,
            1.4416456163169384,
            1.3979557820249364,
            1.4898998932597651,
            1.39295911912831,
            1.284377246895075,
            1.5059134195962758,
            1.3897530746031688,
            1.5224763480325734,
            1.473186374916331,
            1.367860043067912,
            1.377195694990315,
            1.3845231787179089,
            1.3648794718765778,
            1.4153812430343926,
            1.419488271301224,
            1.419219569870578,
            1.4873439583962957,
            1.5106535672672314,
            1.4343045945107848,
            1.4384340328933711,
            1.4248038889030987,
            1.340257624082002,
            1.4582898146438432,
            1.4006037738107093,
            1.453377907771706,
            1.2814976672937608,
            1.4041886411676958,
            1.446719393622703,
            1.4349262381362273,
            1.3129783240312063,
            1.4747268574003336,
            1.4485303652651937,
            1.3563140181188076,
            1.5217255253773476,
            1.4693963839287074,
            1.449624425594751,
            1.4270014133077806,
            1.1779530457556422,
            1.517791441678946,
            1.406668404280142,
            1.522396207351309,
            1.3812963022723197,
            1.5884551434189818,
            1.4369536067546675,
            1.2947732533345917,
            1.2998541028572388,
            1.429585037043725,
            1.2073959432248138,
            1.6830120202858494,
            1.2013956272890565,
            1.0788570447521093,
            1.705817161574992,
            2.271224717779226,
            0.811821464847988,
            0.9162861552219334,
            0.8627209754934005,
            0.8615704848834633,
            1.40989235490786,
            0.6892253711682418,
            1.787805972379466,
            0.7461759224922005,
            0.3325085142189005,
            2.350400924682617,
            0.1913166046142578,
            1.1106945168733242,
            0.0,
            0.0,
            0.0,
            0.29129491196004526,
        ]

        assert obj.errors(
            flow=True, error_mode=uproot.behaviors.TProfile._kERRORSPREADI
        ).tolist() == [
            0.2425426377130359,
            0.7421210342302459,
            0.4940066334987832,
            0.2886751345948129,
            0.2886751345948129,
            0.2464980351520863,
            0.5555373736396868,
            0.24357921956140027,
            0.224616129931814,
            0.34906168361481404,
            0.4356334723283742,
            0.5128651082538828,
            0.2086307384620165,
            0.28308077003120913,
            0.2891541406820913,
            0.16769727425722117,
            0.1725773236590863,
            0.12765099099147656,
            0.10176558165942572,
            0.15209837443095275,
            0.11509671433352467,
            0.10149120489291587,
            0.11432069747168126,
            0.09759737443630617,
            0.0925726825400381,
            0.06761852807106097,
            0.07883833461255244,
            0.06391971743421765,
            0.07016808339801081,
            0.0679063456384074,
            0.05330254783019173,
            0.056304893803072076,
            0.055238305812566516,
            0.047974962128087315,
            0.042558147198316985,
            0.04422411577185198,
            0.0408986879854767,
            0.03453675368752007,
            0.039438577439864786,
            0.03461426584130604,
            0.036187944978430614,
            0.034085467706933194,
            0.03170797279308202,
            0.031219377450826796,
            0.03011256422687173,
            0.02926608780683337,
            0.0301281364334744,
            0.029773650810830235,
            0.029748389712173053,
            0.03081957669527989,
            0.03132949553456636,
            0.02939420318612115,
            0.029258470846132534,
            0.02930430026995912,
            0.02804401796249436,
            0.031175984988258274,
            0.030108329759273612,
            0.03149116682767534,
            0.029094905772258012,
            0.03256760040302268,
            0.034455467521643364,
            0.03480207320474039,
            0.032712202513451534,
            0.03860859020725239,
            0.03885261043325975,
            0.03856340740992072,
            0.04624045482680718,
            0.04543317885660241,
            0.04864621055120345,
            0.05203738725490573,
            0.043244016740287015,
            0.05850656051444226,
            0.059709748394490884,
            0.06594229969906718,
            0.07220151434675717,
            0.08170131663135467,
            0.08712811029061408,
            0.08092332833341198,
            0.09191356506835095,
            0.10837656197125221,
            0.10509032780349721,
            0.1549338147492931,
            0.12013956272890565,
            0.11435861802671626,
            0.18394299511064918,
            0.36368702093446753,
            0.13346262669376094,
            0.18325723104438668,
            0.17988975869975438,
            0.1926530171606879,
            0.352473088726965,
            0.18420322865597596,
            0.5959353241264886,
            0.21540243485684468,
            0.11755951260322403,
            1.6619844323502102,
            0.1352812684763272,
            0.4534391377411209,
            0.2886751345948129,
            0.0,
            0.2886751345948129,
            0.16817919583370047,
        ]

        assert obj.errors(
            flow=True, error_mode=uproot.behaviors.TProfile._kERRORSPREADG
        ).tolist() == [
            0.7071067811865475,
            0.7071067811865475,
            0.5773502691896258,
            1.0,
            1.0,
            0.7071067811865475,
            0.5,
            0.4082482904638631,
            0.2886751345948129,
            0.35355339059327373,
            0.3333333333333333,
            0.2581988897471611,
            0.2581988897471611,
            0.1796053020267749,
            0.1690308509457033,
            0.15811388300841897,
            0.125,
            0.125,
            0.1111111111111111,
            0.09622504486493763,
            0.08980265101338746,
            0.08006407690254357,
            0.0778498944161523,
            0.06917144638660747,
            0.06178020632152154,
            0.058025885318565944,
            0.050507627227610534,
            0.048112522432468816,
            0.04632410546120795,
            0.04381079543383235,
            0.04068942293855797,
            0.03901371573204352,
            0.035623524993954825,
            0.033277916281986085,
            0.03044312827739915,
            0.02968260885977624,
            0.029361010975735173,
            0.026889882837002246,
            0.026189140043946204,
            0.024906774069335894,
            0.023769134427076417,
            0.023137240669137377,
            0.023180714250535184,
            0.022668802672263903,
            0.021749411414517784,
            0.021442250696755896,
            0.021286234067143354,
            0.020974918506045256,
            0.020961090407515925,
            0.020721216851891204,
            0.020739033894608506,
            0.02049369659597791,
            0.020340502363726694,
            0.02056725174474318,
            0.02092434876593436,
            0.02137845624045064,
            0.02149667901961739,
            0.021667569500871973,
            0.022703830459324992,
            0.023193180352135665,
            0.023816275411477048,
            0.024253562503633298,
            0.024914503091731197,
            0.026180163474687157,
            0.026822089039291005,
            0.028432506701809173,
            0.0303868562731382,
            0.030919620705155318,
            0.033557802760701215,
            0.03646624787447364,
            0.036711154910717615,
            0.03854716722458499,
            0.04244763599780089,
            0.043314808182421,
            0.05227083734893167,
            0.05143444998736397,
            0.06063390625908324,
            0.0625,
            0.07071067811865475,
            0.07580980435789034,
            0.08703882797784893,
            0.09205746178983235,
            0.1,
            0.105999788000636,
            0.10783277320343841,
            0.16012815380508713,
            0.1643989873053573,
            0.2,
            0.20851441405707477,
            0.22360679774997896,
            0.25,
            0.2672612419124244,
            0.3333333333333333,
            0.2886751345948129,
            0.35355339059327373,
            0.7071067811865475,
            0.7071067811865475,
            0.4082482904638631,
            1.0,
            0.0,
            1.0,
            0.5773502691896258,
        ]


def test_boost_1d():
    boost_histogram = pytest.importorskip("boost_histogram")
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        f["hpx"].to_boost()


def test_boost_2d():
    boost_histogram = pytest.importorskip("boost_histogram")
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        f["hpxpy"].to_boost()


def test_hist_1d():
    hist = pytest.importorskip("hist")
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        f["hpx"].to_hist()


def test_hist_2d():
    hist = pytest.importorskip("hist")
    with uproot.open(skhep_testdata.data_path("uproot-hepdata-example.root")) as f:
        f["hpxpy"].to_hist()
