#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

import warnings, unittest

from ll import url


class CommonTests:
	def test_fileext(self):
		u = self.type("/gurk/hurz")
		self.assertEqual(u.file, u"hurz")
		self.assertEqual(u.ext, None)
		u.file = u"nx.png"
		self.assertEqual(u.file, u"nx.png")
		self.assertEqual(u.ext, u"png")
		self.assertEqual(unicode(u.path), u"/gurk/n%c3%b6x.png")
		u.ext = "gif"
		self.assertEqual(u.file, u"nx.gif")
		self.assertEqual(u.ext, u"gif")

		u = self.type("/gurk/hurz.")
		self.assertEqual(u.file, u"hurz.")
		self.assertEqual(u.ext, u"")
		u.ext = "gif"
		self.assertEqual(u.file, u"hurz.gif")
		self.assertEqual(u.ext, u"gif")

		u = self.type("/gurk/hurz.png")
		self.assertEqual(u.file, u"hurz.png")
		self.assertEqual(u.ext, u"png")

		u = self.type("/gurk/hurz/")
		self.assertEqual(u.file, u"")
		self.assertEqual(u.ext, None)
		u.ext = "gif"
		self.assertEqual(u.file, u".gif")
		self.assertEqual(u.ext, u"gif")

		self.assertEqual(self.type(".gif").withoutext(), self.type("./"))

	def test_relative(self):
		tests = [
			("./", "./", "./"),
			("./", "/", "./"),
			("/a/b/c", "/a/b/d", "d"),
			#("/a/b/c", "/a/b/c/d", "c/d"),
			("/a/b/c/", "/a/b/c/d", "d"),
			("a/b/c/", "a/b/c/d", "d"),
			("a/b/c/", "/a/b/e/f", "../e/f"),
			("../", "a/b/e/f", "../a/b/e/f"),
			("/c/", "/a/b/e/f", "../a/b/e/f"),
			("c?x=y", "c?x=z", "c?x=z"),
		]
		for (base, rel, res) in tests:
			basepath = url.Path(base)
			relpath = url.Path(rel)
			respath = url.Path(res)
			self.assertEqual(relpath.relative(basepath), respath, "%r.relative(%r) is %r, but should be %r" % (relpath, basepath, relpath.relative(basepath), respath))

	def test_join_list(self):
		self.assertEqual(
			["", "gurk", "gurk/"]/self.type("index.html"),
			map(self.type, ["index.html", "index.html", "gurk/index.html"])
		)

		self.assertEqual(
			self.type("gurk/")/["", "hinz", "kunz"],
			map(self.type, ["gurk/", "gurk/hinz", "gurk/kunz"])
		)

	def test_withfile(self):
		tests = [
			("", "gurk", "gurk"),
			("/", "gurk", "/gurk"),
			("/hurz", "gurk", "/gurk"),
			("/hurz.gif", "gurk.gif", "/gurk.gif"),
		]

		for (org, file, exp) in tests:
			org = self.type(org)
			exp = self.type(exp)
			res = org.withfile(file)
			self.assertEqual(exp, res)

	def test_withoutfile(self):
		tests = [
			("", "./"),
			("/", "/"),
			("/hurz", "/"),
			("hurz", "./"),
			("/gurk/hurz/hinz/", "/gurk/hurz/hinz/"),
			("/gurk/hurz/hinz/kunz", "/gurk/hurz/hinz/"),
		]

		for (org, exp) in tests:
			org = self.type(org)
			exp = self.type(exp)
			res = org.withoutfile()
			self.assertEqual(exp, res)

	def test_withext(self):
		tests = [
			("", "py", ".py"),
			("/", "py", "/.py"),
			("/hurz", "py", "/hurz.py"),
			("/hurz.", "py", "/hurz.py"),
			("/hurz.gif", "py", "/hurz.py"),
			("/hurz.gif.png", "py", "/hurz.gif.py"),
			("/hurz.gif.png", "pyc.py", "/hurz.gif.pyc.py"),
		]

		for (org, ext, exp) in tests:
			org = self.type(org)
			exp = self.type(exp)
			res = org.withext(ext)
			self.assertEqual(exp, res)

	def test_withoutext(self):
		tests = [
			("", ""),
			("/", "/"),
			("/gurk.1/hurz", "/gurk.1/hurz"),
			("/gurk.2/hurz.gif", "/gurk.2/hurz"),
			("/gurk.3/hurz.gif.png", "/gurk.3/hurz.gif"),
			("/gurk.4/hurz.", "/gurk.4/hurz"),
		]

		for (org, exp) in tests:
			org = self.type(org)
			exp = self.type(exp)
			res = org.withoutext()
			self.assertEqual(exp, res)


class PathTest(CommonTests, unittest.TestCase):
	type = url.Path

	def test_join(self):
		tests = [
			("a", "b", "b"),
			("a", "b/////////", "b/"),
			("a/", "b", "a/b"),
			("a/", "..", ""), # FIXME Should this be "./"?
			("a/", "../", ""), # FIXME Should this be "./"?
			("a/", "../b", "b"),
			("a/", "../../b", "../b"),
			("../a/", "../b", "../b"),
			("../a/.", "../b", "../b"),
			(".", "b", "b"),
			("./", "b", "b"),
			("./", "b/.", "b/"),
			("./", "./", ""), # FIXME Should this be "./"?
			("a", "/", "/"),
			("a", "/b", "/b"),
			("a", "/b/c", "/b/c"),
		]

		for (base, rel, res) in tests:
			basepath = url.Path(base)
			relpath = url.Path(rel)
			respath = url.Path(res)
			self.assertEqual(basepath/relpath, respath, "%r/%r is %r, but should be %r" % (basepath, relpath, basepath/relpath, respath))
			# This checks rdiv
			self.assertEqual(str(basepath)/relpath, respath, "%r/%r is %r, but should be %r" % (basepath, relpath, basepath/relpath, respath))

	def test_startswith(self):
		u = url.Path("/gurk/hurz/hinz/kunz/")
		self.assert_(u.startswith("/"))
		self.assert_(u.startswith("/gurk"))
		self.assert_(u.startswith("/gurk/"))
		self.assert_(not u.startswith("/gurk2"))
		self.assert_(u.startswith("/gurk/hurz/hinz/"))
		self.assert_(u.startswith("/gurk/hurz/hinz/kunz"))
		self.assert_(u.startswith("/gurk/hurz/hinz/kunz/"))
		self.assert_(not u.startswith("/gurk/hurz/hinz/kunz/nix"))

	def test_endswith(self):
		u = url.Path("/gurk/hurz/hinz/kunz/")
		self.assert_(not u.endswith("/"))
		self.assert_(u.endswith(""))
		self.assert_(u.endswith([(u"",)]))
		self.assert_(u.endswith("kunz/"))
		self.assert_(u.endswith("hinz/kunz/"))
		self.assert_(not u.endswith("/hinz/kunz/"))
		self.assert_(u.endswith("/gurk/hurz/hinz/kunz/"))
		self.assert_(not u.endswith("gurk/gurk/hurz/hinz/kunz/"))

	def test_item(self):
		p = url.Path("/gurk/hurz/hinz/kunz/")
		self.assertEqual(p[0], ("gurk",))
		self.assertEqual(p[-1], ("",))
		p[1] = "gurk"
		self.assertEqual(p[1], ("gurk",))
		self.assertEqual(p, "/gurk/gurk/hinz/kunz/")
		p[-1] = "foo;bar"
		self.assertEqual(p[4], ("foo", "bar"))
		self.assertEqual(p, "/gurk/gurk/hinz/kunz/foo;bar")

		p[0] = "gar/nix"
		self.assertEqual(p[0], ("gar/nix", ))
		self.assertEqual(p, "/gar%2fnix/gurk/hinz/kunz/foo;bar")

	def test_slice(self):
		p = url.Path("/gurk/hurz/hinz/kunz/")
		self.assertEqual(p[:], "gurk/hurz/hinz/kunz/")
		self.assertEqual(p[:1], url.Path("gurk"))
		self.assertEqual(p[-1:], url.Path(""))
		self.assertEqual(p[-2:], "kunz/")

		p[:] = "kunz/hinz".split("/")
		self.assertEqual(p, "/kunz/hinz")
		p[1:] = "gurk/hurz/gar;nix".split("/")
		self.assertEqual(p, "/kunz/gurk/hurz/gar;nix")

	def test_contains(self):
		p = url.Path("/gurk;hurz/hinz/kunz/")
		self.assert_("gurk" not in p)
		self.assert_("gurk;hurz" in p)
		self.assert_(("gurk", "hurz") in p)
		self.assert_("" in p)
		self.assert_(("",) in p)
		self.assert_(("","") not in p)

	def test_insert(self):
		p = url.Path("/gurk;hurz/hinz/kunz/")
		p.insert(0, "nix")
		self.assertEqual(p, url.Path("/nix/gurk;hurz/hinz/kunz/"))

		p = url.Path("/gurk;hurz/hinz/kunz/")
		p.insert(-1, "nix")
		self.assertEqual(p, url.Path("/gurk;hurz/hinz/kunz/nix/"))

		p = url.Path("gurk;hurz/hinz/kunz/")
		p.insert(0, "nix")
		self.assertEqual(p, url.Path("nix/gurk;hurz/hinz/kunz/"))

		p = url.Path("gurk;hurz/hinz/kunz")
		p.insert(-1, "nix")
		self.assertEqual(p, url.Path("gurk;hurz/hinz/nix/kunz"))


class URLTest(CommonTests, unittest.TestCase):
	type = url.URL

	def test_parse(self):
		base = "http://a/b/c/d;p?q#f"
		u = url.URL(base)
		self.assertEqual(u.scheme, "http")
		self.assertEqual(u.userinfo, None)
		self.assertEqual(u.host, "a")
		self.assertEqual(u.port, None)
		self.assertEqual(u.hostport, "a")
		self.assertEqual(u.server, "a")
		self.assertEqual(u.authority, "a")
		self.assertEqual(u.reg_name, None)
		self.assertEqual(u.path, "/b/c/d;p")
		self.assertEqual(u.path.segments, [("b",), ("c",), ("d", "p")])
		self.assertEqual(u.isabspath, True)
		self.assertEqual(u.query, "q")
		self.assertEqual(u.query_parts, False)
		self.assertEqual(u.frag, "f")
		self.assertEqual(u.opaque_part, None)
		self.assertEqual(u.url, base)

		base = "http://a/b/c/d;p?q=x#f"
		u = url.URL(base)
		self.assertEqual(u.query, "q=x")
		self.assertEqual(u.query_parts, {"q": ["x"]})

	def test_join_rfc2396(self):
		base = "http://a/b/c/d;p?q"
		tests = [
			# RFC2396 Section C.1: Normal Examples
			("g:h",           "g:h"),
			("g",             "http://a/b/c/g"),
			("./g",           "http://a/b/c/g"),
			("g/",            "http://a/b/c/g/"),
			("/g",            "http://a/g"),
			("//g",           "http://g"),
			("?y",            "http://a/b/c/?y"),
			("g?y",           "http://a/b/c/g?y"),
			("#s",            "http://a/b/c/d;p?q#s"),
			("g#s",           "http://a/b/c/g#s"),
			("g?y#s",         "http://a/b/c/g?y#s"),
			(";x",            "http://a/b/c/;x"),
			("g;x",           "http://a/b/c/g;x"),
			("g;x?y#s",       "http://a/b/c/g;x?y#s"),
			(".",             "http://a/b/c/"),
			("./",            "http://a/b/c/"),
			("..",            "http://a/b/"),
			("../",           "http://a/b/"),
			("../g",          "http://a/b/g"),
			("../..",         "http://a/"),
			("../../",        "http://a/"),
			("../../g",       "http://a/g"),
			# RFC2396 Section C.2: Abnormal Examples
			("",              "http://a/b/c/d;p?q"),
			("../../../g",    "http://a/../g"),
			("../../../../g", "http://a/../../g"),
			("/./g",          "http://a/./g"),
			("/../g",         "http://a/../g"),
			("g.",            "http://a/b/c/g."),
			(".g",            "http://a/b/c/.g"),
			("g..",           "http://a/b/c/g.."),
			("..g",           "http://a/b/c/..g"),
			("./../g",        "http://a/b/g"),
			("./g/.",         "http://a/b/c/g/"),
			("g/./h",         "http://a/b/c/g/h"),
			("g/../h",        "http://a/b/c/h"),
			("g;x=1/./y",     "http://a/b/c/g;x=1/y"),
			("g;x=1/../y",    "http://a/b/c/y"),
			("g?y/./x",       "http://a/b/c/g?y/./x"),
			("g?y/../x",      "http://a/b/c/g?y/../x"),
			("g#s/./x",       "http://a/b/c/g#s/./x"),
			("g#s/../x",      "http://a/b/c/g#s/../x"),
			("http:g",        "http:g"), # use the validating version here
		]
		baseurl = url.URL(base)
		for (rel, res) in tests:
			relurl = url.URL(rel)
			resurl = url.URL(res)
			self.assertEqual(baseurl/relurl, resurl, "%r/%r is %r, but should be %r" % (baseurl, relurl, baseurl/relurl, resurl))
			# This checks rdiv
			self.assertEqual(str(baseurl)/relurl, resurl, "%r/%r is %r, but should be %r" % (baseurl, relurl, baseurl/relurl, resurl))

	def test_join(self):
		tests = [
			("http://test.com/index.html", "impress.html", "http://test.com/impress.html"),
			("http://test.com/index.html", "", "http://test.com/index.html"),
			("/bb/cc/", "http:", "http:"),
			("mailto:x@y.z", "index.html", "index.html"),
			("mailto:x@y.z", "", "mailto:x@y.z"),
			("javascript:return ':/:/:';", "index.html", "index.html"),
			("javascript:document.write('http://foo@bar.com:81/foo;bar/bar;foo?x=y#frag');", "index.html", "index.html"),
			("mailto:x@y", "", "mailto:x@y"),
			("http://test.com/gurk/hurz.gif", "/index.html", "http://test.com/index.html"),
			("http://test.com/gurk/hurz.gif", "../", "http://test.com/"),
			("http://test.com/gurk/hurz.gif", "../gurk.gif?foo=bar#nix", "http://test.com/gurk.gif?foo=bar#nix"),
			("http://test.com/gurk/hurz.gif", "../../gurk.gif?foo=bar#nix", "http://test.com/../gurk.gif?foo=bar#nix"),
			("http://test.com/gurk/hurz.gif", "root:gurk.gif", "root:gurk.gif"),
			("root:gurk.gif", "http://test.com/gurk/hurz.gif", "http://test.com/gurk/hurz.gif"),
			("root:gurk/hurz/hinz.gif", "hinz/kunz.gif", "root:gurk/hurz/hinz/kunz.gif"),
			("root:gurk/hurz/hinz.gif", "root:hinz/kunz.gif", "root:hinz/kunz.gif"),
			("http://test.com", "gurk", "http://test.com/gurk")
		]

		for (base, rel, res) in tests:
			baseurl = url.URL(base)
			relurl = url.URL(rel)
			resurl = url.URL(res)
			self.assertEqual(baseurl/relurl, resurl, "%r/%r is %r, but should be %r" % (baseurl, relurl, baseurl/relurl, resurl))
			# This checks rdiv
			self.assertEqual(str(baseurl)/relurl, resurl, "%r/%r is %r, but should be %r" % (baseurl, relurl, baseurl/relurl, resurl))

	def test_normalize(self):
		tests = [
			("", ""),
			("./", ""),
			("/./", "/"),
			("xx", "xx"),
			("xx/yy", "xx/yy"),
			("xx/..", ""),
			("xx/../.", ""),
			("./xx/..", ""),
			("./xx/../.", ""),
			("xx/./..", ""),
			("xx/yy/..", "xx/"),
			("xx//yy/../..", ""),
			("xx//yy/./..", "xx/"),
			("xx//yy//../", "xx/"),
			("xx/../..//", "../"),
			("xx/.././..", ".."), # ".." parts above the root loose their "directoryness", otherwise this would be "../"
			("xx/.", "xx/"),
			("./xx", "xx"),
			("/xx", "/xx"),
			("/./xx", "/xx"),
			("xx/../xx/../xx", "xx"),
		]
		for (u, u2) in tests:
			u = url.URL(u)
			u1 = u.clone()
			u1.path.normalize()
			u2 = url.URL(u2)
			self.assertEqual(u1, u2, "%r normalized is %r, but should be %r" % (u, u1, u2))

	def test_str(self):
		s = "ftp://ftp.livinglogic.de/pub/livinglogic/xist/XIST-42.105.tar.bz2"
		u = url.URL(s)
		self.assertEqual(unicode(u), s)
		self.assertEqual(str(u), s)

	def test_relative(self):
		tests = [
			("./", "./", "./"),
			("cc.html", "./", "./"),
			("./cc.html", "./", "./"),
			("file:./cc.html", "file:./", "./"),
			("root:./cc.html", "file:./", "file:./"),
			("root:xist/Documentation.html", "http://server/", "http://server/"),
			("root:cc.html", "root:", "./"),
			("root:cc.html", "./", "./"),
			("cc.html", "#mark", "#mark"),
			("root:cc.html", "root:#mark", "./#mark"),
			("root:cc.html", "#mark", "#mark"),
			("root:cc.html", "root:cc.html#mark", "#mark"),
			("root:cc.html", "root:dd.html#mark", "dd.html#mark"),
			("root:aa/bb/cc.html", "root:", "../../"),
			("", "", ""),
			("http://server/aa/bb.html", "http://server/aa/cc.html", "cc.html"),
			("/aa/bb.html", "/xx.html", "/xx.html"), # we don't handle URLs without scheme
		]
		for (base, rel, res) in tests:
			baseurl = url.URL(base)
			relurl = url.URL(rel)
			resurl = url.URL(res)
			self.assertEqual(relurl.relative(baseurl), resurl, "%r.relative(%r) is %r, but should be %r" % (relurl, baseurl, relurl.relative(baseurl), resurl))

	def test_query(self):
		u = url.URL("/search?id=13&id")
		self.assertEqual(u.query, "id=13&id")
		self.assertEqual(u.query_parts, 0)

		u.query += "=17"
		self.assertEqual(u.query, "id=13&id=17")
		self.assertEqual(u.query_parts, {u"id": [u"13", u"17"]})

		del u.query_parts["id"]
		u.query_parts["name"] = "gurk"
		self.assertEqual(u.query, "name=gurk")
		self.assertEqual(u.query_parts, {u"name": u"gurk"})

		u.query_parts["name"] = [u"grk"]
		self.assertEqual(u.query, "name=g%c3%bcrk")
		self.assertEqual(u.query_parts, {u"name": [u"grk"]})

		u.query_parts["name"] = u"grk"
		self.assertEqual(u.query, "name=g%c3%bcrk")
		self.assertEqual(u.query_parts, {u"name": u"grk"})

	def test_eq(self):
		u1 = url.URL("HTTP://www.FOO.com/gurk")
		u2 = url.URL("http://www.foo.com:80/gurk")
		self.assertEqual(u1, u2)
		self.assertEqual(hash(u1), hash(u2))

		u1 = url.URL("http://www.foo.com/gurk?id=13&id=17")
		u2 = url.URL("http://www.foo.com/gurk?id=17&id=13")
		self.assertEqual(u1, u2)
		self.assertEqual(hash(u1), hash(u2))

		u1 = url.URL("http://www.foo.com/gurk?gurk=hurz&hinz=kunz")
		u2 = url.URL("http://www.foo.com/gurk?hinz=kunz&gurk=hurz")
		self.assertEqual(u1, u2)
		self.assertEqual(hash(u1), hash(u2))

	def test_withfrag(self):
		u1a = url.URL("x#y")
		u1b = url.URL("x#y")
		u2 = url.URL("x#z")
		self.assertEqual(u1a.withfrag("z"), u2)
		self.assertEqual(u1a, u1b) # make sure withfrag created a new URL

	def test_without(self):
		u1a = url.URL("x#y")
		u1b = url.URL("x#y")
		u2 = url.URL("x")
		u3 = url.URL("x#")
		self.assertEqual(u1a.withoutfrag(), u2)
		self.assertNotEqual(u1a.withoutfrag(), u3)
		self.assertEqual(u1a, u1b) # make sure withfrag created a new URL

	def test_relpathauthority(self):
		u = url.URL("http://www.foo.com/bar/baz;baz")
		u2 = u.clone()
		u2.path = [ map(unicode.upper, seg) for seg in u2.path.segments ]
		self.assert_(not u2.path.isabs)
		self.assertEqual(str(u2), "http://www.foo.com/BAR/BAZ;BAZ")
		del u2.scheme
		del u2.authority
		self.assertEqual(str(u2), "BAR/BAZ;BAZ")


class WindowsTests(unittest.TestCase):
	def test_escape(self):
		self.assertEqual(url.URL("%u0042").file, u"\x42")

		warnings.filterwarnings("error", category=UserWarning)
		self.assertRaises(UserWarning, url.URL, "%u00")
		self.assertRaises(UserWarning, url.URL, "%u00xx")

		warnings.filterwarnings("ignore", category=UserWarning)
		self.assertEqual(url.URL("%u00").file, u"%u00")
		self.assertEqual(url.URL("%u00xx").file, u"%u00xx")


if __name__ == "__main__":
	unittest.main()
