/********************************************************************************
*                                                                               *
*               Condition Variable                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXString.h>
#include <fox/FXStream.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXMutex.h"
#include "FXCondition.h"
using namespace FXEX;
namespace FXEX {

// map
FXIMPLEMENT(FXCondition,FXWaitable,NULL,0)

// FXCondition constructor
FXCondition::FXCondition(FXMutex* m,FXObject *tgt,FXSelector sel) : FXWaitable(tgt,sel) {
  if (!m) fxerror("FXMutex: must specify a real mutex\n");
  conditionMutex=m;
#ifndef WIN32
  FXint status;
  pthread_condattr_t attr;
  status=pthread_condattr_init(&attr);
  FXASSERT(status==0);
  FXMALLOC(&condition,pthread_cond_t,1);
  status=pthread_cond_init((pthread_cond_t*)condition,&attr);
  FXASSERT(status==0);
  pthread_condattr_destroy(&attr);
#else
  condition=CreateMutex();
  FXASSERT(condition!=NULL);
#endif
  }

// FXCondition destructor
FXCondition::~FXCondition() {
#ifndef WIN32
  pthread_cond_destroy((pthread_cond_t*)condition);
  FXFREE(&condition);
#else
  CloseHandle(condition);
#endif
  }

// wait
void FXCondition::wait(){
  if (!conditionMutex->locked())
    fxerror("%s: condition variable requires a locked mutex\n",getClassName());
  FXThreadMutex threadmutex=&((*conditionMutex).mutexHandle);
#ifndef WIN32
  pthread_cond_wait((pthread_cond_t*)condition,(pthread_mutex_t*)threadmutex);
#else
  SignalObjectAndWait(threadmutex,condition,INFINITE,FALSE);
  WaitForSingleObject(threadmutex,INFINITE);
#endif
  FXWaitable::wait();
  }

// wait / pause a thread
FXbool FXCondition::trywait(FXuint ms) {
  if (!conditionMutex->locked())
    fxerror("%s: condition variable requires a locked mutex\n",getClassName());
  FXThreadMutex threadmutex=&((*conditionMutex).mutexHandle);
#ifndef WIN32
  struct timespec ts;
  ts.tv_sec=ms/1000;
  ts.tv_nsec=(ms%1000)*1000000;
  while (1){  // need to handle canceled system call
    FXint rc = pthread_cond_timedwait((pthread_cond_t*)condition,(pthread_mutex_t*)threadmutex,&ts);
    switch (rc) {
      case 0: return FXWaitable::trywait(ms);  // signaled
      case ETIMEDOUT: return FALSE;            // timed out
      }
    }
#else  // defined(WIN32)
  DWORD result;
  result=SignalObjectAndWait(threadmutex,condition,ms,FALSE);
  if (result!=WAIT_OBJECT_0) return FALSE;
  result=WaitForSingleObject(threadmutex,ms);
  if (result!=WAIT_OBJECT_0) return FALSE;
  return FXWaitable::trywait(ms);
#endif
  }

// signal
void FXCondition::activate() {
#ifndef WIN32
  pthread_cond_signal((pthread_cond_t*)condition);
#else 
  FXThreadMutex *threadmutex=conditionMutex->mutexHandle;
  while (1){
    DWORD result = SignalObjectAndWait(&condition,threadmutex,INFINITE,FALSE);
    if(result==WAIT_OBJECT_0) break;
    }
#endif
  FXWaitable::activate();
  }

// broadcast
void FXCondition::activateAll() {
#ifndef WIN32
  pthread_cond_broadcast((pthread_cond_t*)condition);
#else
  PulseEvent(condition);
#endif
  FXWaitable::activateAll();
  }

// set to use a new mutex
void FXCondition::setMutex(FXMutex *m){
  if (!m) fxerror("FXMutex: must specify a real mutex\n");
  if (conditionMutex!=m){
    activateAll();
    conditionMutex=m;
    }
  }

}

