/********************************************************************************
*                                                                               *
*                  Hostname resolver object                                     *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXString.h>
#include <fox/FXStream.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXURL.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXThreadEvent.h"
#include "FXMutex.h"
#include "FXFastMutex.h"
#include "FXAtomic.h"
#include "FXResolver.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 * - uses getaddrinfo() internally, in a child thread
 * - cannot kill a thread which is blocked in getaddrinfo(), must wait until iDNS lookup timeout
 */

// map
FXDEFMAP(FXResolver) FXResolverMap[]={
  FXMAPFUNC(SEL_COMMAND,FXBaseObject::ID_SETVALUE,FXResolver::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXBaseObject::ID_SETSTRINGVALUE,FXResolver::onCmdSetStringValue),
  FXMAPFUNC(SEL_COMMAND,FXBaseObject::ID_GETSTRINGVALUE,FXResolver::onCmdGetStringValue),
  };
FXIMPLEMENT(FXResolver,FXThread,FXResolverMap,ARRAYNUMBER(FXResolverMap))

// serialisation
FXResolver::FXResolver() : FXThread(){
  family=FXSocketFamilyNone;
  type=FXSocketTypeNone;
  fastmutex=new FXFastMutex();
  code=FXIOStatusOk;
  FXCALLOC(sock_addr,struct sockaddr,1);
  }

// ctor
FXResolver::FXResolver(const FXString& host,FXObject *tgt,FXSelector sel) : FXThread(tgt,sel){
  hostname=host;
  family=FXSocketFamilyNone;
  type=FXSocketTypeNone;
  fastmutex=new FXFastMutex();
  code=FXIOStatusOk;
  FXCALLOC(sock_addr,struct sockaddr,1);
  }

// free up all resources
FXResolver::~FXResolver(){
  FXFREE(sock_addr);
  delete fastmutex;
  }

// return status
FXint FXResolver::status(){
  fastmutex->lock();
  FXint c=code;
  fastmutex->unlock();
  return c;
  }

// simple test to see if resolution has occurred
FXbool FXResolver::resolved(){
  fastmutex->lock();
  FXbool r=officialname.length()?TRUE:FALSE;
  fastmutex->unlock();
  return r;
  }

// set to resolve new hostname
void FXResolver::host(const FXString& h){
  fastmutex->lock();
  hostname=h;
  fastmutex->unlock();
  }

// return resolved official/canonomical name
FXString FXResolver::officialName() {
  fastmutex->lock();
  FXString n(officialname);
  fastmutex->unlock();
  return n;
  }

// resolve using a non-blocking implementation...
// if hostname points at the local machine, just look it up using the blocking resolve
// version, since this lookup should be really quick (assuming that the networking stack
// is actually functional)
void FXResolver::resolve(){
  if (isRunning()) fxerror("%s: resolution is already underway.\n",getClassName());
  if (!hostname.length()) fxerror("%s: missing hostname to resolve.\n",getClassName());
  if (hostname=="127.0.0.1" || hostname=="localhost" || "localhost.localdomain" || hostname==FXURL::hostname() || hostname=="0.0.0.0") blockingResolve();
  else start();
  }

// resolve the host info in a worker thread
void FXResolver::run(){
  FXString host;
  FXint fam=PF_INET;
  FXint typ=0;
  fastmutex->lock();
  officialname="";
  code=FXIOStatusOk;
  host=hostname;
  if (family == FXSocketFamilyInet6) fam = PF_INET6;
  if (type == FXSocketTypeTCP) typ = SOCK_STREAM;
  else if (type == FXSocketTypeUDP) typ = SOCK_DGRAM;
  fxmessage("resolver thread resolving: %s\n", host.text());
  fastmutex->unlock();

  struct addrinfo *ainfo;
  struct addrinfo hints;
  hints.ai_flags = 0;
  hints.ai_family = fam;
  hints.ai_socktype = typ;
  hints.ai_protocol = 0;
  hints.ai_addrlen = 0;
  hints.ai_addr = NULL;
  hints.ai_canonname = NULL;
  hints.ai_next = NULL;

  FXint result = ::getaddrinfo(host.text(),NULL,&hints,&ainfo);
  if (result == 0) {
    fastmutex->lock();
    officialname=ainfo->ai_canonname;
    memcpy(sock_addr,ainfo->ai_addr,sizeof(struct sockaddr));
    fastmutex->unlock();
    freeaddrinfo(ainfo);
    signal(SEL_COMMAND);
    }
  else {
    fastmutex->lock();
    code=result;
    fastmutex->unlock();
    signal(SEL_ERROR);
    }
  }

// this method is provided so that we can do local hostname or blocking lookups,
// without needing to start a worker thread
FXbool FXResolver::blockingResolve(){
  FXString host;
  FXint fam=PF_INET;
  FXint typ=0;
  fastmutex->lock();
  officialname="";
  code=FXIOStatusOk;
  host=hostname;
  if (family == FXSocketFamilyInet6) fam = PF_INET6;
  if (type == FXSocketTypeTCP) typ = SOCK_STREAM;
  else if (type == FXSocketTypeUDP) typ = SOCK_DGRAM;
  fxmessage("resolving: %s\n", host.text());
  fastmutex->unlock();

  struct addrinfo *ainfo;
  struct addrinfo hints;
  hints.ai_flags = 0;
  hints.ai_family = fam;
  hints.ai_socktype = typ;
  hints.ai_protocol = 0;
  hints.ai_addrlen = 0;
  hints.ai_addr = NULL;
  hints.ai_canonname = NULL;
  hints.ai_next = NULL;

  FXint result = ::getaddrinfo(host.text(),NULL,&hints,&ainfo);
  FXObject *tgt=getTarget();
  FXSelector msg=getSelector();
  if (result == 0) {
    fastmutex->lock();
    officialname=ainfo->ai_canonname;
    memcpy(sock_addr,ainfo->ai_addr,sizeof(struct sockaddr));
    fastmutex->unlock();
    freeaddrinfo(ainfo);
    if (tgt) tgt->handle(this,FXSEL(SEL_COMMAND,msg),NULL);
    return TRUE;
    }
  else {
    fastmutex->lock();
    code=result;
    fastmutex->unlock();
    if (tgt) tgt->handle(this,FXSEL(SEL_ERROR,msg),NULL);
    return FALSE;
    }

/* FIXME: remove this code, once functional
  if (family != FXSocketFamilyInet)
    fxerror("%s: wrong family - blockingResolve() only supports Inet family.\n",getClassName());

  // get hostname info
  struct hostent * hostaddr;
  FXTRACE((100,"Looking up hostname\n"));
  hostaddr=gethostbyname(hostname.text());

  // On many windows systems, gethostbyname() fails if called with a valid ip address
  // (i.e. "212.212.212.121") The workaround is trying gethostbyaddr() if the previous fails.
  // Linux systems don't have this problem, as man page for gethostbyname() confirms.
  // Since, Linux is not the only unix platform, we use the WIN32 fix under unix as well.
  // Note: we can't use inet_aton instead of the obsolete inet_addr, since windows programmers
  // didn't like it...
#ifndef WIN32
  if (hostaddr == NULL) {
    struct in_addr adr = {0};
    FXint result = inet_aton(hostname.text(),&adr);
    if(result != 0)
      hostaddr = gethostbyaddr((const char *) &(adr.s_addr), sizeof(adr.s_addr), AF_INET);
    }
  if (hostaddr == NULL) {
//    code=h_errno;
    return FALSE;
    }
#else
  if (hostaddr == NULL) {
    unsigned long adr = inet_addr(hostname.text());
    if(adr!=INADDR_NONE) hostaddr = gethostbyaddr((const char *) &adr, sizeof(adr), AF_INET);
    }
  if (hostaddr == NULL) {
 //   code=ERRNO;
    return FALSE;
    }
#endif

  FXString name(hostaddr->h_addr_list[0], hostaddr->h_length );
  fastmutex->lock();
  officialname=name;
  fastmutex->unlock();
  FXObject *tgt=getTarget();
  FXSelector msg=getSelector();
  if (tgt) tgt->handle(this,FXSEL(SEL_COMMAND,msg),NULL);
  return TRUE;
*/
  }

// return reference to internal sockaddr structure
void* FXResolver::socket_address_struct(){
  return (void*)sock_addr;
  }

// set socket family
void FXResolver::setFamily(FXSocketFamily family){
  if(isRunning()) fxerror("%s: cannot set socket family while looking up hostname.\n",getClassName());
  this->family=family;
  }

// get socket family
FXSocketFamily FXResolver::getFamily(){
  fastmutex->lock();
  FXSocketFamily f=family;
  fastmutex->unlock();
  return f;
  }

// set socket type
void FXResolver::setType(FXSocketType type){
  if(isRunning()) fxerror("%s: cannot set socket type while looking up hostname.\n",getClassName());
  this->type=type;
  }

// get socket type
FXSocketType FXResolver::getType(){
  fastmutex->lock();
  FXSocketType t=type;
  fastmutex->unlock();
  return t;
  }

// set hostname to resolve
long FXResolver::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  if (!isRunning()) host((char*)ptr);
  return 1;
  }

// set the hostname to resolve
long FXResolver::onCmdSetStringValue(FXObject*,FXSelector,void *ptr){
  if (!isRunning()) host(*((FXString*)ptr));
  return 1;
  }

// return the official hostname, if set, otherwise return the origonal hostname
long FXResolver::onCmdGetStringValue(FXObject*,FXSelector,void *ptr){
  fastmutex->lock();
  if (officialname.length()) *((FXString*)ptr)=officialname;
  else *((FXString*)ptr)=hostname;
  fastmutex->unlock();
  return 1;
  }

}

