/********************************************************************************
*                                                                               *
*                   Base type for thread objects                                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXFastMutex.h"
#include "FXAtomic.h"
#include "FXMutex.h"
#include "FXThread.h"
using namespace FXEX;
namespace FXEX {

// maps
FXDEFMAP(FXThread) FXThreadMap[]={
  FXMAPFUNC(SEL_COMMAND,FXThread::ID_STOP,FXThread::onStop),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXThread::onCmdGetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXThread::onCmdSetIntValue),
  };
FXIMPLEMENT_ABSTRACT(FXThread,FXRunnable,FXThreadMap,ARRAYNUMBER(FXThreadMap))

// The pthread implementation needs a 'void*' to start executing... WIN32 needs something similar...
// Thus we need this helper routine.
#ifndef WIN32
extern "C" void* ThreadSetup(void *ptr){
#else
extern "C" LPTHREAD_START_ROUTINE ThreadSetup(void *ptr){
#endif
  FXThread *th=(FXThread*)ptr;
  th->signal(SEL_CREATE);
  th->run();
  th->reset();
  th->signal(SEL_DELETE);
  return NULL;
  }

// construct a new thread, executing the functionality in a derived class
FXThread::FXThread(FXObject *tgt,FXSelector sel) : FXRunnable(tgt,sel) {
  thread=NULL;
  stopThread=new FXAtomicInt();
  }

// if the thread is running, wait for it to stop, before freeing this objects' resources
FXThread::~FXThread(){
  wake();
  while (isRunning()) {
    stop();
    fxsleep(100);
    }
  delete stopThread;
  }

// start the thread running
void FXThread::start(){
  mutex->lock();
  if (thread) fxerror("%s: trying to start thread while existing thread is still running\n",getClassName());
  (*stopThread)=0;
#ifndef WIN32
  FXMALLOC(&thread,pthread_t,1);
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED);
  if (pthread_create((pthread_t*)thread,&attr,ThreadSetup,this) != 0 )
    fxerror("%s failure: Could not create thread\n",getClassName());
  pthread_attr_destroy(&attr);
#else
  DWORD tid;
  if((thread=CreateThread(NULL,0,ThreadSetup,this,NULL,&tid))==NULL)
    fxerror("%s failure: Could not create thread\n",getClassName());
#endif
  mutex->unlock();
  }

// request to stop the thread
void FXThread::stop(){
  (*stopThread)++;
  }

// is the thread running
FXbool FXThread::isRunning(){
  mutex->lock();
  FXbool t=thread!=NULL;
  mutex->unlock();
  return t;
  }

// reset the thread
void FXThread::reset(){
  mutex->lock();
  if (thread){
#ifdef WIN32
    CloseHandle(thread);
    thread=NULL;
#else
    FXFREE(&thread);
#endif
    }
  mutex->unlock();
  while (*stopThread) (*stopThread)--;
  }

// yield the thread
void FXThread::yield(){
#ifndef WIN32
  pthread_yield();
#else
  FXRunnable::sleep(0);
#endif
  }

// stop thread request
long FXThread::onStop(FXObject*,FXSelector,void*){
  stop();
  return 1;
  }

// get the current running state
long FXThread::onCmdGetIntValue(FXObject*,FXSelector,void *ptr){
  FXint *i=(FXint*)ptr;
  *i=isRunning();
  return 1;
  }

// handle request to stop the running state
long FXThread::onCmdSetIntValue(FXObject*,FXSelector,void *ptr){
  FXint *i=(FXint*)ptr;
  *stopThread=*i;
  return 1;
  }

}
