/********************************************************************************
*                                                                               *
*             E x t e n d e d   C o m b o   B o x   W i d g e t                 *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998,2002 by Jeroen van der Zijp.   All Rights Reserved.        *
* Modified      2002,2003 by Johan Meijdam,         VORtech Computing.          *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <fox/xincs.h>
#include <fox/fxver.h>
#include <fox/fxdefs.h>
#include <fox/fxkeys.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXSettings.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXId.h>
#include <fox/FXDrawable.h>
#include <fox/FXDC.h>
#include <fox/FXWindow.h>
#include <fox/FXFrame.h>
#include <fox/FXLabel.h>
#include <fox/FXTextField.h>
#include <fox/FXButton.h>
#include <fox/FXMenuButton.h>
#include <fox/FXComposite.h>
#include <fox/FXPacker.h>
#include <fox/FXShell.h>
#include <fox/FXPopup.h>
#include <fox/FXScrollBar.h>
#include <fox/FXScrollArea.h>
#include <fox/FXList.h>
#include <fox/FXComboBox.h>
#include <fox/FXFont.h>
#include <fox/FXIconList.h>
using namespace FX;
#include "FXComboBoxEx.h"
using namespace FXEX;

/*
  Notes:
  - Handling typed text:
    a) Pass string to target only.
    b) Pass string to target & add to list [begin, after/before current, or end].
    c) Pass string to target & replace current item's label.
  - FXComboBoxEx is a text field which may be filled from an FXIconList.
  - FXComboBoxEx is a text field which in turn may fill an FXIconList also.
  - In most other respects, it behaves like a FXTextField.
  - Need to catch up/down arrow keys.
  - Combobox turns OFF GUI Updating while being manipulated.
  - Need some way to size the FXIconList automatically to the number of items.
  - If you leave the list then getCurrentItem() returns the last item under
    cursor which is not the same item shown in FXComboBoxEx.
  - Combobox should also issue notify message, probably.
    FXComboBoxEx::getItem() and FXComboBoxEx::getItemText() are the same; this
    should probably change.
*/

#define COMBOBOX_INS_MASK   (COMBOBOX_REPLACE|COMBOBOX_INSERT_BEFORE|COMBOBOX_INSERT_AFTER|COMBOBOX_INSERT_FIRST|COMBOBOX_INSERT_LAST)
#define COMBOBOX_MASK       (COMBOBOX_STATIC|COMBOBOX_INS_MASK)

static const FXint LINE_SPACING = 4;   // Line spacing between items

/*******************************************************************************/

namespace FXEX {

// Map
FXDEFMAP(FXComboBoxEx) FXComboBoxExMap[]={
  FXMAPFUNC(SEL_FOCUS_UP,0,FXComboBoxEx::onFocusUp),
  FXMAPFUNC(SEL_FOCUS_DOWN,0,FXComboBoxEx::onFocusDown),
  FXMAPFUNC(SEL_FOCUS_SELF,0,FXComboBoxEx::onFocusSelf),
  FXMAPFUNC(SEL_UPDATE,FXComboBoxEx::ID_TEXT,FXComboBoxEx::onUpdFmText),
  FXMAPFUNC(SEL_CLICKED,FXComboBoxEx::ID_LIST,FXComboBoxEx::onListClicked),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,FXComboBoxEx::ID_TEXT,FXComboBoxEx::onTextButton),
  FXMAPFUNC(SEL_CHANGED,FXComboBoxEx::ID_TEXT,FXComboBoxEx::onTextChanged),
  FXMAPFUNC(SEL_COMMAND,FXComboBoxEx::ID_TEXT,FXComboBoxEx::onTextCommand),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETREALVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETREALVALUE,FXComboBoxEx::onFwdToText),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,FXComboBoxEx::onFwdToText),
  };


// Object implementation
FXIMPLEMENT(FXComboBoxEx,FXPacker,FXComboBoxExMap,ARRAYNUMBER(FXComboBoxExMap))

// List box
FXComboBoxEx::FXComboBoxEx(FXComposite *p,FXint cols,FXint nvis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXPacker(p,opts,x,y,w,h, 0,0,0,0, 0,0){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  field=new FXTextField(this,cols,this,FXComboBoxEx::ID_TEXT,0, 0,0,0,0, pl,pr,pt,pb);
  if(options&COMBOBOX_STATIC) field->setEditable(FALSE);
  pane=new FXPopup(this,FRAME_LINE);
  list=new FXIconList(pane,this,FXComboBoxEx::ID_LIST,ICONLIST_ROWS|ICONLIST_BROWSESELECT|ICONLIST_AUTOSIZE|ICONLIST_DETAILED|LAYOUT_FILL_X|LAYOUT_FILL_Y|SCROLLERS_TRACK|HSCROLLER_NEVER);
  if(options&COMBOBOX_STATIC) list->setScrollStyle(SCROLLERS_TRACK|HSCROLLING_OFF);
  appendHeader("",NULL,getWidth()); // Default, empty header with width of the control
  setNumVisible(nvis);  
  button=new FXMenuButton(this,NULL,NULL,pane,FRAME_RAISED|FRAME_THICK|MENUBUTTON_DOWN|MENUBUTTON_ATTACH_RIGHT, 0,0,0,0, 0,0,0,0);
  button->setXOffset(border);
  button->setYOffset(border);
  flags&=~FLAG_UPDATE;  // Never GUI update
  }


// Createwindow
void FXComboBoxEx::create(){
  FXPacker::create();
  pane->create();
  }


// Detach window
void FXComboBoxEx::detach(){
  FXPacker::detach();
  pane->detach();
  }


// Destroy window
void FXComboBoxEx::destroy(){
  pane->destroy();
  FXPacker::destroy();
  }


// Enable the window
void FXComboBoxEx::enable(){
  if(!(flags&FLAG_ENABLED)){
    FXPacker::enable();
    field->enable();
    button->enable();
    }
  }


// Disable the window
void FXComboBoxEx::disable(){
  if(flags&FLAG_ENABLED){
    FXPacker::disable();
    field->disable();
    button->disable();
    }
  }


// Get default width
FXint FXComboBoxEx::getDefaultWidth(){
  FXint ww,pw;
  ww=field->getDefaultWidth()+button->getDefaultWidth()+(border<<1);
  pw=pane->getDefaultWidth();
  return FXMAX(ww,pw);
  }


// Get default height
FXint FXComboBoxEx::getDefaultHeight(){
  FXint th,bh;
  th=field->getDefaultHeight();
  bh=button->getDefaultHeight();
  return FXMAX(th,bh)+(border<<1);
  }


// Recalculate layout
void FXComboBoxEx::layout(){
  FXint buttonWidth,textWidth,itemHeight;
  itemHeight=height-(border<<1);
  buttonWidth=button->getDefaultWidth();
  textWidth=width-buttonWidth-(border<<1);
  field->position(border,border,textWidth,itemHeight);
  button->position(border+textWidth,border,buttonWidth,itemHeight);
  FXint paneheight=(numvisible+1)*(LINE_SPACING+list->getFont()->getFontHeight());
  FXint panewidth=0;
  for (FXint i=0; i<list->getNumHeaders(); ++i) panewidth += list->getHeaderSize(i);
  pane->resize(panewidth,paneheight);
  flags&=~FLAG_DIRTY;
  }


// Forward GUI update of text field to target; but only if pane is not popped
long FXComboBoxEx::onUpdFmText(FXObject*,FXSelector,void*){
  return target && !isPaneShown() && target->handle(this,FXSEL(SEL_UPDATE,message),NULL);
  }


// Command handled in the text field
long FXComboBoxEx::onFwdToText(FXObject* sender,FXSelector sel,void* ptr){
  return field->handle(sender,sel,ptr);
  }


// Forward clicked message from list to target
long FXComboBoxEx::onListClicked(FXObject*,FXSelector,void* ptr){
  button->handle(this,FXSEL(SEL_COMMAND,ID_UNPOST),NULL);    // Unpost the list
  if(0<=((FXint)(FXival)ptr)){
    field->setText(getTextUntilFirstTab(list->getItemText((FXint)(FXival)ptr)));
    if(target){target->handle(this,FXSEL(SEL_COMMAND,message),(void*)getText().text());}
    }
  return 1;
  }


// Pressed left button in text field
long FXComboBoxEx::onTextButton(FXObject*,FXSelector,void*){
  if(options&COMBOBOX_STATIC){
    button->handle(this,FXSEL(SEL_COMMAND,ID_POST),NULL);    // Post the list
    return 1;
    }
  return 0;
  }


// Text has changed
long FXComboBoxEx::onTextChanged(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_CHANGED,message),ptr);
  }


// Text has changed
long FXComboBoxEx::onTextCommand(FXObject*,FXSelector,void* ptr){
  FXint index=list->getCurrentItem();
  if(!(options&COMBOBOX_STATIC)){
    switch(options&COMBOBOX_INS_MASK){
      case COMBOBOX_REPLACE:
        if(0<=index) setItem(index,(FXchar*)ptr,getItemData(index));
        break;
      case COMBOBOX_INSERT_BEFORE:
        if(0<=index) insertItem(index,(FXchar*)ptr);
        break;
      case COMBOBOX_INSERT_AFTER:
        if(0<=index) insertItem(index+1,(FXchar*)ptr);
        break;
      case COMBOBOX_INSERT_FIRST:
        insertItem(0,(FXchar*)ptr);
        break;
      case COMBOBOX_INSERT_LAST:
        appendItem((FXchar*)ptr);
        break;
      }
    }
  return target && target->handle(this,FXSEL(SEL_COMMAND,message),ptr);
  }


// Bounce focus to the text field
long FXComboBoxEx::onFocusSelf(FXObject* sender,FXSelector,void* ptr){
  return field->handle(sender,FXSEL(SEL_FOCUS_SELF,0),ptr);
  }


// Select upper item
long FXComboBoxEx::onFocusUp(FXObject*,FXSelector,void*){
  FXTRACE((100,"FXComboBoxEx::onFocusUp\n"));
  FXint index=getCurrentItem();
  if(index<0) index=getNumItems()-1;
  else if(0<index) index--;
  if(0<=index && index<getNumItems()){
    setCurrentItem(index);
    if(target){target->handle(this,FXSEL(SEL_COMMAND,message),(void*)getText().text());}
    }
  return 1;
  }


// Select lower item
long FXComboBoxEx::onFocusDown(FXObject*,FXSelector,void*){
  FXTRACE((100,"FXComboBoxEx::onFocusDown\n"));
  FXint index=getCurrentItem();
  if(index<0) index=0;
  else if(index<getNumItems()-1) index++;
  if(0<=index && index<getNumItems()){
    setCurrentItem(index);
    if(target){target->handle(this,FXSEL(SEL_COMMAND,message),(void*)getText().text());}
    }
  return 1;
  }


// Return true if editable
FXbool FXComboBoxEx::isEditable() const {
  return field->isEditable();
  }


// Set widget is editable or not
void FXComboBoxEx::setEditable(FXbool edit){
  field->setEditable(edit);
  }


// Set text
void FXComboBoxEx::setText(const FXString& text){
  field->setText(text);
  }


// Obtain text
FXString FXComboBoxEx::getText() const {
  return field->getText();
  }


// Set number of text columns
void FXComboBoxEx::setNumColumns(FXint cols){
  field->setNumColumns(cols);
  }


// Get number of text columns
FXint FXComboBoxEx::getNumColumns() const {
  return field->getNumColumns();
  }


// Get number of items
FXint FXComboBoxEx::getNumItems() const {
  return list->getNumItems();
  }


// Get number of visible items
FXint FXComboBoxEx::getNumVisible() const {
   return numvisible; 
}


// Set number of visible items
void FXComboBoxEx::setNumVisible(FXint nvis){
   numvisible=nvis;
   recalc();
  }


// Is item current
FXbool FXComboBoxEx::isItemCurrent(FXint index) const {
  return list->isItemCurrent(index);
  }


// Change current item
void FXComboBoxEx::setCurrentItem(FXint index){
  list->setCurrentItem(index);
  if(0<=index){
    setText(getTextUntilFirstTab(list->getItemText(index)));
    }
  else{
    setText(FXString::null);
    }
  }


// Get current item
FXint FXComboBoxEx::getCurrentItem() const {
  return list->getCurrentItem();
  }


// Retrieve item
FXString FXComboBoxEx::getItem(FXint index) const {
  return getTextUntilFirstTab(list->getItem(index)->getText());
  }

// Replace text of item at index
void FXComboBoxEx::setItem(FXint index,const FXString& text,void* ptr){
  list->setItem(index,text,NULL,NULL,ptr);
  if(isItemCurrent(index)){
    field->setText(getTextUntilFirstTab(text));
    }
  recalc();
  }


// Insert item at index
void FXComboBoxEx::insertItem(FXint index,const FXString& text,void* ptr){
  list->insertItem(index,text,NULL,NULL,ptr);
  if(isItemCurrent(index)){
    field->setText(getTextUntilFirstTab(text));
    }
  recalc();
  }


// Append item
void FXComboBoxEx::appendItem(const FXString& text,void* ptr){
  list->appendItem(text,NULL,NULL,ptr);
  if(isItemCurrent(getNumItems()-1)){
    field->setText(getTextUntilFirstTab(text));
    }
  recalc();
  }


// Prepend item
void FXComboBoxEx::prependItem(const FXString& text,void* ptr){
  list->prependItem(text,NULL,NULL,ptr);
  if(isItemCurrent(0)){
    field->setText(getTextUntilFirstTab(text));
    }
  recalc();
  }


// Replace text of item at index
void FXComboBoxEx::setItem(FXint index,FXIconItem* item){
  list->setItem(index,item);
  if(isItemCurrent(index)){
    field->setText(getTextUntilFirstTab(item->getText()));
    }
  recalc();
  }


// Insert item at index
void FXComboBoxEx::insertItem(FXint index,FXIconItem* item){
  list->insertItem(index,item);
  if(isItemCurrent(index)){
    field->setText(getTextUntilFirstTab(item->getText()));
    }
  recalc();
  }


// Append item
void FXComboBoxEx::appendItem(FXIconItem* item){
  list->appendItem(item);
  if(isItemCurrent(getNumItems()-1)){
    field->setText(getTextUntilFirstTab(item->getText()));
    }
  recalc();
  }


// Prepend item
void FXComboBoxEx::prependItem(FXIconItem* item){
  list->prependItem(item);
  if(isItemCurrent(0)){
    field->setText(getTextUntilFirstTab(item->getText()));
    }
  recalc();
  }


// Remove given item
void FXComboBoxEx::removeItem(FXint index){
  FXint current=list->getCurrentItem();
  list->removeItem(index);
  if(index==current){
    current=list->getCurrentItem();
    if(0<=current){
      field->setText(getTextUntilFirstTab(list->getItemText(current)));
      }
    else{
      field->setText(FXString::null);
      }
    }
  recalc();
  }


// Remove all items
void FXComboBoxEx::clearItems(){
  field->setText(FXString::null);
  list->clearItems();
  recalc();
  }


// Get item by name
FXint FXComboBoxEx::findItem(const FXString& text,FXint start,FXuint iflags) const {
  return list->findItem(text,start,iflags);
  }


// Set item text
void FXComboBoxEx::setItemText(FXint index,const FXString& txt){
  if(isItemCurrent(index)) setText(txt);
  list->setItemText(index,txt);
  recalc();
  }


// Get item text
FXString FXComboBoxEx::getItemText(FXint index) const {
  return getTextUntilFirstTab(list->getItemText(index));
  }


// Set item data
void FXComboBoxEx::setItemData(FXint index,void* ptr) const {
  list->setItemData(index,ptr);
  }


// Get item data
void* FXComboBoxEx::getItemData(FXint index) const {
  return list->getItemData(index);
  }


// Is the pane shown
FXbool FXComboBoxEx::isPaneShown() const {
  return pane->shown();
  }


// Set font
void FXComboBoxEx::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  field->setFont(fnt);
  list->setFont(fnt);
  recalc();
  }


// Obtain font
FXFont* FXComboBoxEx::getFont() const {
  return field->getFont();
  }


// Change combobox style
void FXComboBoxEx::setComboStyle(FXuint mode){
  FXuint opts=(options&~COMBOBOX_MASK)|(mode&COMBOBOX_MASK);
  if(opts!=options){
    options=opts;
    if(options&COMBOBOX_STATIC){
      field->setEditable(FALSE);                                // Non-editable
      list->setScrollStyle(SCROLLERS_TRACK|HSCROLLING_OFF);     // No scrolling
      }
    else{
      field->setEditable(TRUE);                                 // Editable
      list->setScrollStyle(SCROLLERS_TRACK|HSCROLLER_NEVER);    // Scrollable, but no scrollbar
      }
    recalc();
    }
  }


// Get combobox style
FXuint FXComboBoxEx::getComboStyle() const {
  return (options&COMBOBOX_MASK);
  }


// Set window background color
void FXComboBoxEx::setBackColor(FXColor clr){
  field->setBackColor(clr);
  list->setBackColor(clr);
  }


// Get background color
FXColor FXComboBoxEx::getBackColor() const {
  return field->getBackColor();
  }


// Set text color
void FXComboBoxEx::setTextColor(FXColor clr){
  field->setTextColor(clr);
  list->setTextColor(clr);
  }


// Return text color
FXColor FXComboBoxEx::getTextColor() const {
  return field->getTextColor();
  }


// Set select background color
void FXComboBoxEx::setSelBackColor(FXColor clr){
  field->setSelBackColor(clr);
  list->setSelBackColor(clr);
  }


// Return selected background color
FXColor FXComboBoxEx::getSelBackColor() const {
  return field->getSelBackColor();
  }


// Set selected text color
void FXComboBoxEx::setSelTextColor(FXColor clr){
  field->setSelTextColor(clr);
  list->setSelTextColor(clr);
  }


// Return selected text color
FXColor FXComboBoxEx::getSelTextColor() const {
  return field->getSelTextColor();
  }


// Sort items using current sort function
void FXComboBoxEx::sortItems(){
  list->sortItems();
  }


// Return sort function
FXIconListSortFunc FXComboBoxEx::getSortFunc() const {
  return list->getSortFunc();
  }


// Change sort function
void FXComboBoxEx::setSortFunc(FXIconListSortFunc func){
  list->setSortFunc(func);
  }


// Set help text
void FXComboBoxEx::setHelpText(const FXString& txt){
  field->setHelpText(txt);
  }

// Get help text
FXString FXComboBoxEx::getHelpText() const {
  return field->getHelpText();
  }


// Set tip text
void FXComboBoxEx::setTipText(const FXString& txt){
  field->setTipText(txt);
  }


// Get tip text
FXString FXComboBoxEx::getTipText() const {
  return field->getTipText();
  }


// Save object to stream
void FXComboBoxEx::save(FXStream& store) const {
  FXPacker::save(store);
  store << field;
  store << button;
  store << list;
  store << pane;
  }


// Load object from stream
void FXComboBoxEx::load(FXStream& store){
  FXPacker::load(store);
  store >> field;
  store >> button;
  store >> list;
  store >> pane;
  }


// Delete it
FXComboBoxEx::~FXComboBoxEx(){
  delete pane;
  pane=(FXPopup*)-1;
  field=(FXTextField*)-1;
  button=(FXMenuButton*)-1;
  list=(FXIconList*)-1;
  }


// Return the text from the given string until the first tab character
// Return the given string if it doesn't contain a tab character
FXString FXComboBoxEx::getTextUntilFirstTab(const FXString& in){
  FXint pos = in.find('\t');
  if (pos>0) return in.left(pos);
  return in;
  }

}
