/********************************************************************************
*                                                                               *
*                  Unix Daemon application object                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fx.h>
#include <fox/fxdefs.h>
#include "exincs.h"
#include "FXDaemonApp.h"
using namespace FXEX;

// Just in case
#define bzero(ptr,size) memset(ptr,0,size)

// Regular define
#define SELECT(n,r,w,e,t)  select(n,r,w,e,t)

/*******************************************************************************/
namespace FX {

// Callback Record
struct FXCBSpec {
  FXObject      *target;            // Receiver object
  FXSelector     message;           // Message sent to receiver
  };

// Timer record
struct FXTimer {
  FXTimer       *next;              // Next timeout in list
  FXObject      *target;            // Receiver object
  FXSelector     message;           // Message sent to receiver
  struct timeval due;               // When timer is due
  };

// Signal record
struct FXSignal {
  FXObject      *target;            // Receiver object
  FXSelector     message;           // Message sent to receiver
  FXbool         handlerset;        // Handler was already set
  FXbool         notified;          // Signal has fired
  };

// Idle record
struct FXChore {
  FXChore       *next;              // Next chore in list
  FXObject      *target;            // Receiver object
  FXSelector     message;           // Message sent to receiver
  };

// Input record
struct FXInput {
  FXCBSpec       read;              // Callback spec for read
  FXCBSpec       write;             // Callback spec for write
  FXCBSpec       excpt;             // Callback spec for except
  };

// Recursive Event Loop Invocation
struct FXInvocation {
  FXInvocation **invocation;  // Pointer to variable holding pointer to current invocation
  FXInvocation  *upper;       // Invocation above this one
  FXWindow      *window;      // Modal window (if any)
  FXModality     modality;    // Modality mode
  FXint          code;        // Return code
  FXbool         done;        // True if breaking out

  // Enter modal loop
  FXInvocation(FXInvocation** inv,FXModality mode=MODAL_FOR_NONE,FXWindow* win=NULL):invocation(inv),upper(*inv),window(win),modality(mode),code(0),done(FALSE){
    *invocation=this;
    }

  // Exit modal loop
 ~FXInvocation(){
    *invocation=upper;
    }
  };


// data structures/members for SysV message queue
struct SysVmsgbuffer {
  long           mtype;             // SysV message type - must be greater than 0
  char          *mtext;             // the data to send
  };

struct FXSysVdata {
  FXint          bytesread;         // the number of bytes that have been written into the buffer
  SysVmsgbuffer  buffer;            // the place where the system call stores the data
  };

}; // namespace FX

namespace FXEX {

// System V IPC message queue record
struct FXSysVMessage {
  FXSysVMessage *next;              // Next SysV message queue
  FXObject      *target;            // Receiver object
  FXSelector     message;           // Message sent to receiver
  FXint          msgid;             // SysV message ID
  FXint          msgtype;           // SysV message type
  FXint          maxsize;           // max size of SysV message for this ID/type
  FXSysVdata     data;              // data received
  };


/*******************************************************************************/


// Application object
FXDaemonApp* FXDaemonApp::app=NULL;

// Version number that the library has been compiled with
const FXuchar FXDaemonApp::version[3]={FOX_MAJOR,FOX_MINOR,FOX_LEVEL};

// Copyright notice
const FXuchar FXDaemonApp::copyright[80]="Copyright (C) 1997,2003 Jeroen van der Zijp. All Rights Reserved.";

// Largest number of signals on this system
const FXint FXDaemonApp::MAXSIGNALS = 64;


/*******************************************************************************/

// Map
FXDEFMAP(FXDaemonApp) FXDaemonAppMap[]={
  FXMAPFUNC(SEL_COMMAND,FXDaemonApp::ID_QUIT,FXDaemonApp::onCmdQuit),
  FXMAPFUNC(SEL_COMMAND,FXDaemonApp::ID_DUMP,FXDaemonApp::onCmdDump),
  };
FXIMPLEMENT(FXDaemonApp,FXObject,FXDaemonAppMap,ARRAYNUMBER(FXDaemonAppMap))

/*******************************************************************************/

// Initialize application object
FXDaemonApp::FXDaemonApp(const FXString& name,const FXString& vendor):registry(name,vendor){

  // Test if application object already exists
  if(app){fxerror("Warning: Trying to construct multiple application objects.\n");}

  // Initialize private platform independent data
  timers=NULL;
  chores=NULL;
  timerrecs=NULL;
  chorerecs=NULL;
  invocation=NULL;                        // Modal loop invocation
  FXCALLOC(&inputs,FXInput,8);            // Input file descriptors
  ninputs=8;                              // Number of these
  maxinput=-1;                            // Maximum input number
  signals=NULL;                           // Signals array
  nsignals=0;                             // Number of signal handlers set
  initialized=FALSE;                      // Not yet initialized
  sysVmessages=NULL;

  // File descriptors
  FXCALLOC(&r_fds,fd_set,1);              // Read File Descriptor set
  FXCALLOC(&w_fds,fd_set,1);              // Write File Descriptor set
  FXCALLOC(&e_fds,fd_set,1);              // Except File Descriptor set

  // Pointer to FXDaemonApp
  app=this;
  }

/*******************************************************************************/


// Compare times
static inline int operator<(const struct timeval& a,const struct timeval& b){
  return (a.tv_sec<b.tv_sec) || (a.tv_sec==b.tv_sec && a.tv_usec<b.tv_usec);
  }

// Add timeout, sorted by time
FXTimer* FXDaemonApp::addTimeout(FXint ms,FXObject* tgt,FXSelector sel){
  if(ms<1){ fxerror("%s::addTimeout: wait time should be greater than 0\n",getClassName()); }
  FXTimer *t;
  if(timerrecs){
    t=timerrecs;
    timerrecs=t->next;
    }
  else{
    t=new FXTimer;
    }
  t->target=tgt;
  t->message=sel;
  gettimeofday(&t->due,NULL);
  t->due.tv_sec+=ms/1000;
  t->due.tv_usec+=(ms%1000)*1000;
  if(t->due.tv_usec>=1000000){
    t->due.tv_usec-=1000000;
    t->due.tv_sec+=1;
    }
  FXTimer **hh=&timers;
  FXTimer *h=*hh;
  while(h && (h->due < t->due)){
    hh=&h->next;
    h=*hh;
    }
  t->next=h;
  *hh=t;
  return t;
  }


// Remove timeout from the list
FXTimer* FXDaemonApp::removeTimeout(FXTimer *t){
  if(t){
    FXTimer *h,**hh;
    for(h=timers,hh=&timers; h!=t; hh=&h->next,h=h->next){
      if(h==NULL) return NULL;
      }
    FXASSERT(h==t);
    *hh=t->next;
    t->next=timerrecs;
    timerrecs=t;
    }
  return NULL;
  }


/*******************************************************************************/


// Signal handler; note this is a single write operation
// which can not be interrupted by another handler!
void FXDaemonApp::signalhandler(int sig){
  app->signals[sig].notified=TRUE;
  }


// Add a signal message
void FXDaemonApp::addSignal(FXint sig,FXObject* tgt,FXSelector sel,FXuint flags){
  void (*handler)(int);
  if(sig<0 || MAXSIGNALS<sig){ fxerror("%s::addSignal: bad signal number\n",getClassName()); }

  // First signal added allocates the array
  if(nsignals==0){FXCALLOC(&signals,FXSignal,MAXSIGNALS);}

  // May have just changed the message and/or target
  signals[sig].target=tgt;
  signals[sig].message=sel;

  // Has handler been set?
  if(!signals[sig].handlerset){
    handler=signalhandler;
    struct sigaction sigact;
    sigact.sa_handler=handler;
    sigemptyset(&sigact.sa_mask);
    sigact.sa_flags=flags;
    if(sigaction(sig,&sigact,NULL)!=0){ fxwarning("%s::addSignal: error setting signal handler\n",getClassName()); }
    signals[sig].notified=FALSE;
    signals[sig].handlerset=TRUE;
    nsignals++;
    }
  }


// Remove all signal messages for signal sig
void FXDaemonApp::removeSignal(FXint sig){
  if(sig<0 || MAXSIGNALS<sig){ fxerror("%s::removeSignal: bad signal number\n",getClassName()); }

  // Has a handler been set?
  if(signals[sig].handlerset){
    struct sigaction sigact;
    sigact.sa_handler=SIG_DFL;
    sigemptyset(&sigact.sa_mask);
    sigact.sa_flags=0;
    if(sigaction(sig,&sigact,NULL)!=0){ fxwarning("%s::removeSignal: error removing signal handler\n",getClassName()); }
    signals[sig].target=NULL;
    signals[sig].message=0;
    signals[sig].handlerset=FALSE;
    signals[sig].notified=FALSE;
    nsignals--;

    // Last signal removed deletes the array
    if(nsignals==0){FXFREE(&signals);}
    }
  }


/*******************************************************************************/


// Add chore to the END of the list
FXChore* FXDaemonApp::addChore(FXObject* tgt,FXSelector sel){
  register FXChore *c,**cc;
  if(chorerecs){
    c=chorerecs;
    chorerecs=c->next;
    }
  else{
    c=new FXChore;
    }
  for(cc=&chores; *cc; cc=&(*cc)->next);
  c->target=tgt;
  c->message=sel;
  c->next=NULL;
  *cc=c;
  return c;
  }


// Remove chore from the list
FXChore* FXDaemonApp::removeChore(FXChore *c){
  register FXChore *h,**hh;
  if(c){
    for(h=chores,hh=&chores; h!=c; hh=&h->next,h=h->next){
      if(h==NULL) return NULL;
      }
    FXASSERT(h==c);
    *hh=c->next;
    c->next=chorerecs;
    chorerecs=c;
    }
  return NULL;
  }


/*******************************************************************************/


// Add input
FXbool FXDaemonApp::addInput(FXInputHandle fd,FXuint mode,FXObject *tgt,FXSelector sel){
  if(mode==INPUT_NONE) return FALSE;
  if(fd<0 || fd>=FD_SETSIZE) return FALSE;
  if(fd>=ninputs){                    // Grow table of callbacks
    FXRESIZE(&inputs,FXInput,fd+1);
    memset(&inputs[ninputs],0,sizeof(FXInput)*(fd+1-ninputs));
    ninputs=fd+1;
    }
  FXASSERT(inputs);
  FXASSERT(fd<ninputs);
  if(mode&INPUT_READ){
    inputs[fd].read.target=tgt;
    inputs[fd].read.message=sel;
    FD_SET(fd,(fd_set*)r_fds);
    }
  if(mode&INPUT_WRITE){
    inputs[fd].write.target=tgt;
    inputs[fd].write.message=sel;
    FD_SET(fd,(fd_set*)w_fds);
    }
  if(mode&INPUT_EXCEPT){
    inputs[fd].excpt.target=tgt;
    inputs[fd].excpt.message=sel;
    FD_SET(fd,(fd_set*)e_fds);
    }
  if(fd>maxinput) maxinput=fd;
  return TRUE;
  }


// Remove input
FXbool FXDaemonApp::removeInput(FXInputHandle fd,FXuint mode){
  if(mode==INPUT_NONE) return FALSE;
  if(fd<0 || fd>maxinput) return FALSE;
  if(mode&INPUT_READ){
    inputs[fd].read.target=NULL;
    inputs[fd].read.message=0;
    FD_CLR(fd,(fd_set*)r_fds);
    }
  if(mode&INPUT_WRITE){
    inputs[fd].write.target=NULL;
    inputs[fd].write.message=0;
    FD_CLR(fd,(fd_set*)w_fds);
    }
  if(mode&INPUT_EXCEPT){
    inputs[fd].excpt.target=NULL;
    inputs[fd].excpt.message=0;
    FD_CLR(fd,(fd_set*)e_fds);
    }
  while(0<=maxinput){                   // Limit number of fd's to test if possible
    if(inputs[maxinput].read.target || inputs[maxinput].write.target || inputs[maxinput].excpt.target) break;
    maxinput--;
    }
  return TRUE;
  }


/*******************************************************************************/

// Currently we can only receive messages - not sure how to do to send a message...
FXSysVMessage* FXDaemonApp::addMessageQueue(FXint messageid,FXint maxMsgSize,FXObject* tgt,FXSelector sel,FXint messagetype){
  register FXSysVMessage **q, *que;
  que = new FXSysVMessage;
  que->next = NULL;
  que->target = tgt;
  que->message = sel;
  que->msgid = messageid;
  que->msgtype = messagetype;
  que->maxsize = maxMsgSize;
  que->data.bytesread = 0;
  que->data.buffer.mtype = 1;
  FXMALLOC (&(que->data.buffer.mtext),char,maxMsgSize);
  // check to make sure that programmer doesnt make us check for multiple same message queues/types
  if (sysVmessages){
    for(q=&sysVmessages; *q; q=&(*q)->next){
      if((*q)->msgid == messageid) {
        if ((*q)->msgtype==0)
          fxerror("SYS V msg type duplicate - previous all, new %i\n",messagetype);
        if (messagetype==0)
          fxerror("SYS V msg type duplicate - previous %i, new all\n",(*q)->msgtype);
        if ((*q)->msgtype==messagetype)
          fxerror("SYS V msg type duplicate - previous %i, new %i\n",(*q)->msgtype,messagetype);
        }
      }
    que->next=sysVmessages;
    }
  sysVmessages=que;
  return que;
  }


FXSysVMessage* FXDaemonApp::removeMessageQueue(FXSysVMessage* que){
  if (que && sysVmessages){
    register FXSysVMessage *q;
    if (sysVmessages == que){
      q = sysVmessages;
      sysVmessages=sysVmessages->next;
      FXFREE (&(q->data.buffer.mtext));
      delete q;
      }
    else{
      for (q=sysVmessages; q; q=q->next){
        if ( q->next != NULL && q->next == que ){
	  q->next = que->next;
          FXFREE (&(que->data.buffer.mtext));
          delete q;
          break;
          }
        }
      }
    }
  return NULL;
  }


/*******************************************************************************/

// Get an event
FXbool FXDaemonApp::getNextEvent(FXRawEvent& ev,FXbool blocking){
  struct timeval now,delta;
  register FXChore *c;
  register int maxfds;
  register int nfds;
  register int fff;
  register int sig;
  fd_set readfds;
  fd_set writefds;
  fd_set exceptfds;

  // Handle all past due timers
  gettimeofday(&now,NULL);
  while(timers){
    register FXTimer* t=timers;
    if(now < t->due) break;
    timers=t->next;
    if(t->target) t->target->handle(this,FXSEL(SEL_TIMEOUT,t->message),NULL);
    t->next=timerrecs;
    timerrecs=t;
    }

  // Check non-immediate signals that may have fired
  if(nsignals){
    for(sig=0; sig<MAXSIGNALS; sig++){
      if(signals[sig].notified){
        signals[sig].notified=FALSE;
        if(signals[sig].target && signals[sig].target->handle(this,FXSEL(SEL_SIGNAL,signals[sig].message),(void*)sig)) return FALSE;
        }
      }
    }

  // Check SysV message queues for any messages
  // Note: we process all message queues before returning, this ensures a complete
  //       scan of each queue, rather than finding data on the first queue only, then returning
  if (sysVmessages){
    register FXSysVMessage *q;
    FXbool workdone=FALSE;
    for (q=sysVmessages; q; q=q->next){
      q->data.bytesread= msgrcv (q->msgid, &(q->data.buffer), q->maxsize, q->msgtype, MSG_NOERROR|IPC_NOWAIT);
      if (q->data.bytesread < 0 && errno!=EINTR) {
        fxerror("Application terminated: lost SysV message queue errno=%d\n",errno);
        return FALSE;
        }
      if (q->data.bytesread > 0){
        workdone=TRUE;
        q->target->handle(this,FXSEL(SEL_COMMAND,q->message), &q->data );
        }
      }
    if (workdone) return FALSE;
    }

  // Prepare fd's to watch
  FD_ZERO(&readfds);
  FD_ZERO(&writefds);
  FD_ZERO(&exceptfds);

  // Add other inputs...
  maxfds=ninputs;

  delta.tv_usec=0;
  delta.tv_sec=0;

  // Do a quick poll for any ready events (and ONLY event!)
  nfds=SELECT(maxfds+1,&readfds,&writefds,&exceptfds,&delta);

  // No events, do idle processing
  if(nfds==0){

    // Do our chores :-)
    c=chores;
    if(c){
      chores=c->next;
      if(c->target) c->target->handle(this,FXSEL(SEL_CHORE,c->message),NULL);
      c->next=chorerecs;
      chorerecs=c;
      }

    // There are more chores to do
    if(chores) return FALSE;

    // We're not blocking
    if(!blocking) return FALSE;

    // Now, block till timeout, i/o, or event
    readfds=*((fd_set*)r_fds);
    writefds=*((fd_set*)w_fds);
    exceptfds=*((fd_set*)e_fds);

    // Add other inputs...
    maxfds=ninputs;

    // If there are timers or sysVmessages, we block only for a little while.
    if(timers || sysVmessages){

      if (timers){
        // All that testing above may have taken some time...
        gettimeofday(&now,NULL);

        // Compute how long to wait
        delta.tv_usec=timers->due.tv_usec-now.tv_usec;
        delta.tv_sec=timers->due.tv_sec-now.tv_sec;
        while(delta.tv_usec<0){
          delta.tv_usec+=1000000;
          delta.tv_sec-=1;
          }
        FXASSERT(0<=delta.tv_usec && delta.tv_usec<1000000);

        // Some timers are already due; do them right away!
        if(delta.tv_sec<0 || (delta.tv_sec==0 && delta.tv_usec==0)) return FALSE;
        }

      if (sysVmessages){
        // Notes:
	// 1. we _dont_ do a quick poll here for any new SysV messages, since it may
	//    result in a starvation of handling select() based events
	// 2. thus we choose some resonable timeout, unless we have already computed
	//    a timer based timeout - so we use the timer based usec value or a valid
	//    resonable SysV polling value (say half a second between polls)
        if (!timers) delta.tv_usec = 500000;
        else if (delta.tv_sec > 0) delta.tv_sec = 0;
        }

      // Block till timer (timeout) or event or interrupt
      nfds=SELECT(maxfds+1,&readfds,&writefds,&exceptfds,&delta);
      }

    // If no timers or no SysV messages, we block till event or interrupt
    else{
      nfds=SELECT(maxfds+1,&readfds,&writefds,&exceptfds,NULL);
      }
    }

  // Timed out or interrupted
  if(nfds<=0){
    if(nfds<0 && errno!=EAGAIN && errno!=EINTR){fxerror("Application terminated: interrupt or lost connection errno=%d\n",errno);}
    return FALSE;
    }

  // Try I/O channels
  for(fff=0; fff<=maxinput; fff++){
    if(FD_ISSET(fff,&readfds)){
      if(inputs[fff].read.target) inputs[fff].read.target->handle(this,FXSEL(SEL_IO_READ,inputs[fff].read.message),(void*)fff);
      }
    if(FD_ISSET(fff,&writefds)){
      if(inputs[fff].write.target) inputs[fff].write.target->handle(this,FXSEL(SEL_IO_WRITE,inputs[fff].write.message),(void*)fff);
      }
    if(FD_ISSET(fff,&exceptfds)){
      if(inputs[fff].excpt.target) inputs[fff].excpt.target->handle(this,FXSEL(SEL_IO_EXCEPT,inputs[fff].read.message),(void*)fff);
      }
    }

  // Regular event
  return TRUE;
  }



/*******************************************************************************/


// Peek for event
FXbool FXDaemonApp::peekEvent(){
  struct timeval delta;
  fd_set readfds;
  fd_set writefds;
  fd_set exceptfds;
  int maxfds;
  int nfds;

  // If we are using message queues, check each queue for any messages
  // Note: using msgctl() is about the best we can do
  // -> Unix provides no support for select()ing on message queues
  if (sysVmessages){
    struct msqid_ds buffer;
    register FXSysVMessage *q;
    for (q=sysVmessages; q; q=q->next){
      if ( msgctl (q->msgid, IPC_STAT, &buffer) < 0 )
        {fxerror("Application terminated: lost SysV message queue errno=%d\n",errno);}
      if ( buffer.msg_qnum ) return TRUE;
      }
    }

  // Prepare fd's to watch
  FD_ZERO(&readfds);
  FD_ZERO(&writefds);
  FD_ZERO(&exceptfds);

  // Add other inputs...
  maxfds=ninputs;

  delta.tv_usec=0;
  delta.tv_sec=0;

  // Do a quick poll for any ready events
  nfds=SELECT(maxfds+1,&readfds,&writefds,&exceptfds,&delta);

  // Interrupt
  if(nfds<0 && errno!=EAGAIN && errno!=EINTR){
    fxerror("Application terminated: interrupt or lost connection errno=%d\n",errno);
    }

  // I/O activity
  return nfds>0 ? TRUE : FALSE;
  }


/*******************************************************************************/

// Dispatch event to widget
FXbool FXDaemonApp::dispatchEvent(FXRawEvent& ev){
  return TRUE;
  }

/*******************************************************************************/

// Run application
FXint FXDaemonApp::run(){
  FXInvocation inv(&invocation);
  FXTRACE((100,"Start run\n"));
  while(!inv.done){
    runOneEvent();
    }
  FXTRACE((100,"End run\n"));
  return inv.code;
  }


// Run till some flag becomes non-zero
FXint FXDaemonApp::runUntil(FXuint& condition){
  FXInvocation inv(&invocation);
  FXTRACE((100,"Start runUntil\n"));
  while(!inv.done && condition==0){
    runOneEvent();
    }
  FXTRACE((100,"End runUntil\n"));
  return condition;
  }


// Run event loop while events are available
FXint FXDaemonApp::runWhileEvents(){
  FXInvocation inv(&invocation);
  FXTRACE((100,"Start runWhileEvents\n"));
  while(!inv.done && peekEvent()){
    runOneEvent();
    }
  FXTRACE((100,"End runWhileEvents\n"));
  return 0;
  }


// Perform one event dispatch
void FXDaemonApp::runOneEvent(){
  FXRawEvent ev;
  if(getNextEvent(ev)){
    dispatchEvent(ev);
    }
  }


// Break out of topmost event loop, closing all nested loops also
void FXDaemonApp::stop(FXint value){
  for(FXInvocation* inv=invocation; inv; inv=inv->upper){
    inv->done=TRUE;
    inv->code=0;
    if(inv->upper==NULL){
      inv->code=value;
      return;
      }
    }
  }


// Initialize application
void FXDaemonApp::init(int& argc,char** argv){
  FXint i,j;

  // Verify implementation invariants
  FXASSERT(sizeof(FXuchar)==1);
  FXASSERT(sizeof(FXbool)==1);
  FXASSERT(sizeof(FXchar)==1);
  FXASSERT(sizeof(FXushort)==2);
  FXASSERT(sizeof(FXshort)==2);
  FXASSERT(sizeof(FXuint)==4);
  FXASSERT(sizeof(FXint)==4);
  FXASSERT(sizeof(FXfloat)==4);
  FXASSERT(sizeof(FXdouble)==8);

  // Long is not always available on all implementations
#ifdef FX_LONG
  FXASSERT(sizeof(FXulong)==8);
  FXASSERT(sizeof(FXlong)==8);
#endif

  // Parse out FOX args
  i=j=1;
  while(j<argc){

    // Set trace level
    if(strcmp(argv[j],"-tracelevel")==0){
      j++;
      if(j>=argc){
        fxwarning("%s:init: missing argument for -tracelevel.\n",getClassName());
        ::exit(1);
        }
      if(sscanf(argv[j],"%d",&fxTraceLevel)!=1){
        fxwarning("%s::init: expected trace level number.\n",getClassName());
        ::exit(1);
        }
      j++;
      continue;
      }

    // Copy program arguments
    argv[i++]=argv[j++];
    }

  // Adjust argment count
  argv[i]=NULL;
  argc=i;

  // Read the registry
  registry.read();

  }


// Exit application, but not the process
void FXDaemonApp::exit(FXint code){
  FXTRACE((100,"%s::exit\n",getClassName()));

  // Write the registry
  registry.write();

  // Exit the program
  stop(code);
  }


// Create application's windows
void FXDaemonApp::create(){
  FXTRACE((100,"%s::create\n",getClassName()));
  }


// Detach application's windows
void FXDaemonApp::detach(){
  FXTRACE((100,"%s::detach\n",getClassName()));
  }


// Destroy application's windows
void FXDaemonApp::destroy(){
  FXTRACE((100,"%s::destroy\n",getClassName()));
  }


// Handle quit
long FXDaemonApp::onCmdQuit(FXObject*,FXSelector,void*){
  exit(0);
  return 1;
  }

/*******************************************************************************/

// Beep
void FXDaemonApp::beep(){
  if(!initialized){ fxerror("%s::beep: Should open display first.\n",getClassName()); }
  // FIXME do something here to make a sound
  }


// Dump widgets
long FXDaemonApp::onCmdDump(FXObject*,FXSelector,void*){
  return 1;
  }


// Save to stream
void FXDaemonApp::save(FXStream& store) const {
  FXObject::save(store);
  }


// Load from stream
void FXDaemonApp::load(FXStream& store){
  FXObject::load(store);
  }


// Virtual destructor
FXDaemonApp::~FXDaemonApp(){
  register FXTimer *t;
  register FXChore *c;
  register FXSysVMessage *q;

  FXFREE(&inputs);
  FXFREE(&r_fds);
  FXFREE(&w_fds);
  FXFREE(&e_fds);
  ninputs=0;

  // Free signals list
  FXFREE(&signals);
  nsignals=0;

  // Kill outstanding timers
  while(timers){
    t=timers;
    timers=timers->next;
    delete t;
    }

  // Free recycled timer records
  while(timerrecs){
    t=timerrecs;
    timerrecs=timerrecs->next;
    delete t;
    }

  // Free SysVmessages
  while(sysVmessages){
    q=sysVmessages;
    sysVmessages=sysVmessages->next;
    FXFREE (&(q->data.buffer.mtext));
    delete q;
    }

  // Kill outstanding chores
  while(chores){
    c=chores;
    chores=chores->next;
    delete c;
    }

  // Free recycled chore records
  while(chorerecs){
    c=chorerecs;
    chorerecs=chorerecs->next;
    delete c;
    }

  // Do this last
  app=NULL;
  }

}

