/* Free Foundation Classes for string ver.0.1.5
 * 
 * This software is in the public domain.
 * There are no restrictions on any sort of usage of this software.
 * 
 * $ffcstring: ffcstring.cpp,v 1.44.1 2001/09/17 22:57:47 Toshihiro Inoue Exp $
 */

#include "ffcstring.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifdef WIN32
#  define VSNPRINTF(a, b, c, d) _vsnprintf(a, b, c, d)
#else
#  define VSNPRINTF(a, b, c, d) vsnprintf(a, b, c, d)
#endif

FFCString operator+(const char* str1, const FFCString& str2)
{
  return FFCString(str1) + str2;
}


void FFCString::__init_class__()
{
  buffer = 0;
  size = 0;
  max = 0;
}


void FFCString::init()
{
  buffer = new char[16];
  max = 16;
  erase();
}


FFCString::FFCString()
{
  __init_class__();
  
  init();
}


FFCString::FFCString(const FFCString& s)
{
  __init_class__();
  
  init();
  operator=(s);
}


FFCString::FFCString(const char* s)
{
  __init_class__();
  
  init();
  operator=(s);
}


FFCString::FFCString(const char* s, int len)
{
  __init_class__();
  
  init();
  setString(s, len);
}


FFCString::~FFCString()
{
  delete [] buffer;
}


void FFCString::setMaxSize(int sz)
{
  if(max > sz) return;
  
  while(max <= sz) max += max;
  char* newBuf = new char[max];
  newBuf[0] = '\0';
  if(buffer) {
    if(size > 0) memcpy(newBuf, buffer, size + 1);
    delete [] buffer;
  }
  buffer = newBuf;
}


void FFCString::erase()
{
  buffer[0] = '\0';
  size = 0;
}


FFCString& FFCString::operator=(const FFCString& s)
{
  erase();
  if(s.size < 0) return *this;
  
  setMaxSize(s.size);
  memcpy(buffer, s.buffer, s.size + 1);
  size = s.size;
  return *this;
}


FFCString& FFCString::operator=(const char* s)
{
  int len = 0;
  
  erase();
  if(!s) return *this;
  
  len = strlen(s);
  if(len > 0) {
    setMaxSize(len);
    strcpy(buffer, s);
    size = len;
  }
  return *this;
}


FFCString& FFCString::setString(const char* s, int len /*= -1*/)
{
  erase();
  if(!s) return *this;
  
  if(len < 0) len = strlen(s);
  if(len > 0) {
    setMaxSize(len);
    memcpy(buffer, s, len);
    buffer[len] = '\0';
    size = len;
  }
  return *this;
}


FFCString& FFCString::sprintf(const char* fmt,  ...)
{
  va_list ap;
  int len = 0;
  
  size = 0;
  for(;;) {
    va_start(ap, fmt);
    len = VSNPRINTF(buffer, max, fmt, ap);
    va_end(ap);
    if(0 <= len && len < max) break;
    setMaxSize(max * 2 - 1);
  }
  size = len;
  return *this;
}


FFCString& FFCString::operator+=(char ch)
{
  setMaxSize(size + 1);
  buffer[size++] = ch;
  buffer[size] = '\0';
  return *this;
}


FFCString& FFCString::operator+=(unsigned short ch)
{
  if(ch < 0x0080) {
    setMaxSize(size + 1);
    buffer[size++] = char(ch);
    buffer[size] = '\0';
  } else if(ch < 0x0800) {
    setMaxSize(size + 2);
    buffer[size++] = char(0xc0 + ((ch >> 6) & 0x001f));
    buffer[size++] = char(0x80 + (ch & 0x003f));
    buffer[size] = '\0';
  } else {
    setMaxSize(size + 3);
    buffer[size++] = char(0xe0 + ((ch >> 12) & 0x000f));
    buffer[size++] = char(0x80 + ((ch >>  6) & 0x003f));
    buffer[size++] = char(0x80 + (ch & 0x003f));
    buffer[size] = '\0';
  }
  return *this;
}


FFCString& FFCString::operator+=(const char* s)
{
  int len = 0;
  
  if(!s) return *this;
  
  len = strlen(s);
  if(len < 1) return *this;
  
  setMaxSize(size + len);
  strcpy(&buffer[size], s);
  size += len;
  return *this;
}


FFCString& FFCString::operator+=(const FFCString& s)
{
  if(s.size < 1) return *this;
  
  setMaxSize(size + s.size);
  memcpy(&buffer[size], s.buffer, s.size + 1);
  size += s.size;
  return *this;
}


FFCString& FFCString::append(const char* s, int len)
{
  if(!s || len < 1) return *this;
  
  setMaxSize(size + len);
  memcpy(&buffer[size], s, len);
  size += len;
  buffer[size] = '\0';
  return *this;
}


FFCString FFCString::operator+(const FFCString& str) const
{
  FFCString ret;
  
  ret = *this;
  ret += str;
  return ret;
}


FFCString FFCString::operator+(const char* str) const
{
  FFCString ret;
  
  ret = *this;
  ret += str;
  return ret;
}


bool FFCString::operator==(const FFCString& str) const
{
  return !strcmp(buffer, str.buffer);
}


bool FFCString::operator==(const char* str) const
{
  return !strcmp(buffer, str);
}


bool FFCString::operator!=(const FFCString& str) const
{
  return strcmp(buffer, str.buffer) != 0;
}


bool FFCString::operator!=(const char* str) const
{
  return strcmp(buffer, str) != 0;
}


bool FFCString::operator<(const FFCString& str) const
{
  return strcmp(buffer, str.buffer) < 0;
}


bool FFCString::operator<(const char* str) const
{
  return strcmp(buffer, str) < 0;
}


bool FFCString::operator>(const FFCString& str) const
{
  return strcmp(buffer, str.buffer) > 0;
}


bool FFCString::operator>(const char* str) const
{
  return strcmp(buffer, str) > 0;
}


bool FFCString::operator<=(const FFCString& str) const
{
  return strcmp(buffer, str.buffer) <= 0;
}


bool FFCString::operator<=(const char* str) const
{
  return strcmp(buffer, str) <= 0;
}


bool FFCString::operator>=(const FFCString& str) const
{
  return strcmp(buffer, str.buffer) >= 0;
}


bool FFCString::operator>=(const char* str) const
{
  return strcmp(buffer, str) >= 0;
}


bool FFCString::prefix(const FFCString& str) const
{
  if(size < str.size) return false;
  return !strncmp(buffer, str.buffer, str.size);
}


bool FFCString::prefix(const char* str) const
{
  int len = 0;
  
  len = strlen(str);
  if(size < len) return false;
  return !strncmp(buffer, str, len);
}


bool FFCString::suffix(const FFCString& str) const
{
  if(size < str.size) return false;
  return !strncmp(buffer + size - str.size, str.buffer, str.size);
}


bool FFCString::suffix(const char* str) const
{
  int len = 0;
  
  len = strlen(str);
  if(size < len) return false;
  return !strncmp(buffer + size - len, str, len);
}


FFCString FFCString::lower() const
{
  FFCString ret;
  int i = 0;
  const char* p = 0;
  
  for(i = 0, p = buffer; i < size; i++, p++) {
    if(*p < 'A' || *p > 'Z') {
      ret += *p;
    } else {
      ret += char(*p + 32);
    }
  }
  return ret;
}


FFCString FFCString::upper() const
{
  FFCString ret;
  int i = 0;
  const char* p = 0;
  
  for(i = 0, p = buffer; i < size; i++, p++) {
    if(*p < 'a' || *p > 'z') {
      ret += *p;
    } else {
      ret += char(*p - 32);
    }
  }
  return ret;
}


FFCString FFCString::rmSpace() const
{
  const char* ps = 0;
  const char* pe = 0;
  
  for(ps = buffer; *ps; ps++) {
    if(*ps != ' ' && *ps != '\t' && *ps != '\r' && *ps != '\n') break;
  }
  
  for(pe = buffer + size; pe > ps; pe--) {
    if(*pe != ' ' && *pe != '\t' && *pe != '\r' && *pe != '\n') break;
  }
  
  return FFCString(ps, pe - ps);
}


int FFCString::toInt()
{
  return atoi(buffer);
}


FFCString FFCString::mid(int pos, int len /*= -1*/) const
{
  int max = 0;
  
  max = size - pos;
  if(max < 1) return "";
  
  if(len < 0 || len > max) len = max;
  return FFCString(&buffer[pos], len);
}


FFCString FFCString::left(int len) const
{
  if(len < 1) return "";
  
  if(len > size) len = size;
  return FFCString(buffer, len);
}


FFCString FFCString::right(int len) const
{
  if(len < 1) return "";
  
  if(len > size) len = size;
  return FFCString(&buffer[size - len], len);
}


FFCString FFCString::insert(int pos, const FFCString& s) const
{
  return left(pos) + s + mid(pos);
}


FFCString FFCString::remove(int pos, int len /*= -1*/) const
{
  FFCString ret;
  
  if(len == 0 || pos < 0 || pos >= size) return *this;
  if(len < 0) return left(pos);
  if(pos + len > size) len = size - pos;
  return left(pos) + mid(pos + len);
}


int FFCString::find(char ch, int pos /*= 0*/) const
{
  const char* p = 0;
  
  if(pos < 0) {
    pos = 0;
  } else if(pos >= size) {
    return -1;
  }
  p = strchr(&buffer[pos], ch);
  if(!p) return -1;
  return p - buffer;
}


int FFCString::find(const char* str, int pos /*= 0*/) const
{
  const char* p = 0;
  
  if(pos < 0) {
    pos = 0;
  } else if(pos >= size) {
    return -1;
  }
  p = strstr(&buffer[pos], str);
  if(!p) return -1;
  return p - buffer;
}


int FFCString::rfind(char ch) const
{
  const char* p = 0;
  
  p = strrchr(buffer, ch);
  if(!p) return -1;
  return p - buffer;
}


FFCString FFCString::replace(char ch1, char ch2) const
{
  FFCString ret;
  int i = 0;
  char* ch = 0;
  
  ch = buffer;
  for(i = 0; i < size; i++, ch++) {
    if(*ch == ch1) {
      ret += ch2;
    } else {
      ret += *ch;
    }
  }
  return ret;
}


FFCString FFCString::directReplace(char ch1, char ch2)
{
  char* ch = 0;
  
  for(ch = buffer; *ch; ch++) {
    if(*ch == ch1) *ch = ch2;
  }
  return *this;
}


FFCString& FFCString::operator-=(int len)
{
  if(len >= size) {
    erase();
  } else if(len > 0) {
    buffer[size - len] = '\0';
    size -= len;
  }
  return *this;
}


FFCString& FFCString::directInsert(int pos, const FFCString& s)
{
  if(pos > size) pos = size;
  setMaxSize(size + s.size);
  if(pos < size) {
    memcpy(&buffer[pos + s.size], &buffer[pos], size - pos + 1);
  }
  memcpy(&buffer[pos], s.buffer, s.size);
  size += s.size;
  return *this;
}


FFCString& FFCString::directRemove(int pos, int len /*= -1*/)
{
  if(len == 0 || pos < 0 || pos >= size) return *this;
  if(len < 0) len = size - pos;
  if(pos + len > size) len = size - pos;
  memcpy(&buffer[pos], &buffer[pos + len], size - (pos + len) + 1);
  size -= len;
  return *this;
}


void FFCString::memcpy(void* dst, const void* src, int len)
{
  char* d = 0;
  const char* s = 0;
  
  if(len < 0) return;
  
  d = (char*)dst;
  s = (const char*)src;
  if(dst < src) {
    for(int i = 0; i < len; i++, d++, s++) *d = *s;
  } else if(src < dst) {
    d += len - 1;
    s += len - 1;
    for(int i = 0; i < len; i++, d--, s--) *d = *s;
  }
}
