/********************************************************************************
*                                                                               *
*                  Variant class - a container for any data type                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
* Copyright (C) 2003 by Mathew Robertson.       All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "FXVariant.h"
using namespace FXEX;
namespace FXEX {

// map
FXDEFMAP(FXVariant) FXVariantMap[]={
//  FXMAPFUNC(SEL_COMMAND,FXVariant::ID_VALUE,FXVariant::onCmdValue),
//  FXMAPFUNC(SEL_CHANGED,FXVariant::ID_VALUE,FXVariant::onCmdValue),
//  FXMAPFUNC(SEL_UPDATE,FXVariant::ID_VALUE,FXVariant::onUpdValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXVariant::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXVariant::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETREALVALUE,FXVariant::onCmdSetRealValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FXVariant::onCmdSetStringValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXVariant::onCmdGetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETREALVALUE,FXVariant::onCmdGetRealValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,FXVariant::onCmdGetStringValue),
  };
FXIMPLEMENT(FXVariant,FXBaseObject,FXVariantMap,ARRAYNUMBER(FXVariantMap))

// constructors for all the types
FXVariant::FXVariant(FXObject *tgt,FXSelector sel) : FXBaseObject(tgt,sel) {
  type=DT_UNASSIGNED;
  clear();
  }
FXVariant::FXVariant(const FXVariant &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const bool &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXuchar &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXchar &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXushort &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXshort &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXuint &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXint &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXulong &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXlong &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXfloat &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXdouble &v,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=v;
  }
FXVariant::FXVariant(const FXString &s,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=s;
  }
FXVariant::FXVariant(const FXWString &s,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=s;
  }
FXVariant::FXVariant(const FXCharset &c,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=c;
  }
FXVariant::FXVariant(FXObject *o,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=o;
  }
FXVariant::FXVariant(void* p,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  *this=p;
  }
FXVariant::FXVariant(const void* p,unsigned long size,FXObject *tgt,FXSelector sel):FXBaseObject(tgt,sel){
  type=DT_UNASSIGNED;
  setData(p,size);
  }

// destructor
FXVariant::~FXVariant(){
  clear();
  }

// return the natural size of the instance of the variant
unsigned long FXVariant::getSize() const {
  switch(type){
    case DT_UNASSIGNED:	return 0;
    case DT_NULL:	return 0;
    case DT_BOOL:	return sizeof(bool);
    case DT_FXUCHAR:	return sizeof(data.fxuchar);
    case DT_FXCHAR:	return sizeof(data.fxchar);
    case DT_FXUSHORT:	return sizeof(data.fxushort);
    case DT_FXSHORT:	return sizeof(data.fxshort);
    case DT_FXUINT:	return sizeof(data.fxuint);
    case DT_FXINT:	return sizeof(data.fxint);
    case DT_FXULONG:	return sizeof(data.fxulong);
    case DT_FXLONG:	return sizeof(data.fxlong);
    case DT_FXFLOAT:	return sizeof(data.fxfloat);
    case DT_FXDOUBLE:	return sizeof(data.fxdouble);
    case DT_FXCHARSET:	return 256;
    case DT_FXSTRING:	return (fxstring.length()+1)*sizeof(FXchar);
    case DT_FXWSTRING:	return (fxwstring.length()+1)*sizeof(FXwchar);
    case DT_FXOBJECT:	return sizeof(fxobject);
    case DT_POINTER:	return sizeof(pointer);
    case DT_RAW:	return raw.size;
    }
  return 0;
  }

// empty the contents of this variant
void FXVariant::clear() {
  switch(type){
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:       data.boolean=false; break;
    case DT_FXUCHAR:    data.fxuchar=0; break;
    case DT_FXCHAR:     data.fxchar=0; break;
    case DT_FXUSHORT:   data.fxushort=0; break;
    case DT_FXSHORT:    data.fxshort=0; break;
    case DT_FXUINT:     data.fxuint=0; break;
    case DT_FXINT:      data.fxint=0; break;
    case DT_FXULONG:    data.fxulong=0; break;
    case DT_FXLONG:     data.fxlong=0; break;
    case DT_FXFLOAT:    data.fxfloat=0.0; break;
    case DT_FXDOUBLE:   data.fxdouble=0.0; break;
    case DT_FXCHARSET:  fxcharset.clear(); break;
    case DT_FXSTRING:   fxstring.clear(); break;
    case DT_FXWSTRING:  fxwstring.clear(); break;
    case DT_FXOBJECT:   fxobject=NULL; break;
    case DT_POINTER:    pointer=NULL; break;
    case DT_RAW:        FXFREE(&raw.pointer); raw.size=0; break;
    }
  type=DT_UNASSIGNED;
  }

// set variant type to be specific data type
void FXVariant::setType(FXint type) {
  clear();
  this->type=type;
  }

// assign from another variant
FXVariant &FXVariant::operator=(const FXVariant &v) {
  setType(v.type);
  switch(type){
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:       data.boolean=v.data.boolean; break;
    case DT_FXUCHAR:    data.fxuchar=v.data.fxuchar; break;
    case DT_FXCHAR:     data.fxchar=v.data.fxchar; break;
    case DT_FXUSHORT:   data.fxushort=v.data.fxushort; break;
    case DT_FXSHORT:    data.fxshort=v.data.fxshort; break;
    case DT_FXUINT:     data.fxuint=v.data.fxuint; break;
    case DT_FXINT:      data.fxint=v.data.fxint; break;
    case DT_FXULONG:    data.fxulong=v.data.fxulong; break;
    case DT_FXLONG:     data.fxlong=v.data.fxlong; break;
    case DT_FXFLOAT:    data.fxfloat=v.data.fxfloat; break;
    case DT_FXDOUBLE:   data.fxdouble=v.data.fxdouble; break;
    case DT_FXCHARSET:  fxcharset=v.fxcharset; break;
    case DT_FXSTRING:   fxstring=v.fxstring; break;
    case DT_FXWSTRING:  fxwstring=v.fxwstring; break;
    case DT_FXOBJECT:   fxobject=v.fxobject; break;
    case DT_POINTER:    pointer=v.pointer; break;
    case DT_RAW:        setData(v.raw.pointer,v.raw.size); break;
    }
  return *this;
  }

// assign from bool
FXVariant &FXVariant::operator=(const bool &v) {
  clear();
  type=DT_BOOL;
  data.boolean=v;
  return *this;
  }

// assign from FXuchar
FXVariant &FXVariant::operator=(const FXuchar &v) {
  clear();
  type=DT_FXUCHAR;
  data.fxuchar=v;
  return *this;
  }

// assign from FXchar
FXVariant &FXVariant::operator=(const FXchar &v) {
  clear();
  type=DT_FXCHAR;
  data.fxchar=v;
  return *this;
  }

// assign from FXushort
FXVariant &FXVariant::operator=(const FXushort &v) {
  clear();
  type=DT_FXUSHORT;
  data.fxushort=v;
  return *this;
  }

// assign from FXshort
FXVariant &FXVariant::operator=(const FXshort &v) {
  clear();
  type=DT_FXSHORT;
  data.fxshort=v;
  return *this;
  }

// assign from FXuint
FXVariant &FXVariant::operator=(const FXuint &v) {
  clear();
  type=DT_FXUINT;
  data.fxuint=v;
  return *this;
  }

// assign from FXint
FXVariant &FXVariant::operator=(const FXint &v) {
  clear();
  type=DT_FXINT;
  data.fxint=v;
  return *this;
  }

// assign from FXulong
FXVariant &FXVariant::operator=(const FXulong &v) {
  clear();
  type=DT_FXULONG;
  data.fxulong=v;
  return *this;
  }

// assign from FXlong
FXVariant &FXVariant::operator=(const FXlong &v) {
  clear();
  type=DT_FXLONG;
  data.fxlong=v;
  return *this;
  }

// assign from FXfloat
FXVariant &FXVariant::operator=(const FXfloat &v) {
  clear();
  type=DT_FXFLOAT;
  data.fxfloat=v;
  return *this;
  }

// assign from FXdouble
FXVariant &FXVariant::operator=(const FXdouble &v) {
  clear();
  type=DT_FXDOUBLE;
  data.fxdouble=v;
  return *this;
  }

// assign from FXCharset
FXVariant &FXVariant::operator=(const FXCharset &c) {
  clear();
  type=DT_FXCHARSET;
  fxcharset=c;
  return *this;
  }

// assign from FXString
FXVariant &FXVariant::operator=(const FXString &s) {
  clear();
  type=DT_FXSTRING;
  fxstring=s;
  return *this;
  }

// assign from FXWString
FXVariant &FXVariant::operator=(const FXWString &s) {
  clear();
  type=DT_FXWSTRING;
  fxwstring=s;
  return *this;
  }

// assign from FXObject*
FXVariant &FXVariant::operator=(FXObject *o) {
  clear();
  type=DT_FXOBJECT;
  fxobject=o;
  return *this;
  }

// assign from pointer
FXVariant &FXVariant::operator=(void *p) {
  clear();
  type=DT_POINTER;
  pointer=p;
  return *this;
  }

// copy some raw data
void FXVariant::setData(const void *p,unsigned long size){
  clear();
  type=DT_RAW;
  if (FXMEMDUP(raw.pointer,FXuchar,p,size)) raw.size=size;
  }

// return bool'ized version of value
bool FXVariant::asBool() const {
  switch(type){
    case DT_UNASSIGNED: return false;
    case DT_NULL:       return false;
    case DT_BOOL:       return data.boolean;
    case DT_FXUCHAR:    return data.fxuchar!=0;
    case DT_FXCHAR:     return data.fxchar!=0;
    case DT_FXUSHORT:   return data.fxushort!=0;
    case DT_FXSHORT:    return data.fxshort!=0;
    case DT_FXUINT:     return data.fxuint!=0;
    case DT_FXINT:      return data.fxint!=0;
    case DT_FXULONG:    return data.fxulong!=0;
    case DT_FXLONG:     return data.fxlong!=0;
    case DT_FXFLOAT:    return data.fxfloat!=0;
    case DT_FXDOUBLE:   return data.fxdouble!=0;
    case DT_FXCHARSET: {
      FXString s((FXCharset)fxcharset);
      if (FXIntVal(s)) return true;
      s.lower();
      if (s[0]=='t' || s[0]=='y' || s[0]=='o') return true;
      return false;
      }
    case DT_FXSTRING: {
      if (FXIntVal(fxstring)) return true;
      FXString s=fxstring; s.lower();
      if (s[0]=='t' || s[0]=='y' || s[0]=='o') return true;
      return false;
      }
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return false;
  }

// return FXuchar'ized version of value
FXuchar FXVariant::asFXuchar() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXuchar)data.boolean;
    case DT_FXUCHAR:    return (FXuchar)data.fxuchar;
    case DT_FXCHAR:     return (FXuchar)data.fxchar;
    case DT_FXUSHORT:   return (FXuchar)data.fxushort;
    case DT_FXSHORT:    return (FXuchar)data.fxshort;
    case DT_FXUINT:     return (FXuchar)data.fxuint;
    case DT_FXINT:      return (FXuchar)data.fxint;
    case DT_FXULONG:    return (FXuchar)data.fxulong;
    case DT_FXLONG:     return (FXuchar)data.fxlong;
    case DT_FXFLOAT:    return (FXuchar)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXuchar)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXuchar)FXUIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXuchar)FXUIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXchar'ized version of value
FXchar FXVariant::asFXchar() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXchar)data.boolean;
    case DT_FXUCHAR:    return (FXchar)data.fxuchar;
    case DT_FXCHAR:     return (FXchar)data.fxchar;
    case DT_FXUSHORT:   return (FXchar)data.fxushort;
    case DT_FXSHORT:    return (FXchar)data.fxshort;
    case DT_FXUINT:     return (FXchar)data.fxuint;
    case DT_FXINT:      return (FXchar)data.fxint;
    case DT_FXULONG:    return (FXchar)data.fxulong;
    case DT_FXLONG:     return (FXchar)data.fxlong;
    case DT_FXFLOAT:    return (FXchar)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXchar)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXchar)FXIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXchar)FXIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXushort'ized version of value
FXushort FXVariant::asFXushort() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXushort)data.boolean;
    case DT_FXUCHAR:    return (FXushort)data.fxuchar;
    case DT_FXCHAR:     return (FXushort)data.fxchar;
    case DT_FXUSHORT:   return (FXushort)data.fxushort;
    case DT_FXSHORT:    return (FXushort)data.fxshort;
    case DT_FXUINT:     return (FXushort)data.fxuint;
    case DT_FXINT:      return (FXushort)data.fxint;
    case DT_FXULONG:    return (FXushort)data.fxulong;
    case DT_FXLONG:     return (FXushort)data.fxlong;
    case DT_FXFLOAT:    return (FXushort)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXushort)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXushort)FXUIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXushort)FXUIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXshort'ized version of value
FXshort FXVariant::asFXshort() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXshort)data.boolean;
    case DT_FXUCHAR:    return (FXshort)data.fxuchar;
    case DT_FXCHAR:     return (FXshort)data.fxchar;
    case DT_FXUSHORT:   return (FXshort)data.fxushort;
    case DT_FXSHORT:    return (FXshort)data.fxshort;
    case DT_FXUINT:     return (FXshort)data.fxuint;
    case DT_FXINT:      return (FXshort)data.fxint;
    case DT_FXULONG:    return (FXshort)data.fxulong;
    case DT_FXLONG:     return (FXshort)data.fxlong;
    case DT_FXFLOAT:    return (FXshort)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXshort)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXshort)FXIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXshort)FXIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXuint'ized version of value
FXuint FXVariant::asFXuint() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXuint)data.boolean;
    case DT_FXUCHAR:    return (FXuint)data.fxuchar;
    case DT_FXCHAR:     return (FXuint)data.fxchar;
    case DT_FXUSHORT:   return (FXuint)data.fxushort;
    case DT_FXSHORT:    return (FXuint)data.fxshort;
    case DT_FXUINT:     return (FXuint)data.fxuint;
    case DT_FXINT:      return (FXuint)data.fxint;
    case DT_FXULONG:    return (FXuint)data.fxulong;
    case DT_FXLONG:     return (FXuint)data.fxlong;
    case DT_FXFLOAT:    return (FXuint)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXuint)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXuint)FXUIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXuint)FXUIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXint'ized version of value
FXint FXVariant::asFXint() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXint)data.boolean;
    case DT_FXUCHAR:    return (FXint)data.fxuchar;
    case DT_FXCHAR:     return (FXint)data.fxchar;
    case DT_FXUSHORT:   return (FXint)data.fxushort;
    case DT_FXSHORT:    return (FXint)data.fxshort;
    case DT_FXUINT:     return (FXint)data.fxuint;
    case DT_FXINT:      return (FXint)data.fxint;
    case DT_FXULONG:    return (FXint)data.fxulong;
    case DT_FXLONG:     return (FXint)data.fxlong;
    case DT_FXFLOAT:    return (FXint)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXint)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXint)FXIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXint)FXIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXulong'ized version of value
FXulong FXVariant::asFXulong() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXulong)data.boolean;
    case DT_FXUCHAR:    return (FXulong)data.fxuchar;
    case DT_FXCHAR:     return (FXulong)data.fxchar;
    case DT_FXUSHORT:   return (FXulong)data.fxushort;
    case DT_FXSHORT:    return (FXulong)data.fxshort;
    case DT_FXUINT:     return (FXulong)data.fxuint;
    case DT_FXINT:      return (FXulong)data.fxint;
    case DT_FXULONG:    return (FXulong)data.fxulong;
    case DT_FXLONG:     return (FXulong)data.fxlong;
    case DT_FXFLOAT:    return (FXulong)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXulong)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXulong)FXUIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXulong)FXUIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXlong'ized version of value
FXlong FXVariant::asFXlong() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return (FXlong)data.boolean;
    case DT_FXUCHAR:    return (FXlong)data.fxuchar;
    case DT_FXCHAR:     return (FXlong)data.fxchar;
    case DT_FXUSHORT:   return (FXlong)data.fxushort;
    case DT_FXSHORT:    return (FXlong)data.fxshort;
    case DT_FXUINT:     return (FXlong)data.fxuint;
    case DT_FXINT:      return (FXlong)data.fxint;
    case DT_FXULONG:    return (FXlong)data.fxulong;
    case DT_FXLONG:     return (FXlong)data.fxlong;
    case DT_FXFLOAT:    return (FXlong)roundl((FXdouble)data.fxfloat);
    case DT_FXDOUBLE:   return (FXlong)roundl(data.fxdouble);
    case DT_FXCHARSET:  return (FXlong)FXIntVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return (FXlong)FXIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0;
  }

// return FXfloat'ized version of value
FXfloat FXVariant::asFXfloat() const {
  switch(type){
    case DT_UNASSIGNED: return 0.0;
    case DT_NULL:       return 0.0;
    case DT_BOOL:       return data.boolean?1.0:0.0;
    case DT_FXUCHAR:    return 1.0*data.fxuchar;
    case DT_FXCHAR:     return 1.0*data.fxchar;
    case DT_FXUSHORT:   return 1.0*data.fxushort;
    case DT_FXSHORT:    return 1.0*data.fxshort;
    case DT_FXUINT:     return 1.0*data.fxuint;
    case DT_FXINT:      return 1.0*data.fxint;
    case DT_FXULONG:    return 1.0*data.fxulong;
    case DT_FXLONG:     return 1.0*data.fxlong;
    case DT_FXFLOAT:    return data.fxfloat;
    case DT_FXDOUBLE:   return (FXfloat)data.fxdouble;
    case DT_FXCHARSET:  return FXFloatVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return FXFloatVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0.0;
  }

// return FXdouble'ized version of valuez
FXdouble FXVariant::asFXdouble() const {
  switch(type){
    case DT_UNASSIGNED: return 0.0;
    case DT_NULL:       return 0.0;
    case DT_BOOL:       return data.boolean?1.0:0.0;
    case DT_FXUCHAR:    return 1.0*data.fxuchar;
    case DT_FXCHAR:     return 1.0*data.fxchar;
    case DT_FXUSHORT:   return 1.0*data.fxushort;
    case DT_FXSHORT:    return 1.0*data.fxshort;
    case DT_FXUINT:     return 1.0*data.fxuint;
    case DT_FXINT:      return 1.0*data.fxint;
    case DT_FXULONG:    return 1.0*data.fxulong;
    case DT_FXLONG:     return 1.0*data.fxlong;
    case DT_FXFLOAT:    return (FXdouble)data.fxfloat;
    case DT_FXDOUBLE:   return data.fxdouble;
    case DT_FXCHARSET:  return FXDoubleVal((FXCharset)fxcharset);
    case DT_FXSTRING:   return FXDoubleVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject!=NULL;
    case DT_POINTER:    return pointer!=NULL;
    case DT_RAW:        return raw.size!=0;
    }
  return 0.0;
  }

// return FXCharset'ized version of value
FXCharset FXVariant::asFXCharset() const {
  switch(type){
    case DT_UNASSIGNED: return FXCharset();
    case DT_NULL:       return FXCharset();
    case DT_BOOL:       return data.boolean?FXCharset('1'):FXCharset('0');
    case DT_FXUCHAR:    return FXCharset((FXchar)data.fxuchar);
    case DT_FXCHAR:     return FXCharset((FXchar)data.fxchar);
    case DT_FXUSHORT:   return FXCharset(FXStringVal((FXuint)data.fxushort));
    case DT_FXSHORT:    return FXCharset(FXStringVal((FXint)data.fxshort));
    case DT_FXUINT:     return FXCharset(FXStringVal((FXuint)data.fxuint));
    case DT_FXINT:      return FXCharset(FXStringVal((FXint)data.fxint));
    case DT_FXULONG:    return FXCharset(FXStringVal((FXuint)data.fxulong));
    case DT_FXLONG:     return FXCharset(FXStringVal((FXint)data.fxlong));
    case DT_FXFLOAT:    return FXCharset(FXStringVal(data.fxfloat));
    case DT_FXDOUBLE:   return FXCharset(FXStringVal(data.fxdouble));
    case DT_FXCHARSET:  return fxcharset;
    case DT_FXSTRING:   return FXCharset(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return fxobject?FXCharset('1'):FXCharset('0');
    case DT_POINTER:    return pointer?FXCharset('1'):FXCharset('0');
    case DT_RAW:        return raw.size?FXCharset('1'):FXCharset('0');
    }
  return FXCharset();
  }

// return FXString'ized version of value
FXString FXVariant::asFXString() const {
  switch(type){
    case DT_UNASSIGNED: return FXString::null;
    case DT_NULL:       return FXString::null;
    case DT_BOOL:       return data.boolean?FXString("1"):FXString("0");
    case DT_FXUCHAR:    return FXStringVal((FXuint)data.fxuchar);
    case DT_FXCHAR:     return FXStringVal((FXint)data.fxchar);
    case DT_FXUSHORT:   return FXStringVal((FXuint)data.fxushort);
    case DT_FXSHORT:    return FXStringVal((FXint)data.fxshort);
    case DT_FXUINT:     return FXStringVal((FXuint)data.fxuint);
    case DT_FXINT:      return FXStringVal((FXint)data.fxint);
    case DT_FXULONG:    return FXStringVal((FXuint)data.fxulong);
    case DT_FXLONG:     return FXStringVal((FXint)data.fxlong);
    case DT_FXFLOAT:    return FXStringVal(data.fxfloat);
    case DT_FXDOUBLE:   return FXStringVal(data.fxdouble);
    case DT_FXCHARSET:  return FXString((FXCharset)fxcharset);
    case DT_FXSTRING:   return fxstring;
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return FXString(fxobject->getClassName());
    case DT_POINTER:    return FXStringFormat("%p",pointer);
    case DT_RAW:        return FXStringFormat("%p,%lu",raw.pointer,raw.size);
    }
  return FXString::null;
  }

// return FXWString'ized version of value
FXWString FXVariant::asFXWString() const {
  fxerror("FIXME");
/*
  switch(type){
    case DT_UNASSIGNED: return FXWString::null;
    case DT_NULL:       return FXWString::null;
    case DT_BOOL:       return data.boolean?FXWString("1"):FXWString("0");
    case DT_FXUCHAR:    return FXWStringVal((FXuint)data.fxuchar);
    case DT_FXCHAR:     return FXWStringVal((FXint)data.fxchar);
    case DT_FXUSHORT:   return FXWStringVal((FXuint)data.fxushort);
    case DT_FXSHORT:    return FXWStringVal((FXint)data.fxshort);
    case DT_FXUINT:     return FXWStringVal((FXuint)data.fxuint);
    case DT_FXINT:      return FXWStringVal((FXint)data.fxint);
    case DT_FXULONG:    return FXWStringVal((FXuint)data.fxulong);
    case DT_FXLONG:     return FXWStringVal((FXint)data.fxlong);
    case DT_FXFLOAT:    return FXWStringVal(data.fxfloat);
    case DT_FXDOUBLE:   return FXWStringVal(data.fxdouble);
    case DT_FXCHARSET:  return FXWString((FXCharset)fxcharset);
    case DT_FXSTRING:   fxerror("FIXME");
    case DT_FXWSTRING:  return fxwstring;
    case DT_FXOBJECT:   return FXWString(fxobject->getClassName());
    case DT_POINTER:    return FXWStringFormat("%p",pointer);
    case DT_RAW:        return FXWStringFormat("%p,%lu",raw.pointer,raw.size);
    }
*/
  return FXWString::null;
  }

// return FXObject'ized version of value
FXObject* FXVariant::asFXObject() const {
  switch(type){
    case DT_UNASSIGNED: return NULL;
    case DT_NULL:       return NULL;
    case DT_BOOL:       return NULL;
    case DT_FXUCHAR:    return NULL;
    case DT_FXCHAR:     return NULL;
    case DT_FXUSHORT:   return NULL;
    case DT_FXSHORT:    return NULL;
    case DT_FXUINT:     return NULL;
    case DT_FXINT:      return NULL;
    case DT_FXULONG:    return NULL;
    case DT_FXLONG:     return NULL;
    case DT_FXFLOAT:    return NULL;
    case DT_FXDOUBLE:   return NULL;
    case DT_FXCHARSET:  return NULL;
    case DT_FXSTRING:   return NULL;
    case DT_FXWSTRING:  return NULL;
    case DT_FXOBJECT:   return fxobject;
    case DT_POINTER:    return NULL;
    case DT_RAW:        return NULL;
    }
  return NULL;
  }

// return pointer'ized version of value
void* FXVariant::asPointer() const {
  switch(type){
    case DT_UNASSIGNED: return NULL;
    case DT_NULL:       return NULL;
    case DT_BOOL:       return (void*)&data.boolean;
    case DT_FXUCHAR:    return (void*)&data.fxuchar;
    case DT_FXCHAR:     return (void*)&data.fxchar;
    case DT_FXUSHORT:   return (void*)&data.fxushort;
    case DT_FXSHORT:    return (void*)&data.fxshort;
    case DT_FXUINT:     return (void*)&data.fxuint;
    case DT_FXINT:      return (void*)&data.fxint;
    case DT_FXULONG:    return (void*)&data.fxulong;
    case DT_FXLONG:     return (void*)&data.fxlong;
    case DT_FXFLOAT:    return (void*)&data.fxfloat;
    case DT_FXDOUBLE:   return (void*)&data.fxdouble;
    case DT_FXCHARSET:  return (void*)&fxcharset;
    case DT_FXSTRING:   return (void*)fxstring.text();
    case DT_FXWSTRING:  return (void*)fxwstring.text();
    case DT_FXOBJECT:   return (void*)fxobject;
    case DT_POINTER:    return pointer;
    case DT_RAW:        return raw.pointer;
    }
  return NULL;
  }

// return as a raw value
void* FXVariant::asRaw() const {
  switch(type){
    case DT_UNASSIGNED: return NULL;
    case DT_NULL:       return NULL;
    case DT_BOOL:       return (void*)&data.boolean;
    case DT_FXUCHAR:    return (void*)&data.fxuchar;
    case DT_FXCHAR:     return (void*)&data.fxchar;
    case DT_FXUSHORT:   return (void*)&data.fxushort;
    case DT_FXSHORT:    return (void*)&data.fxshort;
    case DT_FXUINT:     return (void*)&data.fxuint;
    case DT_FXINT:      return (void*)&data.fxint;
    case DT_FXULONG:    return (void*)&data.fxulong;
    case DT_FXLONG:     return (void*)&data.fxlong;
    case DT_FXFLOAT:    return (void*)&data.fxfloat;
    case DT_FXDOUBLE:   return (void*)&data.fxdouble;
//FIXME    case DT_FXCHARSET:  return (void*)fxcharset.text();
    case DT_FXSTRING:   return (void*)fxstring.text();
    case DT_FXWSTRING:  return (void*)fxwstring.text();
    case DT_FXOBJECT:   return (void*)fxobject;
    case DT_POINTER:    return (void*)(&(pointer));
    case DT_RAW:        return raw.pointer;
    }
  return NULL;
  }

// return address for value
FXuval FXVariant::asAddress() const {
  switch(type){
    case DT_UNASSIGNED: return 0;
    case DT_NULL:       return 0;
    case DT_BOOL:       return 0;
    case DT_FXUCHAR:    return 0;
    case DT_FXCHAR:     return 0;
    case DT_FXUSHORT:   return 0;
    case DT_FXSHORT:    return 0;
    case DT_FXUINT:     return (FXuval)data.fxuint;
    case DT_FXINT:      return 0;
    case DT_FXULONG:    return (FXuval)data.fxulong;
    case DT_FXLONG:     return 0;
    case DT_FXFLOAT:    return 0;
    case DT_FXDOUBLE:   return 0;
    case DT_FXCHARSET:  return 0;
    case DT_FXSTRING:
      if (fxstring.find("0X")!=0) return FXUIntVal(fxstring,16);
      if (fxstring.find("0x")!=0) return FXUIntVal(fxstring,16);
      return FXUIntVal(fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return (FXuval)fxobject;
    case DT_POINTER:    return (FXuval)pointer;
    case DT_RAW:        return (FXuval)raw.pointer;
    }
  return 0;
  }

// toRaw() requires special handling so that we get our deep copy.
FXVariant &FXVariant::toRaw() {
  if (type==DT_RAW) return *this;
  const void *ptr = NULL;
  FXVariant cpy(*this);
  switch(type){
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:       ptr = &cpy.data.boolean; break;
    case DT_FXUCHAR:    ptr = &cpy.data.fxuchar; break;
    case DT_FXCHAR:     ptr = &cpy.data.fxchar; break;
    case DT_FXUSHORT:   ptr = &cpy.data.fxushort; break;
    case DT_FXSHORT:    ptr = &cpy.data.fxshort; break;
    case DT_FXUINT:     ptr = &cpy.data.fxuint; break;
    case DT_FXINT:      ptr = &cpy.data.fxint; break;
    case DT_FXULONG:    ptr = &cpy.data.fxulong; break;
    case DT_FXLONG:     ptr = &cpy.data.fxulong; break;
    case DT_FXFLOAT:    ptr = &cpy.data.fxfloat; break;
    case DT_FXDOUBLE:   ptr = &cpy.data.fxdouble; break;
    case DT_FXCHARSET:  ptr = cpy.fxcharset.text(); break;
    case DT_FXSTRING:   ptr = cpy.fxstring.text(); break;
    case DT_FXWSTRING:  ptr = cpy.fxwstring.text();; break;
    case DT_FXOBJECT:   ptr = cpy.fxobject; break;
    case DT_POINTER:    ptr = &cpy.pointer; break;
    }
  if (ptr) setData(ptr, getSize());
  else setType(DT_RAW);
  return *this;
  }

// convert to type, by type specification
FXVariant &FXVariant::convertTo(FXint type) {
  switch(type){
    case DT_UNASSIGNED: clear(); break;
    case DT_NULL:       setType(DT_NULL); break;
    case DT_BOOL:       return toBool();
    case DT_FXUCHAR:    return toFXuchar();
    case DT_FXCHAR:     return toFXchar();
    case DT_FXUSHORT:   return toFXushort();
    case DT_FXSHORT:    return toFXshort();
    case DT_FXUINT:     return toFXuint();
    case DT_FXINT:      return toFXint();
    case DT_FXULONG:    return toFXulong();
    case DT_FXLONG:     return toFXlong();
    case DT_FXFLOAT:    return toFXfloat();
    case DT_FXDOUBLE:   return toFXdouble();
    case DT_FXCHARSET:  return toFXCharset();
    case DT_FXSTRING:   return toFXString();
    case DT_FXWSTRING:  return toFXWString();
    case DT_FXOBJECT:   return toFXObject();
    case DT_POINTER:    return toPointer();
    case DT_RAW:        return toRaw();
    }
  return *this;
  }

/*
// value changed from widget
long FXVariant::onCmdValue(FXObject *sender,FXSelector sel,void *ptr){
  switch(type){
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:
     FXint i=data.boolean?1:0;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.boolean=i?1:0;
     break;
    case DT_FXUCHAR:
     FXint i=(FXint)data.fxuchar;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxuchar=(FXuchar)i;
     break;
    case DT_FXCHAR:
     FXint i=(FXint)data.fxchar;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxchar=(FXchar)i;
     break;
    case DT_FXUSHORT:
     FXint i=(FXint)data.fxushort;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxushort=(FXushort)i;
     break;
    case DT_FXSHORT:
     FXint i=(FXint)data.fxshort;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxshort=(FXshort)i;
     break;
    case DT_FXUINT:
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&data.fxuint);
     break;
    case DT_FXINT:
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&data.fxint);
     break;
    case DT_FXULONG:
     FXint i=(FXint)data.fxulong;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxulong=(FXulong)i;
     break;
    case DT_FXLONG:
     FXint i=(FXint)data.fxlong;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETINTVALUE),(void*)&i);
     data.fxlong=(FXlong)i;
     break;
    case DT_FXFLOAT:
     FXdouble d=(FXdouble)data.fxfloat;
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETREALVALUE),(void*)&d);
     data.fxfloat=(FXfloat)i;
     break;
    case DT_FXDOUBLE:
     sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETREALVALUE),(void*)&data.fxdouble);
     break;
    case DT_FXCHARSET:
      FXString s((FXCharset)fxcharset);
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE),(void*)&s);
      fxcharset=s;
    case DT_FXSTRING:
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE),(void*)&fxstring);
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   break;
    case DT_POINTER:    break;
    case DT_RAW:        break;
    }
  if(target) target->handle(this,FXSEL(FXSELTYPE(sel),message),NULL);
  return 1;
  }

// value changed from widget
long FXVariant::onUpdValue(FXObject *sender,FXSelector sel,void *ptr){
  switch(type){
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:
      FXint i=data.boolean?1:0;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXUCHAR:
      FXint i=(FXint)data.fxuchar;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXCHAR:
      FXint i=(FXint)data.fxchar;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXUSHORT:
      FXint i=(FXint)data.fxushort;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXSHORT:
      FXint i=(FXint)data.fxshort;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXUINT:
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&data.fxuint);
      break;
    case DT_FXINT:
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&data.fxint);
      break;
    case DT_FXULONG:
      FXint i=(FXint)data.fxulong;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXLONG:
      FXint i=(FXint)data.fxlong;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&i);
      break;
    case DT_FXFLOAT:
      FXdouble d=(FXint)data.fxfloat;
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETREALVALUE),(void*)&d);
      break;
    case DT_FXDOUBLE:
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETREALVALUE),(void*)&data.fxdouble);
      break;
    case DT_FXCHARSET:
      FXString s((FXCharset)fxcharset);
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&s);
      break;
    case DT_FXSTRING:
      sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&fxstring);
      break;
    case DT_FXWSTRING:  fxerror("FIXME");
    case DT_FXOBJECT:   return 0;
    case DT_POINTER:    return pointer;
    case DT_RAW:        return raw.pointer;
    }
  return 1;
  }
*/

// update value from a message
long FXVariant::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  FXchar *c=(FXchar*)ptr;
  *this=FXString(c);
  return 1;
  }

// update value from a message
long FXVariant::onCmdSetIntValue(FXObject*,FXSelector,void* ptr){
  FXint *i=(FXint*)ptr;
  *this=*i;
  return 1;
  }

// update value from a message
long FXVariant::onCmdSetRealValue(FXObject*,FXSelector,void* ptr){
  FXdouble *d=(FXdouble*)ptr;
  *this=*d;
  return 1;
  }

// update value from a message
long FXVariant::onCmdSetStringValue(FXObject*,FXSelector,void* ptr){
  FXString *s=(FXString*)ptr;
  *this=*s;
  return 1;
  }

// obtain value from a message
long FXVariant::onCmdGetIntValue(FXObject*,FXSelector,void* ptr){
  *((FXint*)ptr)=asFXint();
  return 1;
  }

// obtain value from a message
long FXVariant::onCmdGetRealValue(FXObject*,FXSelector,void* ptr){
  *((FXdouble*)ptr)=asFXdouble();
  return 1;
  }

// obtain value from a message
long FXVariant::onCmdGetStringValue(FXObject*,FXSelector,void* ptr){
  *((FXString*)ptr)=asFXString();
  return 1;
  }

// save object to stream
void FXVariant::save(FXStream &store) const {
  FXBaseObject::save(store);
  store << type;
  switch (type) {
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:       { FXbool b=data.boolean?1:0; store << b; } break;
    case DT_FXUCHAR:    store << data.fxuchar; break;
    case DT_FXCHAR:     store << data.fxchar; break;
    case DT_FXUSHORT:   store << data.fxushort; break;
    case DT_FXSHORT:    store << data.fxshort; break;
    case DT_FXUINT:     store << data.fxuint; break;
    case DT_FXINT:      store << data.fxint; break;
    case DT_FXULONG:    store << data.fxulong; break;
    case DT_FXLONG:     store << data.fxlong; break;
    case DT_FXFLOAT:    store << data.fxfloat; break;
    case DT_FXDOUBLE:   store << data.fxdouble; break;
    case DT_FXCHARSET:  store << fxcharset; break;
    case DT_FXSTRING:   store << fxstring; break;
    case DT_FXWSTRING:  store << fxwstring; break;
    case DT_FXOBJECT:   store.saveObject(fxobject); break;
    case DT_POINTER:    break;
    case DT_RAW: {
      store << ((FXulong)raw.size);
      if (raw.size) store.save((FXuchar*)raw.pointer,raw.size);
      } break;
    }
  }

// load object from stream
void FXVariant::load(FXStream &store) {
  FXBaseObject::load(store);
  store >> type;
  switch (type) {
    case DT_UNASSIGNED: break;
    case DT_NULL:       break;
    case DT_BOOL:       { FXbool b; store >> b; data.boolean=b?true:false; } break;
    case DT_FXUCHAR:    store >> data.fxuchar; break;
    case DT_FXCHAR:     store >> data.fxchar; break;
    case DT_FXUSHORT:   store >> data.fxushort; break;
    case DT_FXSHORT:    store >> data.fxshort; break;
    case DT_FXUINT:     store >> data.fxuint; break;
    case DT_FXINT:      store >> data.fxint; break;
    case DT_FXULONG:    store >> data.fxulong; break;
    case DT_FXLONG:     store >> data.fxlong; break;
    case DT_FXFLOAT:    store >> data.fxfloat; break;
    case DT_FXDOUBLE:   store >> data.fxdouble; break;
    case DT_FXCHARSET:  store >> fxcharset; break;
    case DT_FXSTRING:   store >> fxstring; break;
    case DT_FXWSTRING:  store >> fxwstring; break;
    case DT_FXOBJECT:   store.loadObject(fxobject); break;
    case DT_POINTER:    type=DT_NULL; break;
    case DT_RAW: {
      store >> ((FXulong)raw.size);
      if (raw.size) store.load((FXuchar*)raw.pointer,raw.size);
      } break;
    }
  }

// operator ==
FXbool operator==(const FXVariant &v1,const FXVariant &v2){
  if (v1.type == v2.type) {
    switch (v1.type) {
      case v1.DT_UNASSIGNED: return TRUE;
      case v1.DT_NULL:       return TRUE;
      case v1.DT_BOOL:       return v1.data.boolean  == v2.data.boolean;
      case v1.DT_FXUCHAR:    return v1.data.fxuchar  == v2.data.fxuchar;
      case v1.DT_FXCHAR:     return v1.data.fxchar   == v2.data.fxchar;
      case v1.DT_FXUSHORT:   return v1.data.fxushort == v2.data.fxushort;
      case v1.DT_FXSHORT:    return v1.data.fxshort  == v2.data.fxshort;
      case v1.DT_FXUINT:     return v1.data.fxuint   == v2.data.fxuint;
      case v1.DT_FXINT:      return v1.data.fxint    == v2.data.fxint;
      case v1.DT_FXULONG:    return v1.data.fxulong  == v2.data.fxulong;
      case v1.DT_FXLONG:     return v1.data.fxlong   == v2.data.fxlong;
      case v1.DT_FXFLOAT:    return v1.data.fxfloat  == v2.data.fxfloat;
      case v1.DT_FXDOUBLE:   return v1.data.fxdouble == v2.data.fxdouble;
      case v1.DT_FXCHARSET:  return v1.fxcharset     == v2.fxcharset;
      case v1.DT_FXSTRING:   return v1.fxstring      == v2.fxstring;
      case v1.DT_FXWSTRING:  return v1.fxwstring     == v2.fxwstring;
      case v1.DT_FXOBJECT:   return v1.fxobject      == v2.fxobject;
      case v1.DT_POINTER:    return v1.pointer       == v2.pointer;
      case v1.DT_RAW:        return v1.raw.pointer==v2.raw.pointer && v1.raw.size==v2.raw.size;
      }
    }
  return FALSE;
  }

// operator !=
FXbool operator!=(const FXVariant &v1,const FXVariant &v2){
  if (v1.type == v2.type) {
    switch (v1.type) {
      case v1.DT_UNASSIGNED: return FALSE;
      case v1.DT_NULL:       return FALSE;
      case v1.DT_BOOL:       return v1.data.boolean  != v2.data.boolean;
      case v1.DT_FXUCHAR:    return v1.data.fxuchar  != v2.data.fxuchar;
      case v1.DT_FXCHAR:     return v1.data.fxchar   != v2.data.fxchar;
      case v1.DT_FXUSHORT:   return v1.data.fxushort != v2.data.fxushort;
      case v1.DT_FXSHORT:    return v1.data.fxshort  != v2.data.fxshort;
      case v1.DT_FXUINT:     return v1.data.fxuint   != v2.data.fxuint;
      case v1.DT_FXINT:      return v1.data.fxint    != v2.data.fxint;
      case v1.DT_FXULONG:    return v1.data.fxulong  != v2.data.fxulong;
      case v1.DT_FXLONG:     return v1.data.fxlong   != v2.data.fxlong;
      case v1.DT_FXFLOAT:    return v1.data.fxfloat  != v2.data.fxfloat;
      case v1.DT_FXDOUBLE:   return v1.data.fxdouble != v2.data.fxdouble;
      case v1.DT_FXCHARSET:  return v1.fxcharset     != v2.fxcharset;
      case v1.DT_FXSTRING:   return v1.fxstring      != v2.fxstring;
      case v1.DT_FXWSTRING:  return v1.fxwstring     != v2.fxwstring;
      case v1.DT_FXOBJECT:   return v1.fxobject      != v2.fxobject;
      case v1.DT_POINTER:    return v1.pointer       != v2.pointer;
      case v1.DT_RAW:        return v1.raw.pointer!=v2.raw.pointer || v1.raw.size!=v2.raw.size;
      }
    }
  return TRUE;
  }

}
