/********************************************************************************
*                                                                               *
*                  XML file/stream reader object                                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXXmlReader.h"
using namespace FXEX;
namespace FXEX {

/*
 * This FXXmlReader, in reality is really a 'tag' reader (or more specifically an SGML reader)
 *
 * TODO
 * - we still need suppport for serialisation, though I'm not sure how this would go
 *   if we were using a socket...
 */

// map
FXDEFMAP (FXXmlReader) FXXmlReaderMap[]={
  FXMAPFUNC(SEL_COMMAND,FXXmlReader::ID_XML_READER,FXXmlReader::onParse),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXXmlReader::onGetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXXmlReader::onSetIntValue),
  };
FXIMPLEMENT(FXXmlReader,FXBaseObject,FXXmlReaderMap,ARRAYNUMBER(FXXmlReaderMap))

// ctor
FXXmlReader::FXXmlReader(FXApp* a,FXStream* s,FXObject* tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  stream=s;
  tagMode=FALSE;
  state=FALSE;
  }

// dtor
FXXmlReader::~FXXmlReader(){
  if (tagMode && target) target->handle(this,FXSEL(SEL_TAG,message),(void*)&tag);
  stream=(FXStream*)-1;
  }

// Parse the stream...
//
// To handle the end of a tag:
// - we check to make sure that we are currently in 'tag mode',
//   if not we generate a parse error
// - note that this method 'has memory' of the last call to parse() so that if we called
//   parse() on a stream and the stream became empty just after the tag became open
//   then the second call to parse() will correctly close the tag.
// - then we generate a SEL_TAG message with the data inside the tag
//
// Note: we dont check content/tag length since we do want to actually generate the
//       content/tag event, without the contents/tag, since some clients may care about
//       the formatting of the origonal XML document.
//
// FIXME This implementation has a flaw in that, since this class is essentially XML-tag-
//       stateless, if the content contains, say some javascript with a greater than sign,
//       it will close the 'script' tag prematurely.  I guess a better way to implement it
//       would be to fire a FOX event when we received the tag ending, that way if the
//       target decides to return 1 from the handle, then we dont close the tag.
FXbool FXXmlReader::parse() {
  if (!stream) return FALSE;
  state=TRUE;
  FXchar ch;
  while (state && getNext(ch)){

    // Handle the end of a tag
    if (ch == '>') {
      if (!tagMode) { 
        readError(ERROR_UNMATCHED_CLOSE_BRACE);
        continue;
        }
      else {
        // forward the tag
        if (target) target->handle(this,FXSEL(SEL_TAG,message),(void*)&tag);
        tag.clear();
        tagMode=FALSE;
        continue;
        }
      }

    // Handle the start of a tag
    if (ch == '<') {
      // Forward the content...
      if (target) target->handle(this,FXSEL(SEL_CONTENT,message),(void*)&content);
      content.clear();
      tagMode=TRUE;
      }

    // If we are reading in a tag, we just update the tag string, otherwise
    // add any remaining stuff to the content
    if (tagMode) tag += ch;
    else content += ch;
    }

  // If we got here and we have some content, generate the content event
  if (content.length()) {
    if (target) target->handle(this,FXSEL(SEL_CONTENT,message),(void*)content.text());
    content.clear();
    }

  // If the parent stopped the parse, just return the state indicating that we can
  // continue parsing the stream, but only if the stream is not empty
  if (!state && stream->status()==FXStreamOK) return TRUE;

  // Otherwise, we must have hit the end of the stream
  // Note: just because we hit the end of the stream, doesn't mean more data wont
  //       become available soon, say from a socket.  It is then up to the owner
  //       to call parse() again to continue parsing the input stream.
  return FALSE;
  }

// helper - returns the next item from the stream if it can,
// otherwise it returns FALSE if no more characters are available
// Note: at the moment we are assuming that a stream is based on chars
//       but this may change to UTF8/16/32 in the long term
FXbool FXXmlReader::getNext(FXchar &c){
  if (stream && stream->status()==FXStreamOK) {
    *stream >> c;
    // TODO we could add a progress callback here
    if (stream->status() != FXStreamEnd) return TRUE;
    }
  return FALSE;
  }

// generate a 'read error' event
void FXXmlReader::readError(FXint errval) {
  if (target) target->handle(this,FXSEL(SEL_ERROR,message),(void*)&errval);
  }

// handle event to begin/stop parsing
long FXXmlReader::onParse(FXObject*,FXSelector,void*){
  if (state) stop();
  else parse();
  return 1;
  }

// handle GETINTVALUE request - forward the state
long FXXmlReader::onGetIntValue(FXObject *sender,FXSelector,void*){
  FXint val=(FXint)state;
  sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&val);
  return 1;
  }

// handle SETINTVALUE request - set the state
long FXXmlReader::onSetIntValue(FXObject*,FXSelector,void *ptr){
  *((FXint*)ptr)=state;
  return 1;
  }

}

