#include "fxex.h"
#define MAX_VALUE 10000000

/* FXThreadFunction demo
 *
 * This demo demonstrates the simple use of using a global function as the worker thread,
 * such that the thread is executed when a button is pressed.  The thread simulates a
 * long running calculation, which finally exits.
 *
 * When the thread has computed the value, it generates a SEL_THREAD event.  This event
 * signals the GUI that the worker thread is finished.
 *
 * By using an atomic variable to store the results of the calculation, and connecting
 * it to an atomic data target, the GUI automatically gets the updated value every time
 * the GUI runs through it SEL_UPDATE event loop.
 *
 * Note, for this demo we generate some sort of FOX event, so as to cause the implicate
 * SEL_UPDATE loop to be run ... normally you dont need to do this, but the demo appears
 * to be broken, unless you move the mouse over the widgets while the thread is runnning.
 */

// These properties are global
void thread_function();  // the worker thread
FXThreadFunction *thread;
FXAtomicInt calculation_result;

// main window
class MainWindow : public FXMainWindow {
  FXDECLARE(MainWindow)
private:
  FXAtomicIntDataTarget  datatarget;
protected:
  MainWindow(){}
public:
  enum {
    ID_MAINWINDOW=FXMainWindow::ID_LAST,
    ID_THREAD,
    ID_START_THREAD,
    ID_TIMER,
    ID_LAST
    };
public:
  long onClose(FXObject*,FXSelector,void*);
  long onThreadEvent(FXObject*,FXSelector,void*);
  long onStartThread(FXObject*,FXSelector,void*);
  long onTimer(FXObject*,FXSelector,void*);
public:
  MainWindow(FXApp *a);
  virtual void create();
  ~MainWindow();
  };

// map
FXDEFMAP(MainWindow) MainWindowMap[]={
  FXMAPFUNC(SEL_SIGNAL,MainWindow::ID_MAINWINDOW,MainWindow::onClose),
  FXMAPFUNC(SEL_CLOSE,MainWindow::ID_MAINWINDOW,MainWindow::onClose),
  // handle button event
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_START_THREAD,MainWindow::onStartThread),
  // generic event
  FXMAPFUNC(SEL_TIMEOUT,MainWindow::ID_TIMER,MainWindow::onTimer),
  // handle worker thread event,
  // - SEL_THREAD is generated from the signal() function call
  // - SEL_CREATE / SEL_DESTROY are implicately generated when a thread has
  //   just started or is about to stop
  FXMAPFUNC(SEL_THREAD,MainWindow::ID_THREAD,MainWindow::onThreadEvent),
  FXMAPFUNC(SEL_CREATE,MainWindow::ID_THREAD,MainWindow::onThreadEvent),
  FXMAPFUNC(SEL_DESTROY,MainWindow::ID_THREAD,MainWindow::onThreadEvent),
  };
FXIMPLEMENT(MainWindow,FXMainWindow,MainWindowMap,ARRAYNUMBER(MainWindowMap))

//*************************** Main window ************************************
MainWindow::MainWindow(FXApp *a) : FXMainWindow(a,"FXThreadFunction demo") {
  setTarget(this);
  setSelector(ID_MAINWINDOW);
  setX(50);
  setY(50);
  calculation_result=0;
  datatarget.connect(calculation_result);
  new FXTextField(this,10,&datatarget,FXDataTarget::ID_VALUE,TEXTFIELD_READONLY|TEXTFIELD_INTEGER|LAYOUT_FILL_X);
  new FXButton(this,"press to start/stop thread",NULL,this,ID_START_THREAD);
  }

// dtor
MainWindow::~MainWindow(){
  getApp()->removeTimeout(this,ID_TIMER);
  }

// create FOX resources
void MainWindow::create(){
  FXMainWindow::create();
  show();
  onTimer(NULL,0,NULL);
  }

// handle close window event
long MainWindow::onClose(FXObject*,FXSelector,void*){
  getApp()->exit();
  return 1;
  }

// handle a message from the thread
long MainWindow::onThreadEvent(FXObject*,FXSelector sel,void*){
  FXshort type=FXSELTYPE(sel);
  switch(type){
    case SEL_CREATE  : fxmessage("thread event: SEL_CREATE \n"); break;
    case SEL_DESTROY : fxmessage("thread event: SEL_DESTROY\n"); break;
    case SEL_THREAD  : fxmessage("thread event: SEL_THREAD \n"); break;
    default: fxmessage("thread event: %i\n",type);
    }
  return 1;
  }

// Start the thread.
// Note: using a thread function means that any/all forms of synchronisation must be handled
//       by global variables, etc.
// Since the 'calculation_result' variable is an atomic variable, we can simply sets its
// value to the 'MAX_VALUE' constant, to stop the worker thread.
long MainWindow::onStartThread(FXObject*,FXSelector,void*){
  if (calculation_result==0) thread->start();
  else calculation_result=MAX_VALUE;
  return 1;
  }

// generate a generic update event (needed for the demo only)
long MainWindow::onTimer(FXObject*,FXSelector,void*){
  getApp()->addTimeout(this,ID_TIMER,1000);
  return 1;
  }


// A worker thread is self contained - it shouldn't affect the performance
// of the main thread (not counting CPU usage, since the OS still schedules thread
// execution).
void thread_function(){

  // some interesting calculation
  while (calculation_result < MAX_VALUE){
    calculation_result++;

    // notice that we yield here - this to avoid the OS from giving the worker thread
    // too much CPU time, thus interfering with the general responsiveness of the GUI,
    // you may or may not want to so this...
    thread->yield();
    }

  // you can signal the main thread at anytime; here we do it at the end
  thread->signal();
  calculation_result=0;
  }


/* start it up */
int main(int argc, char *argv[]){
  FXApp app("FXThreadFunction","FoxExTest");
  app.init(argc,argv);
  MainWindow *w=new MainWindow(&app);
  app.create();
  thread=new FXThreadFunction(thread_function,w,MainWindow::ID_THREAD);
  return app.run();
  }

