/********************************************************************************
*                                                                               *
*                  Variant class - a container for any data type                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
* Copyright (C) 2003 by Mathew Robertson.       All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXVARIANT_H
#define FXVARIANT_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {

/**
 * Variant objects can (... its aim is to ...) be assignable to any data type.
 * They accept ID_GET/SET events so at to make connecting to widgets/datatargets easier.
 * FIXME: add support for FXWString
 *
 * Note: we only provide a single pair of operator==/operator!= for the FXVariant type
 *       since you can always use the getType() and 'asXXXXX()' methods to get the type
 *       you want.
 */
class FXAPI FXVariant : public FXBaseObject {
  FXDECLARE(FXVariant)

protected:
  FXint type;           // type of variant
  union {               // basic types
    bool boolean;
    FXuchar fxuchar;
    FXchar fxchar;
    FXushort fxushort;
    FXshort fxshort;
    FXuint fxuint;
    FXint fxint;
    FXulong fxulong;
    FXlong fxlong;
    FXfloat fxfloat;
    FXdouble fxdouble;
    } data;
  FXCharset fxcharset;  // charset
  FXString fxstring;    // string
  FXWString fxwstring;  // wide string
  FXObject *fxobject;	// point to FXObject type
  void *pointer;        // assignment to pointer
  struct {              // hold some raw data
    unsigned long size;
    void *pointer;
    } raw;

public:
  /**
   * Supported data types
   * - allows for subclassing (use your own data type)
   * - use the pointer types when you just want to point at anaother object
   */
  enum {
    DT_UNASSIGNED=-1,
    DT_NULL,
    DT_BOOL,
    DT_FXUCHAR,
    DT_FXCHAR,
    DT_FXUSHORT,
    DT_FXSHORT,
    DT_FXUINT,
    DT_FXINT,
    DT_FXULONG,
    DT_FXLONG,
    DT_FXFLOAT,
    DT_FXDOUBLE,
    DT_FXCHARSET,
    DT_FXSTRING,
    DT_FXWSTRING,
    DT_FXOBJECT,
    DT_POINTER,
    DT_RAW,
    DT_LAST
    };

public:
  enum {
    ID_VALUE=FXBaseObject::ID_LAST,
    ID_LAST
    };

public:
/*
  long onCmdValue(FXObject*,FXSelector,void*);
  long onUpdValue(FXObject*,FXSelector,void*);
*/
  long onCmdSetValue(FXObject*,FXSelector,void*);
  long onCmdSetIntValue(FXObject*,FXSelector,void*);
  long onCmdSetRealValue(FXObject*,FXSelector,void*);
  long onCmdSetStringValue(FXObject*,FXSelector,void*);
  long onCmdGetIntValue(FXObject*,FXSelector,void*);
  long onCmdGetRealValue(FXObject*,FXSelector,void*);
  long onCmdGetStringValue(FXObject*,FXSelector,void*);

public:
  /**
   * constructors for all the known types
   * takes a deep copy of data type (where possible)
   */
  FXVariant(FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXVariant &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const bool &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXuchar &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXchar &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXushort &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXshort &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXuint &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXint &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXulong &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXlong &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXfloat &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXdouble &v,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXCharset &c,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXString &s,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(const FXWString &s,FXObject *tgt=NULL,FXSelector sel=0);
  FXVariant(FXObject *o,FXObject *tgt=NULL,FXSelector sel=0); // point at FXObject
  FXVariant(void *p,FXObject *tgt=NULL,FXSelector sel=0); // pointer/reference
  FXVariant(const void *p,unsigned long size,FXObject *tgt=NULL,FXSelector sel=0); // raw data type

  /**
   * Return the number of bytes needed to store hold the data for the current type
   * this will often be the natural size of the data type
   */
  unsigned long getSize() const;

  /// clears/resets the contents of the variant
  virtual void clear();

  /// set the date type, to a specific type
  void setType(FXint type);

  /// return data type
  FXint getType() const { return type; }

  /// test data type
  FXbool isUnassigned() const   { return type==DT_UNASSIGNED; }
  FXbool isNull() const         { return type==DT_NULL; }
  FXbool isBool() const         { return type==DT_BOOL; }
  FXbool isFXuchar() const      { return type==DT_FXUCHAR; }
  FXbool isFXchar() const       { return type==DT_FXCHAR; }
  FXbool isFXushort() const     { return type==DT_FXUSHORT; }
  FXbool isFXshort() const      { return type==DT_FXSHORT; }
  FXbool isFXuint() const       { return type==DT_FXUINT; }
  FXbool isFXint() const        { return type==DT_FXINT; }
  FXbool isFXfloat() const      { return type==DT_FXFLOAT; }
  FXbool isFXdouble() const     { return type==DT_FXDOUBLE; }
  FXbool isFXCharset() const    { return type==DT_FXCHARSET; }
  FXbool isFXString() const     { return type==DT_FXSTRING; }
  FXbool isFXWString() const    { return type==DT_FXWSTRING; }
  FXbool isFXObject() const     { return type==DT_FXOBJECT; }
  FXbool isPointer() const      { return type==DT_POINTER; }
  FXbool isRaw() const          { return type==DT_RAW; }

  /// assignment operators
  FXVariant& operator=(const FXVariant &v);
  FXVariant& operator=(const bool &v);
  FXVariant& operator=(const FXuchar &v);
  FXVariant& operator=(const FXchar &v);
  FXVariant& operator=(const FXushort &v);
  FXVariant& operator=(const FXshort &v);
  FXVariant& operator=(const FXuint &v);
  FXVariant& operator=(const FXint &v);
  FXVariant& operator=(const FXulong &v);
  FXVariant& operator=(const FXlong &v);
  FXVariant& operator=(const FXfloat &v);
  FXVariant& operator=(const FXdouble &v);
  FXVariant& operator=(const FXCharset &c);
  FXVariant& operator=(const FXString &s);
  FXVariant& operator=(const FXWString &s);
  FXVariant& operator=(FXObject *o);
  FXVariant& operator=(void *p);  // set to new pointer

  /// copy raw data, of specified length
  void setData(const void *p,unsigned long size);

  /**
   * Try and convert the data to something useful; returns the result
   * Note: you may or may not get something useful out of this
   */
  virtual bool asBool() const;
  virtual FXuchar asFXuchar() const;
  virtual FXchar asFXchar() const;
  virtual FXushort asFXushort() const;
  virtual FXshort asFXshort() const;
  virtual FXuint asFXuint() const;
  virtual FXint asFXint() const;
  virtual FXulong asFXulong() const;
  virtual FXlong asFXlong() const;
  virtual FXfloat asFXfloat() const;
  virtual FXdouble asFXdouble() const;
  virtual FXCharset asFXCharset() const;
  virtual FXString asFXString() const;
  virtual FXWString asFXWString() const;
  virtual FXObject* asFXObject() const;
  virtual void* asPointer() const;
  virtual void* asRaw() const;
  virtual FXuval asAddress() const;  // special case for pointers

  /**
   * return operator()  (eg: as used in 'if' conditions)
   * FIXME: Should we throw an error if the required type is not the internal type,
   * rather than trying to convert it to that type?
   */
  operator bool() const		{ return asBool(); }
  operator FXuchar() const	{ return asFXuchar(); }
  operator FXchar() const	{ return asFXchar(); }
  operator FXushort() const	{ return asFXushort(); }
  operator FXshort() const	{ return asFXshort(); }
  operator FXuint() const	{ return asFXuint(); }
  operator FXint() const	{ return asFXint(); }
  operator FXulong() const	{ return asFXulong(); }
  operator FXlong() const	{ return asFXlong(); }
  operator FXfloat() const	{ return asFXfloat(); }
  operator FXdouble() const	{ return asFXdouble(); }
  operator FXCharset() const	{ return asFXCharset(); }
  operator FXString() const	{ return asFXString(); }
  operator FXWString() const	{ return asFXWString(); }
  operator FXObject*() const	{ return asFXObject(); }
  operator void*() const	{ return asPointer(); }

  /**
   * convert internal representation to different type, if possible
   * note: this may result in loss of data
   */
  FXVariant& toBool()		{ *this = asBool(); return *this; }
  FXVariant& toFXuchar()	{ *this = asFXuchar(); return *this; }
  FXVariant& toFXchar()		{ *this = asFXchar(); return *this; }
  FXVariant& toFXushort()	{ *this = asFXushort(); return *this; }
  FXVariant& toFXshort()	{ *this = asFXshort(); return *this; }
  FXVariant& toFXuint()		{ *this = asFXuint(); return *this; }
  FXVariant& toFXint()		{ *this = asFXint(); return *this; }
  FXVariant& toFXulong()	{ *this = asFXulong(); return *this; }
  FXVariant& toFXlong()		{ *this = asFXlong(); return *this; }
  FXVariant& toFXfloat()	{ *this = asFXfloat(); return *this; }
  FXVariant& toFXdouble()	{ *this = asFXdouble(); return *this; }
  FXVariant& toFXCharset()	{ *this = asFXCharset(); return *this; }
  FXVariant& toFXString()	{ *this = asFXString(); return *this; }
  FXVariant& toFXWString()	{ *this = asFXWString(); return *this; }
  FXVariant& toFXObject()	{ *this = asFXObject(); return *this; }
  FXVariant& toPointer()	{ *this = asPointer(); return *this; }
  FXVariant& toRaw();

  /// convert to a specific type
  FXVariant& convertTo(FXint type);

  /// equality...
  friend FXAPI FXbool operator==(const FXVariant &v1,const FXVariant &v2);

  /// in-equality...
  friend FXAPI FXbool operator!=(const FXVariant &v1,const FXVariant &v2);

  /// save to stream
  virtual void save(FXStream &store) const;

  /// load from stream
  virtual void load(FXStream &store);

  /// destructor (& cleanup)
  virtual ~FXVariant();
  };

} // namespace FXEX
#endif // FXVARIANT_H
