/********************************************************************************
*                                                                               *
*                  Database query-result interface                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#define _GNU_SOURCE 1
#include "fxexdefs.h"
#include "FXArray.h"
#include "FXDatabaseField.h"
#include "FXDatabaseInterface.h"
#include "FXDatabaseQuery.h"
using namespace FXEX;
namespace FXEX {

// map
FXDEFMAP(FXDatabaseQuery) FXDatabaseQueryMap[]={
  FXMAPFUNC(SEL_EVENT,      FXDatabaseQuery::ID_DISCONNECT,     FXDatabaseQuery::onDisconnect),
  FXMAPFUNC(SEL_EVENT,      FXDatabaseQuery::ID_CONNECT,        FXDatabaseQuery::onConnect),
  FXMAPFUNC(SEL_CHORE,      FXDatabaseQuery::ID_DESTROY,        FXDatabaseQuery::onDestroy),
  FXMAPFUNC(SEL_EVENT,      FXDatabaseQuery::ID_DESTROY,        FXDatabaseQuery::onFree),
};

FXIMPLEMENT_ABSTRACT(FXDatabaseQuery,FXBaseObject,FXDatabaseQueryMap,ARRAYNUMBER(FXDatabaseQueryMap))

FXDatabaseQuery::FXDatabaseQuery() : FXBaseObject() {}

FXDatabaseQuery::FXDatabaseQuery(FXDatabaseInterface *dbi, FXObject *tgt=NULL, FXSelector sel=0):FXBaseObject(dbi->getApp(),tgt,sel){
  database=dbi;
  query="";
  state = dbi ? rsClose : rsInvalid;
  readOnly = TRUE;
}

FXDatabaseQuery::~FXDatabaseQuery() {
  getApp()->removeChore(this,ID_DESTROY);
  FXDatabaseQuery::deleteFields();
  database=(FXDatabaseInterface*)-1;
}

FXint FXDatabaseQuery::findFldTarget(FXObject *tgt) {
  FXint x;
  FXint n = fldTargets.no();
  for(x=0; x<n; ++x)
    if(fldTargets[x].tgt==tgt) return x;
  return -1;
}

void FXDatabaseQuery::initTarget(FXint tgt) {
  FXObject *t = fldTargets[tgt].tgt;
  FXint m = fldTargets[tgt].msg;
  FXDatabaseField *fld;
  FXdbEvent e;
  e.state = state;
  e.dbi = database;
  e.dbq = this;
  e.dbf = NULL;
  e.data = (void *) NULL;
  if(!database) {
    e.type = dbDisconnect;
    t->handle(this, MKUINT(m, SEL_EVENT), &e);
  } else {
    e.type = dbConnect;
    t->handle(this, MKUINT(m, SEL_EVENT), &e);
    if(isOpen()) {
      bindTarget(tgt);
      e.type = dbOpen;
      t->handle(this, MKUINT(m, SEL_EVENT), &e);
      if(state==rsAddNew) {
        e.type = dbAddNew;
        t->handle(this, MKUINT(m, SEL_EVENT), &e);
      } else if(state==rsModify) {
        fld = fields[fldTargets[tgt].fldPos];
        fld->handle(this, MKUINT(FXDatabaseField::ID_REFRESHOLD, SEL_EVENT), &e);
        e.type = dbEdit;
        t->handle(this, MKUINT(m, SEL_EVENT), &e);
      } else {
        FXASSERT(state==rsRead);
        if(recordCount()>0) {
          fld = fields[fldTargets[tgt].fldPos];
          e.type = dbRefresh;
          fld->handle(this, MKUINT(FXDatabaseField::ID_EVENT, SEL_EVENT), &e);
        }
      }
    } else {
      e.type = dbClose;
      t->handle(this, MKUINT(m, SEL_EVENT), &e);
    }
  }
}

void FXDatabaseQuery::addFldTarget(FXObject *tgt, FXSelector msg, const FXString &fld) {
  FXfldTarget t;
  if(findFldTarget(tgt)>=0) FXDatabaseInterface::dbThrow("Target already selected", -1);
  t.tgt = tgt;
  t.msg = msg;
  t.fldName = fld;
  t.fldPos = -1;
  fldTargets.append(t);
  initTarget(fldTargets.no()-1);
}

void FXDatabaseQuery::addFldTarget(FXObject *tgt, FXSelector msg, FXint fld) {
  FXfldTarget t;
  if(findFldTarget(tgt)>=0) FXDatabaseInterface::dbThrow("Target already selected", -1);
  t.tgt = tgt;
  t.msg = msg;
  t.fldName = "";
  t.fldPos = fld;
  fldTargets.append(t);
  initTarget(fldTargets.no()-1);
}

void FXDatabaseQuery::removeFldTarget(FXObject *tgt) {
  FXint x = findFldTarget(tgt);
  if(x<0) FXDatabaseInterface::dbThrow("No target found", -1);
  if(isOpen()) { unbindTarget(x); };
  fldTargets.remove(x);
}

FXbool FXDatabaseQuery::isOpen() const {
  return (state!=rsInvalid) && (state!=rsClose);
}

FXbool FXDatabaseQuery::isEditable() const {
  return (state==rsAddNew) || (state==rsModify);
}

FXbool FXDatabaseQuery::bindTarget(FXint pos) {
  FXfldTarget *t = &fldTargets[pos];
  FXString n = t->fldName;
  FXint p = t->fldPos;
  FXASSERT(isOpen());
  if(p<0) {
    FXint x;
    FXint m = fields.no();
    for(x=0; x<m; ++x)
      if(!comparecase(n, fields[x]->name())) break;
    if(x<m) p = x;
  }
  if(p>=fields.no()) p = 0;
  if(p<0) {
    fxwarning("Cannot bind target\n");
    return FALSE;
  }
  fields[p]->setTarget(t->tgt);
  fields[p]->setSelector(t->msg);
  fldTargets[pos].fldPos = p;
  return TRUE;
}

FXbool FXDatabaseQuery::unbindTarget(FXint pos) {
  FXObject *tgt = fldTargets[pos].tgt;
  FXint x;
  FXint n = fields.no();
  for(x=0; x<n; ++x)
    if(fields[x]->getTarget()==tgt) {
      fields[x]->setTarget(NULL);
      fields[x]->setSelector(0);
      return TRUE;
    }
  fxwarning("Cannot unbind target");
  return FALSE;
}

void FXDatabaseQuery::checkState(FXrsState should_be) const {
  if(state!=should_be) FXDatabaseInterface::dbThrow("Invalid query state", -1);
  }

void FXDatabaseQuery::checkOpen(FXbool open) const {
  if(state==rsInvalid) FXDatabaseInterface::dbThrow("Invalid query state", -1);
  if(state==rsClose) {
    if(!open) return;
  } else {
    if(open) return;
  }
  FXDatabaseInterface::dbThrow(open ? "Query not open" : "query open", -1);
}

void FXDatabaseQuery::checkEditable(FXbool editable) const {
  checkOpen(TRUE);
  if(state==rsAddNew || state==rsModify) {
    if(!editable) FXDatabaseInterface::dbThrow("Missing Update or CancelUpdate", -1);
  } else {
    if(editable) FXDatabaseInterface::dbThrow("Query is not editable", -1);
  }
}

// save resouces to stream
void FXDatabaseQuery::save(FXStream& store) const {
  FXBaseObject::save(store);
  store << database;
  store << query;
  store << readOnly;
  store.save((FXint *) &state, sizeof(state));
  fields.save(store);
}

// load resources from stream
void FXDatabaseQuery::load(FXStream& store){
  FXBaseObject::load(store);
  store >> database;
  store >> query;
  store >> readOnly;
  store.load((FXint *) &state, sizeof(state));
  fields.load(store);
}

FXint FXDatabaseQuery::fieldCount() {
  checkOpen(TRUE);
  return fields.no();
}

void FXDatabaseQuery::broadcastMessage(FXdbEvType ev, void *data) {
  FXdbEvent e;
  FXint cnt = fields.no();
  e.type = ev;
  e.state = state;
  e.dbi = database;
  e.dbq = this;
  e.dbf = NULL;
  e.data = data;
  if(target) target->handle(this, MKUINT(message, SEL_EVENT), &e);
  while(cnt)
    fields[--cnt]->handle(this, MKUINT(FXDatabaseField::ID_EVENT, SEL_EVENT), &e);
}

void FXDatabaseQuery::deleteFields() {
  FXint cnt = fields.no();
  FXint x;
  for(x=0; x<cnt; ++x) {
    delete fields[x];
  }
  fields.clear();
}

long FXDatabaseQuery::onDisconnect(FXObject *, FXSelector, void *) {
  FXASSERT(database);
  if(isOpen()) {
    if(state!=rsRead) CancelUpdate();
    Close();
  }
  broadcastMessage(dbDisconnect);
  database = NULL;
  return 1;
}

long FXDatabaseQuery::onConnect(FXObject *, FXSelector, void *data) {
  FXASSERT(!database);
  database = (FXDatabaseInterface *) data;
  broadcastMessage(dbConnect);
  return 1;
}

void FXDatabaseQuery::Open(const FXString &query, FXbool readOnly) {
  FXint x;
  FXint n = fldTargets.no();
  this->query = query;
  this->readOnly = readOnly;
  state = rsRead;
  for(x=0; x<n; ++x) bindTarget(x);
  broadcastMessage(dbOpen);
}

void FXDatabaseQuery::Close() {
  FXint x;
  FXint n = fldTargets.no();
  for(x=0; x<n; ++x) unbindTarget(x);
  state = rsClose;
  broadcastMessage(dbClose);
}

void FXDatabaseQuery::addNew() {
  state = rsAddNew;
  broadcastMessage(dbAddNew);
}

void FXDatabaseQuery::Edit() {
  state = rsModify;
  broadcastMessage(dbEdit, (void *) currentPos());
}

void FXDatabaseQuery::Delete() {
  state = rsDelete;
  broadcastMessage(dbEdit, (void *) currentPos());
}

void FXDatabaseQuery::Update() {
  state = rsRead;
  broadcastMessage(dbUpdate);
}

void FXDatabaseQuery::CancelUpdate() {
  state = rsRead;
  broadcastMessage(dbCancelUpdate);
}

void FXDatabaseQuery::notifyMove(FXuint currentPos) {
  broadcastMessage(dbRefresh, (void *) currentPos);
}

void FXDatabaseQuery::Free() {
  if(database) database->handle(this, MKUINT(FXDatabaseInterface::ID_QRYDETACH, SEL_EVENT), NULL);
  database = NULL;
  handle(this, MKUINT(ID_DESTROY, SEL_EVENT), NULL);
}

long FXDatabaseQuery::onFree(FXObject *, FXSelector, void *) {
  broadcastMessage(dbDestroy);
  getApp()->addChore(this,ID_DESTROY);
  return 1;
}

long FXDatabaseQuery::onDestroy(FXObject *, FXSelector, void *) {
  delete this;
  return 1;
}

FXDatabaseField &FXDatabaseQuery::operator[](FXint pos) {
  checkOpen(TRUE);
  if(pos<0 || pos>=fields.no()) {
    FXDatabaseInterface::dbThrow("Bound error", -1);
  }
  return *fields[pos];
}

}
