/********************************************************************************
*                                                                               *
*                  S t r i n g    T o k e n i z e r    C l a s s                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXSTRINGTOKENIZER_H
#define FXSTRINGTOKENIZER_H

namespace FXEX {

/// String tokenizer options
enum {
  TOKENIZER_NORMAL=0,
  TOKENIZER_QUOTE_SINGLE=0x00000001,  // dont break into tokens within single quotes
  TOKENIZER_QUOTE_DOUBLE=0x00000002   // dont break into tokens within double quotes
  };

/**
 * Tokenise a string by the seperators.
 *
 * Note:
 * 1. The seperators are single characters such as tab, comma's.
 * 2. You can use multiple seperators at any one time.
 * 3. It will skip over multiple adjacent seperators, so as to retrieve the next/previous token.
 * 4. You can enable single and/or double quoting mode so as to avoid seperating tokens
 *    within the single/double quotes.  If a double quote occurs inside a pair of single
 *    quotes, then the double quote will be broken, since it occurs within a matched pair.
 *    This applies to the other case - a single quote will be broken if matched within
 *    a double quote pair.
 * 5. Double/single quotes can be uses as the seperators of tokens, though it wont make
 *    sense when used with the TOKENIZER_QUOTE_xxx modes.
 */
class FXAPI FXStringTokenizer {
  protected:
    FXString str;           // the string to be broken up
    FXString seperators;    // seperators that breaks up the tokens
    FXint index;            // the current location within the string
    FXuint options;         // string options
   
  protected:
    /// indicates if a character is one of the seperators
    FXbool isSep(const FXchar c);

  public:
    /// constuctors
    FXStringTokenizer(const FXString& s="",const FXString& seps=" \t\n\r",FXuint opts=TOKENIZER_NORMAL);
    FXStringTokenizer(const FXString& s,const FXchar* seps,FXuint opts=TOKENIZER_NORMAL);
    FXStringTokenizer(const FXString& s,const FXchar sep,FXuint opts=TOKENIZER_NORMAL);

    /// length of text
    FXint length() { return str.length(); }

    /// get the next token
    FXString next();

    /// has more next tokens
    FXbool hasNext();

    /// get the previous token
    FXString prev();

    /// has more previous tokens
    FXbool hasPrev();

    /// reset the tokenizer to the start of the string
    void reset() { index=-1; }

    /// reset the tokenizer to the end of the string (so you can traverse backwards)
    void resetEnd() { index=str.length(); }

    /// get the remaining text left at the end of the string
    FXString remaining() const { return str.right(str.length()-index+1); }

    /// get the remaining text left at the start of the string
    FXString begining() const { return str.left(index); }

    /// set the seperators
    void setSeperator(const FXchar sep) { seperators=FXString(sep,1); }
    void setSeperator(const FXchar* seps) { seperators=seps; }
    void setSeperators(const FXString& seps) { seperators=seps; }

    /// add other seperators to the current set
    void addSeperator(const FXchar sep) { seperators+=FXString(sep,1); }
    void addSeperators(const FXchar* seps) { seperators+=seps; }
    void addSeperators(const FXString& seps) { seperators+=seps; }

    /// get the current seperators
    FXString getSeperators() const { return seperators; }

    /// set the tokenized string to a new string
    void setText(const FXString& s);

    /// get the text
    FXString getText() const { return str; }

    /// set the tokenized string to a another tokenizer
    FXStringTokenizer& operator=(const FXStringTokenizer& s);

    /// set the tokenized string to a new string
    FXStringTokenizer& operator=(const FXString& s);

    /// Return a non-const reference to the ith character
    FXchar& operator[](FXint i){ return str[i]; }

    /// Return a const reference to the ith character
    const FXchar& operator[](FXint i) const { return str[i]; }

    /// Comparison operators ==
    friend FXAPI FXbool operator==(const FXStringTokenizer &s1,const FXStringTokenizer &s2);
    friend FXAPI FXbool operator==(const FXString &s1,const FXStringTokenizer &s2);
    friend FXAPI FXbool operator==(const FXStringTokenizer &s1,const FXString &s2);

    /// Comparison operators !=
    friend FXAPI FXbool operator!=(const FXStringTokenizer &s1,const FXStringTokenizer &s2);
    friend FXAPI FXbool operator!=(const FXString &s1,const FXStringTokenizer &s2);
    friend FXAPI FXbool operator!=(const FXStringTokenizer &s1,const FXString &s2);

    /// Saving to a stream
    friend FXAPI FXStream& operator<<(FXStream& store,const FXStringTokenizer& s);

    /// Load from a stream
    friend FXAPI FXStream& operator>>(FXStream& store,FXStringTokenizer& s);

    /// dtor
    virtual ~FXStringTokenizer();
  };

} // namespace FXEX
#endif // FXSTRINGTOKENIZER_H
