/********************************************************************************
*                                                                               *
*               Synchronisation mechanism                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXSYNCHRONISE_H
#define FXSYNCHRONISE_H

namespace FXEX {

/**
 * The following macros serialises a section of code in FXRunnable and
 * derived classes (ie FXThread)
 */
#define FXSYNCHRONISE    FXSynchronise _fxsynchronise_lock(*this);

/**
 * A Synchronise class serialises access to the class resources,
 * during the current block scope. This has the advantage that
 * mutex.unlock() will be called automatically when an exception
 * is thrown, ie. the 'synchronise' goes out of scope.
 * ie. auto lock/unlock of mutex
 *
 * As an alternative to:
 * {
 *   mutex.lock();
 *   .....
 *   mutex.unlock();
 * }
 *
 * you can use a single instance of the FXSynchronise class:
 * {
 *   FXSynchronise(mutex);
 *   ....
 * }
 *
 * for cases in FXRunnable derived classes, you could use the FXSYNCHRONISE
 * macro in FXThread derived clases:
 * {
 *   FXSYNCHRONISE
 *   ...
 * }
 */
class FXAPI FXSynchronise {
private:
  FXLockable& lockable;

private:
  // dummy copy constructor and operator= to prevent copying
  FXSynchronise(const FXSynchronise&);
  FXSynchronise& operator=(const FXSynchronise&);

public:
  /// on construction, lock the lockable
  inline FXSynchronise(FXLockable& l) : lockable(l) { lockable.lock(); }

  /// lock the mutex using the FXSYNCHRONISE macro
  inline FXSynchronise(FXRunnable& r) : lockable(*r.mutex) { lockable.lock(); }

  /// unlock the lockable on destruction
  inline ~FXSynchronise() { lockable.unlock(); }
  };

} // namespace FXEX
#endif // FXSYNCHRONISE_H
