/********************************************************************************
*                                                                               *
*                  Stream Buffering object                                      *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXBUFFEREDSTREAM_H
#define FXBUFFEREDSTREAM_H

#ifndef FXSTREAM
#include <fox/FXStream.h>
using namespace FX;
#endif
namespace FXEX {

/**
 * This class add support to FXStream's for:
 * 1. buffering
 * 2. stream enhancement such as compression, data munging, etc
 *
 *
 * This functionality is kinda cool, we can do something like this:
 * 
 * FXFileStream fstream = new FXFileStream(...);
 * fstream.open(...);
 * FXBufferedStream bstream = new FXBufferedStream(fstream);
 * ...
 * bstream << object;
 * ...
 * // when buffer becomes full (or closed), bstream automatically flushes into fstream
 *
 *
 * Or even better:
 *
 * FXSocketStream fstream = new FXSocketStream(...);
 * fstream.open(...);
 * FXBzipStream bstream = new FXBzipStream(fstream,compression_level);
 * ...
 * bstream << object;
 * ...
 * // when the buffer becomes full (or closed), bstream automatically flushes bzip
 * // compressed data over the socket!
 *
 *
 * Child should subclass this class and overload the bufferLoad() and bufferSave()
 * method so that they can mung the data when the buffer is full/empty.
 */
class FXAPI FXBufferedStream : public FXMemoryStream {
private:
  FXStream  *stream;     // embellish this stream

private:
  /// These are overloaded to generate the buffer full/empty events
  virtual void saveItems(const void *buf,unsigned long n);
  virtual void loadItems(void *buf,unsigned long n);

protected:
  /// overloaded by subclasses to handle a full buffer during a save condition
  virtual FXbool bufferSave();

  /// overloaded by subclasses to handle a buffer which has completed loading the current buffer
  virtual FXbool bufferLoad();

public:
  /// Point this enhanced stream at the base stream
  FXBufferedStream(FXStream* s,const FXObject* cont=NULL);

  /// Create an enhanced stream
  /// - the subclass should figure out how much space to allocate in the buffer (using setSpace())
  FXbool open(FXStreamDirection save_or_load);

  /// Create an enhanced stream of a specific buffer size
  FXbool open(FXuint sp,FXStreamDirection save_or_load);

  /// Stop using an enhanced stream
  FXbool close();

  /// Put the buffer into the base stream
  void flush();

  /// Save to stream
  virtual FXStream& operator<<(const FXuchar& v);
  virtual FXStream& operator<<(const FXchar& v);

  /// Load from stream
  virtual FXStream& operator>>(FXuchar& v);
  virtual FXStream& operator>>(FXchar& v);

  /// Destructor
  virtual ~FXBufferedStream();
  };

} // namespace FXEX
#endif // FXBUFFEREDSTREAM_H
